"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.totalForResource = exports.add = exports.totalMemory = exports.totalCPU = exports.ResourceStatus = exports.quantityToScalar = exports.findSuffix = exports.podsForNode = void 0;
const underscore_1 = require("underscore");
async function podsForNode(api, nodeName) {
    const allPods = await api.listPodForAllNamespaces();
    return allPods.body.items.filter((pod) => pod.spec.nodeName === nodeName);
}
exports.podsForNode = podsForNode;
function findSuffix(quantity) {
    let ix = quantity.length - 1;
    while (ix >= 0 && !/[\.0-9]/.test(quantity.charAt(ix))) {
        ix--;
    }
    return ix === -1 ? '' : quantity.substring(ix + 1);
}
exports.findSuffix = findSuffix;
function quantityToScalar(quantity) {
    if (!quantity) {
        return 0;
    }
    const suffix = findSuffix(quantity);
    if (suffix === '') {
        const num = Number(quantity).valueOf();
        if (isNaN(num)) {
            throw new Error('Unknown quantity ' + quantity);
        }
        return num;
    }
    switch (suffix) {
        case 'm':
            return Number(quantity.substr(0, quantity.length - 1)).valueOf() / 1000.0;
        case 'Ki':
            return BigInt(quantity.substr(0, quantity.length - 2)) * BigInt(1024);
        case 'Mi':
            return BigInt(quantity.substr(0, quantity.length - 2)) * BigInt(1024 * 1024);
        case 'Gi':
            return BigInt(quantity.substr(0, quantity.length - 2)) * BigInt(1024 * 1024 * 1024);
        case 'Ti':
            return (BigInt(quantity.substr(0, quantity.length - 2)) * BigInt(1024 * 1024 * 1024) * BigInt(1024));
        case 'Pi':
            return (BigInt(quantity.substr(0, quantity.length - 2)) *
                BigInt(1024 * 1024 * 1024) *
                BigInt(1024 * 1024));
        case 'Ei':
            return (BigInt(quantity.substr(0, quantity.length - 2)) *
                BigInt(1024 * 1024 * 1024) *
                BigInt(1024 * 1024 * 1024));
        default:
            throw new Error(`Unknown suffix: ${suffix}`);
    }
}
exports.quantityToScalar = quantityToScalar;
class ResourceStatus {
    constructor(request, limit, resourceType) {
        this.request = request;
        this.limit = limit;
        this.resourceType = resourceType;
    }
}
exports.ResourceStatus = ResourceStatus;
function totalCPU(pod) {
    return totalForResource(pod, 'cpu');
}
exports.totalCPU = totalCPU;
function totalMemory(pod) {
    return totalForResource(pod, 'memory');
}
exports.totalMemory = totalMemory;
function add(n1, n2) {
    if (underscore_1.isNumber(n1) && underscore_1.isNumber(n2)) {
        return n1 + n2;
    }
    if (underscore_1.isNumber(n1)) {
        return BigInt(Math.round(n1)) + n2;
    }
    else if (underscore_1.isNumber(n2)) {
        return n1 + BigInt(Math.round(n2));
    }
    return (n1 + n2);
}
exports.add = add;
function totalForResource(pod, resource) {
    let reqTotal = 0;
    let limitTotal = 0;
    pod.spec.containers.forEach((container) => {
        if (container.resources) {
            if (container.resources.requests) {
                reqTotal = add(reqTotal, quantityToScalar(container.resources.requests[resource]));
            }
            if (container.resources.limits) {
                limitTotal = add(limitTotal, quantityToScalar(container.resources.limits[resource]));
            }
        }
    });
    return new ResourceStatus(reqTotal, limitTotal, resource);
}
exports.totalForResource = totalForResource;
//# sourceMappingURL=util.js.map