"use strict";

const atRuleParamIndex = require("../../utils/atRuleParamIndex");
const mediaFeatureColonSpaceChecker = require("../mediaFeatureColonSpaceChecker");
const ruleMessages = require("../../utils/ruleMessages");
const validateOptions = require("../../utils/validateOptions");
const whitespaceChecker = require("../../utils/whitespaceChecker");

const ruleName = "media-feature-colon-space-before";

const messages = ruleMessages(ruleName, {
  expectedBefore: () => 'Expected single space before ":"',
  rejectedBefore: () => 'Unexpected whitespace before ":"'
});

const rule = function(expectation, options, context) {
  const checker = whitespaceChecker("space", expectation, messages);

  return (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
      actual: expectation,
      possible: ["always", "never"]
    });

    if (!validOptions) {
      return;
    }

    let fixData;

    mediaFeatureColonSpaceChecker({
      root,
      result,
      locationChecker: checker.before,
      checkedRuleName: ruleName,
      fix: context.fix
        ? (atRule, index) => {
            const paramColonIndex = index - atRuleParamIndex(atRule);

            fixData = fixData || new Map();
            const colonIndices = fixData.get(atRule) || [];

            colonIndices.push(paramColonIndex);
            fixData.set(atRule, colonIndices);

            return true;
          }
        : null
    });

    if (fixData) {
      fixData.forEach((colonIndices, atRule) => {
        let params = atRule.raws.params
          ? atRule.raws.params.raw
          : atRule.params;

        colonIndices
          .sort((a, b) => b - a)
          .forEach(index => {
            const beforeColon = params.slice(0, index);
            const afterColon = params.slice(index);

            if (expectation === "always") {
              params = beforeColon.replace(/\s*$/, " ") + afterColon;
            } else if (expectation === "never") {
              params = beforeColon.replace(/\s*$/, "") + afterColon;
            }
          });

        if (atRule.raws.params) {
          atRule.raws.params.raw = params;
        } else {
          atRule.params = params;
        }
      });
    }
  };
};

rule.ruleName = ruleName;
rule.messages = messages;
module.exports = rule;
