#!/usr/bin/env perl
# Copyright 2009 The Go Authors. All rights reserved.
# Use of this source code is governed by a BSD-style
# license that can be found in the LICENSE file.

use strict;

if($ENV{'GOARCH'} eq "" || $ENV{'GOOS'} eq "") {
	print STDERR "GOARCH or GOOS not defined in environment\n";
	exit 1;
}

# Check that we are using the new build system if we should
if($ENV{'GOLANG_SYS_BUILD'} ne "docker") {
	print STDERR "In the new build system, mksysnum should not be called directly.\n";
	print STDERR "See README.md\n";
	exit 1;
}

my $command = "$0 ". join(' ', @ARGV);

print <<EOF;
// $command
// Code generated by the command above; see README.md. DO NOT EDIT.

// +build $ENV{'GOARCH'},$ENV{'GOOS'}

package unix

const(
EOF

my $offset = 0;

sub fmt {
	my ($name, $num) = @_;
	if($num > 999){
		# ignore deprecated syscalls that are no longer implemented
		# https://git.kernel.org/cgit/linux/kernel/git/torvalds/linux.git/tree/include/uapi/asm-generic/unistd.h?id=refs/heads/master#n716
		return;
	}
	$name =~ y/a-z/A-Z/;
	$num = $num + $offset;
	print "	SYS_$name = $num;\n";
}

my $prev;
open(CC, "$ENV{'CC'} -E -dD @ARGV |") || die "can't run $ENV{'CC'}";
while(<CC>){
	if(/^#define __NR_Linux\s+([0-9]+)/){
		# mips/mips64: extract offset
		$offset = $1;
	}
	elsif(/^#define __NR(\w*)_SYSCALL_BASE\s+([0-9]+)/){
		# arm: extract offset
		$offset = $1;
	}
	elsif(/^#define __NR_syscalls\s+/) {
		# ignore redefinitions of __NR_syscalls
	}
	elsif(/^#define __NR_(\w*)Linux_syscalls\s+/) {
		# mips/mips64: ignore definitions about the number of syscalls
	}
	elsif(/^#define __NR_(\w+)\s+([0-9]+)/){
		$prev = $2;
		fmt($1, $2);
	}
	elsif(/^#define __NR3264_(\w+)\s+([0-9]+)/){
		$prev = $2;
		fmt($1, $2);
	}
	elsif(/^#define __NR_(\w+)\s+\(\w+\+\s*([0-9]+)\)/){
		fmt($1, $prev+$2)
	}
	elsif(/^#define __NR_(\w+)\s+\(__NR_Linux \+ ([0-9]+)/){
		fmt($1, $2);
	}
	elsif(/^#define __NR_(\w+)\s+\(__NR_SYSCALL_BASE \+ ([0-9]+)/){
		fmt($1, $2);
	}
}

print <<EOF;
)
EOF
