//
// Copyright (c) 2021 Red Hat, Inc.
// This program and the accompanying materials are made
// available under the terms of the Eclipse Public License 2.0
// which is available at https://www.eclipse.org/legal/epl-2.0/
//
// SPDX-License-Identifier: EPL-2.0
//
// Contributors:
//   Red Hat, Inc. - initial API and implementation
//
package che

import (
	"os"
	"reflect"
	"testing"

	orgv1 "github.com/eclipse/che-operator/pkg/apis/org/v1"
	"github.com/eclipse/che-operator/pkg/deploy"
	"github.com/eclipse/che-operator/pkg/util"
	"github.com/google/go-cmp/cmp"
	configv1 "github.com/openshift/api/config/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	fakeDiscovery "k8s.io/client-go/discovery/fake"
	fakeclientset "k8s.io/client-go/kubernetes/fake"
	"k8s.io/client-go/kubernetes/scheme"
	"sigs.k8s.io/controller-runtime/pkg/client/fake"
	"sigs.k8s.io/controller-runtime/pkg/log/zap"
	logf "sigs.k8s.io/controller-runtime/pkg/runtime/log"
)

func TestReadProxyConfiguration(t *testing.T) {
	type testCase struct {
		name              string
		openShiftVersion  string
		cheCluster        *orgv1.CheCluster
		clusterProxy      *configv1.Proxy
		initObjects       []runtime.Object
		expectedProxyConf *deploy.Proxy
	}

	testCases := []testCase{
		{
			name:             "Test no proxy configured",
			openShiftVersion: "4",
			clusterProxy: &configv1.Proxy{
				ObjectMeta: metav1.ObjectMeta{
					Name: "cluster",
				},
			},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
			},
			initObjects:       []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{},
		},
		{
			name:             "Test checluster proxy configured, OpenShift 4.x",
			openShiftVersion: "4",
			clusterProxy: &configv1.Proxy{
				ObjectMeta: metav1.ObjectMeta{
					Name: "cluster",
				},
			},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
				Spec: orgv1.CheClusterSpec{
					Server: orgv1.CheClusterSpecServer{
						ProxyURL:      "http://proxy",
						ProxyPort:     "3128",
						NonProxyHosts: "host1",
					},
				},
			},
			initObjects: []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{
				HttpProxy:        "http://proxy:3128",
				HttpUser:         "",
				HttpPassword:     "",
				HttpHost:         "proxy",
				HttpPort:         "3128",
				HttpsProxy:       "http://proxy:3128",
				HttpsUser:        "",
				HttpsPassword:    "",
				HttpsHost:        "proxy",
				HttpsPort:        "3128",
				NoProxy:          "host1,.svc",
				TrustedCAMapName: "",
			},
		},
		{
			name:             "Test checluster proxy configured, nonProxy merged, OpenShift 4.x",
			openShiftVersion: "4",
			clusterProxy: &configv1.Proxy{
				ObjectMeta: metav1.ObjectMeta{
					Name: "cluster",
				},
				Status: configv1.ProxyStatus{
					HTTPProxy:  "http://proxy:3128",
					HTTPSProxy: "http://proxy:3128",
					NoProxy:    "host2",
				},
			},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
				Spec: orgv1.CheClusterSpec{
					Server: orgv1.CheClusterSpecServer{
						ProxyURL:      "http://proxy",
						ProxyPort:     "3128",
						NonProxyHosts: "host1",
					},
				},
			},
			initObjects: []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{
				HttpProxy:        "http://proxy:3128",
				HttpUser:         "",
				HttpPassword:     "",
				HttpHost:         "proxy",
				HttpPort:         "3128",
				HttpsProxy:       "http://proxy:3128",
				HttpsUser:        "",
				HttpsPassword:    "",
				HttpsHost:        "proxy",
				HttpsPort:        "3128",
				NoProxy:          "host1,host2",
				TrustedCAMapName: "",
			},
		},
		{
			name:             "Test cluster wide proxy configured, OpenShift 4.x",
			openShiftVersion: "4",
			clusterProxy: &configv1.Proxy{
				ObjectMeta: metav1.ObjectMeta{
					Name: "cluster",
				},
				Status: configv1.ProxyStatus{
					HTTPProxy:  "http://proxy:3128",
					HTTPSProxy: "http://proxy:3128",
					NoProxy:    "host1",
				},
			},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
				Spec: orgv1.CheClusterSpec{
					Server: orgv1.CheClusterSpecServer{},
				},
			},
			initObjects: []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{
				HttpProxy:        "http://proxy:3128",
				HttpUser:         "",
				HttpPassword:     "",
				HttpHost:         "proxy",
				HttpPort:         "3128",
				HttpsProxy:       "http://proxy:3128",
				HttpsUser:        "",
				HttpsPassword:    "",
				HttpsHost:        "proxy",
				HttpsPort:        "3128",
				NoProxy:          "host1",
				TrustedCAMapName: "",
			},
		},
		{
			name:             "Test cluster wide proxy configured, nonProxy merged, OpenShift 4.x",
			openShiftVersion: "4",
			clusterProxy: &configv1.Proxy{
				ObjectMeta: metav1.ObjectMeta{
					Name: "cluster",
				},
				Status: configv1.ProxyStatus{
					HTTPProxy:  "http://proxy:3128",
					HTTPSProxy: "http://proxy:3128",
					NoProxy:    "host1",
				},
			},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
				Spec: orgv1.CheClusterSpec{
					Server: orgv1.CheClusterSpecServer{
						NonProxyHosts: "host2",
					},
				},
			},
			initObjects: []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{
				HttpProxy:        "http://proxy:3128",
				HttpUser:         "",
				HttpPassword:     "",
				HttpHost:         "proxy",
				HttpPort:         "3128",
				HttpsProxy:       "http://proxy:3128",
				HttpsUser:        "",
				HttpsPassword:    "",
				HttpsHost:        "proxy",
				HttpsPort:        "3128",
				NoProxy:          "host1,host2",
				TrustedCAMapName: "",
			},
		},
		{
			name:             "Test checluster proxy configured, OpenShift 3.x",
			openShiftVersion: "3",
			clusterProxy:     &configv1.Proxy{},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
				Spec: orgv1.CheClusterSpec{
					Server: orgv1.CheClusterSpecServer{
						ProxyURL:      "http://proxy",
						ProxyPort:     "3128",
						NonProxyHosts: "host1",
					},
				},
			},
			initObjects: []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{
				HttpProxy:        "http://proxy:3128",
				HttpUser:         "",
				HttpPassword:     "",
				HttpHost:         "proxy",
				HttpPort:         "3128",
				HttpsProxy:       "http://proxy:3128",
				HttpsUser:        "",
				HttpsPassword:    "",
				HttpsHost:        "proxy",
				HttpsPort:        "3128",
				NoProxy:          "host1",
				TrustedCAMapName: "",
			},
		},
		{
			name:             "Test checluster proxy configured, OpenShift 3.x and k8s, svc usage",
			openShiftVersion: "3",
			clusterProxy:     &configv1.Proxy{},
			cheCluster: &orgv1.CheCluster{
				ObjectMeta: metav1.ObjectMeta{
					Namespace: "eclipse-che",
				},
				Spec: orgv1.CheClusterSpec{
					Server: orgv1.CheClusterSpecServer{
						ProxyURL:                   "http://proxy",
						ProxyPort:                  "3128",
						NonProxyHosts:              "host1",
						UseInternalClusterSVCNames: true,
					},
				},
			},
			initObjects: []runtime.Object{},
			expectedProxyConf: &deploy.Proxy{
				HttpProxy:        "http://proxy:3128",
				HttpUser:         "",
				HttpPassword:     "",
				HttpHost:         "proxy",
				HttpPort:         "3128",
				HttpsProxy:       "http://proxy:3128",
				HttpsUser:        "",
				HttpsPassword:    "",
				HttpsHost:        "proxy",
				HttpsPort:        "3128",
				NoProxy:          "host1,.svc",
				TrustedCAMapName: "",
			},
		},
	}

	for _, testCase := range testCases {
		t.Run(testCase.name, func(t *testing.T) {
			logf.SetLogger(zap.LoggerTo(os.Stdout, true))
			orgv1.SchemeBuilder.AddToScheme(scheme.Scheme)
			testCase.initObjects = append(testCase.initObjects, testCase.clusterProxy, testCase.cheCluster)

			scheme := scheme.Scheme
			orgv1.SchemeBuilder.AddToScheme(scheme)
			scheme.AddKnownTypes(configv1.SchemeGroupVersion, &configv1.Proxy{})

			cli := fake.NewFakeClientWithScheme(scheme, testCase.initObjects...)
			nonCachedClient := fake.NewFakeClientWithScheme(scheme, testCase.initObjects...)
			clientSet := fakeclientset.NewSimpleClientset()
			fakeDiscovery, _ := clientSet.Discovery().(*fakeDiscovery.FakeDiscovery)
			fakeDiscovery.Fake.Resources = []*metav1.APIResourceList{}

			os.Setenv("OPENSHIFT_VERSION", testCase.openShiftVersion)
			util.IsOpenShift, util.IsOpenShift4, _ = util.DetectOpenShift()

			r := &ReconcileChe{
				client:          cli,
				nonCachedClient: nonCachedClient,
				discoveryClient: fakeDiscovery,
				scheme:          scheme,
			}

			actualProxyConf, err := r.getProxyConfiguration(testCase.cheCluster)
			if err != nil {
				t.Fatalf("Error reading proxy configuration: %v", err)
			}

			if !reflect.DeepEqual(testCase.expectedProxyConf, actualProxyConf) {
				t.Errorf("Expected deployment and deployment returned from API server differ (-want, +got): %v", cmp.Diff(testCase.expectedProxyConf, actualProxyConf))
			}
		})
	}
}
