"use strict";
/*
 * Copyright (C) 2017 TypeFox and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const lsp = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const logger_1 = require("./logger");
const snippet_proposal_provider_1 = require("./snippet-proposal-provider");
const text_document_1 = require("./text-document");
const text_editor_1 = require("./text-editor");
const types_1 = require("./types");
const utils_1 = require("./utils");
const window_1 = require("./window");
const workspace_1 = require("./workspace");
const goBuild_1 = require("../src/goBuild");
const commands_1 = require("./commands");
const goCodeAction_1 = require("../src/goCodeAction");
const goSuggest_1 = require("../src/goSuggest");
const goDeclaration_1 = require("../src/goDeclaration");
const goFormat_1 = require("../src/goFormat");
const goOutline_1 = require("../src/goOutline");
const goExtraInfo_1 = require("../src/goExtraInfo");
const goReferences_1 = require("../src/goReferences");
const goReferencesCodelens_1 = require("../src/goReferencesCodelens");
const goRunTestCodelens_1 = require("../src/goRunTestCodelens");
const goRename_1 = require("../src/goRename");
const goSignature_1 = require("../src/goSignature");
const goSymbol_1 = require("../src/goSymbol");
const vscode_1 = require("./vscode");
const activate_1 = require("./activate");
exports.WORKSPACE_EDIT_COMMAND = 'workspace-edit';
class LspServer {
    constructor(options) {
        this.options = options;
        this.openedDocumentUris = new Map();
        this.hasStartedInitialBuild = false;
        this.activation = new Promise((resolve, reject) => {
            this.activated = resolve;
        });
        this.logger = new logger_1.PrefixingLogger(options.logger, '[lspserver]');
    }
    initialize(params) {
        return __awaiter(this, void 0, void 0, function* () {
            this.logger.log('initialize', params);
            this.initializeParams = params;
            workspace_1.workspace.workspaceFolders.push(new workspace_1.WorkspaceFolder(vscode_uri_1.default.parse(params.rootUri)));
            this.initializeResult = {
                capabilities: {
                    textDocumentSync: lsp.TextDocumentSyncKind.Incremental,
                    completionProvider: {
                        triggerCharacters: ['.'],
                        resolveProvider: false
                    },
                    codeActionProvider: true,
                    codeLensProvider: {
                        resolveProvider: true
                    },
                    definitionProvider: true,
                    documentFormattingProvider: true,
                    documentHighlightProvider: true,
                    documentSymbolProvider: true,
                    executeCommandProvider: {
                        commands: [exports.WORKSPACE_EDIT_COMMAND]
                    },
                    hoverProvider: true,
                    renameProvider: true,
                    referencesProvider: true,
                    signatureHelpProvider: {
                        triggerCharacters: ['(', ',']
                    },
                    workspaceSymbolProvider: true,
                }
            };
            this.logger.log('onInitialize result', this.initializeResult);
            return this.initializeResult;
        });
    }
    initialized(params) {
        return __awaiter(this, void 0, void 0, function* () {
            this.logger.log('initialized', params);
            return activate_1.activate(this.options.lspClient, this, this.logger).then(() => {
                this.codeActionProvider = new goCodeAction_1.GoCodeActionProvider();
                this.completionItemProvider = new goSuggest_1.GoCompletionItemProvider();
                this.definitionProvider = new goDeclaration_1.GoDefinitionProvider();
                this.documentSymbolProvider = new goOutline_1.GoDocumentSymbolProvider();
                this.formattingProvider = new goFormat_1.GoDocumentFormattingEditProvider();
                this.hoverProvider = new goExtraInfo_1.GoHoverProvider();
                this.renameProvider = new goRename_1.GoRenameProvider();
                this.referenceProvider = new goReferences_1.GoReferenceProvider();
                this.referenceCodeLensProvider = new goReferencesCodelens_1.GoReferencesCodeLensProvider();
                this.signatureHelpProvider = new goSignature_1.GoSignatureHelpProvider();
                this.snippetProposalProvider = new snippet_proposal_provider_1.SnippetProposalProvider(this.logger);
                this.testCodeLensProvider = new goRunTestCodelens_1.GoRunTestCodeLensProvider();
                this.workspaceSymbolProvider = new goSymbol_1.GoWorkspaceSymbolProvider();
                this.activated();
            });
        });
    }
    didOpenTextDocument(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            const path = utils_1.uriToPath(params.textDocument.uri);
            this.logger.log('onDidOpenTextDocument', params, path);
            const document = new text_document_1.TextDocument(params.textDocument);
            this.openedDocumentUris.set(params.textDocument.uri, document);
            const editor = new text_editor_1.TextEditor(document, this.options.lspClient);
            window_1.window.visibleTextEditors.push(editor);
            window_1.window.activeTextEditor = editor;
            // doBuild requires an active editor so we start the initial build this late
            this.hasStartedInitialBuild = true;
            goBuild_1.buildCode(false);
        });
    }
    didCloseTextDocument(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            const path = utils_1.uriToPath(params.textDocument.uri);
            this.logger.log('onDidCloseTextDocument', params, path);
            this.openedDocumentUris.delete(params.textDocument.uri);
            const uri = vscode_uri_1.default.parse(params.textDocument.uri);
            let i = 0;
            for (; i < window_1.window.visibleTextEditors.length; ++i) {
                if (window_1.window.visibleTextEditors[i].document.uri === uri)
                    break;
            }
            if (i < window_1.window.visibleTextEditors.length)
                window_1.window.visibleTextEditors.splice(i);
        });
    }
    didChangeTextDocument(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('onDidCloseTextDocument', params, (document) => __awaiter(this, void 0, void 0, function* () {
                document.apply(params.contentChanges, params.textDocument.version);
            }));
        });
    }
    didSaveTextDocument(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('onDidSaveTextDocument', params, (document) => __awaiter(this, void 0, void 0, function* () {
                document.save();
            })).then(() => goBuild_1.buildCode(false));
        });
    }
    definition(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('definition', params, (document) => __awaiter(this, void 0, void 0, function* () {
                const definition = yield this.definitionProvider.provideDefinition(document, params.position, lsp.CancellationToken.None);
                if (definition)
                    return definition;
                else
                    return [];
            }));
        });
    }
    documentSymbol(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('symbol', params, (document) => __awaiter(this, void 0, void 0, function* () {
                const symbolInformation = yield this.documentSymbolProvider.provideDocumentSymbols(document, lsp.CancellationToken.None);
                symbolInformation.forEach(symbol => {
                    symbol.location = new types_1.Location(symbol.range, utils_1.uriToStringUri(document.uri));
                });
                return symbolInformation;
            }));
        });
    }
    completion(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('completion', params, (document) => __awaiter(this, void 0, void 0, function* () {
                const items = yield this.completionItemProvider.provideCompletionItems(document, params.position, lsp.CancellationToken.None);
                return {
                    isIncomplete: false,
                    items: items.concat(this.snippetProposalProvider.proposals)
                };
            }));
        });
    }
    hover(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('hover', params, (document) => __awaiter(this, void 0, void 0, function* () {
                return this.hoverProvider.provideHover(document, params.position, lsp.CancellationToken.None);
            }));
        });
    }
    rename(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('onRename', params, (document) => __awaiter(this, void 0, void 0, function* () {
                return this.renameProvider.provideRenameEdits(document, params.position, params.newName, lsp.CancellationToken.None);
            }));
        });
    }
    references(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('onReferences', params, (document) => __awaiter(this, void 0, void 0, function* () {
                return this.referenceProvider.provideReferences(document, params.position, { includeDeclaration: true }, lsp.CancellationToken.None);
            }));
        });
    }
    documentFormatting(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('format', params, (document) => __awaiter(this, void 0, void 0, function* () {
                return this.formattingProvider.provideDocumentFormattingEdits(document, params.options, lsp.CancellationToken.None);
            }));
        });
    }
    signatureHelp(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('signatureHelp', params, (document) => __awaiter(this, void 0, void 0, function* () {
                return this.signatureHelpProvider.provideSignatureHelp(document, params.position, lsp.CancellationToken.None);
            }));
        });
    }
    codeAction(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('codeAction', params, (document) => __awaiter(this, void 0, void 0, function* () {
                return this.codeActionProvider.provideCodeActions(document, new types_1.Range(params.range), params.context, lsp.CancellationToken.None);
            }));
        });
    }
    codeLens(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            return this.executeOnDocument('codeLens', params, (document) => __awaiter(this, void 0, void 0, function* () {
                const referenceCodeLenses = yield this.referenceCodeLensProvider.provideCodeLenses(document, lsp.CancellationToken.None);
                const testCodeLenses = yield this.testCodeLensProvider.provideCodeLenses(document, lsp.CancellationToken.None);
                const allCodeLenses = referenceCodeLenses.concat(testCodeLenses);
                allCodeLenses.forEach(codeLens => codeLens.data = {
                    textDocument: params.textDocument
                });
                return allCodeLenses;
            }));
        });
    }
    codeLensResolve(codeLens) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            if (!codeLens.command) {
                codeLens.document = this.getOpenDocument(codeLens.data.textDocument.uri);
                return this.referenceCodeLensProvider.resolveCodeLens(codeLens, lsp.CancellationToken.None);
            }
            else {
                return codeLens;
            }
        });
    }
    executeCommand(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            this.logger.log('executeCommand', params);
            const args = params.arguments;
            const document = this.getOpenDocument(args[args.length - 2]);
            const selection = args[args.length - 1];
            this.activateEditor(document, selection);
            return commands_1.commands.executeCommand(params.command, ...args.slice(0, -2));
        });
    }
    documentHighlight(arg) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            this.logger.log('documentHighlight', arg);
            // TODO
            return [];
        });
    }
    workspaceSymbol(params) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.activation;
            this.logger.log('symbol', params);
            return this.workspaceSymbolProvider.provideWorkspaceSymbols(params.query, lsp.CancellationToken.None);
        });
    }
    executeOnDocument(serviceName, params, lambda) {
        const path = utils_1.uriToPath(params.textDocument.uri);
        this.logger.log(serviceName, params, path);
        const document = this.getOpenDocument(params.textDocument.uri);
        this.activateEditor(document);
        return lambda.call(this, document).catch(err => {
            this.options.lspClient.showMessage({
                message: err,
                type: lsp.MessageType.Error
            });
            throw new lsp.ResponseError(lsp.ErrorCodes.InvalidRequest, err, err);
        });
    }
    getOpenDocument(uri) {
        const doc = this.openedDocumentUris.get(uri);
        if (doc) {
            return doc;
        }
        else {
            throw new Error('Document ' + uri + ' has not been opened.');
        }
    }
    activateEditor(document, selection) {
        window_1.window.activeTextEditor = window_1.window.visibleTextEditors.find(editor => editor.document.uri === document.uri);
        if (selection)
            window_1.window.activeTextEditor.selection = new vscode_1.Selection(new types_1.Range(selection.start, selection.end));
        return document;
    }
}
exports.LspServer = LspServer;
//# sourceMappingURL=lsp-server.js.map