"use strict";
/*
 * Copyright (C) 2017 TypeFox and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai = require("chai");
const path = require("path");
const lsp = require("vscode-languageserver");
const util_1 = require("util");
const lsp_server_1 = require("./lsp-server");
const test_utils_1 = require("./test-utils");
const logger_1 = require("./logger");
const utils_1 = require("./utils");
const text_document_1 = require("./text-document");
const goPackages_1 = require("../src/goPackages");
const vscode_1 = require("./vscode");
const activate_1 = require("./activate");
const os_1 = require("os");
const goBuild_1 = require("../src/goBuild");
function position(document, match, count = 1) {
    const doc = new text_document_1.TextDocument(document);
    let i = 0;
    let idx = -1;
    while (i++ < count) {
        idx = doc.getText().indexOf(match, idx + 1);
    }
    const pos = doc.positionAt(idx);
    return {
        line: pos.line,
        character: pos.character
    };
}
exports.position = position;
const assert = chai.assert;
const expect = chai.expect;
let diagnosticResolve = [];
let server;
let client;
before(() => __awaiter(this, void 0, void 0, function* () {
    client = {
        publishDiagnostics(args) {
            if (diagnosticResolve.length > 0) {
                const resolve = diagnosticResolve.splice(0, 1)[0];
                if (resolve)
                    resolve(args);
            }
        },
        showMessage(args) {
            throw args; // should not be called.
        },
        logMessage(args) {
            console.log(args.message);
        },
        applyWorkspaceEdit(args) {
            return __awaiter(this, void 0, void 0, function* () {
                throw new Error('unsupported');
            });
        },
        sendTelemetryEvent(args) { },
        showMessageRequest(args) {
            return __awaiter(this, void 0, void 0, function* () {
                throw new Error('unsupported');
            });
        },
        showInformationMessage(msg, ...options) {
            return __awaiter(this, void 0, void 0, function* () {
                // throw new Error('unsupported');
                return Promise.resolve('');
            });
        },
        registerCapability(args) {
            return __awaiter(this, void 0, void 0, function* () {
                // throw new Error('unsupported');
            });
        }
    };
    const logger = new logger_1.ConsoleLogger();
    server = new lsp_server_1.LspServer({
        logger: logger,
        lspClient: client,
    });
    vscode_1.workspace.getConfiguration('go')['toolsGopath'] = path.resolve(os_1.homedir(), 'go');
    yield server.initialize({
        rootPath: undefined,
        rootUri: test_utils_1.uri(test_utils_1.testWorkspacePath()),
        processId: 42,
        capabilities: {}
    });
    activate_1.mockActivate(client);
}));
function openNewDocument(fileName, contents) {
    const path = test_utils_1.testFilePath(fileName);
    test_utils_1.writeContents(path, contents);
    const doc = {
        uri: test_utils_1.uri(path),
        languageId: 'go',
        version: 1,
        text: contents
    };
    server.didOpenTextDocument({
        textDocument: doc
    });
    return doc;
}
exports.openNewDocument = openNewDocument;
describe('completion', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package main
			
			func main() {
				fmt.Println("hello world")
			}	
		`);
        yield goPackages_1.getImportablePackages(utils_1.uriToPath(doc.uri), true);
        const pos = position(doc, 'fmt.');
        pos.character += 4;
        const proposals = yield server.completion({
            textDocument: doc,
            position: pos
        });
        assert.isTrue(proposals.items.length > 40);
        const item = proposals.items[0];
        assert.isTrue(item.detail !== undefined);
        assert.isTrue(proposals.items.some(item => item.insertTextFormat === undefined), 'Expected text proposals');
        assert.isTrue(proposals.items.some(item => item.insertTextFormat === lsp.InsertTextFormat.Snippet), 'Expected snippet proposals');
        assert.isTrue(proposals.items.some(item => {
            if (item.additionalTextEdits !== undefined) {
                const additionalText = item.additionalTextEdits[0].newText;
                return additionalText.indexOf('import') >= 0 && additionalText.indexOf('"fmt"') >= 0;
            }
            return false;
        }));
    }));
});
describe('definition', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package main
			
			func main() {
				foo()
			}	

			func foo() {
			}
		`);
        yield goPackages_1.getImportablePackages(utils_1.uriToPath(doc.uri), true);
        const pos = position(doc, 'foo');
        const definition = yield server.definition({
            textDocument: doc,
            position: pos
        });
        const definitionPos = position(doc, 'foo', 2);
        assert.isFalse(util_1.isArray(definition));
        assert.equal(definitionPos.line, definition.range.start.line);
        assert.equal(definitionPos.character, definition.range.start.character);
    }));
});
describe('diagnostics', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package main
			
			func main() {
				foo2()
			}	

			func foo() {
			}
		`);
        let diagnosticPromise0 = new Promise((resolve, reject) => {
            diagnosticResolve.push(resolve);
        });
        let diagnosticPromise1 = new Promise((resolve, reject) => {
            diagnosticResolve.push(resolve);
        });
        goBuild_1.buildCode(false);
        const diagnostics0 = (yield diagnosticPromise0).diagnostics;
        assert.equal(0, diagnostics0.length);
        const diagnostics1 = (yield diagnosticPromise1).diagnostics;
        assert.equal(1, diagnostics1.length);
        assert.equal('undefined: foo2', diagnostics1[0].message);
    }));
});
describe('symbol', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package main
			
			import "fmt"

			func main() {
				fmt.Println("");
			}	

			func foo() {
			}
		`);
        const symbols = yield server.documentSymbol({
            textDocument: doc
        });
        assert.equal(3, symbols.length);
        assert.equal('main', symbols[0].name);
        assert.equal('main', symbols[1].name);
        assert.equal('foo', symbols[2].name);
    }));
});
describe('formatting', () => {
    it('full document formatting', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', ` package
test 
import  
"fmt"

func foo(x string) { 
fmt   . Println("    ");	foo("")
}`);
        const edits = yield server.documentFormatting({
            textDocument: doc,
            options: {
                tabSize: 4,
                insertSpaces: true
            }
        });
        const result = vscode_1.applyEdits(doc.text, edits);
        assert.equal(`package test

import "fmt"

func foo(x string) {
	fmt.Println("    ")
	foo("")
}
`, result);
    }));
});
describe('signatureHelp', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package test
			func foo(x string, baz bool) {
				foo("",true)
			}			
		`);
        let result = yield server.signatureHelp({
            textDocument: doc,
            position: position(doc, '""')
        });
        assert.equal('x string', result.signatures[result.activeSignature].parameters[result.activeParameter].label);
        result = yield server.signatureHelp({
            textDocument: doc,
            position: position(doc, 'true)')
        });
        assert.equal('baz bool', result.signatures[result.activeSignature].parameters[result.activeParameter].label);
    }));
});
describe('hover', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package test
			// Foo is dangerous 
			func Foo(x string, baz bool) {
				return Foo("",true);
			}			
		`);
        let result = yield server.hover({
            textDocument: doc,
            position: position(doc, 'Foo(x')
        });
        assert.equal('Foo func(x string, baz bool)', result.contents[0].value);
        assert.equal('Foo is dangerous\n', result.contents[1]);
    }));
});
// describe('rename', () => {
// 	it('simple test', async () => {
// 		const doc = openNewDocument('main.go', `
// 			package test
//
// 			func foo(x string) string {
// 				return foo("");
// 			}
// 		`);
// 		let result = await server.rename({
// 			textDocument: doc,
// 			position: position(doc, 'foo'),
// 			newName: 'bar'
// 		});
// 		// TODO expectation
// 	});
// });
describe('references', () => {
    it('simple test', () => __awaiter(this, void 0, void 0, function* () {
        const doc = openNewDocument('main.go', `
			package test
			
			func foo(x string) string {
				return foo("");
			}
		`);
        let result = yield server.references({
            textDocument: doc,
            position: position(doc, 'foo')
        });
        assert.equal(2, result.length);
    }));
});
//# sourceMappingURL=lsp-server.spec.js.map