"use strict";
/*
 * Copyright (C) 2017 TypeFox and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const lsp = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
function applyEdits(before, edits) {
    const sorted = edits.sort((a, b) => {
        if (a.range.start.line === b.range.start.line) {
            return a.range.start.character - b.range.start.character;
        }
        return a.range.start.line - b.range.start.line;
    });
    const doc = lsp.TextDocument.create('', '', 0, before);
    let currentDoc = '';
    let offset = 0;
    for (const edit of sorted) {
        const startOffset = doc.offsetAt(edit.range.start);
        currentDoc += before.substr(offset, startOffset - offset) + edit.newText;
        offset = doc.offsetAt(edit.range.end);
    }
    return currentDoc + before.substr(offset);
}
exports.applyEdits = applyEdits;
class TextDocument {
    constructor(doc) {
        this.version = 0;
        this.lastAccessed = new Date().getTime();
        this.text = doc.text;
        this.uri = vscode_uri_1.default.parse(doc.uri);
        if (lsp.VersionedTextDocumentIdentifier.is(doc)) {
            this.version = doc.version;
        }
    }
    get lines() {
        return this.text.split('\n');
    }
    get lineCount() {
        return this.lines.length;
    }
    get languageId() {
        return 'go';
    }
    getText(range) {
        if (!range)
            return this.text;
        const offset = this.offsetAt(range.start);
        const length = this.offsetAt(range.end) - offset;
        return this.text.substr(offset, length);
    }
    getWordRangeAtPosition(position) {
        const lines = this.lines;
        const line = Math.min(lines.length - 1, Math.max(0, position.line));
        const lineText = lines[line];
        const character = Math.min(lineText.length - 1, Math.max(0, position.character));
        let startChar = character;
        while (startChar > 0 && !/\s/.test(lineText.charAt(startChar - 1)))
            --startChar;
        let endChar = character;
        while (endChar < lineText.length - 1 && !/\s/.test(lineText.charAt(endChar)))
            ++endChar;
        if (startChar === endChar)
            return undefined;
        else
            return lsp.Range.create(line, startChar, line, endChar);
    }
    lineAt(line) {
        return {
            text: this.lines[line]
        };
    }
    getPosition(offset) {
        if (offset > this.text.length) {
            throw new Error('offset ' + offset + ' is out of bounds. Document length was ' + this.text.length);
        }
        const lines = this.lines;
        let currentOffSet = 0;
        for (let i = 0; i < lines.length; i++) {
            const l = lines[i];
            if (currentOffSet + l.length > offset) {
                return {
                    line: i,
                    character: offset - currentOffSet
                };
            }
            else {
                currentOffSet += l.length + 1;
            }
        }
        return {
            line: lines.length - 1,
            character: lines[lines.length - 1].length
        };
    }
    positionAt(offset) {
        return this.getPosition(offset);
    }
    offsetAt(position) {
        const lines = this.text.split('\n');
        let currentOffSet = 0;
        for (let i = 0; i < lines.length; i++) {
            const l = lines[i];
            if (position.line === i) {
                if (l.length < position.character) {
                    throw new Error(`Position ${JSON.stringify(position)} is out of range. Line [${i}] only has length ${l.length}.`);
                }
                return currentOffSet + position.character;
            }
            else {
                currentOffSet += l.length + 1;
            }
        }
        throw new Error(`Position ${JSON.stringify(position)} is out of range. Document only has ${lines.length} lines.`);
    }
    apply(contentChanges, version) {
        this.applyEdits(contentChanges.map(e => {
            const range = e.range || lsp.Range.create(lsp.Position.create(0, 0), this.getPosition(e.rangeLength || 0));
            return lsp.TextEdit.replace(range, e.text);
        }));
        this.version = version;
    }
    applyEdits(edits) {
        this.text = applyEdits(this.text, edits);
        this.lastAccessed = new Date().getTime();
    }
    save() {
        return __awaiter(this, void 0, void 0, function* () {
            // TODO sync with disc?
        });
    }
    get fileName() {
        return this.uri.fsPath;
    }
    get isUntitled() {
        return false;
    }
    get isDirty() {
        return false;
    }
}
exports.TextDocument = TextDocument;
class TextDocumentChangeEvent {
}
exports.TextDocumentChangeEvent = TextDocumentChangeEvent;
//# sourceMappingURL=text-document.js.map