"use strict";
/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
const vscode = require("vscode");
const path = require("path");
const goPath_1 = require("./goPath");
const cp = require("child_process");
const vscode_extension_telemetry_1 = require("vscode-extension-telemetry");
const fs = require("fs");
const os = require("os");
const goStatus_1 = require("./goStatus");
const goMain_1 = require("./goMain");
const extensionId = 'lukehoban.Go';
const extensionVersion = vscode.extensions.getExtension(extensionId).packageJSON.version;
const aiKey = 'AIF-d9b70cd4-b9f9-4d70-929b-a071c400b217';
exports.goKeywords = [
    'break',
    'case',
    'chan',
    'const',
    'continue',
    'default',
    'defer',
    'else',
    'fallthrough',
    'for',
    'func',
    'go',
    'goto',
    'if',
    'import',
    'interface',
    'map',
    'package',
    'range',
    'return',
    'select',
    'struct',
    'switch',
    'type',
    'var'
];
let goVersion = null;
let vendorSupport = null;
let telemtryReporter;
let toolsGopath;
function byteOffsetAt(document, position) {
    let offset = document.offsetAt(position);
    let text = document.getText();
    return Buffer.byteLength(text.substr(0, offset));
}
exports.byteOffsetAt = byteOffsetAt;
function parseFilePrelude(text) {
    let lines = text.split('\n');
    let ret = { imports: [], pkg: null };
    for (let i = 0; i < lines.length; i++) {
        let line = lines[i];
        let pkgMatch = line.match(/^(\s)*package(\s)+(\w+)/);
        if (pkgMatch) {
            ret.pkg = { start: i, end: i, name: pkgMatch[3] };
        }
        if (line.match(/^(\s)*import(\s)+\(/)) {
            ret.imports.push({ kind: 'multi', start: i, end: -1 });
        }
        if (line.match(/^(\s)*import(\s)+[^\(]/)) {
            ret.imports.push({ kind: 'single', start: i, end: i });
        }
        if (line.match(/^(\s)*\)/)) {
            if (ret.imports[ret.imports.length - 1].end === -1) {
                ret.imports[ret.imports.length - 1].end = i;
            }
        }
        if (line.match(/^(\s)*(func|const|type|var)/)) {
            break;
        }
    }
    return ret;
}
exports.parseFilePrelude = parseFilePrelude;
// Takes a Go function signature like:
//     (foo, bar string, baz number) (string, string)
// and returns an array of parameter strings:
//     ["foo", "bar string", "baz string"]
// Takes care of balancing parens so to not get confused by signatures like:
//     (pattern string, handler func(ResponseWriter, *Request)) {
function parameters(signature) {
    let ret = [];
    let parenCount = 0;
    let lastStart = 1;
    for (let i = 1; i < signature.length; i++) {
        switch (signature[i]) {
            case '(':
                parenCount++;
                break;
            case ')':
                parenCount--;
                if (parenCount < 0) {
                    if (i > lastStart) {
                        ret.push(signature.substring(lastStart, i));
                    }
                    return ret;
                }
                break;
            case ',':
                if (parenCount === 0) {
                    ret.push(signature.substring(lastStart, i));
                    lastStart = i + 2;
                }
                break;
        }
    }
    return null;
}
exports.parameters = parameters;
function canonicalizeGOPATHPrefix(filename) {
    let gopath = getCurrentGoPath();
    if (!gopath)
        return filename;
    let workspaces = gopath.split(path.delimiter);
    let filenameLowercase = filename.toLowerCase();
    // In case of multiple workspaces, find current workspace by checking if current file is
    // under any of the workspaces in $GOPATH
    let currentWorkspace = null;
    for (let workspace of workspaces) {
        // In case of nested workspaces, (example: both /Users/me and /Users/me/a/b/c are in $GOPATH)
        // both parent & child workspace in the nested workspaces pair can make it inside the above if block
        // Therefore, the below check will take longer (more specific to current file) of the two
        if (filenameLowercase.substring(0, workspace.length) === workspace.toLowerCase()
            && (!currentWorkspace || workspace.length > currentWorkspace.length)) {
            currentWorkspace = workspace;
        }
    }
    if (!currentWorkspace)
        return filename;
    return currentWorkspace + filename.slice(currentWorkspace.length);
}
exports.canonicalizeGOPATHPrefix = canonicalizeGOPATHPrefix;
/**
 * Gets version of Go based on the output of the command `go version`.
 * Returns null if go is being used from source/tip in which case `go version` will not return release tag like go1.6.3
 */
function getGoVersion() {
    let goRuntimePath = goPath_1.getGoRuntimePath();
    if (!goRuntimePath) {
        vscode.window.showInformationMessage('Cannot find "go" binary. Update PATH or GOROOT appropriately');
        return Promise.resolve(null);
    }
    if (goVersion) {
        /* __GDPR__
           "getGoVersion" : {
              "version" : { "classification": "SystemMetaData", "purpose": "FeatureInsight" }
           }
         */
        sendTelemetryEvent('getGoVersion', { version: `${goVersion.major}.${goVersion.minor}` });
        return Promise.resolve(goVersion);
    }
    return new Promise((resolve, reject) => {
        cp.execFile(goRuntimePath, ['version'], {}, (err, stdout, stderr) => {
            let matches = /go version go(\d).(\d).*/.exec(stdout);
            if (matches) {
                goVersion = {
                    major: parseInt(matches[1]),
                    minor: parseInt(matches[2])
                };
                /* __GDPR__
                   "getGoVersion" : {
                      "version" : { "classification": "SystemMetaData", "purpose": "FeatureInsight" }
                   }
                 */
                sendTelemetryEvent('getGoVersion', { version: `${goVersion.major}.${goVersion.minor}` });
            }
            else {
                /* __GDPR__
                   "getGoVersion" : {
                      "version" : { "classification": "SystemMetaData", "purpose": "FeatureInsight" }
                   }
                 */
                sendTelemetryEvent('getGoVersion', { version: stdout });
            }
            return resolve(goVersion);
        });
    });
}
exports.getGoVersion = getGoVersion;
/**
 * Returns boolean denoting if current version of Go supports vendoring
 */
function isVendorSupported() {
    if (vendorSupport != null) {
        return Promise.resolve(vendorSupport);
    }
    return getGoVersion().then(version => {
        if (!version) {
            return process.env['GO15VENDOREXPERIMENT'] === '0' ? false : true;
        }
        switch (version.major) {
            case 0:
                vendorSupport = false;
                break;
            case 1:
                vendorSupport = (version.minor > 6 || ((version.minor === 5 || version.minor === 6) && process.env['GO15VENDOREXPERIMENT'] === '1')) ? true : false;
                break;
            default:
                vendorSupport = true;
                break;
        }
        return vendorSupport;
    });
}
exports.isVendorSupported = isVendorSupported;
/**
 * Returns boolean indicating if GOPATH is set or not
 * If not set, then prompts user to do set GOPATH
 */
function isGoPathSet() {
    if (!getCurrentGoPath()) {
        vscode.window.showInformationMessage('Set GOPATH environment variable and restart VS Code or set GOPATH in Workspace settings', 'Set GOPATH in Workspace Settings').then(selected => {
            if (selected === 'Set GOPATH in Workspace Settings') {
                vscode.commands.executeCommand('workbench.action.openWorkspaceSettings');
            }
        });
        return false;
    }
    return true;
}
exports.isGoPathSet = isGoPathSet;
function sendTelemetryEvent(eventName, properties, measures) {
    telemtryReporter = telemtryReporter ? telemtryReporter : new vscode_extension_telemetry_1.default(extensionId, extensionVersion, aiKey);
    telemtryReporter.sendTelemetryEvent(eventName, properties, measures);
}
exports.sendTelemetryEvent = sendTelemetryEvent;
function isPositionInString(document, position) {
    let lineText = document.lineAt(position.line).text;
    let lineTillCurrentPosition = lineText.substr(0, position.character);
    // Count the number of double quotes in the line till current position. Ignore escaped double quotes
    let doubleQuotesCnt = (lineTillCurrentPosition.match(/\"/g) || []).length;
    let escapedDoubleQuotesCnt = (lineTillCurrentPosition.match(/\\\"/g) || []).length;
    doubleQuotesCnt -= escapedDoubleQuotesCnt;
    return doubleQuotesCnt % 2 === 1;
}
exports.isPositionInString = isPositionInString;
function getToolsGopath(useCache = true) {
    if (!useCache || !toolsGopath) {
        toolsGopath = resolveToolsGopath();
    }
    return toolsGopath;
}
exports.getToolsGopath = getToolsGopath;
function resolveToolsGopath() {
    let toolsGopathForWorkspace = vscode.workspace.getConfiguration('go')['toolsGopath'] || '';
    // In case of single root, use resolvePath to resolve ~ and ${workspaceRoot}
    if (!vscode.workspace.workspaceFolders || vscode.workspace.workspaceFolders.length <= 1) {
        return resolvePath(toolsGopathForWorkspace);
    }
    // In case of multi-root, resolve ~ and ignore ${workspaceRoot}
    if (toolsGopathForWorkspace.startsWith('~')) {
        toolsGopathForWorkspace = path.join(os.homedir(), toolsGopathForWorkspace.substr(1));
    }
    if (toolsGopathForWorkspace && toolsGopathForWorkspace.trim() && !/\${workspaceRoot}/.test(toolsGopathForWorkspace)) {
        return toolsGopathForWorkspace;
    }
    // If any of the folders in multi root have toolsGopath set, use it.
    for (let i = 0; i < vscode.workspace.workspaceFolders.length; i++) {
        let toolsGopath = vscode.workspace.getConfiguration('go', vscode.workspace.workspaceFolders[i].uri).inspect('toolsGopath').workspaceFolderValue;
        toolsGopath = resolvePath(toolsGopath, vscode.workspace.workspaceFolders[i].uri.fsPath);
        if (toolsGopath) {
            return toolsGopath;
        }
    }
}
function getBinPath(tool) {
    return goPath_1.getBinPathWithPreferredGopath(tool, getToolsGopath(), getCurrentGoPath());
}
exports.getBinPath = getBinPath;
function getFileArchive(document) {
    let fileContents = document.getText();
    return document.fileName + '\n' + Buffer.byteLength(fileContents, 'utf8') + '\n' + fileContents;
}
exports.getFileArchive = getFileArchive;
function getToolsEnvVars() {
    const config = vscode.workspace.getConfiguration('go', vscode.window.activeTextEditor ? vscode.window.activeTextEditor.document.uri : null);
    const toolsEnvVars = config['toolsEnvVars'];
    let gopath = getCurrentGoPath();
    let envVars = Object.assign({}, process.env, gopath ? { GOPATH: gopath } : {});
    if (toolsEnvVars && typeof toolsEnvVars === 'object') {
        Object.keys(toolsEnvVars).forEach(key => envVars[key] = resolvePath(toolsEnvVars[key]));
    }
    return envVars;
}
exports.getToolsEnvVars = getToolsEnvVars;
function getCurrentGoPath(workspaceUri) {
    if (!workspaceUri && vscode.window.activeTextEditor && vscode.workspace.getWorkspaceFolder(vscode.window.activeTextEditor.document.uri)) {
        workspaceUri = vscode.workspace.getWorkspaceFolder(vscode.window.activeTextEditor.document.uri).uri;
    }
    const config = vscode.workspace.getConfiguration('go', workspaceUri);
    let currentRoot = workspaceUri ? workspaceUri.fsPath : vscode.workspace.rootPath;
    // Workaround for issue in https://github.com/Microsoft/vscode/issues/9448#issuecomment-244804026
    if (process.platform === 'win32' && currentRoot) {
        currentRoot = currentRoot.substr(0, 1).toUpperCase() + currentRoot.substr(1);
    }
    const configGopath = config['gopath'] ? resolvePath(config['gopath'], currentRoot) : '';
    const inferredGopath = config['inferGopath'] === true ? goPath_1.getInferredGopath(currentRoot) : '';
    return inferredGopath ? inferredGopath : (configGopath || process.env['GOPATH']);
}
exports.getCurrentGoPath = getCurrentGoPath;
function getExtensionCommands() {
    let pkgJSON = vscode.extensions.getExtension(extensionId).packageJSON;
    if (!pkgJSON.contributes || !pkgJSON.contributes.commands) {
        return;
    }
    let extensionCommands = vscode.extensions.getExtension(extensionId).packageJSON.contributes.commands.filter(x => x.command !== 'go.show.commands');
    return extensionCommands;
}
exports.getExtensionCommands = getExtensionCommands;
class LineBuffer {
    constructor() {
        this.buf = '';
        this.lineListeners = [];
        this.lastListeners = [];
    }
    append(chunk) {
        this.buf += chunk;
        do {
            const idx = this.buf.indexOf('\n');
            if (idx === -1) {
                break;
            }
            this.fireLine(this.buf.substring(0, idx));
            this.buf = this.buf.substring(idx + 1);
        } while (true);
    }
    done() {
        this.fireDone(this.buf !== '' ? this.buf : null);
    }
    fireLine(line) {
        this.lineListeners.forEach(listener => listener(line));
    }
    fireDone(last) {
        this.lastListeners.forEach(listener => listener(last));
    }
    onLine(listener) {
        this.lineListeners.push(listener);
    }
    onDone(listener) {
        this.lastListeners.push(listener);
    }
}
exports.LineBuffer = LineBuffer;
function timeout(millis) {
    return new Promise((resolve, reject) => {
        setTimeout(() => resolve(), millis);
    });
}
exports.timeout = timeout;
/**
 * Exapnds ~ to homedir in non-Windows platform and resolves ${workspaceRoot}
 */
function resolvePath(inputPath, workspaceRoot) {
    if (!inputPath || !inputPath.trim())
        return inputPath;
    if (!workspaceRoot && vscode.workspace.workspaceFolders) {
        if (vscode.workspace.workspaceFolders.length === 1) {
            workspaceRoot = vscode.workspace.rootPath;
        }
        else if (vscode.window.activeTextEditor && vscode.workspace.getWorkspaceFolder(vscode.window.activeTextEditor.document.uri)) {
            workspaceRoot = vscode.workspace.getWorkspaceFolder(vscode.window.activeTextEditor.document.uri).uri.fsPath;
        }
    }
    if (workspaceRoot) {
        inputPath = inputPath.replace(/\${workspaceRoot}/g, workspaceRoot);
    }
    return goPath_1.resolveHomeDir(inputPath);
}
exports.resolvePath = resolvePath;
/**
 * Returns the import path in a passed in string.
 * @param text The string to search for an import path
 */
function getImportPath(text) {
    // Catch cases like `import alias "importpath"` and `import "importpath"`
    let singleLineImportMatches = text.match(/^\s*import\s+([a-z,A-Z,_,\.]\w*\s+)?\"([^\"]+)\"/);
    if (singleLineImportMatches) {
        return singleLineImportMatches[2];
    }
    // Catch cases like `alias "importpath"` and "importpath"
    let groupImportMatches = text.match(/^\s*([a-z,A-Z,_,\.]\w*\s+)?\"([^\"]+)\"/);
    if (groupImportMatches) {
        return groupImportMatches[2];
    }
    return '';
}
exports.getImportPath = getImportPath;
// TODO: Add unit tests for the below
/**
 * Guess the package name based on parent directory name of the given file
 *
 * Cases:
 * - dir 'go-i18n' -> 'i18n'
 * - dir 'go-spew' -> 'spew'
 * - dir 'kingpin' -> 'kingpin'
 * - dir 'go-expand-tilde' -> 'tilde'
 * - dir 'gax-go' -> 'gax'
 * - dir 'go-difflib' -> 'difflib'
 * - dir 'jwt-go' -> 'jwt'
 * - dir 'go-radix' -> 'radix'
 *
 * @param {string} filePath.
 */
function guessPackageNameFromFile(filePath) {
    return new Promise((resolve, reject) => {
        const goFilename = path.basename(filePath);
        if (goFilename === 'main.go') {
            return resolve(['main']);
        }
        const directoryPath = path.dirname(filePath);
        const dirName = path.basename(directoryPath);
        let segments = dirName.split(/[\.-]/);
        segments = segments.filter(val => val !== 'go');
        if (segments.length === 0 || !/[a-zA-Z_]\w*/.test(segments[segments.length - 1])) {
            return reject();
        }
        const proposedPkgName = segments[segments.length - 1];
        if (goFilename.endsWith('_test.go')) {
            return resolve([proposedPkgName, proposedPkgName + '_test']);
        }
        fs.stat(path.join(directoryPath, 'main.go'), (err, stats) => {
            if (stats && stats.isFile()) {
                return resolve(['main']);
            }
            return resolve([proposedPkgName]);
        });
    });
}
exports.guessPackageNameFromFile = guessPackageNameFromFile;
/**
 * Runs given Go tool and returns errors/warnings that can be fed to the Problems Matcher
 * @param args Arguments to be passed while running given tool
 * @param cwd cwd that will passed in the env object while running given tool
 * @param severity error or warning
 * @param useStdErr If true, the stderr of the output of the given tool will be used, else stdout will be used
 * @param toolName The name of the Go tool to run. If none is provided, the go runtime itself is used
 * @param printUnexpectedOutput If true, then output that doesnt match expected format is printed to the output channel
 */
function runTool(args, cwd, severity, useStdErr, toolName, env, printUnexpectedOutput, token) {
    let goRuntimePath = goPath_1.getGoRuntimePath();
    let cmd = toolName ? getBinPath(toolName) : goRuntimePath;
    let p;
    if (token) {
        token.onCancellationRequested(() => {
            if (p) {
                killTree(p.pid);
            }
        });
    }
    return new Promise((resolve, reject) => {
        p = cp.execFile(cmd, args, { env: env, cwd: cwd }, (err, stdout, stderr) => {
            try {
                if (err && err.code === 'ENOENT') {
                    // Since the tool is run on save which can be frequent
                    // we avoid sending explicit notification if tool is missing
                    console.log(`Cannot find ${toolName ? toolName : goRuntimePath}`);
                    return resolve([]);
                }
                if (err && stderr && !useStdErr) {
                    goStatus_1.outputChannel.appendLine(['Error while running tool:', cmd, ...args].join(' '));
                    goStatus_1.outputChannel.appendLine(stderr);
                    return resolve([]);
                }
                let lines = (useStdErr ? stderr : stdout).toString().split('\n');
                goStatus_1.outputChannel.appendLine(['Finished running tool:', cmd, ...args].join(' '));
                let ret = [];
                let unexpectedOutput = false;
                let atleastSingleMatch = false;
                for (let i = 0; i < lines.length; i++) {
                    if (lines[i][0] === '\t' && ret.length > 0) {
                        ret[ret.length - 1].msg += '\n' + lines[i];
                        continue;
                    }
                    let match = /^([^:]*: )?((.:)?[^:]*):(\d+)(:(\d+)?)?:(?:\w+:)? (.*)$/.exec(lines[i]);
                    if (!match) {
                        if (printUnexpectedOutput && useStdErr && stderr)
                            unexpectedOutput = true;
                        continue;
                    }
                    atleastSingleMatch = true;
                    let [_, __, file, ___, lineStr, ____, charStr, msg] = match;
                    let line = +lineStr;
                    // Building skips vendor folders,
                    // But vet and lint take in directories and not import paths, so no way to skip them
                    // So prune out the results from vendor folders herehere.
                    if (!path.isAbsolute(file) && (file.startsWith(`vendor${path.sep}`) || file.indexOf(`${path.sep}vendor${path.sep}`) > -1)) {
                        continue;
                    }
                    file = path.resolve(cwd, file);
                    ret.push({ file, line, msg, severity });
                    goStatus_1.outputChannel.appendLine(`${file}:${line}: ${msg}`);
                }
                if (!atleastSingleMatch && unexpectedOutput && vscode.window.activeTextEditor) {
                    goStatus_1.outputChannel.appendLine(stderr);
                    if (err) {
                        ret.push({
                            file: vscode.window.activeTextEditor.document.fileName,
                            line: 1,
                            msg: stderr,
                            severity: 'error'
                        });
                    }
                }
                goStatus_1.outputChannel.appendLine('');
                resolve(ret);
            }
            catch (e) {
                reject(e);
            }
        });
    });
}
exports.runTool = runTool;
function handleDiagnosticErrors(document, errors, diagnosticSeverity) {
    if (diagnosticSeverity === undefined || diagnosticSeverity === vscode.DiagnosticSeverity.Error) {
        goMain_1.errorDiagnosticCollection.clear();
    }
    if (diagnosticSeverity === undefined || diagnosticSeverity === vscode.DiagnosticSeverity.Warning) {
        goMain_1.warningDiagnosticCollection.clear();
    }
    let diagnosticMap = new Map();
    errors.forEach(error => {
        let canonicalFile = vscode.Uri.file(error.file).toString();
        let startColumn = 0;
        let endColumn = 1;
        if (document && document.uri.toString() === canonicalFile) {
            let range = new vscode.Range(error.line - 1, 0, error.line - 1, document.lineAt(error.line - 1).range.end.character + 1);
            let text = document.getText(range);
            let [_, leading, trailing] = /^(\s*).*(\s*)$/.exec(text);
            startColumn = leading.length;
            endColumn = text.length - trailing.length;
        }
        let range = new vscode.Range(error.line - 1, startColumn, error.line - 1, endColumn);
        let severity = mapSeverityToVSCodeSeverity(error.severity);
        let diagnostic = new vscode.Diagnostic(range, error.msg, severity);
        let diagnostics = diagnosticMap.get(canonicalFile);
        if (!diagnostics) {
            diagnostics = new Map();
        }
        if (!diagnostics[severity]) {
            diagnostics[severity] = [];
        }
        diagnostics[severity].push(diagnostic);
        diagnosticMap.set(canonicalFile, diagnostics);
    });
    diagnosticMap.forEach((diagMap, file) => {
        const fileUri = vscode.Uri.parse(file);
        if (diagnosticSeverity === undefined || diagnosticSeverity === vscode.DiagnosticSeverity.Error) {
            const newErrors = diagMap[vscode.DiagnosticSeverity.Error];
            let existingWarnings = goMain_1.warningDiagnosticCollection.get(fileUri);
            goMain_1.errorDiagnosticCollection.set(fileUri, newErrors);
            // If there are warnings on current file, remove the ones co-inciding with the new errors
            if (newErrors && existingWarnings) {
                const errorLines = newErrors.map(x => x.range.start.line);
                existingWarnings = existingWarnings.filter(x => errorLines.indexOf(x.range.start.line) === -1);
                goMain_1.warningDiagnosticCollection.set(fileUri, existingWarnings);
            }
        }
        if (diagnosticSeverity === undefined || diagnosticSeverity === vscode.DiagnosticSeverity.Warning) {
            const existingErrors = goMain_1.errorDiagnosticCollection.get(fileUri);
            let newWarnings = diagMap[vscode.DiagnosticSeverity.Warning];
            // If there are errors on current file, ignore the new warnings co-inciding with them
            if (existingErrors && newWarnings) {
                const errorLines = existingErrors.map(x => x.range.start.line);
                newWarnings = newWarnings.filter(x => errorLines.indexOf(x.range.start.line) === -1);
            }
            goMain_1.warningDiagnosticCollection.set(fileUri, newWarnings);
        }
    });
}
exports.handleDiagnosticErrors = handleDiagnosticErrors;
;
function mapSeverityToVSCodeSeverity(sev) {
    switch (sev) {
        case 'error': return vscode.DiagnosticSeverity.Error;
        case 'warning': return vscode.DiagnosticSeverity.Warning;
        default: return vscode.DiagnosticSeverity.Error;
    }
}
function getWorkspaceFolderPath(fileUri) {
    if (fileUri) {
        let workspace = vscode.workspace.getWorkspaceFolder(fileUri);
        if (workspace) {
            return workspace.uri.fsPath;
        }
    }
    // fall back to the first workspace
    let folders = vscode.workspace.workspaceFolders;
    if (folders && folders.length) {
        return folders[0].uri.fsPath;
    }
}
exports.getWorkspaceFolderPath = getWorkspaceFolderPath;
function killProcess(p) {
    if (p) {
        try {
            p.kill();
        }
        catch (e) {
            console.log('Error killing process: ' + e);
            if (e && e.message && e.stack) {
                let matches = e.stack.match(/(src.go[a-z,A-Z]+\.js)/g);
                if (matches) {
                    /* __GDPR__
                       "errorKillingProcess" : {
                          "message" : { "classification": "SystemMetaData", "purpose": "FeatureInsight" },
                          "stack": { "classification": "CustomerContent", "purpose": "FeatureInsight" }
                       }
                     */
                    sendTelemetryEvent('errorKillingProcess', { message: e.message, stack: matches });
                }
            }
        }
    }
}
exports.killProcess = killProcess;
function killTree(processId) {
    if (process.platform === 'win32') {
        const TASK_KILL = 'C:\\Windows\\System32\\taskkill.exe';
        // when killing a process in Windows its child processes are *not* killed but become root processes.
        // Therefore we use TASKKILL.EXE
        try {
            cp.execSync(`${TASK_KILL} /F /T /PID ${processId}`);
        }
        catch (err) {
        }
    }
    else {
        // on linux and OS X we kill all direct and indirect child processes as well
        try {
            const cmd = path.join(__dirname, '../../../scripts/terminateProcess.sh');
            cp.spawnSync(cmd, [processId.toString()]);
        }
        catch (err) {
        }
    }
}
exports.killTree = killTree;
//# sourceMappingURL=util.js.map