/**********************************************************************
 * Copyright (c) 2018-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
import * as yargs from 'yargs';
/**
 * Init all sources of extensions and plugins by cloning them, creating symlinks, update package.json, etc.
 * @author Florent Benoit
 */
export declare class InitSources {
    readonly rootFolder: string;
    readonly packagesFolder: string;
    readonly pluginsFolder: string;
    readonly cheTheiaFolder: string;
    readonly assemblyFolder: string;
    readonly theiaVersion: string;
    /**
     * Prefix for extensions.
     */
    static readonly PREFIX_PACKAGES_EXTENSIONS = "@che-";
    static readonly DEFAULT_EXTENSIONS_URI = "https://raw.githubusercontent.com/eclipse-che/che-theia/master/che-theia-init-sources.yml";
    static argBuilder: (theYargs: yargs.Argv) => yargs.Argv;
    /**
     * Source clone locations could be replaced from the command line --alias option.
     */
    sourceLocationAliases: Map<string, string>;
    /**
     * Set of global dependencies
     */
    private globalDevDependencies;
    /**
     * Will clone sources keeping or omiting the history
     */
    private keepHistory;
    /**
     * extensions processed
     */
    extensions: ISource[];
    /**
     * Constructor
     */
    constructor(rootFolder: string, packagesFolder: string, pluginsFolder: string, cheTheiaFolder: string, assemblyFolder: string, theiaVersion: string);
    /**
     * Keep or omit git history when cloning sources
     */
    set keepGitHistory(value: boolean);
    /**
     * Install all extensions
     */
    generate(extensionsPath: string, isDevMode?: boolean): Promise<void>;
    /**
     * Update configs/root-compilation.tsconfig.json
     */
    initRootCompilationUnits(): Promise<void>;
    /**
     * Scan package.json file and grab all dev dependencies and store them in globalDevDependencies variable
     */
    initGlobalDependencies(): Promise<void>;
    /**
     * Adds an extension to the current theia
     * @param extension the extension to add
     */
    addExtension(extension: ISource): Promise<void>;
    /**
     * perform update of devDependencies or dependencies in package.json file of the cloned extension
     */
    updateDependencies(extension: ISource, rewrite?: boolean): Promise<void>;
    /**
     * Update the given dependency by comparing with global dependencies or checking if it's a theia dependency.
     * @param dependencyKey the key of dependency
     * @param dependencyValue its original value
     */
    updateDependency(dependencyKey: string, dependencyValue: string): string;
    /**
     * Insert the given extension into the package.json of the assembly.
     * @param extension the given extension
     */
    insertExtensionIntoAssembly(extension: ISource): Promise<void>;
    symlink(source: ISource): Promise<void>;
    pluginsSymlink(source: ISource): Promise<void>;
    /**
     * Clone the given extension with the correct branch/tag
     * @param extension the extension to clone
     */
    clone(extension: ISource): Promise<void>;
    initSourceLocationAliases(alias: string[] | undefined): Promise<void>;
    readConfigurationAndGenerate(configPath: string | undefined, dev: boolean): Promise<void>;
}
/**
 * Source's interface
 */
export interface ISource {
    source: string;
    checkoutTo: string;
    type: string;
    extensions: string[];
    plugins: string[];
    clonedDir: string;
    extSymbolicLinks: string[];
    pluginSymbolicLinks: string[];
}
//# sourceMappingURL=init-sources.d.ts.map