"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.unpublish = exports.publish = void 0;
const fs = require("fs");
const GalleryInterfaces_1 = require("azure-devops-node-api/interfaces/GalleryInterfaces");
const package_1 = require("./package");
const tmp = require("tmp");
const store_1 = require("./store");
const util_1 = require("./util");
const denodeify = require("denodeify");
const zip_1 = require("./zip");
const tmpName = denodeify(tmp.tmpName);
function publish(options = {}) {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        if (options.packagePath) {
            if (options.version) {
                throw new Error(`Both options not supported simultaneously: 'packagePath' and 'version'.`);
            }
            else if (options.targets) {
                throw new Error(`Both options not supported simultaneously: 'packagePath' and 'target'.`);
            }
            for (const packagePath of options.packagePath) {
                const vsix = yield zip_1.readVSIXPackage(packagePath);
                let target;
                try {
                    target = (_a = vsix.xmlManifest.PackageManifest.Metadata[0].Identity[0].$.TargetPlatform) !== null && _a !== void 0 ? _a : undefined;
                }
                catch (err) {
                    throw new Error(`Invalid extension VSIX manifest. ${err}`);
                }
                yield _publish(packagePath, vsix.manifest, Object.assign(Object.assign({}, options), { target }));
            }
        }
        else {
            const cwd = options.cwd || process.cwd();
            const manifest = yield package_1.readManifest(cwd);
            yield package_1.prepublish(cwd, manifest, options.useYarn);
            yield package_1.versionBump(options);
            if (options.targets) {
                for (const target of options.targets) {
                    const packagePath = yield tmpName();
                    const packageResult = yield package_1.pack(Object.assign(Object.assign({}, options), { target, packagePath }));
                    yield _publish(packagePath, packageResult.manifest, Object.assign(Object.assign({}, options), { target }));
                }
            }
            else {
                const packagePath = yield tmpName();
                const packageResult = yield package_1.pack(Object.assign(Object.assign({}, options), { packagePath }));
                yield _publish(packagePath, packageResult.manifest, options);
            }
        }
    });
}
exports.publish = publish;
function _publish(packagePath, manifest, options) {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        if (!options.noVerify && manifest.enableProposedApi) {
            throw new Error("Extensions using proposed API (enableProposedApi: true) can't be published to the Marketplace");
        }
        const pat = (_a = options.pat) !== null && _a !== void 0 ? _a : (yield store_1.getPublisher(manifest.publisher)).pat;
        const api = yield util_1.getGalleryAPI(pat);
        const packageStream = fs.createReadStream(packagePath);
        const name = `${manifest.publisher}.${manifest.name}`;
        const description = options.target
            ? `${name} (${options.target}) v${manifest.version}`
            : `${name} v${manifest.version}`;
        util_1.log.info(`Publishing '${description}'...`);
        let extension = null;
        try {
            try {
                extension = yield api.getExtension(null, manifest.publisher, manifest.name, null, GalleryInterfaces_1.ExtensionQueryFlags.IncludeVersions);
            }
            catch (err) {
                if (err.statusCode !== 404) {
                    throw err;
                }
            }
            if (extension) {
                const sameVersion = extension.versions.filter(v => v.version === manifest.version);
                if (sameVersion.length > 0) {
                    if (!options.target) {
                        throw new Error(`${description} already exists.`);
                    }
                    if (sameVersion.some(v => !v.targetPlatform)) {
                        throw new Error(`${name} (no target) v${manifest.version} already exists.`);
                    }
                    if (sameVersion.some(v => v.targetPlatform === options.target)) {
                        throw new Error(`${description} already exists.`);
                    }
                }
                try {
                    yield api.updateExtension(undefined, packageStream, manifest.publisher, manifest.name);
                }
                catch (err) {
                    if (err.statusCode === 409) {
                        throw new Error(`${description} already exists.`);
                    }
                    else {
                        throw err;
                    }
                }
            }
            else {
                yield api.createExtension(undefined, packageStream);
            }
        }
        catch (err) {
            const message = (err && err.message) || '';
            if (/Personal Access Token used has expired/.test(message)) {
                err.message = `${err.message}\n\nYou're using an expired Personal Access Token, please get a new PAT.\nMore info: https://aka.ms/vscodepat`;
            }
            else if (/Invalid Resource/.test(message)) {
                err.message = `${err.message}\n\nYou're likely using an expired Personal Access Token, please get a new PAT.\nMore info: https://aka.ms/vscodepat`;
            }
            throw err;
        }
        util_1.log.info(`Extension URL (might take a few minutes): ${util_1.getPublishedUrl(name)}`);
        util_1.log.info(`Hub URL: ${util_1.getHubUrl(manifest.publisher, manifest.name)}`);
        util_1.log.done(`Published ${description}.`);
    });
}
function unpublish(options = {}) {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        let publisher, name;
        if (options.id) {
            [publisher, name] = options.id.split('.');
        }
        else {
            const manifest = yield package_1.readManifest(options.cwd);
            publisher = manifest.publisher;
            name = manifest.name;
        }
        const fullName = `${publisher}.${name}`;
        if (!options.force) {
            const answer = yield util_1.read(`This will FOREVER delete '${fullName}'! Are you sure? [y/N] `);
            if (!/^y$/i.test(answer)) {
                throw new Error('Aborted');
            }
        }
        const pat = (_a = options.pat) !== null && _a !== void 0 ? _a : (yield store_1.getPublisher(publisher)).pat;
        const api = yield util_1.getGalleryAPI(pat);
        yield api.deleteExtension(publisher, name);
        util_1.log.done(`Deleted extension: ${fullName}!`);
    });
}
exports.unpublish = unpublish;
//# sourceMappingURL=publish.js.map