"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.listPublishers = exports.deletePublisher = exports.logoutPublisher = exports.loginPublisher = exports.getPublisher = exports.verifyPat = exports.KeytarStore = exports.FileStore = void 0;
const fs = require("fs");
const path = require("path");
const osenv_1 = require("osenv");
const util_1 = require("./util");
const validation_1 = require("./validation");
const package_1 = require("./package");
class FileStore {
    constructor(path, publishers) {
        this.path = path;
        this.publishers = publishers;
    }
    static open(path = FileStore.DefaultPath) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const rawStore = yield fs.promises.readFile(path, 'utf8');
                return new FileStore(path, JSON.parse(rawStore).publishers);
            }
            catch (err) {
                if (err.code === 'ENOENT') {
                    return new FileStore(path, []);
                }
                else if (/SyntaxError/.test(err)) {
                    throw new Error(`Error parsing file store: ${path}`);
                }
                throw err;
            }
        });
    }
    get size() {
        return this.publishers.length;
    }
    save() {
        return __awaiter(this, void 0, void 0, function* () {
            yield fs.promises.writeFile(this.path, JSON.stringify({ publishers: this.publishers }), { mode: '0600' });
        });
    }
    deleteStore() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield fs.promises.unlink(this.path);
            }
            catch (_a) {
                // noop
            }
        });
    }
    get(name) {
        return this.publishers.filter(p => p.name === name)[0];
    }
    add(publisher) {
        return __awaiter(this, void 0, void 0, function* () {
            this.publishers = [...this.publishers.filter(p => p.name !== publisher.name), publisher];
            yield this.save();
        });
    }
    delete(name) {
        return __awaiter(this, void 0, void 0, function* () {
            this.publishers = this.publishers.filter(p => p.name !== name);
            yield this.save();
        });
    }
    [Symbol.iterator]() {
        return this.publishers[Symbol.iterator]();
    }
}
exports.FileStore = FileStore;
FileStore.DefaultPath = path.join(osenv_1.home(), '.vsce');
class KeytarStore {
    constructor(keytar, serviceName, publishers) {
        this.keytar = keytar;
        this.serviceName = serviceName;
        this.publishers = publishers;
    }
    static open(serviceName = 'vscode-vsce') {
        return __awaiter(this, void 0, void 0, function* () {
            const keytar = yield Promise.resolve().then(() => require('keytar'));
            const creds = yield keytar.findCredentials(serviceName);
            return new KeytarStore(keytar, serviceName, creds.map(({ account, password }) => ({ name: account, pat: password })));
        });
    }
    get size() {
        return this.publishers.length;
    }
    get(name) {
        return this.publishers.filter(p => p.name === name)[0];
    }
    add(publisher) {
        return __awaiter(this, void 0, void 0, function* () {
            this.publishers = [...this.publishers.filter(p => p.name !== publisher.name), publisher];
            yield this.keytar.setPassword(this.serviceName, publisher.name, publisher.pat);
        });
    }
    delete(name) {
        return __awaiter(this, void 0, void 0, function* () {
            this.publishers = this.publishers.filter(p => p.name !== name);
            yield this.keytar.deletePassword(this.serviceName, name);
        });
    }
    [Symbol.iterator]() {
        return this.publishers[Symbol.iterator]();
    }
}
exports.KeytarStore = KeytarStore;
function verifyPat(pat, publisherName) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!pat) {
            throw new Error('The Personal Access Token is mandatory.');
        }
        if (!publisherName) {
            try {
                publisherName = (yield package_1.readManifest()).publisher;
            }
            catch (error) {
                throw new Error(`Can not read the publisher's name. Either supply it as an argument or run vsce from the extension folder. Additional information:\n\n${error}`);
            }
        }
        try {
            // If the caller of the `getRoleAssignments` API has any of the roles
            // (Creator, Owner, Contributor, Reader) on the publisher, we get a 200,
            // otherwise we get a 403.
            const api = yield util_1.getSecurityRolesAPI(pat);
            yield api.getRoleAssignments('gallery.publisher', publisherName);
        }
        catch (error) {
            throw new Error('The Personal Access Token verification has failed. Additional information:\n\n' + error);
        }
        console.log(`The Personal Access Token verification succeeded for the publisher '${publisherName}'.`);
    });
}
exports.verifyPat = verifyPat;
function requestPAT(publisherName) {
    return __awaiter(this, void 0, void 0, function* () {
        console.log('https://marketplace.visualstudio.com/manage/publishers/');
        const pat = yield util_1.read(`Personal Access Token for publisher '${publisherName}':`, { silent: true, replace: '*' });
        yield verifyPat(pat, publisherName);
        return pat;
    });
}
function openDefaultStore() {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        if (/^file$/i.test((_a = process.env['VSCE_STORE']) !== null && _a !== void 0 ? _a : '')) {
            return yield FileStore.open();
        }
        const keytarStore = yield KeytarStore.open();
        const fileStore = yield FileStore.open();
        // migrate from file store
        if (fileStore.size) {
            for (const publisher of fileStore) {
                yield keytarStore.add(publisher);
            }
            yield fileStore.deleteStore();
            util_1.log.info(`Migrated ${fileStore.size} publishers to system credential manager. Deleted local store '${fileStore.path}'.`);
        }
        return keytarStore;
    });
}
function getPublisher(publisherName) {
    return __awaiter(this, void 0, void 0, function* () {
        validation_1.validatePublisher(publisherName);
        const store = yield openDefaultStore();
        let publisher = store.get(publisherName);
        if (publisher) {
            return publisher;
        }
        const pat = yield requestPAT(publisherName);
        publisher = { name: publisherName, pat };
        yield store.add(publisher);
        return publisher;
    });
}
exports.getPublisher = getPublisher;
function loginPublisher(publisherName) {
    return __awaiter(this, void 0, void 0, function* () {
        validation_1.validatePublisher(publisherName);
        const store = yield openDefaultStore();
        let publisher = store.get(publisherName);
        if (publisher) {
            console.log(`Publisher '${publisherName}' is already known`);
            const answer = yield util_1.read('Do you want to overwrite its PAT? [y/N] ');
            if (!/^y$/i.test(answer)) {
                throw new Error('Aborted');
            }
        }
        const pat = yield requestPAT(publisherName);
        publisher = { name: publisherName, pat };
        yield store.add(publisher);
        return publisher;
    });
}
exports.loginPublisher = loginPublisher;
function logoutPublisher(publisherName) {
    return __awaiter(this, void 0, void 0, function* () {
        validation_1.validatePublisher(publisherName);
        const store = yield openDefaultStore();
        const publisher = store.get(publisherName);
        if (!publisher) {
            throw new Error(`Unknown publisher '${publisherName}'`);
        }
        yield store.delete(publisherName);
    });
}
exports.logoutPublisher = logoutPublisher;
function deletePublisher(publisherName) {
    return __awaiter(this, void 0, void 0, function* () {
        const publisher = yield getPublisher(publisherName);
        const answer = yield util_1.read(`This will FOREVER delete '${publisherName}'! Are you sure? [y/N] `);
        if (!/^y$/i.test(answer)) {
            throw new Error('Aborted');
        }
        const api = yield util_1.getGalleryAPI(publisher.pat);
        yield api.deletePublisher(publisherName);
        const store = yield openDefaultStore();
        yield store.delete(publisherName);
        util_1.log.done(`Deleted publisher '${publisherName}'.`);
    });
}
exports.deletePublisher = deletePublisher;
function listPublishers() {
    return __awaiter(this, void 0, void 0, function* () {
        const store = yield openDefaultStore();
        for (const publisher of store) {
            console.log(publisher.name);
        }
    });
}
exports.listPublishers = listPublishers;
//# sourceMappingURL=store.js.map