/**********************************************************************
 * Copyright (c) 2018-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
/**
 * Handle the parsing of node packages with Yarn.
 * It allows to grab direct/production dependencies (not the dev dependencies)
 * @author Florent Benoit
 */
export declare class Yarn {
    readonly rootFolder: string;
    private readonly dependenciesDirectory;
    private readonly forbiddenPackages;
    private readonly excludedPackages;
    /**
     * Command to grab dependencies
     */
    static readonly YARN_GET_DEPENDENCIES = "yarn list --json --prod";
    /**
     * Command to grab yarn configuration.
     */
    static readonly YARN_GET_CONFIG = "yarn config current --json";
    constructor(rootFolder: string, dependenciesDirectory: string, forbiddenPackages: string[], excludedPackages: string[]);
    /**
     * Get package.json dependency paths (not including dev dependencies)
     */
    getDependencies(rootModule: string): Promise<string[]>;
    /**
     * Exclude dependencies which defined on excludedPackages
     *
     * @protected
     * @param {string} pkg
     * @returns
     * @memberof Yarn
     */
    protected isExcluded(pkg: string): boolean;
    /**
     * Find from children all the direct dependencies. Also exclude some dependencies by not analyzing them.
     * Allow as well to report error in case of a forbidden dependency found
     * @param children the list of dependencies to analyze
     * @param nodeTreeDependencies the object containing the tree of dependencies
     * @param subsetDependencies  the
     */
    protected findDependencies(children: string[], nodeTreeDependencies: Map<string, string[]>, subsetDependencies: string[]): void;
    /**
     * Insert the given node into the Map/tree of dependencies
     * @param yarnNode the node to insert
     * @param nodeTreeDependencies the tree to update
     */
    protected insertNode(yarnNode: IYarnNode, nodeTreeDependencies: Map<string, string[]>): void;
    /**
     * Add a node package (entry of yarn list) to the given array.
     * Also loop on all children and call ourself back
     * @param nodeModulesFolder the node_modules location
     * @param yarnNode the node entry to add
     * @param packages the array representing all node dependencies
     */
    protected addNodePackage(nodeModulesFolder: string, moduleName: string, packages: INodePackage[]): Promise<void>;
}
/**
 * Describes a node package entry (a name and a path)
 */
export interface INodePackage {
    name: string;
    path: string;
}
/**
 * Describes parsed result of yarn/json output
 */
export interface IYarnNode {
    name: string;
    children: IYarnNode[];
}
/**
 * Returns the full package name of an installed package or the emtpy string if not installed
 */
export declare function getFullPackageName(rootFolder: string, name: string): Promise<string>;
/**
 * Returns the version of an installed package or the emtpy string if not installed
 */
export declare function getPackageVersion(rootFolder: string, name: string): Promise<string>;
//# sourceMappingURL=yarn.d.ts.map