#!/bin/bash

# Define docker user env variable
export DOCKER_USER="andrewschoen"

# Define brew registry location
export BREW_REGISTRY=registry-proxy.engineering.redhat.com/rh-osbs

# Changing the prompt when using -x to be more informative
export PS4='[${FUNCNAME:-main}@${BASH_SOURCE}:${LINENO}] '

export OCP_LATEST_RELEASED_VERSION="4.8"

# This map maps the image names that we use for our upstream testing
# (quay.io images) to the pullable image version (i.e. using DS testing
# registry) and final image version (using redhat.io registry) separated
# by '|' character . It is used to translate the current quay.io images
# and their tags to the latest available image identified by SHA.
declare -A DS_MAP
DS_MAP=(
    [quay.io/rhceph-dev/rhceph]="registry.redhat.io/rhceph/rhceph-4-rhel8"
    [quay.io/rhceph-dev/cephcsi]="registry.redhat.io/ocs4/cephcsi-rhel8"
    [quay.io/rhceph-dev/mcg-core]="registry.redhat.io/ocs4/mcg-core-rhel8"
    [quay.io/rhceph-dev/mcg-operator]="registry.redhat.io/ocs4/mcg-rhel8-operator"
    [quay.io/rhceph-dev/ocs-operator]="registry.redhat.io/ocs4/ocs-rhel8-operator"
    [quay.io/rhceph-dev/ocs-must-gather]="registry.redhat.io/ocs4/ocs-must-gather-rhel8"
    [quay.io/rhceph-dev/rook-ceph]="registry.redhat.io/ocs4/rook-ceph-rhel8-operator"
    [quay.io/rhceph-dev/volume-replication-operator]="registry.redhat.io/ocs4/volume-replication-rhel8-operator"
    [quay.io/rhceph-dev/ose-csi-node-driver-registrar]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [quay.io/rhceph-dev/ose-csi-external-resizer]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel7"
    [quay.io/rhceph-dev/ose-csi-external-provisioner]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7"
    [quay.io/rhceph-dev/ose-csi-external-attacher]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [quay.io/rhceph-dev/ose-csi-external-snapshotter]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel7"
)

# Map of all the sidecars -- containers that we use but are not built by us
declare -A SIDECAR_MAP
SIDECAR_MAP=(
    [rhceph]="registry.redhat.io/rhceph/rhceph-4-rhel8"
    [csi-node-driver-registrar]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [csi-external-resizer]="registry.redhat.io/openshift4/ose-csi-external-resizer"
    [csi-external-provisioner]="registry.redhat.io/openshift4/ose-csi-external-provisioner"
    [csi-external-attacher]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [csi-external-snapshotter]="registry.redhat.io/openshift4/ose-csi-external-snapshotter"
    [rhel8-postgresql-12]="registry.redhat.io/rhel8/postgresql-12"
)

if  [ ${OCS_VERSION//./} -gt 45 ]
then
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-resizer]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel8"
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-provisioner]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel8"
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-snapshotter]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel8"
fi

declare -A UPSTREAM_REPO_MAP
UPSTREAM_REPO_MAP=(
    [cephcsi]="https://github.com/openshift/ceph-csi"
    [noobaa-core]="https://github.com/noobaa/noobaa-core"
    [noobaa-operator]="https://github.com/noobaa/noobaa-operator"
    [ocs-operator]="https://github.com/openshift/ocs-operator"
    [rook-ceph]="https://github.com/openshift/rook"
)

if  [ ${OCS_VERSION//./} -ge 48 ]
then
    UPSTREAM_REPO_MAP[volume-replication-operator]="https://github.com/csi-addons/volume-replication-operator"
fi

declare -A CEPHCSI_UPSTREAM_BRANCH_MAP
CEPHCSI_UPSTREAM_BRANCH_MAP=(
    [4.3]="release-4.3"
    [4.4]="release-4.4"
    [4.5]="release-4.5"
    [4.6]="release-4.6"
    [4.7]="release-4.7"
    [4.8]="release-4.8"
)

declare -A MCG_CORE_UPSTREAM_BRANCH_MAP
MCG_CORE_UPSTREAM_BRANCH_MAP=(
    [4.3]="5.3"
    [4.4]="5.4"
    [4.5]="5.5"
    [4.6]="5.6"
    [4.7]="5.7"
    [4.8]="5.8"
)

declare -A MCG_OPERATOR_UPSTREAM_BRANCH_MAP
MCG_OPERATOR_UPSTREAM_BRANCH_MAP=(
    [4.3]="2.1"
    [4.4]="2.2"
    [4.5]="2.3"
    [4.6]="5.6"
    [4.7]="5.7"
    [4.8]="5.8"
)

declare -A OCS_OPERATOR_UPSTREAM_BRANCH_MAP
OCS_OPERATOR_UPSTREAM_BRANCH_MAP=(
    [4.3]="release-4.3"
    [4.4]="release-4.4"
    [4.5]="release-4.5"
    [4.6]="release-4.6"
    [4.7]="release-4.7"
    [4.8]="release-4.8"
)

declare -A ROOK_CEPH_UPSTREAM_BRANCH_MAP
ROOK_CEPH_UPSTREAM_BRANCH_MAP=(
    [4.3]="ocs-4.3"
    [4.4]="release-4.4"
    [4.5]="release-4.5"
    [4.6]="release-4.6"
    [4.7]="release-4.7"
    [4.8]="release-4.8"
)

declare -A VOLUME_REPLICATION_UPSTREAM_BRANCH_MAP
VOLUME_REPLICATION_UPSTREAM_BRANCH_MAP=(
    [4.3]=""
    [4.4]=""
    [4.5]=""
    [4.6]=""
    [4.7]=""
    [4.8]="release-v0.1"
)

declare -A ROOK_CSI_REGISTRAR_REPO
ROOK_CSI_REGISTRAR_REPO=(
    [4.3]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [4.4]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [4.5]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [4.6]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [4.7]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [4.8]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
)

declare -A ROOK_CSI_REGISTRAR_TAG
ROOK_CSI_REGISTRAR_TAG=(
    [4.3]="v4.3"
    [4.4]="v4.3"
    [4.5]="v4.3"
    [4.6]="v4.6"
    [4.7]="v4.7"
    [4.8]="v4.8"
)

if [ "$(echo ${ROOK_CSI_REGISTRAR_TAG[$OCS_VERSION]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" ]
then
    DS_MAP[quay.io/rhceph-dev/ose-csi-node-driver-registrar]="${BREW_REGISTRY}/openshift-ose-csi-node-driver-registrar"
    SIDECAR_MAP[csi-node-driver-registrar]="${BREW_REGISTRY}/openshift-ose-csi-node-driver-registrar"
    OCS_OPERATOR_DS_MAP[quay.io/rhceph-dev/ose-csi-node-driver-registrar]="${BREW_REGISTRY}/openshift-ose-csi-node-driver-registrar"
    ROOK_CSI_REGISTRAR_REPO[$OCS_VERSION]="${BREW_REGISTRY}/openshift-ose-csi-node-driver-registrar"
fi

declare -A ROOK_CSI_RESIZER_REPO
ROOK_CSI_RESIZER_REPO=(
    [4.3]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel7"
    [4.4]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel7"
    [4.5]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel7"
    [4.6]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel8"
    [4.7]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel8"
    [4.8]="registry.redhat.io/openshift4/ose-csi-external-resizer-rhel8"
)

declare -A ROOK_CSI_RESIZER_TAG
ROOK_CSI_RESIZER_TAG=(
    [4.3]="v4.3"
    [4.4]="v4.3"
    [4.5]="v4.3"
    [4.6]="v4.6"
    [4.7]="v4.7"
    [4.8]="v4.8"
)

if [ "$(echo ${ROOK_CSI_RESIZER_TAG[$OCS_VERSION]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" ]
then
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-resizer]="${BREW_REGISTRY}/openshift-ose-csi-external-resizer"
    SIDECAR_MAP[csi-external-resizer]="${BREW_REGISTRY}/openshift-ose-csi-external-resizer"
    OCS_OPERATOR_DS_MAP[quay.io/rhceph-dev/ose-csi-external-resizer]="${BREW_REGISTRY}/openshift-ose-csi-external-resizer"
    ROOK_CSI_RESIZER_REPO[$OCS_VERSION]="${BREW_REGISTRY}/openshift-ose-csi-external-resizer"
fi

declare -A ROOK_CSI_PROVISIONER_REPO
ROOK_CSI_PROVISIONER_REPO=(
    [4.3]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7"
    [4.4]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7"
    [4.5]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7"
    [4.6]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel8"
    [4.7]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel8"
    [4.8]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel8"
)

declare -A ROOK_CSI_PROVISIONER_TAG
ROOK_CSI_PROVISIONER_TAG=(
    [4.3]="v4.3"
    [4.4]="v4.3"
    [4.5]="v4.3"
    [4.6]="v4.6"
    [4.7]="v4.7"
    [4.8]="v4.8"
)

if [ "$(echo ${ROOK_CSI_PROVISIONER_TAG[$OCS_VERSION]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" ]
then
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-provisioner]="${BREW_REGISTRY}/openshift-ose-csi-external-provisioner"
    SIDECAR_MAP[csi-external-provisioner]="${BREW_REGISTRY}/openshift-ose-csi-external-provisioner"
    OCS_OPERATOR_DS_MAP[quay.io/rhceph-dev/ose-csi-external-provisioner]="${BREW_REGISTRY}/openshift-ose-csi-external-provisioner"
    ROOK_CSI_PROVISIONER_REPO[$OCS_VERSION]="${BREW_REGISTRY}/openshift-ose-csi-external-provisioner"
fi

declare -A ROOK_CSI_ATTACHER_REPO
ROOK_CSI_ATTACHER_REPO=(
    [4.3]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [4.4]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [4.5]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [4.6]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [4.7]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [4.8]="registry.redhat.io/openshift4/ose-csi-external-attacher"
)

declare -A ROOK_CSI_ATTACHER_TAG
ROOK_CSI_ATTACHER_TAG=(
    [4.3]="v4.3"
    [4.4]="v4.3"
    [4.5]="v4.3"
    [4.6]="v4.6"
    [4.7]="v4.7"
    [4.8]="v4.8"
)

if [ "$(echo ${ROOK_CSI_ATTACHER_TAG[$OCS_VERSION]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" ]
then
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-attacher]="${BREW_REGISTRY}/openshift-ose-csi-external-attacher"
    SIDECAR_MAP[csi-external-attacher]="${BREW_REGISTRY}/openshift-ose-csi-external-attacher"
    OCS_OPERATOR_DS_MAP[quay.io/rhceph-dev/ose-csi-external-attacher]="${BREW_REGISTRY}/openshift-ose-csi-external-attacher"
    ROOK_CSI_ATTACHER_REPO[$OCS_VERSION]="${BREW_REGISTRY}/openshift-ose-csi-external-attacher"
fi

declare -A ROOK_CSI_SNAPSHOTTER_REPO
ROOK_CSI_SNAPSHOTTER_REPO=(
    [4.3]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel7"
    [4.4]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel7"
    [4.5]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel7"
    [4.6]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel8"
    [4.7]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel8"
    [4.8]="registry.redhat.io/openshift4/ose-csi-external-snapshotter-rhel8"
)

declare -A ROOK_CSI_SNAPSHOTTER_TAG
ROOK_CSI_SNAPSHOTTER_TAG=(
    [4.3]=""
    [4.4]=""
    [4.5]=""
    [4.6]="v4.6"
    [4.7]="v4.6"
    [4.8]="v4.8"
)


if [ "$(echo ${ROOK_CSI_SNAPSHOTTER_TAG[$OCS_VERSION]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" ]
then
    DS_MAP[quay.io/rhceph-dev/ose-csi-external-snapshotter]="${BREW_REGISTRY}/openshift-ose-csi-external-snapshotter"
    SIDECAR_MAP[csi-external-snapshotter]="${BREW_REGISTRY}/openshift-ose-csi-external-snapshotter"
    OCS_OPERATOR_DS_MAP[quay.io/rhceph-dev/ose-csi-external-snapshotter]="${BREW_REGISTRY}/openshift-ose-csi-external-snapshotter"
    ROOK_CSI_SNAPSHOTTER_REPO[$OCS_VERSION]="${BREW_REGISTRY}/openshift-ose-csi-external-snapshotter"
fi

declare -A NOOBAA_DB_REPO
NOOBAA_DB_REPO=(
    [4.3]="registry.redhat.io/rhscl/mongodb-36-rhel7"
    [4.4]="registry.redhat.io/rhscl/mongodb-36-rhel7"
    [4.5]="registry.redhat.io/rhscl/mongodb-36-rhel7"
    [4.6]="registry.redhat.io/rhscl/mongodb-36-rhel7"
    [4.7]="registry.redhat.io/rhel8/postgresql-12"
    [4.8]="registry.redhat.io/rhel8/postgresql-12"
)

declare -A NOOBAA_DB_TAG
NOOBAA_DB_TAG=(
    [4.3]="latest"
    [4.4]="latest"
    [4.5]="latest"
    [4.6]="latest"
    [4.7]="latest"
    [4.8]="latest"
)

declare -A OSE_OPERATOR_REGISTRY_TAG
OSE_OPERATOR_REGISTRY_TAG=(
    [4.3]="v4.5"
    [4.4]="v4.5"
    [4.5]="v4.5"
    [4.6]="v4.6"
    [4.7]="v4.7"
    [4.8]="v4.8"
)

declare -A CEPH_TAGS
CEPH_TAGS=(
    [4.3]="4-20"
    [4.4]="4-28"
    [4.5]="4-32"
    [4.6]="4-53"
    [4.7]="4-53"
    [4.8]="4-59"
)

function get_oc_tool() {
    echo "setting up oc"
    pushd $(mktemp -d)
    # Download the OSP clients
    curl -s "https://mirror.openshift.com/pub/openshift-v4/clients/ocp/4.5.2/openshift-client-linux-4.5.2.tar.gz" -o oc.tar.gz
    # Unpack oc binary and cleanup
    tar -xzf oc.tar.gz oc
    rm -f oc.tar.gz
    # export OC_TOOL and make sure it is executable
    export OC_TOOL=$(readlink -f ./oc)
    chmod +x "$OC_TOOL"
    popd

    # For debug. Don't want to fail if we cannot login at this stage for some reason
    $OC_TOOL version || true
    $OC_TOOL project || true
    $OC_TOOL project cpaas-rhocs || true
}

function get_opm() {
    # Make sure wget is installed
    test -x /usr/bin/wget || sudo dnf install -y /usr/bin/wget
    # Download the OPM tool
    wget -q "https://github.com/operator-framework/operator-registry/releases/download/v1.12.3/linux-amd64-opm"
    export OPM_TOOL=$(readlink -f ./linux-amd64-opm)
    chmod +x "$OPM_TOOL"
}

function ocs_mirror() {
    for i in $(seq 1 10); do
        sudo "$OC_TOOL" image mirror --filter-by-os='.*' "$1" "$2" && return 0 || :
    done
    return 1
}

function ocs_build() {
    # make sure jq is installed
    test -x /usr/bin/jq || sudo dnf install -y /usr/bin/jq
    local _component=$(cat Dockerfile|egrep -e 'com.redhat.component="' | sed -e 's|^.*com.redhat.component="\(.*\)".*$|\1|')
    local _version=$(cat Dockerfile|egrep -e 'version="' | sed -e 's|^.*version="\(.*\)".*$|\1|')
    local _release=$(cat Dockerfile|egrep -e 'release="' | sed -e 's|^.*release="\(.*\)".*$|\1|')
    for i in $(seq 1 3); do
        rhpkg container-build --signing-intent release "$@" && return 0 || :
        local state=$(brew --noauth call --json getBuild "$_component-$_version-$_release" | jq .state)
        if [ $state = 1 ]
        then
            echo "The brew build is completed"
            return 0
        fi
    done
    return 1
}

function tag_to_sha () {
        local _VAR="$1"
        local _IMAGE="$2"
        local _TAG="$3"

        # If tag is empty, there is no such image for the release
        test -n "$_TAG" || return 0

        # We used image digests prior to 4.5, we use manifest digests now
        case "$OCS_VERSION" in
        4.[234])
                local _IMAGE_SHA=$("$OC_TOOL" image info $_IMAGE:$_TAG --filter-by-os=linux/amd64 | egrep '^Digest:' | awk '{print $2}')
                ;;
        *)
                local _IMAGE_SHA=$("$OC_TOOL" image info $_IMAGE:$_TAG --filter-by-os=linux/amd64 | egrep '^Manifest List:' | awk '{print $3}')
                ;;
        esac

        # Make sure we could get the IMAGE SHA, fail otherwise (we use set -eux)
        test -n "$_IMAGE_SHA"

        # Return image@sha
        echo "$_IMAGE@$_IMAGE_SHA"

        eval export "$_VAR"="$_IMAGE@$_IMAGE_SHA"
}

function get_golang_1_16() {
        # Upgrade golang
        rm -rf __glng__
        mkdir __glng__
        pushd __glng__
        brew download-build -a x86_64 -a noarch golang-1.16.4-3.module+el8.5.0+11462+061f83ad
        dnf update -y ./golang-*.rpm
        popd
        rm -rf __glng__
}

function map_pull_spec() {
    local img="${1:?Image pull spec should be provided}"

    echo ${img/registry-proxy.engineering.redhat.com\/rh-osbs\/ocs4-/registry.redhat.io\/ocs4\/}
}

function map_sidecar_pull_spec() {
    local img="${1:?Image pull spec should be provided}"
    declare -A prod_map
    prod_map=(
        [registry-proxy.engineering.redhat.com/rh-osbs/rhceph]="registry.redhat.io/rhceph/rhceph-4-rhel8"
        [registry-proxy.engineering.redhat.com/rh-osbs/openshift-ose-csi-node-driver-registrar]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
        [registry-proxy.engineering.redhat.com/rh-osbs/openshift-ose-csi-external-resizer]="registry.redhat.io/openshift4/ose-csi-external-resizer"
        [registry-proxy.engineering.redhat.com/rh-osbs/openshift-ose-csi-external-provisioner]="registry.redhat.io/openshift4/ose-csi-external-provisioner"
        [registry-proxy.engineering.redhat.com/rh-osbs/openshift-ose-csi-external-attacher]="registry.redhat.io/openshift4/ose-csi-external-attacher"
        [registry-proxy.engineering.redhat.com/rh-osbs/openshift-ose-csi-external-snapshotter]="registry.redhat.io/openshift4/ose-csi-external-snapshotter"
        [registry-proxy.engineering.redhat.com/rh-osbs/rhel8-postgresql-12]="registry.redhat.io/rhel8/postgresql-12"
    )

    for k in "${!prod_map[@]}"; do
        [[ "$img" == ${k}* ]] || continue
        local sha="${img#*@}"
        local reg_n_repo="${prod_map[$k]}"
        echo "${reg_n_repo}@${sha}"
        return 0
    done

    echo "Failed to find mapping for $img" 1>&2
    return 1
}
