"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const findUp = require("find-up");
const constants_1 = require("../constants");
const jsonc_1 = require("../utils/jsonc");
const ExtensionValidator_1 = require("../validation/ExtensionValidator");
const SchemaValidator_1 = require("../validation/SchemaValidator");
exports.loadSchema = (relativePath) => {
    const pkgDir = path.dirname(findUp.sync('package.json', { cwd: __dirname }));
    const schemaPath = [
        path.resolve(pkgDir, 'schema'),
        path.resolve(pkgDir, 'generated/schema'),
    ].find((p) => fs.existsSync(p) && fs.statSync(p).isDirectory());
    return require(path.resolve(schemaPath, relativePath)).default;
};
exports.validateExtensionsFileSchema = (ext, description = constants_1.extensionsFile) => {
    const schema = exports.loadSchema('console-extensions');
    return new SchemaValidator_1.SchemaValidator(description).validate(schema, ext);
};
const emitJSON = (compilation, filename, data) => {
    const content = JSON.stringify(data, null, 2);
    // webpack compilation.emitAsset API requires the source argument to implement
    // methods which aren't strictly needed for processing the asset. In this case,
    // we just provide the content (source) and its length (size).
    // TODO(vojtech): revisit after bumping webpack 5 to latest stable version
    // eslint-disable-next-line @typescript-eslint/ban-ts-ignore
    // @ts-ignore
    compilation.emitAsset(filename, {
        source: () => content,
        size: () => content.length,
    });
};
class ConsoleAssetPlugin {
    constructor(pkg) {
        this.pkg = pkg;
        const ext = jsonc_1.parseJSONC(path.resolve(process.cwd(), constants_1.extensionsFile));
        exports.validateExtensionsFileSchema(ext).report();
        this.manifest = {
            name: pkg.consolePlugin.name,
            version: pkg.consolePlugin.version,
            displayName: pkg.consolePlugin.displayName,
            description: pkg.consolePlugin.description,
            dependencies: pkg.consolePlugin.dependencies,
            disableStaticPlugins: pkg.consolePlugin.disableStaticPlugins,
            extensions: ext,
        };
    }
    apply(compiler) {
        const errors = [];
        const addErrorsToCompilation = (compilation) => {
            errors.forEach((e) => {
                // TODO(vojtech): revisit after bumping webpack 5 to latest stable version
                // eslint-disable-next-line @typescript-eslint/ban-ts-ignore
                // @ts-ignore
                compilation.errors.push(new Error(e));
            });
        };
        compiler.hooks.afterCompile.tap(ConsoleAssetPlugin.name, (compilation) => {
            const result = new ExtensionValidator_1.ExtensionValidator(constants_1.extensionsFile).validate(compilation, this.manifest.extensions, this.pkg.consolePlugin.exposedModules || {});
            if (result.hasErrors()) {
                errors.push(result.formatErrors());
            }
        });
        compiler.hooks.shouldEmit.tap(ConsoleAssetPlugin.name, (compilation) => {
            addErrorsToCompilation(compilation);
            return errors.length === 0;
        });
        compiler.hooks.emit.tap(ConsoleAssetPlugin.name, (compilation) => {
            emitJSON(compilation, constants_1.pluginManifestFile, this.manifest);
        });
    }
}
exports.ConsoleAssetPlugin = ConsoleAssetPlugin;
//# sourceMappingURL=ConsoleAssetPlugin.js.map