//! Cached tables for precalculated values.

use crate::util::*;

/// Precalculated table for a digit to a character.
///
/// Unoptimized table for radix N always, which translates a single digit to a
/// character, and also useful for radix-N float encoding.
const DIGIT_TO_CHAR: [u8; 36] = [b'0', b'1', b'2', b'3', b'4', b'5', b'6', b'7', b'8', b'9', b'A', b'B', b'C', b'D', b'E', b'F', b'G', b'H', b'I', b'J', b'K', b'L', b'M', b'N', b'O', b'P', b'Q', b'R', b'S', b'T', b'U', b'V', b'W', b'X', b'Y', b'Z'];

/// Get character from digit.
#[inline(always)]
#[allow(dead_code)]
pub(crate) fn digit_to_char<T: Integer>(digit: T) -> u8 {
    debug_assert!(digit.as_i32() >= 0 && digit.as_i32() < 36, "digit_to_char() invalid character.");
    index!(DIGIT_TO_CHAR[digit.as_usize()])
}

// Conditionally compile the precompiled radix**2 tables.
// These tables take `2 * (value % (radix^2))`, and return
// two consecutive values corresponding to both digits.
//
// Total array storage: 32 KB.
// Provides ~5x performance enhancement.
//
// These arrays are cache-friendly, for each BASE[2-36] table,
// elements can be access sequentially 2-at-a-time, preventing as many
// cache misses inside inner loops. For example, accessing the two elements
// for a remainder of `3` for the radix^2 in radix 2 will give you `1` and `1`,
// at indexes 6 and 7.

cfg_if! {
if #[cfg(feature = "table")] {
pub(crate) const DIGIT_TO_BASE10_SQUARED: [u8; 200] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9'];
}}  // cfg_if

cfg_if! {
if #[cfg(all(feature = "radix", feature = "table"))] {
pub(crate) const DIGIT_TO_BASE2_SQUARED: [u8; 8] = [b'0', b'0', b'0', b'1', b'1', b'0', b'1', b'1'];
pub(crate) const DIGIT_TO_BASE3_SQUARED: [u8; 18] = [b'0', b'0', b'0', b'1', b'0', b'2', b'1', b'0', b'1', b'1', b'1', b'2', b'2', b'0', b'2', b'1', b'2', b'2'];
pub(crate) const DIGIT_TO_BASE4_SQUARED: [u8; 32] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3'];
pub(crate) const DIGIT_TO_BASE5_SQUARED: [u8; 50] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4'];
pub(crate) const DIGIT_TO_BASE6_SQUARED: [u8; 72] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5'];
pub(crate) const DIGIT_TO_BASE7_SQUARED: [u8; 98] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6'];
pub(crate) const DIGIT_TO_BASE8_SQUARED: [u8; 128] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7'];
pub(crate) const DIGIT_TO_BASE9_SQUARED: [u8; 162] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8'];
pub(crate) const DIGIT_TO_BASE11_SQUARED: [u8; 242] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A'];
pub(crate) const DIGIT_TO_BASE12_SQUARED: [u8; 288] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B'];
pub(crate) const DIGIT_TO_BASE13_SQUARED: [u8; 338] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C'];
pub(crate) const DIGIT_TO_BASE14_SQUARED: [u8; 392] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D'];
pub(crate) const DIGIT_TO_BASE15_SQUARED: [u8; 450] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E'];
pub(crate) const DIGIT_TO_BASE16_SQUARED: [u8; 512] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F'];
pub(crate) const DIGIT_TO_BASE17_SQUARED: [u8; 578] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G'];
pub(crate) const DIGIT_TO_BASE18_SQUARED: [u8; 648] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H'];
pub(crate) const DIGIT_TO_BASE19_SQUARED: [u8; 722] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I'];
pub(crate) const DIGIT_TO_BASE20_SQUARED: [u8; 800] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J'];
pub(crate) const DIGIT_TO_BASE21_SQUARED: [u8; 882] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K'];
pub(crate) const DIGIT_TO_BASE22_SQUARED: [u8; 968] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L'];
pub(crate) const DIGIT_TO_BASE23_SQUARED: [u8; 1058] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M'];
pub(crate) const DIGIT_TO_BASE24_SQUARED: [u8; 1152] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N'];
pub(crate) const DIGIT_TO_BASE25_SQUARED: [u8; 1250] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O'];
pub(crate) const DIGIT_TO_BASE26_SQUARED: [u8; 1352] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P'];
pub(crate) const DIGIT_TO_BASE27_SQUARED: [u8; 1458] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q'];
pub(crate) const DIGIT_TO_BASE28_SQUARED: [u8; 1568] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R'];
pub(crate) const DIGIT_TO_BASE29_SQUARED: [u8; 1682] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S'];
pub(crate) const DIGIT_TO_BASE30_SQUARED: [u8; 1800] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T'];
pub(crate) const DIGIT_TO_BASE31_SQUARED: [u8; 1922] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'0', b'U', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'1', b'U', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'2', b'U', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'3', b'U', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'4', b'U', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'5', b'U', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'6', b'U', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'7', b'U', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'8', b'U', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'9', b'U', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'A', b'U', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'B', b'U', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'C', b'U', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'D', b'U', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'E', b'U', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'F', b'U', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'G', b'U', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'H', b'U', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'I', b'U', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'J', b'U', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'K', b'U', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'L', b'U', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'M', b'U', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'N', b'U', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'O', b'U', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'P', b'U', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'Q', b'U', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'R', b'U', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'S', b'U', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T', b'T', b'U', b'U', b'0', b'U', b'1', b'U', b'2', b'U', b'3', b'U', b'4', b'U', b'5', b'U', b'6', b'U', b'7', b'U', b'8', b'U', b'9', b'U', b'A', b'U', b'B', b'U', b'C', b'U', b'D', b'U', b'E', b'U', b'F', b'U', b'G', b'U', b'H', b'U', b'I', b'U', b'J', b'U', b'K', b'U', b'L', b'U', b'M', b'U', b'N', b'U', b'O', b'U', b'P', b'U', b'Q', b'U', b'R', b'U', b'S', b'U', b'T', b'U', b'U'];
pub(crate) const DIGIT_TO_BASE32_SQUARED: [u8; 2048] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'0', b'U', b'0', b'V', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'1', b'U', b'1', b'V', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'2', b'U', b'2', b'V', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'3', b'U', b'3', b'V', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'4', b'U', b'4', b'V', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'5', b'U', b'5', b'V', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'6', b'U', b'6', b'V', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'7', b'U', b'7', b'V', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'8', b'U', b'8', b'V', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'9', b'U', b'9', b'V', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'A', b'U', b'A', b'V', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'B', b'U', b'B', b'V', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'C', b'U', b'C', b'V', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'D', b'U', b'D', b'V', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'E', b'U', b'E', b'V', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'F', b'U', b'F', b'V', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'G', b'U', b'G', b'V', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'H', b'U', b'H', b'V', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'I', b'U', b'I', b'V', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'J', b'U', b'J', b'V', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'K', b'U', b'K', b'V', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'L', b'U', b'L', b'V', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'M', b'U', b'M', b'V', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'N', b'U', b'N', b'V', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'O', b'U', b'O', b'V', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'P', b'U', b'P', b'V', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'Q', b'U', b'Q', b'V', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'R', b'U', b'R', b'V', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'S', b'U', b'S', b'V', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T', b'T', b'U', b'T', b'V', b'U', b'0', b'U', b'1', b'U', b'2', b'U', b'3', b'U', b'4', b'U', b'5', b'U', b'6', b'U', b'7', b'U', b'8', b'U', b'9', b'U', b'A', b'U', b'B', b'U', b'C', b'U', b'D', b'U', b'E', b'U', b'F', b'U', b'G', b'U', b'H', b'U', b'I', b'U', b'J', b'U', b'K', b'U', b'L', b'U', b'M', b'U', b'N', b'U', b'O', b'U', b'P', b'U', b'Q', b'U', b'R', b'U', b'S', b'U', b'T', b'U', b'U', b'U', b'V', b'V', b'0', b'V', b'1', b'V', b'2', b'V', b'3', b'V', b'4', b'V', b'5', b'V', b'6', b'V', b'7', b'V', b'8', b'V', b'9', b'V', b'A', b'V', b'B', b'V', b'C', b'V', b'D', b'V', b'E', b'V', b'F', b'V', b'G', b'V', b'H', b'V', b'I', b'V', b'J', b'V', b'K', b'V', b'L', b'V', b'M', b'V', b'N', b'V', b'O', b'V', b'P', b'V', b'Q', b'V', b'R', b'V', b'S', b'V', b'T', b'V', b'U', b'V', b'V'];
pub(crate) const DIGIT_TO_BASE33_SQUARED: [u8; 2178] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'0', b'U', b'0', b'V', b'0', b'W', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'1', b'U', b'1', b'V', b'1', b'W', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'2', b'U', b'2', b'V', b'2', b'W', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'3', b'U', b'3', b'V', b'3', b'W', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'4', b'U', b'4', b'V', b'4', b'W', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'5', b'U', b'5', b'V', b'5', b'W', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'6', b'U', b'6', b'V', b'6', b'W', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'7', b'U', b'7', b'V', b'7', b'W', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'8', b'U', b'8', b'V', b'8', b'W', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'9', b'U', b'9', b'V', b'9', b'W', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'A', b'U', b'A', b'V', b'A', b'W', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'B', b'U', b'B', b'V', b'B', b'W', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'C', b'U', b'C', b'V', b'C', b'W', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'D', b'U', b'D', b'V', b'D', b'W', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'E', b'U', b'E', b'V', b'E', b'W', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'F', b'U', b'F', b'V', b'F', b'W', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'G', b'U', b'G', b'V', b'G', b'W', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'H', b'U', b'H', b'V', b'H', b'W', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'I', b'U', b'I', b'V', b'I', b'W', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'J', b'U', b'J', b'V', b'J', b'W', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'K', b'U', b'K', b'V', b'K', b'W', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'L', b'U', b'L', b'V', b'L', b'W', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'M', b'U', b'M', b'V', b'M', b'W', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'N', b'U', b'N', b'V', b'N', b'W', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'O', b'U', b'O', b'V', b'O', b'W', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'P', b'U', b'P', b'V', b'P', b'W', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'Q', b'U', b'Q', b'V', b'Q', b'W', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'R', b'U', b'R', b'V', b'R', b'W', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'S', b'U', b'S', b'V', b'S', b'W', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T', b'T', b'U', b'T', b'V', b'T', b'W', b'U', b'0', b'U', b'1', b'U', b'2', b'U', b'3', b'U', b'4', b'U', b'5', b'U', b'6', b'U', b'7', b'U', b'8', b'U', b'9', b'U', b'A', b'U', b'B', b'U', b'C', b'U', b'D', b'U', b'E', b'U', b'F', b'U', b'G', b'U', b'H', b'U', b'I', b'U', b'J', b'U', b'K', b'U', b'L', b'U', b'M', b'U', b'N', b'U', b'O', b'U', b'P', b'U', b'Q', b'U', b'R', b'U', b'S', b'U', b'T', b'U', b'U', b'U', b'V', b'U', b'W', b'V', b'0', b'V', b'1', b'V', b'2', b'V', b'3', b'V', b'4', b'V', b'5', b'V', b'6', b'V', b'7', b'V', b'8', b'V', b'9', b'V', b'A', b'V', b'B', b'V', b'C', b'V', b'D', b'V', b'E', b'V', b'F', b'V', b'G', b'V', b'H', b'V', b'I', b'V', b'J', b'V', b'K', b'V', b'L', b'V', b'M', b'V', b'N', b'V', b'O', b'V', b'P', b'V', b'Q', b'V', b'R', b'V', b'S', b'V', b'T', b'V', b'U', b'V', b'V', b'V', b'W', b'W', b'0', b'W', b'1', b'W', b'2', b'W', b'3', b'W', b'4', b'W', b'5', b'W', b'6', b'W', b'7', b'W', b'8', b'W', b'9', b'W', b'A', b'W', b'B', b'W', b'C', b'W', b'D', b'W', b'E', b'W', b'F', b'W', b'G', b'W', b'H', b'W', b'I', b'W', b'J', b'W', b'K', b'W', b'L', b'W', b'M', b'W', b'N', b'W', b'O', b'W', b'P', b'W', b'Q', b'W', b'R', b'W', b'S', b'W', b'T', b'W', b'U', b'W', b'V', b'W', b'W'];
pub(crate) const DIGIT_TO_BASE34_SQUARED: [u8; 2312] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'0', b'U', b'0', b'V', b'0', b'W', b'0', b'X', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'1', b'U', b'1', b'V', b'1', b'W', b'1', b'X', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'2', b'U', b'2', b'V', b'2', b'W', b'2', b'X', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'3', b'U', b'3', b'V', b'3', b'W', b'3', b'X', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'4', b'U', b'4', b'V', b'4', b'W', b'4', b'X', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'5', b'U', b'5', b'V', b'5', b'W', b'5', b'X', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'6', b'U', b'6', b'V', b'6', b'W', b'6', b'X', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'7', b'U', b'7', b'V', b'7', b'W', b'7', b'X', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'8', b'U', b'8', b'V', b'8', b'W', b'8', b'X', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'9', b'U', b'9', b'V', b'9', b'W', b'9', b'X', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'A', b'U', b'A', b'V', b'A', b'W', b'A', b'X', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'B', b'U', b'B', b'V', b'B', b'W', b'B', b'X', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'C', b'U', b'C', b'V', b'C', b'W', b'C', b'X', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'D', b'U', b'D', b'V', b'D', b'W', b'D', b'X', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'E', b'U', b'E', b'V', b'E', b'W', b'E', b'X', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'F', b'U', b'F', b'V', b'F', b'W', b'F', b'X', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'G', b'U', b'G', b'V', b'G', b'W', b'G', b'X', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'H', b'U', b'H', b'V', b'H', b'W', b'H', b'X', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'I', b'U', b'I', b'V', b'I', b'W', b'I', b'X', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'J', b'U', b'J', b'V', b'J', b'W', b'J', b'X', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'K', b'U', b'K', b'V', b'K', b'W', b'K', b'X', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'L', b'U', b'L', b'V', b'L', b'W', b'L', b'X', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'M', b'U', b'M', b'V', b'M', b'W', b'M', b'X', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'N', b'U', b'N', b'V', b'N', b'W', b'N', b'X', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'O', b'U', b'O', b'V', b'O', b'W', b'O', b'X', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'P', b'U', b'P', b'V', b'P', b'W', b'P', b'X', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'Q', b'U', b'Q', b'V', b'Q', b'W', b'Q', b'X', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'R', b'U', b'R', b'V', b'R', b'W', b'R', b'X', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'S', b'U', b'S', b'V', b'S', b'W', b'S', b'X', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T', b'T', b'U', b'T', b'V', b'T', b'W', b'T', b'X', b'U', b'0', b'U', b'1', b'U', b'2', b'U', b'3', b'U', b'4', b'U', b'5', b'U', b'6', b'U', b'7', b'U', b'8', b'U', b'9', b'U', b'A', b'U', b'B', b'U', b'C', b'U', b'D', b'U', b'E', b'U', b'F', b'U', b'G', b'U', b'H', b'U', b'I', b'U', b'J', b'U', b'K', b'U', b'L', b'U', b'M', b'U', b'N', b'U', b'O', b'U', b'P', b'U', b'Q', b'U', b'R', b'U', b'S', b'U', b'T', b'U', b'U', b'U', b'V', b'U', b'W', b'U', b'X', b'V', b'0', b'V', b'1', b'V', b'2', b'V', b'3', b'V', b'4', b'V', b'5', b'V', b'6', b'V', b'7', b'V', b'8', b'V', b'9', b'V', b'A', b'V', b'B', b'V', b'C', b'V', b'D', b'V', b'E', b'V', b'F', b'V', b'G', b'V', b'H', b'V', b'I', b'V', b'J', b'V', b'K', b'V', b'L', b'V', b'M', b'V', b'N', b'V', b'O', b'V', b'P', b'V', b'Q', b'V', b'R', b'V', b'S', b'V', b'T', b'V', b'U', b'V', b'V', b'V', b'W', b'V', b'X', b'W', b'0', b'W', b'1', b'W', b'2', b'W', b'3', b'W', b'4', b'W', b'5', b'W', b'6', b'W', b'7', b'W', b'8', b'W', b'9', b'W', b'A', b'W', b'B', b'W', b'C', b'W', b'D', b'W', b'E', b'W', b'F', b'W', b'G', b'W', b'H', b'W', b'I', b'W', b'J', b'W', b'K', b'W', b'L', b'W', b'M', b'W', b'N', b'W', b'O', b'W', b'P', b'W', b'Q', b'W', b'R', b'W', b'S', b'W', b'T', b'W', b'U', b'W', b'V', b'W', b'W', b'W', b'X', b'X', b'0', b'X', b'1', b'X', b'2', b'X', b'3', b'X', b'4', b'X', b'5', b'X', b'6', b'X', b'7', b'X', b'8', b'X', b'9', b'X', b'A', b'X', b'B', b'X', b'C', b'X', b'D', b'X', b'E', b'X', b'F', b'X', b'G', b'X', b'H', b'X', b'I', b'X', b'J', b'X', b'K', b'X', b'L', b'X', b'M', b'X', b'N', b'X', b'O', b'X', b'P', b'X', b'Q', b'X', b'R', b'X', b'S', b'X', b'T', b'X', b'U', b'X', b'V', b'X', b'W', b'X', b'X'];
pub(crate) const DIGIT_TO_BASE35_SQUARED: [u8; 2450] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'0', b'U', b'0', b'V', b'0', b'W', b'0', b'X', b'0', b'Y', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'1', b'U', b'1', b'V', b'1', b'W', b'1', b'X', b'1', b'Y', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'2', b'U', b'2', b'V', b'2', b'W', b'2', b'X', b'2', b'Y', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'3', b'U', b'3', b'V', b'3', b'W', b'3', b'X', b'3', b'Y', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'4', b'U', b'4', b'V', b'4', b'W', b'4', b'X', b'4', b'Y', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'5', b'U', b'5', b'V', b'5', b'W', b'5', b'X', b'5', b'Y', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'6', b'U', b'6', b'V', b'6', b'W', b'6', b'X', b'6', b'Y', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'7', b'U', b'7', b'V', b'7', b'W', b'7', b'X', b'7', b'Y', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'8', b'U', b'8', b'V', b'8', b'W', b'8', b'X', b'8', b'Y', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'9', b'U', b'9', b'V', b'9', b'W', b'9', b'X', b'9', b'Y', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'A', b'U', b'A', b'V', b'A', b'W', b'A', b'X', b'A', b'Y', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'B', b'U', b'B', b'V', b'B', b'W', b'B', b'X', b'B', b'Y', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'C', b'U', b'C', b'V', b'C', b'W', b'C', b'X', b'C', b'Y', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'D', b'U', b'D', b'V', b'D', b'W', b'D', b'X', b'D', b'Y', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'E', b'U', b'E', b'V', b'E', b'W', b'E', b'X', b'E', b'Y', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'F', b'U', b'F', b'V', b'F', b'W', b'F', b'X', b'F', b'Y', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'G', b'U', b'G', b'V', b'G', b'W', b'G', b'X', b'G', b'Y', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'H', b'U', b'H', b'V', b'H', b'W', b'H', b'X', b'H', b'Y', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'I', b'U', b'I', b'V', b'I', b'W', b'I', b'X', b'I', b'Y', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'J', b'U', b'J', b'V', b'J', b'W', b'J', b'X', b'J', b'Y', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'K', b'U', b'K', b'V', b'K', b'W', b'K', b'X', b'K', b'Y', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'L', b'U', b'L', b'V', b'L', b'W', b'L', b'X', b'L', b'Y', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'M', b'U', b'M', b'V', b'M', b'W', b'M', b'X', b'M', b'Y', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'N', b'U', b'N', b'V', b'N', b'W', b'N', b'X', b'N', b'Y', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'O', b'U', b'O', b'V', b'O', b'W', b'O', b'X', b'O', b'Y', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'P', b'U', b'P', b'V', b'P', b'W', b'P', b'X', b'P', b'Y', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'Q', b'U', b'Q', b'V', b'Q', b'W', b'Q', b'X', b'Q', b'Y', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'R', b'U', b'R', b'V', b'R', b'W', b'R', b'X', b'R', b'Y', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'S', b'U', b'S', b'V', b'S', b'W', b'S', b'X', b'S', b'Y', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T', b'T', b'U', b'T', b'V', b'T', b'W', b'T', b'X', b'T', b'Y', b'U', b'0', b'U', b'1', b'U', b'2', b'U', b'3', b'U', b'4', b'U', b'5', b'U', b'6', b'U', b'7', b'U', b'8', b'U', b'9', b'U', b'A', b'U', b'B', b'U', b'C', b'U', b'D', b'U', b'E', b'U', b'F', b'U', b'G', b'U', b'H', b'U', b'I', b'U', b'J', b'U', b'K', b'U', b'L', b'U', b'M', b'U', b'N', b'U', b'O', b'U', b'P', b'U', b'Q', b'U', b'R', b'U', b'S', b'U', b'T', b'U', b'U', b'U', b'V', b'U', b'W', b'U', b'X', b'U', b'Y', b'V', b'0', b'V', b'1', b'V', b'2', b'V', b'3', b'V', b'4', b'V', b'5', b'V', b'6', b'V', b'7', b'V', b'8', b'V', b'9', b'V', b'A', b'V', b'B', b'V', b'C', b'V', b'D', b'V', b'E', b'V', b'F', b'V', b'G', b'V', b'H', b'V', b'I', b'V', b'J', b'V', b'K', b'V', b'L', b'V', b'M', b'V', b'N', b'V', b'O', b'V', b'P', b'V', b'Q', b'V', b'R', b'V', b'S', b'V', b'T', b'V', b'U', b'V', b'V', b'V', b'W', b'V', b'X', b'V', b'Y', b'W', b'0', b'W', b'1', b'W', b'2', b'W', b'3', b'W', b'4', b'W', b'5', b'W', b'6', b'W', b'7', b'W', b'8', b'W', b'9', b'W', b'A', b'W', b'B', b'W', b'C', b'W', b'D', b'W', b'E', b'W', b'F', b'W', b'G', b'W', b'H', b'W', b'I', b'W', b'J', b'W', b'K', b'W', b'L', b'W', b'M', b'W', b'N', b'W', b'O', b'W', b'P', b'W', b'Q', b'W', b'R', b'W', b'S', b'W', b'T', b'W', b'U', b'W', b'V', b'W', b'W', b'W', b'X', b'W', b'Y', b'X', b'0', b'X', b'1', b'X', b'2', b'X', b'3', b'X', b'4', b'X', b'5', b'X', b'6', b'X', b'7', b'X', b'8', b'X', b'9', b'X', b'A', b'X', b'B', b'X', b'C', b'X', b'D', b'X', b'E', b'X', b'F', b'X', b'G', b'X', b'H', b'X', b'I', b'X', b'J', b'X', b'K', b'X', b'L', b'X', b'M', b'X', b'N', b'X', b'O', b'X', b'P', b'X', b'Q', b'X', b'R', b'X', b'S', b'X', b'T', b'X', b'U', b'X', b'V', b'X', b'W', b'X', b'X', b'X', b'Y', b'Y', b'0', b'Y', b'1', b'Y', b'2', b'Y', b'3', b'Y', b'4', b'Y', b'5', b'Y', b'6', b'Y', b'7', b'Y', b'8', b'Y', b'9', b'Y', b'A', b'Y', b'B', b'Y', b'C', b'Y', b'D', b'Y', b'E', b'Y', b'F', b'Y', b'G', b'Y', b'H', b'Y', b'I', b'Y', b'J', b'Y', b'K', b'Y', b'L', b'Y', b'M', b'Y', b'N', b'Y', b'O', b'Y', b'P', b'Y', b'Q', b'Y', b'R', b'Y', b'S', b'Y', b'T', b'Y', b'U', b'Y', b'V', b'Y', b'W', b'Y', b'X', b'Y', b'Y'];
pub(crate) const DIGIT_TO_BASE36_SQUARED: [u8; 2592] = [b'0', b'0', b'0', b'1', b'0', b'2', b'0', b'3', b'0', b'4', b'0', b'5', b'0', b'6', b'0', b'7', b'0', b'8', b'0', b'9', b'0', b'A', b'0', b'B', b'0', b'C', b'0', b'D', b'0', b'E', b'0', b'F', b'0', b'G', b'0', b'H', b'0', b'I', b'0', b'J', b'0', b'K', b'0', b'L', b'0', b'M', b'0', b'N', b'0', b'O', b'0', b'P', b'0', b'Q', b'0', b'R', b'0', b'S', b'0', b'T', b'0', b'U', b'0', b'V', b'0', b'W', b'0', b'X', b'0', b'Y', b'0', b'Z', b'1', b'0', b'1', b'1', b'1', b'2', b'1', b'3', b'1', b'4', b'1', b'5', b'1', b'6', b'1', b'7', b'1', b'8', b'1', b'9', b'1', b'A', b'1', b'B', b'1', b'C', b'1', b'D', b'1', b'E', b'1', b'F', b'1', b'G', b'1', b'H', b'1', b'I', b'1', b'J', b'1', b'K', b'1', b'L', b'1', b'M', b'1', b'N', b'1', b'O', b'1', b'P', b'1', b'Q', b'1', b'R', b'1', b'S', b'1', b'T', b'1', b'U', b'1', b'V', b'1', b'W', b'1', b'X', b'1', b'Y', b'1', b'Z', b'2', b'0', b'2', b'1', b'2', b'2', b'2', b'3', b'2', b'4', b'2', b'5', b'2', b'6', b'2', b'7', b'2', b'8', b'2', b'9', b'2', b'A', b'2', b'B', b'2', b'C', b'2', b'D', b'2', b'E', b'2', b'F', b'2', b'G', b'2', b'H', b'2', b'I', b'2', b'J', b'2', b'K', b'2', b'L', b'2', b'M', b'2', b'N', b'2', b'O', b'2', b'P', b'2', b'Q', b'2', b'R', b'2', b'S', b'2', b'T', b'2', b'U', b'2', b'V', b'2', b'W', b'2', b'X', b'2', b'Y', b'2', b'Z', b'3', b'0', b'3', b'1', b'3', b'2', b'3', b'3', b'3', b'4', b'3', b'5', b'3', b'6', b'3', b'7', b'3', b'8', b'3', b'9', b'3', b'A', b'3', b'B', b'3', b'C', b'3', b'D', b'3', b'E', b'3', b'F', b'3', b'G', b'3', b'H', b'3', b'I', b'3', b'J', b'3', b'K', b'3', b'L', b'3', b'M', b'3', b'N', b'3', b'O', b'3', b'P', b'3', b'Q', b'3', b'R', b'3', b'S', b'3', b'T', b'3', b'U', b'3', b'V', b'3', b'W', b'3', b'X', b'3', b'Y', b'3', b'Z', b'4', b'0', b'4', b'1', b'4', b'2', b'4', b'3', b'4', b'4', b'4', b'5', b'4', b'6', b'4', b'7', b'4', b'8', b'4', b'9', b'4', b'A', b'4', b'B', b'4', b'C', b'4', b'D', b'4', b'E', b'4', b'F', b'4', b'G', b'4', b'H', b'4', b'I', b'4', b'J', b'4', b'K', b'4', b'L', b'4', b'M', b'4', b'N', b'4', b'O', b'4', b'P', b'4', b'Q', b'4', b'R', b'4', b'S', b'4', b'T', b'4', b'U', b'4', b'V', b'4', b'W', b'4', b'X', b'4', b'Y', b'4', b'Z', b'5', b'0', b'5', b'1', b'5', b'2', b'5', b'3', b'5', b'4', b'5', b'5', b'5', b'6', b'5', b'7', b'5', b'8', b'5', b'9', b'5', b'A', b'5', b'B', b'5', b'C', b'5', b'D', b'5', b'E', b'5', b'F', b'5', b'G', b'5', b'H', b'5', b'I', b'5', b'J', b'5', b'K', b'5', b'L', b'5', b'M', b'5', b'N', b'5', b'O', b'5', b'P', b'5', b'Q', b'5', b'R', b'5', b'S', b'5', b'T', b'5', b'U', b'5', b'V', b'5', b'W', b'5', b'X', b'5', b'Y', b'5', b'Z', b'6', b'0', b'6', b'1', b'6', b'2', b'6', b'3', b'6', b'4', b'6', b'5', b'6', b'6', b'6', b'7', b'6', b'8', b'6', b'9', b'6', b'A', b'6', b'B', b'6', b'C', b'6', b'D', b'6', b'E', b'6', b'F', b'6', b'G', b'6', b'H', b'6', b'I', b'6', b'J', b'6', b'K', b'6', b'L', b'6', b'M', b'6', b'N', b'6', b'O', b'6', b'P', b'6', b'Q', b'6', b'R', b'6', b'S', b'6', b'T', b'6', b'U', b'6', b'V', b'6', b'W', b'6', b'X', b'6', b'Y', b'6', b'Z', b'7', b'0', b'7', b'1', b'7', b'2', b'7', b'3', b'7', b'4', b'7', b'5', b'7', b'6', b'7', b'7', b'7', b'8', b'7', b'9', b'7', b'A', b'7', b'B', b'7', b'C', b'7', b'D', b'7', b'E', b'7', b'F', b'7', b'G', b'7', b'H', b'7', b'I', b'7', b'J', b'7', b'K', b'7', b'L', b'7', b'M', b'7', b'N', b'7', b'O', b'7', b'P', b'7', b'Q', b'7', b'R', b'7', b'S', b'7', b'T', b'7', b'U', b'7', b'V', b'7', b'W', b'7', b'X', b'7', b'Y', b'7', b'Z', b'8', b'0', b'8', b'1', b'8', b'2', b'8', b'3', b'8', b'4', b'8', b'5', b'8', b'6', b'8', b'7', b'8', b'8', b'8', b'9', b'8', b'A', b'8', b'B', b'8', b'C', b'8', b'D', b'8', b'E', b'8', b'F', b'8', b'G', b'8', b'H', b'8', b'I', b'8', b'J', b'8', b'K', b'8', b'L', b'8', b'M', b'8', b'N', b'8', b'O', b'8', b'P', b'8', b'Q', b'8', b'R', b'8', b'S', b'8', b'T', b'8', b'U', b'8', b'V', b'8', b'W', b'8', b'X', b'8', b'Y', b'8', b'Z', b'9', b'0', b'9', b'1', b'9', b'2', b'9', b'3', b'9', b'4', b'9', b'5', b'9', b'6', b'9', b'7', b'9', b'8', b'9', b'9', b'9', b'A', b'9', b'B', b'9', b'C', b'9', b'D', b'9', b'E', b'9', b'F', b'9', b'G', b'9', b'H', b'9', b'I', b'9', b'J', b'9', b'K', b'9', b'L', b'9', b'M', b'9', b'N', b'9', b'O', b'9', b'P', b'9', b'Q', b'9', b'R', b'9', b'S', b'9', b'T', b'9', b'U', b'9', b'V', b'9', b'W', b'9', b'X', b'9', b'Y', b'9', b'Z', b'A', b'0', b'A', b'1', b'A', b'2', b'A', b'3', b'A', b'4', b'A', b'5', b'A', b'6', b'A', b'7', b'A', b'8', b'A', b'9', b'A', b'A', b'A', b'B', b'A', b'C', b'A', b'D', b'A', b'E', b'A', b'F', b'A', b'G', b'A', b'H', b'A', b'I', b'A', b'J', b'A', b'K', b'A', b'L', b'A', b'M', b'A', b'N', b'A', b'O', b'A', b'P', b'A', b'Q', b'A', b'R', b'A', b'S', b'A', b'T', b'A', b'U', b'A', b'V', b'A', b'W', b'A', b'X', b'A', b'Y', b'A', b'Z', b'B', b'0', b'B', b'1', b'B', b'2', b'B', b'3', b'B', b'4', b'B', b'5', b'B', b'6', b'B', b'7', b'B', b'8', b'B', b'9', b'B', b'A', b'B', b'B', b'B', b'C', b'B', b'D', b'B', b'E', b'B', b'F', b'B', b'G', b'B', b'H', b'B', b'I', b'B', b'J', b'B', b'K', b'B', b'L', b'B', b'M', b'B', b'N', b'B', b'O', b'B', b'P', b'B', b'Q', b'B', b'R', b'B', b'S', b'B', b'T', b'B', b'U', b'B', b'V', b'B', b'W', b'B', b'X', b'B', b'Y', b'B', b'Z', b'C', b'0', b'C', b'1', b'C', b'2', b'C', b'3', b'C', b'4', b'C', b'5', b'C', b'6', b'C', b'7', b'C', b'8', b'C', b'9', b'C', b'A', b'C', b'B', b'C', b'C', b'C', b'D', b'C', b'E', b'C', b'F', b'C', b'G', b'C', b'H', b'C', b'I', b'C', b'J', b'C', b'K', b'C', b'L', b'C', b'M', b'C', b'N', b'C', b'O', b'C', b'P', b'C', b'Q', b'C', b'R', b'C', b'S', b'C', b'T', b'C', b'U', b'C', b'V', b'C', b'W', b'C', b'X', b'C', b'Y', b'C', b'Z', b'D', b'0', b'D', b'1', b'D', b'2', b'D', b'3', b'D', b'4', b'D', b'5', b'D', b'6', b'D', b'7', b'D', b'8', b'D', b'9', b'D', b'A', b'D', b'B', b'D', b'C', b'D', b'D', b'D', b'E', b'D', b'F', b'D', b'G', b'D', b'H', b'D', b'I', b'D', b'J', b'D', b'K', b'D', b'L', b'D', b'M', b'D', b'N', b'D', b'O', b'D', b'P', b'D', b'Q', b'D', b'R', b'D', b'S', b'D', b'T', b'D', b'U', b'D', b'V', b'D', b'W', b'D', b'X', b'D', b'Y', b'D', b'Z', b'E', b'0', b'E', b'1', b'E', b'2', b'E', b'3', b'E', b'4', b'E', b'5', b'E', b'6', b'E', b'7', b'E', b'8', b'E', b'9', b'E', b'A', b'E', b'B', b'E', b'C', b'E', b'D', b'E', b'E', b'E', b'F', b'E', b'G', b'E', b'H', b'E', b'I', b'E', b'J', b'E', b'K', b'E', b'L', b'E', b'M', b'E', b'N', b'E', b'O', b'E', b'P', b'E', b'Q', b'E', b'R', b'E', b'S', b'E', b'T', b'E', b'U', b'E', b'V', b'E', b'W', b'E', b'X', b'E', b'Y', b'E', b'Z', b'F', b'0', b'F', b'1', b'F', b'2', b'F', b'3', b'F', b'4', b'F', b'5', b'F', b'6', b'F', b'7', b'F', b'8', b'F', b'9', b'F', b'A', b'F', b'B', b'F', b'C', b'F', b'D', b'F', b'E', b'F', b'F', b'F', b'G', b'F', b'H', b'F', b'I', b'F', b'J', b'F', b'K', b'F', b'L', b'F', b'M', b'F', b'N', b'F', b'O', b'F', b'P', b'F', b'Q', b'F', b'R', b'F', b'S', b'F', b'T', b'F', b'U', b'F', b'V', b'F', b'W', b'F', b'X', b'F', b'Y', b'F', b'Z', b'G', b'0', b'G', b'1', b'G', b'2', b'G', b'3', b'G', b'4', b'G', b'5', b'G', b'6', b'G', b'7', b'G', b'8', b'G', b'9', b'G', b'A', b'G', b'B', b'G', b'C', b'G', b'D', b'G', b'E', b'G', b'F', b'G', b'G', b'G', b'H', b'G', b'I', b'G', b'J', b'G', b'K', b'G', b'L', b'G', b'M', b'G', b'N', b'G', b'O', b'G', b'P', b'G', b'Q', b'G', b'R', b'G', b'S', b'G', b'T', b'G', b'U', b'G', b'V', b'G', b'W', b'G', b'X', b'G', b'Y', b'G', b'Z', b'H', b'0', b'H', b'1', b'H', b'2', b'H', b'3', b'H', b'4', b'H', b'5', b'H', b'6', b'H', b'7', b'H', b'8', b'H', b'9', b'H', b'A', b'H', b'B', b'H', b'C', b'H', b'D', b'H', b'E', b'H', b'F', b'H', b'G', b'H', b'H', b'H', b'I', b'H', b'J', b'H', b'K', b'H', b'L', b'H', b'M', b'H', b'N', b'H', b'O', b'H', b'P', b'H', b'Q', b'H', b'R', b'H', b'S', b'H', b'T', b'H', b'U', b'H', b'V', b'H', b'W', b'H', b'X', b'H', b'Y', b'H', b'Z', b'I', b'0', b'I', b'1', b'I', b'2', b'I', b'3', b'I', b'4', b'I', b'5', b'I', b'6', b'I', b'7', b'I', b'8', b'I', b'9', b'I', b'A', b'I', b'B', b'I', b'C', b'I', b'D', b'I', b'E', b'I', b'F', b'I', b'G', b'I', b'H', b'I', b'I', b'I', b'J', b'I', b'K', b'I', b'L', b'I', b'M', b'I', b'N', b'I', b'O', b'I', b'P', b'I', b'Q', b'I', b'R', b'I', b'S', b'I', b'T', b'I', b'U', b'I', b'V', b'I', b'W', b'I', b'X', b'I', b'Y', b'I', b'Z', b'J', b'0', b'J', b'1', b'J', b'2', b'J', b'3', b'J', b'4', b'J', b'5', b'J', b'6', b'J', b'7', b'J', b'8', b'J', b'9', b'J', b'A', b'J', b'B', b'J', b'C', b'J', b'D', b'J', b'E', b'J', b'F', b'J', b'G', b'J', b'H', b'J', b'I', b'J', b'J', b'J', b'K', b'J', b'L', b'J', b'M', b'J', b'N', b'J', b'O', b'J', b'P', b'J', b'Q', b'J', b'R', b'J', b'S', b'J', b'T', b'J', b'U', b'J', b'V', b'J', b'W', b'J', b'X', b'J', b'Y', b'J', b'Z', b'K', b'0', b'K', b'1', b'K', b'2', b'K', b'3', b'K', b'4', b'K', b'5', b'K', b'6', b'K', b'7', b'K', b'8', b'K', b'9', b'K', b'A', b'K', b'B', b'K', b'C', b'K', b'D', b'K', b'E', b'K', b'F', b'K', b'G', b'K', b'H', b'K', b'I', b'K', b'J', b'K', b'K', b'K', b'L', b'K', b'M', b'K', b'N', b'K', b'O', b'K', b'P', b'K', b'Q', b'K', b'R', b'K', b'S', b'K', b'T', b'K', b'U', b'K', b'V', b'K', b'W', b'K', b'X', b'K', b'Y', b'K', b'Z', b'L', b'0', b'L', b'1', b'L', b'2', b'L', b'3', b'L', b'4', b'L', b'5', b'L', b'6', b'L', b'7', b'L', b'8', b'L', b'9', b'L', b'A', b'L', b'B', b'L', b'C', b'L', b'D', b'L', b'E', b'L', b'F', b'L', b'G', b'L', b'H', b'L', b'I', b'L', b'J', b'L', b'K', b'L', b'L', b'L', b'M', b'L', b'N', b'L', b'O', b'L', b'P', b'L', b'Q', b'L', b'R', b'L', b'S', b'L', b'T', b'L', b'U', b'L', b'V', b'L', b'W', b'L', b'X', b'L', b'Y', b'L', b'Z', b'M', b'0', b'M', b'1', b'M', b'2', b'M', b'3', b'M', b'4', b'M', b'5', b'M', b'6', b'M', b'7', b'M', b'8', b'M', b'9', b'M', b'A', b'M', b'B', b'M', b'C', b'M', b'D', b'M', b'E', b'M', b'F', b'M', b'G', b'M', b'H', b'M', b'I', b'M', b'J', b'M', b'K', b'M', b'L', b'M', b'M', b'M', b'N', b'M', b'O', b'M', b'P', b'M', b'Q', b'M', b'R', b'M', b'S', b'M', b'T', b'M', b'U', b'M', b'V', b'M', b'W', b'M', b'X', b'M', b'Y', b'M', b'Z', b'N', b'0', b'N', b'1', b'N', b'2', b'N', b'3', b'N', b'4', b'N', b'5', b'N', b'6', b'N', b'7', b'N', b'8', b'N', b'9', b'N', b'A', b'N', b'B', b'N', b'C', b'N', b'D', b'N', b'E', b'N', b'F', b'N', b'G', b'N', b'H', b'N', b'I', b'N', b'J', b'N', b'K', b'N', b'L', b'N', b'M', b'N', b'N', b'N', b'O', b'N', b'P', b'N', b'Q', b'N', b'R', b'N', b'S', b'N', b'T', b'N', b'U', b'N', b'V', b'N', b'W', b'N', b'X', b'N', b'Y', b'N', b'Z', b'O', b'0', b'O', b'1', b'O', b'2', b'O', b'3', b'O', b'4', b'O', b'5', b'O', b'6', b'O', b'7', b'O', b'8', b'O', b'9', b'O', b'A', b'O', b'B', b'O', b'C', b'O', b'D', b'O', b'E', b'O', b'F', b'O', b'G', b'O', b'H', b'O', b'I', b'O', b'J', b'O', b'K', b'O', b'L', b'O', b'M', b'O', b'N', b'O', b'O', b'O', b'P', b'O', b'Q', b'O', b'R', b'O', b'S', b'O', b'T', b'O', b'U', b'O', b'V', b'O', b'W', b'O', b'X', b'O', b'Y', b'O', b'Z', b'P', b'0', b'P', b'1', b'P', b'2', b'P', b'3', b'P', b'4', b'P', b'5', b'P', b'6', b'P', b'7', b'P', b'8', b'P', b'9', b'P', b'A', b'P', b'B', b'P', b'C', b'P', b'D', b'P', b'E', b'P', b'F', b'P', b'G', b'P', b'H', b'P', b'I', b'P', b'J', b'P', b'K', b'P', b'L', b'P', b'M', b'P', b'N', b'P', b'O', b'P', b'P', b'P', b'Q', b'P', b'R', b'P', b'S', b'P', b'T', b'P', b'U', b'P', b'V', b'P', b'W', b'P', b'X', b'P', b'Y', b'P', b'Z', b'Q', b'0', b'Q', b'1', b'Q', b'2', b'Q', b'3', b'Q', b'4', b'Q', b'5', b'Q', b'6', b'Q', b'7', b'Q', b'8', b'Q', b'9', b'Q', b'A', b'Q', b'B', b'Q', b'C', b'Q', b'D', b'Q', b'E', b'Q', b'F', b'Q', b'G', b'Q', b'H', b'Q', b'I', b'Q', b'J', b'Q', b'K', b'Q', b'L', b'Q', b'M', b'Q', b'N', b'Q', b'O', b'Q', b'P', b'Q', b'Q', b'Q', b'R', b'Q', b'S', b'Q', b'T', b'Q', b'U', b'Q', b'V', b'Q', b'W', b'Q', b'X', b'Q', b'Y', b'Q', b'Z', b'R', b'0', b'R', b'1', b'R', b'2', b'R', b'3', b'R', b'4', b'R', b'5', b'R', b'6', b'R', b'7', b'R', b'8', b'R', b'9', b'R', b'A', b'R', b'B', b'R', b'C', b'R', b'D', b'R', b'E', b'R', b'F', b'R', b'G', b'R', b'H', b'R', b'I', b'R', b'J', b'R', b'K', b'R', b'L', b'R', b'M', b'R', b'N', b'R', b'O', b'R', b'P', b'R', b'Q', b'R', b'R', b'R', b'S', b'R', b'T', b'R', b'U', b'R', b'V', b'R', b'W', b'R', b'X', b'R', b'Y', b'R', b'Z', b'S', b'0', b'S', b'1', b'S', b'2', b'S', b'3', b'S', b'4', b'S', b'5', b'S', b'6', b'S', b'7', b'S', b'8', b'S', b'9', b'S', b'A', b'S', b'B', b'S', b'C', b'S', b'D', b'S', b'E', b'S', b'F', b'S', b'G', b'S', b'H', b'S', b'I', b'S', b'J', b'S', b'K', b'S', b'L', b'S', b'M', b'S', b'N', b'S', b'O', b'S', b'P', b'S', b'Q', b'S', b'R', b'S', b'S', b'S', b'T', b'S', b'U', b'S', b'V', b'S', b'W', b'S', b'X', b'S', b'Y', b'S', b'Z', b'T', b'0', b'T', b'1', b'T', b'2', b'T', b'3', b'T', b'4', b'T', b'5', b'T', b'6', b'T', b'7', b'T', b'8', b'T', b'9', b'T', b'A', b'T', b'B', b'T', b'C', b'T', b'D', b'T', b'E', b'T', b'F', b'T', b'G', b'T', b'H', b'T', b'I', b'T', b'J', b'T', b'K', b'T', b'L', b'T', b'M', b'T', b'N', b'T', b'O', b'T', b'P', b'T', b'Q', b'T', b'R', b'T', b'S', b'T', b'T', b'T', b'U', b'T', b'V', b'T', b'W', b'T', b'X', b'T', b'Y', b'T', b'Z', b'U', b'0', b'U', b'1', b'U', b'2', b'U', b'3', b'U', b'4', b'U', b'5', b'U', b'6', b'U', b'7', b'U', b'8', b'U', b'9', b'U', b'A', b'U', b'B', b'U', b'C', b'U', b'D', b'U', b'E', b'U', b'F', b'U', b'G', b'U', b'H', b'U', b'I', b'U', b'J', b'U', b'K', b'U', b'L', b'U', b'M', b'U', b'N', b'U', b'O', b'U', b'P', b'U', b'Q', b'U', b'R', b'U', b'S', b'U', b'T', b'U', b'U', b'U', b'V', b'U', b'W', b'U', b'X', b'U', b'Y', b'U', b'Z', b'V', b'0', b'V', b'1', b'V', b'2', b'V', b'3', b'V', b'4', b'V', b'5', b'V', b'6', b'V', b'7', b'V', b'8', b'V', b'9', b'V', b'A', b'V', b'B', b'V', b'C', b'V', b'D', b'V', b'E', b'V', b'F', b'V', b'G', b'V', b'H', b'V', b'I', b'V', b'J', b'V', b'K', b'V', b'L', b'V', b'M', b'V', b'N', b'V', b'O', b'V', b'P', b'V', b'Q', b'V', b'R', b'V', b'S', b'V', b'T', b'V', b'U', b'V', b'V', b'V', b'W', b'V', b'X', b'V', b'Y', b'V', b'Z', b'W', b'0', b'W', b'1', b'W', b'2', b'W', b'3', b'W', b'4', b'W', b'5', b'W', b'6', b'W', b'7', b'W', b'8', b'W', b'9', b'W', b'A', b'W', b'B', b'W', b'C', b'W', b'D', b'W', b'E', b'W', b'F', b'W', b'G', b'W', b'H', b'W', b'I', b'W', b'J', b'W', b'K', b'W', b'L', b'W', b'M', b'W', b'N', b'W', b'O', b'W', b'P', b'W', b'Q', b'W', b'R', b'W', b'S', b'W', b'T', b'W', b'U', b'W', b'V', b'W', b'W', b'W', b'X', b'W', b'Y', b'W', b'Z', b'X', b'0', b'X', b'1', b'X', b'2', b'X', b'3', b'X', b'4', b'X', b'5', b'X', b'6', b'X', b'7', b'X', b'8', b'X', b'9', b'X', b'A', b'X', b'B', b'X', b'C', b'X', b'D', b'X', b'E', b'X', b'F', b'X', b'G', b'X', b'H', b'X', b'I', b'X', b'J', b'X', b'K', b'X', b'L', b'X', b'M', b'X', b'N', b'X', b'O', b'X', b'P', b'X', b'Q', b'X', b'R', b'X', b'S', b'X', b'T', b'X', b'U', b'X', b'V', b'X', b'W', b'X', b'X', b'X', b'Y', b'X', b'Z', b'Y', b'0', b'Y', b'1', b'Y', b'2', b'Y', b'3', b'Y', b'4', b'Y', b'5', b'Y', b'6', b'Y', b'7', b'Y', b'8', b'Y', b'9', b'Y', b'A', b'Y', b'B', b'Y', b'C', b'Y', b'D', b'Y', b'E', b'Y', b'F', b'Y', b'G', b'Y', b'H', b'Y', b'I', b'Y', b'J', b'Y', b'K', b'Y', b'L', b'Y', b'M', b'Y', b'N', b'Y', b'O', b'Y', b'P', b'Y', b'Q', b'Y', b'R', b'Y', b'S', b'Y', b'T', b'Y', b'U', b'Y', b'V', b'Y', b'W', b'Y', b'X', b'Y', b'Y', b'Y', b'Z', b'Z', b'0', b'Z', b'1', b'Z', b'2', b'Z', b'3', b'Z', b'4', b'Z', b'5', b'Z', b'6', b'Z', b'7', b'Z', b'8', b'Z', b'9', b'Z', b'A', b'Z', b'B', b'Z', b'C', b'Z', b'D', b'Z', b'E', b'Z', b'F', b'Z', b'G', b'Z', b'H', b'Z', b'I', b'Z', b'J', b'Z', b'K', b'Z', b'L', b'Z', b'M', b'Z', b'N', b'Z', b'O', b'Z', b'P', b'Z', b'Q', b'Z', b'R', b'Z', b'S', b'Z', b'T', b'Z', b'U', b'Z', b'V', b'Z', b'W', b'Z', b'X', b'Z', b'Y', b'Z', b'Z'];
}}  // cfg_if

// EXACT EXPONENT

/// Get exact exponent limit for radix.
pub(crate) trait ExactExponent {
    /// Get min and max exponent limits (exact) from radix.
    fn exponent_limit<T: Integer>(radix: T) -> (i32, i32);

    /// Get the number of digits that can be shifted from exponent to mantissa.
    fn mantissa_limit<T: Integer>(radix: T) -> i32;
}

impl ExactExponent for f32 {
    #[inline]
    fn exponent_limit<T: Integer>(radix: T) -> (i32, i32) {
        debug_assert_radix!(radix);
        #[cfg(not(feature = "radix"))] {
            (-10, 10)
        }

        #[cfg(feature = "radix")] {
            match radix.as_i32() {
                2  => (-149, 127),
                3  => (-15, 15),
                4  => (-74, 63),
                5  => (-10, 10),
                6  => (-15, 15),
                7  => (-8, 8),
                8  => (-49, 42),
                9  => (-7, 7),
                10 => (-10, 10),
                11 => (-6, 6),
                12 => (-15, 15),
                13 => (-6, 6),
                14 => (-8, 8),
                15 => (-6, 6),
                16 => (-37, 31),
                17 => (-5, 5),
                18 => (-7, 7),
                19 => (-5, 5),
                20 => (-10, 10),
                21 => (-5, 5),
                22 => (-6, 6),
                23 => (-5, 5),
                24 => (-15, 15),
                25 => (-5, 5),
                26 => (-6, 6),
                27 => (-5, 5),
                28 => (-8, 8),
                29 => (-4, 4),
                30 => (-6, 6),
                31 => (-4, 4),
                32 => (-29, 25),
                33 => (-4, 4),
                34 => (-5, 5),
                35 => (-4, 4),
                36 => (-7, 7),
                // Invalid radix
                _  => unreachable!(),
            }
        }
    }

    #[inline]
    fn mantissa_limit<T: Integer>(radix: T) -> i32 {
        debug_assert_radix!(radix);
        #[cfg(not(feature = "radix"))] {
            7
        }

        #[cfg(feature = "radix")] {
            match radix.as_i32() {
                2  => 23,
                3  => 15,
                4  => 11,
                5  => 10,
                6  => 9,
                7  => 8,
                8  => 7,
                9  => 7,
                10 => 7,
                11 => 6,
                12 => 6,
                13 => 6,
                14 => 6,
                15 => 6,
                16 => 5,
                17 => 5,
                18 => 5,
                19 => 5,
                20 => 5,
                21 => 5,
                22 => 5,
                23 => 5,
                24 => 5,
                25 => 5,
                26 => 5,
                27 => 5,
                28 => 4,
                29 => 4,
                30 => 4,
                31 => 4,
                32 => 4,
                33 => 4,
                34 => 4,
                35 => 4,
                36 => 4,
                // Invalid radix
                _  => unreachable!(),
            }
        }
    }
}

/// Precalculated min and max exponents for values exactly representable as f64.
///
/// Table of values where `radix**min` and `radix**max` are the limits of types
/// exactly representable as an f64.

impl ExactExponent for f64 {
    #[inline]
    fn exponent_limit<T: Integer>(radix: T) -> (i32, i32) {
        debug_assert_radix!(radix);
        #[cfg(not(feature = "radix"))] {
            (-22, 22)
        }

        #[cfg(feature = "radix")] {
            match radix.as_i32() {
                2  => (-1074, 1023),
                3  => (-33, 33),
                4  => (-537, 511),
                5  => (-22, 22),
                6  => (-33, 33),
                7  => (-18, 18),
                8  => (-358, 341),
                9  => (-16, 16),
                10 => (-22, 22),
                11 => (-15, 15),
                12 => (-33, 33),
                13 => (-14, 14),
                14 => (-18, 18),
                15 => (-13, 13),
                16 => (-268, 255),
                17 => (-12, 12),
                18 => (-16, 16),
                19 => (-12, 12),
                20 => (-22, 22),
                21 => (-12, 12),
                22 => (-15, 15),
                23 => (-11, 11),
                24 => (-33, 33),
                25 => (-11, 11),
                26 => (-14, 14),
                27 => (-11, 11),
                28 => (-18, 18),
                29 => (-10, 10),
                30 => (-13, 13),
                31 => (-10, 10),
                32 => (-214, 204),
                33 => (-10, 10),
                34 => (-12, 12),
                35 => (-10, 10),
                36 => (-16, 16),
                // Invalid radix
                _  => unreachable!(),
            }
        }
    }

    #[inline]
    fn mantissa_limit<T: Integer>(radix: T) -> i32 {
        debug_assert_radix!(radix);
        #[cfg(not(feature = "radix"))] {
            15
        }

        #[cfg(feature = "radix")] {
            match radix.as_i32() {
                2  => 52,
                3  => 33,
                4  => 26,
                5  => 22,
                6  => 20,
                7  => 18,
                8  => 17,
                9  => 16,
                10 => 15,
                11 => 15,
                12 => 14,
                13 => 14,
                14 => 13,
                15 => 13,
                16 => 13,
                17 => 12,
                18 => 12,
                19 => 12,
                20 => 12,
                21 => 12,
                22 => 11,
                23 => 11,
                24 => 11,
                25 => 11,
                26 => 11,
                27 => 11,
                28 => 11,
                29 => 10,
                30 => 10,
                31 => 10,
                32 => 10,
                33 => 10,
                34 => 10,
                35 => 10,
                36 => 10,
                // Invalid radix
                _  => unreachable!(),
            }
        }
    }
}

// Conditionally compile the radix POWI tables.
// These tables contain all the values that can be exactly represented
// by a given float of a certain size.
//
// Total array storage: 2.1 KB (f32) + 21.5 KB (f64).
// The total performance enhancements save ~350+ clock cycles (x86) or
// ~100 clock cycles (x87) for the FYL2X and F2XM1 instructions, require
// to compute a power. This should be a significant performance win.

cfg_if! {
if #[cfg(feature = "correct")] {
// TRANSMUTE

/// Hacky transmute for 32-bit and 64-bit floats.
///
/// Remove when `f32::from_bits` and `f64::from_bits` become const
/// in stable.
#[cfg(feature = "radix")]
union Transmute<U: Copy, F: Copy> {
    u: U,
    f: F,
}

// TABLE POW

/// Calculate powers using pre-calculated lookup tables.
/// No error-checking occurs, these methods are not safe.
pub(crate) trait TablePower {
    /// Exponent
    const POW2_EXPONENT_BIAS: i32;

    /// Get power of 2 from exponent.
    #[cfg(feature = "radix")]
    fn table_pow2(exponent: i32) -> Self;

    /// Get power of 2 from exponent.
    fn table_pow<T: Integer>(radix: T, exponent: i32) -> Self;
}

// F32

/// Hacky transmute for 32-bit values.
#[cfg(feature = "radix")]
type F32Transmute = Transmute<u32, f32>;

/// Precalculated values of 2**i for i in range [-149, 127].
#[cfg(feature = "radix")]
#[allow(const_err)]     // This warning is fixed in 1.26.0
const F32_POW2: [f32; 277] = unsafe {[
    // Denormal floats ([-149, -127])
    F32Transmute { u: 0x00000001 }.f,
    F32Transmute { u: 0x00000002 }.f,
    F32Transmute { u: 0x00000004 }.f,
    F32Transmute { u: 0x00000008 }.f,
    F32Transmute { u: 0x00000010 }.f,
    F32Transmute { u: 0x00000020 }.f,
    F32Transmute { u: 0x00000040 }.f,
    F32Transmute { u: 0x00000080 }.f,
    F32Transmute { u: 0x00000100 }.f,
    F32Transmute { u: 0x00000200 }.f,
    F32Transmute { u: 0x00000400 }.f,
    F32Transmute { u: 0x00000800 }.f,
    F32Transmute { u: 0x00001000 }.f,
    F32Transmute { u: 0x00002000 }.f,
    F32Transmute { u: 0x00004000 }.f,
    F32Transmute { u: 0x00008000 }.f,
    F32Transmute { u: 0x00010000 }.f,
    F32Transmute { u: 0x00020000 }.f,
    F32Transmute { u: 0x00040000 }.f,
    F32Transmute { u: 0x00080000 }.f,
    F32Transmute { u: 0x00100000 }.f,
    F32Transmute { u: 0x00200000 }.f,
    F32Transmute { u: 0x00400000 }.f,

    // Regular floats (exponent only, [-126, 127])
    F32Transmute { u: 0x00800000 }.f,
    F32Transmute { u: 0x01000000 }.f,
    F32Transmute { u: 0x01800000 }.f,
    F32Transmute { u: 0x02000000 }.f,
    F32Transmute { u: 0x02800000 }.f,
    F32Transmute { u: 0x03000000 }.f,
    F32Transmute { u: 0x03800000 }.f,
    F32Transmute { u: 0x04000000 }.f,
    F32Transmute { u: 0x04800000 }.f,
    F32Transmute { u: 0x05000000 }.f,
    F32Transmute { u: 0x05800000 }.f,
    F32Transmute { u: 0x06000000 }.f,
    F32Transmute { u: 0x06800000 }.f,
    F32Transmute { u: 0x07000000 }.f,
    F32Transmute { u: 0x07800000 }.f,
    F32Transmute { u: 0x08000000 }.f,
    F32Transmute { u: 0x08800000 }.f,
    F32Transmute { u: 0x09000000 }.f,
    F32Transmute { u: 0x09800000 }.f,
    F32Transmute { u: 0x0A000000 }.f,
    F32Transmute { u: 0x0A800000 }.f,
    F32Transmute { u: 0x0B000000 }.f,
    F32Transmute { u: 0x0B800000 }.f,
    F32Transmute { u: 0x0C000000 }.f,
    F32Transmute { u: 0x0C800000 }.f,
    F32Transmute { u: 0x0D000000 }.f,
    F32Transmute { u: 0x0D800000 }.f,
    F32Transmute { u: 0x0E000000 }.f,
    F32Transmute { u: 0x0E800000 }.f,
    F32Transmute { u: 0x0F000000 }.f,
    F32Transmute { u: 0x0F800000 }.f,
    F32Transmute { u: 0x10000000 }.f,
    F32Transmute { u: 0x10800000 }.f,
    F32Transmute { u: 0x11000000 }.f,
    F32Transmute { u: 0x11800000 }.f,
    F32Transmute { u: 0x12000000 }.f,
    F32Transmute { u: 0x12800000 }.f,
    F32Transmute { u: 0x13000000 }.f,
    F32Transmute { u: 0x13800000 }.f,
    F32Transmute { u: 0x14000000 }.f,
    F32Transmute { u: 0x14800000 }.f,
    F32Transmute { u: 0x15000000 }.f,
    F32Transmute { u: 0x15800000 }.f,
    F32Transmute { u: 0x16000000 }.f,
    F32Transmute { u: 0x16800000 }.f,
    F32Transmute { u: 0x17000000 }.f,
    F32Transmute { u: 0x17800000 }.f,
    F32Transmute { u: 0x18000000 }.f,
    F32Transmute { u: 0x18800000 }.f,
    F32Transmute { u: 0x19000000 }.f,
    F32Transmute { u: 0x19800000 }.f,
    F32Transmute { u: 0x1A000000 }.f,
    F32Transmute { u: 0x1A800000 }.f,
    F32Transmute { u: 0x1B000000 }.f,
    F32Transmute { u: 0x1B800000 }.f,
    F32Transmute { u: 0x1C000000 }.f,
    F32Transmute { u: 0x1C800000 }.f,
    F32Transmute { u: 0x1D000000 }.f,
    F32Transmute { u: 0x1D800000 }.f,
    F32Transmute { u: 0x1E000000 }.f,
    F32Transmute { u: 0x1E800000 }.f,
    F32Transmute { u: 0x1F000000 }.f,
    F32Transmute { u: 0x1F800000 }.f,
    F32Transmute { u: 0x20000000 }.f,
    F32Transmute { u: 0x20800000 }.f,
    F32Transmute { u: 0x21000000 }.f,
    F32Transmute { u: 0x21800000 }.f,
    F32Transmute { u: 0x22000000 }.f,
    F32Transmute { u: 0x22800000 }.f,
    F32Transmute { u: 0x23000000 }.f,
    F32Transmute { u: 0x23800000 }.f,
    F32Transmute { u: 0x24000000 }.f,
    F32Transmute { u: 0x24800000 }.f,
    F32Transmute { u: 0x25000000 }.f,
    F32Transmute { u: 0x25800000 }.f,
    F32Transmute { u: 0x26000000 }.f,
    F32Transmute { u: 0x26800000 }.f,
    F32Transmute { u: 0x27000000 }.f,
    F32Transmute { u: 0x27800000 }.f,
    F32Transmute { u: 0x28000000 }.f,
    F32Transmute { u: 0x28800000 }.f,
    F32Transmute { u: 0x29000000 }.f,
    F32Transmute { u: 0x29800000 }.f,
    F32Transmute { u: 0x2A000000 }.f,
    F32Transmute { u: 0x2A800000 }.f,
    F32Transmute { u: 0x2B000000 }.f,
    F32Transmute { u: 0x2B800000 }.f,
    F32Transmute { u: 0x2C000000 }.f,
    F32Transmute { u: 0x2C800000 }.f,
    F32Transmute { u: 0x2D000000 }.f,
    F32Transmute { u: 0x2D800000 }.f,
    F32Transmute { u: 0x2E000000 }.f,
    F32Transmute { u: 0x2E800000 }.f,
    F32Transmute { u: 0x2F000000 }.f,
    F32Transmute { u: 0x2F800000 }.f,
    F32Transmute { u: 0x30000000 }.f,
    F32Transmute { u: 0x30800000 }.f,
    F32Transmute { u: 0x31000000 }.f,
    F32Transmute { u: 0x31800000 }.f,
    F32Transmute { u: 0x32000000 }.f,
    F32Transmute { u: 0x32800000 }.f,
    F32Transmute { u: 0x33000000 }.f,
    F32Transmute { u: 0x33800000 }.f,
    F32Transmute { u: 0x34000000 }.f,
    F32Transmute { u: 0x34800000 }.f,
    F32Transmute { u: 0x35000000 }.f,
    F32Transmute { u: 0x35800000 }.f,
    F32Transmute { u: 0x36000000 }.f,
    F32Transmute { u: 0x36800000 }.f,
    F32Transmute { u: 0x37000000 }.f,
    F32Transmute { u: 0x37800000 }.f,
    F32Transmute { u: 0x38000000 }.f,
    F32Transmute { u: 0x38800000 }.f,
    F32Transmute { u: 0x39000000 }.f,
    F32Transmute { u: 0x39800000 }.f,
    F32Transmute { u: 0x3A000000 }.f,
    F32Transmute { u: 0x3A800000 }.f,
    F32Transmute { u: 0x3B000000 }.f,
    F32Transmute { u: 0x3B800000 }.f,
    F32Transmute { u: 0x3C000000 }.f,
    F32Transmute { u: 0x3C800000 }.f,
    F32Transmute { u: 0x3D000000 }.f,
    F32Transmute { u: 0x3D800000 }.f,
    F32Transmute { u: 0x3E000000 }.f,
    F32Transmute { u: 0x3E800000 }.f,
    F32Transmute { u: 0x3F000000 }.f,
    F32Transmute { u: 0x3F800000 }.f,
    F32Transmute { u: 0x40000000 }.f,
    F32Transmute { u: 0x40800000 }.f,
    F32Transmute { u: 0x41000000 }.f,
    F32Transmute { u: 0x41800000 }.f,
    F32Transmute { u: 0x42000000 }.f,
    F32Transmute { u: 0x42800000 }.f,
    F32Transmute { u: 0x43000000 }.f,
    F32Transmute { u: 0x43800000 }.f,
    F32Transmute { u: 0x44000000 }.f,
    F32Transmute { u: 0x44800000 }.f,
    F32Transmute { u: 0x45000000 }.f,
    F32Transmute { u: 0x45800000 }.f,
    F32Transmute { u: 0x46000000 }.f,
    F32Transmute { u: 0x46800000 }.f,
    F32Transmute { u: 0x47000000 }.f,
    F32Transmute { u: 0x47800000 }.f,
    F32Transmute { u: 0x48000000 }.f,
    F32Transmute { u: 0x48800000 }.f,
    F32Transmute { u: 0x49000000 }.f,
    F32Transmute { u: 0x49800000 }.f,
    F32Transmute { u: 0x4A000000 }.f,
    F32Transmute { u: 0x4A800000 }.f,
    F32Transmute { u: 0x4B000000 }.f,
    F32Transmute { u: 0x4B800000 }.f,
    F32Transmute { u: 0x4C000000 }.f,
    F32Transmute { u: 0x4C800000 }.f,
    F32Transmute { u: 0x4D000000 }.f,
    F32Transmute { u: 0x4D800000 }.f,
    F32Transmute { u: 0x4E000000 }.f,
    F32Transmute { u: 0x4E800000 }.f,
    F32Transmute { u: 0x4F000000 }.f,
    F32Transmute { u: 0x4F800000 }.f,
    F32Transmute { u: 0x50000000 }.f,
    F32Transmute { u: 0x50800000 }.f,
    F32Transmute { u: 0x51000000 }.f,
    F32Transmute { u: 0x51800000 }.f,
    F32Transmute { u: 0x52000000 }.f,
    F32Transmute { u: 0x52800000 }.f,
    F32Transmute { u: 0x53000000 }.f,
    F32Transmute { u: 0x53800000 }.f,
    F32Transmute { u: 0x54000000 }.f,
    F32Transmute { u: 0x54800000 }.f,
    F32Transmute { u: 0x55000000 }.f,
    F32Transmute { u: 0x55800000 }.f,
    F32Transmute { u: 0x56000000 }.f,
    F32Transmute { u: 0x56800000 }.f,
    F32Transmute { u: 0x57000000 }.f,
    F32Transmute { u: 0x57800000 }.f,
    F32Transmute { u: 0x58000000 }.f,
    F32Transmute { u: 0x58800000 }.f,
    F32Transmute { u: 0x59000000 }.f,
    F32Transmute { u: 0x59800000 }.f,
    F32Transmute { u: 0x5A000000 }.f,
    F32Transmute { u: 0x5A800000 }.f,
    F32Transmute { u: 0x5B000000 }.f,
    F32Transmute { u: 0x5B800000 }.f,
    F32Transmute { u: 0x5C000000 }.f,
    F32Transmute { u: 0x5C800000 }.f,
    F32Transmute { u: 0x5D000000 }.f,
    F32Transmute { u: 0x5D800000 }.f,
    F32Transmute { u: 0x5E000000 }.f,
    F32Transmute { u: 0x5E800000 }.f,
    F32Transmute { u: 0x5F000000 }.f,
    F32Transmute { u: 0x5F800000 }.f,
    F32Transmute { u: 0x60000000 }.f,
    F32Transmute { u: 0x60800000 }.f,
    F32Transmute { u: 0x61000000 }.f,
    F32Transmute { u: 0x61800000 }.f,
    F32Transmute { u: 0x62000000 }.f,
    F32Transmute { u: 0x62800000 }.f,
    F32Transmute { u: 0x63000000 }.f,
    F32Transmute { u: 0x63800000 }.f,
    F32Transmute { u: 0x64000000 }.f,
    F32Transmute { u: 0x64800000 }.f,
    F32Transmute { u: 0x65000000 }.f,
    F32Transmute { u: 0x65800000 }.f,
    F32Transmute { u: 0x66000000 }.f,
    F32Transmute { u: 0x66800000 }.f,
    F32Transmute { u: 0x67000000 }.f,
    F32Transmute { u: 0x67800000 }.f,
    F32Transmute { u: 0x68000000 }.f,
    F32Transmute { u: 0x68800000 }.f,
    F32Transmute { u: 0x69000000 }.f,
    F32Transmute { u: 0x69800000 }.f,
    F32Transmute { u: 0x6A000000 }.f,
    F32Transmute { u: 0x6A800000 }.f,
    F32Transmute { u: 0x6B000000 }.f,
    F32Transmute { u: 0x6B800000 }.f,
    F32Transmute { u: 0x6C000000 }.f,
    F32Transmute { u: 0x6C800000 }.f,
    F32Transmute { u: 0x6D000000 }.f,
    F32Transmute { u: 0x6D800000 }.f,
    F32Transmute { u: 0x6E000000 }.f,
    F32Transmute { u: 0x6E800000 }.f,
    F32Transmute { u: 0x6F000000 }.f,
    F32Transmute { u: 0x6F800000 }.f,
    F32Transmute { u: 0x70000000 }.f,
    F32Transmute { u: 0x70800000 }.f,
    F32Transmute { u: 0x71000000 }.f,
    F32Transmute { u: 0x71800000 }.f,
    F32Transmute { u: 0x72000000 }.f,
    F32Transmute { u: 0x72800000 }.f,
    F32Transmute { u: 0x73000000 }.f,
    F32Transmute { u: 0x73800000 }.f,
    F32Transmute { u: 0x74000000 }.f,
    F32Transmute { u: 0x74800000 }.f,
    F32Transmute { u: 0x75000000 }.f,
    F32Transmute { u: 0x75800000 }.f,
    F32Transmute { u: 0x76000000 }.f,
    F32Transmute { u: 0x76800000 }.f,
    F32Transmute { u: 0x77000000 }.f,
    F32Transmute { u: 0x77800000 }.f,
    F32Transmute { u: 0x78000000 }.f,
    F32Transmute { u: 0x78800000 }.f,
    F32Transmute { u: 0x79000000 }.f,
    F32Transmute { u: 0x79800000 }.f,
    F32Transmute { u: 0x7A000000 }.f,
    F32Transmute { u: 0x7A800000 }.f,
    F32Transmute { u: 0x7B000000 }.f,
    F32Transmute { u: 0x7B800000 }.f,
    F32Transmute { u: 0x7C000000 }.f,
    F32Transmute { u: 0x7C800000 }.f,
    F32Transmute { u: 0x7D000000 }.f,
    F32Transmute { u: 0x7D800000 }.f,
    F32Transmute { u: 0x7E000000 }.f,
    F32Transmute { u: 0x7E800000 }.f,
    F32Transmute { u: 0x7F000000 }.f,
]};

/// Precalculated values of radix**i for i in range [0, arr.len()-1].
/// Each value can be **exactly** represented as that type.
const F32_POW10: [f32; 11] = [1.0, 10.0, 100.0, 1000.0, 10000.0, 100000.0, 1000000.0, 10000000.0, 100000000.0, 1000000000.0, 10000000000.0];

#[cfg(feature = "radix")]
const F32_POW3: [f32; 16] = [1.0, 3.0, 9.0, 27.0, 81.0, 243.0, 729.0, 2187.0, 6561.0, 19683.0, 59049.0, 177147.0, 531441.0, 1594323.0, 4782969.0, 14348907.0];

#[cfg(feature = "radix")]
const F32_POW5: [f32; 11] = [1.0, 5.0, 25.0, 125.0, 625.0, 3125.0, 15625.0, 78125.0, 390625.0, 1953125.0, 9765625.0];

#[cfg(feature = "radix")]
const F32_POW6: [f32; 16] = [1.0, 6.0, 36.0, 216.0, 1296.0, 7776.0, 46656.0, 279936.0, 1679616.0, 10077696.0, 60466176.0, 362797056.0, 2176782336.0, 13060694016.0, 78364164096.0, 470184984576.0];

#[cfg(feature = "radix")]
const F32_POW7: [f32; 9] = [1.0, 7.0, 49.0, 343.0, 2401.0, 16807.0, 117649.0, 823543.0, 5764801.0];

#[cfg(feature = "radix")]
const F32_POW9: [f32; 8] = [1.0, 9.0, 81.0, 729.0, 6561.0, 59049.0, 531441.0, 4782969.0];

#[cfg(feature = "radix")]
const F32_POW11: [f32; 7] = [1.0, 11.0, 121.0, 1331.0, 14641.0, 161051.0, 1771561.0];

#[cfg(feature = "radix")]
const F32_POW12: [f32; 16] = [1.0, 12.0, 144.0, 1728.0, 20736.0, 248832.0, 2985984.0, 35831808.0, 429981696.0, 5159780352.0, 61917364224.0, 743008370688.0, 8916100448256.0, 106993205379072.0, 1283918464548864.0, 15407021574586368.0];

#[cfg(feature = "radix")]
const F32_POW13: [f32; 7] = [1.0, 13.0, 169.0, 2197.0, 28561.0, 371293.0, 4826809.0];

#[cfg(feature = "radix")]
const F32_POW14: [f32; 9] = [1.0, 14.0, 196.0, 2744.0, 38416.0, 537824.0, 7529536.0, 105413504.0, 1475789056.0];

#[cfg(feature = "radix")]
const F32_POW15: [f32; 7] = [1.0, 15.0, 225.0, 3375.0, 50625.0, 759375.0, 11390625.0];

#[cfg(feature = "radix")]
const F32_POW17: [f32; 6] = [1.0, 17.0, 289.0, 4913.0, 83521.0, 1419857.0];

#[cfg(feature = "radix")]
const F32_POW18: [f32; 8] = [1.0, 18.0, 324.0, 5832.0, 104976.0, 1889568.0, 34012224.0, 612220032.0];

#[cfg(feature = "radix")]
const F32_POW19: [f32; 6] = [1.0, 19.0, 361.0, 6859.0, 130321.0, 2476099.0];

#[cfg(feature = "radix")]
const F32_POW20: [f32; 11] = [1.0, 20.0, 400.0, 8000.0, 160000.0, 3200000.0, 64000000.0, 1280000000.0, 25600000000.0, 512000000000.0, 10240000000000.0];

#[cfg(feature = "radix")]
const F32_POW21: [f32; 6] = [1.0, 21.0, 441.0, 9261.0, 194481.0, 4084101.0];

#[cfg(feature = "radix")]
const F32_POW22: [f32; 7] = [1.0, 22.0, 484.0, 10648.0, 234256.0, 5153632.0, 113379904.0];

#[cfg(feature = "radix")]
const F32_POW23: [f32; 6] = [1.0, 23.0, 529.0, 12167.0, 279841.0, 6436343.0];

#[cfg(feature = "radix")]
const F32_POW24: [f32; 16] = [1.0, 24.0, 576.0, 13824.0, 331776.0, 7962624.0, 191102976.0, 4586471424.0, 110075314176.0, 2641807540224.0, 63403380965376.0, 1521681143169024.0, 36520347436056576.0, 876488338465357824.0, 21035720123168587776.0, 504857282956046106624.0];

#[cfg(feature = "radix")]
const F32_POW25: [f32; 6] = [1.0, 25.0, 625.0, 15625.0, 390625.0, 9765625.0];

#[cfg(feature = "radix")]
const F32_POW26: [f32; 7] = [1.0, 26.0, 676.0, 17576.0, 456976.0, 11881376.0, 308915776.0];

#[cfg(feature = "radix")]
const F32_POW27: [f32; 6] = [1.0, 27.0, 729.0, 19683.0, 531441.0, 14348907.0];

#[cfg(feature = "radix")]
const F32_POW28: [f32; 9] = [1.0, 28.0, 784.0, 21952.0, 614656.0, 17210368.0, 481890304.0, 13492928512.0, 377801998336.0];

#[cfg(feature = "radix")]
const F32_POW29: [f32; 5] = [1.0, 29.0, 841.0, 24389.0, 707281.0];

#[cfg(feature = "radix")]
const F32_POW30: [f32; 7] = [1.0, 30.0, 900.0, 27000.0, 810000.0, 24300000.0, 729000000.0];

#[cfg(feature = "radix")]
const F32_POW31: [f32; 5] = [1.0, 31.0, 961.0, 29791.0, 923521.0];

#[cfg(feature = "radix")]
const F32_POW33: [f32; 5] = [1.0, 33.0, 1089.0, 35937.0, 1185921.0];

#[cfg(feature = "radix")]
const F32_POW34: [f32; 6] = [1.0, 34.0, 1156.0, 39304.0, 1336336.0, 45435424.0];

#[cfg(feature = "radix")]
const F32_POW35: [f32; 5] = [1.0, 35.0, 1225.0, 42875.0, 1500625.0];

#[cfg(feature = "radix")]
const F32_POW36: [f32; 8] = [1.0, 36.0, 1296.0, 46656.0, 1679616.0, 60466176.0, 2176782336.0, 78364164096.0];

// Compile-time guarantees for our tables.
const_assert!(F32_POW10[1] / F32_POW10[0] == 10.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW2[1] / F32_POW2[0] == 2.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW3[1] / F32_POW3[0] == 3.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW5[1] / F32_POW5[0] == 5.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW6[1] / F32_POW6[0] == 6.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW7[1] / F32_POW7[0] == 7.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW9[1] / F32_POW9[0] == 9.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW11[1] / F32_POW11[0] == 11.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW12[1] / F32_POW12[0] == 12.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW13[1] / F32_POW13[0] == 13.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW14[1] / F32_POW14[0] == 14.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW15[1] / F32_POW15[0] == 15.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW17[1] / F32_POW17[0] == 17.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW18[1] / F32_POW18[0] == 18.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW19[1] / F32_POW19[0] == 19.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW20[1] / F32_POW20[0] == 20.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW21[1] / F32_POW21[0] == 21.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW22[1] / F32_POW22[0] == 22.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW23[1] / F32_POW23[0] == 23.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW24[1] / F32_POW24[0] == 24.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW25[1] / F32_POW25[0] == 25.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW26[1] / F32_POW26[0] == 26.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW27[1] / F32_POW27[0] == 27.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW28[1] / F32_POW28[0] == 28.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW29[1] / F32_POW29[0] == 29.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW30[1] / F32_POW30[0] == 30.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW31[1] / F32_POW31[0] == 31.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW33[1] / F32_POW33[0] == 33.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW34[1] / F32_POW34[0] == 34.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW35[1] / F32_POW35[0] == 35.0);

#[cfg(feature = "radix")]
const_assert!(F32_POW36[1] / F32_POW36[0] == 36.0);

impl TablePower for f32 {
    const POW2_EXPONENT_BIAS: i32 = 149;

    #[cfg(feature = "radix")]
    #[inline]
    fn table_pow2(exponent: i32) -> f32 {
        debug_assert!(exponent + Self::POW2_EXPONENT_BIAS >= 0, "table_pow2() have negative exponent.");
        F32_POW2[(exponent + Self::POW2_EXPONENT_BIAS).as_usize()]
    }

    #[inline]
    fn table_pow<T: Integer>(radix: T, exponent: i32) -> f32 {
        debug_assert!(exponent >= 0, "table_pow() have negative exponent.");
        debug_assert_radix!(radix);
        let exponent = exponent as usize;

        #[cfg(not(feature = "radix"))] {
            F32_POW10[exponent]
        }

        #[cfg(feature = "radix")] {
            match radix.as_i32() {
                3  => F32_POW3 [exponent],
                5  => F32_POW5 [exponent],
                6  => F32_POW6 [exponent],
                7  => F32_POW7 [exponent],
                9  => F32_POW9 [exponent],
                10 => F32_POW10[exponent],
                11 => F32_POW11[exponent],
                12 => F32_POW12[exponent],
                13 => F32_POW13[exponent],
                14 => F32_POW14[exponent],
                15 => F32_POW15[exponent],
                17 => F32_POW17[exponent],
                18 => F32_POW18[exponent],
                19 => F32_POW19[exponent],
                20 => F32_POW20[exponent],
                21 => F32_POW21[exponent],
                22 => F32_POW22[exponent],
                23 => F32_POW23[exponent],
                24 => F32_POW24[exponent],
                25 => F32_POW25[exponent],
                26 => F32_POW26[exponent],
                27 => F32_POW27[exponent],
                28 => F32_POW28[exponent],
                29 => F32_POW29[exponent],
                30 => F32_POW30[exponent],
                31 => F32_POW31[exponent],
                33 => F32_POW33[exponent],
                34 => F32_POW34[exponent],
                35 => F32_POW35[exponent],
                36 => F32_POW36[exponent],
                // Invalid radix
                _  => unreachable!(),
            }
        }
    }
}

// F64

/// Hacky transmute for 64-bit values.
#[cfg(feature = "radix")]
type F64Transmute = Transmute<u64, f64>;

/// Precalculated values of 2**i for i in range [-1074, 1023].
#[cfg(feature = "radix")]
#[allow(const_err)]     // This warning is fixed in 1.26.0
const F64_POW2: [f64; 2098] = unsafe {[
    // Denormal floats ([-1074, -1023])
    F64Transmute { u: 0x0000000000000001 }.f,
    F64Transmute { u: 0x0000000000000002 }.f,
    F64Transmute { u: 0x0000000000000004 }.f,
    F64Transmute { u: 0x0000000000000008 }.f,
    F64Transmute { u: 0x0000000000000010 }.f,
    F64Transmute { u: 0x0000000000000020 }.f,
    F64Transmute { u: 0x0000000000000040 }.f,
    F64Transmute { u: 0x0000000000000080 }.f,
    F64Transmute { u: 0x0000000000000100 }.f,
    F64Transmute { u: 0x0000000000000200 }.f,
    F64Transmute { u: 0x0000000000000400 }.f,
    F64Transmute { u: 0x0000000000000800 }.f,
    F64Transmute { u: 0x0000000000001000 }.f,
    F64Transmute { u: 0x0000000000002000 }.f,
    F64Transmute { u: 0x0000000000004000 }.f,
    F64Transmute { u: 0x0000000000008000 }.f,
    F64Transmute { u: 0x0000000000010000 }.f,
    F64Transmute { u: 0x0000000000020000 }.f,
    F64Transmute { u: 0x0000000000040000 }.f,
    F64Transmute { u: 0x0000000000080000 }.f,
    F64Transmute { u: 0x0000000000100000 }.f,
    F64Transmute { u: 0x0000000000200000 }.f,
    F64Transmute { u: 0x0000000000400000 }.f,
    F64Transmute { u: 0x0000000000800000 }.f,
    F64Transmute { u: 0x0000000001000000 }.f,
    F64Transmute { u: 0x0000000002000000 }.f,
    F64Transmute { u: 0x0000000004000000 }.f,
    F64Transmute { u: 0x0000000008000000 }.f,
    F64Transmute { u: 0x0000000010000000 }.f,
    F64Transmute { u: 0x0000000020000000 }.f,
    F64Transmute { u: 0x0000000040000000 }.f,
    F64Transmute { u: 0x0000000080000000 }.f,
    F64Transmute { u: 0x0000000100000000 }.f,
    F64Transmute { u: 0x0000000200000000 }.f,
    F64Transmute { u: 0x0000000400000000 }.f,
    F64Transmute { u: 0x0000000800000000 }.f,
    F64Transmute { u: 0x0000001000000000 }.f,
    F64Transmute { u: 0x0000002000000000 }.f,
    F64Transmute { u: 0x0000004000000000 }.f,
    F64Transmute { u: 0x0000008000000000 }.f,
    F64Transmute { u: 0x0000010000000000 }.f,
    F64Transmute { u: 0x0000020000000000 }.f,
    F64Transmute { u: 0x0000040000000000 }.f,
    F64Transmute { u: 0x0000080000000000 }.f,
    F64Transmute { u: 0x0000100000000000 }.f,
    F64Transmute { u: 0x0000200000000000 }.f,
    F64Transmute { u: 0x0000400000000000 }.f,
    F64Transmute { u: 0x0000800000000000 }.f,
    F64Transmute { u: 0x0001000000000000 }.f,
    F64Transmute { u: 0x0002000000000000 }.f,
    F64Transmute { u: 0x0004000000000000 }.f,
    F64Transmute { u: 0x0008000000000000 }.f,

    // Regular floats (exponent only, [-1022, 1023])
    F64Transmute { u: 0x0010000000000000 }.f,
    F64Transmute { u: 0x0020000000000000 }.f,
    F64Transmute { u: 0x0030000000000000 }.f,
    F64Transmute { u: 0x0040000000000000 }.f,
    F64Transmute { u: 0x0050000000000000 }.f,
    F64Transmute { u: 0x0060000000000000 }.f,
    F64Transmute { u: 0x0070000000000000 }.f,
    F64Transmute { u: 0x0080000000000000 }.f,
    F64Transmute { u: 0x0090000000000000 }.f,
    F64Transmute { u: 0x00A0000000000000 }.f,
    F64Transmute { u: 0x00B0000000000000 }.f,
    F64Transmute { u: 0x00C0000000000000 }.f,
    F64Transmute { u: 0x00D0000000000000 }.f,
    F64Transmute { u: 0x00E0000000000000 }.f,
    F64Transmute { u: 0x00F0000000000000 }.f,
    F64Transmute { u: 0x0100000000000000 }.f,
    F64Transmute { u: 0x0110000000000000 }.f,
    F64Transmute { u: 0x0120000000000000 }.f,
    F64Transmute { u: 0x0130000000000000 }.f,
    F64Transmute { u: 0x0140000000000000 }.f,
    F64Transmute { u: 0x0150000000000000 }.f,
    F64Transmute { u: 0x0160000000000000 }.f,
    F64Transmute { u: 0x0170000000000000 }.f,
    F64Transmute { u: 0x0180000000000000 }.f,
    F64Transmute { u: 0x0190000000000000 }.f,
    F64Transmute { u: 0x01A0000000000000 }.f,
    F64Transmute { u: 0x01B0000000000000 }.f,
    F64Transmute { u: 0x01C0000000000000 }.f,
    F64Transmute { u: 0x01D0000000000000 }.f,
    F64Transmute { u: 0x01E0000000000000 }.f,
    F64Transmute { u: 0x01F0000000000000 }.f,
    F64Transmute { u: 0x0200000000000000 }.f,
    F64Transmute { u: 0x0210000000000000 }.f,
    F64Transmute { u: 0x0220000000000000 }.f,
    F64Transmute { u: 0x0230000000000000 }.f,
    F64Transmute { u: 0x0240000000000000 }.f,
    F64Transmute { u: 0x0250000000000000 }.f,
    F64Transmute { u: 0x0260000000000000 }.f,
    F64Transmute { u: 0x0270000000000000 }.f,
    F64Transmute { u: 0x0280000000000000 }.f,
    F64Transmute { u: 0x0290000000000000 }.f,
    F64Transmute { u: 0x02A0000000000000 }.f,
    F64Transmute { u: 0x02B0000000000000 }.f,
    F64Transmute { u: 0x02C0000000000000 }.f,
    F64Transmute { u: 0x02D0000000000000 }.f,
    F64Transmute { u: 0x02E0000000000000 }.f,
    F64Transmute { u: 0x02F0000000000000 }.f,
    F64Transmute { u: 0x0300000000000000 }.f,
    F64Transmute { u: 0x0310000000000000 }.f,
    F64Transmute { u: 0x0320000000000000 }.f,
    F64Transmute { u: 0x0330000000000000 }.f,
    F64Transmute { u: 0x0340000000000000 }.f,
    F64Transmute { u: 0x0350000000000000 }.f,
    F64Transmute { u: 0x0360000000000000 }.f,
    F64Transmute { u: 0x0370000000000000 }.f,
    F64Transmute { u: 0x0380000000000000 }.f,
    F64Transmute { u: 0x0390000000000000 }.f,
    F64Transmute { u: 0x03A0000000000000 }.f,
    F64Transmute { u: 0x03B0000000000000 }.f,
    F64Transmute { u: 0x03C0000000000000 }.f,
    F64Transmute { u: 0x03D0000000000000 }.f,
    F64Transmute { u: 0x03E0000000000000 }.f,
    F64Transmute { u: 0x03F0000000000000 }.f,
    F64Transmute { u: 0x0400000000000000 }.f,
    F64Transmute { u: 0x0410000000000000 }.f,
    F64Transmute { u: 0x0420000000000000 }.f,
    F64Transmute { u: 0x0430000000000000 }.f,
    F64Transmute { u: 0x0440000000000000 }.f,
    F64Transmute { u: 0x0450000000000000 }.f,
    F64Transmute { u: 0x0460000000000000 }.f,
    F64Transmute { u: 0x0470000000000000 }.f,
    F64Transmute { u: 0x0480000000000000 }.f,
    F64Transmute { u: 0x0490000000000000 }.f,
    F64Transmute { u: 0x04A0000000000000 }.f,
    F64Transmute { u: 0x04B0000000000000 }.f,
    F64Transmute { u: 0x04C0000000000000 }.f,
    F64Transmute { u: 0x04D0000000000000 }.f,
    F64Transmute { u: 0x04E0000000000000 }.f,
    F64Transmute { u: 0x04F0000000000000 }.f,
    F64Transmute { u: 0x0500000000000000 }.f,
    F64Transmute { u: 0x0510000000000000 }.f,
    F64Transmute { u: 0x0520000000000000 }.f,
    F64Transmute { u: 0x0530000000000000 }.f,
    F64Transmute { u: 0x0540000000000000 }.f,
    F64Transmute { u: 0x0550000000000000 }.f,
    F64Transmute { u: 0x0560000000000000 }.f,
    F64Transmute { u: 0x0570000000000000 }.f,
    F64Transmute { u: 0x0580000000000000 }.f,
    F64Transmute { u: 0x0590000000000000 }.f,
    F64Transmute { u: 0x05A0000000000000 }.f,
    F64Transmute { u: 0x05B0000000000000 }.f,
    F64Transmute { u: 0x05C0000000000000 }.f,
    F64Transmute { u: 0x05D0000000000000 }.f,
    F64Transmute { u: 0x05E0000000000000 }.f,
    F64Transmute { u: 0x05F0000000000000 }.f,
    F64Transmute { u: 0x0600000000000000 }.f,
    F64Transmute { u: 0x0610000000000000 }.f,
    F64Transmute { u: 0x0620000000000000 }.f,
    F64Transmute { u: 0x0630000000000000 }.f,
    F64Transmute { u: 0x0640000000000000 }.f,
    F64Transmute { u: 0x0650000000000000 }.f,
    F64Transmute { u: 0x0660000000000000 }.f,
    F64Transmute { u: 0x0670000000000000 }.f,
    F64Transmute { u: 0x0680000000000000 }.f,
    F64Transmute { u: 0x0690000000000000 }.f,
    F64Transmute { u: 0x06A0000000000000 }.f,
    F64Transmute { u: 0x06B0000000000000 }.f,
    F64Transmute { u: 0x06C0000000000000 }.f,
    F64Transmute { u: 0x06D0000000000000 }.f,
    F64Transmute { u: 0x06E0000000000000 }.f,
    F64Transmute { u: 0x06F0000000000000 }.f,
    F64Transmute { u: 0x0700000000000000 }.f,
    F64Transmute { u: 0x0710000000000000 }.f,
    F64Transmute { u: 0x0720000000000000 }.f,
    F64Transmute { u: 0x0730000000000000 }.f,
    F64Transmute { u: 0x0740000000000000 }.f,
    F64Transmute { u: 0x0750000000000000 }.f,
    F64Transmute { u: 0x0760000000000000 }.f,
    F64Transmute { u: 0x0770000000000000 }.f,
    F64Transmute { u: 0x0780000000000000 }.f,
    F64Transmute { u: 0x0790000000000000 }.f,
    F64Transmute { u: 0x07A0000000000000 }.f,
    F64Transmute { u: 0x07B0000000000000 }.f,
    F64Transmute { u: 0x07C0000000000000 }.f,
    F64Transmute { u: 0x07D0000000000000 }.f,
    F64Transmute { u: 0x07E0000000000000 }.f,
    F64Transmute { u: 0x07F0000000000000 }.f,
    F64Transmute { u: 0x0800000000000000 }.f,
    F64Transmute { u: 0x0810000000000000 }.f,
    F64Transmute { u: 0x0820000000000000 }.f,
    F64Transmute { u: 0x0830000000000000 }.f,
    F64Transmute { u: 0x0840000000000000 }.f,
    F64Transmute { u: 0x0850000000000000 }.f,
    F64Transmute { u: 0x0860000000000000 }.f,
    F64Transmute { u: 0x0870000000000000 }.f,
    F64Transmute { u: 0x0880000000000000 }.f,
    F64Transmute { u: 0x0890000000000000 }.f,
    F64Transmute { u: 0x08A0000000000000 }.f,
    F64Transmute { u: 0x08B0000000000000 }.f,
    F64Transmute { u: 0x08C0000000000000 }.f,
    F64Transmute { u: 0x08D0000000000000 }.f,
    F64Transmute { u: 0x08E0000000000000 }.f,
    F64Transmute { u: 0x08F0000000000000 }.f,
    F64Transmute { u: 0x0900000000000000 }.f,
    F64Transmute { u: 0x0910000000000000 }.f,
    F64Transmute { u: 0x0920000000000000 }.f,
    F64Transmute { u: 0x0930000000000000 }.f,
    F64Transmute { u: 0x0940000000000000 }.f,
    F64Transmute { u: 0x0950000000000000 }.f,
    F64Transmute { u: 0x0960000000000000 }.f,
    F64Transmute { u: 0x0970000000000000 }.f,
    F64Transmute { u: 0x0980000000000000 }.f,
    F64Transmute { u: 0x0990000000000000 }.f,
    F64Transmute { u: 0x09A0000000000000 }.f,
    F64Transmute { u: 0x09B0000000000000 }.f,
    F64Transmute { u: 0x09C0000000000000 }.f,
    F64Transmute { u: 0x09D0000000000000 }.f,
    F64Transmute { u: 0x09E0000000000000 }.f,
    F64Transmute { u: 0x09F0000000000000 }.f,
    F64Transmute { u: 0x0A00000000000000 }.f,
    F64Transmute { u: 0x0A10000000000000 }.f,
    F64Transmute { u: 0x0A20000000000000 }.f,
    F64Transmute { u: 0x0A30000000000000 }.f,
    F64Transmute { u: 0x0A40000000000000 }.f,
    F64Transmute { u: 0x0A50000000000000 }.f,
    F64Transmute { u: 0x0A60000000000000 }.f,
    F64Transmute { u: 0x0A70000000000000 }.f,
    F64Transmute { u: 0x0A80000000000000 }.f,
    F64Transmute { u: 0x0A90000000000000 }.f,
    F64Transmute { u: 0x0AA0000000000000 }.f,
    F64Transmute { u: 0x0AB0000000000000 }.f,
    F64Transmute { u: 0x0AC0000000000000 }.f,
    F64Transmute { u: 0x0AD0000000000000 }.f,
    F64Transmute { u: 0x0AE0000000000000 }.f,
    F64Transmute { u: 0x0AF0000000000000 }.f,
    F64Transmute { u: 0x0B00000000000000 }.f,
    F64Transmute { u: 0x0B10000000000000 }.f,
    F64Transmute { u: 0x0B20000000000000 }.f,
    F64Transmute { u: 0x0B30000000000000 }.f,
    F64Transmute { u: 0x0B40000000000000 }.f,
    F64Transmute { u: 0x0B50000000000000 }.f,
    F64Transmute { u: 0x0B60000000000000 }.f,
    F64Transmute { u: 0x0B70000000000000 }.f,
    F64Transmute { u: 0x0B80000000000000 }.f,
    F64Transmute { u: 0x0B90000000000000 }.f,
    F64Transmute { u: 0x0BA0000000000000 }.f,
    F64Transmute { u: 0x0BB0000000000000 }.f,
    F64Transmute { u: 0x0BC0000000000000 }.f,
    F64Transmute { u: 0x0BD0000000000000 }.f,
    F64Transmute { u: 0x0BE0000000000000 }.f,
    F64Transmute { u: 0x0BF0000000000000 }.f,
    F64Transmute { u: 0x0C00000000000000 }.f,
    F64Transmute { u: 0x0C10000000000000 }.f,
    F64Transmute { u: 0x0C20000000000000 }.f,
    F64Transmute { u: 0x0C30000000000000 }.f,
    F64Transmute { u: 0x0C40000000000000 }.f,
    F64Transmute { u: 0x0C50000000000000 }.f,
    F64Transmute { u: 0x0C60000000000000 }.f,
    F64Transmute { u: 0x0C70000000000000 }.f,
    F64Transmute { u: 0x0C80000000000000 }.f,
    F64Transmute { u: 0x0C90000000000000 }.f,
    F64Transmute { u: 0x0CA0000000000000 }.f,
    F64Transmute { u: 0x0CB0000000000000 }.f,
    F64Transmute { u: 0x0CC0000000000000 }.f,
    F64Transmute { u: 0x0CD0000000000000 }.f,
    F64Transmute { u: 0x0CE0000000000000 }.f,
    F64Transmute { u: 0x0CF0000000000000 }.f,
    F64Transmute { u: 0x0D00000000000000 }.f,
    F64Transmute { u: 0x0D10000000000000 }.f,
    F64Transmute { u: 0x0D20000000000000 }.f,
    F64Transmute { u: 0x0D30000000000000 }.f,
    F64Transmute { u: 0x0D40000000000000 }.f,
    F64Transmute { u: 0x0D50000000000000 }.f,
    F64Transmute { u: 0x0D60000000000000 }.f,
    F64Transmute { u: 0x0D70000000000000 }.f,
    F64Transmute { u: 0x0D80000000000000 }.f,
    F64Transmute { u: 0x0D90000000000000 }.f,
    F64Transmute { u: 0x0DA0000000000000 }.f,
    F64Transmute { u: 0x0DB0000000000000 }.f,
    F64Transmute { u: 0x0DC0000000000000 }.f,
    F64Transmute { u: 0x0DD0000000000000 }.f,
    F64Transmute { u: 0x0DE0000000000000 }.f,
    F64Transmute { u: 0x0DF0000000000000 }.f,
    F64Transmute { u: 0x0E00000000000000 }.f,
    F64Transmute { u: 0x0E10000000000000 }.f,
    F64Transmute { u: 0x0E20000000000000 }.f,
    F64Transmute { u: 0x0E30000000000000 }.f,
    F64Transmute { u: 0x0E40000000000000 }.f,
    F64Transmute { u: 0x0E50000000000000 }.f,
    F64Transmute { u: 0x0E60000000000000 }.f,
    F64Transmute { u: 0x0E70000000000000 }.f,
    F64Transmute { u: 0x0E80000000000000 }.f,
    F64Transmute { u: 0x0E90000000000000 }.f,
    F64Transmute { u: 0x0EA0000000000000 }.f,
    F64Transmute { u: 0x0EB0000000000000 }.f,
    F64Transmute { u: 0x0EC0000000000000 }.f,
    F64Transmute { u: 0x0ED0000000000000 }.f,
    F64Transmute { u: 0x0EE0000000000000 }.f,
    F64Transmute { u: 0x0EF0000000000000 }.f,
    F64Transmute { u: 0x0F00000000000000 }.f,
    F64Transmute { u: 0x0F10000000000000 }.f,
    F64Transmute { u: 0x0F20000000000000 }.f,
    F64Transmute { u: 0x0F30000000000000 }.f,
    F64Transmute { u: 0x0F40000000000000 }.f,
    F64Transmute { u: 0x0F50000000000000 }.f,
    F64Transmute { u: 0x0F60000000000000 }.f,
    F64Transmute { u: 0x0F70000000000000 }.f,
    F64Transmute { u: 0x0F80000000000000 }.f,
    F64Transmute { u: 0x0F90000000000000 }.f,
    F64Transmute { u: 0x0FA0000000000000 }.f,
    F64Transmute { u: 0x0FB0000000000000 }.f,
    F64Transmute { u: 0x0FC0000000000000 }.f,
    F64Transmute { u: 0x0FD0000000000000 }.f,
    F64Transmute { u: 0x0FE0000000000000 }.f,
    F64Transmute { u: 0x0FF0000000000000 }.f,
    F64Transmute { u: 0x1000000000000000 }.f,
    F64Transmute { u: 0x1010000000000000 }.f,
    F64Transmute { u: 0x1020000000000000 }.f,
    F64Transmute { u: 0x1030000000000000 }.f,
    F64Transmute { u: 0x1040000000000000 }.f,
    F64Transmute { u: 0x1050000000000000 }.f,
    F64Transmute { u: 0x1060000000000000 }.f,
    F64Transmute { u: 0x1070000000000000 }.f,
    F64Transmute { u: 0x1080000000000000 }.f,
    F64Transmute { u: 0x1090000000000000 }.f,
    F64Transmute { u: 0x10A0000000000000 }.f,
    F64Transmute { u: 0x10B0000000000000 }.f,
    F64Transmute { u: 0x10C0000000000000 }.f,
    F64Transmute { u: 0x10D0000000000000 }.f,
    F64Transmute { u: 0x10E0000000000000 }.f,
    F64Transmute { u: 0x10F0000000000000 }.f,
    F64Transmute { u: 0x1100000000000000 }.f,
    F64Transmute { u: 0x1110000000000000 }.f,
    F64Transmute { u: 0x1120000000000000 }.f,
    F64Transmute { u: 0x1130000000000000 }.f,
    F64Transmute { u: 0x1140000000000000 }.f,
    F64Transmute { u: 0x1150000000000000 }.f,
    F64Transmute { u: 0x1160000000000000 }.f,
    F64Transmute { u: 0x1170000000000000 }.f,
    F64Transmute { u: 0x1180000000000000 }.f,
    F64Transmute { u: 0x1190000000000000 }.f,
    F64Transmute { u: 0x11A0000000000000 }.f,
    F64Transmute { u: 0x11B0000000000000 }.f,
    F64Transmute { u: 0x11C0000000000000 }.f,
    F64Transmute { u: 0x11D0000000000000 }.f,
    F64Transmute { u: 0x11E0000000000000 }.f,
    F64Transmute { u: 0x11F0000000000000 }.f,
    F64Transmute { u: 0x1200000000000000 }.f,
    F64Transmute { u: 0x1210000000000000 }.f,
    F64Transmute { u: 0x1220000000000000 }.f,
    F64Transmute { u: 0x1230000000000000 }.f,
    F64Transmute { u: 0x1240000000000000 }.f,
    F64Transmute { u: 0x1250000000000000 }.f,
    F64Transmute { u: 0x1260000000000000 }.f,
    F64Transmute { u: 0x1270000000000000 }.f,
    F64Transmute { u: 0x1280000000000000 }.f,
    F64Transmute { u: 0x1290000000000000 }.f,
    F64Transmute { u: 0x12A0000000000000 }.f,
    F64Transmute { u: 0x12B0000000000000 }.f,
    F64Transmute { u: 0x12C0000000000000 }.f,
    F64Transmute { u: 0x12D0000000000000 }.f,
    F64Transmute { u: 0x12E0000000000000 }.f,
    F64Transmute { u: 0x12F0000000000000 }.f,
    F64Transmute { u: 0x1300000000000000 }.f,
    F64Transmute { u: 0x1310000000000000 }.f,
    F64Transmute { u: 0x1320000000000000 }.f,
    F64Transmute { u: 0x1330000000000000 }.f,
    F64Transmute { u: 0x1340000000000000 }.f,
    F64Transmute { u: 0x1350000000000000 }.f,
    F64Transmute { u: 0x1360000000000000 }.f,
    F64Transmute { u: 0x1370000000000000 }.f,
    F64Transmute { u: 0x1380000000000000 }.f,
    F64Transmute { u: 0x1390000000000000 }.f,
    F64Transmute { u: 0x13A0000000000000 }.f,
    F64Transmute { u: 0x13B0000000000000 }.f,
    F64Transmute { u: 0x13C0000000000000 }.f,
    F64Transmute { u: 0x13D0000000000000 }.f,
    F64Transmute { u: 0x13E0000000000000 }.f,
    F64Transmute { u: 0x13F0000000000000 }.f,
    F64Transmute { u: 0x1400000000000000 }.f,
    F64Transmute { u: 0x1410000000000000 }.f,
    F64Transmute { u: 0x1420000000000000 }.f,
    F64Transmute { u: 0x1430000000000000 }.f,
    F64Transmute { u: 0x1440000000000000 }.f,
    F64Transmute { u: 0x1450000000000000 }.f,
    F64Transmute { u: 0x1460000000000000 }.f,
    F64Transmute { u: 0x1470000000000000 }.f,
    F64Transmute { u: 0x1480000000000000 }.f,
    F64Transmute { u: 0x1490000000000000 }.f,
    F64Transmute { u: 0x14A0000000000000 }.f,
    F64Transmute { u: 0x14B0000000000000 }.f,
    F64Transmute { u: 0x14C0000000000000 }.f,
    F64Transmute { u: 0x14D0000000000000 }.f,
    F64Transmute { u: 0x14E0000000000000 }.f,
    F64Transmute { u: 0x14F0000000000000 }.f,
    F64Transmute { u: 0x1500000000000000 }.f,
    F64Transmute { u: 0x1510000000000000 }.f,
    F64Transmute { u: 0x1520000000000000 }.f,
    F64Transmute { u: 0x1530000000000000 }.f,
    F64Transmute { u: 0x1540000000000000 }.f,
    F64Transmute { u: 0x1550000000000000 }.f,
    F64Transmute { u: 0x1560000000000000 }.f,
    F64Transmute { u: 0x1570000000000000 }.f,
    F64Transmute { u: 0x1580000000000000 }.f,
    F64Transmute { u: 0x1590000000000000 }.f,
    F64Transmute { u: 0x15A0000000000000 }.f,
    F64Transmute { u: 0x15B0000000000000 }.f,
    F64Transmute { u: 0x15C0000000000000 }.f,
    F64Transmute { u: 0x15D0000000000000 }.f,
    F64Transmute { u: 0x15E0000000000000 }.f,
    F64Transmute { u: 0x15F0000000000000 }.f,
    F64Transmute { u: 0x1600000000000000 }.f,
    F64Transmute { u: 0x1610000000000000 }.f,
    F64Transmute { u: 0x1620000000000000 }.f,
    F64Transmute { u: 0x1630000000000000 }.f,
    F64Transmute { u: 0x1640000000000000 }.f,
    F64Transmute { u: 0x1650000000000000 }.f,
    F64Transmute { u: 0x1660000000000000 }.f,
    F64Transmute { u: 0x1670000000000000 }.f,
    F64Transmute { u: 0x1680000000000000 }.f,
    F64Transmute { u: 0x1690000000000000 }.f,
    F64Transmute { u: 0x16A0000000000000 }.f,
    F64Transmute { u: 0x16B0000000000000 }.f,
    F64Transmute { u: 0x16C0000000000000 }.f,
    F64Transmute { u: 0x16D0000000000000 }.f,
    F64Transmute { u: 0x16E0000000000000 }.f,
    F64Transmute { u: 0x16F0000000000000 }.f,
    F64Transmute { u: 0x1700000000000000 }.f,
    F64Transmute { u: 0x1710000000000000 }.f,
    F64Transmute { u: 0x1720000000000000 }.f,
    F64Transmute { u: 0x1730000000000000 }.f,
    F64Transmute { u: 0x1740000000000000 }.f,
    F64Transmute { u: 0x1750000000000000 }.f,
    F64Transmute { u: 0x1760000000000000 }.f,
    F64Transmute { u: 0x1770000000000000 }.f,
    F64Transmute { u: 0x1780000000000000 }.f,
    F64Transmute { u: 0x1790000000000000 }.f,
    F64Transmute { u: 0x17A0000000000000 }.f,
    F64Transmute { u: 0x17B0000000000000 }.f,
    F64Transmute { u: 0x17C0000000000000 }.f,
    F64Transmute { u: 0x17D0000000000000 }.f,
    F64Transmute { u: 0x17E0000000000000 }.f,
    F64Transmute { u: 0x17F0000000000000 }.f,
    F64Transmute { u: 0x1800000000000000 }.f,
    F64Transmute { u: 0x1810000000000000 }.f,
    F64Transmute { u: 0x1820000000000000 }.f,
    F64Transmute { u: 0x1830000000000000 }.f,
    F64Transmute { u: 0x1840000000000000 }.f,
    F64Transmute { u: 0x1850000000000000 }.f,
    F64Transmute { u: 0x1860000000000000 }.f,
    F64Transmute { u: 0x1870000000000000 }.f,
    F64Transmute { u: 0x1880000000000000 }.f,
    F64Transmute { u: 0x1890000000000000 }.f,
    F64Transmute { u: 0x18A0000000000000 }.f,
    F64Transmute { u: 0x18B0000000000000 }.f,
    F64Transmute { u: 0x18C0000000000000 }.f,
    F64Transmute { u: 0x18D0000000000000 }.f,
    F64Transmute { u: 0x18E0000000000000 }.f,
    F64Transmute { u: 0x18F0000000000000 }.f,
    F64Transmute { u: 0x1900000000000000 }.f,
    F64Transmute { u: 0x1910000000000000 }.f,
    F64Transmute { u: 0x1920000000000000 }.f,
    F64Transmute { u: 0x1930000000000000 }.f,
    F64Transmute { u: 0x1940000000000000 }.f,
    F64Transmute { u: 0x1950000000000000 }.f,
    F64Transmute { u: 0x1960000000000000 }.f,
    F64Transmute { u: 0x1970000000000000 }.f,
    F64Transmute { u: 0x1980000000000000 }.f,
    F64Transmute { u: 0x1990000000000000 }.f,
    F64Transmute { u: 0x19A0000000000000 }.f,
    F64Transmute { u: 0x19B0000000000000 }.f,
    F64Transmute { u: 0x19C0000000000000 }.f,
    F64Transmute { u: 0x19D0000000000000 }.f,
    F64Transmute { u: 0x19E0000000000000 }.f,
    F64Transmute { u: 0x19F0000000000000 }.f,
    F64Transmute { u: 0x1A00000000000000 }.f,
    F64Transmute { u: 0x1A10000000000000 }.f,
    F64Transmute { u: 0x1A20000000000000 }.f,
    F64Transmute { u: 0x1A30000000000000 }.f,
    F64Transmute { u: 0x1A40000000000000 }.f,
    F64Transmute { u: 0x1A50000000000000 }.f,
    F64Transmute { u: 0x1A60000000000000 }.f,
    F64Transmute { u: 0x1A70000000000000 }.f,
    F64Transmute { u: 0x1A80000000000000 }.f,
    F64Transmute { u: 0x1A90000000000000 }.f,
    F64Transmute { u: 0x1AA0000000000000 }.f,
    F64Transmute { u: 0x1AB0000000000000 }.f,
    F64Transmute { u: 0x1AC0000000000000 }.f,
    F64Transmute { u: 0x1AD0000000000000 }.f,
    F64Transmute { u: 0x1AE0000000000000 }.f,
    F64Transmute { u: 0x1AF0000000000000 }.f,
    F64Transmute { u: 0x1B00000000000000 }.f,
    F64Transmute { u: 0x1B10000000000000 }.f,
    F64Transmute { u: 0x1B20000000000000 }.f,
    F64Transmute { u: 0x1B30000000000000 }.f,
    F64Transmute { u: 0x1B40000000000000 }.f,
    F64Transmute { u: 0x1B50000000000000 }.f,
    F64Transmute { u: 0x1B60000000000000 }.f,
    F64Transmute { u: 0x1B70000000000000 }.f,
    F64Transmute { u: 0x1B80000000000000 }.f,
    F64Transmute { u: 0x1B90000000000000 }.f,
    F64Transmute { u: 0x1BA0000000000000 }.f,
    F64Transmute { u: 0x1BB0000000000000 }.f,
    F64Transmute { u: 0x1BC0000000000000 }.f,
    F64Transmute { u: 0x1BD0000000000000 }.f,
    F64Transmute { u: 0x1BE0000000000000 }.f,
    F64Transmute { u: 0x1BF0000000000000 }.f,
    F64Transmute { u: 0x1C00000000000000 }.f,
    F64Transmute { u: 0x1C10000000000000 }.f,
    F64Transmute { u: 0x1C20000000000000 }.f,
    F64Transmute { u: 0x1C30000000000000 }.f,
    F64Transmute { u: 0x1C40000000000000 }.f,
    F64Transmute { u: 0x1C50000000000000 }.f,
    F64Transmute { u: 0x1C60000000000000 }.f,
    F64Transmute { u: 0x1C70000000000000 }.f,
    F64Transmute { u: 0x1C80000000000000 }.f,
    F64Transmute { u: 0x1C90000000000000 }.f,
    F64Transmute { u: 0x1CA0000000000000 }.f,
    F64Transmute { u: 0x1CB0000000000000 }.f,
    F64Transmute { u: 0x1CC0000000000000 }.f,
    F64Transmute { u: 0x1CD0000000000000 }.f,
    F64Transmute { u: 0x1CE0000000000000 }.f,
    F64Transmute { u: 0x1CF0000000000000 }.f,
    F64Transmute { u: 0x1D00000000000000 }.f,
    F64Transmute { u: 0x1D10000000000000 }.f,
    F64Transmute { u: 0x1D20000000000000 }.f,
    F64Transmute { u: 0x1D30000000000000 }.f,
    F64Transmute { u: 0x1D40000000000000 }.f,
    F64Transmute { u: 0x1D50000000000000 }.f,
    F64Transmute { u: 0x1D60000000000000 }.f,
    F64Transmute { u: 0x1D70000000000000 }.f,
    F64Transmute { u: 0x1D80000000000000 }.f,
    F64Transmute { u: 0x1D90000000000000 }.f,
    F64Transmute { u: 0x1DA0000000000000 }.f,
    F64Transmute { u: 0x1DB0000000000000 }.f,
    F64Transmute { u: 0x1DC0000000000000 }.f,
    F64Transmute { u: 0x1DD0000000000000 }.f,
    F64Transmute { u: 0x1DE0000000000000 }.f,
    F64Transmute { u: 0x1DF0000000000000 }.f,
    F64Transmute { u: 0x1E00000000000000 }.f,
    F64Transmute { u: 0x1E10000000000000 }.f,
    F64Transmute { u: 0x1E20000000000000 }.f,
    F64Transmute { u: 0x1E30000000000000 }.f,
    F64Transmute { u: 0x1E40000000000000 }.f,
    F64Transmute { u: 0x1E50000000000000 }.f,
    F64Transmute { u: 0x1E60000000000000 }.f,
    F64Transmute { u: 0x1E70000000000000 }.f,
    F64Transmute { u: 0x1E80000000000000 }.f,
    F64Transmute { u: 0x1E90000000000000 }.f,
    F64Transmute { u: 0x1EA0000000000000 }.f,
    F64Transmute { u: 0x1EB0000000000000 }.f,
    F64Transmute { u: 0x1EC0000000000000 }.f,
    F64Transmute { u: 0x1ED0000000000000 }.f,
    F64Transmute { u: 0x1EE0000000000000 }.f,
    F64Transmute { u: 0x1EF0000000000000 }.f,
    F64Transmute { u: 0x1F00000000000000 }.f,
    F64Transmute { u: 0x1F10000000000000 }.f,
    F64Transmute { u: 0x1F20000000000000 }.f,
    F64Transmute { u: 0x1F30000000000000 }.f,
    F64Transmute { u: 0x1F40000000000000 }.f,
    F64Transmute { u: 0x1F50000000000000 }.f,
    F64Transmute { u: 0x1F60000000000000 }.f,
    F64Transmute { u: 0x1F70000000000000 }.f,
    F64Transmute { u: 0x1F80000000000000 }.f,
    F64Transmute { u: 0x1F90000000000000 }.f,
    F64Transmute { u: 0x1FA0000000000000 }.f,
    F64Transmute { u: 0x1FB0000000000000 }.f,
    F64Transmute { u: 0x1FC0000000000000 }.f,
    F64Transmute { u: 0x1FD0000000000000 }.f,
    F64Transmute { u: 0x1FE0000000000000 }.f,
    F64Transmute { u: 0x1FF0000000000000 }.f,
    F64Transmute { u: 0x2000000000000000 }.f,
    F64Transmute { u: 0x2010000000000000 }.f,
    F64Transmute { u: 0x2020000000000000 }.f,
    F64Transmute { u: 0x2030000000000000 }.f,
    F64Transmute { u: 0x2040000000000000 }.f,
    F64Transmute { u: 0x2050000000000000 }.f,
    F64Transmute { u: 0x2060000000000000 }.f,
    F64Transmute { u: 0x2070000000000000 }.f,
    F64Transmute { u: 0x2080000000000000 }.f,
    F64Transmute { u: 0x2090000000000000 }.f,
    F64Transmute { u: 0x20A0000000000000 }.f,
    F64Transmute { u: 0x20B0000000000000 }.f,
    F64Transmute { u: 0x20C0000000000000 }.f,
    F64Transmute { u: 0x20D0000000000000 }.f,
    F64Transmute { u: 0x20E0000000000000 }.f,
    F64Transmute { u: 0x20F0000000000000 }.f,
    F64Transmute { u: 0x2100000000000000 }.f,
    F64Transmute { u: 0x2110000000000000 }.f,
    F64Transmute { u: 0x2120000000000000 }.f,
    F64Transmute { u: 0x2130000000000000 }.f,
    F64Transmute { u: 0x2140000000000000 }.f,
    F64Transmute { u: 0x2150000000000000 }.f,
    F64Transmute { u: 0x2160000000000000 }.f,
    F64Transmute { u: 0x2170000000000000 }.f,
    F64Transmute { u: 0x2180000000000000 }.f,
    F64Transmute { u: 0x2190000000000000 }.f,
    F64Transmute { u: 0x21A0000000000000 }.f,
    F64Transmute { u: 0x21B0000000000000 }.f,
    F64Transmute { u: 0x21C0000000000000 }.f,
    F64Transmute { u: 0x21D0000000000000 }.f,
    F64Transmute { u: 0x21E0000000000000 }.f,
    F64Transmute { u: 0x21F0000000000000 }.f,
    F64Transmute { u: 0x2200000000000000 }.f,
    F64Transmute { u: 0x2210000000000000 }.f,
    F64Transmute { u: 0x2220000000000000 }.f,
    F64Transmute { u: 0x2230000000000000 }.f,
    F64Transmute { u: 0x2240000000000000 }.f,
    F64Transmute { u: 0x2250000000000000 }.f,
    F64Transmute { u: 0x2260000000000000 }.f,
    F64Transmute { u: 0x2270000000000000 }.f,
    F64Transmute { u: 0x2280000000000000 }.f,
    F64Transmute { u: 0x2290000000000000 }.f,
    F64Transmute { u: 0x22A0000000000000 }.f,
    F64Transmute { u: 0x22B0000000000000 }.f,
    F64Transmute { u: 0x22C0000000000000 }.f,
    F64Transmute { u: 0x22D0000000000000 }.f,
    F64Transmute { u: 0x22E0000000000000 }.f,
    F64Transmute { u: 0x22F0000000000000 }.f,
    F64Transmute { u: 0x2300000000000000 }.f,
    F64Transmute { u: 0x2310000000000000 }.f,
    F64Transmute { u: 0x2320000000000000 }.f,
    F64Transmute { u: 0x2330000000000000 }.f,
    F64Transmute { u: 0x2340000000000000 }.f,
    F64Transmute { u: 0x2350000000000000 }.f,
    F64Transmute { u: 0x2360000000000000 }.f,
    F64Transmute { u: 0x2370000000000000 }.f,
    F64Transmute { u: 0x2380000000000000 }.f,
    F64Transmute { u: 0x2390000000000000 }.f,
    F64Transmute { u: 0x23A0000000000000 }.f,
    F64Transmute { u: 0x23B0000000000000 }.f,
    F64Transmute { u: 0x23C0000000000000 }.f,
    F64Transmute { u: 0x23D0000000000000 }.f,
    F64Transmute { u: 0x23E0000000000000 }.f,
    F64Transmute { u: 0x23F0000000000000 }.f,
    F64Transmute { u: 0x2400000000000000 }.f,
    F64Transmute { u: 0x2410000000000000 }.f,
    F64Transmute { u: 0x2420000000000000 }.f,
    F64Transmute { u: 0x2430000000000000 }.f,
    F64Transmute { u: 0x2440000000000000 }.f,
    F64Transmute { u: 0x2450000000000000 }.f,
    F64Transmute { u: 0x2460000000000000 }.f,
    F64Transmute { u: 0x2470000000000000 }.f,
    F64Transmute { u: 0x2480000000000000 }.f,
    F64Transmute { u: 0x2490000000000000 }.f,
    F64Transmute { u: 0x24A0000000000000 }.f,
    F64Transmute { u: 0x24B0000000000000 }.f,
    F64Transmute { u: 0x24C0000000000000 }.f,
    F64Transmute { u: 0x24D0000000000000 }.f,
    F64Transmute { u: 0x24E0000000000000 }.f,
    F64Transmute { u: 0x24F0000000000000 }.f,
    F64Transmute { u: 0x2500000000000000 }.f,
    F64Transmute { u: 0x2510000000000000 }.f,
    F64Transmute { u: 0x2520000000000000 }.f,
    F64Transmute { u: 0x2530000000000000 }.f,
    F64Transmute { u: 0x2540000000000000 }.f,
    F64Transmute { u: 0x2550000000000000 }.f,
    F64Transmute { u: 0x2560000000000000 }.f,
    F64Transmute { u: 0x2570000000000000 }.f,
    F64Transmute { u: 0x2580000000000000 }.f,
    F64Transmute { u: 0x2590000000000000 }.f,
    F64Transmute { u: 0x25A0000000000000 }.f,
    F64Transmute { u: 0x25B0000000000000 }.f,
    F64Transmute { u: 0x25C0000000000000 }.f,
    F64Transmute { u: 0x25D0000000000000 }.f,
    F64Transmute { u: 0x25E0000000000000 }.f,
    F64Transmute { u: 0x25F0000000000000 }.f,
    F64Transmute { u: 0x2600000000000000 }.f,
    F64Transmute { u: 0x2610000000000000 }.f,
    F64Transmute { u: 0x2620000000000000 }.f,
    F64Transmute { u: 0x2630000000000000 }.f,
    F64Transmute { u: 0x2640000000000000 }.f,
    F64Transmute { u: 0x2650000000000000 }.f,
    F64Transmute { u: 0x2660000000000000 }.f,
    F64Transmute { u: 0x2670000000000000 }.f,
    F64Transmute { u: 0x2680000000000000 }.f,
    F64Transmute { u: 0x2690000000000000 }.f,
    F64Transmute { u: 0x26A0000000000000 }.f,
    F64Transmute { u: 0x26B0000000000000 }.f,
    F64Transmute { u: 0x26C0000000000000 }.f,
    F64Transmute { u: 0x26D0000000000000 }.f,
    F64Transmute { u: 0x26E0000000000000 }.f,
    F64Transmute { u: 0x26F0000000000000 }.f,
    F64Transmute { u: 0x2700000000000000 }.f,
    F64Transmute { u: 0x2710000000000000 }.f,
    F64Transmute { u: 0x2720000000000000 }.f,
    F64Transmute { u: 0x2730000000000000 }.f,
    F64Transmute { u: 0x2740000000000000 }.f,
    F64Transmute { u: 0x2750000000000000 }.f,
    F64Transmute { u: 0x2760000000000000 }.f,
    F64Transmute { u: 0x2770000000000000 }.f,
    F64Transmute { u: 0x2780000000000000 }.f,
    F64Transmute { u: 0x2790000000000000 }.f,
    F64Transmute { u: 0x27A0000000000000 }.f,
    F64Transmute { u: 0x27B0000000000000 }.f,
    F64Transmute { u: 0x27C0000000000000 }.f,
    F64Transmute { u: 0x27D0000000000000 }.f,
    F64Transmute { u: 0x27E0000000000000 }.f,
    F64Transmute { u: 0x27F0000000000000 }.f,
    F64Transmute { u: 0x2800000000000000 }.f,
    F64Transmute { u: 0x2810000000000000 }.f,
    F64Transmute { u: 0x2820000000000000 }.f,
    F64Transmute { u: 0x2830000000000000 }.f,
    F64Transmute { u: 0x2840000000000000 }.f,
    F64Transmute { u: 0x2850000000000000 }.f,
    F64Transmute { u: 0x2860000000000000 }.f,
    F64Transmute { u: 0x2870000000000000 }.f,
    F64Transmute { u: 0x2880000000000000 }.f,
    F64Transmute { u: 0x2890000000000000 }.f,
    F64Transmute { u: 0x28A0000000000000 }.f,
    F64Transmute { u: 0x28B0000000000000 }.f,
    F64Transmute { u: 0x28C0000000000000 }.f,
    F64Transmute { u: 0x28D0000000000000 }.f,
    F64Transmute { u: 0x28E0000000000000 }.f,
    F64Transmute { u: 0x28F0000000000000 }.f,
    F64Transmute { u: 0x2900000000000000 }.f,
    F64Transmute { u: 0x2910000000000000 }.f,
    F64Transmute { u: 0x2920000000000000 }.f,
    F64Transmute { u: 0x2930000000000000 }.f,
    F64Transmute { u: 0x2940000000000000 }.f,
    F64Transmute { u: 0x2950000000000000 }.f,
    F64Transmute { u: 0x2960000000000000 }.f,
    F64Transmute { u: 0x2970000000000000 }.f,
    F64Transmute { u: 0x2980000000000000 }.f,
    F64Transmute { u: 0x2990000000000000 }.f,
    F64Transmute { u: 0x29A0000000000000 }.f,
    F64Transmute { u: 0x29B0000000000000 }.f,
    F64Transmute { u: 0x29C0000000000000 }.f,
    F64Transmute { u: 0x29D0000000000000 }.f,
    F64Transmute { u: 0x29E0000000000000 }.f,
    F64Transmute { u: 0x29F0000000000000 }.f,
    F64Transmute { u: 0x2A00000000000000 }.f,
    F64Transmute { u: 0x2A10000000000000 }.f,
    F64Transmute { u: 0x2A20000000000000 }.f,
    F64Transmute { u: 0x2A30000000000000 }.f,
    F64Transmute { u: 0x2A40000000000000 }.f,
    F64Transmute { u: 0x2A50000000000000 }.f,
    F64Transmute { u: 0x2A60000000000000 }.f,
    F64Transmute { u: 0x2A70000000000000 }.f,
    F64Transmute { u: 0x2A80000000000000 }.f,
    F64Transmute { u: 0x2A90000000000000 }.f,
    F64Transmute { u: 0x2AA0000000000000 }.f,
    F64Transmute { u: 0x2AB0000000000000 }.f,
    F64Transmute { u: 0x2AC0000000000000 }.f,
    F64Transmute { u: 0x2AD0000000000000 }.f,
    F64Transmute { u: 0x2AE0000000000000 }.f,
    F64Transmute { u: 0x2AF0000000000000 }.f,
    F64Transmute { u: 0x2B00000000000000 }.f,
    F64Transmute { u: 0x2B10000000000000 }.f,
    F64Transmute { u: 0x2B20000000000000 }.f,
    F64Transmute { u: 0x2B30000000000000 }.f,
    F64Transmute { u: 0x2B40000000000000 }.f,
    F64Transmute { u: 0x2B50000000000000 }.f,
    F64Transmute { u: 0x2B60000000000000 }.f,
    F64Transmute { u: 0x2B70000000000000 }.f,
    F64Transmute { u: 0x2B80000000000000 }.f,
    F64Transmute { u: 0x2B90000000000000 }.f,
    F64Transmute { u: 0x2BA0000000000000 }.f,
    F64Transmute { u: 0x2BB0000000000000 }.f,
    F64Transmute { u: 0x2BC0000000000000 }.f,
    F64Transmute { u: 0x2BD0000000000000 }.f,
    F64Transmute { u: 0x2BE0000000000000 }.f,
    F64Transmute { u: 0x2BF0000000000000 }.f,
    F64Transmute { u: 0x2C00000000000000 }.f,
    F64Transmute { u: 0x2C10000000000000 }.f,
    F64Transmute { u: 0x2C20000000000000 }.f,
    F64Transmute { u: 0x2C30000000000000 }.f,
    F64Transmute { u: 0x2C40000000000000 }.f,
    F64Transmute { u: 0x2C50000000000000 }.f,
    F64Transmute { u: 0x2C60000000000000 }.f,
    F64Transmute { u: 0x2C70000000000000 }.f,
    F64Transmute { u: 0x2C80000000000000 }.f,
    F64Transmute { u: 0x2C90000000000000 }.f,
    F64Transmute { u: 0x2CA0000000000000 }.f,
    F64Transmute { u: 0x2CB0000000000000 }.f,
    F64Transmute { u: 0x2CC0000000000000 }.f,
    F64Transmute { u: 0x2CD0000000000000 }.f,
    F64Transmute { u: 0x2CE0000000000000 }.f,
    F64Transmute { u: 0x2CF0000000000000 }.f,
    F64Transmute { u: 0x2D00000000000000 }.f,
    F64Transmute { u: 0x2D10000000000000 }.f,
    F64Transmute { u: 0x2D20000000000000 }.f,
    F64Transmute { u: 0x2D30000000000000 }.f,
    F64Transmute { u: 0x2D40000000000000 }.f,
    F64Transmute { u: 0x2D50000000000000 }.f,
    F64Transmute { u: 0x2D60000000000000 }.f,
    F64Transmute { u: 0x2D70000000000000 }.f,
    F64Transmute { u: 0x2D80000000000000 }.f,
    F64Transmute { u: 0x2D90000000000000 }.f,
    F64Transmute { u: 0x2DA0000000000000 }.f,
    F64Transmute { u: 0x2DB0000000000000 }.f,
    F64Transmute { u: 0x2DC0000000000000 }.f,
    F64Transmute { u: 0x2DD0000000000000 }.f,
    F64Transmute { u: 0x2DE0000000000000 }.f,
    F64Transmute { u: 0x2DF0000000000000 }.f,
    F64Transmute { u: 0x2E00000000000000 }.f,
    F64Transmute { u: 0x2E10000000000000 }.f,
    F64Transmute { u: 0x2E20000000000000 }.f,
    F64Transmute { u: 0x2E30000000000000 }.f,
    F64Transmute { u: 0x2E40000000000000 }.f,
    F64Transmute { u: 0x2E50000000000000 }.f,
    F64Transmute { u: 0x2E60000000000000 }.f,
    F64Transmute { u: 0x2E70000000000000 }.f,
    F64Transmute { u: 0x2E80000000000000 }.f,
    F64Transmute { u: 0x2E90000000000000 }.f,
    F64Transmute { u: 0x2EA0000000000000 }.f,
    F64Transmute { u: 0x2EB0000000000000 }.f,
    F64Transmute { u: 0x2EC0000000000000 }.f,
    F64Transmute { u: 0x2ED0000000000000 }.f,
    F64Transmute { u: 0x2EE0000000000000 }.f,
    F64Transmute { u: 0x2EF0000000000000 }.f,
    F64Transmute { u: 0x2F00000000000000 }.f,
    F64Transmute { u: 0x2F10000000000000 }.f,
    F64Transmute { u: 0x2F20000000000000 }.f,
    F64Transmute { u: 0x2F30000000000000 }.f,
    F64Transmute { u: 0x2F40000000000000 }.f,
    F64Transmute { u: 0x2F50000000000000 }.f,
    F64Transmute { u: 0x2F60000000000000 }.f,
    F64Transmute { u: 0x2F70000000000000 }.f,
    F64Transmute { u: 0x2F80000000000000 }.f,
    F64Transmute { u: 0x2F90000000000000 }.f,
    F64Transmute { u: 0x2FA0000000000000 }.f,
    F64Transmute { u: 0x2FB0000000000000 }.f,
    F64Transmute { u: 0x2FC0000000000000 }.f,
    F64Transmute { u: 0x2FD0000000000000 }.f,
    F64Transmute { u: 0x2FE0000000000000 }.f,
    F64Transmute { u: 0x2FF0000000000000 }.f,
    F64Transmute { u: 0x3000000000000000 }.f,
    F64Transmute { u: 0x3010000000000000 }.f,
    F64Transmute { u: 0x3020000000000000 }.f,
    F64Transmute { u: 0x3030000000000000 }.f,
    F64Transmute { u: 0x3040000000000000 }.f,
    F64Transmute { u: 0x3050000000000000 }.f,
    F64Transmute { u: 0x3060000000000000 }.f,
    F64Transmute { u: 0x3070000000000000 }.f,
    F64Transmute { u: 0x3080000000000000 }.f,
    F64Transmute { u: 0x3090000000000000 }.f,
    F64Transmute { u: 0x30A0000000000000 }.f,
    F64Transmute { u: 0x30B0000000000000 }.f,
    F64Transmute { u: 0x30C0000000000000 }.f,
    F64Transmute { u: 0x30D0000000000000 }.f,
    F64Transmute { u: 0x30E0000000000000 }.f,
    F64Transmute { u: 0x30F0000000000000 }.f,
    F64Transmute { u: 0x3100000000000000 }.f,
    F64Transmute { u: 0x3110000000000000 }.f,
    F64Transmute { u: 0x3120000000000000 }.f,
    F64Transmute { u: 0x3130000000000000 }.f,
    F64Transmute { u: 0x3140000000000000 }.f,
    F64Transmute { u: 0x3150000000000000 }.f,
    F64Transmute { u: 0x3160000000000000 }.f,
    F64Transmute { u: 0x3170000000000000 }.f,
    F64Transmute { u: 0x3180000000000000 }.f,
    F64Transmute { u: 0x3190000000000000 }.f,
    F64Transmute { u: 0x31A0000000000000 }.f,
    F64Transmute { u: 0x31B0000000000000 }.f,
    F64Transmute { u: 0x31C0000000000000 }.f,
    F64Transmute { u: 0x31D0000000000000 }.f,
    F64Transmute { u: 0x31E0000000000000 }.f,
    F64Transmute { u: 0x31F0000000000000 }.f,
    F64Transmute { u: 0x3200000000000000 }.f,
    F64Transmute { u: 0x3210000000000000 }.f,
    F64Transmute { u: 0x3220000000000000 }.f,
    F64Transmute { u: 0x3230000000000000 }.f,
    F64Transmute { u: 0x3240000000000000 }.f,
    F64Transmute { u: 0x3250000000000000 }.f,
    F64Transmute { u: 0x3260000000000000 }.f,
    F64Transmute { u: 0x3270000000000000 }.f,
    F64Transmute { u: 0x3280000000000000 }.f,
    F64Transmute { u: 0x3290000000000000 }.f,
    F64Transmute { u: 0x32A0000000000000 }.f,
    F64Transmute { u: 0x32B0000000000000 }.f,
    F64Transmute { u: 0x32C0000000000000 }.f,
    F64Transmute { u: 0x32D0000000000000 }.f,
    F64Transmute { u: 0x32E0000000000000 }.f,
    F64Transmute { u: 0x32F0000000000000 }.f,
    F64Transmute { u: 0x3300000000000000 }.f,
    F64Transmute { u: 0x3310000000000000 }.f,
    F64Transmute { u: 0x3320000000000000 }.f,
    F64Transmute { u: 0x3330000000000000 }.f,
    F64Transmute { u: 0x3340000000000000 }.f,
    F64Transmute { u: 0x3350000000000000 }.f,
    F64Transmute { u: 0x3360000000000000 }.f,
    F64Transmute { u: 0x3370000000000000 }.f,
    F64Transmute { u: 0x3380000000000000 }.f,
    F64Transmute { u: 0x3390000000000000 }.f,
    F64Transmute { u: 0x33A0000000000000 }.f,
    F64Transmute { u: 0x33B0000000000000 }.f,
    F64Transmute { u: 0x33C0000000000000 }.f,
    F64Transmute { u: 0x33D0000000000000 }.f,
    F64Transmute { u: 0x33E0000000000000 }.f,
    F64Transmute { u: 0x33F0000000000000 }.f,
    F64Transmute { u: 0x3400000000000000 }.f,
    F64Transmute { u: 0x3410000000000000 }.f,
    F64Transmute { u: 0x3420000000000000 }.f,
    F64Transmute { u: 0x3430000000000000 }.f,
    F64Transmute { u: 0x3440000000000000 }.f,
    F64Transmute { u: 0x3450000000000000 }.f,
    F64Transmute { u: 0x3460000000000000 }.f,
    F64Transmute { u: 0x3470000000000000 }.f,
    F64Transmute { u: 0x3480000000000000 }.f,
    F64Transmute { u: 0x3490000000000000 }.f,
    F64Transmute { u: 0x34A0000000000000 }.f,
    F64Transmute { u: 0x34B0000000000000 }.f,
    F64Transmute { u: 0x34C0000000000000 }.f,
    F64Transmute { u: 0x34D0000000000000 }.f,
    F64Transmute { u: 0x34E0000000000000 }.f,
    F64Transmute { u: 0x34F0000000000000 }.f,
    F64Transmute { u: 0x3500000000000000 }.f,
    F64Transmute { u: 0x3510000000000000 }.f,
    F64Transmute { u: 0x3520000000000000 }.f,
    F64Transmute { u: 0x3530000000000000 }.f,
    F64Transmute { u: 0x3540000000000000 }.f,
    F64Transmute { u: 0x3550000000000000 }.f,
    F64Transmute { u: 0x3560000000000000 }.f,
    F64Transmute { u: 0x3570000000000000 }.f,
    F64Transmute { u: 0x3580000000000000 }.f,
    F64Transmute { u: 0x3590000000000000 }.f,
    F64Transmute { u: 0x35A0000000000000 }.f,
    F64Transmute { u: 0x35B0000000000000 }.f,
    F64Transmute { u: 0x35C0000000000000 }.f,
    F64Transmute { u: 0x35D0000000000000 }.f,
    F64Transmute { u: 0x35E0000000000000 }.f,
    F64Transmute { u: 0x35F0000000000000 }.f,
    F64Transmute { u: 0x3600000000000000 }.f,
    F64Transmute { u: 0x3610000000000000 }.f,
    F64Transmute { u: 0x3620000000000000 }.f,
    F64Transmute { u: 0x3630000000000000 }.f,
    F64Transmute { u: 0x3640000000000000 }.f,
    F64Transmute { u: 0x3650000000000000 }.f,
    F64Transmute { u: 0x3660000000000000 }.f,
    F64Transmute { u: 0x3670000000000000 }.f,
    F64Transmute { u: 0x3680000000000000 }.f,
    F64Transmute { u: 0x3690000000000000 }.f,
    F64Transmute { u: 0x36A0000000000000 }.f,
    F64Transmute { u: 0x36B0000000000000 }.f,
    F64Transmute { u: 0x36C0000000000000 }.f,
    F64Transmute { u: 0x36D0000000000000 }.f,
    F64Transmute { u: 0x36E0000000000000 }.f,
    F64Transmute { u: 0x36F0000000000000 }.f,
    F64Transmute { u: 0x3700000000000000 }.f,
    F64Transmute { u: 0x3710000000000000 }.f,
    F64Transmute { u: 0x3720000000000000 }.f,
    F64Transmute { u: 0x3730000000000000 }.f,
    F64Transmute { u: 0x3740000000000000 }.f,
    F64Transmute { u: 0x3750000000000000 }.f,
    F64Transmute { u: 0x3760000000000000 }.f,
    F64Transmute { u: 0x3770000000000000 }.f,
    F64Transmute { u: 0x3780000000000000 }.f,
    F64Transmute { u: 0x3790000000000000 }.f,
    F64Transmute { u: 0x37A0000000000000 }.f,
    F64Transmute { u: 0x37B0000000000000 }.f,
    F64Transmute { u: 0x37C0000000000000 }.f,
    F64Transmute { u: 0x37D0000000000000 }.f,
    F64Transmute { u: 0x37E0000000000000 }.f,
    F64Transmute { u: 0x37F0000000000000 }.f,
    F64Transmute { u: 0x3800000000000000 }.f,
    F64Transmute { u: 0x3810000000000000 }.f,
    F64Transmute { u: 0x3820000000000000 }.f,
    F64Transmute { u: 0x3830000000000000 }.f,
    F64Transmute { u: 0x3840000000000000 }.f,
    F64Transmute { u: 0x3850000000000000 }.f,
    F64Transmute { u: 0x3860000000000000 }.f,
    F64Transmute { u: 0x3870000000000000 }.f,
    F64Transmute { u: 0x3880000000000000 }.f,
    F64Transmute { u: 0x3890000000000000 }.f,
    F64Transmute { u: 0x38A0000000000000 }.f,
    F64Transmute { u: 0x38B0000000000000 }.f,
    F64Transmute { u: 0x38C0000000000000 }.f,
    F64Transmute { u: 0x38D0000000000000 }.f,
    F64Transmute { u: 0x38E0000000000000 }.f,
    F64Transmute { u: 0x38F0000000000000 }.f,
    F64Transmute { u: 0x3900000000000000 }.f,
    F64Transmute { u: 0x3910000000000000 }.f,
    F64Transmute { u: 0x3920000000000000 }.f,
    F64Transmute { u: 0x3930000000000000 }.f,
    F64Transmute { u: 0x3940000000000000 }.f,
    F64Transmute { u: 0x3950000000000000 }.f,
    F64Transmute { u: 0x3960000000000000 }.f,
    F64Transmute { u: 0x3970000000000000 }.f,
    F64Transmute { u: 0x3980000000000000 }.f,
    F64Transmute { u: 0x3990000000000000 }.f,
    F64Transmute { u: 0x39A0000000000000 }.f,
    F64Transmute { u: 0x39B0000000000000 }.f,
    F64Transmute { u: 0x39C0000000000000 }.f,
    F64Transmute { u: 0x39D0000000000000 }.f,
    F64Transmute { u: 0x39E0000000000000 }.f,
    F64Transmute { u: 0x39F0000000000000 }.f,
    F64Transmute { u: 0x3A00000000000000 }.f,
    F64Transmute { u: 0x3A10000000000000 }.f,
    F64Transmute { u: 0x3A20000000000000 }.f,
    F64Transmute { u: 0x3A30000000000000 }.f,
    F64Transmute { u: 0x3A40000000000000 }.f,
    F64Transmute { u: 0x3A50000000000000 }.f,
    F64Transmute { u: 0x3A60000000000000 }.f,
    F64Transmute { u: 0x3A70000000000000 }.f,
    F64Transmute { u: 0x3A80000000000000 }.f,
    F64Transmute { u: 0x3A90000000000000 }.f,
    F64Transmute { u: 0x3AA0000000000000 }.f,
    F64Transmute { u: 0x3AB0000000000000 }.f,
    F64Transmute { u: 0x3AC0000000000000 }.f,
    F64Transmute { u: 0x3AD0000000000000 }.f,
    F64Transmute { u: 0x3AE0000000000000 }.f,
    F64Transmute { u: 0x3AF0000000000000 }.f,
    F64Transmute { u: 0x3B00000000000000 }.f,
    F64Transmute { u: 0x3B10000000000000 }.f,
    F64Transmute { u: 0x3B20000000000000 }.f,
    F64Transmute { u: 0x3B30000000000000 }.f,
    F64Transmute { u: 0x3B40000000000000 }.f,
    F64Transmute { u: 0x3B50000000000000 }.f,
    F64Transmute { u: 0x3B60000000000000 }.f,
    F64Transmute { u: 0x3B70000000000000 }.f,
    F64Transmute { u: 0x3B80000000000000 }.f,
    F64Transmute { u: 0x3B90000000000000 }.f,
    F64Transmute { u: 0x3BA0000000000000 }.f,
    F64Transmute { u: 0x3BB0000000000000 }.f,
    F64Transmute { u: 0x3BC0000000000000 }.f,
    F64Transmute { u: 0x3BD0000000000000 }.f,
    F64Transmute { u: 0x3BE0000000000000 }.f,
    F64Transmute { u: 0x3BF0000000000000 }.f,
    F64Transmute { u: 0x3C00000000000000 }.f,
    F64Transmute { u: 0x3C10000000000000 }.f,
    F64Transmute { u: 0x3C20000000000000 }.f,
    F64Transmute { u: 0x3C30000000000000 }.f,
    F64Transmute { u: 0x3C40000000000000 }.f,
    F64Transmute { u: 0x3C50000000000000 }.f,
    F64Transmute { u: 0x3C60000000000000 }.f,
    F64Transmute { u: 0x3C70000000000000 }.f,
    F64Transmute { u: 0x3C80000000000000 }.f,
    F64Transmute { u: 0x3C90000000000000 }.f,
    F64Transmute { u: 0x3CA0000000000000 }.f,
    F64Transmute { u: 0x3CB0000000000000 }.f,
    F64Transmute { u: 0x3CC0000000000000 }.f,
    F64Transmute { u: 0x3CD0000000000000 }.f,
    F64Transmute { u: 0x3CE0000000000000 }.f,
    F64Transmute { u: 0x3CF0000000000000 }.f,
    F64Transmute { u: 0x3D00000000000000 }.f,
    F64Transmute { u: 0x3D10000000000000 }.f,
    F64Transmute { u: 0x3D20000000000000 }.f,
    F64Transmute { u: 0x3D30000000000000 }.f,
    F64Transmute { u: 0x3D40000000000000 }.f,
    F64Transmute { u: 0x3D50000000000000 }.f,
    F64Transmute { u: 0x3D60000000000000 }.f,
    F64Transmute { u: 0x3D70000000000000 }.f,
    F64Transmute { u: 0x3D80000000000000 }.f,
    F64Transmute { u: 0x3D90000000000000 }.f,
    F64Transmute { u: 0x3DA0000000000000 }.f,
    F64Transmute { u: 0x3DB0000000000000 }.f,
    F64Transmute { u: 0x3DC0000000000000 }.f,
    F64Transmute { u: 0x3DD0000000000000 }.f,
    F64Transmute { u: 0x3DE0000000000000 }.f,
    F64Transmute { u: 0x3DF0000000000000 }.f,
    F64Transmute { u: 0x3E00000000000000 }.f,
    F64Transmute { u: 0x3E10000000000000 }.f,
    F64Transmute { u: 0x3E20000000000000 }.f,
    F64Transmute { u: 0x3E30000000000000 }.f,
    F64Transmute { u: 0x3E40000000000000 }.f,
    F64Transmute { u: 0x3E50000000000000 }.f,
    F64Transmute { u: 0x3E60000000000000 }.f,
    F64Transmute { u: 0x3E70000000000000 }.f,
    F64Transmute { u: 0x3E80000000000000 }.f,
    F64Transmute { u: 0x3E90000000000000 }.f,
    F64Transmute { u: 0x3EA0000000000000 }.f,
    F64Transmute { u: 0x3EB0000000000000 }.f,
    F64Transmute { u: 0x3EC0000000000000 }.f,
    F64Transmute { u: 0x3ED0000000000000 }.f,
    F64Transmute { u: 0x3EE0000000000000 }.f,
    F64Transmute { u: 0x3EF0000000000000 }.f,
    F64Transmute { u: 0x3F00000000000000 }.f,
    F64Transmute { u: 0x3F10000000000000 }.f,
    F64Transmute { u: 0x3F20000000000000 }.f,
    F64Transmute { u: 0x3F30000000000000 }.f,
    F64Transmute { u: 0x3F40000000000000 }.f,
    F64Transmute { u: 0x3F50000000000000 }.f,
    F64Transmute { u: 0x3F60000000000000 }.f,
    F64Transmute { u: 0x3F70000000000000 }.f,
    F64Transmute { u: 0x3F80000000000000 }.f,
    F64Transmute { u: 0x3F90000000000000 }.f,
    F64Transmute { u: 0x3FA0000000000000 }.f,
    F64Transmute { u: 0x3FB0000000000000 }.f,
    F64Transmute { u: 0x3FC0000000000000 }.f,
    F64Transmute { u: 0x3FD0000000000000 }.f,
    F64Transmute { u: 0x3FE0000000000000 }.f,
    F64Transmute { u: 0x3FF0000000000000 }.f,
    F64Transmute { u: 0x4000000000000000 }.f,
    F64Transmute { u: 0x4010000000000000 }.f,
    F64Transmute { u: 0x4020000000000000 }.f,
    F64Transmute { u: 0x4030000000000000 }.f,
    F64Transmute { u: 0x4040000000000000 }.f,
    F64Transmute { u: 0x4050000000000000 }.f,
    F64Transmute { u: 0x4060000000000000 }.f,
    F64Transmute { u: 0x4070000000000000 }.f,
    F64Transmute { u: 0x4080000000000000 }.f,
    F64Transmute { u: 0x4090000000000000 }.f,
    F64Transmute { u: 0x40A0000000000000 }.f,
    F64Transmute { u: 0x40B0000000000000 }.f,
    F64Transmute { u: 0x40C0000000000000 }.f,
    F64Transmute { u: 0x40D0000000000000 }.f,
    F64Transmute { u: 0x40E0000000000000 }.f,
    F64Transmute { u: 0x40F0000000000000 }.f,
    F64Transmute { u: 0x4100000000000000 }.f,
    F64Transmute { u: 0x4110000000000000 }.f,
    F64Transmute { u: 0x4120000000000000 }.f,
    F64Transmute { u: 0x4130000000000000 }.f,
    F64Transmute { u: 0x4140000000000000 }.f,
    F64Transmute { u: 0x4150000000000000 }.f,
    F64Transmute { u: 0x4160000000000000 }.f,
    F64Transmute { u: 0x4170000000000000 }.f,
    F64Transmute { u: 0x4180000000000000 }.f,
    F64Transmute { u: 0x4190000000000000 }.f,
    F64Transmute { u: 0x41A0000000000000 }.f,
    F64Transmute { u: 0x41B0000000000000 }.f,
    F64Transmute { u: 0x41C0000000000000 }.f,
    F64Transmute { u: 0x41D0000000000000 }.f,
    F64Transmute { u: 0x41E0000000000000 }.f,
    F64Transmute { u: 0x41F0000000000000 }.f,
    F64Transmute { u: 0x4200000000000000 }.f,
    F64Transmute { u: 0x4210000000000000 }.f,
    F64Transmute { u: 0x4220000000000000 }.f,
    F64Transmute { u: 0x4230000000000000 }.f,
    F64Transmute { u: 0x4240000000000000 }.f,
    F64Transmute { u: 0x4250000000000000 }.f,
    F64Transmute { u: 0x4260000000000000 }.f,
    F64Transmute { u: 0x4270000000000000 }.f,
    F64Transmute { u: 0x4280000000000000 }.f,
    F64Transmute { u: 0x4290000000000000 }.f,
    F64Transmute { u: 0x42A0000000000000 }.f,
    F64Transmute { u: 0x42B0000000000000 }.f,
    F64Transmute { u: 0x42C0000000000000 }.f,
    F64Transmute { u: 0x42D0000000000000 }.f,
    F64Transmute { u: 0x42E0000000000000 }.f,
    F64Transmute { u: 0x42F0000000000000 }.f,
    F64Transmute { u: 0x4300000000000000 }.f,
    F64Transmute { u: 0x4310000000000000 }.f,
    F64Transmute { u: 0x4320000000000000 }.f,
    F64Transmute { u: 0x4330000000000000 }.f,
    F64Transmute { u: 0x4340000000000000 }.f,
    F64Transmute { u: 0x4350000000000000 }.f,
    F64Transmute { u: 0x4360000000000000 }.f,
    F64Transmute { u: 0x4370000000000000 }.f,
    F64Transmute { u: 0x4380000000000000 }.f,
    F64Transmute { u: 0x4390000000000000 }.f,
    F64Transmute { u: 0x43A0000000000000 }.f,
    F64Transmute { u: 0x43B0000000000000 }.f,
    F64Transmute { u: 0x43C0000000000000 }.f,
    F64Transmute { u: 0x43D0000000000000 }.f,
    F64Transmute { u: 0x43E0000000000000 }.f,
    F64Transmute { u: 0x43F0000000000000 }.f,
    F64Transmute { u: 0x4400000000000000 }.f,
    F64Transmute { u: 0x4410000000000000 }.f,
    F64Transmute { u: 0x4420000000000000 }.f,
    F64Transmute { u: 0x4430000000000000 }.f,
    F64Transmute { u: 0x4440000000000000 }.f,
    F64Transmute { u: 0x4450000000000000 }.f,
    F64Transmute { u: 0x4460000000000000 }.f,
    F64Transmute { u: 0x4470000000000000 }.f,
    F64Transmute { u: 0x4480000000000000 }.f,
    F64Transmute { u: 0x4490000000000000 }.f,
    F64Transmute { u: 0x44A0000000000000 }.f,
    F64Transmute { u: 0x44B0000000000000 }.f,
    F64Transmute { u: 0x44C0000000000000 }.f,
    F64Transmute { u: 0x44D0000000000000 }.f,
    F64Transmute { u: 0x44E0000000000000 }.f,
    F64Transmute { u: 0x44F0000000000000 }.f,
    F64Transmute { u: 0x4500000000000000 }.f,
    F64Transmute { u: 0x4510000000000000 }.f,
    F64Transmute { u: 0x4520000000000000 }.f,
    F64Transmute { u: 0x4530000000000000 }.f,
    F64Transmute { u: 0x4540000000000000 }.f,
    F64Transmute { u: 0x4550000000000000 }.f,
    F64Transmute { u: 0x4560000000000000 }.f,
    F64Transmute { u: 0x4570000000000000 }.f,
    F64Transmute { u: 0x4580000000000000 }.f,
    F64Transmute { u: 0x4590000000000000 }.f,
    F64Transmute { u: 0x45A0000000000000 }.f,
    F64Transmute { u: 0x45B0000000000000 }.f,
    F64Transmute { u: 0x45C0000000000000 }.f,
    F64Transmute { u: 0x45D0000000000000 }.f,
    F64Transmute { u: 0x45E0000000000000 }.f,
    F64Transmute { u: 0x45F0000000000000 }.f,
    F64Transmute { u: 0x4600000000000000 }.f,
    F64Transmute { u: 0x4610000000000000 }.f,
    F64Transmute { u: 0x4620000000000000 }.f,
    F64Transmute { u: 0x4630000000000000 }.f,
    F64Transmute { u: 0x4640000000000000 }.f,
    F64Transmute { u: 0x4650000000000000 }.f,
    F64Transmute { u: 0x4660000000000000 }.f,
    F64Transmute { u: 0x4670000000000000 }.f,
    F64Transmute { u: 0x4680000000000000 }.f,
    F64Transmute { u: 0x4690000000000000 }.f,
    F64Transmute { u: 0x46A0000000000000 }.f,
    F64Transmute { u: 0x46B0000000000000 }.f,
    F64Transmute { u: 0x46C0000000000000 }.f,
    F64Transmute { u: 0x46D0000000000000 }.f,
    F64Transmute { u: 0x46E0000000000000 }.f,
    F64Transmute { u: 0x46F0000000000000 }.f,
    F64Transmute { u: 0x4700000000000000 }.f,
    F64Transmute { u: 0x4710000000000000 }.f,
    F64Transmute { u: 0x4720000000000000 }.f,
    F64Transmute { u: 0x4730000000000000 }.f,
    F64Transmute { u: 0x4740000000000000 }.f,
    F64Transmute { u: 0x4750000000000000 }.f,
    F64Transmute { u: 0x4760000000000000 }.f,
    F64Transmute { u: 0x4770000000000000 }.f,
    F64Transmute { u: 0x4780000000000000 }.f,
    F64Transmute { u: 0x4790000000000000 }.f,
    F64Transmute { u: 0x47A0000000000000 }.f,
    F64Transmute { u: 0x47B0000000000000 }.f,
    F64Transmute { u: 0x47C0000000000000 }.f,
    F64Transmute { u: 0x47D0000000000000 }.f,
    F64Transmute { u: 0x47E0000000000000 }.f,
    F64Transmute { u: 0x47F0000000000000 }.f,
    F64Transmute { u: 0x4800000000000000 }.f,
    F64Transmute { u: 0x4810000000000000 }.f,
    F64Transmute { u: 0x4820000000000000 }.f,
    F64Transmute { u: 0x4830000000000000 }.f,
    F64Transmute { u: 0x4840000000000000 }.f,
    F64Transmute { u: 0x4850000000000000 }.f,
    F64Transmute { u: 0x4860000000000000 }.f,
    F64Transmute { u: 0x4870000000000000 }.f,
    F64Transmute { u: 0x4880000000000000 }.f,
    F64Transmute { u: 0x4890000000000000 }.f,
    F64Transmute { u: 0x48A0000000000000 }.f,
    F64Transmute { u: 0x48B0000000000000 }.f,
    F64Transmute { u: 0x48C0000000000000 }.f,
    F64Transmute { u: 0x48D0000000000000 }.f,
    F64Transmute { u: 0x48E0000000000000 }.f,
    F64Transmute { u: 0x48F0000000000000 }.f,
    F64Transmute { u: 0x4900000000000000 }.f,
    F64Transmute { u: 0x4910000000000000 }.f,
    F64Transmute { u: 0x4920000000000000 }.f,
    F64Transmute { u: 0x4930000000000000 }.f,
    F64Transmute { u: 0x4940000000000000 }.f,
    F64Transmute { u: 0x4950000000000000 }.f,
    F64Transmute { u: 0x4960000000000000 }.f,
    F64Transmute { u: 0x4970000000000000 }.f,
    F64Transmute { u: 0x4980000000000000 }.f,
    F64Transmute { u: 0x4990000000000000 }.f,
    F64Transmute { u: 0x49A0000000000000 }.f,
    F64Transmute { u: 0x49B0000000000000 }.f,
    F64Transmute { u: 0x49C0000000000000 }.f,
    F64Transmute { u: 0x49D0000000000000 }.f,
    F64Transmute { u: 0x49E0000000000000 }.f,
    F64Transmute { u: 0x49F0000000000000 }.f,
    F64Transmute { u: 0x4A00000000000000 }.f,
    F64Transmute { u: 0x4A10000000000000 }.f,
    F64Transmute { u: 0x4A20000000000000 }.f,
    F64Transmute { u: 0x4A30000000000000 }.f,
    F64Transmute { u: 0x4A40000000000000 }.f,
    F64Transmute { u: 0x4A50000000000000 }.f,
    F64Transmute { u: 0x4A60000000000000 }.f,
    F64Transmute { u: 0x4A70000000000000 }.f,
    F64Transmute { u: 0x4A80000000000000 }.f,
    F64Transmute { u: 0x4A90000000000000 }.f,
    F64Transmute { u: 0x4AA0000000000000 }.f,
    F64Transmute { u: 0x4AB0000000000000 }.f,
    F64Transmute { u: 0x4AC0000000000000 }.f,
    F64Transmute { u: 0x4AD0000000000000 }.f,
    F64Transmute { u: 0x4AE0000000000000 }.f,
    F64Transmute { u: 0x4AF0000000000000 }.f,
    F64Transmute { u: 0x4B00000000000000 }.f,
    F64Transmute { u: 0x4B10000000000000 }.f,
    F64Transmute { u: 0x4B20000000000000 }.f,
    F64Transmute { u: 0x4B30000000000000 }.f,
    F64Transmute { u: 0x4B40000000000000 }.f,
    F64Transmute { u: 0x4B50000000000000 }.f,
    F64Transmute { u: 0x4B60000000000000 }.f,
    F64Transmute { u: 0x4B70000000000000 }.f,
    F64Transmute { u: 0x4B80000000000000 }.f,
    F64Transmute { u: 0x4B90000000000000 }.f,
    F64Transmute { u: 0x4BA0000000000000 }.f,
    F64Transmute { u: 0x4BB0000000000000 }.f,
    F64Transmute { u: 0x4BC0000000000000 }.f,
    F64Transmute { u: 0x4BD0000000000000 }.f,
    F64Transmute { u: 0x4BE0000000000000 }.f,
    F64Transmute { u: 0x4BF0000000000000 }.f,
    F64Transmute { u: 0x4C00000000000000 }.f,
    F64Transmute { u: 0x4C10000000000000 }.f,
    F64Transmute { u: 0x4C20000000000000 }.f,
    F64Transmute { u: 0x4C30000000000000 }.f,
    F64Transmute { u: 0x4C40000000000000 }.f,
    F64Transmute { u: 0x4C50000000000000 }.f,
    F64Transmute { u: 0x4C60000000000000 }.f,
    F64Transmute { u: 0x4C70000000000000 }.f,
    F64Transmute { u: 0x4C80000000000000 }.f,
    F64Transmute { u: 0x4C90000000000000 }.f,
    F64Transmute { u: 0x4CA0000000000000 }.f,
    F64Transmute { u: 0x4CB0000000000000 }.f,
    F64Transmute { u: 0x4CC0000000000000 }.f,
    F64Transmute { u: 0x4CD0000000000000 }.f,
    F64Transmute { u: 0x4CE0000000000000 }.f,
    F64Transmute { u: 0x4CF0000000000000 }.f,
    F64Transmute { u: 0x4D00000000000000 }.f,
    F64Transmute { u: 0x4D10000000000000 }.f,
    F64Transmute { u: 0x4D20000000000000 }.f,
    F64Transmute { u: 0x4D30000000000000 }.f,
    F64Transmute { u: 0x4D40000000000000 }.f,
    F64Transmute { u: 0x4D50000000000000 }.f,
    F64Transmute { u: 0x4D60000000000000 }.f,
    F64Transmute { u: 0x4D70000000000000 }.f,
    F64Transmute { u: 0x4D80000000000000 }.f,
    F64Transmute { u: 0x4D90000000000000 }.f,
    F64Transmute { u: 0x4DA0000000000000 }.f,
    F64Transmute { u: 0x4DB0000000000000 }.f,
    F64Transmute { u: 0x4DC0000000000000 }.f,
    F64Transmute { u: 0x4DD0000000000000 }.f,
    F64Transmute { u: 0x4DE0000000000000 }.f,
    F64Transmute { u: 0x4DF0000000000000 }.f,
    F64Transmute { u: 0x4E00000000000000 }.f,
    F64Transmute { u: 0x4E10000000000000 }.f,
    F64Transmute { u: 0x4E20000000000000 }.f,
    F64Transmute { u: 0x4E30000000000000 }.f,
    F64Transmute { u: 0x4E40000000000000 }.f,
    F64Transmute { u: 0x4E50000000000000 }.f,
    F64Transmute { u: 0x4E60000000000000 }.f,
    F64Transmute { u: 0x4E70000000000000 }.f,
    F64Transmute { u: 0x4E80000000000000 }.f,
    F64Transmute { u: 0x4E90000000000000 }.f,
    F64Transmute { u: 0x4EA0000000000000 }.f,
    F64Transmute { u: 0x4EB0000000000000 }.f,
    F64Transmute { u: 0x4EC0000000000000 }.f,
    F64Transmute { u: 0x4ED0000000000000 }.f,
    F64Transmute { u: 0x4EE0000000000000 }.f,
    F64Transmute { u: 0x4EF0000000000000 }.f,
    F64Transmute { u: 0x4F00000000000000 }.f,
    F64Transmute { u: 0x4F10000000000000 }.f,
    F64Transmute { u: 0x4F20000000000000 }.f,
    F64Transmute { u: 0x4F30000000000000 }.f,
    F64Transmute { u: 0x4F40000000000000 }.f,
    F64Transmute { u: 0x4F50000000000000 }.f,
    F64Transmute { u: 0x4F60000000000000 }.f,
    F64Transmute { u: 0x4F70000000000000 }.f,
    F64Transmute { u: 0x4F80000000000000 }.f,
    F64Transmute { u: 0x4F90000000000000 }.f,
    F64Transmute { u: 0x4FA0000000000000 }.f,
    F64Transmute { u: 0x4FB0000000000000 }.f,
    F64Transmute { u: 0x4FC0000000000000 }.f,
    F64Transmute { u: 0x4FD0000000000000 }.f,
    F64Transmute { u: 0x4FE0000000000000 }.f,
    F64Transmute { u: 0x4FF0000000000000 }.f,
    F64Transmute { u: 0x5000000000000000 }.f,
    F64Transmute { u: 0x5010000000000000 }.f,
    F64Transmute { u: 0x5020000000000000 }.f,
    F64Transmute { u: 0x5030000000000000 }.f,
    F64Transmute { u: 0x5040000000000000 }.f,
    F64Transmute { u: 0x5050000000000000 }.f,
    F64Transmute { u: 0x5060000000000000 }.f,
    F64Transmute { u: 0x5070000000000000 }.f,
    F64Transmute { u: 0x5080000000000000 }.f,
    F64Transmute { u: 0x5090000000000000 }.f,
    F64Transmute { u: 0x50A0000000000000 }.f,
    F64Transmute { u: 0x50B0000000000000 }.f,
    F64Transmute { u: 0x50C0000000000000 }.f,
    F64Transmute { u: 0x50D0000000000000 }.f,
    F64Transmute { u: 0x50E0000000000000 }.f,
    F64Transmute { u: 0x50F0000000000000 }.f,
    F64Transmute { u: 0x5100000000000000 }.f,
    F64Transmute { u: 0x5110000000000000 }.f,
    F64Transmute { u: 0x5120000000000000 }.f,
    F64Transmute { u: 0x5130000000000000 }.f,
    F64Transmute { u: 0x5140000000000000 }.f,
    F64Transmute { u: 0x5150000000000000 }.f,
    F64Transmute { u: 0x5160000000000000 }.f,
    F64Transmute { u: 0x5170000000000000 }.f,
    F64Transmute { u: 0x5180000000000000 }.f,
    F64Transmute { u: 0x5190000000000000 }.f,
    F64Transmute { u: 0x51A0000000000000 }.f,
    F64Transmute { u: 0x51B0000000000000 }.f,
    F64Transmute { u: 0x51C0000000000000 }.f,
    F64Transmute { u: 0x51D0000000000000 }.f,
    F64Transmute { u: 0x51E0000000000000 }.f,
    F64Transmute { u: 0x51F0000000000000 }.f,
    F64Transmute { u: 0x5200000000000000 }.f,
    F64Transmute { u: 0x5210000000000000 }.f,
    F64Transmute { u: 0x5220000000000000 }.f,
    F64Transmute { u: 0x5230000000000000 }.f,
    F64Transmute { u: 0x5240000000000000 }.f,
    F64Transmute { u: 0x5250000000000000 }.f,
    F64Transmute { u: 0x5260000000000000 }.f,
    F64Transmute { u: 0x5270000000000000 }.f,
    F64Transmute { u: 0x5280000000000000 }.f,
    F64Transmute { u: 0x5290000000000000 }.f,
    F64Transmute { u: 0x52A0000000000000 }.f,
    F64Transmute { u: 0x52B0000000000000 }.f,
    F64Transmute { u: 0x52C0000000000000 }.f,
    F64Transmute { u: 0x52D0000000000000 }.f,
    F64Transmute { u: 0x52E0000000000000 }.f,
    F64Transmute { u: 0x52F0000000000000 }.f,
    F64Transmute { u: 0x5300000000000000 }.f,
    F64Transmute { u: 0x5310000000000000 }.f,
    F64Transmute { u: 0x5320000000000000 }.f,
    F64Transmute { u: 0x5330000000000000 }.f,
    F64Transmute { u: 0x5340000000000000 }.f,
    F64Transmute { u: 0x5350000000000000 }.f,
    F64Transmute { u: 0x5360000000000000 }.f,
    F64Transmute { u: 0x5370000000000000 }.f,
    F64Transmute { u: 0x5380000000000000 }.f,
    F64Transmute { u: 0x5390000000000000 }.f,
    F64Transmute { u: 0x53A0000000000000 }.f,
    F64Transmute { u: 0x53B0000000000000 }.f,
    F64Transmute { u: 0x53C0000000000000 }.f,
    F64Transmute { u: 0x53D0000000000000 }.f,
    F64Transmute { u: 0x53E0000000000000 }.f,
    F64Transmute { u: 0x53F0000000000000 }.f,
    F64Transmute { u: 0x5400000000000000 }.f,
    F64Transmute { u: 0x5410000000000000 }.f,
    F64Transmute { u: 0x5420000000000000 }.f,
    F64Transmute { u: 0x5430000000000000 }.f,
    F64Transmute { u: 0x5440000000000000 }.f,
    F64Transmute { u: 0x5450000000000000 }.f,
    F64Transmute { u: 0x5460000000000000 }.f,
    F64Transmute { u: 0x5470000000000000 }.f,
    F64Transmute { u: 0x5480000000000000 }.f,
    F64Transmute { u: 0x5490000000000000 }.f,
    F64Transmute { u: 0x54A0000000000000 }.f,
    F64Transmute { u: 0x54B0000000000000 }.f,
    F64Transmute { u: 0x54C0000000000000 }.f,
    F64Transmute { u: 0x54D0000000000000 }.f,
    F64Transmute { u: 0x54E0000000000000 }.f,
    F64Transmute { u: 0x54F0000000000000 }.f,
    F64Transmute { u: 0x5500000000000000 }.f,
    F64Transmute { u: 0x5510000000000000 }.f,
    F64Transmute { u: 0x5520000000000000 }.f,
    F64Transmute { u: 0x5530000000000000 }.f,
    F64Transmute { u: 0x5540000000000000 }.f,
    F64Transmute { u: 0x5550000000000000 }.f,
    F64Transmute { u: 0x5560000000000000 }.f,
    F64Transmute { u: 0x5570000000000000 }.f,
    F64Transmute { u: 0x5580000000000000 }.f,
    F64Transmute { u: 0x5590000000000000 }.f,
    F64Transmute { u: 0x55A0000000000000 }.f,
    F64Transmute { u: 0x55B0000000000000 }.f,
    F64Transmute { u: 0x55C0000000000000 }.f,
    F64Transmute { u: 0x55D0000000000000 }.f,
    F64Transmute { u: 0x55E0000000000000 }.f,
    F64Transmute { u: 0x55F0000000000000 }.f,
    F64Transmute { u: 0x5600000000000000 }.f,
    F64Transmute { u: 0x5610000000000000 }.f,
    F64Transmute { u: 0x5620000000000000 }.f,
    F64Transmute { u: 0x5630000000000000 }.f,
    F64Transmute { u: 0x5640000000000000 }.f,
    F64Transmute { u: 0x5650000000000000 }.f,
    F64Transmute { u: 0x5660000000000000 }.f,
    F64Transmute { u: 0x5670000000000000 }.f,
    F64Transmute { u: 0x5680000000000000 }.f,
    F64Transmute { u: 0x5690000000000000 }.f,
    F64Transmute { u: 0x56A0000000000000 }.f,
    F64Transmute { u: 0x56B0000000000000 }.f,
    F64Transmute { u: 0x56C0000000000000 }.f,
    F64Transmute { u: 0x56D0000000000000 }.f,
    F64Transmute { u: 0x56E0000000000000 }.f,
    F64Transmute { u: 0x56F0000000000000 }.f,
    F64Transmute { u: 0x5700000000000000 }.f,
    F64Transmute { u: 0x5710000000000000 }.f,
    F64Transmute { u: 0x5720000000000000 }.f,
    F64Transmute { u: 0x5730000000000000 }.f,
    F64Transmute { u: 0x5740000000000000 }.f,
    F64Transmute { u: 0x5750000000000000 }.f,
    F64Transmute { u: 0x5760000000000000 }.f,
    F64Transmute { u: 0x5770000000000000 }.f,
    F64Transmute { u: 0x5780000000000000 }.f,
    F64Transmute { u: 0x5790000000000000 }.f,
    F64Transmute { u: 0x57A0000000000000 }.f,
    F64Transmute { u: 0x57B0000000000000 }.f,
    F64Transmute { u: 0x57C0000000000000 }.f,
    F64Transmute { u: 0x57D0000000000000 }.f,
    F64Transmute { u: 0x57E0000000000000 }.f,
    F64Transmute { u: 0x57F0000000000000 }.f,
    F64Transmute { u: 0x5800000000000000 }.f,
    F64Transmute { u: 0x5810000000000000 }.f,
    F64Transmute { u: 0x5820000000000000 }.f,
    F64Transmute { u: 0x5830000000000000 }.f,
    F64Transmute { u: 0x5840000000000000 }.f,
    F64Transmute { u: 0x5850000000000000 }.f,
    F64Transmute { u: 0x5860000000000000 }.f,
    F64Transmute { u: 0x5870000000000000 }.f,
    F64Transmute { u: 0x5880000000000000 }.f,
    F64Transmute { u: 0x5890000000000000 }.f,
    F64Transmute { u: 0x58A0000000000000 }.f,
    F64Transmute { u: 0x58B0000000000000 }.f,
    F64Transmute { u: 0x58C0000000000000 }.f,
    F64Transmute { u: 0x58D0000000000000 }.f,
    F64Transmute { u: 0x58E0000000000000 }.f,
    F64Transmute { u: 0x58F0000000000000 }.f,
    F64Transmute { u: 0x5900000000000000 }.f,
    F64Transmute { u: 0x5910000000000000 }.f,
    F64Transmute { u: 0x5920000000000000 }.f,
    F64Transmute { u: 0x5930000000000000 }.f,
    F64Transmute { u: 0x5940000000000000 }.f,
    F64Transmute { u: 0x5950000000000000 }.f,
    F64Transmute { u: 0x5960000000000000 }.f,
    F64Transmute { u: 0x5970000000000000 }.f,
    F64Transmute { u: 0x5980000000000000 }.f,
    F64Transmute { u: 0x5990000000000000 }.f,
    F64Transmute { u: 0x59A0000000000000 }.f,
    F64Transmute { u: 0x59B0000000000000 }.f,
    F64Transmute { u: 0x59C0000000000000 }.f,
    F64Transmute { u: 0x59D0000000000000 }.f,
    F64Transmute { u: 0x59E0000000000000 }.f,
    F64Transmute { u: 0x59F0000000000000 }.f,
    F64Transmute { u: 0x5A00000000000000 }.f,
    F64Transmute { u: 0x5A10000000000000 }.f,
    F64Transmute { u: 0x5A20000000000000 }.f,
    F64Transmute { u: 0x5A30000000000000 }.f,
    F64Transmute { u: 0x5A40000000000000 }.f,
    F64Transmute { u: 0x5A50000000000000 }.f,
    F64Transmute { u: 0x5A60000000000000 }.f,
    F64Transmute { u: 0x5A70000000000000 }.f,
    F64Transmute { u: 0x5A80000000000000 }.f,
    F64Transmute { u: 0x5A90000000000000 }.f,
    F64Transmute { u: 0x5AA0000000000000 }.f,
    F64Transmute { u: 0x5AB0000000000000 }.f,
    F64Transmute { u: 0x5AC0000000000000 }.f,
    F64Transmute { u: 0x5AD0000000000000 }.f,
    F64Transmute { u: 0x5AE0000000000000 }.f,
    F64Transmute { u: 0x5AF0000000000000 }.f,
    F64Transmute { u: 0x5B00000000000000 }.f,
    F64Transmute { u: 0x5B10000000000000 }.f,
    F64Transmute { u: 0x5B20000000000000 }.f,
    F64Transmute { u: 0x5B30000000000000 }.f,
    F64Transmute { u: 0x5B40000000000000 }.f,
    F64Transmute { u: 0x5B50000000000000 }.f,
    F64Transmute { u: 0x5B60000000000000 }.f,
    F64Transmute { u: 0x5B70000000000000 }.f,
    F64Transmute { u: 0x5B80000000000000 }.f,
    F64Transmute { u: 0x5B90000000000000 }.f,
    F64Transmute { u: 0x5BA0000000000000 }.f,
    F64Transmute { u: 0x5BB0000000000000 }.f,
    F64Transmute { u: 0x5BC0000000000000 }.f,
    F64Transmute { u: 0x5BD0000000000000 }.f,
    F64Transmute { u: 0x5BE0000000000000 }.f,
    F64Transmute { u: 0x5BF0000000000000 }.f,
    F64Transmute { u: 0x5C00000000000000 }.f,
    F64Transmute { u: 0x5C10000000000000 }.f,
    F64Transmute { u: 0x5C20000000000000 }.f,
    F64Transmute { u: 0x5C30000000000000 }.f,
    F64Transmute { u: 0x5C40000000000000 }.f,
    F64Transmute { u: 0x5C50000000000000 }.f,
    F64Transmute { u: 0x5C60000000000000 }.f,
    F64Transmute { u: 0x5C70000000000000 }.f,
    F64Transmute { u: 0x5C80000000000000 }.f,
    F64Transmute { u: 0x5C90000000000000 }.f,
    F64Transmute { u: 0x5CA0000000000000 }.f,
    F64Transmute { u: 0x5CB0000000000000 }.f,
    F64Transmute { u: 0x5CC0000000000000 }.f,
    F64Transmute { u: 0x5CD0000000000000 }.f,
    F64Transmute { u: 0x5CE0000000000000 }.f,
    F64Transmute { u: 0x5CF0000000000000 }.f,
    F64Transmute { u: 0x5D00000000000000 }.f,
    F64Transmute { u: 0x5D10000000000000 }.f,
    F64Transmute { u: 0x5D20000000000000 }.f,
    F64Transmute { u: 0x5D30000000000000 }.f,
    F64Transmute { u: 0x5D40000000000000 }.f,
    F64Transmute { u: 0x5D50000000000000 }.f,
    F64Transmute { u: 0x5D60000000000000 }.f,
    F64Transmute { u: 0x5D70000000000000 }.f,
    F64Transmute { u: 0x5D80000000000000 }.f,
    F64Transmute { u: 0x5D90000000000000 }.f,
    F64Transmute { u: 0x5DA0000000000000 }.f,
    F64Transmute { u: 0x5DB0000000000000 }.f,
    F64Transmute { u: 0x5DC0000000000000 }.f,
    F64Transmute { u: 0x5DD0000000000000 }.f,
    F64Transmute { u: 0x5DE0000000000000 }.f,
    F64Transmute { u: 0x5DF0000000000000 }.f,
    F64Transmute { u: 0x5E00000000000000 }.f,
    F64Transmute { u: 0x5E10000000000000 }.f,
    F64Transmute { u: 0x5E20000000000000 }.f,
    F64Transmute { u: 0x5E30000000000000 }.f,
    F64Transmute { u: 0x5E40000000000000 }.f,
    F64Transmute { u: 0x5E50000000000000 }.f,
    F64Transmute { u: 0x5E60000000000000 }.f,
    F64Transmute { u: 0x5E70000000000000 }.f,
    F64Transmute { u: 0x5E80000000000000 }.f,
    F64Transmute { u: 0x5E90000000000000 }.f,
    F64Transmute { u: 0x5EA0000000000000 }.f,
    F64Transmute { u: 0x5EB0000000000000 }.f,
    F64Transmute { u: 0x5EC0000000000000 }.f,
    F64Transmute { u: 0x5ED0000000000000 }.f,
    F64Transmute { u: 0x5EE0000000000000 }.f,
    F64Transmute { u: 0x5EF0000000000000 }.f,
    F64Transmute { u: 0x5F00000000000000 }.f,
    F64Transmute { u: 0x5F10000000000000 }.f,
    F64Transmute { u: 0x5F20000000000000 }.f,
    F64Transmute { u: 0x5F30000000000000 }.f,
    F64Transmute { u: 0x5F40000000000000 }.f,
    F64Transmute { u: 0x5F50000000000000 }.f,
    F64Transmute { u: 0x5F60000000000000 }.f,
    F64Transmute { u: 0x5F70000000000000 }.f,
    F64Transmute { u: 0x5F80000000000000 }.f,
    F64Transmute { u: 0x5F90000000000000 }.f,
    F64Transmute { u: 0x5FA0000000000000 }.f,
    F64Transmute { u: 0x5FB0000000000000 }.f,
    F64Transmute { u: 0x5FC0000000000000 }.f,
    F64Transmute { u: 0x5FD0000000000000 }.f,
    F64Transmute { u: 0x5FE0000000000000 }.f,
    F64Transmute { u: 0x5FF0000000000000 }.f,
    F64Transmute { u: 0x6000000000000000 }.f,
    F64Transmute { u: 0x6010000000000000 }.f,
    F64Transmute { u: 0x6020000000000000 }.f,
    F64Transmute { u: 0x6030000000000000 }.f,
    F64Transmute { u: 0x6040000000000000 }.f,
    F64Transmute { u: 0x6050000000000000 }.f,
    F64Transmute { u: 0x6060000000000000 }.f,
    F64Transmute { u: 0x6070000000000000 }.f,
    F64Transmute { u: 0x6080000000000000 }.f,
    F64Transmute { u: 0x6090000000000000 }.f,
    F64Transmute { u: 0x60A0000000000000 }.f,
    F64Transmute { u: 0x60B0000000000000 }.f,
    F64Transmute { u: 0x60C0000000000000 }.f,
    F64Transmute { u: 0x60D0000000000000 }.f,
    F64Transmute { u: 0x60E0000000000000 }.f,
    F64Transmute { u: 0x60F0000000000000 }.f,
    F64Transmute { u: 0x6100000000000000 }.f,
    F64Transmute { u: 0x6110000000000000 }.f,
    F64Transmute { u: 0x6120000000000000 }.f,
    F64Transmute { u: 0x6130000000000000 }.f,
    F64Transmute { u: 0x6140000000000000 }.f,
    F64Transmute { u: 0x6150000000000000 }.f,
    F64Transmute { u: 0x6160000000000000 }.f,
    F64Transmute { u: 0x6170000000000000 }.f,
    F64Transmute { u: 0x6180000000000000 }.f,
    F64Transmute { u: 0x6190000000000000 }.f,
    F64Transmute { u: 0x61A0000000000000 }.f,
    F64Transmute { u: 0x61B0000000000000 }.f,
    F64Transmute { u: 0x61C0000000000000 }.f,
    F64Transmute { u: 0x61D0000000000000 }.f,
    F64Transmute { u: 0x61E0000000000000 }.f,
    F64Transmute { u: 0x61F0000000000000 }.f,
    F64Transmute { u: 0x6200000000000000 }.f,
    F64Transmute { u: 0x6210000000000000 }.f,
    F64Transmute { u: 0x6220000000000000 }.f,
    F64Transmute { u: 0x6230000000000000 }.f,
    F64Transmute { u: 0x6240000000000000 }.f,
    F64Transmute { u: 0x6250000000000000 }.f,
    F64Transmute { u: 0x6260000000000000 }.f,
    F64Transmute { u: 0x6270000000000000 }.f,
    F64Transmute { u: 0x6280000000000000 }.f,
    F64Transmute { u: 0x6290000000000000 }.f,
    F64Transmute { u: 0x62A0000000000000 }.f,
    F64Transmute { u: 0x62B0000000000000 }.f,
    F64Transmute { u: 0x62C0000000000000 }.f,
    F64Transmute { u: 0x62D0000000000000 }.f,
    F64Transmute { u: 0x62E0000000000000 }.f,
    F64Transmute { u: 0x62F0000000000000 }.f,
    F64Transmute { u: 0x6300000000000000 }.f,
    F64Transmute { u: 0x6310000000000000 }.f,
    F64Transmute { u: 0x6320000000000000 }.f,
    F64Transmute { u: 0x6330000000000000 }.f,
    F64Transmute { u: 0x6340000000000000 }.f,
    F64Transmute { u: 0x6350000000000000 }.f,
    F64Transmute { u: 0x6360000000000000 }.f,
    F64Transmute { u: 0x6370000000000000 }.f,
    F64Transmute { u: 0x6380000000000000 }.f,
    F64Transmute { u: 0x6390000000000000 }.f,
    F64Transmute { u: 0x63A0000000000000 }.f,
    F64Transmute { u: 0x63B0000000000000 }.f,
    F64Transmute { u: 0x63C0000000000000 }.f,
    F64Transmute { u: 0x63D0000000000000 }.f,
    F64Transmute { u: 0x63E0000000000000 }.f,
    F64Transmute { u: 0x63F0000000000000 }.f,
    F64Transmute { u: 0x6400000000000000 }.f,
    F64Transmute { u: 0x6410000000000000 }.f,
    F64Transmute { u: 0x6420000000000000 }.f,
    F64Transmute { u: 0x6430000000000000 }.f,
    F64Transmute { u: 0x6440000000000000 }.f,
    F64Transmute { u: 0x6450000000000000 }.f,
    F64Transmute { u: 0x6460000000000000 }.f,
    F64Transmute { u: 0x6470000000000000 }.f,
    F64Transmute { u: 0x6480000000000000 }.f,
    F64Transmute { u: 0x6490000000000000 }.f,
    F64Transmute { u: 0x64A0000000000000 }.f,
    F64Transmute { u: 0x64B0000000000000 }.f,
    F64Transmute { u: 0x64C0000000000000 }.f,
    F64Transmute { u: 0x64D0000000000000 }.f,
    F64Transmute { u: 0x64E0000000000000 }.f,
    F64Transmute { u: 0x64F0000000000000 }.f,
    F64Transmute { u: 0x6500000000000000 }.f,
    F64Transmute { u: 0x6510000000000000 }.f,
    F64Transmute { u: 0x6520000000000000 }.f,
    F64Transmute { u: 0x6530000000000000 }.f,
    F64Transmute { u: 0x6540000000000000 }.f,
    F64Transmute { u: 0x6550000000000000 }.f,
    F64Transmute { u: 0x6560000000000000 }.f,
    F64Transmute { u: 0x6570000000000000 }.f,
    F64Transmute { u: 0x6580000000000000 }.f,
    F64Transmute { u: 0x6590000000000000 }.f,
    F64Transmute { u: 0x65A0000000000000 }.f,
    F64Transmute { u: 0x65B0000000000000 }.f,
    F64Transmute { u: 0x65C0000000000000 }.f,
    F64Transmute { u: 0x65D0000000000000 }.f,
    F64Transmute { u: 0x65E0000000000000 }.f,
    F64Transmute { u: 0x65F0000000000000 }.f,
    F64Transmute { u: 0x6600000000000000 }.f,
    F64Transmute { u: 0x6610000000000000 }.f,
    F64Transmute { u: 0x6620000000000000 }.f,
    F64Transmute { u: 0x6630000000000000 }.f,
    F64Transmute { u: 0x6640000000000000 }.f,
    F64Transmute { u: 0x6650000000000000 }.f,
    F64Transmute { u: 0x6660000000000000 }.f,
    F64Transmute { u: 0x6670000000000000 }.f,
    F64Transmute { u: 0x6680000000000000 }.f,
    F64Transmute { u: 0x6690000000000000 }.f,
    F64Transmute { u: 0x66A0000000000000 }.f,
    F64Transmute { u: 0x66B0000000000000 }.f,
    F64Transmute { u: 0x66C0000000000000 }.f,
    F64Transmute { u: 0x66D0000000000000 }.f,
    F64Transmute { u: 0x66E0000000000000 }.f,
    F64Transmute { u: 0x66F0000000000000 }.f,
    F64Transmute { u: 0x6700000000000000 }.f,
    F64Transmute { u: 0x6710000000000000 }.f,
    F64Transmute { u: 0x6720000000000000 }.f,
    F64Transmute { u: 0x6730000000000000 }.f,
    F64Transmute { u: 0x6740000000000000 }.f,
    F64Transmute { u: 0x6750000000000000 }.f,
    F64Transmute { u: 0x6760000000000000 }.f,
    F64Transmute { u: 0x6770000000000000 }.f,
    F64Transmute { u: 0x6780000000000000 }.f,
    F64Transmute { u: 0x6790000000000000 }.f,
    F64Transmute { u: 0x67A0000000000000 }.f,
    F64Transmute { u: 0x67B0000000000000 }.f,
    F64Transmute { u: 0x67C0000000000000 }.f,
    F64Transmute { u: 0x67D0000000000000 }.f,
    F64Transmute { u: 0x67E0000000000000 }.f,
    F64Transmute { u: 0x67F0000000000000 }.f,
    F64Transmute { u: 0x6800000000000000 }.f,
    F64Transmute { u: 0x6810000000000000 }.f,
    F64Transmute { u: 0x6820000000000000 }.f,
    F64Transmute { u: 0x6830000000000000 }.f,
    F64Transmute { u: 0x6840000000000000 }.f,
    F64Transmute { u: 0x6850000000000000 }.f,
    F64Transmute { u: 0x6860000000000000 }.f,
    F64Transmute { u: 0x6870000000000000 }.f,
    F64Transmute { u: 0x6880000000000000 }.f,
    F64Transmute { u: 0x6890000000000000 }.f,
    F64Transmute { u: 0x68A0000000000000 }.f,
    F64Transmute { u: 0x68B0000000000000 }.f,
    F64Transmute { u: 0x68C0000000000000 }.f,
    F64Transmute { u: 0x68D0000000000000 }.f,
    F64Transmute { u: 0x68E0000000000000 }.f,
    F64Transmute { u: 0x68F0000000000000 }.f,
    F64Transmute { u: 0x6900000000000000 }.f,
    F64Transmute { u: 0x6910000000000000 }.f,
    F64Transmute { u: 0x6920000000000000 }.f,
    F64Transmute { u: 0x6930000000000000 }.f,
    F64Transmute { u: 0x6940000000000000 }.f,
    F64Transmute { u: 0x6950000000000000 }.f,
    F64Transmute { u: 0x6960000000000000 }.f,
    F64Transmute { u: 0x6970000000000000 }.f,
    F64Transmute { u: 0x6980000000000000 }.f,
    F64Transmute { u: 0x6990000000000000 }.f,
    F64Transmute { u: 0x69A0000000000000 }.f,
    F64Transmute { u: 0x69B0000000000000 }.f,
    F64Transmute { u: 0x69C0000000000000 }.f,
    F64Transmute { u: 0x69D0000000000000 }.f,
    F64Transmute { u: 0x69E0000000000000 }.f,
    F64Transmute { u: 0x69F0000000000000 }.f,
    F64Transmute { u: 0x6A00000000000000 }.f,
    F64Transmute { u: 0x6A10000000000000 }.f,
    F64Transmute { u: 0x6A20000000000000 }.f,
    F64Transmute { u: 0x6A30000000000000 }.f,
    F64Transmute { u: 0x6A40000000000000 }.f,
    F64Transmute { u: 0x6A50000000000000 }.f,
    F64Transmute { u: 0x6A60000000000000 }.f,
    F64Transmute { u: 0x6A70000000000000 }.f,
    F64Transmute { u: 0x6A80000000000000 }.f,
    F64Transmute { u: 0x6A90000000000000 }.f,
    F64Transmute { u: 0x6AA0000000000000 }.f,
    F64Transmute { u: 0x6AB0000000000000 }.f,
    F64Transmute { u: 0x6AC0000000000000 }.f,
    F64Transmute { u: 0x6AD0000000000000 }.f,
    F64Transmute { u: 0x6AE0000000000000 }.f,
    F64Transmute { u: 0x6AF0000000000000 }.f,
    F64Transmute { u: 0x6B00000000000000 }.f,
    F64Transmute { u: 0x6B10000000000000 }.f,
    F64Transmute { u: 0x6B20000000000000 }.f,
    F64Transmute { u: 0x6B30000000000000 }.f,
    F64Transmute { u: 0x6B40000000000000 }.f,
    F64Transmute { u: 0x6B50000000000000 }.f,
    F64Transmute { u: 0x6B60000000000000 }.f,
    F64Transmute { u: 0x6B70000000000000 }.f,
    F64Transmute { u: 0x6B80000000000000 }.f,
    F64Transmute { u: 0x6B90000000000000 }.f,
    F64Transmute { u: 0x6BA0000000000000 }.f,
    F64Transmute { u: 0x6BB0000000000000 }.f,
    F64Transmute { u: 0x6BC0000000000000 }.f,
    F64Transmute { u: 0x6BD0000000000000 }.f,
    F64Transmute { u: 0x6BE0000000000000 }.f,
    F64Transmute { u: 0x6BF0000000000000 }.f,
    F64Transmute { u: 0x6C00000000000000 }.f,
    F64Transmute { u: 0x6C10000000000000 }.f,
    F64Transmute { u: 0x6C20000000000000 }.f,
    F64Transmute { u: 0x6C30000000000000 }.f,
    F64Transmute { u: 0x6C40000000000000 }.f,
    F64Transmute { u: 0x6C50000000000000 }.f,
    F64Transmute { u: 0x6C60000000000000 }.f,
    F64Transmute { u: 0x6C70000000000000 }.f,
    F64Transmute { u: 0x6C80000000000000 }.f,
    F64Transmute { u: 0x6C90000000000000 }.f,
    F64Transmute { u: 0x6CA0000000000000 }.f,
    F64Transmute { u: 0x6CB0000000000000 }.f,
    F64Transmute { u: 0x6CC0000000000000 }.f,
    F64Transmute { u: 0x6CD0000000000000 }.f,
    F64Transmute { u: 0x6CE0000000000000 }.f,
    F64Transmute { u: 0x6CF0000000000000 }.f,
    F64Transmute { u: 0x6D00000000000000 }.f,
    F64Transmute { u: 0x6D10000000000000 }.f,
    F64Transmute { u: 0x6D20000000000000 }.f,
    F64Transmute { u: 0x6D30000000000000 }.f,
    F64Transmute { u: 0x6D40000000000000 }.f,
    F64Transmute { u: 0x6D50000000000000 }.f,
    F64Transmute { u: 0x6D60000000000000 }.f,
    F64Transmute { u: 0x6D70000000000000 }.f,
    F64Transmute { u: 0x6D80000000000000 }.f,
    F64Transmute { u: 0x6D90000000000000 }.f,
    F64Transmute { u: 0x6DA0000000000000 }.f,
    F64Transmute { u: 0x6DB0000000000000 }.f,
    F64Transmute { u: 0x6DC0000000000000 }.f,
    F64Transmute { u: 0x6DD0000000000000 }.f,
    F64Transmute { u: 0x6DE0000000000000 }.f,
    F64Transmute { u: 0x6DF0000000000000 }.f,
    F64Transmute { u: 0x6E00000000000000 }.f,
    F64Transmute { u: 0x6E10000000000000 }.f,
    F64Transmute { u: 0x6E20000000000000 }.f,
    F64Transmute { u: 0x6E30000000000000 }.f,
    F64Transmute { u: 0x6E40000000000000 }.f,
    F64Transmute { u: 0x6E50000000000000 }.f,
    F64Transmute { u: 0x6E60000000000000 }.f,
    F64Transmute { u: 0x6E70000000000000 }.f,
    F64Transmute { u: 0x6E80000000000000 }.f,
    F64Transmute { u: 0x6E90000000000000 }.f,
    F64Transmute { u: 0x6EA0000000000000 }.f,
    F64Transmute { u: 0x6EB0000000000000 }.f,
    F64Transmute { u: 0x6EC0000000000000 }.f,
    F64Transmute { u: 0x6ED0000000000000 }.f,
    F64Transmute { u: 0x6EE0000000000000 }.f,
    F64Transmute { u: 0x6EF0000000000000 }.f,
    F64Transmute { u: 0x6F00000000000000 }.f,
    F64Transmute { u: 0x6F10000000000000 }.f,
    F64Transmute { u: 0x6F20000000000000 }.f,
    F64Transmute { u: 0x6F30000000000000 }.f,
    F64Transmute { u: 0x6F40000000000000 }.f,
    F64Transmute { u: 0x6F50000000000000 }.f,
    F64Transmute { u: 0x6F60000000000000 }.f,
    F64Transmute { u: 0x6F70000000000000 }.f,
    F64Transmute { u: 0x6F80000000000000 }.f,
    F64Transmute { u: 0x6F90000000000000 }.f,
    F64Transmute { u: 0x6FA0000000000000 }.f,
    F64Transmute { u: 0x6FB0000000000000 }.f,
    F64Transmute { u: 0x6FC0000000000000 }.f,
    F64Transmute { u: 0x6FD0000000000000 }.f,
    F64Transmute { u: 0x6FE0000000000000 }.f,
    F64Transmute { u: 0x6FF0000000000000 }.f,
    F64Transmute { u: 0x7000000000000000 }.f,
    F64Transmute { u: 0x7010000000000000 }.f,
    F64Transmute { u: 0x7020000000000000 }.f,
    F64Transmute { u: 0x7030000000000000 }.f,
    F64Transmute { u: 0x7040000000000000 }.f,
    F64Transmute { u: 0x7050000000000000 }.f,
    F64Transmute { u: 0x7060000000000000 }.f,
    F64Transmute { u: 0x7070000000000000 }.f,
    F64Transmute { u: 0x7080000000000000 }.f,
    F64Transmute { u: 0x7090000000000000 }.f,
    F64Transmute { u: 0x70A0000000000000 }.f,
    F64Transmute { u: 0x70B0000000000000 }.f,
    F64Transmute { u: 0x70C0000000000000 }.f,
    F64Transmute { u: 0x70D0000000000000 }.f,
    F64Transmute { u: 0x70E0000000000000 }.f,
    F64Transmute { u: 0x70F0000000000000 }.f,
    F64Transmute { u: 0x7100000000000000 }.f,
    F64Transmute { u: 0x7110000000000000 }.f,
    F64Transmute { u: 0x7120000000000000 }.f,
    F64Transmute { u: 0x7130000000000000 }.f,
    F64Transmute { u: 0x7140000000000000 }.f,
    F64Transmute { u: 0x7150000000000000 }.f,
    F64Transmute { u: 0x7160000000000000 }.f,
    F64Transmute { u: 0x7170000000000000 }.f,
    F64Transmute { u: 0x7180000000000000 }.f,
    F64Transmute { u: 0x7190000000000000 }.f,
    F64Transmute { u: 0x71A0000000000000 }.f,
    F64Transmute { u: 0x71B0000000000000 }.f,
    F64Transmute { u: 0x71C0000000000000 }.f,
    F64Transmute { u: 0x71D0000000000000 }.f,
    F64Transmute { u: 0x71E0000000000000 }.f,
    F64Transmute { u: 0x71F0000000000000 }.f,
    F64Transmute { u: 0x7200000000000000 }.f,
    F64Transmute { u: 0x7210000000000000 }.f,
    F64Transmute { u: 0x7220000000000000 }.f,
    F64Transmute { u: 0x7230000000000000 }.f,
    F64Transmute { u: 0x7240000000000000 }.f,
    F64Transmute { u: 0x7250000000000000 }.f,
    F64Transmute { u: 0x7260000000000000 }.f,
    F64Transmute { u: 0x7270000000000000 }.f,
    F64Transmute { u: 0x7280000000000000 }.f,
    F64Transmute { u: 0x7290000000000000 }.f,
    F64Transmute { u: 0x72A0000000000000 }.f,
    F64Transmute { u: 0x72B0000000000000 }.f,
    F64Transmute { u: 0x72C0000000000000 }.f,
    F64Transmute { u: 0x72D0000000000000 }.f,
    F64Transmute { u: 0x72E0000000000000 }.f,
    F64Transmute { u: 0x72F0000000000000 }.f,
    F64Transmute { u: 0x7300000000000000 }.f,
    F64Transmute { u: 0x7310000000000000 }.f,
    F64Transmute { u: 0x7320000000000000 }.f,
    F64Transmute { u: 0x7330000000000000 }.f,
    F64Transmute { u: 0x7340000000000000 }.f,
    F64Transmute { u: 0x7350000000000000 }.f,
    F64Transmute { u: 0x7360000000000000 }.f,
    F64Transmute { u: 0x7370000000000000 }.f,
    F64Transmute { u: 0x7380000000000000 }.f,
    F64Transmute { u: 0x7390000000000000 }.f,
    F64Transmute { u: 0x73A0000000000000 }.f,
    F64Transmute { u: 0x73B0000000000000 }.f,
    F64Transmute { u: 0x73C0000000000000 }.f,
    F64Transmute { u: 0x73D0000000000000 }.f,
    F64Transmute { u: 0x73E0000000000000 }.f,
    F64Transmute { u: 0x73F0000000000000 }.f,
    F64Transmute { u: 0x7400000000000000 }.f,
    F64Transmute { u: 0x7410000000000000 }.f,
    F64Transmute { u: 0x7420000000000000 }.f,
    F64Transmute { u: 0x7430000000000000 }.f,
    F64Transmute { u: 0x7440000000000000 }.f,
    F64Transmute { u: 0x7450000000000000 }.f,
    F64Transmute { u: 0x7460000000000000 }.f,
    F64Transmute { u: 0x7470000000000000 }.f,
    F64Transmute { u: 0x7480000000000000 }.f,
    F64Transmute { u: 0x7490000000000000 }.f,
    F64Transmute { u: 0x74A0000000000000 }.f,
    F64Transmute { u: 0x74B0000000000000 }.f,
    F64Transmute { u: 0x74C0000000000000 }.f,
    F64Transmute { u: 0x74D0000000000000 }.f,
    F64Transmute { u: 0x74E0000000000000 }.f,
    F64Transmute { u: 0x74F0000000000000 }.f,
    F64Transmute { u: 0x7500000000000000 }.f,
    F64Transmute { u: 0x7510000000000000 }.f,
    F64Transmute { u: 0x7520000000000000 }.f,
    F64Transmute { u: 0x7530000000000000 }.f,
    F64Transmute { u: 0x7540000000000000 }.f,
    F64Transmute { u: 0x7550000000000000 }.f,
    F64Transmute { u: 0x7560000000000000 }.f,
    F64Transmute { u: 0x7570000000000000 }.f,
    F64Transmute { u: 0x7580000000000000 }.f,
    F64Transmute { u: 0x7590000000000000 }.f,
    F64Transmute { u: 0x75A0000000000000 }.f,
    F64Transmute { u: 0x75B0000000000000 }.f,
    F64Transmute { u: 0x75C0000000000000 }.f,
    F64Transmute { u: 0x75D0000000000000 }.f,
    F64Transmute { u: 0x75E0000000000000 }.f,
    F64Transmute { u: 0x75F0000000000000 }.f,
    F64Transmute { u: 0x7600000000000000 }.f,
    F64Transmute { u: 0x7610000000000000 }.f,
    F64Transmute { u: 0x7620000000000000 }.f,
    F64Transmute { u: 0x7630000000000000 }.f,
    F64Transmute { u: 0x7640000000000000 }.f,
    F64Transmute { u: 0x7650000000000000 }.f,
    F64Transmute { u: 0x7660000000000000 }.f,
    F64Transmute { u: 0x7670000000000000 }.f,
    F64Transmute { u: 0x7680000000000000 }.f,
    F64Transmute { u: 0x7690000000000000 }.f,
    F64Transmute { u: 0x76A0000000000000 }.f,
    F64Transmute { u: 0x76B0000000000000 }.f,
    F64Transmute { u: 0x76C0000000000000 }.f,
    F64Transmute { u: 0x76D0000000000000 }.f,
    F64Transmute { u: 0x76E0000000000000 }.f,
    F64Transmute { u: 0x76F0000000000000 }.f,
    F64Transmute { u: 0x7700000000000000 }.f,
    F64Transmute { u: 0x7710000000000000 }.f,
    F64Transmute { u: 0x7720000000000000 }.f,
    F64Transmute { u: 0x7730000000000000 }.f,
    F64Transmute { u: 0x7740000000000000 }.f,
    F64Transmute { u: 0x7750000000000000 }.f,
    F64Transmute { u: 0x7760000000000000 }.f,
    F64Transmute { u: 0x7770000000000000 }.f,
    F64Transmute { u: 0x7780000000000000 }.f,
    F64Transmute { u: 0x7790000000000000 }.f,
    F64Transmute { u: 0x77A0000000000000 }.f,
    F64Transmute { u: 0x77B0000000000000 }.f,
    F64Transmute { u: 0x77C0000000000000 }.f,
    F64Transmute { u: 0x77D0000000000000 }.f,
    F64Transmute { u: 0x77E0000000000000 }.f,
    F64Transmute { u: 0x77F0000000000000 }.f,
    F64Transmute { u: 0x7800000000000000 }.f,
    F64Transmute { u: 0x7810000000000000 }.f,
    F64Transmute { u: 0x7820000000000000 }.f,
    F64Transmute { u: 0x7830000000000000 }.f,
    F64Transmute { u: 0x7840000000000000 }.f,
    F64Transmute { u: 0x7850000000000000 }.f,
    F64Transmute { u: 0x7860000000000000 }.f,
    F64Transmute { u: 0x7870000000000000 }.f,
    F64Transmute { u: 0x7880000000000000 }.f,
    F64Transmute { u: 0x7890000000000000 }.f,
    F64Transmute { u: 0x78A0000000000000 }.f,
    F64Transmute { u: 0x78B0000000000000 }.f,
    F64Transmute { u: 0x78C0000000000000 }.f,
    F64Transmute { u: 0x78D0000000000000 }.f,
    F64Transmute { u: 0x78E0000000000000 }.f,
    F64Transmute { u: 0x78F0000000000000 }.f,
    F64Transmute { u: 0x7900000000000000 }.f,
    F64Transmute { u: 0x7910000000000000 }.f,
    F64Transmute { u: 0x7920000000000000 }.f,
    F64Transmute { u: 0x7930000000000000 }.f,
    F64Transmute { u: 0x7940000000000000 }.f,
    F64Transmute { u: 0x7950000000000000 }.f,
    F64Transmute { u: 0x7960000000000000 }.f,
    F64Transmute { u: 0x7970000000000000 }.f,
    F64Transmute { u: 0x7980000000000000 }.f,
    F64Transmute { u: 0x7990000000000000 }.f,
    F64Transmute { u: 0x79A0000000000000 }.f,
    F64Transmute { u: 0x79B0000000000000 }.f,
    F64Transmute { u: 0x79C0000000000000 }.f,
    F64Transmute { u: 0x79D0000000000000 }.f,
    F64Transmute { u: 0x79E0000000000000 }.f,
    F64Transmute { u: 0x79F0000000000000 }.f,
    F64Transmute { u: 0x7A00000000000000 }.f,
    F64Transmute { u: 0x7A10000000000000 }.f,
    F64Transmute { u: 0x7A20000000000000 }.f,
    F64Transmute { u: 0x7A30000000000000 }.f,
    F64Transmute { u: 0x7A40000000000000 }.f,
    F64Transmute { u: 0x7A50000000000000 }.f,
    F64Transmute { u: 0x7A60000000000000 }.f,
    F64Transmute { u: 0x7A70000000000000 }.f,
    F64Transmute { u: 0x7A80000000000000 }.f,
    F64Transmute { u: 0x7A90000000000000 }.f,
    F64Transmute { u: 0x7AA0000000000000 }.f,
    F64Transmute { u: 0x7AB0000000000000 }.f,
    F64Transmute { u: 0x7AC0000000000000 }.f,
    F64Transmute { u: 0x7AD0000000000000 }.f,
    F64Transmute { u: 0x7AE0000000000000 }.f,
    F64Transmute { u: 0x7AF0000000000000 }.f,
    F64Transmute { u: 0x7B00000000000000 }.f,
    F64Transmute { u: 0x7B10000000000000 }.f,
    F64Transmute { u: 0x7B20000000000000 }.f,
    F64Transmute { u: 0x7B30000000000000 }.f,
    F64Transmute { u: 0x7B40000000000000 }.f,
    F64Transmute { u: 0x7B50000000000000 }.f,
    F64Transmute { u: 0x7B60000000000000 }.f,
    F64Transmute { u: 0x7B70000000000000 }.f,
    F64Transmute { u: 0x7B80000000000000 }.f,
    F64Transmute { u: 0x7B90000000000000 }.f,
    F64Transmute { u: 0x7BA0000000000000 }.f,
    F64Transmute { u: 0x7BB0000000000000 }.f,
    F64Transmute { u: 0x7BC0000000000000 }.f,
    F64Transmute { u: 0x7BD0000000000000 }.f,
    F64Transmute { u: 0x7BE0000000000000 }.f,
    F64Transmute { u: 0x7BF0000000000000 }.f,
    F64Transmute { u: 0x7C00000000000000 }.f,
    F64Transmute { u: 0x7C10000000000000 }.f,
    F64Transmute { u: 0x7C20000000000000 }.f,
    F64Transmute { u: 0x7C30000000000000 }.f,
    F64Transmute { u: 0x7C40000000000000 }.f,
    F64Transmute { u: 0x7C50000000000000 }.f,
    F64Transmute { u: 0x7C60000000000000 }.f,
    F64Transmute { u: 0x7C70000000000000 }.f,
    F64Transmute { u: 0x7C80000000000000 }.f,
    F64Transmute { u: 0x7C90000000000000 }.f,
    F64Transmute { u: 0x7CA0000000000000 }.f,
    F64Transmute { u: 0x7CB0000000000000 }.f,
    F64Transmute { u: 0x7CC0000000000000 }.f,
    F64Transmute { u: 0x7CD0000000000000 }.f,
    F64Transmute { u: 0x7CE0000000000000 }.f,
    F64Transmute { u: 0x7CF0000000000000 }.f,
    F64Transmute { u: 0x7D00000000000000 }.f,
    F64Transmute { u: 0x7D10000000000000 }.f,
    F64Transmute { u: 0x7D20000000000000 }.f,
    F64Transmute { u: 0x7D30000000000000 }.f,
    F64Transmute { u: 0x7D40000000000000 }.f,
    F64Transmute { u: 0x7D50000000000000 }.f,
    F64Transmute { u: 0x7D60000000000000 }.f,
    F64Transmute { u: 0x7D70000000000000 }.f,
    F64Transmute { u: 0x7D80000000000000 }.f,
    F64Transmute { u: 0x7D90000000000000 }.f,
    F64Transmute { u: 0x7DA0000000000000 }.f,
    F64Transmute { u: 0x7DB0000000000000 }.f,
    F64Transmute { u: 0x7DC0000000000000 }.f,
    F64Transmute { u: 0x7DD0000000000000 }.f,
    F64Transmute { u: 0x7DE0000000000000 }.f,
    F64Transmute { u: 0x7DF0000000000000 }.f,
    F64Transmute { u: 0x7E00000000000000 }.f,
    F64Transmute { u: 0x7E10000000000000 }.f,
    F64Transmute { u: 0x7E20000000000000 }.f,
    F64Transmute { u: 0x7E30000000000000 }.f,
    F64Transmute { u: 0x7E40000000000000 }.f,
    F64Transmute { u: 0x7E50000000000000 }.f,
    F64Transmute { u: 0x7E60000000000000 }.f,
    F64Transmute { u: 0x7E70000000000000 }.f,
    F64Transmute { u: 0x7E80000000000000 }.f,
    F64Transmute { u: 0x7E90000000000000 }.f,
    F64Transmute { u: 0x7EA0000000000000 }.f,
    F64Transmute { u: 0x7EB0000000000000 }.f,
    F64Transmute { u: 0x7EC0000000000000 }.f,
    F64Transmute { u: 0x7ED0000000000000 }.f,
    F64Transmute { u: 0x7EE0000000000000 }.f,
    F64Transmute { u: 0x7EF0000000000000 }.f,
    F64Transmute { u: 0x7F00000000000000 }.f,
    F64Transmute { u: 0x7F10000000000000 }.f,
    F64Transmute { u: 0x7F20000000000000 }.f,
    F64Transmute { u: 0x7F30000000000000 }.f,
    F64Transmute { u: 0x7F40000000000000 }.f,
    F64Transmute { u: 0x7F50000000000000 }.f,
    F64Transmute { u: 0x7F60000000000000 }.f,
    F64Transmute { u: 0x7F70000000000000 }.f,
    F64Transmute { u: 0x7F80000000000000 }.f,
    F64Transmute { u: 0x7F90000000000000 }.f,
    F64Transmute { u: 0x7FA0000000000000 }.f,
    F64Transmute { u: 0x7FB0000000000000 }.f,
    F64Transmute { u: 0x7FC0000000000000 }.f,
    F64Transmute { u: 0x7FD0000000000000 }.f,
    F64Transmute { u: 0x7FE0000000000000 }.f,
]};

/// Precalculated values of radix**i for i in range [0, arr.len()-1].
/// Each value can be **exactly** represented as that type.
const F64_POW10: [f64; 23] = [1.0, 10.0, 100.0, 1000.0, 10000.0, 100000.0, 1000000.0, 10000000.0, 100000000.0, 1000000000.0, 10000000000.0, 100000000000.0, 1000000000000.0, 10000000000000.0, 100000000000000.0, 1000000000000000.0, 10000000000000000.0, 100000000000000000.0, 1000000000000000000.0, 10000000000000000000.0, 100000000000000000000.0, 1000000000000000000000.0, 10000000000000000000000.0];

#[cfg(feature = "radix")]
const F64_POW3: [f64; 34] = [1.0, 3.0, 9.0, 27.0, 81.0, 243.0, 729.0, 2187.0, 6561.0, 19683.0, 59049.0, 177147.0, 531441.0, 1594323.0, 4782969.0, 14348907.0, 43046721.0, 129140163.0, 387420489.0, 1162261467.0, 3486784401.0, 10460353203.0, 31381059609.0, 94143178827.0, 282429536481.0, 847288609443.0, 2541865828329.0, 7625597484987.0, 22876792454961.0, 68630377364883.0, 205891132094649.0, 617673396283947.0, 1853020188851841.0, 5559060566555523.0];

#[cfg(feature = "radix")]
const F64_POW5: [f64; 23] = [1.0, 5.0, 25.0, 125.0, 625.0, 3125.0, 15625.0, 78125.0, 390625.0, 1953125.0, 9765625.0, 48828125.0, 244140625.0, 1220703125.0, 6103515625.0, 30517578125.0, 152587890625.0, 762939453125.0, 3814697265625.0, 19073486328125.0, 95367431640625.0, 476837158203125.0, 2384185791015625.0];

#[cfg(feature = "radix")]
const F64_POW6: [f64; 34] = [1.0, 6.0, 36.0, 216.0, 1296.0, 7776.0, 46656.0, 279936.0, 1679616.0, 10077696.0, 60466176.0, 362797056.0, 2176782336.0, 13060694016.0, 78364164096.0, 470184984576.0, 2821109907456.0, 16926659444736.0, 101559956668416.0, 609359740010496.0, 3656158440062976.0, 21936950640377856.0, 131621703842267136.0, 789730223053602816.0, 4738381338321616896.0, 28430288029929701376.0, 170581728179578208256.0, 1023490369077469249536.0, 6140942214464815497216.0, 36845653286788892983296.0, 221073919720733357899776.0, 1326443518324400147398656.0, 7958661109946400884391936.0, 47751966659678405306351616.0];

#[cfg(feature = "radix")]
const F64_POW7: [f64; 19] = [1.0, 7.0, 49.0, 343.0, 2401.0, 16807.0, 117649.0, 823543.0, 5764801.0, 40353607.0, 282475249.0, 1977326743.0, 13841287201.0, 96889010407.0, 678223072849.0, 4747561509943.0, 33232930569601.0, 232630513987207.0, 1628413597910449.0];

#[cfg(feature = "radix")]
const F64_POW9: [f64; 17] = [1.0, 9.0, 81.0, 729.0, 6561.0, 59049.0, 531441.0, 4782969.0, 43046721.0, 387420489.0, 3486784401.0, 31381059609.0, 282429536481.0, 2541865828329.0, 22876792454961.0, 205891132094649.0, 1853020188851841.0];

#[cfg(feature = "radix")]
const F64_POW11: [f64; 16] = [1.0, 11.0, 121.0, 1331.0, 14641.0, 161051.0, 1771561.0, 19487171.0, 214358881.0, 2357947691.0, 25937424601.0, 285311670611.0, 3138428376721.0, 34522712143931.0, 379749833583241.0, 4177248169415651.0];

#[cfg(feature = "radix")]
const F64_POW12: [f64; 34] = [1.0, 12.0, 144.0, 1728.0, 20736.0, 248832.0, 2985984.0, 35831808.0, 429981696.0, 5159780352.0, 61917364224.0, 743008370688.0, 8916100448256.0, 106993205379072.0, 1283918464548864.0, 15407021574586368.0, 184884258895036416.0, 2218611106740436992.0, 26623333280885243904.0, 319479999370622926848.0, 3833759992447475122176.0, 46005119909369701466112.0, 552061438912436417593344.0, 6624737266949237011120128.0, 79496847203390844133441536.0, 953962166440690129601298432.0, 11447545997288281555215581184.0, 137370551967459378662586974208.0, 1648446623609512543951043690496.0, 19781359483314150527412524285952.0, 237376313799769806328950291431424.0, 2848515765597237675947403497177088.0, 34182189187166852111368841966125056.0, 410186270246002225336426103593500672.0];

#[cfg(feature = "radix")]
const F64_POW13: [f64; 15] = [1.0, 13.0, 169.0, 2197.0, 28561.0, 371293.0, 4826809.0, 62748517.0, 815730721.0, 10604499373.0, 137858491849.0, 1792160394037.0, 23298085122481.0, 302875106592253.0, 3937376385699289.0];

#[cfg(feature = "radix")]
const F64_POW14: [f64; 19] = [1.0, 14.0, 196.0, 2744.0, 38416.0, 537824.0, 7529536.0, 105413504.0, 1475789056.0, 20661046784.0, 289254654976.0, 4049565169664.0, 56693912375296.0, 793714773254144.0, 11112006825558016.0, 155568095557812224.0, 2177953337809371136.0, 30491346729331195904.0, 426878854210636742656.0];

#[cfg(feature = "radix")]
const F64_POW15: [f64; 14] = [1.0, 15.0, 225.0, 3375.0, 50625.0, 759375.0, 11390625.0, 170859375.0, 2562890625.0, 38443359375.0, 576650390625.0, 8649755859375.0, 129746337890625.0, 1946195068359375.0];

#[cfg(feature = "radix")]
const F64_POW17: [f64; 13] = [1.0, 17.0, 289.0, 4913.0, 83521.0, 1419857.0, 24137569.0, 410338673.0, 6975757441.0, 118587876497.0, 2015993900449.0, 34271896307633.0, 582622237229761.0];

#[cfg(feature = "radix")]
const F64_POW18: [f64; 17] = [1.0, 18.0, 324.0, 5832.0, 104976.0, 1889568.0, 34012224.0, 612220032.0, 11019960576.0, 198359290368.0, 3570467226624.0, 64268410079232.0, 1156831381426176.0, 20822964865671168.0, 374813367582081024.0, 6746640616477458432.0, 121439531096594251776.0];

#[cfg(feature = "radix")]
const F64_POW19: [f64; 13] = [1.0, 19.0, 361.0, 6859.0, 130321.0, 2476099.0, 47045881.0, 893871739.0, 16983563041.0, 322687697779.0, 6131066257801.0, 116490258898219.0, 2213314919066161.0];

#[cfg(feature = "radix")]
const F64_POW20: [f64; 23] = [1.0, 20.0, 400.0, 8000.0, 160000.0, 3200000.0, 64000000.0, 1280000000.0, 25600000000.0, 512000000000.0, 10240000000000.0, 204800000000000.0, 4096000000000000.0, 81920000000000000.0, 1638400000000000000.0, 32768000000000000000.0, 655360000000000000000.0, 13107200000000000000000.0, 262144000000000000000000.0, 5242880000000000000000000.0, 104857600000000000000000000.0, 2097152000000000000000000000.0, 41943040000000000000000000000.0];

#[cfg(feature = "radix")]
const F64_POW21: [f64; 13] = [1.0, 21.0, 441.0, 9261.0, 194481.0, 4084101.0, 85766121.0, 1801088541.0, 37822859361.0, 794280046581.0, 16679880978201.0, 350277500542221.0, 7355827511386641.0];

#[cfg(feature = "radix")]
const F64_POW22: [f64; 16] = [1.0, 22.0, 484.0, 10648.0, 234256.0, 5153632.0, 113379904.0, 2494357888.0, 54875873536.0, 1207269217792.0, 26559922791424.0, 584318301411328.0, 12855002631049216.0, 282810057883082752.0, 6221821273427820544.0, 136880068015412051968.0];

#[cfg(feature = "radix")]
const F64_POW23: [f64; 12] = [1.0, 23.0, 529.0, 12167.0, 279841.0, 6436343.0, 148035889.0, 3404825447.0, 78310985281.0, 1801152661463.0, 41426511213649.0, 952809757913927.0];

#[cfg(feature = "radix")]
const F64_POW24: [f64; 34] = [1.0, 24.0, 576.0, 13824.0, 331776.0, 7962624.0, 191102976.0, 4586471424.0, 110075314176.0, 2641807540224.0, 63403380965376.0, 1521681143169024.0, 36520347436056576.0, 876488338465357824.0, 21035720123168587776.0, 504857282956046106624.0, 12116574790945106558976.0, 290797794982682557415424.0, 6979147079584381377970176.0, 167499529910025153071284224.0, 4019988717840603673710821376.0, 96479729228174488169059713024.0, 2315513501476187716057433112576.0, 55572324035428505185378394701824.0, 1333735776850284124449081472843776.0, 32009658644406818986777955348250624.0, 768231807465763655682670928358014976.0, 18437563379178327736384102280592359424.0, 442501521100279865673218454734216626176.0, 10620036506406716776157242913621199028224.0, 254880876153761202627773829926908776677376.0, 6117141027690268863066571918245810640257024.0, 146811384664566452713597726037899455366168576.0, 3523473231949594865126345424909586928788045824.0];

#[cfg(feature = "radix")]
const F64_POW25: [f64; 12] = [1.0, 25.0, 625.0, 15625.0, 390625.0, 9765625.0, 244140625.0, 6103515625.0, 152587890625.0, 3814697265625.0, 95367431640625.0, 2384185791015625.0];

#[cfg(feature = "radix")]
const F64_POW26: [f64; 15] = [1.0, 26.0, 676.0, 17576.0, 456976.0, 11881376.0, 308915776.0, 8031810176.0, 208827064576.0, 5429503678976.0, 141167095653376.0, 3670344486987776.0, 95428956661682176.0, 2481152873203736576.0, 64509974703297150976.0];

#[cfg(feature = "radix")]
const F64_POW27: [f64; 12] = [1.0, 27.0, 729.0, 19683.0, 531441.0, 14348907.0, 387420489.0, 10460353203.0, 282429536481.0, 7625597484987.0, 205891132094649.0, 5559060566555523.0];

#[cfg(feature = "radix")]
const F64_POW28: [f64; 19] = [1.0, 28.0, 784.0, 21952.0, 614656.0, 17210368.0, 481890304.0, 13492928512.0, 377801998336.0, 10578455953408.0, 296196766695424.0, 8293509467471872.0, 232218265089212416.0, 6502111422497947648.0, 182059119829942534144.0, 5097655355238390956032.0, 142734349946674946768896.0, 3996561798506898509529088.0, 111903730358193158266814464.0];

#[cfg(feature = "radix")]
const F64_POW29: [f64; 11] = [1.0, 29.0, 841.0, 24389.0, 707281.0, 20511149.0, 594823321.0, 17249876309.0, 500246412961.0, 14507145975869.0, 420707233300201.0];

#[cfg(feature = "radix")]
const F64_POW30: [f64; 14] = [1.0, 30.0, 900.0, 27000.0, 810000.0, 24300000.0, 729000000.0, 21870000000.0, 656100000000.0, 19683000000000.0, 590490000000000.0, 17714700000000000.0, 531441000000000000.0, 15943230000000000000.0];

#[cfg(feature = "radix")]
const F64_POW31: [f64; 11] = [1.0, 31.0, 961.0, 29791.0, 923521.0, 28629151.0, 887503681.0, 27512614111.0, 852891037441.0, 26439622160671.0, 819628286980801.0];

#[cfg(feature = "radix")]
const F64_POW33: [f64; 11] = [1.0, 33.0, 1089.0, 35937.0, 1185921.0, 39135393.0, 1291467969.0, 42618442977.0, 1406408618241.0, 46411484401953.0, 1531578985264449.0];

#[cfg(feature = "radix")]
const F64_POW34: [f64; 13] = [1.0, 34.0, 1156.0, 39304.0, 1336336.0, 45435424.0, 1544804416.0, 52523350144.0, 1785793904896.0, 60716992766464.0, 2064377754059776.0, 70188843638032384.0, 2386420683693101056.0];

#[cfg(feature = "radix")]
const F64_POW35: [f64; 11] = [1.0, 35.0, 1225.0, 42875.0, 1500625.0, 52521875.0, 1838265625.0, 64339296875.0, 2251875390625.0, 78815638671875.0, 2758547353515625.0];

#[cfg(feature = "radix")]
const F64_POW36: [f64; 17] = [1.0, 36.0, 1296.0, 46656.0, 1679616.0, 60466176.0, 2176782336.0, 78364164096.0, 2821109907456.0, 101559956668416.0, 3656158440062976.0, 131621703842267136.0, 4738381338321616896.0, 170581728179578208256.0, 6140942214464815497216.0, 221073919720733357899776.0, 7958661109946400884391936.0];

// Compile-time guarantees for our tables.
const_assert!(F64_POW10[1] / F64_POW10[0] == 10.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW2[1] / F64_POW2[0] == 2.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW3[1] / F64_POW3[0] == 3.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW5[1] / F64_POW5[0] == 5.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW6[1] / F64_POW6[0] == 6.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW7[1] / F64_POW7[0] == 7.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW9[1] / F64_POW9[0] == 9.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW11[1] / F64_POW11[0] == 11.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW12[1] / F64_POW12[0] == 12.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW13[1] / F64_POW13[0] == 13.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW14[1] / F64_POW14[0] == 14.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW15[1] / F64_POW15[0] == 15.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW17[1] / F64_POW17[0] == 17.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW18[1] / F64_POW18[0] == 18.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW19[1] / F64_POW19[0] == 19.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW20[1] / F64_POW20[0] == 20.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW21[1] / F64_POW21[0] == 21.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW22[1] / F64_POW22[0] == 22.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW23[1] / F64_POW23[0] == 23.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW24[1] / F64_POW24[0] == 24.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW25[1] / F64_POW25[0] == 25.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW26[1] / F64_POW26[0] == 26.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW27[1] / F64_POW27[0] == 27.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW28[1] / F64_POW28[0] == 28.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW29[1] / F64_POW29[0] == 29.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW30[1] / F64_POW30[0] == 30.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW31[1] / F64_POW31[0] == 31.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW33[1] / F64_POW33[0] == 33.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW34[1] / F64_POW34[0] == 34.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW35[1] / F64_POW35[0] == 35.0);

#[cfg(feature = "radix")]
const_assert!(F64_POW36[1] / F64_POW36[0] == 36.0);

impl TablePower for f64 {
    const POW2_EXPONENT_BIAS: i32 = 1074;

    #[cfg(feature = "radix")]
    #[inline]
    fn table_pow2(exponent: i32) -> f64 {
        debug_assert!(exponent + Self::POW2_EXPONENT_BIAS >= 0, "table_pow2() have negative exponent.");
        F64_POW2[(exponent + Self::POW2_EXPONENT_BIAS).as_usize()]
    }

    #[inline]
    fn table_pow<T: Integer>(radix: T, exponent: i32) -> f64 {
        debug_assert!(exponent >= 0, "table_pow() have negative exponent.");
        debug_assert_radix!(radix);
        let exponent = exponent as usize;

        #[cfg(not(feature = "radix"))] {
            F64_POW10[exponent]
        }

        #[cfg(feature = "radix")] {
            match radix.as_i32() {
                3  => F64_POW3 [exponent],
                5  => F64_POW5 [exponent],
                6  => F64_POW6 [exponent],
                7  => F64_POW7 [exponent],
                9  => F64_POW9 [exponent],
                10 => F64_POW10[exponent],
                11 => F64_POW11[exponent],
                12 => F64_POW12[exponent],
                13 => F64_POW13[exponent],
                14 => F64_POW14[exponent],
                15 => F64_POW15[exponent],
                17 => F64_POW17[exponent],
                18 => F64_POW18[exponent],
                19 => F64_POW19[exponent],
                20 => F64_POW20[exponent],
                21 => F64_POW21[exponent],
                22 => F64_POW22[exponent],
                23 => F64_POW23[exponent],
                24 => F64_POW24[exponent],
                25 => F64_POW25[exponent],
                26 => F64_POW26[exponent],
                27 => F64_POW27[exponent],
                28 => F64_POW28[exponent],
                29 => F64_POW29[exponent],
                30 => F64_POW30[exponent],
                31 => F64_POW31[exponent],
                33 => F64_POW33[exponent],
                34 => F64_POW34[exponent],
                35 => F64_POW35[exponent],
                36 => F64_POW36[exponent],
                // Invalid radix
                _  => unreachable!(),
            }
        }
    }
}

}}   // cfg_if
