use k8s_openapi::{
    api::core::v1::{Container, Pod, PodSpec},
    apimachinery::pkg::apis::meta::v1::ObjectMeta,
};
use k8s_test_framework::{Framework, Interface, Reader};
use std::env;

pub mod metrics;

pub const BUSYBOX_IMAGE: &str = "busybox:1.28";

pub fn get_namespace() -> String {
    env::var("NAMESPACE").unwrap_or_else(|_| "test-vector".to_string())
}

pub fn get_namespace_appended(suffix: &str) -> String {
    format!("{}-{}", get_namespace(), suffix)
}

/// Gets a name we can use for roles to prevent them conflicting with other tests.
/// Uses the provided namespace as the root.
pub fn get_override_name(suffix: &str) -> String {
    format!("{}-{}", get_namespace(), suffix)
}

/// Adds a fullnameOverride entry to the given config. This allows multiple tests
/// to be run against the same cluster without the role anmes clashing.
pub fn config_override_name(config: &str, name: &str) -> String {
    format!("fullnameOverride: \"{}\"\n{}", name, config)
}

pub fn make_framework() -> Framework {
    let interface = Interface::from_env().expect("interface is not ready");
    Framework::new(interface)
}

pub fn collect_btree<'a>(
    items: impl IntoIterator<Item = (&'a str, &'a str)> + 'a,
) -> Option<std::collections::BTreeMap<String, String>> {
    let collected: std::collections::BTreeMap<String, String> = items
        .into_iter()
        .map(|(key, val)| (key.to_owned(), val.to_owned()))
        .collect();
    if collected.is_empty() {
        return None;
    }
    Some(collected)
}

pub fn make_test_container<'a>(name: &'a str, command: &'a str) -> Container {
    Container {
        name: name.to_owned(),
        image: Some(BUSYBOX_IMAGE.to_owned()),
        command: Some(vec!["sh".to_owned()]),
        args: Some(vec!["-c".to_owned(), command.to_owned()]),
        ..Container::default()
    }
}

pub fn make_test_pod_with_containers<'a>(
    namespace: &'a str,
    name: &'a str,
    labels: impl IntoIterator<Item = (&'a str, &'a str)> + 'a,
    annotations: impl IntoIterator<Item = (&'a str, &'a str)> + 'a,
    containers: Vec<Container>,
) -> Pod {
    Pod {
        metadata: ObjectMeta {
            name: Some(name.to_owned()),
            namespace: Some(namespace.to_owned()),
            labels: collect_btree(labels),
            annotations: collect_btree(annotations),
            ..ObjectMeta::default()
        },
        spec: Some(PodSpec {
            containers,
            restart_policy: Some("Never".to_owned()),
            ..PodSpec::default()
        }),
        ..Pod::default()
    }
}

pub fn make_test_pod<'a>(
    namespace: &'a str,
    name: &'a str,
    command: &'a str,
    labels: impl IntoIterator<Item = (&'a str, &'a str)> + 'a,
    annotations: impl IntoIterator<Item = (&'a str, &'a str)> + 'a,
) -> Pod {
    make_test_pod_with_containers(
        namespace,
        name,
        labels,
        annotations,
        vec![make_test_container(name, command)],
    )
}

pub fn parse_json(s: &str) -> Result<serde_json::Value, serde_json::Error> {
    serde_json::from_str(s)
}

pub fn generate_long_string(a: usize, b: usize) -> String {
    (0..a).fold(String::new(), |mut acc, i| {
        let istr = i.to_string();
        for _ in 0..b {
            acc.push_str(&istr);
        }
        acc
    })
}

/// Read the first line from vector logs and assert that it matches the expected
/// one.
/// This allows detecting the situations where things have gone very wrong.
pub async fn smoke_check_first_line(log_reader: &mut Reader) {
    // Wait for first line as a smoke check.
    let first_line = log_reader
        .read_line()
        .await
        .expect("unable to read first line");
    let expected_pat = "INFO vector::app: Log level is enabled. level=\"info\"\n";
    assert!(
        first_line.ends_with(expected_pat),
        "Expected a line ending with {:?} but got {:?}; vector might be malfunctioning",
        expected_pat,
        first_line
    );
}

pub enum FlowControlCommand {
    GoOn,
    Terminate,
}

pub async fn look_for_log_line<P>(
    log_reader: &mut Reader,
    mut predicate: P,
) -> Result<(), Box<dyn std::error::Error>>
where
    P: FnMut(serde_json::Value) -> FlowControlCommand,
{
    let mut lines_till_we_give_up = 10000;
    while let Some(line) = log_reader.read_line().await {
        println!("Got line: {:?}", line);

        lines_till_we_give_up -= 1;
        if lines_till_we_give_up <= 0 {
            println!("Giving up");
            log_reader.kill().await?;
            break;
        }

        if !line.starts_with('{') {
            // This isn't a json, must be an entry from Vector's own log stream.
            continue;
        }

        let val = match parse_json(&line) {
            Ok(val) => val,
            Err(err) if err.is_eof() => {
                // We got an EOF error, this is most likely some very long line,
                // we don't produce lines this bing is our test cases, so we'll
                // just skip the error - as if it wasn't a JSON string.
                println!("The JSON line we just got was incomplete, most likely it was was too long, so we're skipping it");
                continue;
            }
            Err(err) => return Err(err.into()),
        };

        match predicate(val) {
            FlowControlCommand::GoOn => {
                // Not what we were looking for, go on.
            }
            FlowControlCommand::Terminate => {
                // We are told we should stop, request that log reader is
                // killed.
                // This doesn't immediately stop the reading because we want to
                // process the pending buffers first.
                log_reader.kill().await?;
            }
        }
    }

    // Ensure log reader exited.
    log_reader.wait().await.expect("log reader wait failed");

    Ok(())
}
