/*
 * Application Insights JavaScript SDK - Web Analytics, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 */
/**
 * ApplicationInsights.ts
 * @copyright Microsoft 2018
 */
import { __assignFn as __assign, __extendsFn as __extends } from "@microsoft/applicationinsights-shims";
import { PageViewPerformance, PageView, RemoteDependencyData, Event as EventTelemetry, createTelemetryItem, Metric, Exception, Trace, PropertiesPluginIdentifier, AnalyticsPluginIdentifier, stringToBoolOrDefault, createDomEvent, strNotSpecified, isCrossOriginError, utlDisableStorage, utlEnableStorage, dataSanitizeString, createDistributedTraceContextFromTrace } from "@microsoft/applicationinsights-common";
import { BaseTelemetryPlugin, getWindow, getDocument, getHistory, getLocation, objForEachKey, isString, isFunction, isNullOrUndefined, arrForEach, generateW3CId, dumpObj, getExceptionName, safeGetCookieMgr, hasHistory, strUndefined, objDefineAccessors, InstrumentEvent, eventOn, eventOff, mergeEvtNamespace, createUniqueNamespace, throwError, isUndefined, hasWindow, createProcessTelemetryContext } from "@microsoft/applicationinsights-core-js";
import { PageViewManager } from "./Telemetry/PageViewManager";
import { PageVisitTimeManager } from "./Telemetry/PageVisitTimeManager";
import { PageViewPerformanceManager } from "./Telemetry/PageViewPerformanceManager";
import dynamicProto from "@microsoft/dynamicproto-js";
import { Timing } from "./Timing";
"use strict";
var durationProperty = "duration";
var strEvent = "event";
function _dispatchEvent(target, evnt) {
    if (target && target.dispatchEvent && evnt) {
        target.dispatchEvent(evnt);
    }
}
function _getReason(error) {
    if (error && error.reason) {
        var reason = error.reason;
        if (!isString(reason) && isFunction(reason.toString)) {
            return reason.toString();
        }
        return dumpObj(reason);
    }
    // Pass the original object down which will eventually get evaluated for any message or description
    return error || "";
}
var MinMilliSeconds = 60000;
function _configMilliseconds(value, defValue) {
    value = value || defValue;
    if (value < MinMilliSeconds) {
        value = MinMilliSeconds;
    }
    return value;
}
function _getDefaultConfig(config) {
    if (!config) {
        config = {};
    }
    // set default values
    config.sessionRenewalMs = _configMilliseconds(config.sessionRenewalMs, 30 * 60 * 1000);
    config.sessionExpirationMs = _configMilliseconds(config.sessionExpirationMs, 24 * 60 * 60 * 1000);
    config.disableExceptionTracking = stringToBoolOrDefault(config.disableExceptionTracking);
    config.autoTrackPageVisitTime = stringToBoolOrDefault(config.autoTrackPageVisitTime);
    config.overridePageViewDuration = stringToBoolOrDefault(config.overridePageViewDuration);
    config.enableUnhandledPromiseRejectionTracking = stringToBoolOrDefault(config.enableUnhandledPromiseRejectionTracking);
    if (isNaN(config.samplingPercentage) || config.samplingPercentage <= 0 || config.samplingPercentage >= 100) {
        config.samplingPercentage = 100;
    }
    config.isStorageUseDisabled = stringToBoolOrDefault(config.isStorageUseDisabled);
    config.isBrowserLinkTrackingEnabled = stringToBoolOrDefault(config.isBrowserLinkTrackingEnabled);
    config.enableAutoRouteTracking = stringToBoolOrDefault(config.enableAutoRouteTracking);
    config.namePrefix = config.namePrefix || "";
    config.enableDebug = stringToBoolOrDefault(config.enableDebug);
    config.disableFlushOnBeforeUnload = stringToBoolOrDefault(config.disableFlushOnBeforeUnload);
    config.disableFlushOnUnload = stringToBoolOrDefault(config.disableFlushOnUnload, config.disableFlushOnBeforeUnload);
    return config;
}
function _updateStorageUsage(extConfig) {
    // Not resetting the storage usage as someone may have manually called utlDisableStorage, so this will only
    // reset based if the configuration option is provided
    if (!isUndefined(extConfig.isStorageUseDisabled)) {
        if (extConfig.isStorageUseDisabled) {
            utlDisableStorage();
        }
        else {
            utlEnableStorage();
        }
    }
}
var AnalyticsPlugin = /** @class */ (function (_super) {
    __extends(AnalyticsPlugin, _super);
    function AnalyticsPlugin() {
        var _this = _super.call(this) || this;
        _this.identifier = AnalyticsPluginIdentifier; // do not change name or priority
        _this.priority = 180; // take from reserved priority range 100- 200
        _this.autoRoutePVDelay = 500; // ms; Time to wait after a route change before triggering a pageview to allow DOM changes to take place
        var _eventTracking;
        var _pageTracking;
        var _pageViewManager;
        var _pageViewPerformanceManager;
        var _pageVisitTimeManager;
        var _preInitTelemetryInitializers;
        var _isBrowserLinkTrackingEnabled;
        var _browserLinkInitializerAdded;
        var _enableAutoRouteTracking;
        var _historyListenerAdded;
        var _disableExceptionTracking;
        var _autoExceptionInstrumented;
        var _enableUnhandledPromiseRejectionTracking;
        var _autoUnhandledPromiseInstrumented;
        // Counts number of trackAjax invocations.
        // By default we only monitor X ajax call per view to avoid too much load.
        // Default value is set in config.
        // This counter keeps increasing even after the limit is reached.
        var _trackAjaxAttempts = 0;
        // array with max length of 2 that store current url and previous url for SPA page route change trackPageview use.
        var _prevUri; // Assigned in the constructor
        var _currUri;
        var _evtNamespace;
        dynamicProto(AnalyticsPlugin, _this, function (_self, _base) {
            var _addHook = _base._addHook;
            _initDefaults();
            _self.getCookieMgr = function () {
                return safeGetCookieMgr(_self.core);
            };
            _self.processTelemetry = function (env, itemCtx) {
                _self.processNext(env, itemCtx);
            };
            _self.trackEvent = function (event, customProperties) {
                try {
                    var telemetryItem = createTelemetryItem(event, EventTelemetry.dataType, EventTelemetry.envelopeType, _self.diagLog(), customProperties);
                    _self.core.track(telemetryItem);
                }
                catch (e) {
                    _throwInternal(2 /* eLoggingSeverity.WARNING */, 39 /* _eInternalMessageId.TrackTraceFailed */, "trackTrace failed, trace will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * Start timing an extended event. Call `stopTrackEvent` to log the event when it ends.
             * @param   name    A string that identifies this event uniquely within the document.
             */
            _self.startTrackEvent = function (name) {
                try {
                    _eventTracking.start(name);
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 29 /* _eInternalMessageId.StartTrackEventFailed */, "startTrackEvent failed, event will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * Log an extended event that you started timing with `startTrackEvent`.
             * @param   name    The string you used to identify this event in `startTrackEvent`.
             * @param   properties  map[string, string] - additional data used to filter events and metrics in the portal. Defaults to empty.
             * @param   measurements    map[string, number] - metrics associated with this event, displayed in Metrics Explorer on the portal. Defaults to empty.
             */
            _self.stopTrackEvent = function (name, properties, measurements) {
                try {
                    _eventTracking.stop(name, undefined, properties); // Todo: Fix to pass measurements once type is updated
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 30 /* _eInternalMessageId.StopTrackEventFailed */, "stopTrackEvent failed, event will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * @description Log a diagnostic message
             * @param {ITraceTelemetry} trace
             * @param ICustomProperties.
             * @memberof ApplicationInsights
             */
            _self.trackTrace = function (trace, customProperties) {
                try {
                    var telemetryItem = createTelemetryItem(trace, Trace.dataType, Trace.envelopeType, _self.diagLog(), customProperties);
                    _self.core.track(telemetryItem);
                }
                catch (e) {
                    _throwInternal(2 /* eLoggingSeverity.WARNING */, 39 /* _eInternalMessageId.TrackTraceFailed */, "trackTrace failed, trace will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * @description Log a numeric value that is not associated with a specific event. Typically
             * used to send regular reports of performance indicators. To send single measurement, just
             * use the name and average fields of {@link IMetricTelemetry}. If you take measurements
             * frequently, you can reduce the telemetry bandwidth by aggregating multiple measurements
             * and sending the resulting average at intervals
             * @param {IMetricTelemetry} metric input object argument. Only name and average are mandatory.
             * @param {{[key: string]: any}} customProperties additional data used to filter metrics in the
             * portal. Defaults to empty.
             * @memberof ApplicationInsights
             */
            _self.trackMetric = function (metric, customProperties) {
                try {
                    var telemetryItem = createTelemetryItem(metric, Metric.dataType, Metric.envelopeType, _self.diagLog(), customProperties);
                    _self.core.track(telemetryItem);
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 36 /* _eInternalMessageId.TrackMetricFailed */, "trackMetric failed, metric will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * Logs that a page or other item was viewed.
             * @param IPageViewTelemetry The string you used as the name in startTrackPage. Defaults to the document title.
             * @param customProperties Additional data used to filter events and metrics. Defaults to empty.
             * If a user wants to provide duration for pageLoad, it'll have to be in pageView.properties.duration
             */
            _self.trackPageView = function (pageView, customProperties) {
                try {
                    var inPv = pageView || {};
                    _pageViewManager.trackPageView(inPv, __assign(__assign(__assign({}, inPv.properties), inPv.measurements), customProperties));
                    if (_self.config.autoTrackPageVisitTime) {
                        _pageVisitTimeManager.trackPreviousPageVisit(inPv.name, inPv.uri);
                    }
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 37 /* _eInternalMessageId.TrackPVFailed */, "trackPageView failed, page view will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * Create a page view telemetry item and send it to the SDK pipeline through the core.track API
             * @param pageView Page view item to be sent
             * @param properties Custom properties (Part C) that a user can add to the telemetry item
             * @param systemProperties System level properties (Part A) that a user can add to the telemetry item
             */
            _self.sendPageViewInternal = function (pageView, properties, systemProperties) {
                var doc = getDocument();
                if (doc) {
                    pageView.refUri = pageView.refUri === undefined ? doc.referrer : pageView.refUri;
                }
                var telemetryItem = createTelemetryItem(pageView, PageView.dataType, PageView.envelopeType, _self.diagLog(), properties, systemProperties);
                _self.core.track(telemetryItem);
                // reset ajaxes counter
                _trackAjaxAttempts = 0;
            };
            /**
             * @ignore INTERNAL ONLY
             * @param pageViewPerformance
             * @param properties
             */
            _self.sendPageViewPerformanceInternal = function (pageViewPerformance, properties, systemProperties) {
                var telemetryItem = createTelemetryItem(pageViewPerformance, PageViewPerformance.dataType, PageViewPerformance.envelopeType, _self.diagLog(), properties, systemProperties);
                _self.core.track(telemetryItem);
            };
            /**
             * Send browser performance metrics.
             * @param pageViewPerformance
             * @param customProperties
             */
            _self.trackPageViewPerformance = function (pageViewPerformance, customProperties) {
                var inPvp = pageViewPerformance || {};
                try {
                    _pageViewPerformanceManager.populatePageViewPerformanceEvent(inPvp);
                    _self.sendPageViewPerformanceInternal(inPvp, customProperties);
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 37 /* _eInternalMessageId.TrackPVFailed */, "trackPageViewPerformance failed, page view will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * Starts the timer for tracking a page load time. Use this instead of `trackPageView` if you want to control when the page view timer starts and stops,
             * but don't want to calculate the duration yourself. This method doesn't send any telemetry. Call `stopTrackPage` to log the end of the page view
             * and send the event.
             * @param name A string that idenfities this item, unique within this HTML document. Defaults to the document title.
             */
            _self.startTrackPage = function (name) {
                try {
                    if (typeof name !== "string") {
                        var doc = getDocument();
                        name = doc && doc.title || "";
                    }
                    _pageTracking.start(name);
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 31 /* _eInternalMessageId.StartTrackFailed */, "startTrackPage failed, page view may not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * Stops the timer that was started by calling `startTrackPage` and sends the pageview load time telemetry with the specified properties and measurements.
             * The duration of the page view will be the time between calling `startTrackPage` and `stopTrackPage`.
             * @param   name  The string you used as the name in startTrackPage. Defaults to the document title.
             * @param   url   String - a relative or absolute URL that identifies the page or other item. Defaults to the window location.
             * @param   properties  map[string, string] - additional data used to filter pages and metrics in the portal. Defaults to empty.
             * @param   measurements    map[string, number] - metrics associated with this page, displayed in Metrics Explorer on the portal. Defaults to empty.
             */
            _self.stopTrackPage = function (name, url, properties, measurement) {
                try {
                    if (typeof name !== "string") {
                        var doc = getDocument();
                        name = doc && doc.title || "";
                    }
                    if (typeof url !== "string") {
                        var loc = getLocation();
                        url = loc && loc.href || "";
                    }
                    _pageTracking.stop(name, url, properties, measurement);
                    if (_self.config.autoTrackPageVisitTime) {
                        _pageVisitTimeManager.trackPreviousPageVisit(name, url);
                    }
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 32 /* _eInternalMessageId.StopTrackFailed */, "stopTrackPage failed, page view will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
            * @ignore INTERNAL ONLY
            * @param exception
            * @param properties
            * @param systemProperties
            */
            _self.sendExceptionInternal = function (exception, customProperties, systemProperties) {
                var theError = exception.exception || exception.error || new Error(strNotSpecified);
                var exceptionPartB = new Exception(_self.diagLog(), theError, exception.properties || customProperties, exception.measurements, exception.severityLevel, exception.id).toInterface();
                var telemetryItem = createTelemetryItem(exceptionPartB, Exception.dataType, Exception.envelopeType, _self.diagLog(), customProperties, systemProperties);
                _self.core.track(telemetryItem);
            };
            /**
             * Log an exception you have caught.
             *
             * @param {IExceptionTelemetry} exception   Object which contains exception to be sent
             * @param {{[key: string]: any}} customProperties   Additional data used to filter pages and metrics in the portal. Defaults to empty.
             *
             * Any property of type double will be considered a measurement, and will be treated by Application Insights as a metric.
             * @memberof ApplicationInsights
             */
            _self.trackException = function (exception, customProperties) {
                if (exception && !exception.exception && exception.error) {
                    exception.exception = exception.error;
                }
                try {
                    _self.sendExceptionInternal(exception, customProperties);
                }
                catch (e) {
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 35 /* _eInternalMessageId.TrackExceptionFailed */, "trackException failed, exception will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                }
            };
            /**
             * @description Custom error handler for Application Insights Analytics
             * @param {IAutoExceptionTelemetry} exception
             * @memberof ApplicationInsights
             */
            _self._onerror = function (exception) {
                var error = exception && exception.error;
                var evt = exception && exception.evt;
                try {
                    if (!evt) {
                        var _window = getWindow();
                        if (_window) {
                            evt = _window[strEvent];
                        }
                    }
                    var url = (exception && exception.url) || (getDocument() || {}).URL;
                    // If no error source is provided assume the default window.onerror handler
                    var errorSrc = exception.errorSrc || "window.onerror@" + url + ":" + (exception.lineNumber || 0) + ":" + (exception.columnNumber || 0);
                    var properties = {
                        errorSrc: errorSrc,
                        url: url,
                        lineNumber: exception.lineNumber || 0,
                        columnNumber: exception.columnNumber || 0,
                        message: exception.message
                    };
                    if (isCrossOriginError(exception.message, exception.url, exception.lineNumber, exception.columnNumber, exception.error)) {
                        _sendCORSException(Exception.CreateAutoException("Script error: The browser's same-origin policy prevents us from getting the details of this exception. Consider using the 'crossorigin' attribute.", url, exception.lineNumber || 0, exception.columnNumber || 0, error, evt, null, errorSrc), properties);
                    }
                    else {
                        if (!exception.errorSrc) {
                            exception.errorSrc = errorSrc;
                        }
                        _self.trackException({ exception: exception, severityLevel: 3 /* eSeverityLevel.Error */ }, properties);
                    }
                }
                catch (e) {
                    var errorString = error ? (error.name + ", " + error.message) : "null";
                    _throwInternal(1 /* eLoggingSeverity.CRITICAL */, 11 /* _eInternalMessageId.ExceptionWhileLoggingError */, "_onError threw exception while logging error, error will not be collected: "
                        + getExceptionName(e), { exception: dumpObj(e), errorString: errorString });
                }
            };
            _self.addTelemetryInitializer = function (telemetryInitializer) {
                if (_self.core) {
                    // Just add to the core
                    return _self.core.addTelemetryInitializer(telemetryInitializer);
                }
                // Handle "pre-initialization" telemetry initializers (for backward compatibility)
                if (!_preInitTelemetryInitializers) {
                    _preInitTelemetryInitializers = [];
                }
                _preInitTelemetryInitializers.push(telemetryInitializer);
            };
            _self.initialize = function (config, core, extensions, pluginChain) {
                if (_self.isInitialized()) {
                    return;
                }
                if (isNullOrUndefined(core)) {
                    throwError("Error initializing");
                }
                _base.initialize(config, core, extensions, pluginChain);
                try {
                    _evtNamespace = mergeEvtNamespace(createUniqueNamespace(_self.identifier), core.evtNamespace && core.evtNamespace());
                    if (_preInitTelemetryInitializers) {
                        arrForEach(_preInitTelemetryInitializers, function (initializer) {
                            core.addTelemetryInitializer(initializer);
                        });
                        _preInitTelemetryInitializers = null;
                    }
                    var extConfig = _populateDefaults(config);
                    _updateStorageUsage(extConfig);
                    _pageViewPerformanceManager = new PageViewPerformanceManager(_self.core);
                    _pageViewManager = new PageViewManager(_this, extConfig.overridePageViewDuration, _self.core, _pageViewPerformanceManager);
                    _pageVisitTimeManager = new PageVisitTimeManager(_self.diagLog(), function (pageName, pageUrl, pageVisitTime) { return trackPageVisitTime(pageName, pageUrl, pageVisitTime); });
                    _updateBrowserLinkTracking(extConfig, config);
                    _eventTracking = new Timing(_self.diagLog(), "trackEvent");
                    _eventTracking.action =
                        function (name, url, duration, properties) {
                            if (!properties) {
                                properties = {};
                            }
                            properties[durationProperty] = duration.toString();
                            _self.trackEvent({ name: name, properties: properties });
                        };
                    // initialize page view timing
                    _pageTracking = new Timing(_self.diagLog(), "trackPageView");
                    _pageTracking.action = function (name, url, duration, properties, measurements) {
                        // duration must be a custom property in order for the collector to extract it
                        if (isNullOrUndefined(properties)) {
                            properties = {};
                        }
                        properties[durationProperty] = duration.toString();
                        var pageViewItem = {
                            name: name,
                            uri: url,
                            properties: properties,
                            measurements: measurements
                        };
                        _self.sendPageViewInternal(pageViewItem, properties);
                    };
                    if (hasWindow()) {
                        _updateExceptionTracking(extConfig);
                        _updateLocationChange(extConfig);
                    }
                }
                catch (e) {
                    // resetting the initialized state because of failure
                    _self.setInitialized(false);
                    throw e;
                }
            };
            _self._doTeardown = function (unloadCtx, unloadState) {
                _pageViewManager && _pageViewManager.teardown(unloadCtx, unloadState);
                // Just register to remove all events associated with this namespace
                eventOff(window, null, null, _evtNamespace);
                _initDefaults();
            };
            function _populateDefaults(config) {
                var ctx = createProcessTelemetryContext(null, config, _self.core);
                var identifier = _self.identifier;
                // load default values if specified
                var defaults = _getDefaultConfig(config);
                var extConfig = _self.config = ctx.getExtCfg(identifier);
                if (defaults !== undefined) {
                    objForEachKey(defaults, function (field, value) {
                        // for each unspecified field, set the default value
                        extConfig[field] = ctx.getConfig(identifier, field, value);
                        if (extConfig[field] === undefined) {
                            extConfig = value;
                        }
                    });
                }
                return extConfig;
            }
            function _updateBrowserLinkTracking(extConfig, config) {
                _isBrowserLinkTrackingEnabled = extConfig.isBrowserLinkTrackingEnabled || config.isBrowserLinkTrackingEnabled;
                _addDefaultTelemetryInitializers();
            }
            /**
             * Log a page visit time
             * @param    pageName    Name of page
             * @param    pageVisitDuration Duration of visit to the page in milleseconds
             */
            function trackPageVisitTime(pageName, pageUrl, pageVisitTime) {
                var properties = { PageName: pageName, PageUrl: pageUrl };
                _self.trackMetric({
                    name: "PageVisitTime",
                    average: pageVisitTime,
                    max: pageVisitTime,
                    min: pageVisitTime,
                    sampleCount: 1
                }, properties);
            }
            function _addDefaultTelemetryInitializers() {
                if (!_browserLinkInitializerAdded && _isBrowserLinkTrackingEnabled) {
                    var browserLinkPaths_1 = ["/browserLinkSignalR/", "/__browserLink/"];
                    var dropBrowserLinkRequests = function (envelope) {
                        if (_isBrowserLinkTrackingEnabled && envelope.baseType === RemoteDependencyData.dataType) {
                            var remoteData = envelope.baseData;
                            if (remoteData) {
                                for (var i = 0; i < browserLinkPaths_1.length; i++) {
                                    if (remoteData.target && remoteData.target.indexOf(browserLinkPaths_1[i]) >= 0) {
                                        return false;
                                    }
                                }
                            }
                        }
                        return true;
                    };
                    _self.addTelemetryInitializer(dropBrowserLinkRequests);
                    _browserLinkInitializerAdded = true;
                }
            }
            function _sendCORSException(exception, properties) {
                var telemetryItem = createTelemetryItem(exception, Exception.dataType, Exception.envelopeType, _self.diagLog(), properties);
                _self.core.track(telemetryItem);
            }
            function _updateExceptionTracking(extConfig) {
                var _window = getWindow();
                var locn = getLocation(true);
                _disableExceptionTracking = extConfig.disableExceptionTracking;
                if (!_disableExceptionTracking && !_autoExceptionInstrumented && !extConfig.autoExceptionInstrumented) {
                    // We want to enable exception auto collection and it has not been done so yet
                    _addHook(InstrumentEvent(_window, "onerror", {
                        ns: _evtNamespace,
                        rsp: function (callDetails, message, url, lineNumber, columnNumber, error) {
                            if (!_disableExceptionTracking && callDetails.rslt !== true) {
                                _self._onerror(Exception.CreateAutoException(message, url, lineNumber, columnNumber, error, callDetails.evt));
                            }
                        }
                    }, false));
                    _autoExceptionInstrumented = true;
                }
                _addUnhandledPromiseRejectionTracking(extConfig, _window, locn);
            }
            function _updateLocationChange(extConfig) {
                var win = getWindow();
                var locn = getLocation(true);
                _enableAutoRouteTracking = extConfig.enableAutoRouteTracking === true;
                /**
                 * Create a custom "locationchange" event which is triggered each time the history object is changed
                 */
                if (win && _enableAutoRouteTracking && hasHistory()) {
                    var _history = getHistory();
                    if (isFunction(_history.pushState) && isFunction(_history.replaceState) && typeof Event !== strUndefined) {
                        _addHistoryListener(extConfig, win, _history, locn);
                    }
                }
            }
            function _getDistributedTraceCtx() {
                var distributedTraceCtx = null;
                if (_self.core && _self.core.getTraceCtx) {
                    distributedTraceCtx = _self.core.getTraceCtx(false);
                }
                if (!distributedTraceCtx) {
                    // Fallback when using an older Core and PropertiesPlugin
                    var properties = _self.core.getPlugin(PropertiesPluginIdentifier);
                    if (properties) {
                        var context = properties.plugin.context;
                        if (context) {
                            distributedTraceCtx = createDistributedTraceContextFromTrace(context.telemetryTrace);
                        }
                    }
                }
                return distributedTraceCtx;
            }
            /**
             * Create a custom "locationchange" event which is triggered each time the history object is changed
             */
            function _addHistoryListener(extConfig, win, history, locn) {
                function _popstateHandler() {
                    if (_enableAutoRouteTracking) {
                        _dispatchEvent(win, createDomEvent(extConfig.namePrefix + "locationchange"));
                    }
                }
                function _locationChangeHandler() {
                    // We always track the changes (if the handler is installed) to handle the feature being disabled between location changes
                    if (_currUri) {
                        _prevUri = _currUri;
                        _currUri = locn && locn.href || "";
                    }
                    else {
                        _currUri = locn && locn.href || "";
                    }
                    if (_enableAutoRouteTracking) {
                        var distributedTraceCtx = _getDistributedTraceCtx();
                        if (distributedTraceCtx) {
                            distributedTraceCtx.setTraceId(generateW3CId());
                            var traceLocationName = "_unknown_";
                            if (locn && locn.pathname) {
                                traceLocationName = locn.pathname + (locn.hash || "");
                            }
                            // This populates the ai.operation.name which has a maximum size of 1024 so we need to sanitize it
                            distributedTraceCtx.setName(dataSanitizeString(_self.diagLog(), traceLocationName));
                        }
                        setTimeout((function (uri) {
                            // todo: override start time so that it is not affected by autoRoutePVDelay
                            _self.trackPageView({ refUri: uri, properties: { duration: 0 } }); // SPA route change loading durations are undefined, so send 0
                        }).bind(this, _prevUri), _self.autoRoutePVDelay);
                    }
                }
                if (!_historyListenerAdded) {
                    _addHook(InstrumentEvent(history, "pushState", {
                        ns: _evtNamespace,
                        rsp: function () {
                            if (_enableAutoRouteTracking) {
                                _dispatchEvent(win, createDomEvent(extConfig.namePrefix + "pushState"));
                                _dispatchEvent(win, createDomEvent(extConfig.namePrefix + "locationchange"));
                            }
                        }
                    }, true));
                    _addHook(InstrumentEvent(history, "replaceState", {
                        ns: _evtNamespace,
                        rsp: function () {
                            if (_enableAutoRouteTracking) {
                                _dispatchEvent(win, createDomEvent(extConfig.namePrefix + "replaceState"));
                                _dispatchEvent(win, createDomEvent(extConfig.namePrefix + "locationchange"));
                            }
                        }
                    }, true));
                    eventOn(win, extConfig.namePrefix + "popstate", _popstateHandler, _evtNamespace);
                    eventOn(win, extConfig.namePrefix + "locationchange", _locationChangeHandler, _evtNamespace);
                    _historyListenerAdded = true;
                }
            }
            function _addUnhandledPromiseRejectionTracking(extConfig, _window, _location) {
                _enableUnhandledPromiseRejectionTracking = extConfig.enableUnhandledPromiseRejectionTracking === true;
                if (_enableUnhandledPromiseRejectionTracking && !_autoUnhandledPromiseInstrumented) {
                    // We want to enable exception auto collection and it has not been done so yet
                    _addHook(InstrumentEvent(_window, "onunhandledrejection", {
                        ns: _evtNamespace,
                        rsp: function (callDetails, error) {
                            if (_enableUnhandledPromiseRejectionTracking && callDetails.rslt !== true) { // handled could be typeof function
                                _self._onerror(Exception.CreateAutoException(_getReason(error), _location ? _location.href : "", 0, 0, error, callDetails.evt));
                            }
                        }
                    }, false));
                    _autoUnhandledPromiseInstrumented = true;
                    extConfig.autoUnhandledPromiseInstrumented = _autoUnhandledPromiseInstrumented;
                }
            }
            /**
             * This method will throw exceptions in debug mode or attempt to log the error as a console warning.
             * @param severity {LoggingSeverity} - The severity of the log message
             * @param message {_InternalLogMessage} - The log message.
             */
            function _throwInternal(severity, msgId, msg, properties, isUserAct) {
                _self.diagLog().throwInternal(severity, msgId, msg, properties, isUserAct);
            }
            function _initDefaults() {
                _eventTracking = null;
                _pageTracking = null;
                _pageViewManager = null;
                _pageViewPerformanceManager = null;
                _pageVisitTimeManager = null;
                _preInitTelemetryInitializers = null;
                _isBrowserLinkTrackingEnabled = false;
                _browserLinkInitializerAdded = false;
                _enableAutoRouteTracking = false;
                _historyListenerAdded = false;
                _disableExceptionTracking = false;
                _autoExceptionInstrumented = false;
                _enableUnhandledPromiseRejectionTracking = false;
                _autoUnhandledPromiseInstrumented = false;
                // Counts number of trackAjax invocations.
                // By default we only monitor X ajax call per view to avoid too much load.
                // Default value is set in config.
                // This counter keeps increasing even after the limit is reached.
                _trackAjaxAttempts = 0;
                // array with max length of 2 that store current url and previous url for SPA page route change trackPageview use.
                var location = getLocation(true);
                _prevUri = location && location.href || "";
                _currUri = null;
                _evtNamespace = null;
            }
            // For backward compatibility
            objDefineAccessors(_self, "_pageViewManager", function () { return _pageViewManager; });
            objDefineAccessors(_self, "_pageViewPerformanceManager", function () { return _pageViewPerformanceManager; });
            objDefineAccessors(_self, "_pageVisitTimeManager", function () { return _pageVisitTimeManager; });
            objDefineAccessors(_self, "_evtNamespace", function () { return "." + _evtNamespace; });
        });
        return _this;
    }
// Removed Stub for AnalyticsPlugin.prototype.getCookieMgr.
// Removed Stub for AnalyticsPlugin.prototype.processTelemetry.
// Removed Stub for AnalyticsPlugin.prototype.trackEvent.
// Removed Stub for AnalyticsPlugin.prototype.startTrackEvent.
// Removed Stub for AnalyticsPlugin.prototype.stopTrackEvent.
// Removed Stub for AnalyticsPlugin.prototype.trackTrace.
// Removed Stub for AnalyticsPlugin.prototype.trackMetric.
// Removed Stub for AnalyticsPlugin.prototype.trackPageView.
// Removed Stub for AnalyticsPlugin.prototype.sendPageViewInternal.
// Removed Stub for AnalyticsPlugin.prototype.sendPageViewPerformanceInternal.
// Removed Stub for AnalyticsPlugin.prototype.trackPageViewPerformance.
// Removed Stub for AnalyticsPlugin.prototype.startTrackPage.
// Removed Stub for AnalyticsPlugin.prototype.stopTrackPage.
// Removed Stub for AnalyticsPlugin.prototype.sendExceptionInternal.
// Removed Stub for AnalyticsPlugin.prototype.trackException.
// Removed Stub for AnalyticsPlugin.prototype._onerror.
// Removed Stub for AnalyticsPlugin.prototype.addTelemetryInitializer.
// Removed Stub for AnalyticsPlugin.prototype.initialize.
    AnalyticsPlugin.Version = "2.8.4"; // Not currently used anywhere
    AnalyticsPlugin.getDefaultConfig = _getDefaultConfig;
    return AnalyticsPlugin;
}(BaseTelemetryPlugin));
export { AnalyticsPlugin };
