/*
 * Application Insights JavaScript SDK - Web Analytics, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 */


import { utlCanUseSessionStorage, utlGetSessionStorage, utlRemoveSessionStorage, utlSetSessionStorage } from "@microsoft/applicationinsights-common";
import { hasJSON, getJSON, dateNow, dumpObj, throwError, _warnToConsole } from "@microsoft/applicationinsights-core-js";
/**
 * Used to track page visit durations
 */
var PageVisitTimeManager = /** @class */ (function () {
    /**
     * Creates a new instance of PageVisitTimeManager
     * @param pageVisitTimeTrackingHandler Delegate that will be called to send telemetry data to AI (when trackPreviousPageVisit is called)
     * @returns {}
     */
    function PageVisitTimeManager(logger, pageVisitTimeTrackingHandler) {
        this.prevPageVisitDataKeyName = "prevPageVisitData";
        this.pageVisitTimeTrackingHandler = pageVisitTimeTrackingHandler;
        this._logger = logger;
    }
    /**
    * Tracks the previous page visit time telemetry (if exists) and starts timing of new page visit time
    * @param currentPageName Name of page to begin timing for visit duration
    * @param currentPageUrl Url of page to begin timing for visit duration
    */
    PageVisitTimeManager.prototype.trackPreviousPageVisit = function (currentPageName, currentPageUrl) {
        try {
            // Restart timer for new page view
            var prevPageVisitTimeData = this.restartPageVisitTimer(currentPageName, currentPageUrl);
            // If there was a page already being timed, track the visit time for it now.
            if (prevPageVisitTimeData) {
                this.pageVisitTimeTrackingHandler(prevPageVisitTimeData.pageName, prevPageVisitTimeData.pageUrl, prevPageVisitTimeData.pageVisitTime);
            }
        }
        catch (e) {
            _warnToConsole(this._logger, "Auto track page visit time failed, metric will not be collected: " + dumpObj(e));
        }
    };
    /**
     * Stops timing of current page (if exists) and starts timing for duration of visit to pageName
     * @param pageName Name of page to begin timing visit duration
     * @returns {PageVisitData} Page visit data (including duration) of pageName from last call to start or restart, if exists. Null if not.
     */
    PageVisitTimeManager.prototype.restartPageVisitTimer = function (pageName, pageUrl) {
        try {
            var prevPageVisitData = this.stopPageVisitTimer();
            this.startPageVisitTimer(pageName, pageUrl);
            return prevPageVisitData;
        }
        catch (e) {
            _warnToConsole(this._logger, "Call to restart failed: " + dumpObj(e));
            return null;
        }
    };
    /**
     * Starts timing visit duration of pageName
     * @param pageName
     * @returns {}
     */
    PageVisitTimeManager.prototype.startPageVisitTimer = function (pageName, pageUrl) {
        try {
            if (utlCanUseSessionStorage()) {
                if (utlGetSessionStorage(this._logger, this.prevPageVisitDataKeyName) != null) {
                    throwError("Cannot call startPageVisit consecutively without first calling stopPageVisit");
                }
                var currPageVisitData = new PageVisitData(pageName, pageUrl);
                var currPageVisitDataStr = getJSON().stringify(currPageVisitData);
                utlSetSessionStorage(this._logger, this.prevPageVisitDataKeyName, currPageVisitDataStr);
            }
        }
        catch (e) {
            // TODO: Remove this catch in next phase, since if start is called twice in a row the exception needs to be propagated out
            _warnToConsole(this._logger, "Call to start failed: " + dumpObj(e));
        }
    };
    /**
     * Stops timing of current page, if exists.
     * @returns {PageVisitData} Page visit data (including duration) of pageName from call to start, if exists. Null if not.
     */
    PageVisitTimeManager.prototype.stopPageVisitTimer = function () {
        try {
            if (utlCanUseSessionStorage()) {
                // Define end time of page's visit
                var pageVisitEndTime = dateNow();
                // Try to retrieve  page name and start time from session storage
                var pageVisitDataJsonStr = utlGetSessionStorage(this._logger, this.prevPageVisitDataKeyName);
                if (pageVisitDataJsonStr && hasJSON()) {
                    // if previous page data exists, set end time of visit
                    var prevPageVisitData = getJSON().parse(pageVisitDataJsonStr);
                    prevPageVisitData.pageVisitTime = pageVisitEndTime - prevPageVisitData.pageVisitStartTime;
                    // Remove data from storage since we already used it
                    utlRemoveSessionStorage(this._logger, this.prevPageVisitDataKeyName);
                    // Return page visit data
                    return prevPageVisitData;
                }
                else {
                    return null;
                }
            }
            return null;
        }
        catch (e) {
            _warnToConsole(this._logger, "Stop page visit timer failed: " + dumpObj(e));
            return null;
        }
    };
    return PageVisitTimeManager;
}());
export { PageVisitTimeManager };
var PageVisitData = /** @class */ (function () {
    function PageVisitData(pageName, pageUrl) {
        this.pageVisitStartTime = dateNow();
        this.pageName = pageName;
        this.pageUrl = pageUrl;
    }
    return PageVisitData;
}());
export { PageVisitData };
