/*
 * Application Insights JavaScript SDK - Channel, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 */
var _a;
import { __assignFn as __assign, __extendsFn as __extends } from "@microsoft/applicationinsights-shims";
import { SessionStorageSendBuffer, ArraySendBuffer } from "./SendBuffer";
import { DependencyEnvelopeCreator, EventEnvelopeCreator, ExceptionEnvelopeCreator, MetricEnvelopeCreator, PageViewEnvelopeCreator, PageViewPerformanceEnvelopeCreator, TraceEnvelopeCreator } from "./EnvelopeCreator";
import { Serializer } from "./Serializer"; // todo move to channel
import { DisabledPropertyName, RequestHeaders, PageView, Event, Trace, Exception, Metric, PageViewPerformance, RemoteDependencyData, ProcessLegacy, BreezeChannelIdentifier, SampleRate, isInternalApplicationInsightsEndpoint, utlCanUseSessionStorage } from "@microsoft/applicationinsights-common";
import { getWindow, getNavigator, getJSON, BaseTelemetryPlugin, objForEachKey, isNullOrUndefined, arrForEach, dateNow, dumpObj, getExceptionName, getIEVersion, objKeys, isBeaconsSupported, isFetchSupported, useXDomainRequest, isXhrSupported, isArray, createUniqueNamespace, mergeEvtNamespace, _throwInternal, _warnToConsole } from "@microsoft/applicationinsights-core-js";
import { createOfflineListener } from "./Offline";
import { Sample } from "./TelemetryProcessors/Sample";
import dynamicProto from "@microsoft/dynamicproto-js";
var FetchSyncRequestSizeLimitBytes = 65000; // approx 64kb (the current Edge, Firefox and Chrome max limit)
function _getResponseText(xhr) {
    try {
        return xhr.responseText;
    }
    catch (e) {
        // Best effort, as XHR may throw while XDR wont so just ignore
    }
    return null;
}
function _getDefaultAppInsightsChannelConfig() {
    // set default values
    return {
        endpointUrl: function () { return "https://dc.services.visualstudio.com/v2/track"; },
        emitLineDelimitedJson: function () { return false; },
        maxBatchInterval: function () { return 15000; },
        maxBatchSizeInBytes: function () { return 102400; },
        disableTelemetry: function () { return false; },
        enableSessionStorageBuffer: function () { return true; },
        isRetryDisabled: function () { return false; },
        isBeaconApiDisabled: function () { return true; },
        disableXhr: function () { return false; },
        onunloadDisableFetch: function () { return false; },
        onunloadDisableBeacon: function () { return false; },
        instrumentationKey: function () { return undefined; },
        namePrefix: function () { return undefined; },
        samplingPercentage: function () { return 100; },
        customHeaders: function () { return undefined; },
        convertUndefined: function () { return undefined; },
        eventsLimitInMem: function () { return 10000; }
    };
}
var EnvelopeTypeCreator = (_a = {},
    _a[Event.dataType] = EventEnvelopeCreator,
    _a[Trace.dataType] = TraceEnvelopeCreator,
    _a[PageView.dataType] = PageViewEnvelopeCreator,
    _a[PageViewPerformance.dataType] = PageViewPerformanceEnvelopeCreator,
    _a[Exception.dataType] = ExceptionEnvelopeCreator,
    _a[Metric.dataType] = MetricEnvelopeCreator,
    _a[RemoteDependencyData.dataType] = DependencyEnvelopeCreator,
    _a);
var Sender = /** @class */ (function (_super) {
    __extends(Sender, _super);
    function Sender() {
        var _this = _super.call(this) || this;
        _this.priority = 1001;
        _this.identifier = BreezeChannelIdentifier;
        /**
         * The configuration for this sender instance
         */
        _this._senderConfig = _getDefaultAppInsightsChannelConfig();
        // Don't set the defaults here, set them in the _initDefaults() as this is also called during unload
        var _consecutiveErrors; // How many times in a row a retryable error condition has occurred.
        var _retryAt; // The time to retry at in milliseconds from 1970/01/01 (this makes the timer calculation easy).
        var _lastSend; // The time of the last send operation.
        var _paused; // Flag indicating that the sending should be paused
        var _timeoutHandle; // Handle to the timer for delayed sending of batches of data.
        var _serializer;
        var _stamp_specific_redirects;
        var _headers;
        var _syncFetchPayload = 0; // Keep track of the outstanding sync fetch payload total (as sync fetch has limits)
        var _fallbackSender; // The sender to use if the payload size is too large
        var _syncUnloadSender; // The identified sender to use for the synchronous unload stage
        var _offlineListener;
        var _evtNamespace;
        dynamicProto(Sender, _this, function (_self, _base) {
            _initDefaults();
            _self.pause = function () {
                _clearScheduledTimer();
                _paused = true;
            };
            _self.resume = function () {
                if (_paused) {
                    _paused = false;
                    _retryAt = null;
                    // flush if we have exceeded the max-size already
                    if (_self._buffer.size() > _self._senderConfig.maxBatchSizeInBytes()) {
                        _self.triggerSend(true, null, 10 /* SendRequestReason.MaxBatchSize */);
                    }
                    _setupTimer();
                }
            };
            _self.flush = function (isAsync, callBack, sendReason) {
                if (isAsync === void 0) { isAsync = true; }
                if (!_paused) {
                    // Clear the normal schedule timer as we are going to try and flush ASAP
                    _clearScheduledTimer();
                    try {
                        _self.triggerSend(isAsync, null, sendReason || 1 /* SendRequestReason.ManualFlush */);
                    }
                    catch (e) {
                        _throwInternal(_self.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 22 /* _eInternalMessageId.FlushFailed */, "flush failed, telemetry will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                    }
                }
            };
            _self.onunloadFlush = function () {
                if (!_paused) {
                    if ((_self._senderConfig.onunloadDisableBeacon() === false || _self._senderConfig.isBeaconApiDisabled() === false) && isBeaconsSupported()) {
                        try {
                            _self.triggerSend(true, _doUnloadSend, 2 /* SendRequestReason.Unload */);
                        }
                        catch (e) {
                            _throwInternal(_self.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 20 /* _eInternalMessageId.FailedToSendQueuedTelemetry */, "failed to flush with beacon sender on page unload, telemetry will not be collected: " + getExceptionName(e), { exception: dumpObj(e) });
                        }
                    }
                    else {
                        _self.flush();
                    }
                }
            };
            _self.addHeader = function (name, value) {
                _headers[name] = value;
            };
            _self.initialize = function (config, core, extensions, pluginChain) {
                if (_self.isInitialized()) {
                    _throwInternal(_self.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 28 /* _eInternalMessageId.SenderNotInitialized */, "Sender is already initialized");
                }
                _base.initialize(config, core, extensions, pluginChain);
                var ctx = _self._getTelCtx();
                var identifier = _self.identifier;
                _serializer = new Serializer(core.logger);
                _consecutiveErrors = 0;
                _retryAt = null;
                _lastSend = 0;
                _self._sender = null;
                _stamp_specific_redirects = 0;
                var diagLog = _self.diagLog();
                _evtNamespace = mergeEvtNamespace(createUniqueNamespace("Sender"), core.evtNamespace && core.evtNamespace());
                _offlineListener = createOfflineListener(_evtNamespace);
                var defaultConfig = _getDefaultAppInsightsChannelConfig();
                objForEachKey(defaultConfig, function (field, value) {
                    _self._senderConfig[field] = function () { return ctx.getConfig(identifier, field, value()); };
                });
                _self._buffer = (_self._senderConfig.enableSessionStorageBuffer() && utlCanUseSessionStorage())
                    ? new SessionStorageSendBuffer(diagLog, _self._senderConfig) : new ArraySendBuffer(diagLog, _self._senderConfig);
                _self._sample = new Sample(_self._senderConfig.samplingPercentage(), diagLog);
                if (!_validateInstrumentationKey(config)) {
                    _throwInternal(diagLog, 1 /* eLoggingSeverity.CRITICAL */, 100 /* _eInternalMessageId.InvalidInstrumentationKey */, "Invalid Instrumentation key " + config.instrumentationKey);
                }
                if (!isInternalApplicationInsightsEndpoint(_self._senderConfig.endpointUrl()) && _self._senderConfig.customHeaders() && _self._senderConfig.customHeaders().length > 0) {
                    arrForEach(_self._senderConfig.customHeaders(), function (customHeader) {
                        _this.addHeader(customHeader.header, customHeader.value);
                    });
                }
                var senderConfig = _self._senderConfig;
                var sendPostFunc = null;
                if (!senderConfig.disableXhr() && useXDomainRequest()) {
                    sendPostFunc = _xdrSender; // IE 8 and 9
                }
                else if (!senderConfig.disableXhr() && isXhrSupported()) {
                    sendPostFunc = _xhrSender;
                }
                if (!sendPostFunc && isFetchSupported()) {
                    sendPostFunc = _fetchSender;
                }
                // always fallback to XHR
                _fallbackSender = sendPostFunc || _xhrSender;
                if (!senderConfig.isBeaconApiDisabled() && isBeaconsSupported()) {
                    // Config is set to always used beacon sending
                    sendPostFunc = _beaconSender;
                }
                _self._sender = sendPostFunc || _xhrSender;
                if (!senderConfig.onunloadDisableFetch() && isFetchSupported(true)) {
                    // Try and use the fetch with keepalive
                    _syncUnloadSender = _fetchKeepAliveSender;
                }
                else if (isBeaconsSupported()) {
                    // Try and use sendBeacon
                    _syncUnloadSender = _beaconSender;
                }
                else if (!senderConfig.disableXhr() && useXDomainRequest()) {
                    _syncUnloadSender = _xdrSender; // IE 8 and 9
                }
                else if (!senderConfig.disableXhr() && isXhrSupported()) {
                    _syncUnloadSender = _xhrSender;
                }
                else {
                    _syncUnloadSender = _fallbackSender;
                }
            };
            _self.processTelemetry = function (telemetryItem, itemCtx) {
                itemCtx = _self._getTelCtx(itemCtx);
                try {
                    // if master off switch is set, don't send any data
                    if (_self._senderConfig.disableTelemetry()) {
                        // Do not send/save data
                        return;
                    }
                    // validate input
                    if (!telemetryItem) {
                        _throwInternal(itemCtx.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 7 /* _eInternalMessageId.CannotSendEmptyTelemetry */, "Cannot send empty telemetry");
                        return;
                    }
                    // validate event
                    if (telemetryItem.baseData && !telemetryItem.baseType) {
                        _throwInternal(itemCtx.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 70 /* _eInternalMessageId.InvalidEvent */, "Cannot send telemetry without baseData and baseType");
                        return;
                    }
                    if (!telemetryItem.baseType) {
                        // Default
                        telemetryItem.baseType = "EventData";
                    }
                    // ensure a sender was constructed
                    if (!_self._sender) {
                        _throwInternal(itemCtx.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 28 /* _eInternalMessageId.SenderNotInitialized */, "Sender was not initialized");
                        return;
                    }
                    // check if this item should be sampled in, else add sampleRate tag
                    if (!_isSampledIn(telemetryItem)) {
                        // Item is sampled out, do not send it
                        _throwInternal(itemCtx.diagLog(), 2 /* eLoggingSeverity.WARNING */, 33 /* _eInternalMessageId.TelemetrySampledAndNotSent */, "Telemetry item was sampled out and not sent", { SampleRate: _self._sample.sampleRate });
                        return;
                    }
                    else {
                        telemetryItem[SampleRate] = _self._sample.sampleRate;
                    }
                    var convertUndefined = _self._senderConfig.convertUndefined() || undefined;
                    // construct an envelope that Application Insights endpoint can understand
                    // if ikey of telemetry is provided and not empty, envelope will use this iKey instead of senderConfig iKey
                    var defaultEnvelopeIkey = telemetryItem.iKey || _self._senderConfig.instrumentationKey();
                    var aiEnvelope_1 = Sender.constructEnvelope(telemetryItem, defaultEnvelopeIkey, itemCtx.diagLog(), convertUndefined);
                    if (!aiEnvelope_1) {
                        _throwInternal(itemCtx.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 47 /* _eInternalMessageId.CreateEnvelopeError */, "Unable to create an AppInsights envelope");
                        return;
                    }
                    var doNotSendItem_1 = false;
                    // this is for running in legacy mode, where customer may already have a custom initializer present
                    if (telemetryItem.tags && telemetryItem.tags[ProcessLegacy]) {
                        arrForEach(telemetryItem.tags[ProcessLegacy], function (callBack) {
                            try {
                                if (callBack && callBack(aiEnvelope_1) === false) {
                                    doNotSendItem_1 = true;
                                    _warnToConsole(itemCtx.diagLog(), "Telemetry processor check returns false");
                                }
                            }
                            catch (e) {
                                // log error but dont stop executing rest of the telemetry initializers
                                // doNotSendItem = true;
                                _throwInternal(itemCtx.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 64 /* _eInternalMessageId.TelemetryInitializerFailed */, "One of telemetry initializers failed, telemetry item will not be sent: " + getExceptionName(e), { exception: dumpObj(e) }, true);
                            }
                        });
                        delete telemetryItem.tags[ProcessLegacy];
                    }
                    if (doNotSendItem_1) {
                        return; // do not send, no need to execute next plugin
                    }
                    // check if the incoming payload is too large, truncate if necessary
                    var payload = _serializer.serialize(aiEnvelope_1);
                    // flush if we would exceed the max-size limit by adding this item
                    var buffer = _self._buffer;
                    var bufferSize = buffer.size();
                    if ((bufferSize + payload.length) > _self._senderConfig.maxBatchSizeInBytes()) {
                        _self.triggerSend(true, null, 10 /* SendRequestReason.MaxBatchSize */);
                    }
                    // enqueue the payload
                    buffer.enqueue(payload);
                    // ensure an invocation timeout is set
                    _setupTimer();
                }
                catch (e) {
                    _throwInternal(itemCtx.diagLog(), 2 /* eLoggingSeverity.WARNING */, 12 /* _eInternalMessageId.FailedAddingTelemetryToBuffer */, "Failed adding telemetry to the sender's buffer, some telemetry will be lost: " + getExceptionName(e), { exception: dumpObj(e) });
                }
                // hand off the telemetry item to the next plugin
                _self.processNext(telemetryItem, itemCtx);
            };
            /**
             * xhr state changes
             */
            _self._xhrReadyStateChange = function (xhr, payload, countOfItemsInPayload) {
                if (xhr.readyState === 4) {
                    _checkResponsStatus(xhr.status, payload, xhr.responseURL, countOfItemsInPayload, _formatErrorMessageXhr(xhr), _getResponseText(xhr) || xhr.response);
                }
            };
            /**
             * Immediately send buffered data
             * @param async {boolean} - Indicates if the events should be sent asynchronously
             * @param forcedSender {SenderFunction} - Indicates the forcedSender, undefined if not passed
             */
            _self.triggerSend = function (async, forcedSender, sendReason) {
                if (async === void 0) { async = true; }
                if (!_paused) {
                    try {
                        var buffer = _self._buffer;
                        // Send data only if disableTelemetry is false
                        if (!_self._senderConfig.disableTelemetry()) {
                            if (buffer.count() > 0) {
                                var payload = buffer.getItems();
                                _notifySendRequest(sendReason || 0 /* SendRequestReason.Undefined */, async);
                                // invoke send
                                if (forcedSender) {
                                    forcedSender.call(_this, payload, async);
                                }
                                else {
                                    _self._sender(payload, async);
                                }
                            }
                            // update lastSend time to enable throttling
                            _lastSend = +new Date;
                        }
                        else {
                            buffer.clear();
                        }
                        _clearScheduledTimer();
                    }
                    catch (e) {
                        /* Ignore this error for IE under v10 */
                        var ieVer = getIEVersion();
                        if (!ieVer || ieVer > 9) {
                            _throwInternal(_self.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 40 /* _eInternalMessageId.TransmissionFailed */, "Telemetry transmission failed, some telemetry will be lost: " + getExceptionName(e), { exception: dumpObj(e) });
                        }
                    }
                }
            };
            _self._doTeardown = function (unloadCtx, unloadState) {
                _self.onunloadFlush();
                _offlineListener.unload();
                _initDefaults();
            };
            /**
             * error handler
             */
            _self._onError = function (payload, message, event) {
                _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 26 /* _eInternalMessageId.OnError */, "Failed to send telemetry.", { message: message });
                _self._buffer.clearSent(payload);
            };
            /**
             * partial success handler
             */
            _self._onPartialSuccess = function (payload, results) {
                var failed = [];
                var retry = [];
                // Iterate through the reversed array of errors so that splicing doesn't have invalid indexes after the first item.
                var errors = results.errors.reverse();
                for (var _i = 0, errors_1 = errors; _i < errors_1.length; _i++) {
                    var error = errors_1[_i];
                    var extracted = payload.splice(error.index, 1)[0];
                    if (_isRetriable(error.statusCode)) {
                        retry.push(extracted);
                    }
                    else {
                        // All other errors, including: 402 (Monthly quota exceeded) and 439 (Too many requests and refresh cache).
                        failed.push(extracted);
                    }
                }
                if (payload.length > 0) {
                    _self._onSuccess(payload, results.itemsAccepted);
                }
                if (failed.length > 0) {
                    _self._onError(failed, _formatErrorMessageXhr(null, ["partial success", results.itemsAccepted, "of", results.itemsReceived].join(" ")));
                }
                if (retry.length > 0) {
                    _resendPayload(retry);
                    _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 40 /* _eInternalMessageId.TransmissionFailed */, "Partial success. " +
                        "Delivered: " + payload.length + ", Failed: " + failed.length +
                        ". Will retry to send " + retry.length + " our of " + results.itemsReceived + " items");
                }
            };
            /**
             * success handler
             */
            _self._onSuccess = function (payload, countOfItemsInPayload) {
                _self._buffer.clearSent(payload);
            };
            /**
             * xdr state changes
             */
            _self._xdrOnLoad = function (xdr, payload) {
                var responseText = _getResponseText(xdr);
                if (xdr && (responseText + "" === "200" || responseText === "")) {
                    _consecutiveErrors = 0;
                    _self._onSuccess(payload, 0);
                }
                else {
                    var results = _parseResponse(responseText);
                    if (results && results.itemsReceived && results.itemsReceived > results.itemsAccepted
                        && !_self._senderConfig.isRetryDisabled()) {
                        _self._onPartialSuccess(payload, results);
                    }
                    else {
                        _self._onError(payload, _formatErrorMessageXdr(xdr));
                    }
                }
            };
            function _isSampledIn(envelope) {
                return _self._sample.isSampledIn(envelope);
            }
            function _checkResponsStatus(status, payload, responseUrl, countOfItemsInPayload, errorMessage, res) {
                var response = null;
                if (!_self._appId) {
                    response = _parseResponse(res);
                    if (response && response.appId) {
                        _self._appId = response.appId;
                    }
                }
                if ((status < 200 || status >= 300) && status !== 0) {
                    // Update End Point url if permanent redirect or moved permanently
                    // Updates the end point url before retry
                    if (status === 301 || status === 307 || status === 308) {
                        if (!_checkAndUpdateEndPointUrl(responseUrl)) {
                            _self._onError(payload, errorMessage);
                            return;
                        }
                    }
                    if (!_self._senderConfig.isRetryDisabled() && _isRetriable(status)) {
                        _resendPayload(payload);
                        _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 40 /* _eInternalMessageId.TransmissionFailed */, ". " +
                            "Response code " + status + ". Will retry to send " + payload.length + " items.");
                    }
                    else {
                        _self._onError(payload, errorMessage);
                    }
                }
                else if (_offlineListener && !_offlineListener.isOnline()) { // offline
                    // Note: Don't check for status == 0, since adblock gives this code
                    if (!_self._senderConfig.isRetryDisabled()) {
                        var offlineBackOffMultiplier = 10; // arbritrary number
                        _resendPayload(payload, offlineBackOffMultiplier);
                        _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 40 /* _eInternalMessageId.TransmissionFailed */, ". Offline - Response Code: ".concat(status, ". Offline status: ").concat(!_offlineListener.isOnline(), ". Will retry to send ").concat(payload.length, " items."));
                    }
                }
                else {
                    // check if the xhr's responseURL or fetch's response.url is same as endpoint url
                    // TODO after 10 redirects force send telemetry with 'redirect=false' as query parameter.
                    _checkAndUpdateEndPointUrl(responseUrl);
                    if (status === 206) {
                        if (!response) {
                            response = _parseResponse(res);
                        }
                        if (response && !_self._senderConfig.isRetryDisabled()) {
                            _self._onPartialSuccess(payload, response);
                        }
                        else {
                            _self._onError(payload, errorMessage);
                        }
                    }
                    else {
                        _consecutiveErrors = 0;
                        _self._onSuccess(payload, countOfItemsInPayload);
                    }
                }
            }
            function _checkAndUpdateEndPointUrl(responseUrl) {
                // Maximum stamp specific redirects allowed(uncomment this when breeze is ready with not allowing redirects feature)
                if (_stamp_specific_redirects >= 10) {
                    //  _self._senderConfig.endpointUrl = () => Sender._getDefaultAppInsightsChannelConfig().endpointUrl()+"/?redirect=false";
                    //  _stamp_specific_redirects = 0;
                    return false;
                }
                if (!isNullOrUndefined(responseUrl) && responseUrl !== "") {
                    if (responseUrl !== _self._senderConfig.endpointUrl()) {
                        _self._senderConfig.endpointUrl = function () { return responseUrl; };
                        ++_stamp_specific_redirects;
                        return true;
                    }
                }
                return false;
            }
            function _doUnloadSend(payload, isAsync) {
                if (_syncUnloadSender) {
                    // We are unloading so always call the sender with sync set to false
                    _syncUnloadSender(payload, false);
                }
                else {
                    // Fallback to the previous beacon Sender (which causes a CORB warning on chrome now)
                    _beaconSender(payload, isAsync);
                }
            }
            function _doBeaconSend(payload) {
                var nav = getNavigator();
                var buffer = _self._buffer;
                var url = _self._senderConfig.endpointUrl();
                var batch = _self._buffer.batchPayloads(payload);
                // Chrome only allows CORS-safelisted values for the sendBeacon data argument
                // see: https://bugs.chromium.org/p/chromium/issues/detail?id=720283
                var plainTextBatch = new Blob([batch], { type: "text/plain;charset=UTF-8" });
                // The sendBeacon method returns true if the user agent is able to successfully queue the data for transfer. Otherwise it returns false.
                var queued = nav.sendBeacon(url, plainTextBatch);
                if (queued) {
                    buffer.markAsSent(payload);
                    // no response from beaconSender, clear buffer
                    _self._onSuccess(payload, payload.length);
                }
                return queued;
            }
            /**
             * Send Beacon API request
             * @param payload {string} - The data payload to be sent.
             * @param isAsync {boolean} - not used
             * Note: Beacon API does not support custom headers and we are not able to get
             * appId from the backend for the correct correlation.
             */
            function _beaconSender(payload, isAsync) {
                if (isArray(payload) && payload.length > 0) {
                    // The sendBeacon method returns true if the user agent is able to successfully queue the data for transfer. Otherwise it returns false.
                    if (!_doBeaconSend(payload)) {
                        // Failed to send entire payload so try and split data and try to send as much events as possible
                        var droppedPayload = [];
                        for (var lp = 0; lp < payload.length; lp++) {
                            var thePayload = payload[lp];
                            if (!_doBeaconSend([thePayload])) {
                                // Can't send anymore, so split the batch and drop the rest
                                droppedPayload.push(thePayload);
                            }
                        }
                        if (droppedPayload.length > 0) {
                            _fallbackSender && _fallbackSender(droppedPayload, true);
                            _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 40 /* _eInternalMessageId.TransmissionFailed */, ". " + "Failed to send telemetry with Beacon API, retried with normal sender.");
                        }
                    }
                }
            }
            /**
             * Send XMLHttpRequest
             * @param payload {string} - The data payload to be sent.
             * @param isAsync {boolean} - Indicates if the request should be sent asynchronously
             */
            function _xhrSender(payload, isAsync) {
                var xhr = new XMLHttpRequest();
                var endPointUrl = _self._senderConfig.endpointUrl();
                try {
                    xhr[DisabledPropertyName] = true;
                }
                catch (e) {
                    // If the environment has locked down the XMLHttpRequest (preventExtensions and/or freeze), this would
                    // cause the request to fail and we no telemetry would be sent
                }
                xhr.open("POST", endPointUrl, isAsync);
                xhr.setRequestHeader("Content-type", "application/json");
                // append Sdk-Context request header only in case of breeze endpoint
                if (isInternalApplicationInsightsEndpoint(endPointUrl)) {
                    xhr.setRequestHeader(RequestHeaders.sdkContextHeader, RequestHeaders.sdkContextHeaderAppIdRequest);
                }
                arrForEach(objKeys(_headers), function (headerName) {
                    xhr.setRequestHeader(headerName, _headers[headerName]);
                });
                xhr.onreadystatechange = function () { return _self._xhrReadyStateChange(xhr, payload, payload.length); };
                xhr.onerror = function (event) { return _self._onError(payload, _formatErrorMessageXhr(xhr), event); };
                // compose an array of payloads
                var batch = _self._buffer.batchPayloads(payload);
                xhr.send(batch);
                _self._buffer.markAsSent(payload);
            }
            function _fetchKeepAliveSender(payload, isAsync) {
                if (isArray(payload)) {
                    var payloadSize = payload.length;
                    for (var lp = 0; lp < payload.length; lp++) {
                        payloadSize += payload[lp].length;
                    }
                    if ((_syncFetchPayload + payloadSize) <= FetchSyncRequestSizeLimitBytes) {
                        _doFetchSender(payload, false);
                    }
                    else if (isBeaconsSupported()) {
                        // Fallback to beacon sender as we at least get told which events can't be scheduled
                        _beaconSender(payload, isAsync);
                    }
                    else {
                        // Payload is going to be too big so just try and send via XHR
                        _fallbackSender && _fallbackSender(payload, true);
                        _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 40 /* _eInternalMessageId.TransmissionFailed */, ". " + "Failed to send telemetry with Beacon API, retried with xhrSender.");
                    }
                }
            }
            /**
             * Send fetch API request
             * @param payload {string} - The data payload to be sent.
             * @param isAsync {boolean} - not used
             */
            function _fetchSender(payload, isAsync) {
                _doFetchSender(payload, true);
            }
            /**
             * Send fetch API request
             * @param payload {string} - The data payload to be sent.
             * @param isAsync {boolean} - For fetch this identifies whether we are "unloading" (false) or a normal request
             */
            function _doFetchSender(payload, isAsync) {
                var _a;
                var endPointUrl = _self._senderConfig.endpointUrl();
                var batch = _self._buffer.batchPayloads(payload);
                var plainTextBatch = new Blob([batch], { type: "application/json" });
                var requestHeaders = new Headers();
                var batchLength = batch.length;
                var ignoreResponse = false;
                var responseHandled = false;
                // append Sdk-Context request header only in case of breeze endpoint
                if (isInternalApplicationInsightsEndpoint(endPointUrl)) {
                    requestHeaders.append(RequestHeaders.sdkContextHeader, RequestHeaders.sdkContextHeaderAppIdRequest);
                }
                arrForEach(objKeys(_headers), function (headerName) {
                    requestHeaders.append(headerName, _headers[headerName]);
                });
                var init = (_a = {
                        method: "POST",
                        headers: requestHeaders,
                        body: plainTextBatch
                    },
                    _a[DisabledPropertyName] = true // Mark so we don't attempt to track this request
                ,
                    _a);
                if (!isAsync) {
                    init.keepalive = true;
                    // As a sync request (during unload), it is unlikely that we will get a chance to process the response so
                    // just like beacon send assume that the events have been accepted and processed
                    ignoreResponse = true;
                    _syncFetchPayload += batchLength;
                }
                var request = new Request(endPointUrl, init);
                try {
                    // Also try and tag the request (just in case the value in init is not copied over)
                    request[DisabledPropertyName] = true;
                }
                catch (e) {
                    // If the environment has locked down the XMLHttpRequest (preventExtensions and/or freeze), this would
                    // cause the request to fail and we no telemetry would be sent
                }
                _self._buffer.markAsSent(payload);
                try {
                    fetch(request).then(function (response) {
                        if (!isAsync) {
                            _syncFetchPayload -= batchLength;
                            batchLength = 0;
                        }
                        if (!responseHandled) {
                            responseHandled = true;
                            /**
                             * The Promise returned from fetch() won’t reject on HTTP error status even if the response is an HTTP 404 or 500.
                             * Instead, it will resolve normally (with ok status set to false), and it will only reject on network failure
                             * or if anything prevented the request from completing.
                             */
                            if (!response.ok) {
                                _self._onError(payload, response.statusText);
                            }
                            else {
                                response.text().then(function (text) {
                                    _checkResponsStatus(response.status, payload, response.url, payload.length, response.statusText, text);
                                });
                            }
                        }
                    })["catch"](function (error) {
                        if (!isAsync) {
                            _syncFetchPayload -= batchLength;
                            batchLength = 0;
                        }
                        if (!responseHandled) {
                            responseHandled = true;
                            _self._onError(payload, error.message);
                        }
                    });
                }
                catch (e) {
                    if (!responseHandled) {
                        _self._onError(payload, dumpObj(e));
                    }
                }
                if (ignoreResponse && !responseHandled) {
                    // Assume success during unload processing as we most likely won't get the response
                    responseHandled = true;
                    _self._onSuccess(payload, payload.length);
                }
            }
            /**
             * Parses the response from the backend.
             * @param response - XMLHttpRequest or XDomainRequest response
             */
            function _parseResponse(response) {
                try {
                    if (response && response !== "") {
                        var result = getJSON().parse(response);
                        if (result && result.itemsReceived && result.itemsReceived >= result.itemsAccepted &&
                            result.itemsReceived - result.itemsAccepted === result.errors.length) {
                            return result;
                        }
                    }
                }
                catch (e) {
                    _throwInternal(_self.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 43 /* _eInternalMessageId.InvalidBackendResponse */, "Cannot parse the response. " + getExceptionName(e), {
                        response: response
                    });
                }
                return null;
            }
            /**
             * Resend payload. Adds payload back to the send buffer and setup a send timer (with exponential backoff).
             * @param payload
             */
            function _resendPayload(payload, linearFactor) {
                if (linearFactor === void 0) { linearFactor = 1; }
                if (!payload || payload.length === 0) {
                    return;
                }
                var buffer = _self._buffer;
                buffer.clearSent(payload);
                _consecutiveErrors++;
                for (var _i = 0, payload_1 = payload; _i < payload_1.length; _i++) {
                    var item = payload_1[_i];
                    buffer.enqueue(item);
                }
                // setup timer
                _setRetryTime(linearFactor);
                _setupTimer();
            }
            /**
             * Calculates the time to wait before retrying in case of an error based on
             * http://en.wikipedia.org/wiki/Exponential_backoff
             */
            function _setRetryTime(linearFactor) {
                var SlotDelayInSeconds = 10;
                var delayInSeconds;
                if (_consecutiveErrors <= 1) {
                    delayInSeconds = SlotDelayInSeconds;
                }
                else {
                    var backOffSlot = (Math.pow(2, _consecutiveErrors) - 1) / 2;
                    // tslint:disable-next-line:insecure-random
                    var backOffDelay = Math.floor(Math.random() * backOffSlot * SlotDelayInSeconds) + 1;
                    backOffDelay = linearFactor * backOffDelay;
                    delayInSeconds = Math.max(Math.min(backOffDelay, 3600), SlotDelayInSeconds);
                }
                // TODO: Log the backoff time like the C# version does.
                var retryAfterTimeSpan = dateNow() + (delayInSeconds * 1000);
                // TODO: Log the retry at time like the C# version does.
                _retryAt = retryAfterTimeSpan;
            }
            /**
             * Sets up the timer which triggers actually sending the data.
             */
            function _setupTimer() {
                if (!_timeoutHandle && !_paused) {
                    var retryInterval = _retryAt ? Math.max(0, _retryAt - dateNow()) : 0;
                    var timerValue = Math.max(_self._senderConfig.maxBatchInterval(), retryInterval);
                    _timeoutHandle = setTimeout(function () {
                        _timeoutHandle = null;
                        _self.triggerSend(true, null, 1 /* SendRequestReason.NormalSchedule */);
                    }, timerValue);
                }
            }
            function _clearScheduledTimer() {
                clearTimeout(_timeoutHandle);
                _timeoutHandle = null;
                _retryAt = null;
            }
            /**
             * Checks if the SDK should resend the payload after receiving this status code from the backend.
             * @param statusCode
             */
            function _isRetriable(statusCode) {
                return statusCode === 408 // Timeout
                    || statusCode === 429 // Too many requests.
                    || statusCode === 500 // Internal server error.
                    || statusCode === 503; // Service unavailable.
            }
            function _formatErrorMessageXhr(xhr, message) {
                if (xhr) {
                    return "XMLHttpRequest,Status:" + xhr.status + ",Response:" + _getResponseText(xhr) || xhr.response || "";
                }
                return message;
            }
            /**
             * Send XDomainRequest
             * @param payload {string} - The data payload to be sent.
             * @param isAsync {boolean} - Indicates if the request should be sent asynchronously
             *
             * Note: XDomainRequest does not support sync requests. This 'isAsync' parameter is added
             * to maintain consistency with the xhrSender's contract
             * Note: XDomainRequest does not support custom headers and we are not able to get
             * appId from the backend for the correct correlation.
             */
            function _xdrSender(payload, isAsync) {
                var buffer = _self._buffer;
                var _window = getWindow();
                var xdr = new XDomainRequest();
                xdr.onload = function () { return _self._xdrOnLoad(xdr, payload); };
                xdr.onerror = function (event) { return _self._onError(payload, _formatErrorMessageXdr(xdr), event); };
                // XDomainRequest requires the same protocol as the hosting page.
                // If the protocol doesn't match, we can't send the telemetry :(.
                var hostingProtocol = _window && _window.location && _window.location.protocol || "";
                if (_self._senderConfig.endpointUrl().lastIndexOf(hostingProtocol, 0) !== 0) {
                    _throwInternal(_self.diagLog(), 2 /* eLoggingSeverity.WARNING */, 40 /* _eInternalMessageId.TransmissionFailed */, ". " +
                        "Cannot send XDomain request. The endpoint URL protocol doesn't match the hosting page protocol.");
                    buffer.clear();
                    return;
                }
                var endpointUrl = _self._senderConfig.endpointUrl().replace(/^(https?:)/, "");
                xdr.open("POST", endpointUrl);
                // compose an array of payloads
                var batch = buffer.batchPayloads(payload);
                xdr.send(batch);
                buffer.markAsSent(payload);
            }
            function _formatErrorMessageXdr(xdr, message) {
                if (xdr) {
                    return "XDomainRequest,Response:" + _getResponseText(xdr) || "";
                }
                return message;
            }
            // Using function lookups for backward compatibility as the getNotifyMgr() did not exist until after v2.5.6
            function _getNotifyMgr() {
                var func = "getNotifyMgr";
                if (_self.core[func]) {
                    return _self.core[func]();
                }
                // using _self.core['_notificationManager'] for backward compatibility
                return _self.core["_notificationManager"];
            }
            function _notifySendRequest(sendRequest, isAsync) {
                var manager = _getNotifyMgr();
                if (manager && manager.eventsSendRequest) {
                    try {
                        manager.eventsSendRequest(sendRequest, isAsync);
                    }
                    catch (e) {
                        _throwInternal(_self.diagLog(), 1 /* eLoggingSeverity.CRITICAL */, 74 /* _eInternalMessageId.NotificationException */, "send request notification failed: " + getExceptionName(e), { exception: dumpObj(e) });
                    }
                }
            }
            /**
             * Validate UUID Format
             * Specs taken from https://tools.ietf.org/html/rfc4122 and breeze repo
             */
            function _validateInstrumentationKey(config) {
                var disableIKeyValidationFlag = isNullOrUndefined(config.disableInstrumentationKeyValidation) ? false : config.disableInstrumentationKeyValidation;
                if (disableIKeyValidationFlag) {
                    return true;
                }
                var UUID_Regex = "^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$";
                var regexp = new RegExp(UUID_Regex);
                return regexp.test(config.instrumentationKey);
            }
            function _initDefaults() {
                _self._sender = null;
                _self._buffer = null;
                _self._appId = null;
                _self._sample = null;
                _headers = {};
                _offlineListener = null;
                _consecutiveErrors = 0;
                _retryAt = null;
                _lastSend = null;
                _paused = false;
                _timeoutHandle = null;
                _serializer = null;
                _stamp_specific_redirects = 0;
                _syncFetchPayload = 0;
                _fallbackSender = null;
                _syncUnloadSender = null;
                _evtNamespace = null;
            }
        });
        return _this;
    }
    Sender.constructEnvelope = function (orig, iKey, logger, convertUndefined) {
        var envelope;
        if (iKey !== orig.iKey && !isNullOrUndefined(iKey)) {
            envelope = __assign(__assign({}, orig), { iKey: iKey });
        }
        else {
            envelope = orig;
        }
        var creator = EnvelopeTypeCreator[envelope.baseType] || EventEnvelopeCreator;
        return creator(logger, envelope, convertUndefined);
    };
// Removed Stub for Sender.prototype.pause.
// Removed Stub for Sender.prototype.resume.
// Removed Stub for Sender.prototype.flush.
// Removed Stub for Sender.prototype.onunloadFlush.
// Removed Stub for Sender.prototype.initialize.
// Removed Stub for Sender.prototype.processTelemetry.
// Removed Stub for Sender.prototype._xhrReadyStateChange.
// Removed Stub for Sender.prototype.triggerSend.
// Removed Stub for Sender.prototype._onError.
// Removed Stub for Sender.prototype._onPartialSuccess.
// Removed Stub for Sender.prototype._onSuccess.
// Removed Stub for Sender.prototype._xdrOnLoad.
// Removed Stub for Sender.prototype.addHeader.
    return Sender;
}(BaseTelemetryPlugin));
export { Sender };
