/*
 * Application Insights JavaScript SDK - Common, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 */


import { __assignFn as __assign } from "@microsoft/applicationinsights-shims";
import { dataSanitizeException, dataSanitizeMeasurements, dataSanitizeMessage, dataSanitizeProperties, dataSanitizeString } from "./Common/DataSanitizer";
import { isNullOrUndefined, arrMap, isString, strTrim, isArray, isError, arrForEach, isObject, isFunction } from "@microsoft/applicationinsights-core-js";
import { strNotSpecified } from "../Constants";
var NoMethod = "<no_method>";
var strError = "error";
var strStack = "stack";
var strStackDetails = "stackDetails";
var strErrorSrc = "errorSrc";
var strMessage = "message";
var strDescription = "description";
function _stringify(value, convertToString) {
    var result = value;
    if (result && !isString(result)) {
        if (JSON && JSON.stringify) {
            result = JSON.stringify(value);
            if (convertToString && (!result || result === "{}")) {
                if (isFunction(value.toString)) {
                    result = value.toString();
                }
                else {
                    result = "" + value;
                }
            }
        }
        else {
            result = "" + value + " - (Missing JSON.stringify)";
        }
    }
    return result || "";
}
function _formatMessage(theEvent, errorType) {
    var evtMessage = theEvent;
    if (theEvent) {
        if (evtMessage && !isString(evtMessage)) {
            evtMessage = theEvent[strMessage] || theEvent[strDescription] || evtMessage;
        }
        // Make sure the message is a string
        if (evtMessage && !isString(evtMessage)) {
            // tslint:disable-next-line: prefer-conditional-expression
            evtMessage = _stringify(evtMessage, true);
        }
        if (theEvent["filename"]) {
            // Looks like an event object with filename
            evtMessage = evtMessage + " @" + (theEvent["filename"] || "") + ":" + (theEvent["lineno"] || "?") + ":" + (theEvent["colno"] || "?");
        }
    }
    // Automatically add the error type to the message if it does already appear to be present
    if (errorType && errorType !== "String" && errorType !== "Object" && errorType !== "Error" && (evtMessage || "").indexOf(errorType) === -1) {
        evtMessage = errorType + ": " + evtMessage;
    }
    return evtMessage || "";
}
function _isExceptionDetailsInternal(value) {
    try {
        if (isObject(value)) {
            return "hasFullStack" in value && "typeName" in value;
        }
    }
    catch (e) {
        // This can happen with some native browser objects, but should not happen for the type we are checking for
    }
    return false;
}
function _isExceptionInternal(value) {
    try {
        if (isObject(value)) {
            return ("ver" in value && "exceptions" in value && "properties" in value);
        }
    }
    catch (e) {
        // This can happen with some native browser objects, but should not happen for the type we are checking for
    }
    return false;
}
function _isStackDetails(details) {
    return details && details.src && isString(details.src) && details.obj && isArray(details.obj);
}
function _convertStackObj(errorStack) {
    var src = errorStack || "";
    if (!isString(src)) {
        if (isString(src[strStack])) {
            src = src[strStack];
        }
        else {
            src = "" + src;
        }
    }
    var items = src.split("\n");
    return {
        src: src,
        obj: items
    };
}
function _getOperaStack(errorMessage) {
    var stack = [];
    var lines = errorMessage.split("\n");
    for (var lp = 0; lp < lines.length; lp++) {
        var entry = lines[lp];
        if (lines[lp + 1]) {
            entry += "@" + lines[lp + 1];
            lp++;
        }
        stack.push(entry);
    }
    return {
        src: errorMessage,
        obj: stack
    };
}
function _getStackFromErrorObj(errorObj) {
    var details = null;
    if (errorObj) {
        try {
            /* Using bracket notation is support older browsers (IE 7/8 -- dont remember the version) that throw when using dot
            notation for undefined objects and we don't want to loose the error from being reported */
            if (errorObj[strStack]) {
                // Chrome/Firefox
                details = _convertStackObj(errorObj[strStack]);
            }
            else if (errorObj[strError] && errorObj[strError][strStack]) {
                // Edge error event provides the stack and error object
                details = _convertStackObj(errorObj[strError][strStack]);
            }
            else if (errorObj["exception"] && errorObj.exception[strStack]) {
                details = _convertStackObj(errorObj.exception[strStack]);
            }
            else if (_isStackDetails(errorObj)) {
                details = errorObj;
            }
            else if (_isStackDetails(errorObj[strStackDetails])) {
                details = errorObj[strStackDetails];
            }
            else if (window && window["opera"] && errorObj[strMessage]) {
                // Opera
                details = _getOperaStack(errorObj.message);
            }
            else if (isString(errorObj)) {
                details = _convertStackObj(errorObj);
            }
            else {
                var evtMessage = errorObj[strMessage] || errorObj[strDescription] || "";
                if (isString(errorObj[strErrorSrc])) {
                    if (evtMessage) {
                        evtMessage += "\n";
                    }
                    evtMessage += " from " + errorObj[strErrorSrc];
                }
                if (evtMessage) {
                    details = _convertStackObj(evtMessage);
                }
            }
        }
        catch (e) {
            // something unexpected happened so to avoid failing to report any error lets swallow the exception
            // and fallback to the callee/caller method
            details = _convertStackObj(e);
        }
    }
    return details || {
        src: "",
        obj: null
    };
}
function _formatStackTrace(stackDetails) {
    var stack = "";
    if (stackDetails) {
        if (stackDetails.obj) {
            arrForEach(stackDetails.obj, function (entry) {
                stack += entry + "\n";
            });
        }
        else {
            stack = stackDetails.src || "";
        }
    }
    return stack;
}
function _parseStack(stack) {
    var parsedStack;
    var frames = stack.obj;
    if (frames && frames.length > 0) {
        parsedStack = [];
        var level_1 = 0;
        var totalSizeInBytes_1 = 0;
        arrForEach(frames, function (frame) {
            var theFrame = frame.toString();
            if (_StackFrame.regex.test(theFrame)) {
                var parsedFrame = new _StackFrame(theFrame, level_1++);
                totalSizeInBytes_1 += parsedFrame.sizeInBytes;
                parsedStack.push(parsedFrame);
            }
        });
        // DP Constraint - exception parsed stack must be < 32KB
        // remove frames from the middle to meet the threshold
        var exceptionParsedStackThreshold = 32 * 1024;
        if (totalSizeInBytes_1 > exceptionParsedStackThreshold) {
            var left = 0;
            var right = parsedStack.length - 1;
            var size = 0;
            var acceptedLeft = left;
            var acceptedRight = right;
            while (left < right) {
                // check size
                var lSize = parsedStack[left].sizeInBytes;
                var rSize = parsedStack[right].sizeInBytes;
                size += lSize + rSize;
                if (size > exceptionParsedStackThreshold) {
                    // remove extra frames from the middle
                    var howMany = acceptedRight - acceptedLeft + 1;
                    parsedStack.splice(acceptedLeft, howMany);
                    break;
                }
                // update pointers
                acceptedLeft = left;
                acceptedRight = right;
                left++;
                right--;
            }
        }
    }
    return parsedStack;
}
function _getErrorType(errorType) {
    // Gets the Error Type by passing the constructor (used to get the true type of native error object).
    var typeName = "";
    if (errorType) {
        typeName = errorType.typeName || errorType.name || "";
        if (!typeName) {
            try {
                var funcNameRegex = /function (.{1,200})\(/;
                var results = (funcNameRegex).exec((errorType).constructor.toString());
                typeName = (results && results.length > 1) ? results[1] : "";
            }
            catch (e) {
                // eslint-disable-next-line no-empty -- Ignoring any failures as nothing we can do
            }
        }
    }
    return typeName;
}
/**
 * Formats the provided errorObj for display and reporting, it may be a String, Object, integer or undefined depending on the browser.
 * @param errorObj The supplied errorObj
 */
export function _formatErrorCode(errorObj) {
    if (errorObj) {
        try {
            if (!isString(errorObj)) {
                var errorType = _getErrorType(errorObj);
                var result = _stringify(errorObj, false);
                if (!result || result === "{}") {
                    if (errorObj[strError]) {
                        // Looks like an MS Error Event
                        errorObj = errorObj[strError];
                        errorType = _getErrorType(errorObj);
                    }
                    result = _stringify(errorObj, true);
                }
                if (result.indexOf(errorType) !== 0 && errorType !== "String") {
                    return errorType + ":" + result;
                }
                return result;
            }
        }
        catch (e) {
            // eslint-disable-next-line no-empty -- Ignoring any failures as nothing we can do
        }
    }
    // Fallback to just letting the object format itself into a string
    return "" + (errorObj || "");
}
var Exception = /** @class */ (function () {
    /**
     * Constructs a new instance of the ExceptionTelemetry object
     */
    function Exception(logger, exception, properties, measurements, severityLevel, id) {
        this.aiDataContract = {
            ver: 1 /* FieldType.Required */,
            exceptions: 1 /* FieldType.Required */,
            severityLevel: 0 /* FieldType.Default */,
            properties: 0 /* FieldType.Default */,
            measurements: 0 /* FieldType.Default */
        };
        var _self = this;
        _self.ver = 2; // TODO: handle the CS"4.0" ==> breeze 2 conversion in a better way
        if (!_isExceptionInternal(exception)) {
            if (!properties) {
                properties = {};
            }
            _self.exceptions = [new _ExceptionDetails(logger, exception, properties)];
            _self.properties = dataSanitizeProperties(logger, properties);
            _self.measurements = dataSanitizeMeasurements(logger, measurements);
            if (severityLevel) {
                _self.severityLevel = severityLevel;
            }
            if (id) {
                _self.id = id;
            }
        }
        else {
            _self.exceptions = exception.exceptions || [];
            _self.properties = exception.properties;
            _self.measurements = exception.measurements;
            if (exception.severityLevel) {
                _self.severityLevel = exception.severityLevel;
            }
            if (exception.id) {
                _self.id = exception.id;
            }
            if (exception.problemGroup) {
                _self.problemGroup = exception.problemGroup;
            }
            // bool/int types, use isNullOrUndefined
            if (!isNullOrUndefined(exception.isManual)) {
                _self.isManual = exception.isManual;
            }
        }
    }
    Exception.CreateAutoException = function (message, url, lineNumber, columnNumber, error, evt, stack, errorSrc) {
        var errorType = _getErrorType(error || evt || message);
        return {
            message: _formatMessage(message, errorType),
            url: url,
            lineNumber: lineNumber,
            columnNumber: columnNumber,
            error: _formatErrorCode(error || evt || message),
            evt: _formatErrorCode(evt || message),
            typeName: errorType,
            stackDetails: _getStackFromErrorObj(stack || error || evt),
            errorSrc: errorSrc
        };
    };
    Exception.CreateFromInterface = function (logger, exception, properties, measurements) {
        var exceptions = exception.exceptions
            && arrMap(exception.exceptions, function (ex) { return _ExceptionDetails.CreateFromInterface(logger, ex); });
        var exceptionData = new Exception(logger, __assign(__assign({}, exception), { exceptions: exceptions }), properties, measurements);
        return exceptionData;
    };
    Exception.prototype.toInterface = function () {
        var _a = this, exceptions = _a.exceptions, properties = _a.properties, measurements = _a.measurements, severityLevel = _a.severityLevel, problemGroup = _a.problemGroup, id = _a.id, isManual = _a.isManual;
        var exceptionDetailsInterface = exceptions instanceof Array
            && arrMap(exceptions, function (exception) { return exception.toInterface(); })
            || undefined;
        return {
            ver: "4.0",
            exceptions: exceptionDetailsInterface,
            severityLevel: severityLevel,
            properties: properties,
            measurements: measurements,
            problemGroup: problemGroup,
            id: id,
            isManual: isManual
        };
    };
    /**
     * Creates a simple exception with 1 stack frame. Useful for manual constracting of exception.
     */
    Exception.CreateSimpleException = function (message, typeName, assembly, fileName, details, line) {
        return {
            exceptions: [
                {
                    hasFullStack: true,
                    message: message,
                    stack: details,
                    typeName: typeName
                }
            ]
        };
    };
    Exception.envelopeType = "Microsoft.ApplicationInsights.{0}.Exception";
    Exception.dataType = "ExceptionData";
    Exception.formatError = _formatErrorCode;
    return Exception;
}());
export { Exception };
var _ExceptionDetails = /** @class */ (function () {
    function _ExceptionDetails(logger, exception, properties) {
        this.aiDataContract = {
            id: 0 /* FieldType.Default */,
            outerId: 0 /* FieldType.Default */,
            typeName: 1 /* FieldType.Required */,
            message: 1 /* FieldType.Required */,
            hasFullStack: 0 /* FieldType.Default */,
            stack: 0 /* FieldType.Default */,
            parsedStack: 2 /* FieldType.Array */
        };
        var _self = this;
        if (!_isExceptionDetailsInternal(exception)) {
            var error = exception;
            var evt = error && error.evt;
            if (!isError(error)) {
                error = error[strError] || evt || error;
            }
            _self.typeName = dataSanitizeString(logger, _getErrorType(error)) || strNotSpecified;
            _self.message = dataSanitizeMessage(logger, _formatMessage(exception || error, _self.typeName)) || strNotSpecified;
            var stack = exception[strStackDetails] || _getStackFromErrorObj(exception);
            _self.parsedStack = _parseStack(stack);
            _self[strStack] = dataSanitizeException(logger, _formatStackTrace(stack));
            _self.hasFullStack = isArray(_self.parsedStack) && _self.parsedStack.length > 0;
            if (properties) {
                properties.typeName = properties.typeName || _self.typeName;
            }
        }
        else {
            _self.typeName = exception.typeName;
            _self.message = exception.message;
            _self[strStack] = exception[strStack];
            _self.parsedStack = exception.parsedStack || [];
            _self.hasFullStack = exception.hasFullStack;
        }
    }
    _ExceptionDetails.prototype.toInterface = function () {
        var _self = this;
        var parsedStack = _self.parsedStack instanceof Array
            && arrMap(_self.parsedStack, function (frame) { return frame.toInterface(); });
        var exceptionDetailsInterface = {
            id: _self.id,
            outerId: _self.outerId,
            typeName: _self.typeName,
            message: _self.message,
            hasFullStack: _self.hasFullStack,
            stack: _self[strStack],
            parsedStack: parsedStack || undefined
        };
        return exceptionDetailsInterface;
    };
    _ExceptionDetails.CreateFromInterface = function (logger, exception) {
        var parsedStack = (exception.parsedStack instanceof Array
            && arrMap(exception.parsedStack, function (frame) { return _StackFrame.CreateFromInterface(frame); }))
            || exception.parsedStack;
        var exceptionDetails = new _ExceptionDetails(logger, __assign(__assign({}, exception), { parsedStack: parsedStack }));
        return exceptionDetails;
    };
    return _ExceptionDetails;
}());
export { _ExceptionDetails };
var _StackFrame = /** @class */ (function () {
    function _StackFrame(sourceFrame, level) {
        this.aiDataContract = {
            level: 1 /* FieldType.Required */,
            method: 1 /* FieldType.Required */,
            assembly: 0 /* FieldType.Default */,
            fileName: 0 /* FieldType.Default */,
            line: 0 /* FieldType.Default */
        };
        var _self = this;
        _self.sizeInBytes = 0;
        // Not converting this to isString() as typescript uses this logic to "understand" the different
        // types for the 2 different code paths
        if (typeof sourceFrame === "string") {
            var frame = sourceFrame;
            _self.level = level;
            _self.method = NoMethod;
            _self.assembly = strTrim(frame);
            _self.fileName = "";
            _self.line = 0;
            var matches = frame.match(_StackFrame.regex);
            if (matches && matches.length >= 5) {
                _self.method = strTrim(matches[2]) || _self.method;
                _self.fileName = strTrim(matches[4]);
                _self.line = parseInt(matches[5]) || 0;
            }
        }
        else {
            _self.level = sourceFrame.level;
            _self.method = sourceFrame.method;
            _self.assembly = sourceFrame.assembly;
            _self.fileName = sourceFrame.fileName;
            _self.line = sourceFrame.line;
            _self.sizeInBytes = 0;
        }
        _self.sizeInBytes += _self.method.length;
        _self.sizeInBytes += _self.fileName.length;
        _self.sizeInBytes += _self.assembly.length;
        // todo: these might need to be removed depending on how the back-end settles on their size calculation
        _self.sizeInBytes += _StackFrame.baseSize;
        _self.sizeInBytes += _self.level.toString().length;
        _self.sizeInBytes += _self.line.toString().length;
    }
    _StackFrame.CreateFromInterface = function (frame) {
        return new _StackFrame(frame, null /* level is available in frame interface */);
    };
    _StackFrame.prototype.toInterface = function () {
        var _self = this;
        return {
            level: _self.level,
            method: _self.method,
            assembly: _self.assembly,
            fileName: _self.fileName,
            line: _self.line
        };
    };
    // regex to match stack frames from ie/chrome/ff
    // methodName=$2, fileName=$4, lineNo=$5, column=$6
    _StackFrame.regex = /^([\s]+at)?[\s]{0,50}([^\@\()]+?)[\s]{0,50}(\@|\()([^\(\n]+):([0-9]+):([0-9]+)(\)?)$/;
    _StackFrame.baseSize = 58; // '{"method":"","level":,"assembly":"","fileName":"","line":}'.length
    return _StackFrame;
}());
export { _StackFrame };
