/*
 * Application Insights JavaScript SDK - Dependencies Plugin, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 */


import { __assignFn as __assign, __extendsFn as __extends } from "@microsoft/applicationinsights-shims";
import { RequestHeaders, CorrelationIdHelper, createTelemetryItem, RemoteDependencyData, dateTimeUtilsNow, DisabledPropertyName, PropertiesPluginIdentifier, isInternalApplicationInsightsEndpoint, formatTraceParent, createTraceParent, createDistributedTraceContextFromTrace } from "@microsoft/applicationinsights-common";
import { isNullOrUndefined, arrForEach, isString, strTrim, isFunction, BaseTelemetryPlugin, getLocation, getGlobal, strPrototype, InstrumentFunc, InstrumentProto, getPerformance, objForEachKey, generateW3CId, getIEVersion, dumpObj, isXhrSupported, eventOn, mergeEvtNamespace, createUniqueNamespace, createProcessTelemetryContext, _throwInternal, getExceptionName } from "@microsoft/applicationinsights-core-js";
import { ajaxRecord } from "./ajaxRecord";
import dynamicProto from "@microsoft/dynamicproto-js";
var AJAX_MONITOR_PREFIX = "ai.ajxmn.";
var strDiagLog = "diagLog";
var strAjaxData = "ajaxData";
var strFetch = "fetch";
var strTrackDependencyDataInternal = "trackDependencyDataInternal"; // Using string to help with minification
// Using a global value so that to handle same iKey with multiple app insights instances (mostly for testing)
var _markCount = 0;
/** @Ignore */
function _supportsFetch() {
    var _global = getGlobal();
    if (!_global ||
        isNullOrUndefined(_global.Request) ||
        isNullOrUndefined(_global.Request[strPrototype]) ||
        isNullOrUndefined(_global[strFetch])) {
        return null;
    }
    return _global[strFetch];
}
/**
 * Determines whether ajax monitoring can be enabled on this document
 * @returns True if Ajax monitoring is supported on this page, otherwise false
 * @ignore
 */
function _supportsAjaxMonitoring(ajaxMonitorInstance) {
    var result = false;
    if (isXhrSupported()) {
        var proto = XMLHttpRequest[strPrototype];
        result = !isNullOrUndefined(proto) &&
            !isNullOrUndefined(proto.open) && // eslint-disable-line security/detect-non-literal-fs-filename -- false positive
            !isNullOrUndefined(proto.send) &&
            !isNullOrUndefined(proto.abort);
    }
    var ieVer = getIEVersion();
    if (ieVer && ieVer < 9) {
        result = false;
    }
    if (result) {
        // Disable if the XmlHttpRequest can't be extended or hooked
        try {
            var xhr = new XMLHttpRequest();
            xhr[strAjaxData] = {};
            // Check that we can update the prototype
            var theOpen = XMLHttpRequest[strPrototype].open;
            XMLHttpRequest[strPrototype].open = theOpen;
        }
        catch (e) {
            // We can't decorate the xhr object so disable monitoring
            result = false;
            _throwInternalCritical(ajaxMonitorInstance, 15 /* _eInternalMessageId.FailedMonitorAjaxOpen */, "Failed to enable XMLHttpRequest monitoring, extension is not supported", {
                exception: dumpObj(e)
            });
        }
    }
    return result;
}
/** @Ignore */
function _getFailedAjaxDiagnosticsMessage(xhr) {
    var result = "";
    try {
        if (!isNullOrUndefined(xhr) &&
            !isNullOrUndefined(xhr[strAjaxData]) &&
            !isNullOrUndefined(xhr[strAjaxData].requestUrl)) {
            result += "(url: '" + xhr[strAjaxData].requestUrl + "')";
        }
    }
    catch (e) {
        // eslint-disable-next-line no-empty
    }
    return result;
}
/** @ignore */
function _throwInternalCritical(ajaxMonitorInstance, msgId, message, properties, isUserAct) {
    _throwInternal(ajaxMonitorInstance[strDiagLog](), 1 /* eLoggingSeverity.CRITICAL */, msgId, message, properties, isUserAct);
}
/** @ignore */
function _throwInternalWarning(ajaxMonitorInstance, msgId, message, properties, isUserAct) {
    _throwInternal(ajaxMonitorInstance[strDiagLog](), 2 /* eLoggingSeverity.WARNING */, msgId, message, properties, isUserAct);
}
/** @Ignore */
function _createErrorCallbackFunc(ajaxMonitorInstance, internalMessage, message) {
    // tslint:disable-next-line
    return function (args) {
        _throwInternalCritical(ajaxMonitorInstance, internalMessage, message, {
            ajaxDiagnosticsMessage: _getFailedAjaxDiagnosticsMessage(args.inst),
            exception: dumpObj(args.err)
        });
    };
}
function _indexOf(value, match) {
    if (value && match) {
        return value.indexOf(match);
    }
    return -1;
}
function _processDependencyListeners(listeners, core, ajaxData, xhr, input, init) {
    var initializersCount = listeners.length;
    if (initializersCount > 0) {
        var details = {
            core: core,
            xhr: xhr,
            input: input,
            init: init,
            traceId: ajaxData.traceID,
            spanId: ajaxData.spanID,
            traceFlags: ajaxData.traceFlags
        };
        for (var i = 0; i < initializersCount; ++i) {
            var dependencyListener = listeners[i];
            if (dependencyListener && dependencyListener.fn) {
                try {
                    dependencyListener.fn.call(null, details);
                }
                catch (e) {
                    var core_1 = details.core;
                    _throwInternal(core_1 && core_1.logger, 1 /* eLoggingSeverity.CRITICAL */, 64 /* _eInternalMessageId.TelemetryInitializerFailed */, "Dependency listener [#" + i + "] failed: " + getExceptionName(e), { exception: dumpObj(e) }, true);
                }
            }
        }
        ajaxData.traceID = details.traceId;
        ajaxData.spanID = details.spanId;
        ajaxData.traceFlags = details.traceFlags;
    }
}
var AjaxMonitor = /** @class */ (function (_super) {
    __extends(AjaxMonitor, _super);
    function AjaxMonitor() {
        var _this = _super.call(this) || this;
        _this.identifier = AjaxMonitor.identifier;
        _this.priority = 120;
        var _fetchInitialized; // fetch monitoring initialized
        var _xhrInitialized; // XHR monitoring initialized
        var _currentWindowHost;
        var _config;
        var _enableRequestHeaderTracking;
        var _enableAjaxErrorStatusText;
        var _trackAjaxAttempts;
        var _context;
        var _isUsingW3CHeaders;
        var _isUsingAIHeaders;
        var _markPrefix;
        var _enableAjaxPerfTracking;
        var _maxAjaxCallsPerView;
        var _enableResponseHeaderTracking;
        var _disabledUrls;
        var _disableAjaxTracking;
        var _disableFetchTracking;
        var _excludeRequestFromAutoTrackingPatterns;
        var _addRequestContext;
        var _evtNamespace;
        var _dependencyListenerId;
        var _dependencyListeners;
        dynamicProto(AjaxMonitor, _this, function (_self, _base) {
            var _addHook = _base._addHook;
            _initDefaults();
            _self.initialize = function (config, core, extensions, pluginChain) {
                if (!_self.isInitialized()) {
                    _base.initialize(config, core, extensions, pluginChain);
                    _evtNamespace = mergeEvtNamespace(createUniqueNamespace("ajax"), core && core.evtNamespace && core.evtNamespace());
                    _populateDefaults(config);
                    _instrumentXhr();
                    _instrumentFetch();
                    _populateContext();
                }
            };
            _self._doTeardown = function () {
                _initDefaults();
            };
            _self.trackDependencyData = function (dependency, properties) {
                _self[strTrackDependencyDataInternal](dependency, properties);
            };
            _self.includeCorrelationHeaders = function (ajaxData, input, init, xhr) {
                // Test Hook to allow the overriding of the location host
                var currentWindowHost = _self["_currentWindowHost"] || _currentWindowHost;
                _processDependencyListeners(_dependencyListeners, _self.core, ajaxData, xhr, input, init);
                if (input) { // Fetch
                    if (CorrelationIdHelper.canIncludeCorrelationHeader(_config, ajaxData.getAbsoluteUrl(), currentWindowHost)) {
                        if (!init) {
                            init = {};
                        }
                        // init headers override original request headers
                        // so, if they exist use only them, otherwise use request's because they should have been applied in the first place
                        // not using original request headers will result in them being lost
                        init.headers = new Headers(init.headers || (input instanceof Request ? (input.headers || {}) : {}));
                        if (_isUsingAIHeaders) {
                            var id = "|" + ajaxData.traceID + "." + ajaxData.spanID;
                            init.headers.set(RequestHeaders[3 /* eRequestHeaders.requestIdHeader */], id);
                            if (_enableRequestHeaderTracking) {
                                ajaxData.requestHeaders[RequestHeaders[3 /* eRequestHeaders.requestIdHeader */]] = id;
                            }
                        }
                        var appId = _config.appId || (_context && _context.appId());
                        if (appId) {
                            init.headers.set(RequestHeaders[0 /* eRequestHeaders.requestContextHeader */], RequestHeaders[2 /* eRequestHeaders.requestContextAppIdFormat */] + appId);
                            if (_enableRequestHeaderTracking) {
                                ajaxData.requestHeaders[RequestHeaders[0 /* eRequestHeaders.requestContextHeader */]] = RequestHeaders[2 /* eRequestHeaders.requestContextAppIdFormat */] + appId;
                            }
                        }
                        if (_isUsingW3CHeaders) {
                            var traceFlags = ajaxData.traceFlags;
                            if (isNullOrUndefined(traceFlags)) {
                                traceFlags = 0x01;
                            }
                            var traceParent = formatTraceParent(createTraceParent(ajaxData.traceID, ajaxData.spanID, traceFlags));
                            init.headers.set(RequestHeaders[4 /* eRequestHeaders.traceParentHeader */], traceParent);
                            if (_enableRequestHeaderTracking) {
                                ajaxData.requestHeaders[RequestHeaders[4 /* eRequestHeaders.traceParentHeader */]] = traceParent;
                            }
                        }
                    }
                    return init;
                }
                else if (xhr) { // XHR
                    if (CorrelationIdHelper.canIncludeCorrelationHeader(_config, ajaxData.getAbsoluteUrl(), currentWindowHost)) {
                        if (_isUsingAIHeaders) {
                            var id = "|" + ajaxData.traceID + "." + ajaxData.spanID;
                            xhr.setRequestHeader(RequestHeaders[3 /* eRequestHeaders.requestIdHeader */], id);
                            if (_enableRequestHeaderTracking) {
                                ajaxData.requestHeaders[RequestHeaders[3 /* eRequestHeaders.requestIdHeader */]] = id;
                            }
                        }
                        var appId = _config.appId || (_context && _context.appId());
                        if (appId) {
                            xhr.setRequestHeader(RequestHeaders[0 /* eRequestHeaders.requestContextHeader */], RequestHeaders[2 /* eRequestHeaders.requestContextAppIdFormat */] + appId);
                            if (_enableRequestHeaderTracking) {
                                ajaxData.requestHeaders[RequestHeaders[0 /* eRequestHeaders.requestContextHeader */]] = RequestHeaders[2 /* eRequestHeaders.requestContextAppIdFormat */] + appId;
                            }
                        }
                        if (_isUsingW3CHeaders) {
                            var traceFlags = ajaxData.traceFlags;
                            if (isNullOrUndefined(traceFlags)) {
                                traceFlags = 0x01;
                            }
                            var traceParent = formatTraceParent(createTraceParent(ajaxData.traceID, ajaxData.spanID, traceFlags));
                            xhr.setRequestHeader(RequestHeaders[4 /* eRequestHeaders.traceParentHeader */], traceParent);
                            if (_enableRequestHeaderTracking) {
                                ajaxData.requestHeaders[RequestHeaders[4 /* eRequestHeaders.traceParentHeader */]] = traceParent;
                            }
                        }
                    }
                    return xhr;
                }
                return undefined;
            };
            _self[strTrackDependencyDataInternal] = function (dependency, properties, systemProperties) {
                if (_maxAjaxCallsPerView === -1 || _trackAjaxAttempts < _maxAjaxCallsPerView) {
                    // Hack since expected format in w3c mode is |abc.def.
                    // Non-w3c format is |abc.def
                    // @todo Remove if better solution is available, e.g. handle in portal
                    if ((_config.distributedTracingMode === 2 /* eDistributedTracingModes.W3C */
                        || _config.distributedTracingMode === 1 /* eDistributedTracingModes.AI_AND_W3C */)
                        && typeof dependency.id === "string" && dependency.id[dependency.id.length - 1] !== ".") {
                        dependency.id += ".";
                    }
                    if (isNullOrUndefined(dependency.startTime)) {
                        dependency.startTime = new Date();
                    }
                    var item = createTelemetryItem(dependency, RemoteDependencyData.dataType, RemoteDependencyData.envelopeType, _self[strDiagLog](), properties, systemProperties);
                    _self.core.track(item);
                }
                else if (_trackAjaxAttempts === _maxAjaxCallsPerView) {
                    _throwInternalCritical(_self, 55 /* _eInternalMessageId.MaxAjaxPerPVExceeded */, "Maximum ajax per page view limit reached, ajax monitoring is paused until the next trackPageView(). In order to increase the limit set the maxAjaxCallsPerView configuration parameter.", true);
                }
                ++_trackAjaxAttempts;
            };
            _self.addDependencyListener = function (dependencyListener) {
                var theInitializer = {
                    id: _dependencyListenerId++,
                    fn: dependencyListener
                };
                _dependencyListeners.push(theInitializer);
                var handler = {
                    remove: function () {
                        arrForEach(_dependencyListeners, function (initializer, idx) {
                            if (initializer.id === theInitializer.id) {
                                _dependencyListeners.splice(idx, 1);
                                return -1;
                            }
                        });
                    }
                };
                return handler;
            };
            function _initDefaults() {
                var location = getLocation();
                _fetchInitialized = false; // fetch monitoring initialized
                _xhrInitialized = false; // XHR monitoring initialized
                _currentWindowHost = location && location.host && location.host.toLowerCase();
                _config = AjaxMonitor.getEmptyConfig();
                _enableRequestHeaderTracking = false;
                _enableAjaxErrorStatusText = false;
                _trackAjaxAttempts = 0;
                _context = null;
                _isUsingW3CHeaders = false;
                _isUsingAIHeaders = false;
                _markPrefix = null;
                _enableAjaxPerfTracking = false;
                _maxAjaxCallsPerView = 0;
                _enableResponseHeaderTracking = false;
                _disabledUrls = {};
                _disableAjaxTracking = false;
                _disableFetchTracking = false;
                _excludeRequestFromAutoTrackingPatterns = null;
                _addRequestContext = null;
                _evtNamespace = null;
                _dependencyListenerId = 0;
                _dependencyListeners = [];
            }
            function _populateDefaults(config) {
                var ctx = createProcessTelemetryContext(null, config, _self.core);
                // Reset to the empty config
                _config = AjaxMonitor.getEmptyConfig();
                var defaultConfig = AjaxMonitor.getDefaultConfig();
                objForEachKey(defaultConfig, function (field, value) {
                    _config[field] = ctx.getConfig(AjaxMonitor.identifier, field, value);
                });
                var distributedTracingMode = _config.distributedTracingMode;
                _enableRequestHeaderTracking = _config.enableRequestHeaderTracking;
                _enableAjaxErrorStatusText = _config.enableAjaxErrorStatusText;
                _enableAjaxPerfTracking = _config.enableAjaxPerfTracking;
                _maxAjaxCallsPerView = _config.maxAjaxCallsPerView;
                _enableResponseHeaderTracking = _config.enableResponseHeaderTracking;
                _excludeRequestFromAutoTrackingPatterns = _config.excludeRequestFromAutoTrackingPatterns;
                _addRequestContext = _config.addRequestContext;
                _isUsingAIHeaders = distributedTracingMode === 0 /* eDistributedTracingModes.AI */ || distributedTracingMode === 1 /* eDistributedTracingModes.AI_AND_W3C */;
                _isUsingW3CHeaders = distributedTracingMode === 1 /* eDistributedTracingModes.AI_AND_W3C */ || distributedTracingMode === 2 /* eDistributedTracingModes.W3C */;
                if (_enableAjaxPerfTracking) {
                    var iKey = config.instrumentationKey || "unkwn";
                    if (iKey.length > 5) {
                        _markPrefix = AJAX_MONITOR_PREFIX + iKey.substring(iKey.length - 5) + ".";
                    }
                    else {
                        _markPrefix = AJAX_MONITOR_PREFIX + iKey + ".";
                    }
                }
                _disableAjaxTracking = !!_config.disableAjaxTracking;
                _disableFetchTracking = !!_config.disableFetchTracking;
            }
            function _populateContext() {
                var propExt = _self.core.getPlugin(PropertiesPluginIdentifier);
                if (propExt) {
                    _context = propExt.plugin.context; // we could move IPropertiesPlugin to common as well
                }
            }
            // discard the header if it's defined as ignoreHeaders in ICorrelationConfig
            function _canIncludeHeaders(header) {
                var rlt = true;
                if (header || _config.ignoreHeaders) {
                    arrForEach(_config.ignoreHeaders, (function (key) {
                        if (key.toLowerCase() === header.toLowerCase()) {
                            rlt = false;
                            return -1;
                        }
                    }));
                }
                return rlt;
            }
            // Fetch Stuff
            function _instrumentFetch() {
                var fetch = _supportsFetch();
                if (!fetch) {
                    return;
                }
                var global = getGlobal();
                var isPolyfill = fetch.polyfill;
                if (!_disableFetchTracking && !_fetchInitialized) {
                    _addHook(InstrumentFunc(global, strFetch, {
                        ns: _evtNamespace,
                        // Add request hook
                        req: function (callDetails, input, init) {
                            var fetchData;
                            if (!_disableFetchTracking && _fetchInitialized &&
                                !_isDisabledRequest(null, input, init) &&
                                // If we have a polyfil and XHR instrumented then let XHR report otherwise we get duplicates
                                !(isPolyfill && _xhrInitialized)) {
                                var ctx = callDetails.ctx();
                                fetchData = _createFetchRecord(input, init);
                                var newInit = _self.includeCorrelationHeaders(fetchData, input, init);
                                if (newInit !== init) {
                                    callDetails.set(1, newInit);
                                }
                                ctx.data = fetchData;
                            }
                        },
                        rsp: function (callDetails, input) {
                            if (!_disableFetchTracking) {
                                var fetchData_1 = callDetails.ctx().data;
                                if (fetchData_1) {
                                    // Replace the result with the new promise from this code
                                    callDetails.rslt = callDetails.rslt.then(function (response) {
                                        _reportFetchMetrics(callDetails, (response || {}).status, input, response, fetchData_1, function () {
                                            var ajaxResponse = {
                                                statusText: response.statusText,
                                                headerMap: null,
                                                correlationContext: _getFetchCorrelationContext(response)
                                            };
                                            if (_enableResponseHeaderTracking) {
                                                var responseHeaderMap_1 = {};
                                                response.headers.forEach(function (value, name) {
                                                    if (_canIncludeHeaders(name)) {
                                                        responseHeaderMap_1[name] = value;
                                                    }
                                                });
                                                ajaxResponse.headerMap = responseHeaderMap_1;
                                            }
                                            return ajaxResponse;
                                        });
                                        return response;
                                    })["catch"](function (reason) {
                                        _reportFetchMetrics(callDetails, 0, input, null, fetchData_1, null, { error: reason.message });
                                        throw reason;
                                    });
                                }
                            }
                        },
                        // Create an error callback to report any hook errors
                        hkErr: _createErrorCallbackFunc(_self, 15 /* _eInternalMessageId.FailedMonitorAjaxOpen */, "Failed to monitor Window.fetch, monitoring data for this fetch call may be incorrect.")
                    }));
                    _fetchInitialized = true;
                }
                else if (isPolyfill) {
                    // If fetch is a polyfill we need to capture the request to ensure that we correctly track
                    // disabled request URLS (i.e. internal urls) to ensure we don't end up in a constant loop
                    // of reporting ourselves, for example React Native uses a polyfill for fetch
                    // Note: Polyfill implementations that don't support the "poyyfill" tag are not supported
                    // the workaround is to add a polyfill property to your fetch implementation before initializing
                    // App Insights
                    _addHook(InstrumentFunc(global, strFetch, {
                        ns: _evtNamespace,
                        req: function (callDetails, input, init) {
                            // Just call so that we record any disabled URL
                            _isDisabledRequest(null, input, init);
                        }
                    }));
                }
                if (isPolyfill) {
                    // retag the instrumented fetch with the same polyfill settings this is mostly for testing
                    // But also supports multiple App Insights usages
                    global[strFetch].polyfill = isPolyfill;
                }
            }
            function _hookProto(target, funcName, callbacks) {
                _addHook(InstrumentProto(target, funcName, callbacks));
            }
            function _instrumentXhr() {
                if (_supportsAjaxMonitoring(_self) && !_disableAjaxTracking && !_xhrInitialized) {
                    // Instrument open
                    _hookProto(XMLHttpRequest, "open", {
                        ns: _evtNamespace,
                        req: function (args, method, url, async) {
                            if (!_disableAjaxTracking) {
                                var xhr = args.inst;
                                var ajaxData = xhr[strAjaxData];
                                if (!_isDisabledRequest(xhr, url) && _isMonitoredXhrInstance(xhr, true)) {
                                    if (!ajaxData || !ajaxData.xhrMonitoringState.openDone) {
                                        // Only create a single ajaxData (even when multiple AI instances are running)
                                        _openHandler(xhr, method, url, async);
                                    }
                                    // always attach to the on ready state change (required for handling multiple instances)
                                    _attachToOnReadyStateChange(xhr);
                                }
                            }
                        },
                        hkErr: _createErrorCallbackFunc(_self, 15 /* _eInternalMessageId.FailedMonitorAjaxOpen */, "Failed to monitor XMLHttpRequest.open, monitoring data for this ajax call may be incorrect.")
                    });
                    // Instrument send
                    _hookProto(XMLHttpRequest, "send", {
                        ns: _evtNamespace,
                        req: function (args, context) {
                            if (!_disableAjaxTracking) {
                                var xhr = args.inst;
                                var ajaxData = xhr[strAjaxData];
                                if (_isMonitoredXhrInstance(xhr) && !ajaxData.xhrMonitoringState.sendDone) {
                                    _createMarkId("xhr", ajaxData);
                                    ajaxData.requestSentTime = dateTimeUtilsNow();
                                    _self.includeCorrelationHeaders(ajaxData, undefined, undefined, xhr);
                                    ajaxData.xhrMonitoringState.sendDone = true;
                                }
                            }
                        },
                        hkErr: _createErrorCallbackFunc(_self, 17 /* _eInternalMessageId.FailedMonitorAjaxSend */, "Failed to monitor XMLHttpRequest, monitoring data for this ajax call may be incorrect.")
                    });
                    // Instrument abort
                    _hookProto(XMLHttpRequest, "abort", {
                        ns: _evtNamespace,
                        req: function (args) {
                            if (!_disableAjaxTracking) {
                                var xhr = args.inst;
                                var ajaxData = xhr[strAjaxData];
                                if (_isMonitoredXhrInstance(xhr) && !ajaxData.xhrMonitoringState.abortDone) {
                                    ajaxData.aborted = 1;
                                    ajaxData.xhrMonitoringState.abortDone = true;
                                }
                            }
                        },
                        hkErr: _createErrorCallbackFunc(_self, 13 /* _eInternalMessageId.FailedMonitorAjaxAbort */, "Failed to monitor XMLHttpRequest.abort, monitoring data for this ajax call may be incorrect.")
                    });
                    // Instrument setRequestHeader
                    _hookProto(XMLHttpRequest, "setRequestHeader", {
                        ns: _evtNamespace,
                        req: function (args, header, value) {
                            if (!_disableAjaxTracking && _enableRequestHeaderTracking) {
                                var xhr = args.inst;
                                if (_isMonitoredXhrInstance(xhr) && _canIncludeHeaders(header)) {
                                    xhr[strAjaxData].requestHeaders[header] = value;
                                }
                            }
                        },
                        hkErr: _createErrorCallbackFunc(_self, 71 /* _eInternalMessageId.FailedMonitorAjaxSetRequestHeader */, "Failed to monitor XMLHttpRequest.setRequestHeader, monitoring data for this ajax call may be incorrect.")
                    });
                    _xhrInitialized = true;
                }
            }
            function _isDisabledRequest(xhr, request, init) {
                var isDisabled = false;
                var theUrl = ((!isString(request) ? (request || {}).url || "" : request) || "").toLowerCase();
                // check excludeRequestFromAutoTrackingPatterns before stripping off any query string
                arrForEach(_excludeRequestFromAutoTrackingPatterns, function (regex) {
                    var theRegex = regex;
                    if (isString(regex)) {
                        theRegex = new RegExp(regex);
                    }
                    if (!isDisabled) {
                        isDisabled = theRegex.test(theUrl);
                    }
                });
                // if request url matches with exclude regex pattern, return true and no need to check for headers
                if (isDisabled) {
                    return isDisabled;
                }
                var idx = _indexOf(theUrl, "?");
                var idx2 = _indexOf(theUrl, "#");
                if (idx === -1 || (idx2 !== -1 && idx2 < idx)) {
                    idx = idx2;
                }
                if (idx !== -1) {
                    // Strip off any Query string
                    theUrl = theUrl.substring(0, idx);
                }
                // check that this instance is not not used by ajax call performed inside client side monitoring to send data to collector
                if (!isNullOrUndefined(xhr)) {
                    // Look on the XMLHttpRequest of the URL string value
                    isDisabled = xhr[DisabledPropertyName] === true || theUrl[DisabledPropertyName] === true;
                }
                else if (!isNullOrUndefined(request)) { // fetch
                    // Look for DisabledPropertyName in either Request or RequestInit
                    isDisabled = (typeof request === "object" ? request[DisabledPropertyName] === true : false) ||
                        (init ? init[DisabledPropertyName] === true : false);
                }
                // Also add extra check just in case the XHR or fetch objects where not decorated with the DisableProperty due to sealing or freezing
                if (!isDisabled && theUrl && isInternalApplicationInsightsEndpoint(theUrl)) {
                    isDisabled = true;
                }
                if (isDisabled) {
                    // Add the disabled url if not present
                    if (!_disabledUrls[theUrl]) {
                        _disabledUrls[theUrl] = 1;
                    }
                }
                else {
                    // Check to see if the url is listed as disabled
                    if (_disabledUrls[theUrl]) {
                        isDisabled = true;
                    }
                }
                return isDisabled;
            }
            /// <summary>Verifies that particalar instance of XMLHttpRequest needs to be monitored</summary>
            /// <param name="excludeAjaxDataValidation">Optional parameter. True if ajaxData must be excluded from verification</param>
            /// <returns type="bool">True if instance needs to be monitored, otherwise false</returns>
            function _isMonitoredXhrInstance(xhr, excludeAjaxDataValidation) {
                var ajaxValidation = true;
                var initialized = _xhrInitialized;
                if (!isNullOrUndefined(xhr)) {
                    ajaxValidation = excludeAjaxDataValidation === true || !isNullOrUndefined(xhr[strAjaxData]);
                }
                // checking to see that all interested functions on xhr were instrumented
                return initialized
                    // checking on ajaxData to see that it was not removed in user code
                    && ajaxValidation;
            }
            function _getDistributedTraceCtx() {
                var distributedTraceCtx = null;
                if (_self.core && _self.core.getTraceCtx) {
                    distributedTraceCtx = _self.core.getTraceCtx(false);
                }
                // Fall back
                if (!distributedTraceCtx && _context && _context.telemetryTrace) {
                    distributedTraceCtx = createDistributedTraceContextFromTrace(_context.telemetryTrace);
                }
                return distributedTraceCtx;
            }
            function _openHandler(xhr, method, url, async) {
                var distributedTraceCtx = _getDistributedTraceCtx();
                var traceID = (distributedTraceCtx && distributedTraceCtx.getTraceId()) || generateW3CId();
                var spanID = generateW3CId().substr(0, 16);
                var ajaxData = new ajaxRecord(traceID, spanID, _self[strDiagLog]());
                ajaxData.traceFlags = distributedTraceCtx && distributedTraceCtx.getTraceFlags();
                ajaxData.method = method;
                ajaxData.requestUrl = url;
                ajaxData.xhrMonitoringState.openDone = true;
                ajaxData.requestHeaders = {};
                ajaxData.async = async;
                ajaxData.errorStatusText = _enableAjaxErrorStatusText;
                xhr[strAjaxData] = ajaxData;
            }
            function _attachToOnReadyStateChange(xhr) {
                xhr[strAjaxData].xhrMonitoringState.stateChangeAttached = eventOn(xhr, "readystatechange", function () {
                    try {
                        if (xhr && xhr.readyState === 4 && _isMonitoredXhrInstance(xhr)) {
                            _onAjaxComplete(xhr);
                        }
                    }
                    catch (e) {
                        var exceptionText = dumpObj(e);
                        // ignore messages with c00c023f, as this a known IE9 XHR abort issue
                        if (!exceptionText || _indexOf(exceptionText.toLowerCase(), "c00c023f") === -1) {
                            _throwInternalCritical(_self, 16 /* _eInternalMessageId.FailedMonitorAjaxRSC */, "Failed to monitor XMLHttpRequest 'readystatechange' event handler, monitoring data for this ajax call may be incorrect.", {
                                ajaxDiagnosticsMessage: _getFailedAjaxDiagnosticsMessage(xhr),
                                exception: exceptionText
                            });
                        }
                    }
                }, _evtNamespace);
            }
            function _getResponseText(xhr) {
                try {
                    var responseType = xhr.responseType;
                    if (responseType === "" || responseType === "text") {
                        // As per the specification responseText is only valid if the type is an empty string or "text"
                        return xhr.responseText;
                    }
                }
                catch (e) {
                    // This shouldn't happen because of the above check -- but just in case, so just ignore
                }
                return null;
            }
            function _onAjaxComplete(xhr) {
                var ajaxData = xhr[strAjaxData];
                ajaxData.responseFinishedTime = dateTimeUtilsNow();
                ajaxData.status = xhr.status;
                function _reportXhrError(e, failedProps) {
                    var errorProps = failedProps || {};
                    errorProps["ajaxDiagnosticsMessage"] = _getFailedAjaxDiagnosticsMessage(xhr);
                    if (e) {
                        errorProps["exception"] = dumpObj(e);
                    }
                    _throwInternalWarning(_self, 14 /* _eInternalMessageId.FailedMonitorAjaxDur */, "Failed to calculate the duration of the ajax call, monitoring data for this ajax call won't be sent.", errorProps);
                }
                _findPerfResourceEntry("xmlhttprequest", ajaxData, function () {
                    try {
                        var dependency = ajaxData.CreateTrackItem("Ajax", _enableRequestHeaderTracking, function () {
                            var ajaxResponse = {
                                statusText: xhr.statusText,
                                headerMap: null,
                                correlationContext: _getAjaxCorrelationContext(xhr),
                                type: xhr.responseType,
                                responseText: _getResponseText(xhr),
                                response: xhr.response
                            };
                            if (_enableResponseHeaderTracking) {
                                var headers = xhr.getAllResponseHeaders();
                                if (headers) {
                                    // xhr.getAllResponseHeaders() method returns all the response headers, separated by CRLF, as a string or null
                                    // the regex converts the header string into an array of individual headers
                                    var arr = strTrim(headers).split(/[\r\n]+/);
                                    var responseHeaderMap_2 = {};
                                    arrForEach(arr, function (line) {
                                        var parts = line.split(": ");
                                        var header = parts.shift();
                                        var value = parts.join(": ");
                                        if (_canIncludeHeaders(header)) {
                                            responseHeaderMap_2[header] = value;
                                        }
                                    });
                                    ajaxResponse.headerMap = responseHeaderMap_2;
                                }
                            }
                            return ajaxResponse;
                        });
                        var properties = void 0;
                        try {
                            if (!!_addRequestContext) {
                                properties = _addRequestContext({ status: xhr.status, xhr: xhr });
                            }
                        }
                        catch (e) {
                            _throwInternalWarning(_self, 104 /* _eInternalMessageId.FailedAddingCustomDefinedRequestContext */, "Failed to add custom defined request context as configured call back may missing a null check.");
                        }
                        if (dependency) {
                            if (properties !== undefined) {
                                dependency.properties = __assign(__assign({}, dependency.properties), properties);
                            }
                            _self[strTrackDependencyDataInternal](dependency);
                        }
                        else {
                            _reportXhrError(null, {
                                requestSentTime: ajaxData.requestSentTime,
                                responseFinishedTime: ajaxData.responseFinishedTime
                            });
                        }
                    }
                    finally {
                        // cleanup telemetry data
                        try {
                            xhr[strAjaxData] = null;
                        }
                        catch (e) {
                            // May throw in environments that prevent extension or freeze xhr
                        }
                    }
                }, function (e) {
                    _reportXhrError(e, null);
                });
            }
            function _getAjaxCorrelationContext(xhr) {
                try {
                    var responseHeadersString = xhr.getAllResponseHeaders();
                    if (responseHeadersString !== null) {
                        var index = _indexOf(responseHeadersString.toLowerCase(), RequestHeaders[8 /* eRequestHeaders.requestContextHeaderLowerCase */]);
                        if (index !== -1) {
                            var responseHeader = xhr.getResponseHeader(RequestHeaders[0 /* eRequestHeaders.requestContextHeader */]);
                            return CorrelationIdHelper.getCorrelationContext(responseHeader);
                        }
                    }
                }
                catch (e) {
                    _throwInternalWarning(_self, 18 /* _eInternalMessageId.FailedMonitorAjaxGetCorrelationHeader */, "Failed to get Request-Context correlation header as it may be not included in the response or not accessible.", {
                        ajaxDiagnosticsMessage: _getFailedAjaxDiagnosticsMessage(xhr),
                        exception: dumpObj(e)
                    });
                }
            }
            function _createMarkId(type, ajaxData) {
                if (ajaxData.requestUrl && _markPrefix && _enableAjaxPerfTracking) {
                    var performance_1 = getPerformance();
                    if (performance_1 && isFunction(performance_1.mark)) {
                        _markCount++;
                        var markId = _markPrefix + type + "#" + _markCount;
                        performance_1.mark(markId);
                        var entries = performance_1.getEntriesByName(markId);
                        if (entries && entries.length === 1) {
                            ajaxData.perfMark = entries[0];
                        }
                    }
                }
            }
            function _findPerfResourceEntry(initiatorType, ajaxData, trackCallback, reportError) {
                var perfMark = ajaxData.perfMark;
                var performance = getPerformance();
                var maxAttempts = _config.maxAjaxPerfLookupAttempts;
                var retryDelay = _config.ajaxPerfLookupDelay;
                var requestUrl = ajaxData.requestUrl;
                var attempt = 0;
                (function locateResourceTiming() {
                    try {
                        if (performance && perfMark) {
                            attempt++;
                            var perfTiming = null;
                            var entries = performance.getEntries();
                            for (var lp = entries.length - 1; lp >= 0; lp--) {
                                var entry = entries[lp];
                                if (entry) {
                                    if (entry.entryType === "resource") {
                                        if (entry.initiatorType === initiatorType &&
                                            (_indexOf(entry.name, requestUrl) !== -1 || _indexOf(requestUrl, entry.name) !== -1)) {
                                            perfTiming = entry;
                                        }
                                    }
                                    else if (entry.entryType === "mark" && entry.name === perfMark.name) {
                                        // We hit the start event
                                        ajaxData.perfTiming = perfTiming;
                                        break;
                                    }
                                    if (entry.startTime < perfMark.startTime - 1000) {
                                        // Fallback to try and reduce the time spent looking for the perf entry
                                        break;
                                    }
                                }
                            }
                        }
                        if (!perfMark || // - we don't have a perfMark or
                            ajaxData.perfTiming || // - we have not found the perf entry or
                            attempt >= maxAttempts || // - we have tried too many attempts or
                            ajaxData.async === false) { // - this is a sync request
                            if (perfMark && isFunction(performance.clearMarks)) {
                                // Remove the mark so we don't fill up the performance resources too much
                                performance.clearMarks(perfMark.name);
                            }
                            ajaxData.perfAttempts = attempt;
                            // just continue and report the track event
                            trackCallback();
                        }
                        else {
                            // We need to wait for the browser to populate the window.performance entry
                            // This needs to be at least 1ms as waiting <= 1 (on firefox) is not enough time for fetch or xhr,
                            // this is a scheduling issue for the browser implementation
                            setTimeout(locateResourceTiming, retryDelay);
                        }
                    }
                    catch (e) {
                        reportError(e);
                    }
                })();
            }
            function _createFetchRecord(input, init) {
                var distributedTraceCtx = _getDistributedTraceCtx();
                var traceID = (distributedTraceCtx && distributedTraceCtx.getTraceId()) || generateW3CId();
                var spanID = generateW3CId().substr(0, 16);
                var ajaxData = new ajaxRecord(traceID, spanID, _self[strDiagLog]());
                ajaxData.traceFlags = distributedTraceCtx && distributedTraceCtx.getTraceFlags();
                ajaxData.requestSentTime = dateTimeUtilsNow();
                ajaxData.errorStatusText = _enableAjaxErrorStatusText;
                if (input instanceof Request) {
                    ajaxData.requestUrl = input ? input.url : "";
                }
                else {
                    ajaxData.requestUrl = input;
                }
                var method = "GET";
                if (init && init.method) {
                    method = init.method;
                }
                else if (input && input instanceof Request) {
                    method = input.method;
                }
                ajaxData.method = method;
                var requestHeaders = {};
                if (_enableRequestHeaderTracking) {
                    var headers = new Headers((init ? init.headers : 0) || (input instanceof Request ? (input.headers || {}) : {}));
                    headers.forEach(function (value, key) {
                        if (_canIncludeHeaders(key)) {
                            requestHeaders[key] = value;
                        }
                    });
                }
                ajaxData.requestHeaders = requestHeaders;
                _createMarkId("fetch", ajaxData);
                return ajaxData;
            }
            function _getFailedFetchDiagnosticsMessage(input) {
                var result = "";
                try {
                    if (!isNullOrUndefined(input)) {
                        if (typeof (input) === "string") {
                            result += "(url: '".concat(input, "')");
                        }
                        else {
                            result += "(url: '".concat(input.url, "')");
                        }
                    }
                }
                catch (e) {
                    _throwInternalCritical(_self, 15 /* _eInternalMessageId.FailedMonitorAjaxOpen */, "Failed to grab failed fetch diagnostics message", { exception: dumpObj(e) });
                }
                return result;
            }
            function _reportFetchMetrics(callDetails, status, input, response, ajaxData, getResponse, properties) {
                if (!ajaxData) {
                    return;
                }
                function _reportFetchError(msgId, e, failedProps) {
                    var errorProps = failedProps || {};
                    errorProps["fetchDiagnosticsMessage"] = _getFailedFetchDiagnosticsMessage(input);
                    if (e) {
                        errorProps["exception"] = dumpObj(e);
                    }
                    _throwInternalWarning(_self, msgId, "Failed to calculate the duration of the fetch call, monitoring data for this fetch call won't be sent.", errorProps);
                }
                ajaxData.responseFinishedTime = dateTimeUtilsNow();
                ajaxData.status = status;
                _findPerfResourceEntry("fetch", ajaxData, function () {
                    var dependency = ajaxData.CreateTrackItem("Fetch", _enableRequestHeaderTracking, getResponse);
                    var properties;
                    try {
                        if (!!_addRequestContext) {
                            properties = _addRequestContext({ status: status, request: input, response: response });
                        }
                    }
                    catch (e) {
                        _throwInternalWarning(_self, 104 /* _eInternalMessageId.FailedAddingCustomDefinedRequestContext */, "Failed to add custom defined request context as configured call back may missing a null check.");
                    }
                    if (dependency) {
                        if (properties !== undefined) {
                            dependency.properties = __assign(__assign({}, dependency.properties), properties);
                        }
                        _self[strTrackDependencyDataInternal](dependency);
                    }
                    else {
                        _reportFetchError(14 /* _eInternalMessageId.FailedMonitorAjaxDur */, null, {
                            requestSentTime: ajaxData.requestSentTime,
                            responseFinishedTime: ajaxData.responseFinishedTime
                        });
                    }
                }, function (e) {
                    _reportFetchError(18 /* _eInternalMessageId.FailedMonitorAjaxGetCorrelationHeader */, e, null);
                });
            }
            function _getFetchCorrelationContext(response) {
                if (response && response.headers) {
                    try {
                        var responseHeader = response.headers.get(RequestHeaders[0 /* eRequestHeaders.requestContextHeader */]);
                        return CorrelationIdHelper.getCorrelationContext(responseHeader);
                    }
                    catch (e) {
                        _throwInternalWarning(_self, 18 /* _eInternalMessageId.FailedMonitorAjaxGetCorrelationHeader */, "Failed to get Request-Context correlation header as it may be not included in the response or not accessible.", {
                            fetchDiagnosticsMessage: _getFailedFetchDiagnosticsMessage(response),
                            exception: dumpObj(e)
                        });
                    }
                }
            }
        });
        return _this;
    }
    AjaxMonitor.getDefaultConfig = function () {
        var config = {
            maxAjaxCallsPerView: 500,
            disableAjaxTracking: false,
            disableFetchTracking: false,
            excludeRequestFromAutoTrackingPatterns: undefined,
            disableCorrelationHeaders: false,
            distributedTracingMode: 1 /* eDistributedTracingModes.AI_AND_W3C */,
            correlationHeaderExcludedDomains: [
                "*.blob.core.windows.net",
                "*.blob.core.chinacloudapi.cn",
                "*.blob.core.cloudapi.de",
                "*.blob.core.usgovcloudapi.net"
            ],
            correlationHeaderDomains: undefined,
            correlationHeaderExcludePatterns: undefined,
            appId: undefined,
            enableCorsCorrelation: false,
            enableRequestHeaderTracking: false,
            enableResponseHeaderTracking: false,
            enableAjaxErrorStatusText: false,
            enableAjaxPerfTracking: false,
            maxAjaxPerfLookupAttempts: 3,
            ajaxPerfLookupDelay: 25,
            ignoreHeaders: [
                "Authorization",
                "X-API-Key",
                "WWW-Authenticate"
            ],
            addRequestContext: undefined
        };
        return config;
    };
    AjaxMonitor.getEmptyConfig = function () {
        var emptyConfig = this.getDefaultConfig();
        objForEachKey(emptyConfig, function (value) {
            emptyConfig[value] = undefined;
        });
        return emptyConfig;
    };
// Removed Stub for AjaxMonitor.prototype.initialize.
    AjaxMonitor.prototype.processTelemetry = function (item, itemCtx) {
        this.processNext(item, itemCtx);
    };
// Removed Stub for AjaxMonitor.prototype.trackDependencyData.
// Removed Stub for AjaxMonitor.prototype.includeCorrelationHeaders.
// Removed Stub for AjaxMonitor.prototype.addDependencyListener.
// Removed Stub for AjaxMonitor.prototype.trackDependencyDataInternal.
    AjaxMonitor.identifier = "AjaxDependencyPlugin";
    return AjaxMonitor;
}(BaseTelemetryPlugin));
export { AjaxMonitor };
