/*
 * Copyright (c) 2018-2024 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */

import common from '@eclipse-che/common';
import { dump } from 'js-yaml';

import devfileApi from '@/services/devfileApi';
import { FakeStoreBuilder } from '@/store/__mocks__/storeBuilder';
import { getEditor } from '@/store/DevfileRegistries/getEditor';

const mockFetchData = jest.fn();
jest.mock('@/services/registry/fetchData', () => ({
  fetchData: (...args: unknown[]) => mockFetchData(...args),
}));

describe('Get Devfile by URL', () => {
  let editor: devfileApi.Devfile;

  beforeEach(() => {
    editor = buildEditor();
    mockFetchData.mockResolvedValueOnce(dump(editor));
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  it('Should throw the "failed to fetch editor yaml" error message', async () => {
    const store = new FakeStoreBuilder().build();

    let errorText: string | undefined;
    try {
      await getEditor('che-incubator/che-idea/next', store.dispatch, store.getState);
    } catch (e) {
      errorText = common.helpers.errors.getMessage(e);
    }

    expect(errorText).toEqual('Failed to fetch editor yaml by id: che-incubator/che-idea/next.');
  });

  it('Should request a devfile content by editor id', async () => {
    const editors = [
      {
        schemaVersion: '2.2.2',
        metadata: {
          name: 'che-idea',
          attributes: {
            publisher: 'che-incubator',
            version: 'next',
          },
        },
      } as devfileApi.Devfile,
    ];
    const store = new FakeStoreBuilder()
      .withDwPlugins({}, {}, false, editors, 'che-incubator/che-idea/next')
      .build();

    try {
      const result = await getEditor('che-incubator/che-idea/next', store.dispatch, store.getState);
      expect(result).toEqual(dump(editors[0]));
    } catch (e) {
      // no-op
    }
  });

  it('Should return an existing devfile content by editor id', async () => {
    const store = new FakeStoreBuilder().build();

    try {
      await getEditor(
        'https://dummy/che-plugin-registry/main/v3/plugins/che-incubator/che-idea/next/devfile.yaml',
        store.dispatch,
        store.getState,
      );
    } catch (e) {
      // no-op
    }

    expect(mockFetchData).toHaveBeenCalledTimes(1);
    expect(mockFetchData).toHaveBeenCalledWith(
      'https://dummy/che-plugin-registry/main/v3/plugins/che-incubator/che-idea/next/devfile.yaml',
    );
  });

  it('Should return an existing devfile content by editor path', async () => {
    const store = new FakeStoreBuilder()
      .withDevfileRegistries({
        devfiles: {
          ['https://dummy/che-plugin-registry/main/v3/plugins/che-incubator/che-idea/next/devfile.yaml']:
            {
              content: dump(editor),
            },
        },
      })
      .build();

    const customEditor = await getEditor(
      'https://dummy/che-plugin-registry/main/v3/plugins/che-incubator/che-idea/next/devfile.yaml',
      store.dispatch,
      store.getState,
    );

    expect(mockFetchData).toHaveBeenCalledTimes(0);
    expect(customEditor.content).toEqual(dump(editor));
    expect(customEditor.editorYamlUrl).toEqual(
      'https://dummy/che-plugin-registry/main/v3/plugins/che-incubator/che-idea/next/devfile.yaml',
    );
  });
});

function buildEditor(): devfileApi.Devfile {
  return {
    schemaVersion: '2.1.0',
    metadata: {
      name: 'ws-skeleton/eclipseide/4.9.0',
      namespace: 'che',
    },
    components: [
      {
        name: 'eclipse-ide',
        container: {
          image: 'docker.io/wsskeleton/eclipse-broadway',
          mountSources: true,
          memoryLimit: '2048M',
        },
      },
    ],
  };
}
