#!/usr/bin/env bash

set -e -u -o pipefail

MANIFEST_FILE_NAME=".cwm.manifest.txt"

function generate_manifest() {
  (cd "$1" && find . -type f -exec ls -la {} \;) | grep -v "$MANIFEST_FILE_NAME" >"$2"
}

function write_manifest() {
  generate_manifest "$1" "$1/$MANIFEST_FILE_NAME"
}

function is_up_to_date() {
  test -f "$1/$MANIFEST_FILE_NAME" || return 1

  # Writing to $1 also have a nice side-effect of updating containing directory mtime
  # mtime of $1 is used by IDE to clean-up old JBR and client distributions

  local temp_manifest
  temp_manifest=$(mktemp "$1/$MANIFEST_FILE_NAME.XXXXXXXX")
  trap "rm -f '$temp_manifest'" EXIT RETURN

  generate_manifest "$1" "$temp_manifest"
  cmp -s "$temp_manifest" "$1/$MANIFEST_FILE_NAME" || {
    echo " ! Content of the directory '$1' is different from recorded manifest '$1/$MANIFEST_FILE_NAME', re-downloading and re-extracting content there"
    if command -v diff >/dev/null 2>&1; then
      diff -u "$1/$MANIFEST_FILE_NAME" "$temp_manifest"
    else
      echo " ('diff' is missing)"
    fi
    return 1
  }
}

function check_sha256() {
  local file
  local expected_checksum
  local file_source_moniker

  file_source_moniker="$1"
  file="$2"
  expected_checksum="$3"

  if command -v shasum >/dev/null 2>&1; then
    echo "$expected_checksum  $file" | shasum -a 256 --status -c || {
      echo " ! Checksum mismatch for $file, file $file was downloaded from $file_source_moniker"
      return 1
    }
    return 0
  fi

  if command -v sha256sum >/dev/null 2>&1; then
    echo "$expected_checksum  $file" | sha256sum --status -c || {
      echo " ! Checksum mismatch for $file, file $file was downloaded from $file_source_moniker"
      return 1
    }
    return 0
  fi

  echo "Both 'shasum' and 'sha256sum' utilities are missing. Please install one of them and run this script again."
  return 1
}

check_command() {
  local LOCAL_COMMAND_OUTPUT
  LOCAL_COMMAND_OUTPUT=$(command -v "$1" ||:)
  if [[ -z "$LOCAL_COMMAND_OUTPUT" ]]; then
    echo "A required tool is missing: $1" >&2
    exit 1
  fi
}

check_command uname
check_command tar
check_command unzip

function download() {
  if command -v curl >/dev/null 2>&1; then
    curl --progress-bar --fail --show-error --location --output "$1" "$2"
  elif command -v wget >/dev/null 2>&1; then
    wget --progress=bar -O "$1" "$2"
  else
    echo "ERROR: Please install wget or curl" >&2
    exit 1
  fi
}

case "$(uname -s)" in
Linux)
  LOCALAPPDATA_FOLDER="${XDG_CACHE_HOME-${HOME}/.cache}/JetBrains/JetBrainsClientDist"
  JRE_HOME_SUFFIX=jbr
  ;;
Darwin)
  # shellcheck disable=SC2034
  LOCALAPPDATA_FOLDER="${HOME}/Library/Caches/JetBrains/JetBrainsClientDist"
  # shellcheck disable=SC2034
  JRE_HOME_SUFFIX=jbr/Contents/Home
  ;;
*)
  echo "Unsupported system $(uname -s)" >&2
  exit 1
esac


CWM_CLIENT_VERSION="%%CWM_CLIENT_VERSION%%"
# todo
CWM_RUNTIME_VERSION1="%%CWM_RUNTIME_VERSION1%%"
CWM_RUNTIME_VERSION2="%%CWM_RUNTIME_VERSION2%%"
JOIN_LINK="%%JOIN_LINK%%"

CWM_CLIENT_URL="https://cache-redirector.jetbrains.com/download.jetbrains.com/idea/code-with-me/JetBrainsClient-$CWM_CLIENT_VERSION-no-jdk.sit"
CWM_CLIENT_DIR_NAME="JetBrainsClient-$CWM_CLIENT_VERSION-osx-x64"
CWM_CLIENT_SHA256="b8d33783ef015355c94a2b58ae697020c7c965f69c1f1cb8c42dc36200014056"
JRE_URL="https://cache-redirector.jetbrains.com/download.jetbrains.com/idea/jbr/jbr_jcef-${CWM_RUNTIME_VERSION1}-osx-x64-${CWM_RUNTIME_VERSION2}.tar.gz"
JRE_DIR_NAME="jbr_jcef-11_0_10-b1341.35"
JRE_SHA256="e98e515ce6b2c43a5f1c78e8b5945b9fe81f39421733c1d4aa42ca205d9a7d5c"


LINUX_LAUNCHER_FILE="bin/jetbrains_client.sh"

DIR_SUFFIX="-ulauncher"
CWM_CLIENT_DIR="${LOCALAPPDATA_FOLDER}/$CWM_CLIENT_DIR_NAME$DIR_SUFFIX"
JRE_DIR="$LOCALAPPDATA_FOLDER/$JRE_DIR_NAME$DIR_SUFFIX"

if ! is_up_to_date "$CWM_CLIENT_DIR"; then
  echo "*** Downloading $CWM_CLIENT_DIR_NAME from $CWM_CLIENT_URL"
  rm -rf "$CWM_CLIENT_DIR"
  mkdir -p "$CWM_CLIENT_DIR"
  TEMP_DIST="$CWM_CLIENT_DIR/.dist.data"

  download "$TEMP_DIST" "$CWM_CLIENT_URL"

  echo "*** Checking SHA256 checksum of $CWM_CLIENT_URL"
#  check_sha256 "$CWM_CLIENT_URL" "$TEMP_DIST" "$CWM_CLIENT_SHA256"

  echo "*** Extracting $CWM_CLIENT_DIR_NAME to $CWM_CLIENT_DIR"

  case "$(uname -s)" in
  Linux)
    tar -zxpC "$CWM_CLIENT_DIR" -f "$TEMP_DIST"
    ;;
  Darwin)
    unzip -q "$TEMP_DIST" -d "$CWM_CLIENT_DIR"
    ;;
  *)
    echo "Unsupported system $(uname -s)" >&2
    exit 1
  esac

  rm -f "$TEMP_DIST"

  write_manifest "$CWM_CLIENT_DIR"
fi

if ! is_up_to_date "$JRE_DIR"; then
  echo "*** Downloading JetBrains Runtime $JRE_DIR_NAME from $JRE_URL"
  rm -rf "$JRE_DIR"
  mkdir -p "$JRE_DIR"
  TEMP_TAR="$JRE_DIR/.dist.tar.gz"

  download "$TEMP_TAR" "$JRE_URL"

  echo "*** Checking SHA256 checksum of $JRE_URL"
#  check_sha256 "$JRE_URL" "$TEMP_TAR" "$JRE_SHA256"

  echo "*** Extracting JetBrains Runtime $JRE_DIR_NAME to $JRE_DIR"
  tar -zxpC "$JRE_DIR" -f "$TEMP_TAR"
  rm -f "$TEMP_TAR"

  write_manifest "$JRE_DIR"
fi

# Update mtime on JRE & CWM client dirs. It would be used by a cleanup process to remove older distributions
touch "$JRE_DIR"
touch "$CWM_CLIENT_DIR"

echo "Starting $CWM_CLIENT_DIR_NAME with join link: $JOIN_LINK"

case "$(uname -s)" in
Linux)
  APP_DIR=$(find "$CWM_CLIENT_DIR" -mindepth 1 -maxdepth 1 -type d -name '*-*' | head -n1)
  ln -sf "$JRE_DIR/$JRE_HOME_SUFFIX" "$APP_DIR/jbr"
  exec /bin/sh "$APP_DIR/$LINUX_LAUNCHER_FILE" thinClientSsh "$JOIN_LINK"
  ;;
Darwin)
  ln -sf "$JRE_DIR/jbr" "$CWM_CLIENT_DIR"/*.app/Contents/

  retry_count=5
  for r in $(seq 1 $retry_count); do
    # shellcheck disable=SC2015
    open "$CWM_CLIENT_DIR"/*.app --args thinClientSsh "$JOIN_LINK" && exit 0 ||:
    echo " *** 'open' command failed. Trying to retry $r/$retry_count..."
    sleep 1
  done

  echo "Failed to open $CWM_CLIENT_DIR_NAME. Please report this issue including script output to support@jetbrains.com"
  exit 1
  ;;
*)
  echo "Unsupported system $(uname -s)" >&2
  exit 1
esac