/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.session.web.http;

import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Base64;
import java.util.BitSet;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.session.web.http.CookieSerializer;

public class DefaultCookieSerializer
implements CookieSerializer {
    private static final Log logger;
    private static final BitSet domainValid;
    private Clock clock = Clock.systemUTC();
    private String cookieName = "SESSION";
    private Boolean useSecureCookie;
    private boolean useHttpOnlyCookie = true;
    private String cookiePath;
    private Integer cookieMaxAge;
    private String domainName;
    private Pattern domainNamePattern;
    private String jvmRoute;
    private boolean useBase64Encoding = true;
    private String rememberMeRequestAttribute;
    private String sameSite = "Lax";

    @Override
    public List<String> readCookieValues(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        ArrayList<String> matchingCookieValues = new ArrayList<String>();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                String sessionId;
                if (!this.cookieName.equals(cookie.getName())) continue;
                String string = sessionId = this.useBase64Encoding ? this.base64Decode(cookie.getValue()) : cookie.getValue();
                if (sessionId == null) continue;
                if (this.jvmRoute != null && sessionId.endsWith(this.jvmRoute)) {
                    sessionId = sessionId.substring(0, sessionId.length() - this.jvmRoute.length());
                }
                matchingCookieValues.add(sessionId);
            }
        }
        return matchingCookieValues;
    }

    @Override
    public void writeCookieValue(CookieSerializer.CookieValue cookieValue) {
        String path;
        String domain;
        int maxAge;
        HttpServletRequest request = cookieValue.getRequest();
        HttpServletResponse response = cookieValue.getResponse();
        StringBuilder sb = new StringBuilder();
        sb.append(this.cookieName).append('=');
        String value = this.getValue(cookieValue);
        if (value != null && value.length() > 0) {
            this.validateValue(value);
            sb.append(value);
        }
        if ((maxAge = this.getMaxAge(cookieValue)) > -1) {
            sb.append("; Max-Age=").append(cookieValue.getCookieMaxAge());
            ZonedDateTime expires = maxAge != 0 ? ZonedDateTime.now(this.clock).plusSeconds(maxAge) : Instant.EPOCH.atZone(ZoneOffset.UTC);
            sb.append("; Expires=").append(expires.format(DateTimeFormatter.RFC_1123_DATE_TIME));
        }
        if ((domain = this.getDomainName(request)) != null && domain.length() > 0) {
            this.validateDomain(domain);
            sb.append("; Domain=").append(domain);
        }
        if ((path = this.getCookiePath(request)) != null && path.length() > 0) {
            this.validatePath(path);
            sb.append("; Path=").append(path);
        }
        if (this.isSecureCookie(request)) {
            sb.append("; Secure");
        }
        if (this.useHttpOnlyCookie) {
            sb.append("; HttpOnly");
        }
        if (this.sameSite != null) {
            sb.append("; SameSite=").append(this.sameSite);
        }
        response.addHeader("Set-Cookie", sb.toString());
    }

    private String base64Decode(String base64Value) {
        try {
            byte[] decodedCookieBytes = Base64.getDecoder().decode(base64Value);
            return new String(decodedCookieBytes);
        }
        catch (Exception ex) {
            logger.debug((Object)("Unable to Base64 decode value: " + base64Value));
            return null;
        }
    }

    private String base64Encode(String value) {
        byte[] encodedCookieBytes = Base64.getEncoder().encode(value.getBytes());
        return new String(encodedCookieBytes);
    }

    private String getValue(CookieSerializer.CookieValue cookieValue) {
        String requestedCookieValue = cookieValue.getCookieValue();
        Object actualCookieValue = requestedCookieValue;
        if (this.jvmRoute != null) {
            actualCookieValue = requestedCookieValue + this.jvmRoute;
        }
        if (this.useBase64Encoding) {
            actualCookieValue = this.base64Encode((String)actualCookieValue);
        }
        return actualCookieValue;
    }

    private void validateValue(String value) {
        int start = 0;
        int end = value.length();
        if (end > 1 && value.charAt(0) == '\"' && value.charAt(end - 1) == '\"') {
            start = 1;
            --end;
        }
        char[] chars = value.toCharArray();
        for (int i = start; i < end; ++i) {
            char c = chars[i];
            if (c >= '!' && c != '\"' && c != ',' && c != ';' && c != '\\' && c != '\u007f') continue;
            throw new IllegalArgumentException("Invalid character in cookie value: " + c);
        }
    }

    private int getMaxAge(CookieSerializer.CookieValue cookieValue) {
        int maxAge = cookieValue.getCookieMaxAge();
        if (maxAge < 0) {
            if (this.rememberMeRequestAttribute != null && cookieValue.getRequest().getAttribute(this.rememberMeRequestAttribute) != null) {
                cookieValue.setCookieMaxAge(Integer.MAX_VALUE);
            } else if (this.cookieMaxAge != null) {
                cookieValue.setCookieMaxAge(this.cookieMaxAge);
            }
        }
        return cookieValue.getCookieMaxAge();
    }

    private void validateDomain(String domain) {
        int cur = -1;
        char[] chars = domain.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            int prev = cur;
            cur = chars[i];
            if (domainValid.get(cur) && (prev != 46 && prev != -1 || cur != 46 && cur != 45) && (prev != 45 || cur != 46)) continue;
            throw new IllegalArgumentException("Invalid cookie domain: " + domain);
        }
        if (cur == 46 || cur == 45) {
            throw new IllegalArgumentException("Invalid cookie domain: " + domain);
        }
    }

    private void validatePath(String path) {
        for (char ch : path.toCharArray()) {
            if (ch >= ' ' && ch <= '~' && ch != ';') continue;
            throw new IllegalArgumentException("Invalid cookie path: " + path);
        }
    }

    void setClock(Clock clock) {
        this.clock = clock.withZone(ZoneOffset.UTC);
    }

    public void setUseSecureCookie(boolean useSecureCookie) {
        this.useSecureCookie = useSecureCookie;
    }

    public void setUseHttpOnlyCookie(boolean useHttpOnlyCookie) {
        this.useHttpOnlyCookie = useHttpOnlyCookie;
    }

    private boolean isSecureCookie(HttpServletRequest request) {
        if (this.useSecureCookie == null) {
            return request.isSecure();
        }
        return this.useSecureCookie;
    }

    public void setCookiePath(String cookiePath) {
        this.cookiePath = cookiePath;
    }

    public void setCookieName(String cookieName) {
        if (cookieName == null) {
            throw new IllegalArgumentException("cookieName cannot be null");
        }
        this.cookieName = cookieName;
    }

    public void setCookieMaxAge(int cookieMaxAge) {
        this.cookieMaxAge = cookieMaxAge;
    }

    public void setDomainName(String domainName) {
        if (this.domainNamePattern != null) {
            throw new IllegalStateException("Cannot set both domainName and domainNamePattern");
        }
        this.domainName = domainName;
    }

    public void setDomainNamePattern(String domainNamePattern) {
        if (this.domainName != null) {
            throw new IllegalStateException("Cannot set both domainName and domainNamePattern");
        }
        this.domainNamePattern = Pattern.compile(domainNamePattern, 2);
    }

    public void setJvmRoute(String jvmRoute) {
        this.jvmRoute = "." + jvmRoute;
    }

    public void setUseBase64Encoding(boolean useBase64Encoding) {
        this.useBase64Encoding = useBase64Encoding;
    }

    public void setRememberMeRequestAttribute(String rememberMeRequestAttribute) {
        if (rememberMeRequestAttribute == null) {
            throw new IllegalArgumentException("rememberMeRequestAttribute cannot be null");
        }
        this.rememberMeRequestAttribute = rememberMeRequestAttribute;
    }

    public void setSameSite(String sameSite) {
        this.sameSite = sameSite;
    }

    private String getDomainName(HttpServletRequest request) {
        Matcher matcher;
        if (this.domainName != null) {
            return this.domainName;
        }
        if (this.domainNamePattern != null && (matcher = this.domainNamePattern.matcher(request.getServerName())).matches()) {
            return matcher.group(1);
        }
        return null;
    }

    private String getCookiePath(HttpServletRequest request) {
        if (this.cookiePath == null) {
            String contextPath = request.getContextPath();
            return contextPath != null && contextPath.length() > 0 ? contextPath : "/";
        }
        return this.cookiePath;
    }

    static {
        int c;
        logger = LogFactory.getLog(DefaultCookieSerializer.class);
        domainValid = new BitSet(128);
        for (c = 48; c <= 57; c = (int)((char)(c + 1))) {
            domainValid.set(c);
        }
        for (c = 97; c <= 122; c = (int)((char)(c + 1))) {
            domainValid.set(c);
        }
        for (c = 65; c <= 90; c = (int)((char)(c + 1))) {
            domainValid.set(c);
        }
        domainValid.set(46);
        domainValid.set(45);
    }
}

