/*
 * Decompiled with CFR 0.152.
 */
package org.jobrunr.server;

import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.ServiceLoader;
import java.util.Spliterators;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.StreamSupport;
import org.jobrunr.JobRunrException;
import org.jobrunr.jobs.Job;
import org.jobrunr.jobs.filters.JobDefaultFilters;
import org.jobrunr.jobs.filters.JobFilter;
import org.jobrunr.server.BackgroundJobPerformer;
import org.jobrunr.server.BackgroundJobPerformerFactory;
import org.jobrunr.server.BackgroundJobServerConfiguration;
import org.jobrunr.server.JobActivator;
import org.jobrunr.server.JobZooKeeper;
import org.jobrunr.server.ServerZooKeeper;
import org.jobrunr.server.dashboard.DashboardNotificationManager;
import org.jobrunr.server.jmx.BackgroundJobServerMBean;
import org.jobrunr.server.jmx.JobServerStats;
import org.jobrunr.server.runner.BackgroundJobRunner;
import org.jobrunr.server.runner.BackgroundJobWithIocRunner;
import org.jobrunr.server.runner.BackgroundJobWithoutIocRunner;
import org.jobrunr.server.runner.BackgroundStaticFieldJobWithoutIocRunner;
import org.jobrunr.server.runner.BackgroundStaticJobWithoutIocRunner;
import org.jobrunr.server.strategy.WorkDistributionStrategy;
import org.jobrunr.server.tasks.CheckForNewJobRunrVersion;
import org.jobrunr.server.tasks.CheckIfAllJobsExistTask;
import org.jobrunr.server.tasks.CreateClusterIdIfNotExists;
import org.jobrunr.server.tasks.MigrateFromV5toV6Task;
import org.jobrunr.server.threadpool.JobRunrExecutor;
import org.jobrunr.server.threadpool.ScheduledThreadPoolJobRunrExecutor;
import org.jobrunr.storage.BackgroundJobServerStatus;
import org.jobrunr.storage.StorageProvider;
import org.jobrunr.storage.ThreadSafeStorageProvider;
import org.jobrunr.utils.JobUtils;
import org.jobrunr.utils.mapper.JsonMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackgroundJobServer
implements BackgroundJobServerMBean {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackgroundJobServer.class);
    private final BackgroundJobServerConfiguration configuration;
    private final StorageProvider storageProvider;
    private final DashboardNotificationManager dashboardNotificationManager;
    private final JsonMapper jsonMapper;
    private final List<BackgroundJobRunner> backgroundJobRunners;
    private final JobDefaultFilters jobDefaultFilters;
    private final JobServerStats jobServerStats;
    private final WorkDistributionStrategy workDistributionStrategy;
    private final ServerZooKeeper serverZooKeeper;
    private final JobZooKeeper jobZooKeeper;
    private final BackgroundJobServerLifecycleLock lifecycleLock;
    private final BackgroundJobPerformerFactory backgroundJobPerformerFactory;
    private volatile Instant firstHeartbeat;
    private volatile boolean isRunning;
    private volatile Boolean isMaster;
    private volatile ScheduledThreadPoolExecutor zookeeperThreadPool;
    private JobRunrExecutor jobExecutor;

    public BackgroundJobServer(StorageProvider storageProvider, JsonMapper jsonMapper) {
        this(storageProvider, jsonMapper, null);
    }

    public BackgroundJobServer(StorageProvider storageProvider, JsonMapper jsonMapper, JobActivator jobActivator) {
        this(storageProvider, jsonMapper, jobActivator, BackgroundJobServerConfiguration.usingStandardBackgroundJobServerConfiguration());
    }

    public BackgroundJobServer(StorageProvider storageProvider, JsonMapper jsonMapper, JobActivator jobActivator, BackgroundJobServerConfiguration configuration) {
        if (storageProvider == null) {
            throw new IllegalArgumentException("A StorageProvider is required to use a BackgroundJobServer. Please see the documentation on how to setup a job StorageProvider.");
        }
        this.configuration = configuration;
        this.storageProvider = new ThreadSafeStorageProvider(storageProvider);
        this.dashboardNotificationManager = new DashboardNotificationManager(configuration.getId(), storageProvider);
        this.jsonMapper = jsonMapper;
        this.backgroundJobRunners = this.initializeBackgroundJobRunners(jobActivator);
        this.jobDefaultFilters = new JobDefaultFilters(new JobFilter[0]);
        this.jobServerStats = new JobServerStats();
        this.workDistributionStrategy = this.createWorkDistributionStrategy(configuration);
        this.serverZooKeeper = this.createServerZooKeeper();
        this.jobZooKeeper = this.createJobZooKeeper();
        this.backgroundJobPerformerFactory = this.loadBackgroundJobPerformerFactory();
        this.lifecycleLock = new BackgroundJobServerLifecycleLock();
    }

    @Override
    public UUID getId() {
        return this.configuration.getId();
    }

    @Override
    public void start() {
        this.start(true);
    }

    public void start(boolean guard) {
        if (guard) {
            if (this.isStarted()) {
                return;
            }
            try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
                this.firstHeartbeat = Instant.now();
                this.isRunning = true;
                this.startZooKeepers();
                this.startWorkers();
            }
        }
    }

    @Override
    public void pauseProcessing() {
        if (this.isStopped()) {
            throw new IllegalStateException("First start the BackgroundJobServer before pausing");
        }
        if (this.isPaused()) {
            return;
        }
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            this.isRunning = false;
            this.stopWorkers();
            LOGGER.info("Paused job processing");
        }
    }

    @Override
    public void resumeProcessing() {
        if (this.isStopped()) {
            throw new IllegalStateException("First start the BackgroundJobServer before resuming");
        }
        if (this.isProcessing()) {
            return;
        }
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            this.startWorkers();
            this.isRunning = true;
            LOGGER.info("Resumed job processing");
        }
    }

    @Override
    public void stop() {
        if (this.isStopped()) {
            return;
        }
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            LOGGER.info("BackgroundJobServer and BackgroundJobPerformers - stopping (waiting for all jobs to complete - max 10 seconds)");
            this.isMaster = null;
            this.stopWorkers();
            this.stopZooKeepers();
            this.isRunning = false;
            this.firstHeartbeat = null;
            LOGGER.info("BackgroundJobServer and BackgroundJobPerformers stopped");
        }
    }

    public boolean isAnnounced() {
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            boolean bl = this.isMaster != null;
            return bl;
        }
    }

    public boolean isUnAnnounced() {
        return !this.isAnnounced();
    }

    public boolean isMaster() {
        return this.isAnnounced() && this.isMaster != false;
    }

    void setIsMaster(Boolean isMaster) {
        if (this.isStopped()) {
            return;
        }
        this.isMaster = isMaster;
        if (isMaster != null) {
            LOGGER.info("JobRunr BackgroundJobServer ({}) using {} and {} BackgroundJobPerformers started successfully", new Object[]{this.getId(), this.storageProvider.getStorageProviderInfo().getName(), this.workDistributionStrategy.getWorkerCount()});
            if (isMaster.booleanValue()) {
                this.runStartupTasks();
            }
        } else {
            LOGGER.error("JobRunr BackgroundJobServer failed to start");
        }
    }

    @Override
    public boolean isRunning() {
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            boolean bl = this.isRunning;
            return bl;
        }
    }

    @Override
    public BackgroundJobServerStatus getServerStatus() {
        return new BackgroundJobServerStatus(this.configuration.getId(), this.configuration.getName(), this.workDistributionStrategy.getWorkerCount(), this.configuration.getPollIntervalInSeconds(), this.configuration.getDeleteSucceededJobsAfter(), this.configuration.getPermanentlyDeleteDeletedJobsAfter(), this.firstHeartbeat, Instant.now(), this.isRunning, this.jobServerStats);
    }

    public JobZooKeeper getJobZooKeeper() {
        return this.jobZooKeeper;
    }

    public StorageProvider getStorageProvider() {
        return this.storageProvider;
    }

    public BackgroundJobServerConfiguration getConfiguration() {
        return this.configuration;
    }

    public DashboardNotificationManager getDashboardNotificationManager() {
        return this.dashboardNotificationManager;
    }

    public JsonMapper getJsonMapper() {
        return this.jsonMapper;
    }

    public WorkDistributionStrategy getWorkDistributionStrategy() {
        return this.workDistributionStrategy;
    }

    public void setJobFilters(List<JobFilter> jobFilters) {
        this.jobDefaultFilters.addAll(jobFilters);
    }

    public JobDefaultFilters getJobFilters() {
        return this.jobDefaultFilters;
    }

    BackgroundJobRunner getBackgroundJobRunner(Job job) {
        JobUtils.assertJobExists(job.getJobDetails());
        return this.backgroundJobRunners.stream().filter(jobRunner -> jobRunner.supports(job)).findFirst().orElseThrow(() -> JobRunrException.problematicConfigurationException("Could not find a BackgroundJobRunner: either no JobActivator is registered, your Background Job Class is not registered within the IoC container or your Job does not have a default no-arg constructor."));
    }

    public void processJob(Job job) {
        BackgroundJobPerformer backgroundJobPerformer = this.backgroundJobPerformerFactory.newBackgroundJobPerformer(this, job);
        this.jobExecutor.execute(backgroundJobPerformer);
        LOGGER.debug("Submitted BackgroundJobPerformer for job {} to executor service", (Object)job.getId());
    }

    boolean isStarted() {
        return !this.isStopped();
    }

    boolean isStopped() {
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            boolean bl = this.zookeeperThreadPool == null;
            return bl;
        }
    }

    boolean isPaused() {
        return !this.isProcessing();
    }

    boolean isProcessing() {
        try (BackgroundJobServerLifecycleLock ignored = this.lifecycleLock.lock();){
            boolean bl = this.isRunning;
            return bl;
        }
    }

    private void startZooKeepers() {
        this.zookeeperThreadPool = new ScheduledThreadPoolJobRunrExecutor(2, "backgroundjob-zookeeper-pool");
        this.zookeeperThreadPool.scheduleWithFixedDelay(this.serverZooKeeper, 0L, this.configuration.getPollIntervalInSeconds(), TimeUnit.SECONDS);
        this.zookeeperThreadPool.scheduleWithFixedDelay(this.jobZooKeeper, 1L, this.configuration.getPollIntervalInSeconds(), TimeUnit.SECONDS);
        this.zookeeperThreadPool.scheduleWithFixedDelay(new CheckForNewJobRunrVersion(this), 1L, 8L, TimeUnit.HOURS);
    }

    private void stopZooKeepers() {
        this.serverZooKeeper.stop();
        this.stop(this.zookeeperThreadPool);
        this.zookeeperThreadPool = null;
    }

    private void startWorkers() {
        this.jobExecutor = this.loadJobRunrExecutor();
        this.jobExecutor.start();
    }

    private void stopWorkers() {
        if (this.jobExecutor == null) {
            return;
        }
        this.jobExecutor.stop();
        this.jobExecutor = null;
    }

    private void runStartupTasks() {
        try {
            List<Runnable> startupTasks = Arrays.asList(new CreateClusterIdIfNotExists(this), new CheckIfAllJobsExistTask(this), new CheckForNewJobRunrVersion(this), new MigrateFromV5toV6Task(this));
            startupTasks.forEach(this.jobExecutor::execute);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private List<BackgroundJobRunner> initializeBackgroundJobRunners(JobActivator jobActivator) {
        return Arrays.asList(new BackgroundJobWithIocRunner(jobActivator), new BackgroundJobWithoutIocRunner(), new BackgroundStaticJobWithoutIocRunner(), new BackgroundStaticFieldJobWithoutIocRunner());
    }

    private void stop(ScheduledExecutorService executorService) {
        if (executorService == null) {
            return;
        }
        executorService.shutdown();
        try {
            if (!executorService.awaitTermination(10L, TimeUnit.SECONDS)) {
                LOGGER.info("JobRunr BackgroundJobServer shutdown requested - waiting for jobs to finish (at most 10 seconds)");
                executorService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private ServerZooKeeper createServerZooKeeper() {
        return new ServerZooKeeper(this);
    }

    private JobZooKeeper createJobZooKeeper() {
        return new JobZooKeeper(this);
    }

    private WorkDistributionStrategy createWorkDistributionStrategy(BackgroundJobServerConfiguration configuration) {
        return configuration.getBackgroundJobServerWorkerPolicy().toWorkDistributionStrategy(this);
    }

    private BackgroundJobPerformerFactory loadBackgroundJobPerformerFactory() {
        ServiceLoader<BackgroundJobPerformerFactory> serviceLoader = ServiceLoader.load(BackgroundJobPerformerFactory.class);
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(serviceLoader.iterator(), 16), false).min((a, b) -> Integer.compare(b.getPriority(), a.getPriority())).orElseGet(() -> new BasicBackgroundJobPerformerFactory());
    }

    private JobRunrExecutor loadJobRunrExecutor() {
        ServiceLoader<JobRunrExecutor> serviceLoader = ServiceLoader.load(JobRunrExecutor.class);
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(serviceLoader.iterator(), 16), false).min((a, b) -> Integer.compare(b.getPriority(), a.getPriority())).orElseGet(() -> new ScheduledThreadPoolJobRunrExecutor(this.workDistributionStrategy.getWorkerCount(), "backgroundjob-worker-pool"));
    }

    private static class BackgroundJobServerLifecycleLock
    implements AutoCloseable {
        private final ReentrantLock reentrantLock = new ReentrantLock();

        private BackgroundJobServerLifecycleLock() {
        }

        public BackgroundJobServerLifecycleLock lock() {
            if (this.reentrantLock.isHeldByCurrentThread()) {
                return this;
            }
            this.reentrantLock.lock();
            return this;
        }

        @Override
        public void close() {
            this.reentrantLock.unlock();
        }
    }

    private static class BasicBackgroundJobPerformerFactory
    implements BackgroundJobPerformerFactory {
        private BasicBackgroundJobPerformerFactory() {
        }

        @Override
        public int getPriority() {
            return 10;
        }

        @Override
        public BackgroundJobPerformer newBackgroundJobPerformer(BackgroundJobServer backgroundJobServer, Job job) {
            return new BackgroundJobPerformer(backgroundJobServer, job);
        }
    }
}

