/*
 * Decompiled with CFR 0.152.
 */
package org.jobrunr.server;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import org.jobrunr.server.BackgroundJobServer;
import org.jobrunr.server.dashboard.CpuAllocationIrregularityNotification;
import org.jobrunr.server.dashboard.DashboardNotificationManager;
import org.jobrunr.storage.BackgroundJobServerStatus;
import org.jobrunr.storage.ServerTimedOutException;
import org.jobrunr.storage.StorageProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerZooKeeper
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServerZooKeeper.class);
    private final BackgroundJobServer backgroundJobServer;
    private final StorageProvider storageProvider;
    private final DashboardNotificationManager dashboardNotificationManager;
    private final Duration timeoutDuration;
    private final AtomicInteger restartAttempts;
    private UUID masterId;
    private Instant lastSignalAlive;
    private Instant lastServerTimeoutCheck;

    public ServerZooKeeper(BackgroundJobServer backgroundJobServer) {
        this.backgroundJobServer = backgroundJobServer;
        this.storageProvider = backgroundJobServer.getStorageProvider();
        this.dashboardNotificationManager = backgroundJobServer.getDashboardNotificationManager();
        this.timeoutDuration = Duration.ofSeconds(backgroundJobServer.getConfiguration().getPollIntervalInSeconds()).multipliedBy(4L);
        this.restartAttempts = new AtomicInteger();
        this.lastSignalAlive = Instant.now();
        this.lastServerTimeoutCheck = Instant.now();
    }

    @Override
    public void run() {
        if (this.backgroundJobServer.isStopped()) {
            return;
        }
        try {
            if (this.backgroundJobServer.isUnAnnounced()) {
                this.announceBackgroundJobServer();
            } else {
                this.signalBackgroundJobServerAliveAndDoZooKeeping();
            }
        }
        catch (Exception shouldNotHappen) {
            LOGGER.error("An unrecoverable error occurred. Shutting server down...", (Throwable)shouldNotHappen);
            if (this.masterId == null) {
                this.backgroundJobServer.setIsMaster(null);
            }
            new Thread(this::stopServer).start();
        }
    }

    public synchronized void stop() {
        try {
            this.storageProvider.signalBackgroundJobServerStopped(this.backgroundJobServer.getServerStatus());
        }
        catch (Exception e) {
            LOGGER.error("Error when signalling that BackgroundJobServer stopped", (Throwable)e);
        }
        finally {
            this.masterId = null;
        }
    }

    private void announceBackgroundJobServer() {
        BackgroundJobServerStatus serverStatus = this.backgroundJobServer.getServerStatus();
        this.storageProvider.announceBackgroundJobServer(serverStatus);
        this.determineIfCurrentBackgroundJobServerIsMaster();
        this.lastSignalAlive = serverStatus.getLastHeartbeat();
    }

    private void signalBackgroundJobServerAliveAndDoZooKeeping() {
        try {
            this.signalBackgroundJobServerAlive();
            this.deleteServersThatTimedOut();
            this.determineIfCurrentBackgroundJobServerIsMaster();
        }
        catch (ServerTimedOutException e) {
            if (this.restartAttempts.getAndIncrement() < 3) {
                LOGGER.error("SEVERE ERROR - Server timed out while it's still alive. Are all servers using NTP and in the same timezone? Are you having long GC cycles? Restart attempt {} out of 3", (Object)this.restartAttempts);
                new Thread(this::resetServer).start();
            }
            LOGGER.error("FATAL - Server restarted 3 times but still times out by other servers. Shutting down.");
            new Thread(this::stopServer).start();
        }
    }

    private void signalBackgroundJobServerAlive() {
        BackgroundJobServerStatus serverStatus = this.backgroundJobServer.getServerStatus();
        this.storageProvider.signalBackgroundJobServerAlive(serverStatus);
        this.cpuAllocationIrregularity(this.lastSignalAlive, serverStatus.getLastHeartbeat()).ifPresent(amountOfSeconds -> this.dashboardNotificationManager.notify(new CpuAllocationIrregularityNotification((Integer)amountOfSeconds)));
        this.lastSignalAlive = serverStatus.getLastHeartbeat();
    }

    private void deleteServersThatTimedOut() {
        if (Instant.now().isAfter(this.lastServerTimeoutCheck.plus(this.timeoutDuration))) {
            Instant timedOutInstantUsingLastSignalAlive;
            Instant now = Instant.now();
            Instant defaultTimeoutInstant = now.minus(this.timeoutDuration);
            Instant timedOutInstant = ServerZooKeeper.min(defaultTimeoutInstant, timedOutInstantUsingLastSignalAlive = this.lastSignalAlive.minusMillis(500L));
            int amountOfServersThatTimedOut = this.storageProvider.removeTimedOutBackgroundJobServers(timedOutInstant);
            if (amountOfServersThatTimedOut > 0) {
                LOGGER.info("Removed {} server(s) that timed out", (Object)amountOfServersThatTimedOut);
            }
            this.lastServerTimeoutCheck = now;
        }
    }

    private void determineIfCurrentBackgroundJobServerIsMaster() {
        UUID longestRunningBackgroundJobServerId = this.storageProvider.getLongestRunningBackgroundJobServerId();
        if (this.masterId == null || !this.masterId.equals(longestRunningBackgroundJobServerId)) {
            this.masterId = longestRunningBackgroundJobServerId;
            if (this.masterId.equals(this.backgroundJobServer.getId())) {
                this.backgroundJobServer.setIsMaster(true);
                LOGGER.info("Server {} is master (this BackgroundJobServer)", (Object)this.masterId);
            } else {
                this.backgroundJobServer.setIsMaster(false);
                LOGGER.info("Server {} is master (another BackgroundJobServer)", (Object)this.masterId);
            }
        }
    }

    private void resetServer() {
        this.backgroundJobServer.stop();
        this.backgroundJobServer.start();
    }

    private void stopServer() {
        this.backgroundJobServer.stop();
    }

    private static Instant min(Instant instant1, Instant instant2) {
        Object[] instants = new Instant[]{instant1, instant2};
        Arrays.sort(instants);
        return instants[0];
    }

    private Optional<Integer> cpuAllocationIrregularity(Instant lastSignalAlive, Instant lastHeartbeat) {
        int amount3OfSec;
        int amount2OfSec;
        Instant now = Instant.now();
        int amount1OfSec = (int)Math.abs(lastHeartbeat.getEpochSecond() - lastSignalAlive.getEpochSecond());
        int max = Math.max(amount1OfSec, Math.max(amount2OfSec = (int)(now.getEpochSecond() - lastSignalAlive.getEpochSecond()), amount3OfSec = (int)(now.getEpochSecond() - lastHeartbeat.getEpochSecond())));
        if ((long)max > (long)this.backgroundJobServer.getConfiguration().getPollIntervalInSeconds() * 2L) {
            return Optional.of(max);
        }
        return Optional.empty();
    }
}

