"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/********************************************************************************
 * Copyright (c) 2019 TypeFox and others
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 ********************************************************************************/
const vsce_1 = require("@vscode/vsce");
const util_1 = require("./util");
const registry_1 = require("./registry");
const check_license_1 = require("./check-license");
/**
 * Publishes an extension.
 */
async function publish(options = {}) {
    util_1.addEnvOptions(options);
    const internalPublishOptions = [];
    const packagePaths = options.packagePath || [undefined];
    const targets = options.targets || [undefined];
    for (const packagePath of packagePaths) {
        for (const target of targets) {
            internalPublishOptions.push({ ...options, packagePath: packagePath, target: target });
        }
    }
    return Promise.allSettled(internalPublishOptions.map(publishOptions => doPublish(publishOptions)));
}
exports.publish = publish;
async function doPublish(options = {}) {
    if (!options.pat) {
        throw new Error("A personal access token must be given with the option '--pat'.");
    }
    // if the packagePath is a link to a vsix, don't need to package it
    if (options.packagePath && options.packagePath.endsWith('.vsix')) {
        options.extensionFile = options.packagePath;
        delete options.packagePath;
        delete options.target;
    }
    const registry = new registry_1.Registry(options);
    if (!options.extensionFile) {
        await packageExtension(options, registry);
        console.log(); // new line
    }
    else if (options.preRelease) {
        console.warn("Ignoring option '--pre-release' for prepackaged extension.");
    }
    let extension;
    try {
        extension = await registry.publish(options.extensionFile, options.pat);
    }
    catch (err) {
        if (options.skipDuplicate && err.message.endsWith('is already published.')) {
            console.log(err.message + ' Skipping publish.');
            return;
        }
        else {
            throw err;
        }
    }
    if (extension.error) {
        throw new Error(extension.error);
    }
    const name = `${extension.namespace}.${extension.name}`;
    let description = `${name} v${extension.version}`;
    if (extension.targetPlatform !== 'universal') {
        description += `@${extension.targetPlatform}`;
    }
    console.log(`\ud83d\ude80  Published ${description}`);
    if (extension.warning) {
        console.log(`\n!!  ${extension.warning}`);
    }
}
async function packageExtension(options, registry) {
    if (registry.requiresLicense) {
        await check_license_1.checkLicense(options.packagePath);
    }
    options.extensionFile = await util_1.createTempFile({ postfix: '.vsix' });
    const packageOptions = {
        packagePath: options.extensionFile,
        target: options.target,
        cwd: options.packagePath,
        baseContentUrl: options.baseContentUrl,
        baseImagesUrl: options.baseImagesUrl,
        useYarn: options.yarn,
        dependencies: options.dependencies,
        preRelease: options.preRelease
    };
    await vsce_1.createVSIX(packageOptions);
}
//# sourceMappingURL=publish.js.map