# Assisted Installer Customization

## Manifests

These APIs allows for adding arbitrary manifests to the set generated by `openshift-install create manifests`.
A typical use case would be to create a MachineConfig which would make some persistent customization to a group of nodes.
These will only take effect after the machine config operator is up and running so if a change is needed before or during the installation process, another API will be required.

### Create a cluster manifest

```sh
# base64 encoding of the example from https://docs.openshift.com/container-platform/4.6/installing/install_config/installing-customizing.html 
content=YXBpVmVyc2lvbjogbWFjaGluZWNvbmZpZ3VyYXRpb24ub3BlbnNoaWZ0LmlvL3YxCmtpbmQ6IE1hY2hpbmVDb25maWcKbWV0YWRhdGE6CiAgbGFiZWxzOgogICAgbWFjaGluZWNvbmZpZ3VyYXRpb24ub3BlbnNoaWZ0LmlvL3JvbGU6IG1hc3RlcgogIG5hbWU6IDk5LW9wZW5zaGlmdC1tYWNoaW5lY29uZmlnLW1hc3Rlci1rYXJncwpzcGVjOgogIGtlcm5lbEFyZ3VtZW50czoKICAgIC0gJ2xvZ2xldmVsPTcnCg==
file=99-openshift-machineconfig-master-kargs.yaml
folder=openshift

curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request POST \
    --data "{\"file_name\":\"$file\", \"folder\":\"$folder\", \"content\":\"$content\"}" \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/clusters/$CLUSTER_ID/manifests"
```

### View a manifest’s contents

```sh
file=openshift/99-openshift-machineconfig-master-kargs.yaml

curl --header "Authorization: Bearer $TOKEN" "http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/clusters/$CLUSTER_ID/manifests/files?file_name=$file"
```

## Discovery Ignition

The discovery ignition is used to make changes to the CoreOS live iso image which runs before we actually write anything to the target disk.
An example use case would be configuring a separate container registry to pull the assisted-installer-agent image from.

The discovery ignition must use version 3.1.0 regardless of the version of the cluster that will eventually be created.

### Patch the discovery ignition

```sh
# ignition patch file
{
  "config": "{\"ignition\": {\"version\": \"3.1.0\"}, \"storage\": {\"files\": [{\"path\": \"/etc/containers/registries.conf\", \"mode\": 420, \"overwrite\": true, \"user\": { \"name\": \"root\"},\"contents\": {\"source\": \"data:text/plain;base64,dW5xdWFsaWZpZWQtc2VhcmNoLXJlZ2lzdHJpZXMgPSBbInJlZ2lzdHJ5LmFjY2Vzcy5yZWRoYXQuY29tIiwgImRvY2tlci5pbyJdCltbcmVnaXN0cnldXQogICBwcmVmaXggPSAiIgogICBsb2NhdGlvbiA9ICJxdWF5LmlvL29jcG1ldGFsIgogICBtaXJyb3ItYnktZGlnZXN0LW9ubHkgPSBmYWxzZQogICBbW3JlZ2lzdHJ5Lm1pcnJvcl1dCiAgIGxvY2F0aW9uID0gImxvY2FsLnJlZ2lzdHJ5OjUwMDAvb2NwbWV0YWwiCg==\"}}]}}"
}

curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request PATCH \
    --data @discovery-ign.json \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/infra-envs/$INFRA_ENV_ID"
```

### View the discovery ignition

```sh
curl --header "Authorization: Bearer $TOKEN" "http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/infra-envs/$INFRA_ENV_ID/downloads/'files?file_name=discovery.ign"
```

## Install Config

These endpoints alter the default install config yaml used when running `openshift-install create` commands.
Install config customization reference is available [here](https://github.com/openshift/installer/blob/master/docs/user/customization.md)
Some of this content will be pinned to particular values, but most can be edited.
Note, some of this content will be pinned to particular values by the assisted-installer (can't be overwritten).

You can compose the install-config overrides by creating a json string with the options you wish to set.
An example install config with disabled hyperthreading for the control plane:
```yaml
apiVersion: extensions.hive.openshift.io/v1beta1
baseDomain: example.com
controlPlane:
  name: master
  hyperthreading: Disabled
compute:
- name: worker
  replicas: 5
metadata:
  name: test-cluster
platform: ...
pullSecret: '{"auths": ...}'
sshKey: ssh-ed25519 AAAA...
```
should look like this:
```sh
"{\"controlPlane\":{\"hyperthreading\":\"Disabled\"}}"
```

### Patch the install config

```sh
curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request PATCH \
    --data '"{\"controlPlane\":{\"hyperthreading\":\"Disabled\"}}"' \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/clusters/$CLUSTER_ID/install-config"
```

### View the install config

```sh
curl --header "Authorization: Bearer $TOKEN" "http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/clusters/$CLUSTER_ID/install-config"
```

## Pointer Ignition

The pointer ignition is used to customize the particular host when it reboots into the installed system.
These changes will be made to the host, but will not be tracked as a machine config.
Generally a machine config will be a better fit than editing the pointer ignition, but if some change is specific to an individual host it would be made here.

The pointer ignition override version must match the version of the ignition generated by openshift-installer.
This means that the version required for the override will change depending on the cluster version being installed.

| OCP Version(s)      | Ignition version |
|---------------------|------------------|
| 4.6                 | 3.1.0            |
| 4.7, 4.8, 4.9, 4.10 | 3.2.0            |

### Patch the pointer ignition

```sh
curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request PATCH \
    --data '{"config": "{\"ignition\": {\"version\": \"3.1.0\"}, \"storage\": {\"files\": [{\"path\": \"/etc/example\", \"contents\": {\"source\": \"data:text/plain;base64,SGVsbG8gZnJvbSBob3N0IDg0Njk2NzdiLThlZGEtNDQzOS1iNDQwLTc3ZGM5M2FkZmNlZgo=\"}}]}}"}' \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/infra-envs/$INFRA_ENV_ID/hosts/$HOST_ID/ignition"
```

### View the pointer ignition

```sh
curl --header "Authorization: Bearer $TOKEN" "http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/infra-envs/$INFRA_ENV_ID/$HOST_ID/ignition
```

## Installer Params

This endpoint sets parameters to be passed to the coreos-installer command line in addition to the ones we provide by default.
The only parameters that can be set with this endpoint are: ["--append-karg", "--delete-karg", "-n", "--copy-network", "--network-dir", "--save-partlabel", "--save-partindex", "--image-url"]

### Set the installer params

```sh
curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request PATCH \
    --data '{"args": ["--append-karg", "nameserver=8.8.8.8", "-n"]}' \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/infra-envs/$INFRA_ENV_ID/hosts/$HOST_ID/installer-args"
```

## Adding aditional trust bundle
Any additional trust bundles need to be added to both the discovery ISO and to the install-config. The trust bundle can contain one or more additional CA certificates in the format:
```sh
-----BEGIN CERTIFICATE-----
...base-64-encoded, DER Certificate Authority cert...
-----END CERTIFICATE-----

-----BEGIN CERTIFICATE-----
...base-64-encoded, DER Certificate Authority cert...
-----END CERTIFICATE-----
```
### Add additional trust bundle in discovery ISO
The additional trust bundle must be embeded into discovery ignition override
```sh
request_body=$(mktemp)
jq -n --arg OVERRIDE "{\"ignition\": {\"version\": \"3.1.0\"}, \"storage\": {\"files\": [{\"path\": \"/etc/pki/ca-trust/source/anchors/extra_ca.pem\", \"mode\": 420, \"overwrite\": true, \"user\": { \"name\": \"root\"},\"contents\": {\"source\": \"data:text/plain;base64,$(cat ca.pem | base64 -w 0)\"}}]}}" \
'{
   "ignition_config_override": $OVERRIDE
}' > $request_body
```
Patch the discovery ignition override with the resulting file:
```sh
curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request PATCH \
    --data  @$request_body \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/infra-envs/$INFRA_ENV_ID"
```
### Add additionalTrustbundle in install-config
Create file to patch install config
```sh
install_config_patch=$(mktemp)
jq -n --arg BUNDLE "$(cat ca.pem)" \
'{
    "additionalTrustBundle": $BUNDLE
}| tojson' > $install_config_body
```
Patch the install-config with the resulting file
```sh
curl \
    --header "Content-Type: application/json" \
    --header "Authorization: Bearer $TOKEN" \
    --request PATCH \
    --data  @$install_config_patch \
"http://$ASSISTED_SERVICE_IP:$ASSISTED_SERVICE_PORT/api/assisted-install/v2/clusters/$CLUSTER_ID/install-config"
```