// Copyright 2019 Gregory Petrosyan <gregory.petrosyan@gmail.com>
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

package rapid_test

import (
	"regexp"
	"testing"

	. "pgregory.net/rapid"
)

// Based on https://github.com/rust-lang/regex/blob/master/tests/crates_regex.rs
var crateRegexps = []string{
	`\s*(\d+)(\w)\s*`,                          // autoshutdown-0.1.0: r"\s*(\d+)(\w)\s*"
	`/`,                                        // epub-1.1.1: r"/"
	"^Revision\t+: ([0-9a-fA-F]+)",             // rpi-info-0.2.0: "^Revision\t+: ([0-9a-fA-F]+)"
	"Serial\t+: ([0-9a-fA-F]+)",                // rpi-info-0.2.0: "Serial\t+: ([0-9a-fA-F]+)"
	`^u([0-9]+)(be|le|he)?$`,                   // pnet_macros-0.21.0: r"^u([0-9]+)(be|le|he)?$"
	`^[A-Z]{2}\d{2}[A-Z\d]{1,30}$`,             // iban_validate-1.0.3: r"^[A-Z]{2}\d{2}[A-Z\d]{1,30}$"
	`.*\[(?P<percent>.+)%.*\].*`,               // markifier-0.1.0: r".*\[(?P<percent>.+)%.*\].*"
	`(#include) (\S*)(.*)`,                     // mallumo-0.3.0: r"(#include) (\S*)(.*)"
	`(ERROR: \d+:)(\d+)(: )(.+)`,               // mallumo-0.3.0: r"(ERROR: \d+:)(\d+)(: )(.+)"
	`(\d+\()(\d+)(?:\) : )(.+)`,                // mallumo-0.3.0: r"(\d+\()(\d+)(?:\) : )(.+)"
	`(.+?)(\[.*?\])?`,                          // magnet_more-0.0.1: r"(.+?)(\[.*?\])?"
	`:(?P<k>[a-zA-Z_]+)`,                       // magnet_app-0.0.1: r":(?P<k>[a-zA-Z_]+)"
	`^\d{6}(?:\s*,\s*\d{6})*$`,                 // yubibomb-0.2.0: r"^\d{6}(?:\s*,\s*\d{6})*$"
	`[\\/]([^\\/?]+)(\?.*)?$`,                  // multirust-rs-0.0.4: r"[\\/]([^\\/?]+)(\?.*)?$"
	"\"[a-z]*\":null",                          // hueclient-0.3.2: "\"[a-z]*\":null"
	",+",                                       // hueclient-0.3.2: ",+"
	",\\}",                                     // hueclient-0.3.2: ",\\}"
	"\\{,",                                     // hueclient-0.3.2: "\\{,"
	`[a-zA-Z_\$][a-zA-Z_0-9]*`,                 // aerial-0.1.0: r"[a-zA-Z_\$][a-zA-Z_0-9]*"
	`thi[sng]+`,                                // aerial-0.1.0: r"thi[sng]+"
	`(.+)\s+\((.+?)\)`,                         // rvue-0.1.0: r"(.+)\s+\((.+?)\)"
	`([\d\.]+)\s*out\s*of\s*([\d\.]+)`,         // rvue-0.1.0: r"([\d\.]+)\s*out\s*of\s*([\d\.]+)"
	`^([\d\.]+)\s*(?:\(\))?$`,                  // rvue-0.1.0: r"^([\d\.]+)\s*(?:\(\))?$"
	`([\d\.]+)\s*Points\s*Possible`,            // rvue-0.1.0: r"([\d\.]+)\s*Points\s*Possible"
	`([\d\.]+)\s*/\s*([\d\.]+)`,                // rvue-0.1.0: r"([\d\.]+)\s*/\s*([\d\.]+)"
	`_?([_a-z0-9]+)\s*:\s*([_a-z0-9]+)\s*[,)]`, // rvsim-0.1.0: r"_?([_a-z0-9]+)\s*:\s*([_a-z0-9]+)\s*[,)]"
	"(.*[^\\\\])\\{\\}(.*)",                    // nereon-0.1.4: "(.*[^\\\\])\\{\\}(.*)"
	`((?i)^(.+).s(\d+)e(\d+).*)$`,              // next_episode-0.3.0: r"((?i)^(.+).s(\d+)e(\d+).*)$"
	`[^a-z0-9-]+`,                              // migrant_lib-0.19.2: r"[^a-z0-9-]+"
	`[0-9]{14}_[a-z0-9-]+`,                     // migrant_lib-0.19.2: r"[0-9]{14}_[a-z0-9-]+"
	`([0-9]{14}_)?[a-z0-9-]+`,                  // migrant_lib-0.19.2: r"([0-9]{14}_)?[a-z0-9-]+"
	// minipre-0.2.0: "$_"
	`>\s+<`,                      // minifier-0.0.13: r">\s+<"
	`\s{2,}|[\r\n]`,              // minifier-0.0.13: r"\s{2,}|[\r\n]"
	`<(style|script)[\w|\s].*?>`, // minifier-0.0.13: r"<(style|script)[\w|\s].*?>"
	"<!--(.|\n)*?-->",            // minifier-0.0.13: "<!--(.|\n)*?-->"
	`<\w.*?>`,                    // minifier-0.0.13: r"<\w.*?>"
	` \s+|\s +`,                  // minifier-0.0.13: r" \s+|\s +"
	`\w\s+\w`,                    // minifier-0.0.13: r"\w\s+\w"
	`'\s+>`,                      // minifier-0.0.13: r"'\s+>"
	`\d\s+>`,                     // minifier-0.0.13: r"\d\s+>"
	`(?P<relation>\([^)]+\))|(?P<prop>[a-zA-Z0-9_]+)`, // ggp-rs-0.1.2: r"(?P<relation>\([^)]+\))|(?P<prop>[a-zA-Z0-9_]+)"
	`\((.*)\).`,                   // ggp-rs-0.1.2: r"\((.*)\)."
	"[A-Za-z0-9_]",                // poe-superfilter-0.2.0: "[A-Za-z0-9_]"
	`(\d+)x(\d+)`,                 // poke-a-mango-0.5.0: r"(\d+)x(\d+)"
	`(?P<nmsg>\d+) (?P<size>\d+)`, // pop3-rs-0.1.0: r"(?P<nmsg>\d+) (?P<size>\d+)"
	`(?P<msgid>\d+) (?P<uidl>[\x21-\x7E]{1,70})`, // pop3-rs-0.1.0: r"(?P<msgid>\d+) (?P<uidl>[\x21-\x7E]{1,70})"
	`(<.*>)\r\n$`, // pop3-rs-0.1.0: r"(<.*>)\r\n$"
	`^(?P<status>\+OK|-ERR) (?P<statustext>.*)`, // pop3-rs-0.1.0: r"^(?P<status>\+OK|-ERR) (?P<statustext>.*)"
	`^\.\r\n$`,                      // pop3-1.0.6: r"^\.\r\n$"
	`\+OK(.*)`,                      // pop3-1.0.6: r"\+OK(.*)"
	`-ERR(.*)`,                      // pop3-1.0.6: r"-ERR(.*)"
	`\+OK (\d+) (\d+)\r\n`,          // pop3-1.0.6: r"\+OK (\d+) (\d+)\r\n"
	`(\d+) ([\x21-\x7e]+)\r\n`,      // pop3-1.0.6: r"(\d+) ([\x21-\x7e]+)\r\n"
	`\+OK (\d+) ([\x21-\x7e]+)\r\n`, // pop3-1.0.6: r"\+OK (\d+) ([\x21-\x7e]+)\r\n"
	`(\d+) (\d+)\r\n`,               // pop3-1.0.6: r"(\d+) (\d+)\r\n"
	`\+OK (\d+) (\d+)\r\n`,          // pop3-1.0.6: r"\+OK (\d+) (\d+)\r\n"
	"github:(\\w+)/?(\\w+)?",        // polk-1.1.3: "github:(\\w+)/?(\\w+)?"
	"^[0-9]{5}",                     // geochunk-0.1.5: "^[0-9]{5}"
	`((?:(?:0|1[\d]{0,2}|2(?:[0-4]\d?|5[0-5]?|[6-9])?|[3-9]\d?)\.){3}(?:0|1[\d]{0,2}|2(?:[0-4]\d?|5[0-5]?|[6-9])?|[3-9]\d?))`, // generic-dns-update-1.1.4: r"((?:(?:0|1[\d]{0,2}|2(?:[0-4]\d?|5[0-5]?|[6-9])?|[3-9]\d?)\.){3}(?:0|1[\d]{0,2}|2(?:[0-4]\d?|5[0-5]?|[6-9])?|[3-9]\d?))"
	`((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){5}:([0-9A-Fa-f]{1,4}:)?[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,2}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,3}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){2}:([0-9A-Fa-f]{1,4}:){0,4}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}((\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d)\.){3}(\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d))|(([0-9A-Fa-f]{1,4}:){0,5}:((\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d)\.){3}(\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d))|(::([0-9A-Fa-f]{1,4}:){0,5}((\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d)\.){3}(\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d))|([0-9A-Fa-f]{1,4}::([0-9A-Fa-f]{1,4}:){0,5}[0-9A-Fa-f]{1,4})|(::([0-9A-Fa-f]{1,4}:){0,6}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){1,7}:))`, // generic-dns-update-1.1.4: r"((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){5}:([0-9A-Fa-f]{1,4}:)?[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,2}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,3}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){2}:([0-9A-Fa-f]{1,4}:){0,4}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}((\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d)\.){3}(\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d))|(([0-9A-Fa-f]{1,4}:){0,5}:((\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d)\.){3}(\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d))|(::([0-9A-Fa-f]{1,4}:){0,5}((\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d)\.){3}(\d((25[0-5])|(1\d{2})|(2[0-4]\d)|(\d{1,2}))\d))|([0-9A-Fa-f]{1,4}::([0-9A-Fa-f]{1,4}:){0,5}[0-9A-Fa-f]{1,4})|(::([0-9A-Fa-f]{1,4}:){0,6}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){1,7}:))"
	`<value><string>([0-9.]*)</string></value>`,                        // generic-dns-update-1.1.4: r"<value><string>([0-9.]*)</string></value>"
	`<int>([0-9]+)</int>`,                                              // generic-dns-update-1.1.4: r"<int>([0-9]+)</int>"
	`<int>([0-9]+)</int>`,                                              // generic-dns-update-1.1.4: r"<int>([0-9]+)</int>"
	`<boolean>([0-1]*)</boolean>`,                                      // generic-dns-update-1.1.4: r"<boolean>([0-1]*)</boolean>"
	`(\d*)\.(\d*)\.(\d*)(-(\S*))?`,                                     // generate-nix-pkg-0.3.0: r"(\d*)\.(\d*)\.(\d*)(-(\S*))?"
	`^(\S*) (\d*)\.(\d*)\.(\d*)(-(\S*))?`,                              // generate-nix-pkg-0.3.0: r"^(\S*) (\d*)\.(\d*)\.(\d*)(-(\S*))?"
	`arch/([a-z0-9_])+/`,                                               // genact-0.6.0: r"arch/([a-z0-9_])+/"
	`arch/([a-z0-9_])+/`,                                               // genact-0.6.0: r"arch/([a-z0-9_])+/"
	`^\s*((\*(/\d+)?)|[0-9-,/]+)(\s+((\*(/\d+)?)|[0-9-,/]+)){4,5}\s*$`, // cron_rs-0.1.6: r"^\s*((\*(/\d+)?)|[0-9-,/]+)(\s+((\*(/\d+)?)|[0-9-,/]+)){4,5}\s*$"
	`^([a-zA-Z]+)::(.+)$`,                                              // systemfd-0.3.0: r"^([a-zA-Z]+)::(.+)$"
	"__?hidden#\\d+_",                                                  // symbolic-debuginfo-5.0.2: "__?hidden#\\d+_"
	`^Linux ([^ ]+) (.*) \w+(?: GNU/Linux)?$`,                          // symbolic-minidump-5.0.2: r"^Linux ([^ ]+) (.*) \w+(?: GNU/Linux)?$"
	// graphql-idl-parser-0.1.1: "^(?u:\\#)(?u:[\t-\r - \u{85}-\u{85}\u{a0}-\u{a0}\u{1680}-\u{1680}\u{2000}-\u{200a}\u{2028}-\u{2029}\u{202f}-\u{202f}\u{205f}-\u{205f}\u{3000}-\u{3000}])*(?u:.)+"
	// graphql-idl-parser-0.1.1: "^(?u:=)(?u:[\t-\r - \u{85}-\u{85}\u{a0}-\u{a0}\u{1680}-\u{1680}\u{2000}-\u{200a}\u{2028}-\u{2029}\u{202f}-\u{202f}\u{205f}-\u{205f}\u{3000}-\u{3000}])*(?u:.)+"
	"^([A-Z_-_a-z])([0-9A-Z_-_a-z])*", // graphql-idl-parser-0.1.1: "^(?u:[A-Z_-_a-z])(?u:[0-9A-Z_-_a-z])*"
	"^(!)",                            // graphql-idl-parser-0.1.1: "^(?u:!)"
	"^(\\()",                          // graphql-idl-parser-0.1.1: "^(?u:\\()"
	"^(\\))",                          // graphql-idl-parser-0.1.1: "^(?u:\\))"
	"^(,)",                            // graphql-idl-parser-0.1.1: "^(?u:,)"
	"^(:)",                            // graphql-idl-parser-0.1.1: "^(?u::)"
	"^(@)",                            // graphql-idl-parser-0.1.1: "^(?u:@)"
	"^(\\[)",                          // graphql-idl-parser-0.1.1: "^(?u:\\[)"
	"^(\\])",                          // graphql-idl-parser-0.1.1: "^(?u:\\])"
	"^(enum)",                         // graphql-idl-parser-0.1.1: "^(?u:enum)"
	"^(implements)",                   // graphql-idl-parser-0.1.1: "^(?u:implements)"
	"^(input)",                        // graphql-idl-parser-0.1.1: "^(?u:input)"
	"^(interface)",                    // graphql-idl-parser-0.1.1: "^(?u:interface)"
	"^(scalar)",                       // graphql-idl-parser-0.1.1: "^(?u:scalar)"
	"^(type)",                         // graphql-idl-parser-0.1.1: "^(?u:type)"
	"^(union)",                        // graphql-idl-parser-0.1.1: "^(?u:union)"
	"^(\\{)",                          // graphql-idl-parser-0.1.1: "^(?u:\\{)"
	"^(\\})",                          // graphql-idl-parser-0.1.1: "^(?u:\\})"
	`(?s)/\*(?P<config>.*?)\*/`,       // grimoire-0.1.0: r"(?s)/\*(?P<config>.*?)\*/"
	`[\d]+(?:[~\x{2053}\x{223C}\x{FF5E}][\d]+)?`, // phonenumber-0.2.0+8.9.0: r"[\d]+(?:[~\x{2053}\x{223C}\x{FF5E}][\d]+)?"
	`[, \[\]]`,                 // phonenumber-0.2.0+8.9.0: r"[, \[\]]"
	`[\\/] *x`,                 // phonenumber-0.2.0+8.9.0: r"[\\/] *x"
	`[[\P{N}&&\P{L}]&&[^#]]+$`, // phonenumber-0.2.0+8.9.0: r"[[\P{N}&&\P{L}]&&[^#]]+$"
	`(?:.*?[A-Za-z]){3}.*`,     // phonenumber-0.2.0+8.9.0: r"(?:.*?[A-Za-z]){3}.*"
	`(\D+)`,                    // phonenumber-0.2.0+8.9.0: r"(\D+)"
	`(\$\d)`,                   // phonenumber-0.2.0+8.9.0: r"(\$\d)"
	`\(?\$1\)?`,                // phonenumber-0.2.0+8.9.0: r"\(?\$1\)?"
	`\D`,                       // phone_number-0.1.0: r"\D"
	`^0+`,                      // phone_number-0.1.0: r"^0+"
	`^89`,                      // phone_number-0.1.0: r"^89"
	`^8+`,                      // phone_number-0.1.0: r"^8+"
	`^ *(\^_*\^) *$`,           // phile-0.1.4: r"^ *(\^_*\^) *$"
	// phile-0.1.4: r"^[_\p{XID_Start}]$"
	// phile-0.1.4: r"^\p{XID_Continue}$"
	"%25(?P<hex>[0-9a-fA-F][0-9a-fA-F])", // uritemplate-0.1.2: "%25(?P<hex>[0-9a-fA-F][0-9a-fA-F])"
	"^package://(\\w+)/",                 // urdf-rs-0.4.2: "^package://(\\w+)/"
	`(?P<key>[?&.])`,                     // url-match-0.1.7: r"(?P<key>[?&.])"
	`:(?P<key>[a-zA-Z0-9_-]+)`,           // url-match-0.1.7: r":(?P<key>[a-zA-Z0-9_-]+)"
	`hello world`,                        // tsm-sys-0.1.0: r"hello world"
	"^(?:(?:(?:\\d+:).+)|(?:[^:]+))$",    // deb-version-0.1.0: "^(?:(?:(?:\\d+:).+)|(?:[^:]+))$"
	`^(?i)(a|an|the)\s+`,                 // debcargo-2.1.0: r"^(?i)(a|an|the)\s+"
	`^(?i)(rust\s+)?(implementation|library|tool|crate)\s+(of|to|for)\s+`, // debcargo-2.1.0: r"^(?i)(rust\s+)?(implementation|library|tool|crate)\s+(of|to|for)\s+"
	`^.*\.h$`,     // feaders-0.2.0: r"^.*\.h$"
	`^.*\.c$`,     // feaders-0.2.0: r"^.*\.c$"
	`^.*\.hpp$`,   // feaders-0.2.0: r"^.*\.hpp$"
	`^.*\.cc$`,    // feaders-0.2.0: r"^.*\.cc$"
	`^.*\.cpp$`,   // feaders-0.2.0: r"^.*\.cpp$"
	`CPtr\(\w+\)`, // hyperscan-0.1.6: r"CPtr\(\w+\)"
	`^Version:\s(\d\.\d\.\d)\sFeatures:\s+(\w+)?\sMode:\s(\w+)$`, // hyperscan-0.1.6: r"^Version:\s(\d\.\d\.\d)\sFeatures:\s+(\w+)?\sMode:\s(\w+)$"
	`RawDatabase<Block>\{db: \w+\}`,                              // hyperscan-0.1.6: r"RawDatabase<Block>\{db: \w+\}"
	`RawSerializedDatabase\{p: \w+, len: \d+\}`,                  // hyperscan-0.1.6: r"RawSerializedDatabase\{p: \w+, len: \d+\}"
	`[0-9A-F]+`,                    // ucd-parse-0.1.1: r"[0-9A-F]+"
	`.*`,                           // afsort-0.2.0: r".*"
	`.*`,                           // afsort-0.2.0: r".*"
	`.*`,                           // afsort-0.2.0: r".*"
	`.*`,                           // afsort-0.2.0: r".*"
	`.*`,                           // afsort-0.2.0: r".*"
	`.*`,                           // afsort-0.2.0: r".*"
	`^[a-z]+$`,                     // afsort-0.2.0: r"^[a-z]+$"
	`^[a-z]+$`,                     // afsort-0.2.0: r"^[a-z]+$"
	`(\.git|\.pijul|_darcs|\.hg)$`, // tin-summer-1.21.4: r"(\.git|\.pijul|_darcs|\.hg)$"
	`.*?\.(a|la|lo|o|ll|keter|bc|dyn_o|d|rlib|crate|min\.js|hi|dyn_hi|S|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$`, // tin-drummer-1.0.1: r".*?\.(a|la|lo|o|ll|keter|bc|dyn_o|d|rlib|crate|min\.js|hi|dyn_hi|S|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$"
	`.*?\.(stats|conf|h|out|cache.*|dat|pc|info|\.js)$`, // tin-drummer-1.0.1: r".*?\.(stats|conf|h|out|cache.*|dat|pc|info|\.js)$"
	`.*?\.(exe|a|la|o|ll|keter|bc|dyn_o|d|rlib|crate|min\.js|hi|dyn_hi|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$`, // tin-drummer-1.0.1: r".*?\.(exe|a|la|o|ll|keter|bc|dyn_o|d|rlib|crate|min\.js|hi|dyn_hi|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$"
	`.*?\.(stats|conf|h|out|cache.*|\.js)$`,                   // tin-drummer-1.0.1: r".*?\.(stats|conf|h|out|cache.*|\.js)$"
	`(\.git|\.pijul|_darcs|\.hg)$`,                            // tin-drummer-1.0.1: r"(\.git|\.pijul|_darcs|\.hg)$"
	`.*?\.(dyn_o|out|d|hi|dyn_hi|dump-.*|p_hi|p_o|prof|tix)$`, // tin-drummer-1.0.1: r".*?\.(dyn_o|out|d|hi|dyn_hi|dump-.*|p_hi|p_o|prof|tix)$"
	`.*?\.(ibc)$`,                     // tin-drummer-1.0.1: r".*?\.(ibc)$"
	`\.stack-work|dist-newstyle`,      // tin-drummer-1.0.1: r"\.stack-work|dist-newstyle"
	`_NET_WM_PID\(CARDINAL\) = (\d+)`, // timmy-0.3.0: r"_NET_WM_PID\(CARDINAL\) = (\d+)"
	`today|yesterday|now`,             // timmy-0.3.0: r"today|yesterday|now"
	`(?P<day>\d{1,2})/(?P<month>\d{1,2})(/(?P<year>\d{4}|\d{2}))?`, // timmy-0.3.0: r"(?P<day>\d{1,2})/(?P<month>\d{1,2})(/(?P<year>\d{4}|\d{2}))?"
	`(?P<n>\d+) (days?|ds?)(?P<ago>( ago)?)`,                       // timmy-0.3.0: r"(?P<n>\d+) (days?|ds?)(?P<ago>( ago)?)"
	`(?P<hr>\d{2}):(?P<mins>\d{2})`,                                // timmy-0.3.0: r"(?P<hr>\d{2}):(?P<mins>\d{2})"
	`^(\d+): \d+ windows \(.*\) \[\d+x\d+\]( \(attached\))?`,       // tinfo-0.5.0: r"^(\d+): \d+ windows \(.*\) \[\d+x\d+\]( \(attached\))?"
	`^(\d+):(\d+): (.*) \((\d+) panes\) \[(\d+)x(\d+)\]`,           // tinfo-0.5.0: r"^(\d+):(\d+): (.*) \((\d+) panes\) \[(\d+)x(\d+)\]"
	`(?:\\\{start\\\}|\\\{end\\\})`,                                // timespan-0.0.4: r"(?:\\\{start\\\}|\\\{end\\\})"
	`(.*)\s+-\s+(.*)`,                                              // timespan-0.0.4: r"(.*)\s+-\s+(.*)"
	`(.*)\s+(\w+)$`,                                                // timespan-0.0.4: r"(.*)\s+(\w+)$"
	`(.*)\s+(\w+)$`,                                                // timespan-0.0.4: r"(.*)\s+(\w+)$"
	`(.*)\s+-\s+(.*)`,                                              // timespan-0.0.4: r"(.*)\s+-\s+(.*)"
	`[[:lower:]]`,                                                  // titlecase-0.10.0: r"[[:lower:]]"
	`^\d+ (day|week|month|year)s?$`,                                // tight-0.1.3: r"^\d+ (day|week|month|year)s?$"
	`^\d+ (day|week|month|year)s?$`,                                // tight-0.1.3: r"^\d+ (day|week|month|year)s?$"
	`^[-+]?(0|[1-9][0-9_]*)$`,                                      // yaml-0.2.1: r"^[-+]?(0|[1-9][0-9_]*)$"
	`^([-+]?)0o?([0-7_]+)$`,                                        // yaml-0.2.1: r"^([-+]?)0o?([0-7_]+)$"
	`^([-+]?)0x([0-9a-fA-F_]+)$`,                                   // yaml-0.2.1: r"^([-+]?)0x([0-9a-fA-F_]+)$"
	`^([-+]?)0b([0-1_]+)$`,                                         // yaml-0.2.1: r"^([-+]?)0b([0-1_]+)$"
	`^([-+]?)(\.[0-9]+|[0-9]+(\.[0-9]*)?([eE][-+]?[0-9]+)?)$`,      // yaml-0.2.1: r"^([-+]?)(\.[0-9]+|[0-9]+(\.[0-9]*)?([eE][-+]?[0-9]+)?)$"
	`^[+]?(\.inf|\.Inf|\.INF)$`,                                    // yaml-0.2.1: r"^[+]?(\.inf|\.Inf|\.INF)$"
	`^-(\.inf|\.Inf|\.INF)$`,                                       // yaml-0.2.1: r"^-(\.inf|\.Inf|\.INF)$"
	`^(\.nan|\.NaN|\.NAN)$`,                                        // yaml-0.2.1: r"^(\.nan|\.NaN|\.NAN)$"
	`^(null|Null|NULL|~)$`,                                         // yaml-0.2.1: r"^(null|Null|NULL|~)$"
	`^(true|True|TRUE|yes|Yes|YES)$`,                               // yaml-0.2.1: r"^(true|True|TRUE|yes|Yes|YES)$"
	`^(false|False|FALSE|no|No|NO)$`,                               // yaml-0.2.1: r"^(false|False|FALSE|no|No|NO)$"
	`(?m)^(\S+)/(\S+) (\S+)(?: \((.*)\))?$`,                        // kefia-0.1.0: r"(?m)^(\S+)/(\S+) (\S+)(?: \((.*)\))?$"
	"^(\\s+|;.*?(\n|$))+",                                          // risp-0.7.0: "^(\\s+|;.*?(\n|$))+"
	"^\".*?\"",                                                     // risp-0.7.0: "^\".*?\""
	`^[^\s\{\}()\[\]]+`,                                            // risp-0.7.0: r"^[^\s\{\}()\[\]]+"
	`^-?\d+`,                                                       // risp-0.7.0: r"^-?\d+"
	"^([0-9]+)([KMG])?$",                                           // ripgrep-0.8.1: "^([0-9]+)([KMG])?$"
	`^\w+`,                                                         // riquid-0.0.1: r"^\w+"
	`^\d+`,                                                         // riquid-0.0.1: r"^\d+"
	`\A(0x)?([a-fA-F0-9]+)\z`,                                      // recursive_disassembler-2.1.2: r"\A(0x)?([a-fA-F0-9]+)\z"
	`^[a-zA-Z_][a-zA-Z0-9_]*`,                                      // remake-0.1.0: r"^[a-zA-Z_][a-zA-Z0-9_]*"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{2})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{2})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)`,                     // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)`,                    // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)`,                    // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)`,                    // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)`,                    // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)"
	`'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)`,                    // regex-decode-0.1.0: r"'(?P<title>[^']+)'\s+\((?P<year>\d{4})?\)"
	"[0-9]{3}-[0-9]{3}-[0-9]{4}",                                   // regex-cache-0.2.0: "[0-9]{3}-[0-9]{3}-[0-9]{4}"
	`^\d+$`,                                                        // regex-cache-0.2.0: r"^\d+$"
	`^[a-z]+$`,                                                     // regex-cache-0.2.0: r"^[a-z]+$"
	`^\d+$`,                                                        // regex-cache-0.2.0: r"^\d+$"
	`^\d+$`,                                                        // regex-cache-0.2.0: r"^\d+$"
	`\d{4}-\d{2}-\d{2}`,                                            // regex_dfa-0.5.0: r"\d{4}-\d{2}-\d{2}"
	`^[0-9\p{L} _\\.]{3,16}$`,                                      // reaper-2.0.0: r"^[0-9\p{L} _\\.]{3,16}$"
	`^attachment; filename=(.+)$`,                                  // retdec-0.1.0: r"^attachment; filename=(.+)$"
	`(\\)(?P<head>\$[0-9A-Za-z_{])`,                                // renvsubst-0.1.2: r"(\\)(?P<head>\$[0-9A-Za-z_{])"
	`\$([[:word:]]+)`,                                              // renvsubst-0.1.2: r"\$([[:word:]]+)"
	`\$\{([[:word:]]+)\}`,                                          // renvsubst-0.1.2: r"\$\{([[:word:]]+)\}"
	`'[a-z]+'`,                                                     // rexpect-0.3.0: r"'[a-z]+'"
	`^\d{4}-\d{2}-\d{2}$`,                                          // rexpect-0.3.0: r"^\d{4}-\d{2}-\d{2}$"
	`-\d{2}-`,                                                      // rexpect-0.3.0: r"-\d{2}-"
	"^a(b|c)c*$",                                                   // luther-0.1.0: "^a(b|c)c*$"
	`(\x9B|\x1B\[)[0-?]*[ -/]*[@-~]`,                               // little_boxes-1.6.0: r"(\x9B|\x1B\[)[0-?]*[ -/]*[@-~]"
	"^[a-zA-Z]([a-zA-Z0-9_-]*)$",                                   // libimagentrytag-0.8.0: "^[a-zA-Z]([a-zA-Z0-9_-]*)$"
	`^[Yy](\n?)$`,                                                  // libimaginteraction-0.8.0: r"^[Yy](\n?)$"
	`^[Nn](\n?)$`,                                                  // libimaginteraction-0.8.0: r"^[Nn](\n?)$"
	"^(?P<KEY>([^=]*))=(.*)$",                                      // libimagutil-0.8.0: "^(?P<KEY>([^=]*))=(.*)$"
	"(.*)=(\"(?P<QVALUE>([^\"]*))\"|(?P<VALUE>(.*)))$",             // libimagutil-0.8.0: "(.*)=(\"(?P<QVALUE>([^\"]*))\"|(?P<VALUE>(.*)))$"
	`\s+`,           // linux_ip-0.1.0: r"\s+"
	`\s*[\n\r]+\s*`, // linux_ip-0.1.0: r"\s*[\n\r]+\s*"
	`^([0-9a-fA-F\.:/]+)\s+dev\s+([a-z0-9\.]+)\s*(.*)$`,                                // linux_ip-0.1.0: r"^([0-9a-fA-F\.:/]+)\s+dev\s+([a-z0-9\.]+)\s*(.*)$"
	`^([0-9a-fA-F\.:/]+|default)\s+via\s+([a-z0-9\.:]+)\s+dev\s+([a-z0-9\.]+)\s*(.*)$`, // linux_ip-0.1.0: r"^([0-9a-fA-F\.:/]+|default)\s+via\s+([a-z0-9\.:]+)\s+dev\s+([a-z0-9\.]+)\s*(.*)$"
	`^(blackhole)\s+([0-9a-fA-F\.:/]+)$`,                                               // linux_ip-0.1.0: r"^(blackhole)\s+([0-9a-fA-F\.:/]+)$"
	`^(unreachable)\s+([0-9a-fA-F\.:/]+)\s+dev\s+([a-z0-9\.]+)\s+(.*)$`,                // linux_ip-0.1.0: r"^(unreachable)\s+([0-9a-fA-F\.:/]+)\s+dev\s+([a-z0-9\.]+)\s+(.*)$"
	`\s*[\n\r]+\s*`, // linux_ip-0.1.0: r"\s*[\n\r]+\s*"
	`^\d+:\s+([a-zA-Z0-9\.-]+)(@\S+)*:\s+(.*)$`, // linux_ip-0.1.0: r"^\d+:\s+([a-zA-Z0-9\.-]+)(@\S+)*:\s+(.*)$"
	`\s*link/ether\s+([a-f0-9:]+)\s+.*`,         // linux_ip-0.1.0: r"\s*link/ether\s+([a-f0-9:]+)\s+.*"
	`\s*inet[6]*\s+([0-9a-f:\./]+)\s+.*`,        // linux_ip-0.1.0: r"\s*inet[6]*\s+([0-9a-f:\./]+)\s+.*"
	`[^\w -]`,                                   // linky-0.1.4: r"[^\w -]"
	`^(.*):(\d+): [^ ]* ([^ ]*)$`,               // linky-0.1.4: r"^(.*):(\d+): [^ ]* ([^ ]*)$"
	`^(\d{4}-\d{2}-\d{2})-(\d{3})-(.+)$`,        // limonite-0.2.1: r"^(\d{4}-\d{2}-\d{2})-(\d{3})-(.+)$"
	`^[a-zA-Z]+$`,                               // process-queue-0.1.1: r"^[a-zA-Z]+$"
	`^\{([a-zA-Z_]+)\}$`,                        // pronghorn-0.1.2: r"^\{([a-zA-Z_]+)\}$"
	"(?m:^(\\d{3}) (.+)\r$)",                    // protocol-ftp-client-0.1.1: "(?m:^(\\d{3}) (.+)\r$)"
	"\"(.+)\"",                                  // protocol-ftp-client-0.1.1: "\"(.+)\""
	"(\\w+) [Tt]ype: (\\w+)",                    // protocol-ftp-client-0.1.1: "(\\w+) [Tt]ype: (\\w+)"
	"(?m:^(\\d{3})-.+\r$)",                      // protocol-ftp-client-0.1.1: "(?m:^(\\d{3})-.+\r$)"
	"Entering Passive Mode \\((\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+)\\)", // protocol-ftp-client-0.1.1: "Entering Passive Mode \\((\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+)\\)"
	"(?m:^(.+)\r$)", // protocol-ftp-client-0.1.1: "(?m:^(.+)\r$)"
	"^([d-])(?:[rwx-]{3}){3} +\\d+ +\\w+ +\\w+ +(\\d+) +(.+) +(.+)$",                                                     // protocol-ftp-client-0.1.1: "^([d-])(?:[rwx-]{3}){3} +\\d+ +\\w+ +\\w+ +(\\d+) +(.+) +(.+)$"
	`([\./\-_]{0,1}(19|20)\d{2})[\./\-_]{0,1}(([0-3]{0,1}[0-9][\./\-_])|(\w{3,5}[\./\-_]))([0-3]{0,1}[0-9][\./\-]{0,1})`, // article-date-extractor-0.1.1: r"([\./\-_]{0,1}(19|20)\d{2})[\./\-_]{0,1}(([0-3]{0,1}[0-9][\./\-_])|(\w{3,5}[\./\-_]))([0-3]{0,1}[0-9][\./\-]{0,1})"
	`(?i)publishdate|pubdate|timestamp|article_date|articledate|date`,                                                    // article-date-extractor-0.1.1: r"(?i)publishdate|pubdate|timestamp|article_date|articledate|date"
	`type\((.*)\)`,                               // arthas_plugin-0.1.1: r"type\((.*)\)"
	`Vec<(.*)>`,                                  // arthas_plugin-0.1.1: r"Vec<(.*)>"
	`Option<(.*)>`,                               // arthas_plugin-0.1.1: r"Option<(.*)>"
	`HashMap<[a-z0-9A-Z]+, *(.*)>`,               // arthas_plugin-0.1.1: r"HashMap<[a-z0-9A-Z]+, *(.*)>"
	"Vec *< *(.*) *>",                            // arthas_derive-0.1.0: "Vec *< *(.*) *>"
	`Option *< *(.*) *>`,                         // arthas_derive-0.1.0: r"Option *< *(.*) *>"
	`HashMap *< *[a-z0-9A-Z]+ *, *(.*) *>`,       // arthas_derive-0.1.0: r"HashMap *< *[a-z0-9A-Z]+ *, *(.*) *>"
	`^([\w\-\(\)\.']+)\s+([^\s].*)\s*$`,          // arpabet-0.2.0: r"^([\w\-\(\)\.']+)\s+([^\s].*)\s*$"
	`^;;;\s+`,                                    // arpabet-0.2.0: r"^;;;\s+"
	`/\*.*?\*/|//.*`,                             // glossy_codegen-0.2.0: r"/\*.*?\*/|//.*"
	"^\\s*#\\s*include\\s+<([:print:]+)>\\s*$",   // glossy_codegen-0.2.0: "^\\s*#\\s*include\\s+<([:print:]+)>\\s*$"
	"^\\s*#\\s*include\\s+\"([:print:]+)\"\\s*$", // glossy_codegen-0.2.0: "^\\s*#\\s*include\\s+\"([:print:]+)\"\\s*$"
	`^\s*#\s*version\s+(\d+)`,                    // glossy_codegen-0.2.0: r"^\s*#\s*version\s+(\d+)"
	`^\s*$`,                                      // glossy_codegen-0.2.0: r"^\s*$"
	`(?P<addr>via \S+)`,                          // gluster-1.0.1: r"(?P<addr>via \S+)"
	`(?P<src>src \S+)`,                           // gluster-1.0.1: r"(?P<src>src \S+)"
	`(.*)\[\d+\]`,                                // gl_helpers-0.1.7: r"(.*)\[\d+\]"
	`(\d+).(\d+)`,                                // gl_helpers-0.1.7: r"(\d+).(\d+)"
	`(?P<c>[\\\.\+\*\?\(\)\|\[\]\{\}\^\$])`,      // glr-parser-0.0.1: r"(?P<c>[\\\.\+\*\?\(\)\|\[\]\{\}\^\$])"
	`^\w+$`,                                      // glr-parser-0.0.1: r"^\w+$"
	"'[^']+'",                                    // glr-parser-0.0.1: "'[^']+'"
	`(?m)//.*`,                                   // hoodlum-0.5.0: r"(?m)//.*"
	`^1\d{10}$`,                                  // form-checker-0.2.2: r"^1\d{10}$"
	`(?i)^[\w.%+-]+@(?:[A-Z0-9-]+\.)+[A-Z]{2,4}$`,                         // form-checker-0.2.2: r"(?i)^[\w.%+-]+@(?:[A-Z0-9-]+\.)+[A-Z]{2,4}$"
	`(?P<user_agent>[a-zA-Z0-9-_]+/[0-9\.]+)`,                             // wikibase-0.2.0: r"(?P<user_agent>[a-zA-Z0-9-_]+/[0-9\.]+)"
	`Cell [0-9]{2,} - Address:`,                                           // wifiscanner-0.3.6: r"Cell [0-9]{2,} - Address:"
	`([0-9a-zA-Z]{1}[0-9a-zA-Z]{1}[:]{1}){5}[0-9a-zA-Z]{1}[0-9a-zA-Z]{1}`, // wifiscanner-0.3.6: r"([0-9a-zA-Z]{1}[0-9a-zA-Z]{1}[:]{1}){5}[0-9a-zA-Z]{1}[0-9a-zA-Z]{1}"
	`Signal level=(\d+)/100`,                                              // wifiscanner-0.3.6: r"Signal level=(\d+)/100"
	`(?s)\[b\](.*?)\[/b\]`,                                                // bbcode-1.0.2: r"(?s)\[b\](.*?)\[/b\]"
	`(?s)\[i\](.*?)\[/i\]`,                                                // bbcode-1.0.2: r"(?s)\[i\](.*?)\[/i\]"
	`(?s)\[u\](.*?)\[/u\]`,                                                // bbcode-1.0.2: r"(?s)\[u\](.*?)\[/u\]"
	`(?s)\[s\](.*?)\[/s\]`,                                                // bbcode-1.0.2: r"(?s)\[s\](.*?)\[/s\]"
	`(?s)\[size=(\d+)](.*?)\[/size\]`,                                     // bbcode-1.0.2: r"(?s)\[size=(\d+)](.*?)\[/size\]"
	`(?s)\[color=(.+)](.*?)\[/color\]`,                                    // bbcode-1.0.2: r"(?s)\[color=(.+)](.*?)\[/color\]"
	`(?s)\[center\](.*?)\[/center\]`,                                      // bbcode-1.0.2: r"(?s)\[center\](.*?)\[/center\]"
	`(?s)\[left\](.*?)\[/left\]`,                                          // bbcode-1.0.2: r"(?s)\[left\](.*?)\[/left\]"
	`(?s)\[right\](.*?)\[/right\]`,                                        // bbcode-1.0.2: r"(?s)\[right\](.*?)\[/right\]"
	`(?s)\[table\](.*?)\[/table\]`,                                        // bbcode-1.0.2: r"(?s)\[table\](.*?)\[/table\]"
	`(?s)\[td\](.*?)\[/td\]`,                                              // bbcode-1.0.2: r"(?s)\[td\](.*?)\[/td\]"
	`(?s)\[tr\](.*?)\[/tr\]`,                                              // bbcode-1.0.2: r"(?s)\[tr\](.*?)\[/tr\]"
	`(?s)\[th\](.*?)\[/th\]`,                                              // bbcode-1.0.2: r"(?s)\[th\](.*?)\[/th\]"
	`(?s)\[url\](.*?)\[/url\]`,                                            // bbcode-1.0.2: r"(?s)\[url\](.*?)\[/url\]"
	`(?s)\[url=(.+)\](.*?)\[/url\]`,                                       // bbcode-1.0.2: r"(?s)\[url=(.+)\](.*?)\[/url\]"
	`(?s)\[quote\](.*?)\[/quote\]`,                                        // bbcode-1.0.2: r"(?s)\[quote\](.*?)\[/quote\]"
	`(?s)\[quote=(.+)\](.*?)\[/quote\]`,                                   // bbcode-1.0.2: r"(?s)\[quote=(.+)\](.*?)\[/quote\]"
	`(?s)\[img=(\d+)x(\d+)(\b.*)?\](.*?)\[/img\]`,                         // bbcode-1.0.2: r"(?s)\[img=(\d+)x(\d+)(\b.*)?\](.*?)\[/img\]"
	`(?s)\[img=(.+)(\b.*)?\](.*?)\[/img\]`,                                // bbcode-1.0.2: r"(?s)\[img=(.+)(\b.*)?\](.*?)\[/img\]"
	`(?s)\[img(\b.*)?\](.*?)\[/img\]`,                                     // bbcode-1.0.2: r"(?s)\[img(\b.*)?\](.*?)\[/img\]"
	`(?s)\[ol\](.*?)\[/ol\]`,                                              // bbcode-1.0.2: r"(?s)\[ol\](.*?)\[/ol\]"
	`(?s)\[ul\](.*?)\[/ul\]`,                                              // bbcode-1.0.2: r"(?s)\[ul\](.*?)\[/ul\]"
	`(?s)\[list\](.*?)\[/list\]`,                                          // bbcode-1.0.2: r"(?s)\[list\](.*?)\[/list\]"
	`(?s)\[youtube\](.*?)\[/youtube\]`,                                    // bbcode-1.0.2: r"(?s)\[youtube\](.*?)\[/youtube\]"
	`(?s)\[youtube=(\d+)x(\d+)\](.*?)\[/youtube\]`,                        // bbcode-1.0.2: r"(?s)\[youtube=(\d+)x(\d+)\](.*?)\[/youtube\]"
	`(?s)\[li\](.*?)\[/li\]`,                                              // bbcode-1.0.2: r"(?s)\[li\](.*?)\[/li\]"
	`loop\d+`,                                                             // block-utils-0.5.0: r"loop\d+"
	`ram\d+`,                                                              // block-utils-0.5.0: r"ram\d+"
	`md\d+`,                                                               // block-utils-0.5.0: r"md\d+"
	`^([1-9]) min$`,                                                       // kvvliveapi-0.1.0: r"^([1-9]) min$"
	`(\d{2}):(\d{2}):(\d{2})`,                                             // rfc822_sanitizer-0.3.3: r"(\d{2}):(\d{2}):(\d{2})"
	`(\d{1,2}):(\d{1,2}):(\d{1,2})`,                                       // rfc822_sanitizer-0.3.3: r"(\d{1,2}):(\d{1,2}):(\d{1,2})"
	`[2-9]`,                                                               // faker-0.0.4: r"[2-9]"
	`[1-9]`,                                                               // faker-0.0.4: r"[1-9]"
	`[0-9]`,                                                               // faker-0.0.4: r"[0-9]"
	`\d{10}`,                                                              // faker-0.0.4: r"\d{10}"
	`\d{1}`,                                                               // faker-0.0.4: r"\d{1}"
	`^\w+`,                                                                // faker-0.0.4: r"^\w+"
	`^\w+`,                                                                // faker-0.0.4: r"^\w+"
	`^(\w+\.? ?){2,3}$`,                                                   // faker-0.0.4: r"^(\w+\.? ?){2,3}$"
	`^[A-Z][a-z]+\.?$`,                                                    // faker-0.0.4: r"^[A-Z][a-z]+\.?$"
	`^[A-Z][A-Za-z]*\.?$`,                                                 // faker-0.0.4: r"^[A-Z][A-Za-z]*\.?$"
	`http://lorempixel.com/100/100/\w+`,                                   // faker-0.0.4: r"http://lorempixel.com/100/100/\w+"
	`http://lorempixel.com/100/100/cats`,                                  // faker-0.0.4: r"http://lorempixel.com/100/100/cats"
	"(?i:ß)",                                                              // fancy-regex-0.1.0: "(?i:ß)"
	"(?i:\\x{0587})",                                                      // fancy-regex-0.1.0: "(?i:\\x{0587})"
	"^\\\\([!-/:-@\\[-`\\{-~aftnrv]|[0-7]{1,3}|x[0-9a-fA-F]{2}|x\\{[0-9a-fA-F]{1,6}\\})", // fancy-regex-0.1.0: "^\\\\([!-/:-@\\[-`\\{-~aftnrv]|[0-7]{1,3}|x[0-9a-fA-F]{2}|x\\{[0-9a-fA-F]{1,6}\\})"
	`/([^/])[^/]+/`,                       // fancy-prompt-0.1.5: r"/([^/])[^/]+/"
	`^([^:]+):.*?(?::([^:]+))?$`,          // fancy-prompt-0.1.5: r"^([^:]+):.*?(?::([^:]+))?$"
	`^(/?__\w+__)/(.*)`,                   // fanta-0.2.0: r"^(/?__\w+__)/(.*)"
	`(.)([A-Z])`,                          // fanta-cli-0.1.1: r"(.)([A-Z])"
	"\\{:[^\\s]+\\}",                      // fanta-cli-0.1.1: "\\{:[^\\s]+\\}"
	"(?P<last>[^\r])\n",                   // amethyst_tools-0.7.1: "(?P<last>[^\r])\n"
	`^-?\d+(\.\d)?`,                       // amigo-0.3.1: r"^-?\d+(\.\d)?"
	`^[a-zA-Z_]+[\w-]*[!?_]?`,             // amigo-0.3.1: r"^[a-zA-Z_]+[\w-]*[!?_]?"
	`^\(`,                                 // amigo-0.3.1: r"^\("
	`^\)`,                                 // amigo-0.3.1: r"^\)"
	`^\s+`,                                // amigo-0.3.1: r"^\s+"
	"\x1b\\[[^m]+m",                       // ethcore-logger-1.12.0: "\x1b\\[[^m]+m"
	`__.*?__`,                             // dash2html-1.0.1: r"__.*?__"
	`(?i)@(?:time|clipboard|cursor|date)`, // dash2html-1.0.1: r"(?i)@(?:time|clipboard|cursor|date)"
	`^Microsoft Windows \[Version\s(\d+\.\d+\.\d+)\]$`, // os_type-2.0.0: r"^Microsoft Windows \[Version\s(\d+\.\d+\.\d+)\]$"
	`ProductName:\s([\w\s]+)\n`,                        // os_type-2.0.0: r"ProductName:\s([\w\s]+)\n"
	`ProductVersion:\s(\w+\.\w+\.\w+)`,                 // os_type-2.0.0: r"ProductVersion:\s(\w+\.\w+\.\w+)"
	`BuildVersion:\s(\w+)`,                             // os_type-2.0.0: r"BuildVersion:\s(\w+)"
	`(\w+) Linux release`,                              // os_type-2.0.0: r"(\w+) Linux release"
	`release\s([\w\.]+)`,                               // os_type-2.0.0: r"release\s([\w\.]+)"
	`Distributor ID:\s(\w+)`,                           // os_type-2.0.0: r"Distributor ID:\s(\w+)"
	`Release:\s([\w\.]+)`,                              // os_type-2.0.0: r"Release:\s([\w\.]+)"
	`typename type\-parameter\-\d+\-\d+::.+`,           // bindgen-0.37.0: r"typename type\-parameter\-\d+\-\d+::.+"
	"^+(.*)\r\n",                                       // imap-0.8.1: "^+(.*)\r\n"
	`^ffd8ffe0`,                                        // image-base64-0.1.0: r"^ffd8ffe0"
	`^89504e47`,                                        // image-base64-0.1.0: r"^89504e47"
	`^47494638`,                                        // image-base64-0.1.0: r"^47494638"
	"^(/([^/~]|~[01])*)*$",                             // json-pointer-0.3.2: "^(/([^/~]|~[01])*)*$"
	"^#(/([^/~%]|~[01]|%[0-9a-fA-F]{2})*)*$",           // json-pointer-0.3.2: "^#(/([^/~%]|~[01]|%[0-9a-fA-F]{2})*)*$"
	`^5.5.5-(\d{1,2})\.(\d{1,2})\.(\d{1,3})-MariaDB`,   // mysql_common-0.7.0: r"^5.5.5-(\d{1,2})\.(\d{1,2})\.(\d{1,3})-MariaDB"
	`^(\d{1,2})\.(\d{1,2})\.(\d{1,3})(.*)`,             // mysql_common-0.7.0: r"^(\d{1,2})\.(\d{1,2})\.(\d{1,3})(.*)"
	`^[0-9]{4}[0-9A-Z]{2}[0-9]{3}$`,                    // government_id-0.1.0: r"^[0-9]{4}[0-9A-Z]{2}[0-9]{3}$"
	`UniqueIndexViolation: (\w+)`,                      // ohmers-0.1.1: r"UniqueIndexViolation: (\w+)"
	`(.*) you are (.*)`,                                // eliza-1.0.0: r"(.*) you are (.*)"
	`(.*) you are (.*)`,                                // eliza-1.0.0: r"(.*) you are (.*)"
	`(.*) you are (.*)`,                                // eliza-1.0.0: r"(.*) you are (.*)"
	"^\\s*\\*",                                         // chema-0.0.5: "^\\s*\\*"
	"^\\s*@(\\w+)\\s+(.*)",                             // chema-0.0.5: "^\\s*@(\\w+)\\s+(.*)"
	`^\s*#`,                                            // chord3-0.3.0: r"^\s*#"
	`\{(?P<cmd>\w+)(?::?\s*(?P<arg>.*))?\}`,            // chord3-0.3.0: r"\{(?P<cmd>\w+)(?::?\s*(?P<arg>.*))?\}"
	`\{(eot|end_of_tab):?\s*`,                          // chord3-0.3.0: r"\{(eot|end_of_tab):?\s*"
	`([^\[]*)(?:\[([^\]]*)\])?`,                        // chord3-0.3.0: r"([^\[]*)(?:\[([^\]]*)\])?"
	"^[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$", // checkmail-0.1.1: "^[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$"
	`\b\w\w+\b`,     // cntk-0.2.1: r"\b\w\w+\b"
	`\b\w\w+\b`,     // cntk-0.2.1: r"\b\w\w+\b"
	`\(id: (\d+)\)`, // cniguru-0.1.0: r"\(id: (\d+)\)"
	`^(\d+)\.(\d+)\.(\d+)(?:-([\dA-Za-z-]+(?:\.[\dA-Za-z-]+)*))?(?:\+([\dA-Za-z-]+(?:\.[\dA-Za-z-]+)*))?$`, // upm_lib-0.3.0: r"^(\d+)\.(\d+)\.(\d+)(?:-([\dA-Za-z-]+(?:\.[\dA-Za-z-]+)*))?(?:\+([\dA-Za-z-]+(?:\.[\dA-Za-z-]+)*))?$"
	`^\s*(\*+(\s+))?`,                // avro-0.2.1: r"^\s*(\*+(\s+))?"
	`^\s*(\*+)?`,                     // avro-0.2.1: r"^\s*(\*+)?"
	"[0-9]+",                         // nomi-0.0.2: "[0-9]+"
	"([0-9]+)@(?:nodes|n)?:([^@]+)?", // nodes-0.1.0: "([0-9]+)@(?:nodes|n)?:([^@]+)?"
	`(?i)in (\d+) (second|minute|hour|day|week)s?`, // not-stakkr-1.0.0: r"(?i)in (\d+) (second|minute|hour|day|week)s?"
	"^([A-Za-z0-9 -_:]+)\n-+\n",                    // notetxt-0.0.1: "^([A-Za-z0-9 -_:]+)\n-+\n"
	`^-?[0-9]+(\.[0-9]+)?([eE]-?[0-9]+)?$`,         // nail-0.1.0-pre.0: r"^-?[0-9]+(\.[0-9]+)?([eE]-?[0-9]+)?$"
	`^-?[0-9]+$`,                                   // nail-0.1.0-pre.0: r"^-?[0-9]+$"
	`[^\w\s\pP]+`,                                  // askalono-0.2.0: r"[^\w\s\pP]+"
	`[ \t\p{Zs} \\ / \| \x2044 ]+`,                 // askalono-0.2.0: r"(?x)[ \t\p{Zs} \\ / \| \x2044 ]+"
	`\p{Pd}+`,                                      // askalono-0.2.0: r"\p{Pd}+"
	`\p{Ps}+`,                                      // askalono-0.2.0: r"\p{Ps}+"
	`\p{Pe}+`,                                      // askalono-0.2.0: r"\p{Pe}+"
	`\p{Pc}+`,                                      // askalono-0.2.0: r"\p{Pc}+"
	`[©Ⓒⓒ]`,                                        // askalono-0.2.0: r"[©Ⓒⓒ]"
	`[\r\n\v\f]`,                                   // askalono-0.2.0: r"[\r\n\v\f]"
	`\n{3,}`,                                       // askalono-0.2.0: r"\n{3,}"
	`[^\w\s]+`,                                     // askalono-0.2.0: r"[^\w\s]+"
	`\s+`,                                          // askalono-0.2.0: r"\s+"
	`[^0-9a-zA-Z_]`,                                // assembunny_plus-0.0.3: r"[^0-9a-zA-Z_]"
	`[0-9]`,                                        // assembunny_plus-0.0.3: r"[0-9]"
	`(?m)^Minion (\S*) did not respond\. No job will be sent\.$`, // salt-compressor-0.4.0: r"(?m)^Minion (\S*) did not respond\. No job will be sent\.$"
	`</?[^>]+?>`,                       // sabisabi-0.4.1: r"</?[^>]+?>"
	`\([^)]*\)`,                        // sabisabi-0.4.1: r"\([^)]*\)"
	"@import \"([^\"]*)\";",            // sassers-0.13.5-h28: "@import \"([^\"]*)\";"
	`[A-Za-z\d-]{1,63}$`,               // shadowsocks-0.6.2: r"[A-Za-z\d-]{1,63}$"
	"[abc]+",                           // shkeleton-0.1.5: "[abc]+"
	`([^A-Za-z0-9_\-.,:/@\n])`,         // shellwords-0.1.0: r"([^A-Za-z0-9_\-.,:/@\n])"
	`\n`,                               // shellwords-0.1.0: r"\n"
	"(?P<num>[0-9]+)(?P<units>[dhms])", // shush-0.1.5: "(?P<num>[0-9]+)(?P<units>[dhms])"
	`(?:Chrome|CrMo|CriOS)/([.0-9]+)`,  // woothee-0.8.0: r"(?:Chrome|CrMo|CriOS)/([.0-9]+)"
	`Vivaldi/([.0-9]+)`,                // woothee-0.8.0: r"Vivaldi/([.0-9]+)"
	`Firefox/([.0-9]+)`,                // woothee-0.8.0: r"Firefox/([.0-9]+)"
	`^Mozilla/[.0-9]+ \((?:Mobile|Tablet);(?:.*;)? rv:([.0-9]+)\) Gecko/[.0-9]+ Firefox/[.0-9]+$`, // woothee-0.8.0: r"^Mozilla/[.0-9]+ \((?:Mobile|Tablet);(?:.*;)? rv:([.0-9]+)\) Gecko/[.0-9]+ Firefox/[.0-9]+$"
	`FxiOS/([.0-9]+)`,                                                 // woothee-0.8.0: r"FxiOS/([.0-9]+)"
	`\(([^;)]+);FOMA;`,                                                // woothee-0.8.0: r"\(([^;)]+);FOMA;"
	`jig browser[^;]+; ([^);]+)`,                                      // woothee-0.8.0: r"jig browser[^;]+; ([^);]+)"
	`(?i)rss(?:reader|bar|[-_ /;()]|[ +]*/)`,                          // woothee-0.8.0: r"(?i)rss(?:reader|bar|[-_ /;()]|[ +]*/)"
	`(?i)(?:bot|crawler|spider)(?:[-_ ./;@()]|$)`,                     // woothee-0.8.0: r"(?i)(?:bot|crawler|spider)(?:[-_ ./;@()]|$)"
	`(?i)(?:feed|web) ?parser`,                                        // woothee-0.8.0: r"(?i)(?:feed|web) ?parser"
	`(?i)watch ?dog`,                                                  // woothee-0.8.0: r"(?i)watch ?dog"
	`Edge/([.0-9]+)`,                                                  // woothee-0.8.0: r"Edge/([.0-9]+)"
	`MSIE ([.0-9]+);`,                                                 // woothee-0.8.0: r"MSIE ([.0-9]+);"
	`Version/([.0-9]+)`,                                               // woothee-0.8.0: r"Version/([.0-9]+)"
	`Opera[/ ]([.0-9]+)`,                                              // woothee-0.8.0: r"Opera[/ ]([.0-9]+)"
	`OPR/([.0-9]+)`,                                                   // woothee-0.8.0: r"OPR/([.0-9]+)"
	`Version/([.0-9]+)`,                                               // woothee-0.8.0: r"Version/([.0-9]+)"
	`(?:SoftBank|Vodafone|J-PHONE)/[.0-9]+/([^ /;()]+)`,               // woothee-0.8.0: r"(?:SoftBank|Vodafone|J-PHONE)/[.0-9]+/([^ /;()]+)"
	`Trident/([.0-9]+);`,                                              // woothee-0.8.0: r"Trident/([.0-9]+);"
	` rv:([.0-9]+)`,                                                   // woothee-0.8.0: r" rv:([.0-9]+)"
	`IEMobile/([.0-9]+);`,                                             // woothee-0.8.0: r"IEMobile/([.0-9]+);"
	`(?:WILLCOM|DDIPOCKET);[^/]+/([^ /;()]+)`,                         // woothee-0.8.0: r"(?:WILLCOM|DDIPOCKET);[^/]+/([^ /;()]+)"
	`Windows ([ .a-zA-Z0-9]+)[;\\)]`,                                  // woothee-0.8.0: r"Windows ([ .a-zA-Z0-9]+)[;\\)]"
	`^Phone(?: OS)? ([.0-9]+)`,                                        // woothee-0.8.0: r"^Phone(?: OS)? ([.0-9]+)"
	`iP(hone;|ad;|od) .*like Mac OS X`,                                // woothee-0.8.0: r"iP(hone;|ad;|od) .*like Mac OS X"
	`Version/([.0-9]+)`,                                               // woothee-0.8.0: r"Version/([.0-9]+)"
	`rv:(\d+\.\d+\.\d+)`,                                              // woothee-0.8.0: r"rv:(\d+\.\d+\.\d+)"
	`FreeBSD ([^;\)]+);`,                                              // woothee-0.8.0: r"FreeBSD ([^;\)]+);"
	`CrOS ([^\)]+)\)`,                                                 // woothee-0.8.0: r"CrOS ([^\)]+)\)"
	`Android[- ](\d+\.\d+(?:\.\d+)?)`,                                 // woothee-0.8.0: r"Android[- ](\d+\.\d+(?:\.\d+)?)"
	`PSP \(PlayStation Portable\); ([.0-9]+)\)`,                       // woothee-0.8.0: r"PSP \(PlayStation Portable\); ([.0-9]+)\)"
	`PLAYSTATION 3;? ([.0-9]+)\)`,                                     // woothee-0.8.0: r"PLAYSTATION 3;? ([.0-9]+)\)"
	`PlayStation Vita ([.0-9]+)\)`,                                    // woothee-0.8.0: r"PlayStation Vita ([.0-9]+)\)"
	`PlayStation 4 ([.0-9]+)\)`,                                       // woothee-0.8.0: r"PlayStation 4 ([.0-9]+)\)"
	`BB10(?:.+)Version/([.0-9]+) `,                                    // woothee-0.8.0: r"BB10(?:.+)Version/([.0-9]+) "
	`BlackBerry(?:\d+)/([.0-9]+) `,                                    // woothee-0.8.0: r"BlackBerry(?:\d+)/([.0-9]+) "
	`; CPU(?: iPhone)? OS (\d+_\d+(?:_\d+)?) like Mac OS X`,           // woothee-0.8.0: r"; CPU(?: iPhone)? OS (\d+_\d+(?:_\d+)?) like Mac OS X"
	`Mac OS X (10[._]\d+(?:[._]\d+)?)(?:\)|;)`,                        // woothee-0.8.0: r"Mac OS X (10[._]\d+(?:[._]\d+)?)(?:\)|;)"
	`^(?:Apache-HttpClient/|Jakarta Commons-HttpClient/|Java/)`,       // woothee-0.8.0: r"^(?:Apache-HttpClient/|Jakarta Commons-HttpClient/|Java/)"
	`[- ]HttpClient(/|$)`,                                             // woothee-0.8.0: r"[- ]HttpClient(/|$)"
	`^(?:PHP|WordPress|CakePHP|PukiWiki|PECL::HTTP)(?:/| |$)`,         // woothee-0.8.0: r"^(?:PHP|WordPress|CakePHP|PukiWiki|PECL::HTTP)(?:/| |$)"
	`(?:PEAR HTTP_Request|HTTP_Request)(?: class|2)`,                  // woothee-0.8.0: r"(?:PEAR HTTP_Request|HTTP_Request)(?: class|2)"
	`(?:Rome Client |UnwindFetchor/|ia_archiver |Summify |PostRank/)`, // woothee-0.8.0: r"(?:Rome Client |UnwindFetchor/|ia_archiver |Summify |PostRank/)"
	`Sleipnir/([.0-9]+)`,                                              // woothee-0.8.0: r"Sleipnir/([.0-9]+)"
	`@@[a-z|A-Z|\d]+@@`,                                               // word_replace-0.0.3: r"@@[a-z|A-Z|\d]+@@"
	`\w+`,                                                             // wordcount-0.1.0: r"\w+"
	"^([^=]+)=(.*)$",                                                  // just-0.3.12: "^([^=]+)=(.*)$"
	`:[a-zA-Z_]+?:`,                                                   // emote-0.1.0: r":[a-zA-Z_]+?:"
	`:([a-zA-Z0-9_+-]+):`,                                             // emojicons-1.0.1: r":([a-zA-Z0-9_+-]+):"
	`git-codecommit\.([a-z0-9-]+)\.amazonaws\.com`,                    // git2_codecommit-0.1.2: r"git-codecommit\.([a-z0-9-]+)\.amazonaws\.com"
	`^submodule\.(?P<name>.*)\.(?P<key>[^=]*)=(?P<value>.*)$`,         // git-workarea-3.1.2: r"^submodule\.(?P<name>.*)\.(?P<key>[^=]*)=(?P<value>.*)$"
	`^(?P<command>git-(?:receive|upload)-pack) '(?P<path>.+)'$`,       // git-shell-enforce-directory-1.0.0: r"^(?P<command>git-(?:receive|upload)-pack) '(?P<path>.+)'$"
	`[ \n]:(.*?):`,                                                    // git-journal-1.6.3: r"[ \n]:(.*?):"
	`^git@(?P<host>[[:alnum:]\._-]+):(?P<path>[[:alnum:]\._\-/]+).git$`, // git-find-0.3.2: r"^git@(?P<host>[[:alnum:]\._-]+):(?P<path>[[:alnum:]\._\-/]+).git$"
	`private_token=\w{20}`,                           // gitlab-api-0.6.0: r"private_token=\w{20}"
	"^(http://|https://)",                            // td-client-0.7.0: "^(http://|https://)"
	`--(?P<type>[a-zA-Z]+)-- (?P<contents>.*)`,       // karaconv-0.3.0: r"--(?P<type>[a-zA-Z]+)-- (?P<contents>.*)"
	`(?P<comp>et al\.)(?:\.)`,                        // katana-1.0.2: r"(?P<comp>et al\.)(?:\.)"
	`\.{3}`,                                          // katana-1.0.2: r"\.{3}"
	`(?P<number>[0-9]+)\.(?P<decimal>[0-9]+)`,        // katana-1.0.2: r"(?P<number>[0-9]+)\.(?P<decimal>[0-9]+)"
	`\s\.(?P<nums>[0-9]+)`,                           // katana-1.0.2: r"\s\.(?P<nums>[0-9]+)"
	`(?:[A-Za-z]\.){2,}`,                             // katana-1.0.2: r"(?:[A-Za-z]\.){2,}"
	`(?P<init>[A-Z])(?P<point>\.)`,                   // katana-1.0.2: r"(?P<init>[A-Z])(?P<point>\.)"
	`(?P<title>[A-Z][a-z]{1,3})(\.)`,                 // katana-1.0.2: r"(?P<title>[A-Z][a-z]{1,3})(\.)"
	`&==&(?P<p>[.!?])`,                               // katana-1.0.2: r"&==&(?P<p>[.!?])"
	`&\^&(?P<p>[.!?])`,                               // katana-1.0.2: r"&\^&(?P<p>[.!?])"
	`&\*\*&(?P<p>[.!?])`,                             // katana-1.0.2: r"&\*\*&(?P<p>[.!?])"
	`&=&(?P<p>[.!?])`,                                // katana-1.0.2: r"&=&(?P<p>[.!?])"
	`&##&(?P<p>[.!?])`,                               // katana-1.0.2: r"&##&(?P<p>[.!?])"
	`&\$&(?P<p>[.!?])`,                               // katana-1.0.2: r"&\$&(?P<p>[.!?])"
	`@(?:_|\d+(?:/\d+(?:-\d+)?)?)`,                   // kailua_syntax-1.1.0: r"@(?:_|\d+(?:/\d+(?:-\d+)?)?)"
	`<(\d+)>`,                                        // kailua_syntax-1.1.0: r"<(\d+)>"
	`\((\d+),(\d+),(\d+),(\d+),(\d+),(\d+)\)`,        // ftp-3.0.1: r"\((\d+),(\d+),(\d+),(\d+),(\d+),(\d+)\)"
	`\b(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})\b`, // ftp-3.0.1: r"\b(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})\b"
	`\s+(\d+)\s*$`,                                   // ftp-3.0.1: r"\s+(\d+)\s*$"
	`<countryCode>(.*?)</countryCode>`,               // vat-0.1.0: r"<countryCode>(.*?)</countryCode>"
	`<vatNumber>(.*?)</vatNumber>`,                   // vat-0.1.0: r"<vatNumber>(.*?)</vatNumber>"
	`<name>(.*?)</name>`,                             // vat-0.1.0: r"<name>(.*?)</name>"
	`<address>(?s)(.*?)(?-s)</address>`,              // vat-0.1.0: r"<address>(?s)(.*?)(?-s)</address>"
	`<valid>(true|false)</valid>`,                    // vat-0.1.0: r"<valid>(true|false)</valid>"
	`^ATU\d{8}$`,                                     // vat-0.1.0: r"^ATU\d{8}$"
	`^BE0?\d{9, 10}$`,                                // vat-0.1.0: r"^BE0?\d{9, 10}$"
	`^BG\d{9,10}$`,                                   // vat-0.1.0: r"^BG\d{9,10}$"
	`^HR\d{11}$`,                                     // vat-0.1.0: r"^HR\d{11}$"
	`^CY\d{8}[A-Z]$`,                                 // vat-0.1.0: r"^CY\d{8}[A-Z]$"
	`^CZ\d{8,10}$`,                                   // vat-0.1.0: r"^CZ\d{8,10}$"
	`^DK\d{8}$`,                                      // vat-0.1.0: r"^DK\d{8}$"
	`^EE\d{9}$`,                                      // vat-0.1.0: r"^EE\d{9}$"
	`^FI\d{8}$`,                                      // vat-0.1.0: r"^FI\d{8}$"
	`^FR[A-HJ-NP-Z0-9][A-HJ-NP-Z0-9]\d{9}$`,          // vat-0.1.0: r"^FR[A-HJ-NP-Z0-9][A-HJ-NP-Z0-9]\d{9}$"
	`^DE\d{9}$`,                                      // vat-0.1.0: r"^DE\d{9}$"
	`^EL\d{9}$`,                                      // vat-0.1.0: r"^EL\d{9}$"
	`^HU\d{8}$`,                                      // vat-0.1.0: r"^HU\d{8}$"
	`^IE\d[A-Z0-9\+\*]\d{5}[A-Z]{1,2}$`,              // vat-0.1.0: r"^IE\d[A-Z0-9\+\*]\d{5}[A-Z]{1,2}$"
	`^IT\d{11}$`,                                     // vat-0.1.0: r"^IT\d{11}$"
	`^LV\d{11}$`,                                     // vat-0.1.0: r"^LV\d{11}$"
	`^LT(\d{9}|\d{12})$`,                             // vat-0.1.0: r"^LT(\d{9}|\d{12})$"
	`^LU\d{8}$`,                                      // vat-0.1.0: r"^LU\d{8}$"
	`^MT\d{8}$`,                                      // vat-0.1.0: r"^MT\d{8}$"
	`^NL\d{9}B\d{2}$`,                                // vat-0.1.0: r"^NL\d{9}B\d{2}$"
	`^PL\d{10}$`,                                     // vat-0.1.0: r"^PL\d{10}$"
	`^PT\d{9}$`,                                      // vat-0.1.0: r"^PT\d{9}$"
	`^RO\d{2,10}$`,                                   // vat-0.1.0: r"^RO\d{2,10}$"
	`^SK\d{10}$`,                                     // vat-0.1.0: r"^SK\d{10}$"
	`^SI\d{8}$`,                                      // vat-0.1.0: r"^SI\d{8}$"
	`^ES[A-Z0-9]\d{7}[A-Z0-9]$`,                      // vat-0.1.0: r"^ES[A-Z0-9]\d{7}[A-Z0-9]$"
	`^SE\d{10}01$`,                                   // vat-0.1.0: r"^SE\d{10}01$"
	`^(GB(GD|HA)\d{3}|GB\d{9}|GB\d{12})$`,            // vat-0.1.0: r"^(GB(GD|HA)\d{3}|GB\d{9}|GB\d{12})$"
	`\{\{(.*)\}\}`,                                   // eve-0.1.1: r"\{\{(.*)\}\}"
	"^mio",                                           // egc-0.1.2: "^mio"
	"",                                               // pew-0.2.3: ""
	"",                                               // pew-0.2.3: ""
	"y",                                              // mob-0.4.3: "y"
	"@([a-z]+)",                                      // lit-0.2.8: "@([a-z]+)"
	"([A-Z-]+):(.*)",                                 // lit-0.2.8: "([A-Z-]+):(.*)"
	"^[a-zA-Z_][a-zA-Z0-9_]*$",                       // lit-0.2.8: "^[a-zA-Z_][a-zA-Z0-9_]*$"
	`\d+\.\d+\.\d+`,                                  // avm-1.0.1: r"\d+\.\d+\.\d+"
	`\d+\.\d+\.\d+`,                                  // avm-1.0.1: r"\d+\.\d+\.\d+"
	`^Vec<(.+)>$`,                                    // orm-0.2.0: r"^Vec<(.+)>$"
	`\\(\r\n|\n\r|\n|\r)`,                            // sgf-0.1.5: r"\\(\r\n|\n\r|\n|\r)"
	`\\(.)`,                                          // sgf-0.1.5: r"\\(.)"
	`\r\n|\n\r|\n|\r`,                                // sgf-0.1.5: r"\r\n|\n\r|\n|\r"
	`([\]\\:])`,                                      // sgf-0.1.5: r"([\]\\:])"
	"^Bearer realm=\"(.+?)\",service=\"(.+?)\",scope=\"(.+?)\"$", // dok-0.2.0: "^Bearer realm=\"(.+?)\",service=\"(.+?)\",scope=\"(.+?)\"$"
	`([+-]?\s*\d+[dD]\d+|[+-]?\s*\d+)`,                           // d20-0.1.0: r"([+-]?\s*\d+[dD]\d+|[+-]?\s*\d+)"
	"E",                                                          // dvb-0.3.0: "E"
	"^F",                                                         // dvb-0.3.0: "^F"
	"^S",                                                         // dvb-0.3.0: "^S"
	`Change-Id: (I[a-f0-9]{40})$`,                                // ger-0.2.0: r"Change-Id: (I[a-f0-9]{40})$"
	`(refs|ref|fix|fixes|close|closes)\s+([A-Z]{2,5}-[0-9]{1,5})$`, // ger-0.2.0: r"(refs|ref|fix|fixes|close|closes)\s+([A-Z]{2,5}-[0-9]{1,5})$"
	`(\d+)(\.(\d+))?(\.(\d+))?(.*)`,                                // n5-0.2.1: r"(\d+)(\.(\d+))?(\.(\d+))?(.*)"
	`[A-Za-z0-9]`,                                                  // po-0.1.4: r"[A-Za-z0-9]"
	"path is (‘|')?([^’'\n]*)(’|')?",                               // carnix-0.8.5: "path is (‘|')?([^’'\n]*)(’|')?"
	`^(\S*) (\d*)\.(\d*)\.(\d*)(-(\S*))?(.*)?`,                     // carnix-0.8.5: r"^(\S*) (\d*)\.(\d*)\.(\d*)(-(\S*))?(.*)?"
	`(\d*)\.(\d*)\.(\d*)(-(\S*))?`,                                 // carnix-0.8.5: r"(\d*)\.(\d*)\.(\d*)(-(\S*))?"
	`(\S*)-(\d*)\.(\d*)\.(\d*)(-(\S*))?`,                           // carnix-0.8.5: r"(\S*)-(\d*)\.(\d*)\.(\d*)(-(\S*))?"
	`^# CaseFolding-(\d+)\.(\d+)\.(\d+).txt$`,                      // caseless-0.2.1: r"^# CaseFolding-(\d+)\.(\d+)\.(\d+).txt$"
	`^([0-9A-F]+); [CF]; ([0-9A-F ]+);`,                            // caseless-0.2.1: r"^([0-9A-F]+); [CF]; ([0-9A-F ]+);"
	"\r?\n\r?\n",                                                   // cabot-0.2.0: "\r?\n\r?\n"
	"\r?\n",                                                        // cabot-0.2.0: "\r?\n"
	`^600`,                                                         // card-validate-2.2.1: r"^600"
	`^5019`,                                                        // card-validate-2.2.1: r"^5019"
	`^4`,                                                           // card-validate-2.2.1: r"^4"
	`^(5[1-5]|2[2-7])`,                                             // card-validate-2.2.1: r"^(5[1-5]|2[2-7])"
	`^3[47]`,                                                       // card-validate-2.2.1: r"^3[47]"
	`^3[0689]`,                                                     // card-validate-2.2.1: r"^3[0689]"
	`^6([045]|22)`,                                                 // card-validate-2.2.1: r"^6([045]|22)"
	`^(62|88)`,                                                     // card-validate-2.2.1: r"^(62|88)"
	`^35`,                                                          // card-validate-2.2.1: r"^35"
	`^[0-9]+$`,                                                     // card-validate-2.2.1: r"^[0-9]+$"
	`\d{1,} passed.*filtered out`,                                  // cargo-testify-0.3.0: r"\d{1,} passed.*filtered out"
	`error(:|\[).*`,                                                // cargo-testify-0.3.0: r"error(:|\[).*"
	`<(.*?)>`,                                                      // cargo-wix-0.0.5: r"<(.*?)>"
	`<(.*?)>`,                                                      // cargo-wix-0.0.5: r"<(.*?)>"
	`<(.*?)>`,                                                      // cargo-wix-0.0.5: r"<(.*?)>"
	`<(.*?)>`,                                                      // cargo-wix-0.0.5: r"<(.*?)>"
	`(?m)^incremental: re-using (\d+) out of (\d+) modules$`,      // cargo-incremental-0.1.23: r"(?m)^incremental: re-using (\d+) out of (\d+) modules$"
	"(?m)(warning|error): (.*)\n  --> ([^:]:\\d+:\\d+)$",          // cargo-incremental-0.1.23: "(?m)(warning|error): (.*)\n  --> ([^:]:\\d+:\\d+)$"
	`(?m)^test (.*) \.\.\. (\w+)`,                                 // cargo-incremental-0.1.23: r"(?m)^test (.*) \.\.\. (\w+)"
	`(?m)(\d+) passed; (\d+) failed; (\d+) ignored; \d+ measured`, // cargo-incremental-0.1.23: r"(?m)(\d+) passed; (\d+) failed; (\d+) ignored; \d+ measured"
	`^[^-]+-[0-9a-f]+\.js$`,                                       // cargo-testjs-0.1.2: r"^[^-]+-[0-9a-f]+\.js$"
	`\s*//`,                                                       // cargo-tarpaulin-0.6.2: r"\s*//"
	`/\*`,                                                         // cargo-tarpaulin-0.6.2: r"/\*"
	`\*/`,                                                         // cargo-tarpaulin-0.6.2: r"\*/"
	`^fo`,                                                         // cargo-culture-kit-0.1.0: r"^fo"
	"\\s+",                                                        // cargo-screeps-0.1.3: "\\s+"
	"`(\\S+) v([0-9.]+)",                                          // cargo-brew-0.1.4: r"`(\S+) v([0-9.]+)"
	"^\\[.+\\]",                                                   // cargo-release-0.10.2: "^\\[.+\\]"
	"^\\[\\[.+\\]\\]",                                             // cargo-release-0.10.2: "^\\[\\[.+\\]\\]"
	`^https://github.com/([-_0-9a-zA-Z]+)/([-_0-9a-zA-Z]+)(/|.git)?$`, // cargo-edit-0.3.0-beta.1: r"^https://github.com/([-_0-9a-zA-Z]+)/([-_0-9a-zA-Z]+)(/|.git)?$"
	`^https://gitlab.com/([-_0-9a-zA-Z]+)/([-_0-9a-zA-Z]+)(/|.git)?$`, // cargo-edit-0.3.0-beta.1: r"^https://gitlab.com/([-_0-9a-zA-Z]+)/([-_0-9a-zA-Z]+)(/|.git)?$"
	".*",                            // cargo-disassemble-0.1.1: ".*"
	`(?m)(?P<symbol>_ZN[0-9]+.*E)`,  // cargo-demangle-0.1.2: r"(?m)(?P<symbol>_ZN[0-9]+.*E)"
	`^\s*\}(?:\)*;?|\s*else\s*\{)$`, // cargo-coverage-annotations-0.1.5: r"^\s*\}(?:\)*;?|\s*else\s*\{)$"
	"[\u001b\u009b][\\[()#;?]*(?:[0-9]{1,4}(?:;[0-9]{0,4})*)?[0-9A-PRZcf-nqry=><]", // cargo-urlcrate-1.0.1: "[\u{001b}\u{009b}][\\[()#;?]*(?:[0-9]{1,4}(?:;[0-9]{0,4})*)?[0-9A-PRZcf-nqry=><]"
	`^\s*\*( |$)`,                   // cargo-script-0.2.8: r"^\s*\*( |$)"
	`^(\s+)`,                        // cargo-script-0.2.8: r"^(\s+)"
	`/\*|\*/`,                       // cargo-script-0.2.8: r"/\*|\*/"
	`^\s*//!`,                       // cargo-script-0.2.8: r"^\s*//!"
	`^#![^\[].*?(\r\n|\n)`,          // cargo-script-0.2.8: r"^#![^\[].*?(\r\n|\n)"
	`cargo-install-update\.exe-v.+`, // cargo-update-1.5.2: r"cargo-install-update\.exe-v.+"
	`^<(?:(int|uint|str|float|path):)?([\w_][a-zA-Z0-9_]*)>$`, // canteen-0.4.1: r"^<(?:(int|uint|str|float|path):)?([\w_][a-zA-Z0-9_]*)>$"
	`(.)([A-Z])`, // thruster-cli-0.1.3: r"(.)([A-Z])"
	"([Z]+)$",    // thieves-cant-0.1.0: "([Z]+)$"
	`^@\S+/\S+`,  // codeowners-0.1.3: r"^@\S+/\S+"
	`^@\S+`,      // codeowners-0.1.3: r"^@\S+"
	`^\S+@\S+`,   // codeowners-0.1.3: r"^\S+@\S+"
	`^b0000 {21} complete   20[-0-9T:+]+\s +\d+s\n$`, // conserve-0.4.2: r"^b0000 {21} complete   20[-0-9T:+]+\s +\d+s\n$"
	`(?P<greeting>\S+?) (?P<name>\S+?)$`,             // commodore-0.3.0: r"(?P<greeting>\S+?) (?P<name>\S+?)$"
	"([ \\t]*)```haskell([\\s\\S]*?)```",             // corollary-0.3.0: r"([ \t]*)```haskell([\s\S]*?)```"
	`\b((?:a|b|t)\d*)\b`,                             // corollary-0.3.0: r"\b((?:a|b|t)\d*)\b"
	"NB",                                             // colorizex-0.1.3: "NB"
	`(?i)\[[a-z0-9_-]+\]`,                            // colorstring-0.0.1: r"(?i)\[[a-z0-9_-]+\]"
	`^(?i)(\[[a-z0-9_-]+\])+`,                        // colorstring-0.0.1: r"^(?i)(\[[a-z0-9_-]+\])+"
	"name:(.+)",                                      // cosmogony-0.3.0: "name:(.+)"
	`(?m:^ {0,3}\[[^\]]+\]:.+$)`,                     // cobalt-bin-0.12.1: r"(?m:^ {0,3}\[[^\]]+\]:.+$)"
	`[^\p{L}\p{M}\p{N}\p{Pc} -]`,                     // comrak-0.2.12: r"[^\p{L}\p{M}\p{N}\p{Pc} -]"
	"",                                               // content-blocker-0.2.3: ""
	"(?i)hi",                                         // content-blocker-0.2.3: "(?i)hi"
	"http[s]?://domain.org",                          // content-blocker-0.2.3: "http[s]?://domain.org"
	"(?i)http[s]?://domain.org",                      // content-blocker-0.2.3: "(?i)http[s]?://domain.org"
	"http://domain.org",                              // content-blocker-0.2.3: "http://domain.org"
	"http://domain.org",                              // content-blocker-0.2.3: "http://domain.org"
	"ad.html",                                        // content-blocker-0.2.3: "ad.html"
	"ad.html",                                        // content-blocker-0.2.3: "ad.html"
	"http://domain.org",                              // content-blocker-0.2.3: "http://domain.org"
	"http://domain.org/nocookies.sjs",                // content-blocker-0.2.3: "http://domain.org/nocookies.sjs"
	"http://domain.org/nocookies.sjs",                // content-blocker-0.2.3: "http://domain.org/nocookies.sjs"
	"http://domain.org/hideme.jpg",                   // content-blocker-0.2.3: "http://domain.org/hideme.jpg"
	"http://domain.org/ok.html",                      // content-blocker-0.2.3: "http://domain.org/ok.html"
	"http://domain.org/ok.html\\?except_this=1",      // content-blocker-0.2.3: "http://domain.org/ok.html\\?except_this=1"
	"[A-Za-z0-9=]",                                   // victoria-dom-0.1.2: "[A-Za-z0-9=]"
	`^nsq://`,                                        // numbat-1.0.0: r"^nsq://"
	`[\s\t\r\n]`,                                     // airkorea-0.1.2: r"[\s\t\r\n]"
	`([\{\[,])|([\}\]])`,                             // airkorea-0.1.2: r"([\{\[,])|([\}\]])"
	`[^.\d]+$`,                                       // airkorea-0.1.2: r"[^.\d]+$"
	// rofl-0.0.1: r"\b"
	`--------- beginning of.*`,            // rogcat-0.2.15: r"--------- beginning of.*"
	`a|e|i|o|u`,                           // rogcat-0.2.15: r"a|e|i|o|u"
	`^(\d+)([kMG])$`,                      // rogcat-0.2.15: r"^(\d+)([kMG])$"
	"\\.([A-Za-z0-9]{2,4})$",              // media_filename-0.1.4: "\\.([A-Za-z0-9]{2,4})$"
	"([0-9]{3,4}p|[0-9]{3,4}x[0-9]{3,4})", // media_filename-0.1.4: "([0-9]{3,4}p|[0-9]{3,4}x[0-9]{3,4})"
	"(?:^\\[([^]]+)\\]|- ?([^-]+)$)",      // media_filename-0.1.4: "(?:^\\[([^]]+)\\]|- ?([^-]+)$)"
	"(?:[eE]([0-9]{2,3})|[^0-9A-Za-z]([0-9]{2,3})(?:v[0-9])?[^0-9A-Za-z])", // media_filename-0.1.4: "(?:[eE]([0-9]{2,3})|[^0-9A-Za-z]([0-9]{2,3})(?:v[0-9])?[^0-9A-Za-z])"
	"[sS]([0-9]{1,2})", // media_filename-0.1.4: "[sS]([0-9]{1,2})"
	"((?i)(?:PPV.)?[HP]DTV|(?:HD)?CAM|BRRIP|[^a-z]TS[^a-z]|(?:PPV )?WEB.?DL(?: DVDRip)?|HDRip|DVDRip|CamRip|W[EB]BRip|BluRay|BD|DVD|DvDScr|hdtv)", // media_filename-0.1.4: "((?i)(?:PPV.)?[HP]DTV|(?:HD)?CAM|BRRIP|[^a-z]TS[^a-z]|(?:PPV )?WEB.?DL(?: DVDRip)?|HDRip|DVDRip|CamRip|W[EB]BRip|BluRay|BD|DVD|DvDScr|hdtv)"
	"((19[0-9]|20[01])[0-9])",  // media_filename-0.1.4: "((19[0-9]|20[01])[0-9])"
	"((?i)xvid|x264|h\\.?264)", // media_filename-0.1.4: "((?i)xvid|x264|h\\.?264)"
	"((?i)MP3|DD5\\.?1|Dual[- ]Audio|LiNE|DTS|AAC(?:\\.?2\\.0)?|AC3(?:\\.5\\.1)?)", // media_filename-0.1.4: "((?i)MP3|DD5\\.?1|Dual[- ]Audio|LiNE|DTS|AAC(?:\\.?2\\.0)?|AC3(?:\\.5\\.1)?)"
	"\\[([0-9A-F]{8})\\]",                   // media_filename-0.1.4: "\\[([0-9A-F]{8})\\]"
	`(\d+)[xX](\d+)`,                        // termimage-0.3.2: r"(\d+)[xX](\d+)"
	`.*(\d{4}-\d{2}-\d{2}).*`,               // teensy-0.1.0: r".*(\d{4}-\d{2}-\d{2}).*"
	`<@(.+)>`,                               // telescreen-0.1.3: r"<@(.+)>"
	`^(\d+)`,                                // tempus_fugit-0.4.4: r"^(\d+)"
	"(\\?|\\.|\\*|\\[|\\]|\\(|\\)|\\^|\\$)", // fselect-0.4.1: "(\\?|\\.|\\*|\\[|\\]|\\(|\\)|\\^|\\$)"
	"(%|_|\\?|\\.|\\*|\\[|\\]|\\(|\\)|\\^|\\$)",    // fselect-0.4.1: "(%|_|\\?|\\.|\\*|\\[|\\]|\\(|\\)|\\^|\\$)"
	`^([A-Z]+)(?:\s(.+))?\s*`,                      // fs_eventbridge-0.1.0: r"^([A-Z]+)(?:\s(.+))?\s*"
	`(\w{1,2})\[(.+?)\]`,                           // joseki-0.0.1: r"(\w{1,2})\[(.+?)\]"
	`(?i)in (\d+) (second|minute|hour|day|week)s?`, // tweetr-0.2.1: r"(?i)in (\d+) (second|minute|hour|day|week)s?"
	"^([0-9])+",              // bullet_core-0.1.1: "^(?u:[0-9])+"
	"^([0-9])+(\\.)([0-9])+", // bullet_core-0.1.1: "^(?u:[0-9])+(?u:\\.)(?u:[0-9])+"
	"^([A-Za-zª-ªµ-µº-ºÀ-ÖØ-öø-ˁˆ-ˑˠ-ˤˬ-ˬˮ-ˮͰ-ʹͶ-ͷͺ-ͽͿ-ͿΆ-ΆΈ-ΊΌ-ΌΎ-ΡΣ-ϵϷ-ҁҊ-ԯԱ-Ֆՙ-ՙա-ևא-תװ-ײؠ-يٮ-ٯٱ-ۓە-ەۥ-ۦۮ-ۯۺ-ۼۿ-ۿܐ-ܐܒ-ܯݍ-ޥޱ-ޱߊ-ߪߴ-ߵߺ-ߺࠀ-ࠕࠚ-ࠚࠤ-ࠤࠨ-ࠨࡀ-ࡘࢠ-ࢴऄ-हऽ-ऽॐ-ॐक़-ॡॱ-ঀঅ-ঌএ-ঐও-নপ-রল-লশ-হঽ-ঽৎ-ৎড়-ঢ়য়-ৡৰ-ৱਅ-ਊਏ-ਐਓ-ਨਪ-ਰਲ-ਲ਼ਵ-ਸ਼ਸ-ਹਖ਼-ੜਫ਼-ਫ਼ੲ-ੴઅ-ઍએ-ઑઓ-નપ-રલ-ળવ-હઽ-ઽૐ-ૐૠ-ૡૹ-ૹଅ-ଌଏ-ଐଓ-ନପ-ରଲ-ଳଵ-ହଽ-ଽଡ଼-ଢ଼ୟ-ୡୱ-ୱஃ-ஃஅ-ஊஎ-ஐஒ-கங-சஜ-ஜஞ-டண-தந-பம-ஹௐ-ௐఅ-ఌఎ-ఐఒ-నప-హఽ-ఽౘ-ౚౠ-ౡಅ-ಌಎ-ಐಒ-ನಪ-ಳವ-ಹಽ-ಽೞ-ೞೠ-ೡೱ-ೲഅ-ഌഎ-ഐഒ-ഺഽ-ഽൎ-ൎൟ-ൡൺ-ൿඅ-ඖක-නඳ-රල-ලව-ෆก-ะา-ำเ-ๆກ-ຂຄ-ຄງ-ຈຊ-ຊຍ-ຍດ-ທນ-ຟມ-ຣລ-ລວ-ວສ-ຫອ-ະາ-ຳຽ-ຽເ-ໄໆ-ໆໜ-ໟༀ-ༀཀ-ཇཉ-ཬྈ-ྌက-ဪဿ-ဿၐ-ၕၚ-ၝၡ-ၡၥ-ၦၮ-ၰၵ-ႁႎ-ႎႠ-ჅჇ-ჇჍ-Ⴭა-ჺჼ-ቈቊ-ቍቐ-ቖቘ-ቘቚ-ቝበ-ኈኊ-ኍነ-ኰኲ-ኵኸ-ኾዀ-ዀዂ-ዅወ-ዖዘ-ጐጒ-ጕጘ-ፚᎀ-ᎏᎠ-Ᏽᏸ-ᏽᐁ-ᙬᙯ-ᙿᚁ-ᚚᚠ-ᛪᛱ-ᛸᜀ-ᜌᜎ-ᜑᜠ-ᜱᝀ-ᝑᝠ-ᝬᝮ-ᝰក-ឳៗ-ៗៜ-ៜᠠ-ᡷᢀ-ᢨᢪ-ᢪᢰ-ᣵᤀ-ᤞᥐ-ᥭᥰ-ᥴᦀ-ᦫᦰ-ᧉᨀ-ᨖᨠ-ᩔᪧ-ᪧᬅ-ᬳᭅ-ᭋᮃ-ᮠᮮ-ᮯᮺ-ᯥᰀ-ᰣᱍ-ᱏᱚ-ᱽᳩ-ᳬᳮ-ᳱᳵ-ᳶᴀ-ᶿḀ-ἕἘ-Ἕἠ-ὅὈ-Ὅὐ-ὗὙ-ὙὛ-ὛὝ-ὝὟ-ώᾀ-ᾴᾶ-ᾼι-ιῂ-ῄῆ-ῌῐ-ΐῖ-Ίῠ-Ῥῲ-ῴῶ-ῼⁱ-ⁱⁿ-ⁿₐ-ₜℂ-ℂℇ-ℇℊ-ℓℕ-ℕℙ-ℝℤ-ℤΩ-Ωℨ-ℨK-ℭℯ-ℹℼ-ℿⅅ-ⅉⅎ-ⅎↃ-ↄⰀ-Ⱞⰰ-ⱞⱠ-ⳤⳫ-ⳮⳲ-ⳳⴀ-ⴥⴧ-ⴧⴭ-ⴭⴰ-ⵧⵯ-ⵯⶀ-ⶖⶠ-ⶦⶨ-ⶮⶰ-ⶶⶸ-ⶾⷀ-ⷆⷈ-ⷎⷐ-ⷖⷘ-ⷞⸯ-ⸯ々-〆〱-〵〻-〼ぁ-ゖゝ-ゟァ-ヺー-ヿㄅ-ㄭㄱ-ㆎㆠ-ㆺㇰ-ㇿ㐀-䶵一-鿕ꀀ-ꒌꓐ-ꓽꔀ-ꘌꘐ-ꘟꘪ-ꘫꙀ-ꙮꙿ-ꚝꚠ-ꛥꜗ-ꜟꜢ-ꞈꞋ-ꞭꞰ-ꞷꟷ-ꠁꠃ-ꠅꠇ-ꠊꠌ-ꠢꡀ-ꡳꢂ-ꢳꣲ-ꣷꣻ-ꣻꣽ-ꣽꤊ-ꤥꤰ-ꥆꥠ-ꥼꦄ-ꦲꧏ-ꧏꧠ-ꧤꧦ-ꧯꧺ-ꧾꨀ-ꨨꩀ-ꩂꩄ-ꩋꩠ-ꩶꩺ-ꩺꩾ-ꪯꪱ-ꪱꪵ-ꪶꪹ-ꪽꫀ-ꫀꫂ-ꫂꫛ-ꫝꫠ-ꫪꫲ-ꫴꬁ-ꬆꬉ-ꬎꬑ-ꬖꬠ-ꬦꬨ-ꬮꬰ-ꭚꭜ-ꭥꭰ-ꯢ가-힣ힰ-ퟆퟋ-ퟻ豈-舘並-龎ﬀ-ﬆﬓ-ﬗיִ-יִײַ-ﬨשׁ-זּטּ-לּמּ-מּנּ-סּףּ-פּצּ-ﮱﯓ-ﴽﵐ-ﶏﶒ-ﷇﷰ-ﷻﹰ-ﹴﹶ-ﻼＡ-Ｚａ-ｚｦ-ﾾￂ-ￇￊ-ￏￒ-ￗￚ-ￜ𐀀-𐀋𐀍-𐀦𐀨-𐀺𐀼-𐀽𐀿-𐁍𐁐-𐁝𐂀-𐃺𐊀-𐊜𐊠-𐋐𐌀-𐌟𐌰-𐍀𐍂-𐍉𐍐-𐍵𐎀-𐎝𐎠-𐏃𐏈-𐏏𐐀-𐒝𐔀-𐔧𐔰-𐕣𐘀-𐜶𐝀-𐝕𐝠-𐝧𐠀-𐠅𐠈-𐠈𐠊-𐠵𐠷-𐠸𐠼-𐠼𐠿-𐡕𐡠-𐡶𐢀-𐢞𐣠-𐣲𐣴-𐣵𐤀-𐤕𐤠-𐤹𐦀-𐦷𐦾-𐦿𐨀-𐨀𐨐-𐨓𐨕-𐨗𐨙-𐨳𐩠-𐩼𐪀-𐪜𐫀-𐫇𐫉-𐫤𐬀-𐬵𐭀-𐭕𐭠-𐭲𐮀-𐮑𐰀-𐱈𐲀-𐲲𐳀-𐳲𑀃-𑀷𑂃-𑂯𑃐-𑃨𑄃-𑄦𑅐-𑅲𑅶-𑅶𑆃-𑆲𑇁-𑇄𑇚-𑇚𑇜-𑇜𑈀-𑈑𑈓-𑈫𑊀-𑊆𑊈-𑊈𑊊-𑊍𑊏-𑊝𑊟-𑊨𑊰-𑋞𑌅-𑌌𑌏-𑌐𑌓-𑌨𑌪-𑌰𑌲-𑌳𑌵-𑌹𑌽-𑌽𑍐-𑍐𑍝-𑍡𑒀-𑒯𑓄-𑓅𑓇-𑓇𑖀-𑖮𑗘-𑗛𑘀-𑘯𑙄-𑙄𑚀-𑚪𑜀-𑜙𑢠-𑣟𑣿-𑣿𑫀-𑫸𒀀-𒎙𒒀-𒕃𓀀-𓐮𔐀-𔙆𖠀-𖨸𖩀-𖩞𖫐-𖫭𖬀-𖬯𖭀-𖭃𖭣-𖭷𖭽-𖮏𖼀-𖽄𖽐-𖽐𖾓-𖾟𛀀-𛀁𛰀-𛱪𛱰-𛱼𛲀-𛲈𛲐-𛲙𝐀-𝑔𝑖-𝒜𝒞-𝒟𝒢-𝒢𝒥-𝒦𝒩-𝒬𝒮-𝒹𝒻-𝒻𝒽-𝓃𝓅-𝔅𝔇-𝔊𝔍-𝔔𝔖-𝔜𝔞-𝔹𝔻-𝔾𝕀-𝕄𝕆-𝕆𝕊-𝕐𝕒-𝚥𝚨-𝛀𝛂-𝛚𝛜-𝛺𝛼-𝜔𝜖-𝜴𝜶-𝝎𝝐-𝝮𝝰-𝞈𝞊-𝞨𝞪-𝟂𝟄-𝟋𞠀-𞣄𞸀-𞸃𞸅-𞸟𞸡-𞸢𞸤-𞸤𞸧-𞸧𞸩-𞸲𞸴-𞸷𞸹-𞸹𞸻-𞸻𞹂-𞹂𞹇-𞹇𞹉-𞹉𞹋-𞹋𞹍-𞹏𞹑-𞹒𞹔-𞹔𞹗-𞹗𞹙-𞹙𞹛-𞹛𞹝-𞹝𞹟-𞹟𞹡-𞹢𞹤-𞹤𞹧-𞹪𞹬-𞹲𞹴-𞹷𞹹-𞹼𞹾-𞹾𞺀-𞺉𞺋-𞺛𞺡-𞺣𞺥-𞺩𞺫-𞺻𠀀-𪛖𪜀-𫜴𫝀-𫠝𫠠-𬺡丽-𪘀])+",        // bullet_core-0.1.1: "^(?u:[A-Za-zª-ªµ-µº-ºÀ-ÖØ-öø-ˁˆ-ˑˠ-ˤˬ-ˬˮ-ˮͰ-ʹͶ-ͷͺ-ͽͿ-ͿΆ-ΆΈ-ΊΌ-ΌΎ-ΡΣ-ϵϷ-ҁҊ-ԯԱ-Ֆՙ-ՙա-ևא-תװ-ײؠ-يٮ-ٯٱ-ۓە-ەۥ-ۦۮ-ۯۺ-ۼۿ-ۿܐ-ܐܒ-ܯݍ-ޥޱ-ޱߊ-ߪߴ-ߵߺ-ߺࠀ-ࠕࠚ-ࠚࠤ-ࠤࠨ-ࠨࡀ-ࡘࢠ-ࢴऄ-हऽ-ऽॐ-ॐक़-ॡॱ-ঀঅ-ঌএ-ঐও-নপ-রল-লশ-হঽ-ঽৎ-ৎড়-ঢ়য়-ৡৰ-ৱਅ-ਊਏ-ਐਓ-ਨਪ-ਰਲ-ਲ਼ਵ-ਸ਼ਸ-ਹਖ਼-ੜਫ਼-ਫ਼ੲ-ੴઅ-ઍએ-ઑઓ-નપ-રલ-ળવ-હઽ-ઽૐ-ૐૠ-ૡૹ-ૹଅ-ଌଏ-ଐଓ-ନପ-ରଲ-ଳଵ-ହଽ-ଽଡ଼-ଢ଼ୟ-ୡୱ-ୱஃ-ஃஅ-ஊஎ-ஐஒ-கங-சஜ-ஜஞ-டண-தந-பம-ஹௐ-ௐఅ-ఌఎ-ఐఒ-నప-హఽ-ఽౘ-ౚౠ-ౡಅ-ಌಎ-ಐಒ-ನಪ-ಳವ-ಹಽ-ಽೞ-ೞೠ-ೡೱ-ೲഅ-ഌഎ-ഐഒ-ഺഽ-ഽൎ-ൎൟ-ൡൺ-ൿඅ-ඖක-නඳ-රල-ලව-ෆก-ะา-ำเ-ๆກ-ຂຄ-ຄງ-ຈຊ-ຊຍ-ຍດ-ທນ-ຟມ-ຣລ-ລວ-ວສ-ຫອ-ະາ-ຳຽ-ຽເ-ໄໆ-ໆໜ-ໟༀ-ༀཀ-ཇཉ-ཬྈ-ྌက-ဪဿ-ဿၐ-ၕၚ-ၝၡ-ၡၥ-ၦၮ-ၰၵ-ႁႎ-ႎႠ-ჅჇ-ჇჍ-Ⴭა-ჺჼ-ቈቊ-ቍቐ-ቖቘ-ቘቚ-ቝበ-ኈኊ-ኍነ-ኰኲ-ኵኸ-ኾዀ-ዀዂ-ዅወ-ዖዘ-ጐጒ-ጕጘ-ፚᎀ-ᎏᎠ-Ᏽᏸ-ᏽᐁ-ᙬᙯ-ᙿᚁ-ᚚᚠ-ᛪᛱ-ᛸᜀ-ᜌᜎ-ᜑᜠ-ᜱᝀ-ᝑᝠ-ᝬᝮ-ᝰក-ឳៗ-ៗៜ-ៜᠠ-ᡷᢀ-ᢨᢪ-ᢪᢰ-ᣵᤀ-ᤞᥐ-ᥭᥰ-ᥴᦀ-ᦫᦰ-ᧉᨀ-ᨖᨠ-ᩔᪧ-ᪧᬅ-ᬳᭅ-ᭋᮃ-ᮠᮮ-ᮯᮺ-ᯥᰀ-ᰣᱍ-ᱏᱚ-ᱽᳩ-ᳬᳮ-ᳱᳵ-ᳶᴀ-ᶿḀ-ἕἘ-Ἕἠ-ὅὈ-Ὅὐ-ὗὙ-ὙὛ-ὛὝ-ὝὟ-ώᾀ-ᾴᾶ-ᾼι-ιῂ-ῄῆ-ῌῐ-ΐῖ-Ίῠ-Ῥῲ-ῴῶ-ῼⁱ-ⁱⁿ-ⁿₐ-ₜℂ-ℂℇ-ℇℊ-ℓℕ-ℕℙ-ℝℤ-ℤΩ-Ωℨ-ℨK-ℭℯ-ℹℼ-ℿⅅ-ⅉⅎ-ⅎↃ-ↄⰀ-Ⱞⰰ-ⱞⱠ-ⳤⳫ-ⳮⳲ-ⳳⴀ-ⴥⴧ-ⴧⴭ-ⴭⴰ-ⵧⵯ-ⵯⶀ-ⶖⶠ-ⶦⶨ-ⶮⶰ-ⶶⶸ-ⶾⷀ-ⷆⷈ-ⷎⷐ-ⷖⷘ-ⷞⸯ-ⸯ々-〆〱-〵〻-〼ぁ-ゖゝ-ゟァ-ヺー-ヿㄅ-ㄭㄱ-ㆎㆠ-ㆺㇰ-ㇿ㐀-䶵一-鿕ꀀ-ꒌꓐ-ꓽꔀ-ꘌꘐ-ꘟꘪ-ꘫꙀ-ꙮꙿ-ꚝꚠ-ꛥꜗ-ꜟꜢ-ꞈꞋ-ꞭꞰ-ꞷꟷ-ꠁꠃ-ꠅꠇ-ꠊꠌ-ꠢꡀ-ꡳꢂ-ꢳꣲ-ꣷꣻ-ꣻꣽ-ꣽꤊ-ꤥꤰ-ꥆꥠ-ꥼꦄ-ꦲꧏ-ꧏꧠ-ꧤꧦ-ꧯꧺ-ꧾꨀ-ꨨꩀ-ꩂꩄ-ꩋꩠ-ꩶꩺ-ꩺꩾ-ꪯꪱ-ꪱꪵ-ꪶꪹ-ꪽꫀ-ꫀꫂ-ꫂꫛ-ꫝꫠ-ꫪꫲ-ꫴꬁ-ꬆꬉ-ꬎꬑ-ꬖꬠ-ꬦꬨ-ꬮꬰ-ꭚꭜ-ꭥꭰ-ꯢ가-힣ힰ-ퟆퟋ-ퟻ豈-舘並-龎ﬀ-ﬆﬓ-ﬗיִ-יִײַ-ﬨשׁ-זּטּ-לּמּ-מּנּ-סּףּ-פּצּ-ﮱﯓ-ﴽﵐ-ﶏﶒ-ﷇﷰ-ﷻﹰ-ﹴﹶ-ﻼＡ-Ｚａ-ｚｦ-ﾾￂ-ￇￊ-ￏￒ-ￗￚ-ￜ𐀀-𐀋𐀍-𐀦𐀨-𐀺𐀼-𐀽𐀿-𐁍𐁐-𐁝𐂀-𐃺𐊀-𐊜𐊠-𐋐𐌀-𐌟𐌰-𐍀𐍂-𐍉𐍐-𐍵𐎀-𐎝𐎠-𐏃𐏈-𐏏𐐀-𐒝𐔀-𐔧𐔰-𐕣𐘀-𐜶𐝀-𐝕𐝠-𐝧𐠀-𐠅𐠈-𐠈𐠊-𐠵𐠷-𐠸𐠼-𐠼𐠿-𐡕𐡠-𐡶𐢀-𐢞𐣠-𐣲𐣴-𐣵𐤀-𐤕𐤠-𐤹𐦀-𐦷𐦾-𐦿𐨀-𐨀𐨐-𐨓𐨕-𐨗𐨙-𐨳𐩠-𐩼𐪀-𐪜𐫀-𐫇𐫉-𐫤𐬀-𐬵𐭀-𐭕𐭠-𐭲𐮀-𐮑𐰀-𐱈𐲀-𐲲𐳀-𐳲𑀃-𑀷𑂃-𑂯𑃐-𑃨𑄃-𑄦𑅐-𑅲𑅶-𑅶𑆃-𑆲𑇁-𑇄𑇚-𑇚𑇜-𑇜𑈀-𑈑𑈓-𑈫𑊀-𑊆𑊈-𑊈𑊊-𑊍𑊏-𑊝𑊟-𑊨𑊰-𑋞𑌅-𑌌𑌏-𑌐𑌓-𑌨𑌪-𑌰𑌲-𑌳𑌵-𑌹𑌽-𑌽𑍐-𑍐𑍝-𑍡𑒀-𑒯𑓄-𑓅𑓇-𑓇𑖀-𑖮𑗘-𑗛𑘀-𑘯𑙄-𑙄𑚀-𑚪𑜀-𑜙𑢠-𑣟𑣿-𑣿𑫀-𑫸𒀀-𒎙𒒀-𒕃𓀀-𓐮𔐀-𔙆𖠀-𖨸𖩀-𖩞𖫐-𖫭𖬀-𖬯𖭀-𖭃𖭣-𖭷𖭽-𖮏𖼀-𖽄𖽐-𖽐𖾓-𖾟𛀀-𛀁𛰀-𛱪𛱰-𛱼𛲀-𛲈𛲐-𛲙𝐀-𝑔𝑖-𝒜𝒞-𝒟𝒢-𝒢𝒥-𝒦𝒩-𝒬𝒮-𝒹𝒻-𝒻𝒽-𝓃𝓅-𝔅𝔇-𝔊𝔍-𝔔𝔖-𝔜𝔞-𝔹𝔻-𝔾𝕀-𝕄𝕆-𝕆𝕊-𝕐𝕒-𝚥𝚨-𝛀𝛂-𝛚𝛜-𝛺𝛼-𝜔𝜖-𝜴𝜶-𝝎𝝐-𝝮𝝰-𝞈𝞊-𝞨𝞪-𝟂𝟄-𝟋𞠀-𞣄𞸀-𞸃𞸅-𞸟𞸡-𞸢𞸤-𞸤𞸧-𞸧𞸩-𞸲𞸴-𞸷𞸹-𞸹𞸻-𞸻𞹂-𞹂𞹇-𞹇𞹉-𞹉𞹋-𞹋𞹍-𞹏𞹑-𞹒𞹔-𞹔𞹗-𞹗𞹙-𞹙𞹛-𞹛𞹝-𞹝𞹟-𞹟𞹡-𞹢𞹤-𞹤𞹧-𞹪𞹬-𞹲𞹴-𞹷𞹹-𞹼𞹾-𞹾𞺀-𞺉𞺋-𞺛𞺡-𞺣𞺥-𞺩𞺫-𞺻𠀀-𪛖𪜀-𫜴𫝀-𫠝𫠠-𬺡丽-𪘀])+"
	"^(d/d)(([A-Za-zª-ªµ-µº-ºÀ-ÖØ-öø-ˁˆ-ˑˠ-ˤˬ-ˬˮ-ˮͰ-ʹͶ-ͷͺ-ͽͿ-ͿΆ-ΆΈ-ΊΌ-ΌΎ-ΡΣ-ϵϷ-ҁҊ-ԯԱ-Ֆՙ-ՙա-ևא-תװ-ײؠ-يٮ-ٯٱ-ۓە-ەۥ-ۦۮ-ۯۺ-ۼۿ-ۿܐ-ܐܒ-ܯݍ-ޥޱ-ޱߊ-ߪߴ-ߵߺ-ߺࠀ-ࠕࠚ-ࠚࠤ-ࠤࠨ-ࠨࡀ-ࡘࢠ-ࢴऄ-हऽ-ऽॐ-ॐक़-ॡॱ-ঀঅ-ঌএ-ঐও-নপ-রল-লশ-হঽ-ঽৎ-ৎড়-ঢ়য়-ৡৰ-ৱਅ-ਊਏ-ਐਓ-ਨਪ-ਰਲ-ਲ਼ਵ-ਸ਼ਸ-ਹਖ਼-ੜਫ਼-ਫ਼ੲ-ੴઅ-ઍએ-ઑઓ-નપ-રલ-ળવ-હઽ-ઽૐ-ૐૠ-ૡૹ-ૹଅ-ଌଏ-ଐଓ-ନପ-ରଲ-ଳଵ-ହଽ-ଽଡ଼-ଢ଼ୟ-ୡୱ-ୱஃ-ஃஅ-ஊஎ-ஐஒ-கங-சஜ-ஜஞ-டண-தந-பம-ஹௐ-ௐఅ-ఌఎ-ఐఒ-నప-హఽ-ఽౘ-ౚౠ-ౡಅ-ಌಎ-ಐಒ-ನಪ-ಳವ-ಹಽ-ಽೞ-ೞೠ-ೡೱ-ೲഅ-ഌഎ-ഐഒ-ഺഽ-ഽൎ-ൎൟ-ൡൺ-ൿඅ-ඖක-නඳ-රල-ලව-ෆก-ะา-ำเ-ๆກ-ຂຄ-ຄງ-ຈຊ-ຊຍ-ຍດ-ທນ-ຟມ-ຣລ-ລວ-ວສ-ຫອ-ະາ-ຳຽ-ຽເ-ໄໆ-ໆໜ-ໟༀ-ༀཀ-ཇཉ-ཬྈ-ྌက-ဪဿ-ဿၐ-ၕၚ-ၝၡ-ၡၥ-ၦၮ-ၰၵ-ႁႎ-ႎႠ-ჅჇ-ჇჍ-Ⴭა-ჺჼ-ቈቊ-ቍቐ-ቖቘ-ቘቚ-ቝበ-ኈኊ-ኍነ-ኰኲ-ኵኸ-ኾዀ-ዀዂ-ዅወ-ዖዘ-ጐጒ-ጕጘ-ፚᎀ-ᎏᎠ-Ᏽᏸ-ᏽᐁ-ᙬᙯ-ᙿᚁ-ᚚᚠ-ᛪᛱ-ᛸᜀ-ᜌᜎ-ᜑᜠ-ᜱᝀ-ᝑᝠ-ᝬᝮ-ᝰក-ឳៗ-ៗៜ-ៜᠠ-ᡷᢀ-ᢨᢪ-ᢪᢰ-ᣵᤀ-ᤞᥐ-ᥭᥰ-ᥴᦀ-ᦫᦰ-ᧉᨀ-ᨖᨠ-ᩔᪧ-ᪧᬅ-ᬳᭅ-ᭋᮃ-ᮠᮮ-ᮯᮺ-ᯥᰀ-ᰣᱍ-ᱏᱚ-ᱽᳩ-ᳬᳮ-ᳱᳵ-ᳶᴀ-ᶿḀ-ἕἘ-Ἕἠ-ὅὈ-Ὅὐ-ὗὙ-ὙὛ-ὛὝ-ὝὟ-ώᾀ-ᾴᾶ-ᾼι-ιῂ-ῄῆ-ῌῐ-ΐῖ-Ίῠ-Ῥῲ-ῴῶ-ῼⁱ-ⁱⁿ-ⁿₐ-ₜℂ-ℂℇ-ℇℊ-ℓℕ-ℕℙ-ℝℤ-ℤΩ-Ωℨ-ℨK-ℭℯ-ℹℼ-ℿⅅ-ⅉⅎ-ⅎↃ-ↄⰀ-Ⱞⰰ-ⱞⱠ-ⳤⳫ-ⳮⳲ-ⳳⴀ-ⴥⴧ-ⴧⴭ-ⴭⴰ-ⵧⵯ-ⵯⶀ-ⶖⶠ-ⶦⶨ-ⶮⶰ-ⶶⶸ-ⶾⷀ-ⷆⷈ-ⷎⷐ-ⷖⷘ-ⷞⸯ-ⸯ々-〆〱-〵〻-〼ぁ-ゖゝ-ゟァ-ヺー-ヿㄅ-ㄭㄱ-ㆎㆠ-ㆺㇰ-ㇿ㐀-䶵一-鿕ꀀ-ꒌꓐ-ꓽꔀ-ꘌꘐ-ꘟꘪ-ꘫꙀ-ꙮꙿ-ꚝꚠ-ꛥꜗ-ꜟꜢ-ꞈꞋ-ꞭꞰ-ꞷꟷ-ꠁꠃ-ꠅꠇ-ꠊꠌ-ꠢꡀ-ꡳꢂ-ꢳꣲ-ꣷꣻ-ꣻꣽ-ꣽꤊ-ꤥꤰ-ꥆꥠ-ꥼꦄ-ꦲꧏ-ꧏꧠ-ꧤꧦ-ꧯꧺ-ꧾꨀ-ꨨꩀ-ꩂꩄ-ꩋꩠ-ꩶꩺ-ꩺꩾ-ꪯꪱ-ꪱꪵ-ꪶꪹ-ꪽꫀ-ꫀꫂ-ꫂꫛ-ꫝꫠ-ꫪꫲ-ꫴꬁ-ꬆꬉ-ꬎꬑ-ꬖꬠ-ꬦꬨ-ꬮꬰ-ꭚꭜ-ꭥꭰ-ꯢ가-힣ힰ-ퟆퟋ-ퟻ豈-舘並-龎ﬀ-ﬆﬓ-ﬗיִ-יִײַ-ﬨשׁ-זּטּ-לּמּ-מּנּ-סּףּ-פּצּ-ﮱﯓ-ﴽﵐ-ﶏﶒ-ﷇﷰ-ﷻﹰ-ﹴﹶ-ﻼＡ-Ｚａ-ｚｦ-ﾾￂ-ￇￊ-ￏￒ-ￗￚ-ￜ𐀀-𐀋𐀍-𐀦𐀨-𐀺𐀼-𐀽𐀿-𐁍𐁐-𐁝𐂀-𐃺𐊀-𐊜𐊠-𐋐𐌀-𐌟𐌰-𐍀𐍂-𐍉𐍐-𐍵𐎀-𐎝𐎠-𐏃𐏈-𐏏𐐀-𐒝𐔀-𐔧𐔰-𐕣𐘀-𐜶𐝀-𐝕𐝠-𐝧𐠀-𐠅𐠈-𐠈𐠊-𐠵𐠷-𐠸𐠼-𐠼𐠿-𐡕𐡠-𐡶𐢀-𐢞𐣠-𐣲𐣴-𐣵𐤀-𐤕𐤠-𐤹𐦀-𐦷𐦾-𐦿𐨀-𐨀𐨐-𐨓𐨕-𐨗𐨙-𐨳𐩠-𐩼𐪀-𐪜𐫀-𐫇𐫉-𐫤𐬀-𐬵𐭀-𐭕𐭠-𐭲𐮀-𐮑𐰀-𐱈𐲀-𐲲𐳀-𐳲𑀃-𑀷𑂃-𑂯𑃐-𑃨𑄃-𑄦𑅐-𑅲𑅶-𑅶𑆃-𑆲𑇁-𑇄𑇚-𑇚𑇜-𑇜𑈀-𑈑𑈓-𑈫𑊀-𑊆𑊈-𑊈𑊊-𑊍𑊏-𑊝𑊟-𑊨𑊰-𑋞𑌅-𑌌𑌏-𑌐𑌓-𑌨𑌪-𑌰𑌲-𑌳𑌵-𑌹𑌽-𑌽𑍐-𑍐𑍝-𑍡𑒀-𑒯𑓄-𑓅𑓇-𑓇𑖀-𑖮𑗘-𑗛𑘀-𑘯𑙄-𑙄𑚀-𑚪𑜀-𑜙𑢠-𑣟𑣿-𑣿𑫀-𑫸𒀀-𒎙𒒀-𒕃𓀀-𓐮𔐀-𔙆𖠀-𖨸𖩀-𖩞𖫐-𖫭𖬀-𖬯𖭀-𖭃𖭣-𖭷𖭽-𖮏𖼀-𖽄𖽐-𖽐𖾓-𖾟𛀀-𛀁𛰀-𛱪𛱰-𛱼𛲀-𛲈𛲐-𛲙𝐀-𝑔𝑖-𝒜𝒞-𝒟𝒢-𝒢𝒥-𝒦𝒩-𝒬𝒮-𝒹𝒻-𝒻𝒽-𝓃𝓅-𝔅𝔇-𝔊𝔍-𝔔𝔖-𝔜𝔞-𝔹𝔻-𝔾𝕀-𝕄𝕆-𝕆𝕊-𝕐𝕒-𝚥𝚨-𝛀𝛂-𝛚𝛜-𝛺𝛼-𝜔𝜖-𝜴𝜶-𝝎𝝐-𝝮𝝰-𝞈𝞊-𝞨𝞪-𝟂𝟄-𝟋𞠀-𞣄𞸀-𞸃𞸅-𞸟𞸡-𞸢𞸤-𞸤𞸧-𞸧𞸩-𞸲𞸴-𞸷𞸹-𞸹𞸻-𞸻𞹂-𞹂𞹇-𞹇𞹉-𞹉𞹋-𞹋𞹍-𞹏𞹑-𞹒𞹔-𞹔𞹗-𞹗𞹙-𞹙𞹛-𞹛𞹝-𞹝𞹟-𞹟𞹡-𞹢𞹤-𞹤𞹧-𞹪𞹬-𞹲𞹴-𞹷𞹹-𞹼𞹾-𞹾𞺀-𞺉𞺋-𞺛𞺡-𞺣𞺥-𞺩𞺫-𞺻𠀀-𪛖𪜀-𫜴𫝀-𫠝𫠠-𬺡丽-𪘀])+)", // bullet_core-0.1.1: "^(?u:d/d)((?u:[A-Za-zª-ªµ-µº-ºÀ-ÖØ-öø-ˁˆ-ˑˠ-ˤˬ-ˬˮ-ˮͰ-ʹͶ-ͷͺ-ͽͿ-ͿΆ-ΆΈ-ΊΌ-ΌΎ-ΡΣ-ϵϷ-ҁҊ-ԯԱ-Ֆՙ-ՙա-ևא-תװ-ײؠ-يٮ-ٯٱ-ۓە-ەۥ-ۦۮ-ۯۺ-ۼۿ-ۿܐ-ܐܒ-ܯݍ-ޥޱ-ޱߊ-ߪߴ-ߵߺ-ߺࠀ-ࠕࠚ-ࠚࠤ-ࠤࠨ-ࠨࡀ-ࡘࢠ-ࢴऄ-हऽ-ऽॐ-ॐक़-ॡॱ-ঀঅ-ঌএ-ঐও-নপ-রল-লশ-হঽ-ঽৎ-ৎড়-ঢ়য়-ৡৰ-ৱਅ-ਊਏ-ਐਓ-ਨਪ-ਰਲ-ਲ਼ਵ-ਸ਼ਸ-ਹਖ਼-ੜਫ਼-ਫ਼ੲ-ੴઅ-ઍએ-ઑઓ-નપ-રલ-ળવ-હઽ-ઽૐ-ૐૠ-ૡૹ-ૹଅ-ଌଏ-ଐଓ-ନପ-ରଲ-ଳଵ-ହଽ-ଽଡ଼-ଢ଼ୟ-ୡୱ-ୱஃ-ஃஅ-ஊஎ-ஐஒ-கங-சஜ-ஜஞ-டண-தந-பம-ஹௐ-ௐఅ-ఌఎ-ఐఒ-నప-హఽ-ఽౘ-ౚౠ-ౡಅ-ಌಎ-ಐಒ-ನಪ-ಳವ-ಹಽ-ಽೞ-ೞೠ-ೡೱ-ೲഅ-ഌഎ-ഐഒ-ഺഽ-ഽൎ-ൎൟ-ൡൺ-ൿඅ-ඖක-නඳ-රල-ලව-ෆก-ะา-ำเ-ๆກ-ຂຄ-ຄງ-ຈຊ-ຊຍ-ຍດ-ທນ-ຟມ-ຣລ-ລວ-ວສ-ຫອ-ະາ-ຳຽ-ຽເ-ໄໆ-ໆໜ-ໟༀ-ༀཀ-ཇཉ-ཬྈ-ྌက-ဪဿ-ဿၐ-ၕၚ-ၝၡ-ၡၥ-ၦၮ-ၰၵ-ႁႎ-ႎႠ-ჅჇ-ჇჍ-Ⴭა-ჺჼ-ቈቊ-ቍቐ-ቖቘ-ቘቚ-ቝበ-ኈኊ-ኍነ-ኰኲ-ኵኸ-ኾዀ-ዀዂ-ዅወ-ዖዘ-ጐጒ-ጕጘ-ፚᎀ-ᎏᎠ-Ᏽᏸ-ᏽᐁ-ᙬᙯ-ᙿᚁ-ᚚᚠ-ᛪᛱ-ᛸᜀ-ᜌᜎ-ᜑᜠ-ᜱᝀ-ᝑᝠ-ᝬᝮ-ᝰក-ឳៗ-ៗៜ-ៜᠠ-ᡷᢀ-ᢨᢪ-ᢪᢰ-ᣵᤀ-ᤞᥐ-ᥭᥰ-ᥴᦀ-ᦫᦰ-ᧉᨀ-ᨖᨠ-ᩔᪧ-ᪧᬅ-ᬳᭅ-ᭋᮃ-ᮠᮮ-ᮯᮺ-ᯥᰀ-ᰣᱍ-ᱏᱚ-ᱽᳩ-ᳬᳮ-ᳱᳵ-ᳶᴀ-ᶿḀ-ἕἘ-Ἕἠ-ὅὈ-Ὅὐ-ὗὙ-ὙὛ-ὛὝ-ὝὟ-ώᾀ-ᾴᾶ-ᾼι-ιῂ-ῄῆ-ῌῐ-ΐῖ-Ίῠ-Ῥῲ-ῴῶ-ῼⁱ-ⁱⁿ-ⁿₐ-ₜℂ-ℂℇ-ℇℊ-ℓℕ-ℕℙ-ℝℤ-ℤΩ-Ωℨ-ℨK-ℭℯ-ℹℼ-ℿⅅ-ⅉⅎ-ⅎↃ-ↄⰀ-Ⱞⰰ-ⱞⱠ-ⳤⳫ-ⳮⳲ-ⳳⴀ-ⴥⴧ-ⴧⴭ-ⴭⴰ-ⵧⵯ-ⵯⶀ-ⶖⶠ-ⶦⶨ-ⶮⶰ-ⶶⶸ-ⶾⷀ-ⷆⷈ-ⷎⷐ-ⷖⷘ-ⷞⸯ-ⸯ々-〆〱-〵〻-〼ぁ-ゖゝ-ゟァ-ヺー-ヿㄅ-ㄭㄱ-ㆎㆠ-ㆺㇰ-ㇿ㐀-䶵一-鿕ꀀ-ꒌꓐ-ꓽꔀ-ꘌꘐ-ꘟꘪ-ꘫꙀ-ꙮꙿ-ꚝꚠ-ꛥꜗ-ꜟꜢ-ꞈꞋ-ꞭꞰ-ꞷꟷ-ꠁꠃ-ꠅꠇ-ꠊꠌ-ꠢꡀ-ꡳꢂ-ꢳꣲ-ꣷꣻ-ꣻꣽ-ꣽꤊ-ꤥꤰ-ꥆꥠ-ꥼꦄ-ꦲꧏ-ꧏꧠ-ꧤꧦ-ꧯꧺ-ꧾꨀ-ꨨꩀ-ꩂꩄ-ꩋꩠ-ꩶꩺ-ꩺꩾ-ꪯꪱ-ꪱꪵ-ꪶꪹ-ꪽꫀ-ꫀꫂ-ꫂꫛ-ꫝꫠ-ꫪꫲ-ꫴꬁ-ꬆꬉ-ꬎꬑ-ꬖꬠ-ꬦꬨ-ꬮꬰ-ꭚꭜ-ꭥꭰ-ꯢ가-힣ힰ-ퟆퟋ-ퟻ豈-舘並-龎ﬀ-ﬆﬓ-ﬗיִ-יִײַ-ﬨשׁ-זּטּ-לּמּ-מּנּ-סּףּ-פּצּ-ﮱﯓ-ﴽﵐ-ﶏﶒ-ﷇﷰ-ﷻﹰ-ﹴﹶ-ﻼＡ-Ｚａ-ｚｦ-ﾾￂ-ￇￊ-ￏￒ-ￗￚ-ￜ𐀀-𐀋𐀍-𐀦𐀨-𐀺𐀼-𐀽𐀿-𐁍𐁐-𐁝𐂀-𐃺𐊀-𐊜𐊠-𐋐𐌀-𐌟𐌰-𐍀𐍂-𐍉𐍐-𐍵𐎀-𐎝𐎠-𐏃𐏈-𐏏𐐀-𐒝𐔀-𐔧𐔰-𐕣𐘀-𐜶𐝀-𐝕𐝠-𐝧𐠀-𐠅𐠈-𐠈𐠊-𐠵𐠷-𐠸𐠼-𐠼𐠿-𐡕𐡠-𐡶𐢀-𐢞𐣠-𐣲𐣴-𐣵𐤀-𐤕𐤠-𐤹𐦀-𐦷𐦾-𐦿𐨀-𐨀𐨐-𐨓𐨕-𐨗𐨙-𐨳𐩠-𐩼𐪀-𐪜𐫀-𐫇𐫉-𐫤𐬀-𐬵𐭀-𐭕𐭠-𐭲𐮀-𐮑𐰀-𐱈𐲀-𐲲𐳀-𐳲𑀃-𑀷𑂃-𑂯𑃐-𑃨𑄃-𑄦𑅐-𑅲𑅶-𑅶𑆃-𑆲𑇁-𑇄𑇚-𑇚𑇜-𑇜𑈀-𑈑𑈓-𑈫𑊀-𑊆𑊈-𑊈𑊊-𑊍𑊏-𑊝𑊟-𑊨𑊰-𑋞𑌅-𑌌𑌏-𑌐𑌓-𑌨𑌪-𑌰𑌲-𑌳𑌵-𑌹𑌽-𑌽𑍐-𑍐𑍝-𑍡𑒀-𑒯𑓄-𑓅𑓇-𑓇𑖀-𑖮𑗘-𑗛𑘀-𑘯𑙄-𑙄𑚀-𑚪𑜀-𑜙𑢠-𑣟𑣿-𑣿𑫀-𑫸𒀀-𒎙𒒀-𒕃𓀀-𓐮𔐀-𔙆𖠀-𖨸𖩀-𖩞𖫐-𖫭𖬀-𖬯𖭀-𖭃𖭣-𖭷𖭽-𖮏𖼀-𖽄𖽐-𖽐𖾓-𖾟𛀀-𛀁𛰀-𛱪𛱰-𛱼𛲀-𛲈𛲐-𛲙𝐀-𝑔𝑖-𝒜𝒞-𝒟𝒢-𝒢𝒥-𝒦𝒩-𝒬𝒮-𝒹𝒻-𝒻𝒽-𝓃𝓅-𝔅𝔇-𝔊𝔍-𝔔𝔖-𝔜𝔞-𝔹𝔻-𝔾𝕀-𝕄𝕆-𝕆𝕊-𝕐𝕒-𝚥𝚨-𝛀𝛂-𝛚𝛜-𝛺𝛼-𝜔𝜖-𝜴𝜶-𝝎𝝐-𝝮𝝰-𝞈𝞊-𝞨𝞪-𝟂𝟄-𝟋𞠀-𞣄𞸀-𞸃𞸅-𞸟𞸡-𞸢𞸤-𞸤𞸧-𞸧𞸩-𞸲𞸴-𞸷𞸹-𞸹𞸻-𞸻𞹂-𞹂𞹇-𞹇𞹉-𞹉𞹋-𞹋𞹍-𞹏𞹑-𞹒𞹔-𞹔𞹗-𞹗𞹙-𞹙𞹛-𞹛𞹝-𞹝𞹟-𞹟𞹡-𞹢𞹤-𞹤𞹧-𞹪𞹬-𞹲𞹴-𞹷𞹹-𞹼𞹾-𞹾𞺀-𞺉𞺋-𞺛𞺡-𞺣𞺥-𞺩𞺫-𞺻𠀀-𪛖𪜀-𫜴𫝀-𫠝𫠠-𬺡丽-𪘀])+)"
	"^(\\()",                                 // bullet_core-0.1.1: "^(?u:\\()"
	"^(\\))",                                 // bullet_core-0.1.1: "^(?u:\\))"
	"^(\\*)",                                 // bullet_core-0.1.1: "^(?u:\\*)"
	"^(\\+)",                                 // bullet_core-0.1.1: "^(?u:\\+)"
	"^(,)",                                   // bullet_core-0.1.1: "^(?u:,)"
	"^(\\-)",                                 // bullet_core-0.1.1: "^(?u:\\-)"
	"^(/)",                                   // bullet_core-0.1.1: "^(?u:/)"
	"^(\\[)",                                 // bullet_core-0.1.1: "^(?u:\\[)"
	"^(\\])",                                 // bullet_core-0.1.1: "^(?u:\\])"
	"^(\\^)",                                 // bullet_core-0.1.1: "^(?u:\\^)"
	"^(·)",                                   // bullet_core-0.1.1: "^(?u:·)"
	"//+",                                    // actix-web-0.6.13: "//+"
	"//+",                                    // actix-web-0.6.13: "//+"
	`(\S*) .* (\S*) (REACHABLE|STALE|DELAY)`, // althea_kernel_interface-0.1.0: r"(\S*) .* (\S*) (REACHABLE|STALE|DELAY)"
	`-s (.*) --ip6-dst (.*)/.* bcnt = (.*)`,  // althea_kernel_interface-0.1.0: r"-s (.*) --ip6-dst (.*)/.* bcnt = (.*)"
	`\buci(?:\s|$)`,                          // alcibiades-0.3.0: r"\buci(?:\s|$)"
	`\A[a-z0-9._=-]+\z`,                      // ruma-identifiers-0.11.0: r"\A[a-z0-9._=-]+\z"
	`/rusqbins/((?i)[A-F0-9]{8}\-[A-F0-9]{4}\-4[A-F0-9]{3}\-[89AB][A-F0-9]{3}\-[A-F0-9]{12})$`,            // rusqbin-0.2.3: r"/rusqbins/((?i)[A-F0-9]{8}\-[A-F0-9]{4}\-4[A-F0-9]{3}\-[89AB][A-F0-9]{3}\-[A-F0-9]{12})$"
	`/rusqbins/((?i)[A-F0-9]{8}\-[A-F0-9]{4}\-4[A-F0-9]{3}\-[89AB][A-F0-9]{3}\-[A-F0-9]{12})/requests/?$`, // rusqbin-0.2.3: r"/rusqbins/((?i)[A-F0-9]{8}\-[A-F0-9]{4}\-4[A-F0-9]{3}\-[89AB][A-F0-9]{3}\-[A-F0-9]{12})/requests/?$"
	`^(nightly|beta|stable)(?:-(\d{4}-\d{2}-\d{2}))?$`,                                                    // rust-install-0.0.4: r"^(nightly|beta|stable)(?:-(\d{4}-\d{2}-\d{2}))?$"
	"^+(.*)\r\n",                                       // rust_inbox-0.0.5: "^+(.*)\r\n"
	`^\* CAPABILITY (.*)\r\n`,                          // rust_inbox-0.0.5: r"^\* CAPABILITY (.*)\r\n"
	`^([a-zA-Z0-9]+) (OK|NO|BAD)(.*)`,                  // rust_inbox-0.0.5: r"^([a-zA-Z0-9]+) (OK|NO|BAD)(.*)"
	`^\* (\d+) EXISTS\r\n`,                             // rust_inbox-0.0.5: r"^\* (\d+) EXISTS\r\n"
	`^\* (\d+) RECENT\r\n`,                             // rust_inbox-0.0.5: r"^\* (\d+) RECENT\r\n"
	`^\* FLAGS (.+)\r\n`,                               // rust_inbox-0.0.5: r"^\* FLAGS (.+)\r\n"
	`^\* OK \[UNSEEN (\d+)\](.*)\r\n`,                  // rust_inbox-0.0.5: r"^\* OK \[UNSEEN (\d+)\](.*)\r\n"
	`^\* OK \[UIDVALIDITY (\d+)\](.*)\r\n`,             // rust_inbox-0.0.5: r"^\* OK \[UIDVALIDITY (\d+)\](.*)\r\n"
	`^\* OK \[UIDNEXT (\d+)\](.*)\r\n`,                 // rust_inbox-0.0.5: r"^\* OK \[UIDNEXT (\d+)\](.*)\r\n"
	`^\* OK \[PERMANENTFLAGS (.+)\](.*)\r\n`,           // rust_inbox-0.0.5: r"^\* OK \[PERMANENTFLAGS (.+)\](.*)\r\n"
	`^[a-z]+ (\d+)$`,                                   // rustml-0.0.7: r"^[a-z]+ (\d+)$"
	`^[a-z]+ (\d+)$`,                                   // rustml-0.0.7: r"^[a-z]+ (\d+)$"
	`^[a-z]+ (\d+)$`,                                   // rustml-0.0.7: r"^[a-z]+ (\d+)$"
	`([^\\](\\\\)*)\\[\n\r][[:space:]]*`,               // rustfmt-0.10.0: r"([^\\](\\\\)*)\\[\n\r][[:space:]]*"
	`(^\s*$)|(^\s*//\s*rustfmt-[^:]+:\s*\S+)`,          // rustfmt-core-0.4.0: r"(^\s*$)|(^\s*//\s*rustfmt-[^:]+:\s*\S+)"
	"^## `([^`]+)`",                                    // rustfmt-core-0.4.0: r"^## `([^`]+)`"
	`([^\\](\\\\)*)\\[\n\r][[:space:]]*`,               // rustfmt-core-0.4.0: r"([^\\](\\\\)*)\\[\n\r][[:space:]]*"
	`\s;`,                                              // rustfmt-core-0.4.0: r"\s;"
	`^(0x)?([:digit:]+)$`,                              // rust-enum-derive-0.4.0: r"^(0x)?([:digit:]+)$"
	`^([:digit:]+)[:space:]*<<[:space:]*([:digit:]+)$`, // rust-enum-derive-0.4.0: r"^([:digit:]+)[:space:]*<<[:space:]*([:digit:]+)$"
	`^[:space:]*([[:alnum:]_]+)([:space:]*=[:space:]*([:graph:]+))?[:space:]*,`, // rust-enum-derive-0.4.0: r"^[:space:]*([[:alnum:]_]+)([:space:]*=[:space:]*([:graph:]+))?[:space:]*,"
	`^#define[:space:]+([:graph:]+)[:space:]+([:graph:]+)`,                      // rust-enum-derive-0.4.0: r"^#define[:space:]+([:graph:]+)[:space:]+([:graph:]+)"
	`^\s*pub mod (.+);$`,                 // rustsourcebundler-0.2.0: r"^\s*pub mod (.+);$"
	`^\s*pub mod (.+);$`,                 // rustsourcebundler-0.2.0: r"^\s*pub mod (.+);$"
	`([^\\](\\\\)*)\\[\n\r][[:space:]]*`, // rustfmt-nightly-0.8.2: r"([^\\](\\\\)*)\\[\n\r][[:space:]]*"
	`\s;`,                                // rustfmt-nightly-0.8.2: r"\s;"
	`(?s)(.*?)([ \t\r\n]*)(\{\{(\{?\S?\s*?[\w\.\s]*.*?\s*?\}?)\}\})([ \t\r\n]*)`, // rustache-0.1.0: r"(?s)(.*?)([ \t\r\n]*)(\{\{(\{?\S?\s*?[\w\.\s]*.*?\s*?\}?)\}\})([ \t\r\n]*)"
	`_ZN[\$\._[:alnum:]]*`, // rustfilt-0.2.0: r"_ZN[\$\._[:alnum:]]*"
	`(?s)(.*?)([ \t\r\n]*)(\{\{(\{?\S?\s*?[\w\.\s]*.*?\s*?\}?)\}\})([ \t\r\n]*)`, // rustache-lists-0.1.2: r"(?s)(.*?)([ \t\r\n]*)(\{\{(\{?\S?\s*?[\w\.\s]*.*?\s*?\}?)\}\})([ \t\r\n]*)"
	"(.+)=(.+)",      // rural-0.7.3: "(.+)=(.+)"
	"(.*):(.+)",      // rural-0.7.3: "(.*):(.+)"
	"(.+):=(.+)",     // rural-0.7.3: "(.+):=(.+)"
	"(.*)==(.+)",     // rural-0.7.3: "(.*)==(.+)"
	`^\[([^\]]+)\]$`, // rusoto_credential-0.11.0: r"^\[([^\]]+)\]$"
	"([:blank:]*)$",  // rumblebars-0.3.0: "([:blank:]*)$"
	"(\r?\n)[:blank:]*(\\{\\{~?[#!/](?:\\}?[^}])*\\}\\})[:blank:]*(:?\r?\n)?\\z", // rumblebars-0.3.0: "(\r?\n)[:blank:]*(\\{\\{~?[#!/](?:\\}?[^}])*\\}\\})[:blank:]*(:?\r?\n)?\\z"
	"(\r?\n[:blank:]*)(\\{\\{~?>(?:\\}?[^}])*\\}\\})[:blank:]*(:?\r?\n)?\\z",     // rumblebars-0.3.0: "(\r?\n[:blank:]*)(\\{\\{~?>(?:\\}?[^}])*\\}\\})[:blank:]*(:?\r?\n)?\\z"
	"((?:[:blank:]|\r?\n)*)(\r?\n)[:blank:]*$",                                   // rumblebars-0.3.0: "((?:[:blank:]|\r?\n)*)(\r?\n)[:blank:]*$"
	"^([:blank:]*\r?\n)(.*)",      // rumblebars-0.3.0: "^([:blank:]*\r?\n)(.*)"
	`(?P<stamp>[\d-]*)_hello`,     // diesel_cli-1.3.1: r"(?P<stamp>[\d-]*)_hello"
	`(\d+)s`,                      // dishub-0.1.1: r"(\d+)s"
	`\n`,                          // spreadsheet_textconv-0.1.0: r"\n"
	`\r`,                          // spreadsheet_textconv-0.1.0: r"\r"
	`\t`,                          // spreadsheet_textconv-0.1.0: r"\t"
	`DELAY (-?\d+)ms`,             // split_aud-0.1.0: r"DELAY (-?\d+)ms"
	`Trim\((\d+), ?(\d+)\)`,       // split_aud-0.1.0: r"Trim\((\d+), ?(\d+)\)"
	`spotify:[a-z]+:[a-zA-Z0-9]+`, // spotrust-0.0.5: r"spotify:[a-z]+:[a-zA-Z0-9]+"
	`[^\x00-\x7F]`,                // spaceslugs-0.1.0: r"[^\x00-\x7F]"
	`[']+`,                        // spaceslugs-0.1.0: r"[']+"
	`\W+`,                         // spaceslugs-0.1.0: r"\W+"
	`[ ]+`,                        // spaceslugs-0.1.0: r"[ ]+"
	"PHPSESSID=([0-9a-f]+)",       // space_email_api-0.1.1: "PHPSESSID=([0-9a-f]+)"
	"[^0-9.,]",                    // lorikeet-0.7.0: "[^0-9.,]"
	`^(?:\b|(-)?)(\p{L})?((?:(?:\d{1,3}[\.,])+\d{3})|\d+)(?:[\.,](\d{2}))?\b$`, // claude-0.3.0: r"^(?:\b|(-)?)(\p{Currency_Symbol})?((?:(?:\d{1,3}[\.,])+\d{3})|\d+)(?:[\.,](\d{2}))?\b$"
	`<%=\s*(.+?)\s*%>`, // clam-0.1.6: r"<%=\s*(.+?)\s*%>"
	`(\s)`,             // classifier-0.0.3: r"(\s)"
	`(-----BEGIN .*-----\n)((?:(?:[A-Za-z0-9+/]{4})*(?:[A-Za-z0-9+/]{2}==|[A-Za-z0-9+/]{3}=)*\n)+)(-----END .*-----)`, // click-0.3.2: r"(-----BEGIN .*-----\n)((?:(?:[A-Za-z0-9+/]{4})*(?:[A-Za-z0-9+/]{2}==|[A-Za-z0-9+/]{3}=)*\n)+)(-----END .*-----)"
	`-----BEGIN PRIVATE KEY-----`,                               // click-0.3.2: r"-----BEGIN PRIVATE KEY-----"
	`#([A-Z3a-z]*):(.*)`,                                        // ultrastar-txt-0.1.2: r"#([A-Z3a-z]*):(.*)"
	"^-\\s?(-?[0-9]+)\\s*$",                                     // ultrastar-txt-0.1.2: "^-\\s?(-?[0-9]+)\\s*$"
	"^-\\s?(-?[0-9]+)\\s+(-?[0-9]+)",                            // ultrastar-txt-0.1.2: "^-\\s?(-?[0-9]+)\\s+(-?[0-9]+)"
	"^(.)\\s*(-?[0-9]+)\\s+(-?[0-9]+)\\s+(-?[0-9]+)\\s?(.*)",    // ultrastar-txt-0.1.2: "^(.)\\s*(-?[0-9]+)\\s+(-?[0-9]+)\\s+(-?[0-9]+)\\s?(.*)"
	"^P\\s?(-?[0-9]+)",                                          // ultrastar-txt-0.1.2: "^P\\s?(-?[0-9]+)"
	`^template\.add($|\..+$)`,                                   // db-accelerate-2.0.0: r"^template\.add($|\..+$)"
	`^template\.sub($|\..+$)`,                                   // db-accelerate-2.0.0: r"^template\.sub($|\..+$)"
	`(\d+)([cegps])`,                                            // sterling-0.3.0: r"(\d+)([cegps])"
	`[^\w]`,                                                     // stache-0.2.0: r"[^\w]"
	"\"([<>]?)([xcbB\\?hHiIlLqQfdspP]*)\"",                      // strukt-0.1.0: "\"([<>]?)([xcbB\\?hHiIlLqQfdspP]*)\""
	`^STEAM_([0-4]):([0-1]):([0-9]{1,10})$`,                     // steamid-ng-0.3.1: r"^STEAM_([0-4]):([0-1]):([0-9]{1,10})$"
	`^\[([AGMPCgcLTIUai]):([0-4]):([0-9]{1,10})(:([0-9]+))?\]$`, // steamid-ng-0.3.1: r"^\[([AGMPCgcLTIUai]):([0-4]):([0-9]{1,10})(:([0-9]+))?\]$"
	`^\w+`,                            // strscan-0.1.1: r"^\w+"
	`^\s+`,                            // strscan-0.1.1: r"^\s+"
	`^\w+`,                            // strscan-0.1.1: r"^\w+"
	`^\s+`,                            // strscan-0.1.1: r"^\s+"
	`^(\w+)\s+`,                       // strscan-0.1.1: r"^(\w+)\s+"
	`^([a-zA-Z0-9\.-]+)(?:\s+(\d+))$`, // tk-carbon-0.2.0: r"^([a-zA-Z0-9\.-]+)(?:\s+(\d+))$"
	`^([a-zA-Z0-9\.-]+)(?:\s+(\d+))$`, // tk-carbon-0.2.0: r"^([a-zA-Z0-9\.-]+)(?:\s+(\d+))$"
	`extern\s+crate\s+([a-z0-9_]+)\s*;(\s*//(.+))?`, // evalrs-0.0.10: r"extern\s+crate\s+([a-z0-9_]+)\s*;(\s*//(.+))?"
	`(?m)^# `,                           // evalrs-0.0.10: r"(?m)^# "
	`(?m)^\s*fn +main *\( *\)`,          // evalrs-0.0.10: r"(?m)^\s*fn +main *\( *\)"
	`(extern\s+crate\s+[a-z0-9_]+\s*;)`, // evalrs-0.0.10: r"(extern\s+crate\s+[a-z0-9_]+\s*;)"
	"(.*)_t([0-9]+)",                    // gate_build-0.5.0: "(.*)_t([0-9]+)"
	`[^\P{P}-]|\s+-\s+`,                 // rake-0.1.1: r"[^\P{P}-]|\s+-\s+"
	`^.*(?:(?:youtu\.be/|v/|vi/|u/w/|embed/)|(?:(?:watch)?\?v(?:i)?=|\&v(?:i)?=))([^#\&\?]*).*`,                    // rafy-0.2.1: r"^.*(?:(?:youtu\.be/|v/|vi/|u/w/|embed/)|(?:(?:watch)?\?v(?:i)?=|\&v(?:i)?=))([^#\&\?]*).*"
	`^(?P<protocol>.*?)://(?P<public_key>.*?):(?P<secret_key>.*?)@(?P<host>.*?)/(?P<path>.*/)?(?P<project_id>.*)$`, // raven-0.2.1: r"^(?P<protocol>.*?)://(?P<public_key>.*?):(?P<secret_key>.*?)@(?P<host>.*?)/(?P<path>.*/)?(?P<project_id>.*)$"
	`\{[[:space:]]*[^{}]*[[:space:]]*\}`,                           // rargs-0.2.0: r"\{[[:space:]]*[^{}]*[[:space:]]*\}"
	`^\{[[:space:]]*(?P<name>[[:word:]]*)[[:space:]]*\}$`,          // rargs-0.2.0: r"^\{[[:space:]]*(?P<name>[[:word:]]*)[[:space:]]*\}$"
	`^\{[[:space:]]*(?P<num>-?\d+)[[:space:]]*\}$`,                 // rargs-0.2.0: r"^\{[[:space:]]*(?P<num>-?\d+)[[:space:]]*\}$"
	`^\{(?P<left>-?\d*)?\.\.(?P<right>-?\d*)?(?::(?P<sep>.*))?\}$`, // rargs-0.2.0: r"^\{(?P<left>-?\d*)?\.\.(?P<right>-?\d*)?(?::(?P<sep>.*))?\}$"
	`(.*?)[[:space:]]+|(.*?)$`,                                     // rargs-0.2.0: r"(.*?)[[:space:]]+|(.*?)$"
	`[a-zA-Z0-9]{8}`,                                               // indradb-lib-0.15.0: r"[a-zA-Z0-9]{8}"
	`::`,                                                           // fungi-lang-0.1.50: r"::"
	"/hello/(?P<name>[a-zA-Z]+)",                                   // nickel-0.10.1: "/hello/(?P<name>[a-zA-Z]+)"
	"/hello/(?P<name>[a-zA-Z]+)",                                   // nickel-0.10.1: "/hello/(?P<name>[a-zA-Z]+)"
	`\{(\w+)\}`,                                                    // pact_verifier-0.4.0: r"\{(\w+)\}"
	"application/.*json",                                           // pact_matching-0.4.1: "application/.*json"
	"application/json.*",                                           // pact_matching-0.4.1: "application/json.*"
	"application/.*xml",                                            // pact_matching-0.4.1: "application/.*xml"
	"([\"'\\(\\[\\{{<\u201c])(\\s*)(.+?)(\\s*)([\"'\\)\\]\\}}>\u201d])", // pangu-0.2.0: "([\"'\\(\\[\\{{<\u{201c}])(\\s*)(.+?)(\\s*)([\"'\\)\\]\\}}>\u{201d}])"
	"([\\(\\[\\{{<\u201c]+)(\\s*)(.+?)(\\s*)([\\)\\]\\}}>\u201d]+)",     // pangu-0.2.0: "([\\(\\[\\{{<\u{201c}]+)(\\s*)(.+?)(\\s*)([\\)\\]\\}}>\u{201d}]+)"
	`\{-[\s\S]*?-\}`, // parser-haskell-0.2.0: r"\{-[\s\S]*?-\}"
	`(?m);+\s*$`,     // parser-haskell-0.2.0: r"(?m);+\s*$"
	`(?m)^#(if|ifn?def|endif|else|include|elif).*`, // parser-haskell-0.2.0: r"(?m)^#(if|ifn?def|endif|else|include|elif).*"
	`'([^'\\]|\\[A-Z]{1,3}|\\.)'`,                  // parser-haskell-0.2.0: r"'([^'\\]|\\[A-Z]{1,3}|\\.)'"
	`forall\s+(.*?)\.`,                             // parser-haskell-0.2.0: r"forall\s+(.*?)\."
	"\\s{2,}",                                      // html2md-0.2.1: "\\s{2,}"
	"\\n{2,}",                                      // html2md-0.2.1: "\\n{2,}"
	"(?m)(\\S) $",                                  // html2md-0.2.1: "(?m)(\\S) $"
	"(?m)^[-*] ",                                   // html2md-0.2.1: "(?m)^[-*] "
	`#.*$`,                                         // ovpnfile-0.1.2: r"#.*$"
	`^<(\S+)>`,                                     // ovpnfile-0.1.2: r"^<(\S+)>"
	`^</(\S+)>`,                                    // ovpnfile-0.1.2: r"^</(\S+)>"
	`#([:xdigit:]{2})([:xdigit:]{2})([:xdigit:]{2})`,         // screenruster-saver-fractal-0.1.1: r"#([:xdigit:]{2})([:xdigit:]{2})([:xdigit:]{2})"
	`rgb\((?: *(\d{1,3}),)(?: *(\d{1,3}),)(?: *(\d{1,3}))\)`, // scarlet-0.2.2: r"rgb\((?: *(\d{1,3}),)(?: *(\d{1,3}),)(?: *(\d{1,3}))\)"
	`^([\w:]+)<(.+)>$`,                                       // cpp_to_rust_generator-0.2.0: r"^([\w:]+)<(.+)>$"
	`^type-parameter-(\d+)-(\d+)$`,                           // cpp_to_rust_generator-0.2.0: r"^type-parameter-(\d+)-(\d+)$"
	`^([\w~]+)<[^<>]+>$`,                                     // cpp_to_rust_generator-0.2.0: r"^([\w~]+)<[^<>]+>$"
	`(signals|Q_SIGNALS)\s*:`,                                // cpp_to_rust_generator-0.2.0: r"(signals|Q_SIGNALS)\s*:"
	`(slots|Q_SLOTS)\s*:`,                                    // cpp_to_rust_generator-0.2.0: r"(slots|Q_SLOTS)\s*:"
	`(public|protected|private)\s*:`,                         // cpp_to_rust_generator-0.2.0: r"(public|protected|private)\s*:"
	`^([\w:]+)<(.+)>$`,                                       // cpp_to_rust-0.5.3: r"^([\w:]+)<(.+)>$"
	`^type-parameter-(\d+)-(\d+)$`,                           // cpp_to_rust-0.5.3: r"^type-parameter-(\d+)-(\d+)$"
	`^([\w~]+)<[^<>]+>$`,                                     // cpp_to_rust-0.5.3: r"^([\w~]+)<[^<>]+>$"
	`(signals|Q_SIGNALS)\s*:`,                                // cpp_to_rust-0.5.3: r"(signals|Q_SIGNALS)\s*:"
	`(slots|Q_SLOTS)\s*:`,                                    // cpp_to_rust-0.5.3: r"(slots|Q_SLOTS)\s*:"
	`(public|protected|private)\s*:`,                         // cpp_to_rust-0.5.3: r"(public|protected|private)\s*:"
	"(\\d{2}\\.\\d{2}\\.\\d{2}) (\\d{2}:\\d{2}:\\d{2}) (.*)", // fritzbox_logs-0.2.0: "(\\d{2}\\.\\d{2}\\.\\d{2}) (\\d{2}:\\d{2}:\\d{2}) (.*)"
	`mxc://(?P<server>[^/]+)/(?P<media>.+)`,                  // fractal-matrix-api-3.29.0: r"mxc://(?P<server>[^/]+)/(?P<media>.+)"
	`^api-[a-zA-Z0-9]{32}$`,                                  // smtp2go-0.1.4: r"^api-[a-zA-Z0-9]{32}$"
	`^[-a-zA-Z0-9_=@,.;]+$`,                                  // pusher-0.3.1: r"^[-a-zA-Z0-9_=@,.;]+$"
	`\A\d+\.\d+\z`,                                           // pusher-0.3.1: r"\A\d+\.\d+\z"
	`^\.(.+?) +?(.+)$`,                                       // bakervm-0.9.0: r"^\.(.+?) +?(.+)$"
	`^\.([^\s]+)$`,                                           // bakervm-0.9.0: r"^\.([^\s]+)$"
	`^include! +([^\s]+)$`,                                   // bakervm-0.9.0: r"^include! +([^\s]+)$"
	`^@(\d+)$`,                                               // bakervm-0.9.0: r"^@(\d+)$"
	`^true|false$`,                                           // bakervm-0.9.0: r"^true|false$"
	`^(-?\d+)?\.[0-9]+$`,                                     // bakervm-0.9.0: r"^(-?\d+)?\.[0-9]+$"
	`^(-?\d+)?$`,                                             // bakervm-0.9.0: r"^(-?\d+)?$"
	`^#([0-9abcdefABCDEF]{6})$`,                              // bakervm-0.9.0: r"^#([0-9abcdefABCDEF]{6})$"
	`^'(.)'$`,                                                // bakervm-0.9.0: r"^'(.)'$"
	`^\$vi\((\d+)\)$`,                                        // bakervm-0.9.0: r"^\$vi\((\d+)\)$"
	`^\$key\((\d+)\)$`,                                       // bakervm-0.9.0: r"^\$key\((\d+)\)$"
	"(?P<type>[A-Z^']+) (?P<route>[^']+) HTTP/(?P<http>[^']+)", // banana-0.0.2: "(?P<type>[A-Z^']+) (?P<route>[^']+) HTTP/(?P<http>[^']+)"
	`[A-F0-9]{8}`, // serial-key-2.0.0: r"[A-F0-9]{8}"
	// serde-hjson-0.8.1: "[\\\\\"\x00-\x1f\x7f-\u{9f}\u{00ad}\u{0600}-\u{0604}\u{070f}\u{17b4}\u{17b5}\u{200c}-\u{200f}\u{2028}-\u{202f}\u{2060}-\u{206f}\u{feff}\u{fff0}-\u{ffff}]"
	// serde-hjson-0.8.1: "[\x00-\x1f\x7f-\u{9f}\u{00ad}\u{0600}-\u{0604}\u{070f}\u{17b4}\u{17b5}\u{200c}-\u{200f}\u{2028}-\u{202f}\u{2060}-\u{206f}\u{feff}\u{fff0}-\u{ffff}]"
	// serde-hjson-0.8.1: "'''|[\x00-\x09\x0b\x0c\x0e-\x1f\x7f-\u{9f}\u{00ad}\u{0600}-\u{0604}\u{070f}\u{17b4}\u{17b5}\u{200c}-\u{200f}\u{2028}-\u{202f}\u{2060}-\u{206f}\u{feff}\u{fff0}-\u{ffff}]"
	`/todos/(?P<id>\d+)`,                  // serde-odbc-0.1.0: r"/todos/(?P<id>\d+)"
	`^(?:_<)?([a-zA-Z0-9_]+?)(?:\.\.|::)`, // sentry-0.6.0: r"^(?:_<)?([a-zA-Z0-9_]+?)(?:\.\.|::)"
	`[^a-zA-Z0 -]+`,                       // sentiment-0.1.1: r"[^a-zA-Z0 -]+"
	` {2,}`,                               // sentiment-0.1.1: r" {2,}"
	`(?m)//.*`,                            // verilog-0.0.1: r"(?m)//.*"
	"(?P<robot>C3PO)",                     // verex-0.2.2: "(?P<robot>C3PO)"
	">|<|\"|&",                            // handlebars-0.32.4: ">|<|\"|&"
	`^\w+-\w+-[0123456789]{4}$`,           // haikunator-0.1.2: r"^\w+-\w+-[0123456789]{4}$"
	`^\w+@\w+@[0123456789]{4}$`,           // haikunator-0.1.2: r"^\w+@\w+@[0123456789]{4}$"
	`^\w+-\w+-[0123456789abcdef]{4}$`,     // haikunator-0.1.2: r"^\w+-\w+-[0123456789abcdef]{4}$"
	`^\w+-\w+-[0123456789忠犬ハチ公]{10}$`,     // haikunator-0.1.2: r"^\w+-\w+-[0123456789忠犬ハチ公]{10}$"
	`^\w+-\w+$`,                           // haikunator-0.1.2: r"^\w+-\w+$"
	`^\w+-\w+-[foo]{4}$`,                  // haikunator-0.1.2: r"^\w+-\w+-[foo]{4}$"
	`^\w+-\w+-[0123456789忠犬ハチ公]{5}$`,      // haikunator-0.1.2: r"^\w+-\w+-[0123456789忠犬ハチ公]{5}$"
	`(.*)`,                       // bobbin-cli-0.8.3: r"(.*)"
	`rustc (.*)`,                 // bobbin-cli-0.8.3: r"rustc (.*)"
	`cargo (.*)`,                 // bobbin-cli-0.8.3: r"cargo (.*)"
	`xargo (.*)\n`,               // bobbin-cli-0.8.3: r"xargo (.*)\n"
	`Open On-Chip Debugger (.*)`, // bobbin-cli-0.8.3: r"Open On-Chip Debugger (.*)"
	`arm-none-eabi-gcc \(GNU Tools for ARM Embedded Processors[^\)]*\) (.*)`, // bobbin-cli-0.8.3: r"arm-none-eabi-gcc \(GNU Tools for ARM Embedded Processors[^\)]*\) (.*)"
	`(?m).*\nBasic Open Source SAM-BA Application \(BOSSA\) Version (.*)\n`,  // bobbin-cli-0.8.3: r"(?m).*\nBasic Open Source SAM-BA Application \(BOSSA\) Version (.*)\n"
	`(?m)SEGGER J-Link Commander (.*)\n`,                                     // bobbin-cli-0.8.3: r"(?m)SEGGER J-Link Commander (.*)\n"
	`(?m)Teensy Loader, Command Line, Version (.*)\n`,                        // bobbin-cli-0.8.3: r"(?m)Teensy Loader, Command Line, Version (.*)\n"
	`dfu-util (.*)\n`,                      // bobbin-cli-0.8.3: r"dfu-util (.*)\n"
	`^/static/[\w.]+$`,                     // borsholder-0.9.1: r"^/static/[\w.]+$"
	`^/timeline/([0-9]+)$`,                 // borsholder-0.9.1: r"^/timeline/([0-9]+)$"
	"\u001B\\[[\\d;]*[^\\d;]",              // fblog-1.0.1: "\u{001B}\\[[\\d;]*[^\\d;]"
	"\u001B\\[[\\d;]*[^\\d;]",              // fblog-1.0.1: "\u{001B}\\[[\\d;]*[^\\d;]"
	`^\[\d+\]$`,                            // toml-query-0.6.0: r"^\[\d+\]$"
	` (?P<key>[^\s]+):(?P<value>[^\s^/]+)`, // todo-txt-1.1.0: r" (?P<key>[^\s]+):(?P<value>[^\s^/]+)"
	`\band\b`,                              // findr-0.1.5: r"\band\b"
	`\bor\b`,                               // findr-0.1.5: r"\bor\b"
	`\bnot\b`,                              // findr-0.1.5: r"\bnot\b"
	`.*?\.(a|la|lo|o|ll|keter|bc|dyn_o|out|d|rlib|crate|min\.js|hi|dyn_hi|S|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$`, // file-sniffer-3.0.1: r".*?\.(a|la|lo|o|ll|keter|bc|dyn_o|out|d|rlib|crate|min\.js|hi|dyn_hi|S|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$"
	`.*?\.(stats|conf|h|cache.*|dat|pc|info)$`, // file-sniffer-3.0.1: r".*?\.(stats|conf|h|cache.*|dat|pc|info)$"
	`.*?\.(exe|a|la|o|ll|keter|bc|dyn_o|out|d|rlib|crate|min\.js|hi|dyn_hi|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$`, // file-sniffer-3.0.1: r".*?\.(exe|a|la|o|ll|keter|bc|dyn_o|out|d|rlib|crate|min\.js|hi|dyn_hi|jsexe|webapp|js\.externs|ibc|toc|aux|fdb_latexmk|fls|egg-info|whl|js_a|js_hi|jld|ji|js_o|so.*|dump-.*|vmb|crx|orig|elmo|elmi|pyc|mod|p_hi|p_o|prof|tix)$"
	`.*?\.(stats|conf|h|cache.*)$`,  // file-sniffer-3.0.1: r".*?\.(stats|conf|h|cache.*)$"
	`(\.git|\.pijul|_darcs|\.hg)$`,  // file-sniffer-3.0.1: r"(\.git|\.pijul|_darcs|\.hg)$"
	"test",                          // file_logger-0.1.0: "test"
	`foo`,                           // file_scanner-0.2.0: r"foo"
	`a+b`,                           // file_scanner-0.2.0: r"a+b"
	`a[ab]*b`,                       // file_scanner-0.2.0: r"a[ab]*b"
	`\s+`,                           // file_scanner-0.2.0: r"\s+"
	`\s+`,                           // file_scanner-0.2.0: r"\s+"
	`^\s*([^\s]+) %cellsplit<\d+>$`, // cellsplit-0.2.1: r"^\s*([^\s]+) %cellsplit<\d+>$"
	`^\s*([^\s]+) %cellsplit<\d+>$`, // cellsplit-0.2.1: r"^\s*([^\s]+) %cellsplit<\d+>$"
	`^[+\-]?[0-9]+`,                 // aterm-0.20.0: r"^[+\-]?[0-9]+"
	`^[+\-]?[0-9]+\.[0-9]*([eE][+\-]?[0-9]+)?`, // aterm-0.20.0: r"^[+\-]?[0-9]+\.[0-9]*([eE][+\-]?[0-9]+)?"
	`^[*] OK`,                          // atarashii_imap-0.3.0: r"^[*] OK"
	`FLAGS\s\((.+)\)`,                  // atarashii_imap-0.3.0: r"FLAGS\s\((.+)\)"
	`\[PERMANENTFLAGS\s\((.+)\)\]`,     // atarashii_imap-0.3.0: r"\[PERMANENTFLAGS\s\((.+)\)\]"
	`\[UIDVALIDITY\s(\d+)\]`,           // atarashii_imap-0.3.0: r"\[UIDVALIDITY\s(\d+)\]"
	`(\d+)\sEXISTS`,                    // atarashii_imap-0.3.0: r"(\d+)\sEXISTS"
	`(\d+)\sRECENT`,                    // atarashii_imap-0.3.0: r"(\d+)\sRECENT"
	`\[UNSEEN\s(\d+)\]`,                // atarashii_imap-0.3.0: r"\[UNSEEN\s(\d+)\]"
	`\[UIDNEXT\s(\d+)\]`,               // atarashii_imap-0.3.0: r"\[UIDNEXT\s(\d+)\]"
	`\\(\{|\})`,                        // editorconfig-1.0.0: r"\\(\{|\})"
	`(^|[^\\])\\\|`,                    // editorconfig-1.0.0: r"(^|[^\\])\\\|"
	`\[([^\]]*)$`,                      // editorconfig-1.0.0: r"\[([^\]]*)$"
	`\[(.*/.*)\]`,                      // editorconfig-1.0.0: r"\[(.*/.*)\]"
	`\{(-?\d+\\\.\\\.-?\d+)\}`,         // editorconfig-1.0.0: r"\{(-?\d+\\\.\\\.-?\d+)\}"
	`\{([^,]+)\}`,                      // editorconfig-1.0.0: r"\{([^,]+)\}"
	`\{(([^\}].*)?(,|\|)(.*[^\\])?)\}`, // editorconfig-1.0.0: r"\{(([^\}].*)?(,|\|)(.*[^\\])?)\}"
	`^/`,                               // editorconfig-1.0.0: r"^/"
	`(^|[^\\])(\{|\})`,                 // editorconfig-1.0.0: r"(^|[^\\])(\{|\})"
	"^#!.*\n",                          // edmunge-1.0.0: "^#!.*\n"
	`\\N\{(.*?)(?:\}|$)`,               // unicode_names2_macros-0.2.0: r"\\N\{(.*?)(?:\}|$)"
	`^--- (?P<filename>[^\t\n]+)(?:\t(?P<timestamp>[^\n]+))?`,                           // unidiff-0.2.1: r"^--- (?P<filename>[^\t\n]+)(?:\t(?P<timestamp>[^\n]+))?"
	`^\+\+\+ (?P<filename>[^\t\n]+)(?:\t(?P<timestamp>[^\n]+))?`,                        // unidiff-0.2.1: r"^\+\+\+ (?P<filename>[^\t\n]+)(?:\t(?P<timestamp>[^\n]+))?"
	`^@@ -(\d+)(?:,(\d+))? \+(\d+)(?:,(\d+))? @@[ ]?(.*)`,                               // unidiff-0.2.1: r"^@@ -(\d+)(?:,(\d+))? \+(\d+)(?:,(\d+))? @@[ ]?(.*)"
	`^(?P<line_type>[- \n\+\\]?)(?P<value>.*)`,                                          // unidiff-0.2.1: r"^(?P<line_type>[- \n\+\\]?)(?P<value>.*)"
	"/?(?P<zoom>[0-9]?[0-9])/(?P<x>[0-9]{1,10})/(?P<y>[0-9]{1,10})(\\.[a-zA-Z]{3,4})?$", // slippy-map-tiles-0.13.1: "/?(?P<zoom>[0-9]?[0-9])/(?P<x>[0-9]{1,10})/(?P<y>[0-9]{1,10})(\\.[a-zA-Z]{3,4})?$"
	`^(?P<minlon>-?[0-9]{1,3}(\.[0-9]{1,10})?) (?P<minlat>-?[0-9]{1,3}(\.[0-9]{1,10})?) (?P<maxlon>-?[0-9]{1,3}(\.[0-9]{1,10})?) (?P<maxlat>-?[0-9]{1,3}(\.[0-9]{1,10})?)$`, // slippy-map-tiles-0.13.1: r"^(?P<minlon>-?[0-9]{1,3}(\.[0-9]{1,10})?) (?P<minlat>-?[0-9]{1,3}(\.[0-9]{1,10})?) (?P<maxlon>-?[0-9]{1,3}(\.[0-9]{1,10})?) (?P<maxlat>-?[0-9]{1,3}(\.[0-9]{1,10})?)$"
	`^(?P<minlon>-?[0-9]{1,3}(\.[0-9]{1,10})?),(?P<minlat>-?[0-9]{1,3}(\.[0-9]{1,10})?),(?P<maxlon>-?[0-9]{1,3}(\.[0-9]{1,10})?),(?P<maxlat>-?[0-9]{1,3}(\.[0-9]{1,10})?)$`, // slippy-map-tiles-0.13.1: r"^(?P<minlon>-?[0-9]{1,3}(\.[0-9]{1,10})?),(?P<minlat>-?[0-9]{1,3}(\.[0-9]{1,10})?),(?P<maxlon>-?[0-9]{1,3}(\.[0-9]{1,10})?),(?P<maxlat>-?[0-9]{1,3}(\.[0-9]{1,10})?)$"
	`^https?://(.+?):1400/xml`,            // sonos-0.1.2: r"^https?://(.+?):1400/xml"
	`^[a-z]{2}$`,                          // validator_derive-0.7.0: r"^[a-z]{2}$"
	`[a-z]{2}`,                            // validator_derive-0.7.0: r"[a-z]{2}"
	`[a-z]{2}`,                            // validator_derive-0.7.0: r"[a-z]{2}"
	`one of \d+ options`,                  // nginx-config-0.8.0: r"one of \d+ options"
	`[\s,]`,                               // waltz-0.4.0: r"[\s,]"
	`^aws_access_key_id = (.*)`,           // warheadhateus-0.2.1: r"^aws_access_key_id = (.*)"
	`^aws_secret_access_key = (.*)`,       // warheadhateus-0.2.1: r"^aws_secret_access_key = (.*)"
	`^aws_access_key_id = (.*)`,           // warheadhateus-0.2.1: r"^aws_access_key_id = (.*)"
	`^aws_secret_access_key = (.*)`,       // warheadhateus-0.2.1: r"^aws_secret_access_key = (.*)"
	"([\u4E00-\u9FD5a-zA-Z0-9+#&\\._%]+)", // jieba-rs-0.2.2: r"([\u{4E00}-\u{9FD5}a-zA-Z0-9+#&\._%]+)"
	`(\r\n|\s)`,                           // jieba-rs-0.2.2: r"(\r\n|\s)"
	"([\u4E00-\u9FD5]+)",                  // jieba-rs-0.2.2: "([\u{4E00}-\u{9FD5}]+)"
	`[^a-zA-Z0-9+#\n]`,                    // jieba-rs-0.2.2: r"[^a-zA-Z0-9+#\n]"
	"([\u4E00-\u9FD5]+)",                  // jieba-rs-0.2.2: r"([\u{4E00}-\u{9FD5}]+)"
	`([a-zA-Z0-9]+(?:.\d+)?%?)`,           // jieba-rs-0.2.2: r"([a-zA-Z0-9]+(?:.\d+)?%?)"
	`Span\([0-9 ,]*\)`,                    // lalrpop-0.15.2: r"Span\([0-9 ,]*\)"
	`Span\([0-9 ,]*\)`,                    // lalrpop-snap-0.15.2: r"Span\([0-9 ,]*\)"
	`[\S]+`,                               // nlp-tokenize-0.1.0: r"[\S]+"
	"[[:xdigit:]][70]",                    // kbgpg-0.1.2: "[[:xdigit:]][70]"
	`^((?P<address>.*):)?(?P<port>\d+)$`,  // cdbd-0.1.1: r"^((?P<address>.*):)?(?P<port>\d+)$"
	`[\w\s=+-/]+\((\{(.|\n)*\})\);?`,      // mbutiles-0.1.1: r"[\w\s=+-/]+\((\{(.|\n)*\})\);?"
	`^-\d+(?:ms|s|m|h|d|w|y)?$`,           // extrahop-0.2.5: r"^-\d+(?:ms|s|m|h|d|w|y)?$"
	"^((?:.*)-)?ss(0|[1-9][0-9]*)\\.pip$", // pippin-0.1.0: "^((?:.*)-)?ss(0|[1-9][0-9]*)\\.pip$"
	"^((?:.*)-)?ss(0|[1-9][0-9]*)-cl(0|[1-9][0-9]*)\\.piplog$",                       // pippin-0.1.0: "^((?:.*)-)?ss(0|[1-9][0-9]*)-cl(0|[1-9][0-9]*)\\.piplog$"
	"^((?:.*)-)?ss(0|[1-9][0-9]*)\\.pip$",                                            // pippin-0.1.0: "^((?:.*)-)?ss(0|[1-9][0-9]*)\\.pip$"
	"^((?:.*)-)?ss(0|[1-9][0-9]*)-cl(0|[1-9][0-9]*)\\.piplog$",                       // pippin-0.1.0: "^((?:.*)-)?ss(0|[1-9][0-9]*)-cl(0|[1-9][0-9]*)\\.piplog$"
	"^.*pn(0|[1-9][0-9]*)(-ss(0|[1-9][0-9]*)(\\.pip|-cl(0|[1-9][0-9]*)\\.piplog))?$", // pippin-0.1.0: "^.*pn(0|[1-9][0-9]*)(-ss(0|[1-9][0-9]*)(\\.pip|-cl(0|[1-9][0-9]*)\\.piplog))?$"
	"^(.*)-ss(?:0|[1-9][0-9]*)(?:\\.pip|-cl(?:0|[1-9][0-9]*)\\.piplog)$",             // pippin-0.1.0: "^(.*)-ss(?:0|[1-9][0-9]*)(?:\\.pip|-cl(?:0|[1-9][0-9]*)\\.piplog)$"
	`(?i)[āáǎàēéěèōóǒòīíǐìūúǔùüǘǚǜńň]`,                                              // pinyin-0.3.0: r"(?i)[āáǎàēéěèōóǒòīíǐìūúǔùüǘǚǜńň]"
	`([aeoiuvnm])([0-4])$`,           // pinyin-0.3.0: r"([aeoiuvnm])([0-4])$"
	`(?P<value>\d+)(?P<units>[a-z])`, // duration-parser-0.2.0: r"(?P<value>\d+)(?P<units>[a-z])"
	`^\d+\D?$`,                       // dutree-0.2.7: r"^\d+\D?$"
	`^[A-Za-z0-9]*$`,                 // djangohashers-0.3.0: r"^[A-Za-z0-9]*$"
	`^[A-Z][A-Z0-9]{2,}$`,            // rtag-0.3.5: r"^[A-Z][A-Z0-9]{2,}$"
	`^http://www\.emusic\.com`,       // rtag-0.3.5: r"^http://www\.emusic\.com"
	`^[A-Z][A-Z0-9]{2,}`,             // rtag-0.3.5: r"^[A-Z][A-Z0-9]{2,}"
	`(^[\x{0}|\x{feff}|\x{fffe}]*|[\x{0}|\x{feff}|\x{fffe}]*$)`, // rtag-0.3.5: r"(^[\x{0}|\x{feff}|\x{fffe}]*|[\x{0}|\x{feff}|\x{fffe}]*$)"
	`(\d+)[xX](\d+)`,                  // rtow-0.1.0: r"(\d+)[xX](\d+)"
	`\$([a-zA-Z0-9_]+)`,               // pleingres-sql-plugin-0.1.0: r"\$([a-zA-Z0-9_]+)"
	"[\\n]+",                          // dono-2.0.0: "[\\n]+"
	"(?m)^\\n",                        // dono-2.0.0: "(?m)^\\n"
	"(?m)^\\n",                        // dono-2.0.0: "(?m)^\\n"
	`^[0-9A-Za-z\+/]{43}=\.ed25519$`,  // ssb-common-0.3.0: r"^[0-9A-Za-z\+/]{43}=\.ed25519$"
	`^[0-9A-Za-z\+/]{86}==\.ed25519$`, // ssb-common-0.3.0: r"^[0-9A-Za-z\+/]{86}==\.ed25519$"
	`^[0-9A-Za-z\+/]{43}=\.sha256$`,   // ssb-common-0.3.0: r"^[0-9A-Za-z\+/]{43}=\.sha256$"
	`^(?P<major>\d+)\.(?P<minor>\d+)(?:\.(?P<patch>\d+))?(?:(?P<pre0>[a-z]+)(?P<pre1>\d*))?$`, // mozversion-0.1.3: r"^(?P<major>\d+)\.(?P<minor>\d+)(?:\.(?P<patch>\d+))?(?:(?P<pre0>[a-z]+)(?P<pre1>\d*))?$"
	`^(\d+)\.(\d+)$`,          // monger-0.5.6: r"^(\d+)\.(\d+)$"
	`^[rv]2\.6`,               // mongo_rub-0.0.2: r"^[rv]2\.6"
	"body value",              // flow-0.3.5: "body value"
	"start marker",            // flow-0.3.5: "start marker"
	"end marker",              // flow-0.3.5: "end marker"
	"body value",              // flow-0.3.5: "body value"
	"^([A-Za-z/ ]+): (.*)",    // vobsub-0.2.3: "^([A-Za-z/ ]+): (.*)"
	`#([^\s=]+)*`,             // voidmap-1.1.2: r"#([^\s=]+)*"
	`#(\S+)*`,                 // voidmap-1.1.2: r"#(\S+)*"
	`#prio=(\d+)`,             // voidmap-1.1.2: r"#prio=(\d+)"
	`\[(\S+)\]`,               // voidmap-1.1.2: r"\[(\S+)\]"
	`#limit=(\d+)`,            // voidmap-1.1.2: r"#limit=(\d+)"
	`#tagged=(\S+)`,           // voidmap-1.1.2: r"#tagged=(\S+)"
	`#rev\b`,                  // voidmap-1.1.2: r"#rev\b"
	`#done\b`,                 // voidmap-1.1.2: r"#done\b"
	`#open\b`,                 // voidmap-1.1.2: r"#open\b"
	`#since=(\S+)`,            // voidmap-1.1.2: r"#since=(\S+)"
	`#until=(\S+)`,            // voidmap-1.1.2: r"#until=(\S+)"
	`#plot=(\S+)`,             // voidmap-1.1.2: r"#plot=(\S+)"
	`#n=(\d+)`,                // voidmap-1.1.2: r"#n=(\d+)"
	`(\S+)`,                   // voidmap-1.1.2: r"(\S+)"
	`(?P<y>\d+)y`,             // voidmap-1.1.2: r"(?P<y>\d+)y"
	`(?P<m>\d+)m`,             // voidmap-1.1.2: r"(?P<m>\d+)m"
	`(?P<w>\d+)w`,             // voidmap-1.1.2: r"(?P<w>\d+)w"
	`(?P<d>\d+)d`,             // voidmap-1.1.2: r"(?P<d>\d+)d"
	`(?P<h>\d+)h`,             // voidmap-1.1.2: r"(?P<h>\d+)h"
	`C-(.)`,                   // voidmap-1.1.2: r"C-(.)"
	`^\.\./qt[^/]+/`,          // qt_generator-0.2.0: r"^\.\./qt[^/]+/"
	"(href|src)=\"([^\"]*)\"", // qt_generator-0.2.0: "(href|src)=\"([^\"]*)\""
	`[01]{5}`,                 // kryptos-0.6.1: r"[01]{5}"
	"data_batch_[1-5].bin",    // cifar_10_loader-0.2.0: "data_batch_[1-5].bin"
	"test_batch.bin",          // cifar_10_loader-0.2.0: "test_batch.bin"
	`^\d+.\d+s$`,              // circadian-0.6.0: r"^\d+.\d+s$"
	`^\d+:\d+$`,               // circadian-0.6.0: r"^\d+:\d+$"
	`^\d+:\d+m$`,              // circadian-0.6.0: r"^\d+:\d+m$"
	`!!`,                      // cicada-0.8.1: r"!!"
	"^([^`]*)`([^`]+)`(.*)$",  // cicada-0.8.1: r"^([^`]*)`([^`]+)`(.*)$"
	`\*+`,                     // cicada-0.8.1: r"\*+"
	`([^\$]*)\$\{?([A-Za-z0-9\?\$_]+)\}?(.*)`, // cicada-0.8.1: r"([^\$]*)\$\{?([A-Za-z0-9\?\$_]+)\}?(.*)"
	`^ *alias +([a-zA-Z0-9_\.-]+)=(.*)$`,      // cicada-0.8.1: r"^ *alias +([a-zA-Z0-9_\.-]+)=(.*)$"
	`hi`,                                      // vterm-sys-0.1.0: r"hi"
	`.*?\t`,                                   // skim-0.5.0: r".*?\t"
	`.*?[\t ]`,                                // skim-0.5.0: r".*?[\t ]"
	`(\{-?[0-9.,q]*?})`,                       // skim-0.5.0: r"(\{-?[0-9.,q]*?})"
	`[ \t\n]+`,                                // skim-0.5.0: r"[ \t\n]+"
	`[ \t\n]+`,                                // skim-0.5.0: r"[ \t\n]+"
	`([^ |]+( +\| +[^ |]*)+)|( +)`,            // skim-0.5.0: r"([^ |]+( +\| +[^ |]*)+)|( +)"
	` +\| +`,                                  // skim-0.5.0: r" +\| +"
	`^(?P<left>-?\d+)?(?P<sep>\.\.)?(?P<right>-?\d+)?$`, // skim-0.5.0: r"^(?P<left>-?\d+)?(?P<sep>\.\.)?(?P<right>-?\d+)?$"
	",",    // skim-0.5.0: ","
	".*?,", // skim-0.5.0: ".*?,"
	".*?,", // skim-0.5.0: ".*?,"
	",",    // skim-0.5.0: ","
	`\x1B\[(?:([0-9]+;[0-9]+[Hf])|([0-9]+[ABCD])|(s|u|2J|K)|([0-9;]*m)|(=[0-9]+[hI]))`, // skim-0.5.0: r"\x1B\[(?:([0-9]+;[0-9]+[Hf])|([0-9]+[ABCD])|(s|u|2J|K)|([0-9;]*m)|(=[0-9]+[hI]))"
	`[-_./]\z`,                                // egg-mode-text-1.14.7: r"[-_./]\z"
	"^[ \t\r\n\x0c]*[#!]",                     // java-properties-1.1.1: "^[ \t\r\n\x0c]*[#!]"
	`^[ \t\x0c]*[#!][^\r\n]*$`,                // java-properties-1.1.1: r"^[ \t\x0c]*[#!][^\r\n]*$"
	`^([ \t\x0c]*[:=][ \t\x0c]*|[ \t\x0c]+)$`, // java-properties-1.1.1: r"^([ \t\x0c]*[:=][ \t\x0c]*|[ \t\x0c]+)$"
	`:.+\.`,                      // ipaddress-0.1.2: r":.+\."
	`\.`,                         // ipaddress-0.1.2: r"\."
	`:`,                          // ipaddress-0.1.2: r":"
	`v(\d+)\.(\d+)\.(\d+)`,       // iptables-0.2.2: r"v(\d+)\.(\d+)\.(\d+)"
	`^([^-]+)-(.*)\.dat\.gz$`,    // rsure-0.8.1: r"^([^-]+)-(.*)\.dat\.gz$"
	"^(.*?)(<=|<|==|>=|>)(.*?)$", // rs-jsonpath-0.1.0: "^(.*?)(<=|<|==|>=|>)(.*?)$"
	`(\n|^)(\w+):([\n\w\W]+?)(\n(?:\w)|(\n\]))`, // oatie-0.3.0: r"(\n|^)(\w+):([\n\w\W]+?)(\n(?:\w)|(\n\]))"
	"#.*$",                           // weld-0.2.0: "#.*$"
	`^[A-Za-z$_][A-Za-z0-9$_]*$`,     // weld-0.2.0: r"^[A-Za-z$_][A-Za-z0-9$_]*$"
	`^[0-9]+[cC]$`,                   // weld-0.2.0: r"^[0-9]+[cC]$"
	`^0b[0-1]+[cC]$`,                 // weld-0.2.0: r"^0b[0-1]+[cC]$"
	`^0x[0-9a-fA-F]+[cC]$`,           // weld-0.2.0: r"^0x[0-9a-fA-F]+[cC]$"
	`^[0-9]+$`,                       // weld-0.2.0: r"^[0-9]+$"
	`^0b[0-1]+$`,                     // weld-0.2.0: r"^0b[0-1]+$"
	`^0x[0-9a-fA-F]+$`,               // weld-0.2.0: r"^0x[0-9a-fA-F]+$"
	`^[0-9]+[lL]$`,                   // weld-0.2.0: r"^[0-9]+[lL]$"
	`^0b[0-1]+[lL]$`,                 // weld-0.2.0: r"^0b[0-1]+[lL]$"
	`^0x[0-9a-fA-F]+[lL]$`,           // weld-0.2.0: r"^0x[0-9a-fA-F]+[lL]$"
	"([(, ])enum\\b",                 // webgl_generator-0.1.0: "([(, ])enum\\b"
	"\\bAcquireResourcesCallback\\b", // webgl_generator-0.1.0: "\\bAcquireResourcesCallback\\b"
	`^(\d+)(,(\d+))?([acd]).*$`,      // weave-0.2.0: r"^(\d+)(,(\d+))?([acd]).*$"
	`<BinaryState>(\d)(\|-?\d+)*</BinaryState>`,      // wemo-0.0.12: r"<BinaryState>(\d)(\|-?\d+)*</BinaryState>"
	`(http[s]?://[^\s]+)`,                            // webscale-0.9.4: r"(http[s]?://[^\s]+)"
	`^\d+.*$`,                                        // svgrep-1.1.0: r"^\d+.*$"
	`^[\pL\pN]+$`,                                    // ignore-0.4.2: r"^[\pL\pN]+$"
	`^([A-Za-z][0-9A-Za-z_]*)?$`,                     // ommui_string_patterns-0.1.2: r"^([A-Za-z][0-9A-Za-z_]*)?$"
	`^(\S+(?:.*\S)?)?$`,                              // ommui_string_patterns-0.1.2: r"^(\S+(?:.*\S)?)?$"
	"^(?P<min>[0-9]{1,10})(:(?P<max>[0-9]{1,10}))?$", // opcua-types-0.3.0: "^(?P<min>[0-9]{1,10})(:(?P<max>[0-9]{1,10}))?$"
	`^(ns=(?P<ns>[0-9]+);)?(?P<t>[isgb])=(?P<v>.+)$`, // opcua-types-0.3.0: r"^(ns=(?P<ns>[0-9]+);)?(?P<t>[isgb])=(?P<v>.+)$"
	`^(.+?)\s*:\s*(.+)$`,                             // open_read_later-1.1.1: r"^(.+?)\s*:\s*(.+)$"
	`^.*(?:(?:youtu\.be/|v/|vi/|u/w/|embed/)|(?:(?:watch)?\?v(?:i)?=|\&v(?:i)?=))([^#\&\?]*).*`, // youtube-downloader-0.1.0: r"^.*(?:(?:youtu\.be/|v/|vi/|u/w/|embed/)|(?:(?:watch)?\?v(?:i)?=|\&v(?:i)?=))([^#\&\?]*).*"
	".",                       // yobot-0.1.1: "."
	`.`,                       // yobot-0.1.1: r"."
	`.+`,                      // yobot-0.1.1: r".+"
	`.`,                       // yobot-0.1.1: r"."
	`foo`,                     // ubiquity-0.1.5: r"foo"
	`/target/`,                // ubiquity-0.1.5: r"/target/"
	`.DS_Store`,               // ubiquity-0.1.5: r".DS_Store"
	`//.*`,                    // qasm-1.0.0: r"//.*"
	`\{\{ *([a-z\._]+) *\}\}`, // drill-0.3.5: r"\{\{ *([a-z\._]+) *\}\}"
	`^([^\]\[]+)`,             // queryst-2.0.0: r"^([^\]\[]+)"
	`(\[[^\]\[]*\])`,          // queryst-2.0.0: r"(\[[^\]\[]*\])"
	`^/(\w+)$`,                // qui-vive-0.1.0: r"^/(\w+)$"
	`^/key$`,                  // qui-vive-0.1.0: r"^/key$"
	`^/key/(\w+)$`,            // qui-vive-0.1.0: r"^/key/(\w+)$"
	`^/url$`,                  // qui-vive-0.1.0: r"^/url$"
	`^/url/(\w+)$`,            // qui-vive-0.1.0: r"^/url/(\w+)$"
	`^/inv$`,                  // qui-vive-0.1.0: r"^/inv$"
	`^/inv/(\w+)$`,            // qui-vive-0.1.0: r"^/inv/(\w+)$"
	// subdiff-0.1.0: r"\b"
	`^(\d+)/(\d+)$`,                    // substudy-0.4.5: r"^(\d+)/(\d+)$"
	`\s+`,                              // substudy-0.4.5: r"\s+"
	`<[a-z/][^>]*>`,                    // substudy-0.4.5: r"<[a-z/][^>]*>"
	`(\([^)]*\)|♪[^♪]*♪|[A-Z]{2,} ?:)`, // substudy-0.4.5: r"(\([^)]*\)|♪[^♪]*♪|[A-Z]{2,} ?:)"
	`\s+`,                              // substudy-0.4.5: r"\s+"
	`^(\d(-| )?){9}(x|X|\d|(\d(-| )?){3}\d)$`, // isbnid-0.1.3: r"^(\d(-| )?){9}(x|X|\d|(\d(-| )?){3}\d)$"
	`[^0-9X]`, // isbnid-0.1.3: r"[^0-9X]"
	`Intel\(r\) SPMD Program Compiler \(ispc\), (\d+\.\d+\.\d+)`, // ispc-0.3.5: r"Intel\(r\) SPMD Program Compiler \(ispc\), (\d+\.\d+\.\d+)"
}

func TestStringMatching(t *testing.T) {
	t.Parallel()

	for _, expr := range crateRegexps {
		t.Run(expr, func(t *testing.T) {
			re, err := regexp.Compile(expr)
			if err != nil {
				t.Fatalf("failed to compile %q: %v", expr, err)
			}

			Check(t, func(t *T) {
				s := StringMatching(expr).Draw(t, "s").(string)
				if !re.MatchString(s) {
					t.Fatalf("%q does not match %q", s, expr)
				}
			})
		})
	}
}

func TestSliceOfBytesMatching(t *testing.T) {
	t.Parallel()

	for _, expr := range crateRegexps {
		t.Run(expr, func(t *testing.T) {
			re, err := regexp.Compile(expr)
			if err != nil {
				t.Fatalf("failed to compile %q: %v", expr, err)
			}

			Check(t, func(t *T) {
				s := SliceOfBytesMatching(expr).Draw(t, "s").([]byte)
				if !re.Match(s) {
					t.Fatalf("%q does not match %q", s, expr)
				}
			})
		})
	}
}
