package fetchfixes

import (
	"context"
	"fmt"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/runtime/schema"
	k8sserial "k8s.io/apimachinery/pkg/runtime/serializer/json"
	"k8s.io/cli-runtime/pkg/genericclioptions"

	"github.com/openshift/oc-compliance/internal/common"
)

type RuleHelper struct {
	kuser      common.KubeClientUser
	gvk        schema.GroupVersionResource
	kind       string
	name       string
	outputPath string
	genericclioptions.IOStreams
}

func NewRuleHelper(kuser common.KubeClientUser, name string, outputPath string, streams genericclioptions.IOStreams) common.ObjectHelper {
	return &RuleHelper{
		kuser: kuser,
		name:  name,
		kind:  "Rule",
		gvk: schema.GroupVersionResource{
			Group:    common.CmpAPIGroup,
			Version:  common.CmpResourceVersion,
			Resource: "rules",
		},
		outputPath: outputPath,
		IOStreams:  streams,
	}
}

func (h *RuleHelper) Handle() error {
	r, err := h.kuser.DynamicClient().Resource(h.gvk).Namespace(h.kuser.GetNamespace()).Get(context.TODO(), h.name, metav1.GetOptions{})
	if err != nil {
		return err
	}
	fixes, err := h.getAvailableFixes(r)
	if err != nil {
		return err
	}

	if len(fixes) == 0 {
		fmt.Fprintf(h.Out, "No fixes to persist for rule '%s'\n", r.GetName())
		return nil
	}

	yamlSerializer := k8sserial.NewYAMLSerializer(k8sserial.DefaultMetaFactory, nil, nil)

	needsSuffix := len(fixes) > 1
	for idx, fix := range fixes {
		fixName := r.GetName()
		if needsSuffix {
			fixName = fmt.Sprintf("%s-%d", r.GetName(), idx)
		}

		path, err := common.PersistObjectToYamlFile(fixName, fix, h.outputPath, yamlSerializer)
		if err != nil {
			return err
		}

		fmt.Fprintf(h.Out, "Persisted rule fix to %s\n", path)
	}

	return nil
}

func (h *RuleHelper) getAvailableFixes(obj *unstructured.Unstructured) ([]*unstructured.Unstructured, error) {
	fixes, found, err := unstructured.NestedSlice(obj.Object, "availableFixes")
	if err != nil {
		return nil, fmt.Errorf("Unable to get fixes of %s/%s of type %s: %s", obj.GetNamespace(), obj.GetName(), h.kind, err)
	}
	if !found {
		// This is not a fatal error. Some rules might not have fixes assigned to them
		return nil, nil
	}

	output := []*unstructured.Unstructured{}
	for _, fixIf := range fixes {
		fix, ok := fixIf.(map[string]interface{})
		if !ok {
			fmt.Fprintf(h.ErrOut, "WARNING: Rule '%s/%s' has a malformed fix. Expected a map.", obj.GetNamespace(), obj.GetName())
			continue
		}
		fixobjraw, found, err := unstructured.NestedMap(fix, "fixObject")
		if err != nil {
			fmt.Fprintf(h.ErrOut, "WARNING: Rule '%s/%s' has a malformed fix. Couldn't get 'fixObject' key.", obj.GetNamespace(), obj.GetName())
			continue
		}
		if !found {
			fmt.Fprintf(h.ErrOut, "WARNING: Rule '%s/%s' has a malformed fix. 'fixObject' key not found.", obj.GetNamespace(), obj.GetName())
			continue
		}
		fixobj := &unstructured.Unstructured{Object: fixobjraw}
		output = append(output, fixobj)
	}
	return output, nil
}
