// Package servicefabric implements the Azure ARM Servicefabric service API version 5.6.*.
//
// Service Fabric REST Client APIs allows management of Service Fabric clusters, applications and services.
package servicefabric

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"github.com/satori/go.uuid"
	"net/http"
)

const (
	// DefaultBaseURI is the default URI used for the service Servicefabric
	DefaultBaseURI = "http://localhost:19080"
)

// BaseClient is the base client for Servicefabric.
type BaseClient struct {
	autorest.Client
	BaseURI string
}

// New creates an instance of the BaseClient client.
func New() BaseClient {
	return NewWithBaseURI(DefaultBaseURI)
}

// NewWithBaseURI creates an instance of the BaseClient client.
func NewWithBaseURI(baseURI string) BaseClient {
	return BaseClient{
		Client:  autorest.NewClientWithUserAgent(UserAgent()),
		BaseURI: baseURI,
	}
}

// CancelOperation the following is a list of APIs that start fault operations that may be cancelled using
// CancelOperation -
// - StartDataLoss
// - StartQuorumLoss
// - StartPartitionRestart
// - StartNodeTransition
//
// If force is false, then the specified user-induced operation will be gracefully stopped and cleaned up.  If force is
// true, the command will be aborted, and some internal state
// may be left behind.  Specifying force as true should be used with care.  Calling this API with force set to true is
// not allowed until this API has already
// been called on the same test command with force set to false first, or unless the test command already has an
// OperationState of OperationState.RollingBack.
// Clarification: OperationState.RollingBack means that the system will/is be cleaning up internal system state caused
// by executing the command.  It will not restore data if the
// test command was to cause data loss.  For example, if you call StartDataLoss then call this API, the system will
// only clean up internal state from running the command.
// It will not restore the target partition's data, if the command progressed far enough to cause data loss.
//
// Important note:  if this API is invoked with force==true, internal state may be left behind.
// Parameters:
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// force - indicates whether to gracefully rollback and clean up internal system state modified by executing
// the user-induced operation.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) CancelOperation(ctx context.Context, operationID uuid.UUID, force bool, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "CancelOperation", err.Error())
	}

	req, err := client.CancelOperationPreparer(ctx, operationID, force, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CancelOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.CancelOperationSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CancelOperation", resp, "Failure sending request")
		return
	}

	result, err = client.CancelOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CancelOperation", resp, "Failure responding to request")
	}

	return
}

// CancelOperationPreparer prepares the CancelOperation request.
func (client BaseClient) CancelOperationPreparer(ctx context.Context, operationID uuid.UUID, force bool, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"Force":       autorest.Encode("query", force),
		"OperationId": autorest.Encode("query", operationID),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Faults/$/Cancel"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CancelOperationSender sends the CancelOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CancelOperationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CancelOperationResponder handles the response to the CancelOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) CancelOperationResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CopyImageStoreContent copies the image store content from the source image store relative path to the destination
// image store relative path.
// Parameters:
// imageStoreCopyDescription - describes the copy description for the image store.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) CopyImageStoreContent(ctx context.Context, imageStoreCopyDescription ImageStoreCopyDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageStoreCopyDescription,
			Constraints: []validation.Constraint{{Target: "imageStoreCopyDescription.RemoteSource", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "imageStoreCopyDescription.RemoteDestination", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "CopyImageStoreContent", err.Error())
	}

	req, err := client.CopyImageStoreContentPreparer(ctx, imageStoreCopyDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CopyImageStoreContent", nil, "Failure preparing request")
		return
	}

	resp, err := client.CopyImageStoreContentSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CopyImageStoreContent", resp, "Failure sending request")
		return
	}

	result, err = client.CopyImageStoreContentResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CopyImageStoreContent", resp, "Failure responding to request")
	}

	return
}

// CopyImageStoreContentPreparer prepares the CopyImageStoreContent request.
func (client BaseClient) CopyImageStoreContentPreparer(ctx context.Context, imageStoreCopyDescription ImageStoreCopyDescription, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/ImageStore/$/Copy"),
		autorest.WithJSON(imageStoreCopyDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CopyImageStoreContentSender sends the CopyImageStoreContent request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CopyImageStoreContentSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CopyImageStoreContentResponder handles the response to the CopyImageStoreContent request. The method always
// closes the http.Response Body.
func (client BaseClient) CopyImageStoreContentResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CreateApplication creates a Service Fabric application using the specified description.
// Parameters:
// applicationDescription - describes the application to be created.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) CreateApplication(ctx context.Context, applicationDescription ApplicationDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: applicationDescription,
			Constraints: []validation.Constraint{{Target: "applicationDescription.Name", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "applicationDescription.TypeName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "applicationDescription.TypeVersion", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "applicationDescription.ApplicationCapacity", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "applicationDescription.ApplicationCapacity.MinimumNodes", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "applicationDescription.ApplicationCapacity.MinimumNodes", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
						{Target: "applicationDescription.ApplicationCapacity.MaximumNodes", Name: validation.Null, Rule: false,
							Chain: []validation.Constraint{{Target: "applicationDescription.ApplicationCapacity.MaximumNodes", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
					}}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "CreateApplication", err.Error())
	}

	req, err := client.CreateApplicationPreparer(ctx, applicationDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateApplication", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateApplicationSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateApplication", resp, "Failure sending request")
		return
	}

	result, err = client.CreateApplicationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateApplication", resp, "Failure responding to request")
	}

	return
}

// CreateApplicationPreparer prepares the CreateApplication request.
func (client BaseClient) CreateApplicationPreparer(ctx context.Context, applicationDescription ApplicationDescription, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Applications/$/Create"),
		autorest.WithJSON(applicationDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateApplicationSender sends the CreateApplication request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateApplicationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateApplicationResponder handles the response to the CreateApplication request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateApplicationResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CreateComposeApplication creates a Service Fabric compose application.
// Parameters:
// createComposeApplicationDescription - describes the compose application that needs to be created.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) CreateComposeApplication(ctx context.Context, createComposeApplicationDescription CreateComposeApplicationDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: createComposeApplicationDescription,
			Constraints: []validation.Constraint{{Target: "createComposeApplicationDescription.ApplicationName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "createComposeApplicationDescription.ComposeFileContent", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "CreateComposeApplication", err.Error())
	}

	req, err := client.CreateComposeApplicationPreparer(ctx, createComposeApplicationDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateComposeApplication", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateComposeApplicationSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateComposeApplication", resp, "Failure sending request")
		return
	}

	result, err = client.CreateComposeApplicationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateComposeApplication", resp, "Failure responding to request")
	}

	return
}

// CreateComposeApplicationPreparer prepares the CreateComposeApplication request.
func (client BaseClient) CreateComposeApplicationPreparer(ctx context.Context, createComposeApplicationDescription CreateComposeApplicationDescription, timeout *int64) (*http.Request, error) {
	const APIVersion = "4.0-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/ComposeDeployments/$/Create"),
		autorest.WithJSON(createComposeApplicationDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateComposeApplicationSender sends the CreateComposeApplication request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateComposeApplicationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateComposeApplicationResponder handles the response to the CreateComposeApplication request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateComposeApplicationResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CreateService creates the specified service.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceDescription - the configuration for the service.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) CreateService(ctx context.Context, applicationID string, serviceDescription BasicServiceDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: serviceDescription,
			Constraints: []validation.Constraint{{Target: "serviceDescription.ServiceName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "serviceDescription.ServiceTypeName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "serviceDescription.PartitionDescription", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "CreateService", err.Error())
	}

	req, err := client.CreateServicePreparer(ctx, applicationID, serviceDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateService", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateServiceSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateService", resp, "Failure sending request")
		return
	}

	result, err = client.CreateServiceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateService", resp, "Failure responding to request")
	}

	return
}

// CreateServicePreparer prepares the CreateService request.
func (client BaseClient) CreateServicePreparer(ctx context.Context, applicationID string, serviceDescription BasicServiceDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetServices/$/Create", pathParameters),
		autorest.WithJSON(serviceDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateServiceSender sends the CreateService request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateServiceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateServiceResponder handles the response to the CreateService request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateServiceResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CreateServiceFromTemplate creates a Service Fabric service from the service template defined in the application
// manifest.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceFromTemplateDescription - describes the service that needs to be created from the template defined in
// the application manifest.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) CreateServiceFromTemplate(ctx context.Context, applicationID string, serviceFromTemplateDescription ServiceFromTemplateDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: serviceFromTemplateDescription,
			Constraints: []validation.Constraint{{Target: "serviceFromTemplateDescription.ApplicationName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "serviceFromTemplateDescription.ServiceName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "serviceFromTemplateDescription.ServiceTypeName", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "CreateServiceFromTemplate", err.Error())
	}

	req, err := client.CreateServiceFromTemplatePreparer(ctx, applicationID, serviceFromTemplateDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateServiceFromTemplate", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateServiceFromTemplateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateServiceFromTemplate", resp, "Failure sending request")
		return
	}

	result, err = client.CreateServiceFromTemplateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "CreateServiceFromTemplate", resp, "Failure responding to request")
	}

	return
}

// CreateServiceFromTemplatePreparer prepares the CreateServiceFromTemplate request.
func (client BaseClient) CreateServiceFromTemplatePreparer(ctx context.Context, applicationID string, serviceFromTemplateDescription ServiceFromTemplateDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetServices/$/CreateFromTemplate", pathParameters),
		autorest.WithJSON(serviceFromTemplateDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateServiceFromTemplateSender sends the CreateServiceFromTemplate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateServiceFromTemplateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateServiceFromTemplateResponder handles the response to the CreateServiceFromTemplate request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateServiceFromTemplateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteApplication deletes an existing Service Fabric application. An application must be created before it can be
// deleted. Deleting an application will delete all services that are part of that application. By default Service
// Fabric will try to close service replicas in a graceful manner and then delete the service. However if service is
// having issues closing the replica gracefully, the delete operation may take a long time or get stuck. Use the
// optional ForceRemove flag to skip the graceful close sequence and forcefully delete the application and all of the
// its services.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// forceRemove - remove a Service Fabric application or service forcefully without going through the graceful
// shutdown sequence. This parameter can be used to forcefully delete an application or service for which
// delete is timing out due to issues in the service code that prevents graceful close of replicas.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) DeleteApplication(ctx context.Context, applicationID string, forceRemove *bool, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "DeleteApplication", err.Error())
	}

	req, err := client.DeleteApplicationPreparer(ctx, applicationID, forceRemove, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteApplication", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteApplicationSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteApplication", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteApplicationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteApplication", resp, "Failure responding to request")
	}

	return
}

// DeleteApplicationPreparer prepares the DeleteApplication request.
func (client BaseClient) DeleteApplicationPreparer(ctx context.Context, applicationID string, forceRemove *bool, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if forceRemove != nil {
		queryParameters["ForceRemove"] = autorest.Encode("query", *forceRemove)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/Delete", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteApplicationSender sends the DeleteApplication request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteApplicationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteApplicationResponder handles the response to the DeleteApplication request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteApplicationResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteImageStoreContent deletes existing image store content being found within the given image store relative path.
// This can be used to delete uploaded application packages once they are provisioned.
// Parameters:
// contentPath - relative path to file or folder in the image store from its root.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) DeleteImageStoreContent(ctx context.Context, contentPath string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "DeleteImageStoreContent", err.Error())
	}

	req, err := client.DeleteImageStoreContentPreparer(ctx, contentPath, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteImageStoreContent", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteImageStoreContentSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteImageStoreContent", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteImageStoreContentResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteImageStoreContent", resp, "Failure responding to request")
	}

	return
}

// DeleteImageStoreContentPreparer prepares the DeleteImageStoreContent request.
func (client BaseClient) DeleteImageStoreContentPreparer(ctx context.Context, contentPath string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"contentPath": autorest.Encode("path", contentPath),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ImageStore/{contentPath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteImageStoreContentSender sends the DeleteImageStoreContent request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteImageStoreContentSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteImageStoreContentResponder handles the response to the DeleteImageStoreContent request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteImageStoreContentResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteService deletes an existing Service Fabric service. A service must be created before it can be deleted. By
// default Service Fabric will try to close service replicas in a graceful manner and then delete the service. However
// if service is having issues closing the replica gracefully, the delete operation may take a long time or get stuck.
// Use the optional ForceRemove flag to skip the graceful close sequence and forcefully delete the service.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// forceRemove - remove a Service Fabric application or service forcefully without going through the graceful
// shutdown sequence. This parameter can be used to forcefully delete an application or service for which
// delete is timing out due to issues in the service code that prevents graceful close of replicas.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) DeleteService(ctx context.Context, serviceID string, forceRemove *bool, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "DeleteService", err.Error())
	}

	req, err := client.DeleteServicePreparer(ctx, serviceID, forceRemove, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteService", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteServiceSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteService", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteServiceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeleteService", resp, "Failure responding to request")
	}

	return
}

// DeleteServicePreparer prepares the DeleteService request.
func (client BaseClient) DeleteServicePreparer(ctx context.Context, serviceID string, forceRemove *bool, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if forceRemove != nil {
		queryParameters["ForceRemove"] = autorest.Encode("query", *forceRemove)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/Delete", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteServiceSender sends the DeleteService request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteServiceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteServiceResponder handles the response to the DeleteService request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteServiceResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeployedServicePackageToNode downloads packages associated with specified service manifest to image cache on
// specified node.
// Parameters:
// nodeName - the name of the node.
// deployServicePackageToNodeDescription - describes information for deploying a service package to a Service
// Fabric node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) DeployedServicePackageToNode(ctx context.Context, nodeName string, deployServicePackageToNodeDescription DeployServicePackageToNodeDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: deployServicePackageToNodeDescription,
			Constraints: []validation.Constraint{{Target: "deployServicePackageToNodeDescription.ServiceManifestName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "deployServicePackageToNodeDescription.ApplicationTypeName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "deployServicePackageToNodeDescription.ApplicationTypeVersion", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "deployServicePackageToNodeDescription.NodeName", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "DeployedServicePackageToNode", err.Error())
	}

	req, err := client.DeployedServicePackageToNodePreparer(ctx, nodeName, deployServicePackageToNodeDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeployedServicePackageToNode", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeployedServicePackageToNodeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeployedServicePackageToNode", resp, "Failure sending request")
		return
	}

	result, err = client.DeployedServicePackageToNodeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DeployedServicePackageToNode", resp, "Failure responding to request")
	}

	return
}

// DeployedServicePackageToNodePreparer prepares the DeployedServicePackageToNode request.
func (client BaseClient) DeployedServicePackageToNodePreparer(ctx context.Context, nodeName string, deployServicePackageToNodeDescription DeployServicePackageToNodeDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/DeployServicePackage", pathParameters),
		autorest.WithJSON(deployServicePackageToNodeDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeployedServicePackageToNodeSender sends the DeployedServicePackageToNode request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeployedServicePackageToNodeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeployedServicePackageToNodeResponder handles the response to the DeployedServicePackageToNode request. The method always
// closes the http.Response Body.
func (client BaseClient) DeployedServicePackageToNodeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DisableNode deactivate a Service Fabric cluster node with the specified deactivation intent. Once the deactivation
// is in progress, the deactivation intent can be increased, but not decreased (for example, a node which is was
// deactivated with the Pause intent can be deactivated further with Restart, but not the other way around. Nodes may
// be reactivated using the Activate a node operation any time after they are deactivated. If the deactivation is not
// complete this will cancel the deactivation. A node which goes down and comes back up while deactivated will still
// need to be reactivated before services will be placed on that node.
// Parameters:
// nodeName - the name of the node.
// deactivationIntentDescription - describes the intent or reason for deactivating the node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) DisableNode(ctx context.Context, nodeName string, deactivationIntentDescription DeactivationIntentDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "DisableNode", err.Error())
	}

	req, err := client.DisableNodePreparer(ctx, nodeName, deactivationIntentDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DisableNode", nil, "Failure preparing request")
		return
	}

	resp, err := client.DisableNodeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DisableNode", resp, "Failure sending request")
		return
	}

	result, err = client.DisableNodeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "DisableNode", resp, "Failure responding to request")
	}

	return
}

// DisableNodePreparer prepares the DisableNode request.
func (client BaseClient) DisableNodePreparer(ctx context.Context, nodeName string, deactivationIntentDescription DeactivationIntentDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/Deactivate", pathParameters),
		autorest.WithJSON(deactivationIntentDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DisableNodeSender sends the DisableNode request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DisableNodeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DisableNodeResponder handles the response to the DisableNode request. The method always
// closes the http.Response Body.
func (client BaseClient) DisableNodeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// EnableNode activates a Service Fabric cluster node which is currently deactivated. Once activated, the node will
// again become a viable target for placing new replicas, and any deactivated replicas remaining on the node will be
// reactivated.
// Parameters:
// nodeName - the name of the node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) EnableNode(ctx context.Context, nodeName string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "EnableNode", err.Error())
	}

	req, err := client.EnableNodePreparer(ctx, nodeName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "EnableNode", nil, "Failure preparing request")
		return
	}

	resp, err := client.EnableNodeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "EnableNode", resp, "Failure sending request")
		return
	}

	result, err = client.EnableNodeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "EnableNode", resp, "Failure responding to request")
	}

	return
}

// EnableNodePreparer prepares the EnableNode request.
func (client BaseClient) EnableNodePreparer(ctx context.Context, nodeName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/Activate", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// EnableNodeSender sends the EnableNode request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) EnableNodeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// EnableNodeResponder handles the response to the EnableNode request. The method always
// closes the http.Response Body.
func (client BaseClient) EnableNodeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// GetAadMetadata gets the Azure Active Directory metadata used for secured connection to cluster.
// This API is not supposed to be called separately. It provides information needed to set up an Azure Active Directory
// secured connection with a Service Fabric cluster.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetAadMetadata(ctx context.Context, timeout *int64) (result AadMetadataObject, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetAadMetadata", err.Error())
	}

	req, err := client.GetAadMetadataPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetAadMetadata", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetAadMetadataSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetAadMetadata", resp, "Failure sending request")
		return
	}

	result, err = client.GetAadMetadataResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetAadMetadata", resp, "Failure responding to request")
	}

	return
}

// GetAadMetadataPreparer prepares the GetAadMetadata request.
func (client BaseClient) GetAadMetadataPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "1.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetAadMetadata"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetAadMetadataSender sends the GetAadMetadata request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetAadMetadataSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetAadMetadataResponder handles the response to the GetAadMetadata request. The method always
// closes the http.Response Body.
func (client BaseClient) GetAadMetadataResponder(resp *http.Response) (result AadMetadataObject, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationHealth returns the heath state of the service fabric application. The response reports either Ok,
// Error or Warning health state. If the entity is not found in the helath store, it will return Error.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// deployedApplicationsHealthStateFilter - allows filtering of the deployed applications health state objects
// returned in the result of application health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states. Only
// deployed applications that match the filter will be returned.\
// All deployed applications are used to evaluate the aggregated health state. If not specified, all entries
// are returned.
// The state values are flag based enumeration, so the value could be a combination of these value obtained
// using bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of deployed applications with HealthState value of
// OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn't match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// servicesHealthStateFilter - allows filtering of the services health state objects returned in the result of
// services health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states.
// Only services that match the filter are returned. All services are used to evaluate the aggregated health
// state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value
// obtained using bitwise 'OR' operator. For example, if the provided value is 6 then health state of services
// with HealthState value of OK (2) and Warning (4) will be returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn�t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationHealth(ctx context.Context, applicationID string, eventsHealthStateFilter *int32, deployedApplicationsHealthStateFilter *int32, servicesHealthStateFilter *int32, timeout *int64) (result ApplicationHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationHealth", err.Error())
	}

	req, err := client.GetApplicationHealthPreparer(ctx, applicationID, eventsHealthStateFilter, deployedApplicationsHealthStateFilter, servicesHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationHealth", resp, "Failure responding to request")
	}

	return
}

// GetApplicationHealthPreparer prepares the GetApplicationHealth request.
func (client BaseClient) GetApplicationHealthPreparer(ctx context.Context, applicationID string, eventsHealthStateFilter *int32, deployedApplicationsHealthStateFilter *int32, servicesHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if deployedApplicationsHealthStateFilter != nil {
		queryParameters["DeployedApplicationsHealthStateFilter"] = autorest.Encode("query", *deployedApplicationsHealthStateFilter)
	} else {
		queryParameters["DeployedApplicationsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if servicesHealthStateFilter != nil {
		queryParameters["ServicesHealthStateFilter"] = autorest.Encode("query", *servicesHealthStateFilter)
	} else {
		queryParameters["ServicesHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationHealthSender sends the GetApplicationHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationHealthResponder handles the response to the GetApplicationHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationHealthResponder(resp *http.Response) (result ApplicationHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationHealthUsingPolicy gets the health of a Service Fabric application. Use EventsHealthStateFilter to
// filter the collection of health events reported on the node based on the health state. Use ClusterHealthPolicies to
// override the health policies used to evaluate the health.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// deployedApplicationsHealthStateFilter - allows filtering of the deployed applications health state objects
// returned in the result of application health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states. Only
// deployed applications that match the filter will be returned.\
// All deployed applications are used to evaluate the aggregated health state. If not specified, all entries
// are returned.
// The state values are flag based enumeration, so the value could be a combination of these value obtained
// using bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of deployed applications with HealthState value of
// OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn't match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// servicesHealthStateFilter - allows filtering of the services health state objects returned in the result of
// services health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states.
// Only services that match the filter are returned. All services are used to evaluate the aggregated health
// state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value
// obtained using bitwise 'OR' operator. For example, if the provided value is 6 then health state of services
// with HealthState value of OK (2) and Warning (4) will be returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn�t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationHealthPolicy - describes the health policies used to evaluate the health of an application or one
// of its children.
// If not present, the health evaluation uses the health policy from application manifest or the default health
// policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationHealthUsingPolicy(ctx context.Context, applicationID string, eventsHealthStateFilter *int32, deployedApplicationsHealthStateFilter *int32, servicesHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (result ApplicationHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationHealthUsingPolicy", err.Error())
	}

	req, err := client.GetApplicationHealthUsingPolicyPreparer(ctx, applicationID, eventsHealthStateFilter, deployedApplicationsHealthStateFilter, servicesHealthStateFilter, applicationHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetApplicationHealthUsingPolicyPreparer prepares the GetApplicationHealthUsingPolicy request.
func (client BaseClient) GetApplicationHealthUsingPolicyPreparer(ctx context.Context, applicationID string, eventsHealthStateFilter *int32, deployedApplicationsHealthStateFilter *int32, servicesHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if deployedApplicationsHealthStateFilter != nil {
		queryParameters["DeployedApplicationsHealthStateFilter"] = autorest.Encode("query", *deployedApplicationsHealthStateFilter)
	} else {
		queryParameters["DeployedApplicationsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if servicesHealthStateFilter != nil {
		queryParameters["ServicesHealthStateFilter"] = autorest.Encode("query", *servicesHealthStateFilter)
	} else {
		queryParameters["ServicesHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if applicationHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(applicationHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationHealthUsingPolicySender sends the GetApplicationHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationHealthUsingPolicyResponder handles the response to the GetApplicationHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationHealthUsingPolicyResponder(resp *http.Response) (result ApplicationHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationInfo returns the information about the application that was created or in the process of being created
// in the Service Fabric cluster and whose name matches the one specified as the parameter. The response includes the
// name, type, status, parameters and other details about the application.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// excludeApplicationParameters - the flag that specifies whether application parameters will be excluded from
// the result.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationInfo(ctx context.Context, applicationID string, excludeApplicationParameters *bool, timeout *int64) (result ApplicationInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationInfo", err.Error())
	}

	req, err := client.GetApplicationInfoPreparer(ctx, applicationID, excludeApplicationParameters, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationInfo", resp, "Failure responding to request")
	}

	return
}

// GetApplicationInfoPreparer prepares the GetApplicationInfo request.
func (client BaseClient) GetApplicationInfoPreparer(ctx context.Context, applicationID string, excludeApplicationParameters *bool, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if excludeApplicationParameters != nil {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", *excludeApplicationParameters)
	} else {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", false)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationInfoSender sends the GetApplicationInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationInfoResponder handles the response to the GetApplicationInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationInfoResponder(resp *http.Response) (result ApplicationInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationInfoList gets the information about the applications that were created or in the process of being
// created in the Service Fabric cluster and match filters specified as the parameter. The response includes the name,
// type, status, parameters and other details about the application. If the applications do not fit in a page, one page
// of results is returned as well as a continuation token which can be used to get the next page.
// Parameters:
// applicationTypeName - the application type name used to filter the applications to query for. This value
// should not contain the application type version.
// excludeApplicationParameters - the flag that specifies whether application parameters will be excluded from
// the result.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationInfoList(ctx context.Context, applicationTypeName string, excludeApplicationParameters *bool, continuationToken string, timeout *int64) (result PagedApplicationInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationInfoList", err.Error())
	}

	req, err := client.GetApplicationInfoListPreparer(ctx, applicationTypeName, excludeApplicationParameters, continuationToken, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationInfoList", resp, "Failure responding to request")
	}

	return
}

// GetApplicationInfoListPreparer prepares the GetApplicationInfoList request.
func (client BaseClient) GetApplicationInfoListPreparer(ctx context.Context, applicationTypeName string, excludeApplicationParameters *bool, continuationToken string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(applicationTypeName) > 0 {
		queryParameters["ApplicationTypeName"] = autorest.Encode("query", applicationTypeName)
	}
	if excludeApplicationParameters != nil {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", *excludeApplicationParameters)
	} else {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", false)
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Applications"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationInfoListSender sends the GetApplicationInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationInfoListResponder handles the response to the GetApplicationInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationInfoListResponder(resp *http.Response) (result PagedApplicationInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationManifest gets the manifest describing an application type. The response contains the application
// manifest XML as a string.
// Parameters:
// applicationTypeName - the name of the application type.
// applicationTypeVersion - the version of the application type.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationManifest(ctx context.Context, applicationTypeName string, applicationTypeVersion string, timeout *int64) (result ApplicationTypeManifest, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationManifest", err.Error())
	}

	req, err := client.GetApplicationManifestPreparer(ctx, applicationTypeName, applicationTypeVersion, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationManifest", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationManifestSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationManifest", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationManifestResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationManifest", resp, "Failure responding to request")
	}

	return
}

// GetApplicationManifestPreparer prepares the GetApplicationManifest request.
func (client BaseClient) GetApplicationManifestPreparer(ctx context.Context, applicationTypeName string, applicationTypeVersion string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationTypeName": autorest.Encode("path", applicationTypeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":            APIVersion,
		"ApplicationTypeVersion": autorest.Encode("query", applicationTypeVersion),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ApplicationTypes/{applicationTypeName}/$/GetApplicationManifest", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationManifestSender sends the GetApplicationManifest request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationManifestSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationManifestResponder handles the response to the GetApplicationManifest request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationManifestResponder(resp *http.Response) (result ApplicationTypeManifest, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationNameInfo the GetApplicationName endpoint returns the name of the application for the specified
// service.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationNameInfo(ctx context.Context, serviceID string, timeout *int64) (result ApplicationNameInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationNameInfo", err.Error())
	}

	req, err := client.GetApplicationNameInfoPreparer(ctx, serviceID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationNameInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationNameInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationNameInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationNameInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationNameInfo", resp, "Failure responding to request")
	}

	return
}

// GetApplicationNameInfoPreparer prepares the GetApplicationNameInfo request.
func (client BaseClient) GetApplicationNameInfoPreparer(ctx context.Context, serviceID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/GetApplicationName", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationNameInfoSender sends the GetApplicationNameInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationNameInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationNameInfoResponder handles the response to the GetApplicationNameInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationNameInfoResponder(resp *http.Response) (result ApplicationNameInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationTypeInfoList returns the information about the application types that are provisioned or in the
// process of being provisioned in the Service Fabric cluster. Each version of an application type is returned as one
// application type. The response includes the name, version, status and other details about the application type. This
// is a paged query, meaning that if not all of the application types fit in a page, one page of results is returned as
// well as a continuation token which can be used to get the next page. For example, if there are 10 application types
// but a page only fits the first 3 application types, or if max results is set to 3, then 3 is returned. To access the
// rest of the results, retrieve subsequent pages by using the returned continuation token in the next query. An empty
// continuation token is returned if there are no subsequent pages.
// Parameters:
// excludeApplicationParameters - the flag that specifies whether application parameters will be excluded from
// the result.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// maxResults - the maximum number of results to be returned as part of the paged queries. This parameter
// defines the upper bound on the number of results returned. The results returned can be less than the
// specified maximum results if they do not fit in the message as per the max message size restrictions defined
// in the configuration. If this parameter is zero or not specified, the paged queries includes as much results
// as possible that fit in the return message.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationTypeInfoList(ctx context.Context, excludeApplicationParameters *bool, continuationToken string, maxResults *int64, timeout *int64) (result PagedApplicationTypeInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxResults,
			Constraints: []validation.Constraint{{Target: "maxResults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxResults", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationTypeInfoList", err.Error())
	}

	req, err := client.GetApplicationTypeInfoListPreparer(ctx, excludeApplicationParameters, continuationToken, maxResults, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationTypeInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationTypeInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationTypeInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationTypeInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationTypeInfoList", resp, "Failure responding to request")
	}

	return
}

// GetApplicationTypeInfoListPreparer prepares the GetApplicationTypeInfoList request.
func (client BaseClient) GetApplicationTypeInfoListPreparer(ctx context.Context, excludeApplicationParameters *bool, continuationToken string, maxResults *int64, timeout *int64) (*http.Request, error) {
	const APIVersion = "4.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if excludeApplicationParameters != nil {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", *excludeApplicationParameters)
	} else {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", false)
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if maxResults != nil {
		queryParameters["MaxResults"] = autorest.Encode("query", *maxResults)
	} else {
		queryParameters["MaxResults"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/ApplicationTypes"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationTypeInfoListSender sends the GetApplicationTypeInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationTypeInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationTypeInfoListResponder handles the response to the GetApplicationTypeInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationTypeInfoListResponder(resp *http.Response) (result PagedApplicationTypeInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationTypeInfoListByName returns the information about the application types that are provisioned or in the
// process of being provisioned in the Service Fabric cluster. These results are of application types whose name match
// exactly the one specified as the parameter, and which comply with the given query parameters. All versions of the
// application type matching the application type name are returned, with each version returned as one application
// type. The response includes the name, version, status and other details about the application type. This is a paged
// query, meaning that if not all of the application types fit in a page, one page of results is returned as well as a
// continuation token which can be used to get the next page. For example, if there are 10 application types but a page
// only fits the first 3 application types, or if max results is set to 3, then 3 is returned. To access the rest of
// the results, retrieve subsequent pages by using the returned continuation token in the next query. An empty
// continuation token is returned if there are no subsequent pages.
// Parameters:
// applicationTypeName - the name of the application type.
// excludeApplicationParameters - the flag that specifies whether application parameters will be excluded from
// the result.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// maxResults - the maximum number of results to be returned as part of the paged queries. This parameter
// defines the upper bound on the number of results returned. The results returned can be less than the
// specified maximum results if they do not fit in the message as per the max message size restrictions defined
// in the configuration. If this parameter is zero or not specified, the paged queries includes as much results
// as possible that fit in the return message.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationTypeInfoListByName(ctx context.Context, applicationTypeName string, excludeApplicationParameters *bool, continuationToken string, maxResults *int64, timeout *int64) (result PagedApplicationTypeInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxResults,
			Constraints: []validation.Constraint{{Target: "maxResults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxResults", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationTypeInfoListByName", err.Error())
	}

	req, err := client.GetApplicationTypeInfoListByNamePreparer(ctx, applicationTypeName, excludeApplicationParameters, continuationToken, maxResults, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationTypeInfoListByName", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationTypeInfoListByNameSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationTypeInfoListByName", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationTypeInfoListByNameResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationTypeInfoListByName", resp, "Failure responding to request")
	}

	return
}

// GetApplicationTypeInfoListByNamePreparer prepares the GetApplicationTypeInfoListByName request.
func (client BaseClient) GetApplicationTypeInfoListByNamePreparer(ctx context.Context, applicationTypeName string, excludeApplicationParameters *bool, continuationToken string, maxResults *int64, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationTypeName": autorest.Encode("path", applicationTypeName),
	}

	const APIVersion = "4.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if excludeApplicationParameters != nil {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", *excludeApplicationParameters)
	} else {
		queryParameters["ExcludeApplicationParameters"] = autorest.Encode("query", false)
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if maxResults != nil {
		queryParameters["MaxResults"] = autorest.Encode("query", *maxResults)
	} else {
		queryParameters["MaxResults"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ApplicationTypes/{applicationTypeName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationTypeInfoListByNameSender sends the GetApplicationTypeInfoListByName request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationTypeInfoListByNameSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationTypeInfoListByNameResponder handles the response to the GetApplicationTypeInfoListByName request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationTypeInfoListByNameResponder(resp *http.Response) (result PagedApplicationTypeInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetApplicationUpgrade returns information about the state of the latest application upgrade along with details to
// aid debugging application health issues.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetApplicationUpgrade(ctx context.Context, applicationID string, timeout *int64) (result ApplicationUpgradeProgressInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetApplicationUpgrade", err.Error())
	}

	req, err := client.GetApplicationUpgradePreparer(ctx, applicationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationUpgrade", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetApplicationUpgradeSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationUpgrade", resp, "Failure sending request")
		return
	}

	result, err = client.GetApplicationUpgradeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetApplicationUpgrade", resp, "Failure responding to request")
	}

	return
}

// GetApplicationUpgradePreparer prepares the GetApplicationUpgrade request.
func (client BaseClient) GetApplicationUpgradePreparer(ctx context.Context, applicationID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetUpgradeProgress", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetApplicationUpgradeSender sends the GetApplicationUpgrade request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetApplicationUpgradeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetApplicationUpgradeResponder handles the response to the GetApplicationUpgrade request. The method always
// closes the http.Response Body.
func (client BaseClient) GetApplicationUpgradeResponder(resp *http.Response) (result ApplicationUpgradeProgressInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetChaosReport you can either specify the ContinuationToken to get the next segment of the Chaos report or you can
// specify the time-range
// through StartTimeUtc and EndTimeUtc, but you cannot specify both the ContinuationToken and the time-range in the
// same call.
// When there are more than 100 Chaos events, the Chaos report is returned in segments where a segment contains no more
// than 100 Chaos events.
// Parameters:
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// startTimeUtc - the count of ticks representing the start time of the time range for which a Chaos report is
// to be generated. Please consult [DateTime.Ticks
// Property](https://msdn.microsoft.com/en-us/library/system.datetime.ticks%28v=vs.110%29) for details about
// tick.
// endTimeUtc - the count of ticks representing the end time of the time range for which a Chaos report is to
// be generated. Please consult [DateTime.Ticks
// Property](https://msdn.microsoft.com/en-us/library/system.datetime.ticks%28v=vs.110%29) for details about
// tick.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetChaosReport(ctx context.Context, continuationToken string, startTimeUtc string, endTimeUtc string, timeout *int64) (result ChaosReport, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetChaosReport", err.Error())
	}

	req, err := client.GetChaosReportPreparer(ctx, continuationToken, startTimeUtc, endTimeUtc, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetChaosReport", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetChaosReportSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetChaosReport", resp, "Failure sending request")
		return
	}

	result, err = client.GetChaosReportResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetChaosReport", resp, "Failure responding to request")
	}

	return
}

// GetChaosReportPreparer prepares the GetChaosReport request.
func (client BaseClient) GetChaosReportPreparer(ctx context.Context, continuationToken string, startTimeUtc string, endTimeUtc string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if len(startTimeUtc) > 0 {
		queryParameters["StartTimeUtc"] = autorest.Encode("query", startTimeUtc)
	}
	if len(endTimeUtc) > 0 {
		queryParameters["EndTimeUtc"] = autorest.Encode("query", endTimeUtc)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Tools/Chaos/$/Report"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetChaosReportSender sends the GetChaosReport request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetChaosReportSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetChaosReportResponder handles the response to the GetChaosReport request. The method always
// closes the http.Response Body.
func (client BaseClient) GetChaosReportResponder(resp *http.Response) (result ChaosReport, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetClusterHealth gets the health of a Service Fabric cluster.
// Use EventsHealthStateFilter to filter the collection of health events reported on the cluster based on the health
// state.
// Similarly, use NodesHealthStateFilter and ApplicationsHealthStateFilter to filter the collection of nodes and
// applications returned based on their aggregated health state.
// Parameters:
// nodesHealthStateFilter - allows filtering of the node health state objects returned in the result of cluster
// health query
// based on their health state. The possible values for this parameter include integer value of one of the
// following health states. Only nodes that match the filter are returned. All nodes are used to evaluate the
// aggregated health state.
// If not specified, all entries are returned.
// The state values are flag based enumeration, so the value could be a combination of these values obtained
// using bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of nodes with HealthState value of OK (2) and
// Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationsHealthStateFilter - allows filtering of the application health state objects returned in the
// result of cluster health
// query based on their health state.
// The possible values for this parameter include integer value obtained from members or bitwise operations
// on members of HealthStateFilter enumeration. Only applications that match the filter are returned.
// All applications are used to evaluate the aggregated health state. If not specified, all entries are
// returned.
// The state values are flag based enumeration, so the value could be a combination of these values obtained
// using bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of applications with HealthState value of OK (2)
// and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetClusterHealth(ctx context.Context, nodesHealthStateFilter *int32, applicationsHealthStateFilter *int32, eventsHealthStateFilter *int32, timeout *int64) (result ClusterHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetClusterHealth", err.Error())
	}

	req, err := client.GetClusterHealthPreparer(ctx, nodesHealthStateFilter, applicationsHealthStateFilter, eventsHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetClusterHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetClusterHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealth", resp, "Failure responding to request")
	}

	return
}

// GetClusterHealthPreparer prepares the GetClusterHealth request.
func (client BaseClient) GetClusterHealthPreparer(ctx context.Context, nodesHealthStateFilter *int32, applicationsHealthStateFilter *int32, eventsHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if nodesHealthStateFilter != nil {
		queryParameters["NodesHealthStateFilter"] = autorest.Encode("query", *nodesHealthStateFilter)
	} else {
		queryParameters["NodesHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if applicationsHealthStateFilter != nil {
		queryParameters["ApplicationsHealthStateFilter"] = autorest.Encode("query", *applicationsHealthStateFilter)
	} else {
		queryParameters["ApplicationsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetClusterHealth"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetClusterHealthSender sends the GetClusterHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetClusterHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetClusterHealthResponder handles the response to the GetClusterHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetClusterHealthResponder(resp *http.Response) (result ClusterHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetClusterHealthChunk gets the health of a Service Fabric cluster using health chunks. Includes the aggregated
// health state of the cluster, but none of the cluster entities.
// To expand the cluster health and get the health state of all or some of the entities, use the POST URI and specify
// the cluster health chunk query description.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetClusterHealthChunk(ctx context.Context, timeout *int64) (result ClusterHealthChunk, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetClusterHealthChunk", err.Error())
	}

	req, err := client.GetClusterHealthChunkPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthChunk", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetClusterHealthChunkSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthChunk", resp, "Failure sending request")
		return
	}

	result, err = client.GetClusterHealthChunkResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthChunk", resp, "Failure responding to request")
	}

	return
}

// GetClusterHealthChunkPreparer prepares the GetClusterHealthChunk request.
func (client BaseClient) GetClusterHealthChunkPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetClusterHealthChunk"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetClusterHealthChunkSender sends the GetClusterHealthChunk request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetClusterHealthChunkSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetClusterHealthChunkResponder handles the response to the GetClusterHealthChunk request. The method always
// closes the http.Response Body.
func (client BaseClient) GetClusterHealthChunkResponder(resp *http.Response) (result ClusterHealthChunk, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetClusterHealthChunkUsingPolicyAndAdvancedFilters gets the health of a Service Fabric cluster using health chunks.
// The health evaluation is done based on the input cluster health chunk query description.
// The query description allows users to specify health policies for evaluating the cluster and its children.
// Users can specify very flexible filters to select which cluster entities to return. The selection can be done based
// on the entities health state and based on the hierarchy.
// The query can return multi-level children of the entities based on the specified filters. For example, it can return
// one application with a specified name, and for this application, return
// only services that are in Error or Warning, and all partitions and replicas for one of these services.
// Parameters:
// clusterHealthChunkQueryDescription - describes the cluster and application health policies used to evaluate
// the cluster health and the filters to select which cluster entities to be returned.
// If the cluster health policy is present, it is used to evaluate the cluster events and the cluster nodes. If
// not present, the health evaluation uses the cluster health policy defined in the cluster manifest or the
// default cluster health policy.
// By default, each application is evaluated using its specific application health policy, defined in the
// application manifest, or the default health policy, if no policy is defined in manifest.
// If the application health policy map is specified, and it has an entry for an application, the specified
// application health policy
// is used to evaluate the application health.
// Users can specify very flexible filters to select which cluster entities to include in response. The
// selection can be done based on the entities health state and based on the hierarchy.
// The query can return multi-level children of the entities based on the specified filters. For example, it
// can return one application with a specified name, and for this application, return
// only services that are in Error or Warning, and all partitions and replicas for one of these services.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetClusterHealthChunkUsingPolicyAndAdvancedFilters(ctx context.Context, clusterHealthChunkQueryDescription *ClusterHealthChunkQueryDescription, timeout *int64) (result ClusterHealthChunk, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetClusterHealthChunkUsingPolicyAndAdvancedFilters", err.Error())
	}

	req, err := client.GetClusterHealthChunkUsingPolicyAndAdvancedFiltersPreparer(ctx, clusterHealthChunkQueryDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthChunkUsingPolicyAndAdvancedFilters", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetClusterHealthChunkUsingPolicyAndAdvancedFiltersSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthChunkUsingPolicyAndAdvancedFilters", resp, "Failure sending request")
		return
	}

	result, err = client.GetClusterHealthChunkUsingPolicyAndAdvancedFiltersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthChunkUsingPolicyAndAdvancedFilters", resp, "Failure responding to request")
	}

	return
}

// GetClusterHealthChunkUsingPolicyAndAdvancedFiltersPreparer prepares the GetClusterHealthChunkUsingPolicyAndAdvancedFilters request.
func (client BaseClient) GetClusterHealthChunkUsingPolicyAndAdvancedFiltersPreparer(ctx context.Context, clusterHealthChunkQueryDescription *ClusterHealthChunkQueryDescription, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetClusterHealthChunk"),
		autorest.WithQueryParameters(queryParameters))
	if clusterHealthChunkQueryDescription != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(clusterHealthChunkQueryDescription))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetClusterHealthChunkUsingPolicyAndAdvancedFiltersSender sends the GetClusterHealthChunkUsingPolicyAndAdvancedFilters request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetClusterHealthChunkUsingPolicyAndAdvancedFiltersSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetClusterHealthChunkUsingPolicyAndAdvancedFiltersResponder handles the response to the GetClusterHealthChunkUsingPolicyAndAdvancedFilters request. The method always
// closes the http.Response Body.
func (client BaseClient) GetClusterHealthChunkUsingPolicyAndAdvancedFiltersResponder(resp *http.Response) (result ClusterHealthChunk, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetClusterHealthUsingPolicy gets the health of a Service Fabric cluster.
// Use EventsHealthStateFilter to filter the collection of health events reported on the cluster based on the health
// state.
// Similarly, use NodesHealthStateFilter and ApplicationsHealthStateFilter to filter the collection of nodes and
// applications returned based on their aggregated health state.
// Use ClusterHealthPolicies to override the health policies used to evaluate the health.
// Parameters:
// nodesHealthStateFilter - allows filtering of the node health state objects returned in the result of cluster
// health query
// based on their health state. The possible values for this parameter include integer value of one of the
// following health states. Only nodes that match the filter are returned. All nodes are used to evaluate the
// aggregated health state.
// If not specified, all entries are returned.
// The state values are flag based enumeration, so the value could be a combination of these values obtained
// using bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of nodes with HealthState value of OK (2) and
// Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationsHealthStateFilter - allows filtering of the application health state objects returned in the
// result of cluster health
// query based on their health state.
// The possible values for this parameter include integer value obtained from members or bitwise operations
// on members of HealthStateFilter enumeration. Only applications that match the filter are returned.
// All applications are used to evaluate the aggregated health state. If not specified, all entries are
// returned.
// The state values are flag based enumeration, so the value could be a combination of these values obtained
// using bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of applications with HealthState value of OK (2)
// and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// clusterHealthPolicies - describes the health policies used to evaluate the cluster health.
// If not present, the health evaluation uses the cluster health policy defined in the cluster manifest or the
// default cluster health policy.
// By default, each application is evaluated using its specific application health policy, defined in the
// application manifest, or the default health policy, if no policy is defined in manifest.
// If the application health policy map is specified, and it has an entry for an application, the specified
// application health policy
// is used to evaluate the application health.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetClusterHealthUsingPolicy(ctx context.Context, nodesHealthStateFilter *int32, applicationsHealthStateFilter *int32, eventsHealthStateFilter *int32, clusterHealthPolicies *ClusterHealthPolicies, timeout *int64) (result ClusterHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetClusterHealthUsingPolicy", err.Error())
	}

	req, err := client.GetClusterHealthUsingPolicyPreparer(ctx, nodesHealthStateFilter, applicationsHealthStateFilter, eventsHealthStateFilter, clusterHealthPolicies, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetClusterHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetClusterHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetClusterHealthUsingPolicyPreparer prepares the GetClusterHealthUsingPolicy request.
func (client BaseClient) GetClusterHealthUsingPolicyPreparer(ctx context.Context, nodesHealthStateFilter *int32, applicationsHealthStateFilter *int32, eventsHealthStateFilter *int32, clusterHealthPolicies *ClusterHealthPolicies, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if nodesHealthStateFilter != nil {
		queryParameters["NodesHealthStateFilter"] = autorest.Encode("query", *nodesHealthStateFilter)
	} else {
		queryParameters["NodesHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if applicationsHealthStateFilter != nil {
		queryParameters["ApplicationsHealthStateFilter"] = autorest.Encode("query", *applicationsHealthStateFilter)
	} else {
		queryParameters["ApplicationsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetClusterHealth"),
		autorest.WithQueryParameters(queryParameters))
	if clusterHealthPolicies != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(clusterHealthPolicies))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetClusterHealthUsingPolicySender sends the GetClusterHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetClusterHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetClusterHealthUsingPolicyResponder handles the response to the GetClusterHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetClusterHealthUsingPolicyResponder(resp *http.Response) (result ClusterHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetClusterManifest get the Service Fabric cluster manifest. The cluster manifest contains properties of the cluster
// that include different node types on the cluster,
// security configurations, fault and upgrade domain topologies etc.
//
// These properties are specified as part of the ClusterConfig.JSON file while deploying a stand alone cluster.
// However, most of the information in the cluster manifest
// is generated internally by service fabric during cluster deployment in other deployment scenarios (for e.g when
// using azuer portal).
//
// The contents of the cluster manifest are for informational purposes only and users are not expected to take a
// dependency on the format of the file contents or its interpretation.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetClusterManifest(ctx context.Context, timeout *int64) (result ClusterManifest, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetClusterManifest", err.Error())
	}

	req, err := client.GetClusterManifestPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterManifest", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetClusterManifestSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterManifest", resp, "Failure sending request")
		return
	}

	result, err = client.GetClusterManifestResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterManifest", resp, "Failure responding to request")
	}

	return
}

// GetClusterManifestPreparer prepares the GetClusterManifest request.
func (client BaseClient) GetClusterManifestPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetClusterManifest"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetClusterManifestSender sends the GetClusterManifest request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetClusterManifestSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetClusterManifestResponder handles the response to the GetClusterManifest request. The method always
// closes the http.Response Body.
func (client BaseClient) GetClusterManifestResponder(resp *http.Response) (result ClusterManifest, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetClusterUpgradeProgress gets the current progress of the ongoing cluster upgrade. If no upgrade is currently in
// progress, gets the last state of the previous cluster upgrade.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetClusterUpgradeProgress(ctx context.Context, timeout *int64) (result ClusterUpgradeProgressObject, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetClusterUpgradeProgress", err.Error())
	}

	req, err := client.GetClusterUpgradeProgressPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterUpgradeProgress", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetClusterUpgradeProgressSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterUpgradeProgress", resp, "Failure sending request")
		return
	}

	result, err = client.GetClusterUpgradeProgressResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetClusterUpgradeProgress", resp, "Failure responding to request")
	}

	return
}

// GetClusterUpgradeProgressPreparer prepares the GetClusterUpgradeProgress request.
func (client BaseClient) GetClusterUpgradeProgressPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetUpgradeProgress"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetClusterUpgradeProgressSender sends the GetClusterUpgradeProgress request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetClusterUpgradeProgressSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetClusterUpgradeProgressResponder handles the response to the GetClusterUpgradeProgress request. The method always
// closes the http.Response Body.
func (client BaseClient) GetClusterUpgradeProgressResponder(resp *http.Response) (result ClusterUpgradeProgressObject, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetComposeApplicationStatus returns the status of compose application that was created or in the process of being
// created in the Service Fabric cluster and whose name matches the one specified as the parameter. The response
// includes the name, status and other details about the application.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetComposeApplicationStatus(ctx context.Context, applicationID string, timeout *int64) (result ComposeApplicationStatusInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetComposeApplicationStatus", err.Error())
	}

	req, err := client.GetComposeApplicationStatusPreparer(ctx, applicationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetComposeApplicationStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetComposeApplicationStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetComposeApplicationStatus", resp, "Failure sending request")
		return
	}

	result, err = client.GetComposeApplicationStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetComposeApplicationStatus", resp, "Failure responding to request")
	}

	return
}

// GetComposeApplicationStatusPreparer prepares the GetComposeApplicationStatus request.
func (client BaseClient) GetComposeApplicationStatusPreparer(ctx context.Context, applicationID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "4.0-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ComposeDeployments/{applicationId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetComposeApplicationStatusSender sends the GetComposeApplicationStatus request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetComposeApplicationStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetComposeApplicationStatusResponder handles the response to the GetComposeApplicationStatus request. The method always
// closes the http.Response Body.
func (client BaseClient) GetComposeApplicationStatusResponder(resp *http.Response) (result ComposeApplicationStatusInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetComposeApplicationStatusList gets the status about the compose applications that were created or in the process
// of being created in the Service Fabric cluster. The response includes the name, status and other details about the
// compose application. If the applications do not fit in a page, one page of results is returned as well as a
// continuation token which can be used to get the next page.
// Parameters:
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// maxResults - the maximum number of results to be returned as part of the paged queries. This parameter
// defines the upper bound on the number of results returned. The results returned can be less than the
// specified maximum results if they do not fit in the message as per the max message size restrictions defined
// in the configuration. If this parameter is zero or not specified, the paged queries includes as much results
// as possible that fit in the return message.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetComposeApplicationStatusList(ctx context.Context, continuationToken string, maxResults *int64, timeout *int64) (result PagedComposeApplicationStatusInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxResults,
			Constraints: []validation.Constraint{{Target: "maxResults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxResults", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetComposeApplicationStatusList", err.Error())
	}

	req, err := client.GetComposeApplicationStatusListPreparer(ctx, continuationToken, maxResults, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetComposeApplicationStatusList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetComposeApplicationStatusListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetComposeApplicationStatusList", resp, "Failure sending request")
		return
	}

	result, err = client.GetComposeApplicationStatusListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetComposeApplicationStatusList", resp, "Failure responding to request")
	}

	return
}

// GetComposeApplicationStatusListPreparer prepares the GetComposeApplicationStatusList request.
func (client BaseClient) GetComposeApplicationStatusListPreparer(ctx context.Context, continuationToken string, maxResults *int64, timeout *int64) (*http.Request, error) {
	const APIVersion = "4.0-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if maxResults != nil {
		queryParameters["MaxResults"] = autorest.Encode("query", *maxResults)
	} else {
		queryParameters["MaxResults"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/ComposeDeployments"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetComposeApplicationStatusListSender sends the GetComposeApplicationStatusList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetComposeApplicationStatusListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetComposeApplicationStatusListResponder handles the response to the GetComposeApplicationStatusList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetComposeApplicationStatusListResponder(resp *http.Response) (result PagedComposeApplicationStatusInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDataLossProgress gets the progress of a data loss operation started with StartDataLoss, using the OperationId.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDataLossProgress(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, timeout *int64) (result PartitionDataLossProgress, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDataLossProgress", err.Error())
	}

	req, err := client.GetDataLossProgressPreparer(ctx, serviceID, partitionID, operationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDataLossProgress", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDataLossProgressSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDataLossProgress", resp, "Failure sending request")
		return
	}

	result, err = client.GetDataLossProgressResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDataLossProgress", resp, "Failure responding to request")
	}

	return
}

// GetDataLossProgressPreparer prepares the GetDataLossProgress request.
func (client BaseClient) GetDataLossProgressPreparer(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"serviceId":   serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"OperationId": autorest.Encode("query", operationID),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Services/{serviceId}/$/GetPartitions/{partitionId}/$/GetDataLossProgress", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDataLossProgressSender sends the GetDataLossProgress request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDataLossProgressSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDataLossProgressResponder handles the response to the GetDataLossProgress request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDataLossProgressResponder(resp *http.Response) (result PartitionDataLossProgress, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedApplicationHealth gets the information about health of an application deployed on a Service Fabric node.
// Use EventsHealthStateFilter to optionally filter for the collection of HealthEvent objects reported on the deployed
// application based on health state. Use DeployedServicePackagesHealthStateFilter to optionally filter for
// DeployedServicePackageHealth children based on health state.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// deployedServicePackagesHealthStateFilter - allows filtering of the deployed service package health state
// objects returned in the result of deployed application health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states.
// Only deployed service packages that match the filter are returned. All deployed service packages are used to
// evaluate the aggregated health state of the deployed application.
// If not specified, all entries are returned.
// The state values are flag based enumeration, so the value can be a combination of these value obtained using
// bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of service packages with HealthState value of OK
// (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedApplicationHealth(ctx context.Context, nodeName string, applicationID string, eventsHealthStateFilter *int32, deployedServicePackagesHealthStateFilter *int32, timeout *int64) (result DeployedApplicationHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedApplicationHealth", err.Error())
	}

	req, err := client.GetDeployedApplicationHealthPreparer(ctx, nodeName, applicationID, eventsHealthStateFilter, deployedServicePackagesHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedApplicationHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedApplicationHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationHealth", resp, "Failure responding to request")
	}

	return
}

// GetDeployedApplicationHealthPreparer prepares the GetDeployedApplicationHealth request.
func (client BaseClient) GetDeployedApplicationHealthPreparer(ctx context.Context, nodeName string, applicationID string, eventsHealthStateFilter *int32, deployedServicePackagesHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if deployedServicePackagesHealthStateFilter != nil {
		queryParameters["DeployedServicePackagesHealthStateFilter"] = autorest.Encode("query", *deployedServicePackagesHealthStateFilter)
	} else {
		queryParameters["DeployedServicePackagesHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedApplicationHealthSender sends the GetDeployedApplicationHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedApplicationHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedApplicationHealthResponder handles the response to the GetDeployedApplicationHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedApplicationHealthResponder(resp *http.Response) (result DeployedApplicationHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedApplicationHealthUsingPolicy gets the information about health of an application deployed on a Service
// Fabric node using the specified policy. Use EventsHealthStateFilter to optionally filter for the collection of
// HealthEvent objects reported on the deployed application based on health state. Use
// DeployedServicePackagesHealthStateFilter to optionally filter for DeployedServicePackageHealth children based on
// health state. Use ApplicationHealthPolicy to optionally override the health policies used to evaluate the health.
// This API only uses 'ConsiderWarningAsError' field of the ApplicationHealthPolicy. The rest of the fields are ignored
// while evaluating the health of the deployed application.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// deployedServicePackagesHealthStateFilter - allows filtering of the deployed service package health state
// objects returned in the result of deployed application health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states.
// Only deployed service packages that match the filter are returned. All deployed service packages are used to
// evaluate the aggregated health state of the deployed application.
// If not specified, all entries are returned.
// The state values are flag based enumeration, so the value can be a combination of these value obtained using
// bitwise 'OR' operator.
// For example, if the provided value is 6 then health state of service packages with HealthState value of OK
// (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationHealthPolicy - describes the health policies used to evaluate the health of an application or one
// of its children.
// If not present, the health evaluation uses the health policy from application manifest or the default health
// policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedApplicationHealthUsingPolicy(ctx context.Context, nodeName string, applicationID string, eventsHealthStateFilter *int32, deployedServicePackagesHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (result DeployedApplicationHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedApplicationHealthUsingPolicy", err.Error())
	}

	req, err := client.GetDeployedApplicationHealthUsingPolicyPreparer(ctx, nodeName, applicationID, eventsHealthStateFilter, deployedServicePackagesHealthStateFilter, applicationHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedApplicationHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedApplicationHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetDeployedApplicationHealthUsingPolicyPreparer prepares the GetDeployedApplicationHealthUsingPolicy request.
func (client BaseClient) GetDeployedApplicationHealthUsingPolicyPreparer(ctx context.Context, nodeName string, applicationID string, eventsHealthStateFilter *int32, deployedServicePackagesHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if deployedServicePackagesHealthStateFilter != nil {
		queryParameters["DeployedServicePackagesHealthStateFilter"] = autorest.Encode("query", *deployedServicePackagesHealthStateFilter)
	} else {
		queryParameters["DeployedServicePackagesHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if applicationHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(applicationHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedApplicationHealthUsingPolicySender sends the GetDeployedApplicationHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedApplicationHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedApplicationHealthUsingPolicyResponder handles the response to the GetDeployedApplicationHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedApplicationHealthUsingPolicyResponder(resp *http.Response) (result DeployedApplicationHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedApplicationInfo gets the information about an application deployed on a Service Fabric node.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedApplicationInfo(ctx context.Context, nodeName string, applicationID string, timeout *int64) (result DeployedApplicationInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedApplicationInfo", err.Error())
	}

	req, err := client.GetDeployedApplicationInfoPreparer(ctx, nodeName, applicationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedApplicationInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedApplicationInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationInfo", resp, "Failure responding to request")
	}

	return
}

// GetDeployedApplicationInfoPreparer prepares the GetDeployedApplicationInfo request.
func (client BaseClient) GetDeployedApplicationInfoPreparer(ctx context.Context, nodeName string, applicationID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedApplicationInfoSender sends the GetDeployedApplicationInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedApplicationInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedApplicationInfoResponder handles the response to the GetDeployedApplicationInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedApplicationInfoResponder(resp *http.Response) (result DeployedApplicationInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedApplicationInfoList gets the list of applications deployed on a Service Fabric node.
// Parameters:
// nodeName - the name of the node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedApplicationInfoList(ctx context.Context, nodeName string, timeout *int64) (result ListDeployedApplicationInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedApplicationInfoList", err.Error())
	}

	req, err := client.GetDeployedApplicationInfoListPreparer(ctx, nodeName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedApplicationInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedApplicationInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedApplicationInfoList", resp, "Failure responding to request")
	}

	return
}

// GetDeployedApplicationInfoListPreparer prepares the GetDeployedApplicationInfoList request.
func (client BaseClient) GetDeployedApplicationInfoListPreparer(ctx context.Context, nodeName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedApplicationInfoListSender sends the GetDeployedApplicationInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedApplicationInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedApplicationInfoListResponder handles the response to the GetDeployedApplicationInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedApplicationInfoListResponder(resp *http.Response) (result ListDeployedApplicationInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedCodePackageInfoList gets the list of code packages deployed on a Service Fabric node for the given
// application.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceManifestName - the name of a service manifest registered as part of an application type in a Service
// Fabric cluster.
// codePackageName - the name of code package specified in service manifest registered as part of an
// application type in a Service Fabric cluster.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedCodePackageInfoList(ctx context.Context, nodeName string, applicationID string, serviceManifestName string, codePackageName string, timeout *int64) (result ListDeployedCodePackageInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedCodePackageInfoList", err.Error())
	}

	req, err := client.GetDeployedCodePackageInfoListPreparer(ctx, nodeName, applicationID, serviceManifestName, codePackageName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedCodePackageInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedCodePackageInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedCodePackageInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedCodePackageInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedCodePackageInfoList", resp, "Failure responding to request")
	}

	return
}

// GetDeployedCodePackageInfoListPreparer prepares the GetDeployedCodePackageInfoList request.
func (client BaseClient) GetDeployedCodePackageInfoListPreparer(ctx context.Context, nodeName string, applicationID string, serviceManifestName string, codePackageName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(serviceManifestName) > 0 {
		queryParameters["ServiceManifestName"] = autorest.Encode("query", serviceManifestName)
	}
	if len(codePackageName) > 0 {
		queryParameters["CodePackageName"] = autorest.Encode("query", codePackageName)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetCodePackages", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedCodePackageInfoListSender sends the GetDeployedCodePackageInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedCodePackageInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedCodePackageInfoListResponder handles the response to the GetDeployedCodePackageInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedCodePackageInfoListResponder(resp *http.Response) (result ListDeployedCodePackageInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServicePackageHealth gets the information about health of service package for a specific application
// deployed on a Service Fabric node. Use EventsHealthStateFilter to optionally filter for the collection of
// HealthEvent objects reported on the deployed service package based on health state.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// servicePackageName - the name of the service package.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServicePackageHealth(ctx context.Context, nodeName string, applicationID string, servicePackageName string, eventsHealthStateFilter *int32, timeout *int64) (result DeployedServicePackageHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServicePackageHealth", err.Error())
	}

	req, err := client.GetDeployedServicePackageHealthPreparer(ctx, nodeName, applicationID, servicePackageName, eventsHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServicePackageHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServicePackageHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageHealth", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServicePackageHealthPreparer prepares the GetDeployedServicePackageHealth request.
func (client BaseClient) GetDeployedServicePackageHealthPreparer(ctx context.Context, nodeName string, applicationID string, servicePackageName string, eventsHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId":      applicationID,
		"nodeName":           autorest.Encode("path", nodeName),
		"servicePackageName": servicePackageName,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServicePackages/{servicePackageName}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServicePackageHealthSender sends the GetDeployedServicePackageHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServicePackageHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServicePackageHealthResponder handles the response to the GetDeployedServicePackageHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServicePackageHealthResponder(resp *http.Response) (result DeployedServicePackageHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServicePackageHealthUsingPolicy gets the information about health of an service package for a specific
// application deployed on a Service Fabric node. using the specified policy. Use EventsHealthStateFilter to optionally
// filter for the collection of HealthEvent objects reported on the deployed service package based on health state. Use
// ApplicationHealthPolicy to optionally override the health policies used to evaluate the health. This API only uses
// 'ConsiderWarningAsError' field of the ApplicationHealthPolicy. The rest of the fields are ignored while evaluating
// the health of the deployed service package.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// servicePackageName - the name of the service package.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationHealthPolicy - describes the health policies used to evaluate the health of an application or one
// of its children.
// If not present, the health evaluation uses the health policy from application manifest or the default health
// policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServicePackageHealthUsingPolicy(ctx context.Context, nodeName string, applicationID string, servicePackageName string, eventsHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (result DeployedServicePackageHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServicePackageHealthUsingPolicy", err.Error())
	}

	req, err := client.GetDeployedServicePackageHealthUsingPolicyPreparer(ctx, nodeName, applicationID, servicePackageName, eventsHealthStateFilter, applicationHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServicePackageHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServicePackageHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServicePackageHealthUsingPolicyPreparer prepares the GetDeployedServicePackageHealthUsingPolicy request.
func (client BaseClient) GetDeployedServicePackageHealthUsingPolicyPreparer(ctx context.Context, nodeName string, applicationID string, servicePackageName string, eventsHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId":      applicationID,
		"nodeName":           autorest.Encode("path", nodeName),
		"servicePackageName": servicePackageName,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServicePackages/{servicePackageName}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if applicationHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(applicationHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServicePackageHealthUsingPolicySender sends the GetDeployedServicePackageHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServicePackageHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServicePackageHealthUsingPolicyResponder handles the response to the GetDeployedServicePackageHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServicePackageHealthUsingPolicyResponder(resp *http.Response) (result DeployedServicePackageHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServicePackageInfoList returns the information about the service packages deployed on a Service Fabric
// node for the given application.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServicePackageInfoList(ctx context.Context, nodeName string, applicationID string, timeout *int64) (result ListDeployedServicePackageInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServicePackageInfoList", err.Error())
	}

	req, err := client.GetDeployedServicePackageInfoListPreparer(ctx, nodeName, applicationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServicePackageInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServicePackageInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageInfoList", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServicePackageInfoListPreparer prepares the GetDeployedServicePackageInfoList request.
func (client BaseClient) GetDeployedServicePackageInfoListPreparer(ctx context.Context, nodeName string, applicationID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServicePackages", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServicePackageInfoListSender sends the GetDeployedServicePackageInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServicePackageInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServicePackageInfoListResponder handles the response to the GetDeployedServicePackageInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServicePackageInfoListResponder(resp *http.Response) (result ListDeployedServicePackageInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServicePackageInfoListByName returns the information about the service packages deployed on a Service
// Fabric node for the given application. These results are of service packages whose name match exactly the service
// package name specified as the parameter.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// servicePackageName - the name of the service package.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServicePackageInfoListByName(ctx context.Context, nodeName string, applicationID string, servicePackageName string, timeout *int64) (result ListDeployedServicePackageInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServicePackageInfoListByName", err.Error())
	}

	req, err := client.GetDeployedServicePackageInfoListByNamePreparer(ctx, nodeName, applicationID, servicePackageName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageInfoListByName", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServicePackageInfoListByNameSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageInfoListByName", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServicePackageInfoListByNameResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServicePackageInfoListByName", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServicePackageInfoListByNamePreparer prepares the GetDeployedServicePackageInfoListByName request.
func (client BaseClient) GetDeployedServicePackageInfoListByNamePreparer(ctx context.Context, nodeName string, applicationID string, servicePackageName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId":      applicationID,
		"nodeName":           autorest.Encode("path", nodeName),
		"servicePackageName": servicePackageName,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServicePackages/{servicePackageName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServicePackageInfoListByNameSender sends the GetDeployedServicePackageInfoListByName request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServicePackageInfoListByNameSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServicePackageInfoListByNameResponder handles the response to the GetDeployedServicePackageInfoListByName request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServicePackageInfoListByNameResponder(resp *http.Response) (result ListDeployedServicePackageInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServiceReplicaDetailInfo gets the details of the replica deployed on a Service Fabric node. The
// information include service kind, service name, current service operation, current service operation start date
// time, partition id, replica/instance id, reported load and other information.
// Parameters:
// nodeName - the name of the node.
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServiceReplicaDetailInfo(ctx context.Context, nodeName string, partitionID uuid.UUID, replicaID string, timeout *int64) (result DeployedServiceReplicaDetailInfoModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServiceReplicaDetailInfo", err.Error())
	}

	req, err := client.GetDeployedServiceReplicaDetailInfoPreparer(ctx, nodeName, partitionID, replicaID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceReplicaDetailInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServiceReplicaDetailInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceReplicaDetailInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServiceReplicaDetailInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceReplicaDetailInfo", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServiceReplicaDetailInfoPreparer prepares the GetDeployedServiceReplicaDetailInfo request.
func (client BaseClient) GetDeployedServiceReplicaDetailInfoPreparer(ctx context.Context, nodeName string, partitionID uuid.UUID, replicaID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName":    autorest.Encode("path", nodeName),
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetPartitions/{partitionId}/$/GetReplicas/{replicaId}/$/GetDetail", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServiceReplicaDetailInfoSender sends the GetDeployedServiceReplicaDetailInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServiceReplicaDetailInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServiceReplicaDetailInfoResponder handles the response to the GetDeployedServiceReplicaDetailInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServiceReplicaDetailInfoResponder(resp *http.Response) (result DeployedServiceReplicaDetailInfoModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServiceReplicaInfoList gets the list containing the information about replicas deployed on a Service
// Fabric node. The information include partition id, replica id, status of the replica, name of the service, name of
// the service type and other information. Use PartitionId or ServiceManifestName query parameters to return
// information about the deployed replicas matching the specified values for those parameters.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// serviceManifestName - the name of a service manifest registered as part of an application type in a Service
// Fabric cluster.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServiceReplicaInfoList(ctx context.Context, nodeName string, applicationID string, partitionID *uuid.UUID, serviceManifestName string, timeout *int64) (result ListDeployedServiceReplicaInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServiceReplicaInfoList", err.Error())
	}

	req, err := client.GetDeployedServiceReplicaInfoListPreparer(ctx, nodeName, applicationID, partitionID, serviceManifestName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceReplicaInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServiceReplicaInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceReplicaInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServiceReplicaInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceReplicaInfoList", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServiceReplicaInfoListPreparer prepares the GetDeployedServiceReplicaInfoList request.
func (client BaseClient) GetDeployedServiceReplicaInfoListPreparer(ctx context.Context, nodeName string, applicationID string, partitionID *uuid.UUID, serviceManifestName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if partitionID != nil {
		queryParameters["PartitionId"] = autorest.Encode("query", *partitionID)
	}
	if len(serviceManifestName) > 0 {
		queryParameters["ServiceManifestName"] = autorest.Encode("query", serviceManifestName)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetReplicas", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServiceReplicaInfoListSender sends the GetDeployedServiceReplicaInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServiceReplicaInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServiceReplicaInfoListResponder handles the response to the GetDeployedServiceReplicaInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServiceReplicaInfoListResponder(resp *http.Response) (result ListDeployedServiceReplicaInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServiceTypeInfoByName gets the information about a specified service type of the application deployed on
// a node in a Service Fabric cluster. The response includes the name of the service type, its registration status, the
// code package that registered it and activation id of the service package.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceTypeName - specifies the name of a Service Fabric service type.
// serviceManifestName - the name of the service manifest to filter the list of deployed service type
// information. If specified, the response will only contain the information about service types that are
// defined in this service manifest.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServiceTypeInfoByName(ctx context.Context, nodeName string, applicationID string, serviceTypeName string, serviceManifestName string, timeout *int64) (result DeployedServiceTypeInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServiceTypeInfoByName", err.Error())
	}

	req, err := client.GetDeployedServiceTypeInfoByNamePreparer(ctx, nodeName, applicationID, serviceTypeName, serviceManifestName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceTypeInfoByName", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServiceTypeInfoByNameSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceTypeInfoByName", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServiceTypeInfoByNameResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceTypeInfoByName", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServiceTypeInfoByNamePreparer prepares the GetDeployedServiceTypeInfoByName request.
func (client BaseClient) GetDeployedServiceTypeInfoByNamePreparer(ctx context.Context, nodeName string, applicationID string, serviceTypeName string, serviceManifestName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId":   applicationID,
		"nodeName":        autorest.Encode("path", nodeName),
		"serviceTypeName": serviceTypeName,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(serviceManifestName) > 0 {
		queryParameters["ServiceManifestName"] = autorest.Encode("query", serviceManifestName)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServiceTypes/{serviceTypeName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServiceTypeInfoByNameSender sends the GetDeployedServiceTypeInfoByName request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServiceTypeInfoByNameSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServiceTypeInfoByNameResponder handles the response to the GetDeployedServiceTypeInfoByName request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServiceTypeInfoByNameResponder(resp *http.Response) (result DeployedServiceTypeInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeployedServiceTypeInfoList gets the list containing the information about service types from the applications
// deployed on a node in a Service Fabric cluster. The response includes the name of the service type, its registration
// status, the code package that registered it and activation id of the service package.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceManifestName - the name of the service manifest to filter the list of deployed service type
// information. If specified, the response will only contain the information about service types that are
// defined in this service manifest.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetDeployedServiceTypeInfoList(ctx context.Context, nodeName string, applicationID string, serviceManifestName string, timeout *int64) (result ListDeployedServiceTypeInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetDeployedServiceTypeInfoList", err.Error())
	}

	req, err := client.GetDeployedServiceTypeInfoListPreparer(ctx, nodeName, applicationID, serviceManifestName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceTypeInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeployedServiceTypeInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceTypeInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeployedServiceTypeInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetDeployedServiceTypeInfoList", resp, "Failure responding to request")
	}

	return
}

// GetDeployedServiceTypeInfoListPreparer prepares the GetDeployedServiceTypeInfoList request.
func (client BaseClient) GetDeployedServiceTypeInfoListPreparer(ctx context.Context, nodeName string, applicationID string, serviceManifestName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(serviceManifestName) > 0 {
		queryParameters["ServiceManifestName"] = autorest.Encode("query", serviceManifestName)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServiceTypes", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeployedServiceTypeInfoListSender sends the GetDeployedServiceTypeInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeployedServiceTypeInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDeployedServiceTypeInfoListResponder handles the response to the GetDeployedServiceTypeInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeployedServiceTypeInfoListResponder(resp *http.Response) (result ListDeployedServiceTypeInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetFaultOperationList gets the a list of user-induced fault operations filtered by provided input.
// Parameters:
// typeFilter - used to filter on OperationType for user-induced operations.
// 65535 - select all
// 1     - select PartitionDataLoss.
// 2     - select PartitionQuorumLoss.
// 4     - select PartitionRestart.
// 8     - select NodeTransition.
// stateFilter - used to filter on OperationState's for user-induced operations.
// 65535 - select All
// 1     - select Running
// 2     - select RollingBack
// 8     - select Completed
// 16    - select Faulted
// 32    - select Cancelled
// 64    - select ForceCancelled
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetFaultOperationList(ctx context.Context, typeFilter int32, stateFilter int32, timeout *int64) (result ListOperationStatus, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetFaultOperationList", err.Error())
	}

	req, err := client.GetFaultOperationListPreparer(ctx, typeFilter, stateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetFaultOperationList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetFaultOperationListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetFaultOperationList", resp, "Failure sending request")
		return
	}

	result, err = client.GetFaultOperationListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetFaultOperationList", resp, "Failure responding to request")
	}

	return
}

// GetFaultOperationListPreparer prepares the GetFaultOperationList request.
func (client BaseClient) GetFaultOperationListPreparer(ctx context.Context, typeFilter int32, stateFilter int32, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"StateFilter": autorest.Encode("query", stateFilter),
		"TypeFilter":  autorest.Encode("query", typeFilter),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Faults/"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetFaultOperationListSender sends the GetFaultOperationList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetFaultOperationListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetFaultOperationListResponder handles the response to the GetFaultOperationList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetFaultOperationListResponder(resp *http.Response) (result ListOperationStatus, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetImageStoreContent returns the information about the image store content at the specified contentPath relative to
// the root of the image store.
// Parameters:
// contentPath - relative path to file or folder in the image store from its root.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetImageStoreContent(ctx context.Context, contentPath string, timeout *int64) (result ImageStoreContent, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetImageStoreContent", err.Error())
	}

	req, err := client.GetImageStoreContentPreparer(ctx, contentPath, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetImageStoreContent", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetImageStoreContentSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetImageStoreContent", resp, "Failure sending request")
		return
	}

	result, err = client.GetImageStoreContentResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetImageStoreContent", resp, "Failure responding to request")
	}

	return
}

// GetImageStoreContentPreparer prepares the GetImageStoreContent request.
func (client BaseClient) GetImageStoreContentPreparer(ctx context.Context, contentPath string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"contentPath": autorest.Encode("path", contentPath),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ImageStore/{contentPath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetImageStoreContentSender sends the GetImageStoreContent request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetImageStoreContentSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetImageStoreContentResponder handles the response to the GetImageStoreContent request. The method always
// closes the http.Response Body.
func (client BaseClient) GetImageStoreContentResponder(resp *http.Response) (result ImageStoreContent, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetImageStoreRootContent returns the information about the image store content at the root of the image store.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetImageStoreRootContent(ctx context.Context, timeout *int64) (result ImageStoreContent, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetImageStoreRootContent", err.Error())
	}

	req, err := client.GetImageStoreRootContentPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetImageStoreRootContent", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetImageStoreRootContentSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetImageStoreRootContent", resp, "Failure sending request")
		return
	}

	result, err = client.GetImageStoreRootContentResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetImageStoreRootContent", resp, "Failure responding to request")
	}

	return
}

// GetImageStoreRootContentPreparer prepares the GetImageStoreRootContent request.
func (client BaseClient) GetImageStoreRootContentPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/ImageStore"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetImageStoreRootContentSender sends the GetImageStoreRootContent request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetImageStoreRootContentSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetImageStoreRootContentResponder handles the response to the GetImageStoreRootContent request. The method always
// closes the http.Response Body.
func (client BaseClient) GetImageStoreRootContentResponder(resp *http.Response) (result ImageStoreContent, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetNodeHealth gets the health of a Service Fabric node. Use EventsHealthStateFilter to filter the collection of
// health events reported on the node based on the health state. If the node that you specify by name does not exist in
// the health store, this returns an error.
// Parameters:
// nodeName - the name of the node.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetNodeHealth(ctx context.Context, nodeName string, eventsHealthStateFilter *int32, timeout *int64) (result NodeHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetNodeHealth", err.Error())
	}

	req, err := client.GetNodeHealthPreparer(ctx, nodeName, eventsHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetNodeHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetNodeHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeHealth", resp, "Failure responding to request")
	}

	return
}

// GetNodeHealthPreparer prepares the GetNodeHealth request.
func (client BaseClient) GetNodeHealthPreparer(ctx context.Context, nodeName string, eventsHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetNodeHealthSender sends the GetNodeHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetNodeHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetNodeHealthResponder handles the response to the GetNodeHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetNodeHealthResponder(resp *http.Response) (result NodeHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetNodeHealthUsingPolicy gets the health of a Service Fabric node. Use EventsHealthStateFilter to filter the
// collection of health events reported on the node based on the health state. Use ClusterHealthPolicy in the POST body
// to override the health policies used to evaluate the health. If the node that you specify by name does not exist in
// the health store, this returns an error.
// Parameters:
// nodeName - the name of the node.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// clusterHealthPolicy - describes the health policies used to evaluate the health of a cluster or node. If not
// present, the health evaluation uses the health policy from cluster manifest or the default health policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetNodeHealthUsingPolicy(ctx context.Context, nodeName string, eventsHealthStateFilter *int32, clusterHealthPolicy *ClusterHealthPolicy, timeout *int64) (result NodeHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetNodeHealthUsingPolicy", err.Error())
	}

	req, err := client.GetNodeHealthUsingPolicyPreparer(ctx, nodeName, eventsHealthStateFilter, clusterHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetNodeHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetNodeHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetNodeHealthUsingPolicyPreparer prepares the GetNodeHealthUsingPolicy request.
func (client BaseClient) GetNodeHealthUsingPolicyPreparer(ctx context.Context, nodeName string, eventsHealthStateFilter *int32, clusterHealthPolicy *ClusterHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if clusterHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(clusterHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetNodeHealthUsingPolicySender sends the GetNodeHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetNodeHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetNodeHealthUsingPolicyResponder handles the response to the GetNodeHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetNodeHealthUsingPolicyResponder(resp *http.Response) (result NodeHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetNodeInfo gets the information about a specific node in the Service Fabric Cluster.The respons include the name,
// status, id, health, uptime and other details about the node.
// Parameters:
// nodeName - the name of the node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetNodeInfo(ctx context.Context, nodeName string, timeout *int64) (result NodeInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetNodeInfo", err.Error())
	}

	req, err := client.GetNodeInfoPreparer(ctx, nodeName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetNodeInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetNodeInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeInfo", resp, "Failure responding to request")
	}

	return
}

// GetNodeInfoPreparer prepares the GetNodeInfo request.
func (client BaseClient) GetNodeInfoPreparer(ctx context.Context, nodeName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetNodeInfoSender sends the GetNodeInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetNodeInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetNodeInfoResponder handles the response to the GetNodeInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetNodeInfoResponder(resp *http.Response) (result NodeInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetNodeInfoList the Nodes endpoint returns information about the nodes in the Service Fabric Cluster. The respons
// include the name, status, id, health, uptime and other details about the node.
// Parameters:
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// nodeStatusFilter - allows filtering the nodes based on the NodeStatus. Only the nodes that are matching the
// specified filter value will be returned. The filter value can be one of the following.
//
// - default - This filter value will match all of the nodes excepts the ones with with status as Unknown or
// Removed.
// - all - This filter value will match all of the nodes.
// - up - This filter value will match nodes that are Up.
// - down - This filter value will match nodes that are Down.
// - enabling - This filter value will match nodes that are in the process of being enabled with status as
// Enabling.
// - disabling - This filter value will match nodes that are in the process of being disabled with status as
// Disabling.
// - disabled - This filter value will match nodes that are Disabled.
// - unknown - This filter value will match nodes whose status is Unknown. A node would be in Unknown state if
// Service Fabric does not have authoritative information about that node. This can happen if the system learns
// about a node at runtime.
// - removed - This filter value will match nodes whose status is Removed. These are the nodes that are removed
// from the cluster using the RemoveNodeState API.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetNodeInfoList(ctx context.Context, continuationToken string, nodeStatusFilter string, timeout *int64) (result PagedNodeInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetNodeInfoList", err.Error())
	}

	req, err := client.GetNodeInfoListPreparer(ctx, continuationToken, nodeStatusFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetNodeInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetNodeInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeInfoList", resp, "Failure responding to request")
	}

	return
}

// GetNodeInfoListPreparer prepares the GetNodeInfoList request.
func (client BaseClient) GetNodeInfoListPreparer(ctx context.Context, continuationToken string, nodeStatusFilter string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if len(string(nodeStatusFilter)) > 0 {
		queryParameters["NodeStatusFilter"] = autorest.Encode("query", nodeStatusFilter)
	} else {
		queryParameters["NodeStatusFilter"] = autorest.Encode("query", "default")
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Nodes"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetNodeInfoListSender sends the GetNodeInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetNodeInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetNodeInfoListResponder handles the response to the GetNodeInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetNodeInfoListResponder(resp *http.Response) (result PagedNodeInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetNodeLoadInfo gets the load information of a Service Fabric node.
// Parameters:
// nodeName - the name of the node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetNodeLoadInfo(ctx context.Context, nodeName string, timeout *int64) (result NodeLoadInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetNodeLoadInfo", err.Error())
	}

	req, err := client.GetNodeLoadInfoPreparer(ctx, nodeName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeLoadInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetNodeLoadInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeLoadInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetNodeLoadInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeLoadInfo", resp, "Failure responding to request")
	}

	return
}

// GetNodeLoadInfoPreparer prepares the GetNodeLoadInfo request.
func (client BaseClient) GetNodeLoadInfoPreparer(ctx context.Context, nodeName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetLoadInformation", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetNodeLoadInfoSender sends the GetNodeLoadInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetNodeLoadInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetNodeLoadInfoResponder handles the response to the GetNodeLoadInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetNodeLoadInfoResponder(resp *http.Response) (result NodeLoadInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetNodeTransitionProgress gets the progress of an operation started with StartNodeTransition using the provided
// OperationId.
// Parameters:
// nodeName - the name of the node.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetNodeTransitionProgress(ctx context.Context, nodeName string, operationID uuid.UUID, timeout *int64) (result NodeTransitionProgress, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetNodeTransitionProgress", err.Error())
	}

	req, err := client.GetNodeTransitionProgressPreparer(ctx, nodeName, operationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeTransitionProgress", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetNodeTransitionProgressSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeTransitionProgress", resp, "Failure sending request")
		return
	}

	result, err = client.GetNodeTransitionProgressResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetNodeTransitionProgress", resp, "Failure responding to request")
	}

	return
}

// GetNodeTransitionProgressPreparer prepares the GetNodeTransitionProgress request.
func (client BaseClient) GetNodeTransitionProgressPreparer(ctx context.Context, nodeName string, operationID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"OperationId": autorest.Encode("query", operationID),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Nodes/{nodeName}/$/GetTransitionProgress", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetNodeTransitionProgressSender sends the GetNodeTransitionProgress request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetNodeTransitionProgressSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetNodeTransitionProgressResponder handles the response to the GetNodeTransitionProgress request. The method always
// closes the http.Response Body.
func (client BaseClient) GetNodeTransitionProgressResponder(resp *http.Response) (result NodeTransitionProgress, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPartitionHealth gets the health information of the specified partition.
// Use EventsHealthStateFilter to filter the collection of health events reported on the service based on the health
// state.
// Use ReplicasHealthStateFilter to filter the collection of ReplicaHealthState objects on the partition.
// If you specify a partition that does not exist in the health store, this cmdlet returns an error.
// Parameters:
// partitionID - the identity of the partition.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// replicasHealthStateFilter - allows filtering the collection of ReplicaHealthState objects on the partition.
// The value can be obtained from members or bitwise operations on members of HealthStateFilter. Only replicas
// that match the filter will be returned. All replicas will be used to evaluate the aggregated health state.
// If not specified, all entries will be returned.The state values are flag based enumeration, so the value
// could be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided
// value is 6 then all of the events with HealthState value of OK (2) and Warning (4) will be returned. The
// possible values for this parameter include integer value of one of the following health states.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetPartitionHealth(ctx context.Context, partitionID uuid.UUID, eventsHealthStateFilter *int32, replicasHealthStateFilter *int32, timeout *int64) (result PartitionHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetPartitionHealth", err.Error())
	}

	req, err := client.GetPartitionHealthPreparer(ctx, partitionID, eventsHealthStateFilter, replicasHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPartitionHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetPartitionHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionHealth", resp, "Failure responding to request")
	}

	return
}

// GetPartitionHealthPreparer prepares the GetPartitionHealth request.
func (client BaseClient) GetPartitionHealthPreparer(ctx context.Context, partitionID uuid.UUID, eventsHealthStateFilter *int32, replicasHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if replicasHealthStateFilter != nil {
		queryParameters["ReplicasHealthStateFilter"] = autorest.Encode("query", *replicasHealthStateFilter)
	} else {
		queryParameters["ReplicasHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPartitionHealthSender sends the GetPartitionHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetPartitionHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPartitionHealthResponder handles the response to the GetPartitionHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetPartitionHealthResponder(resp *http.Response) (result PartitionHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPartitionHealthUsingPolicy gets the health information of the specified partition.
// If the application health policy is specified, the health evaluation uses it to get the aggregated health state.
// If the policy is not specified, the health evaluation uses the application health policy defined in the application
// manifest, or the default health policy, if no policy is defined in the manifest.
// Use EventsHealthStateFilter to filter the collection of health events reported on the partition based on the health
// state.
// Use ReplicasHealthStateFilter to filter the collection of ReplicaHealthState objects on the partition. Use
// ApplicationHealthPolicy in the POST body to override the health policies used to evaluate the health.
// If you specify a partition that does not exist in the health store, this cmdlet returns an error.
// Parameters:
// partitionID - the identity of the partition.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// replicasHealthStateFilter - allows filtering the collection of ReplicaHealthState objects on the partition.
// The value can be obtained from members or bitwise operations on members of HealthStateFilter. Only replicas
// that match the filter will be returned. All replicas will be used to evaluate the aggregated health state.
// If not specified, all entries will be returned.The state values are flag based enumeration, so the value
// could be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided
// value is 6 then all of the events with HealthState value of OK (2) and Warning (4) will be returned. The
// possible values for this parameter include integer value of one of the following health states.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationHealthPolicy - describes the health policies used to evaluate the health of an application or one
// of its children.
// If not present, the health evaluation uses the health policy from application manifest or the default health
// policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetPartitionHealthUsingPolicy(ctx context.Context, partitionID uuid.UUID, eventsHealthStateFilter *int32, replicasHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (result PartitionHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetPartitionHealthUsingPolicy", err.Error())
	}

	req, err := client.GetPartitionHealthUsingPolicyPreparer(ctx, partitionID, eventsHealthStateFilter, replicasHealthStateFilter, applicationHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPartitionHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetPartitionHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetPartitionHealthUsingPolicyPreparer prepares the GetPartitionHealthUsingPolicy request.
func (client BaseClient) GetPartitionHealthUsingPolicyPreparer(ctx context.Context, partitionID uuid.UUID, eventsHealthStateFilter *int32, replicasHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if replicasHealthStateFilter != nil {
		queryParameters["ReplicasHealthStateFilter"] = autorest.Encode("query", *replicasHealthStateFilter)
	} else {
		queryParameters["ReplicasHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if applicationHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(applicationHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPartitionHealthUsingPolicySender sends the GetPartitionHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetPartitionHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPartitionHealthUsingPolicyResponder handles the response to the GetPartitionHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetPartitionHealthUsingPolicyResponder(resp *http.Response) (result PartitionHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPartitionInfo the Partitions endpoint returns information about the specified partition. The response include the
// partition id, partitioning scheme information, keys supported by the partition, status, health and other details
// about the partition.
// Parameters:
// partitionID - the identity of the partition.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetPartitionInfo(ctx context.Context, partitionID uuid.UUID, timeout *int64) (result ServicePartitionInfoModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetPartitionInfo", err.Error())
	}

	req, err := client.GetPartitionInfoPreparer(ctx, partitionID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPartitionInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetPartitionInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionInfo", resp, "Failure responding to request")
	}

	return
}

// GetPartitionInfoPreparer prepares the GetPartitionInfo request.
func (client BaseClient) GetPartitionInfoPreparer(ctx context.Context, partitionID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPartitionInfoSender sends the GetPartitionInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetPartitionInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPartitionInfoResponder handles the response to the GetPartitionInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetPartitionInfoResponder(resp *http.Response) (result ServicePartitionInfoModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPartitionInfoList gets the list of partitions of a Service Fabric service. The response include the partition id,
// partitioning scheme information, keys supported by the partition, status, health and other details about the
// partition.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetPartitionInfoList(ctx context.Context, serviceID string, continuationToken string, timeout *int64) (result PagedServicePartitionInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetPartitionInfoList", err.Error())
	}

	req, err := client.GetPartitionInfoListPreparer(ctx, serviceID, continuationToken, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPartitionInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetPartitionInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionInfoList", resp, "Failure responding to request")
	}

	return
}

// GetPartitionInfoListPreparer prepares the GetPartitionInfoList request.
func (client BaseClient) GetPartitionInfoListPreparer(ctx context.Context, serviceID string, continuationToken string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/GetPartitions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPartitionInfoListSender sends the GetPartitionInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetPartitionInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPartitionInfoListResponder handles the response to the GetPartitionInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetPartitionInfoListResponder(resp *http.Response) (result PagedServicePartitionInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPartitionLoadInformation returns information about the specified partition.
// The response includes a list of load information.
// Each information includes load metric name, value and last reported time in UTC.
// Parameters:
// partitionID - the identity of the partition.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetPartitionLoadInformation(ctx context.Context, partitionID uuid.UUID, timeout *int64) (result PartitionLoadInformation, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetPartitionLoadInformation", err.Error())
	}

	req, err := client.GetPartitionLoadInformationPreparer(ctx, partitionID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionLoadInformation", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPartitionLoadInformationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionLoadInformation", resp, "Failure sending request")
		return
	}

	result, err = client.GetPartitionLoadInformationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionLoadInformation", resp, "Failure responding to request")
	}

	return
}

// GetPartitionLoadInformationPreparer prepares the GetPartitionLoadInformation request.
func (client BaseClient) GetPartitionLoadInformationPreparer(ctx context.Context, partitionID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetLoadInformation", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPartitionLoadInformationSender sends the GetPartitionLoadInformation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetPartitionLoadInformationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPartitionLoadInformationResponder handles the response to the GetPartitionLoadInformation request. The method always
// closes the http.Response Body.
func (client BaseClient) GetPartitionLoadInformationResponder(resp *http.Response) (result PartitionLoadInformation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPartitionRestartProgress gets the progress of a PartitionRestart started with StartPartitionRestart using the
// provided OperationId.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetPartitionRestartProgress(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, timeout *int64) (result PartitionRestartProgress, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetPartitionRestartProgress", err.Error())
	}

	req, err := client.GetPartitionRestartProgressPreparer(ctx, serviceID, partitionID, operationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionRestartProgress", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPartitionRestartProgressSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionRestartProgress", resp, "Failure sending request")
		return
	}

	result, err = client.GetPartitionRestartProgressResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetPartitionRestartProgress", resp, "Failure responding to request")
	}

	return
}

// GetPartitionRestartProgressPreparer prepares the GetPartitionRestartProgress request.
func (client BaseClient) GetPartitionRestartProgressPreparer(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"serviceId":   serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"OperationId": autorest.Encode("query", operationID),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Services/{serviceId}/$/GetPartitions/{partitionId}/$/GetRestartProgress", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPartitionRestartProgressSender sends the GetPartitionRestartProgress request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetPartitionRestartProgressSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPartitionRestartProgressResponder handles the response to the GetPartitionRestartProgress request. The method always
// closes the http.Response Body.
func (client BaseClient) GetPartitionRestartProgressResponder(resp *http.Response) (result PartitionRestartProgress, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetProvisionedFabricCodeVersionInfoList gets a list of information about fabric code versions that are provisioned
// in the cluster. The parameter CodeVersion can be used to optionally filter the output to only that particular
// version.
// Parameters:
// codeVersion - the product version of Service Fabric.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetProvisionedFabricCodeVersionInfoList(ctx context.Context, codeVersion string, timeout *int64) (result ListFabricCodeVersionInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetProvisionedFabricCodeVersionInfoList", err.Error())
	}

	req, err := client.GetProvisionedFabricCodeVersionInfoListPreparer(ctx, codeVersion, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetProvisionedFabricCodeVersionInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetProvisionedFabricCodeVersionInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetProvisionedFabricCodeVersionInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetProvisionedFabricCodeVersionInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetProvisionedFabricCodeVersionInfoList", resp, "Failure responding to request")
	}

	return
}

// GetProvisionedFabricCodeVersionInfoListPreparer prepares the GetProvisionedFabricCodeVersionInfoList request.
func (client BaseClient) GetProvisionedFabricCodeVersionInfoListPreparer(ctx context.Context, codeVersion string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(codeVersion) > 0 {
		queryParameters["CodeVersion"] = autorest.Encode("query", codeVersion)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetProvisionedCodeVersions"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetProvisionedFabricCodeVersionInfoListSender sends the GetProvisionedFabricCodeVersionInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetProvisionedFabricCodeVersionInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetProvisionedFabricCodeVersionInfoListResponder handles the response to the GetProvisionedFabricCodeVersionInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetProvisionedFabricCodeVersionInfoListResponder(resp *http.Response) (result ListFabricCodeVersionInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetProvisionedFabricConfigVersionInfoList gets a list of information about fabric config versions that are
// provisioned in the cluster. The parameter ConfigVersion can be used to optionally filter the output to only that
// particular version.
// Parameters:
// configVersion - the config version of Service Fabric.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetProvisionedFabricConfigVersionInfoList(ctx context.Context, configVersion string, timeout *int64) (result ListFabricConfigVersionInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetProvisionedFabricConfigVersionInfoList", err.Error())
	}

	req, err := client.GetProvisionedFabricConfigVersionInfoListPreparer(ctx, configVersion, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetProvisionedFabricConfigVersionInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetProvisionedFabricConfigVersionInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetProvisionedFabricConfigVersionInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetProvisionedFabricConfigVersionInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetProvisionedFabricConfigVersionInfoList", resp, "Failure responding to request")
	}

	return
}

// GetProvisionedFabricConfigVersionInfoListPreparer prepares the GetProvisionedFabricConfigVersionInfoList request.
func (client BaseClient) GetProvisionedFabricConfigVersionInfoListPreparer(ctx context.Context, configVersion string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(configVersion) > 0 {
		queryParameters["ConfigVersion"] = autorest.Encode("query", configVersion)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/GetProvisionedConfigVersions"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetProvisionedFabricConfigVersionInfoListSender sends the GetProvisionedFabricConfigVersionInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetProvisionedFabricConfigVersionInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetProvisionedFabricConfigVersionInfoListResponder handles the response to the GetProvisionedFabricConfigVersionInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetProvisionedFabricConfigVersionInfoListResponder(resp *http.Response) (result ListFabricConfigVersionInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetQuorumLossProgress gets the progress of a quorum loss operation started with StartQuorumLoss, using the provided
// OperationId.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetQuorumLossProgress(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, timeout *int64) (result PartitionQuorumLossProgress, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetQuorumLossProgress", err.Error())
	}

	req, err := client.GetQuorumLossProgressPreparer(ctx, serviceID, partitionID, operationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetQuorumLossProgress", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetQuorumLossProgressSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetQuorumLossProgress", resp, "Failure sending request")
		return
	}

	result, err = client.GetQuorumLossProgressResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetQuorumLossProgress", resp, "Failure responding to request")
	}

	return
}

// GetQuorumLossProgressPreparer prepares the GetQuorumLossProgress request.
func (client BaseClient) GetQuorumLossProgressPreparer(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"serviceId":   serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"OperationId": autorest.Encode("query", operationID),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Services/{serviceId}/$/GetPartitions/{partitionId}/$/GetQuorumLossProgress", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetQuorumLossProgressSender sends the GetQuorumLossProgress request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetQuorumLossProgressSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetQuorumLossProgressResponder handles the response to the GetQuorumLossProgress request. The method always
// closes the http.Response Body.
func (client BaseClient) GetQuorumLossProgressResponder(resp *http.Response) (result PartitionQuorumLossProgress, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetReplicaHealth gets the health of a Service Fabric replica.
// Use EventsHealthStateFilter to filter the collection of health events reported on the replica based on the health
// state.
// Parameters:
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetReplicaHealth(ctx context.Context, partitionID uuid.UUID, replicaID string, eventsHealthStateFilter *int32, timeout *int64) (result ReplicaHealthModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetReplicaHealth", err.Error())
	}

	req, err := client.GetReplicaHealthPreparer(ctx, partitionID, replicaID, eventsHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetReplicaHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetReplicaHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaHealth", resp, "Failure responding to request")
	}

	return
}

// GetReplicaHealthPreparer prepares the GetReplicaHealth request.
func (client BaseClient) GetReplicaHealthPreparer(ctx context.Context, partitionID uuid.UUID, replicaID string, eventsHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetReplicas/{replicaId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetReplicaHealthSender sends the GetReplicaHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetReplicaHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetReplicaHealthResponder handles the response to the GetReplicaHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetReplicaHealthResponder(resp *http.Response) (result ReplicaHealthModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetReplicaHealthUsingPolicy gets the health of a Service Fabric stateful service replica or stateless service
// instance.
// Use EventsHealthStateFilter to filter the collection of health events reported on the cluster based on the health
// state.
// Use ApplicationHealthPolicy to optionally override the health policies used to evaluate the health. This API only
// uses 'ConsiderWarningAsError' field of the ApplicationHealthPolicy. The rest of the fields are ignored while
// evaluating the health of the replica.
// Parameters:
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationHealthPolicy - describes the health policies used to evaluate the health of an application or one
// of its children.
// If not present, the health evaluation uses the health policy from application manifest or the default health
// policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetReplicaHealthUsingPolicy(ctx context.Context, partitionID uuid.UUID, replicaID string, eventsHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (result ReplicaHealthModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetReplicaHealthUsingPolicy", err.Error())
	}

	req, err := client.GetReplicaHealthUsingPolicyPreparer(ctx, partitionID, replicaID, eventsHealthStateFilter, applicationHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetReplicaHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetReplicaHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetReplicaHealthUsingPolicyPreparer prepares the GetReplicaHealthUsingPolicy request.
func (client BaseClient) GetReplicaHealthUsingPolicyPreparer(ctx context.Context, partitionID uuid.UUID, replicaID string, eventsHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetReplicas/{replicaId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if applicationHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(applicationHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetReplicaHealthUsingPolicySender sends the GetReplicaHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetReplicaHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetReplicaHealthUsingPolicyResponder handles the response to the GetReplicaHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetReplicaHealthUsingPolicyResponder(resp *http.Response) (result ReplicaHealthModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetReplicaInfo the respons include the id, role, status, health, node name, uptime, and other details about the
// replica.
// Parameters:
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetReplicaInfo(ctx context.Context, partitionID uuid.UUID, replicaID string, continuationToken string, timeout *int64) (result ReplicaInfoModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetReplicaInfo", err.Error())
	}

	req, err := client.GetReplicaInfoPreparer(ctx, partitionID, replicaID, continuationToken, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetReplicaInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetReplicaInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaInfo", resp, "Failure responding to request")
	}

	return
}

// GetReplicaInfoPreparer prepares the GetReplicaInfo request.
func (client BaseClient) GetReplicaInfoPreparer(ctx context.Context, partitionID uuid.UUID, replicaID string, continuationToken string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetReplicas/{replicaId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetReplicaInfoSender sends the GetReplicaInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetReplicaInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetReplicaInfoResponder handles the response to the GetReplicaInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetReplicaInfoResponder(resp *http.Response) (result ReplicaInfoModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetReplicaInfoList the GetReplicas endpoint returns information about the replicas of the specified partition. The
// respons include the id, role, status, health, node name, uptime, and other details about the replica.
// Parameters:
// partitionID - the identity of the partition.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetReplicaInfoList(ctx context.Context, partitionID uuid.UUID, continuationToken string, timeout *int64) (result PagedReplicaInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetReplicaInfoList", err.Error())
	}

	req, err := client.GetReplicaInfoListPreparer(ctx, partitionID, continuationToken, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetReplicaInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetReplicaInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetReplicaInfoList", resp, "Failure responding to request")
	}

	return
}

// GetReplicaInfoListPreparer prepares the GetReplicaInfoList request.
func (client BaseClient) GetReplicaInfoListPreparer(ctx context.Context, partitionID uuid.UUID, continuationToken string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetReplicas", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetReplicaInfoListSender sends the GetReplicaInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetReplicaInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetReplicaInfoListResponder handles the response to the GetReplicaInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetReplicaInfoListResponder(resp *http.Response) (result PagedReplicaInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceDescription gets the description of an existing Service Fabric service. A service must be created before
// its description can be obtained.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceDescription(ctx context.Context, serviceID string, timeout *int64) (result ServiceDescriptionModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceDescription", err.Error())
	}

	req, err := client.GetServiceDescriptionPreparer(ctx, serviceID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceDescription", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceDescriptionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceDescription", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceDescriptionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceDescription", resp, "Failure responding to request")
	}

	return
}

// GetServiceDescriptionPreparer prepares the GetServiceDescription request.
func (client BaseClient) GetServiceDescriptionPreparer(ctx context.Context, serviceID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/GetDescription", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceDescriptionSender sends the GetServiceDescription request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceDescriptionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceDescriptionResponder handles the response to the GetServiceDescription request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceDescriptionResponder(resp *http.Response) (result ServiceDescriptionModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceHealth gets the health information of the specified service.
// Use EventsHealthStateFilter to filter the collection of health events reported on the service based on the health
// state.
// Use PartitionsHealthStateFilter to filter the collection of partitions returned.
// If you specify a service that does not exist in the health store, this cmdlet returns an error.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// partitionsHealthStateFilter - allows filtering of the partitions health state objects returned in the result
// of service health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states.
// Only partitions that match the filter are returned. All partitions are used to evaluate the aggregated
// health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value
// obtained using bitwise 'OR' operator. For example, if the provided value is 6 then health state of
// partitions with HealthState value of OK (2) and Warning (4) will be returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn't match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceHealth(ctx context.Context, serviceID string, eventsHealthStateFilter *int32, partitionsHealthStateFilter *int32, timeout *int64) (result ServiceHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceHealth", err.Error())
	}

	req, err := client.GetServiceHealthPreparer(ctx, serviceID, eventsHealthStateFilter, partitionsHealthStateFilter, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceHealthSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceHealth", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceHealth", resp, "Failure responding to request")
	}

	return
}

// GetServiceHealthPreparer prepares the GetServiceHealth request.
func (client BaseClient) GetServiceHealthPreparer(ctx context.Context, serviceID string, eventsHealthStateFilter *int32, partitionsHealthStateFilter *int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if partitionsHealthStateFilter != nil {
		queryParameters["PartitionsHealthStateFilter"] = autorest.Encode("query", *partitionsHealthStateFilter)
	} else {
		queryParameters["PartitionsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceHealthSender sends the GetServiceHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceHealthResponder handles the response to the GetServiceHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceHealthResponder(resp *http.Response) (result ServiceHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceHealthUsingPolicy gets the health information of the specified service.
// If the application health policy is specified, the health evaluation uses it to get the aggregated health state.
// If the policy is not specified, the health evaluation uses the application health policy defined in the application
// manifest, or the default health policy, if no policy is defined in the manifest.
// Use EventsHealthStateFilter to filter the collection of health events reported on the service based on the health
// state.
// Use PartitionsHealthStateFilter to filter the collection of partitions returned.
// If you specify a service that does not exist in the health store, this cmdlet returns an error.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// eventsHealthStateFilter - allows filtering the collection of HealthEvent objects returned based on health
// state.
// The possible values for this parameter include integer value of one of the following health states.
// Only events that match the filter are returned. All events are used to evaluate the aggregated health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value obtained using bitwise 'OR' operator. For example, If the provided value is
// 6 then all of the events with HealthState value of OK (2) and Warning (4) are returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn’t match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// partitionsHealthStateFilter - allows filtering of the partitions health state objects returned in the result
// of service health query based on their health state.
// The possible values for this parameter include integer value of one of the following health states.
// Only partitions that match the filter are returned. All partitions are used to evaluate the aggregated
// health state.
// If not specified, all entries are returned. The state values are flag based enumeration, so the value could
// be a combination of these value
// obtained using bitwise 'OR' operator. For example, if the provided value is 6 then health state of
// partitions with HealthState value of OK (2) and Warning (4) will be returned.
//
// - Default - Default value. Matches any HealthState. The value is zero.
// - None - Filter that doesn't match any HealthState value. Used in order to return no results on a given
// collection of states. The value is 1.
// - Ok - Filter that matches input with HealthState value Ok. The value is 2.
// - Warning - Filter that matches input with HealthState value Warning. The value is 4.
// - Error - Filter that matches input with HealthState value Error. The value is 8.
// - All - Filter that matches input with any HealthState value. The value is 65535.
// applicationHealthPolicy - describes the health policies used to evaluate the health of an application or one
// of its children.
// If not present, the health evaluation uses the health policy from application manifest or the default health
// policy.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceHealthUsingPolicy(ctx context.Context, serviceID string, eventsHealthStateFilter *int32, partitionsHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (result ServiceHealth, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceHealthUsingPolicy", err.Error())
	}

	req, err := client.GetServiceHealthUsingPolicyPreparer(ctx, serviceID, eventsHealthStateFilter, partitionsHealthStateFilter, applicationHealthPolicy, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceHealthUsingPolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceHealthUsingPolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceHealthUsingPolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceHealthUsingPolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceHealthUsingPolicy", resp, "Failure responding to request")
	}

	return
}

// GetServiceHealthUsingPolicyPreparer prepares the GetServiceHealthUsingPolicy request.
func (client BaseClient) GetServiceHealthUsingPolicyPreparer(ctx context.Context, serviceID string, eventsHealthStateFilter *int32, partitionsHealthStateFilter *int32, applicationHealthPolicy *ApplicationHealthPolicy, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if eventsHealthStateFilter != nil {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", *eventsHealthStateFilter)
	} else {
		queryParameters["EventsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if partitionsHealthStateFilter != nil {
		queryParameters["PartitionsHealthStateFilter"] = autorest.Encode("query", *partitionsHealthStateFilter)
	} else {
		queryParameters["PartitionsHealthStateFilter"] = autorest.Encode("query", 0)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/GetHealth", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if applicationHealthPolicy != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(applicationHealthPolicy))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceHealthUsingPolicySender sends the GetServiceHealthUsingPolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceHealthUsingPolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceHealthUsingPolicyResponder handles the response to the GetServiceHealthUsingPolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceHealthUsingPolicyResponder(resp *http.Response) (result ServiceHealth, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceInfo returns the information about specified service belonging to the specified Service Fabric
// application.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceInfo(ctx context.Context, applicationID string, serviceID string, timeout *int64) (result ServiceInfoModel, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceInfo", err.Error())
	}

	req, err := client.GetServiceInfoPreparer(ctx, applicationID, serviceID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceInfo", resp, "Failure responding to request")
	}

	return
}

// GetServiceInfoPreparer prepares the GetServiceInfo request.
func (client BaseClient) GetServiceInfoPreparer(ctx context.Context, applicationID string, serviceID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"serviceId":     serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetServices/{serviceId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceInfoSender sends the GetServiceInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceInfoResponder handles the response to the GetServiceInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceInfoResponder(resp *http.Response) (result ServiceInfoModel, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceInfoList returns the information about all services belonging to the application specified by the
// application id.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// serviceTypeName - the service type name used to filter the services to query for.
// continuationToken - the continuation token parameter is used to obtain next set of results. A continuation
// token with a non empty value is included in the response of the API when the results from the system do not
// fit in a single response. When this value is passed to the next API call, the API returns next set of
// results. If there are no further results then the continuation token does not contain a value. The value of
// this parameter should not be URL encoded.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceInfoList(ctx context.Context, applicationID string, serviceTypeName string, continuationToken string, timeout *int64) (result PagedServiceInfoList, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceInfoList", err.Error())
	}

	req, err := client.GetServiceInfoListPreparer(ctx, applicationID, serviceTypeName, continuationToken, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceInfoList", resp, "Failure responding to request")
	}

	return
}

// GetServiceInfoListPreparer prepares the GetServiceInfoList request.
func (client BaseClient) GetServiceInfoListPreparer(ctx context.Context, applicationID string, serviceTypeName string, continuationToken string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(serviceTypeName) > 0 {
		queryParameters["ServiceTypeName"] = autorest.Encode("query", serviceTypeName)
	}
	if len(continuationToken) > 0 {
		queryParameters["ContinuationToken"] = continuationToken
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/GetServices", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceInfoListSender sends the GetServiceInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceInfoListResponder handles the response to the GetServiceInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceInfoListResponder(resp *http.Response) (result PagedServiceInfoList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceManifest gets the manifest describing a service type. The response contains the service manifest XML as a
// string.
// Parameters:
// applicationTypeName - the name of the application type.
// applicationTypeVersion - the version of the application type.
// serviceManifestName - the name of a service manifest registered as part of an application type in a Service
// Fabric cluster.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceManifest(ctx context.Context, applicationTypeName string, applicationTypeVersion string, serviceManifestName string, timeout *int64) (result ServiceTypeManifest, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceManifest", err.Error())
	}

	req, err := client.GetServiceManifestPreparer(ctx, applicationTypeName, applicationTypeVersion, serviceManifestName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceManifest", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceManifestSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceManifest", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceManifestResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceManifest", resp, "Failure responding to request")
	}

	return
}

// GetServiceManifestPreparer prepares the GetServiceManifest request.
func (client BaseClient) GetServiceManifestPreparer(ctx context.Context, applicationTypeName string, applicationTypeVersion string, serviceManifestName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationTypeName": autorest.Encode("path", applicationTypeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":            APIVersion,
		"ApplicationTypeVersion": autorest.Encode("query", applicationTypeVersion),
		"ServiceManifestName":    autorest.Encode("query", serviceManifestName),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ApplicationTypes/{applicationTypeName}/$/GetServiceManifest", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceManifestSender sends the GetServiceManifest request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceManifestSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceManifestResponder handles the response to the GetServiceManifest request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceManifestResponder(resp *http.Response) (result ServiceTypeManifest, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceNameInfo the GetServiceName endpoint returns the name of the service for the specified partition.
// Parameters:
// partitionID - the identity of the partition.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceNameInfo(ctx context.Context, partitionID uuid.UUID, timeout *int64) (result ServiceNameInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceNameInfo", err.Error())
	}

	req, err := client.GetServiceNameInfoPreparer(ctx, partitionID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceNameInfo", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceNameInfoSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceNameInfo", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceNameInfoResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceNameInfo", resp, "Failure responding to request")
	}

	return
}

// GetServiceNameInfoPreparer prepares the GetServiceNameInfo request.
func (client BaseClient) GetServiceNameInfoPreparer(ctx context.Context, partitionID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetServiceName", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceNameInfoSender sends the GetServiceNameInfo request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceNameInfoSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceNameInfoResponder handles the response to the GetServiceNameInfo request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceNameInfoResponder(resp *http.Response) (result ServiceNameInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetServiceTypeInfoList gets the list containing the information about service types that are supported by a
// provisioned application type in a Service Fabric cluster. The response includes the name of the service type, the
// name and version of the service manifest the type is defined in, kind (stateless or stateless) of the service type
// and other information about it.
// Parameters:
// applicationTypeName - the name of the application type.
// applicationTypeVersion - the version of the application type.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) GetServiceTypeInfoList(ctx context.Context, applicationTypeName string, applicationTypeVersion string, timeout *int64) (result ListServiceTypeInfo, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "GetServiceTypeInfoList", err.Error())
	}

	req, err := client.GetServiceTypeInfoListPreparer(ctx, applicationTypeName, applicationTypeVersion, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceTypeInfoList", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetServiceTypeInfoListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceTypeInfoList", resp, "Failure sending request")
		return
	}

	result, err = client.GetServiceTypeInfoListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "GetServiceTypeInfoList", resp, "Failure responding to request")
	}

	return
}

// GetServiceTypeInfoListPreparer prepares the GetServiceTypeInfoList request.
func (client BaseClient) GetServiceTypeInfoListPreparer(ctx context.Context, applicationTypeName string, applicationTypeVersion string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationTypeName": autorest.Encode("path", applicationTypeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":            APIVersion,
		"ApplicationTypeVersion": autorest.Encode("query", applicationTypeVersion),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ApplicationTypes/{applicationTypeName}/$/GetServiceTypes", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetServiceTypeInfoListSender sends the GetServiceTypeInfoList request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetServiceTypeInfoListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetServiceTypeInfoListResponder handles the response to the GetServiceTypeInfoList request. The method always
// closes the http.Response Body.
func (client BaseClient) GetServiceTypeInfoListResponder(resp *http.Response) (result ListServiceTypeInfo, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// InvokeInfrastructureCommand for clusters that have one or more instances of the Infrastructure Service configured,
// this API provides a way to send infrastructure-specific commands to a particular
// instance of the Infrastructure Service.
//
// Available commands and their corresponding response formats vary depending upon
// the infrastructure on which the cluster is running.
//
// This API supports the Service Fabric platform; it is not meant to be used directly from your code.
// Parameters:
// command - the text of the command to be invoked. The content of the command is infrastructure-specific.
// serviceID - the identity of the infrastructure service. This is  the full name of the infrastructure service
// without the 'fabric:' URI scheme. This parameter required only for the cluster that have more than one
// instance of infrastructure service running.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) InvokeInfrastructureCommand(ctx context.Context, command string, serviceID string, timeout *int64) (result ReadCloser, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "InvokeInfrastructureCommand", err.Error())
	}

	req, err := client.InvokeInfrastructureCommandPreparer(ctx, command, serviceID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "InvokeInfrastructureCommand", nil, "Failure preparing request")
		return
	}

	resp, err := client.InvokeInfrastructureCommandSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "InvokeInfrastructureCommand", resp, "Failure sending request")
		return
	}

	result, err = client.InvokeInfrastructureCommandResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "InvokeInfrastructureCommand", resp, "Failure responding to request")
	}

	return
}

// InvokeInfrastructureCommandPreparer prepares the InvokeInfrastructureCommand request.
func (client BaseClient) InvokeInfrastructureCommandPreparer(ctx context.Context, command string, serviceID string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"Command":     autorest.Encode("query", command),
	}
	if len(serviceID) > 0 {
		queryParameters["ServiceId"] = autorest.Encode("query", serviceID)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/InvokeInfrastructureCommand"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// InvokeInfrastructureCommandSender sends the InvokeInfrastructureCommand request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) InvokeInfrastructureCommandSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// InvokeInfrastructureCommandResponder handles the response to the InvokeInfrastructureCommand request. The method always
// closes the http.Response Body.
func (client BaseClient) InvokeInfrastructureCommandResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// InvokeInfrastructureQuery for clusters that have one or more instances of the Infrastructure Service configured,
// this API provides a way to send infrastructure-specific queries to a particular
// instance of the Infrastructure Service.
//
// Available commands and their corresponding response formats vary depending upon
// the infrastructure on which the cluster is running.
//
// This API supports the Service Fabric platform; it is not meant to be used directly from your code.
// Parameters:
// command - the text of the command to be invoked. The content of the command is infrastructure-specific.
// serviceID - the identity of the infrastructure service. This is  the full name of the infrastructure service
// without the 'fabric:' URI scheme. This parameter required only for the cluster that have more than one
// instance of infrastructure service running.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) InvokeInfrastructureQuery(ctx context.Context, command string, serviceID string, timeout *int64) (result ReadCloser, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "InvokeInfrastructureQuery", err.Error())
	}

	req, err := client.InvokeInfrastructureQueryPreparer(ctx, command, serviceID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "InvokeInfrastructureQuery", nil, "Failure preparing request")
		return
	}

	resp, err := client.InvokeInfrastructureQuerySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "InvokeInfrastructureQuery", resp, "Failure sending request")
		return
	}

	result, err = client.InvokeInfrastructureQueryResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "InvokeInfrastructureQuery", resp, "Failure responding to request")
	}

	return
}

// InvokeInfrastructureQueryPreparer prepares the InvokeInfrastructureQuery request.
func (client BaseClient) InvokeInfrastructureQueryPreparer(ctx context.Context, command string, serviceID string, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"Command":     autorest.Encode("query", command),
	}
	if len(serviceID) > 0 {
		queryParameters["ServiceId"] = autorest.Encode("query", serviceID)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/InvokeInfrastructureQuery"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// InvokeInfrastructureQuerySender sends the InvokeInfrastructureQuery request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) InvokeInfrastructureQuerySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// InvokeInfrastructureQueryResponder handles the response to the InvokeInfrastructureQuery request. The method always
// closes the http.Response Body.
func (client BaseClient) InvokeInfrastructureQueryResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// ProvisionApplicationType provisions or registers a Service Fabric application type with the cluster. This is
// required before any new applications can be instantiated.
// Parameters:
// applicationTypeImageStorePath - the relative path for the application package in the image store specified
// during the prior copy operation.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ProvisionApplicationType(ctx context.Context, applicationTypeImageStorePath ApplicationTypeImageStorePath, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: applicationTypeImageStorePath,
			Constraints: []validation.Constraint{{Target: "applicationTypeImageStorePath.ApplicationTypeBuildPath", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ProvisionApplicationType", err.Error())
	}

	req, err := client.ProvisionApplicationTypePreparer(ctx, applicationTypeImageStorePath, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ProvisionApplicationType", nil, "Failure preparing request")
		return
	}

	resp, err := client.ProvisionApplicationTypeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ProvisionApplicationType", resp, "Failure sending request")
		return
	}

	result, err = client.ProvisionApplicationTypeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ProvisionApplicationType", resp, "Failure responding to request")
	}

	return
}

// ProvisionApplicationTypePreparer prepares the ProvisionApplicationType request.
func (client BaseClient) ProvisionApplicationTypePreparer(ctx context.Context, applicationTypeImageStorePath ApplicationTypeImageStorePath, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/ApplicationTypes/$/Provision"),
		autorest.WithJSON(applicationTypeImageStorePath),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ProvisionApplicationTypeSender sends the ProvisionApplicationType request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ProvisionApplicationTypeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ProvisionApplicationTypeResponder handles the response to the ProvisionApplicationType request. The method always
// closes the http.Response Body.
func (client BaseClient) ProvisionApplicationTypeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RecoverAllPartitions indicates to the Service Fabric cluster that it should attempt to recover any services
// (including system services) which are currently stuck in quorum loss. This operation should only be performed if it
// is known that the replicas that are down cannot be recovered. Incorrect use of this API can cause potential data
// loss.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RecoverAllPartitions(ctx context.Context, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RecoverAllPartitions", err.Error())
	}

	req, err := client.RecoverAllPartitionsPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverAllPartitions", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverAllPartitionsSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverAllPartitions", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverAllPartitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverAllPartitions", resp, "Failure responding to request")
	}

	return
}

// RecoverAllPartitionsPreparer prepares the RecoverAllPartitions request.
func (client BaseClient) RecoverAllPartitionsPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/RecoverAllPartitions"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverAllPartitionsSender sends the RecoverAllPartitions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverAllPartitionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RecoverAllPartitionsResponder handles the response to the RecoverAllPartitions request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverAllPartitionsResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RecoverPartition indicates to the Service Fabric cluster that it should attempt to recover a specific partition
// which is currently stuck in quorum loss. This operation should only be performed if it is known that the replicas
// that are down cannot be recovered. Incorrect use of this API can cause potential data loss.
// Parameters:
// partitionID - the identity of the partition.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RecoverPartition(ctx context.Context, partitionID uuid.UUID, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RecoverPartition", err.Error())
	}

	req, err := client.RecoverPartitionPreparer(ctx, partitionID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverPartition", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverPartitionSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverPartition", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverPartitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverPartition", resp, "Failure responding to request")
	}

	return
}

// RecoverPartitionPreparer prepares the RecoverPartition request.
func (client BaseClient) RecoverPartitionPreparer(ctx context.Context, partitionID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/Recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverPartitionSender sends the RecoverPartition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverPartitionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RecoverPartitionResponder handles the response to the RecoverPartition request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverPartitionResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RecoverServicePartitions indicates to the Service Fabric cluster that it should attempt to recover the specified
// service which is currently stuck in quorum loss. This operation should only be performed if it is known that the
// replicas that are down cannot be recovered. Incorrect use of this API can cause potential data loss.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RecoverServicePartitions(ctx context.Context, serviceID string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RecoverServicePartitions", err.Error())
	}

	req, err := client.RecoverServicePartitionsPreparer(ctx, serviceID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverServicePartitions", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverServicePartitionsSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverServicePartitions", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverServicePartitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverServicePartitions", resp, "Failure responding to request")
	}

	return
}

// RecoverServicePartitionsPreparer prepares the RecoverServicePartitions request.
func (client BaseClient) RecoverServicePartitionsPreparer(ctx context.Context, serviceID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/$/{serviceId}/$/GetPartitions/$/Recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverServicePartitionsSender sends the RecoverServicePartitions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverServicePartitionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RecoverServicePartitionsResponder handles the response to the RecoverServicePartitions request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverServicePartitionsResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RecoverSystemPartitions indicates to the Service Fabric cluster that it should attempt to recover the system
// services which are currently stuck in quorum loss. This operation should only be performed if it is known that the
// replicas that are down cannot be recovered. Incorrect use of this API can cause potential data loss.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RecoverSystemPartitions(ctx context.Context, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RecoverSystemPartitions", err.Error())
	}

	req, err := client.RecoverSystemPartitionsPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverSystemPartitions", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverSystemPartitionsSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverSystemPartitions", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverSystemPartitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RecoverSystemPartitions", resp, "Failure responding to request")
	}

	return
}

// RecoverSystemPartitionsPreparer prepares the RecoverSystemPartitions request.
func (client BaseClient) RecoverSystemPartitionsPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/RecoverSystemPartitions"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverSystemPartitionsSender sends the RecoverSystemPartitions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverSystemPartitionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RecoverSystemPartitionsResponder handles the response to the RecoverSystemPartitions request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverSystemPartitionsResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveComposeApplication deletes an existing Service Fabric compose application. An application must be created
// before it can be deleted.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RemoveComposeApplication(ctx context.Context, applicationID string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RemoveComposeApplication", err.Error())
	}

	req, err := client.RemoveComposeApplicationPreparer(ctx, applicationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveComposeApplication", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveComposeApplicationSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveComposeApplication", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveComposeApplicationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveComposeApplication", resp, "Failure responding to request")
	}

	return
}

// RemoveComposeApplicationPreparer prepares the RemoveComposeApplication request.
func (client BaseClient) RemoveComposeApplicationPreparer(ctx context.Context, applicationID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "4.0-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ComposeDeployments/{applicationId}/$/Delete", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveComposeApplicationSender sends the RemoveComposeApplication request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RemoveComposeApplicationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveComposeApplicationResponder handles the response to the RemoveComposeApplication request. The method always
// closes the http.Response Body.
func (client BaseClient) RemoveComposeApplicationResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveNodeState notifies Service Fabric that the persisted state on a node has been permanently removed or lost.
// This implies that it is not possible to recover the persisted state of that node. This generally happens if a hard
// disk has been wiped clean, or if a hard disk crashes. The node has to be down for this operation to be successful.
// This operation lets Service Fabric know that the replicas on that node no longer exist, and that Service Fabric
// should stop waiting for those replicas to come back up. Do not run this cmdlet if the state on the node has not been
// removed and the node can comes back up with its state intact.
// Parameters:
// nodeName - the name of the node.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RemoveNodeState(ctx context.Context, nodeName string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RemoveNodeState", err.Error())
	}

	req, err := client.RemoveNodeStatePreparer(ctx, nodeName, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveNodeState", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveNodeStateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveNodeState", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveNodeStateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveNodeState", resp, "Failure responding to request")
	}

	return
}

// RemoveNodeStatePreparer prepares the RemoveNodeState request.
func (client BaseClient) RemoveNodeStatePreparer(ctx context.Context, nodeName string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/RemoveNodeState", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveNodeStateSender sends the RemoveNodeState request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RemoveNodeStateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveNodeStateResponder handles the response to the RemoveNodeState request. The method always
// closes the http.Response Body.
func (client BaseClient) RemoveNodeStateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveReplica this API simulates a Service Fabric replica failure by removing a replica from a Service Fabric
// cluster. The removal closes the replica, transitions the replica to the role None, and then removes all of the state
// information of the replica from the cluster. This API tests the replica state removal path, and simulates the report
// fault permanent path through client APIs. Warning - There are no safety checks performed when this API is used.
// Incorrect use of this API can lead to data loss for stateful services.In addition, the forceRemove flag impacts all
// other replicas hosted in the same process.
// Parameters:
// nodeName - the name of the node.
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// forceRemove - remove a Service Fabric application or service forcefully without going through the graceful
// shutdown sequence. This parameter can be used to forcefully delete an application or service for which
// delete is timing out due to issues in the service code that prevents graceful close of replicas.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RemoveReplica(ctx context.Context, nodeName string, partitionID uuid.UUID, replicaID string, forceRemove *bool, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RemoveReplica", err.Error())
	}

	req, err := client.RemoveReplicaPreparer(ctx, nodeName, partitionID, replicaID, forceRemove, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveReplica", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveReplicaSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveReplica", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveReplicaResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RemoveReplica", resp, "Failure responding to request")
	}

	return
}

// RemoveReplicaPreparer prepares the RemoveReplica request.
func (client BaseClient) RemoveReplicaPreparer(ctx context.Context, nodeName string, partitionID uuid.UUID, replicaID string, forceRemove *bool, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName":    autorest.Encode("path", nodeName),
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if forceRemove != nil {
		queryParameters["ForceRemove"] = autorest.Encode("query", *forceRemove)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetPartitions/{partitionId}/$/GetReplicas/{replicaId}/$/Delete", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveReplicaSender sends the RemoveReplica request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RemoveReplicaSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveReplicaResponder handles the response to the RemoveReplica request. The method always
// closes the http.Response Body.
func (client BaseClient) RemoveReplicaResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportApplicationHealth reports health state of the specified Service Fabric application. The report must contain
// the information about the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway Application, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, run GetApplicationHealth and check that the report
// appears in the HealthEvents section.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportApplicationHealth(ctx context.Context, applicationID string, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportApplicationHealth", err.Error())
	}

	req, err := client.ReportApplicationHealthPreparer(ctx, applicationID, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportApplicationHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportApplicationHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportApplicationHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportApplicationHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportApplicationHealth", resp, "Failure responding to request")
	}

	return
}

// ReportApplicationHealthPreparer prepares the ReportApplicationHealth request.
func (client BaseClient) ReportApplicationHealthPreparer(ctx context.Context, applicationID string, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportApplicationHealthSender sends the ReportApplicationHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportApplicationHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportApplicationHealthResponder handles the response to the ReportApplicationHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportApplicationHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportClusterHealth sends a health report on a Service Fabric cluster. The report must contain the information about
// the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway node, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, run GetClusterHealth and check that the report appears in
// the HealthEvents section.
// Parameters:
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportClusterHealth(ctx context.Context, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportClusterHealth", err.Error())
	}

	req, err := client.ReportClusterHealthPreparer(ctx, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportClusterHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportClusterHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportClusterHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportClusterHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportClusterHealth", resp, "Failure responding to request")
	}

	return
}

// ReportClusterHealthPreparer prepares the ReportClusterHealth request.
func (client BaseClient) ReportClusterHealthPreparer(ctx context.Context, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/$/ReportClusterHealth"),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportClusterHealthSender sends the ReportClusterHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportClusterHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportClusterHealthResponder handles the response to the ReportClusterHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportClusterHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportDeployedApplicationHealth reports health state of the application deployed on a Service Fabric node. The
// report must contain the information about the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway Service, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, get deployed application health and check that the report
// appears in the HealthEvents section.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportDeployedApplicationHealth(ctx context.Context, nodeName string, applicationID string, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportDeployedApplicationHealth", err.Error())
	}

	req, err := client.ReportDeployedApplicationHealthPreparer(ctx, nodeName, applicationID, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportDeployedApplicationHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportDeployedApplicationHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportDeployedApplicationHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportDeployedApplicationHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportDeployedApplicationHealth", resp, "Failure responding to request")
	}

	return
}

// ReportDeployedApplicationHealthPreparer prepares the ReportDeployedApplicationHealth request.
func (client BaseClient) ReportDeployedApplicationHealthPreparer(ctx context.Context, nodeName string, applicationID string, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportDeployedApplicationHealthSender sends the ReportDeployedApplicationHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportDeployedApplicationHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportDeployedApplicationHealthResponder handles the response to the ReportDeployedApplicationHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportDeployedApplicationHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportDeployedServicePackageHealth reports health state of the service package of the application deployed on a
// Service Fabric node. The report must contain the information about the source of the health report and property on
// which it is reported.
// The report is sent to a Service Fabric gateway Service, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, get deployed service package health and check that the
// report appears in the HealthEvents section.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// servicePackageName - the name of the service package.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportDeployedServicePackageHealth(ctx context.Context, nodeName string, applicationID string, servicePackageName string, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportDeployedServicePackageHealth", err.Error())
	}

	req, err := client.ReportDeployedServicePackageHealthPreparer(ctx, nodeName, applicationID, servicePackageName, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportDeployedServicePackageHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportDeployedServicePackageHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportDeployedServicePackageHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportDeployedServicePackageHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportDeployedServicePackageHealth", resp, "Failure responding to request")
	}

	return
}

// ReportDeployedServicePackageHealthPreparer prepares the ReportDeployedServicePackageHealth request.
func (client BaseClient) ReportDeployedServicePackageHealthPreparer(ctx context.Context, nodeName string, applicationID string, servicePackageName string, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId":      applicationID,
		"nodeName":           autorest.Encode("path", nodeName),
		"servicePackageName": servicePackageName,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetServicePackages/{servicePackageName}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportDeployedServicePackageHealthSender sends the ReportDeployedServicePackageHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportDeployedServicePackageHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportDeployedServicePackageHealthResponder handles the response to the ReportDeployedServicePackageHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportDeployedServicePackageHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportNodeHealth reports health state of the specified Service Fabric node. The report must contain the information
// about the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway node, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, run GetNodeHealth and check that the report appears in
// the HealthEvents section.
// Parameters:
// nodeName - the name of the node.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportNodeHealth(ctx context.Context, nodeName string, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportNodeHealth", err.Error())
	}

	req, err := client.ReportNodeHealthPreparer(ctx, nodeName, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportNodeHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportNodeHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportNodeHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportNodeHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportNodeHealth", resp, "Failure responding to request")
	}

	return
}

// ReportNodeHealthPreparer prepares the ReportNodeHealth request.
func (client BaseClient) ReportNodeHealthPreparer(ctx context.Context, nodeName string, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportNodeHealthSender sends the ReportNodeHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportNodeHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportNodeHealthResponder handles the response to the ReportNodeHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportNodeHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportPartitionHealth reports health state of the specified Service Fabric partition. The report must contain the
// information about the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway Partition, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, run GetPartitionHealth and check that the report appears
// in the HealthEvents section.
// Parameters:
// partitionID - the identity of the partition.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportPartitionHealth(ctx context.Context, partitionID uuid.UUID, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportPartitionHealth", err.Error())
	}

	req, err := client.ReportPartitionHealthPreparer(ctx, partitionID, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportPartitionHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportPartitionHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportPartitionHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportPartitionHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportPartitionHealth", resp, "Failure responding to request")
	}

	return
}

// ReportPartitionHealthPreparer prepares the ReportPartitionHealth request.
func (client BaseClient) ReportPartitionHealthPreparer(ctx context.Context, partitionID uuid.UUID, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportPartitionHealthSender sends the ReportPartitionHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportPartitionHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportPartitionHealthResponder handles the response to the ReportPartitionHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportPartitionHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportReplicaHealth reports health state of the specified Service Fabric replica. The report must contain the
// information about the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway Replica, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, run GetReplicaHealth and check that the report appears in
// the HealthEvents section.
// Parameters:
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// serviceKind - the kind of service replica (Stateless or Stateful) for which the health is being reported.
// Following are the possible values.
// - Stateless - Does not use Service Fabric to make its state highly available or reliable. The value is 1
// - Stateful - Uses Service Fabric to make its state or part of its state highly available and reliable. The
// value is 2.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportReplicaHealth(ctx context.Context, partitionID uuid.UUID, replicaID string, serviceKind string, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportReplicaHealth", err.Error())
	}

	req, err := client.ReportReplicaHealthPreparer(ctx, partitionID, replicaID, serviceKind, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportReplicaHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportReplicaHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportReplicaHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportReplicaHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportReplicaHealth", resp, "Failure responding to request")
	}

	return
}

// ReportReplicaHealthPreparer prepares the ReportReplicaHealth request.
func (client BaseClient) ReportReplicaHealthPreparer(ctx context.Context, partitionID uuid.UUID, replicaID string, serviceKind string, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"ServiceKind": autorest.Encode("query", serviceKind),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/GetReplicas/{replicaId}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportReplicaHealthSender sends the ReportReplicaHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportReplicaHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportReplicaHealthResponder handles the response to the ReportReplicaHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportReplicaHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReportServiceHealth reports health state of the specified Service Fabric service. The report must contain the
// information about the source of the health report and property on which it is reported.
// The report is sent to a Service Fabric gateway Service, which forwards to the health store.
// The report may be accepted by the gateway, but rejected by the health store after extra validation.
// For example, the health store may reject the report because of an invalid parameter, like a stale sequence number.
// To see whether the report was applied in the health store, run GetServiceHealth and check that the report appears in
// the HealthEvents section.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// healthInformation - describes the health information for the health report. This information needs to be
// present in all of the health reports sent to the health manager.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ReportServiceHealth(ctx context.Context, serviceID string, healthInformation HealthInformation, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: healthInformation,
			Constraints: []validation.Constraint{{Target: "healthInformation.SourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "healthInformation.Property", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ReportServiceHealth", err.Error())
	}

	req, err := client.ReportServiceHealthPreparer(ctx, serviceID, healthInformation, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportServiceHealth", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReportServiceHealthSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportServiceHealth", resp, "Failure sending request")
		return
	}

	result, err = client.ReportServiceHealthResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ReportServiceHealth", resp, "Failure responding to request")
	}

	return
}

// ReportServiceHealthPreparer prepares the ReportServiceHealth request.
func (client BaseClient) ReportServiceHealthPreparer(ctx context.Context, serviceID string, healthInformation HealthInformation, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/ReportHealth", pathParameters),
		autorest.WithJSON(healthInformation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReportServiceHealthSender sends the ReportServiceHealth request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReportServiceHealthSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReportServiceHealthResponder handles the response to the ReportServiceHealth request. The method always
// closes the http.Response Body.
func (client BaseClient) ReportServiceHealthResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ResetPartitionLoad resets the current load of a Service Fabric partition to the default load for the service.
// Parameters:
// partitionID - the identity of the partition.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ResetPartitionLoad(ctx context.Context, partitionID uuid.UUID, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ResetPartitionLoad", err.Error())
	}

	req, err := client.ResetPartitionLoadPreparer(ctx, partitionID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResetPartitionLoad", nil, "Failure preparing request")
		return
	}

	resp, err := client.ResetPartitionLoadSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResetPartitionLoad", resp, "Failure sending request")
		return
	}

	result, err = client.ResetPartitionLoadResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResetPartitionLoad", resp, "Failure responding to request")
	}

	return
}

// ResetPartitionLoadPreparer prepares the ResetPartitionLoad request.
func (client BaseClient) ResetPartitionLoadPreparer(ctx context.Context, partitionID uuid.UUID, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Partitions/{partitionId}/$/ResetLoad", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ResetPartitionLoadSender sends the ResetPartitionLoad request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ResetPartitionLoadSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ResetPartitionLoadResponder handles the response to the ResetPartitionLoad request. The method always
// closes the http.Response Body.
func (client BaseClient) ResetPartitionLoadResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ResolveService resolve a Service Fabric service partition, to get the endpoints of the service replicas.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionKeyType - key type for the partition. This parameter is required if the partition scheme for the
// service is Int64Range or Named. The possible values are following.
// - None (1) - Indicates that the the PartitionKeyValue parameter is not specified. This is valid for the
// partitions with partitioning scheme as Singleton. This is the default value. The value is 1.
// - Int64Range (2) - Indicates that the the PartitionKeyValue parameter is an int64 partition key. This is
// valid for the partitions with partitioning scheme as Int64Range. The value is 2.
// - Named (3) - Indicates that the the PartitionKeyValue parameter is a name of the partition. This is valid
// for the partitions with partitioning scheme as Named. The value is 3.
// partitionKeyValue - partition key. This is required if the partition scheme for the service is Int64Range or
// Named.
// previousRspVersion - the value in the Version field of the response that was received previously. This is
// required if the user knows that the result that was got previously is stale.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ResolveService(ctx context.Context, serviceID string, partitionKeyType *int32, partitionKeyValue string, previousRspVersion string, timeout *int64) (result ResolvedServicePartition, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ResolveService", err.Error())
	}

	req, err := client.ResolveServicePreparer(ctx, serviceID, partitionKeyType, partitionKeyValue, previousRspVersion, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResolveService", nil, "Failure preparing request")
		return
	}

	resp, err := client.ResolveServiceSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResolveService", resp, "Failure sending request")
		return
	}

	result, err = client.ResolveServiceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResolveService", resp, "Failure responding to request")
	}

	return
}

// ResolveServicePreparer prepares the ResolveService request.
func (client BaseClient) ResolveServicePreparer(ctx context.Context, serviceID string, partitionKeyType *int32, partitionKeyValue string, previousRspVersion string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if partitionKeyType != nil {
		queryParameters["PartitionKeyType"] = autorest.Encode("query", *partitionKeyType)
	}
	if len(partitionKeyValue) > 0 {
		queryParameters["PartitionKeyValue"] = partitionKeyValue
	}
	if len(previousRspVersion) > 0 {
		queryParameters["PreviousRspVersion"] = previousRspVersion
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/ResolvePartition", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ResolveServiceSender sends the ResolveService request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ResolveServiceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ResolveServiceResponder handles the response to the ResolveService request. The method always
// closes the http.Response Body.
func (client BaseClient) ResolveServiceResponder(resp *http.Response) (result ResolvedServicePartition, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestartDeployedCodePackage restarts a code package deployed on a Service Fabric node in a cluster. This aborts the
// code package process, which will restart all the user service replicas hosted in that process.
// Parameters:
// nodeName - the name of the node.
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// restartDeployedCodePackageDescription - describes the deployed code package on Service Fabric node to
// restart.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RestartDeployedCodePackage(ctx context.Context, nodeName string, applicationID string, restartDeployedCodePackageDescription RestartDeployedCodePackageDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RestartDeployedCodePackage", err.Error())
	}

	req, err := client.RestartDeployedCodePackagePreparer(ctx, nodeName, applicationID, restartDeployedCodePackageDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartDeployedCodePackage", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestartDeployedCodePackageSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartDeployedCodePackage", resp, "Failure sending request")
		return
	}

	result, err = client.RestartDeployedCodePackageResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartDeployedCodePackage", resp, "Failure responding to request")
	}

	return
}

// RestartDeployedCodePackagePreparer prepares the RestartDeployedCodePackage request.
func (client BaseClient) RestartDeployedCodePackagePreparer(ctx context.Context, nodeName string, applicationID string, restartDeployedCodePackageDescription RestartDeployedCodePackageDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
		"nodeName":      autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetApplications/{applicationId}/$/GetCodePackages/$/Restart", pathParameters),
		autorest.WithJSON(restartDeployedCodePackageDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestartDeployedCodePackageSender sends the RestartDeployedCodePackage request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestartDeployedCodePackageSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RestartDeployedCodePackageResponder handles the response to the RestartDeployedCodePackage request. The method always
// closes the http.Response Body.
func (client BaseClient) RestartDeployedCodePackageResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RestartNode restarts a Service Fabric cluster node that is already started.
// Parameters:
// nodeName - the name of the node.
// restartNodeDescription - the instance of the node to be restarted and a flag indicating the need to take
// dump of the fabric process.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RestartNode(ctx context.Context, nodeName string, restartNodeDescription RestartNodeDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: restartNodeDescription,
			Constraints: []validation.Constraint{{Target: "restartNodeDescription.NodeInstanceID", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RestartNode", err.Error())
	}

	req, err := client.RestartNodePreparer(ctx, nodeName, restartNodeDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartNode", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestartNodeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartNode", resp, "Failure sending request")
		return
	}

	result, err = client.RestartNodeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartNode", resp, "Failure responding to request")
	}

	return
}

// RestartNodePreparer prepares the RestartNode request.
func (client BaseClient) RestartNodePreparer(ctx context.Context, nodeName string, restartNodeDescription RestartNodeDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/Restart", pathParameters),
		autorest.WithJSON(restartNodeDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestartNodeSender sends the RestartNode request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestartNodeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RestartNodeResponder handles the response to the RestartNode request. The method always
// closes the http.Response Body.
func (client BaseClient) RestartNodeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RestartReplica restarts a service replica of a persisted service running on a node. Warning - There are no safety
// checks performed when this API is used. Incorrect use of this API can lead to availability loss for stateful
// services.
// Parameters:
// nodeName - the name of the node.
// partitionID - the identity of the partition.
// replicaID - the identifier of the replica.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RestartReplica(ctx context.Context, nodeName string, partitionID uuid.UUID, replicaID string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RestartReplica", err.Error())
	}

	req, err := client.RestartReplicaPreparer(ctx, nodeName, partitionID, replicaID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartReplica", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestartReplicaSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartReplica", resp, "Failure sending request")
		return
	}

	result, err = client.RestartReplicaResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RestartReplica", resp, "Failure responding to request")
	}

	return
}

// RestartReplicaPreparer prepares the RestartReplica request.
func (client BaseClient) RestartReplicaPreparer(ctx context.Context, nodeName string, partitionID uuid.UUID, replicaID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName":    autorest.Encode("path", nodeName),
		"partitionId": partitionID,
		"replicaId":   replicaID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/GetPartitions/{partitionId}/$/GetReplicas/{replicaId}/$/Restart", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestartReplicaSender sends the RestartReplica request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestartReplicaSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RestartReplicaResponder handles the response to the RestartReplica request. The method always
// closes the http.Response Body.
func (client BaseClient) RestartReplicaResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ResumeApplicationUpgrade resumes an unmonitored manual Service Fabric application upgrade. Service Fabric upgrades
// one upgrade domain at a time. For unmonitored manual upgrades, after Service Fabric finishes an upgrade domain, it
// waits for you to call this API before proceeding to the next upgrade domain.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// resumeApplicationUpgradeDescription - describes the parameters for resuming an application upgrade.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) ResumeApplicationUpgrade(ctx context.Context, applicationID string, resumeApplicationUpgradeDescription ResumeApplicationUpgradeDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: resumeApplicationUpgradeDescription,
			Constraints: []validation.Constraint{{Target: "resumeApplicationUpgradeDescription.UpgradeDomainName", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "ResumeApplicationUpgrade", err.Error())
	}

	req, err := client.ResumeApplicationUpgradePreparer(ctx, applicationID, resumeApplicationUpgradeDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResumeApplicationUpgrade", nil, "Failure preparing request")
		return
	}

	resp, err := client.ResumeApplicationUpgradeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResumeApplicationUpgrade", resp, "Failure sending request")
		return
	}

	result, err = client.ResumeApplicationUpgradeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "ResumeApplicationUpgrade", resp, "Failure responding to request")
	}

	return
}

// ResumeApplicationUpgradePreparer prepares the ResumeApplicationUpgrade request.
func (client BaseClient) ResumeApplicationUpgradePreparer(ctx context.Context, applicationID string, resumeApplicationUpgradeDescription ResumeApplicationUpgradeDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/MoveToNextUpgradeDomain", pathParameters),
		autorest.WithJSON(resumeApplicationUpgradeDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ResumeApplicationUpgradeSender sends the ResumeApplicationUpgrade request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ResumeApplicationUpgradeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ResumeApplicationUpgradeResponder handles the response to the ResumeApplicationUpgrade request. The method always
// closes the http.Response Body.
func (client BaseClient) ResumeApplicationUpgradeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RollbackApplicationUpgrade starts rolling back the current application upgrade to the previous version. This API can
// only be used to rollback the current in-progress upgrade that is rolling forward to new version. If the application
// is not currently being upgraded use StartApplicationUpgrade API to upgrade it to desired version including rolling
// back to a previous version.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) RollbackApplicationUpgrade(ctx context.Context, applicationID string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "RollbackApplicationUpgrade", err.Error())
	}

	req, err := client.RollbackApplicationUpgradePreparer(ctx, applicationID, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RollbackApplicationUpgrade", nil, "Failure preparing request")
		return
	}

	resp, err := client.RollbackApplicationUpgradeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RollbackApplicationUpgrade", resp, "Failure sending request")
		return
	}

	result, err = client.RollbackApplicationUpgradeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "RollbackApplicationUpgrade", resp, "Failure responding to request")
	}

	return
}

// RollbackApplicationUpgradePreparer prepares the RollbackApplicationUpgrade request.
func (client BaseClient) RollbackApplicationUpgradePreparer(ctx context.Context, applicationID string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/RollbackUpgrade", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RollbackApplicationUpgradeSender sends the RollbackApplicationUpgrade request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RollbackApplicationUpgradeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RollbackApplicationUpgradeResponder handles the response to the RollbackApplicationUpgrade request. The method always
// closes the http.Response Body.
func (client BaseClient) RollbackApplicationUpgradeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartApplicationUpgrade validates the supplied application upgrade parameters and starts upgrading the application
// if the parameters are valid.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// applicationUpgradeDescription - describes the parameters for an application upgrade.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartApplicationUpgrade(ctx context.Context, applicationID string, applicationUpgradeDescription ApplicationUpgradeDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: applicationUpgradeDescription,
			Constraints: []validation.Constraint{{Target: "applicationUpgradeDescription.Name", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "applicationUpgradeDescription.TargetApplicationTypeVersion", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "applicationUpgradeDescription.Parameters", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartApplicationUpgrade", err.Error())
	}

	req, err := client.StartApplicationUpgradePreparer(ctx, applicationID, applicationUpgradeDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartApplicationUpgrade", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartApplicationUpgradeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartApplicationUpgrade", resp, "Failure sending request")
		return
	}

	result, err = client.StartApplicationUpgradeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartApplicationUpgrade", resp, "Failure responding to request")
	}

	return
}

// StartApplicationUpgradePreparer prepares the StartApplicationUpgrade request.
func (client BaseClient) StartApplicationUpgradePreparer(ctx context.Context, applicationID string, applicationUpgradeDescription ApplicationUpgradeDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/Upgrade", pathParameters),
		autorest.WithJSON(applicationUpgradeDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartApplicationUpgradeSender sends the StartApplicationUpgrade request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartApplicationUpgradeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartApplicationUpgradeResponder handles the response to the StartApplicationUpgrade request. The method always
// closes the http.Response Body.
func (client BaseClient) StartApplicationUpgradeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartChaos if Chaos is not already running in the cluster, it starts Chaos with the passed in Chaos parameters.
// If Chaos is already running when this call is made, the call fails with the error code
// FABRIC_E_CHAOS_ALREADY_RUNNING.
// Please refer to the article [Induce controlled Chaos in Service Fabric
// clusters](https://docs.microsoft.com/en-us/azure/service-fabric/service-fabric-controlled-chaos) for more details.
// Parameters:
// chaosParameters - describes all the parameters to configure a Chaos run.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartChaos(ctx context.Context, chaosParameters ChaosParameters, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: chaosParameters,
			Constraints: []validation.Constraint{{Target: "chaosParameters.MaxClusterStabilizationTimeoutInSeconds", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "chaosParameters.MaxClusterStabilizationTimeoutInSeconds", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "chaosParameters.MaxClusterStabilizationTimeoutInSeconds", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil},
				}},
				{Target: "chaosParameters.MaxConcurrentFaults", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "chaosParameters.MaxConcurrentFaults", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
						{Target: "chaosParameters.MaxConcurrentFaults", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil},
					}},
				{Target: "chaosParameters.WaitTimeBetweenFaultsInSeconds", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "chaosParameters.WaitTimeBetweenFaultsInSeconds", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
						{Target: "chaosParameters.WaitTimeBetweenFaultsInSeconds", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil},
					}},
				{Target: "chaosParameters.WaitTimeBetweenIterationsInSeconds", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "chaosParameters.WaitTimeBetweenIterationsInSeconds", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
						{Target: "chaosParameters.WaitTimeBetweenIterationsInSeconds", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil},
					}}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartChaos", err.Error())
	}

	req, err := client.StartChaosPreparer(ctx, chaosParameters, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartChaos", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartChaosSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartChaos", resp, "Failure sending request")
		return
	}

	result, err = client.StartChaosResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartChaos", resp, "Failure responding to request")
	}

	return
}

// StartChaosPreparer prepares the StartChaos request.
func (client BaseClient) StartChaosPreparer(ctx context.Context, chaosParameters ChaosParameters, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Tools/Chaos/$/Start"),
		autorest.WithJSON(chaosParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartChaosSender sends the StartChaos request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartChaosSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartChaosResponder handles the response to the StartChaos request. The method always
// closes the http.Response Body.
func (client BaseClient) StartChaosResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartDataLoss this API will induce data loss for the specified partition. It will trigger a call to the OnDataLoss
// API of the partition.
// Actual data loss will depend on the specified DataLossMode
// PartialDataLoss - Only a quorum of replicas are removed and OnDataLoss is triggered for the partition but actual
// data loss depends on the presence of in-flight replication.
// FullDataLoss - All replicas are removed hence all data is lost and OnDataLoss is triggered.
//
// This API should only be called with a stateful service as the target.
//
// Calling this API with a system service as the target is not advised.
//
// Note:  Once this API has been called, it cannot be reversed. Calling CancelOperation will only stop execution and
// clean up internal system state.
// It will not restore data if the command has progressed far enough to cause data loss.
//
// Call the GetDataLossProgress API with the same OperationId to return information on the operation started with this
// API.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// dataLossMode - this enum is passed to the StartDataLoss API to indicate what type of data loss to induce.
// - Invalid - Reserved.  Do not pass into API.
// - PartialDataLoss - PartialDataLoss option will cause a quorum of replicas to go down, triggering an
// OnDataLoss event in the system for the given partition.
// - FullDataLoss - FullDataLoss option will drop all the replicas which means that all the data will be lost.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartDataLoss(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, dataLossMode string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartDataLoss", err.Error())
	}

	req, err := client.StartDataLossPreparer(ctx, serviceID, partitionID, operationID, dataLossMode, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartDataLoss", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartDataLossSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartDataLoss", resp, "Failure sending request")
		return
	}

	result, err = client.StartDataLossResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartDataLoss", resp, "Failure responding to request")
	}

	return
}

// StartDataLossPreparer prepares the StartDataLoss request.
func (client BaseClient) StartDataLossPreparer(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, dataLossMode string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"serviceId":   serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":  APIVersion,
		"DataLossMode": autorest.Encode("query", dataLossMode),
		"OperationId":  autorest.Encode("query", operationID),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Services/{serviceId}/$/GetPartitions/{partitionId}/$/StartDataLoss", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartDataLossSender sends the StartDataLoss request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartDataLossSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartDataLossResponder handles the response to the StartDataLoss request. The method always
// closes the http.Response Body.
func (client BaseClient) StartDataLossResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartNode starts a Service Fabric cluster node that is already stopped.
// Parameters:
// nodeName - the name of the node.
// startNodeDescription - the instance id of the stopped node that needs to be started.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartNode(ctx context.Context, nodeName string, startNodeDescription StartNodeDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: startNodeDescription,
			Constraints: []validation.Constraint{{Target: "startNodeDescription.NodeInstanceID", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartNode", err.Error())
	}

	req, err := client.StartNodePreparer(ctx, nodeName, startNodeDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartNode", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartNodeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartNode", resp, "Failure sending request")
		return
	}

	result, err = client.StartNodeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartNode", resp, "Failure responding to request")
	}

	return
}

// StartNodePreparer prepares the StartNode request.
func (client BaseClient) StartNodePreparer(ctx context.Context, nodeName string, startNodeDescription StartNodeDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/Start", pathParameters),
		autorest.WithJSON(startNodeDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartNodeSender sends the StartNode request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartNodeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartNodeResponder handles the response to the StartNode request. The method always
// closes the http.Response Body.
func (client BaseClient) StartNodeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartNodeTransition starts or stops a cluster node.  A cluster node is a process, not the OS instance itself.  To
// start a node, pass in "Start" for the NodeTransitionType parameter.
// To stop a node, pass in "Stop" for the NodeTransitionType parameter.  This API starts the operation - when the API
// returns the node may not have finished transitioning yet.
// Call GetNodeTransitionProgress with the same OperationId to get the progress of the operation.
// Parameters:
// nodeName - the name of the node.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// nodeTransitionType - indicates the type of transition to perform.  NodeTransitionType.Start will start a
// stopped node.  NodeTransitionType.Stop will stop a node that is up.
// - Invalid - Reserved.  Do not pass into API.
// - Start - Transition a stopped node to up.
// - Stop - Transition an up node to stopped.
// nodeInstanceID - the node instance ID of the target node.  This can be determined through GetNodeInfo API.
// stopDurationInSeconds - the duration, in seconds, to keep the node stopped.  The minimum value is 600, the
// maximum is 14400.  After this time expires, the node will automatically come back up.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartNodeTransition(ctx context.Context, nodeName string, operationID uuid.UUID, nodeTransitionType string, nodeInstanceID string, stopDurationInSeconds int32, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: stopDurationInSeconds,
			Constraints: []validation.Constraint{{Target: "stopDurationInSeconds", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartNodeTransition", err.Error())
	}

	req, err := client.StartNodeTransitionPreparer(ctx, nodeName, operationID, nodeTransitionType, nodeInstanceID, stopDurationInSeconds, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartNodeTransition", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartNodeTransitionSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartNodeTransition", resp, "Failure sending request")
		return
	}

	result, err = client.StartNodeTransitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartNodeTransition", resp, "Failure responding to request")
	}

	return
}

// StartNodeTransitionPreparer prepares the StartNodeTransition request.
func (client BaseClient) StartNodeTransitionPreparer(ctx context.Context, nodeName string, operationID uuid.UUID, nodeTransitionType string, nodeInstanceID string, stopDurationInSeconds int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":           APIVersion,
		"NodeInstanceId":        autorest.Encode("query", nodeInstanceID),
		"NodeTransitionType":    autorest.Encode("query", nodeTransitionType),
		"OperationId":           autorest.Encode("query", operationID),
		"StopDurationInSeconds": autorest.Encode("query", stopDurationInSeconds),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Nodes/{nodeName}/$/StartTransition/", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartNodeTransitionSender sends the StartNodeTransition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartNodeTransitionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartNodeTransitionResponder handles the response to the StartNodeTransition request. The method always
// closes the http.Response Body.
func (client BaseClient) StartNodeTransitionResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartPartitionRestart this API is useful for testing failover.
//
// If used to target a stateless service partition, RestartPartitionMode must be AllReplicasOrInstances.
//
// Call the GetPartitionRestartProgress API using the same OperationId to get the progress.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// restartPartitionMode - - Invalid - Reserved.  Do not pass into API.
// - AllReplicasOrInstances - All replicas or instances in the partition are restarted at once.
// - OnlyActiveSecondaries - Only the secondary replicas are restarted.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartPartitionRestart(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, restartPartitionMode string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartPartitionRestart", err.Error())
	}

	req, err := client.StartPartitionRestartPreparer(ctx, serviceID, partitionID, operationID, restartPartitionMode, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartPartitionRestart", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartPartitionRestartSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartPartitionRestart", resp, "Failure sending request")
		return
	}

	result, err = client.StartPartitionRestartResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartPartitionRestart", resp, "Failure responding to request")
	}

	return
}

// StartPartitionRestartPreparer prepares the StartPartitionRestart request.
func (client BaseClient) StartPartitionRestartPreparer(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, restartPartitionMode string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"serviceId":   serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":          APIVersion,
		"OperationId":          autorest.Encode("query", operationID),
		"RestartPartitionMode": autorest.Encode("query", restartPartitionMode),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Services/{serviceId}/$/GetPartitions/{partitionId}/$/StartRestart", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartPartitionRestartSender sends the StartPartitionRestart request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartPartitionRestartSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartPartitionRestartResponder handles the response to the StartPartitionRestart request. The method always
// closes the http.Response Body.
func (client BaseClient) StartPartitionRestartResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StartQuorumLoss induces quorum loss for a given stateful service partition.  This API is useful for a temporary
// quorum loss situation on your service.
//
// Call the GetQuorumLossProgress API with the same OperationId to return information on the operation started with
// this API.
//
// This can only be called on stateful persisted (HasPersistedState==true) services.  Do not use this API on stateless
// services or stateful in-memory only services.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// partitionID - the identity of the partition.
// operationID - a GUID that identifies a call of this API.  This is passed into the corresponding GetProgress
// API
// quorumLossMode - this enum is passed to the StartQuorumLoss API to indicate what type of quorum loss to
// induce.
// - Invalid - Reserved.  Do not pass into API.
// - QuorumReplicas - Partial Quorum loss mode : Minimum number of replicas for a partition will be down that
// will cause a quorum loss.
// - AllReplicas- Full Quorum loss mode : All replicas for a partition will be down that will cause a quorum
// loss.
// quorumLossDuration - the amount of time for which the partition will be kept in quorum loss.  This must be
// specified in seconds.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StartQuorumLoss(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, quorumLossMode string, quorumLossDuration int32, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StartQuorumLoss", err.Error())
	}

	req, err := client.StartQuorumLossPreparer(ctx, serviceID, partitionID, operationID, quorumLossMode, quorumLossDuration, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartQuorumLoss", nil, "Failure preparing request")
		return
	}

	resp, err := client.StartQuorumLossSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartQuorumLoss", resp, "Failure sending request")
		return
	}

	result, err = client.StartQuorumLossResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StartQuorumLoss", resp, "Failure responding to request")
	}

	return
}

// StartQuorumLossPreparer prepares the StartQuorumLoss request.
func (client BaseClient) StartQuorumLossPreparer(ctx context.Context, serviceID string, partitionID uuid.UUID, operationID uuid.UUID, quorumLossMode string, quorumLossDuration int32, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"partitionId": partitionID,
		"serviceId":   serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version":        APIVersion,
		"OperationId":        autorest.Encode("query", operationID),
		"QuorumLossDuration": autorest.Encode("query", quorumLossDuration),
		"QuorumLossMode":     autorest.Encode("query", quorumLossMode),
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Faults/Services/{serviceId}/$/GetPartitions/{partitionId}/$/StartQuorumLoss", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StartQuorumLossSender sends the StartQuorumLoss request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StartQuorumLossSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StartQuorumLossResponder handles the response to the StartQuorumLoss request. The method always
// closes the http.Response Body.
func (client BaseClient) StartQuorumLossResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StopChaos stops Chaos from scheduling further faults; but, the in-flight faults are not affected.
// Parameters:
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StopChaos(ctx context.Context, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StopChaos", err.Error())
	}

	req, err := client.StopChaosPreparer(ctx, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StopChaos", nil, "Failure preparing request")
		return
	}

	resp, err := client.StopChaosSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StopChaos", resp, "Failure sending request")
		return
	}

	result, err = client.StopChaosResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StopChaos", resp, "Failure responding to request")
	}

	return
}

// StopChaosPreparer prepares the StopChaos request.
func (client BaseClient) StopChaosPreparer(ctx context.Context, timeout *int64) (*http.Request, error) {
	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/Tools/Chaos/$/Stop"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StopChaosSender sends the StopChaos request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StopChaosSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StopChaosResponder handles the response to the StopChaos request. The method always
// closes the http.Response Body.
func (client BaseClient) StopChaosResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StopNode stops a Service Fabric cluster node that is in a started state. The node will stay down until start node is
// called.
// Parameters:
// nodeName - the name of the node.
// stopNodeDescription - the instance id of the stopped node that needs to be stopped.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) StopNode(ctx context.Context, nodeName string, stopNodeDescription StopNodeDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: stopNodeDescription,
			Constraints: []validation.Constraint{{Target: "stopNodeDescription.NodeInstanceID", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "StopNode", err.Error())
	}

	req, err := client.StopNodePreparer(ctx, nodeName, stopNodeDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StopNode", nil, "Failure preparing request")
		return
	}

	resp, err := client.StopNodeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StopNode", resp, "Failure sending request")
		return
	}

	result, err = client.StopNodeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "StopNode", resp, "Failure responding to request")
	}

	return
}

// StopNodePreparer prepares the StopNode request.
func (client BaseClient) StopNodePreparer(ctx context.Context, nodeName string, stopNodeDescription StopNodeDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"nodeName": autorest.Encode("path", nodeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Nodes/{nodeName}/$/Stop", pathParameters),
		autorest.WithJSON(stopNodeDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StopNodeSender sends the StopNode request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) StopNodeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StopNodeResponder handles the response to the StopNode request. The method always
// closes the http.Response Body.
func (client BaseClient) StopNodeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UnprovisionApplicationType removes or unregisters a Service Fabric application type from the cluster. This operation
// can only be performed if all application instance of the application type has been deleted. Once the application
// type is unregistered, no new application instance can be created for this particular application type.
// Parameters:
// applicationTypeName - the name of the application type.
// applicationTypeImageStoreVersion - the version of the application type in the image store.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) UnprovisionApplicationType(ctx context.Context, applicationTypeName string, applicationTypeImageStoreVersion ApplicationTypeImageStoreVersion, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: applicationTypeImageStoreVersion,
			Constraints: []validation.Constraint{{Target: "applicationTypeImageStoreVersion.ApplicationTypeVersion", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "UnprovisionApplicationType", err.Error())
	}

	req, err := client.UnprovisionApplicationTypePreparer(ctx, applicationTypeName, applicationTypeImageStoreVersion, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UnprovisionApplicationType", nil, "Failure preparing request")
		return
	}

	resp, err := client.UnprovisionApplicationTypeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UnprovisionApplicationType", resp, "Failure sending request")
		return
	}

	result, err = client.UnprovisionApplicationTypeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UnprovisionApplicationType", resp, "Failure responding to request")
	}

	return
}

// UnprovisionApplicationTypePreparer prepares the UnprovisionApplicationType request.
func (client BaseClient) UnprovisionApplicationTypePreparer(ctx context.Context, applicationTypeName string, applicationTypeImageStoreVersion ApplicationTypeImageStoreVersion, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationTypeName": autorest.Encode("path", applicationTypeName),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ApplicationTypes/{applicationTypeName}/$/Unprovision", pathParameters),
		autorest.WithJSON(applicationTypeImageStoreVersion),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UnprovisionApplicationTypeSender sends the UnprovisionApplicationType request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UnprovisionApplicationTypeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UnprovisionApplicationTypeResponder handles the response to the UnprovisionApplicationType request. The method always
// closes the http.Response Body.
func (client BaseClient) UnprovisionApplicationTypeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpdateApplicationUpgrade updates the parameters of an ongoing application upgrade from the ones specified at the
// time of starting the application upgrade. This may be required to mitigate stuck application upgrades due to
// incorrect parameters or issues in the application to make progress.
// Parameters:
// applicationID - the identity of the application. This is typically the full name of the application without
// the 'fabric:' URI scheme.
// applicationUpgradeUpdateDescription - describes the parameters for updating an existing application upgrade.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) UpdateApplicationUpgrade(ctx context.Context, applicationID string, applicationUpgradeUpdateDescription ApplicationUpgradeUpdateDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: applicationUpgradeUpdateDescription,
			Constraints: []validation.Constraint{{Target: "applicationUpgradeUpdateDescription.Name", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "UpdateApplicationUpgrade", err.Error())
	}

	req, err := client.UpdateApplicationUpgradePreparer(ctx, applicationID, applicationUpgradeUpdateDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UpdateApplicationUpgrade", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateApplicationUpgradeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UpdateApplicationUpgrade", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateApplicationUpgradeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UpdateApplicationUpgrade", resp, "Failure responding to request")
	}

	return
}

// UpdateApplicationUpgradePreparer prepares the UpdateApplicationUpgrade request.
func (client BaseClient) UpdateApplicationUpgradePreparer(ctx context.Context, applicationID string, applicationUpgradeUpdateDescription ApplicationUpgradeUpdateDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"applicationId": applicationID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Applications/{applicationId}/$/UpdateUpgrade", pathParameters),
		autorest.WithJSON(applicationUpgradeUpdateDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateApplicationUpgradeSender sends the UpdateApplicationUpgrade request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateApplicationUpgradeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateApplicationUpgradeResponder handles the response to the UpdateApplicationUpgrade request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateApplicationUpgradeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpdateService updates the specified service using the given update description.
// Parameters:
// serviceID - the identity of the service. This is typically the full name of the service without the
// 'fabric:' URI scheme.
// serviceUpdateDescription - the updated configuration for the service.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) UpdateService(ctx context.Context, serviceID string, serviceUpdateDescription BasicServiceUpdateDescription, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "UpdateService", err.Error())
	}

	req, err := client.UpdateServicePreparer(ctx, serviceID, serviceUpdateDescription, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UpdateService", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateServiceSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UpdateService", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateServiceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UpdateService", resp, "Failure responding to request")
	}

	return
}

// UpdateServicePreparer prepares the UpdateService request.
func (client BaseClient) UpdateServicePreparer(ctx context.Context, serviceID string, serviceUpdateDescription BasicServiceUpdateDescription, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"serviceId": serviceID,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/Services/{serviceId}/$/Update", pathParameters),
		autorest.WithJSON(serviceUpdateDescription),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateServiceSender sends the UpdateService request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateServiceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateServiceResponder handles the response to the UpdateService request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateServiceResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UploadFile uploads contents of the file to the image store. Use this API if the file is small enough to upload again
// if the connection fails. The file's data needs to be added to the request body. The contents will be uploaded to the
// specified path.
// Parameters:
// contentPath - relative path to file or folder in the image store from its root.
// timeout - the server timeout for performing the operation in seconds. This specifies the time duration that
// the client is willing to wait for the requested operation to complete. The default value for this parameter
// is 60 seconds.
func (client BaseClient) UploadFile(ctx context.Context, contentPath string, timeout *int64) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: timeout,
			Constraints: []validation.Constraint{{Target: "timeout", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "timeout", Name: validation.InclusiveMaximum, Rule: 4294967295, Chain: nil},
					{Target: "timeout", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("servicefabric.BaseClient", "UploadFile", err.Error())
	}

	req, err := client.UploadFilePreparer(ctx, contentPath, timeout)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UploadFile", nil, "Failure preparing request")
		return
	}

	resp, err := client.UploadFileSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UploadFile", resp, "Failure sending request")
		return
	}

	result, err = client.UploadFileResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "servicefabric.BaseClient", "UploadFile", resp, "Failure responding to request")
	}

	return
}

// UploadFilePreparer prepares the UploadFile request.
func (client BaseClient) UploadFilePreparer(ctx context.Context, contentPath string, timeout *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"contentPath": autorest.Encode("path", contentPath),
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 60)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/ImageStore/{contentPath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UploadFileSender sends the UploadFile request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UploadFileSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UploadFileResponder handles the response to the UploadFile request. The method always
// closes the http.Response Body.
func (client BaseClient) UploadFileResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}
