#!/bin/bash

LOG=$2
CONF=/etc/cpfs/opencache.conf

#
# do nothing if config doesn't exist
#
if [ -f ${CONF} ]; then
    . ${CONF}
else
    echo "Skip config open cache because $CONF not exists" >> ${LOG}
    exit
fi

PROC=/proc/fs/lustre
LLITE=${PROC}/llite
SYSFS=/sys/fs/lustre
NS=${SYSFS}/ldlm/namespaces
DEFAULT_TARGET_DATA_CACHE_RATIO=50

setup()
{
    MNTPOINT=$1
    local fsname
    local mem_total
    local mem_pagecache
    local mdt_nr
    local ost_nr
    local file_nr
    local lock_nr

    [[ -z "$TARGET_FILE_NR" || $TARGET_FILE_NR -eq 0 ]] && { echo "Skip open cache config because no TARGET_FILE_NR" >> ${LOG}; return; }
    # verify the given mntpoint is mounted
    [ -z "${MNTPOINT}" ] && { echo "MNTPOINT missing for open cache config" >> ${LOG}; return 1; }
    MOUNTPOINTS=`mount | grep lustre | awk '{ print $3}'`
    [ -z "$MOUNTPOINTS" ] && { echo "mount have no type of lustre" >> ${LOG}; return 1; }
    match_res=`echo "$MOUNTPOINTS" | grep -w "$MNTPOINT"`
    [ -z "$match_res" ] || [ "$match_res" != "$MNTPOINT" ] && { echo "Invalide MNTPOINT($MNTPOINT), please consistent with /etc/cpfs/cpfs-mount.conf" >> ${LOG}; return 1; }
    [ ! -e ${LLITE} ] && return 1

    mem_total=$(free | grep Mem | awk '{print $2}')
    fsname=$(lfs getname ${MNTPOINT} | awk -F'-' '{print $1}')
    fsname_uuid=$(lfs getname ${MNTPOINT} | awk '{print $1}')
    uuid=$(lfs getname ${MNTPOINT} | awk -F'[-| ]' '{print $2}')
    mdt_nr=$(lfs mdts ${MNTPOINT} | grep "^[0-9]*:" | wc -l)
    ost_nr=$(lfs osts ${MNTPOINT} | grep "^[0-9]*:" | wc -l)
    file_nr=$((TARGET_FILE_NR + TARGET_FILE_NR / 5)) # add some extra

    if [ ${mdt_nr} -eq 0 -o ${ost_nr} -eq 0 ]; then
        echo "Only found ${mdt_nr} MDTs and ${ost_nr} OSTs" >> ${LOG}
        return 1
    fi

    # metadata dlm lock lru, add some extra
    lock_nr=${file_nr}
    # if the lock memory is small enough, double the lock count for the case
    # that application do stat() which would get attr lock.
    if [ $((mem_total / (lock_nr * 512 / 1024))) -ge 100 ]; then
        lock_nr=$((lock_nr * 2))
    fi

    echo "Setting up ${MNTPOINT} open cache..." >> ${LOG}

    if [[ "${TARGET_FILE_NR}" && $TARGET_FILE_NR -gt 0 ]]; then
        # enable open lock
        echo 1 > ${LLITE}/${fsname_uuid}/open_cache
        # devide between MDCs
        lock_nr=$((lock_nr / mdt_nr))
        for f in $(ls ${NS}/${fsname}-MDT*-${uuid}/lru_size); do
            echo ${lock_nr} > ${f}
        done
        echo "Set each MDC's lock lru_size ${lock_nr}" >> ${LOG}

        # vfs object cache
        echo ${file_nr} > ${LLITE}/${fsname_uuid}/vfsobj_lru_size
        # lu site cache
        echo off > ${SYSFS}/site_cache_shrink
    fi

    if [[ "${TARGET_DATA_CACHE_RATIO}" && ${TARGET_DATA_CACHE_RATIO} -gt 0 ]]; then
        if [ ${TARGET_DATA_CACHE_RATIO} -ge 100 ]; then
            echo "Data cache config failed because invalid TARGET_DATA_CACHE_RATIO $TARGET_DATA_CACHE_RATIO" >> ${LOG}
            return 1
        fi

        # data dlm lock lru
        lock_nr=$((file_nr / ost_nr))
        for f in $(ls ${NS}/${fsname}-OST*-${uuid}/lru_size); do
            echo ${lock_nr} > $f
        done
        echo "Set each OSC's lock lru_size ${lock_nr}" >> ${LOG}

        # aggressive page cache, TARGET_DATA_CACHE_RATIO of total memory
        mem_pagecache=$((mem_total * TARGET_DATA_CACHE_RATIO / 100 / 1024))
        echo ${mem_pagecache} > ${LLITE}/${fsname_uuid}/max_cached_mb
        echo "Set page cache size ${mem_pagecache} MiB" >> ${LOG}
    fi

    echo "Open cache config of ${MNTPOINT} done" >> ${LOG}
}

teardown()
{
    MNTPOINT=$1
    local fsname
    local mem_total
    local mem_pagecache

    # verify the given mntpoint is mounted
    [ -z "${MNTPOINT}" ] && { echo "MNTPOINT missing for open cache config" >> ${LOG}; return 1; }
    MOUNTPOINTS=`mount | grep lustre | awk '{ print $3}'`
    [ -z "$MOUNTPOINTS" ] && { echo "mount have no type of lustre" >> ${LOG}; return 1; }
    match_res=`echo "$MOUNTPOINTS" | grep -w "$MNTPOINT"`
    [ -z "$match_res" ] || [ "$match_res" != "$MNTPOINT" ] && { echo "Invalide MNTPOINT($MNTPOINT), please consistent with /etc/cpfs/cpfs-mount.conf" >> ${LOG}; return 1; }

    mem_total=$(free | grep Mem | awk '{print $2}')
    fsname=$(lfs getname ${MNTPOINT} | awk -F'-' '{print $1}')
    fsname_uuid=$(lfs getname ${MNTPOINT} | awk '{print $1}')
    uuid=$(lfs getname ${MNTPOINT} | awk -F'[-| ]' '{print $2}')

    echo "Stopping ${MNTPOINT} open cache..." >> ${LOG}

    # disable open lock
    echo 0 > ${LLITE}/${fsname_uuid}/open_cache
    # MDCS
    for f in $(ls ${NS}/${fsname}-MDT*-${uuid}/lru_size); do
        echo 0 > ${f}
    done
    echo "Set each MDC's lock lru_size to 0" >> ${LOG}
    # vfs obejcet cache
    echo 0 > ${LLITE}/${fsname_uuid}/vfsobj_lru_size
    # lu site cache
    echo on > ${SYSFS}/site_cache_shrink
    # data dlm lock lru
    for f in $(ls ${NS}/${fsname}-OST*-${uuid}/lru_size); do
        echo 0 > $f
    done
    echo "Set each OSC's lock lru_size to 0" >> ${LOG}
    # aggressive page cache, 50 of total memory
    mem_pagecache=$((mem_total * DEFAULT_TARGET_DATA_CACHE_RATIO / 100 / 1024))
    echo ${mem_pagecache} > ${LLITE}/${fsname_uuid}/max_cached_mb
    echo "Set page cache size ${mem_pagecache} MiB" >> ${LOG}

    echo "Stop open cache of ${MNTPOINT} done" >> ${LOG}
}

case "$1" in
    start)
        NUM=`echo $MNTPOINTS | awk '{print split($0, A, ";")}'`
        [ $NUM -eq 0 ] && { echo "MNTPOINT missing for open cache config" >> ${LOG}; exit 1; }
        for ((i=1; i <= $NUM; i++))
        do
            MNTPOINT=`echo $MNTPOINTS | awk '{split($0, A, ";"); print A['$i']}'`
            setup $MNTPOINT
            [ $? -eq 0 ] || { echo "Config open cache failed" >> ${LOG}; exit 1; }
        done
        ;;
    stop)
        NUM=`echo $MNTPOINTS | awk '{print split($0, A, ";")}'`
        [ $NUM -eq 0 ] && { echo "MNTPOINT missing for open cache config" >> ${LOG}; exit 1; }
        for ((i=1; i <= $NUM; i++))
        do
            MNTPOINT=`echo $MNTPOINTS | awk '{split($0, A, ";"); print A['$i']}'`
            teardown $MNTPOINT
            [ $? -eq 0 ] || { echo "Stop open cache failed" >> ${LOG}; exit 1; }
        done
        ;;
    *)
        echo "Usage: $0 {start|stop} [log]"
        exit 3
        ;;
    esac

