/*
 * GPL HEADER START
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 only,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License version 2 for more details (a copy is included
 * in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; If not, see
 * http://www.gnu.org/licenses/gpl-2.0.html
 *
 * GPL HEADER END
 */
/*
 * Copyright (c) 2002, 2010, Oracle and/or its affiliates. All rights reserved.
 * Use is subject to license terms.
 *
 * Copyright (c) 2011, 2016, Intel Corporation.
 */
/*
 * This file is part of Lustre, http://www.lustre.org/
 * Lustre is a trademark of Sun Microsystems, Inc.
 *
 * lustre/llite/rw.c
 *
 * Lustre Lite I/O page cache routines shared by different kernel revs
 */

#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/unistd.h>
#include <linux/writeback.h>
#include <asm/uaccess.h>

#include <linux/fs.h>
#include <linux/stat.h>
#include <asm/uaccess.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
/* current_is_kswapd() */
#include <linux/swap.h>
#include <linux/task_io_accounting_ops.h>

#define DEBUG_SUBSYSTEM S_LLITE

#include <obd_cksum.h>
#include "llite_internal.h"
#include <lustre_compat.h>

static void ll_ra_stats_inc_sbi(struct ll_sb_info *sbi, enum ra_stat which);

/**
 * Get readahead pages from the filesystem readahead pool of the client for a
 * thread.
 *
 * /param sbi superblock for filesystem readahead state ll_ra_info
 * /param ria per-thread readahead state
 * /param pages number of pages requested for readahead for the thread.
 *
 * WARNING: This algorithm is used to reduce contention on sbi->ll_lock.
 * It should work well if the ra_max_pages is much greater than the single
 * file's read-ahead window, and not too many threads contending for
 * these readahead pages.
 *
 * TODO: There may be a 'global sync problem' if many threads are trying
 * to get an ra budget that is larger than the remaining readahead pages
 * and reach here at exactly the same time. They will compute /a ret to
 * consume the remaining pages, but will fail at atomic_add_return() and
 * get a zero ra window, although there is still ra space remaining. - Jay */

static unsigned long ll_ra_count_get(struct ll_sb_info *sbi,
				     struct ra_io_arg *ria,
				     unsigned long pages, unsigned long min)
{
        struct ll_ra_info *ra = &sbi->ll_ra_info;
        long ret;
        ENTRY;

        /* If read-ahead pages left are less than 1M, do not do read-ahead,
         * otherwise it will form small read RPC(< 1M), which hurt server
         * performance a lot. */
	ret = min(ra->ra_max_pages - atomic_read(&ra->ra_cur_pages),
		  pages);
        if (ret < 0 || ret < min_t(long, PTLRPC_MAX_BRW_PAGES, pages))
                GOTO(out, ret = 0);

	if (atomic_add_return(ret, &ra->ra_cur_pages) > ra->ra_max_pages) {
		atomic_sub(ret, &ra->ra_cur_pages);
		ret = 0;
	}

out:
	if (ret < min) {
		/* override ra limit for maximum performance */
		atomic_add(min - ret, &ra->ra_cur_pages);
		ret = min;
	}
	RETURN(ret);
}

void ll_ra_count_put(struct ll_sb_info *sbi, unsigned long len)
{
	struct ll_ra_info *ra = &sbi->ll_ra_info;
	atomic_sub(len, &ra->ra_cur_pages);
}

static void ll_ra_stats_inc_sbi(struct ll_sb_info *sbi, enum ra_stat which)
{
	LASSERTF(which < _NR_RA_STAT, "which: %u\n", which);
	lprocfs_counter_incr(sbi->ll_ra_stats, which);
}

void ll_ra_stats_inc(struct inode *inode, enum ra_stat which)
{
	struct ll_sb_info *sbi = ll_i2sbi(inode);
	ll_ra_stats_inc_sbi(sbi, which);
}

#define RAS_CDEBUG(ras) \
	CDEBUG(D_READA,                                                      \
	       "lrp %lu cr %lu cp %lu ws %lu wl %lu nra %lu rpc %lu "        \
	       "r %lu ri %lu csr %lu sf %lu sp %lu sl %lu\n",                \
	       ras->ras_last_readpage, ras->ras_consecutive_requests,        \
	       ras->ras_consecutive_pages, ras->ras_window_start,            \
	       ras->ras_window_len, ras->ras_next_readahead,                 \
	       ras->ras_rpc_size,                                            \
	       ras->ras_requests, ras->ras_request_index,                    \
	       ras->ras_consecutive_stride_requests, ras->ras_stride_offset, \
	       ras->ras_stride_pages, ras->ras_stride_length)

#define RAS_OD_CDEBUG(ras, level) \
        CDEBUG(level,                                                        \
                "OD RAS: tw [%lu, %lu) (%lu), od [%lu, +%lu) m %lu, "        \
                "hit %lu\n",                                                 \
                ras->ras_od_tw_start, ras->ras_od_tw_end,                    \
                ras->ras_od_tw_end - ras->ras_od_tw_start,                   \
                ras->ras_od_start, ras->ras_od_size,                         \
                ras->ras_od_start + ras->ras_od_size -                       \
                        ras->ras_od_async_size,                              \
                ras->ras_od_hit)                                             \

#define RAS_OD_TRACE(ras)	RAS_OD_CDEBUG(ras, D_READA)
#define RAS_OD_WARN(ras)	RAS_OD_CDEBUG(ras, D_WARNING)

static int index_in_window(unsigned long index, unsigned long point,
                           unsigned long before, unsigned long after)
{
        unsigned long start = point - before, end = point + after;

        if (start > point)
               start = 0;
        if (end < point)
               end = ~0;

        return start <= index && index <= end;
}

void ll_ras_enter(struct file *f)
{
	struct ll_file_data *fd = LUSTRE_FPRIVATE(f);
	struct ll_readahead_state *ras = &fd->fd_ras;

	spin_lock(&ras->ras_lock);
	ras->ras_requests++;
	ras->ras_request_index = 0;
	ras->ras_consecutive_requests++;
	spin_unlock(&ras->ras_lock);
}

/**
 * Initiates read-ahead of a page with given index.
 *
 * \retval +ve: page was already uptodate so it will be skipped
 *              from being added;
 * \retval -ve: page wasn't added to \a queue for error;
 * \retval   0: page was added into \a queue for read ahead.
 */
static int ll_read_ahead_page(const struct lu_env *env, struct cl_io *io,
			      struct cl_page_list *queue, pgoff_t index,
			      bool od_marker)
{
	struct cl_object *clob  = io->ci_obj;
	struct inode     *inode = vvp_object_inode(clob);
	struct page      *vmpage;
	struct cl_page   *page;
	struct vvp_page  *vpg;
	enum ra_stat      which = _NR_RA_STAT; /* keep gcc happy */
	int               rc    = 0;
	const char       *msg   = NULL;
	ENTRY;

	vmpage = grab_cache_page_nowait(inode->i_mapping, index);
	if (vmpage == NULL) {
		which = RA_STAT_FAILED_GRAB_PAGE;
		msg   = "g_c_p_n failed";
		GOTO(out, rc = -EBUSY);
	}

	/* Check if vmpage was truncated or reclaimed */
	if (vmpage->mapping != inode->i_mapping) {
		which = RA_STAT_WRONG_GRAB_PAGE;
		msg   = "g_c_p_n returned invalid page";
		GOTO(out, rc = -EBUSY);
	}

	page = cl_page_find(env, clob, vmpage->index, vmpage, CPT_CACHEABLE);
	if (IS_ERR(page)) {
		which = RA_STAT_FAILED_GRAB_PAGE;
		msg   = "cl_page_find failed";
		GOTO(out, rc = PTR_ERR(page));
	}

	lu_ref_add(&page->cp_reference, "ra", current);
	cl_page_assume(env, io, page);
	vpg = cl2vvp_page(cl_object_page_slice(clob, page));
	if (!vpg->vpg_defer_uptodate && !PageUptodate(vmpage)) {
		vpg->vpg_defer_uptodate = 1;
		vpg->vpg_ra_used = 0;
		vpg->vpg_ra_od_marker = !!od_marker;
		cl_page_list_add(queue, page);
	} else {
		/* skip completed pages */
		cl_page_unassume(env, io, page);
		/* This page is already uptodate, returning a positive number
		 * to tell the callers about this */
		rc = 1;
	}

	lu_ref_del(&page->cp_reference, "ra", current);
	cl_page_put(env, page);

out:
	if (vmpage != NULL) {
		if (rc != 0)
			unlock_page(vmpage);
		put_page(vmpage);
	}
	if (msg != NULL) {
		ll_ra_stats_inc(inode, which);
		CDEBUG(D_READA, "%s\n", msg);

	}

	RETURN(rc);
}

#define RIA_DEBUG(ria)                                                       \
        CDEBUG(D_READA, "rs %lu re %lu ro %lu rl %lu rp %lu\n",       \
        ria->ria_start, ria->ria_end, ria->ria_stoff, ria->ria_length,\
        ria->ria_pages)

static inline int stride_io_mode(struct ll_readahead_state *ras)
{
        return ras->ras_consecutive_stride_requests > 1;
}

/* The function calculates how much pages will be read in
 * [off, off + length], in such stride IO area,
 * stride_offset = st_off, stride_lengh = st_len,
 * stride_pages = st_pgs
 *
 *   |------------------|*****|------------------|*****|------------|*****|....
 * st_off
 *   |--- st_pgs     ---|
 *   |-----     st_len   -----|
 *
 *              How many pages it should read in such pattern
 *              |-------------------------------------------------------------|
 *              off
 *              |<------                  length                      ------->|
 *
 *          =   |<----->|  +  |-------------------------------------| +   |---|
 *             start_left                 st_pgs * i                    end_left
 */
static unsigned long
stride_pg_count(pgoff_t st_off, unsigned long st_len, unsigned long st_pgs,
                unsigned long off, unsigned long length)
{
        __u64 start = off > st_off ? off - st_off : 0;
        __u64 end = off + length > st_off ? off + length - st_off : 0;
        unsigned long start_left = 0;
        unsigned long end_left = 0;
        unsigned long pg_count;

        if (st_len == 0 || length == 0 || end == 0)
                return length;

        start_left = do_div(start, st_len);
        if (start_left < st_pgs)
                start_left = st_pgs - start_left;
        else
                start_left = 0;

        end_left = do_div(end, st_len);
        if (end_left > st_pgs)
                end_left = st_pgs;

	CDEBUG(D_READA, "start %llu, end %llu start_left %lu end_left %lu\n",
               start, end, start_left, end_left);

        if (start == end)
                pg_count = end_left - (st_pgs - start_left);
        else
                pg_count = start_left + st_pgs * (end - start - 1) + end_left;

        CDEBUG(D_READA, "st_off %lu, st_len %lu st_pgs %lu off %lu length %lu"
               "pgcount %lu\n", st_off, st_len, st_pgs, off, length, pg_count);

        return pg_count;
}

static int ria_page_count(struct ra_io_arg *ria)
{
        __u64 length = ria->ria_end >= ria->ria_start ?
                       ria->ria_end - ria->ria_start + 1 : 0;

        return stride_pg_count(ria->ria_stoff, ria->ria_length,
                               ria->ria_pages, ria->ria_start,
                               length);
}

static unsigned long ras_align(struct ll_readahead_state *ras,
			       unsigned long index,
			       unsigned long *remainder)
{
	unsigned long rem = index % ras->ras_rpc_size;
	if (remainder != NULL)
		*remainder = rem;
	return index - rem;
}

/*Check whether the index is in the defined ra-window */
static int ras_inside_ra_window(unsigned long idx, struct ra_io_arg *ria)
{
        /* If ria_length == ria_pages, it means non-stride I/O mode,
         * idx should always inside read-ahead window in this case
         * For stride I/O mode, just check whether the idx is inside
         * the ria_pages. */
        return ria->ria_length == 0 || ria->ria_length == ria->ria_pages ||
               (idx >= ria->ria_stoff && (idx - ria->ria_stoff) %
                ria->ria_length < ria->ria_pages);
}

static inline
void adjust_read_ahead_endpoint(struct ll_readahead_state *ras,
				struct cl_read_ahead *ra,
				struct ra_io_arg *ria,
				bool ondemand)
{
	unsigned long end;

	end = ras_align(ras, ria->ria_end + 1, NULL);
	if (ondemand) {
		/* ondemand has its own readahead pace, so only
		 * trim to rpc boundary when we go across it, note
		 * the target window may continue to expand so we'll
		 * likely get nice aligned readahead eventually.
		 */
		if (end > ria->ria_start && !ria->ria_eof) {
			CDEBUG(D_READA, "trim end %lu to rpc boundary %lu\n",
			       ria->ria_end, end - 1);
			ria->ria_end = end - 1;
		}
	} else {
		/* trim it to align with optimal RPC size */
		if (end > 0 && !ria->ria_eof)
			ria->ria_end = end - 1;
	}

	if (ria->ria_end < ria->ria_end_min)
		ria->ria_end = ria->ria_end_min;
	if (ria->ria_end > ra->cra_end)
		ria->ria_end = ra->cra_end;
}

static unsigned long
ll_read_ahead_pages(const struct lu_env *env, struct cl_io *io, bool ondemand,
		    struct cl_page_list *queue, struct ll_readahead_state *ras,
		    struct ra_io_arg *ria, pgoff_t *ra_end)
{
	struct cl_read_ahead ra = { 0 };
	int rc = 0, count = 0;
	bool stride_ria;
	pgoff_t page_idx;

	LASSERT(ria != NULL);
	RIA_DEBUG(ria);

	stride_ria = ria->ria_length > ria->ria_pages && ria->ria_pages > 0;
	for (page_idx = ria->ria_start;
	     page_idx <= ria->ria_end && ria->ria_reserved > 0; page_idx++) {
		if (ras_inside_ra_window(page_idx, ria)) {
			if (ra.cra_end == 0 || ra.cra_end < page_idx) {
				cl_read_ahead_release(env, &ra);

				rc = cl_io_read_ahead(env, io, page_idx, &ra);
				if (rc < 0)
					break;

				CDEBUG(D_READA, "idx: %lu, ra: %lu, rpc: %lu\n",
				       page_idx, ra.cra_end, ra.cra_rpc_size);
				LASSERTF(ra.cra_end >= page_idx,
					 "object: %p, indcies %lu / %lu\n",
					 io->ci_obj, ra.cra_end, page_idx);
				/* update read ahead RPC size.
				 * NB: it's racy but doesn't matter */
				if (ras->ras_rpc_size > ra.cra_rpc_size &&
				    ra.cra_rpc_size > 0)
					ras->ras_rpc_size = ra.cra_rpc_size;

				adjust_read_ahead_endpoint(ras, &ra, ria,
							   ondemand);
			}
			if (page_idx > ria->ria_end)
				break;

			/* If the page is inside the read-ahead window */
			rc = ll_read_ahead_page(env, io, queue, page_idx,
					(ria->ria_marker &&
					 page_idx == ria->ria_marker));
			if (rc < 0)
				break;

			*ra_end = page_idx;
			/* Only subtract from reserve & count the page if we
			 * really did readahead on that page. */
			if (rc == 0) {
				ria->ria_reserved--;
				count++;
			}
                } else if (stride_ria) {
                        /* If it is not in the read-ahead window, and it is
                         * read-ahead mode, then check whether it should skip
                         * the stride gap */
                        pgoff_t offset;
                        /* FIXME: This assertion only is valid when it is for
                         * forward read-ahead, it will be fixed when backward
                         * read-ahead is implemented */
			LASSERTF(page_idx >= ria->ria_stoff,
				"Invalid page_idx %lu rs %lu re %lu ro %lu "
				"rl %lu rp %lu\n", page_idx,
				ria->ria_start, ria->ria_end, ria->ria_stoff,
				ria->ria_length, ria->ria_pages);
                        offset = page_idx - ria->ria_stoff;
                        offset = offset % (ria->ria_length);
                        if (offset > ria->ria_pages) {
                                page_idx += ria->ria_length - offset;
                                CDEBUG(D_READA, "i %lu skip %lu \n", page_idx,
                                       ria->ria_length - offset);
                                continue;
                        }
                }
        }

	cl_read_ahead_release(env, &ra);

	return count;
}

static unsigned long init_od_ra_size(unsigned long size, unsigned long min_ra)
{
	unsigned long newsize = roundup_pow_of_two(size);

	if (newsize <= PTLRPC_MAX_BRW_PAGES / 32)
		newsize = newsize * 4;
	else if (newsize <= PTLRPC_MAX_BRW_PAGES / 4)
		newsize = newsize * 2;
	else
		newsize = PTLRPC_MAX_BRW_PAGES;

	return max_t(unsigned long, newsize, min_ra);
}

static unsigned long next_od_ra_size(struct ll_readahead_state *ras,
				     unsigned long min_ra)
{
	unsigned long cur = ras->ras_od_size;
	unsigned long newsize;

	/* od window might have be moved forward within target window, in
	 * cases such as readahead was truncated at stripe boundary
	 * multiple times. here we calculate using original intended
	 * window size. */
	if (ras->ras_od_tw_start != 0 &&
	    ras->ras_od_tw_start < ras->ras_od_start)
		cur += ras->ras_od_start - ras->ras_od_tw_start;

	cur = max_t(unsigned long, 1, cur);
	if (cur < ras->ras_od_max_window / 16)
		newsize = 4 * cur;
	else
		newsize = 2 * cur;

	newsize = max_t(unsigned long, newsize, min_ra);
	return min_t(unsigned long, newsize, ras->ras_od_max_window);
}

/*
 * limit od window within limit, note it's the per-file maximum not the
 * rpc limit, since we want the target window to be big enough.
 */
static inline void od_trim_window(struct ll_readahead_state *ras)
{
	if (unlikely(ras->ras_od_size > ras->ras_od_max_window)) {
		ras->ras_od_size = ras->ras_od_max_window;
		ras->ras_od_async_size = min_t(unsigned long,
						ras->ras_od_async_size,
						ras->ras_od_size);
	}
}

/*
 * expand or move the target window according to current page index
 * and od window. call need to make sure index is in current window.
 */
static void od_expand_target_window(struct ll_readahead_state *ras,
				    unsigned long index)
{
	od_trim_window(ras);

	/* move start forward, align the start helps keep the end aligned */
	index = min_t(unsigned long, index, ras->ras_od_start);
	if (index / ras->ras_rpc_size >
				ras->ras_od_tw_start / ras->ras_rpc_size)
		ras->ras_od_tw_start = ras_align(ras, index, NULL);

	/* the range has covered everything? */
	if (ras->ras_od_tw_end >= ras->ras_od_start + ras->ras_od_size)
		return;

	/* expand the range */
	ras->ras_od_tw_end = ras->ras_od_start + ras->ras_od_size;

	/* but keep it within limit */
	if (ras->ras_od_tw_start + ras->ras_od_max_window >= ras->ras_od_tw_end)
		return;

	/* trim down the edge of tw and od window */
	ras->ras_od_tw_end = ras->ras_od_tw_start + ras->ras_od_max_window;
	ras->ras_od_size = ras->ras_od_tw_end - ras->ras_od_start;
	ras->ras_od_async_size = ras->ras_od_size;
}

/* reset target window to align to od */
static void od_reset_target_window(struct ll_readahead_state *ras)
{
	od_trim_window(ras);

	ras->ras_od_tw_start = ras->ras_od_start;
	ras->ras_od_tw_end = ras->ras_od_start + ras->ras_od_size;
}

static bool od_page_in_target_window(struct ll_readahead_state *ras,
				     unsigned long index)
{
	return index >= ras->ras_od_tw_start && index < ras->ras_od_tw_end;
}

/*
 * return true if od has not reached the end of target window, this is
 * normally caused by readahead get trimed as stripe/rpc boundary.
 */
static bool od_target_window_has_gap(struct ll_readahead_state *ras)
{
	return ras->ras_od_start + ras->ras_od_size < ras->ras_od_tw_end;
}

/*
 * return true if we have good amount of readahead in target window.
 */
static bool od_target_window_is_full(struct ll_readahead_state *ras)
{
	return ras->ras_od_start + ras->ras_od_size + ras->ras_rpc_size >=
			ras->ras_od_tw_end;
}

/*
 * shrink od window according to the maximum page we can reach @end.
 */
static void od_shrink_window(struct ll_readahead_state *ras,
			     unsigned long end)
{
	if (ras->ras_od_start + ras->ras_od_size > end + 1) {
		if (end == 0)
			ras->ras_od_size = 0;
		else
			ras->ras_od_size = end + 1 - ras->ras_od_start;

		ras->ras_od_async_size = min(ras->ras_od_size,
					     ras->ras_od_async_size);

		CDEBUG(D_READA, "shrink od to %lu: [%lu, +%lu) m %lu\n",
		       end, ras->ras_od_start, ras->ras_od_size,
		       ras->ras_od_start + ras->ras_od_size -
				ras->ras_od_async_size);
	}
}

static void od_clear_state(struct ll_ra_info *rai,
			   struct ll_readahead_state *ras)
{
	ras->ras_od_max_window = rai->ra_max_pages_per_file;
	ras->ras_od_start = 0;
	ras->ras_od_size = 0;
	ras->ras_od_async_size = 0;
	ras->ras_od_hit = 0;
	ras->ras_od_tw_start = 0;
	ras->ras_od_tw_end = 0;
}

/* use may change ra tunables, make sure the window settings are valid. */
static void od_check_window(struct ll_ra_info *rai,
			    struct ll_readahead_state *ras)
{
	/* rare event, simply reset everything */
	if (unlikely(ras->ras_od_max_window != rai->ra_max_pages_per_file ||
		     ras->ras_od_tw_end - ras->ras_od_tw_start >
				ras->ras_od_max_window)) {
		int level = ras->ras_od_max_window !=
					rai->ra_max_pages_per_file ?
				D_READA : D_WARNING;
		CDEBUG(level, "reset od state: max %lu / %lu, tw [%lu, %lu)\n",
		       ras->ras_od_max_window, rai->ra_max_pages_per_file,
		       ras->ras_od_tw_start, ras->ras_od_tw_end);
		od_clear_state(rai, ras);
		return;
	}

	if (unlikely(ras->ras_od_start < ras->ras_od_tw_start ||
		     ras->ras_od_start + ras->ras_od_size >
						ras->ras_od_tw_end ||
		     ras->ras_od_size < ras->ras_od_async_size ||
		     ras->ras_od_tw_start > ras->ras_od_tw_end)) {
		CWARN("invalid od readahead state, reset\n");
		RAS_OD_WARN(ras);
		od_clear_state(rai, ras);
	}
}

/*
 * Ondemand readahead.
 *
 * We use 2 windows scheme. The "od window" implemented the kernel ondemand
 * readahead algorithm with minor changes. There's also a "target window"
 * which encompass the od window, and is usually much larger. Due do
 * limitations of rpc, striping, locking etc., the intended ondemand ra
 * range often can't be satisfied, so the od window maybe truncated to
 * small range, and we try to move od window forward within target window
 * whenever we can. Keep the od window close to the higher end of the
 * target window as much as possible is important to have 1GB/s+ single
 * stream througput.
 */
static int ondemand_readahead(const struct lu_env *env, struct cl_io *io,
			      struct file *file, struct cl_page_list *queue,
			      struct ll_readahead_state *ras,
			      struct vvp_page *vpg)
{
	const unsigned long page_cache_max_scan = 256; /* 1 MiB */
	struct vvp_io *vio = vvp_env_io(env);
	struct ll_thread_info *lti = ll_env_info(env);
	struct ra_io_arg *ria = &lti->lti_ria;
	struct cl_attr *attr = vvp_env_thread_attr(env);
	struct cl_object *clob = io->ci_obj;
	struct inode *inode = vvp_object_inode(clob);
	struct ll_sb_info *sbi = ll_i2sbi(inode);
	struct ll_ra_info *rai = &sbi->ll_ra_info;
	pgoff_t index = vvp_index(vpg);
	unsigned long ra_min_pages;
	unsigned long uninitialized_var(od_mod);
	unsigned long req_size = 1;
	unsigned long len, mlen = 0;
	pgoff_t ra_end = 0;
	bool follow_window = true;
	__u64 kms;
	int ret = 0;
	ENTRY;

	/* req_size is counted after current page instead of original io size
	 * to be able to handle more complex cases */
	if (vio->vui_ra_valid) {
		req_size = vio->vui_ra_start + vio->vui_ra_count - index;
		if (unlikely(req_size == 0 || req_size > U32_MAX)) {
			/* vfs may want to read one more page after the
			 * intended range, it doesn't represent the user
			 * behaviour so skip readahead if we can */
			if (req_size == 0 && vpg->vpg_defer_uptodate) {
				CDEBUG(D_READA, "trailing page is uptodate\n");
				RETURN(0);
			}
			/* impossible but be paranoid */
			if (req_size != 0)
				CDEBUG(D_WARNING, "BUG: %lu + %lu < %lu\n",
					vio->vui_ra_start, vio->vui_ra_count,
					index);
			else
				CDEBUG(D_READA, "[%lu +%lu) read trailing "
					"page %lu\n",
					vio->vui_ra_start, vio->vui_ra_count,
					index);

			/* just pretend it's normal */
			req_size = 1;
		}
	}

	CDEBUG(D_READA, "page %lu (u %d, m %d), req_size %lu\n", index,
	       vpg->vpg_defer_uptodate, vpg->vpg_ra_od_marker, req_size);
	RAS_OD_TRACE(ras);

	cl_object_attr_lock(clob);
	ret = cl_object_attr_get(env, clob, attr);
	cl_object_attr_unlock(clob);
	if (ret != 0)
		RETURN(ret);

	kms = attr->cat_kms;
	if (unlikely(kms == 0)) {
		ll_ra_stats_inc_sbi(sbi, RA_STAT_ZERO_LEN);
		RETURN(0);
	}

	memset(ria, 0, sizeof *ria);
	/* take a snapshot */
	ra_min_pages = rai->ra_od_min_pages;

	spin_lock(&ras->ras_lock);

	/* validate window settings */
	od_check_window(rai, ras);
	if (unlikely(ras->ras_od_max_window < 2)) {
		spin_unlock(&ras->ras_lock);
		RETURN(0);
	}

	if (file && (file->f_mode & FMODE_RANDOM) &&
	    !rai->ra_od_no_vfs_hint) {
		/* but still respect min pages */
		req_size = max_t(unsigned long, req_size, ra_min_pages);

		ria->ria_start = index;
		ria->ria_end = index + req_size - 1;

		follow_window = false;
		ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_RANDOM_SKIP);
	} else if (index == 0) {
		/* beginning of a file, init/reset the window */
		od_mod = ++ras->ras_od_mod;
		ras->ras_od_hit = 1;
		ras->ras_od_start = 0;
		ras->ras_od_size = init_od_ra_size(req_size, ra_min_pages);
		ras->ras_od_async_size = ras->ras_od_size > req_size ?
			ras->ras_od_size - req_size : ras->ras_od_size;
		od_reset_target_window(ras);
	} else if (index == ras->ras_od_start + ras->ras_od_size ||
		   index == ras->ras_od_start + ras->ras_od_size -
				ras->ras_od_async_size ||
		   (vpg->vpg_ra_od_marker &&
		    od_page_in_target_window(ras, index))) {
		unsigned long prev_od_start = ras->ras_od_start;

		/* valid ra trigger offset, or hit marker in target window
		 * and od has already moved */
		CDEBUG(D_READA, "hit trigger\n");
		od_mod = ++ras->ras_od_mod;
		ras->ras_od_hit++;

		/* move window forward and make sure the current index
		 * stays in the window. note if the page is not uptodate,
		 * most likely it's at right edge of current od window,
		 * in rare cases it could be at marker position but original
		 * ra page has been pushed out from memory.
		 */
		if (index + ras->ras_od_max_window >
				ras->ras_od_start + ras->ras_od_size)
			ras->ras_od_start += ras->ras_od_size;

		/* restore to previous window size */
		if (od_target_window_has_gap(ras))
			ras->ras_od_size = ras->ras_od_tw_end - prev_od_start;

		ras->ras_od_size = next_od_ra_size(ras, ra_min_pages);
		ras->ras_od_async_size = ras->ras_od_size;
		od_expand_target_window(ras, index);

		ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_PUSH_WINDOW);
	} else if (vpg->vpg_ra_od_marker) {
		pgoff_t next_hole;

		/* hit marked page without valid ra state, recover window
		 * size from vvp_page */
		CDEBUG(D_READA, "hit marker\n");

		rcu_read_lock();
		next_hole = page_cache_next_hole(file->f_mapping, index + 1,
						 page_cache_max_scan);
		rcu_read_unlock();
		if (next_hole == 0 ||
		    next_hole - index > page_cache_max_scan){
			CDEBUG(D_READA, "next hole %lu is out of range \n",
				next_hole);
			spin_unlock(&ras->ras_lock);
			ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_MARKER_SKIP);
			RETURN(0);
		}

		od_mod = ++ras->ras_od_mod;
		ras->ras_od_hit = 2;
		ras->ras_od_tw_start = ras->ras_od_tw_end = 0;
		ras->ras_od_start = next_hole;
		ras->ras_od_size = next_hole - index;
		ras->ras_od_size = next_od_ra_size(ras, ra_min_pages);
		ras->ras_od_async_size = ras->ras_od_size;
		od_reset_target_window(ras);

		ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_MARKER_RECOVER);
	} else if (req_size > PTLRPC_MAX_BRW_PAGES) {
		/* oversize read, reset window */
		CDEBUG(D_READA, "over sized read, reset window\n");

		od_mod = ++ras->ras_od_mod;
		ras->ras_od_hit = 3;
		ras->ras_od_start = index;
		ras->ras_od_size = init_od_ra_size(req_size, ra_min_pages);
		ras->ras_od_async_size = ras->ras_od_size;
		od_reset_target_window(ras);

		ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_OVERSIZE);
	} else if (0) {
		/* FIXME
		 * it seems we can ignore sequential cache miss here
		 * since it'll be covered by following context search?
		 */
	} else if (vpg->vpg_defer_uptodate &&
		   od_page_in_target_window(ras, index) &&
		   od_target_window_has_gap(ras)) {
		/*
		 * try to close the gap in target window, keep
		 * hit number and target window unchanged.
		 */
		CDEBUG(D_READA, "pickup left over in target window\n");
		od_mod = ++ras->ras_od_mod;
		ras->ras_od_start += ras->ras_od_size;
		ras->ras_od_size = ras->ras_od_tw_end - ras->ras_od_start;
		ras->ras_od_async_size = ras->ras_od_size;

		ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_RESUME_OD);
	} else {
		/* try context readahead */
		pgoff_t head, size;

		rcu_read_lock();
		head = page_cache_prev_hole(file->f_mapping, index - 1,
					    page_cache_max_scan);
		rcu_read_unlock();
		size = index - 1 - head;

		if (size != 0) {
			od_mod = ++ras->ras_od_mod;
			ras->ras_od_hit = 2;

			/* starts from beginning of file, it's a strong
			 * indication of long-run stream or whole file
			 * read, so double the window. */
			if (size >= index) {
				size *= 2;
				ras->ras_od_hit++;
			}

			ras->ras_od_start = index;
			ras->ras_od_size = init_od_ra_size(size + req_size,
							   ra_min_pages);
			ras->ras_od_async_size = ras->ras_od_size;
			od_reset_target_window(ras);

			CDEBUG(D_READA, "recovered from context\n");
			ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_CTX_RECOVER);
		} else {
			/* no history, read as is without pollute rs state */
			ria->ria_start = index;
			ria->ria_end = index + req_size - 1;
			if (ria->ria_end - ria->ria_start + 1 < ra_min_pages) {
				ria->ria_marker = ria->ria_end + 1;
				ria->ria_end = ria->ria_start +
							ra_min_pages - 1;
			}

			CDEBUG(D_READA, "no history, read as is\n");
			ll_ra_stats_inc_sbi(sbi, RA_STAT_OD_NO_HISTORY);
			follow_window = false;
		}
	}

	if (follow_window) {
		/* if this read will hit the marker by itself, trigger it
		 * now and merge the next window into current one */
		if (index == ras->ras_od_start &&
		    ras->ras_od_size == ras->ras_od_async_size) {
			unsigned long async_size;

			async_size = next_od_ra_size(ras, ra_min_pages);
			if (ras->ras_od_size + async_size <
					ras->ras_od_max_window) {
				ras->ras_od_async_size = async_size;
				ras->ras_od_size += ras->ras_od_async_size;
			} else {
				ras->ras_od_async_size =
					ras->ras_od_max_window -
						ras->ras_od_size;
				ras->ras_od_size = ras->ras_od_max_window;
			}

			od_expand_target_window(ras, index);
			CDEBUG(D_READA, "merge next window\n");
		}

		/* fill args */
		ria->ria_start = ras->ras_od_start;
		ria->ria_end = ras->ras_od_start + ras->ras_od_size - 1;
		ria->ria_marker = ras->ras_od_start + ras->ras_od_size -
						ras->ras_od_async_size;
	}
	spin_unlock(&ras->ras_lock);

	/* truncate to kms */
	if (ria->ria_end >= (unsigned long)((kms - 1) >> PAGE_SHIFT)) {
		ria->ria_end = (unsigned long)((kms - 1) >> PAGE_SHIFT);
		ria->ria_eof = true;
		CDEBUG(D_READA, "truncate to kms %lu\n", ria->ria_end);
	}

	/* current page has been added already */
	if (ria->ria_start == index) {
		ria->ria_start++;
		if (ra_min_pages)
			ra_min_pages--;
	}

	CDEBUG(D_READA, "ra [%lu, %lu] (%lu), od [%lu +%lu) m %lu\n",
		ria->ria_start, ria->ria_end,
		ria->ria_end - ria->ria_start + 1,
		ras->ras_od_start, ras->ras_od_size,
		ras->ras_od_start + ras->ras_od_size - ras->ras_od_async_size);

	len = ria->ria_end >= ria->ria_start ?
		ria->ria_end - ria->ria_start + 1 : 0;
	if (len == 0) {
		ll_ra_stats_inc_sbi(sbi, RA_STAT_ZERO_WINDOW);
		RETURN(0);
	}

	/* get the minimal length to cover the read */
	if (!vpg->vpg_defer_uptodate && vio->vui_ra_valid &&
	    vio->vui_ra_start + vio->vui_ra_count > ria->ria_start) {
		mlen = min(vio->vui_ra_start + vio->vui_ra_count,
			   (pgoff_t) (attr->cat_size + PAGE_SIZE) >> PAGE_SHIFT);
		if (mlen > ria->ria_start)
			mlen -= ria->ria_start;
		else
			mlen = len;
	}
	if (mlen != 0 || ra_min_pages != 0) {
		unsigned long remainder;

		mlen = max_t(unsigned long, mlen, ra_min_pages);

		/* trim to RPC boundary */
		ras_align(ras, ria->ria_start, &remainder);
		mlen = min(mlen, ras->ras_rpc_size - remainder);
		ria->ria_end_min = ria->ria_start + mlen;
	}

	ria->ria_reserved = ll_ra_count_get(sbi, ria, len, mlen);
	CDEBUG(D_READA, "reserved pages: %lu/%lu/%lu, cur %d, max %lu\n",
	       ria->ria_reserved, len, mlen,
	       atomic_read(&rai->ra_cur_pages), rai->ra_max_pages);

	if (ria->ria_reserved < len) {
		ll_ra_stats_inc_sbi(sbi, RA_STAT_MAX_IN_FLIGHT);

		if (ria->ria_reserved == 0 && follow_window) {
			spin_lock(&ras->ras_lock);
			if (od_mod == ras->ras_od_mod) {
				ras->ras_od_size = 0;
				ras->ras_od_async_size = 0;
			}
			spin_unlock(&ras->ras_lock);
			RETURN(0);
		}

		ria->ria_end = ria->ria_start + ria->ria_reserved - 1;
	}

	if (follow_window) {
		/* the actual range might have changed, adjust od window. */
		spin_lock(&ras->ras_lock);
		if (od_mod == ras->ras_od_mod)
			od_shrink_window(ras, ria->ria_end);
		spin_unlock(&ras->ras_lock);
	}

	/* finally, make sure marker is sane after all the adjustment */
	ria->ria_marker = min(ria->ria_marker, ria->ria_end);

	ret = ll_read_ahead_pages(env, io, true, queue, ras, ria, &ra_end);
	CDEBUG(D_READA, "marker %lu, ra_end = %lu end = %lu pages = %d\n",
	       ria->ria_marker, ra_end, ria->ria_end, ret);
	if (ria->ria_reserved != 0) {
		if (follow_window) {
			/* to be simple, just clear window so next will
			 * try context search */
			spin_lock(&ras->ras_lock);
			if (od_mod == ras->ras_od_mod)
				od_shrink_window(ras, ra_end);
			spin_unlock(&ras->ras_lock);
		}
		ll_ra_count_put(sbi, ria->ria_reserved);
	}

	RAS_OD_TRACE(ras);
	if (ra_end == ria->ria_end && ra_end == (kms >> PAGE_SHIFT))
		ll_ra_stats_inc_sbi(sbi, RA_STAT_EOF);
	if (ra_end != ria->ria_end)
		ll_ra_stats_inc_sbi(sbi, RA_STAT_FAILED_REACH_END);
	RETURN(ret);
}

/*
 * Before going into expensive readahead algorithm, we try to return
 * as soon as possible if this is a page handled brought in by previous
 * readahead. This optimization is crutial for small sized read stream
 * workload.
 */
static inline
int ll_od_readahead(const struct lu_env *env, struct cl_io *io,
			struct file *file, struct cl_page_list *queue,
			struct ll_readahead_state *ras,
			struct vvp_page *vpg)
{
	if (vpg->vpg_defer_uptodate && !vpg->vpg_ra_od_marker) {
		pgoff_t index = vvp_index(vpg);
		bool push_window = false;

		spin_lock(&ras->ras_lock);
		/*
		 * potentially move window forward if:
		 *  1. at expected index w.r.t. od window;
		 *  2. in target window which still has gap;
		 */
		if ((ras->ras_od_start + ras->ras_od_size == index ||
		     ras->ras_od_start + ras->ras_od_size -
				ras->ras_od_async_size == index) ||
		    (od_page_in_target_window(ras, index) &&
		     od_target_window_has_gap(ras))) {
			push_window = true;
		}
		spin_unlock(&ras->ras_lock);

		if (!push_window)
			return 0;
	}

	return ondemand_readahead(env, io, file, queue, ras, vpg);
}

static int ll_readahead(const struct lu_env *env, struct cl_io *io,
			struct cl_page_list *queue,
			struct ll_readahead_state *ras, bool hit)
{
	struct vvp_io *vio = vvp_env_io(env);
	struct ll_thread_info *lti = ll_env_info(env);
	struct cl_attr *attr = vvp_env_thread_attr(env);
	unsigned long len, mlen = 0;
	pgoff_t ra_end = 0, start = 0, end = 0;
	struct inode *inode;
	struct ra_io_arg *ria = &lti->lti_ria;
	struct cl_object *clob;
	int ret = 0;
	__u64 kms;
	ENTRY;

	clob = io->ci_obj;
	inode = vvp_object_inode(clob);

	memset(ria, 0, sizeof *ria);

	cl_object_attr_lock(clob);
	ret = cl_object_attr_get(env, clob, attr);
	cl_object_attr_unlock(clob);

	if (ret != 0)
		RETURN(ret);
	kms = attr->cat_kms;
	if (kms == 0) {
		ll_ra_stats_inc(inode, RA_STAT_ZERO_LEN);
		RETURN(0);
	}

	spin_lock(&ras->ras_lock);

	/**
	 * Note: other thread might rollback the ras_next_readahead,
	 * if it can not get the full size of prepared pages, see the
	 * end of this function. For stride read ahead, it needs to
	 * make sure the offset is no less than ras_stride_offset,
	 * so that stride read ahead can work correctly.
	 */
	if (stride_io_mode(ras))
		start = max(ras->ras_next_readahead, ras->ras_stride_offset);
	else
		start = ras->ras_next_readahead;

	if (ras->ras_window_len > 0)
		end = ras->ras_window_start + ras->ras_window_len - 1;

	/* Enlarge the RA window to encompass the full read */
	if (vio->vui_ra_valid &&
	    end < vio->vui_ra_start + vio->vui_ra_count - 1)
		end = vio->vui_ra_start + vio->vui_ra_count - 1;

        if (end != 0) {
		unsigned long end_index;

		/* Truncate RA window to end of file */
		end_index = (unsigned long)((kms - 1) >> PAGE_SHIFT);
		if (end_index <= end) {
			end = end_index;
			ria->ria_eof = true;
		}
        }
        ria->ria_start = start;
        ria->ria_end = end;
        /* If stride I/O mode is detected, get stride window*/
        if (stride_io_mode(ras)) {
                ria->ria_stoff = ras->ras_stride_offset;
                ria->ria_length = ras->ras_stride_length;
                ria->ria_pages = ras->ras_stride_pages;
        }
	spin_unlock(&ras->ras_lock);

	if (end == 0) {
		ll_ra_stats_inc(inode, RA_STAT_ZERO_WINDOW);
		RETURN(0);
	}
	len = ria_page_count(ria);
	if (len == 0) {
		ll_ra_stats_inc(inode, RA_STAT_ZERO_WINDOW);
		RETURN(0);
	}

	RAS_CDEBUG(ras);
	CDEBUG(D_READA, DFID": ria: %lu/%lu, bead: %lu/%lu, hit: %d\n",
	       PFID(lu_object_fid(&clob->co_lu)),
	       ria->ria_start, ria->ria_end,
	       vio->vui_ra_valid ? vio->vui_ra_start : 0,
	       vio->vui_ra_valid ? vio->vui_ra_count : 0,
	       hit);

	/* at least to extend the readahead window to cover current read */
	if (!hit && vio->vui_ra_valid &&
	    vio->vui_ra_start + vio->vui_ra_count > ria->ria_start) {
		unsigned long remainder;

		/* to the end of current read window, but for small file
		 * the window usually far exceeds the actual file size,
		 * over read ahead leads to huge amount of pages were
		 * used as cache but for nothing.
		 *
		 * Since it's cached read and we already have lock held,
		 * the current file size is likely reflect the current
		 * actual value in most workload, so let's not exceed it
		 * in combine with kms which we know is valid.
		 *
		 * A side note: when file size is multiple of page size,
		 * vfs trys to read one more page at the end, we accomodate
		 * this behaviour to avoid an extra RPC.
		 */
		mlen = min(vio->vui_ra_start + vio->vui_ra_count,
			   (pgoff_t)(attr->cat_size + PAGE_SIZE) >> PAGE_SHIFT);
		if (mlen > ria->ria_start)
			mlen -= ria->ria_start;
		else
			mlen = len;

		/* trim to RPC boundary */
		ras_align(ras, ria->ria_start, &remainder);
		mlen = min(mlen, ras->ras_rpc_size - remainder);
		ria->ria_end_min = ria->ria_start + mlen;
	}

	ria->ria_reserved = ll_ra_count_get(ll_i2sbi(inode), ria, len, mlen);
	if (ria->ria_reserved < len)
		ll_ra_stats_inc(inode, RA_STAT_MAX_IN_FLIGHT);

	CDEBUG(D_READA, "reserved pages: %lu/%lu/%lu, ra_cur %d, ra_max %lu\n",
	       ria->ria_reserved, len, mlen,
	       atomic_read(&ll_i2sbi(inode)->ll_ra_info.ra_cur_pages),
	       ll_i2sbi(inode)->ll_ra_info.ra_max_pages);

	ret = ll_read_ahead_pages(env, io, false, queue, ras, ria, &ra_end);

	if (ria->ria_reserved != 0)
		ll_ra_count_put(ll_i2sbi(inode), ria->ria_reserved);

	if (ra_end == end && ra_end == (kms >> PAGE_SHIFT))
		ll_ra_stats_inc(inode, RA_STAT_EOF);

	CDEBUG(D_READA, "ra_end = %lu end = %lu stride end = %lu pages = %d\n",
	       ra_end, end, ria->ria_end, ret);

	if (ra_end != end)
		ll_ra_stats_inc(inode, RA_STAT_FAILED_REACH_END);
	if (ra_end > 0) {
		/* update the ras so that the next read-ahead tries from
		 * where we left off. */
		spin_lock(&ras->ras_lock);
		ras->ras_next_readahead = ra_end + 1;
		spin_unlock(&ras->ras_lock);
		RAS_CDEBUG(ras);
	}

	RETURN(ret);
}

static void ras_set_start(struct inode *inode, struct ll_readahead_state *ras,
			  unsigned long index)
{
	ras->ras_window_start = ras_align(ras, index, NULL);
}

/* called with the ras_lock held or from places where it doesn't matter */
static void ras_reset(struct inode *inode, struct ll_readahead_state *ras,
		      unsigned long index)
{
	ras->ras_last_readpage = index;
	ras->ras_consecutive_requests = 0;
	ras->ras_consecutive_pages = 0;
	ras->ras_window_len = 0;
	ras_set_start(inode, ras, index);
	ras->ras_next_readahead = max(ras->ras_window_start, index + 1);

	RAS_CDEBUG(ras);
}

/* called with the ras_lock held or from places where it doesn't matter */
static void ras_stride_reset(struct ll_readahead_state *ras)
{
        ras->ras_consecutive_stride_requests = 0;
        ras->ras_stride_length = 0;
        ras->ras_stride_pages = 0;
        RAS_CDEBUG(ras);
}

void ll_readahead_init(struct inode *inode, struct ll_readahead_state *ras)
{
	spin_lock_init(&ras->ras_lock);
	ras->ras_rpc_size = PTLRPC_MAX_BRW_PAGES;
	ras_reset(inode, ras, 0);
	ras->ras_requests = 0;

	od_clear_state(&ll_i2sbi(inode)->ll_ra_info, ras);
}

/*
 * Check whether the read request is in the stride window.
 * If it is in the stride window, return 1, otherwise return 0.
 */
static int index_in_stride_window(struct ll_readahead_state *ras,
				  unsigned long index)
{
	unsigned long stride_gap;

	if (ras->ras_stride_length == 0 || ras->ras_stride_pages == 0 ||
	    ras->ras_stride_pages == ras->ras_stride_length)
		return 0;

	stride_gap = index - ras->ras_last_readpage - 1;

	/* If it is contiguous read */
	if (stride_gap == 0)
		return ras->ras_consecutive_pages + 1 <= ras->ras_stride_pages;

	/* Otherwise check the stride by itself */
	return (ras->ras_stride_length - ras->ras_stride_pages) == stride_gap &&
		ras->ras_consecutive_pages == ras->ras_stride_pages;
}

static void ras_update_stride_detector(struct ll_readahead_state *ras,
                                       unsigned long index)
{
        unsigned long stride_gap = index - ras->ras_last_readpage - 1;

        if (!stride_io_mode(ras) && (stride_gap != 0 ||
             ras->ras_consecutive_stride_requests == 0)) {
                ras->ras_stride_pages = ras->ras_consecutive_pages;
                ras->ras_stride_length = stride_gap +ras->ras_consecutive_pages;
        }
        LASSERT(ras->ras_request_index == 0);
        LASSERT(ras->ras_consecutive_stride_requests == 0);

        if (index <= ras->ras_last_readpage) {
                /*Reset stride window for forward read*/
                ras_stride_reset(ras);
                return;
        }

        ras->ras_stride_pages = ras->ras_consecutive_pages;
        ras->ras_stride_length = stride_gap +ras->ras_consecutive_pages;

        RAS_CDEBUG(ras);
        return;
}

static unsigned long
stride_page_count(struct ll_readahead_state *ras, unsigned long len)
{
        return stride_pg_count(ras->ras_stride_offset, ras->ras_stride_length,
                               ras->ras_stride_pages, ras->ras_stride_offset,
                               len);
}

/* Stride Read-ahead window will be increased inc_len according to
 * stride I/O pattern */
static void ras_stride_increase_window(struct ll_readahead_state *ras,
                                       struct ll_ra_info *ra,
                                       unsigned long inc_len)
{
        unsigned long left, step, window_len;
        unsigned long stride_len;

        LASSERT(ras->ras_stride_length > 0);
        LASSERTF(ras->ras_window_start + ras->ras_window_len
                 >= ras->ras_stride_offset, "window_start %lu, window_len %lu"
                 " stride_offset %lu\n", ras->ras_window_start,
                 ras->ras_window_len, ras->ras_stride_offset);

        stride_len = ras->ras_window_start + ras->ras_window_len -
                     ras->ras_stride_offset;

        left = stride_len % ras->ras_stride_length;
        window_len = ras->ras_window_len - left;

        if (left < ras->ras_stride_pages)
                left += inc_len;
        else
                left = ras->ras_stride_pages + inc_len;

        LASSERT(ras->ras_stride_pages != 0);

        step = left / ras->ras_stride_pages;
        left %= ras->ras_stride_pages;

        window_len += step * ras->ras_stride_length + left;

        if (stride_page_count(ras, window_len) <= ra->ra_max_pages_per_file)
                ras->ras_window_len = window_len;

        RAS_CDEBUG(ras);
}

static void ras_increase_window(struct inode *inode,
				struct ll_readahead_state *ras,
				struct ll_ra_info *ra)
{
	/* The stretch of ra-window should be aligned with max rpc_size
	 * but current clio architecture does not support retrieve such
	 * information from lower layer. FIXME later
	 */
	if (stride_io_mode(ras)) {
		ras_stride_increase_window(ras, ra, ras->ras_rpc_size);
	} else {
		unsigned long wlen;

		wlen = min(ras->ras_window_len + ras->ras_rpc_size,
			   ra->ra_max_pages_per_file);
		ras->ras_window_len = ras_align(ras, wlen, NULL);
	}
}

static void ras_update(struct ll_sb_info *sbi, struct inode *inode,
		       struct ll_readahead_state *ras, unsigned long index,
		       enum ras_update_flags flags)
{
	struct ll_ra_info *ra = &sbi->ll_ra_info;
	bool hit = flags & LL_RAS_HIT;
	int zero = 0, stride_detect = 0, ra_miss = 0;
	ENTRY;

	if (!hit)
		CDEBUG(D_READA, DFID " pages at %lu miss.\n",
		       PFID(ll_inode2fid(inode)), index);
        ll_ra_stats_inc_sbi(sbi, hit ? RA_STAT_HIT : RA_STAT_MISS);

	/* skip the rest for ondemand algorithm */
	if (ra->ra_ondemand) {
		EXIT;
		return;
	}

	spin_lock(&ras->ras_lock);

        /* reset the read-ahead window in two cases.  First when the app seeks
         * or reads to some other part of the file.  Secondly if we get a
         * read-ahead miss that we think we've previously issued.  This can
         * be a symptom of there being so many read-ahead pages that the VM is
         * reclaiming it before we get to it. */
        if (!index_in_window(index, ras->ras_last_readpage, 8, 8)) {
                zero = 1;
                ll_ra_stats_inc_sbi(sbi, RA_STAT_DISTANT_READPAGE);
        } else if (!hit && ras->ras_window_len &&
                   index < ras->ras_next_readahead &&
                   index_in_window(index, ras->ras_window_start, 0,
                                   ras->ras_window_len)) {
                ra_miss = 1;
                ll_ra_stats_inc_sbi(sbi, RA_STAT_MISS_IN_WINDOW);
        }

        /* On the second access to a file smaller than the tunable
         * ra_max_read_ahead_whole_pages trigger RA on all pages in the
         * file up to ra_max_pages_per_file.  This is simply a best effort
         * and only occurs once per open file.  Normal RA behavior is reverted
         * to for subsequent IO.  The mmap case does not increment
         * ras_requests and thus can never trigger this behavior. */
	if (ras->ras_requests >= 2 && !ras->ras_request_index) {
		__u64 kms_pages;

		kms_pages = (i_size_read(inode) + PAGE_SIZE - 1) >>
			    PAGE_SHIFT;

		CDEBUG(D_READA, "kmsp %llu mwp %lu mp %lu\n", kms_pages,
                       ra->ra_max_read_ahead_whole_pages, ra->ra_max_pages_per_file);

                if (kms_pages &&
                    kms_pages <= ra->ra_max_read_ahead_whole_pages) {
                        ras->ras_window_start = 0;
			ras->ras_next_readahead = index + 1;
                        ras->ras_window_len = min(ra->ra_max_pages_per_file,
                                ra->ra_max_read_ahead_whole_pages);
                        GOTO(out_unlock, 0);
                }
        }
	if (zero) {
		/* check whether it is in stride I/O mode*/
		if (!index_in_stride_window(ras, index)) {
			if (ras->ras_consecutive_stride_requests == 0 &&
			    ras->ras_request_index == 0) {
				ras_update_stride_detector(ras, index);
				ras->ras_consecutive_stride_requests++;
			} else {
				ras_stride_reset(ras);
			}
			ras_reset(inode, ras, index);
			ras->ras_consecutive_pages++;
			GOTO(out_unlock, 0);
		} else {
			ras->ras_consecutive_pages = 0;
			ras->ras_consecutive_requests = 0;
			if (++ras->ras_consecutive_stride_requests > 1)
				stride_detect = 1;
			RAS_CDEBUG(ras);
		}
	} else {
		if (ra_miss) {
			if (index_in_stride_window(ras, index) &&
			    stride_io_mode(ras)) {
				if (index != ras->ras_last_readpage + 1)
					ras->ras_consecutive_pages = 0;
				ras_reset(inode, ras, index);

				/* If stride-RA hit cache miss, the stride
				 * detector will not be reset to avoid the
				 * overhead of redetecting read-ahead mode,
				 * but on the condition that the stride window
				 * is still intersect with normal sequential
				 * read-ahead window. */
				if (ras->ras_window_start <
				    ras->ras_stride_offset)
					ras_stride_reset(ras);
				RAS_CDEBUG(ras);
			} else {
				/* Reset both stride window and normal RA
				 * window */
				ras_reset(inode, ras, index);
				ras->ras_consecutive_pages++;
				ras_stride_reset(ras);
				GOTO(out_unlock, 0);
			}
		} else if (stride_io_mode(ras)) {
			/* If this is contiguous read but in stride I/O mode
			 * currently, check whether stride step still is valid,
			 * if invalid, it will reset the stride ra window*/
			if (!index_in_stride_window(ras, index)) {
				/* Shrink stride read-ahead window to be zero */
				ras_stride_reset(ras);
				ras->ras_window_len = 0;
				ras->ras_next_readahead = index;
			}
		}
	}
	ras->ras_consecutive_pages++;
	ras->ras_last_readpage = index;
	ras_set_start(inode, ras, index);

	if (stride_io_mode(ras)) {
		/* Since stride readahead is sentivite to the offset
		 * of read-ahead, so we use original offset here,
		 * instead of ras_window_start, which is RPC aligned */
		ras->ras_next_readahead = max(index + 1,
					      ras->ras_next_readahead);
		ras->ras_window_start = max(ras->ras_stride_offset,
					    ras->ras_window_start);
	} else {
		if (ras->ras_next_readahead < ras->ras_window_start)
			ras->ras_next_readahead = ras->ras_window_start;
		if (!hit)
			ras->ras_next_readahead = index + 1;
	}
	RAS_CDEBUG(ras);

	/* Trigger RA in the mmap case where ras_consecutive_requests
	 * is not incremented and thus can't be used to trigger RA */
	if (ras->ras_consecutive_pages >= 4 && flags & LL_RAS_MMAP) {
		ras_increase_window(inode, ras, ra);
		/* reset consecutive pages so that the readahead window can
		 * grow gradually. */
		ras->ras_consecutive_pages = 0;
		GOTO(out_unlock, 0);
	}

	/* Initially reset the stride window offset to next_readahead*/
	if (ras->ras_consecutive_stride_requests == 2 && stride_detect) {
		/**
		 * Once stride IO mode is detected, next_readahead should be
		 * reset to make sure next_readahead > stride offset
		 */
		ras->ras_next_readahead = max(index, ras->ras_next_readahead);
		ras->ras_stride_offset = index;
		ras->ras_window_start = max(index, ras->ras_window_start);
	}

	/* The initial ras_window_len is set to the request size.  To avoid
	 * uselessly reading and discarding pages for random IO the window is
	 * only increased once per consecutive request received. */
	if ((ras->ras_consecutive_requests > 1 || stride_detect) &&
	    !ras->ras_request_index)
		ras_increase_window(inode, ras, ra);
	EXIT;
out_unlock:
	RAS_CDEBUG(ras);
	ras->ras_request_index++;
	spin_unlock(&ras->ras_lock);
	return;
}

int ll_writepage(struct page *vmpage, struct writeback_control *wbc)
{
	struct inode	       *inode = vmpage->mapping->host;
	struct ll_inode_info   *lli   = ll_i2info(inode);
        struct lu_env          *env;
        struct cl_io           *io;
        struct cl_page         *page;
        struct cl_object       *clob;
	bool redirtied = false;
	bool unlocked = false;
        int result;
	__u16 refcheck;
        ENTRY;

        LASSERT(PageLocked(vmpage));
        LASSERT(!PageWriteback(vmpage));

	LASSERT(ll_i2dtexp(inode) != NULL);

	env = cl_env_get(&refcheck);
	if (IS_ERR(env))
		GOTO(out, result = PTR_ERR(env));

        clob  = ll_i2info(inode)->lli_clob;
        LASSERT(clob != NULL);

	io = vvp_env_thread_io(env);
        io->ci_obj = clob;
	io->ci_ignore_layout = 1;
        result = cl_io_init(env, io, CIT_MISC, clob);
        if (result == 0) {
                page = cl_page_find(env, clob, vmpage->index,
                                    vmpage, CPT_CACHEABLE);
		if (!IS_ERR(page)) {
			lu_ref_add(&page->cp_reference, "writepage",
				   current);
			cl_page_assume(env, io, page);
			result = cl_page_flush(env, io, page);
			if (result != 0) {
				/*
				 * Re-dirty page on error so it retries write,
				 * but not in case when IO has actually
				 * occurred and completed with an error.
				 */
				if (!PageError(vmpage)) {
					redirty_page_for_writepage(wbc, vmpage);
					result = 0;
					redirtied = true;
				}
			}
			cl_page_disown(env, io, page);
			unlocked = true;
			lu_ref_del(&page->cp_reference,
				   "writepage", current);
			cl_page_put(env, page);
		} else {
			result = PTR_ERR(page);
		}
        }
        cl_io_fini(env, io);

	if (redirtied && wbc->sync_mode == WB_SYNC_ALL) {
		loff_t offset = cl_offset(clob, vmpage->index);

		/* Flush page failed because the extent is being written out.
		 * Wait for the write of extent to be finished to avoid
		 * breaking kernel which assumes ->writepage should mark
		 * PageWriteback or clean the page. */
		result = cl_sync_file_range(inode, offset,
					    offset + PAGE_SIZE - 1,
					    CL_FSYNC_LOCAL, 1);
		if (result > 0) {
			/* actually we may have written more than one page.
			 * decreasing this page because the caller will count
			 * it. */
			wbc->nr_to_write -= result - 1;
			result = 0;
		}
	}

	cl_env_put(env, &refcheck);
	GOTO(out, result);

out:
	if (result < 0) {
		if (!lli->lli_async_rc)
			lli->lli_async_rc = result;
		SetPageError(vmpage);
		if (!unlocked)
			unlock_page(vmpage);
	}
	return result;
}

int ll_writepages(struct address_space *mapping, struct writeback_control *wbc)
{
	struct inode *inode = mapping->host;
	loff_t start;
	loff_t end;
	enum cl_fsync_mode mode;
	int range_whole = 0;
	int result;
	ENTRY;

	if (wbc->range_cyclic) {
		start = mapping->writeback_index << PAGE_SHIFT;
		end = OBD_OBJECT_EOF;
	} else {
		start = wbc->range_start;
		end = wbc->range_end;
		if (end == LLONG_MAX) {
			end = OBD_OBJECT_EOF;
			range_whole = start == 0;
		}
	}

	mode = CL_FSYNC_NONE;
	if (wbc->sync_mode == WB_SYNC_ALL)
		mode = CL_FSYNC_LOCAL;

	if (ll_i2info(inode)->lli_clob == NULL)
		RETURN(0);

	/* for directio, it would call writepages() to evict cached pages
	 * inside the IO context of write, which will cause deadlock at
	 * layout_conf since it waits for active IOs to complete. */
	result = cl_sync_file_range(inode, start, end, mode, 1);
	if (result > 0) {
		wbc->nr_to_write -= result;
		result = 0;
	 }

	if (wbc->range_cyclic || (range_whole && wbc->nr_to_write > 0)) {
		if (end == OBD_OBJECT_EOF)
			mapping->writeback_index = 0;
		else
			mapping->writeback_index = (end >> PAGE_SHIFT) + 1;
	}
	RETURN(result);
}

struct ll_cl_context *ll_cl_find(struct file *file)
{
	struct ll_file_data *fd = LUSTRE_FPRIVATE(file);
	struct ll_cl_context *lcc;
	struct ll_cl_context *found = NULL;

	read_lock(&fd->fd_lock);
	list_for_each_entry(lcc, &fd->fd_lccs, lcc_list) {
		if (lcc->lcc_cookie == current) {
			found = lcc;
			break;
		}
	}
	read_unlock(&fd->fd_lock);

	return found;
}

void ll_cl_add(struct file *file, const struct lu_env *env, struct cl_io *io,
	       enum lcc_type type)
{
	struct ll_file_data *fd = LUSTRE_FPRIVATE(file);
	struct ll_cl_context *lcc = &ll_env_info(env)->lti_io_ctx;

	memset(lcc, 0, sizeof(*lcc));
	INIT_LIST_HEAD(&lcc->lcc_list);
	lcc->lcc_cookie = current;
	lcc->lcc_env = env;
	lcc->lcc_io = io;
	lcc->lcc_type = type;

	write_lock(&fd->fd_lock);
	list_add(&lcc->lcc_list, &fd->fd_lccs);
	write_unlock(&fd->fd_lock);
}

void ll_cl_remove(struct file *file, const struct lu_env *env)
{
	struct ll_file_data *fd = LUSTRE_FPRIVATE(file);
	struct ll_cl_context *lcc = &ll_env_info(env)->lti_io_ctx;

	write_lock(&fd->fd_lock);
	list_del_init(&lcc->lcc_list);
	write_unlock(&fd->fd_lock);
}

static int ll_io_read_page(const struct lu_env *env, struct cl_io *io,
			   struct cl_page *page, struct file *file)
{
	struct inode              *inode  = vvp_object_inode(page->cp_obj);
	struct ll_sb_info         *sbi    = ll_i2sbi(inode);
	struct ll_file_data       *fd     = LUSTRE_FPRIVATE(file);
	struct ll_readahead_state *ras    = &fd->fd_ras;
	struct cl_2queue          *queue  = &io->ci_queue;
	struct vvp_page           *vpg;
	int			   rc = 0;
	bool			   uptodate;
	ENTRY;

	vpg = cl2vvp_page(cl_object_page_slice(page->cp_obj, page));
	uptodate = vpg->vpg_defer_uptodate;
	CDEBUG(D_READA, "page %lu, marker %d, defer_uptodate %d\n",
		vvp_index(vpg), vpg->vpg_ra_od_marker, uptodate);

	if (sbi->ll_ra_info.ra_max_pages_per_file > 0 &&
	    sbi->ll_ra_info.ra_max_pages > 0 &&
	    !vpg->vpg_ra_updated) {
		struct vvp_io *vio = vvp_env_io(env);
		enum ras_update_flags flags = 0;

		if (uptodate)
			flags |= LL_RAS_HIT;
		if (!vio->vui_ra_valid)
			flags |= LL_RAS_MMAP;
		ras_update(sbi, inode, ras, vvp_index(vpg), flags);
	}

	cl_2queue_init(queue);
	if (uptodate) {
		vpg->vpg_ra_used = 1;
		cl_page_export(env, page, 1);
		cl_page_disown(env, io, page);
	} else {
		cl_2queue_add(queue, page);
	}

	if (sbi->ll_ra_info.ra_max_pages_per_file > 0 &&
	    sbi->ll_ra_info.ra_max_pages > 0) {
		int rc2;

		if (sbi->ll_ra_info.ra_ondemand)
			rc2 = ll_od_readahead(env, io, file, &queue->c2_qin,
					      ras, vpg);
		else
			rc2 = ll_readahead(env, io, &queue->c2_qin, ras,
					   uptodate);
		CDEBUG(D_READA, DFID " %d pages read ahead at %lu\n",
		       PFID(ll_inode2fid(inode)), rc2, vvp_index(vpg));
	}

	if (queue->c2_qin.pl_nr > 0) {
		int count = queue->c2_qin.pl_nr;
		rc = cl_io_submit_rw(env, io, CRT_READ, queue);
		if (rc == 0)
			task_io_account_read(PAGE_SIZE * count);
	}

	/*
	 * Unlock unsent pages in case of error.
	 */
	cl_page_list_disown(env, io, &queue->c2_qin);
	cl_2queue_fini(env, queue);

	RETURN(rc);
}

static int ll_fast_readpage(struct ll_cl_context *lcc,
			    struct file *file, struct inode *inode,
			    struct cl_object *clob, struct page *vmpage)
{
	const struct lu_env *env;
	struct ll_sb_info *sbi;
	struct ll_file_data *fd;
	struct ll_readahead_state *ras;
	struct cl_page *page;
	struct vvp_page *vpg;
	int result = -ENODATA;

	/* TODO: need to verify the layout version to make sure
	 * the page is not invalid due to layout change. */
	page = cl_vmpage_page(vmpage, clob);
	if (page == NULL) {
		unlock_page(vmpage);
		return result;
	}

	env = lcc->lcc_env;
	sbi = ll_i2sbi(inode);
	fd = LUSTRE_FPRIVATE(file);
	ras = &fd->fd_ras;

	vpg = cl2vvp_page(cl_object_page_slice(page->cp_obj, page));
	if (vpg->vpg_defer_uptodate) {
		enum ras_update_flags flags = LL_RAS_HIT;
		bool export = false;

		if (lcc->lcc_type == LCC_MMAP)
			flags |= LL_RAS_MMAP;

		/* For fast read, it updates read ahead state only
		 * if the page is hit in cache because non cache page
		 * case will be handled by slow read later. */
		ras_update(sbi, inode, ras, vvp_index(vpg), flags);
		/* avoid duplicate ras_update() call */
		vpg->vpg_ra_updated = 1;

		/* Check if we can issue a readahead RPC, if that is
		 * the case, we can't do fast IO because we will need
		 * a cl_io to issue the RPC. */
		if (sbi->ll_ra_info.ra_ondemand) {
			/*
			 * checking into ras withou locking is slightly
			 * racy but no serious harm.
			 */
			if (!vpg->vpg_ra_od_marker &&
			    (!od_page_in_target_window(ras, vvp_index(vpg)) ||
			     od_target_window_is_full(ras)))
				export = true;
		} else {
			if (ras->ras_window_start + ras->ras_window_len <
			    ras->ras_next_readahead + PTLRPC_MAX_BRW_PAGES)
				export = true;
		}

		if (export) {
			/* export the page and skip io stack */
			vpg->vpg_ra_used = 1;
			cl_page_export(env, page, 1);
			result = 0;
		}
	}

	unlock_page(vmpage);
	cl_page_put(env, page);
	return result;
}

int ll_readpage(struct file *file, struct page *vmpage)
{
	struct inode *inode = file_inode(file);
	struct cl_object *clob = ll_i2info(inode)->lli_clob;
	struct ll_cl_context *lcc;
	const struct lu_env  *env;
	struct cl_io   *io;
	struct cl_page *page;
	int result;
	ENTRY;

	lcc = ll_cl_find(file);
	if (lcc == NULL) {
		unlock_page(vmpage);
		RETURN(-EIO);
	}

	env = lcc->lcc_env;
	io  = lcc->lcc_io;
	if (io == NULL)
		RETURN(ll_fast_readpage(lcc, file, inode, clob, vmpage));

	LASSERT(io->ci_state == CIS_IO_GOING);
	page = cl_page_find(env, clob, vmpage->index, vmpage, CPT_CACHEABLE);
	if (!IS_ERR(page)) {
		LASSERT(page->cp_type == CPT_CACHEABLE);
		if (likely(!PageUptodate(vmpage))) {
			cl_page_assume(env, io, page);
			result = ll_io_read_page(env, io, page, file);
		} else {
			/* Page from a non-object file. */
			unlock_page(vmpage);
			result = 0;
		}
		cl_page_put(env, page);
	} else {
		unlock_page(vmpage);
		result = PTR_ERR(page);
        }
	RETURN(result);
}

int ll_page_sync_io(const struct lu_env *env, struct cl_io *io,
		    struct cl_page *page, enum cl_req_type crt)
{
	struct cl_2queue  *queue;
	int result;

	LASSERT(io->ci_type == CIT_READ || io->ci_type == CIT_WRITE);

	queue = &io->ci_queue;
	cl_2queue_init_page(queue, page);

	result = cl_io_submit_sync(env, io, crt, queue, 0);
	LASSERT(cl_page_is_owned(page, io));

	if (crt == CRT_READ)
		/*
		 * in CRT_WRITE case page is left locked even in case of
		 * error.
		 */
		cl_page_list_disown(env, io, &queue->c2_qin);
	cl_2queue_fini(env, queue);

	return result;
}
