/**
 * (C) Copyright IBM Corp. 2020, 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * IBM OpenAPI SDK Code Generator Version: 3.28.0-55613c9e-20210220-164656
 */

// Package vulnerabilityadvisorv3 : Operations and models for the VulnerabilityAdvisorV3 service
package vulnerabilityadvisorv3

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"reflect"
	"time"

	common "github.com/IBM/container-registry-go-sdk/common"
	"github.com/IBM/go-sdk-core/v5/core"
)

// VulnerabilityAdvisorV3 : Management interface of Vulnerability Advisor for IBM Cloud Container Registry
//
// Version: 3.0.0
type VulnerabilityAdvisorV3 struct {
	Service *core.BaseService

	// The unique ID for your IBM Cloud account. Run 'ibmcloud cr info' to get the ID of the target account.
	Account *string

	// The preferred language code for this request.
	AcceptLanguage *string
}

// DefaultServiceURL is the default URL to make service requests to.
const DefaultServiceURL = "https://us.icr.io"

// DefaultServiceName is the default key used to find external configuration information.
const DefaultServiceName = "vulnerability_advisor"

// VulnerabilityAdvisorV3Options : Service options
type VulnerabilityAdvisorV3Options struct {
	ServiceName   string
	URL           string
	Authenticator core.Authenticator

	// The unique ID for your IBM Cloud account. Run 'ibmcloud cr info' to get the ID of the target account.
	Account *string `validate:"required"`

	// The preferred language code for this request.
	AcceptLanguage *string
}

// NewVulnerabilityAdvisorV3UsingExternalConfig : constructs an instance of VulnerabilityAdvisorV3 with passed in options and external configuration.
func NewVulnerabilityAdvisorV3UsingExternalConfig(options *VulnerabilityAdvisorV3Options) (vulnerabilityAdvisor *VulnerabilityAdvisorV3, err error) {
	if options.ServiceName == "" {
		options.ServiceName = DefaultServiceName
	}

	if options.Authenticator == nil {
		options.Authenticator, err = core.GetAuthenticatorFromEnvironment(options.ServiceName)
		if err != nil {
			return
		}
	}

	vulnerabilityAdvisor, err = NewVulnerabilityAdvisorV3(options)
	if err != nil {
		return
	}

	err = vulnerabilityAdvisor.Service.ConfigureService(options.ServiceName)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = vulnerabilityAdvisor.Service.SetServiceURL(options.URL)
	}
	return
}

// NewVulnerabilityAdvisorV3 : constructs an instance of VulnerabilityAdvisorV3 with passed in options.
func NewVulnerabilityAdvisorV3(options *VulnerabilityAdvisorV3Options) (service *VulnerabilityAdvisorV3, err error) {
	serviceOptions := &core.ServiceOptions{
		URL:           DefaultServiceURL,
		Authenticator: options.Authenticator,
	}

	err = core.ValidateStruct(options, "options")
	if err != nil {
		return
	}

	baseService, err := core.NewBaseService(serviceOptions)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = baseService.SetServiceURL(options.URL)
		if err != nil {
			return
		}
	}

	service = &VulnerabilityAdvisorV3{
		Service:        baseService,
		Account:        options.Account,
		AcceptLanguage: options.AcceptLanguage,
	}

	return
}

// GetServiceURLForRegion returns the service URL to be used for the specified region
func GetServiceURLForRegion(region string) (string, error) {
	return "", fmt.Errorf("service does not support regional URLs")
}

// Clone makes a copy of "vulnerabilityAdvisor" suitable for processing requests.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) Clone() *VulnerabilityAdvisorV3 {
	if core.IsNil(vulnerabilityAdvisor) {
		return nil
	}
	clone := *vulnerabilityAdvisor
	clone.Service = vulnerabilityAdvisor.Service.Clone()
	return &clone
}

// SetServiceURL sets the service URL
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) SetServiceURL(url string) error {
	return vulnerabilityAdvisor.Service.SetServiceURL(url)
}

// GetServiceURL returns the service URL
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) GetServiceURL() string {
	return vulnerabilityAdvisor.Service.GetServiceURL()
}

// SetDefaultHeaders sets HTTP headers to be sent in every request
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) SetDefaultHeaders(headers http.Header) {
	vulnerabilityAdvisor.Service.SetDefaultHeaders(headers)
}

// SetEnableGzipCompression sets the service's EnableGzipCompression field
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) SetEnableGzipCompression(enableGzip bool) {
	vulnerabilityAdvisor.Service.SetEnableGzipCompression(enableGzip)
}

// GetEnableGzipCompression returns the service's EnableGzipCompression field
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) GetEnableGzipCompression() bool {
	return vulnerabilityAdvisor.Service.GetEnableGzipCompression()
}

// EnableRetries enables automatic retries for requests invoked for this service instance.
// If either parameter is specified as 0, then a default value is used instead.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) EnableRetries(maxRetries int, maxRetryInterval time.Duration) {
	vulnerabilityAdvisor.Service.EnableRetries(maxRetries, maxRetryInterval)
}

// DisableRetries disables automatic retries for requests invoked for this service instance.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) DisableRetries() {
	vulnerabilityAdvisor.Service.DisableRetries()
}

// AccountReportQueryPath : Get the vulnerability assessment for all images
// Get the vulnerability assessment for the list of registry images that belong to a specific account.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) AccountReportQueryPath(accountReportQueryPathOptions *AccountReportQueryPathOptions) (result *ScanReportList, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.AccountReportQueryPathWithContext(context.Background(), accountReportQueryPathOptions)
}

// AccountReportQueryPathWithContext is an alternate form of the AccountReportQueryPath method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) AccountReportQueryPathWithContext(ctx context.Context, accountReportQueryPathOptions *AccountReportQueryPathOptions) (result *ScanReportList, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(accountReportQueryPathOptions, "accountReportQueryPathOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/report/account`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range accountReportQueryPathOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "AccountReportQueryPath")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	if accountReportQueryPathOptions.Repository != nil {
		builder.AddQuery("repository", fmt.Sprint(*accountReportQueryPathOptions.Repository))
	}
	if accountReportQueryPathOptions.IncludeIBM != nil {
		builder.AddQuery("includeIBM", fmt.Sprint(*accountReportQueryPathOptions.IncludeIBM))
	}
	if accountReportQueryPathOptions.IncludePrivate != nil {
		builder.AddQuery("includePrivate", fmt.Sprint(*accountReportQueryPathOptions.IncludePrivate))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalScanReportList)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// AccountStatusQueryPath : Get vulnerability assessment status for all images
// Get the vulnerability assessment status for the list of registry images that belong to a specific account.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) AccountStatusQueryPath(accountStatusQueryPathOptions *AccountStatusQueryPathOptions) (result *ScanreportImageSummaryList, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.AccountStatusQueryPathWithContext(context.Background(), accountStatusQueryPathOptions)
}

// AccountStatusQueryPathWithContext is an alternate form of the AccountStatusQueryPath method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) AccountStatusQueryPathWithContext(ctx context.Context, accountStatusQueryPathOptions *AccountStatusQueryPathOptions) (result *ScanreportImageSummaryList, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(accountStatusQueryPathOptions, "accountStatusQueryPathOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/report/account/status`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range accountStatusQueryPathOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "AccountStatusQueryPath")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	if accountStatusQueryPathOptions.Repository != nil {
		builder.AddQuery("repository", fmt.Sprint(*accountStatusQueryPathOptions.Repository))
	}
	if accountStatusQueryPathOptions.IncludeIBM != nil {
		builder.AddQuery("includeIBM", fmt.Sprint(*accountStatusQueryPathOptions.IncludeIBM))
	}
	if accountStatusQueryPathOptions.IncludePrivate != nil {
		builder.AddQuery("includePrivate", fmt.Sprint(*accountStatusQueryPathOptions.IncludePrivate))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalScanreportImageSummaryList)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ImageReportQueryPath : Get vulnerability assessment
// Get the vulnerability assessment for a registry image.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ImageReportQueryPath(imageReportQueryPathOptions *ImageReportQueryPathOptions) (result *ScanReport, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ImageReportQueryPathWithContext(context.Background(), imageReportQueryPathOptions)
}

// ImageReportQueryPathWithContext is an alternate form of the ImageReportQueryPath method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ImageReportQueryPathWithContext(ctx context.Context, imageReportQueryPathOptions *ImageReportQueryPathOptions) (result *ScanReport, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(imageReportQueryPathOptions, "imageReportQueryPathOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(imageReportQueryPathOptions, "imageReportQueryPathOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *imageReportQueryPathOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/report/image/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range imageReportQueryPathOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ImageReportQueryPath")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalScanReport)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ImageStatusQueryPath : Get vulnerability status
// Get the overall vulnerability status for a registry image.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ImageStatusQueryPath(imageStatusQueryPathOptions *ImageStatusQueryPathOptions) (result *ScanreportSummary, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ImageStatusQueryPathWithContext(context.Background(), imageStatusQueryPathOptions)
}

// ImageStatusQueryPathWithContext is an alternate form of the ImageStatusQueryPath method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ImageStatusQueryPathWithContext(ctx context.Context, imageStatusQueryPathOptions *ImageStatusQueryPathOptions) (result *ScanreportSummary, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(imageStatusQueryPathOptions, "imageStatusQueryPathOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(imageStatusQueryPathOptions, "imageStatusQueryPathOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *imageStatusQueryPathOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/report/image/status/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range imageStatusQueryPathOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ImageStatusQueryPath")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalScanreportSummary)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListExemptionAccount : List account level exemptions
// List the exemptions that are specified with account level scope.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListExemptionAccount(listExemptionAccountOptions *ListExemptionAccountOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ListExemptionAccountWithContext(context.Background(), listExemptionAccountOptions)
}

// ListExemptionAccountWithContext is an alternate form of the ListExemptionAccount method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListExemptionAccountWithContext(ctx context.Context, listExemptionAccountOptions *ListExemptionAccountOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listExemptionAccountOptions, "listExemptionAccountOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listExemptionAccountOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ListExemptionAccount")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse []json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetExemptionAccount : Get an account level exemption
// Get details of an exemption that is specified with account level scope.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) GetExemptionAccount(getExemptionAccountOptions *GetExemptionAccountOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.GetExemptionAccountWithContext(context.Background(), getExemptionAccountOptions)
}

// GetExemptionAccountWithContext is an alternate form of the GetExemptionAccount method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) GetExemptionAccountWithContext(ctx context.Context, getExemptionAccountOptions *GetExemptionAccountOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getExemptionAccountOptions, "getExemptionAccountOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getExemptionAccountOptions, "getExemptionAccountOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"issueType": *getExemptionAccountOptions.IssueType,
		"issueID":   *getExemptionAccountOptions.IssueID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/issue/{issueType}/{issueID}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getExemptionAccountOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "GetExemptionAccount")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateExemptionAccount : Create or update an account level exemption
// Create or update an exemption that is specified with account level scope.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) CreateExemptionAccount(createExemptionAccountOptions *CreateExemptionAccountOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.CreateExemptionAccountWithContext(context.Background(), createExemptionAccountOptions)
}

// CreateExemptionAccountWithContext is an alternate form of the CreateExemptionAccount method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) CreateExemptionAccountWithContext(ctx context.Context, createExemptionAccountOptions *CreateExemptionAccountOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createExemptionAccountOptions, "createExemptionAccountOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createExemptionAccountOptions, "createExemptionAccountOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"issueType": *createExemptionAccountOptions.IssueType,
		"issueID":   *createExemptionAccountOptions.IssueID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/issue/{issueType}/{issueID}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createExemptionAccountOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "CreateExemptionAccount")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteExemptionAccount : Delete an account level exemption
// Delete an exemption that is specified with account level scope.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) DeleteExemptionAccount(deleteExemptionAccountOptions *DeleteExemptionAccountOptions) (response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.DeleteExemptionAccountWithContext(context.Background(), deleteExemptionAccountOptions)
}

// DeleteExemptionAccountWithContext is an alternate form of the DeleteExemptionAccount method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) DeleteExemptionAccountWithContext(ctx context.Context, deleteExemptionAccountOptions *DeleteExemptionAccountOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteExemptionAccountOptions, "deleteExemptionAccountOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteExemptionAccountOptions, "deleteExemptionAccountOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"issueType": *deleteExemptionAccountOptions.IssueType,
		"issueID":   *deleteExemptionAccountOptions.IssueID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/issue/{issueType}/{issueID}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteExemptionAccountOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "DeleteExemptionAccount")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vulnerabilityAdvisor.Service.Request(request, nil)

	return
}

// ListExemptionResource : List resource exemptions
// List the exemptions that are specified for a resource (account, registry namespace, repository, or image).
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListExemptionResource(listExemptionResourceOptions *ListExemptionResourceOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ListExemptionResourceWithContext(context.Background(), listExemptionResourceOptions)
}

// ListExemptionResourceWithContext is an alternate form of the ListExemptionResource method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListExemptionResourceWithContext(ctx context.Context, listExemptionResourceOptions *ListExemptionResourceOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listExemptionResourceOptions, "listExemptionResourceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listExemptionResourceOptions, "listExemptionResourceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"resource": *listExemptionResourceOptions.Resource,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/{resource}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listExemptionResourceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ListExemptionResource")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse []json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetExemptionResource : Get details of a resource exemption
// Get an exemption that is specified for a resource (account, registry namespace, repository, or image).
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) GetExemptionResource(getExemptionResourceOptions *GetExemptionResourceOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.GetExemptionResourceWithContext(context.Background(), getExemptionResourceOptions)
}

// GetExemptionResourceWithContext is an alternate form of the GetExemptionResource method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) GetExemptionResourceWithContext(ctx context.Context, getExemptionResourceOptions *GetExemptionResourceOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getExemptionResourceOptions, "getExemptionResourceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getExemptionResourceOptions, "getExemptionResourceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"resource":  *getExemptionResourceOptions.Resource,
		"issueType": *getExemptionResourceOptions.IssueType,
		"issueID":   *getExemptionResourceOptions.IssueID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/{resource}/issue/{issueType}/{issueID}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getExemptionResourceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "GetExemptionResource")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateExemptionResource : Create or update a resource exemption
// Create or update an exemption that is specified for a resource (account, registry namespace, repository, or image).
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) CreateExemptionResource(createExemptionResourceOptions *CreateExemptionResourceOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.CreateExemptionResourceWithContext(context.Background(), createExemptionResourceOptions)
}

// CreateExemptionResourceWithContext is an alternate form of the CreateExemptionResource method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) CreateExemptionResourceWithContext(ctx context.Context, createExemptionResourceOptions *CreateExemptionResourceOptions) (result *Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createExemptionResourceOptions, "createExemptionResourceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createExemptionResourceOptions, "createExemptionResourceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"resource":  *createExemptionResourceOptions.Resource,
		"issueType": *createExemptionResourceOptions.IssueType,
		"issueID":   *createExemptionResourceOptions.IssueID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/{resource}/issue/{issueType}/{issueID}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createExemptionResourceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "CreateExemptionResource")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteExemptionResource : Delete a resource exemption
// Delete an exemption that is specified for a resource (account, registry namespace, repository, or image).
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) DeleteExemptionResource(deleteExemptionResourceOptions *DeleteExemptionResourceOptions) (response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.DeleteExemptionResourceWithContext(context.Background(), deleteExemptionResourceOptions)
}

// DeleteExemptionResourceWithContext is an alternate form of the DeleteExemptionResource method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) DeleteExemptionResourceWithContext(ctx context.Context, deleteExemptionResourceOptions *DeleteExemptionResourceOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteExemptionResourceOptions, "deleteExemptionResourceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteExemptionResourceOptions, "deleteExemptionResourceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"resource":  *deleteExemptionResourceOptions.Resource,
		"issueType": *deleteExemptionResourceOptions.IssueType,
		"issueID":   *deleteExemptionResourceOptions.IssueID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/image/{resource}/issue/{issueType}/{issueID}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteExemptionResourceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "DeleteExemptionResource")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vulnerabilityAdvisor.Service.Request(request, nil)

	return
}

// ExemptHandler : List the types of exemption
// List the types of exemption.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ExemptHandler(exemptHandlerOptions *ExemptHandlerOptions) (result []ExemptionTypeInfo, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ExemptHandlerWithContext(context.Background(), exemptHandlerOptions)
}

// ExemptHandlerWithContext is an alternate form of the ExemptHandler method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ExemptHandlerWithContext(ctx context.Context, exemptHandlerOptions *ExemptHandlerOptions) (result []ExemptionTypeInfo, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(exemptHandlerOptions, "exemptHandlerOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exempt/types`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range exemptHandlerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ExemptHandler")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse []json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemptionTypeInfo)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListAccountExemptions : List all exemptions
// List all of the exemptions in the given account.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListAccountExemptions(listAccountExemptionsOptions *ListAccountExemptionsOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ListAccountExemptionsWithContext(context.Background(), listAccountExemptionsOptions)
}

// ListAccountExemptionsWithContext is an alternate form of the ListAccountExemptions method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListAccountExemptionsWithContext(ctx context.Context, listAccountExemptionsOptions *ListAccountExemptionsOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listAccountExemptionsOptions, "listAccountExemptionsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exemptions/account`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listAccountExemptionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ListAccountExemptions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse []json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ExemptionsAccountDeleteHandler : Delete all exemptions
// Delete all of the exemptions in the given account.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptions *ExemptionsAccountDeleteHandlerOptions) (result *ExemptionDeletionInfo, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ExemptionsAccountDeleteHandlerWithContext(context.Background(), exemptionsAccountDeleteHandlerOptions)
}

// ExemptionsAccountDeleteHandlerWithContext is an alternate form of the ExemptionsAccountDeleteHandler method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ExemptionsAccountDeleteHandlerWithContext(ctx context.Context, exemptionsAccountDeleteHandlerOptions *ExemptionsAccountDeleteHandlerOptions) (result *ExemptionDeletionInfo, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(exemptionsAccountDeleteHandlerOptions, "exemptionsAccountDeleteHandlerOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exemptions/deleteAll`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range exemptionsAccountDeleteHandlerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ExemptionsAccountDeleteHandler")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemptionDeletionInfo)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListImageExemptions : List image exemptions
// List all of the exemptions for an image.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListImageExemptions(listImageExemptionsOptions *ListImageExemptionsOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ListImageExemptionsWithContext(context.Background(), listImageExemptionsOptions)
}

// ListImageExemptionsWithContext is an alternate form of the ListImageExemptions method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListImageExemptionsWithContext(ctx context.Context, listImageExemptionsOptions *ListImageExemptionsOptions) (result []Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listImageExemptionsOptions, "listImageExemptionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listImageExemptionsOptions, "listImageExemptionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"resource": *listImageExemptionsOptions.Resource,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exemptions/image/{resource}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listImageExemptionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ListImageExemptions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	if listImageExemptionsOptions.IncludeScope != nil {
		builder.AddQuery("includeScope", fmt.Sprint(*listImageExemptionsOptions.IncludeScope))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse []json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListBulkImageExemptions : List exemptions for images
// List the exemptions for the given list of images.
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListBulkImageExemptions(listBulkImageExemptionsOptions *ListBulkImageExemptionsOptions) (result map[string][]Exemption, response *core.DetailedResponse, err error) {
	return vulnerabilityAdvisor.ListBulkImageExemptionsWithContext(context.Background(), listBulkImageExemptionsOptions)
}

// ListBulkImageExemptionsWithContext is an alternate form of the ListBulkImageExemptions method which supports a Context parameter
func (vulnerabilityAdvisor *VulnerabilityAdvisorV3) ListBulkImageExemptionsWithContext(ctx context.Context, listBulkImageExemptionsOptions *ListBulkImageExemptionsOptions) (result map[string][]Exemption, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listBulkImageExemptionsOptions, "listBulkImageExemptionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listBulkImageExemptionsOptions, "listBulkImageExemptionsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vulnerabilityAdvisor.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vulnerabilityAdvisor.Service.Options.URL, `/va/api/v3/exemptions/images`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listBulkImageExemptionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vulnerability_advisor", "V3", "ListBulkImageExemptions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")
	if vulnerabilityAdvisor.Account != nil {
		builder.AddHeader("Account", fmt.Sprint(*vulnerabilityAdvisor.Account))
	}
	if vulnerabilityAdvisor.AcceptLanguage != nil {
		builder.AddHeader("Accept-Language", fmt.Sprint(*vulnerabilityAdvisor.AcceptLanguage))
	}

	_, err = builder.SetBodyContentJSON(listBulkImageExemptionsOptions.Body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vulnerabilityAdvisor.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalExemption)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// AccountReportQueryPathOptions : The AccountReportQueryPath options.
type AccountReportQueryPathOptions struct {
	// The name of the repository that you want to see image vulnerability assessments for. For example,
	// us.icr.io/namespace/image.
	Repository *string

	// When set to true, the returned list contains IBM public images and the account images. If not set, or set to false,
	// the list contains only the account images.
	IncludeIBM *string

	// When set to false, the returned list does not contain the private account images. If not set, or set to true, the
	// list contains the private account images.
	IncludePrivate *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAccountReportQueryPathOptions : Instantiate AccountReportQueryPathOptions
func (*VulnerabilityAdvisorV3) NewAccountReportQueryPathOptions() *AccountReportQueryPathOptions {
	return &AccountReportQueryPathOptions{}
}

// SetRepository : Allow user to set Repository
func (options *AccountReportQueryPathOptions) SetRepository(repository string) *AccountReportQueryPathOptions {
	options.Repository = core.StringPtr(repository)
	return options
}

// SetIncludeIBM : Allow user to set IncludeIBM
func (options *AccountReportQueryPathOptions) SetIncludeIBM(includeIBM string) *AccountReportQueryPathOptions {
	options.IncludeIBM = core.StringPtr(includeIBM)
	return options
}

// SetIncludePrivate : Allow user to set IncludePrivate
func (options *AccountReportQueryPathOptions) SetIncludePrivate(includePrivate string) *AccountReportQueryPathOptions {
	options.IncludePrivate = core.StringPtr(includePrivate)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AccountReportQueryPathOptions) SetHeaders(param map[string]string) *AccountReportQueryPathOptions {
	options.Headers = param
	return options
}

// AccountStatusQueryPathOptions : The AccountStatusQueryPath options.
type AccountStatusQueryPathOptions struct {
	// The name of the repository that you want to see image vulnerability assessments for. For example,
	// us.icr.io/namespace/image.
	Repository *string

	// When set to true, the returned list contains IBM public images and the account images. If not set, or set to false,
	// the list contains only the account images.
	IncludeIBM *string

	// When set to false, the returned list does not contain the private account images. If not set, or set to true, the
	// list contains the private account images.
	IncludePrivate *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAccountStatusQueryPathOptions : Instantiate AccountStatusQueryPathOptions
func (*VulnerabilityAdvisorV3) NewAccountStatusQueryPathOptions() *AccountStatusQueryPathOptions {
	return &AccountStatusQueryPathOptions{}
}

// SetRepository : Allow user to set Repository
func (options *AccountStatusQueryPathOptions) SetRepository(repository string) *AccountStatusQueryPathOptions {
	options.Repository = core.StringPtr(repository)
	return options
}

// SetIncludeIBM : Allow user to set IncludeIBM
func (options *AccountStatusQueryPathOptions) SetIncludeIBM(includeIBM string) *AccountStatusQueryPathOptions {
	options.IncludeIBM = core.StringPtr(includeIBM)
	return options
}

// SetIncludePrivate : Allow user to set IncludePrivate
func (options *AccountStatusQueryPathOptions) SetIncludePrivate(includePrivate string) *AccountStatusQueryPathOptions {
	options.IncludePrivate = core.StringPtr(includePrivate)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AccountStatusQueryPathOptions) SetHeaders(param map[string]string) *AccountStatusQueryPathOptions {
	options.Headers = param
	return options
}

// CreateExemptionAccountOptions : The CreateExemptionAccount options.
type CreateExemptionAccountOptions struct {
	// Exemption type, e.g. 'cve' or 'sn' or 'configuration'. See /va/api/v3/exempt/types for more details.
	IssueType *string `validate:"required,ne="`

	// Exemption ID, e.g. 'CVE-2018-9999'. See /va/api/v3/exempt/types for more details.
	IssueID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateExemptionAccountOptions : Instantiate CreateExemptionAccountOptions
func (*VulnerabilityAdvisorV3) NewCreateExemptionAccountOptions(issueType string, issueID string) *CreateExemptionAccountOptions {
	return &CreateExemptionAccountOptions{
		IssueType: core.StringPtr(issueType),
		IssueID:   core.StringPtr(issueID),
	}
}

// SetIssueType : Allow user to set IssueType
func (options *CreateExemptionAccountOptions) SetIssueType(issueType string) *CreateExemptionAccountOptions {
	options.IssueType = core.StringPtr(issueType)
	return options
}

// SetIssueID : Allow user to set IssueID
func (options *CreateExemptionAccountOptions) SetIssueID(issueID string) *CreateExemptionAccountOptions {
	options.IssueID = core.StringPtr(issueID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateExemptionAccountOptions) SetHeaders(param map[string]string) *CreateExemptionAccountOptions {
	options.Headers = param
	return options
}

// CreateExemptionResourceOptions : The CreateExemptionResource options.
type CreateExemptionResourceOptions struct {
	// IBM Cloud Registry resource (namespace, namespace/repository, namespace/repository:tag, or
	// namespace/repository@sha256:hash).
	Resource *string `validate:"required,ne="`

	// Exemption type, e.g. 'cve' or 'sn' or 'configuration'. See /va/api/v3/exempt/types for more details.
	IssueType *string `validate:"required,ne="`

	// Exemption ID, e.g. 'CVE-2018-9999'. See /va/api/v3/exempt/types for more details.
	IssueID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateExemptionResourceOptions : Instantiate CreateExemptionResourceOptions
func (*VulnerabilityAdvisorV3) NewCreateExemptionResourceOptions(resource string, issueType string, issueID string) *CreateExemptionResourceOptions {
	return &CreateExemptionResourceOptions{
		Resource:  core.StringPtr(resource),
		IssueType: core.StringPtr(issueType),
		IssueID:   core.StringPtr(issueID),
	}
}

// SetResource : Allow user to set Resource
func (options *CreateExemptionResourceOptions) SetResource(resource string) *CreateExemptionResourceOptions {
	options.Resource = core.StringPtr(resource)
	return options
}

// SetIssueType : Allow user to set IssueType
func (options *CreateExemptionResourceOptions) SetIssueType(issueType string) *CreateExemptionResourceOptions {
	options.IssueType = core.StringPtr(issueType)
	return options
}

// SetIssueID : Allow user to set IssueID
func (options *CreateExemptionResourceOptions) SetIssueID(issueID string) *CreateExemptionResourceOptions {
	options.IssueID = core.StringPtr(issueID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateExemptionResourceOptions) SetHeaders(param map[string]string) *CreateExemptionResourceOptions {
	options.Headers = param
	return options
}

// DeleteExemptionAccountOptions : The DeleteExemptionAccount options.
type DeleteExemptionAccountOptions struct {
	// Exemption type, e.g. 'cve' or 'sn' or 'configuration'. See /va/api/v3/exempt/types for more details.
	IssueType *string `validate:"required,ne="`

	// Exemption ID, e.g. 'CVE-2018-9999'. See /va/api/v3/exempt/types for more details.
	IssueID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteExemptionAccountOptions : Instantiate DeleteExemptionAccountOptions
func (*VulnerabilityAdvisorV3) NewDeleteExemptionAccountOptions(issueType string, issueID string) *DeleteExemptionAccountOptions {
	return &DeleteExemptionAccountOptions{
		IssueType: core.StringPtr(issueType),
		IssueID:   core.StringPtr(issueID),
	}
}

// SetIssueType : Allow user to set IssueType
func (options *DeleteExemptionAccountOptions) SetIssueType(issueType string) *DeleteExemptionAccountOptions {
	options.IssueType = core.StringPtr(issueType)
	return options
}

// SetIssueID : Allow user to set IssueID
func (options *DeleteExemptionAccountOptions) SetIssueID(issueID string) *DeleteExemptionAccountOptions {
	options.IssueID = core.StringPtr(issueID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteExemptionAccountOptions) SetHeaders(param map[string]string) *DeleteExemptionAccountOptions {
	options.Headers = param
	return options
}

// DeleteExemptionResourceOptions : The DeleteExemptionResource options.
type DeleteExemptionResourceOptions struct {
	// IBM Cloud Registry resource (namespace, namespace/repository, namespace/repository:tag, or
	// namespace/repository@sha256:hash).
	Resource *string `validate:"required,ne="`

	// Exemption type, e.g. 'cve' or 'sn' or 'configuration'. See /va/api/v3/exempt/types for more details.
	IssueType *string `validate:"required,ne="`

	// Exemption ID, e.g. 'CVE-2018-9999'. See /va/api/v3/exempt/types for more details.
	IssueID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteExemptionResourceOptions : Instantiate DeleteExemptionResourceOptions
func (*VulnerabilityAdvisorV3) NewDeleteExemptionResourceOptions(resource string, issueType string, issueID string) *DeleteExemptionResourceOptions {
	return &DeleteExemptionResourceOptions{
		Resource:  core.StringPtr(resource),
		IssueType: core.StringPtr(issueType),
		IssueID:   core.StringPtr(issueID),
	}
}

// SetResource : Allow user to set Resource
func (options *DeleteExemptionResourceOptions) SetResource(resource string) *DeleteExemptionResourceOptions {
	options.Resource = core.StringPtr(resource)
	return options
}

// SetIssueType : Allow user to set IssueType
func (options *DeleteExemptionResourceOptions) SetIssueType(issueType string) *DeleteExemptionResourceOptions {
	options.IssueType = core.StringPtr(issueType)
	return options
}

// SetIssueID : Allow user to set IssueID
func (options *DeleteExemptionResourceOptions) SetIssueID(issueID string) *DeleteExemptionResourceOptions {
	options.IssueID = core.StringPtr(issueID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteExemptionResourceOptions) SetHeaders(param map[string]string) *DeleteExemptionResourceOptions {
	options.Headers = param
	return options
}

// ExemptHandlerOptions : The ExemptHandler options.
type ExemptHandlerOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewExemptHandlerOptions : Instantiate ExemptHandlerOptions
func (*VulnerabilityAdvisorV3) NewExemptHandlerOptions() *ExemptHandlerOptions {
	return &ExemptHandlerOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ExemptHandlerOptions) SetHeaders(param map[string]string) *ExemptHandlerOptions {
	options.Headers = param
	return options
}

// Exemption : Exemption struct
type Exemption struct {
	// The unique ID for the IBM Cloud account for this policy.
	AccountID *string `json:"account_id" validate:"required"`

	// ID of issue being exempted.
	IssueID *string `json:"issue_id" validate:"required"`

	// Type of issue being exempted.
	IssueType *string `json:"issue_type" validate:"required"`

	Scope *ExemptionScope `json:"scope,omitempty"`
}

// UnmarshalExemption unmarshals an instance of Exemption from the specified map of raw messages.
func UnmarshalExemption(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Exemption)
	err = core.UnmarshalPrimitive(m, "account_id", &obj.AccountID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "issue_id", &obj.IssueID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "issue_type", &obj.IssueType)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "scope", &obj.Scope, UnmarshalExemptionScope)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ExemptionScope : ExemptionScope struct
type ExemptionScope struct {
	// The type of scope the exemption applies to: account, namespace, repository, image.
	ScopeType *string `json:"scope_type" validate:"required"`

	// The IBM Cloud Registry namespace included in scope.
	Namespace *string `json:"namespace,omitempty"`

	// The IBM Cloud Registry repository included in scope.
	Repository *string `json:"repository,omitempty"`

	// The IBM Cloud Registry tag included in scope.
	Tag *string `json:"tag,omitempty"`
}

// UnmarshalExemptionScope unmarshals an instance of ExemptionScope from the specified map of raw messages.
func UnmarshalExemptionScope(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ExemptionScope)
	err = core.UnmarshalPrimitive(m, "scope_type", &obj.ScopeType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "namespace", &obj.Namespace)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "repository", &obj.Repository)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "tag", &obj.Tag)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ExemptionsAccountDeleteHandlerOptions : The ExemptionsAccountDeleteHandler options.
type ExemptionsAccountDeleteHandlerOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewExemptionsAccountDeleteHandlerOptions : Instantiate ExemptionsAccountDeleteHandlerOptions
func (*VulnerabilityAdvisorV3) NewExemptionsAccountDeleteHandlerOptions() *ExemptionsAccountDeleteHandlerOptions {
	return &ExemptionsAccountDeleteHandlerOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ExemptionsAccountDeleteHandlerOptions) SetHeaders(param map[string]string) *ExemptionsAccountDeleteHandlerOptions {
	options.Headers = param
	return options
}

// GetExemptionAccountOptions : The GetExemptionAccount options.
type GetExemptionAccountOptions struct {
	// Exemption type, e.g. 'cve' or 'sn' or 'configuration'. See /va/api/v3/exempt/types for more details.
	IssueType *string `validate:"required,ne="`

	// Exemption ID, e.g. 'CVE-2018-9999'. See /va/api/v3/exempt/types for more details.
	IssueID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetExemptionAccountOptions : Instantiate GetExemptionAccountOptions
func (*VulnerabilityAdvisorV3) NewGetExemptionAccountOptions(issueType string, issueID string) *GetExemptionAccountOptions {
	return &GetExemptionAccountOptions{
		IssueType: core.StringPtr(issueType),
		IssueID:   core.StringPtr(issueID),
	}
}

// SetIssueType : Allow user to set IssueType
func (options *GetExemptionAccountOptions) SetIssueType(issueType string) *GetExemptionAccountOptions {
	options.IssueType = core.StringPtr(issueType)
	return options
}

// SetIssueID : Allow user to set IssueID
func (options *GetExemptionAccountOptions) SetIssueID(issueID string) *GetExemptionAccountOptions {
	options.IssueID = core.StringPtr(issueID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetExemptionAccountOptions) SetHeaders(param map[string]string) *GetExemptionAccountOptions {
	options.Headers = param
	return options
}

// GetExemptionResourceOptions : The GetExemptionResource options.
type GetExemptionResourceOptions struct {
	// IBM Cloud Registry resource (namespace, namespace/repository, namespace/repository:tag, or
	// namespace/repository@sha256:hash).
	Resource *string `validate:"required,ne="`

	// Exemption type, e.g. 'cve' or 'sn' or 'configuration'. See /va/api/v3/exempt/types for more details.
	IssueType *string `validate:"required,ne="`

	// Exemption ID, e.g. 'CVE-2018-9999'. See /va/api/v3/exempt/types for more details.
	IssueID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetExemptionResourceOptions : Instantiate GetExemptionResourceOptions
func (*VulnerabilityAdvisorV3) NewGetExemptionResourceOptions(resource string, issueType string, issueID string) *GetExemptionResourceOptions {
	return &GetExemptionResourceOptions{
		Resource:  core.StringPtr(resource),
		IssueType: core.StringPtr(issueType),
		IssueID:   core.StringPtr(issueID),
	}
}

// SetResource : Allow user to set Resource
func (options *GetExemptionResourceOptions) SetResource(resource string) *GetExemptionResourceOptions {
	options.Resource = core.StringPtr(resource)
	return options
}

// SetIssueType : Allow user to set IssueType
func (options *GetExemptionResourceOptions) SetIssueType(issueType string) *GetExemptionResourceOptions {
	options.IssueType = core.StringPtr(issueType)
	return options
}

// SetIssueID : Allow user to set IssueID
func (options *GetExemptionResourceOptions) SetIssueID(issueID string) *GetExemptionResourceOptions {
	options.IssueID = core.StringPtr(issueID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetExemptionResourceOptions) SetHeaders(param map[string]string) *GetExemptionResourceOptions {
	options.Headers = param
	return options
}

// ImageReportQueryPathOptions : The ImageReportQueryPath options.
type ImageReportQueryPathOptions struct {
	// The name of the image. For example, us.icr.io/namespace/repository:tag.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewImageReportQueryPathOptions : Instantiate ImageReportQueryPathOptions
func (*VulnerabilityAdvisorV3) NewImageReportQueryPathOptions(name string) *ImageReportQueryPathOptions {
	return &ImageReportQueryPathOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *ImageReportQueryPathOptions) SetName(name string) *ImageReportQueryPathOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ImageReportQueryPathOptions) SetHeaders(param map[string]string) *ImageReportQueryPathOptions {
	options.Headers = param
	return options
}

// ImageStatusQueryPathOptions : The ImageStatusQueryPath options.
type ImageStatusQueryPathOptions struct {
	// The name of the image. For example, us.icr.io/namespace/repository:tag.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewImageStatusQueryPathOptions : Instantiate ImageStatusQueryPathOptions
func (*VulnerabilityAdvisorV3) NewImageStatusQueryPathOptions(name string) *ImageStatusQueryPathOptions {
	return &ImageStatusQueryPathOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *ImageStatusQueryPathOptions) SetName(name string) *ImageStatusQueryPathOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ImageStatusQueryPathOptions) SetHeaders(param map[string]string) *ImageStatusQueryPathOptions {
	options.Headers = param
	return options
}

// ListAccountExemptionsOptions : The ListAccountExemptions options.
type ListAccountExemptionsOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListAccountExemptionsOptions : Instantiate ListAccountExemptionsOptions
func (*VulnerabilityAdvisorV3) NewListAccountExemptionsOptions() *ListAccountExemptionsOptions {
	return &ListAccountExemptionsOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListAccountExemptionsOptions) SetHeaders(param map[string]string) *ListAccountExemptionsOptions {
	options.Headers = param
	return options
}

// ListBulkImageExemptionsOptions : The ListBulkImageExemptions options.
type ListBulkImageExemptionsOptions struct {
	// List of images.
	Body []string `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListBulkImageExemptionsOptions : Instantiate ListBulkImageExemptionsOptions
func (*VulnerabilityAdvisorV3) NewListBulkImageExemptionsOptions(body []string) *ListBulkImageExemptionsOptions {
	return &ListBulkImageExemptionsOptions{
		Body: body,
	}
}

// SetBody : Allow user to set Body
func (options *ListBulkImageExemptionsOptions) SetBody(body []string) *ListBulkImageExemptionsOptions {
	options.Body = body
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListBulkImageExemptionsOptions) SetHeaders(param map[string]string) *ListBulkImageExemptionsOptions {
	options.Headers = param
	return options
}

// ListExemptionAccountOptions : The ListExemptionAccount options.
type ListExemptionAccountOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListExemptionAccountOptions : Instantiate ListExemptionAccountOptions
func (*VulnerabilityAdvisorV3) NewListExemptionAccountOptions() *ListExemptionAccountOptions {
	return &ListExemptionAccountOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListExemptionAccountOptions) SetHeaders(param map[string]string) *ListExemptionAccountOptions {
	options.Headers = param
	return options
}

// ListExemptionResourceOptions : The ListExemptionResource options.
type ListExemptionResourceOptions struct {
	// IBM Cloud Registry resource (namespace, namespace/repository, namespace/repository:tag, or
	// namespace/repository@sha256:hash).
	Resource *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListExemptionResourceOptions : Instantiate ListExemptionResourceOptions
func (*VulnerabilityAdvisorV3) NewListExemptionResourceOptions(resource string) *ListExemptionResourceOptions {
	return &ListExemptionResourceOptions{
		Resource: core.StringPtr(resource),
	}
}

// SetResource : Allow user to set Resource
func (options *ListExemptionResourceOptions) SetResource(resource string) *ListExemptionResourceOptions {
	options.Resource = core.StringPtr(resource)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListExemptionResourceOptions) SetHeaders(param map[string]string) *ListExemptionResourceOptions {
	options.Headers = param
	return options
}

// ListImageExemptionsOptions : The ListImageExemptions options.
type ListImageExemptionsOptions struct {
	// IBM Cloud Registry resource (namespace, namespace/repository, namespace/repository:tag, or
	// namespace/repository@sha256:hash).
	Resource *string `validate:"required,ne="`

	// Include scope on returned exemptions.
	IncludeScope *bool

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListImageExemptionsOptions : Instantiate ListImageExemptionsOptions
func (*VulnerabilityAdvisorV3) NewListImageExemptionsOptions(resource string) *ListImageExemptionsOptions {
	return &ListImageExemptionsOptions{
		Resource: core.StringPtr(resource),
	}
}

// SetResource : Allow user to set Resource
func (options *ListImageExemptionsOptions) SetResource(resource string) *ListImageExemptionsOptions {
	options.Resource = core.StringPtr(resource)
	return options
}

// SetIncludeScope : Allow user to set IncludeScope
func (options *ListImageExemptionsOptions) SetIncludeScope(includeScope bool) *ListImageExemptionsOptions {
	options.IncludeScope = core.BoolPtr(includeScope)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListImageExemptionsOptions) SetHeaders(param map[string]string) *ListImageExemptionsOptions {
	options.Headers = param
	return options
}

// ScanReport : ScanReport struct
type ScanReport struct {
	// Possible configuration issues found in the docker object.
	ConfigurationIssues []ScanresultConfigurationIssue `json:"configuration_issues" validate:"required"`

	// The unique ID of the report.
	ID *string `json:"id" validate:"required"`

	// The scan time of the report as a UNIX timestamp.
	ScanTime *int64 `json:"scan_time" validate:"required"`

	// Overall vulnerability assessment status: OK, WARN, FAIL, UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information
	// about these status codes, see
	// https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
	Status *string `json:"status" validate:"required"`

	// Vulnerabilities found in the docker object at the scan time.
	Vulnerabilities []ScanresultCVE `json:"vulnerabilities" validate:"required"`
}

// UnmarshalScanReport unmarshals an instance of ScanReport from the specified map of raw messages.
func UnmarshalScanReport(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanReport)
	err = core.UnmarshalModel(m, "configuration_issues", &obj.ConfigurationIssues, UnmarshalScanresultConfigurationIssue)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "scan_time", &obj.ScanTime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vulnerabilities", &obj.Vulnerabilities, UnmarshalScanresultCVE)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanReportList : ScanReportList struct
type ScanReportList struct {
	// A dictionary of image names as key and report.Report object as value.
	Assessments map[string]ScanReport `json:"assessments" validate:"required"`
}

// UnmarshalScanReportList unmarshals an instance of ScanReportList from the specified map of raw messages.
func UnmarshalScanReportList(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanReportList)
	err = core.UnmarshalModel(m, "assessments", &obj.Assessments, UnmarshalScanReport)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ExemptionDeletionInfo : ExemptionDeletionInfo struct
type ExemptionDeletionInfo struct {
	NumberOfExemptionsDeleted *int64 `json:"number_of_exemptions_deleted" validate:"required"`
}

// UnmarshalExemptionDeletionInfo unmarshals an instance of ExemptionDeletionInfo from the specified map of raw messages.
func UnmarshalExemptionDeletionInfo(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ExemptionDeletionInfo)
	err = core.UnmarshalPrimitive(m, "number_of_exemptions_deleted", &obj.NumberOfExemptionsDeleted)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ExemptionTypeInfo : ExemptionTypeInfo struct
type ExemptionTypeInfo struct {
	DisplayName *string `json:"display_name" validate:"required"`

	Identifier *string `json:"identifier" validate:"required"`

	TranslationKey *string `json:"translationKey,omitempty"`

	ValidationRegex *string `json:"validation_regex" validate:"required"`
}

// UnmarshalExemptionTypeInfo unmarshals an instance of ExemptionTypeInfo from the specified map of raw messages.
func UnmarshalExemptionTypeInfo(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ExemptionTypeInfo)
	err = core.UnmarshalPrimitive(m, "display_name", &obj.DisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "identifier", &obj.Identifier)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "translationKey", &obj.TranslationKey)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "validation_regex", &obj.ValidationRegex)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanreportImageSummary : ScanreportImageSummary struct
type ScanreportImageSummary struct {
	// The number of configuration issues found.
	ConfigurationIssueCount *int64 `json:"configuration_issue_count" validate:"required"`

	// The image creation time as a UNIX timestamp.
	CreatedTime *int64 `json:"created_time" validate:"required"`

	// The number of exempt configuration issues found.
	ExemptConfigurationIssueCount *int64 `json:"exempt_configuration_issue_count" validate:"required"`

	// The number of exempt issues found.
	ExemptIssueCount *int64 `json:"exempt_issue_count" validate:"required"`

	// The number of exempt vulnerability issues found.
	ExemptVulnerabilityCount *int64 `json:"exempt_vulnerability_count" validate:"required"`

	// The number of issues found.
	IssueCount *int64 `json:"issue_count" validate:"required"`

	// Full docker image name including tag e.g. us.icr.io/namespace/repository:tag.
	Name *string `json:"name" validate:"required"`

	// The scan time of the report as a UNIX timestamp.
	ScanTime *int64 `json:"scan_time" validate:"required"`

	// Overall vulnerability assessment status: OK, WARN, FAIL, UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information
	// about these status codes, see
	// https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
	Status *string `json:"status" validate:"required"`

	// The number of vulnerability issues found.
	VulnerabilityCount *int64 `json:"vulnerability_count" validate:"required"`
}

// UnmarshalScanreportImageSummary unmarshals an instance of ScanreportImageSummary from the specified map of raw messages.
func UnmarshalScanreportImageSummary(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanreportImageSummary)
	err = core.UnmarshalPrimitive(m, "configuration_issue_count", &obj.ConfigurationIssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_time", &obj.CreatedTime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_configuration_issue_count", &obj.ExemptConfigurationIssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_issue_count", &obj.ExemptIssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_vulnerability_count", &obj.ExemptVulnerabilityCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "issue_count", &obj.IssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "scan_time", &obj.ScanTime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "vulnerability_count", &obj.VulnerabilityCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanreportImageSummaryList : ScanreportImageSummaryList struct
type ScanreportImageSummaryList struct {
	// List of image summaries.
	Images []ScanreportImageSummary `json:"images" validate:"required"`
}

// UnmarshalScanreportImageSummaryList unmarshals an instance of ScanreportImageSummaryList from the specified map of raw messages.
func UnmarshalScanreportImageSummaryList(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanreportImageSummaryList)
	err = core.UnmarshalModel(m, "images", &obj.Images, UnmarshalScanreportImageSummary)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanreportSummary : ScanreportSummary struct
type ScanreportSummary struct {
	// The number of configuration issues found.
	ConfigurationIssueCount *int64 `json:"configuration_issue_count" validate:"required"`

	// The number of exempt configuration issues found.
	ExemptConfigurationIssueCount *int64 `json:"exempt_configuration_issue_count" validate:"required"`

	// The number of exempt issues found.
	ExemptIssueCount *int64 `json:"exempt_issue_count" validate:"required"`

	// The number of exempt vulnerability issues found.
	ExemptVulnerabilityCount *int64 `json:"exempt_vulnerability_count" validate:"required"`

	// The number of issues found.
	IssueCount *int64 `json:"issue_count" validate:"required"`

	// The scan time of the report as a UNIX timestamp.
	ScanTime *int64 `json:"scan_time" validate:"required"`

	// Overall vulnerability assessment status: OK, WARN, FAIL, UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information
	// about these status codes, see
	// https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
	Status *string `json:"status" validate:"required"`

	// The number of vulnerability issues found.
	VulnerabilityCount *int64 `json:"vulnerability_count" validate:"required"`
}

// UnmarshalScanreportSummary unmarshals an instance of ScanreportSummary from the specified map of raw messages.
func UnmarshalScanreportSummary(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanreportSummary)
	err = core.UnmarshalPrimitive(m, "configuration_issue_count", &obj.ConfigurationIssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_configuration_issue_count", &obj.ExemptConfigurationIssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_issue_count", &obj.ExemptIssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_vulnerability_count", &obj.ExemptVulnerabilityCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "issue_count", &obj.IssueCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "scan_time", &obj.ScanTime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "vulnerability_count", &obj.VulnerabilityCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanresultCVE : ScanresultCVE struct
type ScanresultCVE struct {
	// True if this CVE has been exempted by user policy, and false otherwise.
	CveExempt *bool `json:"cve_exempt" validate:"required"`

	// The ID for this CVE.
	CveID *string `json:"cve_id" validate:"required"`

	// Number of security notices that contain fixes for this CVE and are exempted by user policy.
	ExemptSecurityNoticeCount *int64 `json:"exempt_security_notice_count" validate:"required"`

	// 'exempt' if this CVE is exempt or all security notices for this CVE are exempt. 'partial' if this CVE is not exempt
	// but a subset of security notices for this CVE are exempt. 'active' if this CVE is not exempt and no security notices
	// for this CVE are exempt.
	ExemptStatus *string `json:"exempt_status" validate:"required"`

	// Number of security notices that contain fixes for this CVE and are not exempted by user policy.
	SecurityNoticeCount *int64 `json:"security_notice_count" validate:"required"`

	// Security notices that contain fixes for this CVE.
	SecurityNotices []ScanresultSecurityNotice `json:"security_notices" validate:"required"`

	// The summary of the security vulnerability for this CVE.
	Summary *string `json:"summary" validate:"required"`

	// Total number of security notices that contain fixes for this CVE.
	TotalSecurityNoticeCount *int64 `json:"total_security_notice_count" validate:"required"`
}

// UnmarshalScanresultCVE unmarshals an instance of ScanresultCVE from the specified map of raw messages.
func UnmarshalScanresultCVE(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanresultCVE)
	err = core.UnmarshalPrimitive(m, "cve_exempt", &obj.CveExempt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cve_id", &obj.CveID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_security_notice_count", &obj.ExemptSecurityNoticeCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt_status", &obj.ExemptStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "security_notice_count", &obj.SecurityNoticeCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "security_notices", &obj.SecurityNotices, UnmarshalScanresultSecurityNotice)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "summary", &obj.Summary)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_security_notice_count", &obj.TotalSecurityNoticeCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanresultConfigurationIssue : ScanresultConfigurationIssue struct
type ScanresultConfigurationIssue struct {
	// Advice on how to solve this ConfigurationIssue.
	CorrectiveAction *string `json:"corrective_action" validate:"required"`

	// Description of this ConfigurationIssue.
	Description *string `json:"description" validate:"required"`

	// True if this configuration issue is exempted by user policy, and false otherwise.
	Exempt *bool `json:"exempt" validate:"required"`

	// Additional keys and string values about this ConfigurationIssue.
	Meta map[string]string `json:"meta" validate:"required"`

	// The ID of the check which found this ConfigurationIssue.
	Type *string `json:"type" validate:"required"`
}

// UnmarshalScanresultConfigurationIssue unmarshals an instance of ScanresultConfigurationIssue from the specified map of raw messages.
func UnmarshalScanresultConfigurationIssue(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanresultConfigurationIssue)
	err = core.UnmarshalPrimitive(m, "corrective_action", &obj.CorrectiveAction)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "description", &obj.Description)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "exempt", &obj.Exempt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "meta", &obj.Meta)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanresultPackageFixes : ScanresultPackageFixes struct
type ScanresultPackageFixes struct {
	// Advice on how to solve this vulnerability.
	CorrectiveAction *string `json:"corrective_action" validate:"required"`

	// Description of the vulnerability.
	Description *string `json:"description" validate:"required"`

	// The version of this package that contains the fix for this vulnerability.
	FixVersion *string `json:"fix_version" validate:"required"`

	// The version of this package that was found installed at scan time.
	InstalledVersion *string `json:"installed_version" validate:"required"`

	// The name of the package.
	PackageName *string `json:"package_name" validate:"required"`
}

// UnmarshalScanresultPackageFixes unmarshals an instance of ScanresultPackageFixes from the specified map of raw messages.
func UnmarshalScanresultPackageFixes(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanresultPackageFixes)
	err = core.UnmarshalPrimitive(m, "corrective_action", &obj.CorrectiveAction)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "description", &obj.Description)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "fix_version", &obj.FixVersion)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "installed_version", &obj.InstalledVersion)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "package_name", &obj.PackageName)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ScanresultSecurityNotice : ScanresultSecurityNotice struct
type ScanresultSecurityNotice struct {
	// Further information about this security notice.
	Notice *string `json:"notice" validate:"required"`

	// True if this security notice has been exempted by user policy, and false otherwise.
	NoticeExempt *bool `json:"notice_exempt" validate:"required"`

	// The ID for this security notice.
	NoticeID *string `json:"notice_id" validate:"required"`

	// Summary of information about the security vulnerabilities fixed by this security notice.
	Summary *string `json:"summary" validate:"required"`

	// Package updates that contain fixes for this vulnerability.
	VulnerablePackages []ScanresultPackageFixes `json:"vulnerable_packages" validate:"required"`
}

// UnmarshalScanresultSecurityNotice unmarshals an instance of ScanresultSecurityNotice from the specified map of raw messages.
func UnmarshalScanresultSecurityNotice(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ScanresultSecurityNotice)
	err = core.UnmarshalPrimitive(m, "notice", &obj.Notice)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "notice_exempt", &obj.NoticeExempt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "notice_id", &obj.NoticeID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "summary", &obj.Summary)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vulnerable_packages", &obj.VulnerablePackages, UnmarshalScanresultPackageFixes)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}
