package v1

import (
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// +genclient
// +genclient:nonNamespaced
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// Ingress holds cluster-wide information about ingress, including the default ingress domain
// used for routes. The canonical name is `cluster`.
//
// Compatibility level 1: Stable within a major release for a minimum of 12 months or 3 minor releases (whichever is longer).
// +openshift:compatibility-gen:level=1
type Ingress struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata,omitempty"`

	// spec holds user settable values for configuration
	// +kubebuilder:validation:Required
	// +required
	Spec IngressSpec `json:"spec"`
	// status holds observed values from the cluster. They may not be overridden.
	// +optional
	Status IngressStatus `json:"status"`
}

type IngressSpec struct {
	// domain is used to generate a default host name for a route when the
	// route's host name is empty. The generated host name will follow this
	// pattern: "<route-name>.<route-namespace>.<domain>".
	//
	// It is also used as the default wildcard domain suffix for ingress. The
	// default ingresscontroller domain will follow this pattern: "*.<domain>".
	//
	// Once set, changing domain is not currently supported.
	Domain string `json:"domain"`

	// appsDomain is an optional domain to use instead of the one specified
	// in the domain field when a Route is created without specifying an explicit
	// host. If appsDomain is nonempty, this value is used to generate default
	// host values for Route. Unlike domain, appsDomain may be modified after
	// installation.
	// This assumes a new ingresscontroller has been setup with a wildcard
	// certificate.
	// +optional
	AppsDomain string `json:"appsDomain,omitempty"`

	// componentRoutes is an optional list of routes that are managed by OpenShift components
	// that a cluster-admin is able to configure the hostname and serving certificate for.
	// The namespace and name of each route in this list should match an existing entry in the
	// status.componentRoutes list.
	//
	// To determine the set of configurable Routes, look at namespace and name of entries in the
	// .status.componentRoutes list, where participating operators write the status of
	// configurable routes.
	// +optional
	ComponentRoutes []ComponentRouteSpec `json:"componentRoutes,omitempty"`

	// requiredHSTSPolicies specifies HSTS policies that are required to be set on newly created  or updated routes
	// matching the domainPattern/s and namespaceSelector/s that are specified in the policy.
	// Each requiredHSTSPolicy must have at least a domainPattern and a maxAge to validate a route HSTS Policy route
	// annotation, and affect route admission.
	//
	// A candidate route is checked for HSTS Policies if it has the HSTS Policy route annotation:
	// "haproxy.router.openshift.io/hsts_header"
	// E.g. haproxy.router.openshift.io/hsts_header: max-age=31536000;preload;includeSubDomains
	//
	// - For each candidate route, if it matches a requiredHSTSPolicy domainPattern and optional namespaceSelector,
	// then the maxAge, preloadPolicy, and includeSubdomainsPolicy must be valid to be admitted.  Otherwise, the route
	// is rejected.
	// - The first match, by domainPattern and optional namespaceSelector, in the ordering of the RequiredHSTSPolicies
	// determines the route's admission status.
	// - If the candidate route doesn't match any requiredHSTSPolicy domainPattern and optional namespaceSelector,
	// then it may use any HSTS Policy annotation.
	//
	// The HSTS policy configuration may be changed after routes have already been created. An update to a previously
	// admitted route may then fail if the updated route does not conform to the updated HSTS policy configuration.
	// However, changing the HSTS policy configuration will not cause a route that is already admitted to stop working.
	//
	// Note that if there are no RequiredHSTSPolicies, any HSTS Policy annotation on the route is valid.
	// +optional
	RequiredHSTSPolicies []RequiredHSTSPolicy `json:"requiredHSTSPolicies,omitempty"`
}

// ConsumingUser is an alias for string which we add validation to. Currently only service accounts are supported.
// +kubebuilder:validation:Pattern="^system:serviceaccount:[a-z0-9]([-a-z0-9]*[a-z0-9])?:[a-z0-9]([-a-z0-9]*[a-z0-9])?(\\.[a-z0-9]([-a-z0-9]*[a-z0-9])?)*$"
// +kubebuilder:validation:MinLength=1
// +kubebuilder:validation:MaxLength=512
type ConsumingUser string

// Hostname is an alias for hostname string validation.
//
// The left operand of the | is the original kubebuilder hostname validation format, which is incorrect because it
// allows upper case letters, disallows hyphen or number in the TLD, and allows labels to start/end in non-alphanumeric
// characters.  See https://bugzilla.redhat.com/show_bug.cgi?id=2039256.
// ^([a-zA-Z0-9\p{S}\p{L}]((-?[a-zA-Z0-9\p{S}\p{L}]{0,62})?)|([a-zA-Z0-9\p{S}\p{L}](([a-zA-Z0-9-\p{S}\p{L}]{0,61}[a-zA-Z0-9\p{S}\p{L}])?)(\.)){1,}([a-zA-Z\p{L}]){2,63})$
//
// The right operand of the | is a new pattern that mimics the current API route admission validation on hostname,
// except that it allows hostnames longer than the maximum length:
// ^(([a-z0-9][-a-z0-9]{0,61}[a-z0-9]|[a-z0-9]{1,63})[\.]){0,}([a-z0-9][-a-z0-9]{0,61}[a-z0-9]|[a-z0-9]{1,63})$
//
// Both operand patterns are made available so that modifications on ingress spec can still happen after an invalid hostname
// was saved via validation by the incorrect left operand of the | operator.
//
// +kubebuilder:validation:Pattern=`^([a-zA-Z0-9\p{S}\p{L}]((-?[a-zA-Z0-9\p{S}\p{L}]{0,62})?)|([a-zA-Z0-9\p{S}\p{L}](([a-zA-Z0-9-\p{S}\p{L}]{0,61}[a-zA-Z0-9\p{S}\p{L}])?)(\.)){1,}([a-zA-Z\p{L}]){2,63})$|^(([a-z0-9][-a-z0-9]{0,61}[a-z0-9]|[a-z0-9]{1,63})[\.]){0,}([a-z0-9][-a-z0-9]{0,61}[a-z0-9]|[a-z0-9]{1,63})$`
type Hostname string

type IngressStatus struct {
	// componentRoutes is where participating operators place the current route status for routes whose
	// hostnames and serving certificates can be customized by the cluster-admin.
	// +optional
	ComponentRoutes []ComponentRouteStatus `json:"componentRoutes,omitempty"`
}

// ComponentRouteSpec allows for configuration of a route's hostname and serving certificate.
type ComponentRouteSpec struct {
	// namespace is the namespace of the route to customize.
	//
	// The namespace and name of this componentRoute must match a corresponding
	// entry in the list of status.componentRoutes if the route is to be customized.
	// +kubebuilder:validation:Pattern=^[a-z0-9]([-a-z0-9]*[a-z0-9])?$
	// +kubebuilder:validation:MinLength=1
	// +kubebuilder:validation:MaxLength=63
	// +kubebuilder:validation:Required
	// +required
	Namespace string `json:"namespace"`

	// name is the logical name of the route to customize.
	//
	// The namespace and name of this componentRoute must match a corresponding
	// entry in the list of status.componentRoutes if the route is to be customized.
	// +kubebuilder:validation:MinLength=1
	// +kubebuilder:validation:MaxLength=256
	// +kubebuilder:validation:Required
	// +required
	Name string `json:"name"`

	// hostname is the hostname that should be used by the route.
	// +kubebuilder:validation:Required
	// +required
	Hostname Hostname `json:"hostname"`

	// servingCertKeyPairSecret is a reference to a secret of type `kubernetes.io/tls` in the openshift-config namespace.
	// The serving cert/key pair must match and will be used by the operator to fulfill the intent of serving with this name.
	// If the custom hostname uses the default routing suffix of the cluster,
	// the Secret specification for a serving certificate will not be needed.
	// +optional
	ServingCertKeyPairSecret SecretNameReference `json:"servingCertKeyPairSecret"`
}

// ComponentRouteStatus contains information allowing configuration of a route's hostname and serving certificate.
type ComponentRouteStatus struct {
	// namespace is the namespace of the route to customize. It must be a real namespace. Using an actual namespace
	// ensures that no two components will conflict and the same component can be installed multiple times.
	//
	// The namespace and name of this componentRoute must match a corresponding
	// entry in the list of spec.componentRoutes if the route is to be customized.
	// +kubebuilder:validation:Pattern=^[a-z0-9]([-a-z0-9]*[a-z0-9])?$
	// +kubebuilder:validation:MinLength=1
	// +kubebuilder:validation:MaxLength=63
	// +kubebuilder:validation:Required
	// +required
	Namespace string `json:"namespace"`

	// name is the logical name of the route to customize. It does not have to be the actual name of a route resource
	// but it cannot be renamed.
	//
	// The namespace and name of this componentRoute must match a corresponding
	// entry in the list of spec.componentRoutes if the route is to be customized.
	// +kubebuilder:validation:MinLength=1
	// +kubebuilder:validation:MaxLength=256
	// +kubebuilder:validation:Required
	// +required
	Name string `json:"name"`

	// defaultHostname is the hostname of this route prior to customization.
	// +kubebuilder:validation:Required
	// +required
	DefaultHostname Hostname `json:"defaultHostname"`

	// consumingUsers is a slice of ServiceAccounts that need to have read permission on the servingCertKeyPairSecret secret.
	// +kubebuilder:validation:MaxItems=5
	// +optional
	ConsumingUsers []ConsumingUser `json:"consumingUsers,omitempty"`

	// currentHostnames is the list of current names used by the route. Typically, this list should consist of a single
	// hostname, but if multiple hostnames are supported by the route the operator may write multiple entries to this list.
	// +kubebuilder:validation:MinItems=1
	// +optional
	CurrentHostnames []Hostname `json:"currentHostnames,omitempty"`

	// conditions are used to communicate the state of the componentRoutes entry.
	//
	// Supported conditions include Available, Degraded and Progressing.
	//
	// If available is true, the content served by the route can be accessed by users. This includes cases
	// where a default may continue to serve content while the customized route specified by the cluster-admin
	// is being configured.
	//
	// If Degraded is true, that means something has gone wrong trying to handle the componentRoutes entry.
	// The currentHostnames field may or may not be in effect.
	//
	// If Progressing is true, that means the component is taking some action related to the componentRoutes entry.
	// +optional
	Conditions []metav1.Condition `json:"conditions,omitempty"`

	// relatedObjects is a list of resources which are useful when debugging or inspecting how spec.componentRoutes is applied.
	// +kubebuilder:validation:MinItems=1
	// +kubebuilder:validation:Required
	// +required
	RelatedObjects []ObjectReference `json:"relatedObjects"`
}

// Compatibility level 1: Stable within a major release for a minimum of 12 months or 3 minor releases (whichever is longer).
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object
// +openshift:compatibility-gen:level=1
type IngressList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata"`

	Items []Ingress `json:"items"`
}
