/**
 * (C) Copyright IBM Corp. 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package catalogmanagementv1_test

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"time"

	"github.com/IBM/go-sdk-core/v5/core"
	"github.com/IBM/platform-services-go-sdk/catalogmanagementv1"
	"github.com/go-openapi/strfmt"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe(`CatalogManagementV1`, func() {
	var testServer *httptest.Server
	Describe(`Service constructor tests`, func() {
		It(`Instantiate service client`, func() {
			catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
			})
			Expect(catalogManagementService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
				URL: "{BAD_URL_STRING",
			})
			Expect(catalogManagementService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
				URL: "https://catalogmanagementv1/api",
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(catalogManagementService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"CATALOG_MANAGEMENT_URL": "https://catalogmanagementv1/api",
				"CATALOG_MANAGEMENT_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1UsingExternalConfig(&catalogmanagementv1.CatalogManagementV1Options{
				})
				Expect(catalogManagementService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := catalogManagementService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != catalogManagementService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(catalogManagementService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(catalogManagementService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1UsingExternalConfig(&catalogmanagementv1.CatalogManagementV1Options{
					URL: "https://testService/api",
				})
				Expect(catalogManagementService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := catalogManagementService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != catalogManagementService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(catalogManagementService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(catalogManagementService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1UsingExternalConfig(&catalogmanagementv1.CatalogManagementV1Options{
				})
				err := catalogManagementService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := catalogManagementService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != catalogManagementService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(catalogManagementService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(catalogManagementService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"CATALOG_MANAGEMENT_URL": "https://catalogmanagementv1/api",
				"CATALOG_MANAGEMENT_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1UsingExternalConfig(&catalogmanagementv1.CatalogManagementV1Options{
			})

			It(`Instantiate service client with error`, func() {
				Expect(catalogManagementService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"CATALOG_MANAGEMENT_AUTH_TYPE":   "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1UsingExternalConfig(&catalogmanagementv1.CatalogManagementV1Options{
				URL: "{BAD_URL_STRING",
			})

			It(`Instantiate service client with error`, func() {
				Expect(catalogManagementService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = catalogmanagementv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`GetCatalogAccount(getCatalogAccountOptions *GetCatalogAccountOptions) - Operation response error`, func() {
		getCatalogAccountPath := "/catalogaccount"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetCatalogAccount with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountOptions model
				getCatalogAccountOptionsModel := new(catalogmanagementv1.GetCatalogAccountOptions)
				getCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetCatalogAccount(getCatalogAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetCatalogAccount(getCatalogAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAccount(getCatalogAccountOptions *GetCatalogAccountOptions)`, func() {
		getCatalogAccountPath := "/catalogaccount"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "hide_IBM_cloud_catalog": false, "account_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}}`)
				}))
			})
			It(`Invoke GetCatalogAccount successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetCatalogAccountOptions model
				getCatalogAccountOptionsModel := new(catalogmanagementv1.GetCatalogAccountOptions)
				getCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetCatalogAccountWithContext(ctx, getCatalogAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetCatalogAccount(getCatalogAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetCatalogAccountWithContext(ctx, getCatalogAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "hide_IBM_cloud_catalog": false, "account_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}}`)
				}))
			})
			It(`Invoke GetCatalogAccount successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetCatalogAccount(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetCatalogAccountOptions model
				getCatalogAccountOptionsModel := new(catalogmanagementv1.GetCatalogAccountOptions)
				getCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetCatalogAccount(getCatalogAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetCatalogAccount with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountOptions model
				getCatalogAccountOptionsModel := new(catalogmanagementv1.GetCatalogAccountOptions)
				getCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetCatalogAccount(getCatalogAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetCatalogAccount successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountOptions model
				getCatalogAccountOptionsModel := new(catalogmanagementv1.GetCatalogAccountOptions)
				getCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetCatalogAccount(getCatalogAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateCatalogAccount(updateCatalogAccountOptions *UpdateCatalogAccountOptions)`, func() {
		updateCatalogAccountPath := "/catalogaccount"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateCatalogAccountPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateCatalogAccount successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.UpdateCatalogAccount(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the UpdateCatalogAccountOptions model
				updateCatalogAccountOptionsModel := new(catalogmanagementv1.UpdateCatalogAccountOptions)
				updateCatalogAccountOptionsModel.ID = core.StringPtr("testString")
				updateCatalogAccountOptionsModel.HideIBMCloudCatalog = core.BoolPtr(true)
				updateCatalogAccountOptionsModel.AccountFilters = filtersModel
				updateCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.UpdateCatalogAccount(updateCatalogAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke UpdateCatalogAccount with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the UpdateCatalogAccountOptions model
				updateCatalogAccountOptionsModel := new(catalogmanagementv1.UpdateCatalogAccountOptions)
				updateCatalogAccountOptionsModel.ID = core.StringPtr("testString")
				updateCatalogAccountOptionsModel.HideIBMCloudCatalog = core.BoolPtr(true)
				updateCatalogAccountOptionsModel.AccountFilters = filtersModel
				updateCatalogAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.UpdateCatalogAccount(updateCatalogAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAccountAudit(getCatalogAccountAuditOptions *GetCatalogAccountAuditOptions) - Operation response error`, func() {
		getCatalogAccountAuditPath := "/catalogaccount/audit"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountAuditPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetCatalogAccountAudit with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountAuditOptions model
				getCatalogAccountAuditOptionsModel := new(catalogmanagementv1.GetCatalogAccountAuditOptions)
				getCatalogAccountAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetCatalogAccountAudit(getCatalogAccountAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetCatalogAccountAudit(getCatalogAccountAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAccountAudit(getCatalogAccountAuditOptions *GetCatalogAccountAuditOptions)`, func() {
		getCatalogAccountAuditPath := "/catalogaccount/audit"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetCatalogAccountAudit successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetCatalogAccountAuditOptions model
				getCatalogAccountAuditOptionsModel := new(catalogmanagementv1.GetCatalogAccountAuditOptions)
				getCatalogAccountAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetCatalogAccountAuditWithContext(ctx, getCatalogAccountAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetCatalogAccountAudit(getCatalogAccountAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetCatalogAccountAuditWithContext(ctx, getCatalogAccountAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetCatalogAccountAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetCatalogAccountAudit(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetCatalogAccountAuditOptions model
				getCatalogAccountAuditOptionsModel := new(catalogmanagementv1.GetCatalogAccountAuditOptions)
				getCatalogAccountAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetCatalogAccountAudit(getCatalogAccountAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetCatalogAccountAudit with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountAuditOptions model
				getCatalogAccountAuditOptionsModel := new(catalogmanagementv1.GetCatalogAccountAuditOptions)
				getCatalogAccountAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetCatalogAccountAudit(getCatalogAccountAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetCatalogAccountAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountAuditOptions model
				getCatalogAccountAuditOptionsModel := new(catalogmanagementv1.GetCatalogAccountAuditOptions)
				getCatalogAccountAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetCatalogAccountAudit(getCatalogAccountAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAccountFilters(getCatalogAccountFiltersOptions *GetCatalogAccountFiltersOptions) - Operation response error`, func() {
		getCatalogAccountFiltersPath := "/catalogaccount/filters"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountFiltersPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["catalog"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetCatalogAccountFilters with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountFiltersOptions model
				getCatalogAccountFiltersOptionsModel := new(catalogmanagementv1.GetCatalogAccountFiltersOptions)
				getCatalogAccountFiltersOptionsModel.Catalog = core.StringPtr("testString")
				getCatalogAccountFiltersOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetCatalogAccountFilters(getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetCatalogAccountFilters(getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAccountFilters(getCatalogAccountFiltersOptions *GetCatalogAccountFiltersOptions)`, func() {
		getCatalogAccountFiltersPath := "/catalogaccount/filters"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountFiltersPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["catalog"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"account_filters": [{"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}], "catalog_filters": [{"catalog": {"id": "ID", "name": "Name"}, "filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}}]}`)
				}))
			})
			It(`Invoke GetCatalogAccountFilters successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetCatalogAccountFiltersOptions model
				getCatalogAccountFiltersOptionsModel := new(catalogmanagementv1.GetCatalogAccountFiltersOptions)
				getCatalogAccountFiltersOptionsModel.Catalog = core.StringPtr("testString")
				getCatalogAccountFiltersOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetCatalogAccountFiltersWithContext(ctx, getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetCatalogAccountFilters(getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetCatalogAccountFiltersWithContext(ctx, getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAccountFiltersPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["catalog"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"account_filters": [{"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}], "catalog_filters": [{"catalog": {"id": "ID", "name": "Name"}, "filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}}]}`)
				}))
			})
			It(`Invoke GetCatalogAccountFilters successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetCatalogAccountFilters(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetCatalogAccountFiltersOptions model
				getCatalogAccountFiltersOptionsModel := new(catalogmanagementv1.GetCatalogAccountFiltersOptions)
				getCatalogAccountFiltersOptionsModel.Catalog = core.StringPtr("testString")
				getCatalogAccountFiltersOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetCatalogAccountFilters(getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetCatalogAccountFilters with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountFiltersOptions model
				getCatalogAccountFiltersOptionsModel := new(catalogmanagementv1.GetCatalogAccountFiltersOptions)
				getCatalogAccountFiltersOptionsModel.Catalog = core.StringPtr("testString")
				getCatalogAccountFiltersOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetCatalogAccountFilters(getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetCatalogAccountFilters successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAccountFiltersOptions model
				getCatalogAccountFiltersOptionsModel := new(catalogmanagementv1.GetCatalogAccountFiltersOptions)
				getCatalogAccountFiltersOptionsModel.Catalog = core.StringPtr("testString")
				getCatalogAccountFiltersOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetCatalogAccountFilters(getCatalogAccountFiltersOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListCatalogs(listCatalogsOptions *ListCatalogsOptions) - Operation response error`, func() {
		listCatalogsPath := "/catalogs"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listCatalogsPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListCatalogs with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListCatalogsOptions model
				listCatalogsOptionsModel := new(catalogmanagementv1.ListCatalogsOptions)
				listCatalogsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ListCatalogs(listCatalogsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ListCatalogs(listCatalogsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListCatalogs(listCatalogsOptions *ListCatalogsOptions)`, func() {
		listCatalogsPath := "/catalogs"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listCatalogsPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"total_count": 10, "resources": [{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}]}`)
				}))
			})
			It(`Invoke ListCatalogs successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ListCatalogsOptions model
				listCatalogsOptionsModel := new(catalogmanagementv1.ListCatalogsOptions)
				listCatalogsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ListCatalogsWithContext(ctx, listCatalogsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ListCatalogs(listCatalogsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ListCatalogsWithContext(ctx, listCatalogsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listCatalogsPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"total_count": 10, "resources": [{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}]}`)
				}))
			})
			It(`Invoke ListCatalogs successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ListCatalogs(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListCatalogsOptions model
				listCatalogsOptionsModel := new(catalogmanagementv1.ListCatalogsOptions)
				listCatalogsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ListCatalogs(listCatalogsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListCatalogs with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListCatalogsOptions model
				listCatalogsOptionsModel := new(catalogmanagementv1.ListCatalogsOptions)
				listCatalogsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ListCatalogs(listCatalogsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListCatalogs successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListCatalogsOptions model
				listCatalogsOptionsModel := new(catalogmanagementv1.ListCatalogsOptions)
				listCatalogsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ListCatalogs(listCatalogsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateCatalog(createCatalogOptions *CreateCatalogOptions) - Operation response error`, func() {
		createCatalogPath := "/catalogs"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createCatalogPath))
					Expect(req.Method).To(Equal("POST"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateCatalog with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the CreateCatalogOptions model
				createCatalogOptionsModel := new(catalogmanagementv1.CreateCatalogOptions)
				createCatalogOptionsModel.ID = core.StringPtr("testString")
				createCatalogOptionsModel.Rev = core.StringPtr("testString")
				createCatalogOptionsModel.Label = core.StringPtr("testString")
				createCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				createCatalogOptionsModel.Tags = []string{"testString"}
				createCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createCatalogOptionsModel.Disabled = core.BoolPtr(true)
				createCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogFilters = filtersModel
				createCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				createCatalogOptionsModel.Kind = core.StringPtr("testString")
				createCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.CreateCatalog(createCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.CreateCatalog(createCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateCatalog(createCatalogOptions *CreateCatalogOptions)`, func() {
		createCatalogPath := "/catalogs"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createCatalogPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}`)
				}))
			})
			It(`Invoke CreateCatalog successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the CreateCatalogOptions model
				createCatalogOptionsModel := new(catalogmanagementv1.CreateCatalogOptions)
				createCatalogOptionsModel.ID = core.StringPtr("testString")
				createCatalogOptionsModel.Rev = core.StringPtr("testString")
				createCatalogOptionsModel.Label = core.StringPtr("testString")
				createCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				createCatalogOptionsModel.Tags = []string{"testString"}
				createCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createCatalogOptionsModel.Disabled = core.BoolPtr(true)
				createCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogFilters = filtersModel
				createCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				createCatalogOptionsModel.Kind = core.StringPtr("testString")
				createCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.CreateCatalogWithContext(ctx, createCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.CreateCatalog(createCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.CreateCatalogWithContext(ctx, createCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createCatalogPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}`)
				}))
			})
			It(`Invoke CreateCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.CreateCatalog(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the CreateCatalogOptions model
				createCatalogOptionsModel := new(catalogmanagementv1.CreateCatalogOptions)
				createCatalogOptionsModel.ID = core.StringPtr("testString")
				createCatalogOptionsModel.Rev = core.StringPtr("testString")
				createCatalogOptionsModel.Label = core.StringPtr("testString")
				createCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				createCatalogOptionsModel.Tags = []string{"testString"}
				createCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createCatalogOptionsModel.Disabled = core.BoolPtr(true)
				createCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogFilters = filtersModel
				createCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				createCatalogOptionsModel.Kind = core.StringPtr("testString")
				createCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.CreateCatalog(createCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateCatalog with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the CreateCatalogOptions model
				createCatalogOptionsModel := new(catalogmanagementv1.CreateCatalogOptions)
				createCatalogOptionsModel.ID = core.StringPtr("testString")
				createCatalogOptionsModel.Rev = core.StringPtr("testString")
				createCatalogOptionsModel.Label = core.StringPtr("testString")
				createCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				createCatalogOptionsModel.Tags = []string{"testString"}
				createCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createCatalogOptionsModel.Disabled = core.BoolPtr(true)
				createCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogFilters = filtersModel
				createCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				createCatalogOptionsModel.Kind = core.StringPtr("testString")
				createCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.CreateCatalog(createCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the CreateCatalogOptions model
				createCatalogOptionsModel := new(catalogmanagementv1.CreateCatalogOptions)
				createCatalogOptionsModel.ID = core.StringPtr("testString")
				createCatalogOptionsModel.Rev = core.StringPtr("testString")
				createCatalogOptionsModel.Label = core.StringPtr("testString")
				createCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				createCatalogOptionsModel.Tags = []string{"testString"}
				createCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createCatalogOptionsModel.Disabled = core.BoolPtr(true)
				createCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				createCatalogOptionsModel.CatalogFilters = filtersModel
				createCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				createCatalogOptionsModel.Kind = core.StringPtr("testString")
				createCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.CreateCatalog(createCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalog(getCatalogOptions *GetCatalogOptions) - Operation response error`, func() {
		getCatalogPath := "/catalogs/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetCatalog with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogOptions model
				getCatalogOptionsModel := new(catalogmanagementv1.GetCatalogOptions)
				getCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetCatalog(getCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetCatalog(getCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalog(getCatalogOptions *GetCatalogOptions)`, func() {
		getCatalogPath := "/catalogs/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}`)
				}))
			})
			It(`Invoke GetCatalog successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetCatalogOptions model
				getCatalogOptionsModel := new(catalogmanagementv1.GetCatalogOptions)
				getCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetCatalogWithContext(ctx, getCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetCatalog(getCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetCatalogWithContext(ctx, getCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}`)
				}))
			})
			It(`Invoke GetCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetCatalog(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetCatalogOptions model
				getCatalogOptionsModel := new(catalogmanagementv1.GetCatalogOptions)
				getCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetCatalog(getCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetCatalog with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogOptions model
				getCatalogOptionsModel := new(catalogmanagementv1.GetCatalogOptions)
				getCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetCatalog(getCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetCatalogOptions model with no property values
				getCatalogOptionsModelNew := new(catalogmanagementv1.GetCatalogOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetCatalog(getCatalogOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogOptions model
				getCatalogOptionsModel := new(catalogmanagementv1.GetCatalogOptions)
				getCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetCatalog(getCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceCatalog(replaceCatalogOptions *ReplaceCatalogOptions) - Operation response error`, func() {
		replaceCatalogPath := "/catalogs/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceCatalogPath))
					Expect(req.Method).To(Equal("PUT"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceCatalog with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the ReplaceCatalogOptions model
				replaceCatalogOptionsModel := new(catalogmanagementv1.ReplaceCatalogOptions)
				replaceCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceCatalogOptionsModel.ID = core.StringPtr("testString")
				replaceCatalogOptionsModel.Rev = core.StringPtr("testString")
				replaceCatalogOptionsModel.Label = core.StringPtr("testString")
				replaceCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				replaceCatalogOptionsModel.Tags = []string{"testString"}
				replaceCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceCatalogOptionsModel.Disabled = core.BoolPtr(true)
				replaceCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				replaceCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogFilters = filtersModel
				replaceCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				replaceCatalogOptionsModel.Kind = core.StringPtr("testString")
				replaceCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceCatalog(replaceCatalogOptions *ReplaceCatalogOptions)`, func() {
		replaceCatalogPath := "/catalogs/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceCatalogPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}`)
				}))
			})
			It(`Invoke ReplaceCatalog successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the ReplaceCatalogOptions model
				replaceCatalogOptionsModel := new(catalogmanagementv1.ReplaceCatalogOptions)
				replaceCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceCatalogOptionsModel.ID = core.StringPtr("testString")
				replaceCatalogOptionsModel.Rev = core.StringPtr("testString")
				replaceCatalogOptionsModel.Label = core.StringPtr("testString")
				replaceCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				replaceCatalogOptionsModel.Tags = []string{"testString"}
				replaceCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceCatalogOptionsModel.Disabled = core.BoolPtr(true)
				replaceCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				replaceCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogFilters = filtersModel
				replaceCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				replaceCatalogOptionsModel.Kind = core.StringPtr("testString")
				replaceCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ReplaceCatalogWithContext(ctx, replaceCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ReplaceCatalogWithContext(ctx, replaceCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceCatalogPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "label": "Label", "short_description": "ShortDescription", "catalog_icon_url": "CatalogIconURL", "tags": ["Tags"], "url": "URL", "crn": "CRN", "offerings_url": "OfferingsURL", "features": [{"title": "Title", "description": "Description"}], "disabled": true, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "resource_group_id": "ResourceGroupID", "owning_account": "OwningAccount", "catalog_filters": {"include_all": true, "category_filters": {"mapKey": {"include": false, "filter": {"filter_terms": ["FilterTerms"]}}}, "id_filters": {"include": {"filter_terms": ["FilterTerms"]}, "exclude": {"filter_terms": ["FilterTerms"]}}}, "syndication_settings": {"remove_related_components": false, "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "history": {"namespaces": ["Namespaces"], "clusters": [{"region": "Region", "id": "ID", "name": "Name", "resource_group_name": "ResourceGroupName", "type": "Type", "namespaces": ["Namespaces"], "all_namespaces": false}], "last_run": "2019-01-01T12:00:00.000Z"}, "authorization": {"token": "Token", "last_run": "2019-01-01T12:00:00.000Z"}}, "kind": "Kind"}`)
				}))
			})
			It(`Invoke ReplaceCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ReplaceCatalog(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the ReplaceCatalogOptions model
				replaceCatalogOptionsModel := new(catalogmanagementv1.ReplaceCatalogOptions)
				replaceCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceCatalogOptionsModel.ID = core.StringPtr("testString")
				replaceCatalogOptionsModel.Rev = core.StringPtr("testString")
				replaceCatalogOptionsModel.Label = core.StringPtr("testString")
				replaceCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				replaceCatalogOptionsModel.Tags = []string{"testString"}
				replaceCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceCatalogOptionsModel.Disabled = core.BoolPtr(true)
				replaceCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				replaceCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogFilters = filtersModel
				replaceCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				replaceCatalogOptionsModel.Kind = core.StringPtr("testString")
				replaceCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceCatalog with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the ReplaceCatalogOptions model
				replaceCatalogOptionsModel := new(catalogmanagementv1.ReplaceCatalogOptions)
				replaceCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceCatalogOptionsModel.ID = core.StringPtr("testString")
				replaceCatalogOptionsModel.Rev = core.StringPtr("testString")
				replaceCatalogOptionsModel.Label = core.StringPtr("testString")
				replaceCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				replaceCatalogOptionsModel.Tags = []string{"testString"}
				replaceCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceCatalogOptionsModel.Disabled = core.BoolPtr(true)
				replaceCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				replaceCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogFilters = filtersModel
				replaceCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				replaceCatalogOptionsModel.Kind = core.StringPtr("testString")
				replaceCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceCatalogOptions model with no property values
				replaceCatalogOptionsModelNew := new(catalogmanagementv1.ReplaceCatalogOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ReplaceCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				filterTermsModel.FilterTerms = []string{"testString"}

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel

				// Construct an instance of the ReplaceCatalogOptions model
				replaceCatalogOptionsModel := new(catalogmanagementv1.ReplaceCatalogOptions)
				replaceCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceCatalogOptionsModel.ID = core.StringPtr("testString")
				replaceCatalogOptionsModel.Rev = core.StringPtr("testString")
				replaceCatalogOptionsModel.Label = core.StringPtr("testString")
				replaceCatalogOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogIconURL = core.StringPtr("testString")
				replaceCatalogOptionsModel.Tags = []string{"testString"}
				replaceCatalogOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceCatalogOptionsModel.Disabled = core.BoolPtr(true)
				replaceCatalogOptionsModel.ResourceGroupID = core.StringPtr("testString")
				replaceCatalogOptionsModel.OwningAccount = core.StringPtr("testString")
				replaceCatalogOptionsModel.CatalogFilters = filtersModel
				replaceCatalogOptionsModel.SyndicationSettings = syndicationResourceModel
				replaceCatalogOptionsModel.Kind = core.StringPtr("testString")
				replaceCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ReplaceCatalog(replaceCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteCatalog(deleteCatalogOptions *DeleteCatalogOptions)`, func() {
		deleteCatalogPath := "/catalogs/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteCatalogPath))
					Expect(req.Method).To(Equal("DELETE"))

					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteCatalog successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteCatalog(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteCatalogOptions model
				deleteCatalogOptionsModel := new(catalogmanagementv1.DeleteCatalogOptions)
				deleteCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteCatalog(deleteCatalogOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteCatalog with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteCatalogOptions model
				deleteCatalogOptionsModel := new(catalogmanagementv1.DeleteCatalogOptions)
				deleteCatalogOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteCatalogOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteCatalog(deleteCatalogOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteCatalogOptions model with no property values
				deleteCatalogOptionsModelNew := new(catalogmanagementv1.DeleteCatalogOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteCatalog(deleteCatalogOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAudit(getCatalogAuditOptions *GetCatalogAuditOptions) - Operation response error`, func() {
		getCatalogAuditPath := "/catalogs/testString/audit"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAuditPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetCatalogAudit with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAuditOptions model
				getCatalogAuditOptionsModel := new(catalogmanagementv1.GetCatalogAuditOptions)
				getCatalogAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCatalogAudit(getCatalogAuditOptions *GetCatalogAuditOptions)`, func() {
		getCatalogAuditPath := "/catalogs/testString/audit"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetCatalogAudit successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetCatalogAuditOptions model
				getCatalogAuditOptionsModel := new(catalogmanagementv1.GetCatalogAuditOptions)
				getCatalogAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetCatalogAuditWithContext(ctx, getCatalogAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetCatalogAuditWithContext(ctx, getCatalogAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getCatalogAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetCatalogAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetCatalogAudit(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetCatalogAuditOptions model
				getCatalogAuditOptionsModel := new(catalogmanagementv1.GetCatalogAuditOptions)
				getCatalogAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetCatalogAudit with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAuditOptions model
				getCatalogAuditOptionsModel := new(catalogmanagementv1.GetCatalogAuditOptions)
				getCatalogAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetCatalogAuditOptions model with no property values
				getCatalogAuditOptionsModelNew := new(catalogmanagementv1.GetCatalogAuditOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetCatalogAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetCatalogAuditOptions model
				getCatalogAuditOptionsModel := new(catalogmanagementv1.GetCatalogAuditOptions)
				getCatalogAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getCatalogAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetCatalogAudit(getCatalogAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetConsumptionOfferings(getConsumptionOfferingsOptions *GetConsumptionOfferingsOptions) - Operation response error`, func() {
		getConsumptionOfferingsPath := "/offerings"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getConsumptionOfferingsPath))
					Expect(req.Method).To(Equal("GET"))
					// TODO: Add check for digest query parameter
					Expect(req.URL.Query()["catalog"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["select"]).To(Equal([]string{"all"}))
					// TODO: Add check for includeHidden query parameter
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetConsumptionOfferings with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetConsumptionOfferingsOptions model
				getConsumptionOfferingsOptionsModel := new(catalogmanagementv1.GetConsumptionOfferingsOptions)
				getConsumptionOfferingsOptionsModel.Digest = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Catalog = core.StringPtr("testString")
				getConsumptionOfferingsOptionsModel.Select = core.StringPtr("all")
				getConsumptionOfferingsOptionsModel.IncludeHidden = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getConsumptionOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				getConsumptionOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetConsumptionOfferings(getConsumptionOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetConsumptionOfferings(getConsumptionOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetConsumptionOfferings(getConsumptionOfferingsOptions *GetConsumptionOfferingsOptions)`, func() {
		getConsumptionOfferingsPath := "/offerings"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getConsumptionOfferingsPath))
					Expect(req.Method).To(Equal("GET"))

					// TODO: Add check for digest query parameter
					Expect(req.URL.Query()["catalog"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["select"]).To(Equal([]string{"all"}))
					// TODO: Add check for includeHidden query parameter
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}]}`)
				}))
			})
			It(`Invoke GetConsumptionOfferings successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetConsumptionOfferingsOptions model
				getConsumptionOfferingsOptionsModel := new(catalogmanagementv1.GetConsumptionOfferingsOptions)
				getConsumptionOfferingsOptionsModel.Digest = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Catalog = core.StringPtr("testString")
				getConsumptionOfferingsOptionsModel.Select = core.StringPtr("all")
				getConsumptionOfferingsOptionsModel.IncludeHidden = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getConsumptionOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				getConsumptionOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetConsumptionOfferingsWithContext(ctx, getConsumptionOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetConsumptionOfferings(getConsumptionOfferingsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetConsumptionOfferingsWithContext(ctx, getConsumptionOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getConsumptionOfferingsPath))
					Expect(req.Method).To(Equal("GET"))

					// TODO: Add check for digest query parameter
					Expect(req.URL.Query()["catalog"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["select"]).To(Equal([]string{"all"}))
					// TODO: Add check for includeHidden query parameter
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}]}`)
				}))
			})
			It(`Invoke GetConsumptionOfferings successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetConsumptionOfferings(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetConsumptionOfferingsOptions model
				getConsumptionOfferingsOptionsModel := new(catalogmanagementv1.GetConsumptionOfferingsOptions)
				getConsumptionOfferingsOptionsModel.Digest = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Catalog = core.StringPtr("testString")
				getConsumptionOfferingsOptionsModel.Select = core.StringPtr("all")
				getConsumptionOfferingsOptionsModel.IncludeHidden = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getConsumptionOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				getConsumptionOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetConsumptionOfferings(getConsumptionOfferingsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetConsumptionOfferings with error: Operation request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetConsumptionOfferingsOptions model
				getConsumptionOfferingsOptionsModel := new(catalogmanagementv1.GetConsumptionOfferingsOptions)
				getConsumptionOfferingsOptionsModel.Digest = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Catalog = core.StringPtr("testString")
				getConsumptionOfferingsOptionsModel.Select = core.StringPtr("all")
				getConsumptionOfferingsOptionsModel.IncludeHidden = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getConsumptionOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				getConsumptionOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetConsumptionOfferings(getConsumptionOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetConsumptionOfferings successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetConsumptionOfferingsOptions model
				getConsumptionOfferingsOptionsModel := new(catalogmanagementv1.GetConsumptionOfferingsOptions)
				getConsumptionOfferingsOptionsModel.Digest = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Catalog = core.StringPtr("testString")
				getConsumptionOfferingsOptionsModel.Select = core.StringPtr("all")
				getConsumptionOfferingsOptionsModel.IncludeHidden = core.BoolPtr(true)
				getConsumptionOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getConsumptionOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				getConsumptionOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetConsumptionOfferings(getConsumptionOfferingsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOfferings(listOfferingsOptions *ListOfferingsOptions) - Operation response error`, func() {
		listOfferingsPath := "/catalogs/testString/offerings"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOfferingsPath))
					Expect(req.Method).To(Equal("GET"))
					// TODO: Add check for digest query parameter
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListOfferings with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListOfferingsOptions model
				listOfferingsOptionsModel := new(catalogmanagementv1.ListOfferingsOptions)
				listOfferingsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listOfferingsOptionsModel.Digest = core.BoolPtr(true)
				listOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listOfferingsOptionsModel.Name = core.StringPtr("testString")
				listOfferingsOptionsModel.Sort = core.StringPtr("testString")
				listOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ListOfferings(listOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ListOfferings(listOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOfferings(listOfferingsOptions *ListOfferingsOptions)`, func() {
		listOfferingsPath := "/catalogs/testString/offerings"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOfferingsPath))
					Expect(req.Method).To(Equal("GET"))

					// TODO: Add check for digest query parameter
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}]}`)
				}))
			})
			It(`Invoke ListOfferings successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ListOfferingsOptions model
				listOfferingsOptionsModel := new(catalogmanagementv1.ListOfferingsOptions)
				listOfferingsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listOfferingsOptionsModel.Digest = core.BoolPtr(true)
				listOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listOfferingsOptionsModel.Name = core.StringPtr("testString")
				listOfferingsOptionsModel.Sort = core.StringPtr("testString")
				listOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ListOfferingsWithContext(ctx, listOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ListOfferings(listOfferingsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ListOfferingsWithContext(ctx, listOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOfferingsPath))
					Expect(req.Method).To(Equal("GET"))

					// TODO: Add check for digest query parameter
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}]}`)
				}))
			})
			It(`Invoke ListOfferings successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ListOfferings(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListOfferingsOptions model
				listOfferingsOptionsModel := new(catalogmanagementv1.ListOfferingsOptions)
				listOfferingsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listOfferingsOptionsModel.Digest = core.BoolPtr(true)
				listOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listOfferingsOptionsModel.Name = core.StringPtr("testString")
				listOfferingsOptionsModel.Sort = core.StringPtr("testString")
				listOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ListOfferings(listOfferingsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListOfferings with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListOfferingsOptions model
				listOfferingsOptionsModel := new(catalogmanagementv1.ListOfferingsOptions)
				listOfferingsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listOfferingsOptionsModel.Digest = core.BoolPtr(true)
				listOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listOfferingsOptionsModel.Name = core.StringPtr("testString")
				listOfferingsOptionsModel.Sort = core.StringPtr("testString")
				listOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ListOfferings(listOfferingsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListOfferingsOptions model with no property values
				listOfferingsOptionsModelNew := new(catalogmanagementv1.ListOfferingsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ListOfferings(listOfferingsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListOfferings successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListOfferingsOptions model
				listOfferingsOptionsModel := new(catalogmanagementv1.ListOfferingsOptions)
				listOfferingsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listOfferingsOptionsModel.Digest = core.BoolPtr(true)
				listOfferingsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listOfferingsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listOfferingsOptionsModel.Name = core.StringPtr("testString")
				listOfferingsOptionsModel.Sort = core.StringPtr("testString")
				listOfferingsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ListOfferings(listOfferingsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateOffering(createOfferingOptions *CreateOfferingOptions) - Operation response error`, func() {
		createOfferingPath := "/catalogs/testString/offerings"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createOfferingPath))
					Expect(req.Method).To(Equal("POST"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateOffering with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the CreateOfferingOptions model
				createOfferingOptionsModel := new(catalogmanagementv1.CreateOfferingOptions)
				createOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createOfferingOptionsModel.ID = core.StringPtr("testString")
				createOfferingOptionsModel.Rev = core.StringPtr("testString")
				createOfferingOptionsModel.URL = core.StringPtr("testString")
				createOfferingOptionsModel.CRN = core.StringPtr("testString")
				createOfferingOptionsModel.Label = core.StringPtr("testString")
				createOfferingOptionsModel.Name = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				createOfferingOptionsModel.Tags = []string{"testString"}
				createOfferingOptionsModel.Keywords = []string{"testString"}
				createOfferingOptionsModel.Rating = ratingModel
				createOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				createOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				createOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				createOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				createOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				createOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				createOfferingOptionsModel.Metadata = make(map[string]interface{})
				createOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				createOfferingOptionsModel.Hidden = core.BoolPtr(true)
				createOfferingOptionsModel.Provider = core.StringPtr("testString")
				createOfferingOptionsModel.RepoInfo = repoInfoModel
				createOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.CreateOffering(createOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.CreateOffering(createOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateOffering(createOfferingOptions *CreateOfferingOptions)`, func() {
		createOfferingPath := "/catalogs/testString/offerings"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createOfferingPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke CreateOffering successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the CreateOfferingOptions model
				createOfferingOptionsModel := new(catalogmanagementv1.CreateOfferingOptions)
				createOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createOfferingOptionsModel.ID = core.StringPtr("testString")
				createOfferingOptionsModel.Rev = core.StringPtr("testString")
				createOfferingOptionsModel.URL = core.StringPtr("testString")
				createOfferingOptionsModel.CRN = core.StringPtr("testString")
				createOfferingOptionsModel.Label = core.StringPtr("testString")
				createOfferingOptionsModel.Name = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				createOfferingOptionsModel.Tags = []string{"testString"}
				createOfferingOptionsModel.Keywords = []string{"testString"}
				createOfferingOptionsModel.Rating = ratingModel
				createOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				createOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				createOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				createOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				createOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				createOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				createOfferingOptionsModel.Metadata = make(map[string]interface{})
				createOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				createOfferingOptionsModel.Hidden = core.BoolPtr(true)
				createOfferingOptionsModel.Provider = core.StringPtr("testString")
				createOfferingOptionsModel.RepoInfo = repoInfoModel
				createOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.CreateOfferingWithContext(ctx, createOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.CreateOffering(createOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.CreateOfferingWithContext(ctx, createOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createOfferingPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke CreateOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.CreateOffering(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the CreateOfferingOptions model
				createOfferingOptionsModel := new(catalogmanagementv1.CreateOfferingOptions)
				createOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createOfferingOptionsModel.ID = core.StringPtr("testString")
				createOfferingOptionsModel.Rev = core.StringPtr("testString")
				createOfferingOptionsModel.URL = core.StringPtr("testString")
				createOfferingOptionsModel.CRN = core.StringPtr("testString")
				createOfferingOptionsModel.Label = core.StringPtr("testString")
				createOfferingOptionsModel.Name = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				createOfferingOptionsModel.Tags = []string{"testString"}
				createOfferingOptionsModel.Keywords = []string{"testString"}
				createOfferingOptionsModel.Rating = ratingModel
				createOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				createOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				createOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				createOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				createOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				createOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				createOfferingOptionsModel.Metadata = make(map[string]interface{})
				createOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				createOfferingOptionsModel.Hidden = core.BoolPtr(true)
				createOfferingOptionsModel.Provider = core.StringPtr("testString")
				createOfferingOptionsModel.RepoInfo = repoInfoModel
				createOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.CreateOffering(createOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateOffering with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the CreateOfferingOptions model
				createOfferingOptionsModel := new(catalogmanagementv1.CreateOfferingOptions)
				createOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createOfferingOptionsModel.ID = core.StringPtr("testString")
				createOfferingOptionsModel.Rev = core.StringPtr("testString")
				createOfferingOptionsModel.URL = core.StringPtr("testString")
				createOfferingOptionsModel.CRN = core.StringPtr("testString")
				createOfferingOptionsModel.Label = core.StringPtr("testString")
				createOfferingOptionsModel.Name = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				createOfferingOptionsModel.Tags = []string{"testString"}
				createOfferingOptionsModel.Keywords = []string{"testString"}
				createOfferingOptionsModel.Rating = ratingModel
				createOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				createOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				createOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				createOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				createOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				createOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				createOfferingOptionsModel.Metadata = make(map[string]interface{})
				createOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				createOfferingOptionsModel.Hidden = core.BoolPtr(true)
				createOfferingOptionsModel.Provider = core.StringPtr("testString")
				createOfferingOptionsModel.RepoInfo = repoInfoModel
				createOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.CreateOffering(createOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateOfferingOptions model with no property values
				createOfferingOptionsModelNew := new(catalogmanagementv1.CreateOfferingOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.CreateOffering(createOfferingOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the CreateOfferingOptions model
				createOfferingOptionsModel := new(catalogmanagementv1.CreateOfferingOptions)
				createOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createOfferingOptionsModel.ID = core.StringPtr("testString")
				createOfferingOptionsModel.Rev = core.StringPtr("testString")
				createOfferingOptionsModel.URL = core.StringPtr("testString")
				createOfferingOptionsModel.CRN = core.StringPtr("testString")
				createOfferingOptionsModel.Label = core.StringPtr("testString")
				createOfferingOptionsModel.Name = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				createOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				createOfferingOptionsModel.Tags = []string{"testString"}
				createOfferingOptionsModel.Keywords = []string{"testString"}
				createOfferingOptionsModel.Rating = ratingModel
				createOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				createOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				createOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				createOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				createOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				createOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				createOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				createOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				createOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				createOfferingOptionsModel.Metadata = make(map[string]interface{})
				createOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				createOfferingOptionsModel.Hidden = core.BoolPtr(true)
				createOfferingOptionsModel.Provider = core.StringPtr("testString")
				createOfferingOptionsModel.RepoInfo = repoInfoModel
				createOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.CreateOffering(createOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ImportOfferingVersion(importOfferingVersionOptions *ImportOfferingVersionOptions) - Operation response error`, func() {
		importOfferingVersionPath := "/catalogs/testString/offerings/testString/version"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(importOfferingVersionPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					// TODO: Add check for includeConfig query parameter
					// TODO: Add check for isVSI query parameter
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ImportOfferingVersion with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ImportOfferingVersionOptions model
				importOfferingVersionOptionsModel := new(catalogmanagementv1.ImportOfferingVersionOptions)
				importOfferingVersionOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingVersionOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Tags = []string{"testString"}
				importOfferingVersionOptionsModel.TargetKinds = []string{"testString"}
				importOfferingVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingVersionOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingVersionOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingVersionOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingVersionOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingVersionOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ImportOfferingVersion(importOfferingVersionOptions *ImportOfferingVersionOptions)`, func() {
		importOfferingVersionPath := "/catalogs/testString/offerings/testString/version"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(importOfferingVersionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					// TODO: Add check for includeConfig query parameter
					// TODO: Add check for isVSI query parameter
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ImportOfferingVersion successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ImportOfferingVersionOptions model
				importOfferingVersionOptionsModel := new(catalogmanagementv1.ImportOfferingVersionOptions)
				importOfferingVersionOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingVersionOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Tags = []string{"testString"}
				importOfferingVersionOptionsModel.TargetKinds = []string{"testString"}
				importOfferingVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingVersionOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingVersionOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingVersionOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingVersionOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingVersionOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ImportOfferingVersionWithContext(ctx, importOfferingVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ImportOfferingVersionWithContext(ctx, importOfferingVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(importOfferingVersionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					// TODO: Add check for includeConfig query parameter
					// TODO: Add check for isVSI query parameter
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ImportOfferingVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ImportOfferingVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ImportOfferingVersionOptions model
				importOfferingVersionOptionsModel := new(catalogmanagementv1.ImportOfferingVersionOptions)
				importOfferingVersionOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingVersionOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Tags = []string{"testString"}
				importOfferingVersionOptionsModel.TargetKinds = []string{"testString"}
				importOfferingVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingVersionOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingVersionOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingVersionOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingVersionOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingVersionOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ImportOfferingVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ImportOfferingVersionOptions model
				importOfferingVersionOptionsModel := new(catalogmanagementv1.ImportOfferingVersionOptions)
				importOfferingVersionOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingVersionOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Tags = []string{"testString"}
				importOfferingVersionOptionsModel.TargetKinds = []string{"testString"}
				importOfferingVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingVersionOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingVersionOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingVersionOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingVersionOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingVersionOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ImportOfferingVersionOptions model with no property values
				importOfferingVersionOptionsModelNew := new(catalogmanagementv1.ImportOfferingVersionOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke ImportOfferingVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ImportOfferingVersionOptions model
				importOfferingVersionOptionsModel := new(catalogmanagementv1.ImportOfferingVersionOptions)
				importOfferingVersionOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingVersionOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Tags = []string{"testString"}
				importOfferingVersionOptionsModel.TargetKinds = []string{"testString"}
				importOfferingVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingVersionOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingVersionOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingVersionOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingVersionOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingVersionOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ImportOfferingVersion(importOfferingVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ImportOffering(importOfferingOptions *ImportOfferingOptions) - Operation response error`, func() {
		importOfferingPath := "/catalogs/testString/import/offerings"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(importOfferingPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["X-Auth-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["offeringID"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					// TODO: Add check for includeConfig query parameter
					// TODO: Add check for isVSI query parameter
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ImportOffering with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ImportOfferingOptions model
				importOfferingOptionsModel := new(catalogmanagementv1.ImportOfferingOptions)
				importOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingOptionsModel.Tags = []string{"testString"}
				importOfferingOptionsModel.TargetKinds = []string{"testString"}
				importOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingOptionsModel.XAuthToken = core.StringPtr("testString")
				importOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ImportOffering(importOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ImportOffering(importOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ImportOffering(importOfferingOptions *ImportOfferingOptions)`, func() {
		importOfferingPath := "/catalogs/testString/import/offerings"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(importOfferingPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["offeringID"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					// TODO: Add check for includeConfig query parameter
					// TODO: Add check for isVSI query parameter
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ImportOffering successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ImportOfferingOptions model
				importOfferingOptionsModel := new(catalogmanagementv1.ImportOfferingOptions)
				importOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingOptionsModel.Tags = []string{"testString"}
				importOfferingOptionsModel.TargetKinds = []string{"testString"}
				importOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingOptionsModel.XAuthToken = core.StringPtr("testString")
				importOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ImportOfferingWithContext(ctx, importOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ImportOffering(importOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ImportOfferingWithContext(ctx, importOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(importOfferingPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["offeringID"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					// TODO: Add check for includeConfig query parameter
					// TODO: Add check for isVSI query parameter
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ImportOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ImportOffering(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ImportOfferingOptions model
				importOfferingOptionsModel := new(catalogmanagementv1.ImportOfferingOptions)
				importOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingOptionsModel.Tags = []string{"testString"}
				importOfferingOptionsModel.TargetKinds = []string{"testString"}
				importOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingOptionsModel.XAuthToken = core.StringPtr("testString")
				importOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ImportOffering(importOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ImportOffering with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ImportOfferingOptions model
				importOfferingOptionsModel := new(catalogmanagementv1.ImportOfferingOptions)
				importOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingOptionsModel.Tags = []string{"testString"}
				importOfferingOptionsModel.TargetKinds = []string{"testString"}
				importOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingOptionsModel.XAuthToken = core.StringPtr("testString")
				importOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ImportOffering(importOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ImportOfferingOptions model with no property values
				importOfferingOptionsModelNew := new(catalogmanagementv1.ImportOfferingOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ImportOffering(importOfferingOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke ImportOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ImportOfferingOptions model
				importOfferingOptionsModel := new(catalogmanagementv1.ImportOfferingOptions)
				importOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				importOfferingOptionsModel.Tags = []string{"testString"}
				importOfferingOptionsModel.TargetKinds = []string{"testString"}
				importOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				importOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				importOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				importOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				importOfferingOptionsModel.IncludeConfig = core.BoolPtr(true)
				importOfferingOptionsModel.IsVsi = core.BoolPtr(true)
				importOfferingOptionsModel.RepoType = core.StringPtr("testString")
				importOfferingOptionsModel.XAuthToken = core.StringPtr("testString")
				importOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ImportOffering(importOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReloadOffering(reloadOfferingOptions *ReloadOfferingOptions) - Operation response error`, func() {
		reloadOfferingPath := "/catalogs/testString/offerings/testString/reload"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(reloadOfferingPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReloadOffering with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReloadOfferingOptions model
				reloadOfferingOptionsModel := new(catalogmanagementv1.ReloadOfferingOptions)
				reloadOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				reloadOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				reloadOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				reloadOfferingOptionsModel.Tags = []string{"testString"}
				reloadOfferingOptionsModel.TargetKinds = []string{"testString"}
				reloadOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				reloadOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				reloadOfferingOptionsModel.RepoType = core.StringPtr("testString")
				reloadOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ReloadOffering(reloadOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ReloadOffering(reloadOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReloadOffering(reloadOfferingOptions *ReloadOfferingOptions)`, func() {
		reloadOfferingPath := "/catalogs/testString/offerings/testString/reload"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(reloadOfferingPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ReloadOffering successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ReloadOfferingOptions model
				reloadOfferingOptionsModel := new(catalogmanagementv1.ReloadOfferingOptions)
				reloadOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				reloadOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				reloadOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				reloadOfferingOptionsModel.Tags = []string{"testString"}
				reloadOfferingOptionsModel.TargetKinds = []string{"testString"}
				reloadOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				reloadOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				reloadOfferingOptionsModel.RepoType = core.StringPtr("testString")
				reloadOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ReloadOfferingWithContext(ctx, reloadOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ReloadOffering(reloadOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ReloadOfferingWithContext(ctx, reloadOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(reloadOfferingPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["targetVersion"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["zipurl"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["repoType"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ReloadOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ReloadOffering(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ReloadOfferingOptions model
				reloadOfferingOptionsModel := new(catalogmanagementv1.ReloadOfferingOptions)
				reloadOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				reloadOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				reloadOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				reloadOfferingOptionsModel.Tags = []string{"testString"}
				reloadOfferingOptionsModel.TargetKinds = []string{"testString"}
				reloadOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				reloadOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				reloadOfferingOptionsModel.RepoType = core.StringPtr("testString")
				reloadOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ReloadOffering(reloadOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReloadOffering with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReloadOfferingOptions model
				reloadOfferingOptionsModel := new(catalogmanagementv1.ReloadOfferingOptions)
				reloadOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				reloadOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				reloadOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				reloadOfferingOptionsModel.Tags = []string{"testString"}
				reloadOfferingOptionsModel.TargetKinds = []string{"testString"}
				reloadOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				reloadOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				reloadOfferingOptionsModel.RepoType = core.StringPtr("testString")
				reloadOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ReloadOffering(reloadOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReloadOfferingOptions model with no property values
				reloadOfferingOptionsModelNew := new(catalogmanagementv1.ReloadOfferingOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ReloadOffering(reloadOfferingOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke ReloadOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReloadOfferingOptions model
				reloadOfferingOptionsModel := new(catalogmanagementv1.ReloadOfferingOptions)
				reloadOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				reloadOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				reloadOfferingOptionsModel.TargetVersion = core.StringPtr("testString")
				reloadOfferingOptionsModel.Tags = []string{"testString"}
				reloadOfferingOptionsModel.TargetKinds = []string{"testString"}
				reloadOfferingOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				reloadOfferingOptionsModel.Zipurl = core.StringPtr("testString")
				reloadOfferingOptionsModel.RepoType = core.StringPtr("testString")
				reloadOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ReloadOffering(reloadOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOffering(getOfferingOptions *GetOfferingOptions) - Operation response error`, func() {
		getOfferingPath := "/catalogs/testString/offerings/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOffering with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingOptions model
				getOfferingOptionsModel := new(catalogmanagementv1.GetOfferingOptions)
				getOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetOffering(getOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetOffering(getOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOffering(getOfferingOptions *GetOfferingOptions)`, func() {
		getOfferingPath := "/catalogs/testString/offerings/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke GetOffering successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingOptions model
				getOfferingOptionsModel := new(catalogmanagementv1.GetOfferingOptions)
				getOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingWithContext(ctx, getOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOffering(getOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingWithContext(ctx, getOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke GetOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOffering(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingOptions model
				getOfferingOptionsModel := new(catalogmanagementv1.GetOfferingOptions)
				getOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOffering(getOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOffering with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingOptions model
				getOfferingOptionsModel := new(catalogmanagementv1.GetOfferingOptions)
				getOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOffering(getOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingOptions model with no property values
				getOfferingOptionsModelNew := new(catalogmanagementv1.GetOfferingOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOffering(getOfferingOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingOptions model
				getOfferingOptionsModel := new(catalogmanagementv1.GetOfferingOptions)
				getOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOffering(getOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceOffering(replaceOfferingOptions *ReplaceOfferingOptions) - Operation response error`, func() {
		replaceOfferingPath := "/catalogs/testString/offerings/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOfferingPath))
					Expect(req.Method).To(Equal("PUT"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceOffering with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the ReplaceOfferingOptions model
				replaceOfferingOptionsModel := new(catalogmanagementv1.ReplaceOfferingOptions)
				replaceOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingOptionsModel.ID = core.StringPtr("testString")
				replaceOfferingOptionsModel.Rev = core.StringPtr("testString")
				replaceOfferingOptionsModel.URL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.Label = core.StringPtr("testString")
				replaceOfferingOptionsModel.Name = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.Tags = []string{"testString"}
				replaceOfferingOptionsModel.Keywords = []string{"testString"}
				replaceOfferingOptionsModel.Rating = ratingModel
				replaceOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				replaceOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				replaceOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				replaceOfferingOptionsModel.Metadata = make(map[string]interface{})
				replaceOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				replaceOfferingOptionsModel.Hidden = core.BoolPtr(true)
				replaceOfferingOptionsModel.Provider = core.StringPtr("testString")
				replaceOfferingOptionsModel.RepoInfo = repoInfoModel
				replaceOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ReplaceOffering(replaceOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ReplaceOffering(replaceOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceOffering(replaceOfferingOptions *ReplaceOfferingOptions)`, func() {
		replaceOfferingPath := "/catalogs/testString/offerings/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOfferingPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ReplaceOffering successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the ReplaceOfferingOptions model
				replaceOfferingOptionsModel := new(catalogmanagementv1.ReplaceOfferingOptions)
				replaceOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingOptionsModel.ID = core.StringPtr("testString")
				replaceOfferingOptionsModel.Rev = core.StringPtr("testString")
				replaceOfferingOptionsModel.URL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.Label = core.StringPtr("testString")
				replaceOfferingOptionsModel.Name = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.Tags = []string{"testString"}
				replaceOfferingOptionsModel.Keywords = []string{"testString"}
				replaceOfferingOptionsModel.Rating = ratingModel
				replaceOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				replaceOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				replaceOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				replaceOfferingOptionsModel.Metadata = make(map[string]interface{})
				replaceOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				replaceOfferingOptionsModel.Hidden = core.BoolPtr(true)
				replaceOfferingOptionsModel.Provider = core.StringPtr("testString")
				replaceOfferingOptionsModel.RepoInfo = repoInfoModel
				replaceOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ReplaceOfferingWithContext(ctx, replaceOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ReplaceOffering(replaceOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ReplaceOfferingWithContext(ctx, replaceOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOfferingPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ReplaceOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ReplaceOffering(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the ReplaceOfferingOptions model
				replaceOfferingOptionsModel := new(catalogmanagementv1.ReplaceOfferingOptions)
				replaceOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingOptionsModel.ID = core.StringPtr("testString")
				replaceOfferingOptionsModel.Rev = core.StringPtr("testString")
				replaceOfferingOptionsModel.URL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.Label = core.StringPtr("testString")
				replaceOfferingOptionsModel.Name = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.Tags = []string{"testString"}
				replaceOfferingOptionsModel.Keywords = []string{"testString"}
				replaceOfferingOptionsModel.Rating = ratingModel
				replaceOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				replaceOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				replaceOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				replaceOfferingOptionsModel.Metadata = make(map[string]interface{})
				replaceOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				replaceOfferingOptionsModel.Hidden = core.BoolPtr(true)
				replaceOfferingOptionsModel.Provider = core.StringPtr("testString")
				replaceOfferingOptionsModel.RepoInfo = repoInfoModel
				replaceOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ReplaceOffering(replaceOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceOffering with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the ReplaceOfferingOptions model
				replaceOfferingOptionsModel := new(catalogmanagementv1.ReplaceOfferingOptions)
				replaceOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingOptionsModel.ID = core.StringPtr("testString")
				replaceOfferingOptionsModel.Rev = core.StringPtr("testString")
				replaceOfferingOptionsModel.URL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.Label = core.StringPtr("testString")
				replaceOfferingOptionsModel.Name = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.Tags = []string{"testString"}
				replaceOfferingOptionsModel.Keywords = []string{"testString"}
				replaceOfferingOptionsModel.Rating = ratingModel
				replaceOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				replaceOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				replaceOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				replaceOfferingOptionsModel.Metadata = make(map[string]interface{})
				replaceOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				replaceOfferingOptionsModel.Hidden = core.BoolPtr(true)
				replaceOfferingOptionsModel.Provider = core.StringPtr("testString")
				replaceOfferingOptionsModel.RepoInfo = repoInfoModel
				replaceOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ReplaceOffering(replaceOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceOfferingOptions model with no property values
				replaceOfferingOptionsModelNew := new(catalogmanagementv1.ReplaceOfferingOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ReplaceOffering(replaceOfferingOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ReplaceOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")

				// Construct an instance of the ReplaceOfferingOptions model
				replaceOfferingOptionsModel := new(catalogmanagementv1.ReplaceOfferingOptions)
				replaceOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingOptionsModel.ID = core.StringPtr("testString")
				replaceOfferingOptionsModel.Rev = core.StringPtr("testString")
				replaceOfferingOptionsModel.URL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.Label = core.StringPtr("testString")
				replaceOfferingOptionsModel.Name = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingIconURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingDocsURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.OfferingSupportURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.Tags = []string{"testString"}
				replaceOfferingOptionsModel.Keywords = []string{"testString"}
				replaceOfferingOptionsModel.Rating = ratingModel
				replaceOfferingOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceOfferingOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.LongDescription = core.StringPtr("testString")
				replaceOfferingOptionsModel.Features = []catalogmanagementv1.Feature{*featureModel}
				replaceOfferingOptionsModel.Kinds = []catalogmanagementv1.Kind{*kindModel}
				replaceOfferingOptionsModel.PermitRequestIBMPublicPublish = core.BoolPtr(true)
				replaceOfferingOptionsModel.IBMPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicPublishApproved = core.BoolPtr(true)
				replaceOfferingOptionsModel.PublicOriginalCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PublishPublicCRN = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalApprovalRecord = core.StringPtr("testString")
				replaceOfferingOptionsModel.PortalUIURL = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogID = core.StringPtr("testString")
				replaceOfferingOptionsModel.CatalogName = core.StringPtr("testString")
				replaceOfferingOptionsModel.Metadata = make(map[string]interface{})
				replaceOfferingOptionsModel.Disclaimer = core.StringPtr("testString")
				replaceOfferingOptionsModel.Hidden = core.BoolPtr(true)
				replaceOfferingOptionsModel.Provider = core.StringPtr("testString")
				replaceOfferingOptionsModel.RepoInfo = repoInfoModel
				replaceOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ReplaceOffering(replaceOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteOffering(deleteOfferingOptions *DeleteOfferingOptions)`, func() {
		deleteOfferingPath := "/catalogs/testString/offerings/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteOfferingPath))
					Expect(req.Method).To(Equal("DELETE"))

					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteOffering successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteOffering(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteOfferingOptions model
				deleteOfferingOptionsModel := new(catalogmanagementv1.DeleteOfferingOptions)
				deleteOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				deleteOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteOffering(deleteOfferingOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteOffering with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteOfferingOptions model
				deleteOfferingOptionsModel := new(catalogmanagementv1.DeleteOfferingOptions)
				deleteOfferingOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteOfferingOptionsModel.OfferingID = core.StringPtr("testString")
				deleteOfferingOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteOffering(deleteOfferingOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteOfferingOptions model with no property values
				deleteOfferingOptionsModelNew := new(catalogmanagementv1.DeleteOfferingOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteOffering(deleteOfferingOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingAudit(getOfferingAuditOptions *GetOfferingAuditOptions) - Operation response error`, func() {
		getOfferingAuditPath := "/catalogs/testString/offerings/testString/audit"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingAuditPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOfferingAudit with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingAuditOptions model
				getOfferingAuditOptionsModel := new(catalogmanagementv1.GetOfferingAuditOptions)
				getOfferingAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingAuditOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingAudit(getOfferingAuditOptions *GetOfferingAuditOptions)`, func() {
		getOfferingAuditPath := "/catalogs/testString/offerings/testString/audit"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetOfferingAudit successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingAuditOptions model
				getOfferingAuditOptionsModel := new(catalogmanagementv1.GetOfferingAuditOptions)
				getOfferingAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingAuditOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingAuditWithContext(ctx, getOfferingAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingAuditWithContext(ctx, getOfferingAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetOfferingAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingAudit(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingAuditOptions model
				getOfferingAuditOptionsModel := new(catalogmanagementv1.GetOfferingAuditOptions)
				getOfferingAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingAuditOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingAudit with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingAuditOptions model
				getOfferingAuditOptionsModel := new(catalogmanagementv1.GetOfferingAuditOptions)
				getOfferingAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingAuditOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingAuditOptions model with no property values
				getOfferingAuditOptionsModelNew := new(catalogmanagementv1.GetOfferingAuditOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingAuditOptions model
				getOfferingAuditOptionsModel := new(catalogmanagementv1.GetOfferingAuditOptions)
				getOfferingAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingAuditOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingAudit(getOfferingAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceOfferingIcon(replaceOfferingIconOptions *ReplaceOfferingIconOptions) - Operation response error`, func() {
		replaceOfferingIconPath := "/catalogs/testString/offerings/testString/icon/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOfferingIconPath))
					Expect(req.Method).To(Equal("PUT"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceOfferingIcon with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReplaceOfferingIconOptions model
				replaceOfferingIconOptionsModel := new(catalogmanagementv1.ReplaceOfferingIconOptions)
				replaceOfferingIconOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.FileName = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceOfferingIcon(replaceOfferingIconOptions *ReplaceOfferingIconOptions)`, func() {
		replaceOfferingIconPath := "/catalogs/testString/offerings/testString/icon/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOfferingIconPath))
					Expect(req.Method).To(Equal("PUT"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ReplaceOfferingIcon successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ReplaceOfferingIconOptions model
				replaceOfferingIconOptionsModel := new(catalogmanagementv1.ReplaceOfferingIconOptions)
				replaceOfferingIconOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.FileName = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ReplaceOfferingIconWithContext(ctx, replaceOfferingIconOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ReplaceOfferingIconWithContext(ctx, replaceOfferingIconOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOfferingIconPath))
					Expect(req.Method).To(Equal("PUT"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke ReplaceOfferingIcon successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ReplaceOfferingIcon(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ReplaceOfferingIconOptions model
				replaceOfferingIconOptionsModel := new(catalogmanagementv1.ReplaceOfferingIconOptions)
				replaceOfferingIconOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.FileName = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceOfferingIcon with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReplaceOfferingIconOptions model
				replaceOfferingIconOptionsModel := new(catalogmanagementv1.ReplaceOfferingIconOptions)
				replaceOfferingIconOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.FileName = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceOfferingIconOptions model with no property values
				replaceOfferingIconOptionsModelNew := new(catalogmanagementv1.ReplaceOfferingIconOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ReplaceOfferingIcon successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReplaceOfferingIconOptions model
				replaceOfferingIconOptionsModel := new(catalogmanagementv1.ReplaceOfferingIconOptions)
				replaceOfferingIconOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.OfferingID = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.FileName = core.StringPtr("testString")
				replaceOfferingIconOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ReplaceOfferingIcon(replaceOfferingIconOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateOfferingIBM(updateOfferingIBMOptions *UpdateOfferingIBMOptions) - Operation response error`, func() {
		updateOfferingIBMPath := "/catalogs/testString/offerings/testString/publish/allow_request/true"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateOfferingIBMPath))
					Expect(req.Method).To(Equal("POST"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateOfferingIBM with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the UpdateOfferingIBMOptions model
				updateOfferingIBMOptionsModel := new(catalogmanagementv1.UpdateOfferingIBMOptions)
				updateOfferingIBMOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.OfferingID = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.ApprovalType = core.StringPtr("allow_request")
				updateOfferingIBMOptionsModel.Approved = core.StringPtr("true")
				updateOfferingIBMOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateOfferingIBM(updateOfferingIBMOptions *UpdateOfferingIBMOptions)`, func() {
		updateOfferingIBMPath := "/catalogs/testString/offerings/testString/publish/allow_request/true"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateOfferingIBMPath))
					Expect(req.Method).To(Equal("POST"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"allow_request": true, "ibm": false, "public": true, "changed": false}`)
				}))
			})
			It(`Invoke UpdateOfferingIBM successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the UpdateOfferingIBMOptions model
				updateOfferingIBMOptionsModel := new(catalogmanagementv1.UpdateOfferingIBMOptions)
				updateOfferingIBMOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.OfferingID = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.ApprovalType = core.StringPtr("allow_request")
				updateOfferingIBMOptionsModel.Approved = core.StringPtr("true")
				updateOfferingIBMOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.UpdateOfferingIBMWithContext(ctx, updateOfferingIBMOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.UpdateOfferingIBMWithContext(ctx, updateOfferingIBMOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateOfferingIBMPath))
					Expect(req.Method).To(Equal("POST"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"allow_request": true, "ibm": false, "public": true, "changed": false}`)
				}))
			})
			It(`Invoke UpdateOfferingIBM successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.UpdateOfferingIBM(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the UpdateOfferingIBMOptions model
				updateOfferingIBMOptionsModel := new(catalogmanagementv1.UpdateOfferingIBMOptions)
				updateOfferingIBMOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.OfferingID = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.ApprovalType = core.StringPtr("allow_request")
				updateOfferingIBMOptionsModel.Approved = core.StringPtr("true")
				updateOfferingIBMOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateOfferingIBM with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the UpdateOfferingIBMOptions model
				updateOfferingIBMOptionsModel := new(catalogmanagementv1.UpdateOfferingIBMOptions)
				updateOfferingIBMOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.OfferingID = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.ApprovalType = core.StringPtr("allow_request")
				updateOfferingIBMOptionsModel.Approved = core.StringPtr("true")
				updateOfferingIBMOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateOfferingIBMOptions model with no property values
				updateOfferingIBMOptionsModelNew := new(catalogmanagementv1.UpdateOfferingIBMOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateOfferingIBM successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the UpdateOfferingIBMOptions model
				updateOfferingIBMOptionsModel := new(catalogmanagementv1.UpdateOfferingIBMOptions)
				updateOfferingIBMOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.OfferingID = core.StringPtr("testString")
				updateOfferingIBMOptionsModel.ApprovalType = core.StringPtr("allow_request")
				updateOfferingIBMOptionsModel.Approved = core.StringPtr("true")
				updateOfferingIBMOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.UpdateOfferingIBM(updateOfferingIBMOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingUpdates(getOfferingUpdatesOptions *GetOfferingUpdatesOptions) - Operation response error`, func() {
		getOfferingUpdatesPath := "/catalogs/testString/offerings/testString/updates"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingUpdatesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["kind"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["resource_group_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["namespace"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOfferingUpdates with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingUpdatesOptions model
				getOfferingUpdatesOptionsModel := new(catalogmanagementv1.GetOfferingUpdatesOptions)
				getOfferingUpdatesOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Kind = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Version = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ClusterID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Region = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Namespace = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingUpdates(getOfferingUpdatesOptions *GetOfferingUpdatesOptions)`, func() {
		getOfferingUpdatesPath := "/catalogs/testString/offerings/testString/updates"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingUpdatesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["kind"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["resource_group_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["namespace"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"version_locator": "VersionLocator", "version": "Version", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "required_resources": [{"type": "mem", "value": "anyValue"}], "package_version": "PackageVersion", "can_update": false, "messages": {"mapKey": "Inner"}}]`)
				}))
			})
			It(`Invoke GetOfferingUpdates successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingUpdatesOptions model
				getOfferingUpdatesOptionsModel := new(catalogmanagementv1.GetOfferingUpdatesOptions)
				getOfferingUpdatesOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Kind = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Version = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ClusterID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Region = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Namespace = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingUpdatesWithContext(ctx, getOfferingUpdatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingUpdatesWithContext(ctx, getOfferingUpdatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingUpdatesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["kind"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["resource_group_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["namespace"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"version_locator": "VersionLocator", "version": "Version", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "required_resources": [{"type": "mem", "value": "anyValue"}], "package_version": "PackageVersion", "can_update": false, "messages": {"mapKey": "Inner"}}]`)
				}))
			})
			It(`Invoke GetOfferingUpdates successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingUpdates(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingUpdatesOptions model
				getOfferingUpdatesOptionsModel := new(catalogmanagementv1.GetOfferingUpdatesOptions)
				getOfferingUpdatesOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Kind = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Version = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ClusterID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Region = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Namespace = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingUpdates with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingUpdatesOptions model
				getOfferingUpdatesOptionsModel := new(catalogmanagementv1.GetOfferingUpdatesOptions)
				getOfferingUpdatesOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Kind = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Version = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ClusterID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Region = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Namespace = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingUpdatesOptions model with no property values
				getOfferingUpdatesOptionsModelNew := new(catalogmanagementv1.GetOfferingUpdatesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingUpdates successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingUpdatesOptions model
				getOfferingUpdatesOptionsModel := new(catalogmanagementv1.GetOfferingUpdatesOptions)
				getOfferingUpdatesOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.OfferingID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Kind = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Version = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ClusterID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Region = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Namespace = core.StringPtr("testString")
				getOfferingUpdatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingUpdates(getOfferingUpdatesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingAbout(getOfferingAboutOptions *GetOfferingAboutOptions)`, func() {
		getOfferingAboutPath := "/versions/testString/about"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingAboutPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "text/markdown")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `"OperationResponse"`)
				}))
			})
			It(`Invoke GetOfferingAbout successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingAboutOptions model
				getOfferingAboutOptionsModel := new(catalogmanagementv1.GetOfferingAboutOptions)
				getOfferingAboutOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingAboutOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingAboutWithContext(ctx, getOfferingAboutOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingAbout(getOfferingAboutOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingAboutWithContext(ctx, getOfferingAboutOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingAboutPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "text/markdown")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `"OperationResponse"`)
				}))
			})
			It(`Invoke GetOfferingAbout successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingAbout(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingAboutOptions model
				getOfferingAboutOptionsModel := new(catalogmanagementv1.GetOfferingAboutOptions)
				getOfferingAboutOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingAboutOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingAbout(getOfferingAboutOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingAbout with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingAboutOptions model
				getOfferingAboutOptionsModel := new(catalogmanagementv1.GetOfferingAboutOptions)
				getOfferingAboutOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingAboutOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingAbout(getOfferingAboutOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingAboutOptions model with no property values
				getOfferingAboutOptionsModelNew := new(catalogmanagementv1.GetOfferingAboutOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingAbout(getOfferingAboutOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingAbout successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingAboutOptions model
				getOfferingAboutOptionsModel := new(catalogmanagementv1.GetOfferingAboutOptions)
				getOfferingAboutOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingAboutOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingAbout(getOfferingAboutOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingLicense(getOfferingLicenseOptions *GetOfferingLicenseOptions)`, func() {
		getOfferingLicensePath := "/versions/testString/licenses/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingLicensePath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "text/plain")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `"OperationResponse"`)
				}))
			})
			It(`Invoke GetOfferingLicense successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingLicenseOptions model
				getOfferingLicenseOptionsModel := new(catalogmanagementv1.GetOfferingLicenseOptions)
				getOfferingLicenseOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.LicenseID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingLicenseWithContext(ctx, getOfferingLicenseOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingLicense(getOfferingLicenseOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingLicenseWithContext(ctx, getOfferingLicenseOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingLicensePath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "text/plain")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `"OperationResponse"`)
				}))
			})
			It(`Invoke GetOfferingLicense successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingLicense(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingLicenseOptions model
				getOfferingLicenseOptionsModel := new(catalogmanagementv1.GetOfferingLicenseOptions)
				getOfferingLicenseOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.LicenseID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingLicense(getOfferingLicenseOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingLicense with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingLicenseOptions model
				getOfferingLicenseOptionsModel := new(catalogmanagementv1.GetOfferingLicenseOptions)
				getOfferingLicenseOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.LicenseID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingLicense(getOfferingLicenseOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingLicenseOptions model with no property values
				getOfferingLicenseOptionsModelNew := new(catalogmanagementv1.GetOfferingLicenseOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingLicense(getOfferingLicenseOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingLicense successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingLicenseOptions model
				getOfferingLicenseOptionsModel := new(catalogmanagementv1.GetOfferingLicenseOptions)
				getOfferingLicenseOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.LicenseID = core.StringPtr("testString")
				getOfferingLicenseOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingLicense(getOfferingLicenseOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingContainerImages(getOfferingContainerImagesOptions *GetOfferingContainerImagesOptions) - Operation response error`, func() {
		getOfferingContainerImagesPath := "/versions/testString/containerImages"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingContainerImagesPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOfferingContainerImages with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingContainerImagesOptions model
				getOfferingContainerImagesOptionsModel := new(catalogmanagementv1.GetOfferingContainerImagesOptions)
				getOfferingContainerImagesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingContainerImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingContainerImages(getOfferingContainerImagesOptions *GetOfferingContainerImagesOptions)`, func() {
		getOfferingContainerImagesPath := "/versions/testString/containerImages"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingContainerImagesPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"description": "Description", "images": [{"image": "Image"}]}`)
				}))
			})
			It(`Invoke GetOfferingContainerImages successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingContainerImagesOptions model
				getOfferingContainerImagesOptionsModel := new(catalogmanagementv1.GetOfferingContainerImagesOptions)
				getOfferingContainerImagesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingContainerImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingContainerImagesWithContext(ctx, getOfferingContainerImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingContainerImagesWithContext(ctx, getOfferingContainerImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingContainerImagesPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"description": "Description", "images": [{"image": "Image"}]}`)
				}))
			})
			It(`Invoke GetOfferingContainerImages successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingContainerImages(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingContainerImagesOptions model
				getOfferingContainerImagesOptionsModel := new(catalogmanagementv1.GetOfferingContainerImagesOptions)
				getOfferingContainerImagesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingContainerImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingContainerImages with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingContainerImagesOptions model
				getOfferingContainerImagesOptionsModel := new(catalogmanagementv1.GetOfferingContainerImagesOptions)
				getOfferingContainerImagesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingContainerImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingContainerImagesOptions model with no property values
				getOfferingContainerImagesOptionsModelNew := new(catalogmanagementv1.GetOfferingContainerImagesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingContainerImages successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingContainerImagesOptions model
				getOfferingContainerImagesOptionsModel := new(catalogmanagementv1.GetOfferingContainerImagesOptions)
				getOfferingContainerImagesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingContainerImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingContainerImages(getOfferingContainerImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeprecateVersion(deprecateVersionOptions *DeprecateVersionOptions)`, func() {
		deprecateVersionPath := "/versions/testString/deprecate"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deprecateVersionPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke DeprecateVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeprecateVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeprecateVersionOptions model
				deprecateVersionOptionsModel := new(catalogmanagementv1.DeprecateVersionOptions)
				deprecateVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				deprecateVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeprecateVersion(deprecateVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeprecateVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeprecateVersionOptions model
				deprecateVersionOptionsModel := new(catalogmanagementv1.DeprecateVersionOptions)
				deprecateVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				deprecateVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeprecateVersion(deprecateVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeprecateVersionOptions model with no property values
				deprecateVersionOptionsModelNew := new(catalogmanagementv1.DeprecateVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeprecateVersion(deprecateVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`AccountPublishVersion(accountPublishVersionOptions *AccountPublishVersionOptions)`, func() {
		accountPublishVersionPath := "/versions/testString/account-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountPublishVersionPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke AccountPublishVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.AccountPublishVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the AccountPublishVersionOptions model
				accountPublishVersionOptionsModel := new(catalogmanagementv1.AccountPublishVersionOptions)
				accountPublishVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				accountPublishVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.AccountPublishVersion(accountPublishVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke AccountPublishVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the AccountPublishVersionOptions model
				accountPublishVersionOptionsModel := new(catalogmanagementv1.AccountPublishVersionOptions)
				accountPublishVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				accountPublishVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.AccountPublishVersion(accountPublishVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the AccountPublishVersionOptions model with no property values
				accountPublishVersionOptionsModelNew := new(catalogmanagementv1.AccountPublishVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.AccountPublishVersion(accountPublishVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`IBMPublishVersion(ibmPublishVersionOptions *IBMPublishVersionOptions)`, func() {
		ibmPublishVersionPath := "/versions/testString/ibm-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(ibmPublishVersionPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke IBMPublishVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.IBMPublishVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the IBMPublishVersionOptions model
				ibmPublishVersionOptionsModel := new(catalogmanagementv1.IBMPublishVersionOptions)
				ibmPublishVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				ibmPublishVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.IBMPublishVersion(ibmPublishVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke IBMPublishVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the IBMPublishVersionOptions model
				ibmPublishVersionOptionsModel := new(catalogmanagementv1.IBMPublishVersionOptions)
				ibmPublishVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				ibmPublishVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.IBMPublishVersion(ibmPublishVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the IBMPublishVersionOptions model with no property values
				ibmPublishVersionOptionsModelNew := new(catalogmanagementv1.IBMPublishVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.IBMPublishVersion(ibmPublishVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`PublicPublishVersion(publicPublishVersionOptions *PublicPublishVersionOptions)`, func() {
		publicPublishVersionPath := "/versions/testString/public-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(publicPublishVersionPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke PublicPublishVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.PublicPublishVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the PublicPublishVersionOptions model
				publicPublishVersionOptionsModel := new(catalogmanagementv1.PublicPublishVersionOptions)
				publicPublishVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				publicPublishVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.PublicPublishVersion(publicPublishVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke PublicPublishVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublicPublishVersionOptions model
				publicPublishVersionOptionsModel := new(catalogmanagementv1.PublicPublishVersionOptions)
				publicPublishVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				publicPublishVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.PublicPublishVersion(publicPublishVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the PublicPublishVersionOptions model with no property values
				publicPublishVersionOptionsModelNew := new(catalogmanagementv1.PublicPublishVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.PublicPublishVersion(publicPublishVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CommitVersion(commitVersionOptions *CommitVersionOptions)`, func() {
		commitVersionPath := "/versions/testString/commit"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(commitVersionPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(200)
				}))
			})
			It(`Invoke CommitVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.CommitVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the CommitVersionOptions model
				commitVersionOptionsModel := new(catalogmanagementv1.CommitVersionOptions)
				commitVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				commitVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.CommitVersion(commitVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke CommitVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the CommitVersionOptions model
				commitVersionOptionsModel := new(catalogmanagementv1.CommitVersionOptions)
				commitVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				commitVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.CommitVersion(commitVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the CommitVersionOptions model with no property values
				commitVersionOptionsModelNew := new(catalogmanagementv1.CommitVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.CommitVersion(commitVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CopyVersion(copyVersionOptions *CopyVersionOptions)`, func() {
		copyVersionPath := "/versions/testString/copy"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(copyVersionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					res.WriteHeader(200)
				}))
			})
			It(`Invoke CopyVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.CopyVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the CopyVersionOptions model
				copyVersionOptionsModel := new(catalogmanagementv1.CopyVersionOptions)
				copyVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				copyVersionOptionsModel.Tags = []string{"testString"}
				copyVersionOptionsModel.TargetKinds = []string{"testString"}
				copyVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				copyVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.CopyVersion(copyVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke CopyVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the CopyVersionOptions model
				copyVersionOptionsModel := new(catalogmanagementv1.CopyVersionOptions)
				copyVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				copyVersionOptionsModel.Tags = []string{"testString"}
				copyVersionOptionsModel.TargetKinds = []string{"testString"}
				copyVersionOptionsModel.Content = CreateMockByteArray("This is a mock byte array value.")
				copyVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.CopyVersion(copyVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the CopyVersionOptions model with no property values
				copyVersionOptionsModelNew := new(catalogmanagementv1.CopyVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.CopyVersion(copyVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingWorkingCopy(getOfferingWorkingCopyOptions *GetOfferingWorkingCopyOptions) - Operation response error`, func() {
		getOfferingWorkingCopyPath := "/versions/testString/workingcopy"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingWorkingCopyPath))
					Expect(req.Method).To(Equal("POST"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOfferingWorkingCopy with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingWorkingCopyOptions model
				getOfferingWorkingCopyOptionsModel := new(catalogmanagementv1.GetOfferingWorkingCopyOptions)
				getOfferingWorkingCopyOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingWorkingCopyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingWorkingCopy(getOfferingWorkingCopyOptions *GetOfferingWorkingCopyOptions)`, func() {
		getOfferingWorkingCopyPath := "/versions/testString/workingcopy"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingWorkingCopyPath))
					Expect(req.Method).To(Equal("POST"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}`)
				}))
			})
			It(`Invoke GetOfferingWorkingCopy successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingWorkingCopyOptions model
				getOfferingWorkingCopyOptionsModel := new(catalogmanagementv1.GetOfferingWorkingCopyOptions)
				getOfferingWorkingCopyOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingWorkingCopyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingWorkingCopyWithContext(ctx, getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingWorkingCopyWithContext(ctx, getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingWorkingCopyPath))
					Expect(req.Method).To(Equal("POST"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}`)
				}))
			})
			It(`Invoke GetOfferingWorkingCopy successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingWorkingCopy(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingWorkingCopyOptions model
				getOfferingWorkingCopyOptionsModel := new(catalogmanagementv1.GetOfferingWorkingCopyOptions)
				getOfferingWorkingCopyOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingWorkingCopyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingWorkingCopy with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingWorkingCopyOptions model
				getOfferingWorkingCopyOptionsModel := new(catalogmanagementv1.GetOfferingWorkingCopyOptions)
				getOfferingWorkingCopyOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingWorkingCopyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingWorkingCopyOptions model with no property values
				getOfferingWorkingCopyOptionsModelNew := new(catalogmanagementv1.GetOfferingWorkingCopyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingWorkingCopy successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingWorkingCopyOptions model
				getOfferingWorkingCopyOptionsModel := new(catalogmanagementv1.GetOfferingWorkingCopyOptions)
				getOfferingWorkingCopyOptionsModel.VersionLocID = core.StringPtr("testString")
				getOfferingWorkingCopyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingWorkingCopy(getOfferingWorkingCopyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVersion(getVersionOptions *GetVersionOptions) - Operation response error`, func() {
		getVersionPath := "/versions/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVersionPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVersion with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetVersionOptions model
				getVersionOptionsModel := new(catalogmanagementv1.GetVersionOptions)
				getVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				getVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetVersion(getVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetVersion(getVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVersion(getVersionOptions *GetVersionOptions)`, func() {
		getVersionPath := "/versions/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVersionPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke GetVersion successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetVersionOptions model
				getVersionOptionsModel := new(catalogmanagementv1.GetVersionOptions)
				getVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				getVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetVersionWithContext(ctx, getVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetVersion(getVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetVersionWithContext(ctx, getVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVersionPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "name": "Name", "offering_icon_url": "OfferingIconURL", "offering_docs_url": "OfferingDocsURL", "offering_support_url": "OfferingSupportURL", "tags": ["Tags"], "keywords": ["Keywords"], "rating": {"one_star_count": 12, "two_star_count": 12, "three_star_count": 14, "four_star_count": 13}, "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "long_description": "LongDescription", "features": [{"title": "Title", "description": "Description"}], "kinds": [{"id": "ID", "format_kind": "FormatKind", "target_kind": "TargetKind", "metadata": {"mapKey": "anyValue"}, "install_description": "InstallDescription", "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "versions": [{"id": "ID", "_rev": "Rev", "crn": "CRN", "version": "Version", "sha": "Sha", "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "offering_id": "OfferingID", "catalog_id": "CatalogID", "kind_id": "KindID", "tags": ["Tags"], "repo_url": "RepoURL", "source_url": "SourceURL", "tgz_url": "TgzURL", "configuration": [{"key": "Key", "type": "Type", "default_value": "anyValue", "value_constraint": "ValueConstraint", "description": "Description", "required": true, "options": ["anyValue"], "hidden": true}], "metadata": {"mapKey": "anyValue"}, "validation": {"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}, "required_resources": [{"type": "mem", "value": "anyValue"}], "single_instance": true, "install": {"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}, "pre_install": [{"instructions": "Instructions", "script": "Script", "script_permission": "ScriptPermission", "delete_script": "DeleteScript", "scope": "Scope"}], "entitlement": {"provider_name": "ProviderName", "provider_id": "ProviderID", "product_id": "ProductID", "part_numbers": ["PartNumbers"], "image_repo_name": "ImageRepoName"}, "licenses": [{"id": "ID", "name": "Name", "type": "Type", "url": "URL", "description": "Description"}], "image_manifest_url": "ImageManifestURL", "deprecated": true, "package_version": "PackageVersion", "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "version_locator": "VersionLocator", "console_url": "ConsoleURL", "long_description": "LongDescription", "whitelisted_accounts": ["WhitelistedAccounts"]}], "plans": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "additional_features": [{"title": "Title", "description": "Description"}], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "deployments": [{"id": "ID", "label": "Label", "name": "Name", "short_description": "ShortDescription", "long_description": "LongDescription", "metadata": {"mapKey": "anyValue"}, "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z"}]}]}], "permit_request_ibm_public_publish": false, "ibm_publish_approved": true, "public_publish_approved": false, "public_original_crn": "PublicOriginalCRN", "publish_public_crn": "PublishPublicCRN", "portal_approval_record": "PortalApprovalRecord", "portal_ui_url": "PortalUIURL", "catalog_id": "CatalogID", "catalog_name": "CatalogName", "metadata": {"mapKey": "anyValue"}, "disclaimer": "Disclaimer", "hidden": true, "provider": "Provider", "repo_info": {"token": "Token", "type": "Type"}}`)
				}))
			})
			It(`Invoke GetVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVersionOptions model
				getVersionOptionsModel := new(catalogmanagementv1.GetVersionOptions)
				getVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				getVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetVersion(getVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetVersionOptions model
				getVersionOptionsModel := new(catalogmanagementv1.GetVersionOptions)
				getVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				getVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetVersion(getVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVersionOptions model with no property values
				getVersionOptionsModelNew := new(catalogmanagementv1.GetVersionOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetVersion(getVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetVersionOptions model
				getVersionOptionsModel := new(catalogmanagementv1.GetVersionOptions)
				getVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				getVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetVersion(getVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteVersion(deleteVersionOptions *DeleteVersionOptions)`, func() {
		deleteVersionPath := "/versions/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVersionPath))
					Expect(req.Method).To(Equal("DELETE"))

					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVersionOptions model
				deleteVersionOptionsModel := new(catalogmanagementv1.DeleteVersionOptions)
				deleteVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				deleteVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteVersion(deleteVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteVersionOptions model
				deleteVersionOptionsModel := new(catalogmanagementv1.DeleteVersionOptions)
				deleteVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				deleteVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteVersion(deleteVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVersionOptions model with no property values
				deleteVersionOptionsModelNew := new(catalogmanagementv1.DeleteVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteVersion(deleteVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCluster(getClusterOptions *GetClusterOptions) - Operation response error`, func() {
		getClusterPath := "/deploy/kubernetes/clusters/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getClusterPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetCluster with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetClusterOptions model
				getClusterOptionsModel := new(catalogmanagementv1.GetClusterOptions)
				getClusterOptionsModel.ClusterID = core.StringPtr("testString")
				getClusterOptionsModel.Region = core.StringPtr("testString")
				getClusterOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getClusterOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetCluster(getClusterOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetCluster(getClusterOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetCluster(getClusterOptions *GetClusterOptions)`, func() {
		getClusterPath := "/deploy/kubernetes/clusters/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getClusterPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"resource_group_id": "ResourceGroupID", "resource_group_name": "ResourceGroupName", "id": "ID", "name": "Name", "region": "Region"}`)
				}))
			})
			It(`Invoke GetCluster successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetClusterOptions model
				getClusterOptionsModel := new(catalogmanagementv1.GetClusterOptions)
				getClusterOptionsModel.ClusterID = core.StringPtr("testString")
				getClusterOptionsModel.Region = core.StringPtr("testString")
				getClusterOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getClusterOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetClusterWithContext(ctx, getClusterOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetCluster(getClusterOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetClusterWithContext(ctx, getClusterOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getClusterPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"resource_group_id": "ResourceGroupID", "resource_group_name": "ResourceGroupName", "id": "ID", "name": "Name", "region": "Region"}`)
				}))
			})
			It(`Invoke GetCluster successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetCluster(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetClusterOptions model
				getClusterOptionsModel := new(catalogmanagementv1.GetClusterOptions)
				getClusterOptionsModel.ClusterID = core.StringPtr("testString")
				getClusterOptionsModel.Region = core.StringPtr("testString")
				getClusterOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getClusterOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetCluster(getClusterOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetCluster with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetClusterOptions model
				getClusterOptionsModel := new(catalogmanagementv1.GetClusterOptions)
				getClusterOptionsModel.ClusterID = core.StringPtr("testString")
				getClusterOptionsModel.Region = core.StringPtr("testString")
				getClusterOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getClusterOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetCluster(getClusterOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetClusterOptions model with no property values
				getClusterOptionsModelNew := new(catalogmanagementv1.GetClusterOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetCluster(getClusterOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetCluster successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetClusterOptions model
				getClusterOptionsModel := new(catalogmanagementv1.GetClusterOptions)
				getClusterOptionsModel.ClusterID = core.StringPtr("testString")
				getClusterOptionsModel.Region = core.StringPtr("testString")
				getClusterOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getClusterOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetCluster(getClusterOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetNamespaces(getNamespacesOptions *GetNamespacesOptions) - Operation response error`, func() {
		getNamespacesPath := "/deploy/kubernetes/clusters/testString/namespaces"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getNamespacesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetNamespaces with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetNamespacesOptions model
				getNamespacesOptionsModel := new(catalogmanagementv1.GetNamespacesOptions)
				getNamespacesOptionsModel.ClusterID = core.StringPtr("testString")
				getNamespacesOptionsModel.Region = core.StringPtr("testString")
				getNamespacesOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getNamespacesOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getNamespacesOptionsModel.Offset = core.Int64Ptr(int64(38))
				getNamespacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetNamespaces(getNamespacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetNamespaces(getNamespacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetNamespaces(getNamespacesOptions *GetNamespacesOptions)`, func() {
		getNamespacesPath := "/deploy/kubernetes/clusters/testString/namespaces"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getNamespacesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": ["Resources"]}`)
				}))
			})
			It(`Invoke GetNamespaces successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetNamespacesOptions model
				getNamespacesOptionsModel := new(catalogmanagementv1.GetNamespacesOptions)
				getNamespacesOptionsModel.ClusterID = core.StringPtr("testString")
				getNamespacesOptionsModel.Region = core.StringPtr("testString")
				getNamespacesOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getNamespacesOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getNamespacesOptionsModel.Offset = core.Int64Ptr(int64(38))
				getNamespacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetNamespacesWithContext(ctx, getNamespacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetNamespaces(getNamespacesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetNamespacesWithContext(ctx, getNamespacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getNamespacesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": ["Resources"]}`)
				}))
			})
			It(`Invoke GetNamespaces successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetNamespaces(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetNamespacesOptions model
				getNamespacesOptionsModel := new(catalogmanagementv1.GetNamespacesOptions)
				getNamespacesOptionsModel.ClusterID = core.StringPtr("testString")
				getNamespacesOptionsModel.Region = core.StringPtr("testString")
				getNamespacesOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getNamespacesOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getNamespacesOptionsModel.Offset = core.Int64Ptr(int64(38))
				getNamespacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetNamespaces(getNamespacesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetNamespaces with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetNamespacesOptions model
				getNamespacesOptionsModel := new(catalogmanagementv1.GetNamespacesOptions)
				getNamespacesOptionsModel.ClusterID = core.StringPtr("testString")
				getNamespacesOptionsModel.Region = core.StringPtr("testString")
				getNamespacesOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getNamespacesOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getNamespacesOptionsModel.Offset = core.Int64Ptr(int64(38))
				getNamespacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetNamespaces(getNamespacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetNamespacesOptions model with no property values
				getNamespacesOptionsModelNew := new(catalogmanagementv1.GetNamespacesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetNamespaces(getNamespacesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetNamespaces successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetNamespacesOptions model
				getNamespacesOptionsModel := new(catalogmanagementv1.GetNamespacesOptions)
				getNamespacesOptionsModel.ClusterID = core.StringPtr("testString")
				getNamespacesOptionsModel.Region = core.StringPtr("testString")
				getNamespacesOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getNamespacesOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getNamespacesOptionsModel.Offset = core.Int64Ptr(int64(38))
				getNamespacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetNamespaces(getNamespacesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeployOperators(deployOperatorsOptions *DeployOperatorsOptions) - Operation response error`, func() {
		deployOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deployOperatorsPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke DeployOperators with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeployOperatorsOptions model
				deployOperatorsOptionsModel := new(catalogmanagementv1.DeployOperatorsOptions)
				deployOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deployOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Region = core.StringPtr("testString")
				deployOperatorsOptionsModel.Namespaces = []string{"testString"}
				deployOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				deployOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.DeployOperators(deployOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.DeployOperators(deployOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeployOperators(deployOperatorsOptions *DeployOperatorsOptions)`, func() {
		deployOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deployOperatorsPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"phase": "Phase", "message": "Message", "link": "Link", "name": "Name", "version": "Version", "namespace": "Namespace", "package_name": "PackageName", "catalog_id": "CatalogID"}]`)
				}))
			})
			It(`Invoke DeployOperators successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the DeployOperatorsOptions model
				deployOperatorsOptionsModel := new(catalogmanagementv1.DeployOperatorsOptions)
				deployOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deployOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Region = core.StringPtr("testString")
				deployOperatorsOptionsModel.Namespaces = []string{"testString"}
				deployOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				deployOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.DeployOperatorsWithContext(ctx, deployOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.DeployOperators(deployOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.DeployOperatorsWithContext(ctx, deployOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deployOperatorsPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"phase": "Phase", "message": "Message", "link": "Link", "name": "Name", "version": "Version", "namespace": "Namespace", "package_name": "PackageName", "catalog_id": "CatalogID"}]`)
				}))
			})
			It(`Invoke DeployOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.DeployOperators(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the DeployOperatorsOptions model
				deployOperatorsOptionsModel := new(catalogmanagementv1.DeployOperatorsOptions)
				deployOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deployOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Region = core.StringPtr("testString")
				deployOperatorsOptionsModel.Namespaces = []string{"testString"}
				deployOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				deployOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.DeployOperators(deployOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke DeployOperators with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeployOperatorsOptions model
				deployOperatorsOptionsModel := new(catalogmanagementv1.DeployOperatorsOptions)
				deployOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deployOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Region = core.StringPtr("testString")
				deployOperatorsOptionsModel.Namespaces = []string{"testString"}
				deployOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				deployOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.DeployOperators(deployOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the DeployOperatorsOptions model with no property values
				deployOperatorsOptionsModelNew := new(catalogmanagementv1.DeployOperatorsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.DeployOperators(deployOperatorsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeployOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeployOperatorsOptions model
				deployOperatorsOptionsModel := new(catalogmanagementv1.DeployOperatorsOptions)
				deployOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deployOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Region = core.StringPtr("testString")
				deployOperatorsOptionsModel.Namespaces = []string{"testString"}
				deployOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				deployOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deployOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.DeployOperators(deployOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOperators(listOperatorsOptions *ListOperatorsOptions) - Operation response error`, func() {
		listOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatorsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version_locator_id"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListOperators with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListOperatorsOptions model
				listOperatorsOptionsModel := new(catalogmanagementv1.ListOperatorsOptions)
				listOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				listOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				listOperatorsOptionsModel.Region = core.StringPtr("testString")
				listOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				listOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ListOperators(listOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ListOperators(listOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOperators(listOperatorsOptions *ListOperatorsOptions)`, func() {
		listOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatorsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version_locator_id"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"phase": "Phase", "message": "Message", "link": "Link", "name": "Name", "version": "Version", "namespace": "Namespace", "package_name": "PackageName", "catalog_id": "CatalogID"}]`)
				}))
			})
			It(`Invoke ListOperators successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ListOperatorsOptions model
				listOperatorsOptionsModel := new(catalogmanagementv1.ListOperatorsOptions)
				listOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				listOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				listOperatorsOptionsModel.Region = core.StringPtr("testString")
				listOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				listOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ListOperatorsWithContext(ctx, listOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ListOperators(listOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ListOperatorsWithContext(ctx, listOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatorsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version_locator_id"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"phase": "Phase", "message": "Message", "link": "Link", "name": "Name", "version": "Version", "namespace": "Namespace", "package_name": "PackageName", "catalog_id": "CatalogID"}]`)
				}))
			})
			It(`Invoke ListOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ListOperators(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListOperatorsOptions model
				listOperatorsOptionsModel := new(catalogmanagementv1.ListOperatorsOptions)
				listOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				listOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				listOperatorsOptionsModel.Region = core.StringPtr("testString")
				listOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				listOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ListOperators(listOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListOperators with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListOperatorsOptions model
				listOperatorsOptionsModel := new(catalogmanagementv1.ListOperatorsOptions)
				listOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				listOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				listOperatorsOptionsModel.Region = core.StringPtr("testString")
				listOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				listOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ListOperators(listOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListOperatorsOptions model with no property values
				listOperatorsOptionsModelNew := new(catalogmanagementv1.ListOperatorsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ListOperators(listOperatorsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListOperatorsOptions model
				listOperatorsOptionsModel := new(catalogmanagementv1.ListOperatorsOptions)
				listOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				listOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				listOperatorsOptionsModel.Region = core.StringPtr("testString")
				listOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				listOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ListOperators(listOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceOperators(replaceOperatorsOptions *ReplaceOperatorsOptions) - Operation response error`, func() {
		replaceOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOperatorsPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceOperators with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReplaceOperatorsOptions model
				replaceOperatorsOptionsModel := new(catalogmanagementv1.ReplaceOperatorsOptions)
				replaceOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				replaceOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Region = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Namespaces = []string{"testString"}
				replaceOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				replaceOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceOperators(replaceOperatorsOptions *ReplaceOperatorsOptions)`, func() {
		replaceOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOperatorsPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"phase": "Phase", "message": "Message", "link": "Link", "name": "Name", "version": "Version", "namespace": "Namespace", "package_name": "PackageName", "catalog_id": "CatalogID"}]`)
				}))
			})
			It(`Invoke ReplaceOperators successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ReplaceOperatorsOptions model
				replaceOperatorsOptionsModel := new(catalogmanagementv1.ReplaceOperatorsOptions)
				replaceOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				replaceOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Region = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Namespaces = []string{"testString"}
				replaceOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				replaceOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ReplaceOperatorsWithContext(ctx, replaceOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ReplaceOperatorsWithContext(ctx, replaceOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceOperatorsPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"phase": "Phase", "message": "Message", "link": "Link", "name": "Name", "version": "Version", "namespace": "Namespace", "package_name": "PackageName", "catalog_id": "CatalogID"}]`)
				}))
			})
			It(`Invoke ReplaceOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ReplaceOperators(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ReplaceOperatorsOptions model
				replaceOperatorsOptionsModel := new(catalogmanagementv1.ReplaceOperatorsOptions)
				replaceOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				replaceOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Region = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Namespaces = []string{"testString"}
				replaceOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				replaceOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceOperators with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReplaceOperatorsOptions model
				replaceOperatorsOptionsModel := new(catalogmanagementv1.ReplaceOperatorsOptions)
				replaceOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				replaceOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Region = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Namespaces = []string{"testString"}
				replaceOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				replaceOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceOperatorsOptions model with no property values
				replaceOperatorsOptionsModelNew := new(catalogmanagementv1.ReplaceOperatorsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ReplaceOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ReplaceOperatorsOptions model
				replaceOperatorsOptionsModel := new(catalogmanagementv1.ReplaceOperatorsOptions)
				replaceOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				replaceOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Region = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Namespaces = []string{"testString"}
				replaceOperatorsOptionsModel.AllNamespaces = core.BoolPtr(true)
				replaceOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				replaceOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ReplaceOperators(replaceOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteOperators(deleteOperatorsOptions *DeleteOperatorsOptions)`, func() {
		deleteOperatorsPath := "/deploy/kubernetes/olm/operator"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteOperatorsPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["version_locator_id"]).To(Equal([]string{"testString"}))
					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteOperators successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteOperators(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteOperatorsOptions model
				deleteOperatorsOptionsModel := new(catalogmanagementv1.DeleteOperatorsOptions)
				deleteOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deleteOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deleteOperatorsOptionsModel.Region = core.StringPtr("testString")
				deleteOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deleteOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteOperators(deleteOperatorsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteOperators with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteOperatorsOptions model
				deleteOperatorsOptionsModel := new(catalogmanagementv1.DeleteOperatorsOptions)
				deleteOperatorsOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deleteOperatorsOptionsModel.ClusterID = core.StringPtr("testString")
				deleteOperatorsOptionsModel.Region = core.StringPtr("testString")
				deleteOperatorsOptionsModel.VersionLocatorID = core.StringPtr("testString")
				deleteOperatorsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteOperators(deleteOperatorsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteOperatorsOptions model with no property values
				deleteOperatorsOptionsModelNew := new(catalogmanagementv1.DeleteOperatorsOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteOperators(deleteOperatorsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`InstallVersion(installVersionOptions *InstallVersionOptions)`, func() {
		installVersionPath := "/versions/testString/install"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(installVersionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.WriteHeader(202)
				}))
			})
			It(`Invoke InstallVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.InstallVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")

				// Construct an instance of the InstallVersionOptions model
				installVersionOptionsModel := new(catalogmanagementv1.InstallVersionOptions)
				installVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				installVersionOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				installVersionOptionsModel.ClusterID = core.StringPtr("testString")
				installVersionOptionsModel.Region = core.StringPtr("testString")
				installVersionOptionsModel.Namespace = core.StringPtr("testString")
				installVersionOptionsModel.OverrideValues = make(map[string]interface{})
				installVersionOptionsModel.EntitlementApikey = core.StringPtr("testString")
				installVersionOptionsModel.Schematics = deployRequestBodySchematicsModel
				installVersionOptionsModel.Script = core.StringPtr("testString")
				installVersionOptionsModel.ScriptID = core.StringPtr("testString")
				installVersionOptionsModel.VersionLocatorID = core.StringPtr("testString")
				installVersionOptionsModel.VcenterID = core.StringPtr("testString")
				installVersionOptionsModel.VcenterUser = core.StringPtr("testString")
				installVersionOptionsModel.VcenterPassword = core.StringPtr("testString")
				installVersionOptionsModel.VcenterLocation = core.StringPtr("testString")
				installVersionOptionsModel.VcenterDatastore = core.StringPtr("testString")
				installVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.InstallVersion(installVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke InstallVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")

				// Construct an instance of the InstallVersionOptions model
				installVersionOptionsModel := new(catalogmanagementv1.InstallVersionOptions)
				installVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				installVersionOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				installVersionOptionsModel.ClusterID = core.StringPtr("testString")
				installVersionOptionsModel.Region = core.StringPtr("testString")
				installVersionOptionsModel.Namespace = core.StringPtr("testString")
				installVersionOptionsModel.OverrideValues = make(map[string]interface{})
				installVersionOptionsModel.EntitlementApikey = core.StringPtr("testString")
				installVersionOptionsModel.Schematics = deployRequestBodySchematicsModel
				installVersionOptionsModel.Script = core.StringPtr("testString")
				installVersionOptionsModel.ScriptID = core.StringPtr("testString")
				installVersionOptionsModel.VersionLocatorID = core.StringPtr("testString")
				installVersionOptionsModel.VcenterID = core.StringPtr("testString")
				installVersionOptionsModel.VcenterUser = core.StringPtr("testString")
				installVersionOptionsModel.VcenterPassword = core.StringPtr("testString")
				installVersionOptionsModel.VcenterLocation = core.StringPtr("testString")
				installVersionOptionsModel.VcenterDatastore = core.StringPtr("testString")
				installVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.InstallVersion(installVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the InstallVersionOptions model with no property values
				installVersionOptionsModelNew := new(catalogmanagementv1.InstallVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.InstallVersion(installVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`PreinstallVersion(preinstallVersionOptions *PreinstallVersionOptions)`, func() {
		preinstallVersionPath := "/versions/testString/preinstall"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(preinstallVersionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.WriteHeader(202)
				}))
			})
			It(`Invoke PreinstallVersion successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.PreinstallVersion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")

				// Construct an instance of the PreinstallVersionOptions model
				preinstallVersionOptionsModel := new(catalogmanagementv1.PreinstallVersionOptions)
				preinstallVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				preinstallVersionOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				preinstallVersionOptionsModel.ClusterID = core.StringPtr("testString")
				preinstallVersionOptionsModel.Region = core.StringPtr("testString")
				preinstallVersionOptionsModel.Namespace = core.StringPtr("testString")
				preinstallVersionOptionsModel.OverrideValues = make(map[string]interface{})
				preinstallVersionOptionsModel.EntitlementApikey = core.StringPtr("testString")
				preinstallVersionOptionsModel.Schematics = deployRequestBodySchematicsModel
				preinstallVersionOptionsModel.Script = core.StringPtr("testString")
				preinstallVersionOptionsModel.ScriptID = core.StringPtr("testString")
				preinstallVersionOptionsModel.VersionLocatorID = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterID = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterUser = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterPassword = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterLocation = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterDatastore = core.StringPtr("testString")
				preinstallVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.PreinstallVersion(preinstallVersionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke PreinstallVersion with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")

				// Construct an instance of the PreinstallVersionOptions model
				preinstallVersionOptionsModel := new(catalogmanagementv1.PreinstallVersionOptions)
				preinstallVersionOptionsModel.VersionLocID = core.StringPtr("testString")
				preinstallVersionOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				preinstallVersionOptionsModel.ClusterID = core.StringPtr("testString")
				preinstallVersionOptionsModel.Region = core.StringPtr("testString")
				preinstallVersionOptionsModel.Namespace = core.StringPtr("testString")
				preinstallVersionOptionsModel.OverrideValues = make(map[string]interface{})
				preinstallVersionOptionsModel.EntitlementApikey = core.StringPtr("testString")
				preinstallVersionOptionsModel.Schematics = deployRequestBodySchematicsModel
				preinstallVersionOptionsModel.Script = core.StringPtr("testString")
				preinstallVersionOptionsModel.ScriptID = core.StringPtr("testString")
				preinstallVersionOptionsModel.VersionLocatorID = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterID = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterUser = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterPassword = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterLocation = core.StringPtr("testString")
				preinstallVersionOptionsModel.VcenterDatastore = core.StringPtr("testString")
				preinstallVersionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.PreinstallVersion(preinstallVersionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the PreinstallVersionOptions model with no property values
				preinstallVersionOptionsModelNew := new(catalogmanagementv1.PreinstallVersionOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.PreinstallVersion(preinstallVersionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetPreinstall(getPreinstallOptions *GetPreinstallOptions) - Operation response error`, func() {
		getPreinstallPath := "/versions/testString/preinstall"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getPreinstallPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["namespace"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetPreinstall with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetPreinstallOptions model
				getPreinstallOptionsModel := new(catalogmanagementv1.GetPreinstallOptions)
				getPreinstallOptionsModel.VersionLocID = core.StringPtr("testString")
				getPreinstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getPreinstallOptionsModel.ClusterID = core.StringPtr("testString")
				getPreinstallOptionsModel.Region = core.StringPtr("testString")
				getPreinstallOptionsModel.Namespace = core.StringPtr("testString")
				getPreinstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetPreinstall(getPreinstallOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetPreinstall(getPreinstallOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetPreinstall(getPreinstallOptions *GetPreinstallOptions)`, func() {
		getPreinstallPath := "/versions/testString/preinstall"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getPreinstallPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["namespace"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"metadata": {"cluster_id": "ClusterID", "region": "Region", "namespace": "Namespace", "workspace_id": "WorkspaceID", "workspace_name": "WorkspaceName"}, "release": {"deployments": [{"mapKey": "anyValue"}], "replicasets": [{"mapKey": "anyValue"}], "statefulsets": [{"mapKey": "anyValue"}], "pods": [{"mapKey": "anyValue"}], "errors": [{"mapKey": "Inner"}]}, "content_mgmt": {"pods": [{"mapKey": "Inner"}], "errors": [{"mapKey": "Inner"}]}}`)
				}))
			})
			It(`Invoke GetPreinstall successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetPreinstallOptions model
				getPreinstallOptionsModel := new(catalogmanagementv1.GetPreinstallOptions)
				getPreinstallOptionsModel.VersionLocID = core.StringPtr("testString")
				getPreinstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getPreinstallOptionsModel.ClusterID = core.StringPtr("testString")
				getPreinstallOptionsModel.Region = core.StringPtr("testString")
				getPreinstallOptionsModel.Namespace = core.StringPtr("testString")
				getPreinstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetPreinstallWithContext(ctx, getPreinstallOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetPreinstall(getPreinstallOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetPreinstallWithContext(ctx, getPreinstallOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getPreinstallPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["cluster_id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["region"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["namespace"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"metadata": {"cluster_id": "ClusterID", "region": "Region", "namespace": "Namespace", "workspace_id": "WorkspaceID", "workspace_name": "WorkspaceName"}, "release": {"deployments": [{"mapKey": "anyValue"}], "replicasets": [{"mapKey": "anyValue"}], "statefulsets": [{"mapKey": "anyValue"}], "pods": [{"mapKey": "anyValue"}], "errors": [{"mapKey": "Inner"}]}, "content_mgmt": {"pods": [{"mapKey": "Inner"}], "errors": [{"mapKey": "Inner"}]}}`)
				}))
			})
			It(`Invoke GetPreinstall successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetPreinstall(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetPreinstallOptions model
				getPreinstallOptionsModel := new(catalogmanagementv1.GetPreinstallOptions)
				getPreinstallOptionsModel.VersionLocID = core.StringPtr("testString")
				getPreinstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getPreinstallOptionsModel.ClusterID = core.StringPtr("testString")
				getPreinstallOptionsModel.Region = core.StringPtr("testString")
				getPreinstallOptionsModel.Namespace = core.StringPtr("testString")
				getPreinstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetPreinstall(getPreinstallOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetPreinstall with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetPreinstallOptions model
				getPreinstallOptionsModel := new(catalogmanagementv1.GetPreinstallOptions)
				getPreinstallOptionsModel.VersionLocID = core.StringPtr("testString")
				getPreinstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getPreinstallOptionsModel.ClusterID = core.StringPtr("testString")
				getPreinstallOptionsModel.Region = core.StringPtr("testString")
				getPreinstallOptionsModel.Namespace = core.StringPtr("testString")
				getPreinstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetPreinstall(getPreinstallOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetPreinstallOptions model with no property values
				getPreinstallOptionsModelNew := new(catalogmanagementv1.GetPreinstallOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetPreinstall(getPreinstallOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetPreinstall successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetPreinstallOptions model
				getPreinstallOptionsModel := new(catalogmanagementv1.GetPreinstallOptions)
				getPreinstallOptionsModel.VersionLocID = core.StringPtr("testString")
				getPreinstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getPreinstallOptionsModel.ClusterID = core.StringPtr("testString")
				getPreinstallOptionsModel.Region = core.StringPtr("testString")
				getPreinstallOptionsModel.Namespace = core.StringPtr("testString")
				getPreinstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetPreinstall(getPreinstallOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ValidateInstall(validateInstallOptions *ValidateInstallOptions)`, func() {
		validateInstallPath := "/versions/testString/validation/install"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(validateInstallPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.WriteHeader(202)
				}))
			})
			It(`Invoke ValidateInstall successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.ValidateInstall(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")

				// Construct an instance of the ValidateInstallOptions model
				validateInstallOptionsModel := new(catalogmanagementv1.ValidateInstallOptions)
				validateInstallOptionsModel.VersionLocID = core.StringPtr("testString")
				validateInstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				validateInstallOptionsModel.ClusterID = core.StringPtr("testString")
				validateInstallOptionsModel.Region = core.StringPtr("testString")
				validateInstallOptionsModel.Namespace = core.StringPtr("testString")
				validateInstallOptionsModel.OverrideValues = make(map[string]interface{})
				validateInstallOptionsModel.EntitlementApikey = core.StringPtr("testString")
				validateInstallOptionsModel.Schematics = deployRequestBodySchematicsModel
				validateInstallOptionsModel.Script = core.StringPtr("testString")
				validateInstallOptionsModel.ScriptID = core.StringPtr("testString")
				validateInstallOptionsModel.VersionLocatorID = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterID = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterUser = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterPassword = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterLocation = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterDatastore = core.StringPtr("testString")
				validateInstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.ValidateInstall(validateInstallOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke ValidateInstall with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")

				// Construct an instance of the ValidateInstallOptions model
				validateInstallOptionsModel := new(catalogmanagementv1.ValidateInstallOptions)
				validateInstallOptionsModel.VersionLocID = core.StringPtr("testString")
				validateInstallOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				validateInstallOptionsModel.ClusterID = core.StringPtr("testString")
				validateInstallOptionsModel.Region = core.StringPtr("testString")
				validateInstallOptionsModel.Namespace = core.StringPtr("testString")
				validateInstallOptionsModel.OverrideValues = make(map[string]interface{})
				validateInstallOptionsModel.EntitlementApikey = core.StringPtr("testString")
				validateInstallOptionsModel.Schematics = deployRequestBodySchematicsModel
				validateInstallOptionsModel.Script = core.StringPtr("testString")
				validateInstallOptionsModel.ScriptID = core.StringPtr("testString")
				validateInstallOptionsModel.VersionLocatorID = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterID = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterUser = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterPassword = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterLocation = core.StringPtr("testString")
				validateInstallOptionsModel.VcenterDatastore = core.StringPtr("testString")
				validateInstallOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.ValidateInstall(validateInstallOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the ValidateInstallOptions model with no property values
				validateInstallOptionsModelNew := new(catalogmanagementv1.ValidateInstallOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.ValidateInstall(validateInstallOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetValidationStatus(getValidationStatusOptions *GetValidationStatusOptions) - Operation response error`, func() {
		getValidationStatusPath := "/versions/testString/validation/install"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getValidationStatusPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetValidationStatus with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetValidationStatusOptions model
				getValidationStatusOptionsModel := new(catalogmanagementv1.GetValidationStatusOptions)
				getValidationStatusOptionsModel.VersionLocID = core.StringPtr("testString")
				getValidationStatusOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getValidationStatusOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetValidationStatus(getValidationStatusOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetValidationStatus(getValidationStatusOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetValidationStatus(getValidationStatusOptions *GetValidationStatusOptions)`, func() {
		getValidationStatusPath := "/versions/testString/validation/install"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getValidationStatusPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke GetValidationStatus successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetValidationStatusOptions model
				getValidationStatusOptionsModel := new(catalogmanagementv1.GetValidationStatusOptions)
				getValidationStatusOptionsModel.VersionLocID = core.StringPtr("testString")
				getValidationStatusOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getValidationStatusOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetValidationStatusWithContext(ctx, getValidationStatusOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetValidationStatus(getValidationStatusOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetValidationStatusWithContext(ctx, getValidationStatusOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getValidationStatusPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"validated": "2019-01-01T12:00:00.000Z", "requested": "2019-01-01T12:00:00.000Z", "state": "State", "last_operation": "LastOperation", "target": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke GetValidationStatus successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetValidationStatus(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetValidationStatusOptions model
				getValidationStatusOptionsModel := new(catalogmanagementv1.GetValidationStatusOptions)
				getValidationStatusOptionsModel.VersionLocID = core.StringPtr("testString")
				getValidationStatusOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getValidationStatusOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetValidationStatus(getValidationStatusOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetValidationStatus with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetValidationStatusOptions model
				getValidationStatusOptionsModel := new(catalogmanagementv1.GetValidationStatusOptions)
				getValidationStatusOptionsModel.VersionLocID = core.StringPtr("testString")
				getValidationStatusOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getValidationStatusOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetValidationStatus(getValidationStatusOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetValidationStatusOptions model with no property values
				getValidationStatusOptionsModelNew := new(catalogmanagementv1.GetValidationStatusOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetValidationStatus(getValidationStatusOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetValidationStatus successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetValidationStatusOptions model
				getValidationStatusOptionsModel := new(catalogmanagementv1.GetValidationStatusOptions)
				getValidationStatusOptionsModel.VersionLocID = core.StringPtr("testString")
				getValidationStatusOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				getValidationStatusOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetValidationStatus(getValidationStatusOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOverrideValues(getOverrideValuesOptions *GetOverrideValuesOptions)`, func() {
		getOverrideValuesPath := "/versions/testString/validation/overridevalues"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOverrideValuesPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"mapKey": "anyValue"}`)
				}))
			})
			It(`Invoke GetOverrideValues successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOverrideValuesOptions model
				getOverrideValuesOptionsModel := new(catalogmanagementv1.GetOverrideValuesOptions)
				getOverrideValuesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOverrideValuesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOverrideValuesWithContext(ctx, getOverrideValuesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOverrideValues(getOverrideValuesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOverrideValuesWithContext(ctx, getOverrideValuesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOverrideValuesPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"mapKey": "anyValue"}`)
				}))
			})
			It(`Invoke GetOverrideValues successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOverrideValues(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOverrideValuesOptions model
				getOverrideValuesOptionsModel := new(catalogmanagementv1.GetOverrideValuesOptions)
				getOverrideValuesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOverrideValuesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOverrideValues(getOverrideValuesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOverrideValues with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOverrideValuesOptions model
				getOverrideValuesOptionsModel := new(catalogmanagementv1.GetOverrideValuesOptions)
				getOverrideValuesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOverrideValuesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOverrideValues(getOverrideValuesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOverrideValuesOptions model with no property values
				getOverrideValuesOptionsModelNew := new(catalogmanagementv1.GetOverrideValuesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOverrideValues(getOverrideValuesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOverrideValues successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOverrideValuesOptions model
				getOverrideValuesOptionsModel := new(catalogmanagementv1.GetOverrideValuesOptions)
				getOverrideValuesOptionsModel.VersionLocID = core.StringPtr("testString")
				getOverrideValuesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOverrideValues(getOverrideValuesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`SearchObjects(searchObjectsOptions *SearchObjectsOptions) - Operation response error`, func() {
		searchObjectsPath := "/objects"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(searchObjectsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["query"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// TODO: Add check for collapse query parameter
					// TODO: Add check for digest query parameter
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke SearchObjects with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the SearchObjectsOptions model
				searchObjectsOptionsModel := new(catalogmanagementv1.SearchObjectsOptions)
				searchObjectsOptionsModel.Query = core.StringPtr("testString")
				searchObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				searchObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				searchObjectsOptionsModel.Collapse = core.BoolPtr(true)
				searchObjectsOptionsModel.Digest = core.BoolPtr(true)
				searchObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.SearchObjects(searchObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.SearchObjects(searchObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`SearchObjects(searchObjectsOptions *SearchObjectsOptions)`, func() {
		searchObjectsPath := "/objects"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(searchObjectsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["query"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// TODO: Add check for collapse query parameter
					// TODO: Add check for digest query parameter
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}]}`)
				}))
			})
			It(`Invoke SearchObjects successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the SearchObjectsOptions model
				searchObjectsOptionsModel := new(catalogmanagementv1.SearchObjectsOptions)
				searchObjectsOptionsModel.Query = core.StringPtr("testString")
				searchObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				searchObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				searchObjectsOptionsModel.Collapse = core.BoolPtr(true)
				searchObjectsOptionsModel.Digest = core.BoolPtr(true)
				searchObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.SearchObjectsWithContext(ctx, searchObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.SearchObjects(searchObjectsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.SearchObjectsWithContext(ctx, searchObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(searchObjectsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["query"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// TODO: Add check for collapse query parameter
					// TODO: Add check for digest query parameter
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}]}`)
				}))
			})
			It(`Invoke SearchObjects successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.SearchObjects(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the SearchObjectsOptions model
				searchObjectsOptionsModel := new(catalogmanagementv1.SearchObjectsOptions)
				searchObjectsOptionsModel.Query = core.StringPtr("testString")
				searchObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				searchObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				searchObjectsOptionsModel.Collapse = core.BoolPtr(true)
				searchObjectsOptionsModel.Digest = core.BoolPtr(true)
				searchObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.SearchObjects(searchObjectsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke SearchObjects with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the SearchObjectsOptions model
				searchObjectsOptionsModel := new(catalogmanagementv1.SearchObjectsOptions)
				searchObjectsOptionsModel.Query = core.StringPtr("testString")
				searchObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				searchObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				searchObjectsOptionsModel.Collapse = core.BoolPtr(true)
				searchObjectsOptionsModel.Digest = core.BoolPtr(true)
				searchObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.SearchObjects(searchObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the SearchObjectsOptions model with no property values
				searchObjectsOptionsModelNew := new(catalogmanagementv1.SearchObjectsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.SearchObjects(searchObjectsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke SearchObjects successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the SearchObjectsOptions model
				searchObjectsOptionsModel := new(catalogmanagementv1.SearchObjectsOptions)
				searchObjectsOptionsModel.Query = core.StringPtr("testString")
				searchObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				searchObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				searchObjectsOptionsModel.Collapse = core.BoolPtr(true)
				searchObjectsOptionsModel.Digest = core.BoolPtr(true)
				searchObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.SearchObjects(searchObjectsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListObjects(listObjectsOptions *ListObjectsOptions) - Operation response error`, func() {
		listObjectsPath := "/catalogs/testString/objects"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listObjectsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListObjects with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListObjectsOptions model
				listObjectsOptionsModel := new(catalogmanagementv1.ListObjectsOptions)
				listObjectsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listObjectsOptionsModel.Name = core.StringPtr("testString")
				listObjectsOptionsModel.Sort = core.StringPtr("testString")
				listObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ListObjects(listObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ListObjects(listObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListObjects(listObjectsOptions *ListObjectsOptions)`, func() {
		listObjectsPath := "/catalogs/testString/objects"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listObjectsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}]}`)
				}))
			})
			It(`Invoke ListObjects successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the ListObjectsOptions model
				listObjectsOptionsModel := new(catalogmanagementv1.ListObjectsOptions)
				listObjectsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listObjectsOptionsModel.Name = core.StringPtr("testString")
				listObjectsOptionsModel.Sort = core.StringPtr("testString")
				listObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ListObjectsWithContext(ctx, listObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ListObjects(listObjectsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ListObjectsWithContext(ctx, listObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listObjectsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}]}`)
				}))
			})
			It(`Invoke ListObjects successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ListObjects(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListObjectsOptions model
				listObjectsOptionsModel := new(catalogmanagementv1.ListObjectsOptions)
				listObjectsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listObjectsOptionsModel.Name = core.StringPtr("testString")
				listObjectsOptionsModel.Sort = core.StringPtr("testString")
				listObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ListObjects(listObjectsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListObjects with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListObjectsOptions model
				listObjectsOptionsModel := new(catalogmanagementv1.ListObjectsOptions)
				listObjectsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listObjectsOptionsModel.Name = core.StringPtr("testString")
				listObjectsOptionsModel.Sort = core.StringPtr("testString")
				listObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ListObjects(listObjectsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListObjectsOptions model with no property values
				listObjectsOptionsModelNew := new(catalogmanagementv1.ListObjectsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ListObjects(listObjectsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListObjects successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the ListObjectsOptions model
				listObjectsOptionsModel := new(catalogmanagementv1.ListObjectsOptions)
				listObjectsOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				listObjectsOptionsModel.Limit = core.Int64Ptr(int64(1000))
				listObjectsOptionsModel.Offset = core.Int64Ptr(int64(38))
				listObjectsOptionsModel.Name = core.StringPtr("testString")
				listObjectsOptionsModel.Sort = core.StringPtr("testString")
				listObjectsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ListObjects(listObjectsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateObject(createObjectOptions *CreateObjectOptions) - Operation response error`, func() {
		createObjectPath := "/catalogs/testString/objects"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createObjectPath))
					Expect(req.Method).To(Equal("POST"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateObject with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the CreateObjectOptions model
				createObjectOptionsModel := new(catalogmanagementv1.CreateObjectOptions)
				createObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectOptionsModel.ID = core.StringPtr("testString")
				createObjectOptionsModel.Name = core.StringPtr("testString")
				createObjectOptionsModel.Rev = core.StringPtr("testString")
				createObjectOptionsModel.CRN = core.StringPtr("testString")
				createObjectOptionsModel.URL = core.StringPtr("testString")
				createObjectOptionsModel.ParentID = core.StringPtr("testString")
				createObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				createObjectOptionsModel.Label = core.StringPtr("testString")
				createObjectOptionsModel.Tags = []string{"testString"}
				createObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				createObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				createObjectOptionsModel.Kind = core.StringPtr("testString")
				createObjectOptionsModel.Publish = publishObjectModel
				createObjectOptionsModel.State = stateModel
				createObjectOptionsModel.CatalogID = core.StringPtr("testString")
				createObjectOptionsModel.CatalogName = core.StringPtr("testString")
				createObjectOptionsModel.Data = make(map[string]interface{})
				createObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.CreateObject(createObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.CreateObject(createObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateObject(createObjectOptions *CreateObjectOptions)`, func() {
		createObjectPath := "/catalogs/testString/objects"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createObjectPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke CreateObject successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the CreateObjectOptions model
				createObjectOptionsModel := new(catalogmanagementv1.CreateObjectOptions)
				createObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectOptionsModel.ID = core.StringPtr("testString")
				createObjectOptionsModel.Name = core.StringPtr("testString")
				createObjectOptionsModel.Rev = core.StringPtr("testString")
				createObjectOptionsModel.CRN = core.StringPtr("testString")
				createObjectOptionsModel.URL = core.StringPtr("testString")
				createObjectOptionsModel.ParentID = core.StringPtr("testString")
				createObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				createObjectOptionsModel.Label = core.StringPtr("testString")
				createObjectOptionsModel.Tags = []string{"testString"}
				createObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				createObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				createObjectOptionsModel.Kind = core.StringPtr("testString")
				createObjectOptionsModel.Publish = publishObjectModel
				createObjectOptionsModel.State = stateModel
				createObjectOptionsModel.CatalogID = core.StringPtr("testString")
				createObjectOptionsModel.CatalogName = core.StringPtr("testString")
				createObjectOptionsModel.Data = make(map[string]interface{})
				createObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.CreateObjectWithContext(ctx, createObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.CreateObject(createObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.CreateObjectWithContext(ctx, createObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createObjectPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke CreateObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.CreateObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the CreateObjectOptions model
				createObjectOptionsModel := new(catalogmanagementv1.CreateObjectOptions)
				createObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectOptionsModel.ID = core.StringPtr("testString")
				createObjectOptionsModel.Name = core.StringPtr("testString")
				createObjectOptionsModel.Rev = core.StringPtr("testString")
				createObjectOptionsModel.CRN = core.StringPtr("testString")
				createObjectOptionsModel.URL = core.StringPtr("testString")
				createObjectOptionsModel.ParentID = core.StringPtr("testString")
				createObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				createObjectOptionsModel.Label = core.StringPtr("testString")
				createObjectOptionsModel.Tags = []string{"testString"}
				createObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				createObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				createObjectOptionsModel.Kind = core.StringPtr("testString")
				createObjectOptionsModel.Publish = publishObjectModel
				createObjectOptionsModel.State = stateModel
				createObjectOptionsModel.CatalogID = core.StringPtr("testString")
				createObjectOptionsModel.CatalogName = core.StringPtr("testString")
				createObjectOptionsModel.Data = make(map[string]interface{})
				createObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.CreateObject(createObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the CreateObjectOptions model
				createObjectOptionsModel := new(catalogmanagementv1.CreateObjectOptions)
				createObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectOptionsModel.ID = core.StringPtr("testString")
				createObjectOptionsModel.Name = core.StringPtr("testString")
				createObjectOptionsModel.Rev = core.StringPtr("testString")
				createObjectOptionsModel.CRN = core.StringPtr("testString")
				createObjectOptionsModel.URL = core.StringPtr("testString")
				createObjectOptionsModel.ParentID = core.StringPtr("testString")
				createObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				createObjectOptionsModel.Label = core.StringPtr("testString")
				createObjectOptionsModel.Tags = []string{"testString"}
				createObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				createObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				createObjectOptionsModel.Kind = core.StringPtr("testString")
				createObjectOptionsModel.Publish = publishObjectModel
				createObjectOptionsModel.State = stateModel
				createObjectOptionsModel.CatalogID = core.StringPtr("testString")
				createObjectOptionsModel.CatalogName = core.StringPtr("testString")
				createObjectOptionsModel.Data = make(map[string]interface{})
				createObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.CreateObject(createObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateObjectOptions model with no property values
				createObjectOptionsModelNew := new(catalogmanagementv1.CreateObjectOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.CreateObject(createObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the CreateObjectOptions model
				createObjectOptionsModel := new(catalogmanagementv1.CreateObjectOptions)
				createObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectOptionsModel.ID = core.StringPtr("testString")
				createObjectOptionsModel.Name = core.StringPtr("testString")
				createObjectOptionsModel.Rev = core.StringPtr("testString")
				createObjectOptionsModel.CRN = core.StringPtr("testString")
				createObjectOptionsModel.URL = core.StringPtr("testString")
				createObjectOptionsModel.ParentID = core.StringPtr("testString")
				createObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				createObjectOptionsModel.Label = core.StringPtr("testString")
				createObjectOptionsModel.Tags = []string{"testString"}
				createObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				createObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				createObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				createObjectOptionsModel.Kind = core.StringPtr("testString")
				createObjectOptionsModel.Publish = publishObjectModel
				createObjectOptionsModel.State = stateModel
				createObjectOptionsModel.CatalogID = core.StringPtr("testString")
				createObjectOptionsModel.CatalogName = core.StringPtr("testString")
				createObjectOptionsModel.Data = make(map[string]interface{})
				createObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.CreateObject(createObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObject(getObjectOptions *GetObjectOptions) - Operation response error`, func() {
		getObjectPath := "/catalogs/testString/objects/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetObject with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectOptions model
				getObjectOptionsModel := new(catalogmanagementv1.GetObjectOptions)
				getObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetObject(getObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetObject(getObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObject(getObjectOptions *GetObjectOptions)`, func() {
		getObjectPath := "/catalogs/testString/objects/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke GetObject successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetObjectOptions model
				getObjectOptionsModel := new(catalogmanagementv1.GetObjectOptions)
				getObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetObjectWithContext(ctx, getObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetObject(getObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetObjectWithContext(ctx, getObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke GetObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetObjectOptions model
				getObjectOptionsModel := new(catalogmanagementv1.GetObjectOptions)
				getObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetObject(getObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectOptions model
				getObjectOptionsModel := new(catalogmanagementv1.GetObjectOptions)
				getObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetObject(getObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetObjectOptions model with no property values
				getObjectOptionsModelNew := new(catalogmanagementv1.GetObjectOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetObject(getObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectOptions model
				getObjectOptionsModel := new(catalogmanagementv1.GetObjectOptions)
				getObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetObject(getObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceObject(replaceObjectOptions *ReplaceObjectOptions) - Operation response error`, func() {
		replaceObjectPath := "/catalogs/testString/objects/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceObjectPath))
					Expect(req.Method).To(Equal("PUT"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceObject with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the ReplaceObjectOptions model
				replaceObjectOptionsModel := new(catalogmanagementv1.ReplaceObjectOptions)
				replaceObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ID = core.StringPtr("testString")
				replaceObjectOptionsModel.Name = core.StringPtr("testString")
				replaceObjectOptionsModel.Rev = core.StringPtr("testString")
				replaceObjectOptionsModel.CRN = core.StringPtr("testString")
				replaceObjectOptionsModel.URL = core.StringPtr("testString")
				replaceObjectOptionsModel.ParentID = core.StringPtr("testString")
				replaceObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Label = core.StringPtr("testString")
				replaceObjectOptionsModel.Tags = []string{"testString"}
				replaceObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Kind = core.StringPtr("testString")
				replaceObjectOptionsModel.Publish = publishObjectModel
				replaceObjectOptionsModel.State = stateModel
				replaceObjectOptionsModel.CatalogID = core.StringPtr("testString")
				replaceObjectOptionsModel.CatalogName = core.StringPtr("testString")
				replaceObjectOptionsModel.Data = make(map[string]interface{})
				replaceObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.ReplaceObject(replaceObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.ReplaceObject(replaceObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceObject(replaceObjectOptions *ReplaceObjectOptions)`, func() {
		replaceObjectPath := "/catalogs/testString/objects/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceObjectPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke ReplaceObject successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the ReplaceObjectOptions model
				replaceObjectOptionsModel := new(catalogmanagementv1.ReplaceObjectOptions)
				replaceObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ID = core.StringPtr("testString")
				replaceObjectOptionsModel.Name = core.StringPtr("testString")
				replaceObjectOptionsModel.Rev = core.StringPtr("testString")
				replaceObjectOptionsModel.CRN = core.StringPtr("testString")
				replaceObjectOptionsModel.URL = core.StringPtr("testString")
				replaceObjectOptionsModel.ParentID = core.StringPtr("testString")
				replaceObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Label = core.StringPtr("testString")
				replaceObjectOptionsModel.Tags = []string{"testString"}
				replaceObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Kind = core.StringPtr("testString")
				replaceObjectOptionsModel.Publish = publishObjectModel
				replaceObjectOptionsModel.State = stateModel
				replaceObjectOptionsModel.CatalogID = core.StringPtr("testString")
				replaceObjectOptionsModel.CatalogName = core.StringPtr("testString")
				replaceObjectOptionsModel.Data = make(map[string]interface{})
				replaceObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.ReplaceObjectWithContext(ctx, replaceObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.ReplaceObject(replaceObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.ReplaceObjectWithContext(ctx, replaceObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceObjectPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "name": "Name", "_rev": "Rev", "crn": "CRN", "url": "URL", "parent_id": "ParentID", "label_i18n": "LabelI18n", "label": "Label", "tags": ["Tags"], "created": "2019-01-01T12:00:00.000Z", "updated": "2019-01-01T12:00:00.000Z", "short_description": "ShortDescription", "short_description_i18n": "ShortDescriptionI18n", "kind": "Kind", "publish": {"permit_ibm_public_publish": true, "ibm_approved": false, "public_approved": true, "portal_approval_record": "PortalApprovalRecord", "portal_url": "PortalURL"}, "state": {"current": "Current", "current_entered": "2019-01-01T12:00:00.000Z", "pending": "Pending", "pending_requested": "2019-01-01T12:00:00.000Z", "previous": "Previous"}, "catalog_id": "CatalogID", "catalog_name": "CatalogName", "data": {"mapKey": "anyValue"}}`)
				}))
			})
			It(`Invoke ReplaceObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.ReplaceObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the ReplaceObjectOptions model
				replaceObjectOptionsModel := new(catalogmanagementv1.ReplaceObjectOptions)
				replaceObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ID = core.StringPtr("testString")
				replaceObjectOptionsModel.Name = core.StringPtr("testString")
				replaceObjectOptionsModel.Rev = core.StringPtr("testString")
				replaceObjectOptionsModel.CRN = core.StringPtr("testString")
				replaceObjectOptionsModel.URL = core.StringPtr("testString")
				replaceObjectOptionsModel.ParentID = core.StringPtr("testString")
				replaceObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Label = core.StringPtr("testString")
				replaceObjectOptionsModel.Tags = []string{"testString"}
				replaceObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Kind = core.StringPtr("testString")
				replaceObjectOptionsModel.Publish = publishObjectModel
				replaceObjectOptionsModel.State = stateModel
				replaceObjectOptionsModel.CatalogID = core.StringPtr("testString")
				replaceObjectOptionsModel.CatalogName = core.StringPtr("testString")
				replaceObjectOptionsModel.Data = make(map[string]interface{})
				replaceObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.ReplaceObject(replaceObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the ReplaceObjectOptions model
				replaceObjectOptionsModel := new(catalogmanagementv1.ReplaceObjectOptions)
				replaceObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ID = core.StringPtr("testString")
				replaceObjectOptionsModel.Name = core.StringPtr("testString")
				replaceObjectOptionsModel.Rev = core.StringPtr("testString")
				replaceObjectOptionsModel.CRN = core.StringPtr("testString")
				replaceObjectOptionsModel.URL = core.StringPtr("testString")
				replaceObjectOptionsModel.ParentID = core.StringPtr("testString")
				replaceObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Label = core.StringPtr("testString")
				replaceObjectOptionsModel.Tags = []string{"testString"}
				replaceObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Kind = core.StringPtr("testString")
				replaceObjectOptionsModel.Publish = publishObjectModel
				replaceObjectOptionsModel.State = stateModel
				replaceObjectOptionsModel.CatalogID = core.StringPtr("testString")
				replaceObjectOptionsModel.CatalogName = core.StringPtr("testString")
				replaceObjectOptionsModel.Data = make(map[string]interface{})
				replaceObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.ReplaceObject(replaceObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceObjectOptions model with no property values
				replaceObjectOptionsModelNew := new(catalogmanagementv1.ReplaceObjectOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.ReplaceObject(replaceObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ReplaceObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")

				// Construct an instance of the ReplaceObjectOptions model
				replaceObjectOptionsModel := new(catalogmanagementv1.ReplaceObjectOptions)
				replaceObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				replaceObjectOptionsModel.ID = core.StringPtr("testString")
				replaceObjectOptionsModel.Name = core.StringPtr("testString")
				replaceObjectOptionsModel.Rev = core.StringPtr("testString")
				replaceObjectOptionsModel.CRN = core.StringPtr("testString")
				replaceObjectOptionsModel.URL = core.StringPtr("testString")
				replaceObjectOptionsModel.ParentID = core.StringPtr("testString")
				replaceObjectOptionsModel.LabelI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Label = core.StringPtr("testString")
				replaceObjectOptionsModel.Tags = []string{"testString"}
				replaceObjectOptionsModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				replaceObjectOptionsModel.ShortDescription = core.StringPtr("testString")
				replaceObjectOptionsModel.ShortDescriptionI18n = core.StringPtr("testString")
				replaceObjectOptionsModel.Kind = core.StringPtr("testString")
				replaceObjectOptionsModel.Publish = publishObjectModel
				replaceObjectOptionsModel.State = stateModel
				replaceObjectOptionsModel.CatalogID = core.StringPtr("testString")
				replaceObjectOptionsModel.CatalogName = core.StringPtr("testString")
				replaceObjectOptionsModel.Data = make(map[string]interface{})
				replaceObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.ReplaceObject(replaceObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteObject(deleteObjectOptions *DeleteObjectOptions)`, func() {
		deleteObjectPath := "/catalogs/testString/objects/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteObjectPath))
					Expect(req.Method).To(Equal("DELETE"))

					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteObjectOptions model
				deleteObjectOptionsModel := new(catalogmanagementv1.DeleteObjectOptions)
				deleteObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteObject(deleteObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteObjectOptions model
				deleteObjectOptionsModel := new(catalogmanagementv1.DeleteObjectOptions)
				deleteObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteObject(deleteObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteObjectOptions model with no property values
				deleteObjectOptionsModelNew := new(catalogmanagementv1.DeleteObjectOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteObject(deleteObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObjectAudit(getObjectAuditOptions *GetObjectAuditOptions) - Operation response error`, func() {
		getObjectAuditPath := "/catalogs/testString/objects/testString/audit"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAuditPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetObjectAudit with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAuditOptions model
				getObjectAuditOptionsModel := new(catalogmanagementv1.GetObjectAuditOptions)
				getObjectAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetObjectAudit(getObjectAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetObjectAudit(getObjectAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObjectAudit(getObjectAuditOptions *GetObjectAuditOptions)`, func() {
		getObjectAuditPath := "/catalogs/testString/objects/testString/audit"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetObjectAudit successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetObjectAuditOptions model
				getObjectAuditOptionsModel := new(catalogmanagementv1.GetObjectAuditOptions)
				getObjectAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetObjectAuditWithContext(ctx, getObjectAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetObjectAudit(getObjectAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetObjectAuditWithContext(ctx, getObjectAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAuditPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"list": [{"id": "ID", "created": "2019-01-01T12:00:00.000Z", "change_type": "ChangeType", "target_type": "TargetType", "target_id": "TargetID", "who_delegate_email": "WhoDelegateEmail", "message": "Message"}]}`)
				}))
			})
			It(`Invoke GetObjectAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetObjectAudit(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetObjectAuditOptions model
				getObjectAuditOptionsModel := new(catalogmanagementv1.GetObjectAuditOptions)
				getObjectAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetObjectAudit(getObjectAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetObjectAudit with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAuditOptions model
				getObjectAuditOptionsModel := new(catalogmanagementv1.GetObjectAuditOptions)
				getObjectAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetObjectAudit(getObjectAuditOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetObjectAuditOptions model with no property values
				getObjectAuditOptionsModelNew := new(catalogmanagementv1.GetObjectAuditOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetObjectAudit(getObjectAuditOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetObjectAudit successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAuditOptions model
				getObjectAuditOptionsModel := new(catalogmanagementv1.GetObjectAuditOptions)
				getObjectAuditOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAuditOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetObjectAudit(getObjectAuditOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`AccountPublishObject(accountPublishObjectOptions *AccountPublishObjectOptions)`, func() {
		accountPublishObjectPath := "/catalogs/testString/objects/testString/account-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountPublishObjectPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke AccountPublishObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.AccountPublishObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the AccountPublishObjectOptions model
				accountPublishObjectOptionsModel := new(catalogmanagementv1.AccountPublishObjectOptions)
				accountPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				accountPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				accountPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.AccountPublishObject(accountPublishObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke AccountPublishObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the AccountPublishObjectOptions model
				accountPublishObjectOptionsModel := new(catalogmanagementv1.AccountPublishObjectOptions)
				accountPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				accountPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				accountPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.AccountPublishObject(accountPublishObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the AccountPublishObjectOptions model with no property values
				accountPublishObjectOptionsModelNew := new(catalogmanagementv1.AccountPublishObjectOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.AccountPublishObject(accountPublishObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`SharedPublishObject(sharedPublishObjectOptions *SharedPublishObjectOptions)`, func() {
		sharedPublishObjectPath := "/catalogs/testString/objects/testString/shared-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(sharedPublishObjectPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke SharedPublishObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.SharedPublishObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the SharedPublishObjectOptions model
				sharedPublishObjectOptionsModel := new(catalogmanagementv1.SharedPublishObjectOptions)
				sharedPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				sharedPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				sharedPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.SharedPublishObject(sharedPublishObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke SharedPublishObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the SharedPublishObjectOptions model
				sharedPublishObjectOptionsModel := new(catalogmanagementv1.SharedPublishObjectOptions)
				sharedPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				sharedPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				sharedPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.SharedPublishObject(sharedPublishObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the SharedPublishObjectOptions model with no property values
				sharedPublishObjectOptionsModelNew := new(catalogmanagementv1.SharedPublishObjectOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.SharedPublishObject(sharedPublishObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`IBMPublishObject(ibmPublishObjectOptions *IBMPublishObjectOptions)`, func() {
		ibmPublishObjectPath := "/catalogs/testString/objects/testString/ibm-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(ibmPublishObjectPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke IBMPublishObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.IBMPublishObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the IBMPublishObjectOptions model
				ibmPublishObjectOptionsModel := new(catalogmanagementv1.IBMPublishObjectOptions)
				ibmPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				ibmPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				ibmPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.IBMPublishObject(ibmPublishObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke IBMPublishObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the IBMPublishObjectOptions model
				ibmPublishObjectOptionsModel := new(catalogmanagementv1.IBMPublishObjectOptions)
				ibmPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				ibmPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				ibmPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.IBMPublishObject(ibmPublishObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the IBMPublishObjectOptions model with no property values
				ibmPublishObjectOptionsModelNew := new(catalogmanagementv1.IBMPublishObjectOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.IBMPublishObject(ibmPublishObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`PublicPublishObject(publicPublishObjectOptions *PublicPublishObjectOptions)`, func() {
		publicPublishObjectPath := "/catalogs/testString/objects/testString/public-publish"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(publicPublishObjectPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke PublicPublishObject successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.PublicPublishObject(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the PublicPublishObjectOptions model
				publicPublishObjectOptionsModel := new(catalogmanagementv1.PublicPublishObjectOptions)
				publicPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				publicPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				publicPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.PublicPublishObject(publicPublishObjectOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke PublicPublishObject with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PublicPublishObjectOptions model
				publicPublishObjectOptionsModel := new(catalogmanagementv1.PublicPublishObjectOptions)
				publicPublishObjectOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				publicPublishObjectOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				publicPublishObjectOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.PublicPublishObject(publicPublishObjectOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the PublicPublishObjectOptions model with no property values
				publicPublishObjectOptionsModelNew := new(catalogmanagementv1.PublicPublishObjectOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.PublicPublishObject(publicPublishObjectOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateObjectAccess(createObjectAccessOptions *CreateObjectAccessOptions)`, func() {
		createObjectAccessPath := "/catalogs/testString/objects/testString/access/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createObjectAccessPath))
					Expect(req.Method).To(Equal("POST"))

					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateObjectAccess successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.CreateObjectAccess(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the CreateObjectAccessOptions model
				createObjectAccessOptionsModel := new(catalogmanagementv1.CreateObjectAccessOptions)
				createObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				createObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				createObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.CreateObjectAccess(createObjectAccessOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke CreateObjectAccess with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the CreateObjectAccessOptions model
				createObjectAccessOptionsModel := new(catalogmanagementv1.CreateObjectAccessOptions)
				createObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				createObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				createObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				createObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.CreateObjectAccess(createObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the CreateObjectAccessOptions model with no property values
				createObjectAccessOptionsModelNew := new(catalogmanagementv1.CreateObjectAccessOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.CreateObjectAccess(createObjectAccessOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObjectAccess(getObjectAccessOptions *GetObjectAccessOptions) - Operation response error`, func() {
		getObjectAccessPath := "/catalogs/testString/objects/testString/access/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAccessPath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetObjectAccess with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAccessOptions model
				getObjectAccessOptionsModel := new(catalogmanagementv1.GetObjectAccessOptions)
				getObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetObjectAccess(getObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetObjectAccess(getObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObjectAccess(getObjectAccessOptions *GetObjectAccessOptions)`, func() {
		getObjectAccessPath := "/catalogs/testString/objects/testString/access/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAccessPath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "account": "Account", "catalog_id": "CatalogID", "target_id": "TargetID", "create": "2019-01-01T12:00:00.000Z"}`)
				}))
			})
			It(`Invoke GetObjectAccess successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetObjectAccessOptions model
				getObjectAccessOptionsModel := new(catalogmanagementv1.GetObjectAccessOptions)
				getObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetObjectAccessWithContext(ctx, getObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetObjectAccess(getObjectAccessOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetObjectAccessWithContext(ctx, getObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAccessPath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "account": "Account", "catalog_id": "CatalogID", "target_id": "TargetID", "create": "2019-01-01T12:00:00.000Z"}`)
				}))
			})
			It(`Invoke GetObjectAccess successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetObjectAccess(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetObjectAccessOptions model
				getObjectAccessOptionsModel := new(catalogmanagementv1.GetObjectAccessOptions)
				getObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetObjectAccess(getObjectAccessOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetObjectAccess with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAccessOptions model
				getObjectAccessOptionsModel := new(catalogmanagementv1.GetObjectAccessOptions)
				getObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetObjectAccess(getObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetObjectAccessOptions model with no property values
				getObjectAccessOptionsModelNew := new(catalogmanagementv1.GetObjectAccessOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetObjectAccess(getObjectAccessOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetObjectAccess successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAccessOptions model
				getObjectAccessOptionsModel := new(catalogmanagementv1.GetObjectAccessOptions)
				getObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				getObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetObjectAccess(getObjectAccessOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteObjectAccess(deleteObjectAccessOptions *DeleteObjectAccessOptions)`, func() {
		deleteObjectAccessPath := "/catalogs/testString/objects/testString/access/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteObjectAccessPath))
					Expect(req.Method).To(Equal("DELETE"))

					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteObjectAccess successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteObjectAccess(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteObjectAccessOptions model
				deleteObjectAccessOptionsModel := new(catalogmanagementv1.DeleteObjectAccessOptions)
				deleteObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				deleteObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteObjectAccess(deleteObjectAccessOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteObjectAccess with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteObjectAccessOptions model
				deleteObjectAccessOptionsModel := new(catalogmanagementv1.DeleteObjectAccessOptions)
				deleteObjectAccessOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessOptionsModel.AccountIdentifier = core.StringPtr("testString")
				deleteObjectAccessOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteObjectAccess(deleteObjectAccessOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteObjectAccessOptions model with no property values
				deleteObjectAccessOptionsModelNew := new(catalogmanagementv1.DeleteObjectAccessOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteObjectAccess(deleteObjectAccessOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObjectAccessList(getObjectAccessListOptions *GetObjectAccessListOptions) - Operation response error`, func() {
		getObjectAccessListPath := "/catalogs/testString/objects/testString/access"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAccessListPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetObjectAccessList with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAccessListOptions model
				getObjectAccessListOptionsModel := new(catalogmanagementv1.GetObjectAccessListOptions)
				getObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getObjectAccessListOptionsModel.Offset = core.Int64Ptr(int64(38))
				getObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetObjectAccessList(getObjectAccessListOptions *GetObjectAccessListOptions)`, func() {
		getObjectAccessListPath := "/catalogs/testString/objects/testString/access"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAccessListPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "account": "Account", "catalog_id": "CatalogID", "target_id": "TargetID", "create": "2019-01-01T12:00:00.000Z"}]}`)
				}))
			})
			It(`Invoke GetObjectAccessList successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetObjectAccessListOptions model
				getObjectAccessListOptionsModel := new(catalogmanagementv1.GetObjectAccessListOptions)
				getObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getObjectAccessListOptionsModel.Offset = core.Int64Ptr(int64(38))
				getObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetObjectAccessListWithContext(ctx, getObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetObjectAccessListWithContext(ctx, getObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getObjectAccessListPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1000))}))
					Expect(req.URL.Query()["offset"]).To(Equal([]string{fmt.Sprint(int64(38))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"offset": 6, "limit": 5, "total_count": 10, "resource_count": 13, "first": "First", "last": "Last", "prev": "Prev", "next": "Next", "resources": [{"id": "ID", "account": "Account", "catalog_id": "CatalogID", "target_id": "TargetID", "create": "2019-01-01T12:00:00.000Z"}]}`)
				}))
			})
			It(`Invoke GetObjectAccessList successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetObjectAccessList(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetObjectAccessListOptions model
				getObjectAccessListOptionsModel := new(catalogmanagementv1.GetObjectAccessListOptions)
				getObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getObjectAccessListOptionsModel.Offset = core.Int64Ptr(int64(38))
				getObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetObjectAccessList with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAccessListOptions model
				getObjectAccessListOptionsModel := new(catalogmanagementv1.GetObjectAccessListOptions)
				getObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getObjectAccessListOptionsModel.Offset = core.Int64Ptr(int64(38))
				getObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetObjectAccessListOptions model with no property values
				getObjectAccessListOptionsModelNew := new(catalogmanagementv1.GetObjectAccessListOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetObjectAccessList successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetObjectAccessListOptions model
				getObjectAccessListOptionsModel := new(catalogmanagementv1.GetObjectAccessListOptions)
				getObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				getObjectAccessListOptionsModel.Limit = core.Int64Ptr(int64(1000))
				getObjectAccessListOptionsModel.Offset = core.Int64Ptr(int64(38))
				getObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetObjectAccessList(getObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteObjectAccessList(deleteObjectAccessListOptions *DeleteObjectAccessListOptions) - Operation response error`, func() {
		deleteObjectAccessListPath := "/catalogs/testString/objects/testString/access"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteObjectAccessListPath))
					Expect(req.Method).To(Equal("DELETE"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke DeleteObjectAccessList with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteObjectAccessListOptions model
				deleteObjectAccessListOptionsModel := new(catalogmanagementv1.DeleteObjectAccessListOptions)
				deleteObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.Accounts = []string{"testString"}
				deleteObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteObjectAccessList(deleteObjectAccessListOptions *DeleteObjectAccessListOptions)`, func() {
		deleteObjectAccessListPath := "/catalogs/testString/objects/testString/access"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteObjectAccessListPath))
					Expect(req.Method).To(Equal("DELETE"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"errors": {"mapKey": "Inner"}}`)
				}))
			})
			It(`Invoke DeleteObjectAccessList successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the DeleteObjectAccessListOptions model
				deleteObjectAccessListOptionsModel := new(catalogmanagementv1.DeleteObjectAccessListOptions)
				deleteObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.Accounts = []string{"testString"}
				deleteObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.DeleteObjectAccessListWithContext(ctx, deleteObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.DeleteObjectAccessListWithContext(ctx, deleteObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteObjectAccessListPath))
					Expect(req.Method).To(Equal("DELETE"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"errors": {"mapKey": "Inner"}}`)
				}))
			})
			It(`Invoke DeleteObjectAccessList successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.DeleteObjectAccessList(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the DeleteObjectAccessListOptions model
				deleteObjectAccessListOptionsModel := new(catalogmanagementv1.DeleteObjectAccessListOptions)
				deleteObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.Accounts = []string{"testString"}
				deleteObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke DeleteObjectAccessList with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteObjectAccessListOptions model
				deleteObjectAccessListOptionsModel := new(catalogmanagementv1.DeleteObjectAccessListOptions)
				deleteObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.Accounts = []string{"testString"}
				deleteObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the DeleteObjectAccessListOptions model with no property values
				deleteObjectAccessListOptionsModelNew := new(catalogmanagementv1.DeleteObjectAccessListOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteObjectAccessList successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteObjectAccessListOptions model
				deleteObjectAccessListOptionsModel := new(catalogmanagementv1.DeleteObjectAccessListOptions)
				deleteObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				deleteObjectAccessListOptionsModel.Accounts = []string{"testString"}
				deleteObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.DeleteObjectAccessList(deleteObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`AddObjectAccessList(addObjectAccessListOptions *AddObjectAccessListOptions) - Operation response error`, func() {
		addObjectAccessListPath := "/catalogs/testString/objects/testString/access"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(addObjectAccessListPath))
					Expect(req.Method).To(Equal("POST"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke AddObjectAccessList with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the AddObjectAccessListOptions model
				addObjectAccessListOptionsModel := new(catalogmanagementv1.AddObjectAccessListOptions)
				addObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.Accounts = []string{"testString"}
				addObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`AddObjectAccessList(addObjectAccessListOptions *AddObjectAccessListOptions)`, func() {
		addObjectAccessListPath := "/catalogs/testString/objects/testString/access"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(addObjectAccessListPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"errors": {"mapKey": "Inner"}}`)
				}))
			})
			It(`Invoke AddObjectAccessList successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the AddObjectAccessListOptions model
				addObjectAccessListOptionsModel := new(catalogmanagementv1.AddObjectAccessListOptions)
				addObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.Accounts = []string{"testString"}
				addObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.AddObjectAccessListWithContext(ctx, addObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.AddObjectAccessListWithContext(ctx, addObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(addObjectAccessListPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"errors": {"mapKey": "Inner"}}`)
				}))
			})
			It(`Invoke AddObjectAccessList successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.AddObjectAccessList(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the AddObjectAccessListOptions model
				addObjectAccessListOptionsModel := new(catalogmanagementv1.AddObjectAccessListOptions)
				addObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.Accounts = []string{"testString"}
				addObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke AddObjectAccessList with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the AddObjectAccessListOptions model
				addObjectAccessListOptionsModel := new(catalogmanagementv1.AddObjectAccessListOptions)
				addObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.Accounts = []string{"testString"}
				addObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the AddObjectAccessListOptions model with no property values
				addObjectAccessListOptionsModelNew := new(catalogmanagementv1.AddObjectAccessListOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke AddObjectAccessList successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the AddObjectAccessListOptions model
				addObjectAccessListOptionsModel := new(catalogmanagementv1.AddObjectAccessListOptions)
				addObjectAccessListOptionsModel.CatalogIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.ObjectIdentifier = core.StringPtr("testString")
				addObjectAccessListOptionsModel.Accounts = []string{"testString"}
				addObjectAccessListOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.AddObjectAccessList(addObjectAccessListOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateOfferingInstance(createOfferingInstanceOptions *CreateOfferingInstanceOptions) - Operation response error`, func() {
		createOfferingInstancePath := "/instances/offerings"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createOfferingInstancePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateOfferingInstance with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the CreateOfferingInstanceOptions model
				createOfferingInstanceOptionsModel := new(catalogmanagementv1.CreateOfferingInstanceOptions)
				createOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				createOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				createOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateOfferingInstance(createOfferingInstanceOptions *CreateOfferingInstanceOptions)`, func() {
		createOfferingInstancePath := "/instances/offerings"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createOfferingInstancePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "catalog_id": "CatalogID", "offering_id": "OfferingID", "kind_format": "KindFormat", "version": "Version", "cluster_id": "ClusterID", "cluster_region": "ClusterRegion", "cluster_namespaces": ["ClusterNamespaces"], "cluster_all_namespaces": true, "schematics_workspace_id": "SchematicsWorkspaceID", "resource_group_id": "ResourceGroupID"}`)
				}))
			})
			It(`Invoke CreateOfferingInstance successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the CreateOfferingInstanceOptions model
				createOfferingInstanceOptionsModel := new(catalogmanagementv1.CreateOfferingInstanceOptions)
				createOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				createOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				createOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.CreateOfferingInstanceWithContext(ctx, createOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.CreateOfferingInstanceWithContext(ctx, createOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createOfferingInstancePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "catalog_id": "CatalogID", "offering_id": "OfferingID", "kind_format": "KindFormat", "version": "Version", "cluster_id": "ClusterID", "cluster_region": "ClusterRegion", "cluster_namespaces": ["ClusterNamespaces"], "cluster_all_namespaces": true, "schematics_workspace_id": "SchematicsWorkspaceID", "resource_group_id": "ResourceGroupID"}`)
				}))
			})
			It(`Invoke CreateOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.CreateOfferingInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the CreateOfferingInstanceOptions model
				createOfferingInstanceOptionsModel := new(catalogmanagementv1.CreateOfferingInstanceOptions)
				createOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				createOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				createOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateOfferingInstance with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the CreateOfferingInstanceOptions model
				createOfferingInstanceOptionsModel := new(catalogmanagementv1.CreateOfferingInstanceOptions)
				createOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				createOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				createOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateOfferingInstanceOptions model with no property values
				createOfferingInstanceOptionsModelNew := new(catalogmanagementv1.CreateOfferingInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the CreateOfferingInstanceOptions model
				createOfferingInstanceOptionsModel := new(catalogmanagementv1.CreateOfferingInstanceOptions)
				createOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				createOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				createOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				createOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.CreateOfferingInstance(createOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingInstance(getOfferingInstanceOptions *GetOfferingInstanceOptions) - Operation response error`, func() {
		getOfferingInstancePath := "/instances/offerings/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingInstancePath))
					Expect(req.Method).To(Equal("GET"))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOfferingInstance with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingInstanceOptions model
				getOfferingInstanceOptionsModel := new(catalogmanagementv1.GetOfferingInstanceOptions)
				getOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				getOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOfferingInstance(getOfferingInstanceOptions *GetOfferingInstanceOptions)`, func() {
		getOfferingInstancePath := "/instances/offerings/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingInstancePath))
					Expect(req.Method).To(Equal("GET"))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "catalog_id": "CatalogID", "offering_id": "OfferingID", "kind_format": "KindFormat", "version": "Version", "cluster_id": "ClusterID", "cluster_region": "ClusterRegion", "cluster_namespaces": ["ClusterNamespaces"], "cluster_all_namespaces": true, "schematics_workspace_id": "SchematicsWorkspaceID", "resource_group_id": "ResourceGroupID"}`)
				}))
			})
			It(`Invoke GetOfferingInstance successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the GetOfferingInstanceOptions model
				getOfferingInstanceOptionsModel := new(catalogmanagementv1.GetOfferingInstanceOptions)
				getOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				getOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.GetOfferingInstanceWithContext(ctx, getOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.GetOfferingInstanceWithContext(ctx, getOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOfferingInstancePath))
					Expect(req.Method).To(Equal("GET"))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "catalog_id": "CatalogID", "offering_id": "OfferingID", "kind_format": "KindFormat", "version": "Version", "cluster_id": "ClusterID", "cluster_region": "ClusterRegion", "cluster_namespaces": ["ClusterNamespaces"], "cluster_all_namespaces": true, "schematics_workspace_id": "SchematicsWorkspaceID", "resource_group_id": "ResourceGroupID"}`)
				}))
			})
			It(`Invoke GetOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.GetOfferingInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOfferingInstanceOptions model
				getOfferingInstanceOptionsModel := new(catalogmanagementv1.GetOfferingInstanceOptions)
				getOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				getOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOfferingInstance with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingInstanceOptions model
				getOfferingInstanceOptionsModel := new(catalogmanagementv1.GetOfferingInstanceOptions)
				getOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				getOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOfferingInstanceOptions model with no property values
				getOfferingInstanceOptionsModelNew := new(catalogmanagementv1.GetOfferingInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the GetOfferingInstanceOptions model
				getOfferingInstanceOptionsModel := new(catalogmanagementv1.GetOfferingInstanceOptions)
				getOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				getOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.GetOfferingInstance(getOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`PutOfferingInstance(putOfferingInstanceOptions *PutOfferingInstanceOptions) - Operation response error`, func() {
		putOfferingInstancePath := "/instances/offerings/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(putOfferingInstancePath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke PutOfferingInstance with error: Operation response processing error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PutOfferingInstanceOptions model
				putOfferingInstanceOptionsModel := new(catalogmanagementv1.PutOfferingInstanceOptions)
				putOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				putOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				putOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				catalogManagementService.EnableRetries(0, 0)
				result, response, operationErr = catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`PutOfferingInstance(putOfferingInstanceOptions *PutOfferingInstanceOptions)`, func() {
		putOfferingInstancePath := "/instances/offerings/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(putOfferingInstancePath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "catalog_id": "CatalogID", "offering_id": "OfferingID", "kind_format": "KindFormat", "version": "Version", "cluster_id": "ClusterID", "cluster_region": "ClusterRegion", "cluster_namespaces": ["ClusterNamespaces"], "cluster_all_namespaces": true, "schematics_workspace_id": "SchematicsWorkspaceID", "resource_group_id": "ResourceGroupID"}`)
				}))
			})
			It(`Invoke PutOfferingInstance successfully with retries`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())
				catalogManagementService.EnableRetries(0, 0)

				// Construct an instance of the PutOfferingInstanceOptions model
				putOfferingInstanceOptionsModel := new(catalogmanagementv1.PutOfferingInstanceOptions)
				putOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				putOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				putOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := catalogManagementService.PutOfferingInstanceWithContext(ctx, putOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				catalogManagementService.DisableRetries()
				result, response, operationErr := catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = catalogManagementService.PutOfferingInstanceWithContext(ctx, putOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(putOfferingInstancePath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"id": "ID", "_rev": "Rev", "url": "URL", "crn": "CRN", "label": "Label", "catalog_id": "CatalogID", "offering_id": "OfferingID", "kind_format": "KindFormat", "version": "Version", "cluster_id": "ClusterID", "cluster_region": "ClusterRegion", "cluster_namespaces": ["ClusterNamespaces"], "cluster_all_namespaces": true, "schematics_workspace_id": "SchematicsWorkspaceID", "resource_group_id": "ResourceGroupID"}`)
				}))
			})
			It(`Invoke PutOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := catalogManagementService.PutOfferingInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the PutOfferingInstanceOptions model
				putOfferingInstanceOptionsModel := new(catalogmanagementv1.PutOfferingInstanceOptions)
				putOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				putOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				putOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke PutOfferingInstance with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PutOfferingInstanceOptions model
				putOfferingInstanceOptionsModel := new(catalogmanagementv1.PutOfferingInstanceOptions)
				putOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				putOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				putOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the PutOfferingInstanceOptions model with no property values
				putOfferingInstanceOptionsModelNew := new(catalogmanagementv1.PutOfferingInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke PutOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the PutOfferingInstanceOptions model
				putOfferingInstanceOptionsModel := new(catalogmanagementv1.PutOfferingInstanceOptions)
				putOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Rev = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.URL = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CRN = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Label = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.CatalogID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.OfferingID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.KindFormat = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Version = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterRegion = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ClusterNamespaces = []string{"testString"}
				putOfferingInstanceOptionsModel.ClusterAllNamespaces = core.BoolPtr(true)
				putOfferingInstanceOptionsModel.SchematicsWorkspaceID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.ResourceGroupID = core.StringPtr("testString")
				putOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := catalogManagementService.PutOfferingInstance(putOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteOfferingInstance(deleteOfferingInstanceOptions *DeleteOfferingInstanceOptions)`, func() {
		deleteOfferingInstancePath := "/instances/offerings/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteOfferingInstancePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.Header["X-Auth-Refresh-Token"]).ToNot(BeNil())
					Expect(req.Header["X-Auth-Refresh-Token"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteOfferingInstance successfully`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := catalogManagementService.DeleteOfferingInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteOfferingInstanceOptions model
				deleteOfferingInstanceOptionsModel := new(catalogmanagementv1.DeleteOfferingInstanceOptions)
				deleteOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				deleteOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deleteOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = catalogManagementService.DeleteOfferingInstance(deleteOfferingInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteOfferingInstance with error: Operation validation and request error`, func() {
				catalogManagementService, serviceErr := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
				})
				Expect(serviceErr).To(BeNil())
				Expect(catalogManagementService).ToNot(BeNil())

				// Construct an instance of the DeleteOfferingInstanceOptions model
				deleteOfferingInstanceOptionsModel := new(catalogmanagementv1.DeleteOfferingInstanceOptions)
				deleteOfferingInstanceOptionsModel.InstanceIdentifier = core.StringPtr("testString")
				deleteOfferingInstanceOptionsModel.XAuthRefreshToken = core.StringPtr("testString")
				deleteOfferingInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := catalogManagementService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := catalogManagementService.DeleteOfferingInstance(deleteOfferingInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteOfferingInstanceOptions model with no property values
				deleteOfferingInstanceOptionsModelNew := new(catalogmanagementv1.DeleteOfferingInstanceOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = catalogManagementService.DeleteOfferingInstance(deleteOfferingInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Model constructor tests`, func() {
		Context(`Using a service client instance`, func() {
			catalogManagementService, _ := catalogmanagementv1.NewCatalogManagementV1(&catalogmanagementv1.CatalogManagementV1Options{
				URL:           "http://catalogmanagementv1modelgenerator.com",
				Authenticator: &core.NoAuthAuthenticator{},
			})
			It(`Invoke NewAccountPublishObjectOptions successfully`, func() {
				// Construct an instance of the AccountPublishObjectOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				accountPublishObjectOptionsModel := catalogManagementService.NewAccountPublishObjectOptions(catalogIdentifier, objectIdentifier)
				accountPublishObjectOptionsModel.SetCatalogIdentifier("testString")
				accountPublishObjectOptionsModel.SetObjectIdentifier("testString")
				accountPublishObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(accountPublishObjectOptionsModel).ToNot(BeNil())
				Expect(accountPublishObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(accountPublishObjectOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(accountPublishObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewAccountPublishVersionOptions successfully`, func() {
				// Construct an instance of the AccountPublishVersionOptions model
				versionLocID := "testString"
				accountPublishVersionOptionsModel := catalogManagementService.NewAccountPublishVersionOptions(versionLocID)
				accountPublishVersionOptionsModel.SetVersionLocID("testString")
				accountPublishVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(accountPublishVersionOptionsModel).ToNot(BeNil())
				Expect(accountPublishVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(accountPublishVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewAddObjectAccessListOptions successfully`, func() {
				// Construct an instance of the AddObjectAccessListOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				accounts := []string{"testString"}
				addObjectAccessListOptionsModel := catalogManagementService.NewAddObjectAccessListOptions(catalogIdentifier, objectIdentifier, accounts)
				addObjectAccessListOptionsModel.SetCatalogIdentifier("testString")
				addObjectAccessListOptionsModel.SetObjectIdentifier("testString")
				addObjectAccessListOptionsModel.SetAccounts([]string{"testString"})
				addObjectAccessListOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(addObjectAccessListOptionsModel).ToNot(BeNil())
				Expect(addObjectAccessListOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(addObjectAccessListOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(addObjectAccessListOptionsModel.Accounts).To(Equal([]string{"testString"}))
				Expect(addObjectAccessListOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCommitVersionOptions successfully`, func() {
				// Construct an instance of the CommitVersionOptions model
				versionLocID := "testString"
				commitVersionOptionsModel := catalogManagementService.NewCommitVersionOptions(versionLocID)
				commitVersionOptionsModel.SetVersionLocID("testString")
				commitVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(commitVersionOptionsModel).ToNot(BeNil())
				Expect(commitVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(commitVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCopyVersionOptions successfully`, func() {
				// Construct an instance of the CopyVersionOptions model
				versionLocID := "testString"
				copyVersionOptionsModel := catalogManagementService.NewCopyVersionOptions(versionLocID)
				copyVersionOptionsModel.SetVersionLocID("testString")
				copyVersionOptionsModel.SetTags([]string{"testString"})
				copyVersionOptionsModel.SetTargetKinds([]string{"testString"})
				copyVersionOptionsModel.SetContent(*CreateMockByteArray("This is a mock byte array value."))
				copyVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(copyVersionOptionsModel).ToNot(BeNil())
				Expect(copyVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(copyVersionOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(copyVersionOptionsModel.TargetKinds).To(Equal([]string{"testString"}))
				Expect(copyVersionOptionsModel.Content).To(Equal(CreateMockByteArray("This is a mock byte array value.")))
				Expect(copyVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateCatalogOptions successfully`, func() {
				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				Expect(featureModel).ToNot(BeNil())
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")
				Expect(featureModel.Title).To(Equal(core.StringPtr("testString")))
				Expect(featureModel.Description).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				Expect(filterTermsModel).ToNot(BeNil())
				filterTermsModel.FilterTerms = []string{"testString"}
				Expect(filterTermsModel.FilterTerms).To(Equal([]string{"testString"}))

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				Expect(categoryFilterModel).ToNot(BeNil())
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel
				Expect(categoryFilterModel.Include).To(Equal(core.BoolPtr(true)))
				Expect(categoryFilterModel.Filter).To(Equal(filterTermsModel))

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				Expect(idFilterModel).ToNot(BeNil())
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel
				Expect(idFilterModel.Include).To(Equal(filterTermsModel))
				Expect(idFilterModel.Exclude).To(Equal(filterTermsModel))

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				Expect(filtersModel).ToNot(BeNil())
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel
				Expect(filtersModel.IncludeAll).To(Equal(core.BoolPtr(true)))
				Expect(filtersModel.IDFilters).To(Equal(idFilterModel))
				Expect(filtersModel.CategoryFilters["foo"]).To(Equal(*categoryFilterModel))

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				Expect(syndicationClusterModel).ToNot(BeNil())
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)
				Expect(syndicationClusterModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.ResourceGroupName).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.Type).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.Namespaces).To(Equal([]string{"testString"}))
				Expect(syndicationClusterModel.AllNamespaces).To(Equal(core.BoolPtr(true)))

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				Expect(syndicationHistoryModel).ToNot(BeNil())
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				Expect(syndicationHistoryModel.Namespaces).To(Equal([]string{"testString"}))
				Expect(syndicationHistoryModel.Clusters).To(Equal([]catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}))
				Expect(syndicationHistoryModel.LastRun).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				Expect(syndicationAuthorizationModel).ToNot(BeNil())
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				Expect(syndicationAuthorizationModel.Token).To(Equal(core.StringPtr("testString")))
				Expect(syndicationAuthorizationModel.LastRun).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				Expect(syndicationResourceModel).ToNot(BeNil())
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel
				Expect(syndicationResourceModel.RemoveRelatedComponents).To(Equal(core.BoolPtr(true)))
				Expect(syndicationResourceModel.Clusters).To(Equal([]catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}))
				Expect(syndicationResourceModel.History).To(Equal(syndicationHistoryModel))
				Expect(syndicationResourceModel.Authorization).To(Equal(syndicationAuthorizationModel))

				// Construct an instance of the CreateCatalogOptions model
				createCatalogOptionsModel := catalogManagementService.NewCreateCatalogOptions()
				createCatalogOptionsModel.SetID("testString")
				createCatalogOptionsModel.SetRev("testString")
				createCatalogOptionsModel.SetLabel("testString")
				createCatalogOptionsModel.SetShortDescription("testString")
				createCatalogOptionsModel.SetCatalogIconURL("testString")
				createCatalogOptionsModel.SetTags([]string{"testString"})
				createCatalogOptionsModel.SetFeatures([]catalogmanagementv1.Feature{*featureModel})
				createCatalogOptionsModel.SetDisabled(true)
				createCatalogOptionsModel.SetResourceGroupID("testString")
				createCatalogOptionsModel.SetOwningAccount("testString")
				createCatalogOptionsModel.SetCatalogFilters(filtersModel)
				createCatalogOptionsModel.SetSyndicationSettings(syndicationResourceModel)
				createCatalogOptionsModel.SetKind("testString")
				createCatalogOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createCatalogOptionsModel).ToNot(BeNil())
				Expect(createCatalogOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.CatalogIconURL).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(createCatalogOptionsModel.Features).To(Equal([]catalogmanagementv1.Feature{*featureModel}))
				Expect(createCatalogOptionsModel.Disabled).To(Equal(core.BoolPtr(true)))
				Expect(createCatalogOptionsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.OwningAccount).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.CatalogFilters).To(Equal(filtersModel))
				Expect(createCatalogOptionsModel.SyndicationSettings).To(Equal(syndicationResourceModel))
				Expect(createCatalogOptionsModel.Kind).To(Equal(core.StringPtr("testString")))
				Expect(createCatalogOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateObjectAccessOptions successfully`, func() {
				// Construct an instance of the CreateObjectAccessOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				accountIdentifier := "testString"
				createObjectAccessOptionsModel := catalogManagementService.NewCreateObjectAccessOptions(catalogIdentifier, objectIdentifier, accountIdentifier)
				createObjectAccessOptionsModel.SetCatalogIdentifier("testString")
				createObjectAccessOptionsModel.SetObjectIdentifier("testString")
				createObjectAccessOptionsModel.SetAccountIdentifier("testString")
				createObjectAccessOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createObjectAccessOptionsModel).ToNot(BeNil())
				Expect(createObjectAccessOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(createObjectAccessOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(createObjectAccessOptionsModel.AccountIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(createObjectAccessOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateObjectOptions successfully`, func() {
				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				Expect(publishObjectModel).ToNot(BeNil())
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")
				Expect(publishObjectModel.PermitIBMPublicPublish).To(Equal(core.BoolPtr(true)))
				Expect(publishObjectModel.IBMApproved).To(Equal(core.BoolPtr(true)))
				Expect(publishObjectModel.PublicApproved).To(Equal(core.BoolPtr(true)))
				Expect(publishObjectModel.PortalApprovalRecord).To(Equal(core.StringPtr("testString")))
				Expect(publishObjectModel.PortalURL).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				Expect(stateModel).ToNot(BeNil())
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")
				Expect(stateModel.Current).To(Equal(core.StringPtr("testString")))
				Expect(stateModel.CurrentEntered).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(stateModel.Pending).To(Equal(core.StringPtr("testString")))
				Expect(stateModel.PendingRequested).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(stateModel.Previous).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the CreateObjectOptions model
				catalogIdentifier := "testString"
				createObjectOptionsModel := catalogManagementService.NewCreateObjectOptions(catalogIdentifier)
				createObjectOptionsModel.SetCatalogIdentifier("testString")
				createObjectOptionsModel.SetID("testString")
				createObjectOptionsModel.SetName("testString")
				createObjectOptionsModel.SetRev("testString")
				createObjectOptionsModel.SetCRN("testString")
				createObjectOptionsModel.SetURL("testString")
				createObjectOptionsModel.SetParentID("testString")
				createObjectOptionsModel.SetLabelI18n("testString")
				createObjectOptionsModel.SetLabel("testString")
				createObjectOptionsModel.SetTags([]string{"testString"})
				createObjectOptionsModel.SetCreated(CreateMockDateTime("2019-01-01T12:00:00.000Z"))
				createObjectOptionsModel.SetUpdated(CreateMockDateTime("2019-01-01T12:00:00.000Z"))
				createObjectOptionsModel.SetShortDescription("testString")
				createObjectOptionsModel.SetShortDescriptionI18n("testString")
				createObjectOptionsModel.SetKind("testString")
				createObjectOptionsModel.SetPublish(publishObjectModel)
				createObjectOptionsModel.SetState(stateModel)
				createObjectOptionsModel.SetCatalogID("testString")
				createObjectOptionsModel.SetCatalogName("testString")
				createObjectOptionsModel.SetData(make(map[string]interface{}))
				createObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createObjectOptionsModel).ToNot(BeNil())
				Expect(createObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.CRN).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.URL).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.ParentID).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.LabelI18n).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(createObjectOptionsModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(createObjectOptionsModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(createObjectOptionsModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.ShortDescriptionI18n).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Kind).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Publish).To(Equal(publishObjectModel))
				Expect(createObjectOptionsModel.State).To(Equal(stateModel))
				Expect(createObjectOptionsModel.CatalogID).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.CatalogName).To(Equal(core.StringPtr("testString")))
				Expect(createObjectOptionsModel.Data).To(Equal(make(map[string]interface{})))
				Expect(createObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateOfferingInstanceOptions successfully`, func() {
				// Construct an instance of the CreateOfferingInstanceOptions model
				xAuthRefreshToken := "testString"
				createOfferingInstanceOptionsModel := catalogManagementService.NewCreateOfferingInstanceOptions(xAuthRefreshToken)
				createOfferingInstanceOptionsModel.SetXAuthRefreshToken("testString")
				createOfferingInstanceOptionsModel.SetID("testString")
				createOfferingInstanceOptionsModel.SetRev("testString")
				createOfferingInstanceOptionsModel.SetURL("testString")
				createOfferingInstanceOptionsModel.SetCRN("testString")
				createOfferingInstanceOptionsModel.SetLabel("testString")
				createOfferingInstanceOptionsModel.SetCatalogID("testString")
				createOfferingInstanceOptionsModel.SetOfferingID("testString")
				createOfferingInstanceOptionsModel.SetKindFormat("testString")
				createOfferingInstanceOptionsModel.SetVersion("testString")
				createOfferingInstanceOptionsModel.SetClusterID("testString")
				createOfferingInstanceOptionsModel.SetClusterRegion("testString")
				createOfferingInstanceOptionsModel.SetClusterNamespaces([]string{"testString"})
				createOfferingInstanceOptionsModel.SetClusterAllNamespaces(true)
				createOfferingInstanceOptionsModel.SetSchematicsWorkspaceID("testString")
				createOfferingInstanceOptionsModel.SetResourceGroupID("testString")
				createOfferingInstanceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createOfferingInstanceOptionsModel).ToNot(BeNil())
				Expect(createOfferingInstanceOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.URL).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.CRN).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.CatalogID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.KindFormat).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.Version).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.ClusterRegion).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.ClusterNamespaces).To(Equal([]string{"testString"}))
				Expect(createOfferingInstanceOptionsModel.ClusterAllNamespaces).To(Equal(core.BoolPtr(true)))
				Expect(createOfferingInstanceOptionsModel.SchematicsWorkspaceID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingInstanceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateOfferingOptions successfully`, func() {
				// Construct an instance of the Rating model
				ratingModel := new(catalogmanagementv1.Rating)
				Expect(ratingModel).ToNot(BeNil())
				ratingModel.OneStarCount = core.Int64Ptr(int64(38))
				ratingModel.TwoStarCount = core.Int64Ptr(int64(38))
				ratingModel.ThreeStarCount = core.Int64Ptr(int64(38))
				ratingModel.FourStarCount = core.Int64Ptr(int64(38))
				Expect(ratingModel.OneStarCount).To(Equal(core.Int64Ptr(int64(38))))
				Expect(ratingModel.TwoStarCount).To(Equal(core.Int64Ptr(int64(38))))
				Expect(ratingModel.ThreeStarCount).To(Equal(core.Int64Ptr(int64(38))))
				Expect(ratingModel.FourStarCount).To(Equal(core.Int64Ptr(int64(38))))

				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				Expect(featureModel).ToNot(BeNil())
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")
				Expect(featureModel.Title).To(Equal(core.StringPtr("testString")))
				Expect(featureModel.Description).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the Configuration model
				configurationModel := new(catalogmanagementv1.Configuration)
				Expect(configurationModel).ToNot(BeNil())
				configurationModel.Key = core.StringPtr("testString")
				configurationModel.Type = core.StringPtr("testString")
				configurationModel.DefaultValue = core.StringPtr("testString")
				configurationModel.ValueConstraint = core.StringPtr("testString")
				configurationModel.Description = core.StringPtr("testString")
				configurationModel.Required = core.BoolPtr(true)
				configurationModel.Options = []interface{}{"testString"}
				configurationModel.Hidden = core.BoolPtr(true)
				Expect(configurationModel.Key).To(Equal(core.StringPtr("testString")))
				Expect(configurationModel.Type).To(Equal(core.StringPtr("testString")))
				Expect(configurationModel.DefaultValue).To(Equal(core.StringPtr("testString")))
				Expect(configurationModel.ValueConstraint).To(Equal(core.StringPtr("testString")))
				Expect(configurationModel.Description).To(Equal(core.StringPtr("testString")))
				Expect(configurationModel.Required).To(Equal(core.BoolPtr(true)))
				Expect(configurationModel.Options).To(Equal([]interface{}{"testString"}))
				Expect(configurationModel.Hidden).To(Equal(core.BoolPtr(true)))

				// Construct an instance of the Validation model
				validationModel := new(catalogmanagementv1.Validation)
				Expect(validationModel).ToNot(BeNil())
				validationModel.Validated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.Requested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				validationModel.State = core.StringPtr("testString")
				validationModel.LastOperation = core.StringPtr("testString")
				validationModel.Target = make(map[string]interface{})
				Expect(validationModel.Validated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(validationModel.Requested).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(validationModel.State).To(Equal(core.StringPtr("testString")))
				Expect(validationModel.LastOperation).To(Equal(core.StringPtr("testString")))
				Expect(validationModel.Target).To(Equal(make(map[string]interface{})))

				// Construct an instance of the Resource model
				resourceModel := new(catalogmanagementv1.Resource)
				Expect(resourceModel).ToNot(BeNil())
				resourceModel.Type = core.StringPtr("mem")
				resourceModel.Value = core.StringPtr("testString")
				Expect(resourceModel.Type).To(Equal(core.StringPtr("mem")))
				Expect(resourceModel.Value).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the Script model
				scriptModel := new(catalogmanagementv1.Script)
				Expect(scriptModel).ToNot(BeNil())
				scriptModel.Instructions = core.StringPtr("testString")
				scriptModel.Script = core.StringPtr("testString")
				scriptModel.ScriptPermission = core.StringPtr("testString")
				scriptModel.DeleteScript = core.StringPtr("testString")
				scriptModel.Scope = core.StringPtr("testString")
				Expect(scriptModel.Instructions).To(Equal(core.StringPtr("testString")))
				Expect(scriptModel.Script).To(Equal(core.StringPtr("testString")))
				Expect(scriptModel.ScriptPermission).To(Equal(core.StringPtr("testString")))
				Expect(scriptModel.DeleteScript).To(Equal(core.StringPtr("testString")))
				Expect(scriptModel.Scope).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the VersionEntitlement model
				versionEntitlementModel := new(catalogmanagementv1.VersionEntitlement)
				Expect(versionEntitlementModel).ToNot(BeNil())
				versionEntitlementModel.ProviderName = core.StringPtr("testString")
				versionEntitlementModel.ProviderID = core.StringPtr("testString")
				versionEntitlementModel.ProductID = core.StringPtr("testString")
				versionEntitlementModel.PartNumbers = []string{"testString"}
				versionEntitlementModel.ImageRepoName = core.StringPtr("testString")
				Expect(versionEntitlementModel.ProviderName).To(Equal(core.StringPtr("testString")))
				Expect(versionEntitlementModel.ProviderID).To(Equal(core.StringPtr("testString")))
				Expect(versionEntitlementModel.ProductID).To(Equal(core.StringPtr("testString")))
				Expect(versionEntitlementModel.PartNumbers).To(Equal([]string{"testString"}))
				Expect(versionEntitlementModel.ImageRepoName).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the License model
				licenseModel := new(catalogmanagementv1.License)
				Expect(licenseModel).ToNot(BeNil())
				licenseModel.ID = core.StringPtr("testString")
				licenseModel.Name = core.StringPtr("testString")
				licenseModel.Type = core.StringPtr("testString")
				licenseModel.URL = core.StringPtr("testString")
				licenseModel.Description = core.StringPtr("testString")
				Expect(licenseModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(licenseModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(licenseModel.Type).To(Equal(core.StringPtr("testString")))
				Expect(licenseModel.URL).To(Equal(core.StringPtr("testString")))
				Expect(licenseModel.Description).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				Expect(stateModel).ToNot(BeNil())
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")
				Expect(stateModel.Current).To(Equal(core.StringPtr("testString")))
				Expect(stateModel.CurrentEntered).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(stateModel.Pending).To(Equal(core.StringPtr("testString")))
				Expect(stateModel.PendingRequested).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(stateModel.Previous).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the Version model
				versionModel := new(catalogmanagementv1.Version)
				Expect(versionModel).ToNot(BeNil())
				versionModel.ID = core.StringPtr("testString")
				versionModel.Rev = core.StringPtr("testString")
				versionModel.CRN = core.StringPtr("testString")
				versionModel.Version = core.StringPtr("testString")
				versionModel.Sha = core.StringPtr("testString")
				versionModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				versionModel.OfferingID = core.StringPtr("testString")
				versionModel.CatalogID = core.StringPtr("testString")
				versionModel.KindID = core.StringPtr("testString")
				versionModel.Tags = []string{"testString"}
				versionModel.RepoURL = core.StringPtr("testString")
				versionModel.SourceURL = core.StringPtr("testString")
				versionModel.TgzURL = core.StringPtr("testString")
				versionModel.Configuration = []catalogmanagementv1.Configuration{*configurationModel}
				versionModel.Metadata = make(map[string]interface{})
				versionModel.Validation = validationModel
				versionModel.RequiredResources = []catalogmanagementv1.Resource{*resourceModel}
				versionModel.SingleInstance = core.BoolPtr(true)
				versionModel.Install = scriptModel
				versionModel.PreInstall = []catalogmanagementv1.Script{*scriptModel}
				versionModel.Entitlement = versionEntitlementModel
				versionModel.Licenses = []catalogmanagementv1.License{*licenseModel}
				versionModel.ImageManifestURL = core.StringPtr("testString")
				versionModel.Deprecated = core.BoolPtr(true)
				versionModel.PackageVersion = core.StringPtr("testString")
				versionModel.State = stateModel
				versionModel.VersionLocator = core.StringPtr("testString")
				versionModel.ConsoleURL = core.StringPtr("testString")
				versionModel.LongDescription = core.StringPtr("testString")
				versionModel.WhitelistedAccounts = []string{"testString"}
				Expect(versionModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.CRN).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Version).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Sha).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(versionModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(versionModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.CatalogID).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.KindID).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Tags).To(Equal([]string{"testString"}))
				Expect(versionModel.RepoURL).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.SourceURL).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.TgzURL).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Configuration).To(Equal([]catalogmanagementv1.Configuration{*configurationModel}))
				Expect(versionModel.Metadata).To(Equal(make(map[string]interface{})))
				Expect(versionModel.Validation).To(Equal(validationModel))
				Expect(versionModel.RequiredResources).To(Equal([]catalogmanagementv1.Resource{*resourceModel}))
				Expect(versionModel.SingleInstance).To(Equal(core.BoolPtr(true)))
				Expect(versionModel.Install).To(Equal(scriptModel))
				Expect(versionModel.PreInstall).To(Equal([]catalogmanagementv1.Script{*scriptModel}))
				Expect(versionModel.Entitlement).To(Equal(versionEntitlementModel))
				Expect(versionModel.Licenses).To(Equal([]catalogmanagementv1.License{*licenseModel}))
				Expect(versionModel.ImageManifestURL).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.Deprecated).To(Equal(core.BoolPtr(true)))
				Expect(versionModel.PackageVersion).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.State).To(Equal(stateModel))
				Expect(versionModel.VersionLocator).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.ConsoleURL).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.LongDescription).To(Equal(core.StringPtr("testString")))
				Expect(versionModel.WhitelistedAccounts).To(Equal([]string{"testString"}))

				// Construct an instance of the Deployment model
				deploymentModel := new(catalogmanagementv1.Deployment)
				Expect(deploymentModel).ToNot(BeNil())
				deploymentModel.ID = core.StringPtr("testString")
				deploymentModel.Label = core.StringPtr("testString")
				deploymentModel.Name = core.StringPtr("testString")
				deploymentModel.ShortDescription = core.StringPtr("testString")
				deploymentModel.LongDescription = core.StringPtr("testString")
				deploymentModel.Metadata = make(map[string]interface{})
				deploymentModel.Tags = []string{"testString"}
				deploymentModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				deploymentModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				Expect(deploymentModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(deploymentModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(deploymentModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(deploymentModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(deploymentModel.LongDescription).To(Equal(core.StringPtr("testString")))
				Expect(deploymentModel.Metadata).To(Equal(make(map[string]interface{})))
				Expect(deploymentModel.Tags).To(Equal([]string{"testString"}))
				Expect(deploymentModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(deploymentModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))

				// Construct an instance of the Plan model
				planModel := new(catalogmanagementv1.Plan)
				Expect(planModel).ToNot(BeNil())
				planModel.ID = core.StringPtr("testString")
				planModel.Label = core.StringPtr("testString")
				planModel.Name = core.StringPtr("testString")
				planModel.ShortDescription = core.StringPtr("testString")
				planModel.LongDescription = core.StringPtr("testString")
				planModel.Metadata = make(map[string]interface{})
				planModel.Tags = []string{"testString"}
				planModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				planModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				planModel.Deployments = []catalogmanagementv1.Deployment{*deploymentModel}
				Expect(planModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(planModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(planModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(planModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(planModel.LongDescription).To(Equal(core.StringPtr("testString")))
				Expect(planModel.Metadata).To(Equal(make(map[string]interface{})))
				Expect(planModel.Tags).To(Equal([]string{"testString"}))
				Expect(planModel.AdditionalFeatures).To(Equal([]catalogmanagementv1.Feature{*featureModel}))
				Expect(planModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(planModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(planModel.Deployments).To(Equal([]catalogmanagementv1.Deployment{*deploymentModel}))

				// Construct an instance of the Kind model
				kindModel := new(catalogmanagementv1.Kind)
				Expect(kindModel).ToNot(BeNil())
				kindModel.ID = core.StringPtr("testString")
				kindModel.FormatKind = core.StringPtr("testString")
				kindModel.TargetKind = core.StringPtr("testString")
				kindModel.Metadata = make(map[string]interface{})
				kindModel.InstallDescription = core.StringPtr("testString")
				kindModel.Tags = []string{"testString"}
				kindModel.AdditionalFeatures = []catalogmanagementv1.Feature{*featureModel}
				kindModel.Created = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Updated = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				kindModel.Versions = []catalogmanagementv1.Version{*versionModel}
				kindModel.Plans = []catalogmanagementv1.Plan{*planModel}
				Expect(kindModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(kindModel.FormatKind).To(Equal(core.StringPtr("testString")))
				Expect(kindModel.TargetKind).To(Equal(core.StringPtr("testString")))
				Expect(kindModel.Metadata).To(Equal(make(map[string]interface{})))
				Expect(kindModel.InstallDescription).To(Equal(core.StringPtr("testString")))
				Expect(kindModel.Tags).To(Equal([]string{"testString"}))
				Expect(kindModel.AdditionalFeatures).To(Equal([]catalogmanagementv1.Feature{*featureModel}))
				Expect(kindModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(kindModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(kindModel.Versions).To(Equal([]catalogmanagementv1.Version{*versionModel}))
				Expect(kindModel.Plans).To(Equal([]catalogmanagementv1.Plan{*planModel}))

				// Construct an instance of the RepoInfo model
				repoInfoModel := new(catalogmanagementv1.RepoInfo)
				Expect(repoInfoModel).ToNot(BeNil())
				repoInfoModel.Token = core.StringPtr("testString")
				repoInfoModel.Type = core.StringPtr("testString")
				Expect(repoInfoModel.Token).To(Equal(core.StringPtr("testString")))
				Expect(repoInfoModel.Type).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the CreateOfferingOptions model
				catalogIdentifier := "testString"
				createOfferingOptionsModel := catalogManagementService.NewCreateOfferingOptions(catalogIdentifier)
				createOfferingOptionsModel.SetCatalogIdentifier("testString")
				createOfferingOptionsModel.SetID("testString")
				createOfferingOptionsModel.SetRev("testString")
				createOfferingOptionsModel.SetURL("testString")
				createOfferingOptionsModel.SetCRN("testString")
				createOfferingOptionsModel.SetLabel("testString")
				createOfferingOptionsModel.SetName("testString")
				createOfferingOptionsModel.SetOfferingIconURL("testString")
				createOfferingOptionsModel.SetOfferingDocsURL("testString")
				createOfferingOptionsModel.SetOfferingSupportURL("testString")
				createOfferingOptionsModel.SetTags([]string{"testString"})
				createOfferingOptionsModel.SetKeywords([]string{"testString"})
				createOfferingOptionsModel.SetRating(ratingModel)
				createOfferingOptionsModel.SetCreated(CreateMockDateTime("2019-01-01T12:00:00.000Z"))
				createOfferingOptionsModel.SetUpdated(CreateMockDateTime("2019-01-01T12:00:00.000Z"))
				createOfferingOptionsModel.SetShortDescription("testString")
				createOfferingOptionsModel.SetLongDescription("testString")
				createOfferingOptionsModel.SetFeatures([]catalogmanagementv1.Feature{*featureModel})
				createOfferingOptionsModel.SetKinds([]catalogmanagementv1.Kind{*kindModel})
				createOfferingOptionsModel.SetPermitRequestIBMPublicPublish(true)
				createOfferingOptionsModel.SetIBMPublishApproved(true)
				createOfferingOptionsModel.SetPublicPublishApproved(true)
				createOfferingOptionsModel.SetPublicOriginalCRN("testString")
				createOfferingOptionsModel.SetPublishPublicCRN("testString")
				createOfferingOptionsModel.SetPortalApprovalRecord("testString")
				createOfferingOptionsModel.SetPortalUIURL("testString")
				createOfferingOptionsModel.SetCatalogID("testString")
				createOfferingOptionsModel.SetCatalogName("testString")
				createOfferingOptionsModel.SetMetadata(make(map[string]interface{}))
				createOfferingOptionsModel.SetDisclaimer("testString")
				createOfferingOptionsModel.SetHidden(true)
				createOfferingOptionsModel.SetProvider("testString")
				createOfferingOptionsModel.SetRepoInfo(repoInfoModel)
				createOfferingOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createOfferingOptionsModel).ToNot(BeNil())
				Expect(createOfferingOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.URL).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.CRN).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.OfferingIconURL).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.OfferingDocsURL).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.OfferingSupportURL).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(createOfferingOptionsModel.Keywords).To(Equal([]string{"testString"}))
				Expect(createOfferingOptionsModel.Rating).To(Equal(ratingModel))
				Expect(createOfferingOptionsModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(createOfferingOptionsModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(createOfferingOptionsModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.LongDescription).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Features).To(Equal([]catalogmanagementv1.Feature{*featureModel}))
				Expect(createOfferingOptionsModel.Kinds).To(Equal([]catalogmanagementv1.Kind{*kindModel}))
				Expect(createOfferingOptionsModel.PermitRequestIBMPublicPublish).To(Equal(core.BoolPtr(true)))
				Expect(createOfferingOptionsModel.IBMPublishApproved).To(Equal(core.BoolPtr(true)))
				Expect(createOfferingOptionsModel.PublicPublishApproved).To(Equal(core.BoolPtr(true)))
				Expect(createOfferingOptionsModel.PublicOriginalCRN).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.PublishPublicCRN).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.PortalApprovalRecord).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.PortalUIURL).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.CatalogID).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.CatalogName).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Metadata).To(Equal(make(map[string]interface{})))
				Expect(createOfferingOptionsModel.Disclaimer).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.Hidden).To(Equal(core.BoolPtr(true)))
				Expect(createOfferingOptionsModel.Provider).To(Equal(core.StringPtr("testString")))
				Expect(createOfferingOptionsModel.RepoInfo).To(Equal(repoInfoModel))
				Expect(createOfferingOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteCatalogOptions successfully`, func() {
				// Construct an instance of the DeleteCatalogOptions model
				catalogIdentifier := "testString"
				deleteCatalogOptionsModel := catalogManagementService.NewDeleteCatalogOptions(catalogIdentifier)
				deleteCatalogOptionsModel.SetCatalogIdentifier("testString")
				deleteCatalogOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteCatalogOptionsModel).ToNot(BeNil())
				Expect(deleteCatalogOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteCatalogOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteObjectAccessListOptions successfully`, func() {
				// Construct an instance of the DeleteObjectAccessListOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				accounts := []string{"testString"}
				deleteObjectAccessListOptionsModel := catalogManagementService.NewDeleteObjectAccessListOptions(catalogIdentifier, objectIdentifier, accounts)
				deleteObjectAccessListOptionsModel.SetCatalogIdentifier("testString")
				deleteObjectAccessListOptionsModel.SetObjectIdentifier("testString")
				deleteObjectAccessListOptionsModel.SetAccounts([]string{"testString"})
				deleteObjectAccessListOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteObjectAccessListOptionsModel).ToNot(BeNil())
				Expect(deleteObjectAccessListOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectAccessListOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectAccessListOptionsModel.Accounts).To(Equal([]string{"testString"}))
				Expect(deleteObjectAccessListOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteObjectAccessOptions successfully`, func() {
				// Construct an instance of the DeleteObjectAccessOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				accountIdentifier := "testString"
				deleteObjectAccessOptionsModel := catalogManagementService.NewDeleteObjectAccessOptions(catalogIdentifier, objectIdentifier, accountIdentifier)
				deleteObjectAccessOptionsModel.SetCatalogIdentifier("testString")
				deleteObjectAccessOptionsModel.SetObjectIdentifier("testString")
				deleteObjectAccessOptionsModel.SetAccountIdentifier("testString")
				deleteObjectAccessOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteObjectAccessOptionsModel).ToNot(BeNil())
				Expect(deleteObjectAccessOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectAccessOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectAccessOptionsModel.AccountIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectAccessOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteObjectOptions successfully`, func() {
				// Construct an instance of the DeleteObjectOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				deleteObjectOptionsModel := catalogManagementService.NewDeleteObjectOptions(catalogIdentifier, objectIdentifier)
				deleteObjectOptionsModel.SetCatalogIdentifier("testString")
				deleteObjectOptionsModel.SetObjectIdentifier("testString")
				deleteObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteObjectOptionsModel).ToNot(BeNil())
				Expect(deleteObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteOfferingInstanceOptions successfully`, func() {
				// Construct an instance of the DeleteOfferingInstanceOptions model
				instanceIdentifier := "testString"
				xAuthRefreshToken := "testString"
				deleteOfferingInstanceOptionsModel := catalogManagementService.NewDeleteOfferingInstanceOptions(instanceIdentifier, xAuthRefreshToken)
				deleteOfferingInstanceOptionsModel.SetInstanceIdentifier("testString")
				deleteOfferingInstanceOptionsModel.SetXAuthRefreshToken("testString")
				deleteOfferingInstanceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteOfferingInstanceOptionsModel).ToNot(BeNil())
				Expect(deleteOfferingInstanceOptionsModel.InstanceIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteOfferingInstanceOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(deleteOfferingInstanceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteOfferingOptions successfully`, func() {
				// Construct an instance of the DeleteOfferingOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				deleteOfferingOptionsModel := catalogManagementService.NewDeleteOfferingOptions(catalogIdentifier, offeringID)
				deleteOfferingOptionsModel.SetCatalogIdentifier("testString")
				deleteOfferingOptionsModel.SetOfferingID("testString")
				deleteOfferingOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteOfferingOptionsModel).ToNot(BeNil())
				Expect(deleteOfferingOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(deleteOfferingOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(deleteOfferingOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteOperatorsOptions successfully`, func() {
				// Construct an instance of the DeleteOperatorsOptions model
				xAuthRefreshToken := "testString"
				clusterID := "testString"
				region := "testString"
				versionLocatorID := "testString"
				deleteOperatorsOptionsModel := catalogManagementService.NewDeleteOperatorsOptions(xAuthRefreshToken, clusterID, region, versionLocatorID)
				deleteOperatorsOptionsModel.SetXAuthRefreshToken("testString")
				deleteOperatorsOptionsModel.SetClusterID("testString")
				deleteOperatorsOptionsModel.SetRegion("testString")
				deleteOperatorsOptionsModel.SetVersionLocatorID("testString")
				deleteOperatorsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteOperatorsOptionsModel).ToNot(BeNil())
				Expect(deleteOperatorsOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(deleteOperatorsOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(deleteOperatorsOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(deleteOperatorsOptionsModel.VersionLocatorID).To(Equal(core.StringPtr("testString")))
				Expect(deleteOperatorsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteVersionOptions successfully`, func() {
				// Construct an instance of the DeleteVersionOptions model
				versionLocID := "testString"
				deleteVersionOptionsModel := catalogManagementService.NewDeleteVersionOptions(versionLocID)
				deleteVersionOptionsModel.SetVersionLocID("testString")
				deleteVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteVersionOptionsModel).ToNot(BeNil())
				Expect(deleteVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(deleteVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeployOperatorsOptions successfully`, func() {
				// Construct an instance of the DeployOperatorsOptions model
				xAuthRefreshToken := "testString"
				deployOperatorsOptionsModel := catalogManagementService.NewDeployOperatorsOptions(xAuthRefreshToken)
				deployOperatorsOptionsModel.SetXAuthRefreshToken("testString")
				deployOperatorsOptionsModel.SetClusterID("testString")
				deployOperatorsOptionsModel.SetRegion("testString")
				deployOperatorsOptionsModel.SetNamespaces([]string{"testString"})
				deployOperatorsOptionsModel.SetAllNamespaces(true)
				deployOperatorsOptionsModel.SetVersionLocatorID("testString")
				deployOperatorsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deployOperatorsOptionsModel).ToNot(BeNil())
				Expect(deployOperatorsOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(deployOperatorsOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(deployOperatorsOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(deployOperatorsOptionsModel.Namespaces).To(Equal([]string{"testString"}))
				Expect(deployOperatorsOptionsModel.AllNamespaces).To(Equal(core.BoolPtr(true)))
				Expect(deployOperatorsOptionsModel.VersionLocatorID).To(Equal(core.StringPtr("testString")))
				Expect(deployOperatorsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeprecateVersionOptions successfully`, func() {
				// Construct an instance of the DeprecateVersionOptions model
				versionLocID := "testString"
				deprecateVersionOptionsModel := catalogManagementService.NewDeprecateVersionOptions(versionLocID)
				deprecateVersionOptionsModel.SetVersionLocID("testString")
				deprecateVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deprecateVersionOptionsModel).ToNot(BeNil())
				Expect(deprecateVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(deprecateVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetCatalogAccountAuditOptions successfully`, func() {
				// Construct an instance of the GetCatalogAccountAuditOptions model
				getCatalogAccountAuditOptionsModel := catalogManagementService.NewGetCatalogAccountAuditOptions()
				getCatalogAccountAuditOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getCatalogAccountAuditOptionsModel).ToNot(BeNil())
				Expect(getCatalogAccountAuditOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetCatalogAccountFiltersOptions successfully`, func() {
				// Construct an instance of the GetCatalogAccountFiltersOptions model
				getCatalogAccountFiltersOptionsModel := catalogManagementService.NewGetCatalogAccountFiltersOptions()
				getCatalogAccountFiltersOptionsModel.SetCatalog("testString")
				getCatalogAccountFiltersOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getCatalogAccountFiltersOptionsModel).ToNot(BeNil())
				Expect(getCatalogAccountFiltersOptionsModel.Catalog).To(Equal(core.StringPtr("testString")))
				Expect(getCatalogAccountFiltersOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetCatalogAccountOptions successfully`, func() {
				// Construct an instance of the GetCatalogAccountOptions model
				getCatalogAccountOptionsModel := catalogManagementService.NewGetCatalogAccountOptions()
				getCatalogAccountOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getCatalogAccountOptionsModel).ToNot(BeNil())
				Expect(getCatalogAccountOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetCatalogAuditOptions successfully`, func() {
				// Construct an instance of the GetCatalogAuditOptions model
				catalogIdentifier := "testString"
				getCatalogAuditOptionsModel := catalogManagementService.NewGetCatalogAuditOptions(catalogIdentifier)
				getCatalogAuditOptionsModel.SetCatalogIdentifier("testString")
				getCatalogAuditOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getCatalogAuditOptionsModel).ToNot(BeNil())
				Expect(getCatalogAuditOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getCatalogAuditOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetCatalogOptions successfully`, func() {
				// Construct an instance of the GetCatalogOptions model
				catalogIdentifier := "testString"
				getCatalogOptionsModel := catalogManagementService.NewGetCatalogOptions(catalogIdentifier)
				getCatalogOptionsModel.SetCatalogIdentifier("testString")
				getCatalogOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getCatalogOptionsModel).ToNot(BeNil())
				Expect(getCatalogOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getCatalogOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetClusterOptions successfully`, func() {
				// Construct an instance of the GetClusterOptions model
				clusterID := "testString"
				region := "testString"
				xAuthRefreshToken := "testString"
				getClusterOptionsModel := catalogManagementService.NewGetClusterOptions(clusterID, region, xAuthRefreshToken)
				getClusterOptionsModel.SetClusterID("testString")
				getClusterOptionsModel.SetRegion("testString")
				getClusterOptionsModel.SetXAuthRefreshToken("testString")
				getClusterOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getClusterOptionsModel).ToNot(BeNil())
				Expect(getClusterOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(getClusterOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(getClusterOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(getClusterOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetConsumptionOfferingsOptions successfully`, func() {
				// Construct an instance of the GetConsumptionOfferingsOptions model
				getConsumptionOfferingsOptionsModel := catalogManagementService.NewGetConsumptionOfferingsOptions()
				getConsumptionOfferingsOptionsModel.SetDigest(true)
				getConsumptionOfferingsOptionsModel.SetCatalog("testString")
				getConsumptionOfferingsOptionsModel.SetSelect("all")
				getConsumptionOfferingsOptionsModel.SetIncludeHidden(true)
				getConsumptionOfferingsOptionsModel.SetLimit(int64(1000))
				getConsumptionOfferingsOptionsModel.SetOffset(int64(38))
				getConsumptionOfferingsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getConsumptionOfferingsOptionsModel).ToNot(BeNil())
				Expect(getConsumptionOfferingsOptionsModel.Digest).To(Equal(core.BoolPtr(true)))
				Expect(getConsumptionOfferingsOptionsModel.Catalog).To(Equal(core.StringPtr("testString")))
				Expect(getConsumptionOfferingsOptionsModel.Select).To(Equal(core.StringPtr("all")))
				Expect(getConsumptionOfferingsOptionsModel.IncludeHidden).To(Equal(core.BoolPtr(true)))
				Expect(getConsumptionOfferingsOptionsModel.Limit).To(Equal(core.Int64Ptr(int64(1000))))
				Expect(getConsumptionOfferingsOptionsModel.Offset).To(Equal(core.Int64Ptr(int64(38))))
				Expect(getConsumptionOfferingsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetNamespacesOptions successfully`, func() {
				// Construct an instance of the GetNamespacesOptions model
				clusterID := "testString"
				region := "testString"
				xAuthRefreshToken := "testString"
				getNamespacesOptionsModel := catalogManagementService.NewGetNamespacesOptions(clusterID, region, xAuthRefreshToken)
				getNamespacesOptionsModel.SetClusterID("testString")
				getNamespacesOptionsModel.SetRegion("testString")
				getNamespacesOptionsModel.SetXAuthRefreshToken("testString")
				getNamespacesOptionsModel.SetLimit(int64(1000))
				getNamespacesOptionsModel.SetOffset(int64(38))
				getNamespacesOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getNamespacesOptionsModel).ToNot(BeNil())
				Expect(getNamespacesOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(getNamespacesOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(getNamespacesOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(getNamespacesOptionsModel.Limit).To(Equal(core.Int64Ptr(int64(1000))))
				Expect(getNamespacesOptionsModel.Offset).To(Equal(core.Int64Ptr(int64(38))))
				Expect(getNamespacesOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetObjectAccessListOptions successfully`, func() {
				// Construct an instance of the GetObjectAccessListOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				getObjectAccessListOptionsModel := catalogManagementService.NewGetObjectAccessListOptions(catalogIdentifier, objectIdentifier)
				getObjectAccessListOptionsModel.SetCatalogIdentifier("testString")
				getObjectAccessListOptionsModel.SetObjectIdentifier("testString")
				getObjectAccessListOptionsModel.SetLimit(int64(1000))
				getObjectAccessListOptionsModel.SetOffset(int64(38))
				getObjectAccessListOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getObjectAccessListOptionsModel).ToNot(BeNil())
				Expect(getObjectAccessListOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAccessListOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAccessListOptionsModel.Limit).To(Equal(core.Int64Ptr(int64(1000))))
				Expect(getObjectAccessListOptionsModel.Offset).To(Equal(core.Int64Ptr(int64(38))))
				Expect(getObjectAccessListOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetObjectAccessOptions successfully`, func() {
				// Construct an instance of the GetObjectAccessOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				accountIdentifier := "testString"
				getObjectAccessOptionsModel := catalogManagementService.NewGetObjectAccessOptions(catalogIdentifier, objectIdentifier, accountIdentifier)
				getObjectAccessOptionsModel.SetCatalogIdentifier("testString")
				getObjectAccessOptionsModel.SetObjectIdentifier("testString")
				getObjectAccessOptionsModel.SetAccountIdentifier("testString")
				getObjectAccessOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getObjectAccessOptionsModel).ToNot(BeNil())
				Expect(getObjectAccessOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAccessOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAccessOptionsModel.AccountIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAccessOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetObjectAuditOptions successfully`, func() {
				// Construct an instance of the GetObjectAuditOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				getObjectAuditOptionsModel := catalogManagementService.NewGetObjectAuditOptions(catalogIdentifier, objectIdentifier)
				getObjectAuditOptionsModel.SetCatalogIdentifier("testString")
				getObjectAuditOptionsModel.SetObjectIdentifier("testString")
				getObjectAuditOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getObjectAuditOptionsModel).ToNot(BeNil())
				Expect(getObjectAuditOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAuditOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectAuditOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetObjectOptions successfully`, func() {
				// Construct an instance of the GetObjectOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				getObjectOptionsModel := catalogManagementService.NewGetObjectOptions(catalogIdentifier, objectIdentifier)
				getObjectOptionsModel.SetCatalogIdentifier("testString")
				getObjectOptionsModel.SetObjectIdentifier("testString")
				getObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getObjectOptionsModel).ToNot(BeNil())
				Expect(getObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingAboutOptions successfully`, func() {
				// Construct an instance of the GetOfferingAboutOptions model
				versionLocID := "testString"
				getOfferingAboutOptionsModel := catalogManagementService.NewGetOfferingAboutOptions(versionLocID)
				getOfferingAboutOptionsModel.SetVersionLocID("testString")
				getOfferingAboutOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingAboutOptionsModel).ToNot(BeNil())
				Expect(getOfferingAboutOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingAboutOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingAuditOptions successfully`, func() {
				// Construct an instance of the GetOfferingAuditOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				getOfferingAuditOptionsModel := catalogManagementService.NewGetOfferingAuditOptions(catalogIdentifier, offeringID)
				getOfferingAuditOptionsModel.SetCatalogIdentifier("testString")
				getOfferingAuditOptionsModel.SetOfferingID("testString")
				getOfferingAuditOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingAuditOptionsModel).ToNot(BeNil())
				Expect(getOfferingAuditOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingAuditOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingAuditOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingContainerImagesOptions successfully`, func() {
				// Construct an instance of the GetOfferingContainerImagesOptions model
				versionLocID := "testString"
				getOfferingContainerImagesOptionsModel := catalogManagementService.NewGetOfferingContainerImagesOptions(versionLocID)
				getOfferingContainerImagesOptionsModel.SetVersionLocID("testString")
				getOfferingContainerImagesOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingContainerImagesOptionsModel).ToNot(BeNil())
				Expect(getOfferingContainerImagesOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingContainerImagesOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingInstanceOptions successfully`, func() {
				// Construct an instance of the GetOfferingInstanceOptions model
				instanceIdentifier := "testString"
				getOfferingInstanceOptionsModel := catalogManagementService.NewGetOfferingInstanceOptions(instanceIdentifier)
				getOfferingInstanceOptionsModel.SetInstanceIdentifier("testString")
				getOfferingInstanceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingInstanceOptionsModel).ToNot(BeNil())
				Expect(getOfferingInstanceOptionsModel.InstanceIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingInstanceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingLicenseOptions successfully`, func() {
				// Construct an instance of the GetOfferingLicenseOptions model
				versionLocID := "testString"
				licenseID := "testString"
				getOfferingLicenseOptionsModel := catalogManagementService.NewGetOfferingLicenseOptions(versionLocID, licenseID)
				getOfferingLicenseOptionsModel.SetVersionLocID("testString")
				getOfferingLicenseOptionsModel.SetLicenseID("testString")
				getOfferingLicenseOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingLicenseOptionsModel).ToNot(BeNil())
				Expect(getOfferingLicenseOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingLicenseOptionsModel.LicenseID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingLicenseOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingOptions successfully`, func() {
				// Construct an instance of the GetOfferingOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				getOfferingOptionsModel := catalogManagementService.NewGetOfferingOptions(catalogIdentifier, offeringID)
				getOfferingOptionsModel.SetCatalogIdentifier("testString")
				getOfferingOptionsModel.SetOfferingID("testString")
				getOfferingOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingOptionsModel).ToNot(BeNil())
				Expect(getOfferingOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingUpdatesOptions successfully`, func() {
				// Construct an instance of the GetOfferingUpdatesOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				kind := "testString"
				getOfferingUpdatesOptionsModel := catalogManagementService.NewGetOfferingUpdatesOptions(catalogIdentifier, offeringID, kind)
				getOfferingUpdatesOptionsModel.SetCatalogIdentifier("testString")
				getOfferingUpdatesOptionsModel.SetOfferingID("testString")
				getOfferingUpdatesOptionsModel.SetKind("testString")
				getOfferingUpdatesOptionsModel.SetVersion("testString")
				getOfferingUpdatesOptionsModel.SetClusterID("testString")
				getOfferingUpdatesOptionsModel.SetRegion("testString")
				getOfferingUpdatesOptionsModel.SetResourceGroupID("testString")
				getOfferingUpdatesOptionsModel.SetNamespace("testString")
				getOfferingUpdatesOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingUpdatesOptionsModel).ToNot(BeNil())
				Expect(getOfferingUpdatesOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.Kind).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.Version).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.Namespace).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingUpdatesOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOfferingWorkingCopyOptions successfully`, func() {
				// Construct an instance of the GetOfferingWorkingCopyOptions model
				versionLocID := "testString"
				getOfferingWorkingCopyOptionsModel := catalogManagementService.NewGetOfferingWorkingCopyOptions(versionLocID)
				getOfferingWorkingCopyOptionsModel.SetVersionLocID("testString")
				getOfferingWorkingCopyOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOfferingWorkingCopyOptionsModel).ToNot(BeNil())
				Expect(getOfferingWorkingCopyOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getOfferingWorkingCopyOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetOverrideValuesOptions successfully`, func() {
				// Construct an instance of the GetOverrideValuesOptions model
				versionLocID := "testString"
				getOverrideValuesOptionsModel := catalogManagementService.NewGetOverrideValuesOptions(versionLocID)
				getOverrideValuesOptionsModel.SetVersionLocID("testString")
				getOverrideValuesOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getOverrideValuesOptionsModel).ToNot(BeNil())
				Expect(getOverrideValuesOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getOverrideValuesOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetPreinstallOptions successfully`, func() {
				// Construct an instance of the GetPreinstallOptions model
				versionLocID := "testString"
				xAuthRefreshToken := "testString"
				getPreinstallOptionsModel := catalogManagementService.NewGetPreinstallOptions(versionLocID, xAuthRefreshToken)
				getPreinstallOptionsModel.SetVersionLocID("testString")
				getPreinstallOptionsModel.SetXAuthRefreshToken("testString")
				getPreinstallOptionsModel.SetClusterID("testString")
				getPreinstallOptionsModel.SetRegion("testString")
				getPreinstallOptionsModel.SetNamespace("testString")
				getPreinstallOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getPreinstallOptionsModel).ToNot(BeNil())
				Expect(getPreinstallOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getPreinstallOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(getPreinstallOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(getPreinstallOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(getPreinstallOptionsModel.Namespace).To(Equal(core.StringPtr("testString")))
				Expect(getPreinstallOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetValidationStatusOptions successfully`, func() {
				// Construct an instance of the GetValidationStatusOptions model
				versionLocID := "testString"
				xAuthRefreshToken := "testString"
				getValidationStatusOptionsModel := catalogManagementService.NewGetValidationStatusOptions(versionLocID, xAuthRefreshToken)
				getValidationStatusOptionsModel.SetVersionLocID("testString")
				getValidationStatusOptionsModel.SetXAuthRefreshToken("testString")
				getValidationStatusOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getValidationStatusOptionsModel).ToNot(BeNil())
				Expect(getValidationStatusOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getValidationStatusOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(getValidationStatusOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetVersionOptions successfully`, func() {
				// Construct an instance of the GetVersionOptions model
				versionLocID := "testString"
				getVersionOptionsModel := catalogManagementService.NewGetVersionOptions(versionLocID)
				getVersionOptionsModel.SetVersionLocID("testString")
				getVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getVersionOptionsModel).ToNot(BeNil())
				Expect(getVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(getVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewIBMPublishObjectOptions successfully`, func() {
				// Construct an instance of the IBMPublishObjectOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				ibmPublishObjectOptionsModel := catalogManagementService.NewIBMPublishObjectOptions(catalogIdentifier, objectIdentifier)
				ibmPublishObjectOptionsModel.SetCatalogIdentifier("testString")
				ibmPublishObjectOptionsModel.SetObjectIdentifier("testString")
				ibmPublishObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(ibmPublishObjectOptionsModel).ToNot(BeNil())
				Expect(ibmPublishObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(ibmPublishObjectOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(ibmPublishObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewIBMPublishVersionOptions successfully`, func() {
				// Construct an instance of the IBMPublishVersionOptions model
				versionLocID := "testString"
				ibmPublishVersionOptionsModel := catalogManagementService.NewIBMPublishVersionOptions(versionLocID)
				ibmPublishVersionOptionsModel.SetVersionLocID("testString")
				ibmPublishVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(ibmPublishVersionOptionsModel).ToNot(BeNil())
				Expect(ibmPublishVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(ibmPublishVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewImportOfferingOptions successfully`, func() {
				// Construct an instance of the ImportOfferingOptions model
				catalogIdentifier := "testString"
				importOfferingOptionsModel := catalogManagementService.NewImportOfferingOptions(catalogIdentifier)
				importOfferingOptionsModel.SetCatalogIdentifier("testString")
				importOfferingOptionsModel.SetTags([]string{"testString"})
				importOfferingOptionsModel.SetTargetKinds([]string{"testString"})
				importOfferingOptionsModel.SetContent(*CreateMockByteArray("This is a mock byte array value."))
				importOfferingOptionsModel.SetZipurl("testString")
				importOfferingOptionsModel.SetOfferingID("testString")
				importOfferingOptionsModel.SetTargetVersion("testString")
				importOfferingOptionsModel.SetIncludeConfig(true)
				importOfferingOptionsModel.SetIsVsi(true)
				importOfferingOptionsModel.SetRepoType("testString")
				importOfferingOptionsModel.SetXAuthToken("testString")
				importOfferingOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(importOfferingOptionsModel).ToNot(BeNil())
				Expect(importOfferingOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(importOfferingOptionsModel.TargetKinds).To(Equal([]string{"testString"}))
				Expect(importOfferingOptionsModel.Content).To(Equal(CreateMockByteArray("This is a mock byte array value.")))
				Expect(importOfferingOptionsModel.Zipurl).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingOptionsModel.TargetVersion).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingOptionsModel.IncludeConfig).To(Equal(core.BoolPtr(true)))
				Expect(importOfferingOptionsModel.IsVsi).To(Equal(core.BoolPtr(true)))
				Expect(importOfferingOptionsModel.RepoType).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingOptionsModel.XAuthToken).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewImportOfferingVersionOptions successfully`, func() {
				// Construct an instance of the ImportOfferingVersionOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				importOfferingVersionOptionsModel := catalogManagementService.NewImportOfferingVersionOptions(catalogIdentifier, offeringID)
				importOfferingVersionOptionsModel.SetCatalogIdentifier("testString")
				importOfferingVersionOptionsModel.SetOfferingID("testString")
				importOfferingVersionOptionsModel.SetTags([]string{"testString"})
				importOfferingVersionOptionsModel.SetTargetKinds([]string{"testString"})
				importOfferingVersionOptionsModel.SetContent(*CreateMockByteArray("This is a mock byte array value."))
				importOfferingVersionOptionsModel.SetZipurl("testString")
				importOfferingVersionOptionsModel.SetTargetVersion("testString")
				importOfferingVersionOptionsModel.SetIncludeConfig(true)
				importOfferingVersionOptionsModel.SetIsVsi(true)
				importOfferingVersionOptionsModel.SetRepoType("testString")
				importOfferingVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(importOfferingVersionOptionsModel).ToNot(BeNil())
				Expect(importOfferingVersionOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingVersionOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingVersionOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(importOfferingVersionOptionsModel.TargetKinds).To(Equal([]string{"testString"}))
				Expect(importOfferingVersionOptionsModel.Content).To(Equal(CreateMockByteArray("This is a mock byte array value.")))
				Expect(importOfferingVersionOptionsModel.Zipurl).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingVersionOptionsModel.TargetVersion).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingVersionOptionsModel.IncludeConfig).To(Equal(core.BoolPtr(true)))
				Expect(importOfferingVersionOptionsModel.IsVsi).To(Equal(core.BoolPtr(true)))
				Expect(importOfferingVersionOptionsModel.RepoType).To(Equal(core.StringPtr("testString")))
				Expect(importOfferingVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewInstallVersionOptions successfully`, func() {
				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				Expect(deployRequestBodySchematicsModel).ToNot(BeNil())
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")
				Expect(deployRequestBodySchematicsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(deployRequestBodySchematicsModel.Description).To(Equal(core.StringPtr("testString")))
				Expect(deployRequestBodySchematicsModel.Tags).To(Equal([]string{"testString"}))
				Expect(deployRequestBodySchematicsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the InstallVersionOptions model
				versionLocID := "testString"
				xAuthRefreshToken := "testString"
				installVersionOptionsModel := catalogManagementService.NewInstallVersionOptions(versionLocID, xAuthRefreshToken)
				installVersionOptionsModel.SetVersionLocID("testString")
				installVersionOptionsModel.SetXAuthRefreshToken("testString")
				installVersionOptionsModel.SetClusterID("testString")
				installVersionOptionsModel.SetRegion("testString")
				installVersionOptionsModel.SetNamespace("testString")
				installVersionOptionsModel.SetOverrideValues(make(map[string]interface{}))
				installVersionOptionsModel.SetEntitlementApikey("testString")
				installVersionOptionsModel.SetSchematics(deployRequestBodySchematicsModel)
				installVersionOptionsModel.SetScript("testString")
				installVersionOptionsModel.SetScriptID("testString")
				installVersionOptionsModel.SetVersionLocatorID("testString")
				installVersionOptionsModel.SetVcenterID("testString")
				installVersionOptionsModel.SetVcenterUser("testString")
				installVersionOptionsModel.SetVcenterPassword("testString")
				installVersionOptionsModel.SetVcenterLocation("testString")
				installVersionOptionsModel.SetVcenterDatastore("testString")
				installVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(installVersionOptionsModel).ToNot(BeNil())
				Expect(installVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.Namespace).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.OverrideValues).To(Equal(make(map[string]interface{})))
				Expect(installVersionOptionsModel.EntitlementApikey).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.Schematics).To(Equal(deployRequestBodySchematicsModel))
				Expect(installVersionOptionsModel.Script).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.ScriptID).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.VersionLocatorID).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.VcenterID).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.VcenterUser).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.VcenterPassword).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.VcenterLocation).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.VcenterDatastore).To(Equal(core.StringPtr("testString")))
				Expect(installVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListCatalogsOptions successfully`, func() {
				// Construct an instance of the ListCatalogsOptions model
				listCatalogsOptionsModel := catalogManagementService.NewListCatalogsOptions()
				listCatalogsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listCatalogsOptionsModel).ToNot(BeNil())
				Expect(listCatalogsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListObjectsOptions successfully`, func() {
				// Construct an instance of the ListObjectsOptions model
				catalogIdentifier := "testString"
				listObjectsOptionsModel := catalogManagementService.NewListObjectsOptions(catalogIdentifier)
				listObjectsOptionsModel.SetCatalogIdentifier("testString")
				listObjectsOptionsModel.SetLimit(int64(1000))
				listObjectsOptionsModel.SetOffset(int64(38))
				listObjectsOptionsModel.SetName("testString")
				listObjectsOptionsModel.SetSort("testString")
				listObjectsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listObjectsOptionsModel).ToNot(BeNil())
				Expect(listObjectsOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(listObjectsOptionsModel.Limit).To(Equal(core.Int64Ptr(int64(1000))))
				Expect(listObjectsOptionsModel.Offset).To(Equal(core.Int64Ptr(int64(38))))
				Expect(listObjectsOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(listObjectsOptionsModel.Sort).To(Equal(core.StringPtr("testString")))
				Expect(listObjectsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListOfferingsOptions successfully`, func() {
				// Construct an instance of the ListOfferingsOptions model
				catalogIdentifier := "testString"
				listOfferingsOptionsModel := catalogManagementService.NewListOfferingsOptions(catalogIdentifier)
				listOfferingsOptionsModel.SetCatalogIdentifier("testString")
				listOfferingsOptionsModel.SetDigest(true)
				listOfferingsOptionsModel.SetLimit(int64(1000))
				listOfferingsOptionsModel.SetOffset(int64(38))
				listOfferingsOptionsModel.SetName("testString")
				listOfferingsOptionsModel.SetSort("testString")
				listOfferingsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listOfferingsOptionsModel).ToNot(BeNil())
				Expect(listOfferingsOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(listOfferingsOptionsModel.Digest).To(Equal(core.BoolPtr(true)))
				Expect(listOfferingsOptionsModel.Limit).To(Equal(core.Int64Ptr(int64(1000))))
				Expect(listOfferingsOptionsModel.Offset).To(Equal(core.Int64Ptr(int64(38))))
				Expect(listOfferingsOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(listOfferingsOptionsModel.Sort).To(Equal(core.StringPtr("testString")))
				Expect(listOfferingsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListOperatorsOptions successfully`, func() {
				// Construct an instance of the ListOperatorsOptions model
				xAuthRefreshToken := "testString"
				clusterID := "testString"
				region := "testString"
				versionLocatorID := "testString"
				listOperatorsOptionsModel := catalogManagementService.NewListOperatorsOptions(xAuthRefreshToken, clusterID, region, versionLocatorID)
				listOperatorsOptionsModel.SetXAuthRefreshToken("testString")
				listOperatorsOptionsModel.SetClusterID("testString")
				listOperatorsOptionsModel.SetRegion("testString")
				listOperatorsOptionsModel.SetVersionLocatorID("testString")
				listOperatorsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listOperatorsOptionsModel).ToNot(BeNil())
				Expect(listOperatorsOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(listOperatorsOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(listOperatorsOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(listOperatorsOptionsModel.VersionLocatorID).To(Equal(core.StringPtr("testString")))
				Expect(listOperatorsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewPreinstallVersionOptions successfully`, func() {
				// Construct an instance of the DeployRequestBodySchematics model
				deployRequestBodySchematicsModel := new(catalogmanagementv1.DeployRequestBodySchematics)
				Expect(deployRequestBodySchematicsModel).ToNot(BeNil())
				deployRequestBodySchematicsModel.Name = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Description = core.StringPtr("testString")
				deployRequestBodySchematicsModel.Tags = []string{"testString"}
				deployRequestBodySchematicsModel.ResourceGroupID = core.StringPtr("testString")
				Expect(deployRequestBodySchematicsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(deployRequestBodySchematicsModel.Description).To(Equal(core.StringPtr("testString")))
				Expect(deployRequestBodySchematicsModel.Tags).To(Equal([]string{"testString"}))
				Expect(deployRequestBodySchematicsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the PreinstallVersionOptions model
				versionLocID := "testString"
				xAuthRefreshToken := "testString"
				preinstallVersionOptionsModel := catalogManagementService.NewPreinstallVersionOptions(versionLocID, xAuthRefreshToken)
				preinstallVersionOptionsModel.SetVersionLocID("testString")
				preinstallVersionOptionsModel.SetXAuthRefreshToken("testString")
				preinstallVersionOptionsModel.SetClusterID("testString")
				preinstallVersionOptionsModel.SetRegion("testString")
				preinstallVersionOptionsModel.SetNamespace("testString")
				preinstallVersionOptionsModel.SetOverrideValues(make(map[string]interface{}))
				preinstallVersionOptionsModel.SetEntitlementApikey("testString")
				preinstallVersionOptionsModel.SetSchematics(deployRequestBodySchematicsModel)
				preinstallVersionOptionsModel.SetScript("testString")
				preinstallVersionOptionsModel.SetScriptID("testString")
				preinstallVersionOptionsModel.SetVersionLocatorID("testString")
				preinstallVersionOptionsModel.SetVcenterID("testString")
				preinstallVersionOptionsModel.SetVcenterUser("testString")
				preinstallVersionOptionsModel.SetVcenterPassword("testString")
				preinstallVersionOptionsModel.SetVcenterLocation("testString")
				preinstallVersionOptionsModel.SetVcenterDatastore("testString")
				preinstallVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(preinstallVersionOptionsModel).ToNot(BeNil())
				Expect(preinstallVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.Namespace).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.OverrideValues).To(Equal(make(map[string]interface{})))
				Expect(preinstallVersionOptionsModel.EntitlementApikey).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.Schematics).To(Equal(deployRequestBodySchematicsModel))
				Expect(preinstallVersionOptionsModel.Script).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.ScriptID).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.VersionLocatorID).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.VcenterID).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.VcenterUser).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.VcenterPassword).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.VcenterLocation).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.VcenterDatastore).To(Equal(core.StringPtr("testString")))
				Expect(preinstallVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewPublicPublishObjectOptions successfully`, func() {
				// Construct an instance of the PublicPublishObjectOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				publicPublishObjectOptionsModel := catalogManagementService.NewPublicPublishObjectOptions(catalogIdentifier, objectIdentifier)
				publicPublishObjectOptionsModel.SetCatalogIdentifier("testString")
				publicPublishObjectOptionsModel.SetObjectIdentifier("testString")
				publicPublishObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(publicPublishObjectOptionsModel).ToNot(BeNil())
				Expect(publicPublishObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(publicPublishObjectOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(publicPublishObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewPublicPublishVersionOptions successfully`, func() {
				// Construct an instance of the PublicPublishVersionOptions model
				versionLocID := "testString"
				publicPublishVersionOptionsModel := catalogManagementService.NewPublicPublishVersionOptions(versionLocID)
				publicPublishVersionOptionsModel.SetVersionLocID("testString")
				publicPublishVersionOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(publicPublishVersionOptionsModel).ToNot(BeNil())
				Expect(publicPublishVersionOptionsModel.VersionLocID).To(Equal(core.StringPtr("testString")))
				Expect(publicPublishVersionOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewPutOfferingInstanceOptions successfully`, func() {
				// Construct an instance of the PutOfferingInstanceOptions model
				instanceIdentifier := "testString"
				xAuthRefreshToken := "testString"
				putOfferingInstanceOptionsModel := catalogManagementService.NewPutOfferingInstanceOptions(instanceIdentifier, xAuthRefreshToken)
				putOfferingInstanceOptionsModel.SetInstanceIdentifier("testString")
				putOfferingInstanceOptionsModel.SetXAuthRefreshToken("testString")
				putOfferingInstanceOptionsModel.SetID("testString")
				putOfferingInstanceOptionsModel.SetRev("testString")
				putOfferingInstanceOptionsModel.SetURL("testString")
				putOfferingInstanceOptionsModel.SetCRN("testString")
				putOfferingInstanceOptionsModel.SetLabel("testString")
				putOfferingInstanceOptionsModel.SetCatalogID("testString")
				putOfferingInstanceOptionsModel.SetOfferingID("testString")
				putOfferingInstanceOptionsModel.SetKindFormat("testString")
				putOfferingInstanceOptionsModel.SetVersion("testString")
				putOfferingInstanceOptionsModel.SetClusterID("testString")
				putOfferingInstanceOptionsModel.SetClusterRegion("testString")
				putOfferingInstanceOptionsModel.SetClusterNamespaces([]string{"testString"})
				putOfferingInstanceOptionsModel.SetClusterAllNamespaces(true)
				putOfferingInstanceOptionsModel.SetSchematicsWorkspaceID("testString")
				putOfferingInstanceOptionsModel.SetResourceGroupID("testString")
				putOfferingInstanceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(putOfferingInstanceOptionsModel).ToNot(BeNil())
				Expect(putOfferingInstanceOptionsModel.InstanceIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.XAuthRefreshToken).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.URL).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.CRN).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.CatalogID).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.KindFormat).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.Version).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.ClusterID).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.ClusterRegion).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.ClusterNamespaces).To(Equal([]string{"testString"}))
				Expect(putOfferingInstanceOptionsModel.ClusterAllNamespaces).To(Equal(core.BoolPtr(true)))
				Expect(putOfferingInstanceOptionsModel.SchematicsWorkspaceID).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))
				Expect(putOfferingInstanceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewReloadOfferingOptions successfully`, func() {
				// Construct an instance of the ReloadOfferingOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				targetVersion := "testString"
				reloadOfferingOptionsModel := catalogManagementService.NewReloadOfferingOptions(catalogIdentifier, offeringID, targetVersion)
				reloadOfferingOptionsModel.SetCatalogIdentifier("testString")
				reloadOfferingOptionsModel.SetOfferingID("testString")
				reloadOfferingOptionsModel.SetTargetVersion("testString")
				reloadOfferingOptionsModel.SetTags([]string{"testString"})
				reloadOfferingOptionsModel.SetTargetKinds([]string{"testString"})
				reloadOfferingOptionsModel.SetContent(*CreateMockByteArray("This is a mock byte array value."))
				reloadOfferingOptionsModel.SetZipurl("testString")
				reloadOfferingOptionsModel.SetRepoType("testString")
				reloadOfferingOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(reloadOfferingOptionsModel).ToNot(BeNil())
				Expect(reloadOfferingOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(reloadOfferingOptionsModel.OfferingID).To(Equal(core.StringPtr("testString")))
				Expect(reloadOfferingOptionsModel.TargetVersion).To(Equal(core.StringPtr("testString")))
				Expect(reloadOfferingOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(reloadOfferingOptionsModel.TargetKinds).To(Equal([]string{"testString"}))
				Expect(reloadOfferingOptionsModel.Content).To(Equal(CreateMockByteArray("This is a mock byte array value.")))
				Expect(reloadOfferingOptionsModel.Zipurl).To(Equal(core.StringPtr("testString")))
				Expect(reloadOfferingOptionsModel.RepoType).To(Equal(core.StringPtr("testString")))
				Expect(reloadOfferingOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewReplaceCatalogOptions successfully`, func() {
				// Construct an instance of the Feature model
				featureModel := new(catalogmanagementv1.Feature)
				Expect(featureModel).ToNot(BeNil())
				featureModel.Title = core.StringPtr("testString")
				featureModel.Description = core.StringPtr("testString")
				Expect(featureModel.Title).To(Equal(core.StringPtr("testString")))
				Expect(featureModel.Description).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the FilterTerms model
				filterTermsModel := new(catalogmanagementv1.FilterTerms)
				Expect(filterTermsModel).ToNot(BeNil())
				filterTermsModel.FilterTerms = []string{"testString"}
				Expect(filterTermsModel.FilterTerms).To(Equal([]string{"testString"}))

				// Construct an instance of the CategoryFilter model
				categoryFilterModel := new(catalogmanagementv1.CategoryFilter)
				Expect(categoryFilterModel).ToNot(BeNil())
				categoryFilterModel.Include = core.BoolPtr(true)
				categoryFilterModel.Filter = filterTermsModel
				Expect(categoryFilterModel.Include).To(Equal(core.BoolPtr(true)))
				Expect(categoryFilterModel.Filter).To(Equal(filterTermsModel))

				// Construct an instance of the IDFilter model
				idFilterModel := new(catalogmanagementv1.IDFilter)
				Expect(idFilterModel).ToNot(BeNil())
				idFilterModel.Include = filterTermsModel
				idFilterModel.Exclude = filterTermsModel
				Expect(idFilterModel.Include).To(Equal(filterTermsModel))
				Expect(idFilterModel.Exclude).To(Equal(filterTermsModel))

				// Construct an instance of the Filters model
				filtersModel := new(catalogmanagementv1.Filters)
				Expect(filtersModel).ToNot(BeNil())
				filtersModel.IncludeAll = core.BoolPtr(true)
				filtersModel.CategoryFilters = make(map[string]catalogmanagementv1.CategoryFilter)
				filtersModel.IDFilters = idFilterModel
				filtersModel.CategoryFilters["foo"] = *categoryFilterModel
				Expect(filtersModel.IncludeAll).To(Equal(core.BoolPtr(true)))
				Expect(filtersModel.IDFilters).To(Equal(idFilterModel))
				Expect(filtersModel.CategoryFilters["foo"]).To(Equal(*categoryFilterModel))

				// Construct an instance of the SyndicationCluster model
				syndicationClusterModel := new(catalogmanagementv1.SyndicationCluster)
				Expect(syndicationClusterModel).ToNot(BeNil())
				syndicationClusterModel.Region = core.StringPtr("testString")
				syndicationClusterModel.ID = core.StringPtr("testString")
				syndicationClusterModel.Name = core.StringPtr("testString")
				syndicationClusterModel.ResourceGroupName = core.StringPtr("testString")
				syndicationClusterModel.Type = core.StringPtr("testString")
				syndicationClusterModel.Namespaces = []string{"testString"}
				syndicationClusterModel.AllNamespaces = core.BoolPtr(true)
				Expect(syndicationClusterModel.Region).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.ResourceGroupName).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.Type).To(Equal(core.StringPtr("testString")))
				Expect(syndicationClusterModel.Namespaces).To(Equal([]string{"testString"}))
				Expect(syndicationClusterModel.AllNamespaces).To(Equal(core.BoolPtr(true)))

				// Construct an instance of the SyndicationHistory model
				syndicationHistoryModel := new(catalogmanagementv1.SyndicationHistory)
				Expect(syndicationHistoryModel).ToNot(BeNil())
				syndicationHistoryModel.Namespaces = []string{"testString"}
				syndicationHistoryModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationHistoryModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				Expect(syndicationHistoryModel.Namespaces).To(Equal([]string{"testString"}))
				Expect(syndicationHistoryModel.Clusters).To(Equal([]catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}))
				Expect(syndicationHistoryModel.LastRun).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))

				// Construct an instance of the SyndicationAuthorization model
				syndicationAuthorizationModel := new(catalogmanagementv1.SyndicationAuthorization)
				Expect(syndicationAuthorizationModel).ToNot(BeNil())
				syndicationAuthorizationModel.Token = core.StringPtr("testString")
				syndicationAuthorizationModel.LastRun = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				Expect(syndicationAuthorizationModel.Token).To(Equal(core.StringPtr("testString")))
				Expect(syndicationAuthorizationModel.LastRun).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))

				// Construct an instance of the SyndicationResource model
				syndicationResourceModel := new(catalogmanagementv1.SyndicationResource)
				Expect(syndicationResourceModel).ToNot(BeNil())
				syndicationResourceModel.RemoveRelatedComponents = core.BoolPtr(true)
				syndicationResourceModel.Clusters = []catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}
				syndicationResourceModel.History = syndicationHistoryModel
				syndicationResourceModel.Authorization = syndicationAuthorizationModel
				Expect(syndicationResourceModel.RemoveRelatedComponents).To(Equal(core.BoolPtr(true)))
				Expect(syndicationResourceModel.Clusters).To(Equal([]catalogmanagementv1.SyndicationCluster{*syndicationClusterModel}))
				Expect(syndicationResourceModel.History).To(Equal(syndicationHistoryModel))
				Expect(syndicationResourceModel.Authorization).To(Equal(syndicationAuthorizationModel))

				// Construct an instance of the ReplaceCatalogOptions model
				catalogIdentifier := "testString"
				replaceCatalogOptionsModel := catalogManagementService.NewReplaceCatalogOptions(catalogIdentifier)
				replaceCatalogOptionsModel.SetCatalogIdentifier("testString")
				replaceCatalogOptionsModel.SetID("testString")
				replaceCatalogOptionsModel.SetRev("testString")
				replaceCatalogOptionsModel.SetLabel("testString")
				replaceCatalogOptionsModel.SetShortDescription("testString")
				replaceCatalogOptionsModel.SetCatalogIconURL("testString")
				replaceCatalogOptionsModel.SetTags([]string{"testString"})
				replaceCatalogOptionsModel.SetFeatures([]catalogmanagementv1.Feature{*featureModel})
				replaceCatalogOptionsModel.SetDisabled(true)
				replaceCatalogOptionsModel.SetResourceGroupID("testString")
				replaceCatalogOptionsModel.SetOwningAccount("testString")
				replaceCatalogOptionsModel.SetCatalogFilters(filtersModel)
				replaceCatalogOptionsModel.SetSyndicationSettings(syndicationResourceModel)
				replaceCatalogOptionsModel.SetKind("testString")
				replaceCatalogOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(replaceCatalogOptionsModel).ToNot(BeNil())
				Expect(replaceCatalogOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.CatalogIconURL).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(replaceCatalogOptionsModel.Features).To(Equal([]catalogmanagementv1.Feature{*featureModel}))
				Expect(replaceCatalogOptionsModel.Disabled).To(Equal(core.BoolPtr(true)))
				Expect(replaceCatalogOptionsModel.ResourceGroupID).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.OwningAccount).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.CatalogFilters).To(Equal(filtersModel))
				Expect(replaceCatalogOptionsModel.SyndicationSettings).To(Equal(syndicationResourceModel))
				Expect(replaceCatalogOptionsModel.Kind).To(Equal(core.StringPtr("testString")))
				Expect(replaceCatalogOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewReplaceObjectOptions successfully`, func() {
				// Construct an instance of the PublishObject model
				publishObjectModel := new(catalogmanagementv1.PublishObject)
				Expect(publishObjectModel).ToNot(BeNil())
				publishObjectModel.PermitIBMPublicPublish = core.BoolPtr(true)
				publishObjectModel.IBMApproved = core.BoolPtr(true)
				publishObjectModel.PublicApproved = core.BoolPtr(true)
				publishObjectModel.PortalApprovalRecord = core.StringPtr("testString")
				publishObjectModel.PortalURL = core.StringPtr("testString")
				Expect(publishObjectModel.PermitIBMPublicPublish).To(Equal(core.BoolPtr(true)))
				Expect(publishObjectModel.IBMApproved).To(Equal(core.BoolPtr(true)))
				Expect(publishObjectModel.PublicApproved).To(Equal(core.BoolPtr(true)))
				Expect(publishObjectModel.PortalApprovalRecord).To(Equal(core.StringPtr("testString")))
				Expect(publishObjectModel.PortalURL).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the State model
				stateModel := new(catalogmanagementv1.State)
				Expect(stateModel).ToNot(BeNil())
				stateModel.Current = core.StringPtr("testString")
				stateModel.CurrentEntered = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Pending = core.StringPtr("testString")
				stateModel.PendingRequested = CreateMockDateTime("2019-01-01T12:00:00.000Z")
				stateModel.Previous = core.StringPtr("testString")
				Expect(stateModel.Current).To(Equal(core.StringPtr("testString")))
				Expect(stateModel.CurrentEntered).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(stateModel.Pending).To(Equal(core.StringPtr("testString")))
				Expect(stateModel.PendingRequested).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(stateModel.Previous).To(Equal(core.StringPtr("testString")))

				// Construct an instance of the ReplaceObjectOptions model
				catalogIdentifier := "testString"
				objectIdentifier := "testString"
				replaceObjectOptionsModel := catalogManagementService.NewReplaceObjectOptions(catalogIdentifier, objectIdentifier)
				replaceObjectOptionsModel.SetCatalogIdentifier("testString")
				replaceObjectOptionsModel.SetObjectIdentifier("testString")
				replaceObjectOptionsModel.SetID("testString")
				replaceObjectOptionsModel.SetName("testString")
				replaceObjectOptionsModel.SetRev("testString")
				replaceObjectOptionsModel.SetCRN("testString")
				replaceObjectOptionsModel.SetURL("testString")
				replaceObjectOptionsModel.SetParentID("testString")
				replaceObjectOptionsModel.SetLabelI18n("testString")
				replaceObjectOptionsModel.SetLabel("testString")
				replaceObjectOptionsModel.SetTags([]string{"testString"})
				replaceObjectOptionsModel.SetCreated(CreateMockDateTime("2019-01-01T12:00:00.000Z"))
				replaceObjectOptionsModel.SetUpdated(CreateMockDateTime("2019-01-01T12:00:00.000Z"))
				replaceObjectOptionsModel.SetShortDescription("testString")
				replaceObjectOptionsModel.SetShortDescriptionI18n("testString")
				replaceObjectOptionsModel.SetKind("testString")
				replaceObjectOptionsModel.SetPublish(publishObjectModel)
				replaceObjectOptionsModel.SetState(stateModel)
				replaceObjectOptionsModel.SetCatalogID("testString")
				replaceObjectOptionsModel.SetCatalogName("testString")
				replaceObjectOptionsModel.SetData(make(map[string]interface{}))
				replaceObjectOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(replaceObjectOptionsModel).ToNot(BeNil())
				Expect(replaceObjectOptionsModel.CatalogIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.ObjectIdentifier).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.ID).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Rev).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.CRN).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.URL).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.ParentID).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.LabelI18n).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Label).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Tags).To(Equal([]string{"testString"}))
				Expect(replaceObjectOptionsModel.Created).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(replaceObjectOptionsModel.Updated).To(Equal(CreateMockDateTime("2019-01-01T12:00:00.000Z")))
				Expect(replaceObjectOptionsModel.ShortDescription).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.ShortDescriptionI18n).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Kind).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Publish).To(Equal(publishObjectModel))
				Expect(replaceObjectOptionsModel.State).To(Equal(stateModel))
				Expect(replaceObjectOptionsModel.CatalogID).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.CatalogName).To(Equal(core.StringPtr("testString")))
				Expect(replaceObjectOptionsModel.Data).To(Equal(make(map[string]interface{})))
				Expect(replaceObjectOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewReplaceOfferingIconOptions successfully`, func() {
				// Construct an instance of the ReplaceOfferingIconOptions model
				catalogIdentifier := "testString"
				offeringID := "testString"
				fileName := "testString"
				replaceOfferingIconOptionsModel := catalogManagementService.NewReplaceOfferingIconOptions(catalogIdentifier, offeringID, fileName)
				replaceOfferingIconOptionsModel.SetCatalogIdentifier("test