package testhelpers

import (
	"log"
	"net/http"
	"net/http/httputil"
	"os"

	"github.com/Azure/go-autorest/autorest"
)

func buildSender() autorest.Sender {
	return autorest.DecorateSender(&http.Client{
		Transport: &http.Transport{
			Proxy: http.ProxyFromEnvironment,
		},
	}, withRequestLogging())
}

func withRequestLogging() autorest.SendDecorator {
	return func(s autorest.Sender) autorest.Sender {
		return autorest.SenderFunc(func(r *http.Request) (*http.Response, error) {
			shouldLog := os.Getenv("TEST_LOG") != ""

			if shouldLog {
				// strip the authorization header prior to printing
				authHeaderName := "Authorization"
				auth := r.Header.Get(authHeaderName)
				if auth != "" {
					r.Header.Del(authHeaderName)
				}

				// dump request to wire format
				if dump, err := httputil.DumpRequestOut(r, true); err == nil {
					log.Printf("[DEBUG] AzureRM Request: \n%s\n", dump)
				} else {
					// fallback to basic message
					log.Printf("[DEBUG] AzureRM Request: %s to %s\n", r.Method, r.URL)
				}

				// add the auth header back
				if auth != "" {
					r.Header.Add(authHeaderName, auth)
				}
			}

			resp, err := s.Do(r)

			if shouldLog {
				if resp != nil {
					// dump response to wire format
					if dump, err2 := httputil.DumpResponse(resp, true); err2 == nil {
						log.Printf("[DEBUG] AzureRM Response for %s: \n%s\n", r.URL, dump)
					} else {
						// fallback to basic message
						log.Printf("[DEBUG] AzureRM Response: %s for %s\n", resp.Status, r.URL)
					}
				} else {
					log.Printf("[DEBUG] Request to %s completed with no response", r.URL)
				}
			}
			return resp, err
		})
	}
}
