/**
 * (C) Copyright IBM Corp. 2020, 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * IBM OpenAPI SDK Code Generator Version: 3.26.0-4b317b0c-20210127-171701
 */

// Package vpcclassicv1 : Operations and models for the VpcClassicV1 service
package vpcclassicv1

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"reflect"
	"time"

	"github.com/IBM/go-sdk-core/v5/core"
	common "github.com/IBM/vpc-go-sdk/common"
	"github.com/go-openapi/strfmt"
)

// VpcClassicV1 : The IBM Cloud Virtual Private Cloud (VPC) API can be used to programmatically provision and manage
// infrastructure resources, including virtual server instances, subnets, volumes, and load balancers.
//
// Version: 2021-02-09
type VpcClassicV1 struct {
	Service *core.BaseService

	// Requests the version of the API as of a date in the format `YYYY-MM-DD`. Any date up to the current date may be
	// provided. Specify the current date to request the latest version.
	Version *string

	// The infrastructure generation for the request. For the API behavior documented here, use
	// `1`.
	generation *int64
}

// DefaultServiceURL is the default URL to make service requests to.
const DefaultServiceURL = "https://us-south.iaas.cloud.ibm.com/v1"

// DefaultServiceName is the default key used to find external configuration information.
const DefaultServiceName = "vpc_classic"

// VpcClassicV1Options : Service options
type VpcClassicV1Options struct {
	ServiceName   string
	URL           string
	Authenticator core.Authenticator

	// Requests the version of the API as of a date in the format `YYYY-MM-DD`. Any date up to the current date may be
	// provided. Specify the current date to request the latest version.
	Version *string
}

// NewVpcClassicV1UsingExternalConfig : constructs an instance of VpcClassicV1 with passed in options and external configuration.
func NewVpcClassicV1UsingExternalConfig(options *VpcClassicV1Options) (vpcClassic *VpcClassicV1, err error) {
	if options.ServiceName == "" {
		options.ServiceName = DefaultServiceName
	}

	if options.Authenticator == nil {
		options.Authenticator, err = core.GetAuthenticatorFromEnvironment(options.ServiceName)
		if err != nil {
			return
		}
	}

	vpcClassic, err = NewVpcClassicV1(options)
	if err != nil {
		return
	}

	err = vpcClassic.Service.ConfigureService(options.ServiceName)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = vpcClassic.Service.SetServiceURL(options.URL)
	}
	return
}

// NewVpcClassicV1 : constructs an instance of VpcClassicV1 with passed in options.
func NewVpcClassicV1(options *VpcClassicV1Options) (service *VpcClassicV1, err error) {
	serviceOptions := &core.ServiceOptions{
		URL:           DefaultServiceURL,
		Authenticator: options.Authenticator,
	}

	err = core.ValidateStruct(options, "options")
	if err != nil {
		return
	}

	baseService, err := core.NewBaseService(serviceOptions)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = baseService.SetServiceURL(options.URL)
		if err != nil {
			return
		}
	}

	if options.Version == nil {
		options.Version = core.StringPtr("2021-02-09")
	}

	service = &VpcClassicV1{
		Service:    baseService,
		Version:    options.Version,
		generation: core.Int64Ptr(1),
	}

	return
}

// GetServiceURLForRegion returns the service URL to be used for the specified region
func GetServiceURLForRegion(region string) (string, error) {
	return "", fmt.Errorf("service does not support regional URLs")
}

// Clone makes a copy of "vpcClassic" suitable for processing requests.
func (vpcClassic *VpcClassicV1) Clone() *VpcClassicV1 {
	if core.IsNil(vpcClassic) {
		return nil
	}
	clone := *vpcClassic
	clone.Service = vpcClassic.Service.Clone()
	return &clone
}

// SetServiceURL sets the service URL
func (vpcClassic *VpcClassicV1) SetServiceURL(url string) error {
	return vpcClassic.Service.SetServiceURL(url)
}

// GetServiceURL returns the service URL
func (vpcClassic *VpcClassicV1) GetServiceURL() string {
	return vpcClassic.Service.GetServiceURL()
}

// SetDefaultHeaders sets HTTP headers to be sent in every request
func (vpcClassic *VpcClassicV1) SetDefaultHeaders(headers http.Header) {
	vpcClassic.Service.SetDefaultHeaders(headers)
}

// SetEnableGzipCompression sets the service's EnableGzipCompression field
func (vpcClassic *VpcClassicV1) SetEnableGzipCompression(enableGzip bool) {
	vpcClassic.Service.SetEnableGzipCompression(enableGzip)
}

// GetEnableGzipCompression returns the service's EnableGzipCompression field
func (vpcClassic *VpcClassicV1) GetEnableGzipCompression() bool {
	return vpcClassic.Service.GetEnableGzipCompression()
}

// EnableRetries enables automatic retries for requests invoked for this service instance.
// If either parameter is specified as 0, then a default value is used instead.
func (vpcClassic *VpcClassicV1) EnableRetries(maxRetries int, maxRetryInterval time.Duration) {
	vpcClassic.Service.EnableRetries(maxRetries, maxRetryInterval)
}

// DisableRetries disables automatic retries for requests invoked for this service instance.
func (vpcClassic *VpcClassicV1) DisableRetries() {
	vpcClassic.Service.DisableRetries()
}

// ListVpcs : List all VPCs
// This request lists all VPCs in the region. A VPC is a virtual network that belongs to an account and provides logical
// isolation from other networks. A VPC is made up of resources in one or more zones. VPCs are regional, and each VPC
// can contain resources in multiple zones in a region.
func (vpcClassic *VpcClassicV1) ListVpcs(listVpcsOptions *ListVpcsOptions) (result *VPCCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVpcsWithContext(context.Background(), listVpcsOptions)
}

// ListVpcsWithContext is an alternate form of the ListVpcs method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVpcsWithContext(ctx context.Context, listVpcsOptions *ListVpcsOptions) (result *VPCCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVpcsOptions, "listVpcsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVpcsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVpcs")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVpcsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVpcsOptions.Start))
	}
	if listVpcsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVpcsOptions.Limit))
	}
	if listVpcsOptions.ClassicAccess != nil {
		builder.AddQuery("classic_access", fmt.Sprint(*listVpcsOptions.ClassicAccess))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPCCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateVPC : Create a VPC
// This request creates a new VPC from a VPC prototype object. The prototype object is structured in the same way as a
// retrieved VPC, and contains the information necessary to create the new VPC.
func (vpcClassic *VpcClassicV1) CreateVPC(createVPCOptions *CreateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateVPCWithContext(context.Background(), createVPCOptions)
}

// CreateVPCWithContext is an alternate form of the CreateVPC method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateVPCWithContext(ctx context.Context, createVPCOptions *CreateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(createVPCOptions, "createVPCOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createVPCOptions.AddressPrefixManagement != nil {
		body["address_prefix_management"] = createVPCOptions.AddressPrefixManagement
	}
	if createVPCOptions.ClassicAccess != nil {
		body["classic_access"] = createVPCOptions.ClassicAccess
	}
	if createVPCOptions.Name != nil {
		body["name"] = createVPCOptions.Name
	}
	if createVPCOptions.ResourceGroup != nil {
		body["resource_group"] = createVPCOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPC)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteVPC : Delete a VPC
// This request deletes a VPC. This operation cannot be reversed. For this request to succeed, the VPC must not contain
// any instances, subnets, or public gateways. All security groups associated with the VPC are automatically deleted. If
// the default network ACL was automatically created, it is automatically deleted.
func (vpcClassic *VpcClassicV1) DeleteVPC(deleteVPCOptions *DeleteVPCOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteVPCWithContext(context.Background(), deleteVPCOptions)
}

// DeleteVPCWithContext is an alternate form of the DeleteVPC method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteVPCWithContext(ctx context.Context, deleteVPCOptions *DeleteVPCOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCOptions, "deleteVPCOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCOptions, "deleteVPCOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteVPCOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetVPC : Retrieve a VPC
// This request retrieves a single VPC specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetVPC(getVPCOptions *GetVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVPCWithContext(context.Background(), getVPCOptions)
}

// GetVPCWithContext is an alternate form of the GetVPC method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVPCWithContext(ctx context.Context, getVPCOptions *GetVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCOptions, "getVPCOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCOptions, "getVPCOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPCOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPC)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateVPC : Update a VPC
// This request updates a VPC's name.
func (vpcClassic *VpcClassicV1) UpdateVPC(updateVPCOptions *UpdateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateVPCWithContext(context.Background(), updateVPCOptions)
}

// UpdateVPCWithContext is an alternate form of the UpdateVPC method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateVPCWithContext(ctx context.Context, updateVPCOptions *UpdateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCOptions, "updateVPCOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCOptions, "updateVPCOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateVPCOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateVPCOptions.VPCPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPC)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetVPCDefaultSecurityGroup : Retrieve a VPC's default security group
// This request retrieves the default security group for the VPC specified by the identifier in the URL. The default
// security group is applied to any new network interfaces in the VPC that do not specify a security group.
func (vpcClassic *VpcClassicV1) GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions) (result *DefaultSecurityGroup, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVPCDefaultSecurityGroupWithContext(context.Background(), getVPCDefaultSecurityGroupOptions)
}

// GetVPCDefaultSecurityGroupWithContext is an alternate form of the GetVPCDefaultSecurityGroup method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVPCDefaultSecurityGroupWithContext(ctx context.Context, getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions) (result *DefaultSecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCDefaultSecurityGroupOptions, "getVPCDefaultSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCDefaultSecurityGroupOptions, "getVPCDefaultSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPCDefaultSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{id}/default_security_group`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCDefaultSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVPCDefaultSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDefaultSecurityGroup)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVPCAddressPrefixes : List all address prefixes for a VPC
// This request lists all address pool prefixes for a VPC.
func (vpcClassic *VpcClassicV1) ListVPCAddressPrefixes(listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions) (result *AddressPrefixCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVPCAddressPrefixesWithContext(context.Background(), listVPCAddressPrefixesOptions)
}

// ListVPCAddressPrefixesWithContext is an alternate form of the ListVPCAddressPrefixes method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVPCAddressPrefixesWithContext(ctx context.Context, listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions) (result *AddressPrefixCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPCAddressPrefixesOptions, "listVPCAddressPrefixesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPCAddressPrefixesOptions, "listVPCAddressPrefixesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *listVPCAddressPrefixesOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPCAddressPrefixesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVPCAddressPrefixes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVPCAddressPrefixesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPCAddressPrefixesOptions.Start))
	}
	if listVPCAddressPrefixesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPCAddressPrefixesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefixCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateVPCAddressPrefix : Create an address prefix for a VPC
// This request creates a new prefix from a prefix prototype object. The prototype object is structured in the same way
// as a retrieved prefix, and contains the information necessary to create the new prefix.
func (vpcClassic *VpcClassicV1) CreateVPCAddressPrefix(createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateVPCAddressPrefixWithContext(context.Background(), createVPCAddressPrefixOptions)
}

// CreateVPCAddressPrefixWithContext is an alternate form of the CreateVPCAddressPrefix method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateVPCAddressPrefixWithContext(ctx context.Context, createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPCAddressPrefixOptions, "createVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPCAddressPrefixOptions, "createVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *createVPCAddressPrefixOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createVPCAddressPrefixOptions.CIDR != nil {
		body["cidr"] = createVPCAddressPrefixOptions.CIDR
	}
	if createVPCAddressPrefixOptions.Zone != nil {
		body["zone"] = createVPCAddressPrefixOptions.Zone
	}
	if createVPCAddressPrefixOptions.IsDefault != nil {
		body["is_default"] = createVPCAddressPrefixOptions.IsDefault
	}
	if createVPCAddressPrefixOptions.Name != nil {
		body["name"] = createVPCAddressPrefixOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefix)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteVPCAddressPrefix : Delete an address prefix
// This request deletes a prefix. This operation cannot be reversed. The request will fail if any subnets use addresses
// from this prefix.
func (vpcClassic *VpcClassicV1) DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptions *DeleteVPCAddressPrefixOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteVPCAddressPrefixWithContext(context.Background(), deleteVPCAddressPrefixOptions)
}

// DeleteVPCAddressPrefixWithContext is an alternate form of the DeleteVPCAddressPrefix method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteVPCAddressPrefixWithContext(ctx context.Context, deleteVPCAddressPrefixOptions *DeleteVPCAddressPrefixOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCAddressPrefixOptions, "deleteVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCAddressPrefixOptions, "deleteVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *deleteVPCAddressPrefixOptions.VPCID,
		"id":     *deleteVPCAddressPrefixOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetVPCAddressPrefix : Retrieve an address prefix
// This request retrieves a single prefix specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetVPCAddressPrefix(getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVPCAddressPrefixWithContext(context.Background(), getVPCAddressPrefixOptions)
}

// GetVPCAddressPrefixWithContext is an alternate form of the GetVPCAddressPrefix method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVPCAddressPrefixWithContext(ctx context.Context, getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCAddressPrefixOptions, "getVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCAddressPrefixOptions, "getVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *getVPCAddressPrefixOptions.VPCID,
		"id":     *getVPCAddressPrefixOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefix)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateVPCAddressPrefix : Update an address prefix
// This request updates a prefix with the information in a provided prefix patch. The prefix patch object is structured
// in the same way as a retrieved prefix and contains only the information to be updated.
func (vpcClassic *VpcClassicV1) UpdateVPCAddressPrefix(updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateVPCAddressPrefixWithContext(context.Background(), updateVPCAddressPrefixOptions)
}

// UpdateVPCAddressPrefixWithContext is an alternate form of the UpdateVPCAddressPrefix method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateVPCAddressPrefixWithContext(ctx context.Context, updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCAddressPrefixOptions, "updateVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCAddressPrefixOptions, "updateVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *updateVPCAddressPrefixOptions.VPCID,
		"id":     *updateVPCAddressPrefixOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateVPCAddressPrefixOptions.AddressPrefixPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefix)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVPCRoutes : List all routes in a VPC's default routing table
// This request lists all routes in the VPC's default routing table. Each route is zone-specific and directs any packets
// matching its destination CIDR block to a `next_hop` IP address. The most specific route matching a packet's
// destination will be used. If multiple equally-specific routes exist, traffic will be distributed across them.
func (vpcClassic *VpcClassicV1) ListVPCRoutes(listVPCRoutesOptions *ListVPCRoutesOptions) (result *RouteCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVPCRoutesWithContext(context.Background(), listVPCRoutesOptions)
}

// ListVPCRoutesWithContext is an alternate form of the ListVPCRoutes method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVPCRoutesWithContext(ctx context.Context, listVPCRoutesOptions *ListVPCRoutesOptions) (result *RouteCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPCRoutesOptions, "listVPCRoutesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPCRoutesOptions, "listVPCRoutesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *listVPCRoutesOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/routes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPCRoutesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVPCRoutes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVPCRoutesOptions.ZoneName != nil {
		builder.AddQuery("zone.name", fmt.Sprint(*listVPCRoutesOptions.ZoneName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRouteCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateVPCRoute : Create a route in a VPC's default routing table
// This request creates a new route in the VPC's default routing table. The route prototype object is structured in the
// same way as a retrieved route, and contains the information necessary to create the new route. The request will fail
// if the new route will cause a loop.
func (vpcClassic *VpcClassicV1) CreateVPCRoute(createVPCRouteOptions *CreateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateVPCRouteWithContext(context.Background(), createVPCRouteOptions)
}

// CreateVPCRouteWithContext is an alternate form of the CreateVPCRoute method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateVPCRouteWithContext(ctx context.Context, createVPCRouteOptions *CreateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPCRouteOptions, "createVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPCRouteOptions, "createVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *createVPCRouteOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/routes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createVPCRouteOptions.Destination != nil {
		body["destination"] = createVPCRouteOptions.Destination
	}
	if createVPCRouteOptions.NextHop != nil {
		body["next_hop"] = createVPCRouteOptions.NextHop
	}
	if createVPCRouteOptions.Zone != nil {
		body["zone"] = createVPCRouteOptions.Zone
	}
	if createVPCRouteOptions.Name != nil {
		body["name"] = createVPCRouteOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteVPCRoute : Delete a VPC route
// This request deletes a route. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteVPCRoute(deleteVPCRouteOptions *DeleteVPCRouteOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteVPCRouteWithContext(context.Background(), deleteVPCRouteOptions)
}

// DeleteVPCRouteWithContext is an alternate form of the DeleteVPCRoute method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteVPCRouteWithContext(ctx context.Context, deleteVPCRouteOptions *DeleteVPCRouteOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCRouteOptions, "deleteVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCRouteOptions, "deleteVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *deleteVPCRouteOptions.VPCID,
		"id":     *deleteVPCRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetVPCRoute : Retrieve a VPC route
// This request retrieves a single route specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetVPCRoute(getVPCRouteOptions *GetVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVPCRouteWithContext(context.Background(), getVPCRouteOptions)
}

// GetVPCRouteWithContext is an alternate form of the GetVPCRoute method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVPCRouteWithContext(ctx context.Context, getVPCRouteOptions *GetVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCRouteOptions, "getVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCRouteOptions, "getVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *getVPCRouteOptions.VPCID,
		"id":     *getVPCRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateVPCRoute : Update a VPC route
// This request updates a route with the information in a provided route patch. The route patch object is structured in
// the same way as a retrieved route and contains only the information to be updated.
func (vpcClassic *VpcClassicV1) UpdateVPCRoute(updateVPCRouteOptions *UpdateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateVPCRouteWithContext(context.Background(), updateVPCRouteOptions)
}

// UpdateVPCRouteWithContext is an alternate form of the UpdateVPCRoute method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateVPCRouteWithContext(ctx context.Context, updateVPCRouteOptions *UpdateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCRouteOptions, "updateVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCRouteOptions, "updateVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *updateVPCRouteOptions.VPCID,
		"id":     *updateVPCRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpcs/{vpc_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateVPCRouteOptions.RoutePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListSubnets : List all subnets
// This request lists all subnets in the region. Subnets are contiguous ranges of IP addresses specified in CIDR block
// notation. Each subnet is within a particular zone and cannot span multiple zones or regions.
func (vpcClassic *VpcClassicV1) ListSubnets(listSubnetsOptions *ListSubnetsOptions) (result *SubnetCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListSubnetsWithContext(context.Background(), listSubnetsOptions)
}

// ListSubnetsWithContext is an alternate form of the ListSubnets method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListSubnetsWithContext(ctx context.Context, listSubnetsOptions *ListSubnetsOptions) (result *SubnetCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listSubnetsOptions, "listSubnetsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSubnetsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListSubnets")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listSubnetsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSubnetsOptions.Start))
	}
	if listSubnetsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSubnetsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnetCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateSubnet : Create a subnet
// This request creates a new subnet from a subnet prototype object. The prototype object is structured in the same way
// as a retrieved subnet, and contains the information necessary to create the new subnet. For this request to succeed,
// the prototype's CIDR block must not overlap with an existing subnet in the VPC.
func (vpcClassic *VpcClassicV1) CreateSubnet(createSubnetOptions *CreateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateSubnetWithContext(context.Background(), createSubnetOptions)
}

// CreateSubnetWithContext is an alternate form of the CreateSubnet method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateSubnetWithContext(ctx context.Context, createSubnetOptions *CreateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSubnetOptions, "createSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSubnetOptions, "createSubnetOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createSubnetOptions.SubnetPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnet)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteSubnet : Delete a subnet
// This request deletes a subnet. This operation cannot be reversed. For this request to succeed, the subnet must not be
// referenced by any network interfaces, VPN gateways, or load balancers. A delete operation automatically detaches the
// subnet from any network ACLs or public gateways.
func (vpcClassic *VpcClassicV1) DeleteSubnet(deleteSubnetOptions *DeleteSubnetOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteSubnetWithContext(context.Background(), deleteSubnetOptions)
}

// DeleteSubnetWithContext is an alternate form of the DeleteSubnet method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteSubnetWithContext(ctx context.Context, deleteSubnetOptions *DeleteSubnetOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSubnetOptions, "deleteSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSubnetOptions, "deleteSubnetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteSubnetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetSubnet : Retrieve a subnet
// This request retrieves a single subnet specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetSubnet(getSubnetOptions *GetSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	return vpcClassic.GetSubnetWithContext(context.Background(), getSubnetOptions)
}

// GetSubnetWithContext is an alternate form of the GetSubnet method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetSubnetWithContext(ctx context.Context, getSubnetOptions *GetSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetOptions, "getSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetOptions, "getSubnetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnet)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateSubnet : Update a subnet
// This request updates a subnet with the information in a provided subnet patch. The subnet patch object is structured
// in the same way as a retrieved subnet and contains only the information to be updated.
func (vpcClassic *VpcClassicV1) UpdateSubnet(updateSubnetOptions *UpdateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateSubnetWithContext(context.Background(), updateSubnetOptions)
}

// UpdateSubnetWithContext is an alternate form of the UpdateSubnet method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateSubnetWithContext(ctx context.Context, updateSubnetOptions *UpdateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSubnetOptions, "updateSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSubnetOptions, "updateSubnetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateSubnetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateSubnetOptions.SubnetPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnet)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetSubnetNetworkACL : Retrieve a subnet's attached network ACL
// This request retrieves the network ACL attached to the subnet specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetSubnetNetworkACL(getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpcClassic.GetSubnetNetworkACLWithContext(context.Background(), getSubnetNetworkACLOptions)
}

// GetSubnetNetworkACLWithContext is an alternate form of the GetSubnetNetworkACL method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetSubnetNetworkACLWithContext(ctx context.Context, getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetNetworkACLOptions, "getSubnetNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetNetworkACLOptions, "getSubnetNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}/network_acl`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetSubnetNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ReplaceSubnetNetworkACL : Attach a network ACL to a subnet
// This request attaches the network ACL, specified in the request body, to the subnet specified by the subnet
// identifier in the URL. This replaces the existing network ACL on the subnet.
func (vpcClassic *VpcClassicV1) ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpcClassic.ReplaceSubnetNetworkACLWithContext(context.Background(), replaceSubnetNetworkACLOptions)
}

// ReplaceSubnetNetworkACLWithContext is an alternate form of the ReplaceSubnetNetworkACL method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ReplaceSubnetNetworkACLWithContext(ctx context.Context, replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(replaceSubnetNetworkACLOptions, "replaceSubnetNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(replaceSubnetNetworkACLOptions, "replaceSubnetNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *replaceSubnetNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}/network_acl`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range replaceSubnetNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ReplaceSubnetNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(replaceSubnetNetworkACLOptions.NetworkACLIdentity)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UnsetSubnetPublicGateway : Detach a public gateway from a subnet
// This request detaches the public gateway from the subnet specified by the subnet identifier in the URL.
func (vpcClassic *VpcClassicV1) UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptions *UnsetSubnetPublicGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.UnsetSubnetPublicGatewayWithContext(context.Background(), unsetSubnetPublicGatewayOptions)
}

// UnsetSubnetPublicGatewayWithContext is an alternate form of the UnsetSubnetPublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UnsetSubnetPublicGatewayWithContext(ctx context.Context, unsetSubnetPublicGatewayOptions *UnsetSubnetPublicGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(unsetSubnetPublicGatewayOptions, "unsetSubnetPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(unsetSubnetPublicGatewayOptions, "unsetSubnetPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *unsetSubnetPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}/public_gateway`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range unsetSubnetPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UnsetSubnetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetSubnetPublicGateway : Retrieve a subnet's attached public gateway
// This request retrieves the public gateway attached to the subnet specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetSubnetPublicGateway(getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpcClassic.GetSubnetPublicGatewayWithContext(context.Background(), getSubnetPublicGatewayOptions)
}

// GetSubnetPublicGatewayWithContext is an alternate form of the GetSubnetPublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetSubnetPublicGatewayWithContext(ctx context.Context, getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetPublicGatewayOptions, "getSubnetPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetPublicGatewayOptions, "getSubnetPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}/public_gateway`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetSubnetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// SetSubnetPublicGateway : Attach a public gateway to a subnet
// This request attaches the public gateway, specified in the request body, to the subnet specified by the subnet
// identifier in the URL. The public gateway must have the same VPC and zone as the subnet.
func (vpcClassic *VpcClassicV1) SetSubnetPublicGateway(setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpcClassic.SetSubnetPublicGatewayWithContext(context.Background(), setSubnetPublicGatewayOptions)
}

// SetSubnetPublicGatewayWithContext is an alternate form of the SetSubnetPublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) SetSubnetPublicGatewayWithContext(ctx context.Context, setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(setSubnetPublicGatewayOptions, "setSubnetPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(setSubnetPublicGatewayOptions, "setSubnetPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *setSubnetPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/subnets/{id}/public_gateway`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range setSubnetPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "SetSubnetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(setSubnetPublicGatewayOptions.PublicGatewayIdentity)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListImages : List all images
// This request lists all provisionable images available in the region. An image provides source data for a volume.
// Images are either system-provided, or created from another source, such as importing from object storage.
//
// The images will be sorted by their `created_at` property values, with the newest first. Images with identical
// `created_at` values will be secondarily sorted by ascending `id` property values.
func (vpcClassic *VpcClassicV1) ListImages(listImagesOptions *ListImagesOptions) (result *ImageCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListImagesWithContext(context.Background(), listImagesOptions)
}

// ListImagesWithContext is an alternate form of the ListImages method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListImagesWithContext(ctx context.Context, listImagesOptions *ListImagesOptions) (result *ImageCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listImagesOptions, "listImagesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/images`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listImagesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListImages")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listImagesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listImagesOptions.Start))
	}
	if listImagesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listImagesOptions.Limit))
	}
	if listImagesOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listImagesOptions.ResourceGroupID))
	}
	if listImagesOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listImagesOptions.Name))
	}
	if listImagesOptions.Visibility != nil {
		builder.AddQuery("visibility", fmt.Sprint(*listImagesOptions.Visibility))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImageCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateImage : Create an image
// This request creates a new image from an image prototype object. The prototype object is structured in the same way
// as a retrieved image, and contains the information necessary to create the new image. A URL to the image file on
// object storage must be provided.
func (vpcClassic *VpcClassicV1) CreateImage(createImageOptions *CreateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateImageWithContext(context.Background(), createImageOptions)
}

// CreateImageWithContext is an alternate form of the CreateImage method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateImageWithContext(ctx context.Context, createImageOptions *CreateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createImageOptions, "createImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createImageOptions, "createImageOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/images`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createImageOptions.ImagePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImage)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteImage : Delete an image
// This request deletes an image. This operation cannot be reversed. System-provided images are not allowed to be
// deleted. An image with a `status` of `pending`, `tentative`, or `deleting` cannot be deleted.
func (vpcClassic *VpcClassicV1) DeleteImage(deleteImageOptions *DeleteImageOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteImageWithContext(context.Background(), deleteImageOptions)
}

// DeleteImageWithContext is an alternate form of the DeleteImage method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteImageWithContext(ctx context.Context, deleteImageOptions *DeleteImageOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteImageOptions, "deleteImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteImageOptions, "deleteImageOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteImageOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/images/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetImage : Retrieve an image
// This request retrieves a single image specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetImage(getImageOptions *GetImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	return vpcClassic.GetImageWithContext(context.Background(), getImageOptions)
}

// GetImageWithContext is an alternate form of the GetImage method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetImageWithContext(ctx context.Context, getImageOptions *GetImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getImageOptions, "getImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getImageOptions, "getImageOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getImageOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/images/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImage)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateImage : Update an image
// This request updates an image with the information in a provided image patch. The image patch object is structured in
// the same way as a retrieved image and contains only the information to be updated. System-provided images are not
// allowed to be updated. An image with a `status` of `deleting` cannot be updated.
func (vpcClassic *VpcClassicV1) UpdateImage(updateImageOptions *UpdateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateImageWithContext(context.Background(), updateImageOptions)
}

// UpdateImageWithContext is an alternate form of the UpdateImage method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateImageWithContext(ctx context.Context, updateImageOptions *UpdateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateImageOptions, "updateImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateImageOptions, "updateImageOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateImageOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/images/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateImageOptions.ImagePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImage)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListOperatingSystems : List all operating systems
// This request lists all operating systems in the region.
func (vpcClassic *VpcClassicV1) ListOperatingSystems(listOperatingSystemsOptions *ListOperatingSystemsOptions) (result *OperatingSystemCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListOperatingSystemsWithContext(context.Background(), listOperatingSystemsOptions)
}

// ListOperatingSystemsWithContext is an alternate form of the ListOperatingSystems method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListOperatingSystemsWithContext(ctx context.Context, listOperatingSystemsOptions *ListOperatingSystemsOptions) (result *OperatingSystemCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listOperatingSystemsOptions, "listOperatingSystemsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/operating_systems`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listOperatingSystemsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListOperatingSystems")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listOperatingSystemsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listOperatingSystemsOptions.Start))
	}
	if listOperatingSystemsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listOperatingSystemsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalOperatingSystemCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetOperatingSystem : Retrieve an operating system
// This request retrieves a single operating system specified by the name in the URL.
func (vpcClassic *VpcClassicV1) GetOperatingSystem(getOperatingSystemOptions *GetOperatingSystemOptions) (result *OperatingSystem, response *core.DetailedResponse, err error) {
	return vpcClassic.GetOperatingSystemWithContext(context.Background(), getOperatingSystemOptions)
}

// GetOperatingSystemWithContext is an alternate form of the GetOperatingSystem method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetOperatingSystemWithContext(ctx context.Context, getOperatingSystemOptions *GetOperatingSystemOptions) (result *OperatingSystem, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getOperatingSystemOptions, "getOperatingSystemOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getOperatingSystemOptions, "getOperatingSystemOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getOperatingSystemOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/operating_systems/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getOperatingSystemOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetOperatingSystem")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalOperatingSystem)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListKeys : List all keys
// This request lists all keys in the region. A key contains a public SSH key which may be installed on instances when
// they are created. Private keys are not stored.
func (vpcClassic *VpcClassicV1) ListKeys(listKeysOptions *ListKeysOptions) (result *KeyCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListKeysWithContext(context.Background(), listKeysOptions)
}

// ListKeysWithContext is an alternate form of the ListKeys method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListKeysWithContext(ctx context.Context, listKeysOptions *ListKeysOptions) (result *KeyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listKeysOptions, "listKeysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/keys`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listKeysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListKeys")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listKeysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listKeysOptions.Start))
	}
	if listKeysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listKeysOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKeyCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateKey : Create a key
// This request creates a new SSH key from an key prototype object. The prototype object is structured in the same way
// as a retrieved key, and contains the information necessary to create the new key. The public key value must be
// provided.
func (vpcClassic *VpcClassicV1) CreateKey(createKeyOptions *CreateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateKeyWithContext(context.Background(), createKeyOptions)
}

// CreateKeyWithContext is an alternate form of the CreateKey method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateKeyWithContext(ctx context.Context, createKeyOptions *CreateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createKeyOptions, "createKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createKeyOptions, "createKeyOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/keys`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createKeyOptions.PublicKey != nil {
		body["public_key"] = createKeyOptions.PublicKey
	}
	if createKeyOptions.Name != nil {
		body["name"] = createKeyOptions.Name
	}
	if createKeyOptions.ResourceGroup != nil {
		body["resource_group"] = createKeyOptions.ResourceGroup
	}
	if createKeyOptions.Type != nil {
		body["type"] = createKeyOptions.Type
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKey)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteKey : Delete a key
// This request deletes a key. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteKey(deleteKeyOptions *DeleteKeyOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteKeyWithContext(context.Background(), deleteKeyOptions)
}

// DeleteKeyWithContext is an alternate form of the DeleteKey method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteKeyWithContext(ctx context.Context, deleteKeyOptions *DeleteKeyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteKeyOptions, "deleteKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteKeyOptions, "deleteKeyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteKeyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/keys/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetKey : Retrieve a key
// This request retrieves a single key specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetKey(getKeyOptions *GetKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	return vpcClassic.GetKeyWithContext(context.Background(), getKeyOptions)
}

// GetKeyWithContext is an alternate form of the GetKey method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetKeyWithContext(ctx context.Context, getKeyOptions *GetKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getKeyOptions, "getKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getKeyOptions, "getKeyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getKeyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/keys/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKey)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateKey : Update a key
// This request updates a key's name.
func (vpcClassic *VpcClassicV1) UpdateKey(updateKeyOptions *UpdateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateKeyWithContext(context.Background(), updateKeyOptions)
}

// UpdateKeyWithContext is an alternate form of the UpdateKey method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateKeyWithContext(ctx context.Context, updateKeyOptions *UpdateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateKeyOptions, "updateKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateKeyOptions, "updateKeyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateKeyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/keys/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateKeyOptions.KeyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKey)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListInstanceProfiles : List all instance profiles
// This request lists provisionable instance profiles in the region. An instance profile specifies the performance
// characteristics and pricing model for an instance.
func (vpcClassic *VpcClassicV1) ListInstanceProfiles(listInstanceProfilesOptions *ListInstanceProfilesOptions) (result *InstanceProfileCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListInstanceProfilesWithContext(context.Background(), listInstanceProfilesOptions)
}

// ListInstanceProfilesWithContext is an alternate form of the ListInstanceProfiles method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListInstanceProfilesWithContext(ctx context.Context, listInstanceProfilesOptions *ListInstanceProfilesOptions) (result *InstanceProfileCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listInstanceProfilesOptions, "listInstanceProfilesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instance/profiles`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceProfilesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListInstanceProfiles")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listInstanceProfilesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstanceProfilesOptions.Start))
	}
	if listInstanceProfilesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstanceProfilesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceProfileCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetInstanceProfile : Retrieve an instance profile
// This request retrieves a single instance profile specified by the name in the URL.
func (vpcClassic *VpcClassicV1) GetInstanceProfile(getInstanceProfileOptions *GetInstanceProfileOptions) (result *InstanceProfile, response *core.DetailedResponse, err error) {
	return vpcClassic.GetInstanceProfileWithContext(context.Background(), getInstanceProfileOptions)
}

// GetInstanceProfileWithContext is an alternate form of the GetInstanceProfile method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetInstanceProfileWithContext(ctx context.Context, getInstanceProfileOptions *GetInstanceProfileOptions) (result *InstanceProfile, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceProfileOptions, "getInstanceProfileOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceProfileOptions, "getInstanceProfileOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getInstanceProfileOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instance/profiles/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceProfileOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetInstanceProfile")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceProfile)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListInstances : List all instances
// This request lists all instances in the region.
func (vpcClassic *VpcClassicV1) ListInstances(listInstancesOptions *ListInstancesOptions) (result *InstanceCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListInstancesWithContext(context.Background(), listInstancesOptions)
}

// ListInstancesWithContext is an alternate form of the ListInstances method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListInstancesWithContext(ctx context.Context, listInstancesOptions *ListInstancesOptions) (result *InstanceCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listInstancesOptions, "listInstancesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstancesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListInstances")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listInstancesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstancesOptions.Start))
	}
	if listInstancesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstancesOptions.Limit))
	}
	if listInstancesOptions.NetworkInterfacesSubnetID != nil {
		builder.AddQuery("network_interfaces.subnet.id", fmt.Sprint(*listInstancesOptions.NetworkInterfacesSubnetID))
	}
	if listInstancesOptions.NetworkInterfacesSubnetCRN != nil {
		builder.AddQuery("network_interfaces.subnet.crn", fmt.Sprint(*listInstancesOptions.NetworkInterfacesSubnetCRN))
	}
	if listInstancesOptions.NetworkInterfacesSubnetName != nil {
		builder.AddQuery("network_interfaces.subnet.name", fmt.Sprint(*listInstancesOptions.NetworkInterfacesSubnetName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateInstance : Create an instance
// This request provisions a new instance from an instance prototype object. The prototype object is structured in the
// same way as a retrieved instance, and contains the information necessary to provision the new instance. The instance
// is automatically started.
func (vpcClassic *VpcClassicV1) CreateInstance(createInstanceOptions *CreateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateInstanceWithContext(context.Background(), createInstanceOptions)
}

// CreateInstanceWithContext is an alternate form of the CreateInstance method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateInstanceWithContext(ctx context.Context, createInstanceOptions *CreateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceOptions, "createInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceOptions, "createInstanceOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createInstanceOptions.InstancePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstance)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteInstance : Delete an instance
// This request deletes an instance. This operation cannot be reversed. Any floating IPs associated with the instance's
// network interfaces are implicitly disassociated. All flow log collectors with `auto_delete` set to `true` targeting
// the instance and/or the instance's network interfaces are automatically deleted.
func (vpcClassic *VpcClassicV1) DeleteInstance(deleteInstanceOptions *DeleteInstanceOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteInstanceWithContext(context.Background(), deleteInstanceOptions)
}

// DeleteInstanceWithContext is an alternate form of the DeleteInstance method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteInstanceWithContext(ctx context.Context, deleteInstanceOptions *DeleteInstanceOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceOptions, "deleteInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceOptions, "deleteInstanceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteInstanceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetInstance : Retrieve an instance
// This request retrieves a single instance specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetInstance(getInstanceOptions *GetInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	return vpcClassic.GetInstanceWithContext(context.Background(), getInstanceOptions)
}

// GetInstanceWithContext is an alternate form of the GetInstance method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetInstanceWithContext(ctx context.Context, getInstanceOptions *GetInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceOptions, "getInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceOptions, "getInstanceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getInstanceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstance)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateInstance : Update an instance
// This request updates an instance with the information in a provided instance patch. The instance patch object is
// structured in the same way as a retrieved instance and contains only the information to be updated.
func (vpcClassic *VpcClassicV1) UpdateInstance(updateInstanceOptions *UpdateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateInstanceWithContext(context.Background(), updateInstanceOptions)
}

// UpdateInstanceWithContext is an alternate form of the UpdateInstance method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateInstanceWithContext(ctx context.Context, updateInstanceOptions *UpdateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceOptions, "updateInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceOptions, "updateInstanceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateInstanceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceOptions.InstancePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstance)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetInstanceInitialization : Retrieve initialization configuration for an instance
// This request retrieves configuration variables used to initialize the instance, such as SSH keys and the Windows
// administrator password.
func (vpcClassic *VpcClassicV1) GetInstanceInitialization(getInstanceInitializationOptions *GetInstanceInitializationOptions) (result *InstanceInitialization, response *core.DetailedResponse, err error) {
	return vpcClassic.GetInstanceInitializationWithContext(context.Background(), getInstanceInitializationOptions)
}

// GetInstanceInitializationWithContext is an alternate form of the GetInstanceInitialization method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetInstanceInitializationWithContext(ctx context.Context, getInstanceInitializationOptions *GetInstanceInitializationOptions) (result *InstanceInitialization, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceInitializationOptions, "getInstanceInitializationOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceInitializationOptions, "getInstanceInitializationOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getInstanceInitializationOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{id}/initialization`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceInitializationOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetInstanceInitialization")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceInitialization)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateInstanceAction : Create an instance action
// This request creates a new action which will be queued up to run as soon as any pending or running actions have
// completed.
func (vpcClassic *VpcClassicV1) CreateInstanceAction(createInstanceActionOptions *CreateInstanceActionOptions) (result *InstanceAction, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateInstanceActionWithContext(context.Background(), createInstanceActionOptions)
}

// CreateInstanceActionWithContext is an alternate form of the CreateInstanceAction method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateInstanceActionWithContext(ctx context.Context, createInstanceActionOptions *CreateInstanceActionOptions) (result *InstanceAction, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceActionOptions, "createInstanceActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceActionOptions, "createInstanceActionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *createInstanceActionOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/actions`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateInstanceAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createInstanceActionOptions.Type != nil {
		body["type"] = createInstanceActionOptions.Type
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceAction)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListInstanceNetworkInterfaces : List all network interfaces on an instance
// This request lists all network interfaces on an instance. A network interface is an abstract representation of a
// network interface card and connects an instance to a subnet. While each network interface can attach to only one
// subnet, multiple network interfaces can be created to attach to multiple subnets. Multiple interfaces may also attach
// to the same subnet.
func (vpcClassic *VpcClassicV1) ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptions *ListInstanceNetworkInterfacesOptions) (result *NetworkInterfaceUnpaginatedCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListInstanceNetworkInterfacesWithContext(context.Background(), listInstanceNetworkInterfacesOptions)
}

// ListInstanceNetworkInterfacesWithContext is an alternate form of the ListInstanceNetworkInterfaces method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListInstanceNetworkInterfacesWithContext(ctx context.Context, listInstanceNetworkInterfacesOptions *ListInstanceNetworkInterfacesOptions) (result *NetworkInterfaceUnpaginatedCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceNetworkInterfacesOptions, "listInstanceNetworkInterfacesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceNetworkInterfacesOptions, "listInstanceNetworkInterfacesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *listInstanceNetworkInterfacesOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/network_interfaces`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceNetworkInterfacesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListInstanceNetworkInterfaces")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterfaceUnpaginatedCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetInstanceNetworkInterface : Retrieve a network interface
// This request retrieves a single network interface specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptions *GetInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpcClassic.GetInstanceNetworkInterfaceWithContext(context.Background(), getInstanceNetworkInterfaceOptions)
}

// GetInstanceNetworkInterfaceWithContext is an alternate form of the GetInstanceNetworkInterface method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetInstanceNetworkInterfaceWithContext(ctx context.Context, getInstanceNetworkInterfaceOptions *GetInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceNetworkInterfaceOptions, "getInstanceNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceNetworkInterfaceOptions, "getInstanceNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *getInstanceNetworkInterfaceOptions.InstanceID,
		"id":          *getInstanceNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetInstanceNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListInstanceNetworkInterfaceFloatingIps : List all floating IPs associated with a network interface
// This request lists all floating IPs associated with a network interface.
func (vpcClassic *VpcClassicV1) ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptions *ListInstanceNetworkInterfaceFloatingIpsOptions) (result *FloatingIPUnpaginatedCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListInstanceNetworkInterfaceFloatingIpsWithContext(context.Background(), listInstanceNetworkInterfaceFloatingIpsOptions)
}

// ListInstanceNetworkInterfaceFloatingIpsWithContext is an alternate form of the ListInstanceNetworkInterfaceFloatingIps method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListInstanceNetworkInterfaceFloatingIpsWithContext(ctx context.Context, listInstanceNetworkInterfaceFloatingIpsOptions *ListInstanceNetworkInterfaceFloatingIpsOptions) (result *FloatingIPUnpaginatedCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceNetworkInterfaceFloatingIpsOptions, "listInstanceNetworkInterfaceFloatingIpsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceNetworkInterfaceFloatingIpsOptions, "listInstanceNetworkInterfaceFloatingIpsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *listInstanceNetworkInterfaceFloatingIpsOptions.InstanceID,
		"network_interface_id": *listInstanceNetworkInterfaceFloatingIpsOptions.NetworkInterfaceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceNetworkInterfaceFloatingIpsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListInstanceNetworkInterfaceFloatingIps")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIPUnpaginatedCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// RemoveInstanceNetworkInterfaceFloatingIP : Disassociate a floating IP from a network interface
// This request disassociates the specified floating IP from the specified network interface.
func (vpcClassic *VpcClassicV1) RemoveInstanceNetworkInterfaceFloatingIP(removeInstanceNetworkInterfaceFloatingIPOptions *RemoveInstanceNetworkInterfaceFloatingIPOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.RemoveInstanceNetworkInterfaceFloatingIPWithContext(context.Background(), removeInstanceNetworkInterfaceFloatingIPOptions)
}

// RemoveInstanceNetworkInterfaceFloatingIPWithContext is an alternate form of the RemoveInstanceNetworkInterfaceFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) RemoveInstanceNetworkInterfaceFloatingIPWithContext(ctx context.Context, removeInstanceNetworkInterfaceFloatingIPOptions *RemoveInstanceNetworkInterfaceFloatingIPOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeInstanceNetworkInterfaceFloatingIPOptions, "removeInstanceNetworkInterfaceFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeInstanceNetworkInterfaceFloatingIPOptions, "removeInstanceNetworkInterfaceFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *removeInstanceNetworkInterfaceFloatingIPOptions.InstanceID,
		"network_interface_id": *removeInstanceNetworkInterfaceFloatingIPOptions.NetworkInterfaceID,
		"id":                   *removeInstanceNetworkInterfaceFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeInstanceNetworkInterfaceFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "RemoveInstanceNetworkInterfaceFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetInstanceNetworkInterfaceFloatingIP : Retrieve associated floating IP
// This request a retrieves a specified floating IP address if it is associated with the network interface and instance
// specified in the URL.
func (vpcClassic *VpcClassicV1) GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptions *GetInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpcClassic.GetInstanceNetworkInterfaceFloatingIPWithContext(context.Background(), getInstanceNetworkInterfaceFloatingIPOptions)
}

// GetInstanceNetworkInterfaceFloatingIPWithContext is an alternate form of the GetInstanceNetworkInterfaceFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetInstanceNetworkInterfaceFloatingIPWithContext(ctx context.Context, getInstanceNetworkInterfaceFloatingIPOptions *GetInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceNetworkInterfaceFloatingIPOptions, "getInstanceNetworkInterfaceFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceNetworkInterfaceFloatingIPOptions, "getInstanceNetworkInterfaceFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *getInstanceNetworkInterfaceFloatingIPOptions.InstanceID,
		"network_interface_id": *getInstanceNetworkInterfaceFloatingIPOptions.NetworkInterfaceID,
		"id":                   *getInstanceNetworkInterfaceFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceNetworkInterfaceFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetInstanceNetworkInterfaceFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// AddInstanceNetworkInterfaceFloatingIP : Associate a floating IP with a network interface
// This request associates the specified floating IP with the specified network interface, replacing any existing
// association. For this request to succeed, the existing floating IP must not be required by another resource, such as
// a public gateway. A request body is not required, and if supplied, is ignored.
func (vpcClassic *VpcClassicV1) AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptions *AddInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpcClassic.AddInstanceNetworkInterfaceFloatingIPWithContext(context.Background(), addInstanceNetworkInterfaceFloatingIPOptions)
}

// AddInstanceNetworkInterfaceFloatingIPWithContext is an alternate form of the AddInstanceNetworkInterfaceFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) AddInstanceNetworkInterfaceFloatingIPWithContext(ctx context.Context, addInstanceNetworkInterfaceFloatingIPOptions *AddInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addInstanceNetworkInterfaceFloatingIPOptions, "addInstanceNetworkInterfaceFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addInstanceNetworkInterfaceFloatingIPOptions, "addInstanceNetworkInterfaceFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *addInstanceNetworkInterfaceFloatingIPOptions.InstanceID,
		"network_interface_id": *addInstanceNetworkInterfaceFloatingIPOptions.NetworkInterfaceID,
		"id":                   *addInstanceNetworkInterfaceFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addInstanceNetworkInterfaceFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "AddInstanceNetworkInterfaceFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListInstanceVolumeAttachments : List all volumes attachments on an instance
// This request lists all volume attachments on an instance. A volume attachment connects a volume to an instance. Each
// instance may have many volume attachments but each volume attachment connects exactly one instance to exactly one
// volume.
func (vpcClassic *VpcClassicV1) ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptions *ListInstanceVolumeAttachmentsOptions) (result *VolumeAttachmentCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListInstanceVolumeAttachmentsWithContext(context.Background(), listInstanceVolumeAttachmentsOptions)
}

// ListInstanceVolumeAttachmentsWithContext is an alternate form of the ListInstanceVolumeAttachments method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListInstanceVolumeAttachmentsWithContext(ctx context.Context, listInstanceVolumeAttachmentsOptions *ListInstanceVolumeAttachmentsOptions) (result *VolumeAttachmentCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceVolumeAttachmentsOptions, "listInstanceVolumeAttachmentsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceVolumeAttachmentsOptions, "listInstanceVolumeAttachmentsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *listInstanceVolumeAttachmentsOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/volume_attachments`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceVolumeAttachmentsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListInstanceVolumeAttachments")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachmentCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateInstanceVolumeAttachment : Create a volume attachment on an instance
// This request creates a new volume attachment from a volume attachment prototype object. The prototype object is
// structured in the same way as a retrieved volume attachment, and contains the information necessary to create the new
// volume attachment. The creation of a new volume attachment connects a volume to an instance.
func (vpcClassic *VpcClassicV1) CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptions *CreateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateInstanceVolumeAttachmentWithContext(context.Background(), createInstanceVolumeAttachmentOptions)
}

// CreateInstanceVolumeAttachmentWithContext is an alternate form of the CreateInstanceVolumeAttachment method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateInstanceVolumeAttachmentWithContext(ctx context.Context, createInstanceVolumeAttachmentOptions *CreateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceVolumeAttachmentOptions, "createInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceVolumeAttachmentOptions, "createInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *createInstanceVolumeAttachmentOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/volume_attachments`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createInstanceVolumeAttachmentOptions.Volume != nil {
		body["volume"] = createInstanceVolumeAttachmentOptions.Volume
	}
	if createInstanceVolumeAttachmentOptions.DeleteVolumeOnInstanceDelete != nil {
		body["delete_volume_on_instance_delete"] = createInstanceVolumeAttachmentOptions.DeleteVolumeOnInstanceDelete
	}
	if createInstanceVolumeAttachmentOptions.Name != nil {
		body["name"] = createInstanceVolumeAttachmentOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachment)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteInstanceVolumeAttachment : Delete a volume attachment
// This request deletes a volume attachment. The deletion of a volume attachment detaches a volume from an instance.
func (vpcClassic *VpcClassicV1) DeleteInstanceVolumeAttachment(deleteInstanceVolumeAttachmentOptions *DeleteInstanceVolumeAttachmentOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteInstanceVolumeAttachmentWithContext(context.Background(), deleteInstanceVolumeAttachmentOptions)
}

// DeleteInstanceVolumeAttachmentWithContext is an alternate form of the DeleteInstanceVolumeAttachment method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteInstanceVolumeAttachmentWithContext(ctx context.Context, deleteInstanceVolumeAttachmentOptions *DeleteInstanceVolumeAttachmentOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceVolumeAttachmentOptions, "deleteInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceVolumeAttachmentOptions, "deleteInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *deleteInstanceVolumeAttachmentOptions.InstanceID,
		"id":          *deleteInstanceVolumeAttachmentOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/volume_attachments/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetInstanceVolumeAttachment : Retrieve a volume attachment
// This request retrieves a single volume attachment specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptions *GetInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	return vpcClassic.GetInstanceVolumeAttachmentWithContext(context.Background(), getInstanceVolumeAttachmentOptions)
}

// GetInstanceVolumeAttachmentWithContext is an alternate form of the GetInstanceVolumeAttachment method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetInstanceVolumeAttachmentWithContext(ctx context.Context, getInstanceVolumeAttachmentOptions *GetInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceVolumeAttachmentOptions, "getInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceVolumeAttachmentOptions, "getInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *getInstanceVolumeAttachmentOptions.InstanceID,
		"id":          *getInstanceVolumeAttachmentOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/volume_attachments/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachment)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateInstanceVolumeAttachment : Update a volume attachment
// This request updates a volume attachment with the information in a provided volume attachment patch. The volume
// attachment patch object is structured in the same way as a retrieved volume attachment and can contain an updated
// name.
func (vpcClassic *VpcClassicV1) UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptions *UpdateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateInstanceVolumeAttachmentWithContext(context.Background(), updateInstanceVolumeAttachmentOptions)
}

// UpdateInstanceVolumeAttachmentWithContext is an alternate form of the UpdateInstanceVolumeAttachment method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateInstanceVolumeAttachmentWithContext(ctx context.Context, updateInstanceVolumeAttachmentOptions *UpdateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceVolumeAttachmentOptions, "updateInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceVolumeAttachmentOptions, "updateInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *updateInstanceVolumeAttachmentOptions.InstanceID,
		"id":          *updateInstanceVolumeAttachmentOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/instances/{instance_id}/volume_attachments/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceVolumeAttachmentOptions.VolumeAttachmentPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachment)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVolumeProfiles : List all volume profiles
// This request lists all volume profiles available in the region. A volume profile specifies the performance
// characteristics and pricing model for a volume.
func (vpcClassic *VpcClassicV1) ListVolumeProfiles(listVolumeProfilesOptions *ListVolumeProfilesOptions) (result *VolumeProfileCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVolumeProfilesWithContext(context.Background(), listVolumeProfilesOptions)
}

// ListVolumeProfilesWithContext is an alternate form of the ListVolumeProfiles method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVolumeProfilesWithContext(ctx context.Context, listVolumeProfilesOptions *ListVolumeProfilesOptions) (result *VolumeProfileCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVolumeProfilesOptions, "listVolumeProfilesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volume/profiles`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVolumeProfilesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVolumeProfiles")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVolumeProfilesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVolumeProfilesOptions.Start))
	}
	if listVolumeProfilesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVolumeProfilesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeProfileCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetVolumeProfile : Retrieve a volume profile
// This request retrieves a single volume profile specified by the name in the URL.
func (vpcClassic *VpcClassicV1) GetVolumeProfile(getVolumeProfileOptions *GetVolumeProfileOptions) (result *VolumeProfile, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVolumeProfileWithContext(context.Background(), getVolumeProfileOptions)
}

// GetVolumeProfileWithContext is an alternate form of the GetVolumeProfile method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVolumeProfileWithContext(ctx context.Context, getVolumeProfileOptions *GetVolumeProfileOptions) (result *VolumeProfile, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVolumeProfileOptions, "getVolumeProfileOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVolumeProfileOptions, "getVolumeProfileOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getVolumeProfileOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volume/profiles/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVolumeProfileOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVolumeProfile")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeProfile)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVolumes : List all volumes
// This request lists all volumes in the region. Volumes are network-connected block storage devices that may be
// attached to one or more instances in the same region.
func (vpcClassic *VpcClassicV1) ListVolumes(listVolumesOptions *ListVolumesOptions) (result *VolumeCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVolumesWithContext(context.Background(), listVolumesOptions)
}

// ListVolumesWithContext is an alternate form of the ListVolumes method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVolumesWithContext(ctx context.Context, listVolumesOptions *ListVolumesOptions) (result *VolumeCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVolumesOptions, "listVolumesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volumes`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVolumesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVolumes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVolumesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVolumesOptions.Start))
	}
	if listVolumesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVolumesOptions.Limit))
	}
	if listVolumesOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listVolumesOptions.Name))
	}
	if listVolumesOptions.ZoneName != nil {
		builder.AddQuery("zone.name", fmt.Sprint(*listVolumesOptions.ZoneName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateVolume : Create a volume
// This request creates a new volume from a volume prototype object. The prototype object is structured in the same way
// as a retrieved volume, and contains the information necessary to create the new volume.
func (vpcClassic *VpcClassicV1) CreateVolume(createVolumeOptions *CreateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateVolumeWithContext(context.Background(), createVolumeOptions)
}

// CreateVolumeWithContext is an alternate form of the CreateVolume method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateVolumeWithContext(ctx context.Context, createVolumeOptions *CreateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVolumeOptions, "createVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVolumeOptions, "createVolumeOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volumes`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createVolumeOptions.VolumePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolume)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteVolume : Delete a volume
// This request deletes a volume. This operation cannot be reversed. For this request to succeed, the volume must not be
// attached to any instances.
func (vpcClassic *VpcClassicV1) DeleteVolume(deleteVolumeOptions *DeleteVolumeOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteVolumeWithContext(context.Background(), deleteVolumeOptions)
}

// DeleteVolumeWithContext is an alternate form of the DeleteVolume method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteVolumeWithContext(ctx context.Context, deleteVolumeOptions *DeleteVolumeOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVolumeOptions, "deleteVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVolumeOptions, "deleteVolumeOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteVolumeOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volumes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetVolume : Retrieve a volume
// This request retrieves a single volume specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetVolume(getVolumeOptions *GetVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVolumeWithContext(context.Background(), getVolumeOptions)
}

// GetVolumeWithContext is an alternate form of the GetVolume method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVolumeWithContext(ctx context.Context, getVolumeOptions *GetVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVolumeOptions, "getVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVolumeOptions, "getVolumeOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVolumeOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volumes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolume)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateVolume : Update a volume
// This request updates a volume with the information in a provided volume patch. The volume patch object is structured
// in the same way as a retrieved volume and contains only the information to be updated.
func (vpcClassic *VpcClassicV1) UpdateVolume(updateVolumeOptions *UpdateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateVolumeWithContext(context.Background(), updateVolumeOptions)
}

// UpdateVolumeWithContext is an alternate form of the UpdateVolume method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateVolumeWithContext(ctx context.Context, updateVolumeOptions *UpdateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVolumeOptions, "updateVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVolumeOptions, "updateVolumeOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateVolumeOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/volumes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateVolumeOptions.VolumePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolume)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListRegions : List all regions
// This request lists all regions. Each region is a separate geographic area that contains multiple isolated zones.
// Resources can be provisioned into one or more zones in a region. Each zone is isolated, but connected to other zones
// in the same region with low-latency and high-bandwidth links. Regions represent the top-level of fault isolation
// available. Resources deployed within a single region also benefit from the low latency afforded by geographic
// proximity.
func (vpcClassic *VpcClassicV1) ListRegions(listRegionsOptions *ListRegionsOptions) (result *RegionCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListRegionsWithContext(context.Background(), listRegionsOptions)
}

// ListRegionsWithContext is an alternate form of the ListRegions method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListRegionsWithContext(ctx context.Context, listRegionsOptions *ListRegionsOptions) (result *RegionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listRegionsOptions, "listRegionsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/regions`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listRegionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListRegions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRegionCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetRegion : Retrieve a region
// This request retrieves a single region specified by the name in the URL.
func (vpcClassic *VpcClassicV1) GetRegion(getRegionOptions *GetRegionOptions) (result *Region, response *core.DetailedResponse, err error) {
	return vpcClassic.GetRegionWithContext(context.Background(), getRegionOptions)
}

// GetRegionWithContext is an alternate form of the GetRegion method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetRegionWithContext(ctx context.Context, getRegionOptions *GetRegionOptions) (result *Region, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getRegionOptions, "getRegionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getRegionOptions, "getRegionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getRegionOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/regions/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getRegionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetRegion")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRegion)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListRegionZones : List all zones in a region
// This request lists all zones in a region. Zones represent logically-isolated data centers with high-bandwidth and
// low-latency interconnects to other zones in the same region. Faults in a zone do not affect other zones.
func (vpcClassic *VpcClassicV1) ListRegionZones(listRegionZonesOptions *ListRegionZonesOptions) (result *ZoneCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListRegionZonesWithContext(context.Background(), listRegionZonesOptions)
}

// ListRegionZonesWithContext is an alternate form of the ListRegionZones method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListRegionZonesWithContext(ctx context.Context, listRegionZonesOptions *ListRegionZonesOptions) (result *ZoneCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listRegionZonesOptions, "listRegionZonesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listRegionZonesOptions, "listRegionZonesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"region_name": *listRegionZonesOptions.RegionName,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/regions/{region_name}/zones`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listRegionZonesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListRegionZones")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalZoneCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetRegionZone : Retrieve a zone
// This request retrieves a single zone specified by the region and zone names in the URL.
func (vpcClassic *VpcClassicV1) GetRegionZone(getRegionZoneOptions *GetRegionZoneOptions) (result *Zone, response *core.DetailedResponse, err error) {
	return vpcClassic.GetRegionZoneWithContext(context.Background(), getRegionZoneOptions)
}

// GetRegionZoneWithContext is an alternate form of the GetRegionZone method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetRegionZoneWithContext(ctx context.Context, getRegionZoneOptions *GetRegionZoneOptions) (result *Zone, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getRegionZoneOptions, "getRegionZoneOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getRegionZoneOptions, "getRegionZoneOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"region_name": *getRegionZoneOptions.RegionName,
		"name":        *getRegionZoneOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/regions/{region_name}/zones/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getRegionZoneOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetRegionZone")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalZone)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListPublicGateways : List all public gateways
// This request lists all public gateways in the region. A public gateway is a virtual network device associated with a
// VPC, which allows access to the Internet. A public gateway resides in a zone and can be connected to subnets in the
// same zone only.
func (vpcClassic *VpcClassicV1) ListPublicGateways(listPublicGatewaysOptions *ListPublicGatewaysOptions) (result *PublicGatewayCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListPublicGatewaysWithContext(context.Background(), listPublicGatewaysOptions)
}

// ListPublicGatewaysWithContext is an alternate form of the ListPublicGateways method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListPublicGatewaysWithContext(ctx context.Context, listPublicGatewaysOptions *ListPublicGatewaysOptions) (result *PublicGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listPublicGatewaysOptions, "listPublicGatewaysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/public_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listPublicGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListPublicGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listPublicGatewaysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listPublicGatewaysOptions.Start))
	}
	if listPublicGatewaysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listPublicGatewaysOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGatewayCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreatePublicGateway : Create a public gateway
// This request creates a new public gateway from a public gateway prototype object. For this to succeed, the VPC must
// not already have a public gateway in the specified zone.
//
// If a floating IP is provided, it must be unbound. If a floating IP is not provided, one will be created and bound to
// the public gateway. Once a public gateway has been created, its floating IP cannot be unbound. A public gateway must
// be explicitly attached to each subnet it will provide connectivity for.
func (vpcClassic *VpcClassicV1) CreatePublicGateway(createPublicGatewayOptions *CreatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpcClassic.CreatePublicGatewayWithContext(context.Background(), createPublicGatewayOptions)
}

// CreatePublicGatewayWithContext is an alternate form of the CreatePublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreatePublicGatewayWithContext(ctx context.Context, createPublicGatewayOptions *CreatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createPublicGatewayOptions, "createPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createPublicGatewayOptions, "createPublicGatewayOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/public_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreatePublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createPublicGatewayOptions.VPC != nil {
		body["vpc"] = createPublicGatewayOptions.VPC
	}
	if createPublicGatewayOptions.Zone != nil {
		body["zone"] = createPublicGatewayOptions.Zone
	}
	if createPublicGatewayOptions.FloatingIP != nil {
		body["floating_ip"] = createPublicGatewayOptions.FloatingIP
	}
	if createPublicGatewayOptions.Name != nil {
		body["name"] = createPublicGatewayOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeletePublicGateway : Delete a public gateway
// This request deletes a public gateway. This operation cannot be reversed. For this request to succeed, the public
// gateway must not be attached to any subnets. The public gateway's floating IP will be automatically unbound. If the
// floating IP was created when the public gateway was created, it will be deleted.
func (vpcClassic *VpcClassicV1) DeletePublicGateway(deletePublicGatewayOptions *DeletePublicGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeletePublicGatewayWithContext(context.Background(), deletePublicGatewayOptions)
}

// DeletePublicGatewayWithContext is an alternate form of the DeletePublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeletePublicGatewayWithContext(ctx context.Context, deletePublicGatewayOptions *DeletePublicGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deletePublicGatewayOptions, "deletePublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deletePublicGatewayOptions, "deletePublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deletePublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/public_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deletePublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeletePublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetPublicGateway : Retrieve a public gateway
// This request retrieves a single public gateway specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetPublicGateway(getPublicGatewayOptions *GetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpcClassic.GetPublicGatewayWithContext(context.Background(), getPublicGatewayOptions)
}

// GetPublicGatewayWithContext is an alternate form of the GetPublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetPublicGatewayWithContext(ctx context.Context, getPublicGatewayOptions *GetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getPublicGatewayOptions, "getPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getPublicGatewayOptions, "getPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/public_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdatePublicGateway : Update a public gateway
// This request updates a public gateway's name.
func (vpcClassic *VpcClassicV1) UpdatePublicGateway(updatePublicGatewayOptions *UpdatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdatePublicGatewayWithContext(context.Background(), updatePublicGatewayOptions)
}

// UpdatePublicGatewayWithContext is an alternate form of the UpdatePublicGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdatePublicGatewayWithContext(ctx context.Context, updatePublicGatewayOptions *UpdatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updatePublicGatewayOptions, "updatePublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updatePublicGatewayOptions, "updatePublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updatePublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/public_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updatePublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdatePublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updatePublicGatewayOptions.PublicGatewayPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListFloatingIps : List all floating IPs
// This request lists all floating IPs in the region. Floating IPs allow inbound and outbound traffic from the Internet
// to an instance.
func (vpcClassic *VpcClassicV1) ListFloatingIps(listFloatingIpsOptions *ListFloatingIpsOptions) (result *FloatingIPCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListFloatingIpsWithContext(context.Background(), listFloatingIpsOptions)
}

// ListFloatingIpsWithContext is an alternate form of the ListFloatingIps method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListFloatingIpsWithContext(ctx context.Context, listFloatingIpsOptions *ListFloatingIpsOptions) (result *FloatingIPCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listFloatingIpsOptions, "listFloatingIpsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/floating_ips`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listFloatingIpsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListFloatingIps")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listFloatingIpsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listFloatingIpsOptions.Start))
	}
	if listFloatingIpsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listFloatingIpsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIPCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateFloatingIP : Reserve a floating IP
// This request reserves a new floating IP.
func (vpcClassic *VpcClassicV1) CreateFloatingIP(createFloatingIPOptions *CreateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateFloatingIPWithContext(context.Background(), createFloatingIPOptions)
}

// CreateFloatingIPWithContext is an alternate form of the CreateFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateFloatingIPWithContext(ctx context.Context, createFloatingIPOptions *CreateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createFloatingIPOptions, "createFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createFloatingIPOptions, "createFloatingIPOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/floating_ips`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createFloatingIPOptions.FloatingIPPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteFloatingIP : Release a floating IP
// This request disassociates (if associated) and releases a floating IP. This operation cannot be reversed. For this
// request to succeed, the floating IP must not be required by another resource, such as a public gateway.
func (vpcClassic *VpcClassicV1) DeleteFloatingIP(deleteFloatingIPOptions *DeleteFloatingIPOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteFloatingIPWithContext(context.Background(), deleteFloatingIPOptions)
}

// DeleteFloatingIPWithContext is an alternate form of the DeleteFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteFloatingIPWithContext(ctx context.Context, deleteFloatingIPOptions *DeleteFloatingIPOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteFloatingIPOptions, "deleteFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteFloatingIPOptions, "deleteFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetFloatingIP : Retrieve a floating IP
// This request retrieves a single floating IP specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetFloatingIP(getFloatingIPOptions *GetFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpcClassic.GetFloatingIPWithContext(context.Background(), getFloatingIPOptions)
}

// GetFloatingIPWithContext is an alternate form of the GetFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetFloatingIPWithContext(ctx context.Context, getFloatingIPOptions *GetFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getFloatingIPOptions, "getFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getFloatingIPOptions, "getFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateFloatingIP : Update a floating IP
// This request updates a floating IP's name and/or target.
func (vpcClassic *VpcClassicV1) UpdateFloatingIP(updateFloatingIPOptions *UpdateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateFloatingIPWithContext(context.Background(), updateFloatingIPOptions)
}

// UpdateFloatingIPWithContext is an alternate form of the UpdateFloatingIP method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateFloatingIPWithContext(ctx context.Context, updateFloatingIPOptions *UpdateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateFloatingIPOptions, "updateFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateFloatingIPOptions, "updateFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateFloatingIPOptions.FloatingIPPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListNetworkAcls : List all network ACLs
// This request lists all network ACLs in the region. A network ACL defines a set of packet filtering (5-tuple) rules
// for all traffic in and out of a subnet. Both allow and deny rules can be defined, and rules are stateless such that
// reverse traffic in response to allowed traffic is not automatically permitted.
func (vpcClassic *VpcClassicV1) ListNetworkAcls(listNetworkAclsOptions *ListNetworkAclsOptions) (result *NetworkACLCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListNetworkAclsWithContext(context.Background(), listNetworkAclsOptions)
}

// ListNetworkAclsWithContext is an alternate form of the ListNetworkAcls method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListNetworkAclsWithContext(ctx context.Context, listNetworkAclsOptions *ListNetworkAclsOptions) (result *NetworkACLCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listNetworkAclsOptions, "listNetworkAclsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listNetworkAclsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListNetworkAcls")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listNetworkAclsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listNetworkAclsOptions.Start))
	}
	if listNetworkAclsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listNetworkAclsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateNetworkACL : Create a network ACL
// This request creates a new network ACL from a network ACL prototype object. The prototype object is structured in the
// same way as a retrieved network ACL, and contains the information necessary to create the new network ACL.
func (vpcClassic *VpcClassicV1) CreateNetworkACL(createNetworkACLOptions *CreateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateNetworkACLWithContext(context.Background(), createNetworkACLOptions)
}

// CreateNetworkACLWithContext is an alternate form of the CreateNetworkACL method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateNetworkACLWithContext(ctx context.Context, createNetworkACLOptions *CreateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(createNetworkACLOptions, "createNetworkACLOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	if createNetworkACLOptions.NetworkACLPrototype != nil {
		_, err = builder.SetBodyContentJSON(createNetworkACLOptions.NetworkACLPrototype)
		if err != nil {
			return
		}
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteNetworkACL : Delete a network ACL
// This request deletes a network ACL. This operation cannot be reversed. For this request to succeed, the network ACL
// must not be the default network ACL for any VPCs, and the network ACL must not be attached to any subnets.
func (vpcClassic *VpcClassicV1) DeleteNetworkACL(deleteNetworkACLOptions *DeleteNetworkACLOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteNetworkACLWithContext(context.Background(), deleteNetworkACLOptions)
}

// DeleteNetworkACLWithContext is an alternate form of the DeleteNetworkACL method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteNetworkACLWithContext(ctx context.Context, deleteNetworkACLOptions *DeleteNetworkACLOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteNetworkACLOptions, "deleteNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteNetworkACLOptions, "deleteNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetNetworkACL : Retrieve a network ACL
// This request retrieves a single network ACL specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetNetworkACL(getNetworkACLOptions *GetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpcClassic.GetNetworkACLWithContext(context.Background(), getNetworkACLOptions)
}

// GetNetworkACLWithContext is an alternate form of the GetNetworkACL method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetNetworkACLWithContext(ctx context.Context, getNetworkACLOptions *GetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getNetworkACLOptions, "getNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getNetworkACLOptions, "getNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateNetworkACL : Update a network ACL
// This request updates a network ACL's name.
func (vpcClassic *VpcClassicV1) UpdateNetworkACL(updateNetworkACLOptions *UpdateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateNetworkACLWithContext(context.Background(), updateNetworkACLOptions)
}

// UpdateNetworkACLWithContext is an alternate form of the UpdateNetworkACL method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateNetworkACLWithContext(ctx context.Context, updateNetworkACLOptions *UpdateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateNetworkACLOptions, "updateNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateNetworkACLOptions, "updateNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateNetworkACLOptions.NetworkACLPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListNetworkACLRules : List all rules for a network ACL
// This request lists all rules for a network ACL. These rules can allow or deny traffic between a source CIDR block and
// a destination CIDR block over a particular protocol and port range.
func (vpcClassic *VpcClassicV1) ListNetworkACLRules(listNetworkACLRulesOptions *ListNetworkACLRulesOptions) (result *NetworkACLRuleCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListNetworkACLRulesWithContext(context.Background(), listNetworkACLRulesOptions)
}

// ListNetworkACLRulesWithContext is an alternate form of the ListNetworkACLRules method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListNetworkACLRulesWithContext(ctx context.Context, listNetworkACLRulesOptions *ListNetworkACLRulesOptions) (result *NetworkACLRuleCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listNetworkACLRulesOptions, "listNetworkACLRulesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listNetworkACLRulesOptions, "listNetworkACLRulesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *listNetworkACLRulesOptions.NetworkACLID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{network_acl_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listNetworkACLRulesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListNetworkACLRules")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listNetworkACLRulesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listNetworkACLRulesOptions.Start))
	}
	if listNetworkACLRulesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listNetworkACLRulesOptions.Limit))
	}
	if listNetworkACLRulesOptions.Direction != nil {
		builder.AddQuery("direction", fmt.Sprint(*listNetworkACLRulesOptions.Direction))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRuleCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateNetworkACLRule : Create a rule for a network ACL
// This request creates a new rule from a network ACL rule prototype object. The prototype object is structured in the
// same way as a retrieved rule, and contains the information necessary to create the new rule.
func (vpcClassic *VpcClassicV1) CreateNetworkACLRule(createNetworkACLRuleOptions *CreateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateNetworkACLRuleWithContext(context.Background(), createNetworkACLRuleOptions)
}

// CreateNetworkACLRuleWithContext is an alternate form of the CreateNetworkACLRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateNetworkACLRuleWithContext(ctx context.Context, createNetworkACLRuleOptions *CreateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createNetworkACLRuleOptions, "createNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createNetworkACLRuleOptions, "createNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *createNetworkACLRuleOptions.NetworkACLID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{network_acl_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createNetworkACLRuleOptions.NetworkACLRulePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteNetworkACLRule : Delete a network ACL rule
// This request deletes a rule. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteNetworkACLRule(deleteNetworkACLRuleOptions *DeleteNetworkACLRuleOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteNetworkACLRuleWithContext(context.Background(), deleteNetworkACLRuleOptions)
}

// DeleteNetworkACLRuleWithContext is an alternate form of the DeleteNetworkACLRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteNetworkACLRuleWithContext(ctx context.Context, deleteNetworkACLRuleOptions *DeleteNetworkACLRuleOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteNetworkACLRuleOptions, "deleteNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteNetworkACLRuleOptions, "deleteNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *deleteNetworkACLRuleOptions.NetworkACLID,
		"id":             *deleteNetworkACLRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{network_acl_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetNetworkACLRule : Retrieve a network ACL rule
// This request retrieves a single rule specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetNetworkACLRule(getNetworkACLRuleOptions *GetNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.GetNetworkACLRuleWithContext(context.Background(), getNetworkACLRuleOptions)
}

// GetNetworkACLRuleWithContext is an alternate form of the GetNetworkACLRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetNetworkACLRuleWithContext(ctx context.Context, getNetworkACLRuleOptions *GetNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getNetworkACLRuleOptions, "getNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getNetworkACLRuleOptions, "getNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *getNetworkACLRuleOptions.NetworkACLID,
		"id":             *getNetworkACLRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{network_acl_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateNetworkACLRule : Update a network ACL rule
// This request updates a rule with the information in a provided rule patch. The rule patch object is structured in the
// same way as a retrieved rule and contains only the information to be updated.
func (vpcClassic *VpcClassicV1) UpdateNetworkACLRule(updateNetworkACLRuleOptions *UpdateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateNetworkACLRuleWithContext(context.Background(), updateNetworkACLRuleOptions)
}

// UpdateNetworkACLRuleWithContext is an alternate form of the UpdateNetworkACLRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateNetworkACLRuleWithContext(ctx context.Context, updateNetworkACLRuleOptions *UpdateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateNetworkACLRuleOptions, "updateNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateNetworkACLRuleOptions, "updateNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *updateNetworkACLRuleOptions.NetworkACLID,
		"id":             *updateNetworkACLRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/network_acls/{network_acl_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateNetworkACLRuleOptions.NetworkACLRulePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListSecurityGroups : List all security groups
// This request lists all security groups in the region. Security groups provide a way to apply IP filtering rules to
// instances in the associated VPC. With security groups, all traffic is denied by default, and rules added to security
// groups define which traffic the security group permits. Security group rules are stateful such that reverse traffic
// in response to allowed traffic is automatically permitted.
func (vpcClassic *VpcClassicV1) ListSecurityGroups(listSecurityGroupsOptions *ListSecurityGroupsOptions) (result *SecurityGroupCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListSecurityGroupsWithContext(context.Background(), listSecurityGroupsOptions)
}

// ListSecurityGroupsWithContext is an alternate form of the ListSecurityGroups method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListSecurityGroupsWithContext(ctx context.Context, listSecurityGroupsOptions *ListSecurityGroupsOptions) (result *SecurityGroupCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listSecurityGroupsOptions, "listSecurityGroupsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListSecurityGroups")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listSecurityGroupsOptions.VPCID != nil {
		builder.AddQuery("vpc.id", fmt.Sprint(*listSecurityGroupsOptions.VPCID))
	}
	if listSecurityGroupsOptions.VPCCRN != nil {
		builder.AddQuery("vpc.crn", fmt.Sprint(*listSecurityGroupsOptions.VPCCRN))
	}
	if listSecurityGroupsOptions.VPCName != nil {
		builder.AddQuery("vpc.name", fmt.Sprint(*listSecurityGroupsOptions.VPCName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateSecurityGroup : Create a security group
// This request creates a new security group from a security group prototype object. The prototype object is structured
// in the same way as a retrieved security group, and contains the information necessary to create the new security
// group. If security group rules are included in the prototype object, those rules will be added to the security group.
// Each security group is scoped to one VPC. Only network interfaces on instances in that VPC can be added to the
// security group.
func (vpcClassic *VpcClassicV1) CreateSecurityGroup(createSecurityGroupOptions *CreateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateSecurityGroupWithContext(context.Background(), createSecurityGroupOptions)
}

// CreateSecurityGroupWithContext is an alternate form of the CreateSecurityGroup method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateSecurityGroupWithContext(ctx context.Context, createSecurityGroupOptions *CreateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSecurityGroupOptions, "createSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSecurityGroupOptions, "createSecurityGroupOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createSecurityGroupOptions.VPC != nil {
		body["vpc"] = createSecurityGroupOptions.VPC
	}
	if createSecurityGroupOptions.Name != nil {
		body["name"] = createSecurityGroupOptions.Name
	}
	if createSecurityGroupOptions.ResourceGroup != nil {
		body["resource_group"] = createSecurityGroupOptions.ResourceGroup
	}
	if createSecurityGroupOptions.Rules != nil {
		body["rules"] = createSecurityGroupOptions.Rules
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroup)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteSecurityGroup : Delete a security group
// This request deletes a security group. A security group cannot be deleted if it is referenced by any network
// interfaces or other security group rules. Additionally, a VPC's default security group cannot be deleted. This
// operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteSecurityGroup(deleteSecurityGroupOptions *DeleteSecurityGroupOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteSecurityGroupWithContext(context.Background(), deleteSecurityGroupOptions)
}

// DeleteSecurityGroupWithContext is an alternate form of the DeleteSecurityGroup method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteSecurityGroupWithContext(ctx context.Context, deleteSecurityGroupOptions *DeleteSecurityGroupOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSecurityGroupOptions, "deleteSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSecurityGroupOptions, "deleteSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetSecurityGroup : Retrieve a security group
// This request retrieves a single security group specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetSecurityGroup(getSecurityGroupOptions *GetSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	return vpcClassic.GetSecurityGroupWithContext(context.Background(), getSecurityGroupOptions)
}

// GetSecurityGroupWithContext is an alternate form of the GetSecurityGroup method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetSecurityGroupWithContext(ctx context.Context, getSecurityGroupOptions *GetSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupOptions, "getSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupOptions, "getSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroup)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateSecurityGroup : Update a security group
// This request updates a security group with the information provided in a security group patch object. The security
// group patch object is structured in the same way as a retrieved security group and contains only the information to
// be updated.
func (vpcClassic *VpcClassicV1) UpdateSecurityGroup(updateSecurityGroupOptions *UpdateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateSecurityGroupWithContext(context.Background(), updateSecurityGroupOptions)
}

// UpdateSecurityGroupWithContext is an alternate form of the UpdateSecurityGroup method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateSecurityGroupWithContext(ctx context.Context, updateSecurityGroupOptions *UpdateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSecurityGroupOptions, "updateSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSecurityGroupOptions, "updateSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateSecurityGroupOptions.SecurityGroupPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroup)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListSecurityGroupNetworkInterfaces : List all network interfaces associated with a security group
// This request lists all network interfaces associated with a security group, to which the rules in the security group
// are applied.
func (vpcClassic *VpcClassicV1) ListSecurityGroupNetworkInterfaces(listSecurityGroupNetworkInterfacesOptions *ListSecurityGroupNetworkInterfacesOptions) (result *NetworkInterfaceCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListSecurityGroupNetworkInterfacesWithContext(context.Background(), listSecurityGroupNetworkInterfacesOptions)
}

// ListSecurityGroupNetworkInterfacesWithContext is an alternate form of the ListSecurityGroupNetworkInterfaces method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListSecurityGroupNetworkInterfacesWithContext(ctx context.Context, listSecurityGroupNetworkInterfacesOptions *ListSecurityGroupNetworkInterfacesOptions) (result *NetworkInterfaceCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listSecurityGroupNetworkInterfacesOptions, "listSecurityGroupNetworkInterfacesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listSecurityGroupNetworkInterfacesOptions, "listSecurityGroupNetworkInterfacesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *listSecurityGroupNetworkInterfacesOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupNetworkInterfacesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListSecurityGroupNetworkInterfaces")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterfaceCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// RemoveSecurityGroupNetworkInterface : Remove a network interface from a security group
// This request removes a network interface from a security group. Security groups are stateful, so any changes to a
// network interface's security groups are applied to new connections. Existing connections are not affected. If the
// network interface being removed has no other security groups, it will be attached to the VPC's default security
// group.
func (vpcClassic *VpcClassicV1) RemoveSecurityGroupNetworkInterface(removeSecurityGroupNetworkInterfaceOptions *RemoveSecurityGroupNetworkInterfaceOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.RemoveSecurityGroupNetworkInterfaceWithContext(context.Background(), removeSecurityGroupNetworkInterfaceOptions)
}

// RemoveSecurityGroupNetworkInterfaceWithContext is an alternate form of the RemoveSecurityGroupNetworkInterface method which supports a Context parameter
func (vpcClassic *VpcClassicV1) RemoveSecurityGroupNetworkInterfaceWithContext(ctx context.Context, removeSecurityGroupNetworkInterfaceOptions *RemoveSecurityGroupNetworkInterfaceOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeSecurityGroupNetworkInterfaceOptions, "removeSecurityGroupNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeSecurityGroupNetworkInterfaceOptions, "removeSecurityGroupNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *removeSecurityGroupNetworkInterfaceOptions.SecurityGroupID,
		"id":                *removeSecurityGroupNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeSecurityGroupNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "RemoveSecurityGroupNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetSecurityGroupNetworkInterface : Retrieve a network interface in a security group
// This request retrieves a single network interface specified by the identifier in the URL path. The network interface
// must be an existing member of the security group.
func (vpcClassic *VpcClassicV1) GetSecurityGroupNetworkInterface(getSecurityGroupNetworkInterfaceOptions *GetSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpcClassic.GetSecurityGroupNetworkInterfaceWithContext(context.Background(), getSecurityGroupNetworkInterfaceOptions)
}

// GetSecurityGroupNetworkInterfaceWithContext is an alternate form of the GetSecurityGroupNetworkInterface method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetSecurityGroupNetworkInterfaceWithContext(ctx context.Context, getSecurityGroupNetworkInterfaceOptions *GetSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupNetworkInterfaceOptions, "getSecurityGroupNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupNetworkInterfaceOptions, "getSecurityGroupNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *getSecurityGroupNetworkInterfaceOptions.SecurityGroupID,
		"id":                *getSecurityGroupNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetSecurityGroupNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// AddSecurityGroupNetworkInterface : Add a network interface to a security group
// This request adds an existing network interface to an existing security group. When a network interface is added to a
// security group, the security group rules are applied to the network interface. A request body is not required, and if
// supplied, is ignored.
func (vpcClassic *VpcClassicV1) AddSecurityGroupNetworkInterface(addSecurityGroupNetworkInterfaceOptions *AddSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpcClassic.AddSecurityGroupNetworkInterfaceWithContext(context.Background(), addSecurityGroupNetworkInterfaceOptions)
}

// AddSecurityGroupNetworkInterfaceWithContext is an alternate form of the AddSecurityGroupNetworkInterface method which supports a Context parameter
func (vpcClassic *VpcClassicV1) AddSecurityGroupNetworkInterfaceWithContext(ctx context.Context, addSecurityGroupNetworkInterfaceOptions *AddSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addSecurityGroupNetworkInterfaceOptions, "addSecurityGroupNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addSecurityGroupNetworkInterfaceOptions, "addSecurityGroupNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *addSecurityGroupNetworkInterfaceOptions.SecurityGroupID,
		"id":                *addSecurityGroupNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addSecurityGroupNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "AddSecurityGroupNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListSecurityGroupRules : List all rules in a security group
// This request lists all rules in a security group. These rules define what traffic the security group permits.
// Security group rules are stateful, such that reverse traffic in response to allowed traffic is automatically
// permitted.
func (vpcClassic *VpcClassicV1) ListSecurityGroupRules(listSecurityGroupRulesOptions *ListSecurityGroupRulesOptions) (result *SecurityGroupRuleCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListSecurityGroupRulesWithContext(context.Background(), listSecurityGroupRulesOptions)
}

// ListSecurityGroupRulesWithContext is an alternate form of the ListSecurityGroupRules method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListSecurityGroupRulesWithContext(ctx context.Context, listSecurityGroupRulesOptions *ListSecurityGroupRulesOptions) (result *SecurityGroupRuleCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listSecurityGroupRulesOptions, "listSecurityGroupRulesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listSecurityGroupRulesOptions, "listSecurityGroupRulesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *listSecurityGroupRulesOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupRulesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListSecurityGroupRules")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRuleCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateSecurityGroupRule : Create a rule for a security group
// This request creates a new security group rule from a security group rule prototype object. The prototype object is
// structured in the same way as a retrieved security group rule and contains the information necessary to create the
// rule. As part of creating a new rule in a security group, the rule is applied to all the networking interfaces in the
// security group. Rules specify which IP traffic a security group should allow. Security group rules are stateful, such
// that reverse traffic in response to allowed traffic is automatically permitted. A rule allowing inbound TCP traffic
// on port 80 also allows outbound TCP traffic on port 80 without the need for an additional rule.
func (vpcClassic *VpcClassicV1) CreateSecurityGroupRule(createSecurityGroupRuleOptions *CreateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateSecurityGroupRuleWithContext(context.Background(), createSecurityGroupRuleOptions)
}

// CreateSecurityGroupRuleWithContext is an alternate form of the CreateSecurityGroupRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateSecurityGroupRuleWithContext(ctx context.Context, createSecurityGroupRuleOptions *CreateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSecurityGroupRuleOptions, "createSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSecurityGroupRuleOptions, "createSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *createSecurityGroupRuleOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createSecurityGroupRuleOptions.SecurityGroupRulePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteSecurityGroupRule : Delete a security group rule
// This request deletes a security group rule. This operation cannot be reversed. Removing a security group rule will
// not end existing connections allowed by that rule.
func (vpcClassic *VpcClassicV1) DeleteSecurityGroupRule(deleteSecurityGroupRuleOptions *DeleteSecurityGroupRuleOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteSecurityGroupRuleWithContext(context.Background(), deleteSecurityGroupRuleOptions)
}

// DeleteSecurityGroupRuleWithContext is an alternate form of the DeleteSecurityGroupRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteSecurityGroupRuleWithContext(ctx context.Context, deleteSecurityGroupRuleOptions *DeleteSecurityGroupRuleOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSecurityGroupRuleOptions, "deleteSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSecurityGroupRuleOptions, "deleteSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *deleteSecurityGroupRuleOptions.SecurityGroupID,
		"id":                *deleteSecurityGroupRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetSecurityGroupRule : Retrieve a security group rule
// This request retrieves a single security group rule specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetSecurityGroupRule(getSecurityGroupRuleOptions *GetSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.GetSecurityGroupRuleWithContext(context.Background(), getSecurityGroupRuleOptions)
}

// GetSecurityGroupRuleWithContext is an alternate form of the GetSecurityGroupRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetSecurityGroupRuleWithContext(ctx context.Context, getSecurityGroupRuleOptions *GetSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupRuleOptions, "getSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupRuleOptions, "getSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *getSecurityGroupRuleOptions.SecurityGroupID,
		"id":                *getSecurityGroupRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateSecurityGroupRule : Update a security group rule
// This request updates a security group rule with the information provided in a rule patch object. The patch object is
// structured in the same way as a retrieved security group rule and needs to contain only the information to be
// updated.
func (vpcClassic *VpcClassicV1) UpdateSecurityGroupRule(updateSecurityGroupRuleOptions *UpdateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateSecurityGroupRuleWithContext(context.Background(), updateSecurityGroupRuleOptions)
}

// UpdateSecurityGroupRuleWithContext is an alternate form of the UpdateSecurityGroupRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateSecurityGroupRuleWithContext(ctx context.Context, updateSecurityGroupRuleOptions *UpdateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSecurityGroupRuleOptions, "updateSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSecurityGroupRuleOptions, "updateSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *updateSecurityGroupRuleOptions.SecurityGroupID,
		"id":                *updateSecurityGroupRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/security_groups/{security_group_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateSecurityGroupRuleOptions.SecurityGroupRulePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListIkePolicies : List all IKE policies
// This request lists all IKE policies in the region.
func (vpcClassic *VpcClassicV1) ListIkePolicies(listIkePoliciesOptions *ListIkePoliciesOptions) (result *IkePolicyCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListIkePoliciesWithContext(context.Background(), listIkePoliciesOptions)
}

// ListIkePoliciesWithContext is an alternate form of the ListIkePolicies method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListIkePoliciesWithContext(ctx context.Context, listIkePoliciesOptions *ListIkePoliciesOptions) (result *IkePolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listIkePoliciesOptions, "listIkePoliciesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ike_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIkePoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListIkePolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listIkePoliciesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listIkePoliciesOptions.Start))
	}
	if listIkePoliciesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listIkePoliciesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicyCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateIkePolicy : Create an IKE policy
// This request creates a new IKE policy.
func (vpcClassic *VpcClassicV1) CreateIkePolicy(createIkePolicyOptions *CreateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateIkePolicyWithContext(context.Background(), createIkePolicyOptions)
}

// CreateIkePolicyWithContext is an alternate form of the CreateIkePolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateIkePolicyWithContext(ctx context.Context, createIkePolicyOptions *CreateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createIkePolicyOptions, "createIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createIkePolicyOptions, "createIkePolicyOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ike_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createIkePolicyOptions.AuthenticationAlgorithm != nil {
		body["authentication_algorithm"] = createIkePolicyOptions.AuthenticationAlgorithm
	}
	if createIkePolicyOptions.DhGroup != nil {
		body["dh_group"] = createIkePolicyOptions.DhGroup
	}
	if createIkePolicyOptions.EncryptionAlgorithm != nil {
		body["encryption_algorithm"] = createIkePolicyOptions.EncryptionAlgorithm
	}
	if createIkePolicyOptions.IkeVersion != nil {
		body["ike_version"] = createIkePolicyOptions.IkeVersion
	}
	if createIkePolicyOptions.KeyLifetime != nil {
		body["key_lifetime"] = createIkePolicyOptions.KeyLifetime
	}
	if createIkePolicyOptions.Name != nil {
		body["name"] = createIkePolicyOptions.Name
	}
	if createIkePolicyOptions.ResourceGroup != nil {
		body["resource_group"] = createIkePolicyOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteIkePolicy : Delete an IKE policy
// This request deletes an IKE policy. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteIkePolicy(deleteIkePolicyOptions *DeleteIkePolicyOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteIkePolicyWithContext(context.Background(), deleteIkePolicyOptions)
}

// DeleteIkePolicyWithContext is an alternate form of the DeleteIkePolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteIkePolicyWithContext(ctx context.Context, deleteIkePolicyOptions *DeleteIkePolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteIkePolicyOptions, "deleteIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteIkePolicyOptions, "deleteIkePolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteIkePolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ike_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetIkePolicy : Retrieve an IKE policy
// This request retrieves a single IKE policy specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetIkePolicy(getIkePolicyOptions *GetIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.GetIkePolicyWithContext(context.Background(), getIkePolicyOptions)
}

// GetIkePolicyWithContext is an alternate form of the GetIkePolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetIkePolicyWithContext(ctx context.Context, getIkePolicyOptions *GetIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getIkePolicyOptions, "getIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getIkePolicyOptions, "getIkePolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getIkePolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ike_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateIkePolicy : Update an IKE policy
// This request updates the properties of an existing IKE policy.
func (vpcClassic *VpcClassicV1) UpdateIkePolicy(updateIkePolicyOptions *UpdateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateIkePolicyWithContext(context.Background(), updateIkePolicyOptions)
}

// UpdateIkePolicyWithContext is an alternate form of the UpdateIkePolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateIkePolicyWithContext(ctx context.Context, updateIkePolicyOptions *UpdateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateIkePolicyOptions, "updateIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateIkePolicyOptions, "updateIkePolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateIkePolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ike_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateIkePolicyOptions.IkePolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListIkePolicyConnections : List all VPN gateway connections that use a specified IKE policy
// This request lists all VPN gateway connections that use a policy.
func (vpcClassic *VpcClassicV1) ListIkePolicyConnections(listIkePolicyConnectionsOptions *ListIkePolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListIkePolicyConnectionsWithContext(context.Background(), listIkePolicyConnectionsOptions)
}

// ListIkePolicyConnectionsWithContext is an alternate form of the ListIkePolicyConnections method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListIkePolicyConnectionsWithContext(ctx context.Context, listIkePolicyConnectionsOptions *ListIkePolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listIkePolicyConnectionsOptions, "listIkePolicyConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listIkePolicyConnectionsOptions, "listIkePolicyConnectionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *listIkePolicyConnectionsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ike_policies/{id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIkePolicyConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListIkePolicyConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListIpsecPolicies : List all IPsec policies
// This request lists all IPsec policies in the region.
func (vpcClassic *VpcClassicV1) ListIpsecPolicies(listIpsecPoliciesOptions *ListIpsecPoliciesOptions) (result *IPsecPolicyCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListIpsecPoliciesWithContext(context.Background(), listIpsecPoliciesOptions)
}

// ListIpsecPoliciesWithContext is an alternate form of the ListIpsecPolicies method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListIpsecPoliciesWithContext(ctx context.Context, listIpsecPoliciesOptions *ListIpsecPoliciesOptions) (result *IPsecPolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listIpsecPoliciesOptions, "listIpsecPoliciesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ipsec_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIpsecPoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListIpsecPolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listIpsecPoliciesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listIpsecPoliciesOptions.Start))
	}
	if listIpsecPoliciesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listIpsecPoliciesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicyCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateIpsecPolicy : Create an IPsec policy
// This request creates a new IPsec policy.
func (vpcClassic *VpcClassicV1) CreateIpsecPolicy(createIpsecPolicyOptions *CreateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateIpsecPolicyWithContext(context.Background(), createIpsecPolicyOptions)
}

// CreateIpsecPolicyWithContext is an alternate form of the CreateIpsecPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateIpsecPolicyWithContext(ctx context.Context, createIpsecPolicyOptions *CreateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createIpsecPolicyOptions, "createIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createIpsecPolicyOptions, "createIpsecPolicyOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ipsec_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createIpsecPolicyOptions.AuthenticationAlgorithm != nil {
		body["authentication_algorithm"] = createIpsecPolicyOptions.AuthenticationAlgorithm
	}
	if createIpsecPolicyOptions.EncryptionAlgorithm != nil {
		body["encryption_algorithm"] = createIpsecPolicyOptions.EncryptionAlgorithm
	}
	if createIpsecPolicyOptions.Pfs != nil {
		body["pfs"] = createIpsecPolicyOptions.Pfs
	}
	if createIpsecPolicyOptions.KeyLifetime != nil {
		body["key_lifetime"] = createIpsecPolicyOptions.KeyLifetime
	}
	if createIpsecPolicyOptions.Name != nil {
		body["name"] = createIpsecPolicyOptions.Name
	}
	if createIpsecPolicyOptions.ResourceGroup != nil {
		body["resource_group"] = createIpsecPolicyOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteIpsecPolicy : Delete an IPsec policy
// This request deletes an IPsec policy. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteIpsecPolicy(deleteIpsecPolicyOptions *DeleteIpsecPolicyOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteIpsecPolicyWithContext(context.Background(), deleteIpsecPolicyOptions)
}

// DeleteIpsecPolicyWithContext is an alternate form of the DeleteIpsecPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteIpsecPolicyWithContext(ctx context.Context, deleteIpsecPolicyOptions *DeleteIpsecPolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteIpsecPolicyOptions, "deleteIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteIpsecPolicyOptions, "deleteIpsecPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteIpsecPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ipsec_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetIpsecPolicy : Retrieve an IPsec policy
// This request retrieves a single IPsec policy specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetIpsecPolicy(getIpsecPolicyOptions *GetIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.GetIpsecPolicyWithContext(context.Background(), getIpsecPolicyOptions)
}

// GetIpsecPolicyWithContext is an alternate form of the GetIpsecPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetIpsecPolicyWithContext(ctx context.Context, getIpsecPolicyOptions *GetIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getIpsecPolicyOptions, "getIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getIpsecPolicyOptions, "getIpsecPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getIpsecPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ipsec_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateIpsecPolicy : Update an IPsec policy
// This request updates the properties of an existing IPsec policy.
func (vpcClassic *VpcClassicV1) UpdateIpsecPolicy(updateIpsecPolicyOptions *UpdateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateIpsecPolicyWithContext(context.Background(), updateIpsecPolicyOptions)
}

// UpdateIpsecPolicyWithContext is an alternate form of the UpdateIpsecPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateIpsecPolicyWithContext(ctx context.Context, updateIpsecPolicyOptions *UpdateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateIpsecPolicyOptions, "updateIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateIpsecPolicyOptions, "updateIpsecPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateIpsecPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ipsec_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateIpsecPolicyOptions.IPsecPolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListIpsecPolicyConnections : List all VPN gateway connections that use a specified IPsec policy
// This request lists all VPN gateway connections that use a policy.
func (vpcClassic *VpcClassicV1) ListIpsecPolicyConnections(listIpsecPolicyConnectionsOptions *ListIpsecPolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListIpsecPolicyConnectionsWithContext(context.Background(), listIpsecPolicyConnectionsOptions)
}

// ListIpsecPolicyConnectionsWithContext is an alternate form of the ListIpsecPolicyConnections method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListIpsecPolicyConnectionsWithContext(ctx context.Context, listIpsecPolicyConnectionsOptions *ListIpsecPolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listIpsecPolicyConnectionsOptions, "listIpsecPolicyConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listIpsecPolicyConnectionsOptions, "listIpsecPolicyConnectionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *listIpsecPolicyConnectionsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/ipsec_policies/{id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIpsecPolicyConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListIpsecPolicyConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVPNGateways : List all VPN gateways
// This request lists all VPN gateways in the region.
func (vpcClassic *VpcClassicV1) ListVPNGateways(listVPNGatewaysOptions *ListVPNGatewaysOptions) (result *VPNGatewayCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVPNGatewaysWithContext(context.Background(), listVPNGatewaysOptions)
}

// ListVPNGatewaysWithContext is an alternate form of the ListVPNGateways method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVPNGatewaysWithContext(ctx context.Context, listVPNGatewaysOptions *ListVPNGatewaysOptions) (result *VPNGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVPNGatewaysOptions, "listVPNGatewaysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVPNGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVPNGatewaysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPNGatewaysOptions.Start))
	}
	if listVPNGatewaysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPNGatewaysOptions.Limit))
	}
	if listVPNGatewaysOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listVPNGatewaysOptions.ResourceGroupID))
	}
	if listVPNGatewaysOptions.Mode != nil {
		builder.AddQuery("mode", fmt.Sprint(*listVPNGatewaysOptions.Mode))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateVPNGateway : Create a VPN gateway
// This request creates a new VPN gateway.
func (vpcClassic *VpcClassicV1) CreateVPNGateway(createVPNGatewayOptions *CreateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateVPNGatewayWithContext(context.Background(), createVPNGatewayOptions)
}

// CreateVPNGatewayWithContext is an alternate form of the CreateVPNGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateVPNGatewayWithContext(ctx context.Context, createVPNGatewayOptions *CreateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPNGatewayOptions, "createVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPNGatewayOptions, "createVPNGatewayOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createVPNGatewayOptions.VPNGatewayPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteVPNGateway : Delete a VPN gateway
// This request deletes a VPN gateway. A VPN gateway with a `status` of `pending` cannot be deleted. This operation
// deletes all VPN gateway connections associated with this VPN gateway.  This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteVPNGateway(deleteVPNGatewayOptions *DeleteVPNGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteVPNGatewayWithContext(context.Background(), deleteVPNGatewayOptions)
}

// DeleteVPNGatewayWithContext is an alternate form of the DeleteVPNGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteVPNGatewayWithContext(ctx context.Context, deleteVPNGatewayOptions *DeleteVPNGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPNGatewayOptions, "deleteVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPNGatewayOptions, "deleteVPNGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteVPNGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetVPNGateway : Retrieve a VPN gateway
// This request retrieves a single VPN gateway specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetVPNGateway(getVPNGatewayOptions *GetVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVPNGatewayWithContext(context.Background(), getVPNGatewayOptions)
}

// GetVPNGatewayWithContext is an alternate form of the GetVPNGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVPNGatewayWithContext(ctx context.Context, getVPNGatewayOptions *GetVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPNGatewayOptions, "getVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPNGatewayOptions, "getVPNGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPNGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateVPNGateway : Update a VPN gateway
// This request updates the properties of an existing VPN gateway.
func (vpcClassic *VpcClassicV1) UpdateVPNGateway(updateVPNGatewayOptions *UpdateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateVPNGatewayWithContext(context.Background(), updateVPNGatewayOptions)
}

// UpdateVPNGatewayWithContext is an alternate form of the UpdateVPNGateway method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateVPNGatewayWithContext(ctx context.Context, updateVPNGatewayOptions *UpdateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPNGatewayOptions, "updateVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPNGatewayOptions, "updateVPNGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateVPNGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateVPNGatewayOptions.VPNGatewayPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVPNGatewayConnections : List all connections of a VPN gateway
// This request lists all connections of a VPN gateway.
func (vpcClassic *VpcClassicV1) ListVPNGatewayConnections(listVPNGatewayConnectionsOptions *ListVPNGatewayConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVPNGatewayConnectionsWithContext(context.Background(), listVPNGatewayConnectionsOptions)
}

// ListVPNGatewayConnectionsWithContext is an alternate form of the ListVPNGatewayConnections method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVPNGatewayConnectionsWithContext(ctx context.Context, listVPNGatewayConnectionsOptions *ListVPNGatewayConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPNGatewayConnectionsOptions, "listVPNGatewayConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPNGatewayConnectionsOptions, "listVPNGatewayConnectionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *listVPNGatewayConnectionsOptions.VPNGatewayID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewayConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVPNGatewayConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))
	if listVPNGatewayConnectionsOptions.Status != nil {
		builder.AddQuery("status", fmt.Sprint(*listVPNGatewayConnectionsOptions.Status))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateVPNGatewayConnection : Create a connection for a VPN gateway
// This request creates a new VPN gateway connection.
func (vpcClassic *VpcClassicV1) CreateVPNGatewayConnection(createVPNGatewayConnectionOptions *CreateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateVPNGatewayConnectionWithContext(context.Background(), createVPNGatewayConnectionOptions)
}

// CreateVPNGatewayConnectionWithContext is an alternate form of the CreateVPNGatewayConnection method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateVPNGatewayConnectionWithContext(ctx context.Context, createVPNGatewayConnectionOptions *CreateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPNGatewayConnectionOptions, "createVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPNGatewayConnectionOptions, "createVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *createVPNGatewayConnectionOptions.VPNGatewayID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(createVPNGatewayConnectionOptions.VPNGatewayConnectionPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteVPNGatewayConnection : Delete a VPN gateway connection
// This request deletes a VPN gateway connection. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteVPNGatewayConnection(deleteVPNGatewayConnectionOptions *DeleteVPNGatewayConnectionOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteVPNGatewayConnectionWithContext(context.Background(), deleteVPNGatewayConnectionOptions)
}

// DeleteVPNGatewayConnectionWithContext is an alternate form of the DeleteVPNGatewayConnection method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteVPNGatewayConnectionWithContext(ctx context.Context, deleteVPNGatewayConnectionOptions *DeleteVPNGatewayConnectionOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPNGatewayConnectionOptions, "deleteVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPNGatewayConnectionOptions, "deleteVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *deleteVPNGatewayConnectionOptions.VPNGatewayID,
		"id":             *deleteVPNGatewayConnectionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetVPNGatewayConnection : Retrieve a VPN gateway connection
// This request retrieves a single VPN gateway connection specified by the identifier in the URL.
func (vpcClassic *VpcClassicV1) GetVPNGatewayConnection(getVPNGatewayConnectionOptions *GetVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.GetVPNGatewayConnectionWithContext(context.Background(), getVPNGatewayConnectionOptions)
}

// GetVPNGatewayConnectionWithContext is an alternate form of the GetVPNGatewayConnection method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetVPNGatewayConnectionWithContext(ctx context.Context, getVPNGatewayConnectionOptions *GetVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPNGatewayConnectionOptions, "getVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPNGatewayConnectionOptions, "getVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *getVPNGatewayConnectionOptions.VPNGatewayID,
		"id":             *getVPNGatewayConnectionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateVPNGatewayConnection : Update a VPN gateway connection
// This request updates the properties of an existing VPN gateway connection.
func (vpcClassic *VpcClassicV1) UpdateVPNGatewayConnection(updateVPNGatewayConnectionOptions *UpdateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateVPNGatewayConnectionWithContext(context.Background(), updateVPNGatewayConnectionOptions)
}

// UpdateVPNGatewayConnectionWithContext is an alternate form of the UpdateVPNGatewayConnection method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateVPNGatewayConnectionWithContext(ctx context.Context, updateVPNGatewayConnectionOptions *UpdateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPNGatewayConnectionOptions, "updateVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPNGatewayConnectionOptions, "updateVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *updateVPNGatewayConnectionOptions.VPNGatewayID,
		"id":             *updateVPNGatewayConnectionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateVPNGatewayConnectionOptions.VPNGatewayConnectionPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListVPNGatewayConnectionLocalCIDRs : List all local CIDRs for a VPN gateway connection
// This request lists all local CIDRs for a VPN gateway connection.
func (vpcClassic *VpcClassicV1) ListVPNGatewayConnectionLocalCIDRs(listVPNGatewayConnectionLocalCIDRsOptions *ListVPNGatewayConnectionLocalCIDRsOptions) (result *VPNGatewayConnectionLocalCIDRs, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVPNGatewayConnectionLocalCIDRsWithContext(context.Background(), listVPNGatewayConnectionLocalCIDRsOptions)
}

// ListVPNGatewayConnectionLocalCIDRsWithContext is an alternate form of the ListVPNGatewayConnectionLocalCIDRs method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVPNGatewayConnectionLocalCIDRsWithContext(ctx context.Context, listVPNGatewayConnectionLocalCIDRsOptions *ListVPNGatewayConnectionLocalCIDRsOptions) (result *VPNGatewayConnectionLocalCIDRs, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPNGatewayConnectionLocalCIDRsOptions, "listVPNGatewayConnectionLocalCIDRsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPNGatewayConnectionLocalCIDRsOptions, "listVPNGatewayConnectionLocalCIDRsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *listVPNGatewayConnectionLocalCIDRsOptions.VPNGatewayID,
		"id":             *listVPNGatewayConnectionLocalCIDRsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewayConnectionLocalCIDRsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVPNGatewayConnectionLocalCIDRs")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionLocalCIDRs)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// RemoveVPNGatewayConnectionLocalCIDR : Remove a local CIDR from a VPN gateway connection
// This request removes a CIDR from a VPN gateway connection.
func (vpcClassic *VpcClassicV1) RemoveVPNGatewayConnectionLocalCIDR(removeVPNGatewayConnectionLocalCIDROptions *RemoveVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.RemoveVPNGatewayConnectionLocalCIDRWithContext(context.Background(), removeVPNGatewayConnectionLocalCIDROptions)
}

// RemoveVPNGatewayConnectionLocalCIDRWithContext is an alternate form of the RemoveVPNGatewayConnectionLocalCIDR method which supports a Context parameter
func (vpcClassic *VpcClassicV1) RemoveVPNGatewayConnectionLocalCIDRWithContext(ctx context.Context, removeVPNGatewayConnectionLocalCIDROptions *RemoveVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeVPNGatewayConnectionLocalCIDROptions, "removeVPNGatewayConnectionLocalCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeVPNGatewayConnectionLocalCIDROptions, "removeVPNGatewayConnectionLocalCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *removeVPNGatewayConnectionLocalCIDROptions.VPNGatewayID,
		"id":             *removeVPNGatewayConnectionLocalCIDROptions.ID,
		"cidr_prefix":    *removeVPNGatewayConnectionLocalCIDROptions.CIDRPrefix,
		"prefix_length":  *removeVPNGatewayConnectionLocalCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeVPNGatewayConnectionLocalCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "RemoveVPNGatewayConnectionLocalCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// CheckVPNGatewayConnectionLocalCIDR : Check if the specified local CIDR exists on a VPN gateway connection
// This request succeeds if a CIDR exists on the specified VPN gateway connection and fails otherwise.
func (vpcClassic *VpcClassicV1) CheckVPNGatewayConnectionLocalCIDR(checkVPNGatewayConnectionLocalCIDROptions *CheckVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.CheckVPNGatewayConnectionLocalCIDRWithContext(context.Background(), checkVPNGatewayConnectionLocalCIDROptions)
}

// CheckVPNGatewayConnectionLocalCIDRWithContext is an alternate form of the CheckVPNGatewayConnectionLocalCIDR method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CheckVPNGatewayConnectionLocalCIDRWithContext(ctx context.Context, checkVPNGatewayConnectionLocalCIDROptions *CheckVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(checkVPNGatewayConnectionLocalCIDROptions, "checkVPNGatewayConnectionLocalCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(checkVPNGatewayConnectionLocalCIDROptions, "checkVPNGatewayConnectionLocalCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *checkVPNGatewayConnectionLocalCIDROptions.VPNGatewayID,
		"id":             *checkVPNGatewayConnectionLocalCIDROptions.ID,
		"cidr_prefix":    *checkVPNGatewayConnectionLocalCIDROptions.CIDRPrefix,
		"prefix_length":  *checkVPNGatewayConnectionLocalCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range checkVPNGatewayConnectionLocalCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CheckVPNGatewayConnectionLocalCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// AddVPNGatewayConnectionLocalCIDR : Set a local CIDR on a VPN gateway connection
// This request adds the specified CIDR to the specified VPN gateway connection. A request body is not required, and if
// supplied, is ignored. This request succeeds if the CIDR already exists on the specified VPN gateway connection.
func (vpcClassic *VpcClassicV1) AddVPNGatewayConnectionLocalCIDR(addVPNGatewayConnectionLocalCIDROptions *AddVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.AddVPNGatewayConnectionLocalCIDRWithContext(context.Background(), addVPNGatewayConnectionLocalCIDROptions)
}

// AddVPNGatewayConnectionLocalCIDRWithContext is an alternate form of the AddVPNGatewayConnectionLocalCIDR method which supports a Context parameter
func (vpcClassic *VpcClassicV1) AddVPNGatewayConnectionLocalCIDRWithContext(ctx context.Context, addVPNGatewayConnectionLocalCIDROptions *AddVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addVPNGatewayConnectionLocalCIDROptions, "addVPNGatewayConnectionLocalCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addVPNGatewayConnectionLocalCIDROptions, "addVPNGatewayConnectionLocalCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *addVPNGatewayConnectionLocalCIDROptions.VPNGatewayID,
		"id":             *addVPNGatewayConnectionLocalCIDROptions.ID,
		"cidr_prefix":    *addVPNGatewayConnectionLocalCIDROptions.CIDRPrefix,
		"prefix_length":  *addVPNGatewayConnectionLocalCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addVPNGatewayConnectionLocalCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "AddVPNGatewayConnectionLocalCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// ListVPNGatewayConnectionPeerCIDRs : List all peer CIDRs for a VPN gateway connection
// This request lists all peer CIDRs for a VPN gateway connection.
func (vpcClassic *VpcClassicV1) ListVPNGatewayConnectionPeerCIDRs(listVPNGatewayConnectionPeerCIDRsOptions *ListVPNGatewayConnectionPeerCIDRsOptions) (result *VPNGatewayConnectionPeerCIDRs, response *core.DetailedResponse, err error) {
	return vpcClassic.ListVPNGatewayConnectionPeerCIDRsWithContext(context.Background(), listVPNGatewayConnectionPeerCIDRsOptions)
}

// ListVPNGatewayConnectionPeerCIDRsWithContext is an alternate form of the ListVPNGatewayConnectionPeerCIDRs method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListVPNGatewayConnectionPeerCIDRsWithContext(ctx context.Context, listVPNGatewayConnectionPeerCIDRsOptions *ListVPNGatewayConnectionPeerCIDRsOptions) (result *VPNGatewayConnectionPeerCIDRs, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPNGatewayConnectionPeerCIDRsOptions, "listVPNGatewayConnectionPeerCIDRsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPNGatewayConnectionPeerCIDRsOptions, "listVPNGatewayConnectionPeerCIDRsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *listVPNGatewayConnectionPeerCIDRsOptions.VPNGatewayID,
		"id":             *listVPNGatewayConnectionPeerCIDRsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewayConnectionPeerCIDRsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListVPNGatewayConnectionPeerCIDRs")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionPeerCIDRs)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// RemoveVPNGatewayConnectionPeerCIDR : Remove a peer CIDR from a VPN gateway connection
// This request removes a CIDR from a VPN gateway connection.
func (vpcClassic *VpcClassicV1) RemoveVPNGatewayConnectionPeerCIDR(removeVPNGatewayConnectionPeerCIDROptions *RemoveVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.RemoveVPNGatewayConnectionPeerCIDRWithContext(context.Background(), removeVPNGatewayConnectionPeerCIDROptions)
}

// RemoveVPNGatewayConnectionPeerCIDRWithContext is an alternate form of the RemoveVPNGatewayConnectionPeerCIDR method which supports a Context parameter
func (vpcClassic *VpcClassicV1) RemoveVPNGatewayConnectionPeerCIDRWithContext(ctx context.Context, removeVPNGatewayConnectionPeerCIDROptions *RemoveVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeVPNGatewayConnectionPeerCIDROptions, "removeVPNGatewayConnectionPeerCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeVPNGatewayConnectionPeerCIDROptions, "removeVPNGatewayConnectionPeerCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *removeVPNGatewayConnectionPeerCIDROptions.VPNGatewayID,
		"id":             *removeVPNGatewayConnectionPeerCIDROptions.ID,
		"cidr_prefix":    *removeVPNGatewayConnectionPeerCIDROptions.CIDRPrefix,
		"prefix_length":  *removeVPNGatewayConnectionPeerCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeVPNGatewayConnectionPeerCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "RemoveVPNGatewayConnectionPeerCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// CheckVPNGatewayConnectionPeerCIDR : Check if the specified peer CIDR exists on a VPN gateway connection
// This request succeeds if a CIDR exists on the specified VPN gateway connection and fails otherwise.
func (vpcClassic *VpcClassicV1) CheckVPNGatewayConnectionPeerCIDR(checkVPNGatewayConnectionPeerCIDROptions *CheckVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.CheckVPNGatewayConnectionPeerCIDRWithContext(context.Background(), checkVPNGatewayConnectionPeerCIDROptions)
}

// CheckVPNGatewayConnectionPeerCIDRWithContext is an alternate form of the CheckVPNGatewayConnectionPeerCIDR method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CheckVPNGatewayConnectionPeerCIDRWithContext(ctx context.Context, checkVPNGatewayConnectionPeerCIDROptions *CheckVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(checkVPNGatewayConnectionPeerCIDROptions, "checkVPNGatewayConnectionPeerCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(checkVPNGatewayConnectionPeerCIDROptions, "checkVPNGatewayConnectionPeerCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *checkVPNGatewayConnectionPeerCIDROptions.VPNGatewayID,
		"id":             *checkVPNGatewayConnectionPeerCIDROptions.ID,
		"cidr_prefix":    *checkVPNGatewayConnectionPeerCIDROptions.CIDRPrefix,
		"prefix_length":  *checkVPNGatewayConnectionPeerCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range checkVPNGatewayConnectionPeerCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CheckVPNGatewayConnectionPeerCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// AddVPNGatewayConnectionPeerCIDR : Set a peer CIDR on a VPN gateway connection
// This request adds the specified CIDR to the specified VPN gateway connection. A request body is not required, and if
// supplied, is ignored. This request succeeds if the CIDR already exists on the specified VPN gateway connection.
func (vpcClassic *VpcClassicV1) AddVPNGatewayConnectionPeerCIDR(addVPNGatewayConnectionPeerCIDROptions *AddVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.AddVPNGatewayConnectionPeerCIDRWithContext(context.Background(), addVPNGatewayConnectionPeerCIDROptions)
}

// AddVPNGatewayConnectionPeerCIDRWithContext is an alternate form of the AddVPNGatewayConnectionPeerCIDR method which supports a Context parameter
func (vpcClassic *VpcClassicV1) AddVPNGatewayConnectionPeerCIDRWithContext(ctx context.Context, addVPNGatewayConnectionPeerCIDROptions *AddVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addVPNGatewayConnectionPeerCIDROptions, "addVPNGatewayConnectionPeerCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addVPNGatewayConnectionPeerCIDROptions, "addVPNGatewayConnectionPeerCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *addVPNGatewayConnectionPeerCIDROptions.VPNGatewayID,
		"id":             *addVPNGatewayConnectionPeerCIDROptions.ID,
		"cidr_prefix":    *addVPNGatewayConnectionPeerCIDROptions.CIDRPrefix,
		"prefix_length":  *addVPNGatewayConnectionPeerCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addVPNGatewayConnectionPeerCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "AddVPNGatewayConnectionPeerCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// ListLoadBalancers : List all load balancers
// This request lists all load balancers in the region.
func (vpcClassic *VpcClassicV1) ListLoadBalancers(listLoadBalancersOptions *ListLoadBalancersOptions) (result *LoadBalancerCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListLoadBalancersWithContext(context.Background(), listLoadBalancersOptions)
}

// ListLoadBalancersWithContext is an alternate form of the ListLoadBalancers method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListLoadBalancersWithContext(ctx context.Context, listLoadBalancersOptions *ListLoadBalancersOptions) (result *LoadBalancerCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listLoadBalancersOptions, "listLoadBalancersOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListLoadBalancers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateLoadBalancer : Create a load balancer
// This request creates and provisions a new load balancer.
func (vpcClassic *VpcClassicV1) CreateLoadBalancer(createLoadBalancerOptions *CreateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateLoadBalancerWithContext(context.Background(), createLoadBalancerOptions)
}

// CreateLoadBalancerWithContext is an alternate form of the CreateLoadBalancer method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateLoadBalancerWithContext(ctx context.Context, createLoadBalancerOptions *CreateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerOptions, "createLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerOptions, "createLoadBalancerOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createLoadBalancerOptions.IsPublic != nil {
		body["is_public"] = createLoadBalancerOptions.IsPublic
	}
	if createLoadBalancerOptions.Subnets != nil {
		body["subnets"] = createLoadBalancerOptions.Subnets
	}
	if createLoadBalancerOptions.Listeners != nil {
		body["listeners"] = createLoadBalancerOptions.Listeners
	}
	if createLoadBalancerOptions.Logging != nil {
		body["logging"] = createLoadBalancerOptions.Logging
	}
	if createLoadBalancerOptions.Name != nil {
		body["name"] = createLoadBalancerOptions.Name
	}
	if createLoadBalancerOptions.Pools != nil {
		body["pools"] = createLoadBalancerOptions.Pools
	}
	if createLoadBalancerOptions.ResourceGroup != nil {
		body["resource_group"] = createLoadBalancerOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancer)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteLoadBalancer : Delete a load balancer
// This request deletes a load balancer. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteLoadBalancer(deleteLoadBalancerOptions *DeleteLoadBalancerOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteLoadBalancerWithContext(context.Background(), deleteLoadBalancerOptions)
}

// DeleteLoadBalancerWithContext is an alternate form of the DeleteLoadBalancer method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerWithContext(ctx context.Context, deleteLoadBalancerOptions *DeleteLoadBalancerOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerOptions, "deleteLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerOptions, "deleteLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteLoadBalancerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetLoadBalancer : Retrieve a load balancer
// This request retrieves a single load balancer specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetLoadBalancer(getLoadBalancerOptions *GetLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerWithContext(context.Background(), getLoadBalancerOptions)
}

// GetLoadBalancerWithContext is an alternate form of the GetLoadBalancer method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerWithContext(ctx context.Context, getLoadBalancerOptions *GetLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerOptions, "getLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerOptions, "getLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getLoadBalancerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancer)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateLoadBalancer : Update a load balancer
// This request updates a load balancer.
func (vpcClassic *VpcClassicV1) UpdateLoadBalancer(updateLoadBalancerOptions *UpdateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateLoadBalancerWithContext(context.Background(), updateLoadBalancerOptions)
}

// UpdateLoadBalancerWithContext is an alternate form of the UpdateLoadBalancer method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerWithContext(ctx context.Context, updateLoadBalancerOptions *UpdateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerOptions, "updateLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerOptions, "updateLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateLoadBalancerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerOptions.LoadBalancerPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancer)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetLoadBalancerStatistics : List all statistics of a load balancer
// This request lists statistics of a load balancer.
func (vpcClassic *VpcClassicV1) GetLoadBalancerStatistics(getLoadBalancerStatisticsOptions *GetLoadBalancerStatisticsOptions) (result *LoadBalancerStatistics, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerStatisticsWithContext(context.Background(), getLoadBalancerStatisticsOptions)
}

// GetLoadBalancerStatisticsWithContext is an alternate form of the GetLoadBalancerStatistics method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerStatisticsWithContext(ctx context.Context, getLoadBalancerStatisticsOptions *GetLoadBalancerStatisticsOptions) (result *LoadBalancerStatistics, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerStatisticsOptions, "getLoadBalancerStatisticsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerStatisticsOptions, "getLoadBalancerStatisticsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getLoadBalancerStatisticsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{id}/statistics`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerStatisticsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancerStatistics")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerStatistics)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListLoadBalancerListeners : List all listeners for a load balancer
// This request lists all listeners for a load balancer.
func (vpcClassic *VpcClassicV1) ListLoadBalancerListeners(listLoadBalancerListenersOptions *ListLoadBalancerListenersOptions) (result *LoadBalancerListenerCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListLoadBalancerListenersWithContext(context.Background(), listLoadBalancerListenersOptions)
}

// ListLoadBalancerListenersWithContext is an alternate form of the ListLoadBalancerListeners method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListLoadBalancerListenersWithContext(ctx context.Context, listLoadBalancerListenersOptions *ListLoadBalancerListenersOptions) (result *LoadBalancerListenerCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerListenersOptions, "listLoadBalancerListenersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerListenersOptions, "listLoadBalancerListenersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerListenersOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerListenersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListLoadBalancerListeners")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateLoadBalancerListener : Create a listener for a load balancer
// This request creates a new listener for a load balancer.
func (vpcClassic *VpcClassicV1) CreateLoadBalancerListener(createLoadBalancerListenerOptions *CreateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateLoadBalancerListenerWithContext(context.Background(), createLoadBalancerListenerOptions)
}

// CreateLoadBalancerListenerWithContext is an alternate form of the CreateLoadBalancerListener method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateLoadBalancerListenerWithContext(ctx context.Context, createLoadBalancerListenerOptions *CreateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerListenerOptions, "createLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerListenerOptions, "createLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerListenerOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createLoadBalancerListenerOptions.Port != nil {
		body["port"] = createLoadBalancerListenerOptions.Port
	}
	if createLoadBalancerListenerOptions.Protocol != nil {
		body["protocol"] = createLoadBalancerListenerOptions.Protocol
	}
	if createLoadBalancerListenerOptions.AcceptProxyProtocol != nil {
		body["accept_proxy_protocol"] = createLoadBalancerListenerOptions.AcceptProxyProtocol
	}
	if createLoadBalancerListenerOptions.CertificateInstance != nil {
		body["certificate_instance"] = createLoadBalancerListenerOptions.CertificateInstance
	}
	if createLoadBalancerListenerOptions.ConnectionLimit != nil {
		body["connection_limit"] = createLoadBalancerListenerOptions.ConnectionLimit
	}
	if createLoadBalancerListenerOptions.DefaultPool != nil {
		body["default_pool"] = createLoadBalancerListenerOptions.DefaultPool
	}
	if createLoadBalancerListenerOptions.Policies != nil {
		body["policies"] = createLoadBalancerListenerOptions.Policies
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListener)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteLoadBalancerListener : Delete a load balancer listener
// This request deletes a load balancer listener. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerListener(deleteLoadBalancerListenerOptions *DeleteLoadBalancerListenerOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteLoadBalancerListenerWithContext(context.Background(), deleteLoadBalancerListenerOptions)
}

// DeleteLoadBalancerListenerWithContext is an alternate form of the DeleteLoadBalancerListener method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerListenerWithContext(ctx context.Context, deleteLoadBalancerListenerOptions *DeleteLoadBalancerListenerOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerListenerOptions, "deleteLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerListenerOptions, "deleteLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerListenerOptions.LoadBalancerID,
		"id":               *deleteLoadBalancerListenerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetLoadBalancerListener : Retrieve a load balancer listener
// This request retrieves a single listener specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetLoadBalancerListener(getLoadBalancerListenerOptions *GetLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerListenerWithContext(context.Background(), getLoadBalancerListenerOptions)
}

// GetLoadBalancerListenerWithContext is an alternate form of the GetLoadBalancerListener method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerListenerWithContext(ctx context.Context, getLoadBalancerListenerOptions *GetLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerListenerOptions, "getLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerListenerOptions, "getLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerListenerOptions.LoadBalancerID,
		"id":               *getLoadBalancerListenerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListener)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateLoadBalancerListener : Update a load balancer listener
// This request updates a load balancer listener from a listener patch.
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerListener(updateLoadBalancerListenerOptions *UpdateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateLoadBalancerListenerWithContext(context.Background(), updateLoadBalancerListenerOptions)
}

// UpdateLoadBalancerListenerWithContext is an alternate form of the UpdateLoadBalancerListener method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerListenerWithContext(ctx context.Context, updateLoadBalancerListenerOptions *UpdateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerListenerOptions, "updateLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerListenerOptions, "updateLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerListenerOptions.LoadBalancerID,
		"id":               *updateLoadBalancerListenerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerListenerOptions.LoadBalancerListenerPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListener)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListLoadBalancerListenerPolicies : List all policies for a load balancer listener
// This request lists all policies for a load balancer listener.
func (vpcClassic *VpcClassicV1) ListLoadBalancerListenerPolicies(listLoadBalancerListenerPoliciesOptions *ListLoadBalancerListenerPoliciesOptions) (result *LoadBalancerListenerPolicyCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListLoadBalancerListenerPoliciesWithContext(context.Background(), listLoadBalancerListenerPoliciesOptions)
}

// ListLoadBalancerListenerPoliciesWithContext is an alternate form of the ListLoadBalancerListenerPolicies method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListLoadBalancerListenerPoliciesWithContext(ctx context.Context, listLoadBalancerListenerPoliciesOptions *ListLoadBalancerListenerPoliciesOptions) (result *LoadBalancerListenerPolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerListenerPoliciesOptions, "listLoadBalancerListenerPoliciesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerListenerPoliciesOptions, "listLoadBalancerListenerPoliciesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerListenerPoliciesOptions.LoadBalancerID,
		"listener_id":      *listLoadBalancerListenerPoliciesOptions.ListenerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerListenerPoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListLoadBalancerListenerPolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateLoadBalancerListenerPolicy : Create a policy for a load balancer listener
// Creates a new policy for a load balancer listener.
func (vpcClassic *VpcClassicV1) CreateLoadBalancerListenerPolicy(createLoadBalancerListenerPolicyOptions *CreateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateLoadBalancerListenerPolicyWithContext(context.Background(), createLoadBalancerListenerPolicyOptions)
}

// CreateLoadBalancerListenerPolicyWithContext is an alternate form of the CreateLoadBalancerListenerPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateLoadBalancerListenerPolicyWithContext(ctx context.Context, createLoadBalancerListenerPolicyOptions *CreateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerListenerPolicyOptions, "createLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerListenerPolicyOptions, "createLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *createLoadBalancerListenerPolicyOptions.ListenerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createLoadBalancerListenerPolicyOptions.Action != nil {
		body["action"] = createLoadBalancerListenerPolicyOptions.Action
	}
	if createLoadBalancerListenerPolicyOptions.Priority != nil {
		body["priority"] = createLoadBalancerListenerPolicyOptions.Priority
	}
	if createLoadBalancerListenerPolicyOptions.Name != nil {
		body["name"] = createLoadBalancerListenerPolicyOptions.Name
	}
	if createLoadBalancerListenerPolicyOptions.Rules != nil {
		body["rules"] = createLoadBalancerListenerPolicyOptions.Rules
	}
	if createLoadBalancerListenerPolicyOptions.Target != nil {
		body["target"] = createLoadBalancerListenerPolicyOptions.Target
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteLoadBalancerListenerPolicy : Delete a load balancer listener policy
// Deletes a policy of the load balancer listener. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerListenerPolicy(deleteLoadBalancerListenerPolicyOptions *DeleteLoadBalancerListenerPolicyOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteLoadBalancerListenerPolicyWithContext(context.Background(), deleteLoadBalancerListenerPolicyOptions)
}

// DeleteLoadBalancerListenerPolicyWithContext is an alternate form of the DeleteLoadBalancerListenerPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerListenerPolicyWithContext(ctx context.Context, deleteLoadBalancerListenerPolicyOptions *DeleteLoadBalancerListenerPolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerListenerPolicyOptions, "deleteLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerListenerPolicyOptions, "deleteLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *deleteLoadBalancerListenerPolicyOptions.ListenerID,
		"id":               *deleteLoadBalancerListenerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetLoadBalancerListenerPolicy : Retrieve a load balancer listener policy
// Retrieve a single policy specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetLoadBalancerListenerPolicy(getLoadBalancerListenerPolicyOptions *GetLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerListenerPolicyWithContext(context.Background(), getLoadBalancerListenerPolicyOptions)
}

// GetLoadBalancerListenerPolicyWithContext is an alternate form of the GetLoadBalancerListenerPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerListenerPolicyWithContext(ctx context.Context, getLoadBalancerListenerPolicyOptions *GetLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerListenerPolicyOptions, "getLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerListenerPolicyOptions, "getLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *getLoadBalancerListenerPolicyOptions.ListenerID,
		"id":               *getLoadBalancerListenerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateLoadBalancerListenerPolicy : Update a load balancer listener policy
// Updates a policy from a policy patch.
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerListenerPolicy(updateLoadBalancerListenerPolicyOptions *UpdateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateLoadBalancerListenerPolicyWithContext(context.Background(), updateLoadBalancerListenerPolicyOptions)
}

// UpdateLoadBalancerListenerPolicyWithContext is an alternate form of the UpdateLoadBalancerListenerPolicy method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerListenerPolicyWithContext(ctx context.Context, updateLoadBalancerListenerPolicyOptions *UpdateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerListenerPolicyOptions, "updateLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerListenerPolicyOptions, "updateLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *updateLoadBalancerListenerPolicyOptions.ListenerID,
		"id":               *updateLoadBalancerListenerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerListenerPolicyOptions.LoadBalancerListenerPolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicy)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListLoadBalancerListenerPolicyRules : List all rules of a load balancer listener policy
// This request lists all rules of a load balancer listener policy.
func (vpcClassic *VpcClassicV1) ListLoadBalancerListenerPolicyRules(listLoadBalancerListenerPolicyRulesOptions *ListLoadBalancerListenerPolicyRulesOptions) (result *LoadBalancerListenerPolicyRuleCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListLoadBalancerListenerPolicyRulesWithContext(context.Background(), listLoadBalancerListenerPolicyRulesOptions)
}

// ListLoadBalancerListenerPolicyRulesWithContext is an alternate form of the ListLoadBalancerListenerPolicyRules method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListLoadBalancerListenerPolicyRulesWithContext(ctx context.Context, listLoadBalancerListenerPolicyRulesOptions *ListLoadBalancerListenerPolicyRulesOptions) (result *LoadBalancerListenerPolicyRuleCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerListenerPolicyRulesOptions, "listLoadBalancerListenerPolicyRulesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerListenerPolicyRulesOptions, "listLoadBalancerListenerPolicyRulesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerListenerPolicyRulesOptions.LoadBalancerID,
		"listener_id":      *listLoadBalancerListenerPolicyRulesOptions.ListenerID,
		"policy_id":        *listLoadBalancerListenerPolicyRulesOptions.PolicyID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerListenerPolicyRulesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListLoadBalancerListenerPolicyRules")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRuleCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateLoadBalancerListenerPolicyRule : Create a rule for a load balancer listener policy
// Creates a new rule for the load balancer listener policy.
func (vpcClassic *VpcClassicV1) CreateLoadBalancerListenerPolicyRule(createLoadBalancerListenerPolicyRuleOptions *CreateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateLoadBalancerListenerPolicyRuleWithContext(context.Background(), createLoadBalancerListenerPolicyRuleOptions)
}

// CreateLoadBalancerListenerPolicyRuleWithContext is an alternate form of the CreateLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, createLoadBalancerListenerPolicyRuleOptions *CreateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerListenerPolicyRuleOptions, "createLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerListenerPolicyRuleOptions, "createLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *createLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *createLoadBalancerListenerPolicyRuleOptions.PolicyID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createLoadBalancerListenerPolicyRuleOptions.Condition != nil {
		body["condition"] = createLoadBalancerListenerPolicyRuleOptions.Condition
	}
	if createLoadBalancerListenerPolicyRuleOptions.Type != nil {
		body["type"] = createLoadBalancerListenerPolicyRuleOptions.Type
	}
	if createLoadBalancerListenerPolicyRuleOptions.Value != nil {
		body["value"] = createLoadBalancerListenerPolicyRuleOptions.Value
	}
	if createLoadBalancerListenerPolicyRuleOptions.Field != nil {
		body["field"] = createLoadBalancerListenerPolicyRuleOptions.Field
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteLoadBalancerListenerPolicyRule : Delete a load balancer listener policy rule
// Deletes a rule from the load balancer listener policy. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerListenerPolicyRule(deleteLoadBalancerListenerPolicyRuleOptions *DeleteLoadBalancerListenerPolicyRuleOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteLoadBalancerListenerPolicyRuleWithContext(context.Background(), deleteLoadBalancerListenerPolicyRuleOptions)
}

// DeleteLoadBalancerListenerPolicyRuleWithContext is an alternate form of the DeleteLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, deleteLoadBalancerListenerPolicyRuleOptions *DeleteLoadBalancerListenerPolicyRuleOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerListenerPolicyRuleOptions, "deleteLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerListenerPolicyRuleOptions, "deleteLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *deleteLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *deleteLoadBalancerListenerPolicyRuleOptions.PolicyID,
		"id":               *deleteLoadBalancerListenerPolicyRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetLoadBalancerListenerPolicyRule : Retrieve a load balancer listener policy rule
// Retrieves a single rule specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetLoadBalancerListenerPolicyRule(getLoadBalancerListenerPolicyRuleOptions *GetLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerListenerPolicyRuleWithContext(context.Background(), getLoadBalancerListenerPolicyRuleOptions)
}

// GetLoadBalancerListenerPolicyRuleWithContext is an alternate form of the GetLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, getLoadBalancerListenerPolicyRuleOptions *GetLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerListenerPolicyRuleOptions, "getLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerListenerPolicyRuleOptions, "getLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *getLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *getLoadBalancerListenerPolicyRuleOptions.PolicyID,
		"id":               *getLoadBalancerListenerPolicyRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateLoadBalancerListenerPolicyRule : Update a load balancer listener policy rule
// Updates a rule of the load balancer listener policy.
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerListenerPolicyRule(updateLoadBalancerListenerPolicyRuleOptions *UpdateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateLoadBalancerListenerPolicyRuleWithContext(context.Background(), updateLoadBalancerListenerPolicyRuleOptions)
}

// UpdateLoadBalancerListenerPolicyRuleWithContext is an alternate form of the UpdateLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, updateLoadBalancerListenerPolicyRuleOptions *UpdateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerListenerPolicyRuleOptions, "updateLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerListenerPolicyRuleOptions, "updateLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *updateLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *updateLoadBalancerListenerPolicyRuleOptions.PolicyID,
		"id":               *updateLoadBalancerListenerPolicyRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerListenerPolicyRuleOptions.LoadBalancerListenerPolicyRulePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRule)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListLoadBalancerPools : List all pools of a load balancer
// This request lists all pools of a load balancer.
func (vpcClassic *VpcClassicV1) ListLoadBalancerPools(listLoadBalancerPoolsOptions *ListLoadBalancerPoolsOptions) (result *LoadBalancerPoolCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListLoadBalancerPoolsWithContext(context.Background(), listLoadBalancerPoolsOptions)
}

// ListLoadBalancerPoolsWithContext is an alternate form of the ListLoadBalancerPools method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListLoadBalancerPoolsWithContext(ctx context.Context, listLoadBalancerPoolsOptions *ListLoadBalancerPoolsOptions) (result *LoadBalancerPoolCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerPoolsOptions, "listLoadBalancerPoolsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerPoolsOptions, "listLoadBalancerPoolsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerPoolsOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerPoolsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListLoadBalancerPools")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateLoadBalancerPool : Create a load balancer pool
// This request creates a new pool from a pool prototype object.
func (vpcClassic *VpcClassicV1) CreateLoadBalancerPool(createLoadBalancerPoolOptions *CreateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateLoadBalancerPoolWithContext(context.Background(), createLoadBalancerPoolOptions)
}

// CreateLoadBalancerPoolWithContext is an alternate form of the CreateLoadBalancerPool method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateLoadBalancerPoolWithContext(ctx context.Context, createLoadBalancerPoolOptions *CreateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerPoolOptions, "createLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerPoolOptions, "createLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerPoolOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createLoadBalancerPoolOptions.Algorithm != nil {
		body["algorithm"] = createLoadBalancerPoolOptions.Algorithm
	}
	if createLoadBalancerPoolOptions.HealthMonitor != nil {
		body["health_monitor"] = createLoadBalancerPoolOptions.HealthMonitor
	}
	if createLoadBalancerPoolOptions.Protocol != nil {
		body["protocol"] = createLoadBalancerPoolOptions.Protocol
	}
	if createLoadBalancerPoolOptions.Members != nil {
		body["members"] = createLoadBalancerPoolOptions.Members
	}
	if createLoadBalancerPoolOptions.Name != nil {
		body["name"] = createLoadBalancerPoolOptions.Name
	}
	if createLoadBalancerPoolOptions.ProxyProtocol != nil {
		body["proxy_protocol"] = createLoadBalancerPoolOptions.ProxyProtocol
	}
	if createLoadBalancerPoolOptions.SessionPersistence != nil {
		body["session_persistence"] = createLoadBalancerPoolOptions.SessionPersistence
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPool)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteLoadBalancerPool : Delete a load balancer pool
// This request deletes a load balancer pool. This operation cannot be reversed. The pool must not currently be the
// default pool for any listener in the load balancer.
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerPool(deleteLoadBalancerPoolOptions *DeleteLoadBalancerPoolOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteLoadBalancerPoolWithContext(context.Background(), deleteLoadBalancerPoolOptions)
}

// DeleteLoadBalancerPoolWithContext is an alternate form of the DeleteLoadBalancerPool method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerPoolWithContext(ctx context.Context, deleteLoadBalancerPoolOptions *DeleteLoadBalancerPoolOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerPoolOptions, "deleteLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerPoolOptions, "deleteLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerPoolOptions.LoadBalancerID,
		"id":               *deleteLoadBalancerPoolOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetLoadBalancerPool : Retrieve a load balancer pool
// This request retrieves a single pool specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetLoadBalancerPool(getLoadBalancerPoolOptions *GetLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerPoolWithContext(context.Background(), getLoadBalancerPoolOptions)
}

// GetLoadBalancerPoolWithContext is an alternate form of the GetLoadBalancerPool method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerPoolWithContext(ctx context.Context, getLoadBalancerPoolOptions *GetLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerPoolOptions, "getLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerPoolOptions, "getLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerPoolOptions.LoadBalancerID,
		"id":               *getLoadBalancerPoolOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPool)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateLoadBalancerPool : Update a load balancer pool
// This request updates a load balancer pool from a pool patch.
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerPool(updateLoadBalancerPoolOptions *UpdateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateLoadBalancerPoolWithContext(context.Background(), updateLoadBalancerPoolOptions)
}

// UpdateLoadBalancerPoolWithContext is an alternate form of the UpdateLoadBalancerPool method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerPoolWithContext(ctx context.Context, updateLoadBalancerPoolOptions *UpdateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerPoolOptions, "updateLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerPoolOptions, "updateLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerPoolOptions.LoadBalancerID,
		"id":               *updateLoadBalancerPoolOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerPoolOptions.LoadBalancerPoolPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPool)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListLoadBalancerPoolMembers : List all members of a load balancer pool
// This request lists all members of a load balancer pool.
func (vpcClassic *VpcClassicV1) ListLoadBalancerPoolMembers(listLoadBalancerPoolMembersOptions *ListLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ListLoadBalancerPoolMembersWithContext(context.Background(), listLoadBalancerPoolMembersOptions)
}

// ListLoadBalancerPoolMembersWithContext is an alternate form of the ListLoadBalancerPoolMembers method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ListLoadBalancerPoolMembersWithContext(ctx context.Context, listLoadBalancerPoolMembersOptions *ListLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerPoolMembersOptions, "listLoadBalancerPoolMembersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerPoolMembersOptions, "listLoadBalancerPoolMembersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerPoolMembersOptions.LoadBalancerID,
		"pool_id":          *listLoadBalancerPoolMembersOptions.PoolID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerPoolMembersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ListLoadBalancerPoolMembers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMemberCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateLoadBalancerPoolMember : Create a member in a load balancer pool
// This request creates a new member and adds the member to the pool.
func (vpcClassic *VpcClassicV1) CreateLoadBalancerPoolMember(createLoadBalancerPoolMemberOptions *CreateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	return vpcClassic.CreateLoadBalancerPoolMemberWithContext(context.Background(), createLoadBalancerPoolMemberOptions)
}

// CreateLoadBalancerPoolMemberWithContext is an alternate form of the CreateLoadBalancerPoolMember method which supports a Context parameter
func (vpcClassic *VpcClassicV1) CreateLoadBalancerPoolMemberWithContext(ctx context.Context, createLoadBalancerPoolMemberOptions *CreateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerPoolMemberOptions, "createLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerPoolMemberOptions, "createLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *createLoadBalancerPoolMemberOptions.PoolID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "CreateLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if createLoadBalancerPoolMemberOptions.Port != nil {
		body["port"] = createLoadBalancerPoolMemberOptions.Port
	}
	if createLoadBalancerPoolMemberOptions.Target != nil {
		body["target"] = createLoadBalancerPoolMemberOptions.Target
	}
	if createLoadBalancerPoolMemberOptions.Weight != nil {
		body["weight"] = createLoadBalancerPoolMemberOptions.Weight
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMember)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ReplaceLoadBalancerPoolMembers : Replace load balancer pool members
// This request replaces the existing members of the load balancer pool with new members created from the collection of
// member prototype objects.
func (vpcClassic *VpcClassicV1) ReplaceLoadBalancerPoolMembers(replaceLoadBalancerPoolMembersOptions *ReplaceLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	return vpcClassic.ReplaceLoadBalancerPoolMembersWithContext(context.Background(), replaceLoadBalancerPoolMembersOptions)
}

// ReplaceLoadBalancerPoolMembersWithContext is an alternate form of the ReplaceLoadBalancerPoolMembers method which supports a Context parameter
func (vpcClassic *VpcClassicV1) ReplaceLoadBalancerPoolMembersWithContext(ctx context.Context, replaceLoadBalancerPoolMembersOptions *ReplaceLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(replaceLoadBalancerPoolMembersOptions, "replaceLoadBalancerPoolMembersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(replaceLoadBalancerPoolMembersOptions, "replaceLoadBalancerPoolMembersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *replaceLoadBalancerPoolMembersOptions.LoadBalancerID,
		"pool_id":          *replaceLoadBalancerPoolMembersOptions.PoolID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range replaceLoadBalancerPoolMembersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "ReplaceLoadBalancerPoolMembers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	body := make(map[string]interface{})
	if replaceLoadBalancerPoolMembersOptions.Members != nil {
		body["members"] = replaceLoadBalancerPoolMembersOptions.Members
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMemberCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteLoadBalancerPoolMember : Delete a load balancer pool member
// This request deletes a member from the pool. This operation cannot be reversed.
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerPoolMember(deleteLoadBalancerPoolMemberOptions *DeleteLoadBalancerPoolMemberOptions) (response *core.DetailedResponse, err error) {
	return vpcClassic.DeleteLoadBalancerPoolMemberWithContext(context.Background(), deleteLoadBalancerPoolMemberOptions)
}

// DeleteLoadBalancerPoolMemberWithContext is an alternate form of the DeleteLoadBalancerPoolMember method which supports a Context parameter
func (vpcClassic *VpcClassicV1) DeleteLoadBalancerPoolMemberWithContext(ctx context.Context, deleteLoadBalancerPoolMemberOptions *DeleteLoadBalancerPoolMemberOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerPoolMemberOptions, "deleteLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerPoolMemberOptions, "deleteLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *deleteLoadBalancerPoolMemberOptions.PoolID,
		"id":               *deleteLoadBalancerPoolMemberOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "DeleteLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpcClassic.Service.Request(request, nil)

	return
}

// GetLoadBalancerPoolMember : Retrieve a load balancer pool member
// This request retrieves a single member specified by the identifier in the URL path.
func (vpcClassic *VpcClassicV1) GetLoadBalancerPoolMember(getLoadBalancerPoolMemberOptions *GetLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	return vpcClassic.GetLoadBalancerPoolMemberWithContext(context.Background(), getLoadBalancerPoolMemberOptions)
}

// GetLoadBalancerPoolMemberWithContext is an alternate form of the GetLoadBalancerPoolMember method which supports a Context parameter
func (vpcClassic *VpcClassicV1) GetLoadBalancerPoolMemberWithContext(ctx context.Context, getLoadBalancerPoolMemberOptions *GetLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerPoolMemberOptions, "getLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerPoolMemberOptions, "getLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *getLoadBalancerPoolMemberOptions.PoolID,
		"id":               *getLoadBalancerPoolMemberOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "GetLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMember)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateLoadBalancerPoolMember : Update a load balancer pool member
// This request updates an existing member from a member patch.
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerPoolMember(updateLoadBalancerPoolMemberOptions *UpdateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	return vpcClassic.UpdateLoadBalancerPoolMemberWithContext(context.Background(), updateLoadBalancerPoolMemberOptions)
}

// UpdateLoadBalancerPoolMemberWithContext is an alternate form of the UpdateLoadBalancerPoolMember method which supports a Context parameter
func (vpcClassic *VpcClassicV1) UpdateLoadBalancerPoolMemberWithContext(ctx context.Context, updateLoadBalancerPoolMemberOptions *UpdateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerPoolMemberOptions, "updateLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerPoolMemberOptions, "updateLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *updateLoadBalancerPoolMemberOptions.PoolID,
		"id":               *updateLoadBalancerPoolMemberOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpcClassic.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpcClassic.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc_classic", "V1", "UpdateLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpcClassic.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpcClassic.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerPoolMemberOptions.LoadBalancerPoolMemberPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpcClassic.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMember)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// AddInstanceNetworkInterfaceFloatingIPOptions : The AddInstanceNetworkInterfaceFloatingIP options.
type AddInstanceNetworkInterfaceFloatingIPOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	NetworkInterfaceID *string `validate:"required,ne="`

	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddInstanceNetworkInterfaceFloatingIPOptions : Instantiate AddInstanceNetworkInterfaceFloatingIPOptions
func (*VpcClassicV1) NewAddInstanceNetworkInterfaceFloatingIPOptions(instanceID string, networkInterfaceID string, id string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	return &AddInstanceNetworkInterfaceFloatingIPOptions{
		InstanceID:         core.StringPtr(instanceID),
		NetworkInterfaceID: core.StringPtr(networkInterfaceID),
		ID:                 core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetInstanceID(instanceID string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetNetworkInterfaceID : Allow user to set NetworkInterfaceID
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetNetworkInterfaceID(networkInterfaceID string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.NetworkInterfaceID = core.StringPtr(networkInterfaceID)
	return options
}

// SetID : Allow user to set ID
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetID(id string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetHeaders(param map[string]string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.Headers = param
	return options
}

// AddSecurityGroupNetworkInterfaceOptions : The AddSecurityGroupNetworkInterface options.
type AddSecurityGroupNetworkInterfaceOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddSecurityGroupNetworkInterfaceOptions : Instantiate AddSecurityGroupNetworkInterfaceOptions
func (*VpcClassicV1) NewAddSecurityGroupNetworkInterfaceOptions(securityGroupID string, id string) *AddSecurityGroupNetworkInterfaceOptions {
	return &AddSecurityGroupNetworkInterfaceOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *AddSecurityGroupNetworkInterfaceOptions) SetSecurityGroupID(securityGroupID string) *AddSecurityGroupNetworkInterfaceOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *AddSecurityGroupNetworkInterfaceOptions) SetID(id string) *AddSecurityGroupNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddSecurityGroupNetworkInterfaceOptions) SetHeaders(param map[string]string) *AddSecurityGroupNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// AddVPNGatewayConnectionLocalCIDROptions : The AddVPNGatewayConnectionLocalCIDR options.
type AddVPNGatewayConnectionLocalCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddVPNGatewayConnectionLocalCIDROptions : Instantiate AddVPNGatewayConnectionLocalCIDROptions
func (*VpcClassicV1) NewAddVPNGatewayConnectionLocalCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *AddVPNGatewayConnectionLocalCIDROptions {
	return &AddVPNGatewayConnectionLocalCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetVPNGatewayID(vpnGatewayID string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetID(id string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetCIDRPrefix(cidrPrefix string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetPrefixLength(prefixLength string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetHeaders(param map[string]string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.Headers = param
	return options
}

// AddVPNGatewayConnectionPeerCIDROptions : The AddVPNGatewayConnectionPeerCIDR options.
type AddVPNGatewayConnectionPeerCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddVPNGatewayConnectionPeerCIDROptions : Instantiate AddVPNGatewayConnectionPeerCIDROptions
func (*VpcClassicV1) NewAddVPNGatewayConnectionPeerCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *AddVPNGatewayConnectionPeerCIDROptions {
	return &AddVPNGatewayConnectionPeerCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetVPNGatewayID(vpnGatewayID string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetID(id string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetCIDRPrefix(cidrPrefix string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetPrefixLength(prefixLength string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetHeaders(param map[string]string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.Headers = param
	return options
}

// AddressPrefix : AddressPrefix struct
type AddressPrefix struct {
	// The CIDR block for this prefix.
	CIDR *string `json:"cidr" validate:"required"`

	// The date and time that the prefix was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// Indicates whether subnets exist with addresses from this prefix.
	HasSubnets *bool `json:"has_subnets" validate:"required"`

	// The URL for this address prefix.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this address prefix.
	ID *string `json:"id" validate:"required"`

	// Indicates whether this is the default prefix for this zone in this VPC. If a default prefix was automatically
	// created when the VPC was created, the prefix is automatically named using a hyphenated list of randomly-selected
	// words, but may be updated with a user-specified name.
	IsDefault *bool `json:"is_default" validate:"required"`

	// The user-defined name for this address prefix. Names must be unique within the VPC the address prefix resides in.
	Name *string `json:"name" validate:"required"`

	// The zone this address prefix resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// UnmarshalAddressPrefix unmarshals an instance of AddressPrefix from the specified map of raw messages.
func UnmarshalAddressPrefix(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefix)
	err = core.UnmarshalPrimitive(m, "cidr", &obj.CIDR)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "has_subnets", &obj.HasSubnets)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "is_default", &obj.IsDefault)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixCollection : AddressPrefixCollection struct
type AddressPrefixCollection struct {
	// Collection of address prefixes.
	AddressPrefixes []AddressPrefix `json:"address_prefixes" validate:"required"`

	// A link to the first page of resources.
	First *AddressPrefixCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *AddressPrefixCollectionNext `json:"next,omitempty"`
}

// UnmarshalAddressPrefixCollection unmarshals an instance of AddressPrefixCollection from the specified map of raw messages.
func UnmarshalAddressPrefixCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixCollection)
	err = core.UnmarshalModel(m, "address_prefixes", &obj.AddressPrefixes, UnmarshalAddressPrefix)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalAddressPrefixCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalAddressPrefixCollectionNext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixCollectionFirst : A link to the first page of resources.
type AddressPrefixCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalAddressPrefixCollectionFirst unmarshals an instance of AddressPrefixCollectionFirst from the specified map of raw messages.
func UnmarshalAddressPrefixCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type AddressPrefixCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalAddressPrefixCollectionNext unmarshals an instance of AddressPrefixCollectionNext from the specified map of raw messages.
func UnmarshalAddressPrefixCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixPatch : AddressPrefixPatch struct
type AddressPrefixPatch struct {
	// Indicates whether this is the default prefix for this zone in this VPC. Updating to true makes this prefix the
	// default prefix for this zone in this VPC, provided the VPC currently has no default address prefix for this zone.
	// Updating to false removes the default prefix for this zone in this VPC.
	IsDefault *bool `json:"is_default,omitempty"`

	// The user-defined name for this address prefix. Names must be unique within the VPC the address prefix resides in.
	Name *string `json:"name,omitempty"`
}

// UnmarshalAddressPrefixPatch unmarshals an instance of AddressPrefixPatch from the specified map of raw messages.
func UnmarshalAddressPrefixPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixPatch)
	err = core.UnmarshalPrimitive(m, "is_default", &obj.IsDefault)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the AddressPrefixPatch
func (addressPrefixPatch *AddressPrefixPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(addressPrefixPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// CertificateInstanceIdentity : Identifies a certificate instance by a unique property.
// Models which "extend" this model:
// - CertificateInstanceIdentityByCRN
type CertificateInstanceIdentity struct {
	// The CRN for this certificate instance.
	CRN *string `json:"crn,omitempty"`
}

func (*CertificateInstanceIdentity) isaCertificateInstanceIdentity() bool {
	return true
}

type CertificateInstanceIdentityIntf interface {
	isaCertificateInstanceIdentity() bool
}

// UnmarshalCertificateInstanceIdentity unmarshals an instance of CertificateInstanceIdentity from the specified map of raw messages.
func UnmarshalCertificateInstanceIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CertificateInstanceIdentity)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// CertificateInstanceReference : CertificateInstanceReference struct
type CertificateInstanceReference struct {
	// The CRN for this certificate instance.
	CRN *string `json:"crn" validate:"required"`
}

// UnmarshalCertificateInstanceReference unmarshals an instance of CertificateInstanceReference from the specified map of raw messages.
func UnmarshalCertificateInstanceReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CertificateInstanceReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// CheckVPNGatewayConnectionLocalCIDROptions : The CheckVPNGatewayConnectionLocalCIDR options.
type CheckVPNGatewayConnectionLocalCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCheckVPNGatewayConnectionLocalCIDROptions : Instantiate CheckVPNGatewayConnectionLocalCIDROptions
func (*VpcClassicV1) NewCheckVPNGatewayConnectionLocalCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *CheckVPNGatewayConnectionLocalCIDROptions {
	return &CheckVPNGatewayConnectionLocalCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetVPNGatewayID(vpnGatewayID string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetID(id string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetCIDRPrefix(cidrPrefix string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetPrefixLength(prefixLength string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetHeaders(param map[string]string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.Headers = param
	return options
}

// CheckVPNGatewayConnectionPeerCIDROptions : The CheckVPNGatewayConnectionPeerCIDR options.
type CheckVPNGatewayConnectionPeerCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCheckVPNGatewayConnectionPeerCIDROptions : Instantiate CheckVPNGatewayConnectionPeerCIDROptions
func (*VpcClassicV1) NewCheckVPNGatewayConnectionPeerCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *CheckVPNGatewayConnectionPeerCIDROptions {
	return &CheckVPNGatewayConnectionPeerCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetVPNGatewayID(vpnGatewayID string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetID(id string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetCIDRPrefix(cidrPrefix string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetPrefixLength(prefixLength string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetHeaders(param map[string]string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.Headers = param
	return options
}

// CreateFloatingIPOptions : The CreateFloatingIP options.
type CreateFloatingIPOptions struct {
	// The floating IP prototype object.
	FloatingIPPrototype FloatingIPPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateFloatingIPOptions : Instantiate CreateFloatingIPOptions
func (*VpcClassicV1) NewCreateFloatingIPOptions(floatingIPPrototype FloatingIPPrototypeIntf) *CreateFloatingIPOptions {
	return &CreateFloatingIPOptions{
		FloatingIPPrototype: floatingIPPrototype,
	}
}

// SetFloatingIPPrototype : Allow user to set FloatingIPPrototype
func (options *CreateFloatingIPOptions) SetFloatingIPPrototype(floatingIPPrototype FloatingIPPrototypeIntf) *CreateFloatingIPOptions {
	options.FloatingIPPrototype = floatingIPPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateFloatingIPOptions) SetHeaders(param map[string]string) *CreateFloatingIPOptions {
	options.Headers = param
	return options
}

// CreateIkePolicyOptions : The CreateIkePolicy options.
type CreateIkePolicyOptions struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `validate:"required"`

	// The Diffie-Hellman group.
	DhGroup *int64 `validate:"required"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `validate:"required"`

	// The IKE protocol version.
	IkeVersion *int64 `validate:"required"`

	// The key lifetime in seconds.
	KeyLifetime *int64

	// The user-defined name for this IKE policy.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateIkePolicyOptions.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	CreateIkePolicyOptionsAuthenticationAlgorithmMd5Const    = "md5"
	CreateIkePolicyOptionsAuthenticationAlgorithmSha1Const   = "sha1"
	CreateIkePolicyOptionsAuthenticationAlgorithmSha256Const = "sha256"
)

// Constants associated with the CreateIkePolicyOptions.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	CreateIkePolicyOptionsEncryptionAlgorithmAes128Const    = "aes128"
	CreateIkePolicyOptionsEncryptionAlgorithmAes256Const    = "aes256"
	CreateIkePolicyOptionsEncryptionAlgorithmTripleDesConst = "triple_des"
)

// NewCreateIkePolicyOptions : Instantiate CreateIkePolicyOptions
func (*VpcClassicV1) NewCreateIkePolicyOptions(authenticationAlgorithm string, dhGroup int64, encryptionAlgorithm string, ikeVersion int64) *CreateIkePolicyOptions {
	return &CreateIkePolicyOptions{
		AuthenticationAlgorithm: core.StringPtr(authenticationAlgorithm),
		DhGroup:                 core.Int64Ptr(dhGroup),
		EncryptionAlgorithm:     core.StringPtr(encryptionAlgorithm),
		IkeVersion:              core.Int64Ptr(ikeVersion),
	}
}

// SetAuthenticationAlgorithm : Allow user to set AuthenticationAlgorithm
func (options *CreateIkePolicyOptions) SetAuthenticationAlgorithm(authenticationAlgorithm string) *CreateIkePolicyOptions {
	options.AuthenticationAlgorithm = core.StringPtr(authenticationAlgorithm)
	return options
}

// SetDhGroup : Allow user to set DhGroup
func (options *CreateIkePolicyOptions) SetDhGroup(dhGroup int64) *CreateIkePolicyOptions {
	options.DhGroup = core.Int64Ptr(dhGroup)
	return options
}

// SetEncryptionAlgorithm : Allow user to set EncryptionAlgorithm
func (options *CreateIkePolicyOptions) SetEncryptionAlgorithm(encryptionAlgorithm string) *CreateIkePolicyOptions {
	options.EncryptionAlgorithm = core.StringPtr(encryptionAlgorithm)
	return options
}

// SetIkeVersion : Allow user to set IkeVersion
func (options *CreateIkePolicyOptions) SetIkeVersion(ikeVersion int64) *CreateIkePolicyOptions {
	options.IkeVersion = core.Int64Ptr(ikeVersion)
	return options
}

// SetKeyLifetime : Allow user to set KeyLifetime
func (options *CreateIkePolicyOptions) SetKeyLifetime(keyLifetime int64) *CreateIkePolicyOptions {
	options.KeyLifetime = core.Int64Ptr(keyLifetime)
	return options
}

// SetName : Allow user to set Name
func (options *CreateIkePolicyOptions) SetName(name string) *CreateIkePolicyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateIkePolicyOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateIkePolicyOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateIkePolicyOptions) SetHeaders(param map[string]string) *CreateIkePolicyOptions {
	options.Headers = param
	return options
}

// CreateImageOptions : The CreateImage options.
type CreateImageOptions struct {
	// The image prototype object.
	ImagePrototype ImagePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateImageOptions : Instantiate CreateImageOptions
func (*VpcClassicV1) NewCreateImageOptions(imagePrototype ImagePrototypeIntf) *CreateImageOptions {
	return &CreateImageOptions{
		ImagePrototype: imagePrototype,
	}
}

// SetImagePrototype : Allow user to set ImagePrototype
func (options *CreateImageOptions) SetImagePrototype(imagePrototype ImagePrototypeIntf) *CreateImageOptions {
	options.ImagePrototype = imagePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateImageOptions) SetHeaders(param map[string]string) *CreateImageOptions {
	options.Headers = param
	return options
}

// CreateInstanceActionOptions : The CreateInstanceAction options.
type CreateInstanceActionOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The type of action.
	Type *string `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateInstanceActionOptions.Type property.
// The type of action.
const (
	CreateInstanceActionOptionsTypeRebootConst = "reboot"
	CreateInstanceActionOptionsTypeResetConst  = "reset"
	CreateInstanceActionOptionsTypeStartConst  = "start"
	CreateInstanceActionOptionsTypeStopConst   = "stop"
)

// NewCreateInstanceActionOptions : Instantiate CreateInstanceActionOptions
func (*VpcClassicV1) NewCreateInstanceActionOptions(instanceID string, typeVar string) *CreateInstanceActionOptions {
	return &CreateInstanceActionOptions{
		InstanceID: core.StringPtr(instanceID),
		Type:       core.StringPtr(typeVar),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *CreateInstanceActionOptions) SetInstanceID(instanceID string) *CreateInstanceActionOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetType : Allow user to set Type
func (options *CreateInstanceActionOptions) SetType(typeVar string) *CreateInstanceActionOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceActionOptions) SetHeaders(param map[string]string) *CreateInstanceActionOptions {
	options.Headers = param
	return options
}

// CreateInstanceOptions : The CreateInstance options.
type CreateInstanceOptions struct {
	// The instance prototype object.
	InstancePrototype InstancePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceOptions : Instantiate CreateInstanceOptions
func (*VpcClassicV1) NewCreateInstanceOptions(instancePrototype InstancePrototypeIntf) *CreateInstanceOptions {
	return &CreateInstanceOptions{
		InstancePrototype: instancePrototype,
	}
}

// SetInstancePrototype : Allow user to set InstancePrototype
func (options *CreateInstanceOptions) SetInstancePrototype(instancePrototype InstancePrototypeIntf) *CreateInstanceOptions {
	options.InstancePrototype = instancePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceOptions) SetHeaders(param map[string]string) *CreateInstanceOptions {
	options.Headers = param
	return options
}

// CreateInstanceVolumeAttachmentOptions : The CreateInstanceVolumeAttachment options.
type CreateInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The identity of the volume to attach to the instance.
	Volume VolumeIdentityIntf `validate:"required"`

	// If set to true, when deleting the instance the volume will also be deleted.
	DeleteVolumeOnInstanceDelete *bool

	// The user-defined name for this volume attachment. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceVolumeAttachmentOptions : Instantiate CreateInstanceVolumeAttachmentOptions
func (*VpcClassicV1) NewCreateInstanceVolumeAttachmentOptions(instanceID string, volume VolumeIdentityIntf) *CreateInstanceVolumeAttachmentOptions {
	return &CreateInstanceVolumeAttachmentOptions{
		InstanceID: core.StringPtr(instanceID),
		Volume:     volume,
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *CreateInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *CreateInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetVolume : Allow user to set Volume
func (options *CreateInstanceVolumeAttachmentOptions) SetVolume(volume VolumeIdentityIntf) *CreateInstanceVolumeAttachmentOptions {
	options.Volume = volume
	return options
}

// SetDeleteVolumeOnInstanceDelete : Allow user to set DeleteVolumeOnInstanceDelete
func (options *CreateInstanceVolumeAttachmentOptions) SetDeleteVolumeOnInstanceDelete(deleteVolumeOnInstanceDelete bool) *CreateInstanceVolumeAttachmentOptions {
	options.DeleteVolumeOnInstanceDelete = core.BoolPtr(deleteVolumeOnInstanceDelete)
	return options
}

// SetName : Allow user to set Name
func (options *CreateInstanceVolumeAttachmentOptions) SetName(name string) *CreateInstanceVolumeAttachmentOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *CreateInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// CreateIpsecPolicyOptions : The CreateIpsecPolicy options.
type CreateIpsecPolicyOptions struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `validate:"required"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `validate:"required"`

	// Perfect Forward Secrecy.
	Pfs *string `validate:"required"`

	// The key lifetime in seconds.
	KeyLifetime *int64

	// The user-defined name for this IPsec policy.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateIpsecPolicyOptions.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	CreateIpsecPolicyOptionsAuthenticationAlgorithmMd5Const    = "md5"
	CreateIpsecPolicyOptionsAuthenticationAlgorithmSha1Const   = "sha1"
	CreateIpsecPolicyOptionsAuthenticationAlgorithmSha256Const = "sha256"
)

// Constants associated with the CreateIpsecPolicyOptions.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	CreateIpsecPolicyOptionsEncryptionAlgorithmAes128Const    = "aes128"
	CreateIpsecPolicyOptionsEncryptionAlgorithmAes256Const    = "aes256"
	CreateIpsecPolicyOptionsEncryptionAlgorithmTripleDesConst = "triple_des"
)

// Constants associated with the CreateIpsecPolicyOptions.Pfs property.
// Perfect Forward Secrecy.
const (
	CreateIpsecPolicyOptionsPfsDisabledConst = "disabled"
	CreateIpsecPolicyOptionsPfsGroup14Const  = "group_14"
	CreateIpsecPolicyOptionsPfsGroup2Const   = "group_2"
	CreateIpsecPolicyOptionsPfsGroup5Const   = "group_5"
)

// NewCreateIpsecPolicyOptions : Instantiate CreateIpsecPolicyOptions
func (*VpcClassicV1) NewCreateIpsecPolicyOptions(authenticationAlgorithm string, encryptionAlgorithm string, pfs string) *CreateIpsecPolicyOptions {
	return &CreateIpsecPolicyOptions{
		AuthenticationAlgorithm: core.StringPtr(authenticationAlgorithm),
		EncryptionAlgorithm:     core.StringPtr(encryptionAlgorithm),
		Pfs:                     core.StringPtr(pfs),
	}
}

// SetAuthenticationAlgorithm : Allow user to set AuthenticationAlgorithm
func (options *CreateIpsecPolicyOptions) SetAuthenticationAlgorithm(authenticationAlgorithm string) *CreateIpsecPolicyOptions {
	options.AuthenticationAlgorithm = core.StringPtr(authenticationAlgorithm)
	return options
}

// SetEncryptionAlgorithm : Allow user to set EncryptionAlgorithm
func (options *CreateIpsecPolicyOptions) SetEncryptionAlgorithm(encryptionAlgorithm string) *CreateIpsecPolicyOptions {
	options.EncryptionAlgorithm = core.StringPtr(encryptionAlgorithm)
	return options
}

// SetPfs : Allow user to set Pfs
func (options *CreateIpsecPolicyOptions) SetPfs(pfs string) *CreateIpsecPolicyOptions {
	options.Pfs = core.StringPtr(pfs)
	return options
}

// SetKeyLifetime : Allow user to set KeyLifetime
func (options *CreateIpsecPolicyOptions) SetKeyLifetime(keyLifetime int64) *CreateIpsecPolicyOptions {
	options.KeyLifetime = core.Int64Ptr(keyLifetime)
	return options
}

// SetName : Allow user to set Name
func (options *CreateIpsecPolicyOptions) SetName(name string) *CreateIpsecPolicyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateIpsecPolicyOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateIpsecPolicyOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateIpsecPolicyOptions) SetHeaders(param map[string]string) *CreateIpsecPolicyOptions {
	options.Headers = param
	return options
}

// CreateKeyOptions : The CreateKey options.
type CreateKeyOptions struct {
	// A unique public SSH key to import, encoded in PEM format. The key (prior to encoding) must be either 2048 or 4096
	// bits long.
	PublicKey *string `validate:"required"`

	// The user-defined name for this key.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// The crypto-system used by this key.
	Type *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateKeyOptions.Type property.
// The crypto-system used by this key.
const (
	CreateKeyOptionsTypeRsaConst = "rsa"
)

// NewCreateKeyOptions : Instantiate CreateKeyOptions
func (*VpcClassicV1) NewCreateKeyOptions(publicKey string) *CreateKeyOptions {
	return &CreateKeyOptions{
		PublicKey: core.StringPtr(publicKey),
	}
}

// SetPublicKey : Allow user to set PublicKey
func (options *CreateKeyOptions) SetPublicKey(publicKey string) *CreateKeyOptions {
	options.PublicKey = core.StringPtr(publicKey)
	return options
}

// SetName : Allow user to set Name
func (options *CreateKeyOptions) SetName(name string) *CreateKeyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateKeyOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateKeyOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetType : Allow user to set Type
func (options *CreateKeyOptions) SetType(typeVar string) *CreateKeyOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateKeyOptions) SetHeaders(param map[string]string) *CreateKeyOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerListenerOptions : The CreateLoadBalancerListener options.
type CreateLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener port number. Each listener in the load balancer must have a unique
	// `port` and `protocol` combination.
	Port *int64 `validate:"required"`

	// The listener protocol.
	Protocol *string `validate:"required"`

	// If set to `true`, this listener will accept and forward PROXY protocol information. Supported by load balancers in
	// the `application` family (otherwise always `false`).
	AcceptProxyProtocol *bool

	// The certificate instance used for SSL termination. It is applicable only to `https`
	// protocol.
	CertificateInstance CertificateInstanceIdentityIntf

	// The connection limit of the listener.
	ConnectionLimit *int64

	// The default pool associated with the listener. The specified pool must:
	//
	// - Belong to this load balancer
	// - Have the same `protocol` as this listener
	// - Not already be the default pool for another listener.
	DefaultPool LoadBalancerPoolIdentityIntf

	// An array of policies for this listener.
	Policies []LoadBalancerListenerPolicyPrototype

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerListenerOptions.Protocol property.
// The listener protocol.
const (
	CreateLoadBalancerListenerOptionsProtocolHTTPConst  = "http"
	CreateLoadBalancerListenerOptionsProtocolHTTPSConst = "https"
	CreateLoadBalancerListenerOptionsProtocolTCPConst   = "tcp"
)

// NewCreateLoadBalancerListenerOptions : Instantiate CreateLoadBalancerListenerOptions
func (*VpcClassicV1) NewCreateLoadBalancerListenerOptions(loadBalancerID string, port int64, protocol string) *CreateLoadBalancerListenerOptions {
	return &CreateLoadBalancerListenerOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		Port:           core.Int64Ptr(port),
		Protocol:       core.StringPtr(protocol),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPort : Allow user to set Port
func (options *CreateLoadBalancerListenerOptions) SetPort(port int64) *CreateLoadBalancerListenerOptions {
	options.Port = core.Int64Ptr(port)
	return options
}

// SetProtocol : Allow user to set Protocol
func (options *CreateLoadBalancerListenerOptions) SetProtocol(protocol string) *CreateLoadBalancerListenerOptions {
	options.Protocol = core.StringPtr(protocol)
	return options
}

// SetAcceptProxyProtocol : Allow user to set AcceptProxyProtocol
func (options *CreateLoadBalancerListenerOptions) SetAcceptProxyProtocol(acceptProxyProtocol bool) *CreateLoadBalancerListenerOptions {
	options.AcceptProxyProtocol = core.BoolPtr(acceptProxyProtocol)
	return options
}

// SetCertificateInstance : Allow user to set CertificateInstance
func (options *CreateLoadBalancerListenerOptions) SetCertificateInstance(certificateInstance CertificateInstanceIdentityIntf) *CreateLoadBalancerListenerOptions {
	options.CertificateInstance = certificateInstance
	return options
}

// SetConnectionLimit : Allow user to set ConnectionLimit
func (options *CreateLoadBalancerListenerOptions) SetConnectionLimit(connectionLimit int64) *CreateLoadBalancerListenerOptions {
	options.ConnectionLimit = core.Int64Ptr(connectionLimit)
	return options
}

// SetDefaultPool : Allow user to set DefaultPool
func (options *CreateLoadBalancerListenerOptions) SetDefaultPool(defaultPool LoadBalancerPoolIdentityIntf) *CreateLoadBalancerListenerOptions {
	options.DefaultPool = defaultPool
	return options
}

// SetPolicies : Allow user to set Policies
func (options *CreateLoadBalancerListenerOptions) SetPolicies(policies []LoadBalancerListenerPolicyPrototype) *CreateLoadBalancerListenerOptions {
	options.Policies = policies
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerListenerOptions) SetHeaders(param map[string]string) *CreateLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerListenerPolicyOptions : The CreateLoadBalancerListenerPolicy options.
type CreateLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy action.
	Action *string `validate:"required"`

	// Priority of the policy. Lower value indicates higher priority.
	Priority *int64 `validate:"required"`

	// The user-defined name for this policy. Names must be unique within the load balancer listener the policy resides in.
	Name *string

	// An array of rules for this policy.
	Rules []LoadBalancerListenerPolicyRulePrototype

	// When `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify which
	// pool the load balancer forwards the traffic to. When `action` is `redirect`,
	// `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the url and
	// http status code used in the redirect response.
	Target LoadBalancerListenerPolicyTargetPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerListenerPolicyOptions.Action property.
// The policy action.
const (
	CreateLoadBalancerListenerPolicyOptionsActionForwardConst  = "forward"
	CreateLoadBalancerListenerPolicyOptionsActionRedirectConst = "redirect"
	CreateLoadBalancerListenerPolicyOptionsActionRejectConst   = "reject"
)

// NewCreateLoadBalancerListenerPolicyOptions : Instantiate CreateLoadBalancerListenerPolicyOptions
func (*VpcClassicV1) NewCreateLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, action string, priority int64) *CreateLoadBalancerListenerPolicyOptions {
	return &CreateLoadBalancerListenerPolicyOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		Action:         core.StringPtr(action),
		Priority:       core.Int64Ptr(priority),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *CreateLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *CreateLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetAction : Allow user to set Action
func (options *CreateLoadBalancerListenerPolicyOptions) SetAction(action string) *CreateLoadBalancerListenerPolicyOptions {
	options.Action = core.StringPtr(action)
	return options
}

// SetPriority : Allow user to set Priority
func (options *CreateLoadBalancerListenerPolicyOptions) SetPriority(priority int64) *CreateLoadBalancerListenerPolicyOptions {
	options.Priority = core.Int64Ptr(priority)
	return options
}

// SetName : Allow user to set Name
func (options *CreateLoadBalancerListenerPolicyOptions) SetName(name string) *CreateLoadBalancerListenerPolicyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetRules : Allow user to set Rules
func (options *CreateLoadBalancerListenerPolicyOptions) SetRules(rules []LoadBalancerListenerPolicyRulePrototype) *CreateLoadBalancerListenerPolicyOptions {
	options.Rules = rules
	return options
}

// SetTarget : Allow user to set Target
func (options *CreateLoadBalancerListenerPolicyOptions) SetTarget(target LoadBalancerListenerPolicyTargetPrototypeIntf) *CreateLoadBalancerListenerPolicyOptions {
	options.Target = target
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *CreateLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerListenerPolicyRuleOptions : The CreateLoadBalancerListenerPolicyRule options.
type CreateLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The condition of the rule.
	Condition *string `validate:"required"`

	// The type of the rule.
	Type *string `validate:"required"`

	// Value to be matched for rule condition.
	Value *string `validate:"required"`

	// HTTP header field. This is only applicable to "header" rule type.
	Field *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerListenerPolicyRuleOptions.Condition property.
// The condition of the rule.
const (
	CreateLoadBalancerListenerPolicyRuleOptionsConditionContainsConst     = "contains"
	CreateLoadBalancerListenerPolicyRuleOptionsConditionEqualsConst       = "equals"
	CreateLoadBalancerListenerPolicyRuleOptionsConditionMatchesRegexConst = "matches_regex"
)

// Constants associated with the CreateLoadBalancerListenerPolicyRuleOptions.Type property.
// The type of the rule.
const (
	CreateLoadBalancerListenerPolicyRuleOptionsTypeHeaderConst   = "header"
	CreateLoadBalancerListenerPolicyRuleOptionsTypeHostnameConst = "hostname"
	CreateLoadBalancerListenerPolicyRuleOptionsTypePathConst     = "path"
)

// NewCreateLoadBalancerListenerPolicyRuleOptions : Instantiate CreateLoadBalancerListenerPolicyRuleOptions
func (*VpcClassicV1) NewCreateLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, condition string, typeVar string, value string) *CreateLoadBalancerListenerPolicyRuleOptions {
	return &CreateLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
		Condition:      core.StringPtr(condition),
		Type:           core.StringPtr(typeVar),
		Value:          core.StringPtr(value),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetCondition : Allow user to set Condition
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetCondition(condition string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Condition = core.StringPtr(condition)
	return options
}

// SetType : Allow user to set Type
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetType(typeVar string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetValue : Allow user to set Value
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetValue(value string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Value = core.StringPtr(value)
	return options
}

// SetField : Allow user to set Field
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetField(field string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Field = core.StringPtr(field)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerOptions : The CreateLoadBalancer options.
type CreateLoadBalancerOptions struct {
	// Indicates whether this load balancer is public or private.
	IsPublic *bool `validate:"required"`

	// The subnets to provision this load balancer.
	Subnets []SubnetIdentityIntf `validate:"required"`

	// The listeners of this load balancer.
	Listeners []LoadBalancerListenerPrototypeLoadBalancerContext

	// The logging configuration to use for this load balancer. See [VPC Datapath
	// Logging](https://cloud.ibm.com/docs/vpc?topic=vpc-datapath-logging)
	// on the logging format, fields and permitted values.
	//
	// To activate logging, the load balancer profile must support the specified logging
	// type.
	Logging *LoadBalancerLogging

	// The user-defined name for this load balancer. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The pools of this load balancer.
	Pools []LoadBalancerPoolPrototype

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateLoadBalancerOptions : Instantiate CreateLoadBalancerOptions
func (*VpcClassicV1) NewCreateLoadBalancerOptions(isPublic bool, subnets []SubnetIdentityIntf) *CreateLoadBalancerOptions {
	return &CreateLoadBalancerOptions{
		IsPublic: core.BoolPtr(isPublic),
		Subnets:  subnets,
	}
}

// SetIsPublic : Allow user to set IsPublic
func (options *CreateLoadBalancerOptions) SetIsPublic(isPublic bool) *CreateLoadBalancerOptions {
	options.IsPublic = core.BoolPtr(isPublic)
	return options
}

// SetSubnets : Allow user to set Subnets
func (options *CreateLoadBalancerOptions) SetSubnets(subnets []SubnetIdentityIntf) *CreateLoadBalancerOptions {
	options.Subnets = subnets
	return options
}

// SetListeners : Allow user to set Listeners
func (options *CreateLoadBalancerOptions) SetListeners(listeners []LoadBalancerListenerPrototypeLoadBalancerContext) *CreateLoadBalancerOptions {
	options.Listeners = listeners
	return options
}

// SetLogging : Allow user to set Logging
func (options *CreateLoadBalancerOptions) SetLogging(logging *LoadBalancerLogging) *CreateLoadBalancerOptions {
	options.Logging = logging
	return options
}

// SetName : Allow user to set Name
func (options *CreateLoadBalancerOptions) SetName(name string) *CreateLoadBalancerOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetPools : Allow user to set Pools
func (options *CreateLoadBalancerOptions) SetPools(pools []LoadBalancerPoolPrototype) *CreateLoadBalancerOptions {
	options.Pools = pools
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateLoadBalancerOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateLoadBalancerOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerOptions) SetHeaders(param map[string]string) *CreateLoadBalancerOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerPoolMemberOptions : The CreateLoadBalancerPoolMember options.
type CreateLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The port number of the application running in the server member.
	Port *int64 `validate:"required"`

	// The pool member target.
	Target LoadBalancerPoolMemberTargetPrototypeIntf `validate:"required"`

	// Weight of the server member. Applicable only if the pool algorithm is
	// `weighted_round_robin`.
	Weight *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateLoadBalancerPoolMemberOptions : Instantiate CreateLoadBalancerPoolMemberOptions
func (*VpcClassicV1) NewCreateLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, port int64, target LoadBalancerPoolMemberTargetPrototypeIntf) *CreateLoadBalancerPoolMemberOptions {
	return &CreateLoadBalancerPoolMemberOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		Port:           core.Int64Ptr(port),
		Target:         target,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *CreateLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *CreateLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetPort : Allow user to set Port
func (options *CreateLoadBalancerPoolMemberOptions) SetPort(port int64) *CreateLoadBalancerPoolMemberOptions {
	options.Port = core.Int64Ptr(port)
	return options
}

// SetTarget : Allow user to set Target
func (options *CreateLoadBalancerPoolMemberOptions) SetTarget(target LoadBalancerPoolMemberTargetPrototypeIntf) *CreateLoadBalancerPoolMemberOptions {
	options.Target = target
	return options
}

// SetWeight : Allow user to set Weight
func (options *CreateLoadBalancerPoolMemberOptions) SetWeight(weight int64) *CreateLoadBalancerPoolMemberOptions {
	options.Weight = core.Int64Ptr(weight)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *CreateLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerPoolOptions : The CreateLoadBalancerPool options.
type CreateLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The load balancing algorithm.
	Algorithm *string `validate:"required"`

	// The health monitor of this pool.
	HealthMonitor *LoadBalancerPoolHealthMonitorPrototype `validate:"required"`

	// The protocol used for this load balancer pool. Load balancers in the `network` family support `tcp`. Load balancers
	// in the `application` family support `tcp`, `http`, and
	// `https`.
	Protocol *string `validate:"required"`

	// The members for this load balancer pool. For load balancers in the `network` family, the same `port` and `target`
	// tuple cannot be shared by a pool member of any other load balancer in the same VPC.
	Members []LoadBalancerPoolMemberPrototype

	// The user-defined name for this load balancer pool. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The PROXY protocol setting for this pool:
	// - `v1`: Enabled with version 1 (human-readable header format)
	// - `v2`: Enabled with version 2 (binary header format)
	// - `disabled`: Disabled
	//
	// Supported by load balancers in the `application` family (otherwise always `disabled`).
	ProxyProtocol *string

	// The session persistence of this pool.
	SessionPersistence *LoadBalancerPoolSessionPersistencePrototype

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerPoolOptions.Algorithm property.
// The load balancing algorithm.
const (
	CreateLoadBalancerPoolOptionsAlgorithmLeastConnectionsConst   = "least_connections"
	CreateLoadBalancerPoolOptionsAlgorithmRoundRobinConst         = "round_robin"
	CreateLoadBalancerPoolOptionsAlgorithmWeightedRoundRobinConst = "weighted_round_robin"
)

// Constants associated with the CreateLoadBalancerPoolOptions.Protocol property.
// The protocol used for this load balancer pool. Load balancers in the `network` family support `tcp`. Load balancers
// in the `application` family support `tcp`, `http`, and
// `https`.
const (
	CreateLoadBalancerPoolOptionsProtocolHTTPConst  = "http"
	CreateLoadBalancerPoolOptionsProtocolHTTPSConst = "https"
	CreateLoadBalancerPoolOptionsProtocolTCPConst   = "tcp"
)

// Constants associated with the CreateLoadBalancerPoolOptions.ProxyProtocol property.
// The PROXY protocol setting for this pool:
// - `v1`: Enabled with version 1 (human-readable header format)
// - `v2`: Enabled with version 2 (binary header format)
// - `disabled`: Disabled
//
// Supported by load balancers in the `application` family (otherwise always `disabled`).
const (
	CreateLoadBalancerPoolOptionsProxyProtocolDisabledConst = "disabled"
	CreateLoadBalancerPoolOptionsProxyProtocolV1Const       = "v1"
	CreateLoadBalancerPoolOptionsProxyProtocolV2Const       = "v2"
)

// NewCreateLoadBalancerPoolOptions : Instantiate CreateLoadBalancerPoolOptions
func (*VpcClassicV1) NewCreateLoadBalancerPoolOptions(loadBalancerID string, algorithm string, healthMonitor *LoadBalancerPoolHealthMonitorPrototype, protocol string) *CreateLoadBalancerPoolOptions {
	return &CreateLoadBalancerPoolOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		Algorithm:      core.StringPtr(algorithm),
		HealthMonitor:  healthMonitor,
		Protocol:       core.StringPtr(protocol),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetAlgorithm : Allow user to set Algorithm
func (options *CreateLoadBalancerPoolOptions) SetAlgorithm(algorithm string) *CreateLoadBalancerPoolOptions {
	options.Algorithm = core.StringPtr(algorithm)
	return options
}

// SetHealthMonitor : Allow user to set HealthMonitor
func (options *CreateLoadBalancerPoolOptions) SetHealthMonitor(healthMonitor *LoadBalancerPoolHealthMonitorPrototype) *CreateLoadBalancerPoolOptions {
	options.HealthMonitor = healthMonitor
	return options
}

// SetProtocol : Allow user to set Protocol
func (options *CreateLoadBalancerPoolOptions) SetProtocol(protocol string) *CreateLoadBalancerPoolOptions {
	options.Protocol = core.StringPtr(protocol)
	return options
}

// SetMembers : Allow user to set Members
func (options *CreateLoadBalancerPoolOptions) SetMembers(members []LoadBalancerPoolMemberPrototype) *CreateLoadBalancerPoolOptions {
	options.Members = members
	return options
}

// SetName : Allow user to set Name
func (options *CreateLoadBalancerPoolOptions) SetName(name string) *CreateLoadBalancerPoolOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetProxyProtocol : Allow user to set ProxyProtocol
func (options *CreateLoadBalancerPoolOptions) SetProxyProtocol(proxyProtocol string) *CreateLoadBalancerPoolOptions {
	options.ProxyProtocol = core.StringPtr(proxyProtocol)
	return options
}

// SetSessionPersistence : Allow user to set SessionPersistence
func (options *CreateLoadBalancerPoolOptions) SetSessionPersistence(sessionPersistence *LoadBalancerPoolSessionPersistencePrototype) *CreateLoadBalancerPoolOptions {
	options.SessionPersistence = sessionPersistence
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerPoolOptions) SetHeaders(param map[string]string) *CreateLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// CreateNetworkACLOptions : The CreateNetworkACL options.
type CreateNetworkACLOptions struct {
	// The network ACL prototype object.
	NetworkACLPrototype NetworkACLPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateNetworkACLOptions : Instantiate CreateNetworkACLOptions
func (*VpcClassicV1) NewCreateNetworkACLOptions() *CreateNetworkACLOptions {
	return &CreateNetworkACLOptions{}
}

// SetNetworkACLPrototype : Allow user to set NetworkACLPrototype
func (options *CreateNetworkACLOptions) SetNetworkACLPrototype(networkACLPrototype NetworkACLPrototypeIntf) *CreateNetworkACLOptions {
	options.NetworkACLPrototype = networkACLPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateNetworkACLOptions) SetHeaders(param map[string]string) *CreateNetworkACLOptions {
	options.Headers = param
	return options
}

// CreateNetworkACLRuleOptions : The CreateNetworkACLRule options.
type CreateNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The network ACL rule prototype object.
	NetworkACLRulePrototype NetworkACLRulePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateNetworkACLRuleOptions : Instantiate CreateNetworkACLRuleOptions
func (*VpcClassicV1) NewCreateNetworkACLRuleOptions(networkACLID string, networkACLRulePrototype NetworkACLRulePrototypeIntf) *CreateNetworkACLRuleOptions {
	return &CreateNetworkACLRuleOptions{
		NetworkACLID:            core.StringPtr(networkACLID),
		NetworkACLRulePrototype: networkACLRulePrototype,
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *CreateNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *CreateNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetNetworkACLRulePrototype : Allow user to set NetworkACLRulePrototype
func (options *CreateNetworkACLRuleOptions) SetNetworkACLRulePrototype(networkACLRulePrototype NetworkACLRulePrototypeIntf) *CreateNetworkACLRuleOptions {
	options.NetworkACLRulePrototype = networkACLRulePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateNetworkACLRuleOptions) SetHeaders(param map[string]string) *CreateNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// CreatePublicGatewayOptions : The CreatePublicGateway options.
type CreatePublicGatewayOptions struct {
	// The VPC this public gateway will serve.
	VPC VPCIdentityIntf `validate:"required"`

	// The zone where this public gateway will be created.
	Zone ZoneIdentityIntf `validate:"required"`

	FloatingIP PublicGatewayFloatingIPPrototypeIntf

	// The user-defined name for this public gateway. Names must be unique within the VPC the public gateway resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreatePublicGatewayOptions : Instantiate CreatePublicGatewayOptions
func (*VpcClassicV1) NewCreatePublicGatewayOptions(vpc VPCIdentityIntf, zone ZoneIdentityIntf) *CreatePublicGatewayOptions {
	return &CreatePublicGatewayOptions{
		VPC:  vpc,
		Zone: zone,
	}
}

// SetVPC : Allow user to set VPC
func (options *CreatePublicGatewayOptions) SetVPC(vpc VPCIdentityIntf) *CreatePublicGatewayOptions {
	options.VPC = vpc
	return options
}

// SetZone : Allow user to set Zone
func (options *CreatePublicGatewayOptions) SetZone(zone ZoneIdentityIntf) *CreatePublicGatewayOptions {
	options.Zone = zone
	return options
}

// SetFloatingIP : Allow user to set FloatingIP
func (options *CreatePublicGatewayOptions) SetFloatingIP(floatingIP PublicGatewayFloatingIPPrototypeIntf) *CreatePublicGatewayOptions {
	options.FloatingIP = floatingIP
	return options
}

// SetName : Allow user to set Name
func (options *CreatePublicGatewayOptions) SetName(name string) *CreatePublicGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreatePublicGatewayOptions) SetHeaders(param map[string]string) *CreatePublicGatewayOptions {
	options.Headers = param
	return options
}

// CreateSecurityGroupOptions : The CreateSecurityGroup options.
type CreateSecurityGroupOptions struct {
	// The VPC this security group is to be a part of.
	VPC VPCIdentityIntf `validate:"required"`

	// The user-defined name for this security group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words. Security group names must be unique, within the scope of an account.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Array of rule prototype objects for rules to be created for this security group. If unspecified, no rules will be
	// created, resulting in all traffic being denied.
	Rules []SecurityGroupRulePrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSecurityGroupOptions : Instantiate CreateSecurityGroupOptions
func (*VpcClassicV1) NewCreateSecurityGroupOptions(vpc VPCIdentityIntf) *CreateSecurityGroupOptions {
	return &CreateSecurityGroupOptions{
		VPC: vpc,
	}
}

// SetVPC : Allow user to set VPC
func (options *CreateSecurityGroupOptions) SetVPC(vpc VPCIdentityIntf) *CreateSecurityGroupOptions {
	options.VPC = vpc
	return options
}

// SetName : Allow user to set Name
func (options *CreateSecurityGroupOptions) SetName(name string) *CreateSecurityGroupOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateSecurityGroupOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateSecurityGroupOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetRules : Allow user to set Rules
func (options *CreateSecurityGroupOptions) SetRules(rules []SecurityGroupRulePrototypeIntf) *CreateSecurityGroupOptions {
	options.Rules = rules
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSecurityGroupOptions) SetHeaders(param map[string]string) *CreateSecurityGroupOptions {
	options.Headers = param
	return options
}

// CreateSecurityGroupRuleOptions : The CreateSecurityGroupRule options.
type CreateSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The properties of the security group rule to be created.
	SecurityGroupRulePrototype SecurityGroupRulePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSecurityGroupRuleOptions : Instantiate CreateSecurityGroupRuleOptions
func (*VpcClassicV1) NewCreateSecurityGroupRuleOptions(securityGroupID string, securityGroupRulePrototype SecurityGroupRulePrototypeIntf) *CreateSecurityGroupRuleOptions {
	return &CreateSecurityGroupRuleOptions{
		SecurityGroupID:            core.StringPtr(securityGroupID),
		SecurityGroupRulePrototype: securityGroupRulePrototype,
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *CreateSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *CreateSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetSecurityGroupRulePrototype : Allow user to set SecurityGroupRulePrototype
func (options *CreateSecurityGroupRuleOptions) SetSecurityGroupRulePrototype(securityGroupRulePrototype SecurityGroupRulePrototypeIntf) *CreateSecurityGroupRuleOptions {
	options.SecurityGroupRulePrototype = securityGroupRulePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSecurityGroupRuleOptions) SetHeaders(param map[string]string) *CreateSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// CreateSubnetOptions : The CreateSubnet options.
type CreateSubnetOptions struct {
	// The subnet prototype object.
	SubnetPrototype SubnetPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSubnetOptions : Instantiate CreateSubnetOptions
func (*VpcClassicV1) NewCreateSubnetOptions(subnetPrototype SubnetPrototypeIntf) *CreateSubnetOptions {
	return &CreateSubnetOptions{
		SubnetPrototype: subnetPrototype,
	}
}

// SetSubnetPrototype : Allow user to set SubnetPrototype
func (options *CreateSubnetOptions) SetSubnetPrototype(subnetPrototype SubnetPrototypeIntf) *CreateSubnetOptions {
	options.SubnetPrototype = subnetPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSubnetOptions) SetHeaders(param map[string]string) *CreateSubnetOptions {
	options.Headers = param
	return options
}

// CreateVolumeOptions : The CreateVolume options.
type CreateVolumeOptions struct {
	// The volume prototype object.
	VolumePrototype VolumePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVolumeOptions : Instantiate CreateVolumeOptions
func (*VpcClassicV1) NewCreateVolumeOptions(volumePrototype VolumePrototypeIntf) *CreateVolumeOptions {
	return &CreateVolumeOptions{
		VolumePrototype: volumePrototype,
	}
}

// SetVolumePrototype : Allow user to set VolumePrototype
func (options *CreateVolumeOptions) SetVolumePrototype(volumePrototype VolumePrototypeIntf) *CreateVolumeOptions {
	options.VolumePrototype = volumePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVolumeOptions) SetHeaders(param map[string]string) *CreateVolumeOptions {
	options.Headers = param
	return options
}

// CreateVPCAddressPrefixOptions : The CreateVPCAddressPrefix options.
type CreateVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The IPv4 range of the address prefix, expressed in CIDR format. The request must not overlap with any existing
	// address prefixes in the VPC or any of the following reserved address ranges:
	//   - `127.0.0.0/8` (IPv4 loopback addresses)
	//   - `161.26.0.0/16` (IBM services)
	//   - `166.8.0.0/14` (Cloud Service Endpoints)
	//   - `169.254.0.0/16` (IPv4 link-local addresses)
	//   - `224.0.0.0/4` (IPv4 multicast addresses)
	//
	// The prefix length of the address prefix's CIDR must be between `/9` (8,388,608 addresses) and `/29` (8 addresses).
	CIDR *string `validate:"required"`

	// The zone this address prefix is to belong to.
	Zone ZoneIdentityIntf `validate:"required"`

	// Indicates whether this is the default prefix for this zone in this VPC. If true, this prefix will become the default
	// prefix for this zone in this VPC. This fails if the VPC currently has a default address prefix for this zone.
	IsDefault *bool

	// The user-defined name for this address prefix. Names must be unique within the VPC the address prefix resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPCAddressPrefixOptions : Instantiate CreateVPCAddressPrefixOptions
func (*VpcClassicV1) NewCreateVPCAddressPrefixOptions(vpcID string, cidr string, zone ZoneIdentityIntf) *CreateVPCAddressPrefixOptions {
	return &CreateVPCAddressPrefixOptions{
		VPCID: core.StringPtr(vpcID),
		CIDR:  core.StringPtr(cidr),
		Zone:  zone,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *CreateVPCAddressPrefixOptions) SetVPCID(vpcID string) *CreateVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetCIDR : Allow user to set CIDR
func (options *CreateVPCAddressPrefixOptions) SetCIDR(cidr string) *CreateVPCAddressPrefixOptions {
	options.CIDR = core.StringPtr(cidr)
	return options
}

// SetZone : Allow user to set Zone
func (options *CreateVPCAddressPrefixOptions) SetZone(zone ZoneIdentityIntf) *CreateVPCAddressPrefixOptions {
	options.Zone = zone
	return options
}

// SetIsDefault : Allow user to set IsDefault
func (options *CreateVPCAddressPrefixOptions) SetIsDefault(isDefault bool) *CreateVPCAddressPrefixOptions {
	options.IsDefault = core.BoolPtr(isDefault)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCAddressPrefixOptions) SetName(name string) *CreateVPCAddressPrefixOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCAddressPrefixOptions) SetHeaders(param map[string]string) *CreateVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// CreateVPCOptions : The CreateVPC options.
type CreateVPCOptions struct {
	// Indicates whether a default address prefix should be automatically created for each zone in this VPC. If `manual`,
	// this VPC will be created with no default address prefixes.
	AddressPrefixManagement *string

	// Indicates whether this VPC should be connected to Classic Infrastructure. If true, this VPC's resources will have
	// private network connectivity to the account's Classic Infrastructure resources. Only one VPC, per region, may be
	// connected in this way. This value is set at creation and subsequently immutable.
	ClassicAccess *bool

	// The unique user-defined name for this VPC. If unspecified, the name will be a hyphenated list of randomly-selected
	// words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateVPCOptions.AddressPrefixManagement property.
// Indicates whether a default address prefix should be automatically created for each zone in this VPC. If `manual`,
// this VPC will be created with no default address prefixes.
const (
	CreateVPCOptionsAddressPrefixManagementAutoConst   = "auto"
	CreateVPCOptionsAddressPrefixManagementManualConst = "manual"
)

// NewCreateVPCOptions : Instantiate CreateVPCOptions
func (*VpcClassicV1) NewCreateVPCOptions() *CreateVPCOptions {
	return &CreateVPCOptions{}
}

// SetAddressPrefixManagement : Allow user to set AddressPrefixManagement
func (options *CreateVPCOptions) SetAddressPrefixManagement(addressPrefixManagement string) *CreateVPCOptions {
	options.AddressPrefixManagement = core.StringPtr(addressPrefixManagement)
	return options
}

// SetClassicAccess : Allow user to set ClassicAccess
func (options *CreateVPCOptions) SetClassicAccess(classicAccess bool) *CreateVPCOptions {
	options.ClassicAccess = core.BoolPtr(classicAccess)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCOptions) SetName(name string) *CreateVPCOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateVPCOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateVPCOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCOptions) SetHeaders(param map[string]string) *CreateVPCOptions {
	options.Headers = param
	return options
}

// CreateVPCRouteOptions : The CreateVPCRoute options.
type CreateVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The destination of the route. Must not overlap with destinations for existing user-defined routes within the VPC.
	Destination *string `validate:"required"`

	// The next hop that packets will be delivered to.
	NextHop RouteNextHopPrototypeIntf `validate:"required"`

	// The zone to apply the route to. (Traffic from subnets in this zone will be
	// subject to this route.).
	Zone ZoneIdentityIntf `validate:"required"`

	// The user-defined name for this route. If unspecified, the name will be a hyphenated list of randomly-selected words.
	// Names must be unique within the VPC routing table the route resides in.
	Name *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPCRouteOptions : Instantiate CreateVPCRouteOptions
func (*VpcClassicV1) NewCreateVPCRouteOptions(vpcID string, destination string, nextHop RouteNextHopPrototypeIntf, zone ZoneIdentityIntf) *CreateVPCRouteOptions {
	return &CreateVPCRouteOptions{
		VPCID:       core.StringPtr(vpcID),
		Destination: core.StringPtr(destination),
		NextHop:     nextHop,
		Zone:        zone,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *CreateVPCRouteOptions) SetVPCID(vpcID string) *CreateVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetDestination : Allow user to set Destination
func (options *CreateVPCRouteOptions) SetDestination(destination string) *CreateVPCRouteOptions {
	options.Destination = core.StringPtr(destination)
	return options
}

// SetNextHop : Allow user to set NextHop
func (options *CreateVPCRouteOptions) SetNextHop(nextHop RouteNextHopPrototypeIntf) *CreateVPCRouteOptions {
	options.NextHop = nextHop
	return options
}

// SetZone : Allow user to set Zone
func (options *CreateVPCRouteOptions) SetZone(zone ZoneIdentityIntf) *CreateVPCRouteOptions {
	options.Zone = zone
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCRouteOptions) SetName(name string) *CreateVPCRouteOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCRouteOptions) SetHeaders(param map[string]string) *CreateVPCRouteOptions {
	options.Headers = param
	return options
}

// CreateVPNGatewayConnectionOptions : The CreateVPNGatewayConnection options.
type CreateVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection prototype object.
	VPNGatewayConnectionPrototype VPNGatewayConnectionPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPNGatewayConnectionOptions : Instantiate CreateVPNGatewayConnectionOptions
func (*VpcClassicV1) NewCreateVPNGatewayConnectionOptions(vpnGatewayID string, vpnGatewayConnectionPrototype VPNGatewayConnectionPrototypeIntf) *CreateVPNGatewayConnectionOptions {
	return &CreateVPNGatewayConnectionOptions{
		VPNGatewayID:                  core.StringPtr(vpnGatewayID),
		VPNGatewayConnectionPrototype: vpnGatewayConnectionPrototype,
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *CreateVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID string) *CreateVPNGatewayConnectionOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetVPNGatewayConnectionPrototype : Allow user to set VPNGatewayConnectionPrototype
func (options *CreateVPNGatewayConnectionOptions) SetVPNGatewayConnectionPrototype(vpnGatewayConnectionPrototype VPNGatewayConnectionPrototypeIntf) *CreateVPNGatewayConnectionOptions {
	options.VPNGatewayConnectionPrototype = vpnGatewayConnectionPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPNGatewayConnectionOptions) SetHeaders(param map[string]string) *CreateVPNGatewayConnectionOptions {
	options.Headers = param
	return options
}

// CreateVPNGatewayOptions : The CreateVPNGateway options.
type CreateVPNGatewayOptions struct {
	// The VPN gateway prototype object.
	VPNGatewayPrototype VPNGatewayPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPNGatewayOptions : Instantiate CreateVPNGatewayOptions
func (*VpcClassicV1) NewCreateVPNGatewayOptions(vpnGatewayPrototype VPNGatewayPrototypeIntf) *CreateVPNGatewayOptions {
	return &CreateVPNGatewayOptions{
		VPNGatewayPrototype: vpnGatewayPrototype,
	}
}

// SetVPNGatewayPrototype : Allow user to set VPNGatewayPrototype
func (options *CreateVPNGatewayOptions) SetVPNGatewayPrototype(vpnGatewayPrototype VPNGatewayPrototypeIntf) *CreateVPNGatewayOptions {
	options.VPNGatewayPrototype = vpnGatewayPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPNGatewayOptions) SetHeaders(param map[string]string) *CreateVPNGatewayOptions {
	options.Headers = param
	return options
}

// DefaultSecurityGroup : DefaultSecurityGroup struct
type DefaultSecurityGroup struct {
	// The date and time that this security group was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The security group's CRN.
	CRN *string `json:"crn" validate:"required"`

	// The security group's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this security group.
	ID *string `json:"id" validate:"required"`

	// The name of the default security group created for a VPC. The name will be a hyphenated list of randomly-selected
	// words at creation, but may be user-specified with a subsequent request.
	Name *string `json:"name" validate:"required"`

	// Array of rules for the default security group for a VPC. Defaults to allowing all outbound traffic, and allowing all
	// inbound traffic from other interfaces in the VPC's default security group. Rules in the default security group may
	// be changed, added or removed.
	Rules []SecurityGroupRuleIntf `json:"rules" validate:"required"`

	// The VPC this security group is a part of.
	VPC *VPCReference `json:"vpc" validate:"required"`
}

// UnmarshalDefaultSecurityGroup unmarshals an instance of DefaultSecurityGroup from the specified map of raw messages.
func UnmarshalDefaultSecurityGroup(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DefaultSecurityGroup)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DeleteFloatingIPOptions : The DeleteFloatingIP options.
type DeleteFloatingIPOptions struct {
	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteFloatingIPOptions : Instantiate DeleteFloatingIPOptions
func (*VpcClassicV1) NewDeleteFloatingIPOptions(id string) *DeleteFloatingIPOptions {
	return &DeleteFloatingIPOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteFloatingIPOptions) SetID(id string) *DeleteFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteFloatingIPOptions) SetHeaders(param map[string]string) *DeleteFloatingIPOptions {
	options.Headers = param
	return options
}

// DeleteIkePolicyOptions : The DeleteIkePolicy options.
type DeleteIkePolicyOptions struct {
	// The IKE policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteIkePolicyOptions : Instantiate DeleteIkePolicyOptions
func (*VpcClassicV1) NewDeleteIkePolicyOptions(id string) *DeleteIkePolicyOptions {
	return &DeleteIkePolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteIkePolicyOptions) SetID(id string) *DeleteIkePolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteIkePolicyOptions) SetHeaders(param map[string]string) *DeleteIkePolicyOptions {
	options.Headers = param
	return options
}

// DeleteImageOptions : The DeleteImage options.
type DeleteImageOptions struct {
	// The image identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteImageOptions : Instantiate DeleteImageOptions
func (*VpcClassicV1) NewDeleteImageOptions(id string) *DeleteImageOptions {
	return &DeleteImageOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteImageOptions) SetID(id string) *DeleteImageOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteImageOptions) SetHeaders(param map[string]string) *DeleteImageOptions {
	options.Headers = param
	return options
}

// DeleteInstanceOptions : The DeleteInstance options.
type DeleteInstanceOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceOptions : Instantiate DeleteInstanceOptions
func (*VpcClassicV1) NewDeleteInstanceOptions(id string) *DeleteInstanceOptions {
	return &DeleteInstanceOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteInstanceOptions) SetID(id string) *DeleteInstanceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceOptions) SetHeaders(param map[string]string) *DeleteInstanceOptions {
	options.Headers = param
	return options
}

// DeleteInstanceVolumeAttachmentOptions : The DeleteInstanceVolumeAttachment options.
type DeleteInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The volume attachment identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceVolumeAttachmentOptions : Instantiate DeleteInstanceVolumeAttachmentOptions
func (*VpcClassicV1) NewDeleteInstanceVolumeAttachmentOptions(instanceID string, id string) *DeleteInstanceVolumeAttachmentOptions {
	return &DeleteInstanceVolumeAttachmentOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *DeleteInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *DeleteInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceVolumeAttachmentOptions) SetID(id string) *DeleteInstanceVolumeAttachmentOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *DeleteInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// DeleteIpsecPolicyOptions : The DeleteIpsecPolicy options.
type DeleteIpsecPolicyOptions struct {
	// The IPsec policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteIpsecPolicyOptions : Instantiate DeleteIpsecPolicyOptions
func (*VpcClassicV1) NewDeleteIpsecPolicyOptions(id string) *DeleteIpsecPolicyOptions {
	return &DeleteIpsecPolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteIpsecPolicyOptions) SetID(id string) *DeleteIpsecPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteIpsecPolicyOptions) SetHeaders(param map[string]string) *DeleteIpsecPolicyOptions {
	options.Headers = param
	return options
}

// DeleteKeyOptions : The DeleteKey options.
type DeleteKeyOptions struct {
	// The key identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteKeyOptions : Instantiate DeleteKeyOptions
func (*VpcClassicV1) NewDeleteKeyOptions(id string) *DeleteKeyOptions {
	return &DeleteKeyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteKeyOptions) SetID(id string) *DeleteKeyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteKeyOptions) SetHeaders(param map[string]string) *DeleteKeyOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerListenerOptions : The DeleteLoadBalancerListener options.
type DeleteLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerListenerOptions : Instantiate DeleteLoadBalancerListenerOptions
func (*VpcClassicV1) NewDeleteLoadBalancerListenerOptions(loadBalancerID string, id string) *DeleteLoadBalancerListenerOptions {
	return &DeleteLoadBalancerListenerOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerListenerOptions) SetID(id string) *DeleteLoadBalancerListenerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerListenerOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerListenerPolicyOptions : The DeleteLoadBalancerListenerPolicy options.
type DeleteLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerListenerPolicyOptions : Instantiate DeleteLoadBalancerListenerPolicyOptions
func (*VpcClassicV1) NewDeleteLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, id string) *DeleteLoadBalancerListenerPolicyOptions {
	return &DeleteLoadBalancerListenerPolicyOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *DeleteLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *DeleteLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerListenerPolicyOptions) SetID(id string) *DeleteLoadBalancerListenerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerListenerPolicyRuleOptions : The DeleteLoadBalancerListenerPolicyRule options.
type DeleteLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerListenerPolicyRuleOptions : Instantiate DeleteLoadBalancerListenerPolicyRuleOptions
func (*VpcClassicV1) NewDeleteLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, id string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	return &DeleteLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetID(id string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerOptions : The DeleteLoadBalancer options.
type DeleteLoadBalancerOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerOptions : Instantiate DeleteLoadBalancerOptions
func (*VpcClassicV1) NewDeleteLoadBalancerOptions(id string) *DeleteLoadBalancerOptions {
	return &DeleteLoadBalancerOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerOptions) SetID(id string) *DeleteLoadBalancerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerPoolMemberOptions : The DeleteLoadBalancerPoolMember options.
type DeleteLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The member identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerPoolMemberOptions : Instantiate DeleteLoadBalancerPoolMemberOptions
func (*VpcClassicV1) NewDeleteLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, id string) *DeleteLoadBalancerPoolMemberOptions {
	return &DeleteLoadBalancerPoolMemberOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *DeleteLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *DeleteLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerPoolMemberOptions) SetID(id string) *DeleteLoadBalancerPoolMemberOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerPoolOptions : The DeleteLoadBalancerPool options.
type DeleteLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerPoolOptions : Instantiate DeleteLoadBalancerPoolOptions
func (*VpcClassicV1) NewDeleteLoadBalancerPoolOptions(loadBalancerID string, id string) *DeleteLoadBalancerPoolOptions {
	return &DeleteLoadBalancerPoolOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerPoolOptions) SetID(id string) *DeleteLoadBalancerPoolOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerPoolOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// DeleteNetworkACLOptions : The DeleteNetworkACL options.
type DeleteNetworkACLOptions struct {
	// The network ACL identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteNetworkACLOptions : Instantiate DeleteNetworkACLOptions
func (*VpcClassicV1) NewDeleteNetworkACLOptions(id string) *DeleteNetworkACLOptions {
	return &DeleteNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteNetworkACLOptions) SetID(id string) *DeleteNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteNetworkACLOptions) SetHeaders(param map[string]string) *DeleteNetworkACLOptions {
	options.Headers = param
	return options
}

// DeleteNetworkACLRuleOptions : The DeleteNetworkACLRule options.
type DeleteNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteNetworkACLRuleOptions : Instantiate DeleteNetworkACLRuleOptions
func (*VpcClassicV1) NewDeleteNetworkACLRuleOptions(networkACLID string, id string) *DeleteNetworkACLRuleOptions {
	return &DeleteNetworkACLRuleOptions{
		NetworkACLID: core.StringPtr(networkACLID),
		ID:           core.StringPtr(id),
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *DeleteNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *DeleteNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteNetworkACLRuleOptions) SetID(id string) *DeleteNetworkACLRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteNetworkACLRuleOptions) SetHeaders(param map[string]string) *DeleteNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// DeletePublicGatewayOptions : The DeletePublicGateway options.
type DeletePublicGatewayOptions struct {
	// The public gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeletePublicGatewayOptions : Instantiate DeletePublicGatewayOptions
func (*VpcClassicV1) NewDeletePublicGatewayOptions(id string) *DeletePublicGatewayOptions {
	return &DeletePublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeletePublicGatewayOptions) SetID(id string) *DeletePublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeletePublicGatewayOptions) SetHeaders(param map[string]string) *DeletePublicGatewayOptions {
	options.Headers = param
	return options
}

// DeleteSecurityGroupOptions : The DeleteSecurityGroup options.
type DeleteSecurityGroupOptions struct {
	// The security group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSecurityGroupOptions : Instantiate DeleteSecurityGroupOptions
func (*VpcClassicV1) NewDeleteSecurityGroupOptions(id string) *DeleteSecurityGroupOptions {
	return &DeleteSecurityGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteSecurityGroupOptions) SetID(id string) *DeleteSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSecurityGroupOptions) SetHeaders(param map[string]string) *DeleteSecurityGroupOptions {
	options.Headers = param
	return options
}

// DeleteSecurityGroupRuleOptions : The DeleteSecurityGroupRule options.
type DeleteSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSecurityGroupRuleOptions : Instantiate DeleteSecurityGroupRuleOptions
func (*VpcClassicV1) NewDeleteSecurityGroupRuleOptions(securityGroupID string, id string) *DeleteSecurityGroupRuleOptions {
	return &DeleteSecurityGroupRuleOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *DeleteSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *DeleteSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteSecurityGroupRuleOptions) SetID(id string) *DeleteSecurityGroupRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSecurityGroupRuleOptions) SetHeaders(param map[string]string) *DeleteSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// DeleteSubnetOptions : The DeleteSubnet options.
type DeleteSubnetOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSubnetOptions : Instantiate DeleteSubnetOptions
func (*VpcClassicV1) NewDeleteSubnetOptions(id string) *DeleteSubnetOptions {
	return &DeleteSubnetOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteSubnetOptions) SetID(id string) *DeleteSubnetOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSubnetOptions) SetHeaders(param map[string]string) *DeleteSubnetOptions {
	options.Headers = param
	return options
}

// DeleteVolumeOptions : The DeleteVolume options.
type DeleteVolumeOptions struct {
	// The volume identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVolumeOptions : Instantiate DeleteVolumeOptions
func (*VpcClassicV1) NewDeleteVolumeOptions(id string) *DeleteVolumeOptions {
	return &DeleteVolumeOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteVolumeOptions) SetID(id string) *DeleteVolumeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVolumeOptions) SetHeaders(param map[string]string) *DeleteVolumeOptions {
	options.Headers = param
	return options
}

// DeleteVPCAddressPrefixOptions : The DeleteVPCAddressPrefix options.
type DeleteVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The prefix identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCAddressPrefixOptions : Instantiate DeleteVPCAddressPrefixOptions
func (*VpcClassicV1) NewDeleteVPCAddressPrefixOptions(vpcID string, id string) *DeleteVPCAddressPrefixOptions {
	return &DeleteVPCAddressPrefixOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *DeleteVPCAddressPrefixOptions) SetVPCID(vpcID string) *DeleteVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPCAddressPrefixOptions) SetID(id string) *DeleteVPCAddressPrefixOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCAddressPrefixOptions) SetHeaders(param map[string]string) *DeleteVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// DeleteVPCOptions : The DeleteVPC options.
type DeleteVPCOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCOptions : Instantiate DeleteVPCOptions
func (*VpcClassicV1) NewDeleteVPCOptions(id string) *DeleteVPCOptions {
	return &DeleteVPCOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteVPCOptions) SetID(id string) *DeleteVPCOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCOptions) SetHeaders(param map[string]string) *DeleteVPCOptions {
	options.Headers = param
	return options
}

// DeleteVPCRouteOptions : The DeleteVPCRoute options.
type DeleteVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The route identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCRouteOptions : Instantiate DeleteVPCRouteOptions
func (*VpcClassicV1) NewDeleteVPCRouteOptions(vpcID string, id string) *DeleteVPCRouteOptions {
	return &DeleteVPCRouteOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *DeleteVPCRouteOptions) SetVPCID(vpcID string) *DeleteVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPCRouteOptions) SetID(id string) *DeleteVPCRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCRouteOptions) SetHeaders(param map[string]string) *DeleteVPCRouteOptions {
	options.Headers = param
	return options
}

// DeleteVPNGatewayConnectionOptions : The DeleteVPNGatewayConnection options.
type DeleteVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPNGatewayConnectionOptions : Instantiate DeleteVPNGatewayConnectionOptions
func (*VpcClassicV1) NewDeleteVPNGatewayConnectionOptions(vpnGatewayID string, id string) *DeleteVPNGatewayConnectionOptions {
	return &DeleteVPNGatewayConnectionOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *DeleteVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID string) *DeleteVPNGatewayConnectionOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPNGatewayConnectionOptions) SetID(id string) *DeleteVPNGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPNGatewayConnectionOptions) SetHeaders(param map[string]string) *DeleteVPNGatewayConnectionOptions {
	options.Headers = param
	return options
}

// DeleteVPNGatewayOptions : The DeleteVPNGateway options.
type DeleteVPNGatewayOptions struct {
	// The VPN gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPNGatewayOptions : Instantiate DeleteVPNGatewayOptions
func (*VpcClassicV1) NewDeleteVPNGatewayOptions(id string) *DeleteVPNGatewayOptions {
	return &DeleteVPNGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteVPNGatewayOptions) SetID(id string) *DeleteVPNGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPNGatewayOptions) SetHeaders(param map[string]string) *DeleteVPNGatewayOptions {
	options.Headers = param
	return options
}

// EncryptionKeyIdentity : Identifies an encryption key by a unique property.
// Models which "extend" this model:
// - EncryptionKeyIdentityByCRN
type EncryptionKeyIdentity struct {
	// The CRN of the [Key Protect Root
	// Key](https://cloud.ibm.com/docs/key-protect?topic=key-protect-getting-started-tutorial) or [Hyper Protect Crypto
	// Service Root Key](https://cloud.ibm.com/docs/hs-crypto?topic=hs-crypto-get-started) for this resource.
	CRN *string `json:"crn,omitempty"`
}

func (*EncryptionKeyIdentity) isaEncryptionKeyIdentity() bool {
	return true
}

type EncryptionKeyIdentityIntf interface {
	isaEncryptionKeyIdentity() bool
}

// UnmarshalEncryptionKeyIdentity unmarshals an instance of EncryptionKeyIdentity from the specified map of raw messages.
func UnmarshalEncryptionKeyIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EncryptionKeyIdentity)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EncryptionKeyReference : EncryptionKeyReference struct
type EncryptionKeyReference struct {
	// The CRN of the [Key Protect Root
	// Key](https://cloud.ibm.com/docs/key-protect?topic=key-protect-getting-started-tutorial) or [Hyper Protect Crypto
	// Service Root Key](https://cloud.ibm.com/docs/hs-crypto?topic=hs-crypto-get-started) for this resource.
	CRN *string `json:"crn" validate:"required"`
}

// UnmarshalEncryptionKeyReference unmarshals an instance of EncryptionKeyReference from the specified map of raw messages.
func UnmarshalEncryptionKeyReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EncryptionKeyReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIP : FloatingIP struct
type FloatingIP struct {
	// The globally unique IP address.
	Address *string `json:"address" validate:"required"`

	// The date and time that the floating IP was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this floating IP.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this floating IP.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this floating IP.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this floating IP.
	Name *string `json:"name" validate:"required"`

	// The status of the floating IP.
	Status *string `json:"status" validate:"required"`

	// The target of this floating IP.
	Target FloatingIPTargetIntf `json:"target,omitempty"`

	// The zone the floating IP resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the FloatingIP.Status property.
// The status of the floating IP.
const (
	FloatingIPStatusAvailableConst = "available"
	FloatingIPStatusDeletingConst  = "deleting"
	FloatingIPStatusFailedConst    = "failed"
	FloatingIPStatusPendingConst   = "pending"
)

// UnmarshalFloatingIP unmarshals an instance of FloatingIP from the specified map of raw messages.
func UnmarshalFloatingIP(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIP)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFloatingIPTarget)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPByTargetNetworkInterfaceIdentity : The network interface this floating IP is to be bound to.
// Models which "extend" this model:
// - FloatingIPByTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID
// - FloatingIPByTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByHref
type FloatingIPByTargetNetworkInterfaceIdentity struct {
	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`
}

func (*FloatingIPByTargetNetworkInterfaceIdentity) isaFloatingIPByTargetNetworkInterfaceIdentity() bool {
	return true
}

type FloatingIPByTargetNetworkInterfaceIdentityIntf interface {
	isaFloatingIPByTargetNetworkInterfaceIdentity() bool
}

// UnmarshalFloatingIPByTargetNetworkInterfaceIdentity unmarshals an instance of FloatingIPByTargetNetworkInterfaceIdentity from the specified map of raw messages.
func UnmarshalFloatingIPByTargetNetworkInterfaceIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPByTargetNetworkInterfaceIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPCollection : FloatingIPCollection struct
type FloatingIPCollection struct {
	// A link to the first page of resources.
	First *FloatingIPCollectionFirst `json:"first" validate:"required"`

	// Collection of floating IPs.
	FloatingIps []FloatingIP `json:"floating_ips" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *FloatingIPCollectionNext `json:"next,omitempty"`
}

// UnmarshalFloatingIPCollection unmarshals an instance of FloatingIPCollection from the specified map of raw messages.
func UnmarshalFloatingIPCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalFloatingIPCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "floating_ips", &obj.FloatingIps, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalFloatingIPCollectionNext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPCollectionFirst : A link to the first page of resources.
type FloatingIPCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalFloatingIPCollectionFirst unmarshals an instance of FloatingIPCollectionFirst from the specified map of raw messages.
func UnmarshalFloatingIPCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type FloatingIPCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalFloatingIPCollectionNext unmarshals an instance of FloatingIPCollectionNext from the specified map of raw messages.
func UnmarshalFloatingIPCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPPatch : FloatingIPPatch struct
type FloatingIPPatch struct {
	// The unique user-defined name for this floating IP.
	Name *string `json:"name,omitempty"`

	// A new network interface to bind this floating IP to, replacing any existing binding.
	// For this request to succeed, the existing floating IP must not be required by another
	// resource, such as a public gateway.
	Target FloatingIPPatchTargetNetworkInterfaceIdentityIntf `json:"target,omitempty"`
}

// UnmarshalFloatingIPPatch unmarshals an instance of FloatingIPPatch from the specified map of raw messages.
func UnmarshalFloatingIPPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFloatingIPPatchTargetNetworkInterfaceIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the FloatingIPPatch
func (floatingIPPatch *FloatingIPPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(floatingIPPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// FloatingIPPatchTargetNetworkInterfaceIdentity : A new network interface to bind this floating IP to, replacing any existing binding. For this request to succeed, the
// existing floating IP must not be required by another resource, such as a public gateway.
// Models which "extend" this model:
// - FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID
// - FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByHref
type FloatingIPPatchTargetNetworkInterfaceIdentity struct {
	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`
}

func (*FloatingIPPatchTargetNetworkInterfaceIdentity) isaFloatingIPPatchTargetNetworkInterfaceIdentity() bool {
	return true
}

type FloatingIPPatchTargetNetworkInterfaceIdentityIntf interface {
	isaFloatingIPPatchTargetNetworkInterfaceIdentity() bool
}

// UnmarshalFloatingIPPatchTargetNetworkInterfaceIdentity unmarshals an instance of FloatingIPPatchTargetNetworkInterfaceIdentity from the specified map of raw messages.
func UnmarshalFloatingIPPatchTargetNetworkInterfaceIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPPatchTargetNetworkInterfaceIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPPrototype : FloatingIPPrototype struct
// Models which "extend" this model:
// - FloatingIPPrototypeFloatingIPByZone
// - FloatingIPPrototypeFloatingIPByTarget
type FloatingIPPrototype struct {
	// The unique user-defined name for this floating IP. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The identity of the zone to provision a floating IP in.
	Zone ZoneIdentityIntf `json:"zone,omitempty"`

	// The network interface this floating IP is to be bound to.
	Target FloatingIPByTargetNetworkInterfaceIdentityIntf `json:"target,omitempty"`
}

func (*FloatingIPPrototype) isaFloatingIPPrototype() bool {
	return true
}

type FloatingIPPrototypeIntf interface {
	isaFloatingIPPrototype() bool
}

// UnmarshalFloatingIPPrototype unmarshals an instance of FloatingIPPrototype from the specified map of raw messages.
func UnmarshalFloatingIPPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPPrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFloatingIPByTargetNetworkInterfaceIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPReference : FloatingIPReference struct
type FloatingIPReference struct {
	// The globally unique IP address.
	Address *string `json:"address" validate:"required"`

	// The CRN for this floating IP.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this floating IP.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this floating IP.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this floating IP.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalFloatingIPReference unmarshals an instance of FloatingIPReference from the specified map of raw messages.
func UnmarshalFloatingIPReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPReference)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPTarget : The target of this floating IP.
// Models which "extend" this model:
// - FloatingIPTargetNetworkInterfaceReference
// - FloatingIPTargetPublicGatewayReference
type FloatingIPTarget struct {
	// The URL for this network interface.
	Href *string `json:"href,omitempty"`

	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The user-defined name for this network interface.
	Name *string `json:"name,omitempty"`

	// The primary IPv4 address.
	PrimaryIpv4Address *string `json:"primary_ipv4_address,omitempty"`

	// The resource type.
	ResourceType *string `json:"resource_type,omitempty"`

	// The CRN for this public gateway.
	CRN *string `json:"crn,omitempty"`
}

// Constants associated with the FloatingIPTarget.ResourceType property.
// The resource type.
const (
	FloatingIPTargetResourceTypeNetworkInterfaceConst = "network_interface"
)

func (*FloatingIPTarget) isaFloatingIPTarget() bool {
	return true
}

type FloatingIPTargetIntf interface {
	isaFloatingIPTarget() bool
}

// UnmarshalFloatingIPTarget unmarshals an instance of FloatingIPTarget from the specified map of raw messages.
func UnmarshalFloatingIPTarget(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPTarget)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "primary_ipv4_address", &obj.PrimaryIpv4Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPUnpaginatedCollection : FloatingIPUnpaginatedCollection struct
type FloatingIPUnpaginatedCollection struct {
	// Collection of floating IPs.
	FloatingIps []FloatingIP `json:"floating_ips" validate:"required"`
}

// UnmarshalFloatingIPUnpaginatedCollection unmarshals an instance of FloatingIPUnpaginatedCollection from the specified map of raw messages.
func UnmarshalFloatingIPUnpaginatedCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPUnpaginatedCollection)
	err = core.UnmarshalModel(m, "floating_ips", &obj.FloatingIps, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GetFloatingIPOptions : The GetFloatingIP options.
type GetFloatingIPOptions struct {
	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetFloatingIPOptions : Instantiate GetFloatingIPOptions
func (*VpcClassicV1) NewGetFloatingIPOptions(id string) *GetFloatingIPOptions {
	return &GetFloatingIPOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetFloatingIPOptions) SetID(id string) *GetFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetFloatingIPOptions) SetHeaders(param map[string]string) *GetFloatingIPOptions {
	options.Headers = param
	return options
}

// GetIkePolicyOptions : The GetIkePolicy options.
type GetIkePolicyOptions struct {
	// The IKE policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetIkePolicyOptions : Instantiate GetIkePolicyOptions
func (*VpcClassicV1) NewGetIkePolicyOptions(id string) *GetIkePolicyOptions {
	return &GetIkePolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetIkePolicyOptions) SetID(id string) *GetIkePolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetIkePolicyOptions) SetHeaders(param map[string]string) *GetIkePolicyOptions {
	options.Headers = param
	return options
}

// GetImageOptions : The GetImage options.
type GetImageOptions struct {
	// The image identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetImageOptions : Instantiate GetImageOptions
func (*VpcClassicV1) NewGetImageOptions(id string) *GetImageOptions {
	return &GetImageOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetImageOptions) SetID(id string) *GetImageOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetImageOptions) SetHeaders(param map[string]string) *GetImageOptions {
	options.Headers = param
	return options
}

// GetInstanceInitializationOptions : The GetInstanceInitialization options.
type GetInstanceInitializationOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceInitializationOptions : Instantiate GetInstanceInitializationOptions
func (*VpcClassicV1) NewGetInstanceInitializationOptions(id string) *GetInstanceInitializationOptions {
	return &GetInstanceInitializationOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetInstanceInitializationOptions) SetID(id string) *GetInstanceInitializationOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceInitializationOptions) SetHeaders(param map[string]string) *GetInstanceInitializationOptions {
	options.Headers = param
	return options
}

// GetInstanceNetworkInterfaceFloatingIPOptions : The GetInstanceNetworkInterfaceFloatingIP options.
type GetInstanceNetworkInterfaceFloatingIPOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	NetworkInterfaceID *string `validate:"required,ne="`

	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceNetworkInterfaceFloatingIPOptions : Instantiate GetInstanceNetworkInterfaceFloatingIPOptions
func (*VpcClassicV1) NewGetInstanceNetworkInterfaceFloatingIPOptions(instanceID string, networkInterfaceID string, id string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	return &GetInstanceNetworkInterfaceFloatingIPOptions{
		InstanceID:         core.StringPtr(instanceID),
		NetworkInterfaceID: core.StringPtr(networkInterfaceID),
		ID:                 core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetInstanceID(instanceID string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetNetworkInterfaceID : Allow user to set NetworkInterfaceID
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetNetworkInterfaceID(networkInterfaceID string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.NetworkInterfaceID = core.StringPtr(networkInterfaceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetID(id string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetHeaders(param map[string]string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.Headers = param
	return options
}

// GetInstanceNetworkInterfaceOptions : The GetInstanceNetworkInterface options.
type GetInstanceNetworkInterfaceOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceNetworkInterfaceOptions : Instantiate GetInstanceNetworkInterfaceOptions
func (*VpcClassicV1) NewGetInstanceNetworkInterfaceOptions(instanceID string, id string) *GetInstanceNetworkInterfaceOptions {
	return &GetInstanceNetworkInterfaceOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceNetworkInterfaceOptions) SetInstanceID(instanceID string) *GetInstanceNetworkInterfaceOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceNetworkInterfaceOptions) SetID(id string) *GetInstanceNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceNetworkInterfaceOptions) SetHeaders(param map[string]string) *GetInstanceNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// GetInstanceOptions : The GetInstance options.
type GetInstanceOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceOptions : Instantiate GetInstanceOptions
func (*VpcClassicV1) NewGetInstanceOptions(id string) *GetInstanceOptions {
	return &GetInstanceOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetInstanceOptions) SetID(id string) *GetInstanceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceOptions) SetHeaders(param map[string]string) *GetInstanceOptions {
	options.Headers = param
	return options
}

// GetInstanceProfileOptions : The GetInstanceProfile options.
type GetInstanceProfileOptions struct {
	// The instance profile name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceProfileOptions : Instantiate GetInstanceProfileOptions
func (*VpcClassicV1) NewGetInstanceProfileOptions(name string) *GetInstanceProfileOptions {
	return &GetInstanceProfileOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetInstanceProfileOptions) SetName(name string) *GetInstanceProfileOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceProfileOptions) SetHeaders(param map[string]string) *GetInstanceProfileOptions {
	options.Headers = param
	return options
}

// GetInstanceVolumeAttachmentOptions : The GetInstanceVolumeAttachment options.
type GetInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The volume attachment identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceVolumeAttachmentOptions : Instantiate GetInstanceVolumeAttachmentOptions
func (*VpcClassicV1) NewGetInstanceVolumeAttachmentOptions(instanceID string, id string) *GetInstanceVolumeAttachmentOptions {
	return &GetInstanceVolumeAttachmentOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *GetInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceVolumeAttachmentOptions) SetID(id string) *GetInstanceVolumeAttachmentOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *GetInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// GetIpsecPolicyOptions : The GetIpsecPolicy options.
type GetIpsecPolicyOptions struct {
	// The IPsec policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetIpsecPolicyOptions : Instantiate GetIpsecPolicyOptions
func (*VpcClassicV1) NewGetIpsecPolicyOptions(id string) *GetIpsecPolicyOptions {
	return &GetIpsecPolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetIpsecPolicyOptions) SetID(id string) *GetIpsecPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetIpsecPolicyOptions) SetHeaders(param map[string]string) *GetIpsecPolicyOptions {
	options.Headers = param
	return options
}

// GetKeyOptions : The GetKey options.
type GetKeyOptions struct {
	// The key identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetKeyOptions : Instantiate GetKeyOptions
func (*VpcClassicV1) NewGetKeyOptions(id string) *GetKeyOptions {
	return &GetKeyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetKeyOptions) SetID(id string) *GetKeyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetKeyOptions) SetHeaders(param map[string]string) *GetKeyOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerListenerOptions : The GetLoadBalancerListener options.
type GetLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerListenerOptions : Instantiate GetLoadBalancerListenerOptions
func (*VpcClassicV1) NewGetLoadBalancerListenerOptions(loadBalancerID string, id string) *GetLoadBalancerListenerOptions {
	return &GetLoadBalancerListenerOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerListenerOptions) SetID(id string) *GetLoadBalancerListenerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerListenerOptions) SetHeaders(param map[string]string) *GetLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerListenerPolicyOptions : The GetLoadBalancerListenerPolicy options.
type GetLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerListenerPolicyOptions : Instantiate GetLoadBalancerListenerPolicyOptions
func (*VpcClassicV1) NewGetLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, id string) *GetLoadBalancerListenerPolicyOptions {
	return &GetLoadBalancerListenerPolicyOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *GetLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *GetLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerListenerPolicyOptions) SetID(id string) *GetLoadBalancerListenerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *GetLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerListenerPolicyRuleOptions : The GetLoadBalancerListenerPolicyRule options.
type GetLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerListenerPolicyRuleOptions : Instantiate GetLoadBalancerListenerPolicyRuleOptions
func (*VpcClassicV1) NewGetLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, id string) *GetLoadBalancerListenerPolicyRuleOptions {
	return &GetLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetID(id string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerOptions : The GetLoadBalancer options.
type GetLoadBalancerOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerOptions : Instantiate GetLoadBalancerOptions
func (*VpcClassicV1) NewGetLoadBalancerOptions(id string) *GetLoadBalancerOptions {
	return &GetLoadBalancerOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerOptions) SetID(id string) *GetLoadBalancerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerOptions) SetHeaders(param map[string]string) *GetLoadBalancerOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerPoolMemberOptions : The GetLoadBalancerPoolMember options.
type GetLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The member identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerPoolMemberOptions : Instantiate GetLoadBalancerPoolMemberOptions
func (*VpcClassicV1) NewGetLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, id string) *GetLoadBalancerPoolMemberOptions {
	return &GetLoadBalancerPoolMemberOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *GetLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *GetLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerPoolMemberOptions) SetID(id string) *GetLoadBalancerPoolMemberOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *GetLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerPoolOptions : The GetLoadBalancerPool options.
type GetLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerPoolOptions : Instantiate GetLoadBalancerPoolOptions
func (*VpcClassicV1) NewGetLoadBalancerPoolOptions(loadBalancerID string, id string) *GetLoadBalancerPoolOptions {
	return &GetLoadBalancerPoolOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerPoolOptions) SetID(id string) *GetLoadBalancerPoolOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerPoolOptions) SetHeaders(param map[string]string) *GetLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerStatisticsOptions : The GetLoadBalancerStatistics options.
type GetLoadBalancerStatisticsOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerStatisticsOptions : Instantiate GetLoadBalancerStatisticsOptions
func (*VpcClassicV1) NewGetLoadBalancerStatisticsOptions(id string) *GetLoadBalancerStatisticsOptions {
	return &GetLoadBalancerStatisticsOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerStatisticsOptions) SetID(id string) *GetLoadBalancerStatisticsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerStatisticsOptions) SetHeaders(param map[string]string) *GetLoadBalancerStatisticsOptions {
	options.Headers = param
	return options
}

// GetNetworkACLOptions : The GetNetworkACL options.
type GetNetworkACLOptions struct {
	// The network ACL identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetNetworkACLOptions : Instantiate GetNetworkACLOptions
func (*VpcClassicV1) NewGetNetworkACLOptions(id string) *GetNetworkACLOptions {
	return &GetNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetNetworkACLOptions) SetID(id string) *GetNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetNetworkACLOptions) SetHeaders(param map[string]string) *GetNetworkACLOptions {
	options.Headers = param
	return options
}

// GetNetworkACLRuleOptions : The GetNetworkACLRule options.
type GetNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetNetworkACLRuleOptions : Instantiate GetNetworkACLRuleOptions
func (*VpcClassicV1) NewGetNetworkACLRuleOptions(networkACLID string, id string) *GetNetworkACLRuleOptions {
	return &GetNetworkACLRuleOptions{
		NetworkACLID: core.StringPtr(networkACLID),
		ID:           core.StringPtr(id),
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *GetNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *GetNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetID : Allow user to set ID
func (options *GetNetworkACLRuleOptions) SetID(id string) *GetNetworkACLRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetNetworkACLRuleOptions) SetHeaders(param map[string]string) *GetNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// GetOperatingSystemOptions : The GetOperatingSystem options.
type GetOperatingSystemOptions struct {
	// The operating system name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetOperatingSystemOptions : Instantiate GetOperatingSystemOptions
func (*VpcClassicV1) NewGetOperatingSystemOptions(name string) *GetOperatingSystemOptions {
	return &GetOperatingSystemOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetOperatingSystemOptions) SetName(name string) *GetOperatingSystemOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetOperatingSystemOptions) SetHeaders(param map[string]string) *GetOperatingSystemOptions {
	options.Headers = param
	return options
}

// GetPublicGatewayOptions : The GetPublicGateway options.
type GetPublicGatewayOptions struct {
	// The public gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetPublicGatewayOptions : Instantiate GetPublicGatewayOptions
func (*VpcClassicV1) NewGetPublicGatewayOptions(id string) *GetPublicGatewayOptions {
	return &GetPublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetPublicGatewayOptions) SetID(id string) *GetPublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetPublicGatewayOptions) SetHeaders(param map[string]string) *GetPublicGatewayOptions {
	options.Headers = param
	return options
}

// GetRegionOptions : The GetRegion options.
type GetRegionOptions struct {
	// The region name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetRegionOptions : Instantiate GetRegionOptions
func (*VpcClassicV1) NewGetRegionOptions(name string) *GetRegionOptions {
	return &GetRegionOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetRegionOptions) SetName(name string) *GetRegionOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetRegionOptions) SetHeaders(param map[string]string) *GetRegionOptions {
	options.Headers = param
	return options
}

// GetRegionZoneOptions : The GetRegionZone options.
type GetRegionZoneOptions struct {
	// The region name.
	RegionName *string `validate:"required,ne="`

	// The zone name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetRegionZoneOptions : Instantiate GetRegionZoneOptions
func (*VpcClassicV1) NewGetRegionZoneOptions(regionName string, name string) *GetRegionZoneOptions {
	return &GetRegionZoneOptions{
		RegionName: core.StringPtr(regionName),
		Name:       core.StringPtr(name),
	}
}

// SetRegionName : Allow user to set RegionName
func (options *GetRegionZoneOptions) SetRegionName(regionName string) *GetRegionZoneOptions {
	options.RegionName = core.StringPtr(regionName)
	return options
}

// SetName : Allow user to set Name
func (options *GetRegionZoneOptions) SetName(name string) *GetRegionZoneOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetRegionZoneOptions) SetHeaders(param map[string]string) *GetRegionZoneOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupNetworkInterfaceOptions : The GetSecurityGroupNetworkInterface options.
type GetSecurityGroupNetworkInterfaceOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupNetworkInterfaceOptions : Instantiate GetSecurityGroupNetworkInterfaceOptions
func (*VpcClassicV1) NewGetSecurityGroupNetworkInterfaceOptions(securityGroupID string, id string) *GetSecurityGroupNetworkInterfaceOptions {
	return &GetSecurityGroupNetworkInterfaceOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *GetSecurityGroupNetworkInterfaceOptions) SetSecurityGroupID(securityGroupID string) *GetSecurityGroupNetworkInterfaceOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupNetworkInterfaceOptions) SetID(id string) *GetSecurityGroupNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupNetworkInterfaceOptions) SetHeaders(param map[string]string) *GetSecurityGroupNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupOptions : The GetSecurityGroup options.
type GetSecurityGroupOptions struct {
	// The security group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupOptions : Instantiate GetSecurityGroupOptions
func (*VpcClassicV1) NewGetSecurityGroupOptions(id string) *GetSecurityGroupOptions {
	return &GetSecurityGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupOptions) SetID(id string) *GetSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupOptions) SetHeaders(param map[string]string) *GetSecurityGroupOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupRuleOptions : The GetSecurityGroupRule options.
type GetSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupRuleOptions : Instantiate GetSecurityGroupRuleOptions
func (*VpcClassicV1) NewGetSecurityGroupRuleOptions(securityGroupID string, id string) *GetSecurityGroupRuleOptions {
	return &GetSecurityGroupRuleOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *GetSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *GetSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupRuleOptions) SetID(id string) *GetSecurityGroupRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupRuleOptions) SetHeaders(param map[string]string) *GetSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// GetSubnetNetworkACLOptions : The GetSubnetNetworkACL options.
type GetSubnetNetworkACLOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetNetworkACLOptions : Instantiate GetSubnetNetworkACLOptions
func (*VpcClassicV1) NewGetSubnetNetworkACLOptions(id string) *GetSubnetNetworkACLOptions {
	return &GetSubnetNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetNetworkACLOptions) SetID(id string) *GetSubnetNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetNetworkACLOptions) SetHeaders(param map[string]string) *GetSubnetNetworkACLOptions {
	options.Headers = param
	return options
}

// GetSubnetOptions : The GetSubnet options.
type GetSubnetOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetOptions : Instantiate GetSubnetOptions
func (*VpcClassicV1) NewGetSubnetOptions(id string) *GetSubnetOptions {
	return &GetSubnetOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetOptions) SetID(id string) *GetSubnetOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetOptions) SetHeaders(param map[string]string) *GetSubnetOptions {
	options.Headers = param
	return options
}

// GetSubnetPublicGatewayOptions : The GetSubnetPublicGateway options.
type GetSubnetPublicGatewayOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetPublicGatewayOptions : Instantiate GetSubnetPublicGatewayOptions
func (*VpcClassicV1) NewGetSubnetPublicGatewayOptions(id string) *GetSubnetPublicGatewayOptions {
	return &GetSubnetPublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetPublicGatewayOptions) SetID(id string) *GetSubnetPublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetPublicGatewayOptions) SetHeaders(param map[string]string) *GetSubnetPublicGatewayOptions {
	options.Headers = param
	return options
}

// GetVolumeOptions : The GetVolume options.
type GetVolumeOptions struct {
	// The volume identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVolumeOptions : Instantiate GetVolumeOptions
func (*VpcClassicV1) NewGetVolumeOptions(id string) *GetVolumeOptions {
	return &GetVolumeOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVolumeOptions) SetID(id string) *GetVolumeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVolumeOptions) SetHeaders(param map[string]string) *GetVolumeOptions {
	options.Headers = param
	return options
}

// GetVolumeProfileOptions : The GetVolumeProfile options.
type GetVolumeProfileOptions struct {
	// The volume profile name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVolumeProfileOptions : Instantiate GetVolumeProfileOptions
func (*VpcClassicV1) NewGetVolumeProfileOptions(name string) *GetVolumeProfileOptions {
	return &GetVolumeProfileOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetVolumeProfileOptions) SetName(name string) *GetVolumeProfileOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVolumeProfileOptions) SetHeaders(param map[string]string) *GetVolumeProfileOptions {
	options.Headers = param
	return options
}

// GetVPCAddressPrefixOptions : The GetVPCAddressPrefix options.
type GetVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The prefix identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCAddressPrefixOptions : Instantiate GetVPCAddressPrefixOptions
func (*VpcClassicV1) NewGetVPCAddressPrefixOptions(vpcID string, id string) *GetVPCAddressPrefixOptions {
	return &GetVPCAddressPrefixOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *GetVPCAddressPrefixOptions) SetVPCID(vpcID string) *GetVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPCAddressPrefixOptions) SetID(id string) *GetVPCAddressPrefixOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCAddressPrefixOptions) SetHeaders(param map[string]string) *GetVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// GetVPCDefaultSecurityGroupOptions : The GetVPCDefaultSecurityGroup options.
type GetVPCDefaultSecurityGroupOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCDefaultSecurityGroupOptions : Instantiate GetVPCDefaultSecurityGroupOptions
func (*VpcClassicV1) NewGetVPCDefaultSecurityGroupOptions(id string) *GetVPCDefaultSecurityGroupOptions {
	return &GetVPCDefaultSecurityGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPCDefaultSecurityGroupOptions) SetID(id string) *GetVPCDefaultSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCDefaultSecurityGroupOptions) SetHeaders(param map[string]string) *GetVPCDefaultSecurityGroupOptions {
	options.Headers = param
	return options
}

// GetVPCOptions : The GetVPC options.
type GetVPCOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCOptions : Instantiate GetVPCOptions
func (*VpcClassicV1) NewGetVPCOptions(id string) *GetVPCOptions {
	return &GetVPCOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPCOptions) SetID(id string) *GetVPCOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCOptions) SetHeaders(param map[string]string) *GetVPCOptions {
	options.Headers = param
	return options
}

// GetVPCRouteOptions : The GetVPCRoute options.
type GetVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The route identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCRouteOptions : Instantiate GetVPCRouteOptions
func (*VpcClassicV1) NewGetVPCRouteOptions(vpcID string, id string) *GetVPCRouteOptions {
	return &GetVPCRouteOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *GetVPCRouteOptions) SetVPCID(vpcID string) *GetVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPCRouteOptions) SetID(id string) *GetVPCRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCRouteOptions) SetHeaders(param map[string]string) *GetVPCRouteOptions {
	options.Headers = param
	return options
}

// GetVPNGatewayConnectionOptions : The GetVPNGatewayConnection options.
type GetVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPNGatewayConnectionOptions : Instantiate GetVPNGatewayConnectionOptions
func (*VpcClassicV1) NewGetVPNGatewayConnectionOptions(vpnGatewayID string, id string) *GetVPNGatewayConnectionOptions {
	return &GetVPNGatewayConnectionOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *GetVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID string) *GetVPNGatewayConnectionOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPNGatewayConnectionOptions) SetID(id string) *GetVPNGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPNGatewayConnectionOptions) SetHeaders(param map[string]string) *GetVPNGatewayConnectionOptions {
	options.Headers = param
	return options
}

// GetVPNGatewayOptions : The GetVPNGateway options.
type GetVPNGatewayOptions struct {
	// The VPN gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPNGatewayOptions : Instantiate GetVPNGatewayOptions
func (*VpcClassicV1) NewGetVPNGatewayOptions(id string) *GetVPNGatewayOptions {
	return &GetVPNGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPNGatewayOptions) SetID(id string) *GetVPNGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPNGatewayOptions) SetHeaders(param map[string]string) *GetVPNGatewayOptions {
	options.Headers = param
	return options
}

// IkePolicy : IkePolicy struct
type IkePolicy struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `json:"authentication_algorithm" validate:"required"`

	// Collection of references to VPN gateway connections that use this IKE policy.
	Connections []VPNGatewayConnectionReference `json:"connections" validate:"required"`

	// The date and time that this IKE policy was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The Diffie-Hellman group.
	DhGroup *int64 `json:"dh_group" validate:"required"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `json:"encryption_algorithm" validate:"required"`

	// The IKE policy's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this IKE policy.
	ID *string `json:"id" validate:"required"`

	// The IKE protocol version.
	IkeVersion *int64 `json:"ike_version" validate:"required"`

	// The key lifetime in seconds.
	KeyLifetime *int64 `json:"key_lifetime" validate:"required"`

	// The user-defined name for this IKE policy.
	Name *string `json:"name" validate:"required"`

	// The IKE negotiation mode. Only `main` is supported.
	NegotiationMode *string `json:"negotiation_mode" validate:"required"`

	// The resource group for this IKE policy.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the IkePolicy.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	IkePolicyAuthenticationAlgorithmMd5Const    = "md5"
	IkePolicyAuthenticationAlgorithmSha1Const   = "sha1"
	IkePolicyAuthenticationAlgorithmSha256Const = "sha256"
)

// Constants associated with the IkePolicy.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	IkePolicyEncryptionAlgorithmAes128Const    = "aes128"
	IkePolicyEncryptionAlgorithmAes256Const    = "aes256"
	IkePolicyEncryptionAlgorithmTripleDesConst = "triple_des"
)

// Constants associated with the IkePolicy.NegotiationMode property.
// The IKE negotiation mode. Only `main` is supported.
const (
	IkePolicyNegotiationModeMainConst = "main"
)

// Constants associated with the IkePolicy.ResourceType property.
// The resource type.
const (
	IkePolicyResourceTypeIkePolicyConst = "ike_policy"
)

// UnmarshalIkePolicy unmarshals an instance of IkePolicy from the specified map of raw messages.
func UnmarshalIkePolicy(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicy)
	err = core.UnmarshalPrimitive(m, "authentication_algorithm", &obj.AuthenticationAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "connections", &obj.Connections, UnmarshalVPNGatewayConnectionReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "dh_group", &obj.DhGroup)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "encryption_algorithm", &obj.EncryptionAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "ike_version", &obj.IkeVersion)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "key_lifetime", &obj.KeyLifetime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "negotiation_mode", &obj.NegotiationMode)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IkePolicyCollection : IkePolicyCollection struct
type IkePolicyCollection struct {
	// A link to the first page of resources.
	First *IkePolicyCollectionFirst `json:"first" validate:"required"`

	// Collection of IKE policies.
	IkePolicies []IkePolicy `json:"ike_policies" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *IkePolicyCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalIkePolicyCollection unmarshals an instance of IkePolicyCollection from the specified map of raw messages.
func UnmarshalIkePolicyCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicyCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalIkePolicyCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ike_policies", &obj.IkePolicies, UnmarshalIkePolicy)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalIkePolicyCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IkePolicyIdentity : Identifies an IKE policy by a unique property.
// Models which "extend" this model:
// - IkePolicyIdentityByID
// - IkePolicyIdentityByHref
type IkePolicyIdentity struct {
	// The unique identifier for this IKE policy.
	ID *string `json:"id,omitempty"`

	// The IKE policy's canonical URL.
	Href *string `json:"href,omitempty"`
}

func (*IkePolicyIdentity) isaIkePolicyIdentity() bool {
	return true
}

type IkePolicyIdentityIntf interface {
	isaIkePolicyIdentity() bool
}

// UnmarshalIkePolicyIdentity unmarshals an instance of IkePolicyIdentity from the specified map of raw messages.
func UnmarshalIkePolicyIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicyIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IkePolicyPatch : IkePolicyPatch struct
type IkePolicyPatch struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `json:"authentication_algorithm,omitempty"`

	// The Diffie-Hellman group.
	DhGroup *int64 `json:"dh_group,omitempty"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `json:"encryption_algorithm,omitempty"`

	// The IKE protocol version.
	IkeVersion *int64 `json:"ike_version,omitempty"`

	// The key lifetime in seconds.
	KeyLifetime *int64 `json:"key_lifetime,omitempty"`

	// The user-defined name for this IKE policy.
	Name *string `json:"name,omitempty"`
}

// Constants associated with the IkePolicyPatch.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	IkePolicyPatchAuthenticationAlgorithmMd5Const    = "md5"
	IkePolicyPatchAuthenticationAlgorithmSha1Const   = "sha1"
	IkePolicyPatchAuthenticationAlgorithmSha256Const = "sha256"
)

// Constants associated with the IkePolicyPatch.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	IkePolicyPatchEncryptionAlgorithmAes128Const    = "aes128"
	IkePolicyPatchEncryptionAlgorithmAes256Const    = "aes256"
	IkePolicyPatchEncryptionAlgorithmTripleDesConst = "triple_des"
)

// UnmarshalIkePolicyPatch unmarshals an instance of IkePolicyPatch from the specified map of raw messages.
func UnmarshalIkePolicyPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicyPatch)
	err = core.UnmarshalPrimitive(m, "authentication_algorithm", &obj.AuthenticationAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "dh_group", &obj.DhGroup)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "encryption_algorithm", &obj.EncryptionAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "ike_version", &obj.IkeVersion)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "key_lifetime", &obj.KeyLifetime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the IkePolicyPatch
func (ikePolicyPatch *IkePolicyPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(ikePolicyPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// IkePolicyReference : IkePolicyReference struct
type IkePolicyReference struct {
	// The IKE policy's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this IKE policy.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this IKE policy.
	Name *string `json:"name" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the IkePolicyReference.ResourceType property.
// The resource type.
const (
	IkePolicyReferenceResourceTypeIkePolicyConst = "ike_policy"
)

// UnmarshalIkePolicyReference unmarshals an instance of IkePolicyReference from the specified map of raw messages.
func UnmarshalIkePolicyReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicyReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IP : IP struct
type IP struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address" validate:"required"`
}

// UnmarshalIP unmarshals an instance of IP from the specified map of raw messages.
func UnmarshalIP(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IP)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IPsecPolicy : IPsecPolicy struct
type IPsecPolicy struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `json:"authentication_algorithm" validate:"required"`

	// Collection of references to VPN gateway connections that use this IPsec policy.
	Connections []VPNGatewayConnectionReference `json:"connections" validate:"required"`

	// The date and time that this IPsec policy was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The encapsulation mode used. Only `tunnel` is supported.
	EncapsulationMode *string `json:"encapsulation_mode" validate:"required"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `json:"encryption_algorithm" validate:"required"`

	// The IPsec policy's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this IPsec policy.
	ID *string `json:"id" validate:"required"`

	// The key lifetime in seconds.
	KeyLifetime *int64 `json:"key_lifetime" validate:"required"`

	// The user-defined name for this IPsec policy.
	Name *string `json:"name" validate:"required"`

	// Perfect Forward Secrecy.
	Pfs *string `json:"pfs" validate:"required"`

	// The resource group for this IPsec policy.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The transform protocol used. Only `esp` is supported.
	TransformProtocol *string `json:"transform_protocol" validate:"required"`
}

// Constants associated with the IPsecPolicy.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	IPsecPolicyAuthenticationAlgorithmMd5Const    = "md5"
	IPsecPolicyAuthenticationAlgorithmSha1Const   = "sha1"
	IPsecPolicyAuthenticationAlgorithmSha256Const = "sha256"
)

// Constants associated with the IPsecPolicy.EncapsulationMode property.
// The encapsulation mode used. Only `tunnel` is supported.
const (
	IPsecPolicyEncapsulationModeTunnelConst = "tunnel"
)

// Constants associated with the IPsecPolicy.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	IPsecPolicyEncryptionAlgorithmAes128Const    = "aes128"
	IPsecPolicyEncryptionAlgorithmAes256Const    = "aes256"
	IPsecPolicyEncryptionAlgorithmTripleDesConst = "triple_des"
)

// Constants associated with the IPsecPolicy.Pfs property.
// Perfect Forward Secrecy.
const (
	IPsecPolicyPfsDisabledConst = "disabled"
	IPsecPolicyPfsGroup14Const  = "group_14"
	IPsecPolicyPfsGroup2Const   = "group_2"
	IPsecPolicyPfsGroup5Const   = "group_5"
)

// Constants associated with the IPsecPolicy.ResourceType property.
// The resource type.
const (
	IPsecPolicyResourceTypeIpsecPolicyConst = "ipsec_policy"
)

// Constants associated with the IPsecPolicy.TransformProtocol property.
// The transform protocol used. Only `esp` is supported.
const (
	IPsecPolicyTransformProtocolEspConst = "esp"
)

// UnmarshalIPsecPolicy unmarshals an instance of IPsecPolicy from the specified map of raw messages.
func UnmarshalIPsecPolicy(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicy)
	err = core.UnmarshalPrimitive(m, "authentication_algorithm", &obj.AuthenticationAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "connections", &obj.Connections, UnmarshalVPNGatewayConnectionReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "encapsulation_mode", &obj.EncapsulationMode)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "encryption_algorithm", &obj.EncryptionAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "key_lifetime", &obj.KeyLifetime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "pfs", &obj.Pfs)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "transform_protocol", &obj.TransformProtocol)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IPsecPolicyCollection : IPsecPolicyCollection struct
type IPsecPolicyCollection struct {
	// A link to the first page of resources.
	First *IPsecPolicyCollectionFirst `json:"first" validate:"required"`

	// Collection of IPsec policies.
	IpsecPolicies []IPsecPolicy `json:"ipsec_policies" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *IPsecPolicyCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalIPsecPolicyCollection unmarshals an instance of IPsecPolicyCollection from the specified map of raw messages.
func UnmarshalIPsecPolicyCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicyCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalIPsecPolicyCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ipsec_policies", &obj.IpsecPolicies, UnmarshalIPsecPolicy)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalIPsecPolicyCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IPsecPolicyCollectionFirst : A link to the first page of resources.
type IPsecPolicyCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalIPsecPolicyCollectionFirst unmarshals an instance of IPsecPolicyCollectionFirst from the specified map of raw messages.
func UnmarshalIPsecPolicyCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicyCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IPsecPolicyCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type IPsecPolicyCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalIPsecPolicyCollectionNext unmarshals an instance of IPsecPolicyCollectionNext from the specified map of raw messages.
func UnmarshalIPsecPolicyCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicyCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IPsecPolicyIdentity : Identifies an IPsec policy by a unique property.
// Models which "extend" this model:
// - IPsecPolicyIdentityByID
// - IPsecPolicyIdentityByHref
type IPsecPolicyIdentity struct {
	// The unique identifier for this IPsec policy.
	ID *string `json:"id,omitempty"`

	// The IPsec policy's canonical URL.
	Href *string `json:"href,omitempty"`
}

func (*IPsecPolicyIdentity) isaIPsecPolicyIdentity() bool {
	return true
}

type IPsecPolicyIdentityIntf interface {
	isaIPsecPolicyIdentity() bool
}

// UnmarshalIPsecPolicyIdentity unmarshals an instance of IPsecPolicyIdentity from the specified map of raw messages.
func UnmarshalIPsecPolicyIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicyIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IPsecPolicyPatch : IPsecPolicyPatch struct
type IPsecPolicyPatch struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `json:"authentication_algorithm,omitempty"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `json:"encryption_algorithm,omitempty"`

	// The key lifetime in seconds.
	KeyLifetime *int64 `json:"key_lifetime,omitempty"`

	// The user-defined name for this IPsec policy.
	Name *string `json:"name,omitempty"`

	// Perfect Forward Secrecy.
	Pfs *string `json:"pfs,omitempty"`
}

// Constants associated with the IPsecPolicyPatch.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	IPsecPolicyPatchAuthenticationAlgorithmMd5Const    = "md5"
	IPsecPolicyPatchAuthenticationAlgorithmSha1Const   = "sha1"
	IPsecPolicyPatchAuthenticationAlgorithmSha256Const = "sha256"
)

// Constants associated with the IPsecPolicyPatch.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	IPsecPolicyPatchEncryptionAlgorithmAes128Const    = "aes128"
	IPsecPolicyPatchEncryptionAlgorithmAes256Const    = "aes256"
	IPsecPolicyPatchEncryptionAlgorithmTripleDesConst = "triple_des"
)

// Constants associated with the IPsecPolicyPatch.Pfs property.
// Perfect Forward Secrecy.
const (
	IPsecPolicyPatchPfsDisabledConst = "disabled"
	IPsecPolicyPatchPfsGroup14Const  = "group_14"
	IPsecPolicyPatchPfsGroup2Const   = "group_2"
	IPsecPolicyPatchPfsGroup5Const   = "group_5"
)

// UnmarshalIPsecPolicyPatch unmarshals an instance of IPsecPolicyPatch from the specified map of raw messages.
func UnmarshalIPsecPolicyPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicyPatch)
	err = core.UnmarshalPrimitive(m, "authentication_algorithm", &obj.AuthenticationAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "encryption_algorithm", &obj.EncryptionAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "key_lifetime", &obj.KeyLifetime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "pfs", &obj.Pfs)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the IPsecPolicyPatch
func (iPsecPolicyPatch *IPsecPolicyPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(iPsecPolicyPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// IPsecPolicyReference : IPsecPolicyReference struct
type IPsecPolicyReference struct {
	// The IPsec policy's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this IPsec policy.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this IPsec policy.
	Name *string `json:"name" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the IPsecPolicyReference.ResourceType property.
// The resource type.
const (
	IPsecPolicyReferenceResourceTypeIpsecPolicyConst = "ipsec_policy"
)

// UnmarshalIPsecPolicyReference unmarshals an instance of IPsecPolicyReference from the specified map of raw messages.
func UnmarshalIPsecPolicyReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IPsecPolicyReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IkePolicyCollectionFirst : A link to the first page of resources.
type IkePolicyCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalIkePolicyCollectionFirst unmarshals an instance of IkePolicyCollectionFirst from the specified map of raw messages.
func UnmarshalIkePolicyCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicyCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// IkePolicyCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type IkePolicyCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalIkePolicyCollectionNext unmarshals an instance of IkePolicyCollectionNext from the specified map of raw messages.
func UnmarshalIkePolicyCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(IkePolicyCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Image : Image struct
type Image struct {
	// The date and time that the image was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this image.
	CRN *string `json:"crn" validate:"required"`

	// Details for the stored image file.
	File *ImageFile `json:"file" validate:"required"`

	// The URL for this image.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this image.
	ID *string `json:"id" validate:"required"`

	// The minimum size (in gigabytes) of a volume onto which this image may be provisioned.
	//
	// This property may be absent if the image has a `status` of `pending`, `tentative`, or
	// `failed`.
	MinimumProvisionedSize *int64 `json:"minimum_provisioned_size,omitempty"`

	// The user-defined or system-provided name for this image.
	Name *string `json:"name" validate:"required"`

	// The operating system included in this image.
	OperatingSystem *OperatingSystem `json:"operating_system,omitempty"`

	// The resource group for this image.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The status of this image.
	Status *string `json:"status" validate:"required"`

	// Whether the image is publicly visible or private to the account.
	Visibility *string `json:"visibility" validate:"required"`
}

// Constants associated with the Image.Status property.
// The status of this image.
const (
	ImageStatusAvailableConst  = "available"
	ImageStatusDeletingConst   = "deleting"
	ImageStatusDeprecatedConst = "deprecated"
	ImageStatusFailedConst     = "failed"
	ImageStatusPendingConst    = "pending"
	ImageStatusTentativeConst  = "tentative"
	ImageStatusUnusableConst   = "unusable"
)

// Constants associated with the Image.Visibility property.
// Whether the image is publicly visible or private to the account.
const (
	ImageVisibilityPrivateConst = "private"
	ImageVisibilityPublicConst  = "public"
)

// UnmarshalImage unmarshals an instance of Image from the specified map of raw messages.
func UnmarshalImage(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Image)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "file", &obj.File, UnmarshalImageFile)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "minimum_provisioned_size", &obj.MinimumProvisionedSize)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "operating_system", &obj.OperatingSystem, UnmarshalOperatingSystem)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "visibility", &obj.Visibility)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageCollection : ImageCollection struct
type ImageCollection struct {
	// A link to the first page of resources.
	First *ImageCollectionFirst `json:"first" validate:"required"`

	// Collection of images.
	Images []Image `json:"images" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *ImageCollectionNext `json:"next,omitempty"`
}

// UnmarshalImageCollection unmarshals an instance of ImageCollection from the specified map of raw messages.
func UnmarshalImageCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalImageCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "images", &obj.Images, UnmarshalImage)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalImageCollectionNext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageCollectionFirst : A link to the first page of resources.
type ImageCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalImageCollectionFirst unmarshals an instance of ImageCollectionFirst from the specified map of raw messages.
func UnmarshalImageCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type ImageCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalImageCollectionNext unmarshals an instance of ImageCollectionNext from the specified map of raw messages.
func UnmarshalImageCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageFile : ImageFile struct
type ImageFile struct {
	// The size of the stored image file rounded up to the next gigabyte.
	//
	// This property may be absent if the associated image has a `status` of `pending` or
	// `failed`.
	Size *int64 `json:"size,omitempty"`
}

// UnmarshalImageFile unmarshals an instance of ImageFile from the specified map of raw messages.
func UnmarshalImageFile(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageFile)
	err = core.UnmarshalPrimitive(m, "size", &obj.Size)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageFilePrototype : ImageFilePrototype struct
type ImageFilePrototype struct {
	// The Cloud Object Store (COS) location of the image file.
	Href *string `json:"href" validate:"required"`
}

// NewImageFilePrototype : Instantiate ImageFilePrototype (Generic Model Constructor)
func (*VpcClassicV1) NewImageFilePrototype(href string) (model *ImageFilePrototype, err error) {
	model = &ImageFilePrototype{
		Href: core.StringPtr(href),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalImageFilePrototype unmarshals an instance of ImageFilePrototype from the specified map of raw messages.
func UnmarshalImageFilePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageFilePrototype)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageIdentity : Identifies an image by a unique property.
// Models which "extend" this model:
// - ImageIdentityByID
// - ImageIdentityByCRN
// - ImageIdentityByHref
type ImageIdentity struct {
	// The unique identifier for this image.
	ID *string `json:"id,omitempty"`

	// The CRN for this image.
	CRN *string `json:"crn,omitempty"`

	// The URL for this image.
	Href *string `json:"href,omitempty"`
}

func (*ImageIdentity) isaImageIdentity() bool {
	return true
}

type ImageIdentityIntf interface {
	isaImageIdentity() bool
}

// UnmarshalImageIdentity unmarshals an instance of ImageIdentity from the specified map of raw messages.
func UnmarshalImageIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImagePatch : ImagePatch struct
type ImagePatch struct {
	// The unique user-defined name for this image. Names starting with "ibm-" are not allowed.
	Name *string `json:"name,omitempty"`
}

// UnmarshalImagePatch unmarshals an instance of ImagePatch from the specified map of raw messages.
func UnmarshalImagePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImagePatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the ImagePatch
func (imagePatch *ImagePatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(imagePatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// ImagePrototype : ImagePrototype struct
// Models which "extend" this model:
// - ImagePrototypeImageByFile
type ImagePrototype struct {
	// The unique user-defined name for this image. Names starting with "ibm-" are not allowed. If unspecified, the name
	// will be a hyphenated list of randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf `json:"resource_group,omitempty"`

	// The file from which to create the image.
	File *ImageFilePrototype `json:"file,omitempty"`

	// The identity of the [supported operating
	// system](https://cloud.ibm.com/apidocs/vpc#list-operating-systems) included in
	// this image.
	OperatingSystem OperatingSystemIdentityIntf `json:"operating_system,omitempty"`
}

func (*ImagePrototype) isaImagePrototype() bool {
	return true
}

type ImagePrototypeIntf interface {
	isaImagePrototype() bool
}

// UnmarshalImagePrototype unmarshals an instance of ImagePrototype from the specified map of raw messages.
func UnmarshalImagePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImagePrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "file", &obj.File, UnmarshalImageFilePrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "operating_system", &obj.OperatingSystem, UnmarshalOperatingSystemIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ImageReference : ImageReference struct
type ImageReference struct {
	// The CRN for this image.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this image.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this image.
	ID *string `json:"id" validate:"required"`

	// The user-defined or system-provided name for this image.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalImageReference unmarshals an instance of ImageReference from the specified map of raw messages.
func UnmarshalImageReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ImageReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Instance : Instance struct
type Instance struct {
	// The total bandwidth (in megabits per second) shared across the virtual server instance's network interfaces.
	Bandwidth *int64 `json:"bandwidth" validate:"required"`

	// Boot volume attachment.
	BootVolumeAttachment *VolumeAttachmentReferenceInstanceContext `json:"boot_volume_attachment" validate:"required"`

	// The date and time that the virtual server instance was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this virtual server instance.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this virtual server instance.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this virtual server instance.
	ID *string `json:"id" validate:"required"`

	// The image the virtual server instance was provisioned from.
	Image *ImageReference `json:"image,omitempty"`

	// The amount of memory in gigabytes.
	Memory *int64 `json:"memory" validate:"required"`

	// The user-defined name for this virtual server instance (and default system hostname).
	Name *string `json:"name" validate:"required"`

	// Collection of the virtual server instance's network interfaces, including the primary network interface.
	NetworkInterfaces []NetworkInterfaceInstanceContextReference `json:"network_interfaces" validate:"required"`

	// Primary network interface.
	PrimaryNetworkInterface *NetworkInterfaceInstanceContextReference `json:"primary_network_interface" validate:"required"`

	// The profile for this virtual server instance.
	Profile *InstanceProfileReference `json:"profile" validate:"required"`

	// The resource group for this instance.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The status of the virtual server instance.
	Status *string `json:"status" validate:"required"`

	// The virtual server instance VCPU configuration.
	Vcpu *InstanceVcpu `json:"vcpu" validate:"required"`

	// Collection of the virtual server instance's volume attachments, including the boot volume attachment.
	VolumeAttachments []VolumeAttachmentReferenceInstanceContext `json:"volume_attachments" validate:"required"`

	// The VPC the virtual server instance resides in.
	VPC *VPCReference `json:"vpc" validate:"required"`

	// The zone the virtual server instance resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the Instance.Status property.
// The status of the virtual server instance.
const (
	InstanceStatusDeletingConst   = "deleting"
	InstanceStatusFailedConst     = "failed"
	InstanceStatusPausedConst     = "paused"
	InstanceStatusPausingConst    = "pausing"
	InstanceStatusPendingConst    = "pending"
	InstanceStatusRestartingConst = "restarting"
	InstanceStatusResumingConst   = "resuming"
	InstanceStatusRunningConst    = "running"
	InstanceStatusStartingConst   = "starting"
	InstanceStatusStoppedConst    = "stopped"
	InstanceStatusStoppingConst   = "stopping"
)

// UnmarshalInstance unmarshals an instance of Instance from the specified map of raw messages.
func UnmarshalInstance(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Instance)
	err = core.UnmarshalPrimitive(m, "bandwidth", &obj.Bandwidth)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "boot_volume_attachment", &obj.BootVolumeAttachment, UnmarshalVolumeAttachmentReferenceInstanceContext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "image", &obj.Image, UnmarshalImageReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "memory", &obj.Memory)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_interfaces", &obj.NetworkInterfaces, UnmarshalNetworkInterfaceInstanceContextReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "primary_network_interface", &obj.PrimaryNetworkInterface, UnmarshalNetworkInterfaceInstanceContextReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profile", &obj.Profile, UnmarshalInstanceProfileReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vcpu", &obj.Vcpu, UnmarshalInstanceVcpu)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume_attachments", &obj.VolumeAttachments, UnmarshalVolumeAttachmentReferenceInstanceContext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceAction : InstanceAction struct
type InstanceAction struct {
	// The date and time that the action was completed.
	CompletedAt *strfmt.DateTime `json:"completed_at,omitempty"`

	// The date and time that the action was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The URL for this instance action.
	Href *string `json:"href" validate:"required"`

	// The identifier for this instance action.
	ID *string `json:"id" validate:"required"`

	// The date and time that the action was started.
	StartedAt *strfmt.DateTime `json:"started_at,omitempty"`

	// The current status of this action.
	Status *string `json:"status" validate:"required"`

	// The type of action.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the InstanceAction.Status property.
// The current status of this action.
const (
	InstanceActionStatusCompletedConst = "completed"
	InstanceActionStatusFailedConst    = "failed"
	InstanceActionStatusPendingConst   = "pending"
	InstanceActionStatusRunningConst   = "running"
)

// Constants associated with the InstanceAction.Type property.
// The type of action.
const (
	InstanceActionTypeRebootConst = "reboot"
	InstanceActionTypeResetConst  = "reset"
	InstanceActionTypeStartConst  = "start"
	InstanceActionTypeStopConst   = "stop"
)

// UnmarshalInstanceAction unmarshals an instance of InstanceAction from the specified map of raw messages.
func UnmarshalInstanceAction(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceAction)
	err = core.UnmarshalPrimitive(m, "completed_at", &obj.CompletedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "started_at", &obj.StartedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceCollection : InstanceCollection struct
type InstanceCollection struct {
	// A link to the first page of resources.
	First *InstanceCollectionFirst `json:"first" validate:"required"`

	// Collection of virtual server instances.
	Instances []Instance `json:"instances" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *InstanceCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalInstanceCollection unmarshals an instance of InstanceCollection from the specified map of raw messages.
func UnmarshalInstanceCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalInstanceCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "instances", &obj.Instances, UnmarshalInstance)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalInstanceCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceCollectionFirst : A link to the first page of resources.
type InstanceCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalInstanceCollectionFirst unmarshals an instance of InstanceCollectionFirst from the specified map of raw messages.
func UnmarshalInstanceCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type InstanceCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalInstanceCollectionNext unmarshals an instance of InstanceCollectionNext from the specified map of raw messages.
func UnmarshalInstanceCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceInitialization : InstanceInitialization struct
type InstanceInitialization struct {
	// Collection of references to public SSH keys used at instance initialization.
	Keys []KeyReferenceInstanceInitializationContextIntf `json:"keys" validate:"required"`

	Password *InstanceInitializationPassword `json:"password,omitempty"`
}

// UnmarshalInstanceInitialization unmarshals an instance of InstanceInitialization from the specified map of raw messages.
func UnmarshalInstanceInitialization(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceInitialization)
	err = core.UnmarshalModel(m, "keys", &obj.Keys, UnmarshalKeyReferenceInstanceInitializationContext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "password", &obj.Password, UnmarshalInstanceInitializationPassword)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceInitializationPassword : InstanceInitializationPassword struct
type InstanceInitializationPassword struct {
	// The administrator password at initialization, encrypted using `encryption_key`, and returned base64-encoded.
	EncryptedPassword *[]byte `json:"encrypted_password" validate:"required"`

	// The reference to the public SSH key used to encrypt the administrator password.
	EncryptionKey KeyReferenceInstanceInitializationContextIntf `json:"encryption_key" validate:"required"`
}

// UnmarshalInstanceInitializationPassword unmarshals an instance of InstanceInitializationPassword from the specified map of raw messages.
func UnmarshalInstanceInitializationPassword(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceInitializationPassword)
	err = core.UnmarshalPrimitive(m, "encrypted_password", &obj.EncryptedPassword)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "encryption_key", &obj.EncryptionKey, UnmarshalKeyReferenceInstanceInitializationContext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstancePatch : InstancePatch struct
type InstancePatch struct {
	// The user-defined name for this virtual server instance (and default system hostname).
	Name *string `json:"name,omitempty"`
}

// UnmarshalInstancePatch unmarshals an instance of InstancePatch from the specified map of raw messages.
func UnmarshalInstancePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstancePatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the InstancePatch
func (instancePatch *InstancePatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(instancePatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// InstanceProfile : InstanceProfile struct
type InstanceProfile struct {
	Bandwidth InstanceProfileBandwidthIntf `json:"bandwidth" validate:"required"`

	// The CRN for this virtual server instance profile.
	CRN *string `json:"crn" validate:"required"`

	// The product family this virtual server instance profile belongs to.
	Family *string `json:"family,omitempty"`

	// The URL for this virtual server instance profile.
	Href *string `json:"href" validate:"required"`

	// The globally unique name for this virtual server instance profile.
	Name *string `json:"name" validate:"required"`

	PortSpeed InstanceProfilePortSpeedIntf `json:"port_speed" validate:"required"`
}

// UnmarshalInstanceProfile unmarshals an instance of InstanceProfile from the specified map of raw messages.
func UnmarshalInstanceProfile(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfile)
	err = core.UnmarshalModel(m, "bandwidth", &obj.Bandwidth, UnmarshalInstanceProfileBandwidth)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "family", &obj.Family)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "port_speed", &obj.PortSpeed, UnmarshalInstanceProfilePortSpeed)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfileBandwidth : InstanceProfileBandwidth struct
// Models which "extend" this model:
// - InstanceProfileBandwidthFixed
// - InstanceProfileBandwidthRange
// - InstanceProfileBandwidthEnum
// - InstanceProfileBandwidthDependent
type InstanceProfileBandwidth struct {
	// The type for this profile field.
	Type *string `json:"type,omitempty"`

	// The value for this profile field.
	Value *int64 `json:"value,omitempty"`

	// The default value for this profile field.
	Default *int64 `json:"default,omitempty"`

	// The maximum value for this profile field.
	Max *int64 `json:"max,omitempty"`

	// The minimum value for this profile field.
	Min *int64 `json:"min,omitempty"`

	// The increment step value for this profile field.
	Step *int64 `json:"step,omitempty"`

	// The permitted values for this profile field.
	Values []int64 `json:"values,omitempty"`
}

// Constants associated with the InstanceProfileBandwidth.Type property.
// The type for this profile field.
const (
	InstanceProfileBandwidthTypeFixedConst = "fixed"
)

func (*InstanceProfileBandwidth) isaInstanceProfileBandwidth() bool {
	return true
}

type InstanceProfileBandwidthIntf interface {
	isaInstanceProfileBandwidth() bool
}

// UnmarshalInstanceProfileBandwidth unmarshals an instance of InstanceProfileBandwidth from the specified map of raw messages.
func UnmarshalInstanceProfileBandwidth(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfileBandwidth)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "default", &obj.Default)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max", &obj.Max)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "min", &obj.Min)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "step", &obj.Step)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "values", &obj.Values)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfileCollection : InstanceProfileCollection struct
type InstanceProfileCollection struct {
	// A link to the first page of resources.
	First *InstanceProfileCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *InstanceProfileCollectionNext `json:"next,omitempty"`

	// Collection of virtual server instance profiles.
	Profiles []InstanceProfile `json:"profiles" validate:"required"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalInstanceProfileCollection unmarshals an instance of InstanceProfileCollection from the specified map of raw messages.
func UnmarshalInstanceProfileCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfileCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalInstanceProfileCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalInstanceProfileCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profiles", &obj.Profiles, UnmarshalInstanceProfile)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfileCollectionFirst : A link to the first page of resources.
type InstanceProfileCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalInstanceProfileCollectionFirst unmarshals an instance of InstanceProfileCollectionFirst from the specified map of raw messages.
func UnmarshalInstanceProfileCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfileCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfileCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type InstanceProfileCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalInstanceProfileCollectionNext unmarshals an instance of InstanceProfileCollectionNext from the specified map of raw messages.
func UnmarshalInstanceProfileCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfileCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfileIdentity : Identifies an instance profile by a unique property.
// Models which "extend" this model:
// - InstanceProfileIdentityByName
// - InstanceProfileIdentityByCRN
// - InstanceProfileIdentityByHref
type InstanceProfileIdentity struct {
	// The globally unique name for this virtual server instance profile.
	Name *string `json:"name,omitempty"`

	// The CRN for this virtual server instance profile.
	CRN *string `json:"crn,omitempty"`

	// The URL for this virtual server instance profile.
	Href *string `json:"href,omitempty"`
}

func (*InstanceProfileIdentity) isaInstanceProfileIdentity() bool {
	return true
}

type InstanceProfileIdentityIntf interface {
	isaInstanceProfileIdentity() bool
}

// UnmarshalInstanceProfileIdentity unmarshals an instance of InstanceProfileIdentity from the specified map of raw messages.
func UnmarshalInstanceProfileIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfileIdentity)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfilePortSpeed : InstanceProfilePortSpeed struct
// Models which "extend" this model:
// - InstanceProfilePortSpeedFixed
// - InstanceProfilePortSpeedDependent
type InstanceProfilePortSpeed struct {
	// The type for this profile field.
	Type *string `json:"type,omitempty"`

	// The value for this profile field.
	Value *int64 `json:"value,omitempty"`
}

// Constants associated with the InstanceProfilePortSpeed.Type property.
// The type for this profile field.
const (
	InstanceProfilePortSpeedTypeFixedConst = "fixed"
)

func (*InstanceProfilePortSpeed) isaInstanceProfilePortSpeed() bool {
	return true
}

type InstanceProfilePortSpeedIntf interface {
	isaInstanceProfilePortSpeed() bool
}

// UnmarshalInstanceProfilePortSpeed unmarshals an instance of InstanceProfilePortSpeed from the specified map of raw messages.
func UnmarshalInstanceProfilePortSpeed(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfilePortSpeed)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceProfileReference : InstanceProfileReference struct
type InstanceProfileReference struct {
	// The CRN for this virtual server instance profile.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this virtual server instance profile.
	Href *string `json:"href" validate:"required"`

	// The globally unique name for this virtual server instance profile.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalInstanceProfileReference unmarshals an instance of InstanceProfileReference from the specified map of raw messages.
func UnmarshalInstanceProfileReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceProfileReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstancePrototype : InstancePrototype struct
// Models which "extend" this model:
// - InstancePrototypeInstanceByImage
type InstancePrototype struct {
	// The public SSH keys for the administrative user of the virtual server instance. Up to 10 keys may be provided; if no
	// keys are provided the instance will be inaccessible unless the image used provides another means of access. For
	// Windows instances, one of the keys will be used to encrypt the administrator password.
	//
	// Keys will be made available to the virtual server instance as cloud-init vendor data. For cloud-init enabled images,
	// these keys will also be added as SSH authorized keys for the administrative user.
	Keys []KeyIdentityIntf `json:"keys,omitempty"`

	// The user-defined name for this virtual server instance (and default system hostname). If unspecified, the name will
	// be a hyphenated list of randomly-selected words.
	Name *string `json:"name,omitempty"`

	// Collection of additional network interfaces to create for the virtual server instance.
	NetworkInterfaces []NetworkInterfacePrototype `json:"network_interfaces,omitempty"`

	// The profile to use for this virtual server instance.
	Profile InstanceProfileIdentityIntf `json:"profile,omitempty"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf `json:"resource_group,omitempty"`

	// User data to be made available when setting up the virtual server instance.
	UserData *string `json:"user_data,omitempty"`

	// Collection of volume attachments.
	VolumeAttachments []VolumeAttachmentPrototypeInstanceContext `json:"volume_attachments,omitempty"`

	// The VPC the virtual server instance is to be a part of. If provided, must match the
	// VPC tied to the subnets of the instance's network interfaces.
	VPC VPCIdentityIntf `json:"vpc,omitempty"`

	// The boot volume attachment for the virtual server instance.
	BootVolumeAttachment *VolumeAttachmentPrototypeInstanceByImageContext `json:"boot_volume_attachment,omitempty"`

	// The identity of the image to use when provisioning the virtual server instance.
	Image ImageIdentityIntf `json:"image,omitempty"`

	// Primary network interface.
	PrimaryNetworkInterface *NetworkInterfacePrototype `json:"primary_network_interface,omitempty"`

	// The identity of the zone to provision the virtual server instance in.
	Zone ZoneIdentityIntf `json:"zone,omitempty"`
}

func (*InstancePrototype) isaInstancePrototype() bool {
	return true
}

type InstancePrototypeIntf interface {
	isaInstancePrototype() bool
}

// UnmarshalInstancePrototype unmarshals an instance of InstancePrototype from the specified map of raw messages.
func UnmarshalInstancePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstancePrototype)
	err = core.UnmarshalModel(m, "keys", &obj.Keys, UnmarshalKeyIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_interfaces", &obj.NetworkInterfaces, UnmarshalNetworkInterfacePrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profile", &obj.Profile, UnmarshalInstanceProfileIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "user_data", &obj.UserData)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume_attachments", &obj.VolumeAttachments, UnmarshalVolumeAttachmentPrototypeInstanceContext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "boot_volume_attachment", &obj.BootVolumeAttachment, UnmarshalVolumeAttachmentPrototypeInstanceByImageContext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "image", &obj.Image, UnmarshalImageIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "primary_network_interface", &obj.PrimaryNetworkInterface, UnmarshalNetworkInterfacePrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceReference : InstanceReference struct
type InstanceReference struct {
	// The CRN for this virtual server instance.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this virtual server instance.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this virtual server instance.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this virtual server instance (and default system hostname).
	Name *string `json:"name" validate:"required"`
}

// UnmarshalInstanceReference unmarshals an instance of InstanceReference from the specified map of raw messages.
func UnmarshalInstanceReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// InstanceVcpu : The virtual server instance VCPU configuration.
type InstanceVcpu struct {
	// The VCPU architecture.
	Architecture *string `json:"architecture" validate:"required"`

	// The number of VCPUs assigned.
	Count *int64 `json:"count" validate:"required"`
}

// UnmarshalInstanceVcpu unmarshals an instance of InstanceVcpu from the specified map of raw messages.
func UnmarshalInstanceVcpu(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(InstanceVcpu)
	err = core.UnmarshalPrimitive(m, "architecture", &obj.Architecture)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "count", &obj.Count)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Key : Key struct
type Key struct {
	// The date and time that the key was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this key.
	CRN *string `json:"crn" validate:"required"`

	// The fingerprint for this key.  The value is returned base64-encoded and prefixed with the hash algorithm (always
	// `SHA256`).
	Fingerprint *string `json:"fingerprint" validate:"required"`

	// The URL for this key.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this key.
	ID *string `json:"id" validate:"required"`

	// The length of this key (in bits).
	Length *int64 `json:"length" validate:"required"`

	// The user-defined name for this key.
	Name *string `json:"name" validate:"required"`

	// The public SSH key.
	PublicKey *string `json:"public_key" validate:"required"`

	// The resource group for this key.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The crypto-system used by this key.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the Key.Type property.
// The crypto-system used by this key.
const (
	KeyTypeRsaConst = "rsa"
)

// UnmarshalKey unmarshals an instance of Key from the specified map of raw messages.
func UnmarshalKey(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Key)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "fingerprint", &obj.Fingerprint)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "length", &obj.Length)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "public_key", &obj.PublicKey)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// KeyCollection : KeyCollection struct
type KeyCollection struct {
	// A link to the first page of resources.
	First *KeyCollectionFirst `json:"first" validate:"required"`

	// Collection of keys.
	Keys []Key `json:"keys" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *KeyCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalKeyCollection unmarshals an instance of KeyCollection from the specified map of raw messages.
func UnmarshalKeyCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(KeyCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalKeyCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "keys", &obj.Keys, UnmarshalKey)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalKeyCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// KeyCollectionFirst : A link to the first page of resources.
type KeyCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalKeyCollectionFirst unmarshals an instance of KeyCollectionFirst from the specified map of raw messages.
func UnmarshalKeyCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(KeyCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// KeyCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type KeyCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalKeyCollectionNext unmarshals an instance of KeyCollectionNext from the specified map of raw messages.
func UnmarshalKeyCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(KeyCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// KeyIdentity : Identifies a key by a unique property.
// Models which "extend" this model:
// - KeyIdentityByID
// - KeyIdentityByCRN
// - KeyIdentityByHref
// - KeyIdentityKeyIdentityByFingerprint
type KeyIdentity struct {
	// The unique identifier for this key.
	ID *string `json:"id,omitempty"`

	// The CRN for this key.
	CRN *string `json:"crn,omitempty"`

	// The URL for this key.
	Href *string `json:"href,omitempty"`

	// The fingerprint for this key.  The value is returned base64-encoded and prefixed with the hash algorithm (always
	// `SHA256`).
	Fingerprint *string `json:"fingerprint,omitempty"`
}

func (*KeyIdentity) isaKeyIdentity() bool {
	return true
}

type KeyIdentityIntf interface {
	isaKeyIdentity() bool
}

// UnmarshalKeyIdentity unmarshals an instance of KeyIdentity from the specified map of raw messages.
func UnmarshalKeyIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(KeyIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "fingerprint", &obj.Fingerprint)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// KeyPatch : KeyPatch struct
type KeyPatch struct {
	// The user-defined name for this key.
	Name *string `json:"name,omitempty"`
}

// UnmarshalKeyPatch unmarshals an instance of KeyPatch from the specified map of raw messages.
func UnmarshalKeyPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(KeyPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the KeyPatch
func (keyPatch *KeyPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(keyPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// KeyReferenceInstanceInitializationContext : KeyReferenceInstanceInitializationContext struct
// Models which "extend" this model:
// - KeyReferenceInstanceInitializationContextKeyReference
// - KeyReferenceInstanceInitializationContextKeyIdentityByFingerprint
type KeyReferenceInstanceInitializationContext struct {
	// The CRN for this key.
	CRN *string `json:"crn,omitempty"`

	// The fingerprint for this key.  The value is returned base64-encoded and prefixed with the hash algorithm (always
	// `SHA256`).
	Fingerprint *string `json:"fingerprint,omitempty"`

	// The URL for this key.
	Href *string `json:"href,omitempty"`

	// The unique identifier for this key.
	ID *string `json:"id,omitempty"`

	// The user-defined name for this key.
	Name *string `json:"name,omitempty"`
}

func (*KeyReferenceInstanceInitializationContext) isaKeyReferenceInstanceInitializationContext() bool {
	return true
}

type KeyReferenceInstanceInitializationContextIntf interface {
	isaKeyReferenceInstanceInitializationContext() bool
}

// UnmarshalKeyReferenceInstanceInitializationContext unmarshals an instance of KeyReferenceInstanceInitializationContext from the specified map of raw messages.
func UnmarshalKeyReferenceInstanceInitializationContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(KeyReferenceInstanceInitializationContext)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "fingerprint", &obj.Fingerprint)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ListFloatingIpsOptions : The ListFloatingIps options.
type ListFloatingIpsOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListFloatingIpsOptions : Instantiate ListFloatingIpsOptions
func (*VpcClassicV1) NewListFloatingIpsOptions() *ListFloatingIpsOptions {
	return &ListFloatingIpsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListFloatingIpsOptions) SetStart(start string) *ListFloatingIpsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListFloatingIpsOptions) SetLimit(limit int64) *ListFloatingIpsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListFloatingIpsOptions) SetHeaders(param map[string]string) *ListFloatingIpsOptions {
	options.Headers = param
	return options
}

// ListIkePoliciesOptions : The ListIkePolicies options.
type ListIkePoliciesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListIkePoliciesOptions : Instantiate ListIkePoliciesOptions
func (*VpcClassicV1) NewListIkePoliciesOptions() *ListIkePoliciesOptions {
	return &ListIkePoliciesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListIkePoliciesOptions) SetStart(start string) *ListIkePoliciesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListIkePoliciesOptions) SetLimit(limit int64) *ListIkePoliciesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListIkePoliciesOptions) SetHeaders(param map[string]string) *ListIkePoliciesOptions {
	options.Headers = param
	return options
}

// ListIkePolicyConnectionsOptions : The ListIkePolicyConnections options.
type ListIkePolicyConnectionsOptions struct {
	// The IKE policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListIkePolicyConnectionsOptions : Instantiate ListIkePolicyConnectionsOptions
func (*VpcClassicV1) NewListIkePolicyConnectionsOptions(id string) *ListIkePolicyConnectionsOptions {
	return &ListIkePolicyConnectionsOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *ListIkePolicyConnectionsOptions) SetID(id string) *ListIkePolicyConnectionsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListIkePolicyConnectionsOptions) SetHeaders(param map[string]string) *ListIkePolicyConnectionsOptions {
	options.Headers = param
	return options
}

// ListImagesOptions : The ListImages options.
type ListImagesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Filters the collection to resources within one of the resource groups identified in a comma-separated list of
	// resource group identifiers.
	ResourceGroupID *string

	// Filters the collection to resources with the exact specified name.
	Name *string

	// Filters the collection to images with the specified `visibility`.
	Visibility *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the ListImagesOptions.Visibility property.
// Filters the collection to images with the specified `visibility`.
const (
	ListImagesOptionsVisibilityPrivateConst = "private"
	ListImagesOptionsVisibilityPublicConst  = "public"
)

// NewListImagesOptions : Instantiate ListImagesOptions
func (*VpcClassicV1) NewListImagesOptions() *ListImagesOptions {
	return &ListImagesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListImagesOptions) SetStart(start string) *ListImagesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListImagesOptions) SetLimit(limit int64) *ListImagesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetResourceGroupID : Allow user to set ResourceGroupID
func (options *ListImagesOptions) SetResourceGroupID(resourceGroupID string) *ListImagesOptions {
	options.ResourceGroupID = core.StringPtr(resourceGroupID)
	return options
}

// SetName : Allow user to set Name
func (options *ListImagesOptions) SetName(name string) *ListImagesOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetVisibility : Allow user to set Visibility
func (options *ListImagesOptions) SetVisibility(visibility string) *ListImagesOptions {
	options.Visibility = core.StringPtr(visibility)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListImagesOptions) SetHeaders(param map[string]string) *ListImagesOptions {
	options.Headers = param
	return options
}

// ListInstanceNetworkInterfaceFloatingIpsOptions : The ListInstanceNetworkInterfaceFloatingIps options.
type ListInstanceNetworkInterfaceFloatingIpsOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	NetworkInterfaceID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListInstanceNetworkInterfaceFloatingIpsOptions : Instantiate ListInstanceNetworkInterfaceFloatingIpsOptions
func (*VpcClassicV1) NewListInstanceNetworkInterfaceFloatingIpsOptions(instanceID string, networkInterfaceID string) *ListInstanceNetworkInterfaceFloatingIpsOptions {
	return &ListInstanceNetworkInterfaceFloatingIpsOptions{
		InstanceID:         core.StringPtr(instanceID),
		NetworkInterfaceID: core.StringPtr(networkInterfaceID),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *ListInstanceNetworkInterfaceFloatingIpsOptions) SetInstanceID(instanceID string) *ListInstanceNetworkInterfaceFloatingIpsOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetNetworkInterfaceID : Allow user to set NetworkInterfaceID
func (options *ListInstanceNetworkInterfaceFloatingIpsOptions) SetNetworkInterfaceID(networkInterfaceID string) *ListInstanceNetworkInterfaceFloatingIpsOptions {
	options.NetworkInterfaceID = core.StringPtr(networkInterfaceID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListInstanceNetworkInterfaceFloatingIpsOptions) SetHeaders(param map[string]string) *ListInstanceNetworkInterfaceFloatingIpsOptions {
	options.Headers = param
	return options
}

// ListInstanceNetworkInterfacesOptions : The ListInstanceNetworkInterfaces options.
type ListInstanceNetworkInterfacesOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListInstanceNetworkInterfacesOptions : Instantiate ListInstanceNetworkInterfacesOptions
func (*VpcClassicV1) NewListInstanceNetworkInterfacesOptions(instanceID string) *ListInstanceNetworkInterfacesOptions {
	return &ListInstanceNetworkInterfacesOptions{
		InstanceID: core.StringPtr(instanceID),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *ListInstanceNetworkInterfacesOptions) SetInstanceID(instanceID string) *ListInstanceNetworkInterfacesOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListInstanceNetworkInterfacesOptions) SetHeaders(param map[string]string) *ListInstanceNetworkInterfacesOptions {
	options.Headers = param
	return options
}

// ListInstanceProfilesOptions : The ListInstanceProfiles options.
type ListInstanceProfilesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListInstanceProfilesOptions : Instantiate ListInstanceProfilesOptions
func (*VpcClassicV1) NewListInstanceProfilesOptions() *ListInstanceProfilesOptions {
	return &ListInstanceProfilesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListInstanceProfilesOptions) SetStart(start string) *ListInstanceProfilesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListInstanceProfilesOptions) SetLimit(limit int64) *ListInstanceProfilesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListInstanceProfilesOptions) SetHeaders(param map[string]string) *ListInstanceProfilesOptions {
	options.Headers = param
	return options
}

// ListInstanceVolumeAttachmentsOptions : The ListInstanceVolumeAttachments options.
type ListInstanceVolumeAttachmentsOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListInstanceVolumeAttachmentsOptions : Instantiate ListInstanceVolumeAttachmentsOptions
func (*VpcClassicV1) NewListInstanceVolumeAttachmentsOptions(instanceID string) *ListInstanceVolumeAttachmentsOptions {
	return &ListInstanceVolumeAttachmentsOptions{
		InstanceID: core.StringPtr(instanceID),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *ListInstanceVolumeAttachmentsOptions) SetInstanceID(instanceID string) *ListInstanceVolumeAttachmentsOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListInstanceVolumeAttachmentsOptions) SetHeaders(param map[string]string) *ListInstanceVolumeAttachmentsOptions {
	options.Headers = param
	return options
}

// ListInstancesOptions : The ListInstances options.
type ListInstancesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Filters the collection to instances on the subnet of the specified identifier.
	NetworkInterfacesSubnetID *string

	// Filters the collection to instances on the subnet of the specified CRN.
	NetworkInterfacesSubnetCRN *string

	// Filters the collection to instances on the subnet of the specified name.
	NetworkInterfacesSubnetName *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListInstancesOptions : Instantiate ListInstancesOptions
func (*VpcClassicV1) NewListInstancesOptions() *ListInstancesOptions {
	return &ListInstancesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListInstancesOptions) SetStart(start string) *ListInstancesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListInstancesOptions) SetLimit(limit int64) *ListInstancesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetNetworkInterfacesSubnetID : Allow user to set NetworkInterfacesSubnetID
func (options *ListInstancesOptions) SetNetworkInterfacesSubnetID(networkInterfacesSubnetID string) *ListInstancesOptions {
	options.NetworkInterfacesSubnetID = core.StringPtr(networkInterfacesSubnetID)
	return options
}

// SetNetworkInterfacesSubnetCRN : Allow user to set NetworkInterfacesSubnetCRN
func (options *ListInstancesOptions) SetNetworkInterfacesSubnetCRN(networkInterfacesSubnetCRN string) *ListInstancesOptions {
	options.NetworkInterfacesSubnetCRN = core.StringPtr(networkInterfacesSubnetCRN)
	return options
}

// SetNetworkInterfacesSubnetName : Allow user to set NetworkInterfacesSubnetName
func (options *ListInstancesOptions) SetNetworkInterfacesSubnetName(networkInterfacesSubnetName string) *ListInstancesOptions {
	options.NetworkInterfacesSubnetName = core.StringPtr(networkInterfacesSubnetName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListInstancesOptions) SetHeaders(param map[string]string) *ListInstancesOptions {
	options.Headers = param
	return options
}

// ListIpsecPoliciesOptions : The ListIpsecPolicies options.
type ListIpsecPoliciesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListIpsecPoliciesOptions : Instantiate ListIpsecPoliciesOptions
func (*VpcClassicV1) NewListIpsecPoliciesOptions() *ListIpsecPoliciesOptions {
	return &ListIpsecPoliciesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListIpsecPoliciesOptions) SetStart(start string) *ListIpsecPoliciesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListIpsecPoliciesOptions) SetLimit(limit int64) *ListIpsecPoliciesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListIpsecPoliciesOptions) SetHeaders(param map[string]string) *ListIpsecPoliciesOptions {
	options.Headers = param
	return options
}

// ListIpsecPolicyConnectionsOptions : The ListIpsecPolicyConnections options.
type ListIpsecPolicyConnectionsOptions struct {
	// The IPsec policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListIpsecPolicyConnectionsOptions : Instantiate ListIpsecPolicyConnectionsOptions
func (*VpcClassicV1) NewListIpsecPolicyConnectionsOptions(id string) *ListIpsecPolicyConnectionsOptions {
	return &ListIpsecPolicyConnectionsOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *ListIpsecPolicyConnectionsOptions) SetID(id string) *ListIpsecPolicyConnectionsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListIpsecPolicyConnectionsOptions) SetHeaders(param map[string]string) *ListIpsecPolicyConnectionsOptions {
	options.Headers = param
	return options
}

// ListKeysOptions : The ListKeys options.
type ListKeysOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListKeysOptions : Instantiate ListKeysOptions
func (*VpcClassicV1) NewListKeysOptions() *ListKeysOptions {
	return &ListKeysOptions{}
}

// SetStart : Allow user to set Start
func (options *ListKeysOptions) SetStart(start string) *ListKeysOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListKeysOptions) SetLimit(limit int64) *ListKeysOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListKeysOptions) SetHeaders(param map[string]string) *ListKeysOptions {
	options.Headers = param
	return options
}

// ListLoadBalancerListenerPoliciesOptions : The ListLoadBalancerListenerPolicies options.
type ListLoadBalancerListenerPoliciesOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListLoadBalancerListenerPoliciesOptions : Instantiate ListLoadBalancerListenerPoliciesOptions
func (*VpcClassicV1) NewListLoadBalancerListenerPoliciesOptions(loadBalancerID string, listenerID string) *ListLoadBalancerListenerPoliciesOptions {
	return &ListLoadBalancerListenerPoliciesOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *ListLoadBalancerListenerPoliciesOptions) SetLoadBalancerID(loadBalancerID string) *ListLoadBalancerListenerPoliciesOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *ListLoadBalancerListenerPoliciesOptions) SetListenerID(listenerID string) *ListLoadBalancerListenerPoliciesOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListLoadBalancerListenerPoliciesOptions) SetHeaders(param map[string]string) *ListLoadBalancerListenerPoliciesOptions {
	options.Headers = param
	return options
}

// ListLoadBalancerListenerPolicyRulesOptions : The ListLoadBalancerListenerPolicyRules options.
type ListLoadBalancerListenerPolicyRulesOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListLoadBalancerListenerPolicyRulesOptions : Instantiate ListLoadBalancerListenerPolicyRulesOptions
func (*VpcClassicV1) NewListLoadBalancerListenerPolicyRulesOptions(loadBalancerID string, listenerID string, policyID string) *ListLoadBalancerListenerPolicyRulesOptions {
	return &ListLoadBalancerListenerPolicyRulesOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *ListLoadBalancerListenerPolicyRulesOptions) SetLoadBalancerID(loadBalancerID string) *ListLoadBalancerListenerPolicyRulesOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *ListLoadBalancerListenerPolicyRulesOptions) SetListenerID(listenerID string) *ListLoadBalancerListenerPolicyRulesOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *ListLoadBalancerListenerPolicyRulesOptions) SetPolicyID(policyID string) *ListLoadBalancerListenerPolicyRulesOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListLoadBalancerListenerPolicyRulesOptions) SetHeaders(param map[string]string) *ListLoadBalancerListenerPolicyRulesOptions {
	options.Headers = param
	return options
}

// ListLoadBalancerListenersOptions : The ListLoadBalancerListeners options.
type ListLoadBalancerListenersOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListLoadBalancerListenersOptions : Instantiate ListLoadBalancerListenersOptions
func (*VpcClassicV1) NewListLoadBalancerListenersOptions(loadBalancerID string) *ListLoadBalancerListenersOptions {
	return &ListLoadBalancerListenersOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *ListLoadBalancerListenersOptions) SetLoadBalancerID(loadBalancerID string) *ListLoadBalancerListenersOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListLoadBalancerListenersOptions) SetHeaders(param map[string]string) *ListLoadBalancerListenersOptions {
	options.Headers = param
	return options
}

// ListLoadBalancerPoolMembersOptions : The ListLoadBalancerPoolMembers options.
type ListLoadBalancerPoolMembersOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListLoadBalancerPoolMembersOptions : Instantiate ListLoadBalancerPoolMembersOptions
func (*VpcClassicV1) NewListLoadBalancerPoolMembersOptions(loadBalancerID string, poolID string) *ListLoadBalancerPoolMembersOptions {
	return &ListLoadBalancerPoolMembersOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *ListLoadBalancerPoolMembersOptions) SetLoadBalancerID(loadBalancerID string) *ListLoadBalancerPoolMembersOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *ListLoadBalancerPoolMembersOptions) SetPoolID(poolID string) *ListLoadBalancerPoolMembersOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListLoadBalancerPoolMembersOptions) SetHeaders(param map[string]string) *ListLoadBalancerPoolMembersOptions {
	options.Headers = param
	return options
}

// ListLoadBalancerPoolsOptions : The ListLoadBalancerPools options.
type ListLoadBalancerPoolsOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListLoadBalancerPoolsOptions : Instantiate ListLoadBalancerPoolsOptions
func (*VpcClassicV1) NewListLoadBalancerPoolsOptions(loadBalancerID string) *ListLoadBalancerPoolsOptions {
	return &ListLoadBalancerPoolsOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *ListLoadBalancerPoolsOptions) SetLoadBalancerID(loadBalancerID string) *ListLoadBalancerPoolsOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListLoadBalancerPoolsOptions) SetHeaders(param map[string]string) *ListLoadBalancerPoolsOptions {
	options.Headers = param
	return options
}

// ListLoadBalancersOptions : The ListLoadBalancers options.
type ListLoadBalancersOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListLoadBalancersOptions : Instantiate ListLoadBalancersOptions
func (*VpcClassicV1) NewListLoadBalancersOptions() *ListLoadBalancersOptions {
	return &ListLoadBalancersOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListLoadBalancersOptions) SetHeaders(param map[string]string) *ListLoadBalancersOptions {
	options.Headers = param
	return options
}

// ListNetworkACLRulesOptions : The ListNetworkACLRules options.
type ListNetworkACLRulesOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Filters the collection to rules with the specified direction.
	Direction *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the ListNetworkACLRulesOptions.Direction property.
// Filters the collection to rules with the specified direction.
const (
	ListNetworkACLRulesOptionsDirectionInboundConst  = "inbound"
	ListNetworkACLRulesOptionsDirectionOutboundConst = "outbound"
)

// NewListNetworkACLRulesOptions : Instantiate ListNetworkACLRulesOptions
func (*VpcClassicV1) NewListNetworkACLRulesOptions(networkACLID string) *ListNetworkACLRulesOptions {
	return &ListNetworkACLRulesOptions{
		NetworkACLID: core.StringPtr(networkACLID),
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *ListNetworkACLRulesOptions) SetNetworkACLID(networkACLID string) *ListNetworkACLRulesOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetStart : Allow user to set Start
func (options *ListNetworkACLRulesOptions) SetStart(start string) *ListNetworkACLRulesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListNetworkACLRulesOptions) SetLimit(limit int64) *ListNetworkACLRulesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetDirection : Allow user to set Direction
func (options *ListNetworkACLRulesOptions) SetDirection(direction string) *ListNetworkACLRulesOptions {
	options.Direction = core.StringPtr(direction)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListNetworkACLRulesOptions) SetHeaders(param map[string]string) *ListNetworkACLRulesOptions {
	options.Headers = param
	return options
}

// ListNetworkAclsOptions : The ListNetworkAcls options.
type ListNetworkAclsOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListNetworkAclsOptions : Instantiate ListNetworkAclsOptions
func (*VpcClassicV1) NewListNetworkAclsOptions() *ListNetworkAclsOptions {
	return &ListNetworkAclsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListNetworkAclsOptions) SetStart(start string) *ListNetworkAclsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListNetworkAclsOptions) SetLimit(limit int64) *ListNetworkAclsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListNetworkAclsOptions) SetHeaders(param map[string]string) *ListNetworkAclsOptions {
	options.Headers = param
	return options
}

// ListOperatingSystemsOptions : The ListOperatingSystems options.
type ListOperatingSystemsOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListOperatingSystemsOptions : Instantiate ListOperatingSystemsOptions
func (*VpcClassicV1) NewListOperatingSystemsOptions() *ListOperatingSystemsOptions {
	return &ListOperatingSystemsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListOperatingSystemsOptions) SetStart(start string) *ListOperatingSystemsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListOperatingSystemsOptions) SetLimit(limit int64) *ListOperatingSystemsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListOperatingSystemsOptions) SetHeaders(param map[string]string) *ListOperatingSystemsOptions {
	options.Headers = param
	return options
}

// ListPublicGatewaysOptions : The ListPublicGateways options.
type ListPublicGatewaysOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListPublicGatewaysOptions : Instantiate ListPublicGatewaysOptions
func (*VpcClassicV1) NewListPublicGatewaysOptions() *ListPublicGatewaysOptions {
	return &ListPublicGatewaysOptions{}
}

// SetStart : Allow user to set Start
func (options *ListPublicGatewaysOptions) SetStart(start string) *ListPublicGatewaysOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListPublicGatewaysOptions) SetLimit(limit int64) *ListPublicGatewaysOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListPublicGatewaysOptions) SetHeaders(param map[string]string) *ListPublicGatewaysOptions {
	options.Headers = param
	return options
}

// ListRegionZonesOptions : The ListRegionZones options.
type ListRegionZonesOptions struct {
	// The region name.
	RegionName *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListRegionZonesOptions : Instantiate ListRegionZonesOptions
func (*VpcClassicV1) NewListRegionZonesOptions(regionName string) *ListRegionZonesOptions {
	return &ListRegionZonesOptions{
		RegionName: core.StringPtr(regionName),
	}
}

// SetRegionName : Allow user to set RegionName
func (options *ListRegionZonesOptions) SetRegionName(regionName string) *ListRegionZonesOptions {
	options.RegionName = core.StringPtr(regionName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListRegionZonesOptions) SetHeaders(param map[string]string) *ListRegionZonesOptions {
	options.Headers = param
	return options
}

// ListRegionsOptions : The ListRegions options.
type ListRegionsOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListRegionsOptions : Instantiate ListRegionsOptions
func (*VpcClassicV1) NewListRegionsOptions() *ListRegionsOptions {
	return &ListRegionsOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListRegionsOptions) SetHeaders(param map[string]string) *ListRegionsOptions {
	options.Headers = param
	return options
}

// ListSecurityGroupNetworkInterfacesOptions : The ListSecurityGroupNetworkInterfaces options.
type ListSecurityGroupNetworkInterfacesOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListSecurityGroupNetworkInterfacesOptions : Instantiate ListSecurityGroupNetworkInterfacesOptions
func (*VpcClassicV1) NewListSecurityGroupNetworkInterfacesOptions(securityGroupID string) *ListSecurityGroupNetworkInterfacesOptions {
	return &ListSecurityGroupNetworkInterfacesOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *ListSecurityGroupNetworkInterfacesOptions) SetSecurityGroupID(securityGroupID string) *ListSecurityGroupNetworkInterfacesOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListSecurityGroupNetworkInterfacesOptions) SetHeaders(param map[string]string) *ListSecurityGroupNetworkInterfacesOptions {
	options.Headers = param
	return options
}

// ListSecurityGroupRulesOptions : The ListSecurityGroupRules options.
type ListSecurityGroupRulesOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListSecurityGroupRulesOptions : Instantiate ListSecurityGroupRulesOptions
func (*VpcClassicV1) NewListSecurityGroupRulesOptions(securityGroupID string) *ListSecurityGroupRulesOptions {
	return &ListSecurityGroupRulesOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *ListSecurityGroupRulesOptions) SetSecurityGroupID(securityGroupID string) *ListSecurityGroupRulesOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListSecurityGroupRulesOptions) SetHeaders(param map[string]string) *ListSecurityGroupRulesOptions {
	options.Headers = param
	return options
}

// ListSecurityGroupsOptions : The ListSecurityGroups options.
type ListSecurityGroupsOptions struct {
	// Filters the collection to resources in the VPC with the specified identifier.
	VPCID *string

	// Filters the collection to resources in the VPC with the specified CRN.
	VPCCRN *string

	// Filters the collection to resources in the VPC with the exact specified name.
	VPCName *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListSecurityGroupsOptions : Instantiate ListSecurityGroupsOptions
func (*VpcClassicV1) NewListSecurityGroupsOptions() *ListSecurityGroupsOptions {
	return &ListSecurityGroupsOptions{}
}

// SetVPCID : Allow user to set VPCID
func (options *ListSecurityGroupsOptions) SetVPCID(vpcID string) *ListSecurityGroupsOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetVPCCRN : Allow user to set VPCCRN
func (options *ListSecurityGroupsOptions) SetVPCCRN(vpcCRN string) *ListSecurityGroupsOptions {
	options.VPCCRN = core.StringPtr(vpcCRN)
	return options
}

// SetVPCName : Allow user to set VPCName
func (options *ListSecurityGroupsOptions) SetVPCName(vpcName string) *ListSecurityGroupsOptions {
	options.VPCName = core.StringPtr(vpcName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListSecurityGroupsOptions) SetHeaders(param map[string]string) *ListSecurityGroupsOptions {
	options.Headers = param
	return options
}

// ListSubnetsOptions : The ListSubnets options.
type ListSubnetsOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListSubnetsOptions : Instantiate ListSubnetsOptions
func (*VpcClassicV1) NewListSubnetsOptions() *ListSubnetsOptions {
	return &ListSubnetsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListSubnetsOptions) SetStart(start string) *ListSubnetsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListSubnetsOptions) SetLimit(limit int64) *ListSubnetsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListSubnetsOptions) SetHeaders(param map[string]string) *ListSubnetsOptions {
	options.Headers = param
	return options
}

// ListVolumeProfilesOptions : The ListVolumeProfiles options.
type ListVolumeProfilesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVolumeProfilesOptions : Instantiate ListVolumeProfilesOptions
func (*VpcClassicV1) NewListVolumeProfilesOptions() *ListVolumeProfilesOptions {
	return &ListVolumeProfilesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListVolumeProfilesOptions) SetStart(start string) *ListVolumeProfilesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListVolumeProfilesOptions) SetLimit(limit int64) *ListVolumeProfilesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVolumeProfilesOptions) SetHeaders(param map[string]string) *ListVolumeProfilesOptions {
	options.Headers = param
	return options
}

// ListVolumesOptions : The ListVolumes options.
type ListVolumesOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Filters the collection to resources with the exact specified name.
	Name *string

	// Filters the collection to resources in the zone with the exact specified name.
	ZoneName *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVolumesOptions : Instantiate ListVolumesOptions
func (*VpcClassicV1) NewListVolumesOptions() *ListVolumesOptions {
	return &ListVolumesOptions{}
}

// SetStart : Allow user to set Start
func (options *ListVolumesOptions) SetStart(start string) *ListVolumesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListVolumesOptions) SetLimit(limit int64) *ListVolumesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetName : Allow user to set Name
func (options *ListVolumesOptions) SetName(name string) *ListVolumesOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetZoneName : Allow user to set ZoneName
func (options *ListVolumesOptions) SetZoneName(zoneName string) *ListVolumesOptions {
	options.ZoneName = core.StringPtr(zoneName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVolumesOptions) SetHeaders(param map[string]string) *ListVolumesOptions {
	options.Headers = param
	return options
}

// ListVPCAddressPrefixesOptions : The ListVPCAddressPrefixes options.
type ListVPCAddressPrefixesOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVPCAddressPrefixesOptions : Instantiate ListVPCAddressPrefixesOptions
func (*VpcClassicV1) NewListVPCAddressPrefixesOptions(vpcID string) *ListVPCAddressPrefixesOptions {
	return &ListVPCAddressPrefixesOptions{
		VPCID: core.StringPtr(vpcID),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *ListVPCAddressPrefixesOptions) SetVPCID(vpcID string) *ListVPCAddressPrefixesOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetStart : Allow user to set Start
func (options *ListVPCAddressPrefixesOptions) SetStart(start string) *ListVPCAddressPrefixesOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListVPCAddressPrefixesOptions) SetLimit(limit int64) *ListVPCAddressPrefixesOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVPCAddressPrefixesOptions) SetHeaders(param map[string]string) *ListVPCAddressPrefixesOptions {
	options.Headers = param
	return options
}

// ListVPCRoutesOptions : The ListVPCRoutes options.
type ListVPCRoutesOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// Filters the collection to resources in the zone with the exact specified name.
	ZoneName *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVPCRoutesOptions : Instantiate ListVPCRoutesOptions
func (*VpcClassicV1) NewListVPCRoutesOptions(vpcID string) *ListVPCRoutesOptions {
	return &ListVPCRoutesOptions{
		VPCID: core.StringPtr(vpcID),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *ListVPCRoutesOptions) SetVPCID(vpcID string) *ListVPCRoutesOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetZoneName : Allow user to set ZoneName
func (options *ListVPCRoutesOptions) SetZoneName(zoneName string) *ListVPCRoutesOptions {
	options.ZoneName = core.StringPtr(zoneName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVPCRoutesOptions) SetHeaders(param map[string]string) *ListVPCRoutesOptions {
	options.Headers = param
	return options
}

// ListVpcsOptions : The ListVpcs options.
type ListVpcsOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// The `classic_access` parameter filters the returned collection by the supplied field. If the supplied field is
	// `true`, only Classic Access VPCs will be returned. If the supplied field is `false`, only VPCs without Classic
	// Access will be returned.
	ClassicAccess *bool

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVpcsOptions : Instantiate ListVpcsOptions
func (*VpcClassicV1) NewListVpcsOptions() *ListVpcsOptions {
	return &ListVpcsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListVpcsOptions) SetStart(start string) *ListVpcsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListVpcsOptions) SetLimit(limit int64) *ListVpcsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetClassicAccess : Allow user to set ClassicAccess
func (options *ListVpcsOptions) SetClassicAccess(classicAccess bool) *ListVpcsOptions {
	options.ClassicAccess = core.BoolPtr(classicAccess)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVpcsOptions) SetHeaders(param map[string]string) *ListVpcsOptions {
	options.Headers = param
	return options
}

// ListVPNGatewayConnectionLocalCIDRsOptions : The ListVPNGatewayConnectionLocalCIDRs options.
type ListVPNGatewayConnectionLocalCIDRsOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVPNGatewayConnectionLocalCIDRsOptions : Instantiate ListVPNGatewayConnectionLocalCIDRsOptions
func (*VpcClassicV1) NewListVPNGatewayConnectionLocalCIDRsOptions(vpnGatewayID string, id string) *ListVPNGatewayConnectionLocalCIDRsOptions {
	return &ListVPNGatewayConnectionLocalCIDRsOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *ListVPNGatewayConnectionLocalCIDRsOptions) SetVPNGatewayID(vpnGatewayID string) *ListVPNGatewayConnectionLocalCIDRsOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *ListVPNGatewayConnectionLocalCIDRsOptions) SetID(id string) *ListVPNGatewayConnectionLocalCIDRsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVPNGatewayConnectionLocalCIDRsOptions) SetHeaders(param map[string]string) *ListVPNGatewayConnectionLocalCIDRsOptions {
	options.Headers = param
	return options
}

// ListVPNGatewayConnectionPeerCIDRsOptions : The ListVPNGatewayConnectionPeerCIDRs options.
type ListVPNGatewayConnectionPeerCIDRsOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVPNGatewayConnectionPeerCIDRsOptions : Instantiate ListVPNGatewayConnectionPeerCIDRsOptions
func (*VpcClassicV1) NewListVPNGatewayConnectionPeerCIDRsOptions(vpnGatewayID string, id string) *ListVPNGatewayConnectionPeerCIDRsOptions {
	return &ListVPNGatewayConnectionPeerCIDRsOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *ListVPNGatewayConnectionPeerCIDRsOptions) SetVPNGatewayID(vpnGatewayID string) *ListVPNGatewayConnectionPeerCIDRsOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *ListVPNGatewayConnectionPeerCIDRsOptions) SetID(id string) *ListVPNGatewayConnectionPeerCIDRsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVPNGatewayConnectionPeerCIDRsOptions) SetHeaders(param map[string]string) *ListVPNGatewayConnectionPeerCIDRsOptions {
	options.Headers = param
	return options
}

// ListVPNGatewayConnectionsOptions : The ListVPNGatewayConnections options.
type ListVPNGatewayConnectionsOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// Filters the collection to VPN gateway connections with the specified status.
	Status *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListVPNGatewayConnectionsOptions : Instantiate ListVPNGatewayConnectionsOptions
func (*VpcClassicV1) NewListVPNGatewayConnectionsOptions(vpnGatewayID string) *ListVPNGatewayConnectionsOptions {
	return &ListVPNGatewayConnectionsOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *ListVPNGatewayConnectionsOptions) SetVPNGatewayID(vpnGatewayID string) *ListVPNGatewayConnectionsOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetStatus : Allow user to set Status
func (options *ListVPNGatewayConnectionsOptions) SetStatus(status string) *ListVPNGatewayConnectionsOptions {
	options.Status = core.StringPtr(status)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVPNGatewayConnectionsOptions) SetHeaders(param map[string]string) *ListVPNGatewayConnectionsOptions {
	options.Headers = param
	return options
}

// ListVPNGatewaysOptions : The ListVPNGateways options.
type ListVPNGatewaysOptions struct {
	// A server-supplied token determining what resource to start the page on.
	Start *string

	// The number of resources to return on a page.
	Limit *int64

	// Filters the collection to resources within one of the resource groups identified in a comma-separated list of
	// resource group identifiers.
	ResourceGroupID *string

	// Filters the collection to VPN gateways with the specified mode.
	Mode *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the ListVPNGatewaysOptions.Mode property.
// Filters the collection to VPN gateways with the specified mode.
const (
	ListVPNGatewaysOptionsModePolicyConst = "policy"
	ListVPNGatewaysOptionsModeRouteConst  = "route"
)

// NewListVPNGatewaysOptions : Instantiate ListVPNGatewaysOptions
func (*VpcClassicV1) NewListVPNGatewaysOptions() *ListVPNGatewaysOptions {
	return &ListVPNGatewaysOptions{}
}

// SetStart : Allow user to set Start
func (options *ListVPNGatewaysOptions) SetStart(start string) *ListVPNGatewaysOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListVPNGatewaysOptions) SetLimit(limit int64) *ListVPNGatewaysOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetResourceGroupID : Allow user to set ResourceGroupID
func (options *ListVPNGatewaysOptions) SetResourceGroupID(resourceGroupID string) *ListVPNGatewaysOptions {
	options.ResourceGroupID = core.StringPtr(resourceGroupID)
	return options
}

// SetMode : Allow user to set Mode
func (options *ListVPNGatewaysOptions) SetMode(mode string) *ListVPNGatewaysOptions {
	options.Mode = core.StringPtr(mode)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListVPNGatewaysOptions) SetHeaders(param map[string]string) *ListVPNGatewaysOptions {
	options.Headers = param
	return options
}

// LoadBalancer : LoadBalancer struct
type LoadBalancer struct {
	// The date and time that this load balancer was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The load balancer's CRN.
	CRN *string `json:"crn" validate:"required"`

	// Fully qualified domain name assigned to this load balancer.
	Hostname *string `json:"hostname" validate:"required"`

	// The load balancer's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer.
	ID *string `json:"id" validate:"required"`

	// The type of this load balancer, public or private.
	IsPublic *bool `json:"is_public" validate:"required"`

	// The listeners of this load balancer.
	Listeners []LoadBalancerListenerReference `json:"listeners" validate:"required"`

	// The logging configuration for this load balancer.
	Logging *LoadBalancerLogging `json:"logging" validate:"required"`

	// The unique user-defined name for this load balancer.
	Name *string `json:"name" validate:"required"`

	// The operating status of this load balancer.
	OperatingStatus *string `json:"operating_status" validate:"required"`

	// The pools of this load balancer.
	Pools []LoadBalancerPoolReference `json:"pools" validate:"required"`

	// The private IP addresses assigned to this load balancer.
	PrivateIps []IP `json:"private_ips" validate:"required"`

	// The provisioning status of this load balancer.
	ProvisioningStatus *string `json:"provisioning_status" validate:"required"`

	// The public IP addresses assigned to this load balancer.
	//
	// Applicable only for public load balancers.
	PublicIps []IP `json:"public_ips" validate:"required"`

	// The resource group for this load balancer.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The subnets this load balancer is part of.
	Subnets []SubnetReference `json:"subnets" validate:"required"`
}

// Constants associated with the LoadBalancer.OperatingStatus property.
// The operating status of this load balancer.
const (
	LoadBalancerOperatingStatusOfflineConst = "offline"
	LoadBalancerOperatingStatusOnlineConst  = "online"
)

// Constants associated with the LoadBalancer.ProvisioningStatus property.
// The provisioning status of this load balancer.
const (
	LoadBalancerProvisioningStatusActiveConst             = "active"
	LoadBalancerProvisioningStatusCreatePendingConst      = "create_pending"
	LoadBalancerProvisioningStatusDeletePendingConst      = "delete_pending"
	LoadBalancerProvisioningStatusFailedConst             = "failed"
	LoadBalancerProvisioningStatusMaintenancePendingConst = "maintenance_pending"
	LoadBalancerProvisioningStatusUpdatePendingConst      = "update_pending"
)

// UnmarshalLoadBalancer unmarshals an instance of LoadBalancer from the specified map of raw messages.
func UnmarshalLoadBalancer(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancer)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "hostname", &obj.Hostname)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "is_public", &obj.IsPublic)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "listeners", &obj.Listeners, UnmarshalLoadBalancerListenerReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "logging", &obj.Logging, UnmarshalLoadBalancerLogging)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "operating_status", &obj.OperatingStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "pools", &obj.Pools, UnmarshalLoadBalancerPoolReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "private_ips", &obj.PrivateIps, UnmarshalIP)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisioning_status", &obj.ProvisioningStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "public_ips", &obj.PublicIps, UnmarshalIP)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnets", &obj.Subnets, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerCollection : LoadBalancerCollection struct
type LoadBalancerCollection struct {
	// Collection of load balancers.
	LoadBalancers []LoadBalancer `json:"load_balancers" validate:"required"`
}

// UnmarshalLoadBalancerCollection unmarshals an instance of LoadBalancerCollection from the specified map of raw messages.
func UnmarshalLoadBalancerCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerCollection)
	err = core.UnmarshalModel(m, "load_balancers", &obj.LoadBalancers, UnmarshalLoadBalancer)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListener : LoadBalancerListener struct
type LoadBalancerListener struct {
	// If set to `true`, this listener will accept and forward PROXY protocol information. Supported by load balancers in
	// the `application` family (otherwise always `false`).
	AcceptProxyProtocol *bool `json:"accept_proxy_protocol" validate:"required"`

	// The certificate instance used for SSL termination. It is applicable only to `https`
	// protocol.
	CertificateInstance *CertificateInstanceReference `json:"certificate_instance,omitempty"`

	// The connection limit of the listener.
	ConnectionLimit *int64 `json:"connection_limit,omitempty"`

	// The date and time that this listener was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The default pool associated with the listener.
	DefaultPool *LoadBalancerPoolReference `json:"default_pool,omitempty"`

	// The listener's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer listener.
	ID *string `json:"id" validate:"required"`

	// An array of policies for this listener.
	Policies []LoadBalancerListenerPolicyReference `json:"policies,omitempty"`

	// The listener port number. Each listener in the load balancer must have a unique
	// `port` and `protocol` combination.
	Port *int64 `json:"port" validate:"required"`

	// The listener protocol.
	Protocol *string `json:"protocol" validate:"required"`

	// The provisioning status of this listener.
	ProvisioningStatus *string `json:"provisioning_status" validate:"required"`
}

// Constants associated with the LoadBalancerListener.Protocol property.
// The listener protocol.
const (
	LoadBalancerListenerProtocolHTTPConst  = "http"
	LoadBalancerListenerProtocolHTTPSConst = "https"
	LoadBalancerListenerProtocolTCPConst   = "tcp"
)

// Constants associated with the LoadBalancerListener.ProvisioningStatus property.
// The provisioning status of this listener.
const (
	LoadBalancerListenerProvisioningStatusActiveConst             = "active"
	LoadBalancerListenerProvisioningStatusCreatePendingConst      = "create_pending"
	LoadBalancerListenerProvisioningStatusDeletePendingConst      = "delete_pending"
	LoadBalancerListenerProvisioningStatusFailedConst             = "failed"
	LoadBalancerListenerProvisioningStatusMaintenancePendingConst = "maintenance_pending"
	LoadBalancerListenerProvisioningStatusUpdatePendingConst      = "update_pending"
)

// UnmarshalLoadBalancerListener unmarshals an instance of LoadBalancerListener from the specified map of raw messages.
func UnmarshalLoadBalancerListener(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListener)
	err = core.UnmarshalPrimitive(m, "accept_proxy_protocol", &obj.AcceptProxyProtocol)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "certificate_instance", &obj.CertificateInstance, UnmarshalCertificateInstanceReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "connection_limit", &obj.ConnectionLimit)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "default_pool", &obj.DefaultPool, UnmarshalLoadBalancerPoolReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "policies", &obj.Policies, UnmarshalLoadBalancerListenerPolicyReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "protocol", &obj.Protocol)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisioning_status", &obj.ProvisioningStatus)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerCollection : LoadBalancerListenerCollection struct
type LoadBalancerListenerCollection struct {
	// Collection of listeners.
	Listeners []LoadBalancerListener `json:"listeners" validate:"required"`
}

// UnmarshalLoadBalancerListenerCollection unmarshals an instance of LoadBalancerListenerCollection from the specified map of raw messages.
func UnmarshalLoadBalancerListenerCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerCollection)
	err = core.UnmarshalModel(m, "listeners", &obj.Listeners, UnmarshalLoadBalancerListener)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPatch : LoadBalancerListenerPatch struct
type LoadBalancerListenerPatch struct {
	// If set to `true`, this listener will accept and forward PROXY protocol information. Supported by load balancers in
	// the `application` family (otherwise always `false`).
	AcceptProxyProtocol *bool `json:"accept_proxy_protocol,omitempty"`

	// The certificate instance used for SSL termination. It is applicable only to `https`
	// protocol.
	CertificateInstance CertificateInstanceIdentityIntf `json:"certificate_instance,omitempty"`

	// The connection limit of the listener.
	ConnectionLimit *int64 `json:"connection_limit,omitempty"`

	// The default pool associated with the listener. The specified pool must:
	//
	// - Belong to this load balancer
	// - Have the same `protocol` as this listener
	// - Not already be the default pool for another listener.
	DefaultPool LoadBalancerPoolIdentityIntf `json:"default_pool,omitempty"`

	// The listener port number. Each listener in the load balancer must have a unique
	// `port` and `protocol` combination.
	Port *int64 `json:"port,omitempty"`

	// The listener protocol.
	Protocol *string `json:"protocol,omitempty"`
}

// Constants associated with the LoadBalancerListenerPatch.Protocol property.
// The listener protocol.
const (
	LoadBalancerListenerPatchProtocolHTTPConst  = "http"
	LoadBalancerListenerPatchProtocolHTTPSConst = "https"
	LoadBalancerListenerPatchProtocolTCPConst   = "tcp"
)

// UnmarshalLoadBalancerListenerPatch unmarshals an instance of LoadBalancerListenerPatch from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPatch)
	err = core.UnmarshalPrimitive(m, "accept_proxy_protocol", &obj.AcceptProxyProtocol)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "certificate_instance", &obj.CertificateInstance, UnmarshalCertificateInstanceIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "connection_limit", &obj.ConnectionLimit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "default_pool", &obj.DefaultPool, UnmarshalLoadBalancerPoolIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "protocol", &obj.Protocol)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the LoadBalancerListenerPatch
func (loadBalancerListenerPatch *LoadBalancerListenerPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(loadBalancerListenerPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// LoadBalancerListenerPolicy : LoadBalancerListenerPolicy struct
type LoadBalancerListenerPolicy struct {
	// The policy action.
	Action *string `json:"action" validate:"required"`

	// The date and time that this policy was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The listener policy's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The policy's unique identifier.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this policy.
	Name *string `json:"name" validate:"required"`

	// Priority of the policy. Lower value indicates higher priority.
	Priority *int64 `json:"priority" validate:"required"`

	// The provisioning status of this policy.
	ProvisioningStatus *string `json:"provisioning_status" validate:"required"`

	// The rules of this policy.
	Rules []LoadBalancerListenerPolicyRuleReference `json:"rules" validate:"required"`

	// `LoadBalancerPoolReference` is in the response if `action` is `forward`.
	// `LoadBalancerListenerPolicyRedirectURL` is in the response if `action` is `redirect`.
	Target LoadBalancerListenerPolicyTargetIntf `json:"target,omitempty"`
}

// Constants associated with the LoadBalancerListenerPolicy.Action property.
// The policy action.
const (
	LoadBalancerListenerPolicyActionForwardConst  = "forward"
	LoadBalancerListenerPolicyActionRedirectConst = "redirect"
	LoadBalancerListenerPolicyActionRejectConst   = "reject"
)

// Constants associated with the LoadBalancerListenerPolicy.ProvisioningStatus property.
// The provisioning status of this policy.
const (
	LoadBalancerListenerPolicyProvisioningStatusActiveConst             = "active"
	LoadBalancerListenerPolicyProvisioningStatusCreatePendingConst      = "create_pending"
	LoadBalancerListenerPolicyProvisioningStatusDeletePendingConst      = "delete_pending"
	LoadBalancerListenerPolicyProvisioningStatusFailedConst             = "failed"
	LoadBalancerListenerPolicyProvisioningStatusMaintenancePendingConst = "maintenance_pending"
	LoadBalancerListenerPolicyProvisioningStatusUpdatePendingConst      = "update_pending"
)

// UnmarshalLoadBalancerListenerPolicy unmarshals an instance of LoadBalancerListenerPolicy from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicy(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicy)
	err = core.UnmarshalPrimitive(m, "action", &obj.Action)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "priority", &obj.Priority)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisioning_status", &obj.ProvisioningStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalLoadBalancerListenerPolicyRuleReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalLoadBalancerListenerPolicyTarget)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyCollection : LoadBalancerListenerPolicyCollection struct
type LoadBalancerListenerPolicyCollection struct {
	// Collection of policies.
	Policies []LoadBalancerListenerPolicy `json:"policies" validate:"required"`
}

// UnmarshalLoadBalancerListenerPolicyCollection unmarshals an instance of LoadBalancerListenerPolicyCollection from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyCollection)
	err = core.UnmarshalModel(m, "policies", &obj.Policies, UnmarshalLoadBalancerListenerPolicy)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyPatch : LoadBalancerListenerPolicyPatch struct
type LoadBalancerListenerPolicyPatch struct {
	// The user-defined name for this policy. Names must be unique within the load balancer listener the policy resides in.
	Name *string `json:"name,omitempty"`

	// Priority of the policy. Lower value indicates higher priority.
	Priority *int64 `json:"priority,omitempty"`

	// When `action` is `forward`, `LoadBalancerPoolIdentity` specifies which pool the load
	// balancer forwards the traffic to. When `action` is `redirect`,
	// `LoadBalancerListenerPolicyRedirectURLPatch` specifies the url and http
	// status code used in the redirect response.
	Target LoadBalancerListenerPolicyTargetPatchIntf `json:"target,omitempty"`
}

// UnmarshalLoadBalancerListenerPolicyPatch unmarshals an instance of LoadBalancerListenerPolicyPatch from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "priority", &obj.Priority)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalLoadBalancerListenerPolicyTargetPatch)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the LoadBalancerListenerPolicyPatch
func (loadBalancerListenerPolicyPatch *LoadBalancerListenerPolicyPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(loadBalancerListenerPolicyPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// LoadBalancerListenerPolicyPrototype : LoadBalancerListenerPolicyPrototype struct
type LoadBalancerListenerPolicyPrototype struct {
	// The policy action.
	Action *string `json:"action" validate:"required"`

	// The user-defined name for this policy. Names must be unique within the load balancer listener the policy resides in.
	Name *string `json:"name,omitempty"`

	// Priority of the policy. Lower value indicates higher priority.
	Priority *int64 `json:"priority" validate:"required"`

	// An array of rules for this policy.
	Rules []LoadBalancerListenerPolicyRulePrototype `json:"rules,omitempty"`

	// When `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify which
	// pool the load balancer forwards the traffic to. When `action` is `redirect`,
	// `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the url and
	// http status code used in the redirect response.
	Target LoadBalancerListenerPolicyTargetPrototypeIntf `json:"target,omitempty"`
}

// Constants associated with the LoadBalancerListenerPolicyPrototype.Action property.
// The policy action.
const (
	LoadBalancerListenerPolicyPrototypeActionForwardConst  = "forward"
	LoadBalancerListenerPolicyPrototypeActionRedirectConst = "redirect"
	LoadBalancerListenerPolicyPrototypeActionRejectConst   = "reject"
)

// NewLoadBalancerListenerPolicyPrototype : Instantiate LoadBalancerListenerPolicyPrototype (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerListenerPolicyPrototype(action string, priority int64) (model *LoadBalancerListenerPolicyPrototype, err error) {
	model = &LoadBalancerListenerPolicyPrototype{
		Action:   core.StringPtr(action),
		Priority: core.Int64Ptr(priority),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerListenerPolicyPrototype unmarshals an instance of LoadBalancerListenerPolicyPrototype from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyPrototype)
	err = core.UnmarshalPrimitive(m, "action", &obj.Action)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "priority", &obj.Priority)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalLoadBalancerListenerPolicyRulePrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalLoadBalancerListenerPolicyTargetPrototype)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyReference : LoadBalancerListenerPolicyReference struct
type LoadBalancerListenerPolicyReference struct {
	// The listener policy's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The policy's unique identifier.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalLoadBalancerListenerPolicyReference unmarshals an instance of LoadBalancerListenerPolicyReference from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyRule : LoadBalancerListenerPolicyRule struct
type LoadBalancerListenerPolicyRule struct {
	// The condition of the rule.
	Condition *string `json:"condition" validate:"required"`

	// The date and time that this rule was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// HTTP header field. This is only applicable to "header" rule type.
	Field *string `json:"field,omitempty"`

	// The rule's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The rule's unique identifier.
	ID *string `json:"id" validate:"required"`

	// The provisioning status of this rule.
	ProvisioningStatus *string `json:"provisioning_status" validate:"required"`

	// The type of the rule.
	Type *string `json:"type" validate:"required"`

	// Value to be matched for rule condition.
	Value *string `json:"value" validate:"required"`
}

// Constants associated with the LoadBalancerListenerPolicyRule.Condition property.
// The condition of the rule.
const (
	LoadBalancerListenerPolicyRuleConditionContainsConst     = "contains"
	LoadBalancerListenerPolicyRuleConditionEqualsConst       = "equals"
	LoadBalancerListenerPolicyRuleConditionMatchesRegexConst = "matches_regex"
)

// Constants associated with the LoadBalancerListenerPolicyRule.ProvisioningStatus property.
// The provisioning status of this rule.
const (
	LoadBalancerListenerPolicyRuleProvisioningStatusActiveConst             = "active"
	LoadBalancerListenerPolicyRuleProvisioningStatusCreatePendingConst      = "create_pending"
	LoadBalancerListenerPolicyRuleProvisioningStatusDeletePendingConst      = "delete_pending"
	LoadBalancerListenerPolicyRuleProvisioningStatusFailedConst             = "failed"
	LoadBalancerListenerPolicyRuleProvisioningStatusMaintenancePendingConst = "maintenance_pending"
	LoadBalancerListenerPolicyRuleProvisioningStatusUpdatePendingConst      = "update_pending"
)

// Constants associated with the LoadBalancerListenerPolicyRule.Type property.
// The type of the rule.
const (
	LoadBalancerListenerPolicyRuleTypeHeaderConst   = "header"
	LoadBalancerListenerPolicyRuleTypeHostnameConst = "hostname"
	LoadBalancerListenerPolicyRuleTypePathConst     = "path"
)

// UnmarshalLoadBalancerListenerPolicyRule unmarshals an instance of LoadBalancerListenerPolicyRule from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyRule(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyRule)
	err = core.UnmarshalPrimitive(m, "condition", &obj.Condition)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "field", &obj.Field)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisioning_status", &obj.ProvisioningStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyRuleCollection : LoadBalancerListenerPolicyRuleCollection struct
type LoadBalancerListenerPolicyRuleCollection struct {
	// Collection of rules.
	Rules []LoadBalancerListenerPolicyRule `json:"rules" validate:"required"`
}

// UnmarshalLoadBalancerListenerPolicyRuleCollection unmarshals an instance of LoadBalancerListenerPolicyRuleCollection from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyRuleCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyRuleCollection)
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalLoadBalancerListenerPolicyRule)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyRulePatch : LoadBalancerListenerPolicyRulePatch struct
type LoadBalancerListenerPolicyRulePatch struct {
	// The condition of the rule.
	Condition *string `json:"condition,omitempty"`

	// HTTP header field. This is only applicable to "header" rule type.
	Field *string `json:"field,omitempty"`

	// The type of the rule.
	Type *string `json:"type,omitempty"`

	// Value to be matched for rule condition.
	Value *string `json:"value,omitempty"`
}

// Constants associated with the LoadBalancerListenerPolicyRulePatch.Condition property.
// The condition of the rule.
const (
	LoadBalancerListenerPolicyRulePatchConditionContainsConst     = "contains"
	LoadBalancerListenerPolicyRulePatchConditionEqualsConst       = "equals"
	LoadBalancerListenerPolicyRulePatchConditionMatchesRegexConst = "matches_regex"
)

// Constants associated with the LoadBalancerListenerPolicyRulePatch.Type property.
// The type of the rule.
const (
	LoadBalancerListenerPolicyRulePatchTypeHeaderConst   = "header"
	LoadBalancerListenerPolicyRulePatchTypeHostnameConst = "hostname"
	LoadBalancerListenerPolicyRulePatchTypePathConst     = "path"
)

// UnmarshalLoadBalancerListenerPolicyRulePatch unmarshals an instance of LoadBalancerListenerPolicyRulePatch from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyRulePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyRulePatch)
	err = core.UnmarshalPrimitive(m, "condition", &obj.Condition)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "field", &obj.Field)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the LoadBalancerListenerPolicyRulePatch
func (loadBalancerListenerPolicyRulePatch *LoadBalancerListenerPolicyRulePatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(loadBalancerListenerPolicyRulePatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// LoadBalancerListenerPolicyRulePrototype : LoadBalancerListenerPolicyRulePrototype struct
type LoadBalancerListenerPolicyRulePrototype struct {
	// The condition of the rule.
	Condition *string `json:"condition" validate:"required"`

	// HTTP header field. This is only applicable to "header" rule type.
	Field *string `json:"field,omitempty"`

	// The type of the rule.
	Type *string `json:"type" validate:"required"`

	// Value to be matched for rule condition.
	Value *string `json:"value" validate:"required"`
}

// Constants associated with the LoadBalancerListenerPolicyRulePrototype.Condition property.
// The condition of the rule.
const (
	LoadBalancerListenerPolicyRulePrototypeConditionContainsConst     = "contains"
	LoadBalancerListenerPolicyRulePrototypeConditionEqualsConst       = "equals"
	LoadBalancerListenerPolicyRulePrototypeConditionMatchesRegexConst = "matches_regex"
)

// Constants associated with the LoadBalancerListenerPolicyRulePrototype.Type property.
// The type of the rule.
const (
	LoadBalancerListenerPolicyRulePrototypeTypeHeaderConst   = "header"
	LoadBalancerListenerPolicyRulePrototypeTypeHostnameConst = "hostname"
	LoadBalancerListenerPolicyRulePrototypeTypePathConst     = "path"
)

// NewLoadBalancerListenerPolicyRulePrototype : Instantiate LoadBalancerListenerPolicyRulePrototype (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerListenerPolicyRulePrototype(condition string, typeVar string, value string) (model *LoadBalancerListenerPolicyRulePrototype, err error) {
	model = &LoadBalancerListenerPolicyRulePrototype{
		Condition: core.StringPtr(condition),
		Type:      core.StringPtr(typeVar),
		Value:     core.StringPtr(value),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerListenerPolicyRulePrototype unmarshals an instance of LoadBalancerListenerPolicyRulePrototype from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyRulePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyRulePrototype)
	err = core.UnmarshalPrimitive(m, "condition", &obj.Condition)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "field", &obj.Field)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyRuleReference : LoadBalancerListenerPolicyRuleReference struct
type LoadBalancerListenerPolicyRuleReference struct {
	// The rule's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The rule's unique identifier.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalLoadBalancerListenerPolicyRuleReference unmarshals an instance of LoadBalancerListenerPolicyRuleReference from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyRuleReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyRuleReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyTarget : `LoadBalancerPoolReference` is in the response if `action` is `forward`.
// `LoadBalancerListenerPolicyRedirectURL` is in the response if `action` is `redirect`.
// Models which "extend" this model:
// - LoadBalancerListenerPolicyTargetLoadBalancerPoolReference
// - LoadBalancerListenerPolicyTargetLoadBalancerListenerPolicyRedirectURL
type LoadBalancerListenerPolicyTarget struct {
	// The pool's canonical URL.
	Href *string `json:"href,omitempty"`

	// The unique identifier for this load balancer pool.
	ID *string `json:"id,omitempty"`

	// The user-defined name for this load balancer pool.
	Name *string `json:"name,omitempty"`

	// The http status code in the redirect response.
	HTTPStatusCode *int64 `json:"http_status_code,omitempty"`

	// The redirect target URL.
	URL *string `json:"url,omitempty"`
}

func (*LoadBalancerListenerPolicyTarget) isaLoadBalancerListenerPolicyTarget() bool {
	return true
}

type LoadBalancerListenerPolicyTargetIntf interface {
	isaLoadBalancerListenerPolicyTarget() bool
}

// UnmarshalLoadBalancerListenerPolicyTarget unmarshals an instance of LoadBalancerListenerPolicyTarget from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyTarget(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyTarget)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "http_status_code", &obj.HTTPStatusCode)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "url", &obj.URL)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyTargetPatch : When `action` is `forward`, `LoadBalancerPoolIdentity` specifies which pool the load balancer forwards the traffic
// to. When `action` is `redirect`,
// `LoadBalancerListenerPolicyRedirectURLPatch` specifies the url and http status code used in the redirect response.
// Models which "extend" this model:
// - LoadBalancerListenerPolicyTargetPatchLoadBalancerPoolIdentity
// - LoadBalancerListenerPolicyTargetPatchLoadBalancerListenerPolicyRedirectURLPatch
type LoadBalancerListenerPolicyTargetPatch struct {
	// The unique identifier for this load balancer pool.
	ID *string `json:"id,omitempty"`

	// The pool's canonical URL.
	Href *string `json:"href,omitempty"`

	// The http status code in the redirect response.
	HTTPStatusCode *int64 `json:"http_status_code,omitempty"`

	// The redirect target URL.
	URL *string `json:"url,omitempty"`
}

func (*LoadBalancerListenerPolicyTargetPatch) isaLoadBalancerListenerPolicyTargetPatch() bool {
	return true
}

type LoadBalancerListenerPolicyTargetPatchIntf interface {
	isaLoadBalancerListenerPolicyTargetPatch() bool
}

// UnmarshalLoadBalancerListenerPolicyTargetPatch unmarshals an instance of LoadBalancerListenerPolicyTargetPatch from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyTargetPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyTargetPatch)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "http_status_code", &obj.HTTPStatusCode)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "url", &obj.URL)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPolicyTargetPrototype : When `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify which pool the load balancer forwards
// the traffic to. When `action` is `redirect`,
// `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the url and http status code used in the
// redirect response.
// Models which "extend" this model:
// - LoadBalancerListenerPolicyTargetPrototypeLoadBalancerPoolIdentity
// - LoadBalancerListenerPolicyTargetPrototypeLoadBalancerListenerPolicyRedirectURLPrototype
type LoadBalancerListenerPolicyTargetPrototype struct {
	// The unique identifier for this load balancer pool.
	ID *string `json:"id,omitempty"`

	// The pool's canonical URL.
	Href *string `json:"href,omitempty"`

	// The http status code in the redirect response.
	HTTPStatusCode *int64 `json:"http_status_code,omitempty"`

	// The redirect target URL.
	URL *string `json:"url,omitempty"`
}

func (*LoadBalancerListenerPolicyTargetPrototype) isaLoadBalancerListenerPolicyTargetPrototype() bool {
	return true
}

type LoadBalancerListenerPolicyTargetPrototypeIntf interface {
	isaLoadBalancerListenerPolicyTargetPrototype() bool
}

// UnmarshalLoadBalancerListenerPolicyTargetPrototype unmarshals an instance of LoadBalancerListenerPolicyTargetPrototype from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPolicyTargetPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPolicyTargetPrototype)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "http_status_code", &obj.HTTPStatusCode)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "url", &obj.URL)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerPrototypeLoadBalancerContext : LoadBalancerListenerPrototypeLoadBalancerContext struct
type LoadBalancerListenerPrototypeLoadBalancerContext struct {
	// If set to `true`, this listener will accept and forward PROXY protocol information. Supported by load balancers in
	// the `application` family (otherwise always `false`).
	AcceptProxyProtocol *bool `json:"accept_proxy_protocol,omitempty"`

	// The connection limit of the listener.
	ConnectionLimit *int64 `json:"connection_limit,omitempty"`

	// The default pool associated with the listener.
	DefaultPool *LoadBalancerPoolIdentityByName `json:"default_pool,omitempty"`

	// The listener port number. Each listener in the load balancer must have a unique
	// `port` and `protocol` combination.
	Port *int64 `json:"port" validate:"required"`

	// The listener protocol.
	Protocol *string `json:"protocol" validate:"required"`
}

// Constants associated with the LoadBalancerListenerPrototypeLoadBalancerContext.Protocol property.
// The listener protocol.
const (
	LoadBalancerListenerPrototypeLoadBalancerContextProtocolHTTPConst  = "http"
	LoadBalancerListenerPrototypeLoadBalancerContextProtocolHTTPSConst = "https"
	LoadBalancerListenerPrototypeLoadBalancerContextProtocolTCPConst   = "tcp"
)

// NewLoadBalancerListenerPrototypeLoadBalancerContext : Instantiate LoadBalancerListenerPrototypeLoadBalancerContext (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerListenerPrototypeLoadBalancerContext(port int64, protocol string) (model *LoadBalancerListenerPrototypeLoadBalancerContext, err error) {
	model = &LoadBalancerListenerPrototypeLoadBalancerContext{
		Port:     core.Int64Ptr(port),
		Protocol: core.StringPtr(protocol),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerListenerPrototypeLoadBalancerContext unmarshals an instance of LoadBalancerListenerPrototypeLoadBalancerContext from the specified map of raw messages.
func UnmarshalLoadBalancerListenerPrototypeLoadBalancerContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerPrototypeLoadBalancerContext)
	err = core.UnmarshalPrimitive(m, "accept_proxy_protocol", &obj.AcceptProxyProtocol)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "connection_limit", &obj.ConnectionLimit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "default_pool", &obj.DefaultPool, UnmarshalLoadBalancerPoolIdentityByName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "protocol", &obj.Protocol)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerListenerReference : LoadBalancerListenerReference struct
type LoadBalancerListenerReference struct {
	// The listener's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer listener.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalLoadBalancerListenerReference unmarshals an instance of LoadBalancerListenerReference from the specified map of raw messages.
func UnmarshalLoadBalancerListenerReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerListenerReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerLogging : The logging configuration for this load balancer.
type LoadBalancerLogging struct {
	// The datapath logging configuration for this load balancer.
	Datapath *LoadBalancerLoggingDatapath `json:"datapath,omitempty"`
}

// UnmarshalLoadBalancerLogging unmarshals an instance of LoadBalancerLogging from the specified map of raw messages.
func UnmarshalLoadBalancerLogging(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerLogging)
	err = core.UnmarshalModel(m, "datapath", &obj.Datapath, UnmarshalLoadBalancerLoggingDatapath)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerLoggingDatapath : The datapath logging configuration for this load balancer.
type LoadBalancerLoggingDatapath struct {
	// If set to `true`, datapath logging is active for this load balancer.
	Active *bool `json:"active" validate:"required"`
}

// NewLoadBalancerLoggingDatapath : Instantiate LoadBalancerLoggingDatapath (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerLoggingDatapath(active bool) (model *LoadBalancerLoggingDatapath, err error) {
	model = &LoadBalancerLoggingDatapath{
		Active: core.BoolPtr(active),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerLoggingDatapath unmarshals an instance of LoadBalancerLoggingDatapath from the specified map of raw messages.
func UnmarshalLoadBalancerLoggingDatapath(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerLoggingDatapath)
	err = core.UnmarshalPrimitive(m, "active", &obj.Active)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPatch : LoadBalancerPatch struct
type LoadBalancerPatch struct {
	// The logging configuration to use for this load balancer.
	//
	// To activate logging, the load balancer profile must support the specified logging type.
	Logging *LoadBalancerLogging `json:"logging,omitempty"`

	// The unique user-defined name for this load balancer.
	Name *string `json:"name,omitempty"`
}

// UnmarshalLoadBalancerPatch unmarshals an instance of LoadBalancerPatch from the specified map of raw messages.
func UnmarshalLoadBalancerPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPatch)
	err = core.UnmarshalModel(m, "logging", &obj.Logging, UnmarshalLoadBalancerLogging)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the LoadBalancerPatch
func (loadBalancerPatch *LoadBalancerPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(loadBalancerPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// LoadBalancerPool : LoadBalancerPool struct
type LoadBalancerPool struct {
	// The load balancing algorithm.
	Algorithm *string `json:"algorithm" validate:"required"`

	// The date and time that this pool was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The health monitor of this pool.
	HealthMonitor *LoadBalancerPoolHealthMonitor `json:"health_monitor" validate:"required"`

	// The pool's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer pool.
	ID *string `json:"id" validate:"required"`

	// The backend server members of the pool.
	Members []LoadBalancerPoolMemberReference `json:"members,omitempty"`

	// The user-defined name for this load balancer pool.
	Name *string `json:"name" validate:"required"`

	// The protocol used for this load balancer pool.
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the pool on which the
	// unexpected property value was encountered.
	Protocol *string `json:"protocol" validate:"required"`

	// The provisioning status of this pool.
	ProvisioningStatus *string `json:"provisioning_status" validate:"required"`

	// The PROXY protocol setting for this pool:
	// - `v1`: Enabled with version 1 (human-readable header format)
	// - `v2`: Enabled with version 2 (binary header format)
	// - `disabled`: Disabled
	//
	// Supported by load balancers in the `application` family (otherwise always `disabled`).
	ProxyProtocol *string `json:"proxy_protocol" validate:"required"`

	// The session persistence of this pool.
	SessionPersistence *LoadBalancerPoolSessionPersistence `json:"session_persistence,omitempty"`
}

// Constants associated with the LoadBalancerPool.Algorithm property.
// The load balancing algorithm.
const (
	LoadBalancerPoolAlgorithmLeastConnectionsConst   = "least_connections"
	LoadBalancerPoolAlgorithmRoundRobinConst         = "round_robin"
	LoadBalancerPoolAlgorithmWeightedRoundRobinConst = "weighted_round_robin"
)

// Constants associated with the LoadBalancerPool.Protocol property.
// The protocol used for this load balancer pool.
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the pool on which the
// unexpected property value was encountered.
const (
	LoadBalancerPoolProtocolHTTPConst  = "http"
	LoadBalancerPoolProtocolHTTPSConst = "https"
	LoadBalancerPoolProtocolTCPConst   = "tcp"
)

// Constants associated with the LoadBalancerPool.ProvisioningStatus property.
// The provisioning status of this pool.
const (
	LoadBalancerPoolProvisioningStatusActiveConst             = "active"
	LoadBalancerPoolProvisioningStatusCreatePendingConst      = "create_pending"
	LoadBalancerPoolProvisioningStatusDeletePendingConst      = "delete_pending"
	LoadBalancerPoolProvisioningStatusFailedConst             = "failed"
	LoadBalancerPoolProvisioningStatusMaintenancePendingConst = "maintenance_pending"
	LoadBalancerPoolProvisioningStatusUpdatePendingConst      = "update_pending"
)

// Constants associated with the LoadBalancerPool.ProxyProtocol property.
// The PROXY protocol setting for this pool:
// - `v1`: Enabled with version 1 (human-readable header format)
// - `v2`: Enabled with version 2 (binary header format)
// - `disabled`: Disabled
//
// Supported by load balancers in the `application` family (otherwise always `disabled`).
const (
	LoadBalancerPoolProxyProtocolDisabledConst = "disabled"
	LoadBalancerPoolProxyProtocolV1Const       = "v1"
	LoadBalancerPoolProxyProtocolV2Const       = "v2"
)

// UnmarshalLoadBalancerPool unmarshals an instance of LoadBalancerPool from the specified map of raw messages.
func UnmarshalLoadBalancerPool(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPool)
	err = core.UnmarshalPrimitive(m, "algorithm", &obj.Algorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "health_monitor", &obj.HealthMonitor, UnmarshalLoadBalancerPoolHealthMonitor)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "members", &obj.Members, UnmarshalLoadBalancerPoolMemberReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "protocol", &obj.Protocol)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisioning_status", &obj.ProvisioningStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "proxy_protocol", &obj.ProxyProtocol)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "session_persistence", &obj.SessionPersistence, UnmarshalLoadBalancerPoolSessionPersistence)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolCollection : LoadBalancerPoolCollection struct
type LoadBalancerPoolCollection struct {
	// Collection of pools.
	Pools []LoadBalancerPool `json:"pools" validate:"required"`
}

// UnmarshalLoadBalancerPoolCollection unmarshals an instance of LoadBalancerPoolCollection from the specified map of raw messages.
func UnmarshalLoadBalancerPoolCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolCollection)
	err = core.UnmarshalModel(m, "pools", &obj.Pools, UnmarshalLoadBalancerPool)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolHealthMonitor : LoadBalancerPoolHealthMonitor struct
type LoadBalancerPoolHealthMonitor struct {
	// The health check interval in seconds. Interval must be greater than timeout value.
	Delay *int64 `json:"delay" validate:"required"`

	// The health check max retries.
	MaxRetries *int64 `json:"max_retries" validate:"required"`

	// The health check port number. If specified, this overrides the ports specified in the server member resources.
	Port *int64 `json:"port,omitempty"`

	// The health check timeout in seconds.
	Timeout *int64 `json:"timeout" validate:"required"`

	// The protocol type of this load balancer pool health monitor.
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the health monitor on which
	// the unexpected property value was encountered.
	Type *string `json:"type" validate:"required"`

	// The health check URL path. Applicable only if the health monitor `type` is `http` or
	// `https`. This value must be in the format of an [origin-form request
	// target](https://tools.ietf.org/html/rfc7230#section-5.3.1).
	URLPath *string `json:"url_path,omitempty"`
}

// Constants associated with the LoadBalancerPoolHealthMonitor.Type property.
// The protocol type of this load balancer pool health monitor.
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the health monitor on which
// the unexpected property value was encountered.
const (
	LoadBalancerPoolHealthMonitorTypeHTTPConst  = "http"
	LoadBalancerPoolHealthMonitorTypeHTTPSConst = "https"
	LoadBalancerPoolHealthMonitorTypeTCPConst   = "tcp"
)

// UnmarshalLoadBalancerPoolHealthMonitor unmarshals an instance of LoadBalancerPoolHealthMonitor from the specified map of raw messages.
func UnmarshalLoadBalancerPoolHealthMonitor(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolHealthMonitor)
	err = core.UnmarshalPrimitive(m, "delay", &obj.Delay)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max_retries", &obj.MaxRetries)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "timeout", &obj.Timeout)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "url_path", &obj.URLPath)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolHealthMonitorPatch : LoadBalancerPoolHealthMonitorPatch struct
type LoadBalancerPoolHealthMonitorPatch struct {
	// The health check interval in seconds. Interval must be greater than timeout value.
	Delay *int64 `json:"delay" validate:"required"`

	// The health check max retries.
	MaxRetries *int64 `json:"max_retries" validate:"required"`

	// The health check port number. If specified, this overrides the ports specified in the server member resources.
	// Specify `null` to remove an existing port value.
	Port *int64 `json:"port,omitempty"`

	// The health check timeout in seconds.
	Timeout *int64 `json:"timeout" validate:"required"`

	// The protocol type of this load balancer pool health monitor.
	Type *string `json:"type" validate:"required"`

	// The health check URL path. Applicable only if the health monitor `type` is `http` or
	// `https`. This value must be in the format of an [origin-form request
	// target](https://tools.ietf.org/html/rfc7230#section-5.3.1).
	URLPath *string `json:"url_path,omitempty"`
}

// Constants associated with the LoadBalancerPoolHealthMonitorPatch.Type property.
// The protocol type of this load balancer pool health monitor.
const (
	LoadBalancerPoolHealthMonitorPatchTypeHTTPConst  = "http"
	LoadBalancerPoolHealthMonitorPatchTypeHTTPSConst = "https"
	LoadBalancerPoolHealthMonitorPatchTypeTCPConst   = "tcp"
)

// NewLoadBalancerPoolHealthMonitorPatch : Instantiate LoadBalancerPoolHealthMonitorPatch (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolHealthMonitorPatch(delay int64, maxRetries int64, timeout int64, typeVar string) (model *LoadBalancerPoolHealthMonitorPatch, err error) {
	model = &LoadBalancerPoolHealthMonitorPatch{
		Delay:      core.Int64Ptr(delay),
		MaxRetries: core.Int64Ptr(maxRetries),
		Timeout:    core.Int64Ptr(timeout),
		Type:       core.StringPtr(typeVar),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolHealthMonitorPatch unmarshals an instance of LoadBalancerPoolHealthMonitorPatch from the specified map of raw messages.
func UnmarshalLoadBalancerPoolHealthMonitorPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolHealthMonitorPatch)
	err = core.UnmarshalPrimitive(m, "delay", &obj.Delay)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max_retries", &obj.MaxRetries)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "timeout", &obj.Timeout)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "url_path", &obj.URLPath)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolHealthMonitorPrototype : LoadBalancerPoolHealthMonitorPrototype struct
type LoadBalancerPoolHealthMonitorPrototype struct {
	// The health check interval in seconds. Interval must be greater than timeout value.
	Delay *int64 `json:"delay" validate:"required"`

	// The health check max retries.
	MaxRetries *int64 `json:"max_retries" validate:"required"`

	// The health check port number. If specified, this overrides the ports specified in the server member resources.
	Port *int64 `json:"port,omitempty"`

	// The health check timeout in seconds.
	Timeout *int64 `json:"timeout" validate:"required"`

	// The protocol type of this load balancer pool health monitor.
	Type *string `json:"type" validate:"required"`

	// The health check URL path. Applicable only if the health monitor `type` is `http` or
	// `https`. This value must be in the format of an [origin-form request
	// target](https://tools.ietf.org/html/rfc7230#section-5.3.1).
	URLPath *string `json:"url_path,omitempty"`
}

// Constants associated with the LoadBalancerPoolHealthMonitorPrototype.Type property.
// The protocol type of this load balancer pool health monitor.
const (
	LoadBalancerPoolHealthMonitorPrototypeTypeHTTPConst  = "http"
	LoadBalancerPoolHealthMonitorPrototypeTypeHTTPSConst = "https"
	LoadBalancerPoolHealthMonitorPrototypeTypeTCPConst   = "tcp"
)

// NewLoadBalancerPoolHealthMonitorPrototype : Instantiate LoadBalancerPoolHealthMonitorPrototype (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolHealthMonitorPrototype(delay int64, maxRetries int64, timeout int64, typeVar string) (model *LoadBalancerPoolHealthMonitorPrototype, err error) {
	model = &LoadBalancerPoolHealthMonitorPrototype{
		Delay:      core.Int64Ptr(delay),
		MaxRetries: core.Int64Ptr(maxRetries),
		Timeout:    core.Int64Ptr(timeout),
		Type:       core.StringPtr(typeVar),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolHealthMonitorPrototype unmarshals an instance of LoadBalancerPoolHealthMonitorPrototype from the specified map of raw messages.
func UnmarshalLoadBalancerPoolHealthMonitorPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolHealthMonitorPrototype)
	err = core.UnmarshalPrimitive(m, "delay", &obj.Delay)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max_retries", &obj.MaxRetries)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "timeout", &obj.Timeout)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "url_path", &obj.URLPath)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolIdentity : Identifies a load balancer pool by a unique property.
// Models which "extend" this model:
// - LoadBalancerPoolIdentityByID
// - LoadBalancerPoolIdentityByHref
type LoadBalancerPoolIdentity struct {
	// The unique identifier for this load balancer pool.
	ID *string `json:"id,omitempty"`

	// The pool's canonical URL.
	Href *string `json:"href,omitempty"`
}

func (*LoadBalancerPoolIdentity) isaLoadBalancerPoolIdentity() bool {
	return true
}

type LoadBalancerPoolIdentityIntf interface {
	isaLoadBalancerPoolIdentity() bool
}

// UnmarshalLoadBalancerPoolIdentity unmarshals an instance of LoadBalancerPoolIdentity from the specified map of raw messages.
func UnmarshalLoadBalancerPoolIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolIdentityByName : LoadBalancerPoolIdentityByName struct
type LoadBalancerPoolIdentityByName struct {
	// The user-defined name for this load balancer pool.
	Name *string `json:"name" validate:"required"`
}

// NewLoadBalancerPoolIdentityByName : Instantiate LoadBalancerPoolIdentityByName (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolIdentityByName(name string) (model *LoadBalancerPoolIdentityByName, err error) {
	model = &LoadBalancerPoolIdentityByName{
		Name: core.StringPtr(name),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolIdentityByName unmarshals an instance of LoadBalancerPoolIdentityByName from the specified map of raw messages.
func UnmarshalLoadBalancerPoolIdentityByName(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolIdentityByName)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolMember : LoadBalancerPoolMember struct
type LoadBalancerPoolMember struct {
	// The date and time that this member was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// Health of the server member in the pool.
	Health *string `json:"health" validate:"required"`

	// The member's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer pool member.
	ID *string `json:"id" validate:"required"`

	// The port number of the application running in the server member.
	Port *int64 `json:"port" validate:"required"`

	// The provisioning status of this member.
	ProvisioningStatus *string `json:"provisioning_status" validate:"required"`

	// The pool member target.
	Target LoadBalancerPoolMemberTargetIntf `json:"target" validate:"required"`

	// Weight of the server member. Applicable only if the pool algorithm is
	// `weighted_round_robin`.
	Weight *int64 `json:"weight,omitempty"`
}

// Constants associated with the LoadBalancerPoolMember.Health property.
// Health of the server member in the pool.
const (
	LoadBalancerPoolMemberHealthFaultedConst = "faulted"
	LoadBalancerPoolMemberHealthOkConst      = "ok"
	LoadBalancerPoolMemberHealthUnknownConst = "unknown"
)

// Constants associated with the LoadBalancerPoolMember.ProvisioningStatus property.
// The provisioning status of this member.
const (
	LoadBalancerPoolMemberProvisioningStatusActiveConst             = "active"
	LoadBalancerPoolMemberProvisioningStatusCreatePendingConst      = "create_pending"
	LoadBalancerPoolMemberProvisioningStatusDeletePendingConst      = "delete_pending"
	LoadBalancerPoolMemberProvisioningStatusFailedConst             = "failed"
	LoadBalancerPoolMemberProvisioningStatusMaintenancePendingConst = "maintenance_pending"
	LoadBalancerPoolMemberProvisioningStatusUpdatePendingConst      = "update_pending"
)

// UnmarshalLoadBalancerPoolMember unmarshals an instance of LoadBalancerPoolMember from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMember(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMember)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "health", &obj.Health)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisioning_status", &obj.ProvisioningStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalLoadBalancerPoolMemberTarget)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "weight", &obj.Weight)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolMemberCollection : LoadBalancerPoolMemberCollection struct
type LoadBalancerPoolMemberCollection struct {
	// Collection of members.
	Members []LoadBalancerPoolMember `json:"members" validate:"required"`
}

// UnmarshalLoadBalancerPoolMemberCollection unmarshals an instance of LoadBalancerPoolMemberCollection from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMemberCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMemberCollection)
	err = core.UnmarshalModel(m, "members", &obj.Members, UnmarshalLoadBalancerPoolMember)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolMemberPatch : LoadBalancerPoolMemberPatch struct
type LoadBalancerPoolMemberPatch struct {
	// The port number of the application running in the server member.
	Port *int64 `json:"port,omitempty"`

	// The pool member target.
	Target LoadBalancerPoolMemberTargetPrototypeIntf `json:"target,omitempty"`

	// Weight of the server member. Applicable only if the pool algorithm is
	// `weighted_round_robin`.
	Weight *int64 `json:"weight,omitempty"`
}

// UnmarshalLoadBalancerPoolMemberPatch unmarshals an instance of LoadBalancerPoolMemberPatch from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMemberPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMemberPatch)
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalLoadBalancerPoolMemberTargetPrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "weight", &obj.Weight)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the LoadBalancerPoolMemberPatch
func (loadBalancerPoolMemberPatch *LoadBalancerPoolMemberPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(loadBalancerPoolMemberPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// LoadBalancerPoolMemberPrototype : LoadBalancerPoolMemberPrototype struct
type LoadBalancerPoolMemberPrototype struct {
	// The port number of the application running in the server member.
	Port *int64 `json:"port" validate:"required"`

	// The pool member target.
	Target LoadBalancerPoolMemberTargetPrototypeIntf `json:"target" validate:"required"`

	// Weight of the server member. Applicable only if the pool algorithm is
	// `weighted_round_robin`.
	Weight *int64 `json:"weight,omitempty"`
}

// NewLoadBalancerPoolMemberPrototype : Instantiate LoadBalancerPoolMemberPrototype (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolMemberPrototype(port int64, target LoadBalancerPoolMemberTargetPrototypeIntf) (model *LoadBalancerPoolMemberPrototype, err error) {
	model = &LoadBalancerPoolMemberPrototype{
		Port:   core.Int64Ptr(port),
		Target: target,
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolMemberPrototype unmarshals an instance of LoadBalancerPoolMemberPrototype from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMemberPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMemberPrototype)
	err = core.UnmarshalPrimitive(m, "port", &obj.Port)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalLoadBalancerPoolMemberTargetPrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "weight", &obj.Weight)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolMemberReference : LoadBalancerPoolMemberReference struct
type LoadBalancerPoolMemberReference struct {
	// The member's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer pool member.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalLoadBalancerPoolMemberReference unmarshals an instance of LoadBalancerPoolMemberReference from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMemberReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMemberReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolMemberTarget : The pool member target.
// Models which "extend" this model:
// - LoadBalancerPoolMemberTargetIP
type LoadBalancerPoolMemberTarget struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`
}

func (*LoadBalancerPoolMemberTarget) isaLoadBalancerPoolMemberTarget() bool {
	return true
}

type LoadBalancerPoolMemberTargetIntf interface {
	isaLoadBalancerPoolMemberTarget() bool
}

// UnmarshalLoadBalancerPoolMemberTarget unmarshals an instance of LoadBalancerPoolMemberTarget from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMemberTarget(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMemberTarget)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolMemberTargetPrototype : The pool member target.
// Models which "extend" this model:
// - LoadBalancerPoolMemberTargetPrototypeIP
type LoadBalancerPoolMemberTargetPrototype struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`
}

func (*LoadBalancerPoolMemberTargetPrototype) isaLoadBalancerPoolMemberTargetPrototype() bool {
	return true
}

type LoadBalancerPoolMemberTargetPrototypeIntf interface {
	isaLoadBalancerPoolMemberTargetPrototype() bool
}

// UnmarshalLoadBalancerPoolMemberTargetPrototype unmarshals an instance of LoadBalancerPoolMemberTargetPrototype from the specified map of raw messages.
func UnmarshalLoadBalancerPoolMemberTargetPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolMemberTargetPrototype)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolPatch : LoadBalancerPoolPatch struct
type LoadBalancerPoolPatch struct {
	// The load balancing algorithm.
	Algorithm *string `json:"algorithm,omitempty"`

	// The health monitor of this pool.
	HealthMonitor *LoadBalancerPoolHealthMonitorPatch `json:"health_monitor,omitempty"`

	// The user-defined name for this load balancer pool.
	Name *string `json:"name,omitempty"`

	// The protocol used for this load balancer pool.
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the pool on which the
	// unexpected property value was encountered.
	Protocol *string `json:"protocol,omitempty"`

	// The PROXY protocol setting for this pool:
	// - `v1`: Enabled with version 1 (human-readable header format)
	// - `v2`: Enabled with version 2 (binary header format)
	// - `disabled`: Disabled
	//
	// Supported by load balancers in the `application` family (otherwise always `disabled`).
	ProxyProtocol *string `json:"proxy_protocol,omitempty"`

	// The session persistence of this pool.
	SessionPersistence *LoadBalancerPoolSessionPersistencePatch `json:"session_persistence,omitempty"`
}

// Constants associated with the LoadBalancerPoolPatch.Algorithm property.
// The load balancing algorithm.
const (
	LoadBalancerPoolPatchAlgorithmLeastConnectionsConst   = "least_connections"
	LoadBalancerPoolPatchAlgorithmRoundRobinConst         = "round_robin"
	LoadBalancerPoolPatchAlgorithmWeightedRoundRobinConst = "weighted_round_robin"
)

// Constants associated with the LoadBalancerPoolPatch.Protocol property.
// The protocol used for this load balancer pool.
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the pool on which the
// unexpected property value was encountered.
const (
	LoadBalancerPoolPatchProtocolHTTPConst  = "http"
	LoadBalancerPoolPatchProtocolHTTPSConst = "https"
	LoadBalancerPoolPatchProtocolTCPConst   = "tcp"
)

// Constants associated with the LoadBalancerPoolPatch.ProxyProtocol property.
// The PROXY protocol setting for this pool:
// - `v1`: Enabled with version 1 (human-readable header format)
// - `v2`: Enabled with version 2 (binary header format)
// - `disabled`: Disabled
//
// Supported by load balancers in the `application` family (otherwise always `disabled`).
const (
	LoadBalancerPoolPatchProxyProtocolDisabledConst = "disabled"
	LoadBalancerPoolPatchProxyProtocolV1Const       = "v1"
	LoadBalancerPoolPatchProxyProtocolV2Const       = "v2"
)

// UnmarshalLoadBalancerPoolPatch unmarshals an instance of LoadBalancerPoolPatch from the specified map of raw messages.
func UnmarshalLoadBalancerPoolPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolPatch)
	err = core.UnmarshalPrimitive(m, "algorithm", &obj.Algorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "health_monitor", &obj.HealthMonitor, UnmarshalLoadBalancerPoolHealthMonitorPatch)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "protocol", &obj.Protocol)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "proxy_protocol", &obj.ProxyProtocol)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "session_persistence", &obj.SessionPersistence, UnmarshalLoadBalancerPoolSessionPersistencePatch)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the LoadBalancerPoolPatch
func (loadBalancerPoolPatch *LoadBalancerPoolPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(loadBalancerPoolPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// LoadBalancerPoolPrototype : LoadBalancerPoolPrototype struct
type LoadBalancerPoolPrototype struct {
	// The load balancing algorithm.
	Algorithm *string `json:"algorithm" validate:"required"`

	// The health monitor of this pool.
	HealthMonitor *LoadBalancerPoolHealthMonitorPrototype `json:"health_monitor" validate:"required"`

	// The members for this load balancer pool. For load balancers in the `network` family, the same `port` and `target`
	// tuple cannot be shared by a pool member of any other load balancer in the same VPC.
	Members []LoadBalancerPoolMemberPrototype `json:"members,omitempty"`

	// The user-defined name for this load balancer pool. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The protocol used for this load balancer pool. Load balancers in the `network` family support `tcp`. Load balancers
	// in the `application` family support `tcp`, `http`, and
	// `https`.
	Protocol *string `json:"protocol" validate:"required"`

	// The PROXY protocol setting for this pool:
	// - `v1`: Enabled with version 1 (human-readable header format)
	// - `v2`: Enabled with version 2 (binary header format)
	// - `disabled`: Disabled
	//
	// Supported by load balancers in the `application` family (otherwise always `disabled`).
	ProxyProtocol *string `json:"proxy_protocol,omitempty"`

	// The session persistence of this pool.
	SessionPersistence *LoadBalancerPoolSessionPersistencePrototype `json:"session_persistence,omitempty"`
}

// Constants associated with the LoadBalancerPoolPrototype.Algorithm property.
// The load balancing algorithm.
const (
	LoadBalancerPoolPrototypeAlgorithmLeastConnectionsConst   = "least_connections"
	LoadBalancerPoolPrototypeAlgorithmRoundRobinConst         = "round_robin"
	LoadBalancerPoolPrototypeAlgorithmWeightedRoundRobinConst = "weighted_round_robin"
)

// Constants associated with the LoadBalancerPoolPrototype.Protocol property.
// The protocol used for this load balancer pool. Load balancers in the `network` family support `tcp`. Load balancers
// in the `application` family support `tcp`, `http`, and
// `https`.
const (
	LoadBalancerPoolPrototypeProtocolHTTPConst  = "http"
	LoadBalancerPoolPrototypeProtocolHTTPSConst = "https"
	LoadBalancerPoolPrototypeProtocolTCPConst   = "tcp"
)

// Constants associated with the LoadBalancerPoolPrototype.ProxyProtocol property.
// The PROXY protocol setting for this pool:
// - `v1`: Enabled with version 1 (human-readable header format)
// - `v2`: Enabled with version 2 (binary header format)
// - `disabled`: Disabled
//
// Supported by load balancers in the `application` family (otherwise always `disabled`).
const (
	LoadBalancerPoolPrototypeProxyProtocolDisabledConst = "disabled"
	LoadBalancerPoolPrototypeProxyProtocolV1Const       = "v1"
	LoadBalancerPoolPrototypeProxyProtocolV2Const       = "v2"
)

// NewLoadBalancerPoolPrototype : Instantiate LoadBalancerPoolPrototype (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolPrototype(algorithm string, healthMonitor *LoadBalancerPoolHealthMonitorPrototype, protocol string) (model *LoadBalancerPoolPrototype, err error) {
	model = &LoadBalancerPoolPrototype{
		Algorithm:     core.StringPtr(algorithm),
		HealthMonitor: healthMonitor,
		Protocol:      core.StringPtr(protocol),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolPrototype unmarshals an instance of LoadBalancerPoolPrototype from the specified map of raw messages.
func UnmarshalLoadBalancerPoolPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolPrototype)
	err = core.UnmarshalPrimitive(m, "algorithm", &obj.Algorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "health_monitor", &obj.HealthMonitor, UnmarshalLoadBalancerPoolHealthMonitorPrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "members", &obj.Members, UnmarshalLoadBalancerPoolMemberPrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "protocol", &obj.Protocol)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "proxy_protocol", &obj.ProxyProtocol)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "session_persistence", &obj.SessionPersistence, UnmarshalLoadBalancerPoolSessionPersistencePrototype)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolReference : LoadBalancerPoolReference struct
type LoadBalancerPoolReference struct {
	// The pool's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this load balancer pool.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this load balancer pool.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalLoadBalancerPoolReference unmarshals an instance of LoadBalancerPoolReference from the specified map of raw messages.
func UnmarshalLoadBalancerPoolReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolSessionPersistence : LoadBalancerPoolSessionPersistence struct
type LoadBalancerPoolSessionPersistence struct {
	// The session persistence type.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the LoadBalancerPoolSessionPersistence.Type property.
// The session persistence type.
const (
	LoadBalancerPoolSessionPersistenceTypeSourceIPConst = "source_ip"
)

// UnmarshalLoadBalancerPoolSessionPersistence unmarshals an instance of LoadBalancerPoolSessionPersistence from the specified map of raw messages.
func UnmarshalLoadBalancerPoolSessionPersistence(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolSessionPersistence)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolSessionPersistencePatch : LoadBalancerPoolSessionPersistencePatch struct
type LoadBalancerPoolSessionPersistencePatch struct {
	// The session persistence type.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the LoadBalancerPoolSessionPersistencePatch.Type property.
// The session persistence type.
const (
	LoadBalancerPoolSessionPersistencePatchTypeSourceIPConst = "source_ip"
)

// NewLoadBalancerPoolSessionPersistencePatch : Instantiate LoadBalancerPoolSessionPersistencePatch (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolSessionPersistencePatch(typeVar string) (model *LoadBalancerPoolSessionPersistencePatch, err error) {
	model = &LoadBalancerPoolSessionPersistencePatch{
		Type: core.StringPtr(typeVar),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolSessionPersistencePatch unmarshals an instance of LoadBalancerPoolSessionPersistencePatch from the specified map of raw messages.
func UnmarshalLoadBalancerPoolSessionPersistencePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolSessionPersistencePatch)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerPoolSessionPersistencePrototype : LoadBalancerPoolSessionPersistencePrototype struct
type LoadBalancerPoolSessionPersistencePrototype struct {
	// The session persistence type.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the LoadBalancerPoolSessionPersistencePrototype.Type property.
// The session persistence type.
const (
	LoadBalancerPoolSessionPersistencePrototypeTypeSourceIPConst = "source_ip"
)

// NewLoadBalancerPoolSessionPersistencePrototype : Instantiate LoadBalancerPoolSessionPersistencePrototype (Generic Model Constructor)
func (*VpcClassicV1) NewLoadBalancerPoolSessionPersistencePrototype(typeVar string) (model *LoadBalancerPoolSessionPersistencePrototype, err error) {
	model = &LoadBalancerPoolSessionPersistencePrototype{
		Type: core.StringPtr(typeVar),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalLoadBalancerPoolSessionPersistencePrototype unmarshals an instance of LoadBalancerPoolSessionPersistencePrototype from the specified map of raw messages.
func UnmarshalLoadBalancerPoolSessionPersistencePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerPoolSessionPersistencePrototype)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LoadBalancerStatistics : LoadBalancerStatistics struct
type LoadBalancerStatistics struct {
	// Number of active connections of this load balancer.
	ActiveConnections *int64 `json:"active_connections" validate:"required"`

	// Current connection rate (connections per second) of this load balancer.
	ConnectionRate *float32 `json:"connection_rate" validate:"required"`

	// Total number of data processed (bytes) of this load balancer within current calendar month.
	DataProcessedThisMonth *int64 `json:"data_processed_this_month" validate:"required"`

	// Current throughput (Mbps) of this load balancer.
	Throughput *float32 `json:"throughput" validate:"required"`
}

// UnmarshalLoadBalancerStatistics unmarshals an instance of LoadBalancerStatistics from the specified map of raw messages.
func UnmarshalLoadBalancerStatistics(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LoadBalancerStatistics)
	err = core.UnmarshalPrimitive(m, "active_connections", &obj.ActiveConnections)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "connection_rate", &obj.ConnectionRate)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "data_processed_this_month", &obj.DataProcessedThisMonth)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "throughput", &obj.Throughput)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACL : NetworkACL struct
type NetworkACL struct {
	// The date and time that the network ACL was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The URL for this network ACL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network ACL.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this network ACL.
	Name *string `json:"name" validate:"required"`

	// The ordered rules for this network ACL. If no rules exist, all traffic will be allowed.
	Rules []NetworkACLRuleItemIntf `json:"rules" validate:"required"`

	// The subnets to which this network ACL is attached.
	Subnets []SubnetReference `json:"subnets" validate:"required"`
}

// UnmarshalNetworkACL unmarshals an instance of NetworkACL from the specified map of raw messages.
func UnmarshalNetworkACL(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACL)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalNetworkACLRuleItem)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnets", &obj.Subnets, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLCollection : NetworkACLCollection struct
type NetworkACLCollection struct {
	// A link to the first page of resources.
	First *NetworkACLCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// Collection of network ACLs.
	NetworkAcls []NetworkACL `json:"network_acls" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *NetworkACLCollectionNext `json:"next,omitempty"`
}

// UnmarshalNetworkACLCollection unmarshals an instance of NetworkACLCollection from the specified map of raw messages.
func UnmarshalNetworkACLCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalNetworkACLCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_acls", &obj.NetworkAcls, UnmarshalNetworkACL)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalNetworkACLCollectionNext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLCollectionFirst : A link to the first page of resources.
type NetworkACLCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalNetworkACLCollectionFirst unmarshals an instance of NetworkACLCollectionFirst from the specified map of raw messages.
func UnmarshalNetworkACLCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type NetworkACLCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalNetworkACLCollectionNext unmarshals an instance of NetworkACLCollectionNext from the specified map of raw messages.
func UnmarshalNetworkACLCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLIdentity : Identifies a network ACL by a unique property.
// Models which "extend" this model:
// - NetworkACLIdentityByID
// - NetworkACLIdentityByHref
type NetworkACLIdentity struct {
	// The unique identifier for this network ACL.
	ID *string `json:"id,omitempty"`

	// The URL for this network ACL.
	Href *string `json:"href,omitempty"`
}

func (*NetworkACLIdentity) isaNetworkACLIdentity() bool {
	return true
}

type NetworkACLIdentityIntf interface {
	isaNetworkACLIdentity() bool
}

// UnmarshalNetworkACLIdentity unmarshals an instance of NetworkACLIdentity from the specified map of raw messages.
func UnmarshalNetworkACLIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLPatch : NetworkACLPatch struct
type NetworkACLPatch struct {
	// The unique user-defined name for this network ACL.
	Name *string `json:"name,omitempty"`
}

// UnmarshalNetworkACLPatch unmarshals an instance of NetworkACLPatch from the specified map of raw messages.
func UnmarshalNetworkACLPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the NetworkACLPatch
func (networkACLPatch *NetworkACLPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(networkACLPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// NetworkACLPrototype : NetworkACLPrototype struct
// Models which "extend" this model:
// - NetworkACLPrototypeNetworkACLByRules
// - NetworkACLPrototypeNetworkACLBySourceNetworkACL
type NetworkACLPrototype struct {
	// The unique user-defined name for this network ACL. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// Array of prototype objects for rules to create along with this network ACL. If unspecified, rules will be created to
	// allow all traffic.
	Rules []NetworkACLRulePrototypeNetworkACLContextIntf `json:"rules,omitempty"`

	// Network ACL to copy rules from.
	SourceNetworkACL NetworkACLIdentityIntf `json:"source_network_acl,omitempty"`
}

func (*NetworkACLPrototype) isaNetworkACLPrototype() bool {
	return true
}

type NetworkACLPrototypeIntf interface {
	isaNetworkACLPrototype() bool
}

// UnmarshalNetworkACLPrototype unmarshals an instance of NetworkACLPrototype from the specified map of raw messages.
func UnmarshalNetworkACLPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLPrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalNetworkACLRulePrototypeNetworkACLContext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "source_network_acl", &obj.SourceNetworkACL, UnmarshalNetworkACLIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLReference : NetworkACLReference struct
type NetworkACLReference struct {
	// The URL for this network ACL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network ACL.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this network ACL.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalNetworkACLReference unmarshals an instance of NetworkACLReference from the specified map of raw messages.
func UnmarshalNetworkACLReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLRule : NetworkACLRule struct
// Models which "extend" this model:
// - NetworkACLRuleNetworkACLRuleProtocolTcpudp
// - NetworkACLRuleNetworkACLRuleProtocolIcmp
// - NetworkACLRuleNetworkACLRuleProtocolAll
type NetworkACLRule struct {
	// Whether to allow or deny matching traffic.
	Action *string `json:"action" validate:"required"`

	// The rule that this rule is immediately before. If absent, this is the last rule.
	Before *NetworkACLRuleReference `json:"before,omitempty"`

	// The date and time that the rule was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The destination CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Destination *string `json:"destination" validate:"required"`

	// Whether the traffic to be matched is `inbound` or `outbound`.
	Direction *string `json:"direction" validate:"required"`

	// The URL for this network ACL rule.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network ACL rule.
	ID *string `json:"id" validate:"required"`

	// The IP version for this rule.
	IPVersion *string `json:"ip_version" validate:"required"`

	// The user-defined name for this rule. Names must be unique within the network ACL the rule resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string `json:"name" validate:"required"`

	// The protocol to enforce.
	Protocol *string `json:"protocol" validate:"required"`

	// The source CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Source *string `json:"source" validate:"required"`

	// The inclusive upper bound of TCP/UDP destination port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP destination port range.
	PortMin *int64 `json:"port_min,omitempty"`

	// The inclusive upper bound of TCP/UDP source port range.
	SourcePortMax *int64 `json:"source_port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP source port range.
	SourcePortMin *int64 `json:"source_port_min,omitempty"`

	// The ICMP traffic code to allow. If unspecified, all codes are allowed. This can only be specified if type is also
	// specified.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow. If unspecified, all types are allowed by this rule.
	Type *int64 `json:"type,omitempty"`
}

// Constants associated with the NetworkACLRule.Action property.
// Whether to allow or deny matching traffic.
const (
	NetworkACLRuleActionAllowConst = "allow"
	NetworkACLRuleActionDenyConst  = "deny"
)

// Constants associated with the NetworkACLRule.Direction property.
// Whether the traffic to be matched is `inbound` or `outbound`.
const (
	NetworkACLRuleDirectionInboundConst  = "inbound"
	NetworkACLRuleDirectionOutboundConst = "outbound"
)

// Constants associated with the NetworkACLRule.IPVersion property.
// The IP version for this rule.
const (
	NetworkACLRuleIPVersionIpv4Const = "ipv4"
	NetworkACLRuleIPVersionIpv6Const = "ipv6"
)

func (*NetworkACLRule) isaNetworkACLRule() bool {
	return true
}

type NetworkACLRuleIntf interface {
	isaNetworkACLRule() bool
}

// UnmarshalNetworkACLRule unmarshals an instance of NetworkACLRule from the specified map of raw messages.
func UnmarshalNetworkACLRule(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleNetworkACLRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleNetworkACLRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleNetworkACLRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleNetworkACLRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// NetworkACLRuleBeforePatch : The rule to move this rule immediately before. Specify `null` to move this rule after all existing rules.
// Models which "extend" this model:
// - NetworkACLRuleBeforePatchNetworkACLRuleIdentityByID
// - NetworkACLRuleBeforePatchNetworkACLRuleIdentityByHref
type NetworkACLRuleBeforePatch struct {
	// The unique identifier for this network ACL rule.
	ID *string `json:"id,omitempty"`

	// The URL for this network ACL rule.
	Href *string `json:"href,omitempty"`
}

func (*NetworkACLRuleBeforePatch) isaNetworkACLRuleBeforePatch() bool {
	return true
}

type NetworkACLRuleBeforePatchIntf interface {
	isaNetworkACLRuleBeforePatch() bool
}

// UnmarshalNetworkACLRuleBeforePatch unmarshals an instance of NetworkACLRuleBeforePatch from the specified map of raw messages.
func UnmarshalNetworkACLRuleBeforePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLRuleBeforePatch)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLRuleBeforePrototype : The rule to insert this rule immediately before. If omitted, this rule will be inserted after all existing rules.
// Models which "extend" this model:
// - NetworkACLRuleBeforePrototypeNetworkACLRuleIdentityByID
// - NetworkACLRuleBeforePrototypeNetworkACLRuleIdentityByHref
type NetworkACLRuleBeforePrototype struct {
	// The unique identifier for this network ACL rule.
	ID *string `json:"id,omitempty"`

	// The URL for this network ACL rule.
	Href *string `json:"href,omitempty"`
}

func (*NetworkACLRuleBeforePrototype) isaNetworkACLRuleBeforePrototype() bool {
	return true
}

type NetworkACLRuleBeforePrototypeIntf interface {
	isaNetworkACLRuleBeforePrototype() bool
}

// UnmarshalNetworkACLRuleBeforePrototype unmarshals an instance of NetworkACLRuleBeforePrototype from the specified map of raw messages.
func UnmarshalNetworkACLRuleBeforePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLRuleBeforePrototype)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLRuleCollection : NetworkACLRuleCollection struct
type NetworkACLRuleCollection struct {
	// A link to the first page of resources.
	First *NetworkACLRuleCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *NetworkACLRuleCollectionNext `json:"next,omitempty"`

	// Ordered collection of network ACL rules.
	Rules []NetworkACLRuleItemIntf `json:"rules" validate:"required"`
}

// UnmarshalNetworkACLRuleCollection unmarshals an instance of NetworkACLRuleCollection from the specified map of raw messages.
func UnmarshalNetworkACLRuleCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLRuleCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalNetworkACLRuleCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalNetworkACLRuleCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalNetworkACLRuleItem)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLRuleCollectionFirst : A link to the first page of resources.
type NetworkACLRuleCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalNetworkACLRuleCollectionFirst unmarshals an instance of NetworkACLRuleCollectionFirst from the specified map of raw messages.
func UnmarshalNetworkACLRuleCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLRuleCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLRuleCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type NetworkACLRuleCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalNetworkACLRuleCollectionNext unmarshals an instance of NetworkACLRuleCollectionNext from the specified map of raw messages.
func UnmarshalNetworkACLRuleCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLRuleCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkACLRuleItem : NetworkACLRuleItem struct
// Models which "extend" this model:
// - NetworkACLRuleItemNetworkACLRuleProtocolTcpudp
// - NetworkACLRuleItemNetworkACLRuleProtocolIcmp
// - NetworkACLRuleItemNetworkACLRuleProtocolAll
type NetworkACLRuleItem struct {
	// Whether to allow or deny matching traffic.
	Action *string `json:"action" validate:"required"`

	// The rule that this rule is immediately before. In a rule collection, this always
	// refers to the next item in the collection. If absent, this is the last rule.
	Before *NetworkACLRuleReference `json:"before,omitempty"`

	// The date and time that the rule was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The destination CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Destination *string `json:"destination" validate:"required"`

	// Whether the traffic to be matched is `inbound` or `outbound`.
	Direction *string `json:"direction" validate:"required"`

	// The URL for this network ACL rule.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network ACL rule.
	ID *string `json:"id" validate:"required"`

	// The IP version for this rule.
	IPVersion *string `json:"ip_version" validate:"required"`

	// The user-defined name for this rule. Names must be unique within the network ACL the rule resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string `json:"name" validate:"required"`

	// The protocol to enforce.
	Protocol *string `json:"protocol" validate:"required"`

	// The source CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Source *string `json:"source" validate:"required"`

	// The inclusive upper bound of TCP/UDP destination port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP destination port range.
	PortMin *int64 `json:"port_min,omitempty"`

	// The inclusive upper bound of TCP/UDP source port range.
	SourcePortMax *int64 `json:"source_port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP source port range.
	SourcePortMin *int64 `json:"source_port_min,omitempty"`

	// The ICMP traffic code to allow. If unspecified, all codes are allowed. This can only be specified if type is also
	// specified.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow. If unspecified, all types are allowed by this rule.
	Type *int64 `json:"type,omitempty"`
}

// Constants associated with the NetworkACLRuleItem.Action property.
// Whether to allow or deny matching traffic.
const (
	NetworkACLRuleItemActionAllowConst = "allow"
	NetworkACLRuleItemActionDenyConst  = "deny"
)

// Constants associated with the NetworkACLRuleItem.Direction property.
// Whether the traffic to be matched is `inbound` or `outbound`.
const (
	NetworkACLRuleItemDirectionInboundConst  = "inbound"
	NetworkACLRuleItemDirectionOutboundConst = "outbound"
)

// Constants associated with the NetworkACLRuleItem.IPVersion property.
// The IP version for this rule.
const (
	NetworkACLRuleItemIPVersionIpv4Const = "ipv4"
	NetworkACLRuleItemIPVersionIpv6Const = "ipv6"
)

func (*NetworkACLRuleItem) isaNetworkACLRuleItem() bool {
	return true
}

type NetworkACLRuleItemIntf interface {
	isaNetworkACLRuleItem() bool
}

// UnmarshalNetworkACLRuleItem unmarshals an instance of NetworkACLRuleItem from the specified map of raw messages.
func UnmarshalNetworkACLRuleItem(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleItemNetworkACLRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleItemNetworkACLRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleItemNetworkACLRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRuleItemNetworkACLRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// NetworkACLRulePatch : NetworkACLRulePatch struct
// Models which "extend" this model:
// - NetworkACLRulePatchNetworkACLRuleProtocolTcpudp
// - NetworkACLRulePatchNetworkACLRuleProtocolIcmp
// - NetworkACLRulePatchNetworkACLRuleProtocolAll
type NetworkACLRulePatch struct {
	// Whether to allow or deny matching traffic.
	Action *string `json:"action,omitempty"`

	// The rule to move this rule immediately before. Specify `null` to move this rule after
	// all existing rules.
	Before NetworkACLRuleBeforePatchIntf `json:"before,omitempty"`

	// The destination IP address or CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Destination *string `json:"destination,omitempty"`

	// Whether the traffic to be matched is `inbound` or `outbound`.
	Direction *string `json:"direction,omitempty"`

	// The user-defined name for this rule. Names must be unique within the network ACL the rule resides in.
	Name *string `json:"name,omitempty"`

	// The protocol to enforce.
	Protocol *string `json:"protocol,omitempty"`

	// The source IP address or CIDR block.  The CIDR block `0.0.0.0/0` applies to all addresses.
	Source *string `json:"source,omitempty"`

	// The inclusive upper bound of TCP/UDP destination port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP destination port range.
	PortMin *int64 `json:"port_min,omitempty"`

	// The inclusive upper bound of TCP/UDP source port range.
	SourcePortMax *int64 `json:"source_port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP source port range.
	SourcePortMin *int64 `json:"source_port_min,omitempty"`

	// The ICMP traffic code to allow. If unspecified, all codes are allowed. This can only be specified if type is also
	// specified.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow. If unspecified, all types are allowed by this rule.
	Type *int64 `json:"type,omitempty"`
}

// Constants associated with the NetworkACLRulePatch.Action property.
// Whether to allow or deny matching traffic.
const (
	NetworkACLRulePatchActionAllowConst = "allow"
	NetworkACLRulePatchActionDenyConst  = "deny"
)

// Constants associated with the NetworkACLRulePatch.Direction property.
// Whether the traffic to be matched is `inbound` or `outbound`.
const (
	NetworkACLRulePatchDirectionInboundConst  = "inbound"
	NetworkACLRulePatchDirectionOutboundConst = "outbound"
)

func (*NetworkACLRulePatch) isaNetworkACLRulePatch() bool {
	return true
}

type NetworkACLRulePatchIntf interface {
	isaNetworkACLRulePatch() bool
}

// UnmarshalNetworkACLRulePatch unmarshals an instance of NetworkACLRulePatch from the specified map of raw messages.
func UnmarshalNetworkACLRulePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePatchNetworkACLRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePatchNetworkACLRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePatchNetworkACLRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePatchNetworkACLRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// AsPatch returns a generic map representation of the NetworkACLRulePatch
func (networkACLRulePatch *NetworkACLRulePatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(networkACLRulePatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// NetworkACLRulePrototype : NetworkACLRulePrototype struct
// Models which "extend" this model:
// - NetworkACLRulePrototypeNetworkACLRuleProtocolTcpudp
// - NetworkACLRulePrototypeNetworkACLRuleProtocolIcmp
// - NetworkACLRulePrototypeNetworkACLRuleProtocolAll
type NetworkACLRulePrototype struct {
	// Whether to allow or deny matching traffic.
	Action *string `json:"action" validate:"required"`

	// The rule to insert this rule immediately before. If omitted, this rule will be
	// inserted after all existing rules.
	Before NetworkACLRuleBeforePrototypeIntf `json:"before,omitempty"`

	// The destination IP address or CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Destination *string `json:"destination" validate:"required"`

	// Whether the traffic to be matched is `inbound` or `outbound`.
	Direction *string `json:"direction" validate:"required"`

	// The user-defined name for this rule. Names must be unique within the network ACL the rule resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The protocol to enforce.
	Protocol *string `json:"protocol" validate:"required"`

	// The source IP address or CIDR block.  The CIDR block `0.0.0.0/0` applies to all addresses.
	Source *string `json:"source" validate:"required"`

	// The inclusive upper bound of TCP/UDP destination port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP destination port range.
	PortMin *int64 `json:"port_min,omitempty"`

	// The inclusive upper bound of TCP/UDP source port range.
	SourcePortMax *int64 `json:"source_port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP source port range.
	SourcePortMin *int64 `json:"source_port_min,omitempty"`

	// The ICMP traffic code to allow. If unspecified, all codes are allowed. This can only be specified if type is also
	// specified.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow. If unspecified, all types are allowed by this rule.
	Type *int64 `json:"type,omitempty"`
}

// Constants associated with the NetworkACLRulePrototype.Action property.
// Whether to allow or deny matching traffic.
const (
	NetworkACLRulePrototypeActionAllowConst = "allow"
	NetworkACLRulePrototypeActionDenyConst  = "deny"
)

// Constants associated with the NetworkACLRulePrototype.Direction property.
// Whether the traffic to be matched is `inbound` or `outbound`.
const (
	NetworkACLRulePrototypeDirectionInboundConst  = "inbound"
	NetworkACLRulePrototypeDirectionOutboundConst = "outbound"
)

func (*NetworkACLRulePrototype) isaNetworkACLRulePrototype() bool {
	return true
}

type NetworkACLRulePrototypeIntf interface {
	isaNetworkACLRulePrototype() bool
}

// UnmarshalNetworkACLRulePrototype unmarshals an instance of NetworkACLRulePrototype from the specified map of raw messages.
func UnmarshalNetworkACLRulePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// NetworkACLRulePrototypeNetworkACLContext : NetworkACLRulePrototypeNetworkACLContext struct
// Models which "extend" this model:
// - NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTcpudp
// - NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolIcmp
// - NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll
type NetworkACLRulePrototypeNetworkACLContext struct {
	// Whether to allow or deny matching traffic.
	Action *string `json:"action" validate:"required"`

	// The destination IP address or CIDR block. The CIDR block `0.0.0.0/0` applies to all addresses.
	Destination *string `json:"destination" validate:"required"`

	// Whether the traffic to be matched is `inbound` or `outbound`.
	Direction *string `json:"direction" validate:"required"`

	// The user-defined name for this rule. Names must be unique within the network ACL the rule resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The protocol to enforce.
	Protocol *string `json:"protocol" validate:"required"`

	// The source IP address or CIDR block.  The CIDR block `0.0.0.0/0` applies to all addresses.
	Source *string `json:"source" validate:"required"`

	// The inclusive upper bound of TCP/UDP destination port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP destination port range.
	PortMin *int64 `json:"port_min,omitempty"`

	// The inclusive upper bound of TCP/UDP source port range.
	SourcePortMax *int64 `json:"source_port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP source port range.
	SourcePortMin *int64 `json:"source_port_min,omitempty"`

	// The ICMP traffic code to allow. If unspecified, all codes are allowed. This can only be specified if type is also
	// specified.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow. If unspecified, all types are allowed by this rule.
	Type *int64 `json:"type,omitempty"`
}

// Constants associated with the NetworkACLRulePrototypeNetworkACLContext.Action property.
// Whether to allow or deny matching traffic.
const (
	NetworkACLRulePrototypeNetworkACLContextActionAllowConst = "allow"
	NetworkACLRulePrototypeNetworkACLContextActionDenyConst  = "deny"
)

// Constants associated with the NetworkACLRulePrototypeNetworkACLContext.Direction property.
// Whether the traffic to be matched is `inbound` or `outbound`.
const (
	NetworkACLRulePrototypeNetworkACLContextDirectionInboundConst  = "inbound"
	NetworkACLRulePrototypeNetworkACLContextDirectionOutboundConst = "outbound"
)

func (*NetworkACLRulePrototypeNetworkACLContext) isaNetworkACLRulePrototypeNetworkACLContext() bool {
	return true
}

type NetworkACLRulePrototypeNetworkACLContextIntf interface {
	isaNetworkACLRulePrototypeNetworkACLContext() bool
}

// UnmarshalNetworkACLRulePrototypeNetworkACLContext unmarshals an instance of NetworkACLRulePrototypeNetworkACLContext from the specified map of raw messages.
func UnmarshalNetworkACLRulePrototypeNetworkACLContext(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalNetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// NetworkACLRuleReference : NetworkACLRuleReference struct
type NetworkACLRuleReference struct {
	// The URL for this network ACL rule.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network ACL rule.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this network ACL rule.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalNetworkACLRuleReference unmarshals an instance of NetworkACLRuleReference from the specified map of raw messages.
func UnmarshalNetworkACLRuleReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkACLRuleReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkInterface : NetworkInterface struct
type NetworkInterface struct {
	// The date and time that the network interface was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// Array of references to floating IPs associated with this network interface.
	FloatingIps []FloatingIPReference `json:"floating_ips,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network interface.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this network interface.
	Name *string `json:"name" validate:"required"`

	// The network interface port speed in Mbps.
	PortSpeed *int64 `json:"port_speed" validate:"required"`

	// The primary IPv4 address.
	PrimaryIpv4Address *string `json:"primary_ipv4_address" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// Collection of security groups.
	SecurityGroups []SecurityGroupReference `json:"security_groups" validate:"required"`

	// The status of the network interface.
	//
	// Due to a known issue, the value for this property may be `ACTIVE` instead of
	// `available`.
	Status *string `json:"status" validate:"required"`

	// The associated subnet.
	Subnet *SubnetReference `json:"subnet" validate:"required"`

	// The type of this network interface as it relates to an instance.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the NetworkInterface.ResourceType property.
// The resource type.
const (
	NetworkInterfaceResourceTypeNetworkInterfaceConst = "network_interface"
)

// Constants associated with the NetworkInterface.Status property.
// The status of the network interface.
//
// Due to a known issue, the value for this property may be `ACTIVE` instead of
// `available`.
const (
	NetworkInterfaceStatusActiveConst    = "active"
	NetworkInterfaceStatusAvailableConst = "available"
	NetworkInterfaceStatusDeletingConst  = "deleting"
	NetworkInterfaceStatusFailedConst    = "failed"
	NetworkInterfaceStatusPendingConst   = "pending"
)

// Constants associated with the NetworkInterface.Type property.
// The type of this network interface as it relates to an instance.
const (
	NetworkInterfaceTypePrimaryConst   = "primary"
	NetworkInterfaceTypeSecondaryConst = "secondary"
)

// UnmarshalNetworkInterface unmarshals an instance of NetworkInterface from the specified map of raw messages.
func UnmarshalNetworkInterface(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkInterface)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "floating_ips", &obj.FloatingIps, UnmarshalFloatingIPReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "port_speed", &obj.PortSpeed)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "primary_ipv4_address", &obj.PrimaryIpv4Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "security_groups", &obj.SecurityGroups, UnmarshalSecurityGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnet", &obj.Subnet, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkInterfaceCollection : NetworkInterfaceCollection struct
type NetworkInterfaceCollection struct {
	// Collection of network interfaces.
	NetworkInterfaces []NetworkInterface `json:"network_interfaces" validate:"required"`
}

// UnmarshalNetworkInterfaceCollection unmarshals an instance of NetworkInterfaceCollection from the specified map of raw messages.
func UnmarshalNetworkInterfaceCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkInterfaceCollection)
	err = core.UnmarshalModel(m, "network_interfaces", &obj.NetworkInterfaces, UnmarshalNetworkInterface)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkInterfaceInstanceContextReference : NetworkInterfaceInstanceContextReference struct
type NetworkInterfaceInstanceContextReference struct {
	// The URL for this network interface.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network interface.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this network interface.
	Name *string `json:"name" validate:"required"`

	// The primary IPv4 address.
	PrimaryIpv4Address *string `json:"primary_ipv4_address" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The associated subnet.
	Subnet *SubnetReference `json:"subnet" validate:"required"`
}

// Constants associated with the NetworkInterfaceInstanceContextReference.ResourceType property.
// The resource type.
const (
	NetworkInterfaceInstanceContextReferenceResourceTypeNetworkInterfaceConst = "network_interface"
)

// UnmarshalNetworkInterfaceInstanceContextReference unmarshals an instance of NetworkInterfaceInstanceContextReference from the specified map of raw messages.
func UnmarshalNetworkInterfaceInstanceContextReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkInterfaceInstanceContextReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "primary_ipv4_address", &obj.PrimaryIpv4Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnet", &obj.Subnet, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkInterfacePrototype : NetworkInterfacePrototype struct
type NetworkInterfacePrototype struct {
	// The user-defined name for this network interface. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The primary IPv4 address. If specified, it must be an available address on the network interface's subnet. If
	// unspecified, an available address on the subnet will be automatically selected.
	PrimaryIpv4Address *string `json:"primary_ipv4_address,omitempty"`

	// Collection of security groups.
	SecurityGroups []SecurityGroupIdentityIntf `json:"security_groups,omitempty"`

	// The associated subnet.
	Subnet SubnetIdentityIntf `json:"subnet" validate:"required"`
}

// NewNetworkInterfacePrototype : Instantiate NetworkInterfacePrototype (Generic Model Constructor)
func (*VpcClassicV1) NewNetworkInterfacePrototype(subnet SubnetIdentityIntf) (model *NetworkInterfacePrototype, err error) {
	model = &NetworkInterfacePrototype{
		Subnet: subnet,
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalNetworkInterfacePrototype unmarshals an instance of NetworkInterfacePrototype from the specified map of raw messages.
func UnmarshalNetworkInterfacePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkInterfacePrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "primary_ipv4_address", &obj.PrimaryIpv4Address)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "security_groups", &obj.SecurityGroups, UnmarshalSecurityGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnet", &obj.Subnet, UnmarshalSubnetIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkInterfaceReference : NetworkInterfaceReference struct
type NetworkInterfaceReference struct {
	// The URL for this network interface.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network interface.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this network interface.
	Name *string `json:"name" validate:"required"`

	// The primary IPv4 address.
	PrimaryIpv4Address *string `json:"primary_ipv4_address" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the NetworkInterfaceReference.ResourceType property.
// The resource type.
const (
	NetworkInterfaceReferenceResourceTypeNetworkInterfaceConst = "network_interface"
)

// UnmarshalNetworkInterfaceReference unmarshals an instance of NetworkInterfaceReference from the specified map of raw messages.
func UnmarshalNetworkInterfaceReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkInterfaceReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "primary_ipv4_address", &obj.PrimaryIpv4Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// NetworkInterfaceUnpaginatedCollection : NetworkInterfaceUnpaginatedCollection struct
type NetworkInterfaceUnpaginatedCollection struct {
	// Collection of network interfaces.
	NetworkInterfaces []NetworkInterface `json:"network_interfaces" validate:"required"`
}

// UnmarshalNetworkInterfaceUnpaginatedCollection unmarshals an instance of NetworkInterfaceUnpaginatedCollection from the specified map of raw messages.
func UnmarshalNetworkInterfaceUnpaginatedCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(NetworkInterfaceUnpaginatedCollection)
	err = core.UnmarshalModel(m, "network_interfaces", &obj.NetworkInterfaces, UnmarshalNetworkInterface)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OperatingSystem : OperatingSystem struct
type OperatingSystem struct {
	// The operating system architecture.
	Architecture *string `json:"architecture" validate:"required"`

	// A unique, display-friendly name for the operating system.
	DisplayName *string `json:"display_name" validate:"required"`

	// The name of the software family this operating system belongs to.
	Family *string `json:"family" validate:"required"`

	// The URL for this operating system.
	Href *string `json:"href" validate:"required"`

	// The globally unique name for this operating system.
	Name *string `json:"name" validate:"required"`

	// The vendor of the operating system.
	Vendor *string `json:"vendor" validate:"required"`

	// The major release version of this operating system.
	Version *string `json:"version" validate:"required"`
}

// UnmarshalOperatingSystem unmarshals an instance of OperatingSystem from the specified map of raw messages.
func UnmarshalOperatingSystem(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OperatingSystem)
	err = core.UnmarshalPrimitive(m, "architecture", &obj.Architecture)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "display_name", &obj.DisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "family", &obj.Family)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "vendor", &obj.Vendor)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "version", &obj.Version)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OperatingSystemCollection : OperatingSystemCollection struct
type OperatingSystemCollection struct {
	// A link to the first page of resources.
	First *OperatingSystemCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *OperatingSystemCollectionNext `json:"next,omitempty"`

	// Collection of operating systems.
	OperatingSystems []OperatingSystem `json:"operating_systems" validate:"required"`
}

// UnmarshalOperatingSystemCollection unmarshals an instance of OperatingSystemCollection from the specified map of raw messages.
func UnmarshalOperatingSystemCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OperatingSystemCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalOperatingSystemCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalOperatingSystemCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "operating_systems", &obj.OperatingSystems, UnmarshalOperatingSystem)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OperatingSystemCollectionFirst : A link to the first page of resources.
type OperatingSystemCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalOperatingSystemCollectionFirst unmarshals an instance of OperatingSystemCollectionFirst from the specified map of raw messages.
func UnmarshalOperatingSystemCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OperatingSystemCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OperatingSystemCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type OperatingSystemCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalOperatingSystemCollectionNext unmarshals an instance of OperatingSystemCollectionNext from the specified map of raw messages.
func UnmarshalOperatingSystemCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OperatingSystemCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OperatingSystemIdentity : Identifies an operating system by a unique property.
// Models which "extend" this model:
// - OperatingSystemIdentityByName
// - OperatingSystemIdentityByHref
type OperatingSystemIdentity struct {
	// The globally unique name for this operating system.
	Name *string `json:"name,omitempty"`

	// The URL for this operating system.
	Href *string `json:"href,omitempty"`
}

func (*OperatingSystemIdentity) isaOperatingSystemIdentity() bool {
	return true
}

type OperatingSystemIdentityIntf interface {
	isaOperatingSystemIdentity() bool
}

// UnmarshalOperatingSystemIdentity unmarshals an instance of OperatingSystemIdentity from the specified map of raw messages.
func UnmarshalOperatingSystemIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OperatingSystemIdentity)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGateway : PublicGateway struct
type PublicGateway struct {
	// The date and time that the public gateway was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this public gateway.
	CRN *string `json:"crn" validate:"required"`

	// Reference to the floating IP which is bound to this public gateway.
	FloatingIP *PublicGatewayFloatingIP `json:"floating_ip" validate:"required"`

	// The URL for this public gateway.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this public gateway.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this public gateway.
	Name *string `json:"name" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The status of the volume.
	Status *string `json:"status" validate:"required"`

	// The VPC this public gateway serves.
	VPC *VPCReference `json:"vpc" validate:"required"`

	// The zone where this public gateway lives.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the PublicGateway.ResourceType property.
// The resource type.
const (
	PublicGatewayResourceTypePublicGatewayConst = "public_gateway"
)

// Constants associated with the PublicGateway.Status property.
// The status of the volume.
const (
	PublicGatewayStatusAvailableConst = "available"
	PublicGatewayStatusDeletingConst  = "deleting"
	PublicGatewayStatusFailedConst    = "failed"
	PublicGatewayStatusPendingConst   = "pending"
)

// UnmarshalPublicGateway unmarshals an instance of PublicGateway from the specified map of raw messages.
func UnmarshalPublicGateway(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGateway)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "floating_ip", &obj.FloatingIP, UnmarshalPublicGatewayFloatingIP)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayCollection : PublicGatewayCollection struct
type PublicGatewayCollection struct {
	// A link to the first page of resources.
	First *PublicGatewayCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *PublicGatewayCollectionNext `json:"next,omitempty"`

	// Collection of public gateways.
	PublicGateways []PublicGateway `json:"public_gateways" validate:"required"`
}

// UnmarshalPublicGatewayCollection unmarshals an instance of PublicGatewayCollection from the specified map of raw messages.
func UnmarshalPublicGatewayCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalPublicGatewayCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalPublicGatewayCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "public_gateways", &obj.PublicGateways, UnmarshalPublicGateway)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayCollectionFirst : A link to the first page of resources.
type PublicGatewayCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalPublicGatewayCollectionFirst unmarshals an instance of PublicGatewayCollectionFirst from the specified map of raw messages.
func UnmarshalPublicGatewayCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type PublicGatewayCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalPublicGatewayCollectionNext unmarshals an instance of PublicGatewayCollectionNext from the specified map of raw messages.
func UnmarshalPublicGatewayCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayFloatingIP : Reference to the floating IP which is bound to this public gateway.
type PublicGatewayFloatingIP struct {
	// The globally unique IP address.
	Address *string `json:"address" validate:"required"`

	// The CRN for this floating IP.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this floating IP.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this floating IP.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this floating IP.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalPublicGatewayFloatingIP unmarshals an instance of PublicGatewayFloatingIP from the specified map of raw messages.
func UnmarshalPublicGatewayFloatingIP(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayFloatingIP)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayFloatingIPPrototype : PublicGatewayFloatingIPPrototype struct
// Models which "extend" this model:
// - PublicGatewayFloatingIPPrototypeFloatingIPIdentity
// - PublicGatewayFloatingIPPrototypeFloatingIPPrototypeTargetContext
type PublicGatewayFloatingIPPrototype struct {
	// The unique identifier for this floating IP.
	ID *string `json:"id,omitempty"`

	// The CRN for this floating IP.
	CRN *string `json:"crn,omitempty"`

	// The URL for this floating IP.
	Href *string `json:"href,omitempty"`

	// The globally unique IP address.
	Address *string `json:"address,omitempty"`

	// The unique user-defined name for this floating IP. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`
}

func (*PublicGatewayFloatingIPPrototype) isaPublicGatewayFloatingIPPrototype() bool {
	return true
}

type PublicGatewayFloatingIPPrototypeIntf interface {
	isaPublicGatewayFloatingIPPrototype() bool
}

// UnmarshalPublicGatewayFloatingIPPrototype unmarshals an instance of PublicGatewayFloatingIPPrototype from the specified map of raw messages.
func UnmarshalPublicGatewayFloatingIPPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayFloatingIPPrototype)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayIdentity : Identifies a public gateway by a unique property.
// Models which "extend" this model:
// - PublicGatewayIdentityByID
// - PublicGatewayIdentityByCRN
// - PublicGatewayIdentityByHref
type PublicGatewayIdentity struct {
	// The unique identifier for this public gateway.
	ID *string `json:"id,omitempty"`

	// The CRN for this public gateway.
	CRN *string `json:"crn,omitempty"`

	// The URL for this public gateway.
	Href *string `json:"href,omitempty"`
}

func (*PublicGatewayIdentity) isaPublicGatewayIdentity() bool {
	return true
}

type PublicGatewayIdentityIntf interface {
	isaPublicGatewayIdentity() bool
}

// UnmarshalPublicGatewayIdentity unmarshals an instance of PublicGatewayIdentity from the specified map of raw messages.
func UnmarshalPublicGatewayIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PublicGatewayPatch : PublicGatewayPatch struct
type PublicGatewayPatch struct {
	// The user-defined name for this public gateway. Names must be unique within the VPC the public gateway resides in.
	Name *string `json:"name,omitempty"`
}

// UnmarshalPublicGatewayPatch unmarshals an instance of PublicGatewayPatch from the specified map of raw messages.
func UnmarshalPublicGatewayPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the PublicGatewayPatch
func (publicGatewayPatch *PublicGatewayPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(publicGatewayPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// PublicGatewayReference : PublicGatewayReference struct
type PublicGatewayReference struct {
	// The CRN for this public gateway.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this public gateway.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this public gateway.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this public gateway.
	Name *string `json:"name" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the PublicGatewayReference.ResourceType property.
// The resource type.
const (
	PublicGatewayReferenceResourceTypePublicGatewayConst = "public_gateway"
)

// UnmarshalPublicGatewayReference unmarshals an instance of PublicGatewayReference from the specified map of raw messages.
func UnmarshalPublicGatewayReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PublicGatewayReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Region : Region struct
type Region struct {
	// The API endpoint for this region.
	Endpoint *string `json:"endpoint" validate:"required"`

	// The URL for this region.
	Href *string `json:"href" validate:"required"`

	// The globally unique name for this region.
	Name *string `json:"name" validate:"required"`

	// The availability status of this region.
	Status *string `json:"status" validate:"required"`
}

// Constants associated with the Region.Status property.
// The availability status of this region.
const (
	RegionStatusAvailableConst   = "available"
	RegionStatusUnavailableConst = "unavailable"
)

// UnmarshalRegion unmarshals an instance of Region from the specified map of raw messages.
func UnmarshalRegion(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Region)
	err = core.UnmarshalPrimitive(m, "endpoint", &obj.Endpoint)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RegionCollection : RegionCollection struct
type RegionCollection struct {
	// Collection of regions.
	Regions []Region `json:"regions" validate:"required"`
}

// UnmarshalRegionCollection unmarshals an instance of RegionCollection from the specified map of raw messages.
func UnmarshalRegionCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(RegionCollection)
	err = core.UnmarshalModel(m, "regions", &obj.Regions, UnmarshalRegion)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RegionReference : RegionReference struct
type RegionReference struct {
	// The URL for this region.
	Href *string `json:"href" validate:"required"`

	// The globally unique name for this region.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalRegionReference unmarshals an instance of RegionReference from the specified map of raw messages.
func UnmarshalRegionReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(RegionReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RemoveInstanceNetworkInterfaceFloatingIPOptions : The RemoveInstanceNetworkInterfaceFloatingIP options.
type RemoveInstanceNetworkInterfaceFloatingIPOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	NetworkInterfaceID *string `validate:"required,ne="`

	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewRemoveInstanceNetworkInterfaceFloatingIPOptions : Instantiate RemoveInstanceNetworkInterfaceFloatingIPOptions
func (*VpcClassicV1) NewRemoveInstanceNetworkInterfaceFloatingIPOptions(instanceID string, networkInterfaceID string, id string) *RemoveInstanceNetworkInterfaceFloatingIPOptions {
	return &RemoveInstanceNetworkInterfaceFloatingIPOptions{
		InstanceID:         core.StringPtr(instanceID),
		NetworkInterfaceID: core.StringPtr(networkInterfaceID),
		ID:                 core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *RemoveInstanceNetworkInterfaceFloatingIPOptions) SetInstanceID(instanceID string) *RemoveInstanceNetworkInterfaceFloatingIPOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetNetworkInterfaceID : Allow user to set NetworkInterfaceID
func (options *RemoveInstanceNetworkInterfaceFloatingIPOptions) SetNetworkInterfaceID(networkInterfaceID string) *RemoveInstanceNetworkInterfaceFloatingIPOptions {
	options.NetworkInterfaceID = core.StringPtr(networkInterfaceID)
	return options
}

// SetID : Allow user to set ID
func (options *RemoveInstanceNetworkInterfaceFloatingIPOptions) SetID(id string) *RemoveInstanceNetworkInterfaceFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *RemoveInstanceNetworkInterfaceFloatingIPOptions) SetHeaders(param map[string]string) *RemoveInstanceNetworkInterfaceFloatingIPOptions {
	options.Headers = param
	return options
}

// RemoveSecurityGroupNetworkInterfaceOptions : The RemoveSecurityGroupNetworkInterface options.
type RemoveSecurityGroupNetworkInterfaceOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewRemoveSecurityGroupNetworkInterfaceOptions : Instantiate RemoveSecurityGroupNetworkInterfaceOptions
func (*VpcClassicV1) NewRemoveSecurityGroupNetworkInterfaceOptions(securityGroupID string, id string) *RemoveSecurityGroupNetworkInterfaceOptions {
	return &RemoveSecurityGroupNetworkInterfaceOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *RemoveSecurityGroupNetworkInterfaceOptions) SetSecurityGroupID(securityGroupID string) *RemoveSecurityGroupNetworkInterfaceOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *RemoveSecurityGroupNetworkInterfaceOptions) SetID(id string) *RemoveSecurityGroupNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *RemoveSecurityGroupNetworkInterfaceOptions) SetHeaders(param map[string]string) *RemoveSecurityGroupNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// RemoveVPNGatewayConnectionLocalCIDROptions : The RemoveVPNGatewayConnectionLocalCIDR options.
type RemoveVPNGatewayConnectionLocalCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewRemoveVPNGatewayConnectionLocalCIDROptions : Instantiate RemoveVPNGatewayConnectionLocalCIDROptions
func (*VpcClassicV1) NewRemoveVPNGatewayConnectionLocalCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *RemoveVPNGatewayConnectionLocalCIDROptions {
	return &RemoveVPNGatewayConnectionLocalCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *RemoveVPNGatewayConnectionLocalCIDROptions) SetVPNGatewayID(vpnGatewayID string) *RemoveVPNGatewayConnectionLocalCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *RemoveVPNGatewayConnectionLocalCIDROptions) SetID(id string) *RemoveVPNGatewayConnectionLocalCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *RemoveVPNGatewayConnectionLocalCIDROptions) SetCIDRPrefix(cidrPrefix string) *RemoveVPNGatewayConnectionLocalCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *RemoveVPNGatewayConnectionLocalCIDROptions) SetPrefixLength(prefixLength string) *RemoveVPNGatewayConnectionLocalCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *RemoveVPNGatewayConnectionLocalCIDROptions) SetHeaders(param map[string]string) *RemoveVPNGatewayConnectionLocalCIDROptions {
	options.Headers = param
	return options
}

// RemoveVPNGatewayConnectionPeerCIDROptions : The RemoveVPNGatewayConnectionPeerCIDR options.
type RemoveVPNGatewayConnectionPeerCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewRemoveVPNGatewayConnectionPeerCIDROptions : Instantiate RemoveVPNGatewayConnectionPeerCIDROptions
func (*VpcClassicV1) NewRemoveVPNGatewayConnectionPeerCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *RemoveVPNGatewayConnectionPeerCIDROptions {
	return &RemoveVPNGatewayConnectionPeerCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *RemoveVPNGatewayConnectionPeerCIDROptions) SetVPNGatewayID(vpnGatewayID string) *RemoveVPNGatewayConnectionPeerCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *RemoveVPNGatewayConnectionPeerCIDROptions) SetID(id string) *RemoveVPNGatewayConnectionPeerCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *RemoveVPNGatewayConnectionPeerCIDROptions) SetCIDRPrefix(cidrPrefix string) *RemoveVPNGatewayConnectionPeerCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *RemoveVPNGatewayConnectionPeerCIDROptions) SetPrefixLength(prefixLength string) *RemoveVPNGatewayConnectionPeerCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *RemoveVPNGatewayConnectionPeerCIDROptions) SetHeaders(param map[string]string) *RemoveVPNGatewayConnectionPeerCIDROptions {
	options.Headers = param
	return options
}

// ReplaceLoadBalancerPoolMembersOptions : The ReplaceLoadBalancerPoolMembers options.
type ReplaceLoadBalancerPoolMembersOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// Array of pool member prototype objects.
	Members []LoadBalancerPoolMemberPrototype `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewReplaceLoadBalancerPoolMembersOptions : Instantiate ReplaceLoadBalancerPoolMembersOptions
func (*VpcClassicV1) NewReplaceLoadBalancerPoolMembersOptions(loadBalancerID string, poolID string, members []LoadBalancerPoolMemberPrototype) *ReplaceLoadBalancerPoolMembersOptions {
	return &ReplaceLoadBalancerPoolMembersOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		Members:        members,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *ReplaceLoadBalancerPoolMembersOptions) SetLoadBalancerID(loadBalancerID string) *ReplaceLoadBalancerPoolMembersOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *ReplaceLoadBalancerPoolMembersOptions) SetPoolID(poolID string) *ReplaceLoadBalancerPoolMembersOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetMembers : Allow user to set Members
func (options *ReplaceLoadBalancerPoolMembersOptions) SetMembers(members []LoadBalancerPoolMemberPrototype) *ReplaceLoadBalancerPoolMembersOptions {
	options.Members = members
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ReplaceLoadBalancerPoolMembersOptions) SetHeaders(param map[string]string) *ReplaceLoadBalancerPoolMembersOptions {
	options.Headers = param
	return options
}

// ReplaceSubnetNetworkACLOptions : The ReplaceSubnetNetworkACL options.
type ReplaceSubnetNetworkACLOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// The network ACL identity.
	NetworkACLIdentity NetworkACLIdentityIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewReplaceSubnetNetworkACLOptions : Instantiate ReplaceSubnetNetworkACLOptions
func (*VpcClassicV1) NewReplaceSubnetNetworkACLOptions(id string, networkACLIdentity NetworkACLIdentityIntf) *ReplaceSubnetNetworkACLOptions {
	return &ReplaceSubnetNetworkACLOptions{
		ID:                 core.StringPtr(id),
		NetworkACLIdentity: networkACLIdentity,
	}
}

// SetID : Allow user to set ID
func (options *ReplaceSubnetNetworkACLOptions) SetID(id string) *ReplaceSubnetNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetNetworkACLIdentity : Allow user to set NetworkACLIdentity
func (options *ReplaceSubnetNetworkACLOptions) SetNetworkACLIdentity(networkACLIdentity NetworkACLIdentityIntf) *ReplaceSubnetNetworkACLOptions {
	options.NetworkACLIdentity = networkACLIdentity
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ReplaceSubnetNetworkACLOptions) SetHeaders(param map[string]string) *ReplaceSubnetNetworkACLOptions {
	options.Headers = param
	return options
}

// ResourceGroupIdentity : The resource group to use. If unspecified, the account's [default resource
// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
// Models which "extend" this model:
// - ResourceGroupIdentityByID
type ResourceGroupIdentity struct {
	// The unique identifier for this resource group.
	ID *string `json:"id,omitempty"`
}

func (*ResourceGroupIdentity) isaResourceGroupIdentity() bool {
	return true
}

type ResourceGroupIdentityIntf interface {
	isaResourceGroupIdentity() bool
}

// UnmarshalResourceGroupIdentity unmarshals an instance of ResourceGroupIdentity from the specified map of raw messages.
func UnmarshalResourceGroupIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ResourceGroupIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ResourceGroupReference : ResourceGroupReference struct
type ResourceGroupReference struct {
	// The URL for this resource group.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this resource group.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalResourceGroupReference unmarshals an instance of ResourceGroupReference from the specified map of raw messages.
func UnmarshalResourceGroupReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ResourceGroupReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Route : Route struct
type Route struct {
	// The date and time that the route was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The destination of the route.
	Destination *string `json:"destination" validate:"required"`

	// The URL for this route.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this route.
	ID *string `json:"id" validate:"required"`

	// The lifecycle state of the route.
	LifecycleState *string `json:"lifecycle_state" validate:"required"`

	// The user-defined name for this route.
	Name *string `json:"name" validate:"required"`

	// The next hop that packets will be delivered to.
	NextHop RouteNextHopIntf `json:"next_hop" validate:"required"`

	// The zone the route applies to. (Traffic from subnets in this zone will be
	// subject to this route.).
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the Route.LifecycleState property.
// The lifecycle state of the route.
const (
	RouteLifecycleStateDeletedConst   = "deleted"
	RouteLifecycleStateDeletingConst  = "deleting"
	RouteLifecycleStateFailedConst    = "failed"
	RouteLifecycleStatePendingConst   = "pending"
	RouteLifecycleStateStableConst    = "stable"
	RouteLifecycleStateSuspendedConst = "suspended"
	RouteLifecycleStateUpdatingConst  = "updating"
	RouteLifecycleStateWaitingConst   = "waiting"
)

// UnmarshalRoute unmarshals an instance of Route from the specified map of raw messages.
func UnmarshalRoute(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Route)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "destination", &obj.Destination)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "lifecycle_state", &obj.LifecycleState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next_hop", &obj.NextHop, UnmarshalRouteNextHop)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RouteCollection : RouteCollection struct
type RouteCollection struct {
	// Collection of routes.
	Routes []Route `json:"routes" validate:"required"`
}

// UnmarshalRouteCollection unmarshals an instance of RouteCollection from the specified map of raw messages.
func UnmarshalRouteCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(RouteCollection)
	err = core.UnmarshalModel(m, "routes", &obj.Routes, UnmarshalRoute)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RouteNextHop : RouteNextHop struct
// Models which "extend" this model:
// - RouteNextHopIP
type RouteNextHop struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`
}

func (*RouteNextHop) isaRouteNextHop() bool {
	return true
}

type RouteNextHopIntf interface {
	isaRouteNextHop() bool
}

// UnmarshalRouteNextHop unmarshals an instance of RouteNextHop from the specified map of raw messages.
func UnmarshalRouteNextHop(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(RouteNextHop)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RouteNextHopPrototype : The next hop packets will be routed to.
// Models which "extend" this model:
// - RouteNextHopPrototypeRouteNextHopIP
type RouteNextHopPrototype struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`
}

func (*RouteNextHopPrototype) isaRouteNextHopPrototype() bool {
	return true
}

type RouteNextHopPrototypeIntf interface {
	isaRouteNextHopPrototype() bool
}

// UnmarshalRouteNextHopPrototype unmarshals an instance of RouteNextHopPrototype from the specified map of raw messages.
func UnmarshalRouteNextHopPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(RouteNextHopPrototype)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// RoutePatch : RoutePatch struct
type RoutePatch struct {
	// The user-defined name for this route. Names must be unique within the VPC routing table the route resides in.
	Name *string `json:"name,omitempty"`
}

// UnmarshalRoutePatch unmarshals an instance of RoutePatch from the specified map of raw messages.
func UnmarshalRoutePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(RoutePatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the RoutePatch
func (routePatch *RoutePatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(routePatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// SecurityGroup : SecurityGroup struct
type SecurityGroup struct {
	// The date and time that this security group was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The security group's CRN.
	CRN *string `json:"crn" validate:"required"`

	// The security group's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this security group.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this security group. Security group names must be unique, within the scope of an account.
	Name *string `json:"name" validate:"required"`

	// Array of references to network interfaces.
	NetworkInterfaces []NetworkInterfaceReference `json:"network_interfaces" validate:"required"`

	// The resource group for this security group.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// Array of rules for this security group. If no rules exist, all traffic will be denied.
	Rules []SecurityGroupRuleIntf `json:"rules" validate:"required"`

	// The VPC this security group is a part of.
	VPC *VPCReference `json:"vpc" validate:"required"`
}

// UnmarshalSecurityGroup unmarshals an instance of SecurityGroup from the specified map of raw messages.
func UnmarshalSecurityGroup(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroup)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_interfaces", &obj.NetworkInterfaces, UnmarshalNetworkInterfaceReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupCollection : SecurityGroupCollection struct
type SecurityGroupCollection struct {
	// Collection of security groups.
	SecurityGroups []SecurityGroup `json:"security_groups" validate:"required"`
}

// UnmarshalSecurityGroupCollection unmarshals an instance of SecurityGroupCollection from the specified map of raw messages.
func UnmarshalSecurityGroupCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupCollection)
	err = core.UnmarshalModel(m, "security_groups", &obj.SecurityGroups, UnmarshalSecurityGroup)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupIdentity : Identifies a security group by a unique property.
// Models which "extend" this model:
// - SecurityGroupIdentityByID
// - SecurityGroupIdentityByCRN
// - SecurityGroupIdentityByHref
type SecurityGroupIdentity struct {
	// The unique identifier for this security group.
	ID *string `json:"id,omitempty"`

	// The security group's CRN.
	CRN *string `json:"crn,omitempty"`

	// The security group's canonical URL.
	Href *string `json:"href,omitempty"`
}

func (*SecurityGroupIdentity) isaSecurityGroupIdentity() bool {
	return true
}

type SecurityGroupIdentityIntf interface {
	isaSecurityGroupIdentity() bool
}

// UnmarshalSecurityGroupIdentity unmarshals an instance of SecurityGroupIdentity from the specified map of raw messages.
func UnmarshalSecurityGroupIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupPatch : SecurityGroupPatch struct
type SecurityGroupPatch struct {
	// The user-defined name for this security group. Security group names must be unique, within the scope of an account.
	Name *string `json:"name,omitempty"`
}

// UnmarshalSecurityGroupPatch unmarshals an instance of SecurityGroupPatch from the specified map of raw messages.
func UnmarshalSecurityGroupPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the SecurityGroupPatch
func (securityGroupPatch *SecurityGroupPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(securityGroupPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// SecurityGroupReference : SecurityGroupReference struct
type SecurityGroupReference struct {
	// The security group's CRN.
	CRN *string `json:"crn" validate:"required"`

	// The security group's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this security group.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this security group. Security group names must be unique, within the scope of an account.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalSecurityGroupReference unmarshals an instance of SecurityGroupReference from the specified map of raw messages.
func UnmarshalSecurityGroupReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupRule : SecurityGroupRule struct
// Models which "extend" this model:
// - SecurityGroupRuleSecurityGroupRuleProtocolAll
// - SecurityGroupRuleSecurityGroupRuleProtocolIcmp
// - SecurityGroupRuleSecurityGroupRuleProtocolTcpudp
type SecurityGroupRule struct {
	// The direction of traffic to enforce, either `inbound` or `outbound`.
	Direction *string `json:"direction" validate:"required"`

	// The unique identifier for this security group rule.
	ID *string `json:"id" validate:"required"`

	// The IP version to enforce. The format of `remote.address` or `remote.cidr_block` must match this field, if they are
	// used. Alternatively, if `remote` references a security group, then this rule only applies to IP addresses (network
	// interfaces) in that group matching this IP version.
	IPVersion *string `json:"ip_version,omitempty"`

	// The protocol to enforce.
	Protocol *string `json:"protocol,omitempty"`

	// The IP addresses or security groups from which this rule allows traffic (or to
	// which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
	// security group. If omitted, then traffic is allowed from any source (or to any
	// source, for outbound rules).
	Remote SecurityGroupRuleRemoteIntf `json:"remote,omitempty"`

	// The ICMP traffic code to allow.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow.
	Type *int64 `json:"type,omitempty"`

	// The inclusive upper bound of TCP/UDP port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP port range.
	PortMin *int64 `json:"port_min,omitempty"`
}

// Constants associated with the SecurityGroupRule.Direction property.
// The direction of traffic to enforce, either `inbound` or `outbound`.
const (
	SecurityGroupRuleDirectionInboundConst  = "inbound"
	SecurityGroupRuleDirectionOutboundConst = "outbound"
)

// Constants associated with the SecurityGroupRule.IPVersion property.
// The IP version to enforce. The format of `remote.address` or `remote.cidr_block` must match this field, if they are
// used. Alternatively, if `remote` references a security group, then this rule only applies to IP addresses (network
// interfaces) in that group matching this IP version.
const (
	SecurityGroupRuleIPVersionIpv4Const = "ipv4"
)

func (*SecurityGroupRule) isaSecurityGroupRule() bool {
	return true
}

type SecurityGroupRuleIntf interface {
	isaSecurityGroupRule() bool
}

// UnmarshalSecurityGroupRule unmarshals an instance of SecurityGroupRule from the specified map of raw messages.
func UnmarshalSecurityGroupRule(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRuleSecurityGroupRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRuleSecurityGroupRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRuleSecurityGroupRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRuleSecurityGroupRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// SecurityGroupRuleCollection : Collection of rules in a security group.
type SecurityGroupRuleCollection struct {
	// Array of rules.
	Rules []SecurityGroupRuleIntf `json:"rules" validate:"required"`
}

// UnmarshalSecurityGroupRuleCollection unmarshals an instance of SecurityGroupRuleCollection from the specified map of raw messages.
func UnmarshalSecurityGroupRuleCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupRuleCollection)
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupRulePatch : SecurityGroupRulePatch struct
// Models which "extend" this model:
// - SecurityGroupRulePatchSecurityGroupRuleProtocolAll
// - SecurityGroupRulePatchSecurityGroupRuleProtocolIcmp
// - SecurityGroupRulePatchSecurityGroupRuleProtocolTcpudp
type SecurityGroupRulePatch struct {
	// The direction of traffic to enforce, either `inbound` or `outbound`.
	Direction *string `json:"direction,omitempty"`

	// The IP version to enforce. The format of `remote.address` or `remote.cidr_block` must match this field, if they are
	// used. Alternatively, if `remote` references a security group, then this rule only applies to IP addresses (network
	// interfaces) in that group matching this IP version.
	IPVersion *string `json:"ip_version,omitempty"`

	// The protocol to enforce.
	Protocol *string `json:"protocol,omitempty"`

	// The IP addresses or security groups from which this rule will allow traffic (or to
	// which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
	// security group. A CIDR block of `0.0.0.0/0` will allow traffic from any source (or to
	// any source, for outbound rules).
	Remote SecurityGroupRuleRemotePatchIntf `json:"remote,omitempty"`

	// The ICMP traffic code to allow.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow.
	Type *int64 `json:"type,omitempty"`

	// The inclusive upper bound of TCP/UDP port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP port range.
	PortMin *int64 `json:"port_min,omitempty"`
}

// Constants associated with the SecurityGroupRulePatch.Direction property.
// The direction of traffic to enforce, either `inbound` or `outbound`.
const (
	SecurityGroupRulePatchDirectionInboundConst  = "inbound"
	SecurityGroupRulePatchDirectionOutboundConst = "outbound"
)

// Constants associated with the SecurityGroupRulePatch.IPVersion property.
// The IP version to enforce. The format of `remote.address` or `remote.cidr_block` must match this field, if they are
// used. Alternatively, if `remote` references a security group, then this rule only applies to IP addresses (network
// interfaces) in that group matching this IP version.
const (
	SecurityGroupRulePatchIPVersionIpv4Const = "ipv4"
)

func (*SecurityGroupRulePatch) isaSecurityGroupRulePatch() bool {
	return true
}

type SecurityGroupRulePatchIntf interface {
	isaSecurityGroupRulePatch() bool
}

// UnmarshalSecurityGroupRulePatch unmarshals an instance of SecurityGroupRulePatch from the specified map of raw messages.
func UnmarshalSecurityGroupRulePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePatchSecurityGroupRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePatchSecurityGroupRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePatchSecurityGroupRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePatchSecurityGroupRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// AsPatch returns a generic map representation of the SecurityGroupRulePatch
func (securityGroupRulePatch *SecurityGroupRulePatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(securityGroupRulePatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// SecurityGroupRulePrototype : SecurityGroupRulePrototype struct
// Models which "extend" this model:
// - SecurityGroupRulePrototypeSecurityGroupRuleProtocolAll
// - SecurityGroupRulePrototypeSecurityGroupRuleProtocolIcmp
// - SecurityGroupRulePrototypeSecurityGroupRuleProtocolTcpudp
type SecurityGroupRulePrototype struct {
	// The direction of traffic to enforce, either `inbound` or `outbound`.
	Direction *string `json:"direction" validate:"required"`

	// The IP version to enforce. The format of `remote.address` or `remote.cidr_block` must match this field, if they are
	// used. Alternatively, if `remote` references a security group, then this rule only applies to IP addresses (network
	// interfaces) in that group matching this IP version.
	IPVersion *string `json:"ip_version,omitempty"`

	// The protocol to enforce.
	Protocol *string `json:"protocol,omitempty"`

	// The IP addresses or security groups from which this rule will allow traffic (or to
	// which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
	// security group. If omitted, then traffic will be allowed from any source (or to any
	// source, for outbound rules).
	Remote SecurityGroupRuleRemotePrototypeIntf `json:"remote,omitempty"`

	// The ICMP traffic code to allow.
	Code *int64 `json:"code,omitempty"`

	// The ICMP traffic type to allow.
	Type *int64 `json:"type,omitempty"`

	// The inclusive upper bound of TCP/UDP port range.
	PortMax *int64 `json:"port_max,omitempty"`

	// The inclusive lower bound of TCP/UDP port range.
	PortMin *int64 `json:"port_min,omitempty"`
}

// Constants associated with the SecurityGroupRulePrototype.Direction property.
// The direction of traffic to enforce, either `inbound` or `outbound`.
const (
	SecurityGroupRulePrototypeDirectionInboundConst  = "inbound"
	SecurityGroupRulePrototypeDirectionOutboundConst = "outbound"
)

// Constants associated with the SecurityGroupRulePrototype.IPVersion property.
// The IP version to enforce. The format of `remote.address` or `remote.cidr_block` must match this field, if they are
// used. Alternatively, if `remote` references a security group, then this rule only applies to IP addresses (network
// interfaces) in that group matching this IP version.
const (
	SecurityGroupRulePrototypeIPVersionIpv4Const = "ipv4"
)

func (*SecurityGroupRulePrototype) isaSecurityGroupRulePrototype() bool {
	return true
}

type SecurityGroupRulePrototypeIntf interface {
	isaSecurityGroupRulePrototype() bool
}

// UnmarshalSecurityGroupRulePrototype unmarshals an instance of SecurityGroupRulePrototype from the specified map of raw messages.
func UnmarshalSecurityGroupRulePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "protocol", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'protocol': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'protocol' not found in JSON object")
		return
	}
	if discValue == "all" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePrototypeSecurityGroupRuleProtocolAll)
	} else if discValue == "icmp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePrototypeSecurityGroupRuleProtocolIcmp)
	} else if discValue == "tcp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePrototypeSecurityGroupRuleProtocolTcpudp)
	} else if discValue == "udp" {
		err = core.UnmarshalModel(m, "", result, UnmarshalSecurityGroupRulePrototypeSecurityGroupRuleProtocolTcpudp)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'protocol': %s", discValue)
	}
	return
}

// SecurityGroupRuleRemote : The IP addresses or security groups from which this rule allows traffic (or to which, for outbound rules). Can be
// specified as an IP address, a CIDR block, or a security group. If omitted, then traffic is allowed from any source
// (or to any source, for outbound rules).
// Models which "extend" this model:
// - SecurityGroupRuleRemoteIP
// - SecurityGroupRuleRemoteCIDR
// - SecurityGroupRuleRemoteSecurityGroupReference
type SecurityGroupRuleRemote struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`

	// The CIDR block. This property may add support for IPv6 CIDR blocks in the future. When processing a value in this
	// property, verify that the CIDR block is in an expected format. If it is not, log an error. Optionally halt
	// processing and surface the error, or bypass the resource on which the unexpected CIDR block format was encountered.
	CIDRBlock *string `json:"cidr_block,omitempty"`

	// The security group's CRN.
	CRN *string `json:"crn,omitempty"`

	// The security group's canonical URL.
	Href *string `json:"href,omitempty"`

	// The unique identifier for this security group.
	ID *string `json:"id,omitempty"`

	// The user-defined name for this security group. Security group names must be unique, within the scope of an account.
	Name *string `json:"name,omitempty"`
}

func (*SecurityGroupRuleRemote) isaSecurityGroupRuleRemote() bool {
	return true
}

type SecurityGroupRuleRemoteIntf interface {
	isaSecurityGroupRuleRemote() bool
}

// UnmarshalSecurityGroupRuleRemote unmarshals an instance of SecurityGroupRuleRemote from the specified map of raw messages.
func UnmarshalSecurityGroupRuleRemote(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupRuleRemote)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cidr_block", &obj.CIDRBlock)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupRuleRemotePatch : The IP addresses or security groups from which this rule will allow traffic (or to which, for outbound rules). Can be
// specified as an IP address, a CIDR block, or a security group. A CIDR block of `0.0.0.0/0` will allow traffic from
// any source (or to any source, for outbound rules).
// Models which "extend" this model:
// - SecurityGroupRuleRemotePatchIP
// - SecurityGroupRuleRemotePatchCIDR
// - SecurityGroupRuleRemotePatchSecurityGroupIdentity
type SecurityGroupRuleRemotePatch struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`

	// The CIDR block. This property may add support for IPv6 CIDR blocks in the future. When processing a value in this
	// property, verify that the CIDR block is in an expected format. If it is not, log an error. Optionally halt
	// processing and surface the error, or bypass the resource on which the unexpected CIDR block format was encountered.
	CIDRBlock *string `json:"cidr_block,omitempty"`

	// The unique identifier for this security group.
	ID *string `json:"id,omitempty"`

	// The security group's CRN.
	CRN *string `json:"crn,omitempty"`

	// The security group's canonical URL.
	Href *string `json:"href,omitempty"`
}

func (*SecurityGroupRuleRemotePatch) isaSecurityGroupRuleRemotePatch() bool {
	return true
}

type SecurityGroupRuleRemotePatchIntf interface {
	isaSecurityGroupRuleRemotePatch() bool
}

// UnmarshalSecurityGroupRuleRemotePatch unmarshals an instance of SecurityGroupRuleRemotePatch from the specified map of raw messages.
func UnmarshalSecurityGroupRuleRemotePatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupRuleRemotePatch)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cidr_block", &obj.CIDRBlock)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SecurityGroupRuleRemotePrototype : The IP addresses or security groups from which this rule will allow traffic (or to which, for outbound rules). Can be
// specified as an IP address, a CIDR block, or a security group. If omitted, then traffic will be allowed from any
// source (or to any source, for outbound rules).
// Models which "extend" this model:
// - SecurityGroupRuleRemotePrototypeIP
// - SecurityGroupRuleRemotePrototypeCIDR
// - SecurityGroupRuleRemotePrototypeSecurityGroupIdentity
type SecurityGroupRuleRemotePrototype struct {
	// The IP address. This property may add support for IPv6 addresses in the future. When processing a value in this
	// property, verify that the address is in an expected format. If it is not, log an error. Optionally halt processing
	// and surface the error, or bypass the resource on which the unexpected IP address format was encountered.
	Address *string `json:"address,omitempty"`

	// The CIDR block. This property may add support for IPv6 CIDR blocks in the future. When processing a value in this
	// property, verify that the CIDR block is in an expected format. If it is not, log an error. Optionally halt
	// processing and surface the error, or bypass the resource on which the unexpected CIDR block format was encountered.
	CIDRBlock *string `json:"cidr_block,omitempty"`

	// The unique identifier for this security group.
	ID *string `json:"id,omitempty"`

	// The security group's CRN.
	CRN *string `json:"crn,omitempty"`

	// The security group's canonical URL.
	Href *string `json:"href,omitempty"`
}

func (*SecurityGroupRuleRemotePrototype) isaSecurityGroupRuleRemotePrototype() bool {
	return true
}

type SecurityGroupRuleRemotePrototypeIntf interface {
	isaSecurityGroupRuleRemotePrototype() bool
}

// UnmarshalSecurityGroupRuleRemotePrototype unmarshals an instance of SecurityGroupRuleRemotePrototype from the specified map of raw messages.
func UnmarshalSecurityGroupRuleRemotePrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SecurityGroupRuleRemotePrototype)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cidr_block", &obj.CIDRBlock)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SetSubnetPublicGatewayOptions : The SetSubnetPublicGateway options.
type SetSubnetPublicGatewayOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// The public gateway identity.
	PublicGatewayIdentity PublicGatewayIdentityIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewSetSubnetPublicGatewayOptions : Instantiate SetSubnetPublicGatewayOptions
func (*VpcClassicV1) NewSetSubnetPublicGatewayOptions(id string, publicGatewayIdentity PublicGatewayIdentityIntf) *SetSubnetPublicGatewayOptions {
	return &SetSubnetPublicGatewayOptions{
		ID:                    core.StringPtr(id),
		PublicGatewayIdentity: publicGatewayIdentity,
	}
}

// SetID : Allow user to set ID
func (options *SetSubnetPublicGatewayOptions) SetID(id string) *SetSubnetPublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetPublicGatewayIdentity : Allow user to set PublicGatewayIdentity
func (options *SetSubnetPublicGatewayOptions) SetPublicGatewayIdentity(publicGatewayIdentity PublicGatewayIdentityIntf) *SetSubnetPublicGatewayOptions {
	options.PublicGatewayIdentity = publicGatewayIdentity
	return options
}

// SetHeaders : Allow user to set Headers
func (options *SetSubnetPublicGatewayOptions) SetHeaders(param map[string]string) *SetSubnetPublicGatewayOptions {
	options.Headers = param
	return options
}

// Subnet : Subnet struct
type Subnet struct {
	// The number of IPv4 addresses in this subnet that are not in-use, and have not been reserved by the user or the
	// provider.
	AvailableIpv4AddressCount *int64 `json:"available_ipv4_address_count" validate:"required"`

	// The date and time that the subnet was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this subnet.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this subnet.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this subnet.
	ID *string `json:"id" validate:"required"`

	// The IPv4 range of the subnet, expressed in CIDR format.
	Ipv4CIDRBlock *string `json:"ipv4_cidr_block" validate:"required"`

	// The user-defined name for this subnet.
	Name *string `json:"name" validate:"required"`

	// The network ACL for this subnet.
	NetworkACL *NetworkACLReference `json:"network_acl" validate:"required"`

	// The public gateway to handle internet bound traffic for this subnet.
	PublicGateway *PublicGatewayReference `json:"public_gateway,omitempty"`

	// The status of the subnet.
	Status *string `json:"status" validate:"required"`

	// The total number of IPv4 addresses in this subnet.
	//
	// Note: This is calculated as 2<sup>(32 − prefix length)</sup>. For example, the prefix length `/24` gives:<br>
	// 2<sup>(32 − 24)</sup> = 2<sup>8</sup> = 256 addresses.
	TotalIpv4AddressCount *int64 `json:"total_ipv4_address_count" validate:"required"`

	// The VPC this subnet is a part of.
	VPC *VPCReference `json:"vpc" validate:"required"`

	// The zone this subnet resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the Subnet.Status property.
// The status of the subnet.
const (
	SubnetStatusAvailableConst = "available"
	SubnetStatusDeletingConst  = "deleting"
	SubnetStatusFailedConst    = "failed"
	SubnetStatusPendingConst   = "pending"
)

// UnmarshalSubnet unmarshals an instance of Subnet from the specified map of raw messages.
func UnmarshalSubnet(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Subnet)
	err = core.UnmarshalPrimitive(m, "available_ipv4_address_count", &obj.AvailableIpv4AddressCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "ipv4_cidr_block", &obj.Ipv4CIDRBlock)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_acl", &obj.NetworkACL, UnmarshalNetworkACLReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "public_gateway", &obj.PublicGateway, UnmarshalPublicGatewayReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_ipv4_address_count", &obj.TotalIpv4AddressCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SubnetCollection : SubnetCollection struct
type SubnetCollection struct {
	// A link to the first page of resources.
	First *SubnetCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *SubnetCollectionNext `json:"next,omitempty"`

	// Collection of subnets.
	Subnets []Subnet `json:"subnets" validate:"required"`
}

// UnmarshalSubnetCollection unmarshals an instance of SubnetCollection from the specified map of raw messages.
func UnmarshalSubnetCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalSubnetCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalSubnetCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnets", &obj.Subnets, UnmarshalSubnet)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SubnetCollectionFirst : A link to the first page of resources.
type SubnetCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalSubnetCollectionFirst unmarshals an instance of SubnetCollectionFirst from the specified map of raw messages.
func UnmarshalSubnetCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SubnetCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type SubnetCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalSubnetCollectionNext unmarshals an instance of SubnetCollectionNext from the specified map of raw messages.
func UnmarshalSubnetCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SubnetIdentity : Identifies a subnet by a unique property.
// Models which "extend" this model:
// - SubnetIdentityByID
// - SubnetIdentityByCRN
// - SubnetIdentityByHref
type SubnetIdentity struct {
	// The unique identifier for this subnet.
	ID *string `json:"id,omitempty"`

	// The CRN for this subnet.
	CRN *string `json:"crn,omitempty"`

	// The URL for this subnet.
	Href *string `json:"href,omitempty"`
}

func (*SubnetIdentity) isaSubnetIdentity() bool {
	return true
}

type SubnetIdentityIntf interface {
	isaSubnetIdentity() bool
}

// UnmarshalSubnetIdentity unmarshals an instance of SubnetIdentity from the specified map of raw messages.
func UnmarshalSubnetIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SubnetPatch : SubnetPatch struct
type SubnetPatch struct {
	// The user-defined name for this subnet. Names must be unique within the VPC the subnet resides in.
	Name *string `json:"name,omitempty"`

	// The network ACL to use for this subnet.
	NetworkACL NetworkACLIdentityIntf `json:"network_acl,omitempty"`

	// The public gateway to handle internet bound traffic for this subnet.
	PublicGateway PublicGatewayIdentityIntf `json:"public_gateway,omitempty"`
}

// UnmarshalSubnetPatch unmarshals an instance of SubnetPatch from the specified map of raw messages.
func UnmarshalSubnetPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_acl", &obj.NetworkACL, UnmarshalNetworkACLIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "public_gateway", &obj.PublicGateway, UnmarshalPublicGatewayIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the SubnetPatch
func (subnetPatch *SubnetPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(subnetPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// SubnetPrototype : SubnetPrototype struct
// Models which "extend" this model:
// - SubnetPrototypeSubnetByTotalCount
// - SubnetPrototypeSubnetByCIDR
type SubnetPrototype struct {
	// The user-defined name for this subnet. Names must be unique within the VPC the subnet resides in. If unspecified,
	// the name will be a hyphenated list of randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The network ACL to use for this subnet.
	NetworkACL NetworkACLIdentityIntf `json:"network_acl,omitempty"`

	// The public gateway to handle internet bound traffic for this subnet.
	PublicGateway PublicGatewayIdentityIntf `json:"public_gateway,omitempty"`

	// The VPC the subnet is to be a part of.
	VPC VPCIdentityIntf `json:"vpc" validate:"required"`

	// The total number of IPv4 addresses required. Must be a power of 2. The VPC must have a default address prefix in the
	// specified zone, and that prefix must have a free CIDR range with at least this number of addresses.
	TotalIpv4AddressCount *int64 `json:"total_ipv4_address_count,omitempty"`

	// The zone the subnet is to reside in.
	Zone ZoneIdentityIntf `json:"zone,omitempty"`

	// The IPv4 range of the subnet, expressed in CIDR format. The prefix length of the subnet's CIDR must be between `/8`
	// (16,777,216 addresses) and `/29` (8 addresses). The IPv4 range of the subnet's CIDR must fall within an existing
	// address prefix in the VPC. The subnet will be created in the zone of the address prefix that contains the IPv4 CIDR.
	// If zone is specified, it must match the zone of the address prefix that contains the subnet's IPv4 CIDR.
	Ipv4CIDRBlock *string `json:"ipv4_cidr_block,omitempty"`
}

func (*SubnetPrototype) isaSubnetPrototype() bool {
	return true
}

type SubnetPrototypeIntf interface {
	isaSubnetPrototype() bool
}

// UnmarshalSubnetPrototype unmarshals an instance of SubnetPrototype from the specified map of raw messages.
func UnmarshalSubnetPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetPrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "network_acl", &obj.NetworkACL, UnmarshalNetworkACLIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "public_gateway", &obj.PublicGateway, UnmarshalPublicGatewayIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_ipv4_address_count", &obj.TotalIpv4AddressCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "ipv4_cidr_block", &obj.Ipv4CIDRBlock)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// SubnetReference : SubnetReference struct
type SubnetReference struct {
	// The CRN for this subnet.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this subnet.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this subnet.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this subnet.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalSubnetReference unmarshals an instance of SubnetReference from the specified map of raw messages.
func UnmarshalSubnetReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(SubnetReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// UnsetSubnetPublicGatewayOptions : The UnsetSubnetPublicGateway options.
type UnsetSubnetPublicGatewayOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUnsetSubnetPublicGatewayOptions : Instantiate UnsetSubnetPublicGatewayOptions
func (*VpcClassicV1) NewUnsetSubnetPublicGatewayOptions(id string) *UnsetSubnetPublicGatewayOptions {
	return &UnsetSubnetPublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *UnsetSubnetPublicGatewayOptions) SetID(id string) *UnsetSubnetPublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UnsetSubnetPublicGatewayOptions) SetHeaders(param map[string]string) *UnsetSubnetPublicGatewayOptions {
	options.Headers = param
	return options
}

// UpdateFloatingIPOptions : The UpdateFloatingIP options.
type UpdateFloatingIPOptions struct {
	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// The floating IP patch.
	FloatingIPPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateFloatingIPOptions : Instantiate UpdateFloatingIPOptions
func (*VpcClassicV1) NewUpdateFloatingIPOptions(id string, floatingIPPatch map[string]interface{}) *UpdateFloatingIPOptions {
	return &UpdateFloatingIPOptions{
		ID:              core.StringPtr(id),
		FloatingIPPatch: floatingIPPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateFloatingIPOptions) SetID(id string) *UpdateFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetFloatingIPPatch : Allow user to set FloatingIPPatch
func (options *UpdateFloatingIPOptions) SetFloatingIPPatch(floatingIPPatch map[string]interface{}) *UpdateFloatingIPOptions {
	options.FloatingIPPatch = floatingIPPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateFloatingIPOptions) SetHeaders(param map[string]string) *UpdateFloatingIPOptions {
	options.Headers = param
	return options
}

// UpdateIkePolicyOptions : The UpdateIkePolicy options.
type UpdateIkePolicyOptions struct {
	// The IKE policy identifier.
	ID *string `validate:"required,ne="`

	// The IKE policy patch.
	IkePolicyPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateIkePolicyOptions : Instantiate UpdateIkePolicyOptions
func (*VpcClassicV1) NewUpdateIkePolicyOptions(id string, ikePolicyPatch map[string]interface{}) *UpdateIkePolicyOptions {
	return &UpdateIkePolicyOptions{
		ID:             core.StringPtr(id),
		IkePolicyPatch: ikePolicyPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateIkePolicyOptions) SetID(id string) *UpdateIkePolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetIkePolicyPatch : Allow user to set IkePolicyPatch
func (options *UpdateIkePolicyOptions) SetIkePolicyPatch(ikePolicyPatch map[string]interface{}) *UpdateIkePolicyOptions {
	options.IkePolicyPatch = ikePolicyPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateIkePolicyOptions) SetHeaders(param map[string]string) *UpdateIkePolicyOptions {
	options.Headers = param
	return options
}

// UpdateImageOptions : The UpdateImage options.
type UpdateImageOptions struct {
	// The image identifier.
	ID *string `validate:"required,ne="`

	// The image patch.
	ImagePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateImageOptions : Instantiate UpdateImageOptions
func (*VpcClassicV1) NewUpdateImageOptions(id string, imagePatch map[string]interface{}) *UpdateImageOptions {
	return &UpdateImageOptions{
		ID:         core.StringPtr(id),
		ImagePatch: imagePatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateImageOptions) SetID(id string) *UpdateImageOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetImagePatch : Allow user to set ImagePatch
func (options *UpdateImageOptions) SetImagePatch(imagePatch map[string]interface{}) *UpdateImageOptions {
	options.ImagePatch = imagePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateImageOptions) SetHeaders(param map[string]string) *UpdateImageOptions {
	options.Headers = param
	return options
}

// UpdateInstanceOptions : The UpdateInstance options.
type UpdateInstanceOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// The instance patch.
	InstancePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateInstanceOptions : Instantiate UpdateInstanceOptions
func (*VpcClassicV1) NewUpdateInstanceOptions(id string, instancePatch map[string]interface{}) *UpdateInstanceOptions {
	return &UpdateInstanceOptions{
		ID:            core.StringPtr(id),
		InstancePatch: instancePatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateInstanceOptions) SetID(id string) *UpdateInstanceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetInstancePatch : Allow user to set InstancePatch
func (options *UpdateInstanceOptions) SetInstancePatch(instancePatch map[string]interface{}) *UpdateInstanceOptions {
	options.InstancePatch = instancePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateInstanceOptions) SetHeaders(param map[string]string) *UpdateInstanceOptions {
	options.Headers = param
	return options
}

// UpdateInstanceVolumeAttachmentOptions : The UpdateInstanceVolumeAttachment options.
type UpdateInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The volume attachment identifier.
	ID *string `validate:"required,ne="`

	// The volume attachment patch.
	VolumeAttachmentPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateInstanceVolumeAttachmentOptions : Instantiate UpdateInstanceVolumeAttachmentOptions
func (*VpcClassicV1) NewUpdateInstanceVolumeAttachmentOptions(instanceID string, id string, volumeAttachmentPatch map[string]interface{}) *UpdateInstanceVolumeAttachmentOptions {
	return &UpdateInstanceVolumeAttachmentOptions{
		InstanceID:            core.StringPtr(instanceID),
		ID:                    core.StringPtr(id),
		VolumeAttachmentPatch: volumeAttachmentPatch,
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *UpdateInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *UpdateInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateInstanceVolumeAttachmentOptions) SetID(id string) *UpdateInstanceVolumeAttachmentOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetVolumeAttachmentPatch : Allow user to set VolumeAttachmentPatch
func (options *UpdateInstanceVolumeAttachmentOptions) SetVolumeAttachmentPatch(volumeAttachmentPatch map[string]interface{}) *UpdateInstanceVolumeAttachmentOptions {
	options.VolumeAttachmentPatch = volumeAttachmentPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *UpdateInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// UpdateIpsecPolicyOptions : The UpdateIpsecPolicy options.
type UpdateIpsecPolicyOptions struct {
	// The IPsec policy identifier.
	ID *string `validate:"required,ne="`

	// The IPsec policy patch.
	IPsecPolicyPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateIpsecPolicyOptions : Instantiate UpdateIpsecPolicyOptions
func (*VpcClassicV1) NewUpdateIpsecPolicyOptions(id string, iPsecPolicyPatch map[string]interface{}) *UpdateIpsecPolicyOptions {
	return &UpdateIpsecPolicyOptions{
		ID:               core.StringPtr(id),
		IPsecPolicyPatch: iPsecPolicyPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateIpsecPolicyOptions) SetID(id string) *UpdateIpsecPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetIPsecPolicyPatch : Allow user to set IPsecPolicyPatch
func (options *UpdateIpsecPolicyOptions) SetIPsecPolicyPatch(iPsecPolicyPatch map[string]interface{}) *UpdateIpsecPolicyOptions {
	options.IPsecPolicyPatch = iPsecPolicyPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateIpsecPolicyOptions) SetHeaders(param map[string]string) *UpdateIpsecPolicyOptions {
	options.Headers = param
	return options
}

// UpdateKeyOptions : The UpdateKey options.
type UpdateKeyOptions struct {
	// The key identifier.
	ID *string `validate:"required,ne="`

	// The key patch.
	KeyPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateKeyOptions : Instantiate UpdateKeyOptions
func (*VpcClassicV1) NewUpdateKeyOptions(id string, keyPatch map[string]interface{}) *UpdateKeyOptions {
	return &UpdateKeyOptions{
		ID:       core.StringPtr(id),
		KeyPatch: keyPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateKeyOptions) SetID(id string) *UpdateKeyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetKeyPatch : Allow user to set KeyPatch
func (options *UpdateKeyOptions) SetKeyPatch(keyPatch map[string]interface{}) *UpdateKeyOptions {
	options.KeyPatch = keyPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateKeyOptions) SetHeaders(param map[string]string) *UpdateKeyOptions {
	options.Headers = param
	return options
}

// UpdateLoadBalancerListenerOptions : The UpdateLoadBalancerListener options.
type UpdateLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ID *string `validate:"required,ne="`

	// The load balancer listener patch.
	LoadBalancerListenerPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateLoadBalancerListenerOptions : Instantiate UpdateLoadBalancerListenerOptions
func (*VpcClassicV1) NewUpdateLoadBalancerListenerOptions(loadBalancerID string, id string, loadBalancerListenerPatch map[string]interface{}) *UpdateLoadBalancerListenerOptions {
	return &UpdateLoadBalancerListenerOptions{
		LoadBalancerID:            core.StringPtr(loadBalancerID),
		ID:                        core.StringPtr(id),
		LoadBalancerListenerPatch: loadBalancerListenerPatch,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *UpdateLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *UpdateLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateLoadBalancerListenerOptions) SetID(id string) *UpdateLoadBalancerListenerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetLoadBalancerListenerPatch : Allow user to set LoadBalancerListenerPatch
func (options *UpdateLoadBalancerListenerOptions) SetLoadBalancerListenerPatch(loadBalancerListenerPatch map[string]interface{}) *UpdateLoadBalancerListenerOptions {
	options.LoadBalancerListenerPatch = loadBalancerListenerPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateLoadBalancerListenerOptions) SetHeaders(param map[string]string) *UpdateLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// UpdateLoadBalancerListenerPolicyOptions : The UpdateLoadBalancerListenerPolicy options.
type UpdateLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	ID *string `validate:"required,ne="`

	// The listener policy patch.
	LoadBalancerListenerPolicyPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateLoadBalancerListenerPolicyOptions : Instantiate UpdateLoadBalancerListenerPolicyOptions
func (*VpcClassicV1) NewUpdateLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, id string, loadBalancerListenerPolicyPatch map[string]interface{}) *UpdateLoadBalancerListenerPolicyOptions {
	return &UpdateLoadBalancerListenerPolicyOptions{
		LoadBalancerID:                  core.StringPtr(loadBalancerID),
		ListenerID:                      core.StringPtr(listenerID),
		ID:                              core.StringPtr(id),
		LoadBalancerListenerPolicyPatch: loadBalancerListenerPolicyPatch,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *UpdateLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *UpdateLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *UpdateLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *UpdateLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateLoadBalancerListenerPolicyOptions) SetID(id string) *UpdateLoadBalancerListenerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetLoadBalancerListenerPolicyPatch : Allow user to set LoadBalancerListenerPolicyPatch
func (options *UpdateLoadBalancerListenerPolicyOptions) SetLoadBalancerListenerPolicyPatch(loadBalancerListenerPolicyPatch map[string]interface{}) *UpdateLoadBalancerListenerPolicyOptions {
	options.LoadBalancerListenerPolicyPatch = loadBalancerListenerPolicyPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *UpdateLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// UpdateLoadBalancerListenerPolicyRuleOptions : The UpdateLoadBalancerListenerPolicyRule options.
type UpdateLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// The listener policy rule patch.
	LoadBalancerListenerPolicyRulePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateLoadBalancerListenerPolicyRuleOptions : Instantiate UpdateLoadBalancerListenerPolicyRuleOptions
func (*VpcClassicV1) NewUpdateLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, id string, loadBalancerListenerPolicyRulePatch map[string]interface{}) *UpdateLoadBalancerListenerPolicyRuleOptions {
	return &UpdateLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID:                      core.StringPtr(loadBalancerID),
		ListenerID:                          core.StringPtr(listenerID),
		PolicyID:                            core.StringPtr(policyID),
		ID:                                  core.StringPtr(id),
		LoadBalancerListenerPolicyRulePatch: loadBalancerListenerPolicyRulePatch,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *UpdateLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *UpdateLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *UpdateLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *UpdateLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *UpdateLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *UpdateLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateLoadBalancerListenerPolicyRuleOptions) SetID(id string) *UpdateLoadBalancerListenerPolicyRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetLoadBalancerListenerPolicyRulePatch : Allow user to set LoadBalancerListenerPolicyRulePatch
func (options *UpdateLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerListenerPolicyRulePatch(loadBalancerListenerPolicyRulePatch map[string]interface{}) *UpdateLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerListenerPolicyRulePatch = loadBalancerListenerPolicyRulePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *UpdateLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// UpdateLoadBalancerOptions : The UpdateLoadBalancer options.
type UpdateLoadBalancerOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// The load balancer patch.
	LoadBalancerPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateLoadBalancerOptions : Instantiate UpdateLoadBalancerOptions
func (*VpcClassicV1) NewUpdateLoadBalancerOptions(id string, loadBalancerPatch map[string]interface{}) *UpdateLoadBalancerOptions {
	return &UpdateLoadBalancerOptions{
		ID:                core.StringPtr(id),
		LoadBalancerPatch: loadBalancerPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateLoadBalancerOptions) SetID(id string) *UpdateLoadBalancerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetLoadBalancerPatch : Allow user to set LoadBalancerPatch
func (options *UpdateLoadBalancerOptions) SetLoadBalancerPatch(loadBalancerPatch map[string]interface{}) *UpdateLoadBalancerOptions {
	options.LoadBalancerPatch = loadBalancerPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateLoadBalancerOptions) SetHeaders(param map[string]string) *UpdateLoadBalancerOptions {
	options.Headers = param
	return options
}

// UpdateLoadBalancerPoolMemberOptions : The UpdateLoadBalancerPoolMember options.
type UpdateLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The member identifier.
	ID *string `validate:"required,ne="`

	// The load balancer pool member patch.
	LoadBalancerPoolMemberPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateLoadBalancerPoolMemberOptions : Instantiate UpdateLoadBalancerPoolMemberOptions
func (*VpcClassicV1) NewUpdateLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, id string, loadBalancerPoolMemberPatch map[string]interface{}) *UpdateLoadBalancerPoolMemberOptions {
	return &UpdateLoadBalancerPoolMemberOptions{
		LoadBalancerID:              core.StringPtr(loadBalancerID),
		PoolID:                      core.StringPtr(poolID),
		ID:                          core.StringPtr(id),
		LoadBalancerPoolMemberPatch: loadBalancerPoolMemberPatch,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *UpdateLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *UpdateLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *UpdateLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *UpdateLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateLoadBalancerPoolMemberOptions) SetID(id string) *UpdateLoadBalancerPoolMemberOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetLoadBalancerPoolMemberPatch : Allow user to set LoadBalancerPoolMemberPatch
func (options *UpdateLoadBalancerPoolMemberOptions) SetLoadBalancerPoolMemberPatch(loadBalancerPoolMemberPatch map[string]interface{}) *UpdateLoadBalancerPoolMemberOptions {
	options.LoadBalancerPoolMemberPatch = loadBalancerPoolMemberPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *UpdateLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// UpdateLoadBalancerPoolOptions : The UpdateLoadBalancerPool options.
type UpdateLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	ID *string `validate:"required,ne="`

	// The load balancer pool patch.
	LoadBalancerPoolPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateLoadBalancerPoolOptions : Instantiate UpdateLoadBalancerPoolOptions
func (*VpcClassicV1) NewUpdateLoadBalancerPoolOptions(loadBalancerID string, id string, loadBalancerPoolPatch map[string]interface{}) *UpdateLoadBalancerPoolOptions {
	return &UpdateLoadBalancerPoolOptions{
		LoadBalancerID:        core.StringPtr(loadBalancerID),
		ID:                    core.StringPtr(id),
		LoadBalancerPoolPatch: loadBalancerPoolPatch,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *UpdateLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *UpdateLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateLoadBalancerPoolOptions) SetID(id string) *UpdateLoadBalancerPoolOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetLoadBalancerPoolPatch : Allow user to set LoadBalancerPoolPatch
func (options *UpdateLoadBalancerPoolOptions) SetLoadBalancerPoolPatch(loadBalancerPoolPatch map[string]interface{}) *UpdateLoadBalancerPoolOptions {
	options.LoadBalancerPoolPatch = loadBalancerPoolPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateLoadBalancerPoolOptions) SetHeaders(param map[string]string) *UpdateLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// UpdateNetworkACLOptions : The UpdateNetworkACL options.
type UpdateNetworkACLOptions struct {
	// The network ACL identifier.
	ID *string `validate:"required,ne="`

	// The network ACL patch.
	NetworkACLPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateNetworkACLOptions : Instantiate UpdateNetworkACLOptions
func (*VpcClassicV1) NewUpdateNetworkACLOptions(id string, networkACLPatch map[string]interface{}) *UpdateNetworkACLOptions {
	return &UpdateNetworkACLOptions{
		ID:              core.StringPtr(id),
		NetworkACLPatch: networkACLPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateNetworkACLOptions) SetID(id string) *UpdateNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetNetworkACLPatch : Allow user to set NetworkACLPatch
func (options *UpdateNetworkACLOptions) SetNetworkACLPatch(networkACLPatch map[string]interface{}) *UpdateNetworkACLOptions {
	options.NetworkACLPatch = networkACLPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateNetworkACLOptions) SetHeaders(param map[string]string) *UpdateNetworkACLOptions {
	options.Headers = param
	return options
}

// UpdateNetworkACLRuleOptions : The UpdateNetworkACLRule options.
type UpdateNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// The network ACL rule patch.
	NetworkACLRulePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateNetworkACLRuleOptions : Instantiate UpdateNetworkACLRuleOptions
func (*VpcClassicV1) NewUpdateNetworkACLRuleOptions(networkACLID string, id string, networkACLRulePatch map[string]interface{}) *UpdateNetworkACLRuleOptions {
	return &UpdateNetworkACLRuleOptions{
		NetworkACLID:        core.StringPtr(networkACLID),
		ID:                  core.StringPtr(id),
		NetworkACLRulePatch: networkACLRulePatch,
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *UpdateNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *UpdateNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateNetworkACLRuleOptions) SetID(id string) *UpdateNetworkACLRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetNetworkACLRulePatch : Allow user to set NetworkACLRulePatch
func (options *UpdateNetworkACLRuleOptions) SetNetworkACLRulePatch(networkACLRulePatch map[string]interface{}) *UpdateNetworkACLRuleOptions {
	options.NetworkACLRulePatch = networkACLRulePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateNetworkACLRuleOptions) SetHeaders(param map[string]string) *UpdateNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// UpdatePublicGatewayOptions : The UpdatePublicGateway options.
type UpdatePublicGatewayOptions struct {
	// The public gateway identifier.
	ID *string `validate:"required,ne="`

	// The public gateway patch.
	PublicGatewayPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdatePublicGatewayOptions : Instantiate UpdatePublicGatewayOptions
func (*VpcClassicV1) NewUpdatePublicGatewayOptions(id string, publicGatewayPatch map[string]interface{}) *UpdatePublicGatewayOptions {
	return &UpdatePublicGatewayOptions{
		ID:                 core.StringPtr(id),
		PublicGatewayPatch: publicGatewayPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdatePublicGatewayOptions) SetID(id string) *UpdatePublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetPublicGatewayPatch : Allow user to set PublicGatewayPatch
func (options *UpdatePublicGatewayOptions) SetPublicGatewayPatch(publicGatewayPatch map[string]interface{}) *UpdatePublicGatewayOptions {
	options.PublicGatewayPatch = publicGatewayPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdatePublicGatewayOptions) SetHeaders(param map[string]string) *UpdatePublicGatewayOptions {
	options.Headers = param
	return options
}

// UpdateSecurityGroupOptions : The UpdateSecurityGroup options.
type UpdateSecurityGroupOptions struct {
	// The security group identifier.
	ID *string `validate:"required,ne="`

	// The security group patch.
	SecurityGroupPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateSecurityGroupOptions : Instantiate UpdateSecurityGroupOptions
func (*VpcClassicV1) NewUpdateSecurityGroupOptions(id string, securityGroupPatch map[string]interface{}) *UpdateSecurityGroupOptions {
	return &UpdateSecurityGroupOptions{
		ID:                 core.StringPtr(id),
		SecurityGroupPatch: securityGroupPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateSecurityGroupOptions) SetID(id string) *UpdateSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetSecurityGroupPatch : Allow user to set SecurityGroupPatch
func (options *UpdateSecurityGroupOptions) SetSecurityGroupPatch(securityGroupPatch map[string]interface{}) *UpdateSecurityGroupOptions {
	options.SecurityGroupPatch = securityGroupPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateSecurityGroupOptions) SetHeaders(param map[string]string) *UpdateSecurityGroupOptions {
	options.Headers = param
	return options
}

// UpdateSecurityGroupRuleOptions : The UpdateSecurityGroupRule options.
type UpdateSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// The security group rule patch.
	SecurityGroupRulePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateSecurityGroupRuleOptions : Instantiate UpdateSecurityGroupRuleOptions
func (*VpcClassicV1) NewUpdateSecurityGroupRuleOptions(securityGroupID string, id string, securityGroupRulePatch map[string]interface{}) *UpdateSecurityGroupRuleOptions {
	return &UpdateSecurityGroupRuleOptions{
		SecurityGroupID:        core.StringPtr(securityGroupID),
		ID:                     core.StringPtr(id),
		SecurityGroupRulePatch: securityGroupRulePatch,
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *UpdateSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *UpdateSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateSecurityGroupRuleOptions) SetID(id string) *UpdateSecurityGroupRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetSecurityGroupRulePatch : Allow user to set SecurityGroupRulePatch
func (options *UpdateSecurityGroupRuleOptions) SetSecurityGroupRulePatch(securityGroupRulePatch map[string]interface{}) *UpdateSecurityGroupRuleOptions {
	options.SecurityGroupRulePatch = securityGroupRulePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateSecurityGroupRuleOptions) SetHeaders(param map[string]string) *UpdateSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// UpdateSubnetOptions : The UpdateSubnet options.
type UpdateSubnetOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// The subnet patch.
	SubnetPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateSubnetOptions : Instantiate UpdateSubnetOptions
func (*VpcClassicV1) NewUpdateSubnetOptions(id string, subnetPatch map[string]interface{}) *UpdateSubnetOptions {
	return &UpdateSubnetOptions{
		ID:          core.StringPtr(id),
		SubnetPatch: subnetPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateSubnetOptions) SetID(id string) *UpdateSubnetOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetSubnetPatch : Allow user to set SubnetPatch
func (options *UpdateSubnetOptions) SetSubnetPatch(subnetPatch map[string]interface{}) *UpdateSubnetOptions {
	options.SubnetPatch = subnetPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateSubnetOptions) SetHeaders(param map[string]string) *UpdateSubnetOptions {
	options.Headers = param
	return options
}

// UpdateVolumeOptions : The UpdateVolume options.
type UpdateVolumeOptions struct {
	// The volume identifier.
	ID *string `validate:"required,ne="`

	// The volume patch.
	VolumePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateVolumeOptions : Instantiate UpdateVolumeOptions
func (*VpcClassicV1) NewUpdateVolumeOptions(id string, volumePatch map[string]interface{}) *UpdateVolumeOptions {
	return &UpdateVolumeOptions{
		ID:          core.StringPtr(id),
		VolumePatch: volumePatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateVolumeOptions) SetID(id string) *UpdateVolumeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetVolumePatch : Allow user to set VolumePatch
func (options *UpdateVolumeOptions) SetVolumePatch(volumePatch map[string]interface{}) *UpdateVolumeOptions {
	options.VolumePatch = volumePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateVolumeOptions) SetHeaders(param map[string]string) *UpdateVolumeOptions {
	options.Headers = param
	return options
}

// UpdateVPCAddressPrefixOptions : The UpdateVPCAddressPrefix options.
type UpdateVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The prefix identifier.
	ID *string `validate:"required,ne="`

	// The prefix patch.
	AddressPrefixPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateVPCAddressPrefixOptions : Instantiate UpdateVPCAddressPrefixOptions
func (*VpcClassicV1) NewUpdateVPCAddressPrefixOptions(vpcID string, id string, addressPrefixPatch map[string]interface{}) *UpdateVPCAddressPrefixOptions {
	return &UpdateVPCAddressPrefixOptions{
		VPCID:              core.StringPtr(vpcID),
		ID:                 core.StringPtr(id),
		AddressPrefixPatch: addressPrefixPatch,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *UpdateVPCAddressPrefixOptions) SetVPCID(vpcID string) *UpdateVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateVPCAddressPrefixOptions) SetID(id string) *UpdateVPCAddressPrefixOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetAddressPrefixPatch : Allow user to set AddressPrefixPatch
func (options *UpdateVPCAddressPrefixOptions) SetAddressPrefixPatch(addressPrefixPatch map[string]interface{}) *UpdateVPCAddressPrefixOptions {
	options.AddressPrefixPatch = addressPrefixPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateVPCAddressPrefixOptions) SetHeaders(param map[string]string) *UpdateVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// UpdateVPCOptions : The UpdateVPC options.
type UpdateVPCOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// The VPC patch.
	VPCPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateVPCOptions : Instantiate UpdateVPCOptions
func (*VpcClassicV1) NewUpdateVPCOptions(id string, vpcPatch map[string]interface{}) *UpdateVPCOptions {
	return &UpdateVPCOptions{
		ID:       core.StringPtr(id),
		VPCPatch: vpcPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateVPCOptions) SetID(id string) *UpdateVPCOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetVPCPatch : Allow user to set VPCPatch
func (options *UpdateVPCOptions) SetVPCPatch(vpcPatch map[string]interface{}) *UpdateVPCOptions {
	options.VPCPatch = vpcPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateVPCOptions) SetHeaders(param map[string]string) *UpdateVPCOptions {
	options.Headers = param
	return options
}

// UpdateVPCRouteOptions : The UpdateVPCRoute options.
type UpdateVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The route identifier.
	ID *string `validate:"required,ne="`

	// The route patch.
	RoutePatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateVPCRouteOptions : Instantiate UpdateVPCRouteOptions
func (*VpcClassicV1) NewUpdateVPCRouteOptions(vpcID string, id string, routePatch map[string]interface{}) *UpdateVPCRouteOptions {
	return &UpdateVPCRouteOptions{
		VPCID:      core.StringPtr(vpcID),
		ID:         core.StringPtr(id),
		RoutePatch: routePatch,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *UpdateVPCRouteOptions) SetVPCID(vpcID string) *UpdateVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateVPCRouteOptions) SetID(id string) *UpdateVPCRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetRoutePatch : Allow user to set RoutePatch
func (options *UpdateVPCRouteOptions) SetRoutePatch(routePatch map[string]interface{}) *UpdateVPCRouteOptions {
	options.RoutePatch = routePatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateVPCRouteOptions) SetHeaders(param map[string]string) *UpdateVPCRouteOptions {
	options.Headers = param
	return options
}

// UpdateVPNGatewayConnectionOptions : The UpdateVPNGatewayConnection options.
type UpdateVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The VPN gateway connection patch.
	VPNGatewayConnectionPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateVPNGatewayConnectionOptions : Instantiate UpdateVPNGatewayConnectionOptions
func (*VpcClassicV1) NewUpdateVPNGatewayConnectionOptions(vpnGatewayID string, id string, vpnGatewayConnectionPatch map[string]interface{}) *UpdateVPNGatewayConnectionOptions {
	return &UpdateVPNGatewayConnectionOptions{
		VPNGatewayID:              core.StringPtr(vpnGatewayID),
		ID:                        core.StringPtr(id),
		VPNGatewayConnectionPatch: vpnGatewayConnectionPatch,
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *UpdateVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID string) *UpdateVPNGatewayConnectionOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateVPNGatewayConnectionOptions) SetID(id string) *UpdateVPNGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetVPNGatewayConnectionPatch : Allow user to set VPNGatewayConnectionPatch
func (options *UpdateVPNGatewayConnectionOptions) SetVPNGatewayConnectionPatch(vpnGatewayConnectionPatch map[string]interface{}) *UpdateVPNGatewayConnectionOptions {
	options.VPNGatewayConnectionPatch = vpnGatewayConnectionPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateVPNGatewayConnectionOptions) SetHeaders(param map[string]string) *UpdateVPNGatewayConnectionOptions {
	options.Headers = param
	return options
}

// UpdateVPNGatewayOptions : The UpdateVPNGateway options.
type UpdateVPNGatewayOptions struct {
	// The VPN gateway identifier.
	ID *string `validate:"required,ne="`

	// The VPN gateway patch.
	VPNGatewayPatch map[string]interface{} `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateVPNGatewayOptions : Instantiate UpdateVPNGatewayOptions
func (*VpcClassicV1) NewUpdateVPNGatewayOptions(id string, vpnGatewayPatch map[string]interface{}) *UpdateVPNGatewayOptions {
	return &UpdateVPNGatewayOptions{
		ID:              core.StringPtr(id),
		VPNGatewayPatch: vpnGatewayPatch,
	}
}

// SetID : Allow user to set ID
func (options *UpdateVPNGatewayOptions) SetID(id string) *UpdateVPNGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetVPNGatewayPatch : Allow user to set VPNGatewayPatch
func (options *UpdateVPNGatewayOptions) SetVPNGatewayPatch(vpnGatewayPatch map[string]interface{}) *UpdateVPNGatewayOptions {
	options.VPNGatewayPatch = vpnGatewayPatch
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateVPNGatewayOptions) SetHeaders(param map[string]string) *UpdateVPNGatewayOptions {
	options.Headers = param
	return options
}

// VPC : VPC struct
type VPC struct {
	// Indicates whether this VPC is connected to Classic Infrastructure. If true, this VPC's resources have private
	// network connectivity to the account's Classic Infrastructure resources. Only one VPC, per region, may be connected
	// in this way. This value is set at creation and subsequently immutable.
	ClassicAccess *bool `json:"classic_access" validate:"required"`

	// The date and time that the VPC was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this VPC.
	CRN *string `json:"crn" validate:"required"`

	// Array of CSE ([Cloud Service Endpoint](https://cloud.ibm.com/docs/resources?topic=resources-service-endpoints))
	// source IP addresses for the VPC. The VPC will have one CSE source IP address per zone.
	CseSourceIps []VpccseSourceIP `json:"cse_source_ips,omitempty"`

	// The default network ACL to use for subnets created in this VPC.
	DefaultNetworkACL *NetworkACLReference `json:"default_network_acl" validate:"required"`

	// The default security group to use for network interfaces created in this VPC.
	DefaultSecurityGroup *SecurityGroupReference `json:"default_security_group" validate:"required"`

	// The URL for this VPC.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this VPC.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this VPC.
	Name *string `json:"name" validate:"required"`

	// The resource group for this VPC.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The status of this VPC.
	Status *string `json:"status" validate:"required"`
}

// Constants associated with the VPC.Status property.
// The status of this VPC.
const (
	VPCStatusAvailableConst = "available"
	VPCStatusDeletingConst  = "deleting"
	VPCStatusFailedConst    = "failed"
	VPCStatusPendingConst   = "pending"
)

// UnmarshalVPC unmarshals an instance of VPC from the specified map of raw messages.
func UnmarshalVPC(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPC)
	err = core.UnmarshalPrimitive(m, "classic_access", &obj.ClassicAccess)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "cse_source_ips", &obj.CseSourceIps, UnmarshalVpccseSourceIP)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "default_network_acl", &obj.DefaultNetworkACL, UnmarshalNetworkACLReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "default_security_group", &obj.DefaultSecurityGroup, UnmarshalSecurityGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VpccseSourceIP : VpccseSourceIP struct
type VpccseSourceIP struct {
	// The Cloud Service Endpoint source IP address for this zone.
	IP *IP `json:"ip" validate:"required"`

	// The zone this Cloud Service Endpoint source IP belongs to.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// UnmarshalVpccseSourceIP unmarshals an instance of VpccseSourceIP from the specified map of raw messages.
func UnmarshalVpccseSourceIP(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VpccseSourceIP)
	err = core.UnmarshalModel(m, "ip", &obj.IP, UnmarshalIP)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPCCollection : VPCCollection struct
type VPCCollection struct {
	// A link to the first page of resources.
	First *VPCCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *VPCCollectionNext `json:"next,omitempty"`

	// Collection of VPCs.
	Vpcs []VPC `json:"vpcs" validate:"required"`
}

// UnmarshalVPCCollection unmarshals an instance of VPCCollection from the specified map of raw messages.
func UnmarshalVPCCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPCCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalVPCCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalVPCCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpcs", &obj.Vpcs, UnmarshalVPC)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPCCollectionFirst : A link to the first page of resources.
type VPCCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalVPCCollectionFirst unmarshals an instance of VPCCollectionFirst from the specified map of raw messages.
func UnmarshalVPCCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPCCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPCCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type VPCCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalVPCCollectionNext unmarshals an instance of VPCCollectionNext from the specified map of raw messages.
func UnmarshalVPCCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPCCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPCIdentity : Identifies a VPC by a unique property.
// Models which "extend" this model:
// - VPCIdentityByID
// - VPCIdentityByCRN
// - VPCIdentityByHref
type VPCIdentity struct {
	// The unique identifier for this VPC.
	ID *string `json:"id,omitempty"`

	// The CRN for this VPC.
	CRN *string `json:"crn,omitempty"`

	// The URL for this VPC.
	Href *string `json:"href,omitempty"`
}

func (*VPCIdentity) isaVPCIdentity() bool {
	return true
}

type VPCIdentityIntf interface {
	isaVPCIdentity() bool
}

// UnmarshalVPCIdentity unmarshals an instance of VPCIdentity from the specified map of raw messages.
func UnmarshalVPCIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPCIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPCPatch : VPCPatch struct
type VPCPatch struct {
	// The unique user-defined name for this VPC.
	Name *string `json:"name,omitempty"`
}

// UnmarshalVPCPatch unmarshals an instance of VPCPatch from the specified map of raw messages.
func UnmarshalVPCPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPCPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the VPCPatch
func (vpcPatch *VPCPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(vpcPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// VPCReference : VPCReference struct
type VPCReference struct {
	// The CRN for this VPC.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this VPC.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this VPC.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this VPC.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalVPCReference unmarshals an instance of VPCReference from the specified map of raw messages.
func UnmarshalVPCReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPCReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGateway : VPNGateway struct
// Models which "extend" this model:
// - VPNGatewayPolicyMode
type VPNGateway struct {
	// Collection of references to VPN gateway connections.
	Connections []VPNGatewayConnectionReference `json:"connections" validate:"required"`

	// The date and time that this VPN gateway was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The VPN gateway's CRN.
	CRN *string `json:"crn" validate:"required"`

	// The VPN gateway's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this VPN gateway.
	ID *string `json:"id" validate:"required"`

	// Collection of VPN gateway members.
	Members []VPNGatewayMember `json:"members" validate:"required"`

	// The user-defined name for this VPN gateway.
	Name *string `json:"name" validate:"required"`

	// The resource group for this VPN gateway.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The status of the VPN gateway.
	Status *string `json:"status" validate:"required"`

	Subnet *SubnetReference `json:"subnet" validate:"required"`

	// Policy mode VPN gateway.
	Mode *string `json:"mode,omitempty"`
}

// Constants associated with the VPNGateway.ResourceType property.
// The resource type.
const (
	VPNGatewayResourceTypeVPNGatewayConst = "vpn_gateway"
)

// Constants associated with the VPNGateway.Status property.
// The status of the VPN gateway.
const (
	VPNGatewayStatusAvailableConst = "available"
	VPNGatewayStatusDeletingConst  = "deleting"
	VPNGatewayStatusFailedConst    = "failed"
	VPNGatewayStatusPendingConst   = "pending"
)

// Constants associated with the VPNGateway.Mode property.
// Policy mode VPN gateway.
const (
	VPNGatewayModePolicyConst = "policy"
)

func (*VPNGateway) isaVPNGateway() bool {
	return true
}

type VPNGatewayIntf interface {
	isaVPNGateway() bool
}

// UnmarshalVPNGateway unmarshals an instance of VPNGateway from the specified map of raw messages.
func UnmarshalVPNGateway(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGateway)
	err = core.UnmarshalModel(m, "connections", &obj.Connections, UnmarshalVPNGatewayConnectionReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "members", &obj.Members, UnmarshalVPNGatewayMember)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnet", &obj.Subnet, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "mode", &obj.Mode)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayCollection : VPNGatewayCollection struct
type VPNGatewayCollection struct {
	// A link to the first page of resources.
	First *VPNGatewayCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *VPNGatewayCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`

	// Collection of VPN gateways.
	VPNGateways []VPNGatewayIntf `json:"vpn_gateways" validate:"required"`
}

// UnmarshalVPNGatewayCollection unmarshals an instance of VPNGatewayCollection from the specified map of raw messages.
func UnmarshalVPNGatewayCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalVPNGatewayCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalVPNGatewayCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpn_gateways", &obj.VPNGateways, UnmarshalVPNGateway)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayCollectionFirst : A link to the first page of resources.
type VPNGatewayCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalVPNGatewayCollectionFirst unmarshals an instance of VPNGatewayCollectionFirst from the specified map of raw messages.
func UnmarshalVPNGatewayCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type VPNGatewayCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalVPNGatewayCollectionNext unmarshals an instance of VPNGatewayCollectionNext from the specified map of raw messages.
func UnmarshalVPNGatewayCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnection : VPNGatewayConnection struct
// Models which "extend" this model:
// - VPNGatewayConnectionPolicyMode
type VPNGatewayConnection struct {
	// If set to false, the VPN gateway connection is shut down.
	AdminStateUp *bool `json:"admin_state_up" validate:"required"`

	// The authentication mode. Only `psk` is currently supported.
	AuthenticationMode *string `json:"authentication_mode" validate:"required"`

	// The date and time that this VPN gateway connection was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The Dead Peer Detection settings.
	DeadPeerDetection *VPNGatewayConnectionDpd `json:"dead_peer_detection" validate:"required"`

	// The VPN connection's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this VPN gateway connection.
	ID *string `json:"id" validate:"required"`

	// Optional IKE policy configuration. The absence of a policy indicates autonegotiation.
	IkePolicy *IkePolicyReference `json:"ike_policy,omitempty"`

	// Optional IPsec policy configuration. The absence of a policy indicates
	// autonegotiation.
	IpsecPolicy *IPsecPolicyReference `json:"ipsec_policy,omitempty"`

	// The mode of the VPN gateway.
	Mode *string `json:"mode" validate:"required"`

	// The user-defined name for this VPN gateway connection.
	Name *string `json:"name" validate:"required"`

	// The IP address of the peer VPN gateway.
	PeerAddress *string `json:"peer_address" validate:"required"`

	// The preshared key.
	Psk *string `json:"psk" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The status of a VPN gateway connection.
	Status *string `json:"status" validate:"required"`

	// A collection of local CIDRs for this resource.
	LocalCIDRs []string `json:"local_cidrs,omitempty"`

	// A collection of peer CIDRs for this resource.
	PeerCIDRs []string `json:"peer_cidrs,omitempty"`
}

// Constants associated with the VPNGatewayConnection.AuthenticationMode property.
// The authentication mode. Only `psk` is currently supported.
const (
	VPNGatewayConnectionAuthenticationModePskConst = "psk"
)

// Constants associated with the VPNGatewayConnection.Mode property.
// The mode of the VPN gateway.
const (
	VPNGatewayConnectionModePolicyConst = "policy"
	VPNGatewayConnectionModeRouteConst  = "route"
)

// Constants associated with the VPNGatewayConnection.ResourceType property.
// The resource type.
const (
	VPNGatewayConnectionResourceTypeVPNGatewayConnectionConst = "vpn_gateway_connection"
)

// Constants associated with the VPNGatewayConnection.Status property.
// The status of a VPN gateway connection.
const (
	VPNGatewayConnectionStatusDownConst = "down"
	VPNGatewayConnectionStatusUpConst   = "up"
)

func (*VPNGatewayConnection) isaVPNGatewayConnection() bool {
	return true
}

type VPNGatewayConnectionIntf interface {
	isaVPNGatewayConnection() bool
}

// UnmarshalVPNGatewayConnection unmarshals an instance of VPNGatewayConnection from the specified map of raw messages.
func UnmarshalVPNGatewayConnection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnection)
	err = core.UnmarshalPrimitive(m, "admin_state_up", &obj.AdminStateUp)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "authentication_mode", &obj.AuthenticationMode)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "dead_peer_detection", &obj.DeadPeerDetection, UnmarshalVPNGatewayConnectionDpd)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ike_policy", &obj.IkePolicy, UnmarshalIkePolicyReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ipsec_policy", &obj.IpsecPolicy, UnmarshalIPsecPolicyReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "mode", &obj.Mode)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "peer_address", &obj.PeerAddress)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "psk", &obj.Psk)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "local_cidrs", &obj.LocalCIDRs)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "peer_cidrs", &obj.PeerCIDRs)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionCollection : Collection of VPN gateway connections in a VPN gateway.
type VPNGatewayConnectionCollection struct {
	// Array of VPN gateway connections.
	Connections []VPNGatewayConnectionIntf `json:"connections" validate:"required"`
}

// UnmarshalVPNGatewayConnectionCollection unmarshals an instance of VPNGatewayConnectionCollection from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionCollection)
	err = core.UnmarshalModel(m, "connections", &obj.Connections, UnmarshalVPNGatewayConnection)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionDpd : The Dead Peer Detection settings.
type VPNGatewayConnectionDpd struct {
	// Dead Peer Detection actions.
	Action *string `json:"action" validate:"required"`

	// Dead Peer Detection interval in seconds.
	Interval *int64 `json:"interval" validate:"required"`

	// Dead Peer Detection timeout in seconds. Must be at least the interval.
	Timeout *int64 `json:"timeout" validate:"required"`
}

// Constants associated with the VPNGatewayConnectionDpd.Action property.
// Dead Peer Detection actions.
const (
	VPNGatewayConnectionDpdActionClearConst   = "clear"
	VPNGatewayConnectionDpdActionHoldConst    = "hold"
	VPNGatewayConnectionDpdActionNoneConst    = "none"
	VPNGatewayConnectionDpdActionRestartConst = "restart"
)

// UnmarshalVPNGatewayConnectionDpd unmarshals an instance of VPNGatewayConnectionDpd from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionDpd(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionDpd)
	err = core.UnmarshalPrimitive(m, "action", &obj.Action)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "interval", &obj.Interval)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "timeout", &obj.Timeout)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionDpdPrototype : The Dead Peer Detection settings.
type VPNGatewayConnectionDpdPrototype struct {
	// Dead Peer Detection actions.
	Action *string `json:"action,omitempty"`

	// Dead Peer Detection interval in seconds.
	Interval *int64 `json:"interval,omitempty"`

	// Dead Peer Detection timeout in seconds. Must be at least the interval.
	Timeout *int64 `json:"timeout,omitempty"`
}

// Constants associated with the VPNGatewayConnectionDpdPrototype.Action property.
// Dead Peer Detection actions.
const (
	VPNGatewayConnectionDpdPrototypeActionClearConst   = "clear"
	VPNGatewayConnectionDpdPrototypeActionHoldConst    = "hold"
	VPNGatewayConnectionDpdPrototypeActionNoneConst    = "none"
	VPNGatewayConnectionDpdPrototypeActionRestartConst = "restart"
)

// UnmarshalVPNGatewayConnectionDpdPrototype unmarshals an instance of VPNGatewayConnectionDpdPrototype from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionDpdPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionDpdPrototype)
	err = core.UnmarshalPrimitive(m, "action", &obj.Action)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "interval", &obj.Interval)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "timeout", &obj.Timeout)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionLocalCIDRs : VPNGatewayConnectionLocalCIDRs struct
type VPNGatewayConnectionLocalCIDRs struct {
	// A collection of local CIDRs for this resource.
	LocalCIDRs []string `json:"local_cidrs,omitempty"`
}

// UnmarshalVPNGatewayConnectionLocalCIDRs unmarshals an instance of VPNGatewayConnectionLocalCIDRs from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionLocalCIDRs(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionLocalCIDRs)
	err = core.UnmarshalPrimitive(m, "local_cidrs", &obj.LocalCIDRs)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionPatch : VPNGatewayConnectionPatch struct
type VPNGatewayConnectionPatch struct {
	// If set to false, the VPN gateway connection is shut down.
	AdminStateUp *bool `json:"admin_state_up,omitempty"`

	// The Dead Peer Detection settings.
	DeadPeerDetection *VPNGatewayConnectionDpdPrototype `json:"dead_peer_detection,omitempty"`

	// Optional IKE policy configuration. The absence of a policy indicates autonegotiation.
	IkePolicy IkePolicyIdentityIntf `json:"ike_policy,omitempty"`

	// Optional IPsec policy configuration. The absence of a policy indicates
	// autonegotiation.
	IpsecPolicy IPsecPolicyIdentityIntf `json:"ipsec_policy,omitempty"`

	// The user-defined name for this VPN gateway connection.
	Name *string `json:"name,omitempty"`

	// The IP address of the peer VPN gateway.
	PeerAddress *string `json:"peer_address,omitempty"`

	// The preshared key.
	Psk *string `json:"psk,omitempty"`
}

// UnmarshalVPNGatewayConnectionPatch unmarshals an instance of VPNGatewayConnectionPatch from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionPatch)
	err = core.UnmarshalPrimitive(m, "admin_state_up", &obj.AdminStateUp)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "dead_peer_detection", &obj.DeadPeerDetection, UnmarshalVPNGatewayConnectionDpdPrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ike_policy", &obj.IkePolicy, UnmarshalIkePolicyIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ipsec_policy", &obj.IpsecPolicy, UnmarshalIPsecPolicyIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "peer_address", &obj.PeerAddress)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "psk", &obj.Psk)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the VPNGatewayConnectionPatch
func (vpnGatewayConnectionPatch *VPNGatewayConnectionPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(vpnGatewayConnectionPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// VPNGatewayConnectionPeerCIDRs : VPNGatewayConnectionPeerCIDRs struct
type VPNGatewayConnectionPeerCIDRs struct {
	// A collection of peer CIDRs for this resource.
	PeerCIDRs []string `json:"peer_cidrs,omitempty"`
}

// UnmarshalVPNGatewayConnectionPeerCIDRs unmarshals an instance of VPNGatewayConnectionPeerCIDRs from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionPeerCIDRs(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionPeerCIDRs)
	err = core.UnmarshalPrimitive(m, "peer_cidrs", &obj.PeerCIDRs)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionPrototype : VPNGatewayConnectionPrototype struct
// Models which "extend" this model:
// - VPNGatewayConnectionPrototypeVPNGatewayConnectionPolicyModePrototype
type VPNGatewayConnectionPrototype struct {
	// If set to false, the VPN gateway connection is shut down.
	AdminStateUp *bool `json:"admin_state_up,omitempty"`

	// The Dead Peer Detection settings.
	DeadPeerDetection *VPNGatewayConnectionDpdPrototype `json:"dead_peer_detection,omitempty"`

	// Optional IKE policy configuration. The absence of a policy indicates autonegotiation.
	IkePolicy IkePolicyIdentityIntf `json:"ike_policy,omitempty"`

	// Optional IPsec policy configuration. The absence of a policy indicates
	// autonegotiation.
	IpsecPolicy IPsecPolicyIdentityIntf `json:"ipsec_policy,omitempty"`

	// The user-defined name for this VPN gateway connection.
	Name *string `json:"name,omitempty"`

	// The IP address of the peer VPN gateway.
	PeerAddress *string `json:"peer_address" validate:"required"`

	// The preshared key.
	Psk *string `json:"psk" validate:"required"`

	// A collection of local CIDRs for this resource.
	LocalCIDRs []string `json:"local_cidrs,omitempty"`

	// A collection of peer CIDRs for this resource.
	PeerCIDRs []string `json:"peer_cidrs,omitempty"`
}

func (*VPNGatewayConnectionPrototype) isaVPNGatewayConnectionPrototype() bool {
	return true
}

type VPNGatewayConnectionPrototypeIntf interface {
	isaVPNGatewayConnectionPrototype() bool
}

// UnmarshalVPNGatewayConnectionPrototype unmarshals an instance of VPNGatewayConnectionPrototype from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionPrototype)
	err = core.UnmarshalPrimitive(m, "admin_state_up", &obj.AdminStateUp)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "dead_peer_detection", &obj.DeadPeerDetection, UnmarshalVPNGatewayConnectionDpdPrototype)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ike_policy", &obj.IkePolicy, UnmarshalIkePolicyIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ipsec_policy", &obj.IpsecPolicy, UnmarshalIPsecPolicyIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "peer_address", &obj.PeerAddress)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "psk", &obj.Psk)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "local_cidrs", &obj.LocalCIDRs)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "peer_cidrs", &obj.PeerCIDRs)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayConnectionReference : VPNGatewayConnectionReference struct
type VPNGatewayConnectionReference struct {
	// The VPN connection's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this VPN gateway connection.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this VPN connection.
	Name *string `json:"name" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the VPNGatewayConnectionReference.ResourceType property.
// The resource type.
const (
	VPNGatewayConnectionReferenceResourceTypeVPNGatewayConnectionConst = "vpn_gateway_connection"
)

// UnmarshalVPNGatewayConnectionReference unmarshals an instance of VPNGatewayConnectionReference from the specified map of raw messages.
func UnmarshalVPNGatewayConnectionReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayConnectionReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayMember : VPNGatewayMember struct
type VPNGatewayMember struct {
	// The public IP address assigned to the VPN gateway member.
	PublicIP *IP `json:"public_ip" validate:"required"`

	// The high availability role assigned to the VPN gateway member.
	Role *string `json:"role" validate:"required"`

	// The status of the VPN gateway member.
	Status *string `json:"status" validate:"required"`
}

// Constants associated with the VPNGatewayMember.Role property.
// The high availability role assigned to the VPN gateway member.
const (
	VPNGatewayMemberRoleActiveConst  = "active"
	VPNGatewayMemberRoleStandbyConst = "standby"
)

// Constants associated with the VPNGatewayMember.Status property.
// The status of the VPN gateway member.
const (
	VPNGatewayMemberStatusAvailableConst = "available"
	VPNGatewayMemberStatusDeletingConst  = "deleting"
	VPNGatewayMemberStatusFailedConst    = "failed"
	VPNGatewayMemberStatusPendingConst   = "pending"
)

// UnmarshalVPNGatewayMember unmarshals an instance of VPNGatewayMember from the specified map of raw messages.
func UnmarshalVPNGatewayMember(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayMember)
	err = core.UnmarshalModel(m, "public_ip", &obj.PublicIP, UnmarshalIP)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "role", &obj.Role)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VPNGatewayPatch : VPNGatewayPatch struct
type VPNGatewayPatch struct {
	// The user-defined name for this VPN gateway.
	Name *string `json:"name,omitempty"`
}

// UnmarshalVPNGatewayPatch unmarshals an instance of VPNGatewayPatch from the specified map of raw messages.
func UnmarshalVPNGatewayPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the VPNGatewayPatch
func (vpnGatewayPatch *VPNGatewayPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(vpnGatewayPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// VPNGatewayPrototype : VPNGatewayPrototype struct
// Models which "extend" this model:
// - VPNGatewayPrototypeVPNGatewayPolicyModePrototype
type VPNGatewayPrototype struct {
	// The user-defined name for this VPN gateway.
	Name *string `json:"name,omitempty"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf `json:"resource_group,omitempty"`

	// Identifies a subnet by a unique property.
	Subnet SubnetIdentityIntf `json:"subnet" validate:"required"`

	// Policy mode VPN gateway.
	Mode *string `json:"mode,omitempty"`
}

// Constants associated with the VPNGatewayPrototype.Mode property.
// Policy mode VPN gateway.
const (
	VPNGatewayPrototypeModePolicyConst = "policy"
)

func (*VPNGatewayPrototype) isaVPNGatewayPrototype() bool {
	return true
}

type VPNGatewayPrototypeIntf interface {
	isaVPNGatewayPrototype() bool
}

// UnmarshalVPNGatewayPrototype unmarshals an instance of VPNGatewayPrototype from the specified map of raw messages.
func UnmarshalVPNGatewayPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VPNGatewayPrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnet", &obj.Subnet, UnmarshalSubnetIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "mode", &obj.Mode)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Volume : Volume struct
type Volume struct {
	// The capacity of the volume in gigabytes. The specified minimum and maximum capacity values for creating or updating
	// volumes may expand in the future.
	Capacity *int64 `json:"capacity" validate:"required"`

	// The date and time that the volume was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this volume.
	CRN *string `json:"crn" validate:"required"`

	// The type of encryption used on the volume.
	Encryption *string `json:"encryption" validate:"required"`

	// A reference to the root key used to wrap the data encryption key for the volume.
	//
	// This property will be present for volumes with an `encryption` type of
	// `user_managed`.
	EncryptionKey *EncryptionKeyReference `json:"encryption_key,omitempty"`

	// The URL for this volume.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this volume.
	ID *string `json:"id" validate:"required"`

	// The bandwidth for the volume.
	Iops *int64 `json:"iops" validate:"required"`

	// The unique user-defined name for this volume.
	Name *string `json:"name" validate:"required"`

	// The profile this volume uses.
	Profile *VolumeProfileReference `json:"profile" validate:"required"`

	// The resource group for this volume.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The status of the volume.
	//
	// The enumerated values for this property will expand in the future. When processing this property, check for and log
	// unknown values. Optionally halt processing and surface the error, or bypass the volume on which the unexpected
	// property value was encountered.
	Status *string `json:"status" validate:"required"`

	// The collection of volume attachments attaching instances to the volume.
	VolumeAttachments []VolumeAttachmentReferenceVolumeContext `json:"volume_attachments" validate:"required"`

	// The zone this volume resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the Volume.Encryption property.
// The type of encryption used on the volume.
const (
	VolumeEncryptionProviderManagedConst = "provider_managed"
	VolumeEncryptionUserManagedConst     = "user_managed"
)

// Constants associated with the Volume.Status property.
// The status of the volume.
//
// The enumerated values for this property will expand in the future. When processing this property, check for and log
// unknown values. Optionally halt processing and surface the error, or bypass the volume on which the unexpected
// property value was encountered.
const (
	VolumeStatusAvailableConst       = "available"
	VolumeStatusFailedConst          = "failed"
	VolumeStatusPendingConst         = "pending"
	VolumeStatusPendingDeletionConst = "pending_deletion"
	VolumeStatusUnusableConst        = "unusable"
)

// UnmarshalVolume unmarshals an instance of Volume from the specified map of raw messages.
func UnmarshalVolume(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Volume)
	err = core.UnmarshalPrimitive(m, "capacity", &obj.Capacity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "encryption", &obj.Encryption)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "encryption_key", &obj.EncryptionKey, UnmarshalEncryptionKeyReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "iops", &obj.Iops)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profile", &obj.Profile, UnmarshalVolumeProfileReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume_attachments", &obj.VolumeAttachments, UnmarshalVolumeAttachmentReferenceVolumeContext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachment : VolumeAttachment struct
type VolumeAttachment struct {
	// The date and time that the volume was attached.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// If set to true, when deleting the instance the volume will also be deleted.
	DeleteVolumeOnInstanceDelete *bool `json:"delete_volume_on_instance_delete,omitempty"`

	// Information about how the volume is exposed to the instance operating system.
	//
	// This property may be absent if the volume attachment's `status` is not `attached`.
	Device *VolumeAttachmentDevice `json:"device,omitempty"`

	// The URL for this volume attachment.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this volume attachment.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this volume attachment.
	Name *string `json:"name" validate:"required"`

	// The status of this volume attachment.
	Status *string `json:"status" validate:"required"`

	// The type of volume attachment.
	Type *string `json:"type" validate:"required"`

	// The attached volume.
	Volume *VolumeReference `json:"volume" validate:"required"`
}

// Constants associated with the VolumeAttachment.Status property.
// The status of this volume attachment.
const (
	VolumeAttachmentStatusAttachedConst  = "attached"
	VolumeAttachmentStatusAttachingConst = "attaching"
	VolumeAttachmentStatusDeletingConst  = "deleting"
	VolumeAttachmentStatusDetachingConst = "detaching"
)

// Constants associated with the VolumeAttachment.Type property.
// The type of volume attachment.
const (
	VolumeAttachmentTypeBootConst = "boot"
	VolumeAttachmentTypeDataConst = "data"
)

// UnmarshalVolumeAttachment unmarshals an instance of VolumeAttachment from the specified map of raw messages.
func UnmarshalVolumeAttachment(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachment)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "delete_volume_on_instance_delete", &obj.DeleteVolumeOnInstanceDelete)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "device", &obj.Device, UnmarshalVolumeAttachmentDevice)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume", &obj.Volume, UnmarshalVolumeReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentCollection : VolumeAttachmentCollection struct
type VolumeAttachmentCollection struct {
	// Collection of volume attachments.
	VolumeAttachments []VolumeAttachment `json:"volume_attachments" validate:"required"`
}

// UnmarshalVolumeAttachmentCollection unmarshals an instance of VolumeAttachmentCollection from the specified map of raw messages.
func UnmarshalVolumeAttachmentCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentCollection)
	err = core.UnmarshalModel(m, "volume_attachments", &obj.VolumeAttachments, UnmarshalVolumeAttachment)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentDevice : VolumeAttachmentDevice struct
type VolumeAttachmentDevice struct {
	// A unique identifier for the device which is exposed to the instance operating system.
	ID *string `json:"id,omitempty"`
}

// UnmarshalVolumeAttachmentDevice unmarshals an instance of VolumeAttachmentDevice from the specified map of raw messages.
func UnmarshalVolumeAttachmentDevice(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentDevice)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentPatch : VolumeAttachmentPatch struct
type VolumeAttachmentPatch struct {
	// If set to true, when deleting the instance the volume will also be deleted.
	DeleteVolumeOnInstanceDelete *bool `json:"delete_volume_on_instance_delete,omitempty"`

	// The user-defined name for this volume attachment.
	Name *string `json:"name,omitempty"`
}

// UnmarshalVolumeAttachmentPatch unmarshals an instance of VolumeAttachmentPatch from the specified map of raw messages.
func UnmarshalVolumeAttachmentPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentPatch)
	err = core.UnmarshalPrimitive(m, "delete_volume_on_instance_delete", &obj.DeleteVolumeOnInstanceDelete)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the VolumeAttachmentPatch
func (volumeAttachmentPatch *VolumeAttachmentPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(volumeAttachmentPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// VolumeAttachmentPrototypeInstanceByImageContext : VolumeAttachmentPrototypeInstanceByImageContext struct
type VolumeAttachmentPrototypeInstanceByImageContext struct {
	// If set to true, when deleting the instance the volume will also be deleted.
	DeleteVolumeOnInstanceDelete *bool `json:"delete_volume_on_instance_delete,omitempty"`

	// The user-defined name for this volume attachment.
	Name *string `json:"name,omitempty"`

	// A prototype object for a new volume.
	Volume *VolumePrototypeInstanceByImageContext `json:"volume" validate:"required"`
}

// NewVolumeAttachmentPrototypeInstanceByImageContext : Instantiate VolumeAttachmentPrototypeInstanceByImageContext (Generic Model Constructor)
func (*VpcClassicV1) NewVolumeAttachmentPrototypeInstanceByImageContext(volume *VolumePrototypeInstanceByImageContext) (model *VolumeAttachmentPrototypeInstanceByImageContext, err error) {
	model = &VolumeAttachmentPrototypeInstanceByImageContext{
		Volume: volume,
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalVolumeAttachmentPrototypeInstanceByImageContext unmarshals an instance of VolumeAttachmentPrototypeInstanceByImageContext from the specified map of raw messages.
func UnmarshalVolumeAttachmentPrototypeInstanceByImageContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentPrototypeInstanceByImageContext)
	err = core.UnmarshalPrimitive(m, "delete_volume_on_instance_delete", &obj.DeleteVolumeOnInstanceDelete)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume", &obj.Volume, UnmarshalVolumePrototypeInstanceByImageContext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentPrototypeInstanceContext : VolumeAttachmentPrototypeInstanceContext struct
type VolumeAttachmentPrototypeInstanceContext struct {
	// If set to true, when deleting the instance the volume will also be deleted.
	DeleteVolumeOnInstanceDelete *bool `json:"delete_volume_on_instance_delete,omitempty"`

	// The user-defined name for this volume attachment.
	Name *string `json:"name,omitempty"`

	// The identity of the volume to attach to the instance, or a prototype object for a new
	// volume.
	Volume VolumeAttachmentVolumePrototypeInstanceContextIntf `json:"volume" validate:"required"`
}

// NewVolumeAttachmentPrototypeInstanceContext : Instantiate VolumeAttachmentPrototypeInstanceContext (Generic Model Constructor)
func (*VpcClassicV1) NewVolumeAttachmentPrototypeInstanceContext(volume VolumeAttachmentVolumePrototypeInstanceContextIntf) (model *VolumeAttachmentPrototypeInstanceContext, err error) {
	model = &VolumeAttachmentPrototypeInstanceContext{
		Volume: volume,
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalVolumeAttachmentPrototypeInstanceContext unmarshals an instance of VolumeAttachmentPrototypeInstanceContext from the specified map of raw messages.
func UnmarshalVolumeAttachmentPrototypeInstanceContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentPrototypeInstanceContext)
	err = core.UnmarshalPrimitive(m, "delete_volume_on_instance_delete", &obj.DeleteVolumeOnInstanceDelete)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume", &obj.Volume, UnmarshalVolumeAttachmentVolumePrototypeInstanceContext)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentReferenceInstanceContext : VolumeAttachmentReferenceInstanceContext struct
type VolumeAttachmentReferenceInstanceContext struct {
	// Information about how the volume is exposed to the instance operating system.
	//
	// This property may be absent if the volume attachment's `status` is not `attached`.
	Device *VolumeAttachmentDevice `json:"device,omitempty"`

	// The URL for this volume attachment.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this volume attachment.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this volume attachment.
	Name *string `json:"name" validate:"required"`

	// The attached volume.
	Volume *VolumeReference `json:"volume" validate:"required"`
}

// UnmarshalVolumeAttachmentReferenceInstanceContext unmarshals an instance of VolumeAttachmentReferenceInstanceContext from the specified map of raw messages.
func UnmarshalVolumeAttachmentReferenceInstanceContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentReferenceInstanceContext)
	err = core.UnmarshalModel(m, "device", &obj.Device, UnmarshalVolumeAttachmentDevice)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "volume", &obj.Volume, UnmarshalVolumeReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentReferenceVolumeContext : VolumeAttachmentReferenceVolumeContext struct
type VolumeAttachmentReferenceVolumeContext struct {
	// Information about how the volume is exposed to the instance operating system.
	//
	// This property may be absent if the volume attachment's `status` is not `attached`.
	Device *VolumeAttachmentDevice `json:"device,omitempty"`

	// The URL for this volume attachment.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this volume attachment.
	ID *string `json:"id" validate:"required"`

	// The attached instance.
	Instance *InstanceReference `json:"instance" validate:"required"`

	// The user-defined name for this volume attachment.
	Name *string `json:"name" validate:"required"`

	// The type of volume attachment.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the VolumeAttachmentReferenceVolumeContext.Type property.
// The type of volume attachment.
const (
	VolumeAttachmentReferenceVolumeContextTypeBootConst = "boot"
	VolumeAttachmentReferenceVolumeContextTypeDataConst = "data"
)

// UnmarshalVolumeAttachmentReferenceVolumeContext unmarshals an instance of VolumeAttachmentReferenceVolumeContext from the specified map of raw messages.
func UnmarshalVolumeAttachmentReferenceVolumeContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentReferenceVolumeContext)
	err = core.UnmarshalModel(m, "device", &obj.Device, UnmarshalVolumeAttachmentDevice)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "instance", &obj.Instance, UnmarshalInstanceReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeAttachmentVolumePrototypeInstanceContext : The identity of the volume to attach to the instance, or a prototype object for a new volume.
// Models which "extend" this model:
// - VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContext
type VolumeAttachmentVolumePrototypeInstanceContext struct {
	// The identity of the root key to use to wrap the data encryption key for the volume.
	//
	// If this property is not provided, the `encryption` type for the volume will be
	// `provider_managed`.
	EncryptionKey EncryptionKeyIdentityIntf `json:"encryption_key,omitempty"`

	// The bandwidth for the volume.
	Iops *int64 `json:"iops,omitempty"`

	// The unique user-defined name for this volume.
	Name *string `json:"name,omitempty"`

	// The profile to use for this volume.
	Profile VolumeProfileIdentityIntf `json:"profile,omitempty"`

	// The capacity of the volume in gigabytes. The specified minimum and maximum capacity values for creating or updating
	// volumes may expand in the future.
	Capacity *int64 `json:"capacity,omitempty"`
}

func (*VolumeAttachmentVolumePrototypeInstanceContext) isaVolumeAttachmentVolumePrototypeInstanceContext() bool {
	return true
}

type VolumeAttachmentVolumePrototypeInstanceContextIntf interface {
	isaVolumeAttachmentVolumePrototypeInstanceContext() bool
}

// UnmarshalVolumeAttachmentVolumePrototypeInstanceContext unmarshals an instance of VolumeAttachmentVolumePrototypeInstanceContext from the specified map of raw messages.
func UnmarshalVolumeAttachmentVolumePrototypeInstanceContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeAttachmentVolumePrototypeInstanceContext)
	err = core.UnmarshalModel(m, "encryption_key", &obj.EncryptionKey, UnmarshalEncryptionKeyIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "iops", &obj.Iops)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profile", &obj.Profile, UnmarshalVolumeProfileIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "capacity", &obj.Capacity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// VolumeCollection : VolumeCollection struct
type VolumeCollection struct {
	// A link to the first page of resources.
	First *VolumeCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *VolumeCollectionNext `json:"next,omitempty"`

	// Collection of volumes.
	Volumes []Volume `json:"volumes" validate:"required"`
}

// UnmarshalVolumeCollection unmarshals an instance of VolumeCollection from the specified map of raw messages.
func UnmarshalVolumeCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(VolumeCollection)
	err = core.UnmarshalModel(m, "first", &obj.Fir