/**
 * (C) Copyright IBM Corp. 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package vpcv1_test

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"time"

	"github.com/IBM/go-sdk-core/v5/core"
	"github.com/IBM/vpc-go-sdk/vpcv1"
	"github.com/go-openapi/strfmt"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe(`VpcV1`, func() {
	var testServer *httptest.Server
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
				URL:     "https://vpcv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{})
			Expect(vpcService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_URL":       "https://vpcv1/api",
				"VPC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcService, serviceErr := vpcv1.NewVpcV1UsingExternalConfig(&vpcv1.VpcV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcService, serviceErr := vpcv1.NewVpcV1UsingExternalConfig(&vpcv1.VpcV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcService, serviceErr := vpcv1.NewVpcV1UsingExternalConfig(&vpcv1.VpcV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_URL":       "https://vpcv1/api",
				"VPC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcService, serviceErr := vpcv1.NewVpcV1UsingExternalConfig(&vpcv1.VpcV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcService, serviceErr := vpcv1.NewVpcV1UsingExternalConfig(&vpcv1.VpcV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListVpcs(listVpcsOptions *ListVpcsOptions) - Operation response error`, func() {
		version := "testString"
		listVpcsPath := "/vpcs"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVpcsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					// TODO: Add check for classic_access query parameter
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVpcs with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVpcs(listVpcsOptions *ListVpcsOptions)`, func() {
		version := "testString"
		listVpcsPath := "/vpcs"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVpcsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					// TODO: Add check for classic_access query parameter
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132, "vpcs": [{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}]}`)
				}))
			})
			It(`Invoke ListVpcs successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListVpcsWithContext(ctx, listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListVpcsWithContext(ctx, listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVpcsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					// TODO: Add check for classic_access query parameter
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132, "vpcs": [{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}]}`)
				}))
			})
			It(`Invoke ListVpcs successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListVpcs(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVpcs with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListVpcs successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPC(createVPCOptions *CreateVPCOptions) - Operation response error`, func() {
		version := "testString"
		createVPCPath := "/vpcs"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPC with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPC(createVPCOptions *CreateVPCOptions)`, func() {
		version := "testString"
		createVPCPath := "/vpcs"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}`)
				}))
			})
			It(`Invoke CreateVPC successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateVPCWithContext(ctx, createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateVPCWithContext(ctx, createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}`)
				}))
			})
			It(`Invoke CreateVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPC with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteVPC(deleteVPCOptions *DeleteVPCOptions)`, func() {
		version := "testString"
		deleteVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCOptions model
				deleteVPCOptionsModel := new(vpcv1.DeleteVPCOptions)
				deleteVPCOptionsModel.ID = core.StringPtr("testString")
				deleteVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteVPC(deleteVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPC with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCOptions model
				deleteVPCOptionsModel := new(vpcv1.DeleteVPCOptions)
				deleteVPCOptionsModel.ID = core.StringPtr("testString")
				deleteVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteVPC(deleteVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCOptions model with no property values
				deleteVPCOptionsModelNew := new(vpcv1.DeleteVPCOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteVPC(deleteVPCOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPC(getVPCOptions *GetVPCOptions) - Operation response error`, func() {
		version := "testString"
		getVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPC with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPC(getVPCOptions *GetVPCOptions)`, func() {
		version := "testString"
		getVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}`)
				}))
			})
			It(`Invoke GetVPC successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCWithContext(ctx, getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCWithContext(ctx, getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}`)
				}))
			})
			It(`Invoke GetVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPC with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCOptions model with no property values
				getVPCOptionsModelNew := new(vpcv1.GetVPCOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPC(getVPCOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPC(updateVPCOptions *UpdateVPCOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPC with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPC(updateVPCOptions *UpdateVPCOptions)`, func() {
		version := "testString"
		updateVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}`)
				}))
			})
			It(`Invoke UpdateVPC successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateVPCWithContext(ctx, updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateVPCWithContext(ctx, updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "status": "available"}`)
				}))
			})
			It(`Invoke UpdateVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPC with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCOptions model with no property values
				updateVPCOptionsModelNew := new(vpcv1.UpdateVPCOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateVPC(updateVPCOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateVPC successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptions *GetVPCDefaultNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		getVPCDefaultNetworkACLPath := "/vpcs/testString/default_network_acl"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCDefaultNetworkACL with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultNetworkACLOptions model
				getVPCDefaultNetworkACLOptionsModel := new(vpcv1.GetVPCDefaultNetworkACLOptions)
				getVPCDefaultNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptions *GetVPCDefaultNetworkACLOptions)`, func() {
		version := "testString"
		getVPCDefaultNetworkACLPath := "/vpcs/testString/default_network_acl"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "mnemonic-ersatz-eatery-malaise", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"action": "allow", "before": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "destination_port_max": 22, "destination_port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetVPCDefaultNetworkACL successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCDefaultNetworkACLOptions model
				getVPCDefaultNetworkACLOptionsModel := new(vpcv1.GetVPCDefaultNetworkACLOptions)
				getVPCDefaultNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCDefaultNetworkACLWithContext(ctx, getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCDefaultNetworkACLWithContext(ctx, getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "mnemonic-ersatz-eatery-malaise", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"action": "allow", "before": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "destination_port_max": 22, "destination_port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetVPCDefaultNetworkACL successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCDefaultNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCDefaultNetworkACLOptions model
				getVPCDefaultNetworkACLOptionsModel := new(vpcv1.GetVPCDefaultNetworkACLOptions)
				getVPCDefaultNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCDefaultNetworkACL with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultNetworkACLOptions model
				getVPCDefaultNetworkACLOptionsModel := new(vpcv1.GetVPCDefaultNetworkACLOptions)
				getVPCDefaultNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCDefaultNetworkACLOptions model with no property values
				getVPCDefaultNetworkACLOptionsModelNew := new(vpcv1.GetVPCDefaultNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCDefaultNetworkACL successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultNetworkACLOptions model
				getVPCDefaultNetworkACLOptionsModel := new(vpcv1.GetVPCDefaultNetworkACLOptions)
				getVPCDefaultNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptions *GetVPCDefaultRoutingTableOptions) - Operation response error`, func() {
		version := "testString"
		getVPCDefaultRoutingTablePath := "/vpcs/testString/default_routing_table"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCDefaultRoutingTable with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultRoutingTableOptions model
				getVPCDefaultRoutingTableOptionsModel := new(vpcv1.GetVPCDefaultRoutingTableOptions)
				getVPCDefaultRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptions *GetVPCDefaultRoutingTableOptions)`, func() {
		version := "testString"
		getVPCDefaultRoutingTablePath := "/vpcs/testString/default_routing_table"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "milled-easy-equine-machines", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetVPCDefaultRoutingTable successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCDefaultRoutingTableOptions model
				getVPCDefaultRoutingTableOptionsModel := new(vpcv1.GetVPCDefaultRoutingTableOptions)
				getVPCDefaultRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCDefaultRoutingTableWithContext(ctx, getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCDefaultRoutingTableWithContext(ctx, getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "milled-easy-equine-machines", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetVPCDefaultRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCDefaultRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCDefaultRoutingTableOptions model
				getVPCDefaultRoutingTableOptionsModel := new(vpcv1.GetVPCDefaultRoutingTableOptions)
				getVPCDefaultRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCDefaultRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultRoutingTableOptions model
				getVPCDefaultRoutingTableOptionsModel := new(vpcv1.GetVPCDefaultRoutingTableOptions)
				getVPCDefaultRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCDefaultRoutingTableOptions model with no property values
				getVPCDefaultRoutingTableOptionsModelNew := new(vpcv1.GetVPCDefaultRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCDefaultRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultRoutingTableOptions model
				getVPCDefaultRoutingTableOptionsModel := new(vpcv1.GetVPCDefaultRoutingTableOptions)
				getVPCDefaultRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions) - Operation response error`, func() {
		version := "testString"
		getVPCDefaultSecurityGroupPath := "/vpcs/testString/default_security_group"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultSecurityGroupPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions)`, func() {
		version := "testString"
		getVPCDefaultSecurityGroupPath := "/vpcs/testString/default_security_group"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultSecurityGroupPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "observant-chip-emphatic-engraver", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271/rules/6f2a6efe-21e2-401c-b237-620aa26ba16a", "id": "6f2a6efe-21e2-401c-b237-620aa26ba16a", "ip_version": "ipv4", "remote": {"address": "192.168.3.4"}, "protocol": "all"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCDefaultSecurityGroupWithContext(ctx, getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCDefaultSecurityGroupWithContext(ctx, getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultSecurityGroupPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "observant-chip-emphatic-engraver", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271/rules/6f2a6efe-21e2-401c-b237-620aa26ba16a", "id": "6f2a6efe-21e2-401c-b237-620aa26ba16a", "ip_version": "ipv4", "remote": {"address": "192.168.3.4"}, "protocol": "all"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCDefaultSecurityGroup(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCDefaultSecurityGroup with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCDefaultSecurityGroupOptions model with no property values
				getVPCDefaultSecurityGroupOptionsModelNew := new(vpcv1.GetVPCDefaultSecurityGroupOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCAddressPrefixes(listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions) - Operation response error`, func() {
		version := "testString"
		listVPCAddressPrefixesPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCAddressPrefixesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCAddressPrefixes(listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions)`, func() {
		version := "testString"
		listVPCAddressPrefixesPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCAddressPrefixesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address_prefixes": [{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListVPCAddressPrefixesWithContext(ctx, listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListVPCAddressPrefixesWithContext(ctx, listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCAddressPrefixesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address_prefixes": [{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListVPCAddressPrefixes(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVPCAddressPrefixes with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListVPCAddressPrefixesOptions model with no property values
				listVPCAddressPrefixesOptionsModelNew := new(vpcv1.ListVPCAddressPrefixesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCAddressPrefix(createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions) - Operation response error`, func() {
		version := "testString"
		createVPCAddressPrefixPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCAddressPrefix(createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions)`, func() {
		version := "testString"
		createVPCAddressPrefixPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateVPCAddressPrefixWithContext(ctx, createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateVPCAddressPrefixWithContext(ctx, createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVPCAddressPrefixOptions model with no property values
				createVPCAddressPrefixOptionsModelNew := new(vpcv1.CreateVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptions *DeleteVPCAddressPrefixOptions)`, func() {
		version := "testString"
		deleteVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCAddressPrefixOptions model
				deleteVPCAddressPrefixOptionsModel := new(vpcv1.DeleteVPCAddressPrefixOptions)
				deleteVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCAddressPrefixOptions model
				deleteVPCAddressPrefixOptionsModel := new(vpcv1.DeleteVPCAddressPrefixOptions)
				deleteVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCAddressPrefixOptions model with no property values
				deleteVPCAddressPrefixOptionsModelNew := new(vpcv1.DeleteVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCAddressPrefix(getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions) - Operation response error`, func() {
		version := "testString"
		getVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCAddressPrefix with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCAddressPrefix(getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions)`, func() {
		version := "testString"
		getVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCAddressPrefix successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCAddressPrefixWithContext(ctx, getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCAddressPrefixWithContext(ctx, getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCAddressPrefixOptions model with no property values
				getVPCAddressPrefixOptionsModelNew := new(vpcv1.GetVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCAddressPrefix(updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCAddressPrefix(updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions)`, func() {
		version := "testString"
		updateVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateVPCAddressPrefixWithContext(ctx, updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateVPCAddressPrefixWithContext(ctx, updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCAddressPrefixOptions model with no property values
				updateVPCAddressPrefixOptionsModelNew := new(vpcv1.UpdateVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutes(listVPCRoutesOptions *ListVPCRoutesOptions) - Operation response error`, func() {
		version := "testString"
		listVPCRoutesPath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVPCRoutes with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutes(listVPCRoutesOptions *ListVPCRoutesOptions)`, func() {
		version := "testString"
		listVPCRoutesPath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?start=a5e812a2-62c0-4555-86a5-907106760c56&limit=20"}, "routes": [{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCRoutes successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListVPCRoutesWithContext(ctx, listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListVPCRoutesWithContext(ctx, listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?start=a5e812a2-62c0-4555-86a5-907106760c56&limit=20"}, "routes": [{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCRoutes successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListVPCRoutes(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVPCRoutes with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListVPCRoutesOptions model with no property values
				listVPCRoutesOptionsModelNew := new(vpcv1.ListVPCRoutesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ListVPCRoutes(listVPCRoutesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListVPCRoutes successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoute(createVPCRouteOptions *CreateVPCRouteOptions) - Operation response error`, func() {
		version := "testString"
		createVPCRoutePath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPCRoute with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoute(createVPCRouteOptions *CreateVPCRouteOptions)`, func() {
		version := "testString"
		createVPCRoutePath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCRoute successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateVPCRouteWithContext(ctx, createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateVPCRouteWithContext(ctx, createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPCRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVPCRouteOptions model with no property values
				createVPCRouteOptionsModelNew := new(vpcv1.CreateVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateVPCRoute(createVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteVPCRoute(deleteVPCRouteOptions *DeleteVPCRouteOptions)`, func() {
		version := "testString"
		deleteVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCRoutePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCRouteOptions model
				deleteVPCRouteOptionsModel := new(vpcv1.DeleteVPCRouteOptions)
				deleteVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteVPCRoute(deleteVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPCRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCRouteOptions model
				deleteVPCRouteOptionsModel := new(vpcv1.DeleteVPCRouteOptions)
				deleteVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteVPCRoute(deleteVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCRouteOptions model with no property values
				deleteVPCRouteOptionsModelNew := new(vpcv1.DeleteVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteVPCRoute(deleteVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoute(getVPCRouteOptions *GetVPCRouteOptions) - Operation response error`, func() {
		version := "testString"
		getVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCRoute with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoute(getVPCRouteOptions *GetVPCRouteOptions)`, func() {
		version := "testString"
		getVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCRoute successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCRouteWithContext(ctx, getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCRouteWithContext(ctx, getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCRouteOptions model with no property values
				getVPCRouteOptionsModelNew := new(vpcv1.GetVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCRoute(getVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoute(updateVPCRouteOptions *UpdateVPCRouteOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPCRoute with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoute(updateVPCRouteOptions *UpdateVPCRouteOptions)`, func() {
		version := "testString"
		updateVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCRoute successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateVPCRouteWithContext(ctx, updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateVPCRouteWithContext(ctx, updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPCRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCRouteOptions model with no property values
				updateVPCRouteOptionsModelNew := new(vpcv1.UpdateVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateVPCRoute(updateVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateVPCRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutingTables(listVPCRoutingTablesOptions *ListVPCRoutingTablesOptions) - Operation response error`, func() {
		version := "testString"
		listVPCRoutingTablesPath := "/vpcs/testString/routing_tables"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutingTablesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// TODO: Add check for is_default query parameter
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVPCRoutingTables with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutingTablesOptions model
				listVPCRoutingTablesOptionsModel := new(vpcv1.ListVPCRoutingTablesOptions)
				listVPCRoutingTablesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTablesOptionsModel.IsDefault = core.BoolPtr(true)
				listVPCRoutingTablesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutingTables(listVPCRoutingTablesOptions *ListVPCRoutingTablesOptions)`, func() {
		version := "testString"
		listVPCRoutingTablesPath := "/vpcs/testString/routing_tables"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutingTablesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// TODO: Add check for is_default query parameter
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "routing_tables": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCRoutingTables successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListVPCRoutingTablesOptions model
				listVPCRoutingTablesOptionsModel := new(vpcv1.ListVPCRoutingTablesOptions)
				listVPCRoutingTablesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTablesOptionsModel.IsDefault = core.BoolPtr(true)
				listVPCRoutingTablesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListVPCRoutingTablesWithContext(ctx, listVPCRoutingTablesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListVPCRoutingTablesWithContext(ctx, listVPCRoutingTablesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutingTablesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// TODO: Add check for is_default query parameter
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "routing_tables": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCRoutingTables successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListVPCRoutingTables(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVPCRoutingTablesOptions model
				listVPCRoutingTablesOptionsModel := new(vpcv1.ListVPCRoutingTablesOptions)
				listVPCRoutingTablesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTablesOptionsModel.IsDefault = core.BoolPtr(true)
				listVPCRoutingTablesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVPCRoutingTables with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutingTablesOptions model
				listVPCRoutingTablesOptionsModel := new(vpcv1.ListVPCRoutingTablesOptions)
				listVPCRoutingTablesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTablesOptionsModel.IsDefault = core.BoolPtr(true)
				listVPCRoutingTablesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListVPCRoutingTablesOptions model with no property values
				listVPCRoutingTablesOptionsModelNew := new(vpcv1.ListVPCRoutingTablesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListVPCRoutingTables successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutingTablesOptions model
				listVPCRoutingTablesOptionsModel := new(vpcv1.ListVPCRoutingTablesOptions)
				listVPCRoutingTablesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTablesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTablesOptionsModel.IsDefault = core.BoolPtr(true)
				listVPCRoutingTablesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListVPCRoutingTables(listVPCRoutingTablesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoutingTable(createVPCRoutingTableOptions *CreateVPCRoutingTableOptions) - Operation response error`, func() {
		version := "testString"
		createVPCRoutingTablePath := "/vpcs/testString/routing_tables"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutingTablePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPCRoutingTable with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RoutePrototype model
				routePrototypeModel := new(vpcv1.RoutePrototype)
				routePrototypeModel.Action = core.StringPtr("delegate")
				routePrototypeModel.Destination = core.StringPtr("192.168.3.0/24")
				routePrototypeModel.Name = core.StringPtr("my-route-2")
				routePrototypeModel.NextHop = routeNextHopPrototypeModel
				routePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateVPCRoutingTableOptions model
				createVPCRoutingTableOptionsModel := new(vpcv1.CreateVPCRoutingTableOptions)
				createVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableOptionsModel.Name = core.StringPtr("my-routing-table-2")
				createVPCRoutingTableOptionsModel.RouteDirectLinkIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteVPCZoneIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.Routes = []vpcv1.RoutePrototype{*routePrototypeModel}
				createVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoutingTable(createVPCRoutingTableOptions *CreateVPCRoutingTableOptions)`, func() {
		version := "testString"
		createVPCRoutingTablePath := "/vpcs/testString/routing_tables"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutingTablePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke CreateVPCRoutingTable successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RoutePrototype model
				routePrototypeModel := new(vpcv1.RoutePrototype)
				routePrototypeModel.Action = core.StringPtr("delegate")
				routePrototypeModel.Destination = core.StringPtr("192.168.3.0/24")
				routePrototypeModel.Name = core.StringPtr("my-route-2")
				routePrototypeModel.NextHop = routeNextHopPrototypeModel
				routePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateVPCRoutingTableOptions model
				createVPCRoutingTableOptionsModel := new(vpcv1.CreateVPCRoutingTableOptions)
				createVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableOptionsModel.Name = core.StringPtr("my-routing-table-2")
				createVPCRoutingTableOptionsModel.RouteDirectLinkIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteVPCZoneIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.Routes = []vpcv1.RoutePrototype{*routePrototypeModel}
				createVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateVPCRoutingTableWithContext(ctx, createVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateVPCRoutingTableWithContext(ctx, createVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutingTablePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke CreateVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateVPCRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RoutePrototype model
				routePrototypeModel := new(vpcv1.RoutePrototype)
				routePrototypeModel.Action = core.StringPtr("delegate")
				routePrototypeModel.Destination = core.StringPtr("192.168.3.0/24")
				routePrototypeModel.Name = core.StringPtr("my-route-2")
				routePrototypeModel.NextHop = routeNextHopPrototypeModel
				routePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateVPCRoutingTableOptions model
				createVPCRoutingTableOptionsModel := new(vpcv1.CreateVPCRoutingTableOptions)
				createVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableOptionsModel.Name = core.StringPtr("my-routing-table-2")
				createVPCRoutingTableOptionsModel.RouteDirectLinkIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteVPCZoneIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.Routes = []vpcv1.RoutePrototype{*routePrototypeModel}
				createVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPCRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RoutePrototype model
				routePrototypeModel := new(vpcv1.RoutePrototype)
				routePrototypeModel.Action = core.StringPtr("delegate")
				routePrototypeModel.Destination = core.StringPtr("192.168.3.0/24")
				routePrototypeModel.Name = core.StringPtr("my-route-2")
				routePrototypeModel.NextHop = routeNextHopPrototypeModel
				routePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateVPCRoutingTableOptions model
				createVPCRoutingTableOptionsModel := new(vpcv1.CreateVPCRoutingTableOptions)
				createVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableOptionsModel.Name = core.StringPtr("my-routing-table-2")
				createVPCRoutingTableOptionsModel.RouteDirectLinkIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteVPCZoneIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.Routes = []vpcv1.RoutePrototype{*routePrototypeModel}
				createVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVPCRoutingTableOptions model with no property values
				createVPCRoutingTableOptionsModelNew := new(vpcv1.CreateVPCRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RoutePrototype model
				routePrototypeModel := new(vpcv1.RoutePrototype)
				routePrototypeModel.Action = core.StringPtr("delegate")
				routePrototypeModel.Destination = core.StringPtr("192.168.3.0/24")
				routePrototypeModel.Name = core.StringPtr("my-route-2")
				routePrototypeModel.NextHop = routeNextHopPrototypeModel
				routePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateVPCRoutingTableOptions model
				createVPCRoutingTableOptionsModel := new(vpcv1.CreateVPCRoutingTableOptions)
				createVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableOptionsModel.Name = core.StringPtr("my-routing-table-2")
				createVPCRoutingTableOptionsModel.RouteDirectLinkIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.RouteVPCZoneIngress = core.BoolPtr(true)
				createVPCRoutingTableOptionsModel.Routes = []vpcv1.RoutePrototype{*routePrototypeModel}
				createVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateVPCRoutingTable(createVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteVPCRoutingTable(deleteVPCRoutingTableOptions *DeleteVPCRoutingTableOptions)`, func() {
		version := "testString"
		deleteVPCRoutingTablePath := "/vpcs/testString/routing_tables/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCRoutingTablePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteVPCRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCRoutingTableOptions model
				deleteVPCRoutingTableOptionsModel := new(vpcv1.DeleteVPCRoutingTableOptions)
				deleteVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteVPCRoutingTable(deleteVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPCRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCRoutingTableOptions model
				deleteVPCRoutingTableOptionsModel := new(vpcv1.DeleteVPCRoutingTableOptions)
				deleteVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteVPCRoutingTable(deleteVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCRoutingTableOptions model with no property values
				deleteVPCRoutingTableOptionsModelNew := new(vpcv1.DeleteVPCRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteVPCRoutingTable(deleteVPCRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoutingTable(getVPCRoutingTableOptions *GetVPCRoutingTableOptions) - Operation response error`, func() {
		version := "testString"
		getVPCRoutingTablePath := "/vpcs/testString/routing_tables/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCRoutingTable with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRoutingTableOptions model
				getVPCRoutingTableOptionsModel := new(vpcv1.GetVPCRoutingTableOptions)
				getVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoutingTable(getVPCRoutingTableOptions *GetVPCRoutingTableOptions)`, func() {
		version := "testString"
		getVPCRoutingTablePath := "/vpcs/testString/routing_tables/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetVPCRoutingTable successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCRoutingTableOptions model
				getVPCRoutingTableOptionsModel := new(vpcv1.GetVPCRoutingTableOptions)
				getVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCRoutingTableWithContext(ctx, getVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCRoutingTableWithContext(ctx, getVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCRoutingTableOptions model
				getVPCRoutingTableOptionsModel := new(vpcv1.GetVPCRoutingTableOptions)
				getVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRoutingTableOptions model
				getVPCRoutingTableOptionsModel := new(vpcv1.GetVPCRoutingTableOptions)
				getVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCRoutingTableOptions model with no property values
				getVPCRoutingTableOptionsModelNew := new(vpcv1.GetVPCRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRoutingTableOptions model
				getVPCRoutingTableOptionsModel := new(vpcv1.GetVPCRoutingTableOptions)
				getVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCRoutingTable(getVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoutingTable(updateVPCRoutingTableOptions *UpdateVPCRoutingTableOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCRoutingTablePath := "/vpcs/testString/routing_tables/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutingTablePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPCRoutingTable with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutingTablePatch model
				routingTablePatchModel := new(vpcv1.RoutingTablePatch)
				routingTablePatchModel.Name = core.StringPtr("my-routing-table-2")
				routingTablePatchModel.RouteDirectLinkIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteVPCZoneIngress = core.BoolPtr(true)
				routingTablePatchModelAsPatch, asPatchErr := routingTablePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableOptions model
				updateVPCRoutingTableOptionsModel := new(vpcv1.UpdateVPCRoutingTableOptions)
				updateVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.RoutingTablePatch = routingTablePatchModelAsPatch
				updateVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoutingTable(updateVPCRoutingTableOptions *UpdateVPCRoutingTableOptions)`, func() {
		version := "testString"
		updateVPCRoutingTablePath := "/vpcs/testString/routing_tables/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutingTablePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke UpdateVPCRoutingTable successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the RoutingTablePatch model
				routingTablePatchModel := new(vpcv1.RoutingTablePatch)
				routingTablePatchModel.Name = core.StringPtr("my-routing-table-2")
				routingTablePatchModel.RouteDirectLinkIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteVPCZoneIngress = core.BoolPtr(true)
				routingTablePatchModelAsPatch, asPatchErr := routingTablePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableOptions model
				updateVPCRoutingTableOptionsModel := new(vpcv1.UpdateVPCRoutingTableOptions)
				updateVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.RoutingTablePatch = routingTablePatchModelAsPatch
				updateVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateVPCRoutingTableWithContext(ctx, updateVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateVPCRoutingTableWithContext(ctx, updateVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutingTablePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke UpdateVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateVPCRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RoutingTablePatch model
				routingTablePatchModel := new(vpcv1.RoutingTablePatch)
				routingTablePatchModel.Name = core.StringPtr("my-routing-table-2")
				routingTablePatchModel.RouteDirectLinkIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteVPCZoneIngress = core.BoolPtr(true)
				routingTablePatchModelAsPatch, asPatchErr := routingTablePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableOptions model
				updateVPCRoutingTableOptionsModel := new(vpcv1.UpdateVPCRoutingTableOptions)
				updateVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.RoutingTablePatch = routingTablePatchModelAsPatch
				updateVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPCRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutingTablePatch model
				routingTablePatchModel := new(vpcv1.RoutingTablePatch)
				routingTablePatchModel.Name = core.StringPtr("my-routing-table-2")
				routingTablePatchModel.RouteDirectLinkIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteVPCZoneIngress = core.BoolPtr(true)
				routingTablePatchModelAsPatch, asPatchErr := routingTablePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableOptions model
				updateVPCRoutingTableOptionsModel := new(vpcv1.UpdateVPCRoutingTableOptions)
				updateVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.RoutingTablePatch = routingTablePatchModelAsPatch
				updateVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCRoutingTableOptions model with no property values
				updateVPCRoutingTableOptionsModelNew := new(vpcv1.UpdateVPCRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateVPCRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutingTablePatch model
				routingTablePatchModel := new(vpcv1.RoutingTablePatch)
				routingTablePatchModel.Name = core.StringPtr("my-routing-table-2")
				routingTablePatchModel.RouteDirectLinkIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteTransitGatewayIngress = core.BoolPtr(true)
				routingTablePatchModel.RouteVPCZoneIngress = core.BoolPtr(true)
				routingTablePatchModelAsPatch, asPatchErr := routingTablePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableOptions model
				updateVPCRoutingTableOptionsModel := new(vpcv1.UpdateVPCRoutingTableOptions)
				updateVPCRoutingTableOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableOptionsModel.RoutingTablePatch = routingTablePatchModelAsPatch
				updateVPCRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateVPCRoutingTable(updateVPCRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptions *ListVPCRoutingTableRoutesOptions) - Operation response error`, func() {
		version := "testString"
		listVPCRoutingTableRoutesPath := "/vpcs/testString/routing_tables/testString/routes"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutingTableRoutesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVPCRoutingTableRoutes with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutingTableRoutesOptions model
				listVPCRoutingTableRoutesOptionsModel := new(vpcv1.ListVPCRoutingTableRoutesOptions)
				listVPCRoutingTableRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.RoutingTableID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTableRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptions *ListVPCRoutingTableRoutesOptions)`, func() {
		version := "testString"
		listVPCRoutingTableRoutesPath := "/vpcs/testString/routing_tables/testString/routes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutingTableRoutesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?start=a5e812a2-62c0-4555-86a5-907106760c56&limit=20"}, "routes": [{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCRoutingTableRoutes successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListVPCRoutingTableRoutesOptions model
				listVPCRoutingTableRoutesOptionsModel := new(vpcv1.ListVPCRoutingTableRoutesOptions)
				listVPCRoutingTableRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.RoutingTableID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTableRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListVPCRoutingTableRoutesWithContext(ctx, listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListVPCRoutingTableRoutesWithContext(ctx, listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutingTableRoutesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/1a15dca5-7e33-45e1-b7c5-bc690e569531/routes?start=a5e812a2-62c0-4555-86a5-907106760c56&limit=20"}, "routes": [{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVPCRoutingTableRoutes successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListVPCRoutingTableRoutes(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVPCRoutingTableRoutesOptions model
				listVPCRoutingTableRoutesOptionsModel := new(vpcv1.ListVPCRoutingTableRoutesOptions)
				listVPCRoutingTableRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.RoutingTableID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTableRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVPCRoutingTableRoutes with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutingTableRoutesOptions model
				listVPCRoutingTableRoutesOptionsModel := new(vpcv1.ListVPCRoutingTableRoutesOptions)
				listVPCRoutingTableRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.RoutingTableID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTableRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListVPCRoutingTableRoutesOptions model with no property values
				listVPCRoutingTableRoutesOptionsModelNew := new(vpcv1.ListVPCRoutingTableRoutesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListVPCRoutingTableRoutes successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutingTableRoutesOptions model
				listVPCRoutingTableRoutesOptionsModel := new(vpcv1.ListVPCRoutingTableRoutesOptions)
				listVPCRoutingTableRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.RoutingTableID = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Start = core.StringPtr("testString")
				listVPCRoutingTableRoutesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCRoutingTableRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptions *CreateVPCRoutingTableRouteOptions) - Operation response error`, func() {
		version := "testString"
		createVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPCRoutingTableRoute with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRoutingTableRouteOptions model
				createVPCRoutingTableRouteOptionsModel := new(vpcv1.CreateVPCRoutingTableRouteOptions)
				createVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRoutingTableRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRoutingTableRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRoutingTableRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRoutingTableRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptions *CreateVPCRoutingTableRouteOptions)`, func() {
		version := "testString"
		createVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCRoutingTableRoute successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRoutingTableRouteOptions model
				createVPCRoutingTableRouteOptionsModel := new(vpcv1.CreateVPCRoutingTableRouteOptions)
				createVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRoutingTableRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRoutingTableRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRoutingTableRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRoutingTableRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateVPCRoutingTableRouteWithContext(ctx, createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateVPCRoutingTableRouteWithContext(ctx, createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateVPCRoutingTableRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRoutingTableRouteOptions model
				createVPCRoutingTableRouteOptionsModel := new(vpcv1.CreateVPCRoutingTableRouteOptions)
				createVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRoutingTableRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRoutingTableRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRoutingTableRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRoutingTableRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPCRoutingTableRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRoutingTableRouteOptions model
				createVPCRoutingTableRouteOptionsModel := new(vpcv1.CreateVPCRoutingTableRouteOptions)
				createVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRoutingTableRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRoutingTableRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRoutingTableRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRoutingTableRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVPCRoutingTableRouteOptions model with no property values
				createVPCRoutingTableRouteOptionsModelNew := new(vpcv1.CreateVPCRoutingTableRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the CreateVPCRoutingTableRouteOptions model
				createVPCRoutingTableRouteOptionsModel := new(vpcv1.CreateVPCRoutingTableRouteOptions)
				createVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				createVPCRoutingTableRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRoutingTableRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRoutingTableRouteOptionsModel.Action = core.StringPtr("delegate")
				createVPCRoutingTableRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRoutingTableRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteVPCRoutingTableRoute(deleteVPCRoutingTableRouteOptions *DeleteVPCRoutingTableRouteOptions)`, func() {
		version := "testString"
		deleteVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteVPCRoutingTableRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCRoutingTableRouteOptions model
				deleteVPCRoutingTableRouteOptionsModel := new(vpcv1.DeleteVPCRoutingTableRouteOptions)
				deleteVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				deleteVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteVPCRoutingTableRoute(deleteVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPCRoutingTableRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCRoutingTableRouteOptions model
				deleteVPCRoutingTableRouteOptionsModel := new(vpcv1.DeleteVPCRoutingTableRouteOptions)
				deleteVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				deleteVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteVPCRoutingTableRoute(deleteVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCRoutingTableRouteOptions model with no property values
				deleteVPCRoutingTableRouteOptionsModelNew := new(vpcv1.DeleteVPCRoutingTableRouteOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteVPCRoutingTableRoute(deleteVPCRoutingTableRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptions *GetVPCRoutingTableRouteOptions) - Operation response error`, func() {
		version := "testString"
		getVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCRoutingTableRoute with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRoutingTableRouteOptions model
				getVPCRoutingTableRouteOptionsModel := new(vpcv1.GetVPCRoutingTableRouteOptions)
				getVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptions *GetVPCRoutingTableRouteOptions)`, func() {
		version := "testString"
		getVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCRoutingTableRoute successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCRoutingTableRouteOptions model
				getVPCRoutingTableRouteOptionsModel := new(vpcv1.GetVPCRoutingTableRouteOptions)
				getVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetVPCRoutingTableRouteWithContext(ctx, getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetVPCRoutingTableRouteWithContext(ctx, getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetVPCRoutingTableRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCRoutingTableRouteOptions model
				getVPCRoutingTableRouteOptionsModel := new(vpcv1.GetVPCRoutingTableRouteOptions)
				getVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCRoutingTableRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRoutingTableRouteOptions model
				getVPCRoutingTableRouteOptionsModel := new(vpcv1.GetVPCRoutingTableRouteOptions)
				getVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCRoutingTableRouteOptions model with no property values
				getVPCRoutingTableRouteOptionsModelNew := new(vpcv1.GetVPCRoutingTableRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetVPCRoutingTableRouteOptions model
				getVPCRoutingTableRouteOptionsModel := new(vpcv1.GetVPCRoutingTableRouteOptions)
				getVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptions *UpdateVPCRoutingTableRouteOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPCRoutingTableRoute with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableRouteOptions model
				updateVPCRoutingTableRouteOptionsModel := new(vpcv1.UpdateVPCRoutingTableRouteOptions)
				updateVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptions *UpdateVPCRoutingTableRouteOptions)`, func() {
		version := "testString"
		updateVPCRoutingTableRoutePath := "/vpcs/testString/routing_tables/testString/routes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCRoutingTableRoute successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableRouteOptions model
				updateVPCRoutingTableRouteOptionsModel := new(vpcv1.UpdateVPCRoutingTableRouteOptions)
				updateVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateVPCRoutingTableRouteWithContext(ctx, updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateVPCRoutingTableRouteWithContext(ctx, updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutingTableRoutePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateVPCRoutingTableRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableRouteOptions model
				updateVPCRoutingTableRouteOptionsModel := new(vpcv1.UpdateVPCRoutingTableRouteOptions)
				updateVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPCRoutingTableRoute with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableRouteOptions model
				updateVPCRoutingTableRouteOptionsModel := new(vpcv1.UpdateVPCRoutingTableRouteOptions)
				updateVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCRoutingTableRouteOptions model with no property values
				updateVPCRoutingTableRouteOptionsModelNew := new(vpcv1.UpdateVPCRoutingTableRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateVPCRoutingTableRoute successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRoutingTableRouteOptions model
				updateVPCRoutingTableRouteOptionsModel := new(vpcv1.UpdateVPCRoutingTableRouteOptions)
				updateVPCRoutingTableRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutingTableID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRoutingTableRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRoutingTableRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListSubnets(listSubnetsOptions *ListSubnetsOptions) - Operation response error`, func() {
		version := "testString"
		listSubnetsPath := "/subnets"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["routing_table.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["routing_table.name"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListSubnets with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableName = core.StringPtr("testString")
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListSubnets(listSubnetsOptions *ListSubnetsOptions)`, func() {
		version := "testString"
		listSubnetsPath := "/subnets"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["routing_table.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["routing_table.name"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "subnets": [{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListSubnets successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableName = core.StringPtr("testString")
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListSubnetsWithContext(ctx, listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListSubnetsWithContext(ctx, listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["routing_table.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["routing_table.name"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "subnets": [{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListSubnets successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListSubnets(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableName = core.StringPtr("testString")
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListSubnets with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableName = core.StringPtr("testString")
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListSubnets successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableID = core.StringPtr("testString")
				listSubnetsOptionsModel.RoutingTableName = core.StringPtr("testString")
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateSubnet(createSubnetOptions *CreateSubnetOptions) - Operation response error`, func() {
		version := "testString"
		createSubnetPath := "/subnets"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateSubnet with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.IPVersion = core.StringPtr("ipv4")
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.ResourceGroup = resourceGroupIdentityModel
				subnetPrototypeModel.RoutingTable = routingTableIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateSubnet(createSubnetOptions *CreateSubnetOptions)`, func() {
		version := "testString"
		createSubnetPath := "/subnets"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateSubnet successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.IPVersion = core.StringPtr("ipv4")
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.ResourceGroup = resourceGroupIdentityModel
				subnetPrototypeModel.RoutingTable = routingTableIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateSubnetWithContext(ctx, createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateSubnetWithContext(ctx, createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.IPVersion = core.StringPtr("ipv4")
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.ResourceGroup = resourceGroupIdentityModel
				subnetPrototypeModel.RoutingTable = routingTableIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateSubnet with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.IPVersion = core.StringPtr("ipv4")
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.ResourceGroup = resourceGroupIdentityModel
				subnetPrototypeModel.RoutingTable = routingTableIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateSubnetOptions model with no property values
				createSubnetOptionsModelNew := new(vpcv1.CreateSubnetOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateSubnet(createSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.IPVersion = core.StringPtr("ipv4")
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.ResourceGroup = resourceGroupIdentityModel
				subnetPrototypeModel.RoutingTable = routingTableIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteSubnet(deleteSubnetOptions *DeleteSubnetOptions)`, func() {
		version := "testString"
		deleteSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteSubnetPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteSubnetOptions model
				deleteSubnetOptionsModel := new(vpcv1.DeleteSubnetOptions)
				deleteSubnetOptionsModel.ID = core.StringPtr("testString")
				deleteSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteSubnet(deleteSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteSubnet with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteSubnetOptions model
				deleteSubnetOptionsModel := new(vpcv1.DeleteSubnetOptions)
				deleteSubnetOptionsModel.ID = core.StringPtr("testString")
				deleteSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteSubnet(deleteSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteSubnetOptions model with no property values
				deleteSubnetOptionsModelNew := new(vpcv1.DeleteSubnetOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteSubnet(deleteSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnet(getSubnetOptions *GetSubnetOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnet with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnet(getSubnetOptions *GetSubnetOptions)`, func() {
		version := "testString"
		getSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnet successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetSubnetWithContext(ctx, getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetSubnetWithContext(ctx, getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnet with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetOptions model with no property values
				getSubnetOptionsModelNew := new(vpcv1.GetSubnetOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetSubnet(getSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateSubnet(updateSubnetOptions *UpdateSubnetOptions) - Operation response error`, func() {
		version := "testString"
		updateSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateSubnet with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModel.RoutingTable = routingTableIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateSubnet(updateSubnetOptions *UpdateSubnetOptions)`, func() {
		version := "testString"
		updateSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateSubnet successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModel.RoutingTable = routingTableIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateSubnetWithContext(ctx, updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateSubnetWithContext(ctx, updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ip_version": "ipv4", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "routing_table": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-routing-table-1", "resource_type": "routing_table"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModel.RoutingTable = routingTableIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateSubnet with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModel.RoutingTable = routingTableIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateSubnetOptions model with no property values
				updateSubnetOptionsModelNew := new(vpcv1.UpdateSubnetOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateSubnet(updateSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateSubnet successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("6885e83f-03b2-4603-8a86-db2a0f55c840")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModel.RoutingTable = routingTableIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetNetworkACL(getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnetNetworkACL with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetNetworkACL(getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions)`, func() {
		version := "testString"
		getSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"action": "allow", "before": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "destination_port_max": 22, "destination_port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetSubnetNetworkACL successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetSubnetNetworkACLWithContext(ctx, getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetSubnetNetworkACLWithContext(ctx, getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"action": "allow", "before": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "destination_port_max": 22, "destination_port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetSubnetNetworkACL successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetSubnetNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnetNetworkACL with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetNetworkACLOptions model with no property values
				getSubnetNetworkACLOptionsModelNew := new(vpcv1.GetSubnetNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetSubnetNetworkACL successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		replaceSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions)`, func() {
		version := "testString"
		replaceSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"action": "allow", "before": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "destination_port_max": 22, "destination_port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ReplaceSubnetNetworkACLWithContext(ctx, replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ReplaceSubnetNetworkACLWithContext(ctx, replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::network-acl:a4e28308-8ee7-46ab-8108-9f881f22bdbf", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "rules": [{"action": "allow", "before": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "destination_port_max": 22, "destination_port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ReplaceSubnetNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceSubnetNetworkACL with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceSubnetNetworkACLOptions model with no property values
				replaceSubnetNetworkACLOptionsModelNew := new(vpcv1.ReplaceSubnetNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptions *UnsetSubnetPublicGatewayOptions)`, func() {
		version := "testString"
		unsetSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(unsetSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke UnsetSubnetPublicGateway successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.UnsetSubnetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the UnsetSubnetPublicGatewayOptions model
				unsetSubnetPublicGatewayOptionsModel := new(vpcv1.UnsetSubnetPublicGatewayOptions)
				unsetSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				unsetSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke UnsetSubnetPublicGateway with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the UnsetSubnetPublicGatewayOptions model
				unsetSubnetPublicGatewayOptionsModel := new(vpcv1.UnsetSubnetPublicGatewayOptions)
				unsetSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				unsetSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the UnsetSubnetPublicGatewayOptions model with no property values
				unsetSubnetPublicGatewayOptionsModelNew := new(vpcv1.UnsetSubnetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetPublicGateway(getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnetPublicGateway with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetPublicGateway(getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions)`, func() {
		version := "testString"
		getSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnetPublicGateway successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetSubnetPublicGatewayWithContext(ctx, getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetSubnetPublicGatewayWithContext(ctx, getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnetPublicGateway successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetSubnetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnetPublicGateway with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetPublicGatewayOptions model with no property values
				getSubnetPublicGatewayOptionsModelNew := new(vpcv1.GetSubnetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetSubnetPublicGateway successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`SetSubnetPublicGateway(setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		setSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(setSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke SetSubnetPublicGateway with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`SetSubnetPublicGateway(setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions)`, func() {
		version := "testString"
		setSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(setSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke SetSubnetPublicGateway successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.SetSubnetPublicGatewayWithContext(ctx, setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.SetSubnetPublicGatewayWithContext(ctx, setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(setSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke SetSubnetPublicGateway successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.SetSubnetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke SetSubnetPublicGateway with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the SetSubnetPublicGatewayOptions model with no property values
				setSubnetPublicGatewayOptionsModelNew := new(vpcv1.SetSubnetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke SetSubnetPublicGateway successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetRoutingTable(getSubnetRoutingTableOptions *GetSubnetRoutingTableOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetRoutingTablePath := "/subnets/testString/routing_table"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnetRoutingTable with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetRoutingTableOptions model
				getSubnetRoutingTableOptionsModel := new(vpcv1.GetSubnetRoutingTableOptions)
				getSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetRoutingTable(getSubnetRoutingTableOptions *GetSubnetRoutingTableOptions)`, func() {
		version := "testString"
		getSubnetRoutingTablePath := "/subnets/testString/routing_table"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetSubnetRoutingTable successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetRoutingTableOptions model
				getSubnetRoutingTableOptionsModel := new(vpcv1.GetSubnetRoutingTableOptions)
				getSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetSubnetRoutingTableWithContext(ctx, getSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetSubnetRoutingTableWithContext(ctx, getSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetRoutingTablePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetSubnetRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetSubnetRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetRoutingTableOptions model
				getSubnetRoutingTableOptionsModel := new(vpcv1.GetSubnetRoutingTableOptions)
				getSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnetRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetRoutingTableOptions model
				getSubnetRoutingTableOptionsModel := new(vpcv1.GetSubnetRoutingTableOptions)
				getSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetRoutingTableOptions model with no property values
				getSubnetRoutingTableOptionsModelNew := new(vpcv1.GetSubnetRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetSubnetRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetRoutingTableOptions model
				getSubnetRoutingTableOptionsModel := new(vpcv1.GetSubnetRoutingTableOptions)
				getSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				getSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetSubnetRoutingTable(getSubnetRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptions *ReplaceSubnetRoutingTableOptions) - Operation response error`, func() {
		version := "testString"
		replaceSubnetRoutingTablePath := "/subnets/testString/routing_table"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetRoutingTablePath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceSubnetRoutingTable with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("1a15dca5-7e33-45e1-b7c5-bc690e569531")

				// Construct an instance of the ReplaceSubnetRoutingTableOptions model
				replaceSubnetRoutingTableOptionsModel := new(vpcv1.ReplaceSubnetRoutingTableOptions)
				replaceSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetRoutingTableOptionsModel.RoutingTableIdentity = routingTableIdentityModel
				replaceSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptions *ReplaceSubnetRoutingTableOptions)`, func() {
		version := "testString"
		replaceSubnetRoutingTablePath := "/subnets/testString/routing_table"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetRoutingTablePath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke ReplaceSubnetRoutingTable successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("1a15dca5-7e33-45e1-b7c5-bc690e569531")

				// Construct an instance of the ReplaceSubnetRoutingTableOptions model
				replaceSubnetRoutingTableOptionsModel := new(vpcv1.ReplaceSubnetRoutingTableOptions)
				replaceSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetRoutingTableOptionsModel.RoutingTableIdentity = routingTableIdentityModel
				replaceSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ReplaceSubnetRoutingTableWithContext(ctx, replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ReplaceSubnetRoutingTableWithContext(ctx, replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetRoutingTablePath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/982d72b7-db1b-4606-afb2-ed6bd4b0bed1/routing_tables/6885e83f-03b2-4603-8a86-db2a0f55c840", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "lifecycle_state": "stable", "name": "my-routing-table-1", "resource_type": "routing_table", "route_direct_link_ingress": true, "route_transit_gateway_ingress": true, "route_vpc_zone_ingress": false, "routes": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "name": "my-route-1"}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke ReplaceSubnetRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ReplaceSubnetRoutingTable(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("1a15dca5-7e33-45e1-b7c5-bc690e569531")

				// Construct an instance of the ReplaceSubnetRoutingTableOptions model
				replaceSubnetRoutingTableOptionsModel := new(vpcv1.ReplaceSubnetRoutingTableOptions)
				replaceSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetRoutingTableOptionsModel.RoutingTableIdentity = routingTableIdentityModel
				replaceSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceSubnetRoutingTable with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("1a15dca5-7e33-45e1-b7c5-bc690e569531")

				// Construct an instance of the ReplaceSubnetRoutingTableOptions model
				replaceSubnetRoutingTableOptionsModel := new(vpcv1.ReplaceSubnetRoutingTableOptions)
				replaceSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetRoutingTableOptionsModel.RoutingTableIdentity = routingTableIdentityModel
				replaceSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceSubnetRoutingTableOptions model with no property values
				replaceSubnetRoutingTableOptionsModelNew := new(vpcv1.ReplaceSubnetRoutingTableOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke ReplaceSubnetRoutingTable successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the RoutingTableIdentityByID model
				routingTableIdentityModel := new(vpcv1.RoutingTableIdentityByID)
				routingTableIdentityModel.ID = core.StringPtr("1a15dca5-7e33-45e1-b7c5-bc690e569531")

				// Construct an instance of the ReplaceSubnetRoutingTableOptions model
				replaceSubnetRoutingTableOptionsModel := new(vpcv1.ReplaceSubnetRoutingTableOptions)
				replaceSubnetRoutingTableOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetRoutingTableOptionsModel.RoutingTableIdentity = routingTableIdentityModel
				replaceSubnetRoutingTableOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListSubnetReservedIps(listSubnetReservedIpsOptions *ListSubnetReservedIpsOptions) - Operation response error`, func() {
		version := "testString"
		listSubnetReservedIpsPath := "/subnets/testString/reserved_ips"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetReservedIpsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"name"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListSubnetReservedIps with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListSubnetReservedIpsOptions model
				listSubnetReservedIpsOptionsModel := new(vpcv1.ListSubnetReservedIpsOptions)
				listSubnetReservedIpsOptionsModel.SubnetID = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Start = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetReservedIpsOptionsModel.Sort = core.StringPtr("name")
				listSubnetReservedIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListSubnetReservedIps(listSubnetReservedIpsOptions *ListSubnetReservedIpsOptions)`, func() {
		version := "testString"
		listSubnetReservedIpsPath := "/subnets/testString/reserved_ips"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetReservedIpsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"name"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "reserved_ips": [{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListSubnetReservedIps successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListSubnetReservedIpsOptions model
				listSubnetReservedIpsOptionsModel := new(vpcv1.ListSubnetReservedIpsOptions)
				listSubnetReservedIpsOptionsModel.SubnetID = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Start = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetReservedIpsOptionsModel.Sort = core.StringPtr("name")
				listSubnetReservedIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListSubnetReservedIpsWithContext(ctx, listSubnetReservedIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListSubnetReservedIpsWithContext(ctx, listSubnetReservedIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetReservedIpsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["sort"]).To(Equal([]string{"name"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "reserved_ips": [{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListSubnetReservedIps successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListSubnetReservedIps(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListSubnetReservedIpsOptions model
				listSubnetReservedIpsOptionsModel := new(vpcv1.ListSubnetReservedIpsOptions)
				listSubnetReservedIpsOptionsModel.SubnetID = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Start = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetReservedIpsOptionsModel.Sort = core.StringPtr("name")
				listSubnetReservedIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListSubnetReservedIps with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListSubnetReservedIpsOptions model
				listSubnetReservedIpsOptionsModel := new(vpcv1.ListSubnetReservedIpsOptions)
				listSubnetReservedIpsOptionsModel.SubnetID = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Start = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetReservedIpsOptionsModel.Sort = core.StringPtr("name")
				listSubnetReservedIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListSubnetReservedIpsOptions model with no property values
				listSubnetReservedIpsOptionsModelNew := new(vpcv1.ListSubnetReservedIpsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListSubnetReservedIps successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListSubnetReservedIpsOptions model
				listSubnetReservedIpsOptionsModel := new(vpcv1.ListSubnetReservedIpsOptions)
				listSubnetReservedIpsOptionsModel.SubnetID = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Start = core.StringPtr("testString")
				listSubnetReservedIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetReservedIpsOptionsModel.Sort = core.StringPtr("name")
				listSubnetReservedIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListSubnetReservedIps(listSubnetReservedIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateSubnetReservedIP(createSubnetReservedIPOptions *CreateSubnetReservedIPOptions) - Operation response error`, func() {
		version := "testString"
		createSubnetReservedIPPath := "/subnets/testString/reserved_ips"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetReservedIPPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateSubnetReservedIP with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID model
				reservedIPTargetPrototypeModel := new(vpcv1.ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID)
				reservedIPTargetPrototypeModel.ID = core.StringPtr("d7cc5196-9864-48c4-82d8-3f30da41fcc5")

				// Construct an instance of the CreateSubnetReservedIPOptions model
				createSubnetReservedIPOptionsModel := new(vpcv1.CreateSubnetReservedIPOptions)
				createSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				createSubnetReservedIPOptionsModel.AutoDelete = core.BoolPtr(false)
				createSubnetReservedIPOptionsModel.Name = core.StringPtr("my-reserved-ip")
				createSubnetReservedIPOptionsModel.Target = reservedIPTargetPrototypeModel
				createSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateSubnetReservedIP(createSubnetReservedIPOptions *CreateSubnetReservedIPOptions)`, func() {
		version := "testString"
		createSubnetReservedIPPath := "/subnets/testString/reserved_ips"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetReservedIPPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}`)
				}))
			})
			It(`Invoke CreateSubnetReservedIP successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID model
				reservedIPTargetPrototypeModel := new(vpcv1.ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID)
				reservedIPTargetPrototypeModel.ID = core.StringPtr("d7cc5196-9864-48c4-82d8-3f30da41fcc5")

				// Construct an instance of the CreateSubnetReservedIPOptions model
				createSubnetReservedIPOptionsModel := new(vpcv1.CreateSubnetReservedIPOptions)
				createSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				createSubnetReservedIPOptionsModel.AutoDelete = core.BoolPtr(false)
				createSubnetReservedIPOptionsModel.Name = core.StringPtr("my-reserved-ip")
				createSubnetReservedIPOptionsModel.Target = reservedIPTargetPrototypeModel
				createSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateSubnetReservedIPWithContext(ctx, createSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateSubnetReservedIPWithContext(ctx, createSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetReservedIPPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}`)
				}))
			})
			It(`Invoke CreateSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateSubnetReservedIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID model
				reservedIPTargetPrototypeModel := new(vpcv1.ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID)
				reservedIPTargetPrototypeModel.ID = core.StringPtr("d7cc5196-9864-48c4-82d8-3f30da41fcc5")

				// Construct an instance of the CreateSubnetReservedIPOptions model
				createSubnetReservedIPOptionsModel := new(vpcv1.CreateSubnetReservedIPOptions)
				createSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				createSubnetReservedIPOptionsModel.AutoDelete = core.BoolPtr(false)
				createSubnetReservedIPOptionsModel.Name = core.StringPtr("my-reserved-ip")
				createSubnetReservedIPOptionsModel.Target = reservedIPTargetPrototypeModel
				createSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateSubnetReservedIP with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID model
				reservedIPTargetPrototypeModel := new(vpcv1.ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID)
				reservedIPTargetPrototypeModel.ID = core.StringPtr("d7cc5196-9864-48c4-82d8-3f30da41fcc5")

				// Construct an instance of the CreateSubnetReservedIPOptions model
				createSubnetReservedIPOptionsModel := new(vpcv1.CreateSubnetReservedIPOptions)
				createSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				createSubnetReservedIPOptionsModel.AutoDelete = core.BoolPtr(false)
				createSubnetReservedIPOptionsModel.Name = core.StringPtr("my-reserved-ip")
				createSubnetReservedIPOptionsModel.Target = reservedIPTargetPrototypeModel
				createSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateSubnetReservedIPOptions model with no property values
				createSubnetReservedIPOptionsModelNew := new(vpcv1.CreateSubnetReservedIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID model
				reservedIPTargetPrototypeModel := new(vpcv1.ReservedIPTargetPrototypeEndpointGatewayIdentityEndpointGatewayIdentityByID)
				reservedIPTargetPrototypeModel.ID = core.StringPtr("d7cc5196-9864-48c4-82d8-3f30da41fcc5")

				// Construct an instance of the CreateSubnetReservedIPOptions model
				createSubnetReservedIPOptionsModel := new(vpcv1.CreateSubnetReservedIPOptions)
				createSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				createSubnetReservedIPOptionsModel.AutoDelete = core.BoolPtr(false)
				createSubnetReservedIPOptionsModel.Name = core.StringPtr("my-reserved-ip")
				createSubnetReservedIPOptionsModel.Target = reservedIPTargetPrototypeModel
				createSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateSubnetReservedIP(createSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteSubnetReservedIP(deleteSubnetReservedIPOptions *DeleteSubnetReservedIPOptions)`, func() {
		version := "testString"
		deleteSubnetReservedIPPath := "/subnets/testString/reserved_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteSubnetReservedIPPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteSubnetReservedIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteSubnetReservedIPOptions model
				deleteSubnetReservedIPOptionsModel := new(vpcv1.DeleteSubnetReservedIPOptions)
				deleteSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				deleteSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				deleteSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteSubnetReservedIP(deleteSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteSubnetReservedIP with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteSubnetReservedIPOptions model
				deleteSubnetReservedIPOptionsModel := new(vpcv1.DeleteSubnetReservedIPOptions)
				deleteSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				deleteSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				deleteSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteSubnetReservedIP(deleteSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteSubnetReservedIPOptions model with no property values
				deleteSubnetReservedIPOptionsModelNew := new(vpcv1.DeleteSubnetReservedIPOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteSubnetReservedIP(deleteSubnetReservedIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetReservedIP(getSubnetReservedIPOptions *GetSubnetReservedIPOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetReservedIPPath := "/subnets/testString/reserved_ips/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetReservedIPPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnetReservedIP with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetReservedIPOptions model
				getSubnetReservedIPOptionsModel := new(vpcv1.GetSubnetReservedIPOptions)
				getSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetReservedIP(getSubnetReservedIPOptions *GetSubnetReservedIPOptions)`, func() {
		version := "testString"
		getSubnetReservedIPPath := "/subnets/testString/reserved_ips/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetReservedIPPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}`)
				}))
			})
			It(`Invoke GetSubnetReservedIP successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetReservedIPOptions model
				getSubnetReservedIPOptionsModel := new(vpcv1.GetSubnetReservedIPOptions)
				getSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetSubnetReservedIPWithContext(ctx, getSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetSubnetReservedIPWithContext(ctx, getSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetReservedIPPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}`)
				}))
			})
			It(`Invoke GetSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetSubnetReservedIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetReservedIPOptions model
				getSubnetReservedIPOptionsModel := new(vpcv1.GetSubnetReservedIPOptions)
				getSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnetReservedIP with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetReservedIPOptions model
				getSubnetReservedIPOptionsModel := new(vpcv1.GetSubnetReservedIPOptions)
				getSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetReservedIPOptions model with no property values
				getSubnetReservedIPOptionsModelNew := new(vpcv1.GetSubnetReservedIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetSubnetReservedIPOptions model
				getSubnetReservedIPOptionsModel := new(vpcv1.GetSubnetReservedIPOptions)
				getSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				getSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetSubnetReservedIP(getSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateSubnetReservedIP(updateSubnetReservedIPOptions *UpdateSubnetReservedIPOptions) - Operation response error`, func() {
		version := "testString"
		updateSubnetReservedIPPath := "/subnets/testString/reserved_ips/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetReservedIPPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateSubnetReservedIP with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ReservedIPPatch model
				reservedIPPatchModel := new(vpcv1.ReservedIPPatch)
				reservedIPPatchModel.AutoDelete = core.BoolPtr(false)
				reservedIPPatchModel.Name = core.StringPtr("my-reserved-ip")
				reservedIPPatchModelAsPatch, asPatchErr := reservedIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetReservedIPOptions model
				updateSubnetReservedIPOptionsModel := new(vpcv1.UpdateSubnetReservedIPOptions)
				updateSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ReservedIPPatch = reservedIPPatchModelAsPatch
				updateSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateSubnetReservedIP(updateSubnetReservedIPOptions *UpdateSubnetReservedIPOptions)`, func() {
		version := "testString"
		updateSubnetReservedIPPath := "/subnets/testString/reserved_ips/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetReservedIPPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}`)
				}))
			})
			It(`Invoke UpdateSubnetReservedIP successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ReservedIPPatch model
				reservedIPPatchModel := new(vpcv1.ReservedIPPatch)
				reservedIPPatchModel.AutoDelete = core.BoolPtr(false)
				reservedIPPatchModel.Name = core.StringPtr("my-reserved-ip")
				reservedIPPatchModelAsPatch, asPatchErr := reservedIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetReservedIPOptions model
				updateSubnetReservedIPOptionsModel := new(vpcv1.UpdateSubnetReservedIPOptions)
				updateSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ReservedIPPatch = reservedIPPatchModelAsPatch
				updateSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateSubnetReservedIPWithContext(ctx, updateSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateSubnetReservedIPWithContext(ctx, updateSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetReservedIPPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "192.168.3.4", "auto_delete": true, "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e/reserved_ips/6d353a0f-aeb1-4ae1-832e-1110d10981bb", "id": "6d353a0f-aeb1-4ae1-832e-1110d10981bb", "name": "my-reserved-ip", "owner": "user", "resource_type": "subnet_reserved_ip", "target": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::endpoint-gateway:d7cc5196-9864-48c4-82d8-3f30da41fcc5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/endpoint_gateways/d7cc5196-9864-48c4-82d8-3f30da41fcc5", "id": "d7cc5196-9864-48c4-82d8-3f30da41fcc5", "name": "my-endpoint-gateway", "resource_type": "endpoint_gateway"}}`)
				}))
			})
			It(`Invoke UpdateSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateSubnetReservedIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ReservedIPPatch model
				reservedIPPatchModel := new(vpcv1.ReservedIPPatch)
				reservedIPPatchModel.AutoDelete = core.BoolPtr(false)
				reservedIPPatchModel.Name = core.StringPtr("my-reserved-ip")
				reservedIPPatchModelAsPatch, asPatchErr := reservedIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetReservedIPOptions model
				updateSubnetReservedIPOptionsModel := new(vpcv1.UpdateSubnetReservedIPOptions)
				updateSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ReservedIPPatch = reservedIPPatchModelAsPatch
				updateSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateSubnetReservedIP with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ReservedIPPatch model
				reservedIPPatchModel := new(vpcv1.ReservedIPPatch)
				reservedIPPatchModel.AutoDelete = core.BoolPtr(false)
				reservedIPPatchModel.Name = core.StringPtr("my-reserved-ip")
				reservedIPPatchModelAsPatch, asPatchErr := reservedIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetReservedIPOptions model
				updateSubnetReservedIPOptionsModel := new(vpcv1.UpdateSubnetReservedIPOptions)
				updateSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ReservedIPPatch = reservedIPPatchModelAsPatch
				updateSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateSubnetReservedIPOptions model with no property values
				updateSubnetReservedIPOptionsModelNew := new(vpcv1.UpdateSubnetReservedIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateSubnetReservedIP successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ReservedIPPatch model
				reservedIPPatchModel := new(vpcv1.ReservedIPPatch)
				reservedIPPatchModel.AutoDelete = core.BoolPtr(false)
				reservedIPPatchModel.Name = core.StringPtr("my-reserved-ip")
				reservedIPPatchModelAsPatch, asPatchErr := reservedIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetReservedIPOptions model
				updateSubnetReservedIPOptionsModel := new(vpcv1.UpdateSubnetReservedIPOptions)
				updateSubnetReservedIPOptionsModel.SubnetID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ID = core.StringPtr("testString")
				updateSubnetReservedIPOptionsModel.ReservedIPPatch = reservedIPPatchModelAsPatch
				updateSubnetReservedIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateSubnetReservedIP(updateSubnetReservedIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListImages(listImagesOptions *ListImagesOptions) - Operation response error`, func() {
		version := "testString"
		listImagesPath := "/images"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImagesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["visibility"]).To(Equal([]string{"private"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListImages with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListImages(listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListImages(listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListImages(listImagesOptions *ListImagesOptions)`, func() {
		version := "testString"
		listImagesPath := "/images"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImagesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["visibility"]).To(Equal([]string{"private"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?limit=20"}, "images": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListImages successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListImagesWithContext(ctx, listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListImages(listImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListImagesWithContext(ctx, listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImagesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["visibility"]).To(Equal([]string{"private"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?limit=20"}, "images": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListImages successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListImages(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListImages(listImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListImages with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListImages(listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListImages successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListImages(listImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateImage(createImageOptions *CreateImageOptions) - Operation response error`, func() {
		version := "testString"
		createImagePath := "/images"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createImagePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateImage with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.EncryptedDataKey = core.StringPtr("testString")
				imagePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateImage(createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateImage(createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateImage(createImageOptions *CreateImageOptions)`, func() {
		version := "testString"
		createImagePath := "/images"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createImagePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}`)
				}))
			})
			It(`Invoke CreateImage successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.EncryptedDataKey = core.StringPtr("testString")
				imagePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateImageWithContext(ctx, createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateImage(createImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateImageWithContext(ctx, createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createImagePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}`)
				}))
			})
			It(`Invoke CreateImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.EncryptedDataKey = core.StringPtr("testString")
				imagePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateImage(createImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateImage with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.EncryptedDataKey = core.StringPtr("testString")
				imagePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateImage(createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateImageOptions model with no property values
				createImageOptionsModelNew := new(vpcv1.CreateImageOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateImage(createImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.EncryptedDataKey = core.StringPtr("testString")
				imagePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateImage(createImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteImage(deleteImageOptions *DeleteImageOptions)`, func() {
		version := "testString"
		deleteImagePath := "/images/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteImagePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(202)
				}))
			})
			It(`Invoke DeleteImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteImageOptions model
				deleteImageOptionsModel := new(vpcv1.DeleteImageOptions)
				deleteImageOptionsModel.ID = core.StringPtr("testString")
				deleteImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteImage(deleteImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteImage with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteImageOptions model
				deleteImageOptionsModel := new(vpcv1.DeleteImageOptions)
				deleteImageOptionsModel.ID = core.StringPtr("testString")
				deleteImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteImage(deleteImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteImageOptions model with no property values
				deleteImageOptionsModelNew := new(vpcv1.DeleteImageOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteImage(deleteImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetImage(getImageOptions *GetImageOptions) - Operation response error`, func() {
		version := "testString"
		getImagePath := "/images/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getImagePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetImage with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetImage(getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetImage(getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetImage(getImageOptions *GetImageOptions)`, func() {
		version := "testString"
		getImagePath := "/images/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getImagePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}`)
				}))
			})
			It(`Invoke GetImage successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetImageWithContext(ctx, getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetImage(getImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetImageWithContext(ctx, getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getImagePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}`)
				}))
			})
			It(`Invoke GetImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetImage(getImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetImage with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetImage(getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetImageOptions model with no property values
				getImageOptionsModelNew := new(vpcv1.GetImageOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetImage(getImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetImage(getImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateImage(updateImageOptions *UpdateImageOptions) - Operation response error`, func() {
		version := "testString"
		updateImagePath := "/images/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateImagePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateImage with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateImage(updateImageOptions *UpdateImageOptions)`, func() {
		version := "testString"
		updateImagePath := "/images/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateImagePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}`)
				}))
			})
			It(`Invoke UpdateImage successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateImageWithContext(ctx, updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateImageWithContext(ctx, updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateImagePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "encryption": "user_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "file": {"checksums": {"sha256": "e992a84f113d3a35d2145ca3e7aca4fc95fe6daf470a08d8af3422ee59c92e15"}, "size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "source_volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}, "status": "available", "status_reasons": [{"code": "encryption_key_deleted", "message": "Message", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "visibility": "private"}`)
				}))
			})
			It(`Invoke UpdateImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateImage with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateImageOptions model with no property values
				updateImageOptionsModelNew := new(vpcv1.UpdateImageOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateImage(updateImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateImage successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOperatingSystems(listOperatingSystemsOptions *ListOperatingSystemsOptions) - Operation response error`, func() {
		version := "testString"
		listOperatingSystemsPath := "/operating_systems"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatingSystemsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListOperatingSystems with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOperatingSystems(listOperatingSystemsOptions *ListOperatingSystemsOptions)`, func() {
		version := "testString"
		listOperatingSystemsPath := "/operating_systems"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatingSystemsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "operating_systems": [{"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}]}`)
				}))
			})
			It(`Invoke ListOperatingSystems successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListOperatingSystemsWithContext(ctx, listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListOperatingSystemsWithContext(ctx, listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatingSystemsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "operating_systems": [{"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}]}`)
				}))
			})
			It(`Invoke ListOperatingSystems successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListOperatingSystems(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListOperatingSystems with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListOperatingSystems successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOperatingSystem(getOperatingSystemOptions *GetOperatingSystemOptions) - Operation response error`, func() {
		version := "testString"
		getOperatingSystemPath := "/operating_systems/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOperatingSystemPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOperatingSystem with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOperatingSystem(getOperatingSystemOptions *GetOperatingSystemOptions)`, func() {
		version := "testString"
		getOperatingSystemPath := "/operating_systems/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOperatingSystemPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}`)
				}))
			})
			It(`Invoke GetOperatingSystem successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetOperatingSystemWithContext(ctx, getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetOperatingSystemWithContext(ctx, getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOperatingSystemPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"architecture": "amd64", "dedicated_host_only": false, "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}`)
				}))
			})
			It(`Invoke GetOperatingSystem successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetOperatingSystem(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOperatingSystem with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOperatingSystemOptions model with no property values
				getOperatingSystemOptionsModelNew := new(vpcv1.GetOperatingSystemOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetOperatingSystem(getOperatingSystemOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetOperatingSystem successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListKeys(listKeysOptions *ListKeysOptions) - Operation response error`, func() {
		version := "testString"
		listKeysPath := "/keys"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listKeysPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListKeys with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcv1.ListKeysOptions)
				listKeysOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListKeys(listKeysOptions *ListKeysOptions)`, func() {
		version := "testString"
		listKeysPath := "/keys"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listKeysPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"keys": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}]}`)
				}))
			})
			It(`Invoke ListKeys successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcv1.ListKeysOptions)
				listKeysOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListKeysWithContext(ctx, listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListKeysWithContext(ctx, listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listKeysPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"keys": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}]}`)
				}))
			})
			It(`Invoke ListKeys successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListKeys(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcv1.ListKeysOptions)
				listKeysOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListKeys with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcv1.ListKeysOptions)
				listKeysOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListKeys successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcv1.ListKeysOptions)
				listKeysOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateKey(createKeyOptions *CreateKeyOptions) - Operation response error`, func() {
		version := "testString"
		createKeyPath := "/keys"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createKeyPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateKey with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateKey(createKeyOptions *CreateKeyOptions)`, func() {
		version := "testString"
		createKeyPath := "/keys"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createKeyPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke CreateKey successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateKeyWithContext(ctx, createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateKeyWithContext(ctx, createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createKeyPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke CreateKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateKey with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateKeyOptions model with no property values
				createKeyOptionsModelNew := new(vpcv1.CreateKeyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateKey(createKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteKey(deleteKeyOptions *DeleteKeyOptions)`, func() {
		version := "testString"
		deleteKeyPath := "/keys/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteKeyPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteKeyOptions model
				deleteKeyOptionsModel := new(vpcv1.DeleteKeyOptions)
				deleteKeyOptionsModel.ID = core.StringPtr("testString")
				deleteKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteKey(deleteKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteKey with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteKeyOptions model
				deleteKeyOptionsModel := new(vpcv1.DeleteKeyOptions)
				deleteKeyOptionsModel.ID = core.StringPtr("testString")
				deleteKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteKey(deleteKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteKeyOptions model with no property values
				deleteKeyOptionsModelNew := new(vpcv1.DeleteKeyOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteKey(deleteKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetKey(getKeyOptions *GetKeyOptions) - Operation response error`, func() {
		version := "testString"
		getKeyPath := "/keys/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getKeyPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetKey with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetKey(getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetKey(getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetKey(getKeyOptions *GetKeyOptions)`, func() {
		version := "testString"
		getKeyPath := "/keys/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getKeyPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke GetKey successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetKeyWithContext(ctx, getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetKey(getKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetKeyWithContext(ctx, getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getKeyPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke GetKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetKey(getKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetKey with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetKey(getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetKeyOptions model with no property values
				getKeyOptionsModelNew := new(vpcv1.GetKeyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetKey(getKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetKey(getKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateKey(updateKeyOptions *UpdateKeyOptions) - Operation response error`, func() {
		version := "testString"
		updateKeyPath := "/keys/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateKeyPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateKey with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateKey(updateKeyOptions *UpdateKeyOptions)`, func() {
		version := "testString"
		updateKeyPath := "/keys/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateKeyPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke UpdateKey successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateKeyWithContext(ctx, updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateKeyWithContext(ctx, updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateKeyPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke UpdateKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateKey with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateKeyOptions model with no property values
				updateKeyOptionsModelNew := new(vpcv1.UpdateKeyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateKey(updateKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateKey successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceProfiles(listInstanceProfilesOptions *ListInstanceProfilesOptions) - Operation response error`, func() {
		version := "testString"
		listInstanceProfilesPath := "/instance/profiles"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceProfilesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstanceProfiles with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceProfiles(listInstanceProfilesOptions *ListInstanceProfilesOptions)`, func() {
		version := "testString"
		listInstanceProfilesPath := "/instance/profiles"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceProfilesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"profiles": [{"bandwidth": {"type": "fixed", "value": 20000}, "disks": [{"quantity": {"type": "fixed", "value": 4}, "size": {"type": "fixed", "value": 100}, "supported_interface_types": {"default": "virtio_blk", "type": "enum", "values": ["virtio_blk"]}}], "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "memory": {"type": "fixed", "value": 16}, "name": "bc1-4x16", "os_architecture": {"default": "Default", "type": "enum", "values": ["amd64"]}, "port_speed": {"type": "fixed", "value": 1000}, "vcpu_architecture": {"default": "Default", "type": "fixed", "value": "amd64"}, "vcpu_count": {"type": "fixed", "value": 16}}]}`)
				}))
			})
			It(`Invoke ListInstanceProfiles successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListInstanceProfilesWithContext(ctx, listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListInstanceProfilesWithContext(ctx, listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceProfilesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"profiles": [{"bandwidth": {"type": "fixed", "value": 20000}, "disks": [{"quantity": {"type": "fixed", "value": 4}, "size": {"type": "fixed", "value": 100}, "supported_interface_types": {"default": "virtio_blk", "type": "enum", "values": ["virtio_blk"]}}], "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "memory": {"type": "fixed", "value": 16}, "name": "bc1-4x16", "os_architecture": {"default": "Default", "type": "enum", "values": ["amd64"]}, "port_speed": {"type": "fixed", "value": 1000}, "vcpu_architecture": {"default": "Default", "type": "fixed", "value": "amd64"}, "vcpu_count": {"type": "fixed", "value": 16}}]}`)
				}))
			})
			It(`Invoke ListInstanceProfiles successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListInstanceProfiles(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstanceProfiles with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListInstanceProfiles successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceProfile(getInstanceProfileOptions *GetInstanceProfileOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceProfilePath := "/instance/profiles/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceProfilePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceProfile with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceProfile(getInstanceProfileOptions *GetInstanceProfileOptions)`, func() {
		version := "testString"
		getInstanceProfilePath := "/instance/profiles/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceProfilePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": {"type": "fixed", "value": 20000}, "disks": [{"quantity": {"type": "fixed", "value": 4}, "size": {"type": "fixed", "value": 100}, "supported_interface_types": {"default": "virtio_blk", "type": "enum", "values": ["virtio_blk"]}}], "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "memory": {"type": "fixed", "value": 16}, "name": "bc1-4x16", "os_architecture": {"default": "Default", "type": "enum", "values": ["amd64"]}, "port_speed": {"type": "fixed", "value": 1000}, "vcpu_architecture": {"default": "Default", "type": "fixed", "value": "amd64"}, "vcpu_count": {"type": "fixed", "value": 16}}`)
				}))
			})
			It(`Invoke GetInstanceProfile successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetInstanceProfileWithContext(ctx, getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetInstanceProfileWithContext(ctx, getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceProfilePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": {"type": "fixed", "value": 20000}, "disks": [{"quantity": {"type": "fixed", "value": 4}, "size": {"type": "fixed", "value": 100}, "supported_interface_types": {"default": "virtio_blk", "type": "enum", "values": ["virtio_blk"]}}], "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "memory": {"type": "fixed", "value": 16}, "name": "bc1-4x16", "os_architecture": {"default": "Default", "type": "enum", "values": ["amd64"]}, "port_speed": {"type": "fixed", "value": 1000}, "vcpu_architecture": {"default": "Default", "type": "fixed", "value": "amd64"}, "vcpu_count": {"type": "fixed", "value": 16}}`)
				}))
			})
			It(`Invoke GetInstanceProfile successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetInstanceProfile(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceProfile with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceProfileOptions model with no property values
				getInstanceProfileOptionsModelNew := new(vpcv1.GetInstanceProfileOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetInstanceProfile(getInstanceProfileOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetInstanceProfile successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceTemplates(listInstanceTemplatesOptions *ListInstanceTemplatesOptions) - Operation response error`, func() {
		version := "testString"
		listInstanceTemplatesPath := "/instance/templates"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceTemplatesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstanceTemplates with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstanceTemplatesOptions model
				listInstanceTemplatesOptionsModel := new(vpcv1.ListInstanceTemplatesOptions)
				listInstanceTemplatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListInstanceTemplates(listInstanceTemplatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListInstanceTemplates(listInstanceTemplatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceTemplates(listInstanceTemplatesOptions *ListInstanceTemplatesOptions)`, func() {
		version := "testString"
		listInstanceTemplatesPath := "/instance/templates"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceTemplatesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "templates": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstanceTemplates successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListInstanceTemplatesOptions model
				listInstanceTemplatesOptionsModel := new(vpcv1.ListInstanceTemplatesOptions)
				listInstanceTemplatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListInstanceTemplatesWithContext(ctx, listInstanceTemplatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListInstanceTemplates(listInstanceTemplatesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListInstanceTemplatesWithContext(ctx, listInstanceTemplatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceTemplatesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "templates": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstanceTemplates successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListInstanceTemplates(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstanceTemplatesOptions model
				listInstanceTemplatesOptionsModel := new(vpcv1.ListInstanceTemplatesOptions)
				listInstanceTemplatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListInstanceTemplates(listInstanceTemplatesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstanceTemplates with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstanceTemplatesOptions model
				listInstanceTemplatesOptionsModel := new(vpcv1.ListInstanceTemplatesOptions)
				listInstanceTemplatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListInstanceTemplates(listInstanceTemplatesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListInstanceTemplates successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstanceTemplatesOptions model
				listInstanceTemplatesOptionsModel := new(vpcv1.ListInstanceTemplatesOptions)
				listInstanceTemplatesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListInstanceTemplates(listInstanceTemplatesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceTemplate(createInstanceTemplateOptions *CreateInstanceTemplateOptions) - Operation response error`, func() {
		version := "testString"
		createInstanceTemplatePath := "/instance/templates"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceTemplatePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstanceTemplate with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("1e09281b-f177-46fb-baf1-bc152b2e391a")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID)
				volumeAttachmentVolumePrototypeInstanceContextModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("dc201ab2-8536-4904-86a8-084d84582133")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("general-purpose")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("3f9a2d96-830e-4100-9b4c-663225a3f872")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstanceTemplatePrototypeInstanceByImage model
				instanceTemplatePrototypeModel := new(vpcv1.InstanceTemplatePrototypeInstanceByImage)
				instanceTemplatePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instanceTemplatePrototypeModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instanceTemplatePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instanceTemplatePrototypeModel.Profile = instanceProfileIdentityModel
				instanceTemplatePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instanceTemplatePrototypeModel.UserData = core.StringPtr("testString")
				instanceTemplatePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instanceTemplatePrototypeModel.VPC = vpcIdentityModel
				instanceTemplatePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instanceTemplatePrototypeModel.Image = imageIdentityModel
				instanceTemplatePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instanceTemplatePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceTemplateOptions model
				createInstanceTemplateOptionsModel := new(vpcv1.CreateInstanceTemplateOptions)
				createInstanceTemplateOptionsModel.InstanceTemplatePrototype = instanceTemplatePrototypeModel
				createInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceTemplate(createInstanceTemplateOptions *CreateInstanceTemplateOptions)`, func() {
		version := "testString"
		createInstanceTemplatePath := "/instance/templates"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceTemplatePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateInstanceTemplate successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("1e09281b-f177-46fb-baf1-bc152b2e391a")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID)
				volumeAttachmentVolumePrototypeInstanceContextModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("dc201ab2-8536-4904-86a8-084d84582133")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("general-purpose")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("3f9a2d96-830e-4100-9b4c-663225a3f872")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstanceTemplatePrototypeInstanceByImage model
				instanceTemplatePrototypeModel := new(vpcv1.InstanceTemplatePrototypeInstanceByImage)
				instanceTemplatePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instanceTemplatePrototypeModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instanceTemplatePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instanceTemplatePrototypeModel.Profile = instanceProfileIdentityModel
				instanceTemplatePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instanceTemplatePrototypeModel.UserData = core.StringPtr("testString")
				instanceTemplatePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instanceTemplatePrototypeModel.VPC = vpcIdentityModel
				instanceTemplatePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instanceTemplatePrototypeModel.Image = imageIdentityModel
				instanceTemplatePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instanceTemplatePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceTemplateOptions model
				createInstanceTemplateOptionsModel := new(vpcv1.CreateInstanceTemplateOptions)
				createInstanceTemplateOptionsModel.InstanceTemplatePrototype = instanceTemplatePrototypeModel
				createInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateInstanceTemplateWithContext(ctx, createInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateInstanceTemplateWithContext(ctx, createInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceTemplatePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateInstanceTemplate(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("1e09281b-f177-46fb-baf1-bc152b2e391a")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID)
				volumeAttachmentVolumePrototypeInstanceContextModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("dc201ab2-8536-4904-86a8-084d84582133")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("general-purpose")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("3f9a2d96-830e-4100-9b4c-663225a3f872")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstanceTemplatePrototypeInstanceByImage model
				instanceTemplatePrototypeModel := new(vpcv1.InstanceTemplatePrototypeInstanceByImage)
				instanceTemplatePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instanceTemplatePrototypeModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instanceTemplatePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instanceTemplatePrototypeModel.Profile = instanceProfileIdentityModel
				instanceTemplatePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instanceTemplatePrototypeModel.UserData = core.StringPtr("testString")
				instanceTemplatePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instanceTemplatePrototypeModel.VPC = vpcIdentityModel
				instanceTemplatePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instanceTemplatePrototypeModel.Image = imageIdentityModel
				instanceTemplatePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instanceTemplatePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceTemplateOptions model
				createInstanceTemplateOptionsModel := new(vpcv1.CreateInstanceTemplateOptions)
				createInstanceTemplateOptionsModel.InstanceTemplatePrototype = instanceTemplatePrototypeModel
				createInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstanceTemplate with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("1e09281b-f177-46fb-baf1-bc152b2e391a")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID)
				volumeAttachmentVolumePrototypeInstanceContextModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("dc201ab2-8536-4904-86a8-084d84582133")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("general-purpose")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("3f9a2d96-830e-4100-9b4c-663225a3f872")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstanceTemplatePrototypeInstanceByImage model
				instanceTemplatePrototypeModel := new(vpcv1.InstanceTemplatePrototypeInstanceByImage)
				instanceTemplatePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instanceTemplatePrototypeModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instanceTemplatePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instanceTemplatePrototypeModel.Profile = instanceProfileIdentityModel
				instanceTemplatePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instanceTemplatePrototypeModel.UserData = core.StringPtr("testString")
				instanceTemplatePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instanceTemplatePrototypeModel.VPC = vpcIdentityModel
				instanceTemplatePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instanceTemplatePrototypeModel.Image = imageIdentityModel
				instanceTemplatePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instanceTemplatePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceTemplateOptions model
				createInstanceTemplateOptionsModel := new(vpcv1.CreateInstanceTemplateOptions)
				createInstanceTemplateOptionsModel.InstanceTemplatePrototype = instanceTemplatePrototypeModel
				createInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateInstanceTemplateOptions model with no property values
				createInstanceTemplateOptionsModelNew := new(vpcv1.CreateInstanceTemplateOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("1e09281b-f177-46fb-baf1-bc152b2e391a")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumeIdentityVolumeIdentityByID)
				volumeAttachmentVolumePrototypeInstanceContextModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("dc201ab2-8536-4904-86a8-084d84582133")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("general-purpose")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("3f9a2d96-830e-4100-9b4c-663225a3f872")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstanceTemplatePrototypeInstanceByImage model
				instanceTemplatePrototypeModel := new(vpcv1.InstanceTemplatePrototypeInstanceByImage)
				instanceTemplatePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instanceTemplatePrototypeModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instanceTemplatePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instanceTemplatePrototypeModel.Profile = instanceProfileIdentityModel
				instanceTemplatePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instanceTemplatePrototypeModel.UserData = core.StringPtr("testString")
				instanceTemplatePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instanceTemplatePrototypeModel.VPC = vpcIdentityModel
				instanceTemplatePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instanceTemplatePrototypeModel.Image = imageIdentityModel
				instanceTemplatePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instanceTemplatePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceTemplateOptions model
				createInstanceTemplateOptionsModel := new(vpcv1.CreateInstanceTemplateOptions)
				createInstanceTemplateOptionsModel.InstanceTemplatePrototype = instanceTemplatePrototypeModel
				createInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateInstanceTemplate(createInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteInstanceTemplate(deleteInstanceTemplateOptions *DeleteInstanceTemplateOptions)`, func() {
		version := "testString"
		deleteInstanceTemplatePath := "/instance/templates/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteInstanceTemplatePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteInstanceTemplate(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteInstanceTemplateOptions model
				deleteInstanceTemplateOptionsModel := new(vpcv1.DeleteInstanceTemplateOptions)
				deleteInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteInstanceTemplate(deleteInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteInstanceTemplate with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteInstanceTemplateOptions model
				deleteInstanceTemplateOptionsModel := new(vpcv1.DeleteInstanceTemplateOptions)
				deleteInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteInstanceTemplate(deleteInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteInstanceTemplateOptions model with no property values
				deleteInstanceTemplateOptionsModelNew := new(vpcv1.DeleteInstanceTemplateOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteInstanceTemplate(deleteInstanceTemplateOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceTemplate(getInstanceTemplateOptions *GetInstanceTemplateOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceTemplatePath := "/instance/templates/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceTemplatePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceTemplate with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceTemplateOptions model
				getInstanceTemplateOptionsModel := new(vpcv1.GetInstanceTemplateOptions)
				getInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				getInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceTemplate(getInstanceTemplateOptions *GetInstanceTemplateOptions)`, func() {
		version := "testString"
		getInstanceTemplatePath := "/instance/templates/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceTemplatePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstanceTemplate successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceTemplateOptions model
				getInstanceTemplateOptionsModel := new(vpcv1.GetInstanceTemplateOptions)
				getInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				getInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetInstanceTemplateWithContext(ctx, getInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetInstanceTemplateWithContext(ctx, getInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceTemplatePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetInstanceTemplate(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceTemplateOptions model
				getInstanceTemplateOptionsModel := new(vpcv1.GetInstanceTemplateOptions)
				getInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				getInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceTemplate with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceTemplateOptions model
				getInstanceTemplateOptionsModel := new(vpcv1.GetInstanceTemplateOptions)
				getInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				getInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceTemplateOptions model with no property values
				getInstanceTemplateOptionsModelNew := new(vpcv1.GetInstanceTemplateOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceTemplateOptions model
				getInstanceTemplateOptionsModel := new(vpcv1.GetInstanceTemplateOptions)
				getInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				getInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetInstanceTemplate(getInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateInstanceTemplate(updateInstanceTemplateOptions *UpdateInstanceTemplateOptions) - Operation response error`, func() {
		version := "testString"
		updateInstanceTemplatePath := "/instance/templates/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstanceTemplatePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateInstanceTemplate with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the InstanceTemplatePatch model
				instanceTemplatePatchModel := new(vpcv1.InstanceTemplatePatch)
				instanceTemplatePatchModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePatchModelAsPatch, asPatchErr := instanceTemplatePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceTemplateOptions model
				updateInstanceTemplateOptionsModel := new(vpcv1.UpdateInstanceTemplateOptions)
				updateInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				updateInstanceTemplateOptionsModel.InstanceTemplatePatch = instanceTemplatePatchModelAsPatch
				updateInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateInstanceTemplate(updateInstanceTemplateOptions *UpdateInstanceTemplateOptions)`, func() {
		version := "testString"
		updateInstanceTemplatePath := "/instance/templates/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstanceTemplatePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateInstanceTemplate successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the InstanceTemplatePatch model
				instanceTemplatePatchModel := new(vpcv1.InstanceTemplatePatch)
				instanceTemplatePatchModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePatchModelAsPatch, asPatchErr := instanceTemplatePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceTemplateOptions model
				updateInstanceTemplateOptionsModel := new(vpcv1.UpdateInstanceTemplateOptions)
				updateInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				updateInstanceTemplateOptionsModel.InstanceTemplatePatch = instanceTemplatePatchModelAsPatch
				updateInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateInstanceTemplateWithContext(ctx, updateInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateInstanceTemplateWithContext(ctx, updateInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstanceTemplatePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance-template:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/templates/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "keys": [{"id": "a6b1a881-2ce8-41a3-80fc-36316a73f803"}], "name": "my-instance-template", "network_interfaces": [{"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}], "placement_target": {"id": "1e09281b-f177-46fb-baf1-bc152b2e391a"}, "profile": {"name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "user_data": "UserData", "volume_attachments": [{"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5"}}], "vpc": {"id": "4727d842-f94f-4a2d-824a-9bc9b02c523b"}, "boot_volume_attachment": {"delete_volume_on_instance_delete": true, "name": "my-volume-attachment", "volume": {"capacity": 100, "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "iops": 10000, "name": "my-volume", "profile": {"name": "general-purpose"}}}, "image": {"id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8"}, "primary_network_interface": {"allow_ip_spoofing": true, "name": "my-network-interface", "primary_ipv4_address": "10.0.0.5", "security_groups": [{"id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271"}], "subnet": {"id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e"}}, "zone": {"name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateInstanceTemplate(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the InstanceTemplatePatch model
				instanceTemplatePatchModel := new(vpcv1.InstanceTemplatePatch)
				instanceTemplatePatchModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePatchModelAsPatch, asPatchErr := instanceTemplatePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceTemplateOptions model
				updateInstanceTemplateOptionsModel := new(vpcv1.UpdateInstanceTemplateOptions)
				updateInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				updateInstanceTemplateOptionsModel.InstanceTemplatePatch = instanceTemplatePatchModelAsPatch
				updateInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateInstanceTemplate with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the InstanceTemplatePatch model
				instanceTemplatePatchModel := new(vpcv1.InstanceTemplatePatch)
				instanceTemplatePatchModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePatchModelAsPatch, asPatchErr := instanceTemplatePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceTemplateOptions model
				updateInstanceTemplateOptionsModel := new(vpcv1.UpdateInstanceTemplateOptions)
				updateInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				updateInstanceTemplateOptionsModel.InstanceTemplatePatch = instanceTemplatePatchModelAsPatch
				updateInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateInstanceTemplateOptions model with no property values
				updateInstanceTemplateOptionsModelNew := new(vpcv1.UpdateInstanceTemplateOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateInstanceTemplate successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the InstanceTemplatePatch model
				instanceTemplatePatchModel := new(vpcv1.InstanceTemplatePatch)
				instanceTemplatePatchModel.Name = core.StringPtr("my-instance-template")
				instanceTemplatePatchModelAsPatch, asPatchErr := instanceTemplatePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceTemplateOptions model
				updateInstanceTemplateOptionsModel := new(vpcv1.UpdateInstanceTemplateOptions)
				updateInstanceTemplateOptionsModel.ID = core.StringPtr("testString")
				updateInstanceTemplateOptionsModel.InstanceTemplatePatch = instanceTemplatePatchModelAsPatch
				updateInstanceTemplateOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateInstanceTemplate(updateInstanceTemplateOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstances(listInstancesOptions *ListInstancesOptions) - Operation response error`, func() {
		version := "testString"
		listInstancesPath := "/instances"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstancesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.crn"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.name"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstances with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listInstancesOptionsModel.Name = core.StringPtr("testString")
				listInstancesOptionsModel.VPCID = core.StringPtr("testString")
				listInstancesOptionsModel.VPCCRN = core.StringPtr("testString")
				listInstancesOptionsModel.VPCName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstances(listInstancesOptions *ListInstancesOptions)`, func() {
		version := "testString"
		listInstancesPath := "/instances"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstancesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.crn"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.name"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?limit=20"}, "instances": [{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstances successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listInstancesOptionsModel.Name = core.StringPtr("testString")
				listInstancesOptionsModel.VPCID = core.StringPtr("testString")
				listInstancesOptionsModel.VPCCRN = core.StringPtr("testString")
				listInstancesOptionsModel.VPCName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.ListInstancesWithContext(ctx, listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.ListInstancesWithContext(ctx, listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstancesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))
					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.id"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.crn"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["vpc.name"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?limit=20"}, "instances": [{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstances successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.ListInstances(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listInstancesOptionsModel.Name = core.StringPtr("testString")
				listInstancesOptionsModel.VPCID = core.StringPtr("testString")
				listInstancesOptionsModel.VPCCRN = core.StringPtr("testString")
				listInstancesOptionsModel.VPCName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstances with error: Operation request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listInstancesOptionsModel.Name = core.StringPtr("testString")
				listInstancesOptionsModel.VPCID = core.StringPtr("testString")
				listInstancesOptionsModel.VPCCRN = core.StringPtr("testString")
				listInstancesOptionsModel.VPCName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke ListInstances successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listInstancesOptionsModel.Name = core.StringPtr("testString")
				listInstancesOptionsModel.VPCID = core.StringPtr("testString")
				listInstancesOptionsModel.VPCCRN = core.StringPtr("testString")
				listInstancesOptionsModel.VPCName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstance(createInstanceOptions *CreateInstanceOptions) - Operation response error`, func() {
		version := "testString"
		createInstancePath := "/instances"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstancePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstance with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("0787-8c2a09be-ee18-4af2-8ef4-6a6060732221")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstance(createInstanceOptions *CreateInstanceOptions)`, func() {
		version := "testString"
		createInstancePath := "/instances"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstancePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateInstance successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("0787-8c2a09be-ee18-4af2-8ef4-6a6060732221")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateInstanceWithContext(ctx, createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateInstanceWithContext(ctx, createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstancePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("0787-8c2a09be-ee18-4af2-8ef4-6a6060732221")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstance with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("0787-8c2a09be-ee18-4af2-8ef4-6a6060732221")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateInstanceOptions model with no property values
				createInstanceOptionsModelNew := new(vpcv1.CreateInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateInstance(createInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.AllowIPSpoofing = core.BoolPtr(true)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID model
				instancePlacementTargetPrototypeModel := new(vpcv1.InstancePlacementTargetPrototypeDedicatedHostIdentityDedicatedHostIdentityByID)
				instancePlacementTargetPrototypeModel.ID = core.StringPtr("0787-8c2a09be-ee18-4af2-8ef4-6a6060732221")

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.PlacementTarget = instancePlacementTargetPrototypeModel
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`DeleteInstance(deleteInstanceOptions *DeleteInstanceOptions)`, func() {
		version := "testString"
		deleteInstancePath := "/instances/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteInstancePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcService.DeleteInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteInstanceOptions model
				deleteInstanceOptionsModel := new(vpcv1.DeleteInstanceOptions)
				deleteInstanceOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcService.DeleteInstance(deleteInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteInstance with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the DeleteInstanceOptions model
				deleteInstanceOptionsModel := new(vpcv1.DeleteInstanceOptions)
				deleteInstanceOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcService.DeleteInstance(deleteInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteInstanceOptions model with no property values
				deleteInstanceOptionsModelNew := new(vpcv1.DeleteInstanceOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcService.DeleteInstance(deleteInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstance(getInstanceOptions *GetInstanceOptions) - Operation response error`, func() {
		version := "testString"
		getInstancePath := "/instances/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstancePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstance with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstance(getInstanceOptions *GetInstanceOptions)`, func() {
		version := "testString"
		getInstancePath := "/instances/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstancePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstance successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetInstanceWithContext(ctx, getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetInstanceWithContext(ctx, getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstancePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstance with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceOptions model with no property values
				getInstanceOptionsModelNew := new(vpcv1.GetInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetInstance(getInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateInstance(updateInstanceOptions *UpdateInstanceOptions) - Operation response error`, func() {
		version := "testString"
		updateInstancePath := "/instances/testString"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstancePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateInstance with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the InstancePatchProfileInstanceProfileIdentityByName model
				instancePatchProfileModel := new(vpcv1.InstancePatchProfileInstanceProfileIdentityByName)
				instancePatchProfileModel.Name = core.StringPtr("bc1-4x16")

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModel.Profile = instancePatchProfileModel
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateInstance(updateInstanceOptions *UpdateInstanceOptions)`, func() {
		version := "testString"
		updateInstancePath := "/instances/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstancePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateInstance successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the InstancePatchProfileInstanceProfileIdentityByName model
				instancePatchProfileModel := new(vpcv1.InstancePatchProfileInstanceProfileIdentityByName)
				instancePatchProfileModel.Name = core.StringPtr("bc1-4x16")

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModel.Profile = instancePatchProfileModel
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.UpdateInstanceWithContext(ctx, updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.UpdateInstanceWithContext(ctx, updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstancePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "disks": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/disks/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "interface_type": "virtio_blk", "name": "my-instance-disk", "resource_type": "instance_disk", "size": 100}], "gpu": {"count": 1, "manufacturer": "nvidia", "memory": 1, "model": "Tesla V100"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345", "name": "my-resource-group"}, "startable": false, "status": "deleting", "status_reasons": [{"code": "cannot_start_storage", "message": "The virtual server instance is unusable because the encryption key for the boot volume has been deleted", "more_info": "https://cloud.ibm.com/docs/key-protect?topic=key-protect-restore-keys"}], "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.UpdateInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the InstancePatchProfileInstanceProfileIdentityByName model
				instancePatchProfileModel := new(vpcv1.InstancePatchProfileInstanceProfileIdentityByName)
				instancePatchProfileModel.Name = core.StringPtr("bc1-4x16")

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModel.Profile = instancePatchProfileModel
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateInstance with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the InstancePatchProfileInstanceProfileIdentityByName model
				instancePatchProfileModel := new(vpcv1.InstancePatchProfileInstanceProfileIdentityByName)
				instancePatchProfileModel.Name = core.StringPtr("bc1-4x16")

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModel.Profile = instancePatchProfileModel
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateInstanceOptions model with no property values
				updateInstanceOptionsModelNew := new(vpcv1.UpdateInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.UpdateInstance(updateInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke UpdateInstance successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the InstancePatchProfileInstanceProfileIdentityByName model
				instancePatchProfileModel := new(vpcv1.InstancePatchProfileInstanceProfileIdentityByName)
				instancePatchProfileModel.Name = core.StringPtr("bc1-4x16")

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModel.Profile = instancePatchProfileModel
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceInitialization(getInstanceInitializationOptions *GetInstanceInitializationOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceInitializationPath := "/instances/testString/initialization"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceInitializationPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceInitialization with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceInitialization(getInstanceInitializationOptions *GetInstanceInitializationOptions)`, func() {
		version := "testString"
		getInstanceInitializationPath := "/instances/testString/initialization"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceInitializationPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"keys": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}], "password": {"encrypted_password": "VGhpcyBpcyBhbiBlbmNvZGVkIGJ5dGUgYXJyYXku", "encryption_key": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}}}`)
				}))
			})
			It(`Invoke GetInstanceInitialization successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.GetInstanceInitializationWithContext(ctx, getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.GetInstanceInitializationWithContext(ctx, getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceInitializationPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"keys": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}], "password": {"encrypted_password": "VGhpcyBpcyBhbiBlbmNvZGVkIGJ5dGUgYXJyYXku", "encryption_key": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "deleted": {"more_info": "https://cloud.ibm.com/apidocs/vpc#deleted-resources"}, "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}}}`)
				}))
			})
			It(`Invoke GetInstanceInitialization successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.GetInstanceInitialization(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceInitialization with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceInitializationOptions model with no property values
				getInstanceInitializationOptionsModelNew := new(vpcv1.GetInstanceInitializationOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(200)
				}))
			})
			It(`Invoke GetInstanceInitialization successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceAction(createInstanceActionOptions *CreateInstanceActionOptions) - Operation response error`, func() {
		version := "testString"
		createInstanceActionPath := "/instances/testString/actions"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceActionPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstanceAction with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Force = core.BoolPtr(true)
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceAction(createInstanceActionOptions *CreateInstanceActionOptions)`, func() {
		version := "testString"
		createInstanceActionPath := "/instances/testString/actions"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceActionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"completed_at": "2019-01-01T12:00:00.000Z", "created_at": "2019-01-01T12:00:00.000Z", "force": false, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/actions/109a1b6e-1242-4de1-be44-38705e9474ed", "id": "109a1b6e-1242-4de1-be44-38705e9474ed", "started_at": "2019-01-01T12:00:00.000Z", "status": "completed", "type": "reboot"}`)
				}))
			})
			It(`Invoke CreateInstanceAction successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Force = core.BoolPtr(true)
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateInstanceActionWithContext(ctx, createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateInstanceActionWithContext(ctx, createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceActionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"completed_at": "2019-01-01T12:00:00.000Z", "created_at": "2019-01-01T12:00:00.000Z", "force": false, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/actions/109a1b6e-1242-4de1-be44-38705e9474ed", "id": "109a1b6e-1242-4de1-be44-38705e9474ed", "started_at": "2019-01-01T12:00:00.000Z", "status": "completed", "type": "reboot"}`)
				}))
			})
			It(`Invoke CreateInstanceAction successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateInstanceAction(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Force = core.BoolPtr(true)
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstanceAction with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Force = core.BoolPtr(true)
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateInstanceActionOptions model with no property values
				createInstanceActionOptionsModelNew := new(vpcv1.CreateInstanceActionOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcService.CreateInstanceAction(createInstanceActionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint with missing response body`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Set success status code with no respoonse body
					res.WriteHeader(201)
				}))
			})
			It(`Invoke CreateInstanceAction successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Force = core.BoolPtr(true)
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation
				result, response, operationErr := vpcService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())

				// Verify a nil result
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptions *CreateInstanceConsoleAccessTokenOptions) - Operation response error`, func() {
		version := "testString"
		createInstanceConsoleAccessTokenPath := "/instances/testString/console_access_token"
		Context(`Using mock server endpoint with invalid JSON response`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceConsoleAccessTokenPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstanceConsoleAccessToken with error: Operation response processing error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceConsoleAccessTokenOptions model
				createInstanceConsoleAccessTokenOptionsModel := new(vpcv1.CreateInstanceConsoleAccessTokenOptions)
				createInstanceConsoleAccessTokenOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceConsoleAccessTokenOptionsModel.ConsoleType = core.StringPtr("serial")
				createInstanceConsoleAccessTokenOptionsModel.Force = core.BoolPtr(false)
				createInstanceConsoleAccessTokenOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcService.CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcService.EnableRetries(0, 0)
				result, response, operationErr = vpcService.CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptions *CreateInstanceConsoleAccessTokenOptions)`, func() {
		version := "testString"
		createInstanceConsoleAccessTokenPath := "/instances/testString/console_access_token"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceConsoleAccessTokenPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"access_token": "VGhpcyBJcyBhIHRva2Vu", "console_type": "serial", "created_at": "2020-07-27T21:50:14.000Z", "expires_at": "2020-07-27T21:51:14.000Z", "force": false, "href": "wss://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/console?access_token=VGhpcyBJcyBhIHRva2Vu"}`)
				}))
			})
			It(`Invoke CreateInstanceConsoleAccessToken successfully with retries`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())
				vpcService.EnableRetries(0, 0)

				// Construct an instance of the CreateInstanceConsoleAccessTokenOptions model
				createInstanceConsoleAccessTokenOptionsModel := new(vpcv1.CreateInstanceConsoleAccessTokenOptions)
				createInstanceConsoleAccessTokenOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceConsoleAccessTokenOptionsModel.ConsoleType = core.StringPtr("serial")
				createInstanceConsoleAccessTokenOptionsModel.Force = core.BoolPtr(false)
				createInstanceConsoleAccessTokenOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcService.CreateInstanceConsoleAccessTokenWithContext(ctx, createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcService.DisableRetries()
				result, response, operationErr := vpcService.CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcService.CreateInstanceConsoleAccessTokenWithContext(ctx, createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceConsoleAccessTokenPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(2))}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"access_token": "VGhpcyBJcyBhIHRva2Vu", "console_type": "serial", "created_at": "2020-07-27T21:50:14.000Z", "expires_at": "2020-07-27T21:51:14.000Z", "force": false, "href": "wss://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/console?access_token=VGhpcyBJcyBhIHRva2Vu"}`)
				}))
			})
			It(`Invoke CreateInstanceConsoleAccessToken successfully`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcService.CreateInstanceConsoleAccessToken(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the CreateInstanceConsoleAccessTokenOptions model
				createInstanceConsoleAccessTokenOptionsModel := new(vpcv1.CreateInstanceConsoleAccessTokenOptions)
				createInstanceConsoleAccessTokenOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceConsoleAccessTokenOptionsModel.ConsoleType = core.StringPtr("serial")
				createInstanceConsoleAccessTokenOptionsModel.Force = core.BoolPtr(false)
				createInstanceConsoleAccessTokenOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcService.CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstanceConsoleAccessToken with error: Operation validation and request error`, func() {
				vpcService, serviceErr := vpcv1.NewVpcV1(&vpcv1.VpcV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceConsoleAccessTokenOptions model
				createInstanceConsoleAccessTokenOptionsModel := new(vpcv1.CreateInstanceConsoleAccessTokenOptions)
				createInstanceConsoleAccessTokenOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceConsoleAccessTokenOptionsModel.ConsoleType = core.StringPtr("serial")
				createInstanceConsoleAccessTokenOptionsModel.Force = core.BoolPtr(false)
				createInstanceConsoleAccessTokenOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcService.CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second ins