// +build integration

/**
 * (C) Copyright IBM Corp. 2020, 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package vulnerabilityadvisorv3_test

import (
	"fmt"
	"os"
	"strings"

	"github.com/IBM/container-registry-go-sdk/vulnerabilityadvisorv3"
	"github.com/IBM/go-sdk-core/v5/core"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

/**
 * This file contains an integration test for the vulnerabilityadvisorv3 package.
 *
 * Notes:
 *
* Your configuration file (vulnerability_advisor_v3.env) should contain the following variables.
VULNERABILITY_ADVISOR_URL=[Registry URL, eg https://uk.icr.io]
VULNERABILITY_ADVISOR_AUTH_TYPE=iam
VULNERABILITY_ADVISOR_AUTH_URL=https://iam.cloud.ibm.com/identity/token
VULNERABILITY_ADVISOR_APIKEY=[An IAM Apikey]
VULNERABILITY_ADVISOR_ACCOUNT_ID=[Your test account ID]
VULNERABILITY_ADVISOR_NAMESPACE=[Namespace name used by the tests]
VULNERABILITY_ADVISOR_SEED_IMAGE=[An existing namespace/repo:tag for which to get VA scans etc.]
 *
 * The integration test will automatically skip tests if the required config file is not available.
*/

var _ = Describe(`VulnerabilityAdvisorV3 Integration Tests`, func() {

	const externalConfigFile = "../vulnerability_advisor_v3.env"

	var (
		err                         error
		vulnerabilityAdvisorService *vulnerabilityadvisorv3.VulnerabilityAdvisorV3
		serviceURL                  string
		baseNamespace               string
		accountID                   string
		registryDNSName             string
		seedImage                   string
		config                      map[string]string
	)

	var shouldSkipTest = func() {
		Skip("External configuration is not available, skipping tests...")
	}

	Describe(`External configuration`, func() {
		It("Successfully load the configuration", func() {
			_, err = os.Stat(externalConfigFile)
			if err != nil {
				Skip("External configuration file not found, skipping tests: " + err.Error())
			}

			os.Setenv("IBM_CREDENTIALS_FILE", externalConfigFile)
			config, err = core.GetServiceProperties(vulnerabilityadvisorv3.DefaultServiceName)
			if err != nil {
				Skip("Error loading service properties, skipping tests: " + err.Error())
			}
			serviceURL = config["URL"]
			if serviceURL == "" {
				Skip("Unable to load service URL configuration property, skipping tests")
			}
			baseNamespace = config["NAMESPACE"]
			if baseNamespace == "" {
				Skip("Unable to load baseNamespace configuration property, skipping tests")
			}
			accountID = config["ACCOUNT_ID"]
			if accountID == "" {
				Skip("Unable to load accountID configuration property, skipping tests")
			}
			registryDNSName = strings.TrimPrefix(serviceURL, "https://")
			seedImage = config["SEED_IMAGE"]
			if seedImage == "" {
				Skip("Unable to load seedImage configuration property, skipping tests")
			}

			fmt.Printf("Service URL: %s\n", serviceURL)
			shouldSkipTest = func() {}
		})
	})

	Describe(`Client initialization`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It("Successfully construct the service client instance", func() {

			vulnerabilityAdvisorServiceOptions := &vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				Account: core.StringPtr(accountID),
			}

			vulnerabilityAdvisorService, err = vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(vulnerabilityAdvisorServiceOptions)

			Expect(err).To(BeNil())
			Expect(vulnerabilityAdvisorService).ToNot(BeNil())
			Expect(vulnerabilityAdvisorService.Service.Options.URL).To(Equal(serviceURL))
		})
	})

	Describe(`AccountReportQueryPath - Get the vulnerability assessment for all images`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`AccountReportQueryPath(accountReportQueryPathOptions *AccountReportQueryPathOptions)`, func() {

			accountReportQueryPathOptions := &vulnerabilityadvisorv3.AccountReportQueryPathOptions{
				IncludePrivate: core.StringPtr("true"),
			}

			scanReportList, response, err := vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanReportList).ToNot(BeNil())

		})
	})

	Describe(`AccountStatusQueryPath - Get vulnerability assessment status for all images`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`AccountStatusQueryPath(accountStatusQueryPathOptions *AccountStatusQueryPathOptions)`, func() {

			accountStatusQueryPathOptions := &vulnerabilityadvisorv3.AccountStatusQueryPathOptions{
				IncludePrivate: core.StringPtr("true"),
			}

			scanreportImageSummaryList, response, err := vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanreportImageSummaryList).ToNot(BeNil())

		})
	})

	Describe(`ImageStatusQueryPath - Get vulnerability status`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ImageStatusQueryPath(imageStatusQueryPathOptions *ImageStatusQueryPathOptions)`, func() {

			imageStatusQueryPathOptions := &vulnerabilityadvisorv3.ImageStatusQueryPathOptions{
				Name: core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
			}

			scanreportSummary, response, err := vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanreportSummary).ToNot(BeNil())

		})
	})

	Describe(`ImageReportQueryPath - Get vulnerability assessment status`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ImageReportQueryPath(imageReportQueryPathOptions *ImageReportQueryPathOptions)`, func() {

			imageReportQueryPathOptions := &vulnerabilityadvisorv3.ImageReportQueryPathOptions{
				Name: core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
			}

			scanReport, response, err := vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanReport).ToNot(BeNil())

		})
	})

	Describe(`CreateExemptionAccount - Create or update an account level exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`CreateExemptionAccount(createExemptionAccountOptions *CreateExemptionAccountOptions)`, func() {

			createExemptionAccountOptions := &vulnerabilityadvisorv3.CreateExemptionAccountOptions{
				IssueType: core.StringPtr("cve"),
				IssueID:   core.StringPtr("CVE-2020-0001"),
			}

			exemption, response, err := vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(201))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`ListExemptionAccount - List account level exemptions`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ListExemptionAccount(listExemptionAccountOptions *ListExemptionAccountOptions)`, func() {

			listExemptionAccountOptions := &vulnerabilityadvisorv3.ListExemptionAccountOptions{}

			exemption, response, err := vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`GetExemptionAccount - Get an account level exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`GetExemptionAccount(getExemptionAccountOptions *GetExemptionAccountOptions)`, func() {

			getExemptionAccountOptions := &vulnerabilityadvisorv3.GetExemptionAccountOptions{
				IssueType: core.StringPtr("cve"),
				IssueID:   core.StringPtr("CVE-2020-0001"),
			}

			exemption, response, err := vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`DeleteExemptionAccount - Delete an account level exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`DeleteExemptionAccount(deleteExemptionAccountOptions *DeleteExemptionAccountOptions)`, func() {

			deleteExemptionAccountOptions := &vulnerabilityadvisorv3.DeleteExemptionAccountOptions{
				IssueType: core.StringPtr("cve"),
				IssueID:   core.StringPtr("CVE-2020-0001"),
			}

			response, err := vulnerabilityAdvisorService.DeleteExemptionAccount(deleteExemptionAccountOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))

		})
	})

	Describe(`CreateExemptionResource - Create or update a resource exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`CreateExemptionResource(createExemptionResourceOptions *CreateExemptionResourceOptions)`, func() {

			createExemptionResourceOptions := &vulnerabilityadvisorv3.CreateExemptionResourceOptions{
				IssueType: core.StringPtr("cve"),
				IssueID:   core.StringPtr("CVE-2020-0001"),
				Resource:  core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
			}

			exemption, response, err := vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(201))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`ListExemptionResource - List resource exemptions`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ListExemptionResource(listExemptionResourceOptions *ListExemptionResourceOptions)`, func() {

			listExemptionResourceOptions := &vulnerabilityadvisorv3.ListExemptionResourceOptions{
				Resource: core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
			}

			exemption, response, err := vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`ListImageExemptions - List image exemptions`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ListImageExemptions(listImageExemptionsOptions *ListImageExemptionsOptions)`, func() {

			listImageExemptionsOptions := &vulnerabilityadvisorv3.ListImageExemptionsOptions{
				Resource:     core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
				IncludeScope: core.BoolPtr(true),
			}

			exemption, response, err := vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`ListBulkImageExemptions - List exemptions for images`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ListBulkImageExemptions(listBulkImageExemptionsOptions *ListBulkImageExemptionsOptions)`, func() {

			listBulkImageExemptionsOptions := &vulnerabilityadvisorv3.ListBulkImageExemptionsOptions{
				Body: []string{fmt.Sprintf("%s/%s", registryDNSName, seedImage)},
			}

			mapStringExemption, response, err := vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(mapStringExemption).ToNot(BeNil())

		})
	})

	Describe(`ListAccountExemptions - List all exemptions`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ListAccountExemptions(listAccountExemptionsOptions *ListAccountExemptionsOptions)`, func() {

			listAccountExemptionsOptions := &vulnerabilityadvisorv3.ListAccountExemptionsOptions{}

			exemption, response, err := vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`GetExemptionResource - Get details of a resource exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`GetExemptionResource(getExemptionResourceOptions *GetExemptionResourceOptions)`, func() {

			getExemptionResourceOptions := &vulnerabilityadvisorv3.GetExemptionResourceOptions{
				IssueType: core.StringPtr("cve"),
				IssueID:   core.StringPtr("CVE-2020-0001"),
				Resource:  core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
			}

			exemption, response, err := vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
	})

	Describe(`DeleteExemptionResource - Delete a resource exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`DeleteExemptionResource(deleteExemptionResourceOptions *DeleteExemptionResourceOptions)`, func() {

			deleteExemptionResourceOptions := &vulnerabilityadvisorv3.DeleteExemptionResourceOptions{
				IssueType: core.StringPtr("cve"),
				IssueID:   core.StringPtr("CVE-2020-0001"),
				Resource:  core.StringPtr(fmt.Sprintf("%s/%s", registryDNSName, seedImage)),
			}

			response, err := vulnerabilityAdvisorService.DeleteExemptionResource(deleteExemptionResourceOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))

		})
	})

	Describe(`ExemptHandler - List the types of exemption`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ExemptHandler(exemptHandlerOptions *ExemptHandlerOptions)`, func() {

			exemptHandlerOptions := &vulnerabilityadvisorv3.ExemptHandlerOptions{}

			exemptionTypeInfo, response, err := vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemptionTypeInfo).ToNot(BeNil())

		})
	})

	Describe(`ExemptionsAccountDeleteHandler - Delete all exemptions`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptions *ExemptionsAccountDeleteHandlerOptions)`, func() {

			exemptionsAccountDeleteHandlerOptions := &vulnerabilityadvisorv3.ExemptionsAccountDeleteHandlerOptions{}

			exemptionDeletionInfo, response, err := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptions)

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemptionDeletionInfo).ToNot(BeNil())

		})
	})
})

//
// Utility functions are declared in the unit test file
//
