/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package networking

import (
	"fmt"

	"github.com/gophercloud/gophercloud/openstack/networking/v2/extensions/attributestags"
	"github.com/gophercloud/gophercloud/openstack/networking/v2/extensions/security/groups"
	"github.com/gophercloud/gophercloud/openstack/networking/v2/extensions/security/rules"

	infrav1 "sigs.k8s.io/cluster-api-provider-openstack/api/v1beta1"
	"sigs.k8s.io/cluster-api-provider-openstack/pkg/record"
)

const (
	secGroupPrefix     string = "k8s"
	controlPlaneSuffix string = "controlplane"
	workerSuffix       string = "worker"
	bastionSuffix      string = "bastion"
	remoteGroupIDSelf  string = "self"
)

var defaultRules = []infrav1.SecurityGroupRule{
	{
		Direction:      "egress",
		Description:    "Full open",
		EtherType:      "IPv4",
		PortRangeMin:   0,
		PortRangeMax:   0,
		Protocol:       "",
		RemoteIPPrefix: "",
	},
	{
		Direction:      "egress",
		Description:    "Full open",
		EtherType:      "IPv6",
		PortRangeMin:   0,
		PortRangeMax:   0,
		Protocol:       "",
		RemoteIPPrefix: "",
	},
}

// ReconcileSecurityGroups reconcile the security groups.
func (s *Service) ReconcileSecurityGroups(openStackCluster *infrav1.OpenStackCluster, clusterName string) error {
	s.logger.Info("Reconciling security groups", "cluster", clusterName)
	if !openStackCluster.Spec.ManagedSecurityGroups {
		s.logger.V(4).Info("No need to reconcile security groups", "cluster", clusterName)
		return nil
	}

	secControlPlaneGroupName := getSecControlPlaneGroupName(clusterName)
	secWorkerGroupName := getSecWorkerGroupName(clusterName)
	secGroupNames := map[string]string{
		controlPlaneSuffix: secControlPlaneGroupName,
		workerSuffix:       secWorkerGroupName,
	}

	if openStackCluster.Spec.Bastion != nil && openStackCluster.Spec.Bastion.Enabled {
		secBastionGroupName := getSecBastionGroupName(clusterName)
		secGroupNames[bastionSuffix] = secBastionGroupName
	}

	// create security groups first, because desired rules use group ids.
	for _, v := range secGroupNames {
		if err := s.createSecurityGroupIfNotExists(openStackCluster, v); err != nil {
			return err
		}
	}
	// create desired security groups
	desiredSecGroups, err := s.generateDesiredSecGroups(openStackCluster, secGroupNames)
	if err != nil {
		return err
	}

	observedSecGroups := make(map[string]*infrav1.SecurityGroup)
	for k, desiredSecGroup := range desiredSecGroups {
		var err error
		observedSecGroups[k], err = s.getSecurityGroupByName(desiredSecGroup.Name)

		if err != nil {
			return err
		}

		if observedSecGroups[k].ID != "" {
			observedSecGroup, err := s.reconcileGroupRules(desiredSecGroup, *observedSecGroups[k])
			if err != nil {
				return err
			}
			observedSecGroups[k] = &observedSecGroup
			continue
		}
	}

	openStackCluster.Status.ControlPlaneSecurityGroup = observedSecGroups[controlPlaneSuffix]
	openStackCluster.Status.WorkerSecurityGroup = observedSecGroups[workerSuffix]
	openStackCluster.Status.BastionSecurityGroup = observedSecGroups[bastionSuffix]

	return nil
}

func (s *Service) generateDesiredSecGroups(openStackCluster *infrav1.OpenStackCluster, secGroupNames map[string]string) (map[string]infrav1.SecurityGroup, error) {
	desiredSecGroups := make(map[string]infrav1.SecurityGroup)

	var secControlPlaneGroupID string
	var secWorkerGroupID string
	var secBastionGroupID string
	for i, v := range secGroupNames {
		secGroup, err := s.getSecurityGroupByName(v)
		if err != nil {
			return desiredSecGroups, err
		}
		switch i {
		case controlPlaneSuffix:
			secControlPlaneGroupID = secGroup.ID
		case workerSuffix:
			secWorkerGroupID = secGroup.ID
		case bastionSuffix:
			secBastionGroupID = secGroup.ID
		}
	}

	// Start with the default rules
	controlPlaneRules := append([]infrav1.SecurityGroupRule{}, defaultRules...)
	workerRules := append([]infrav1.SecurityGroupRule{}, defaultRules...)

	// Allow all traffic, including from outside the cluster, to access the API
	controlPlaneRules = append(controlPlaneRules,
		infrav1.SecurityGroupRule{
			Description:  "Kubernetes API",
			Direction:    "ingress",
			EtherType:    "IPv4",
			PortRangeMin: 6443,
			PortRangeMax: 6443,
			Protocol:     "tcp",
		},
	)
	// Allow all traffic, including from outside the cluster, to access node port services
	workerRules = append(workerRules,
		infrav1.SecurityGroupRule{
			Description:  "Node Port Services",
			Direction:    "ingress",
			EtherType:    "IPv4",
			PortRangeMin: 30000,
			PortRangeMax: 32767,
			Protocol:     "tcp",
		},
	)

	if openStackCluster.Spec.AllowAllInClusterTraffic {
		// Permit all ingress from the cluster security groups
		controlPlaneRules = append(controlPlaneRules,
			[]infrav1.SecurityGroupRule{
				{
					Description:   "In-cluster Ingress",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  0,
					PortRangeMax:  0,
					Protocol:      "",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "In-cluster Ingress",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  0,
					PortRangeMax:  0,
					Protocol:      "",
					RemoteGroupID: secWorkerGroupID,
				},
			}...,
		)
		workerRules = append(workerRules,
			[]infrav1.SecurityGroupRule{
				{
					Description:   "In-cluster Ingress",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  0,
					PortRangeMax:  0,
					Protocol:      "",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "In-cluster Ingress",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  0,
					PortRangeMax:  0,
					Protocol:      "",
					RemoteGroupID: secControlPlaneGroupID,
				},
			}...,
		)
	} else {
		// Permit traffic for etcd, kubelet and Calico only
		controlPlaneRules = append(controlPlaneRules,
			[]infrav1.SecurityGroupRule{
				{
					Description:   "Etcd",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  2379,
					PortRangeMax:  2380,
					Protocol:      "tcp",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					// kubeadm says this is needed
					Description:   "Kubelet API",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  10250,
					PortRangeMax:  10250,
					Protocol:      "tcp",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					// This is needed to support metrics-server deployments
					Description:   "Kubelet API",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  10250,
					PortRangeMax:  10250,
					Protocol:      "tcp",
					RemoteGroupID: secWorkerGroupID,
				},
				{
					Description:   "BGP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  179,
					PortRangeMax:  179,
					Protocol:      "tcp",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "BGP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  179,
					PortRangeMax:  179,
					Protocol:      "tcp",
					RemoteGroupID: secWorkerGroupID,
				},
				{
					Description:   "IP-in-IP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					Protocol:      "4",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "IP-in-IP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					Protocol:      "4",
					RemoteGroupID: secWorkerGroupID,
				},
			}...,
		)
		workerRules = append(workerRules,
			[]infrav1.SecurityGroupRule{
				{
					// This is needed to support metrics-server deployments
					Description:   "Kubelet API",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  10250,
					PortRangeMax:  10250,
					Protocol:      "tcp",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "Kubelet API",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  10250,
					PortRangeMax:  10250,
					Protocol:      "tcp",
					RemoteGroupID: secControlPlaneGroupID,
				},
				{
					Description:   "BGP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  179,
					PortRangeMax:  179,
					Protocol:      "tcp",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "BGP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  179,
					PortRangeMax:  179,
					Protocol:      "tcp",
					RemoteGroupID: secControlPlaneGroupID,
				},
				{
					Description:   "IP-in-IP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					Protocol:      "4",
					RemoteGroupID: remoteGroupIDSelf,
				},
				{
					Description:   "IP-in-IP (calico)",
					Direction:     "ingress",
					EtherType:     "IPv4",
					Protocol:      "4",
					RemoteGroupID: secControlPlaneGroupID,
				},
			}...,
		)
	}

	if openStackCluster.Spec.Bastion != nil && openStackCluster.Spec.Bastion.Enabled {
		controlPlaneRules = append(controlPlaneRules,
			[]infrav1.SecurityGroupRule{
				{
					Description:   "SSH",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  22,
					PortRangeMax:  22,
					Protocol:      "tcp",
					RemoteGroupID: secBastionGroupID,
				},
			}...,
		)
		workerRules = append(workerRules,
			[]infrav1.SecurityGroupRule{
				{
					Description:   "SSH",
					Direction:     "ingress",
					EtherType:     "IPv4",
					PortRangeMin:  22,
					PortRangeMax:  22,
					Protocol:      "tcp",
					RemoteGroupID: secBastionGroupID,
				},
			}...,
		)
		desiredSecGroups[bastionSuffix] = infrav1.SecurityGroup{
			Name: secGroupNames[bastionSuffix],
			Rules: append(
				[]infrav1.SecurityGroupRule{
					{
						Description:  "SSH",
						Direction:    "ingress",
						EtherType:    "IPv4",
						PortRangeMin: 22,
						PortRangeMax: 22,
						Protocol:     "tcp",
					},
				},
				defaultRules...,
			),
		}
	}

	desiredSecGroups[controlPlaneSuffix] = infrav1.SecurityGroup{
		Name:  secGroupNames[controlPlaneSuffix],
		Rules: controlPlaneRules,
	}

	desiredSecGroups[workerSuffix] = infrav1.SecurityGroup{
		Name:  secGroupNames[workerSuffix],
		Rules: workerRules,
	}

	return desiredSecGroups, nil
}

func (s *Service) GetSecurityGroups(securityGroupParams []infrav1.SecurityGroupParam) ([]string, error) {
	var sgIDs []string
	for _, sg := range securityGroupParams {
		listOpts := groups.ListOpts(sg.Filter)
		if listOpts.ProjectID == "" {
			listOpts.ProjectID = s.projectID
		}
		listOpts.Name = sg.Name
		listOpts.ID = sg.UUID
		SGList, err := s.client.ListSecGroup(listOpts)
		if err != nil {
			return nil, err
		}

		if len(SGList) == 0 {
			return nil, fmt.Errorf("security group %s not found", sg.Name)
		}

		for _, group := range SGList {
			if isDuplicate(sgIDs, group.ID) {
				continue
			}
			sgIDs = append(sgIDs, group.ID)
		}
	}
	return sgIDs, nil
}

func (s *Service) DeleteSecurityGroups(openStackCluster *infrav1.OpenStackCluster, clusterName string) error {
	secGroupNames := []string{
		getSecControlPlaneGroupName(clusterName),
		getSecWorkerGroupName(clusterName),
	}
	for _, secGroupName := range secGroupNames {
		if err := s.deleteSecurityGroup(openStackCluster, secGroupName); err != nil {
			return err
		}
	}

	return nil
}

func (s *Service) DeleteBastionSecurityGroup(openStackCluster *infrav1.OpenStackCluster, clusterName string) error {
	secBastionGroupName := getSecBastionGroupName(clusterName)
	return s.deleteSecurityGroup(openStackCluster, secBastionGroupName)
}

func (s *Service) deleteSecurityGroup(openStackCluster *infrav1.OpenStackCluster, name string) error {
	group, err := s.getSecurityGroupByName(name)
	if err != nil {
		return err
	}
	if group.ID == "" {
		// nothing to do
		return nil
	}
	err = s.client.DeleteSecGroup(group.ID)
	if err != nil {
		record.Warnf(openStackCluster, "FailedDeleteSecurityGroup", "Failed to delete security group %s with id %s: %v", group.Name, group.ID, err)
		return err
	}

	record.Eventf(openStackCluster, "SuccessfulDeleteSecurityGroup", "Deleted security group %s with id %s", group.Name, group.ID)
	return nil
}

// reconcileGroupRules reconciles an already existing observed group by deleting rules not needed anymore and
// creating rules that are missing.
func (s *Service) reconcileGroupRules(desired, observed infrav1.SecurityGroup) (infrav1.SecurityGroup, error) {
	rulesToDelete := []infrav1.SecurityGroupRule{}
	// fills rulesToDelete by calculating observed - desired
	for _, observedRule := range observed.Rules {
		deleteRule := true
		for _, desiredRule := range desired.Rules {
			r := desiredRule
			if r.RemoteGroupID == remoteGroupIDSelf {
				r.RemoteGroupID = observed.ID
			}
			if r.Equal(observedRule) {
				deleteRule = false
				break
			}
		}
		if deleteRule {
			rulesToDelete = append(rulesToDelete, observedRule)
		}
	}

	rulesToCreate := []infrav1.SecurityGroupRule{}
	reconciledRules := make([]infrav1.SecurityGroupRule, 0, len(desired.Rules))
	// fills rulesToCreate by calculating desired - observed
	// also adds rules which are in observed and desired to reconciledRules.
	for _, desiredRule := range desired.Rules {
		r := desiredRule
		if r.RemoteGroupID == remoteGroupIDSelf {
			r.RemoteGroupID = observed.ID
		}
		createRule := true
		for _, observedRule := range observed.Rules {
			if r.Equal(observedRule) {
				// add already existing rules to reconciledRules because we won't touch them anymore
				reconciledRules = append(reconciledRules, observedRule)
				createRule = false
				break
			}
		}
		if createRule {
			rulesToCreate = append(rulesToCreate, desiredRule)
		}
	}

	s.logger.V(4).Info("Deleting rules not needed anymore for group", "name", observed.Name, "amount", len(rulesToDelete))
	for _, rule := range rulesToDelete {
		s.logger.V(6).Info("Deleting rule", "ruleID", rule.ID, "groupName", observed.Name)
		err := s.client.DeleteSecGroupRule(rule.ID)
		if err != nil {
			return infrav1.SecurityGroup{}, err
		}
	}

	s.logger.V(4).Info("Creating new rules needed for group", "name", observed.Name, "amount", len(rulesToCreate))
	for _, rule := range rulesToCreate {
		r := rule
		r.SecurityGroupID = observed.ID
		if r.RemoteGroupID == remoteGroupIDSelf {
			r.RemoteGroupID = observed.ID
		}
		newRule, err := s.createRule(r)
		if err != nil {
			return infrav1.SecurityGroup{}, err
		}
		reconciledRules = append(reconciledRules, newRule)
	}
	observed.Rules = reconciledRules

	return observed, nil
}

func (s *Service) createSecurityGroupIfNotExists(openStackCluster *infrav1.OpenStackCluster, groupName string) error {
	secGroup, err := s.getSecurityGroupByName(groupName)
	if err != nil {
		return err
	}
	if secGroup == nil || secGroup.ID == "" {
		s.logger.V(6).Info("Group doesn't exist, creating it.", "name", groupName)

		createOpts := groups.CreateOpts{
			Name:        groupName,
			Description: "Cluster API managed group",
		}
		s.logger.V(6).Info("Creating group", "name", groupName)

		group, err := s.client.CreateSecGroup(createOpts)
		if err != nil {
			record.Warnf(openStackCluster, "FailedCreateSecurityGroup", "Failed to create security group %s: %v", groupName, err)
			return err
		}

		if len(openStackCluster.Spec.Tags) > 0 {
			_, err = s.client.ReplaceAllAttributesTags("security-groups", group.ID, attributestags.ReplaceAllOpts{
				Tags: openStackCluster.Spec.Tags,
			})
			if err != nil {
				return err
			}
		}

		record.Eventf(openStackCluster, "SuccessfulCreateSecurityGroup", "Created security group %s with id %s", groupName, group.ID)
		return nil
	}

	sInfo := fmt.Sprintf("Reuse Existing SecurityGroup %s with %s", groupName, secGroup.ID)
	s.logger.V(6).Info(sInfo)

	return nil
}

func (s *Service) getSecurityGroupByName(name string) (*infrav1.SecurityGroup, error) {
	opts := groups.ListOpts{
		Name: name,
	}

	s.logger.V(6).Info("Attempting to fetch security group with", "name", name)
	allGroups, err := s.client.ListSecGroup(opts)
	if err != nil {
		return &infrav1.SecurityGroup{}, err
	}

	switch len(allGroups) {
	case 0:
		return &infrav1.SecurityGroup{}, nil
	case 1:
		return convertOSSecGroupToConfigSecGroup(allGroups[0]), nil
	}

	return &infrav1.SecurityGroup{}, fmt.Errorf("more than one security group found named: %s", name)
}

func (s *Service) createRule(r infrav1.SecurityGroupRule) (infrav1.SecurityGroupRule, error) {
	dir := rules.RuleDirection(r.Direction)
	proto := rules.RuleProtocol(r.Protocol)
	etherType := rules.RuleEtherType(r.EtherType)

	createOpts := rules.CreateOpts{
		Description:    r.Description,
		Direction:      dir,
		PortRangeMin:   r.PortRangeMin,
		PortRangeMax:   r.PortRangeMax,
		Protocol:       proto,
		EtherType:      etherType,
		RemoteGroupID:  r.RemoteGroupID,
		RemoteIPPrefix: r.RemoteIPPrefix,
		SecGroupID:     r.SecurityGroupID,
	}
	s.logger.V(6).Info("Creating rule", "Description", r.Description, "Direction", dir, "PortRangeMin", r.PortRangeMin, "PortRangeMax", r.PortRangeMax, "Proto", proto, "etherType", etherType, "RemoteGroupID", r.RemoteGroupID, "RemoteIPPrefix", r.RemoteIPPrefix, "SecurityGroupID", r.SecurityGroupID)
	rule, err := s.client.CreateSecGroupRule(createOpts)
	if err != nil {
		return infrav1.SecurityGroupRule{}, err
	}
	return convertOSSecGroupRuleToConfigSecGroupRule(*rule), nil
}

func getSecControlPlaneGroupName(clusterName string) string {
	return fmt.Sprintf("%s-cluster-%s-secgroup-%s", secGroupPrefix, clusterName, controlPlaneSuffix)
}

func getSecWorkerGroupName(clusterName string) string {
	return fmt.Sprintf("%s-cluster-%s-secgroup-%s", secGroupPrefix, clusterName, workerSuffix)
}

func getSecBastionGroupName(clusterName string) string {
	return fmt.Sprintf("%s-cluster-%s-secgroup-%s", secGroupPrefix, clusterName, bastionSuffix)
}

func convertOSSecGroupToConfigSecGroup(osSecGroup groups.SecGroup) *infrav1.SecurityGroup {
	securityGroupRules := make([]infrav1.SecurityGroupRule, len(osSecGroup.Rules))
	for i, rule := range osSecGroup.Rules {
		securityGroupRules[i] = convertOSSecGroupRuleToConfigSecGroupRule(rule)
	}
	return &infrav1.SecurityGroup{
		ID:    osSecGroup.ID,
		Name:  osSecGroup.Name,
		Rules: securityGroupRules,
	}
}

func convertOSSecGroupRuleToConfigSecGroupRule(osSecGroupRule rules.SecGroupRule) infrav1.SecurityGroupRule {
	return infrav1.SecurityGroupRule{
		ID:              osSecGroupRule.ID,
		Direction:       osSecGroupRule.Direction,
		Description:     osSecGroupRule.Description,
		EtherType:       osSecGroupRule.EtherType,
		SecurityGroupID: osSecGroupRule.SecGroupID,
		PortRangeMin:    osSecGroupRule.PortRangeMin,
		PortRangeMax:    osSecGroupRule.PortRangeMax,
		Protocol:        osSecGroupRule.Protocol,
		RemoteGroupID:   osSecGroupRule.RemoteGroupID,
		RemoteIPPrefix:  osSecGroupRule.RemoteIPPrefix,
	}
}

func isDuplicate(list []string, name string) bool {
	if len(list) == 0 {
		return false
	}
	for _, element := range list {
		if element == name {
			return true
		}
	}
	return false
}
