/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package networking

import (
	"fmt"

	"github.com/go-logr/logr"
	"github.com/gophercloud/gophercloud"
	"github.com/gophercloud/gophercloud/openstack"
	"github.com/gophercloud/gophercloud/openstack/networking/v2/extensions/attributestags"
	"github.com/gophercloud/utils/openstack/clientconfig"
	"k8s.io/apimachinery/pkg/runtime"

	"sigs.k8s.io/cluster-api-provider-openstack/pkg/cloud/services/provider"
	"sigs.k8s.io/cluster-api-provider-openstack/pkg/record"
)

const (
	networkPrefix string = "k8s-clusterapi"
	trunkResource string = "trunks"
	portResource  string = "ports"
)

// Service interfaces with the OpenStack Networking API.
// It will create a network related infrastructure for the cluster, like network, subnet, router, security groups.
type Service struct {
	projectID string
	client    NetworkClient
	logger    logr.Logger
}

// NewService returns an instance of the networking service.
func NewService(providerClient *gophercloud.ProviderClient, clientOpts *clientconfig.ClientOpts, logger logr.Logger) (*Service, error) {
	serviceClient, err := openstack.NewNetworkV2(providerClient, gophercloud.EndpointOpts{
		Region: clientOpts.RegionName,
	})
	if err != nil {
		return nil, fmt.Errorf("failed to create networking service providerClient: %v", err)
	}

	if clientOpts.AuthInfo == nil {
		return nil, fmt.Errorf("failed to get project id: authInfo must be set")
	}

	projectID, err := provider.GetProjectID(providerClient, clientOpts)
	if err != nil {
		return nil, fmt.Errorf("error retrieveing project id: %v", err)
	}

	return &Service{
		projectID: projectID,
		client:    networkClient{serviceClient},
		logger:    logger,
	}, nil
}

// NewTestService returns a Service with no initialisation. It should only be used by tests.
func NewTestService(projectID string, client NetworkClient, logger logr.Logger) *Service {
	return &Service{
		projectID: projectID,
		client:    client,
		logger:    logger,
	}
}

// replaceAllAttributesTags replaces all tags on a neworking resource.
// the value of resourceType must match one of the allowed constants: trunkResource or portResource.
func (s *Service) replaceAllAttributesTags(eventObject runtime.Object, resourceType string, resourceID string, tags []string) error {
	if resourceType != trunkResource && resourceType != portResource {
		record.Warnf(eventObject, "FailedReplaceAllAttributesTags", "Invalid resourceType argument in function call")
		panic(fmt.Errorf("invalid argument: resourceType, %s, does not match allowed arguments: %s or %s", resourceType, trunkResource, portResource))
	}

	_, err := s.client.ReplaceAllAttributesTags(resourceType, resourceID, attributestags.ReplaceAllOpts{
		Tags: tags,
	})
	if err != nil {
		record.Warnf(eventObject, "FailedReplaceAllAttributesTags", "Failed to replace all attributestags, %s: %v", resourceID, err)
		return err
	}

	record.Eventf(eventObject, "SuccessfulReplaceAllAttributeTags", "Replaced all attributestags for %s with tags %s", resourceID, tags)
	return nil
}
