package utils

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
)

// FakeFilesystem allows to setup isolated fake files structure used for the tests.
type FakeFilesystem struct {
	RootDir  string
	Dirs     []string
	Files    map[string][]byte
	Symlinks map[string]string
}

// Use function creates entire files structure and returns a function to tear it down. Example usage: defer fs.Use()()
func (fs *FakeFilesystem) Use() func() {
	// create the new fake fs root dir in /tmp/sriov...
	tmpDir, err := ioutil.TempDir("", "sriov")
	if err != nil {
		panic(fmt.Errorf("error creating fake root dir: %s", err.Error()))
	}
	fs.RootDir = tmpDir

	for _, dir := range fs.Dirs {
		err := os.MkdirAll(path.Join(fs.RootDir, dir), 0755)
		if err != nil {
			panic(fmt.Errorf("error creating fake directory: %s", err.Error()))
		}

	}
	for filename, body := range fs.Files {
		err := ioutil.WriteFile(path.Join(fs.RootDir, filename), body, 0644)
		if err != nil {
			panic(fmt.Errorf("error creating fake file: %s", err.Error()))
		}
	}
	err = os.MkdirAll(path.Join(fs.RootDir, "usr/share/hwdata"), 0755)
	if err != nil {
		panic(fmt.Errorf("error creating fake directory: %s", err.Error()))
	}

	// TODO: Remove writing pci.ids file once ghw is mocked
	// This is to fix the CI failure where ghw lib fails to
	// unzip pci.ids file downloaded from internet.
	pciData, err := ioutil.ReadFile("/usr/share/hwdata/pci.ids")
	if err != nil {
		panic(fmt.Errorf("error reading file: %s", err.Error()))
	}
	err = ioutil.WriteFile(path.Join(fs.RootDir, "usr/share/hwdata/pci.ids"), pciData, 0644)
	if err != nil {
		panic(fmt.Errorf("error creating fake file: %s", err.Error()))
	}

	for link, target := range fs.Symlinks {
		err = os.Symlink(target, path.Join(fs.RootDir, link))
		if err != nil {
			panic(fmt.Errorf("error creating fake symlink: %s", err.Error()))
		}
	}

	sysBusPci = path.Join(fs.RootDir, "/sys/bus/pci/devices")

	return func() {
		// remove temporary fake fs
		err := os.RemoveAll(fs.RootDir)
		if err != nil {
			panic(fmt.Errorf("error tearing down fake filesystem: %s", err.Error()))
		}
	}
}
