package rpm

import (
	"context"
	"net/http"
	"os/exec"
	"testing"
	"time"

	"github.com/google/go-cmp/cmp"
	"github.com/quay/zlog"

	"github.com/quay/claircore"
	"github.com/quay/claircore/test/fetch"
)

func TestScan(t *testing.T) {
	hash, err := claircore.ParseDigest("sha256:729ec3a6ada3a6d26faca9b4779a037231f1762f759ef34c08bdd61bf52cd704")
	if err != nil {
		t.Fatal(err)
	}
	want := []*claircore.Package{
		&claircore.Package{
			Name:           "tzdata",
			Version:        "2019a-1.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "tzdata", Version: "2019a-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d7e32f5f28a029a7fd4ed9a1d0ac3be568c2c5a926e787202a1c3fbbaf70cf5c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "ncurses-base",
			Version:        "6.1-7.20180224.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "ncurses", Version: "6.1-7.20180224.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:198e189345f93894eb71fac81c54816b6d303d03831abd15bdc98079f0f86fcb|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dnf-data",
			Version:        "4.0.9.2-5.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dnf", Version: "4.0.9.2-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:1d39c148148c09be1273ea8bfa5626d39f30a7d3633499f972ecd4415a9257e8|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dbus-common",
			Version:        "1:1.12.8-7.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dbus", Version: "1.12.8-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:78aad94d6baf0eec001772b09959cb2fb758d57484ad4fc171f211914b014ad9|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "setup",
			Version:        "2.12.2-1.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "setup", Version: "2.12.2-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b61ced6a392c34c287a0686d81f66d3e660c40a6108ac5698f4a4696aed4fff2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "basesystem",
			Version:        "11-5.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "basesystem", Version: "11-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:fa127960f98c176c3e57432cbd96a74767f5fa803db325c0eacae18b47d52105|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libselinux",
			Version:        "2.8-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libselinux", Version: "2.8-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:01fd401648619764cbf68a489e83451602c898495fa4520e41c48c48f62a4e36|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "glibc-minimal-langpack",
			Version:        "2.28-42.el8.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "glibc", Version: "2.28-42.el8.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:fa127960f98c176c3e57432cbd96a74767f5fa803db325c0eacae18b47d52105|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "glibc",
			Version:        "2.28-42.el8.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "glibc", Version: "2.28-42.el8.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:ca20e19a3a4cc6b4b8b84aea620975c1a51a5d9e84649808ae15075a07f0d7ea|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libsepol",
			Version:        "2.8-2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libsepol", Version: "2.8-2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:ee8fecca7fde8168278953711f935a04c309eda37d9f5647cecf25ff659f7730|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "xz-libs",
			Version:        "5.2.4-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "xz", Version: "5.2.4-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8694f6056932283de5b0f277bda721ce9ff5d7d217a12175cde6f23187c2a4b1|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libcap",
			Version:        "2.25-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libcap", Version: "2.25-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:897f61e794f569c44820f011b6e4fbfc19fc79e4a50faca94a0c5da658ccda91|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libgpg-error",
			Version:        "1.31-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libgpg-error", Version: "1.31-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:ef78b78f406942ae056c86b3096b0eabc9f2bf5dd63f9685c6d7a28b385c2f4a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libcom_err",
			Version:        "1.44.3-2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "e2fsprogs", Version: "1.44.3-2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:bebfaaf923400e5f7cad4000c963f7717c36f4e5f82f0d72f3db86f80a493621|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libxml2",
			Version:        "2.9.7-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libxml2", Version: "2.9.7-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:f09f1d58921b7a592b9a7c2836676ef4c8b55cd7192395ed3b74ee771fbbb589|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "expat",
			Version:        "2.2.5-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "expat", Version: "2.2.5-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:08d977c11da01f64f916d0e6d26200ca286391b5166df9998ba5c2dbfebcb691|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libuuid",
			Version:        "2.32.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "util-linux", Version: "2.32.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7b1331cda22a580eff4882fd7b7b92cc8066e9e88b5072569b4015a6ff19866a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "chkconfig",
			Version:        "1.11-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "chkconfig", Version: "1.11-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:bd0102959bf6176a832d60f7ac16d42270e3b6422a2299326266a75e06014ae9|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gmp",
			Version:        "1:6.1.2-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gmp", Version: "6.1.2-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8262c6291fbd17ff6043261d8f9769bf5a3304639da7c25c6145baa55b37462b|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libattr",
			Version:        "2.4.48-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "attr", Version: "2.4.48-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:4305eea35033526e55f5043bdc963e58b79a1f9500bb1a6f5b937b552306ca4e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "coreutils-single",
			Version:        "8.30-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "coreutils", Version: "8.30-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2bd25cb124f007586a6e47d0d8f17e19deef98d0ad3471221c916fa08c01ba48|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libblkid",
			Version:        "2.32.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "util-linux", Version: "2.32.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:f4ffcd322ff77998f1940ea057480e2aad29ae28e4546c3e7eb2b0cb339b6a97|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libcap-ng",
			Version:        "0.7.9-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libcap-ng", Version: "0.7.9-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:972a95960a481af19fd0a71750abbc33a5537ef403f81d6c0d8281e26fbe65b4|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libffi",
			Version:        "3.1-18.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libffi", Version: "3.1-18.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:9b3b30a8a81f7d1b03d43ccc0b356a4cc7ed5d309ab7030ffb9a13b864272f29|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "lua-libs",
			Version:        "5.3.4-10.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "lua", Version: "5.3.4-10.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:ae89b936f5d6d9da2a0358a2dd1f1a364c8947b868cb86a0a15fdd167ecebf8e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "p11-kit",
			Version:        "0.23.14-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "p11-kit", Version: "0.23.14-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:17efd6d31955ee8f51e840ecbaa906e2f9b39e0c9bc16f67f6d6af2d33f09792|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gzip",
			Version:        "1.9-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gzip", Version: "1.9-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7a65a5e253d82c342a2e56e37d7d2ca8c59d43153f877e4c23bf2bc06c47f87c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libassuan",
			Version:        "2.5.1-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libassuan", Version: "2.5.1-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:125a1910e1bd3aedee5c6aa7b7a0f0a758178849b004916a4ca25d569a54cc8d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libidn2",
			Version:        "2.0.5-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libidn2", Version: "2.0.5-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b7e550df988802e651ff53d9f74a09f04439875fbf04728dd7dc21affa2bb400|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gdbm-libs",
			Version:        "1:1.18-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gdbm", Version: "1.18-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:5a5327261b2078028882c990e909e9d30fbf796319b619ac460b8f867c1111a6|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libtasn1",
			Version:        "4.13-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libtasn1", Version: "4.13-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2f5301c3c9c5095667e754aee9dd9289c730bd22fd16f608ee190abe06d111c8|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "lzo",
			Version:        "2.08-14.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "lzo", Version: "2.08-14.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:89c7940be2ba590c182539c7cbbc6d875ca9d91899af70262c1ba7aad0e88a7a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "grep",
			Version:        "3.1-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "grep", Version: "3.1-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:eb9a23a41775c978a663c37cae8b8166e0cf95120b2df4554104879afc4dec84|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "glib2",
			Version:        "2.56.4-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "glib2", Version: "2.56.4-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:de8a077d6d178c638670fb73ead324c6e4b306371f6b1760c88c74dc1dc17e4c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dbus-libs",
			Version:        "1:1.12.8-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dbus", Version: "1.12.8-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d35f0c6f32c7c91af1960f54753d1480751c211a017fbc6d3990ef9c48d0c1d2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "openssl-libs",
			Version:        "1:1.1.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "openssl", Version: "1.1.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8fd2a342447a88a225120b87bbcb2772223d6d9bf708ef64fdee39eb9f891f92|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "kmod-libs",
			Version:        "25-11.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "kmod", Version: "25-11.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:bb6192209d173080d53d6cd1835ce8b3bd8f873770cab307a065ec6dae9fcf7c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "kmod",
			Version:        "25-11.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "kmod", Version: "25-11.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:434f99c93cd522ec93798f0c22e38eab3209eaa2aad0842967c13bf3da1e1ecf|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libarchive",
			Version:        "3.3.2-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libarchive", Version: "3.3.2-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:db90fd1df0e1640d5e5793fcfdce6355a86f98fa65bd32d649d3ea093695d1bc|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dhcp-libs",
			Version:        "12:4.3.6-30.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dhcp", Version: "4.3.6-30.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2c55ad3dff760f9fb0369a7bf4a8e9c2f62643c5a6784a836ee5d5a020a20136|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "procps-ng",
			Version:        "3.3.15-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "procps-ng", Version: "3.3.15-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:e5bb29d135facd79163c6c9a2d172778a96a77b03b368d6009f081046b40931e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "squashfs-tools",
			Version:        "4.3-17.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "squashfs-tools", Version: "4.3-17.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:e6850f0458d5389ee6da189f668936a35320ec388cf78cb2705f404fc790b121|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libsemanage",
			Version:        "2.8-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libsemanage", Version: "2.8-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:881856796644b3feb1a813cabe229cf49f03ce010dc6355aaa3a311c8888ebf2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dbus-daemon",
			Version:        "1:1.12.8-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dbus", Version: "1.12.8-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:5f998117eaec75d9f037ca5eb011dad0417dac04df588e882d569da452c4bf6e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libfdisk",
			Version:        "2.32.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "util-linux", Version: "2.32.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:4a249ff76734cd511c62bd2c868bc3849bd018a511b7f2673992527da86f5c66|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "mpfr",
			Version:        "3.1.6-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "mpfr", Version: "3.1.6-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d87079c0d4956f1d1040142a27ed979025702e30825679f991ac728194a659e4|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gnutls",
			Version:        "3.6.5-2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gnutls", Version: "3.6.5-2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:4ff33e6e9f7c7e0808603ba867a347dc2210f74e629b4041cf2756f9ae5294c8|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libcomps",
			Version:        "0.1.8-13.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libcomps", Version: "0.1.8-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:80b60fb3c4e4298985f621f0e90fa8ed23606f93ab1e10d597dca654dc067484|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libksba",
			Version:        "1.3.5-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libksba", Version: "1.3.5-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d5dad8c759e726a82bd638e021a4cc9e28aeeef396ac19b4a9c8b005faaa720c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "cpio",
			Version:        "2.12-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "cpio", Version: "2.12-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7c20ffcc208509068a0ebbc230cbc61ad5318bbadebce8458944507338c2b24f|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "ipcalc",
			Version:        "0.2.4-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "ipcalc", Version: "0.2.4-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:6fea29fb4705ce72e9434c67359d8d28adecb9a33c74448e32a8631f0f73fe59|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "iproute",
			Version:        "4.18.0-11.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "iproute", Version: "4.18.0-11.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:76d8af499d1fcc15ce44916c4f8f979344cea3283c482283be77535e1ad4615e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libpkgconf",
			Version:        "1.4.2-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pkgconf", Version: "1.4.2-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b26a3437a2c18586d600d672c81242e8bf82ed749848c130fd05f57aaadac68e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "pkgconf-pkg-config",
			Version:        "1.4.2-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pkgconf", Version: "1.4.2-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:eff64ebb4a63a0d685adc24caa7b8d91845aa82451af90018a803fafb71d4811|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "iptables-libs",
			Version:        "1.8.2-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "iptables", Version: "1.8.2-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:5133c7170a13b625089db4043d58a9b861024a6a003d835ea11265110a670a58|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libsigsegv",
			Version:        "2.11-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libsigsegv", Version: "2.11-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8ef967becb561e4a833980ee6912d22d5a96331f9bc987c8747d968b5bf0f1d7|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libverto",
			Version:        "0.3.0-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libverto", Version: "0.3.0-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:37d6350891458ab76e74b450d7b5345f5ea930b433880deb62861b40efabbf4d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libtirpc",
			Version:        "1.1.4-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libtirpc", Version: "1.1.4-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2c2c3b2f61bfb031fe47913d988b260553ec8c731cf9caf0fbf8a7fa17a02588|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "platform-python-pip",
			Version:        "9.0.3-13.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "python-pip", Version: "9.0.3-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:a17ca26c61d2358056d2d84a9b6cd6d045a52315b4a071d45259cc5d9826e6e2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "platform-python",
			Version:        "3.6.8-1.el8.0.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "python3", Version: "3.6.8-1.el8.0.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8b6d99e81232845cdf0300a88aca99fcf7b69b837eece6c7c421d74be7a01084|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libpwquality",
			Version:        "1.4.0-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libpwquality", Version: "1.4.0-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d0d4c07d84d6b99525b9159cfa27a94d310319c472424c9dc4338cf3291bbce5|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "util-linux",
			Version:        "2.32.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "util-linux", Version: "2.32.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d3a9eb395bf4bdab98e73f30bcb2bf253acd06b9ae61c9e8617a8addbedd3e21|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "curl",
			Version:        "7.61.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "curl", Version: "7.61.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:49a45974fa83806e3942ebcad2c8bf72e6f3337fd3b9f94ebd526c0b438b98db|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "rpm-libs",
			Version:        "4.14.2-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "rpm", Version: "4.14.2-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:c0de3a7135f47b510eddde4d885e00afd072bd41ef45b1b597b1081d00c2a796|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "device-mapper",
			Version:        "8:1.02.155-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "lvm2", Version: "2.03.02-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b01c1445b0d9d05d9dc80e38e4b293d156cad5472994ace856e67bf6ae5d6a7d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "cryptsetup-libs",
			Version:        "2.0.6-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "cryptsetup", Version: "2.0.6-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:03717069bc76838257d6254f03aace77f5e9c4c47dbaa5917c15871327820622|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "elfutils-libs",
			Version:        "0.174-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "elfutils", Version: "0.174-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7aabee2dda39d1e1081ac6cdd6fab8990e8aab183ba9026897b0feaa755a74eb|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "systemd",
			Version:        "239-13.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "systemd", Version: "239-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:18edae8342701d20cfbb4881d1e6a13da188788282e45d7844591d36dccc6198|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "iputils",
			Version:        "20180629-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "iputils", Version: "20180629-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:89e074b2d8fe4e935a648edf0853354e728c1f42f7c5c8ffee24ad969fbfe085|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libkcapi-hmaccalc",
			Version:        "1.1.1-16_1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libkcapi", Version: "1.1.1-16_1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:1db37d679dbc3e413d5c785c69a02ff3db9df0d7a8ac8552ad728da4df5d4239|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dracut",
			Version:        "049-10.git20190115.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dracut", Version: "049-10.git20190115.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:a612253fab9c7daa22e13cdfe8b0bbebd03be5b6839206248c53bd5549b30da3|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-libcomps",
			Version:        "0.1.8-13.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libcomps", Version: "0.1.8-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:9ce8ebad74c1dd410e8d300f024a6cb43e1611fe447a02139c66f8a3db63710d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-iniparse",
			Version:        "0.4-31.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "python-iniparse", Version: "0.4-31.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8f54f5b9503c405dea9e8d55438d3d364e961bd4adc80942757f82b03614427e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dhcp-client",
			Version:        "12:4.3.6-30.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dhcp", Version: "4.3.6-30.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:c50cefb76514f0e995490ce4998955b3b2323ca2de7f4f665de1b65d68a64386|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "cyrus-sasl-lib",
			Version:        "2.1.27-0.3rc7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "cyrus-sasl", Version: "2.1.27-0.3rc7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:cc76a7d3107fd59aab4997f9d72355ecb55a22b02d15154c7f752a6dcbac446d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libyaml",
			Version:        "0.1.7-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libyaml", Version: "0.1.7-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d7948b4fd90d0e92bcfd9e9f5fc9d2fd871259e528dbc594fdfb528e83f6f41d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "npth",
			Version:        "1.5-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "npth", Version: "1.5-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:239d0769bf245c8714b159c9d1fb31618d86e85afdaca085ca7c145bcf81307c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gpgme",
			Version:        "1.10.0-6.el8.0.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gpgme", Version: "1.10.0-6.el8.0.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:f56057290d8aace26d2142f45c55193d59d6ff93c2f25d3ea8751ced7c139518|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libdnf",
			Version:        "0.22.5-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libdnf", Version: "0.22.5-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:6a025ed38a3633e5f89c28d3fbc84d9dec0bb04549ae045ddd3728738032cb3a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-hawkey",
			Version:        "0.22.5-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libdnf", Version: "0.22.5-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d8ae3aa132427a2e26900d332c1993e249e23c584fb365d2e138e9913741359e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "rpm-build-libs",
			Version:        "4.14.2-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "rpm", Version: "4.14.2-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:10937ed3ff0ccfc8ef1114cbb45160f559cbf055f0cd6668e151ba1ecdf6a077|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-dnf",
			Version:        "4.0.9.2-5.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dnf", Version: "4.0.9.2-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:60b7ea649dcdb08af67e003a19074197a5a5226d6cd7189be14bd2437d3eba8e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "yum",
			Version:        "4.0.9.2-5.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dnf", Version: "4.0.9.2-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:edcce9280e083d452153361916428af9987ad521b14d744ed8af455e796db5c3|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "binutils",
			Version:        "2.30-49.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "binutils", Version: "2.30-49.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:352f73bb853581e9037bd4cffb67e81a35d2453610d60eb18c4293f212fc01e2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "vim-minimal",
			Version:        "2:8.0.1763-10.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "vim", Version: "8.0.1763-10.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:27cef2a26cbe4b678aa716df3ffcd12af8a60dbfc91b5f17b0257bc12f26ae8c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "less",
			Version:        "530-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "less", Version: "530-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:bb2e5b9746e4f479ebcc5d3c10fe8ecfd3aa74ee2e0a18b0c3375b221592d59a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "rootfiles",
			Version:        "8.1-22.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "rootfiles", Version: "8.1-22.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2a98466dc0e09984721d45c869a18d77a8914602e16e070b294b28d4632385c7|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libgcc",
			Version:        "8.2.1-3.5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gcc", Version: "8.2.1-3.5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:847423825bfc1689f71dacfddc03a1da120090816358eb08cfed0d5aa2bb4cfa|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "pkgconf-m4",
			Version:        "1.4.2-1.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pkgconf", Version: "1.4.2-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7e4935774062919285cf636e45243330e3c34b818bc5723c53ea7659e6eb9068|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libreport-filesystem",
			Version:        "2.9.5-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libreport", Version: "2.9.5-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:4139659800d97238058035061cb9621c82756a62c41651c4a690fa865d1a3f2b|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dhcp-common",
			Version:        "12:4.3.6-30.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dhcp", Version: "4.3.6-30.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:320f84bfbf31a518c3763485e5fc2d8dc3853ed9e06737343bb0b7714cc81f48|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "centos-release",
			Version:        "8.0-0.1905.0.9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "centos-release", Version: "8.0-0.1905.0.9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:fe84e3755fc24a31e60dd8d5d3e910ecf01683bd266ef63e3b45c7f7c6d7c196|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "filesystem",
			Version:        "3.8-2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "filesystem", Version: "3.8-2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:28d85a65f1cea65de75a2cc90a1027dfc6745dcfc0f0568fc9a8502230f2a2fd|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "pcre2",
			Version:        "10.32-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pcre2", Version: "10.32-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d14d99ac7f78178d0307273296c5a121ac62aa4f4160c328d697ae80014c7d91|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "ncurses-libs",
			Version:        "6.1-7.20180224.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "ncurses", Version: "6.1-7.20180224.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:973c3259be26361f886c5e87e6e88230b54e24111d30d2d34149fd16946480ee|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "glibc-common",
			Version:        "2.28-42.el8.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "glibc", Version: "2.28-42.el8.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:12cd1a5700218c9f54bb43b7833c8fa0105fd12eb156da7ad53495b7d362ff4a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "bash",
			Version:        "4.4.19-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "bash", Version: "4.4.19-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b4c385ed5f03c511ea189937cca0806f2b72e62e6e55b2c7f58b927b8f09ebf9|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "zlib",
			Version:        "1.2.11-10.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "zlib", Version: "1.2.11-10.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:294731059e720f86656d3dce46a700dfca01a4ac861f8cf42e80eb7dc15c95ae|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "bzip2-libs",
			Version:        "1.0.6-26.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "bzip2", Version: "1.0.6-26.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:47c78745b26ed5be27eb31c3f2603f5b9572fb3801799acab06a2dbbc510dbcb|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "info",
			Version:        "6.5-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "texinfo", Version: "6.5-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:211d7e76cdffb417db117ca3d10abc9ecd3e60132c928cd5bb4440a2f6db14bb|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "elfutils-libelf",
			Version:        "0.174-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "elfutils", Version: "0.174-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2287da8c39b3bd442ef8637a4acefbf3d4a6a3249d57145866334710d1f7188b|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libxcrypt",
			Version:        "4.1.1-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libxcrypt", Version: "4.1.1-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:4824a32a4a9b5d154f7ca7bf808012eecd70e1bebe249aec064b0121a54e3d8d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "sqlite-libs",
			Version:        "3.26.0-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "sqlite", Version: "3.26.0-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:3330aeb4246959fa47ae7ca7b853f419ac33b9039bc1b9f2cc68afcbc57f4a64|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libstdc++",
			Version:        "8.2.1-3.5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gcc", Version: "8.2.1-3.5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d3f35907911a5556f5ba1f5eb6b326051f13e2bb9c608729b59a53a4d05002a5|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "popt",
			Version:        "1.16-14.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "popt", Version: "1.16-14.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:f4e07c1baa0a68c09bac8b913b7851b8595d6d33bacd53534127d6f56760707e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "readline",
			Version:        "7.0-10.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "readline", Version: "7.0-10.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:fc00ff55f5026585d089ef1b6c6226e0ce1b60da8b6cd75e19b6b61e499f7f4a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "json-c",
			Version:        "0.13.1-0.2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "json-c", Version: "0.13.1-0.2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:17835daff24726949b52b534bb294ea014fec5ad3398704aa142d4a54e0e684c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libacl",
			Version:        "2.2.53-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "acl", Version: "2.2.53-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8e89edde49807da4391927c04fe5236a0da7d36e7437002386fedda1f5a4e527|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "sed",
			Version:        "4.5-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "sed", Version: "4.5-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:682b016c8b5a80d4f797d4c9f1b74357f42b8b0fc382c76c35407345aba8f35e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libmount",
			Version:        "2.32.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "util-linux", Version: "2.32.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:25229bfc1bdf1e5d00600719813ec431805349db57fa3288a61cf2f4109582c5|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "audit-libs",
			Version:        "3.0-0.10.20180831git0047a6c.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "audit", Version: "3.0-0.10.20180831git0047a6c.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d84a722fba1d9be6626c5df380b7139a17e019382b133bf77de9527321334ea4|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libsmartcols",
			Version:        "2.32.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "util-linux", Version: "2.32.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:e318db8fdde219b303b17a47696d3e15762006e1b3a9fda56e8f70aba831b104|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "lz4-libs",
			Version:        "1.8.1.2-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "lz4", Version: "1.8.1.2-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:9b61743ad828418c11cf6cc5df84ffe33df8029cbf66dd4201fd807d577e79b2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libgcrypt",
			Version:        "1.8.3-2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libgcrypt", Version: "1.8.3-2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b018ee0c3c9c7cd43eb12fe8d9f1d87e895e3edc17dd14bbbba4ce952b405c90|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "cracklib",
			Version:        "2.9.6-15.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "cracklib", Version: "2.9.6-15.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:fd0ef8f701b21567253a6fb60fa62ba4613861f2b8cdbdb86c3d1ff9fbc4efff|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libunistring",
			Version:        "0.9.9-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libunistring", Version: "0.9.9-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:6cfbea1d92f367e7c123690b5f85787c197449eadcb9f1125bfa563e66c00959|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "file-libs",
			Version:        "5.33-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "file", Version: "5.33-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:4cc51b1e501ecc26f44d35edb9339dfd244e2d3f69e6461380d35f3e0c60b431|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "keyutils-libs",
			Version:        "1.5.10-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "keyutils", Version: "1.5.10-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:abae45bc135c128d35073e3ace81da43505ef991d41e9cfa903bbb7ca0322fd1|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "p11-kit-trust",
			Version:        "0.23.14-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "p11-kit", Version: "0.23.14-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:3f93570de11b5bdc1620b49a6e70b4792deff2bc7f1f2a1da595bb92930553d4|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "pcre",
			Version:        "8.42-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pcre", Version: "8.42-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:822a78f1a57b439511d3edd2d943f51f0393e6cb6713bc4a8cca994326f360ac|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "systemd-libs",
			Version:        "239-13.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "systemd", Version: "239-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:3026b57c21d436ee69d56c0d6cb3ead0b2efb383f5000e5da35221f34ac387b8|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "crypto-policies",
			Version:        "20181217-6.git9a35207.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "crypto-policies", Version: "20181217-6.git9a35207.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:41950f001f00292a7a46f668ec915de747b9b00fb2cb90c37ce37424f082028f|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "ca-certificates",
			Version:        "2018.2.24-6.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "ca-certificates", Version: "2018.2.24-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:a3778bda947716315a2aa5fdb3dba7358d507d9d910ae509e1a90178884b831f|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libdb",
			Version:        "5.3.28-36.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libdb", Version: "5.3.28-36.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2bde2c8d9a7a826a3c675e898096ff694d0f0daf1ff2c0198c4b48df3b1a62df|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "ima-evm-utils",
			Version:        "1.1-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "ima-evm-utils", Version: "1.1-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:5507d55b18ed310c90a7ba3397210efdd19b7036dfa7aeaf335fa9bf7c714bea|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libdb-utils",
			Version:        "5.3.28-36.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libdb", Version: "5.3.28-36.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7c11e7a417437b36c80d7e6df1c87d0aca248dbd3fbe054418522d737e462165|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dbus-tools",
			Version:        "1:1.12.8-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dbus", Version: "1.12.8-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2385478eb665b11efcc60ee45ab4a4e1d4836f4263142bad1733f6301b9e2130|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libusbx",
			Version:        "1.0.22-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libusbx", Version: "1.0.22-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:60c7ca280fc3b6d01fca78bcc74fed36fc249cbd095c1e5b12c44a970bb14f41|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "xz",
			Version:        "5.2.4-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "xz", Version: "5.2.4-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:26e73b37d5e13d2e80c7eb6c52bd662538059d89f3b4b73a13a019467496991d|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gdbm",
			Version:        "1:1.18-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gdbm", Version: "1.18-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:38a0a084c7d171b9f00ca5fa33126cf60606af87996dc7dfdae578078df065e2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "shadow-utils",
			Version:        "2:4.6-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "shadow-utils", Version: "4.6-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:0c67edd6e2020b5393a377026ba914276d6cac44c7ebc01dd83d72bc59100a52|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libutempter",
			Version:        "1.1.6-14.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libutempter", Version: "1.1.6-14.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:de4deec2a205db42de5d15899ed3fa75b2c0a68fcce053ebfb488262c38c1272|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "acl",
			Version:        "2.2.53-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "acl", Version: "2.2.53-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:65af1230ea97986fe3e79a8381ab7deea40fd9073dc380e58e1070b954912297|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "nettle",
			Version:        "3.4.1-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "nettle", Version: "3.4.1-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2e9fc253e3b189524c710e32784dee974f8d68e00cc13f5cd376f72c3c507a37|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "snappy",
			Version:        "1.1.7-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "snappy", Version: "1.1.7-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7a5daa20e22e94cc46f956a90d1d8b425bcc934eefb6c5a08a3840df28c17e20|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libmetalink",
			Version:        "0.1.3-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libmetalink", Version: "0.1.3-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:a18613c99fa5c0c718cb4e16023331047de8f1e050388c48bcc1ff07d77ee074|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "findutils",
			Version:        "1:4.6.0-20.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "findutils", Version: "4.6.0-20.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d81c8b387d276fd99bf90d58db3c06b4385f93051d6cf66b3d7b1cc04458d94f|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "ethtool",
			Version:        "2:4.16-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "ethtool", Version: "4.16-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:959880aa8d1418eefe9de0d504f0e6843e2c2998c0194816713b2678d132de23|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libmnl",
			Version:        "1.0.4-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libmnl", Version: "1.0.4-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:481d76ead6bd438882cd12e4cddeb086a386dec5886055e85cfb78fda081fe29|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libnghttp2",
			Version:        "1.33.0-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "nghttp2", Version: "1.33.0-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:cca0b5ec230d1a27453a8d85176d5c9c00e7604269339b1dc1b912a8b90aee97|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "pkgconf",
			Version:        "1.4.2-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pkgconf", Version: "1.4.2-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:7a4067f2f48fa82bff14356d601702e226cdd0246e2ff1953d8fde384457846c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libpcap",
			Version:        "14:1.9.0-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libpcap", Version: "1.9.0-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d0442eca6382626563cf35b783b5f41cc666426e4d9ff6741a1a2ae54ec51079|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libseccomp",
			Version:        "2.3.3-3.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libseccomp", Version: "2.3.3-3.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:3e8ef42392e684455ed7c63500c296268dc6500bc86b35b57c9d21873b214bdc|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gawk",
			Version:        "4.2.1-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gawk", Version: "4.2.1-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:63b029da02286255977312a750ece107b8f28a29df9d0f85de2bbd18d9928498|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "krb5-libs",
			Version:        "1.16.1-22.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "krb5", Version: "1.16.1-22.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:cbcc23ff27479b930f74c47c4da9e426106adc4439526f00d2100df131d389b8|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libnsl2",
			Version:        "1.2.0-2.20180605git4a062cf.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libnsl2", Version: "1.2.0-2.20180605git4a062cf.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8ea76352c8e248f3c26b773d9de3a160960e6a6fb76ddb8583175a2227d516ef|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "platform-python-setuptools",
			Version:        "39.2.0-4.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "python-setuptools", Version: "39.2.0-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:bf1f60991ba170187ce0a8a0742c4c656613886777e04ebe88d4077516eafcdd|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-libs",
			Version:        "3.6.8-1.el8.0.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "python3", Version: "3.6.8-1.el8.0.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:6b878c69858ac7b1da44c91dc7dd25c857c9085d636fbbc96dd0fdbcbd16b43c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "pam",
			Version:        "1.3.1-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "pam", Version: "1.3.1-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:f6801a288957cf9da1c3f25cf923e08ac4a8cce76c8830bebe20d06756c7c8c6|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libcurl-minimal",
			Version:        "7.61.1-8.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "curl", Version: "7.61.1-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:dd3252e3c4326c9566a15c9a9fdf068822970cbc85e8f9bb3e0bc6c8f38d2763|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "rpm",
			Version:        "4.14.2-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "rpm", Version: "4.14.2-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:cfa9f7711b0ff9f9c2c8879b1d902d5cd211c6dcde089ef4c514c1ef7fc197e6|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libsolv",
			Version:        "0.6.35-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libsolv", Version: "0.6.35-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:9cc540d9c389e9b8ffa3fb5f25c6ee126f81c142043e5f33499ff4b0eb68d815|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "device-mapper-libs",
			Version:        "8:1.02.155-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "lvm2", Version: "2.03.02-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:2accf23b7986d0a834c7033c2a003de4f0e957a6598d3ee3dbe40d5af395f738|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "elfutils-default-yama-scope",
			Version:        "0.174-6.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "elfutils", Version: "0.174-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:e59050e14d6b11165c09f020deb0c58cac3a9fd7cdb6c1a2da429fb34c9ee961|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "systemd-pam",
			Version:        "239-13.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "systemd", Version: "239-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:04a90a0a7cee7bbc1fe433592a664fd2fc7695ad62e63d0319fb21a28aa657e5|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dbus",
			Version:        "1:1.12.8-7.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dbus", Version: "1.12.8-7.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:fa127960f98c176c3e57432cbd96a74767f5fa803db325c0eacae18b47d52105|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libkcapi",
			Version:        "1.1.1-16_1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libkcapi", Version: "1.1.1-16_1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:85e0a0bf27ae8bd85227284bce4d5755d9df4d76636e9e674d40245c1f1d53e3|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "systemd-udev",
			Version:        "239-13.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "systemd", Version: "239-13.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:06f37071663c4c6007dfcc252c52caea5207059ed41368acd232bdcc2bf4df54|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dracut-squash",
			Version:        "049-10.git20190115.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dracut", Version: "049-10.git20190115.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d83c42b5837f392e1f2fadcec976f864da0c4a445453931784c3b0ef26efe268|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-six",
			Version:        "1.11.0-8.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "python-six", Version: "1.11.0-8.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:3cf7b959035f68ffa0568a421cb52b5b9ac18d0ed33b1758ced4c53824dec052|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "bind-export-libs",
			Version:        "32:9.11.4-16.P2.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "bind", Version: "9.11.4-16.P2.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:937fc7723d74e019dc1ed08a96d184798fe5ef787e3acf6149def2701d71543a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dracut-network",
			Version:        "049-10.git20190115.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dracut", Version: "049-10.git20190115.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:32c2c2ff8aa3b0b359f0caaa363d3c60bca96b17f60a11798ceeebd404822225|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "openldap",
			Version:        "2.4.46-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "openldap", Version: "2.4.46-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:10d4086b87d0241e00c2974a21b81f7fd191ff63a9a5fe7b645e124b68e1133a|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "libmodulemd1",
			Version:        "1.8.0-5.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libmodulemd", Version: "2.0.0-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:552186ee8a462d8aa824886240f0ab6595aacbda85961ba46fd6faa01093d4f8|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "gnupg2",
			Version:        "2.2.9-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gnupg2", Version: "2.2.9-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:c8ff770f39becd78672e71a097c8dc85f0bbc224c34c7cbde9174f4f37f4eb78|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "librepo",
			Version:        "1.9.2-1.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "librepo", Version: "1.9.2-1.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:b2ce0987d742e94ee121873d7991c93c20f55e07fca1cdb48b17b7dddc978038|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-libdnf",
			Version:        "0.22.5-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "libdnf", Version: "0.22.5-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:a9ff479162d2b4410d3c66dc30670ac95159a5a1e3d55c647d72acaea82b9383|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-gpg",
			Version:        "1.10.0-6.el8.0.1",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "gpgme", Version: "1.10.0-6.el8.0.1", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:d864cd6a05dc2ec6b186542cb06b859e9be60663641aab3a4137c0974ee4bd5c|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "python3-rpm",
			Version:        "4.14.2-9.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "rpm", Version: "4.14.2-9.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:e1afb8156d811bdaeb6fc3185584a0cd5f5657be011f9a94bc4bb86fb26eeecf|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "dnf",
			Version:        "4.0.9.2-5.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "dnf", Version: "4.0.9.2-5.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:f181fecf80dbb41748774e6d2eff24f0497f29f1bc5be4a85cdbfed9c7a45a8f|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "kexec-tools",
			Version:        "2.0.17-28.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "kexec-tools", Version: "2.0.17-28.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:8d30b05d73c50d584dedd2795787309ece57fa908f8f3fad1d7d060c5827accb|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "tar",
			Version:        "2:1.30-4.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "tar", Version: "1.30-4.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:38c5c9ba7c090fdbd9d68c0aa30bc8fa82800fa5651d7d5a9e23195f90596b2e|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "hostname",
			Version:        "3.20-6.el8",
			Arch:           "x86_64",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "hostname", Version: "3.20-6.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:923455b73c2fdb3dd5bfb805e00ba058c6654c1b1a69a73d79c85fea83832ae2|key:05b555b38483c65d",
		},
		&claircore.Package{
			Name:           "langpacks-en",
			Version:        "1.0-12.el8",
			Arch:           "noarch",
			Kind:           claircore.BINARY,
			Source:         &claircore.Package{Name: "langpacks", Version: "1.0-12.el8", Kind: claircore.SOURCE},
			PackageDB:      "/var/lib/rpm",
			RepositoryHint: "hash:sha256:3c320217c4867e58178cf01c1e2d31629e794b1ee5c1473a5d6bf31c56d7d76c|key:05b555b38483c65d",
		},
	}

	// Check for the bits the package needs.
	for _, exe := range []string{
		"tar", "rpm",
	} {
		if _, err := exec.LookPath(exe); err != nil {
			t.Skipf("skipping test: missing needed utility %q (%v)", exe, err)
		}
	}
	ctx := zlog.Test(context.Background(), t)
	l := &claircore.Layer{
		Hash: hash,
	}

	tctx, done := context.WithTimeout(ctx, 2*time.Minute)
	defer done()
	n, err := fetch.Layer(tctx, t, http.DefaultClient, "docker.io", "library/centos", hash)
	if err != nil {
		t.Fatal(err)
	}
	defer n.Close()
	l.SetLocal(n.Name())

	s := &Scanner{}
	got, err := s.Scan(ctx, l)
	if err != nil {
		t.Fatal(err)
	}
	t.Logf("found %d packages", len(got))
	if !cmp.Equal(got, want) {
		t.Fatal(cmp.Diff(got, want))
	}
}
