"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _ = require("lodash");
const readPkg = require("read-pkg");
const webpack = require("webpack");
const webpack_sources_1 = require("webpack-sources");
const constants_1 = require("../constants");
const shared_modules_1 = require("../shared-modules");
const SchemaValidator_1 = require("../validation/SchemaValidator");
const ConsoleAssetPlugin_1 = require("./ConsoleAssetPlugin");
exports.validatePackageFileSchema = (pkg, description = 'package.json') => {
    const schema = ConsoleAssetPlugin_1.loadSchema('plugin-package');
    const validator = new SchemaValidator_1.SchemaValidator(description);
    if (pkg.consolePlugin) {
        validator.validate(schema, pkg.consolePlugin, 'pkg.consolePlugin');
        validator.assert.validDNSSubdomainName(pkg.consolePlugin.name, 'pkg.consolePlugin.name');
        validator.assert.validSemverString(pkg.consolePlugin.version, 'pkg.consolePlugin.version');
        if (_.isPlainObject(pkg.consolePlugin.dependencies)) {
            Object.entries(pkg.consolePlugin.dependencies).forEach(([depName, versionRange]) => {
                validator.assert.validSemverRangeString(versionRange, `pkg.consolePlugin.dependencies['${depName}']`);
            });
        }
    }
    else {
        validator.result.addError('pkg.consolePlugin object is missing');
    }
    return validator.result;
};
const remoteEntryLibraryType = 'jsonp';
const remoteEntryCallback = 'window.loadPluginEntry';
class ConsoleRemotePlugin {
    constructor() {
        this.pkg = readPkg.sync({ normalize: false });
        exports.validatePackageFileSchema(this.pkg).report();
    }
    apply(compiler) {
        if (!compiler.options.output.enabledLibraryTypes.includes(remoteEntryLibraryType)) {
            compiler.options.output.enabledLibraryTypes.push(remoteEntryLibraryType);
        }
        // Apply relevant webpack plugins
        compiler.hooks.afterPlugins.tap(ConsoleRemotePlugin.name, () => {
            new webpack.container.ContainerPlugin({
                name: this.pkg.consolePlugin.name,
                library: { type: remoteEntryLibraryType, name: remoteEntryCallback },
                filename: constants_1.remoteEntryFile,
                exposes: this.pkg.consolePlugin.exposedModules || {},
                overridables: shared_modules_1.sharedPluginModules.filter((m) => {
                    // ContainerPlugin throws 'module not found' error if an overridable cannot be resolved.
                    // All shared plugin modules are mandatory *except* the Console internal API module.
                    return m !== '@openshift-console/dynamic-plugin-sdk-internal'
                        ? true
                        : !!Object.assign(Object.assign({}, this.pkg.devDependencies), this.pkg.dependencies)[m];
                }),
            }).apply(compiler);
            // Generate additional Console plugin assets
            new ConsoleAssetPlugin_1.ConsoleAssetPlugin(this.pkg).apply(compiler);
            // Ignore require calls for modules that reside in Console monorepo packages
            new webpack.IgnorePlugin({
                resourceRegExp: /^@console\//,
                contextRegExp: /node_modules\/@openshift-console\/dynamic-plugin-sdk/,
            }).apply(compiler);
        });
        // Post-process generated remote entry source
        // TODO(vojtech): fix 'webpack-sources' type incompatibility when updating to latest webpack 5
        compiler.hooks.emit.tap(ConsoleRemotePlugin.name, (compilation) => {
            compilation.updateAsset(constants_1.remoteEntryFile, (source) => {
                const newSource = new webpack_sources_1.ReplaceSource(source);
                newSource.insert(remoteEntryCallback.length + 1, `'${this.pkg.consolePlugin.name}@${this.pkg.consolePlugin.version}',`);
                return newSource;
            });
        });
        // Skip processing entry option if it's missing or empty
        // TODO(vojtech): latest webpack 5 allows `entry: {}` so use that & remove following code
        if (_.isPlainObject(compiler.options.entry) && _.isEmpty(compiler.options.entry)) {
            compiler.hooks.entryOption.tap(ConsoleRemotePlugin.name, () => {
                return true;
            });
        }
        // Set default publicPath if output.publicPath option is missing or empty
        // TODO(vojtech): mainTemplate is deprecated in latest webpack 5, adapt code accordingly
        if (_.isEmpty(compiler.options.output.publicPath)) {
            compiler.hooks.thisCompilation.tap(ConsoleRemotePlugin.name, (compilation) => {
                compilation.mainTemplate.hooks.requireExtensions.tap(ConsoleRemotePlugin.name, () => {
                    const pluginBaseURL = `/api/plugins/${this.pkg.consolePlugin.name}/`;
                    return `${webpack.RuntimeGlobals.publicPath} = "${pluginBaseURL}";`;
                });
            });
        }
    }
}
exports.ConsoleRemotePlugin = ConsoleRemotePlugin;
//# sourceMappingURL=ConsoleRemotePlugin.js.map