# CSI-Addons Operation: ReclaimSpace

## Terminology

| Term     | Definition                                                                            |
| -------- | ------------------------------------------------------------------------------------- |
| VolumeID | The identifier of the volume generated by the plugin.                                 |
| CO       | Container Orchestration system that communicates with plugins using CSI service RPCs. |
| SP       | Storage Provider, the vendor of a CSI plugin implementation.                          |
| RPC      | [Remote Procedure Call](https://en.wikipedia.org/wiki/Remote_procedure_call).         |

## Objective

Define an extension to the CSI Specification that will enable storage vendors
(SP) to develop controllers/plugins that can free unused storage allocations
from existing volumes.

### Goals in MVP

The new extension will define a procedure that

* can be called for existing volumes
* functions for Filesystem and Block mode volumes
* interacts with the Controller and Node-Plugin
* makes it possible for the SP to free unused space from volumes

### Non-Goals in MVP

* Freeing unused storage allocations for inactive volumes

## Solution Overview

This specification defines an interface along with the minimum operational and
packaging recommendations for a storage provider (SP) to implement a space
reclaim operation for volumes. The interface declares the RPCs that a plugin
MUST expose.

## RPC Interface

* **Controller Service**: The Controller plugin MAY implement this RPC.
* **Node Service**: The Node plugin MAY implement this RPC.

```protobuf
syntax = "proto3";
package reclaimspace;

import "github.com/container-storage-interface/spec/lib/go/csi/csi.proto";
import "google/protobuf/descriptor.proto";

option go_package = "github.com/csi-addons/spec/lib/go/reclaimspace";

// ReclaimSpaceController holds the RPC method for running discard operations
// on the inactive (not required to be staged/published) volume.
service ReclaimSpaceController {
  // ControllerReclaimSpace is a procedure that gets called on the CSI
  // Controller.
  rpc ControllerReclaimSpace (ControllerReclaimSpaceRequest)
  returns (ControllerReclaimSpaceResponse) {}
}

// ReclaimSpaceNode holds the RPC method for running discard operations on the
// active (staged/published) volume.
service ReclaimSpaceNode {
  // NodeReclaimSpace is a procedure that gets called on the CSI NodePlugin.
  rpc NodeReclaimSpace (NodeReclaimSpaceRequest)
  returns (NodeReclaimSpaceResponse) {}
}
```

### ControllerReclaimSpace

```protobuf
// ControllerReclaimSpaceRequest contains the information needed to identify
// the volume by the SP and access any backend services so that space can be
// reclaimed.
message ControllerReclaimSpaceRequest {
  // The ID of the volume. This field is REQUIRED.
  string volume_id = 1;

  // Plugin specific parameters passed in as opaque key-value pairs.
  map<string, string> parameters = 2;
  // Secrets required by the plugin to complete the request.
  map<string, string> secrets = 3 [(csi.v1.csi_secret) = true];
}

// ControllerReclaimSpaceResponse holds the information about the result of the
// ControllerReclaimSpaceRequest call.
message ControllerReclaimSpaceResponse {
  // This field is OPTIONAL. This allows the SP to inform the CO about the
  // storage consumption before the ReclaimSpace operation was executed.
  StorageConsumption pre_usage = 1;

  // This field is OPTIONAL. This allows the SP to inform the CO about the
  // storage consumption after the ReclaimSpace operation was executed.
  StorageConsumption post_usage = 2;
}

// StorageConsumption contains the usage in bytes.
message StorageConsumption {
  // This field is REQUIRED. usage_bytes contains the consumed storage in
  // bytes.
  int64 usage_bytes = 1;
}
```

#### ControllerReclaimSpace Errors

| Condition                    | gRPC Code          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | Recovery Behavior                                                                                                                                                                                                                     |
| ---------------------------- | ------------------ | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| Missing required field       | 3 INVALID_ARGUMENT | Indicates that a required field is missing from the request.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | Caller MUST fix the request by adding the missing required field before retrying.                                                                                                                                                     |
| Volume does not exist        | 5 NOT_FOUND        | Indicates that a volume corresponding to the specified `volume_id` does not exist.                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | Caller MUST verify that the `volume_id` is correct and that the volume is accessible and has not been deleted before retrying with exponential back off.                                                                              |
| Operation pending for volume | 10 ABORTED         | Indicates that there is already an operation pending for the specified `volume_id`. In general the CSI-Addons CO plugin is responsible for ensuring that there is no more than one call "in-flight" per `volume_id` at a given time. However, in some circumstances, the CSI-Addons CO plugin MAY lose state (for example when the it crashes and restarts), and MAY issue multiple calls simultaneously for the same `volume_id`. The CSI-driver, SHOULD handle this as gracefully as possible, and MAY return this error code to reject secondary calls. | Caller SHOULD ensure that there are no other calls pending for the specified `volume_id`, and then retry with exponential back off.                                                                                                   |
| Not authenticated            | 16 UNAUTHENTICATED | The invoked RPC does not carry secrets that are valid for authentication.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | Caller SHALL either fix the secrets provided in the RPC, or otherwise regalvanize said secrets such that they will pass authentication by the Plugin for the attempted RPC, after which point the caller MAY retry the attempted RPC. |
| Error is Unknown             | 2 UNKNOWN          | Indicates that a unknown error is generated                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | Caller MUST study the logs before retrying                                                                                                                                                                                            |

### NodeReclaimSpace

```protobuf
// NodeReclaimSpaceRequest contains the information needed to identify the
// location where the volume is mounted so that local filesystem or
// block-device operations to reclaim space can be executed.
message NodeReclaimSpaceRequest {
  // The ID of the volume. This field is REQUIRED.
  string volume_id = 1;

  // The path on which volume is available. This field is REQUIRED.
  // This field overrides the general CSI size limit.
  // SP SHOULD support the maximum path length allowed by the operating
  // system/filesystem, but, at a minimum, SP MUST accept a max path
  // length of at least 128 bytes.
  string volume_path = 2;

  // The path where the volume is staged, if the plugin has the
  // STAGE_UNSTAGE_VOLUME capability, otherwise empty.
  // If not empty, it MUST be an absolute path in the root
  // filesystem of the process serving this request.
  // This field is OPTIONAL.
  // This field overrides the general CSI size limit.
  // SP SHOULD support the maximum path length allowed by the operating
  // system/filesystem, but, at a minimum, SP MUST accept a max path
  // length of at least 128 bytes.
  string staging_target_path = 3;

  // Volume capability describing how the CO intends to use this volume.
  // This allows SP to determine if volume is being used as a block
  // device or mounted file system. For example - if volume is being
  // used as a block device the SP MAY choose to skip calling filesystem
  // operations to reclaim space, but still perform rest of the housekeeping
  // needed for reducing the size of the volume. If volume_capability is
  // omitted the SP MAY determine access_type from given volume_path for the
  // volume and perform space reduction. This is an OPTIONAL field.
  csi.v1.VolumeCapability volume_capability = 4;

  // Secrets required by plugin to complete the reclaim space operation.
  // This field is OPTIONAL.
  map<string, string> secrets = 5 [(csi.v1.csi_secret) = true];
}

// NodeReclaimSpaceResponse holds the information about the result of the
// NodeReclaimSpaceRequest call.
message NodeReclaimSpaceResponse {
  // This field is OPTIONAL. This allows the SP to inform the CO about the
  // storage consumption before the ReclaimSpace operation was executed.
  StorageConsumption pre_usage = 1;

  // This field is OPTIONAL. This allows the SP to inform the CO about the
  // storage consumption after the ReclaimSpace operation was executed.
  StorageConsumption post_usage = 2;
}
```

#### NodeReclaimSpace Errors

| Condition                    | gRPC Code          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | Recovery Behavior                                                                                                                                                                                                                     |
| ---------------------------- | ------------------ | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| Missing required field       | 3 INVALID_ARGUMENT | Indicates that a required field is missing from the request.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | Caller MUST fix the request by adding the missing required field before retrying.                                                                                                                                                     |
| Volume does not exist        | 5 NOT_FOUND        | Indicates that a volume corresponding to the specified `volume_id` does not exist.                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | Caller MUST verify that the `volume_id` is correct and that the volume is accessible and has not been deleted before retrying with exponential back off.                                                                              |
| Operation pending for volume | 10 ABORTED         | Indicates that there is already an operation pending for the specified `volume_id`. In general the CSI-Addons CO plugin is responsible for ensuring that there is no more than one call "in-flight" per `volume_id` at a given time. However, in some circumstances, the CSI-Addons CO plugin MAY lose state (for example when the it crashes and restarts), and MAY issue multiple calls simultaneously for the same `volume_id`. The CSI-driver, SHOULD handle this as gracefully as possible, and MAY return this error code to reject secondary calls. | Caller SHOULD ensure that there are no other calls pending for the specified `volume_id`, and then retry with exponential back off.                                                                                                   |
| Not authenticated            | 16 UNAUTHENTICATED | The invoked RPC does not carry secrets that are valid for authentication.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | Caller SHALL either fix the secrets provided in the RPC, or otherwise regalvanize said secrets such that they will pass authentication by the Plugin for the attempted RPC, after which point the caller MAY retry the attempted RPC. |
| Error is Unknown             | 2 UNKNOWN          | Indicates that a unknown error is generated                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | Caller MUST study the logs before retrying                                                                                                                                                                                            |
