// Copyright (C) 2018-2022 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

#include <gtest/gtest.h>

#include "base_reference_test.hpp"
#include "openvino/op/constant.hpp"
#include "openvino/op/idft.hpp"

using namespace reference_tests;
using namespace ov;

namespace {

struct IDFTParams {
    template <class T>
    IDFTParams(const Shape& input_shape,
              const Shape& expected_shape,
              const element::Type_t& input_type,
              const element::Type_t& expected_type,
              const std::vector<T>& input_value,
              const std::vector<T>& expected_value,
              const std::shared_ptr<op::v0::Constant>& axes,
              const std::shared_ptr<op::v0::Constant>& signal) {
        m_input_shape = input_shape;
        m_expected_shape = expected_shape;
        m_input_type = input_type;
        m_expected_type = expected_type;
        m_input_value = CreateTensor(input_type, input_value);
        m_expected_value = CreateTensor(expected_type, expected_value);
        m_axes = axes;
        m_signal = signal;
    }

    Shape m_input_shape;
    Shape m_expected_shape;
    element::Type_t m_input_type;
    element::Type_t m_expected_type;
    ov::Tensor m_input_value;
    ov::Tensor m_expected_value;
    std::shared_ptr<op::v0::Constant> m_axes;
    std::shared_ptr<op::v0::Constant> m_signal;
};

class ReferenceIDFTLayerTest : public testing::TestWithParam<IDFTParams>, public CommonReferenceTest {
public:
    void SetUp() override {
        auto params = GetParam();
        if (params.m_signal != NULL) {
            function = CreateFunctionWithSignal(params);
        } else {
            function = CreateFunction(params);
        }

        inputData = {params.m_input_value};
        refOutData = {params.m_expected_value};
    }

    static std::string getTestCaseName(const testing::TestParamInfo<IDFTParams>& obj) {
        const auto param = obj.param;
        std::ostringstream result;

        result << "input_shape1=" << param.m_input_shape << "; ";
        result << "output_shape=" << param.m_expected_shape << "; ";
        result << "input_type1=" << param.m_input_type << "; ";
        result << "output_type=" << param.m_expected_type << "; ";
        result << "transpose1=" << param.m_axes;

        return result.str();
    }

private:
    static std::shared_ptr<Model> CreateFunction(IDFTParams& p) {
        auto in = std::make_shared<op::v0::Parameter>(p.m_input_type, p.m_input_shape);
        auto idft = std::make_shared<op::v7::IDFT>(in, p.m_axes);

        return std::make_shared<ov::Model>(idft, ParameterVector{in});
    }

    static std::shared_ptr<Model> CreateFunctionWithSignal(IDFTParams& p) {
        auto in = std::make_shared<op::v0::Parameter>(p.m_input_type, p.m_input_shape);
        auto idft = std::make_shared<op::v7::IDFT>(in, p.m_axes, p.m_signal);

        return std::make_shared<ov::Model>(idft, ParameterVector{in});
    }
};

TEST_P(ReferenceIDFTLayerTest, CompareWithHardcodedRefs) {
    Exec();
}

static const std::vector<float> expected_result = {
    0.85943836,   0.009941814, 0.004292889, 0.54598427,   0.8270831,   0.49770153,  0.9035636,   0.19274887,
    0.8589833,    0.88759327,  0.72343576,  0.057539318,  0.915801,    0.63455844,  0.25069925,  0.045601673,
    0.29793364,   0.8492151,   0.6885839,   0.57419384,   0.009737609, 0.68192583,  0.7614807,   0.37603703,
    0.51804876,   0.033039097, 0.63702065,  0.78960556,   0.5007368,   0.7248742,   0.2040932,   0.1211606,
    0.76035476,   0.44004318,  0.95635134,  0.82913375,   0.225465,    0.009166263, 0.05445403,  0.5885675,
    0.87822133,   0.14324947,  0.68606305,  0.3274419,    0.9169595,   0.732179,    0.04614906,  0.03505424,
    0.84526163,   0.9972937,   0.89781004,  0.9987864,    0.24641308,  0.34678686,  0.22731997,  0.95805293,
    0.595993,     0.8537836,   0.9174756,   0.17441267,   0.86681056,  0.15913424,  0.6638066,   0.522398,
    0.51548326,   0.024979044, 0.1731268,   0.068090245,  0.6125645,   0.4865482,   0.2873719,   0.35936728,
    0.64452374,   0.27963468,  0.59981745,  0.6309508,    0.507604,    0.23389837,  0.77500635,  0.4462004,
    0.53165394,   0.6535075,   0.4306448,   0.21468966,   0.6925882,   0.11183031,  0.25347117,  0.2209481,
    0.8060583,    0.34712377,  0.78980505,  0.16110454,   0.6376819,   0.78736854,  0.909368,    0.6915289,
    0.24747796,   0.32442623,  0.22714981,  0.23976989,   0.25199527,  0.28412706,  0.32461873,  0.51917267,
    0.8394496,    0.6324911,   0.28498915,  0.8887276,    0.90213394,  0.16050571,  0.32190812,  0.67677563,
    0.8594967,    0.28917953,  0.1931407,   0.8282108,    0.14881423,  0.18073067,  0.8490643,   0.2356146,
    0.86200285,   0.57409924,  0.94718546,  0.092213534,  0.34502912,  0.4719212,   0.60031396,  0.22602181,
    0.3067876,    0.49529344,  0.11133887,  0.47633907,   0.13236542,  0.69677263,  0.8490109,   0.6685073,
    0.24199674,   0.7983137,   0.37593383,  0.74520975,   0.16743147,  0.84144354,  0.93073046,  0.55940866,
    0.67484015,   0.077098235, 0.69045097,  0.06949082,   0.6804774,   0.79804176,  0.49027568,  0.8843709,
    0.5665486,    0.91798306,  0.47884017,  0.94707423,   0.98279756,  0.62054926,  0.8134105,   0.01336217,
    0.78324115,   0.9938295,   0.99227554,  0.66681916,   0.38842493,  0.3835454,   0.120395586, 0.5478275,
    0.13309076,   0.9468553,   0.24595714,  0.0057277656, 0.14570542,  0.31220108,  0.41687667,  0.679465,
    0.5731583,    0.7383743,   0.013198466, 0.34619793,   0.9278514,   0.48510832,  0.46039802,  0.8171329,
    0.5041023,    0.37600085,  0.124404594, 0.4201713,    0.7470036,   0.7340853,   0.8449047,   0.137517,
    0.14771219,   0.99655616,  0.2178388,   0.4121613,    0.8655656,   0.32849622,  0.7574791,   0.95230037,
    0.5806251,    0.9598742,   0.7183528,   0.042957753,  0.2926446,   0.5882527,   0.05208914,  0.3216481,
    0.5205192,    0.5095992,   0.011508227, 0.5209922,    0.78207654,  0.34570032,  0.7968098,   0.4619513,
    0.0047925604, 0.029039407, 0.7673424,   0.571703,     0.44400942,  0.82529145,  0.29335254,  0.34418115,
    0.48119327,   0.38321403,  0.31083322,  0.7179562,    0.41055596,  0.06207573,  0.8747831,   0.6018095,
    0.4483476,    0.16189687,  0.8611539,   0.79723805,   0.42178747,  0.95597315,  0.5534858,   0.33466807,
    0.36827618,   0.60728735,  0.6582703,   0.6790265,    0.870856,    0.8868432,   0.43860948,  0.32468447,
    0.77624434,   0.3403061,   0.14144918,  0.23022941,   0.07176102,  0.06941459,  0.37346482,  0.9120822,
    0.65890974,   0.77746564,  0.4515671,   0.45455948,   0.15909587,  0.8017096,   0.6259673,   0.6117355,
    0.77020043,   0.08495594,  0.30376136,  0.55266386,   0.8497134,   0.91790336,  0.86088765,  0.88179666,
    0.9009849,    0.97200614,  0.94119,     0.77911216,   0.8057816,   0.14040896,  0.66522235,  0.6649202,
    0.048396785,  0.75035393,  0.4520953,   0.9877601,    0.46115568,  0.2167145,   0.9271302,   0.39395386,
    0.68578094,   0.576275,    0.20754486,  0.5408786,    0.46040633,  0.18199016,  0.66303253,  0.6288556,
    0.14313427,   0.91675115,  0.36198065,  0.51337945,   0.84241706,  0.22333568,  0.38011634,  0.024615016,
    0.19370414,   0.23593484,  0.32207185,  0.47971123,   0.6202779,   0.6944977,   0.43612957,  0.07961436,
    0.57468814,   0.100025274, 0.42476946,  0.95338464,   0.666547,    0.8683905,   0.52689695,  0.6284723,
    0.85813546,   0.4865953,   0.8269112,   0.08833949,   0.69269264,  0.41784903,  0.5969149,   0.07599888,
    0.14184453,   0.49042618,  0.44027725,  0.6256328,    0.2716237,   0.0999099,   0.09831784,  0.92469853,
    0.24196884,   0.9073526,   0.7523511,   0.7761173,    0.28489882,  0.96349007,  0.5884645,   0.74933976,
    0.06400105,   0.4376275,   0.34752035,  0.6006149,    0.034923803, 0.066874385, 0.9790322,   0.5558188,
    0.97579825,   0.025802653, 0.537738,    0.24921915,   0.111012295, 0.85987717,  0.781183,    0.69588315,
    0.94621634,   0.74946797,  0.6949375,   0.009165181,  0.91075164,  0.72913235,  0.25934777,  0.19463088,
    0.5283304,    0.9241759,   0.0563183,   0.74323857,   0.43722472,  0.2958358,   0.85980684,  0.029655656,
    0.362904,     0.19682994,  0.37778872,  0.09406928,   0.23010127,  0.44393733,  0.420214,    0.39723217,
    0.13777487,   0.06385251,  0.9535715,   0.89861375,   0.2463547,   0.673834,    0.8008994,   0.0861585,
    0.6613363,    0.79498637,  0.79322547,  0.083214305,  0.577025,    0.58655965,  0.119723536, 0.0012204717};

static const std::vector<float> idft1d_input_data = {
    6.329814,     4.2950764,   -0.8105316,   -0.7187835,   -0.059136264, 0.2709784,     0.82793635,  0.57937646,
    0.5997731,    -1.3291739,  1.188664,     1.462941,     -0.01811248,  -1.8314927,    0.16004556,  -2.219835,
    1.0620322,    -1.0679832,  -0.68610185,  0.658314,     4.627743,     4.5935497,     -0.78950775, -0.32600924,
    -1.4706655,   -1.1615934,  0.708719,     1.4568751,    -1.0970218,   -0.39268675,   -0.5990571,  -0.81545514,
    -0.39174145,  -0.420258,   0.55279106,   2.339339,     -0.59915966,  1.3964193,     -0.8447231,  0.14907542,
    6.2576666,    5.5670385,   0.25636938,   -1.7026355,   1.161571,     0.12042561,    0.19768336,  -1.3421875,
    -0.90698814,  1.4111948,   0.70803046,   0.5795436,    1.2021728,    -0.5199567,    -2.558736,   -0.80762154,
    1.1657354,    -0.8685272,  1.2987087,    -1.0047817,   5.6461143,    3.2111988,     0.2361581,   0.3099669,
    0.6179653,    0.099535145, 1.0438079,    -0.016701937, -0.88529384,  -0.12907594,   0.64785606,  -0.8428119,
    -0.058392793, -1.0488291,  -0.4019828,   0.20333555,   0.45051938,   0.45967662,    1.3713523,   -0.6549525,
    5.5258985,    3.7522945,   -1.8860855,   -0.2230255,   0.8160669,    -0.46607828,   0.123957604, 0.61024696,
    0.26978388,   0.9723815,   0.3050212,    0.69621503,   0.27244493,   -1.0805726,    0.20593566,  1.5653824,
    -0.27690098,  0.8950307,   -0.039584313, -0.18680441,  4.975611,     4.6955333,     0.19031112,  -0.8860659,
    0.91665065,   -0.5264673,  -0.4547393,   1.1623507,    -1.4774656,   1.671129,      1.028168,    -1.6014669,
    -1.2178835,   -0.13447604, -0.14712845,  -0.6739672,   -0.3273949,   -0.9012072,    -0.9661755,  0.03590688,
    4.771964,     5.244689,    -0.03415192,  -0.37281254,  -0.49070793,  -0.65789306,   0.8143984,   -0.8913989,
    -0.19890547,  0.17876014,  -0.9956009,   0.82810897,   0.55270624,   -0.023768127,  1.5358362,   0.6981953,
    0.23165298,   0.51040155,  2.4328363,    0.2267083,    6.4758024,    5.72882,       -0.8707881,  -1.110683,
    0.12478554,   1.3484334,   0.3689712,    0.29180524,   -0.8149491,   -0.0922713,    -0.33161288, 0.78140867,
    -0.9623072,   0.8999919,   -2.1120539,   0.84492886,   -1.5347936,   0.7440938,     1.3312622,   -1.0220959,
    3.8123238,    5.62084,     1.3551373,    0.6460793,    -0.21639234,  -1.2077228,    1.1639122,   -0.05263084,
    0.48105645,   -0.5892652,  0.2349168,    1.128768,     0.42568994,   0.36398163,    -1.2250046,  2.3513904,
    0.64331245,   0.8099514,   1.1574583,    0.8668997,    5.59726,      5.659527,      0.48095328,  0.59446967,
    1.1849049,    1.4709316,   -1.2589264,   -0.11577609,  0.6299068,    -1.4621243,    0.7872094,   0.18096408,
    0.5553762,    -2.0060503,  -0.4373122,   0.9938256,    0.89633095,   -0.5491595,    0.8428093,   0.084472984,
    4.52676,      4.351716,    0.73079205,   0.8098516,    0.27877963,   -0.0073297992, 0.36545974,  0.6745955,
    -2.3818088,   1.5816333,   -0.16544427,  0.51321346,   -0.23699868,  -0.13254744,   1.551896,    0.62098134,
    0.7739359,    1.6108581,   0.36288044,   -0.42423314,  5.0995026,    5.1843014,     -1.1968713,  1.1790991,
    -0.018864498, -0.7500831,  0.0879575,    0.22010106,   1.1136081,    2.2893274,     -0.6877146,  -0.40740123,
    0.046427906,  0.8681825,   -0.50678635,  0.23051873,   0.35328788,   -0.45622703,   0.1495475,   -0.104907334,
    4.8094087,    5.2818966,   0.49697292,   0.29568392,   -0.4144543,   -0.64546454,   0.31737912,  -0.8962374,
    -1.0404948,   0.91764164,  0.6826862,    0.08073502,   0.33942595,   0.053232975,   -1.1867946,  0.51120156,
    -1.1452568,   -1.4197243,  0.82389224,   1.8939058,    6.882805,     6.4072084,     -1.3024135,  -0.22483894,
    -0.22082287,  1.0370905,   -0.7639439,   0.6950346,    -0.731326,    0.16821115,    0.0887468,   -0.5732441,
    -0.40715322,  -0.96244293, -0.89126545,  1.3140129,    -0.42358512,  1.7674587,     -0.6400819,  -1.6113993,
    4.4106574,    5.706909,    -1.1110737,   0.10560027,   -1.1108764,   0.34190884,    2.1167603,   -0.067495525,
    -0.16237324,  0.2604496,   -0.8129095,   -0.42274237,  -1.1412699,   -0.0011268258, -0.63462454, -0.15172139,
    -0.7164279,   0.14801888,  -0.3538928,   1.583736,     4.9876184,    4.2879796,     -0.8491325,  0.5345522,
    -0.60507995,  -0.9020085,  1.0447598,    0.21135187,   -0.4787205,   -0.3230412,    0.8076494,   -0.04361339,
    0.62797767,   0.15487206,  -0.23772183,  0.69546384,   1.8609382,    -1.7030516,    1.2658813,   -0.6791475,
    4.921037,     4.8929176,   -0.0124401,   -0.6873918,   -0.21879943,  -0.48610657,   0.36776963,  0.12423802,
    -0.7854952,   0.48838156,  -0.5085067,   -0.08865434,  1.1653454,    0.81965554,    -0.6399579,  -1.0967884,
    1.4099771,    -0.15370974, 2.8824244,    1.0534087,    4.7045717,    5.2045445,     -0.6350576,  2.5321684,
    0.6987691,    -0.53839976, -0.09889791,  0.5662097,    0.4088725,    0.635128,      -1.763303,   -0.49720347,
    -1.0772469,   1.2422445,   -0.3619956,   -1.311133,    1.5846866,    1.0530244,     -0.61141044, 0.74831486,
    5.433625,     3.9661994,   2.006918,     -0.8703619,   -0.7658511,   0.0811044,     0.83877516,  -0.63553256,
    -0.67563355,  1.7368636,   0.9372277,    1.8246815,    0.8615329,    -0.18161502,   0.62479717,  0.2028623,
    0.159001,     1.860977,    0.04177074,   -0.49050322,  4.9402246,    4.0296063,     -0.74729615, -0.27802998,
    -0.8077982,   -0.5414143,  0.467114,     0.9016136,    2.1971147,    -1.466963,     -1.2350414,  1.0967304,
    -0.95607626,  0.51462483,  0.28838068,   1.0117096,    -0.21846394,  0.114624545,   -1.627146,   -0.9431294};

static const std::vector<float> idft2d_input_data = {
    54.020195,    48.368538,   -1.8721353,   -3.7894967,   2.5850394,   -0.7094516,   3.5357249,    1.6819549,
    -3.4001002,   0.23887074,  2.9735894,    2.3982158,    0.3599546,   -5.801426,    -4.427606,    5.2949734,
    1.7113355,    1.428697,    5.8978443,    -0.8472582,   -3.288164,   -0.099487126, -0.33851182,  2.614974,
    -2.766882,    0.18681616,  0.34976268,   -0.2601711,   4.998401,    -2.9831958,   -1.6652081,   0.53361464,
    -0.9001389,   -3.6454318,  -3.7148805,   -0.68562484,  2.0633714,   -2.2154818,   -3.3697965,   3.5273929,
    1.5474558,    -1.6305131,  -5.3327236,   0.54002213,   -1.6671672,  2.4493377,    -2.2604918,   1.4117424,
    2.1797671,    2.5013056,   0.8525213,    1.6570821,    1.717532,    -2.101283,    4.6570606,    -3.6786642,
    0.8912736,    -0.4010569,  -5.9480867,   1.441097,     2.1150498,   -1.4524796,   -3.5035098,   3.0815587,
    -3.3185432,   4.7882123,   5.64722,      -1.1192517,   1.8302126,   -2.5760055,   -0.41363025,  3.2350469,
    1.4296081,    0.8722873,   6.1752787,    -1.7328868,   2.312786,    4.4069357,    1.7721124,    3.3802934,
    -0.53283703,  3.7646027,   4.440572,     -4.353462,    -2.7639425,  3.6855025,    1.8912748,    -2.5849285,
    -2.9895856,   1.1341677,   1.4818796,    0.7482485,    -1.3077981,  1.0669674,    -0.76039124,  -10.8791685,
    2.998129,     -4.2489543,  0.41752052,   -0.45298803,  -0.62486386, 0.5913104,    -0.36638862,  -0.9528576,
    -0.16223967,  -3.171127,   2.7200532,    -3.8751457,   3.8895426,   1.0489256,    -0.091531515, 6.992935,
    4.5098467,    -0.38218838, 0.6637606,    -2.1199496,   3.9403267,   -0.870952,    2.4287906,    1.9679271,
    3.652341,     -4.4909067,  -1.4710087,   0.5256169,    5.4580984,   -2.6554706,   -0.058261395, 3.6613276,
    0.5612789,    1.0594783,   4.5429516,    -1.447232,    -2.388829,   0.52541757,   -6.1111097,   -2.3621864,
    -1.4885365,   -2.6265867,  -4.4030347,   0.27728367,   3.9584684,   -3.7618577,   -3.128574,    -2.8671994,
    1.4171265,    0.02298975,  -2.0790722,   1.6526843,    0.59488124,  -3.2548752,   -0.82249254,  1.3645289,
    -2.9066925,   -3.4377484,  -2.501403,    -2.821631,    -4.427053,   -2.3529994,   0.6670886,    -4.7455816,
    -2.160026,    -1.0587022,  1.1341916,    -0.9469211,   0.67554307,  -4.0473633,   -1.2422556,   4.538533,
    -0.739814,    -3.22405,    1.2332113,    -4.0489397,   -4.560828,   -3.5195189,   6.7066355,    -2.8439593,
    -0.43901098,  -3.9980454,  -4.2256207,   3.0529652,    4.6105156,   2.720234,     2.3327744,    -1.0400636,
    -0.048398018, 2.1603358,   -0.22459112,  0.6870126,    -0.926849,   -7.2363615,   3.7953386,    3.195907,
    3.8662248,    -1.8919971,  0.91311014,   -0.36923724,  3.0576966,   0.19861764,   -0.09782998,  -1.0179963,
    50.71621,     49.313248,   -2.6195984,   3.396334,     -3.1849973,  -2.4107025,   4.7431326,    1.7938776,
    -2.5362587,   6.287631,    -2.656609,    1.4825039,    -0.77803206, 2.3750808,    -1.9940716,   2.0271082,
    3.6380908,    2.822246,    2.2938647,    1.0260472,    3.248794,    -3.05949,     2.0979533,    3.565119,
    1.9497933,    0.2390036,   -2.255065,    0.7849397,    1.9622431,   4.2382064,    -3.2529292,   0.78572094,
    -2.9386084,   0.66875017,  5.743927,     4.850876,     -4.8014383,  6.371132,     -2.6618924,   -1.8847032,
    -1.7702236,   -1.1031301,  1.4129921,    -0.080709964, -2.7634878,  -3.6456683,   1.4174454,    3.4085226,
    3.10102,      0.114031196, -2.4092412,   0.27725983,   2.8974152,   -1.866328,    -0.68216217,  2.249536,
    -0.42875588,  -5.8182187,  5.347006,     -6.2936745,   0.8000201,   3.651592,     1.3155181,    2.3413098,
    2.1600244,    1.8733575,   -2.4694557,   0.39358342,   2.020084,    -0.062472403, -4.131041,    -1.5137839,
    -2.0354557,   1.1957052,   -0.6644075,   -2.0442688,   2.0753646,   4.874056,     -0.090800405, 1.3911223,
    0.68129027,   -4.0028048,  -0.8021738,   0.43866205,   2.7812133,   0.4525791,    -0.87565154,  1.2364697,
    -2.725146,    2.7965212,   4.148448,     -1.9204504,   -0.61004305, -4.790703,    3.1498234,    0.79403657,
    5.305445,     0.2803253,   -3.67164,     -4.3974924,   -2.5132315,  -0.9139994,   6.841936,     -4.089568,
    -1.2774054,   0.9789283,   3.269153,     -3.3947415,   -7.5553513,  3.682307,     2.9227152,    2.3319635,
    2.754105,     -1.2598821,  1.4247041,    -1.8540356,   -2.675635,   1.2705915,    5.2202816,    6.206577,
    0.4957786,    2.1150033,   5.8791704,    2.8043785,    -0.37886655, 0.011162788,  -1.0408137,   -1.5385519,
    -8.079001,    -0.68451786, 2.3513699,    3.0877895,    2.6497078,   1.3670976,    0.77233493,   2.2921152,
    -1.2679763,   2.113087,    4.990262,     -0.046566606, 0.015865922, 1.1569002,    -4.8347507,   1.9560149,
    1.979923,     2.34512,     -0.9634773,   4.3939066,    -6.2031984,  0.8311275,    -2.7740612,   -2.9296994,
    -3.4624243,   -1.4588313,  2.4724,       -0.79477566,  -0.4295609,  5.8110385,    -2.6649034,   -2.270977,
    -2.5511568,   -3.1238616,  -4.46209,     0.16335368,   1.9146351,   1.0459399,    2.8069792,    -0.4705832,
    -4.0632596,   -2.220704,   1.7770543,    -0.5791014,   -2.2041528,  3.026476,     5.324942,     -0.7805673,
    5.9275556,    0.14159381,  -0.81569004,  4.1947803,    -3.8557377,  -0.5163199,   2.478963,     -2.396379,
    -0.3930376,   -0.96302,    -0.9776549,   0.13852966,   0.26078847,  0.8342015,    2.3698487,    4.109933,
    1.3575013,    -0.5828376,  -0.028537825, -0.53020877,  0.39626116,  -1.7572733,   -4.31769,     -2.1674476};

static const std::vector<float> idft3d_input_data = {
    104.7364,    97.68179,     -4.491728,    -0.39316452, -0.59995466, -3.1201572,  8.278858,    3.4758341,
    -5.9363585,  6.5265055,    0.3169801,    3.8807175,   -0.418082,   -3.4263492,  -6.4216776,  7.3220854,
    5.3494234,   4.2509427,    8.191702,     0.17879319,  -0.03937006, -3.1589758,  1.7594413,   6.180092,
    -0.8170867,  0.42582142,   -1.9053001,   0.52476853,  6.9606423,   1.255014,    -4.9181366,  1.319335,
    -3.838747,   -2.9766817,   2.0290484,    4.16525,     -2.7380676,  4.155652,    -6.0316873,  1.6426877,
    -0.2227689,  -2.7336447,   -3.919732,    0.45931256,  -4.4306555,  -1.1963288,  -0.8430467,  4.8202653,
    5.280785,    2.6153364,    -1.556721,    1.9343407,   4.614946,    -3.96761,    3.9748988,   -1.4291265,
    0.46251905,  -6.2192726,   -0.60107887,  -4.852579,   2.9150705,   2.1991146,   -2.1879911,  5.4228687,
    -1.158518,   6.661569,     3.1777658,    -0.7256692,  3.8502965,   -2.6384768,  -4.544671,   1.721262,
    -0.6058461,  2.067991,     5.5108714,    -3.7771575,  4.388153,    9.280992,    1.681312,    4.7714148,
    0.14845347,  -0.23820269,  3.6383984,    -3.9147997,  0.017270446, 4.138083,    1.0156215,   -1.3484575,
    -5.7147317,  3.9306912,    5.630328,     -1.1722009,  -1.9178381,  -3.7237349,  2.3894331,   -10.085134,
    8.303572,    -3.9686286,   -3.2541199,   -4.850478,   -3.1380959,  -0.32268947, 6.475547,    -5.0424256,
    -1.4396465,  -2.1921992,   5.9892044,    -7.269888,   -3.665809,   4.7312326,   2.8311844,   9.324896,
    7.2639513,   -1.6420703,   2.0884657,    -3.9739842,  1.2646922,   0.39964193,  7.649071,    8.174507,
    4.148118,    -2.3759027,   4.4081597,    3.3299959,   5.0792284,   -2.6443086,  -1.0990746,  2.1227744,
    -7.517721,   0.3749615,    6.894322,     1.6405574,   0.26087707,  1.8925169,   -5.3387756,  -0.07007182,
    -2.7565134,  -0.51350284,  0.5872268,    0.23071745,  3.9743357,   -2.6049578,  -7.963324,   -0.9111862,
    3.3970497,   2.368112,     -3.0425484,   6.0465913,   -5.608317,   -2.4237492,  -3.5965526,  -1.5651696,
    -6.369116,   -4.896579,    -0.029001951, -3.616405,   -4.8566127,  3.4580388,   -1.9978137,  -7.016559,
    -4.71118,    -4.1825647,   -3.3278992,   -0.7835678,  2.5901778,   -3.0014238,  1.5647203,   4.06795,
    -4.803074,   -5.444754,    3.0102665,    -4.6280394,  -6.764982,   -0.49304247, 12.031577,   -3.6245267,
    5.488541,    -3.8564541,   -5.04131,     7.2477474,   0.7547778,   2.2039144,   4.8117356,   -3.4364424,
    -0.44143593, 1.1973162,    -1.2022457,   0.8255428,   -0.66605973, -6.4021583,  6.1651874,   7.3058405,
    5.2237253,   -2.4748354,   0.88457155,   -0.89944726, 3.453958,    -1.558656,   -4.4155188,  -3.1854444,
    3.303988,    -0.9447114,   0.7474582,    -7.185831,   5.770039,    1.7012511,   -1.2074116,  -0.11192033,
    -0.86384296, -6.048759,    5.6302013,    0.9157127,   1.1379871,   -8.176507,   -2.433535,   3.2678652,
    -1.9267552,  -1.393548,    3.6039736,    -1.873306,   -6.536957,   2.9600024,   -2.4364662,  -0.95014465,
    -4.716674,   -0.052186966, 2.6048284,    -1.0451086,  3.036159,    -7.221403,   1.5877211,   -0.25210607,
    2.0384693,   -4.3141813,   -9.458808,    -5.5365014,  6.8648105,   -8.586614,   -0.7079052,  5.412094,
    3.3176801,   -0.5273831,   -6.745717,    0.62073076,  1.0963198,   6.0950055,   -3.677938,   -1.9967818,
    -0.921252,   2.387275,     3.261763,     1.3798212,   -1.1798835,  -0.23495495, 5.339221,    -5.928199,
    1.3200281,   5.417163,     -11.295093,   7.7347717,   1.3150296,   -5.1040716,  -4.8190293,  0.74024755,
    -5.4785676,  2.914854,     8.116676,     -1.5128357,  -0.1898706,  -2.5135324,  3.7174103,   4.7488313,
    3.4650638,   -0.32341766,  6.8396864,    0.31138325,  0.2374219,   -0.46712062, 1.8629129,   1.9891711,
    -1.2141278,  7.7674093,    5.2427464,    -4.792124,   -5.5451555,  3.2329237,   2.766926,    -3.8213987,
    -0.26443875, -1.6623533,   -2.6665692,   2.6686997,   -0.6977545,  5.85767,     -3.9102163,  -11.673204,
    -2.3073153,  -4.529278,    4.0891604,    3.9445055,   1.8883687,   1.50531,     -7.2083244,  3.1367111,
    1.1151649,   -4.1500554,   -0.54910004,  -0.48040384, 11.444895,   -2.6333811,  -3.0142484,  4.6609726,
    1.755743,    0.87769306,   -0.7609439,   -0.26591438, 6.615961,    -2.141545,   -2.7914915,  -4.2386503,
    3.1565619,   -6.6059103,   -7.35018,     -2.2787585,  5.836963,    -2.6666338,  0.98255026,  5.199881,
    8.640279,    1.7439961,    2.191582,     -4.535021,   -5.038538,   -0.841679,   -6.8834453,  -4.654301,
    -0.220559,   -4.7396717,   -9.393296,    0.32385087,  3.9426038,   -4.9187584,  1.7061774,   -4.8232145,
    -0.5627973,  -2.3221302,   -1.1155958,   -2.7412212,  6.798079,    -4.0860014,  1.9515686,   4.2942266,
    0.5557329,   -1.9789174,   -4.973804,    -2.0268555,  -3.9974911,  -8.164038,   3.3319929,   -2.474605,
    0.39113098,  2.0651584,    5.5962815,    -1.1102749,  -1.2390921,  -5.0933027,  -4.0492353,  5.009116,
    3.323446,    -1.0033474,   -0.54384375,  -3.4698372,  -2.3566747,  -6.545992,   1.3816929,   -2.0633929,
    -6.3665648,  -4.13964,     -3.4099324,   -1.1418146,  8.466255,    3.2365537,   -0.14618888, 1.3563147,
    0.3446387,   3.1233552,    0.7530624,    0.548483,    -1.1876376,  -8.070564,   1.4254899,   -0.9140264,
    2.5087235,   -1.3091599,   0.9416502,    0.16097029,  2.6614356,   1.9558911,   4.219861,    1.1494511};

static const std::vector<float> expected_bfloat_idft1d_result = {
    0.859375,   0.00897217, 0.003479,  0.542969,    0.824219,  0.496094,  0.898438,  0.191406,  0.855469,
    0.882812,   0.71875,    0.057373,  0.914062,    0.632812,  0.25,      0.0437012, 0.296875,  0.84375,
    0.6875,     0.570312,   0.010498,  0.679688,    0.757812,  0.371094,  0.515625,  0.0302734, 0.632812,
    0.785156,   0.5,        0.71875,   0.204102,    0.117188,  0.757812,  0.4375,    0.953125,  0.824219,
    0.226562,   0.00640869, 0.0541992, 0.585938,    0.875,     0.142578,  0.6875,    0.328125,  0.914062,
    0.730469,   0.0463867,  0.034668,  0.84375,     0.996094,  0.898438,  0.996094,  0.244141,  0.34375,
    0.226562,   0.957031,   0.59375,   0.851562,    0.914062,  0.173828,  0.867188,  0.160156,  0.660156,
    0.523438,   0.511719,   0.0249023, 0.171875,    0.0683594, 0.609375,  0.486328,  0.285156,  0.359375,
    0.640625,   0.28125,    0.597656,  0.632812,    0.507812,  0.234375,  0.773438,  0.445312,  0.527344,
    0.652344,   0.427734,   0.214844,  0.6875,      0.111328,  0.25,      0.220703,  0.800781,  0.347656,
    0.785156,   0.161133,   0.632812,  0.785156,    0.90625,   0.6875,    0.246094,  0.324219,  0.224609,
    0.240234,   0.25,       0.285156,  0.324219,    0.515625,  0.835938,  0.632812,  0.283203,  0.882812,
    0.902344,   0.160156,   0.320312,  0.675781,    0.859375,  0.289062,  0.193359,  0.824219,  0.148438,
    0.181641,   0.84375,    0.234375,  0.859375,    0.574219,  0.945312,  0.0927734, 0.34375,   0.472656,
    0.59375,    0.225586,   0.304688,  0.496094,    0.108398,  0.476562,  0.129883,  0.695312,  0.84375,
    0.667969,   0.240234,   0.796875,  0.375,       0.742188,  0.167969,  0.839844,  0.929688,  0.558594,
    0.671875,   0.0771484,  0.6875,    0.0693359,   0.679688,  0.796875,  0.490234,  0.882812,  0.5625,
    0.914062,   0.476562,   0.945312,  0.980469,    0.617188,  0.8125,    0.0131836, 0.78125,   0.992188,
    0.992188,   0.664062,   0.388672,  0.384766,    0.121094,  0.546875,  0.132812,  0.945312,  0.246094,
    0.00799561, 0.145508,   0.3125,    0.416016,    0.679688,  0.570312,  0.734375,  0.0142822, 0.347656,
    0.929688,   0.484375,   0.460938,  0.816406,    0.5,       0.375,     0.124023,  0.417969,  0.746094,
    0.734375,   0.84375,    0.138672,  0.148438,    0.992188,  0.216797,  0.412109,  0.867188,  0.328125,
    0.757812,   0.949219,   0.578125,  0.957031,    0.714844,  0.043457,  0.289062,  0.585938,  0.0498047,
    0.320312,   0.515625,   0.507812,  0.0098877,   0.519531,  0.777344,  0.34375,   0.792969,  0.460938,
    0.00314331, 0.029541,   0.765625,  0.570312,    0.441406,  0.820312,  0.292969,  0.34375,   0.480469,
    0.382812,   0.3125,     0.71875,   0.410156,    0.0639648, 0.875,     0.601562,  0.447266,  0.163086,
    0.859375,   0.796875,   0.421875,  0.953125,    0.550781,  0.333984,  0.367188,  0.605469,  0.65625,
    0.679688,   0.867188,   0.882812,  0.4375,      0.324219,  0.773438,  0.339844,  0.142578,  0.230469,
    0.0722656,  0.0693359,  0.373047,  0.910156,    0.65625,   0.777344,  0.453125,  0.453125,  0.158203,
    0.800781,   0.625,      0.609375,  0.769531,    0.0864258, 0.302734,  0.554688,  0.847656,  0.914062,
    0.859375,   0.878906,   0.898438,  0.96875,     0.9375,    0.777344,  0.804688,  0.140625,  0.664062,
    0.664062,   0.0478516,  0.746094,  0.451172,    0.984375,  0.460938,  0.214844,  0.921875,  0.392578,
    0.683594,   0.570312,   0.207031,  0.539062,    0.460938,  0.181641,  0.664062,  0.625,     0.143555,
    0.914062,   0.361328,   0.511719,  0.84375,     0.222656,  0.380859,  0.0246582, 0.194336,  0.234375,
    0.324219,   0.478516,   0.621094,  0.695312,    0.4375,    0.0795898, 0.574219,  0.100098,  0.425781,
    0.953125,   0.664062,   0.867188,  0.527344,    0.625,     0.851562,  0.484375,  0.824219,  0.0869141,
    0.6875,     0.416016,   0.59375,   0.0751953,   0.140625,  0.488281,  0.4375,    0.625,     0.269531,
    0.0986328,  0.0976562,  0.921875,  0.241211,    0.902344,  0.75,      0.773438,  0.28125,   0.960938,
    0.585938,   0.746094,   0.0625,    0.4375,      0.347656,  0.597656,  0.0332031, 0.0644531, 0.976562,
    0.554688,   0.96875,    0.024292,  0.535156,    0.248047,  0.110352,  0.859375,  0.78125,   0.695312,
    0.945312,   0.75,       0.695312,  0.00915527,  0.910156,  0.726562,  0.259766,  0.194336,  0.527344,
    0.921875,   0.0576172,  0.742188,  0.4375,      0.296875,  0.859375,  0.0297852, 0.363281,  0.198242,
    0.376953,   0.0947266,  0.229492,  0.439453,    0.419922,  0.394531,  0.136719,  0.0625,    0.953125,
    0.890625,   0.246094,   0.671875,  0.796875,    0.0839844, 0.660156,  0.789062,  0.789062,  0.0810547,
    0.578125,   0.582031,   0.120117,  -0.000337601};

static const std::vector<float> expected_bfloat_idft2d_result = {
    0.859375,   0.00817871, 0.00500488, 0.542969,   0.828125,  0.496094,  0.902344,  0.191406,  0.859375,  0.882812,
    0.722656,   0.0566406,  0.914062,   0.632812,   0.25,      0.0444336, 0.296875,  0.847656,  0.6875,    0.570312,
    0.00891113, 0.679688,   0.757812,   0.375,      0.515625,  0.0317383, 0.636719,  0.789062,  0.5,       0.722656,
    0.205078,   0.120117,   0.757812,   0.4375,     0.957031,  0.828125,  0.225586,  0.0078125, 0.0546875, 0.585938,
    0.875,      0.142578,   0.683594,   0.326172,   0.914062,  0.730469,  0.046875,  0.0327148, 0.84375,   0.996094,
    0.898438,   0.996094,   0.24707,    0.34375,    0.226562,  0.953125,  0.59375,   0.851562,  0.917969,  0.173828,
    0.867188,   0.158203,   0.664062,   0.519531,   0.515625,  0.0244141, 0.171875,  0.0664062, 0.609375,  0.484375,
    0.287109,   0.357422,   0.640625,   0.277344,   0.601562,  0.628906,  0.507812,  0.232422,  0.773438,  0.445312,
    0.53125,    0.648438,   0.429688,   0.213867,   0.691406,  0.112305,  0.253906,  0.21875,   0.804688,  0.345703,
    0.789062,   0.160156,   0.640625,   0.785156,   0.90625,   0.6875,    0.246094,  0.322266,  0.226562,  0.238281,
    0.251953,   0.28125,    0.324219,   0.515625,   0.835938,  0.628906,  0.283203,  0.882812,  0.898438,  0.15918,
    0.320312,   0.671875,   0.859375,   0.285156,   0.193359,  0.828125,  0.149414,  0.179688,  0.847656,  0.234375,
    0.859375,   0.570312,   0.945312,   0.0917969,  0.34375,   0.470703,  0.597656,  0.225586,  0.304688,  0.494141,
    0.11084,    0.474609,   0.131836,   0.695312,   0.847656,  0.664062,  0.242188,  0.796875,  0.375,     0.742188,
    0.166016,   0.839844,   0.929688,   0.558594,   0.671875,  0.0761719, 0.6875,    0.0693359, 0.679688,  0.796875,
    0.490234,   0.882812,   0.5625,     0.914062,   0.478516,  0.945312,  0.980469,  0.617188,  0.8125,    0.0130615,
    0.78125,    0.992188,   0.992188,   0.664062,   0.386719,  0.380859,  0.120605,  0.546875,  0.134766,  0.945312,
    0.24707,    0.00476074, 0.144531,   0.3125,     0.416016,  0.675781,  0.570312,  0.734375,  0.0135498, 0.345703,
    0.925781,   0.482422,   0.460938,   0.816406,   0.503906,  0.375,     0.125,     0.419922,  0.746094,  0.734375,
    0.84375,    0.136719,   0.148438,   0.992188,   0.216797,  0.410156,  0.863281,  0.328125,  0.757812,  0.949219,
    0.578125,   0.960938,   0.714844,   0.0424805,  0.289062,  0.585938,  0.0510254, 0.320312,  0.515625,  0.507812,
    0.00964355, 0.519531,   0.78125,    0.34375,    0.796875,  0.460938,  0.0030365, 0.0292969, 0.765625,  0.570312,
    0.441406,   0.820312,   0.289062,   0.34375,    0.480469,  0.382812,  0.308594,  0.71875,   0.410156,  0.0625,
    0.871094,   0.601562,   0.445312,   0.162109,   0.859375,  0.796875,  0.419922,  0.953125,  0.546875,  0.332031,
    0.367188,   0.605469,   0.65625,    0.675781,   0.867188,  0.882812,  0.4375,    0.324219,  0.773438,  0.339844,
    0.138672,   0.228516,   0.0703125,  0.0693359,  0.371094,  0.914062,  0.65625,   0.773438,  0.449219,  0.453125,
    0.157227,   0.796875,   0.625,      0.609375,   0.765625,  0.0839844, 0.300781,  0.554688,  0.84375,   0.914062,
    0.859375,   0.878906,   0.898438,   0.96875,    0.9375,    0.777344,  0.804688,  0.138672,  0.660156,  0.664062,
    0.0456543,  0.75,       0.449219,   0.984375,   0.457031,  0.216797,  0.921875,  0.394531,  0.683594,  0.574219,
    0.205078,   0.539062,   0.458984,   0.180664,   0.660156,  0.628906,  0.142578,  0.914062,  0.359375,  0.511719,
    0.839844,   0.222656,   0.378906,   0.0241699,  0.191406,  0.234375,  0.320312,  0.480469,  0.617188,  0.695312,
    0.433594,   0.0786133,  0.570312,   0.0996094,  0.421875,  0.953125,  0.664062,  0.867188,  0.523438,  0.625,
    0.851562,   0.484375,   0.820312,   0.0888672,  0.6875,    0.416016,  0.59375,   0.0766602, 0.140625,  0.490234,
    0.4375,     0.625,      0.269531,   0.0986328,  0.097168,  0.921875,  0.240234,  0.90625,   0.75,      0.773438,
    0.28125,    0.960938,   0.585938,   0.75,       0.0620117, 0.4375,    0.347656,  0.601562,  0.0322266, 0.0673828,
    0.976562,   0.554688,   0.96875,    0.0263672,  0.53125,   0.248047,  0.108398,  0.859375,  0.777344,  0.695312,
    0.941406,   0.75,       0.6875,     0.00842285, 0.90625,   0.726562,  0.257812,  0.193359,  0.523438,  0.921875,
    0.0554199,  0.742188,   0.435547,   0.294922,   0.859375,  0.0292969, 0.359375,  0.197266,  0.375,     0.09375,
    0.228516,   0.445312,   0.417969,   0.396484,   0.136719,  0.0644531, 0.949219,  0.898438,  0.244141,  0.671875,
    0.796875,   0.0859375,  0.65625,    0.792969,   0.789062,  0.0820312, 0.570312,  0.585938,  0.117676,  0.000293732};

static const std::vector<float> expected_bfloat_idft3d_result = {
    0.859375,   0.0102539, 0.00363159, 0.546875,   0.824219,  0.496094,  0.898438,   0.191406,   0.855469,  0.882813,
    0.71875,    0.0576172, 0.914063,   0.632813,   0.25,      0.045166,  0.296875,   0.84375,    0.6875,    0.570313,
    0.00927734, 0.679688,  0.757813,   0.375,      0.515625,  0.0334473, 0.632813,   0.785156,   0.5,       0.722656,
    0.205078,   0.120117,  0.757813,   0.4375,     0.953125,  0.828125,  0.224609,   0.00952148, 0.0541992, 0.585938,
    0.875,      0.142578,  0.683594,   0.324219,   0.914063,  0.726563,  0.0454102,  0.0366211,  0.84375,   0.996094,
    0.894531,   0.996094,  0.246094,   0.347656,   0.226563,  0.953125,  0.59375,    0.851563,   0.914063,  0.173828,
    0.863281,   0.15918,   0.660156,   0.523438,   0.515625,  0.0249023, 0.171875,   0.0678711,  0.609375,  0.484375,
    0.285156,   0.359375,  0.640625,   0.279297,   0.597656,  0.628906,  0.507813,   0.233398,   0.773438,  0.445313,
    0.53125,    0.648438,  0.429688,   0.212891,   0.6875,    0.110352,  0.253906,   0.21875,    0.804688,  0.347656,
    0.789063,   0.162109,  0.636719,   0.785156,   0.90625,   0.6875,    0.248047,   0.324219,   0.226563,  0.240234,
    0.25,       0.283203,  0.324219,   0.515625,   0.835938,  0.632813,  0.285156,   0.882813,   0.898438,  0.160156,
    0.320313,   0.671875,  0.855469,   0.289063,   0.191406,  0.828125,  0.148438,   0.179688,   0.84375,   0.236328,
    0.859375,   0.570313,  0.945313,   0.0917969,  0.34375,   0.470703,  0.597656,   0.224609,   0.304688,  0.494141,
    0.110352,   0.476563,  0.131836,   0.695313,   0.84375,   0.664063,  0.241211,   0.796875,   0.375,     0.742188,
    0.166016,   0.839844,  0.929688,   0.554688,   0.671875,  0.0761719, 0.6875,     0.0698242,  0.679688,  0.796875,
    0.488281,   0.882813,  0.5625,     0.914063,   0.476563,  0.945313,  0.976563,   0.617188,   0.808594,  0.0130615,
    0.78125,    0.992188,  0.988281,   0.664063,   0.386719,  0.382813,  0.120117,   0.546875,   0.132813,  0.945313,
    0.244141,   0.0055542, 0.145508,   0.3125,     0.414063,  0.679688,  0.570313,   0.734375,   0.0133057, 0.34375,
    0.921875,   0.484375,  0.460938,   0.8125,     0.5,       0.375,     0.123047,   0.417969,   0.742188,  0.730469,
    0.84375,    0.136719,  0.146484,   0.992188,   0.216797,  0.410156,  0.863281,   0.328125,   0.757813,  0.953125,
    0.578125,   0.957031,  0.714844,   0.0419922,  0.291016,  0.585938,  0.0517578,  0.320313,   0.515625,  0.507813,
    0.0106201,  0.515625,  0.78125,    0.34375,    0.796875,  0.460938,  0.00439453, 0.0290527,  0.765625,  0.570313,
    0.441406,   0.820313,  0.292969,   0.339844,   0.480469,  0.382813,  0.308594,   0.71875,    0.410156,  0.0620117,
    0.871094,   0.601563,  0.445313,   0.160156,   0.859375,  0.796875,  0.421875,   0.953125,   0.550781,  0.333984,
    0.367188,   0.605469,  0.65625,    0.679688,   0.867188,  0.882813,  0.4375,     0.324219,   0.773438,  0.339844,
    0.140625,   0.228516,  0.0712891,  0.0698242,  0.373047,  0.910156,  0.65625,    0.773438,   0.449219,  0.453125,
    0.158203,   0.796875,  0.625,      0.609375,   0.765625,  0.0839844, 0.302734,   0.550781,   0.847656,  0.914063,
    0.859375,   0.878906,  0.898438,   0.96875,    0.9375,    0.773438,  0.804688,   0.140625,   0.664063,  0.664063,
    0.0480957,  0.75,      0.451172,   0.984375,   0.460938,  0.21582,   0.921875,   0.392578,   0.683594,  0.574219,
    0.207031,   0.539063,  0.460938,   0.181641,   0.664063,  0.625,     0.141602,   0.914063,   0.359375,  0.511719,
    0.839844,   0.222656,  0.378906,   0.0244141,  0.193359,  0.234375,  0.320313,   0.476563,   0.617188,  0.691406,
    0.433594,   0.0791016, 0.570313,   0.0996094,  0.423828,  0.953125,  0.664063,   0.867188,   0.523438,  0.625,
    0.855469,   0.484375,  0.828125,   0.0878906,  0.6875,    0.416016,  0.59375,    0.0761719,  0.140625,  0.490234,
    0.4375,     0.625,     0.269531,   0.0996094,  0.0976563, 0.921875,  0.240234,   0.90625,    0.75,      0.773438,
    0.283203,   0.960938,  0.585938,   0.746094,   0.0634766, 0.435547,  0.345703,   0.597656,   0.0341797, 0.0644531,
    0.976563,   0.554688,  0.972656,   0.0266113,  0.535156,  0.248047,  0.111328,   0.859375,   0.78125,   0.695313,
    0.945313,   0.75,      0.691406,   0.00946045, 0.90625,   0.726563,  0.257813,   0.193359,   0.523438,  0.921875,
    0.0556641,  0.742188,  0.435547,   0.294922,   0.859375,  0.0296631, 0.359375,   0.196289,   0.375,     0.09375,
    0.229492,   0.443359,  0.417969,   0.396484,   0.137695,  0.0634766, 0.953125,   0.898438,   0.246094,  0.671875,
    0.796875,   0.0859375, 0.660156,   0.792969,   0.789063,  0.0820313, 0.574219,   0.585938,   0.119141,  0.00128174};

static const std::vector<float> expected_float16_idft1d_result = {
    0.859375,   0.0100555,  0.00403976, 0.545898,   0.82666,   0.498047,  0.90332,   0.192871,  0.858887,
    0.887695,   0.723145,   0.0578003,  0.915527,   0.634766,  0.250488,  0.0457764, 0.297607,  0.849609,
    0.688477,   0.574219,   0.00984192, 0.682129,   0.761719,  0.375977,  0.518066,  0.0329285, 0.637207,
    0.789551,   0.500977,   0.725098,   0.204224,   0.121155,  0.760254,  0.440186,  0.956543,  0.829102,
    0.225708,   0.00914001, 0.0545654,  0.588379,   0.878418,  0.143188,  0.686035,  0.327393,  0.916992,
    0.731934,   0.0461426,  0.0350342,  0.845215,   0.99707,   0.897949,  0.998535,  0.24646,   0.34668,
    0.227417,   0.958008,   0.596191,   0.853516,   0.91748,   0.174438,  0.866699,  0.15918,   0.663574,
    0.522461,   0.515137,   0.0249939,  0.172974,   0.0680542, 0.612305,  0.486572,  0.287354,  0.359375,
    0.644531,   0.279541,   0.599609,   0.630859,   0.507324,  0.233887,  0.774902,  0.446045,  0.531738,
    0.65332,    0.430908,   0.214722,   0.692871,   0.111816,  0.253662,  0.220947,  0.806152,  0.347168,
    0.790039,   0.161011,   0.637695,   0.787598,   0.909668,  0.691406,  0.247559,  0.324463,  0.227295,
    0.239746,   0.251953,   0.283936,   0.324707,   0.519043,  0.839355,  0.632324,  0.285156,  0.888672,
    0.902344,   0.160522,   0.322021,   0.676758,   0.859375,  0.289062,  0.193237,  0.828125,  0.148926,
    0.180664,   0.849121,   0.235596,   0.862305,   0.574219,  0.947266,  0.0922852, 0.345215,  0.471924,
    0.600586,   0.226074,   0.306885,   0.495361,   0.111389,  0.476562,  0.132446,  0.696777,  0.849121,
    0.668457,   0.242188,   0.79834,    0.376221,   0.745605,  0.16748,   0.841797,  0.930664,  0.55957,
    0.674805,   0.0773315,  0.69043,    0.0697021,  0.680664,  0.79834,   0.490234,  0.884766,  0.566895,
    0.918457,   0.479004,   0.947266,   0.98291,    0.620605,  0.813477,  0.0135803, 0.783203,  0.993652,
    0.992188,   0.666992,   0.388428,   0.383545,   0.120422,  0.547852,  0.133057,  0.946777,  0.245972,
    0.00572968, 0.145752,   0.312256,   0.416992,   0.679688,  0.573242,  0.738281,  0.0131683, 0.346191,
    0.927734,   0.485107,   0.460449,   0.816895,   0.504395,  0.375977,  0.124512,  0.42041,   0.74707,
    0.734375,   0.844727,   0.137573,   0.147827,   0.996582,  0.217896,  0.412109,  0.865723,  0.328613,
    0.757324,   0.952148,   0.580566,   0.959961,   0.718262,  0.0430298, 0.292725,  0.587891,  0.052063,
    0.321777,   0.520508,   0.509766,   0.011673,   0.520996,  0.782227,  0.345703,  0.796875,  0.461914,
    0.00485229, 0.0290375,  0.76709,    0.571777,   0.443848,  0.825195,  0.293213,  0.343994,  0.480957,
    0.383057,   0.310791,   0.717773,   0.4104,     0.0620422, 0.874512,  0.601562,  0.448242,  0.161865,
    0.86084,    0.797363,   0.421631,   0.956055,   0.553223,  0.334717,  0.368164,  0.606934,  0.658203,
    0.679199,   0.870605,   0.886719,   0.438721,   0.324707,  0.775879,  0.340332,  0.141357,  0.230103,
    0.0716553,  0.069458,   0.373535,   0.912109,   0.658691,  0.777344,  0.451416,  0.454346,  0.159058,
    0.801758,   0.625977,   0.611816,   0.77002,    0.0848389, 0.303711,  0.552734,  0.849609,  0.917969,
    0.86084,    0.881836,   0.900879,   0.972168,   0.941406,  0.778809,  0.805664,  0.140259,  0.665039,
    0.665039,   0.0482788,  0.750488,   0.452148,   0.987793,  0.461182,  0.216675,  0.927246,  0.394043,
    0.685547,   0.576172,   0.207397,   0.541016,   0.460449,  0.181885,  0.663086,  0.628906,  0.143066,
    0.916992,   0.361816,   0.513184,   0.842285,   0.223389,  0.380127,  0.0247345, 0.193726,  0.236084,
    0.322021,   0.479736,   0.620605,   0.694824,   0.436279,  0.0797729, 0.574707,  0.100037,  0.424805,
    0.953613,   0.666504,   0.868652,   0.526855,   0.628418,  0.858398,  0.486816,  0.827148,  0.088501,
    0.692871,   0.417969,   0.597168,   0.0761719,  0.141968,  0.490479,  0.440186,  0.625977,  0.271729,
    0.100037,   0.0982666,  0.924805,   0.242065,   0.907715,  0.752441,  0.776367,  0.284668,  0.963379,
    0.588379,   0.749023,   0.0639648,  0.4375,     0.347412,  0.600586,  0.03479,   0.0667725, 0.979004,
    0.555664,   0.975586,   0.02565,    0.537598,   0.249146,  0.110657,  0.859863,  0.78125,   0.695801,
    0.946289,   0.749512,   0.695312,   0.00930023, 0.910645,  0.729004,  0.259277,  0.194824,  0.52832,
    0.924316,   0.0562134,  0.743652,   0.437256,   0.295898,  0.859863,  0.0296478, 0.362793,  0.196777,
    0.37793,    0.0940552,  0.230225,   0.444092,   0.42041,   0.397461,  0.137939,  0.0640259, 0.953613,
    0.898926,   0.246338,   0.673828,   0.80127,    0.0863647, 0.661621,  0.79541,   0.793457,  0.0834351,
    0.577148,   0.586914,   0.119934,   0.00131321};

static const std::vector<float> expected_float16_idft2d_result = {
    0.859375,   0.0100098,  0.00434875, 0.545898,   0.827148,  0.497803,  0.903809,  0.192871,   0.858887,  0.887695,
    0.723633,   0.0575562,  0.916016,   0.634277,   0.250732,  0.0457764, 0.298096,  0.849121,   0.688477,  0.574219,
    0.00975037, 0.682129,   0.761719,   0.375977,   0.518066,  0.0331116, 0.637207,  0.789551,   0.500977,  0.725098,
    0.204224,   0.121216,   0.760254,   0.440186,   0.956543,  0.829102,  0.225586,  0.00925446, 0.0545044, 0.588867,
    0.878418,   0.143311,   0.686523,   0.327393,   0.916992,  0.732422,  0.0462341, 0.0350342,  0.845215,  0.997559,
    0.897949,   0.998535,   0.24646,    0.346924,   0.227417,  0.958496,  0.596191,  0.854004,   0.91748,   0.174438,
    0.867188,   0.15918,    0.664062,   0.522461,   0.515625,  0.0251007, 0.17334,   0.0681152,  0.612793,  0.486572,
    0.287354,   0.359375,   0.644531,   0.279785,   0.600098,  0.630859,  0.507812,  0.234009,   0.775391,  0.446289,
    0.531738,   0.653809,   0.430908,   0.214844,   0.692871,  0.111877,  0.253418,  0.221069,   0.806152,  0.347168,
    0.790039,   0.161133,   0.637695,   0.787598,   0.909668,  0.691895,  0.247681,  0.324463,   0.227173,  0.239746,
    0.251953,   0.28418,    0.324707,   0.519043,   0.839355,  0.632812,  0.285156,  0.888672,   0.902344,  0.160522,
    0.322021,   0.676758,   0.859863,   0.289307,   0.193237,  0.828125,  0.148926,  0.180664,   0.849121,  0.235596,
    0.862305,   0.574219,   0.947266,   0.0922852,  0.344971,  0.471924,  0.600586,  0.226074,   0.306885,  0.495361,
    0.11145,    0.476562,   0.132568,   0.696777,   0.849121,  0.668457,  0.242065,  0.79834,    0.375977,  0.745117,
    0.167603,   0.841309,   0.930664,   0.55957,    0.674805,  0.0771484, 0.69043,   0.0695801,  0.680664,  0.79834,
    0.490479,   0.884277,   0.566406,   0.917969,   0.479004,  0.947266,  0.98291,   0.620605,   0.813477,  0.0133667,
    0.783203,   0.993652,   0.992188,   0.666992,   0.388428,  0.383545,  0.120544,  0.547852,   0.133057,  0.946777,
    0.246094,   0.00577164, 0.145752,   0.3125,     0.416992,  0.679688,  0.573242,  0.738281,   0.0131683, 0.346436,
    0.927734,   0.485352,   0.460693,   0.817383,   0.504395,  0.375977,  0.124451,  0.420166,   0.74707,   0.734375,
    0.845215,   0.137573,   0.147705,   0.996582,   0.217896,  0.412109,  0.865723,  0.328613,   0.757324,  0.952148,
    0.580566,   0.959961,   0.718262,   0.0429993,  0.292725,  0.588379,  0.0520935, 0.321777,   0.520508,  0.509766,
    0.0115356,  0.520996,   0.782227,   0.345703,   0.796875,  0.461914,  0.0048027, 0.0290375,  0.767578,  0.571777,
    0.444092,   0.825195,   0.293457,   0.344238,   0.481201,  0.383057,  0.310791,  0.717773,   0.410645,  0.0620728,
    0.875,      0.602051,   0.448486,   0.161865,   0.861328,  0.797363,  0.421875,  0.956055,   0.553711,  0.334717,
    0.368164,   0.607422,   0.658203,   0.679199,   0.871094,  0.886719,  0.438721,  0.324707,   0.776367,  0.340332,
    0.141479,   0.230225,   0.0717163,  0.0693359,  0.373535,  0.912109,  0.658691,  0.777344,   0.45166,   0.45459,
    0.15918,    0.801758,   0.625977,   0.611328,   0.77002,   0.0849609, 0.303711,  0.552734,   0.849609,  0.917969,
    0.86084,    0.881836,   0.900879,   0.972168,   0.941406,  0.779297,  0.805664,  0.140503,   0.665527,  0.665039,
    0.0483704,  0.750488,   0.452148,   0.987793,   0.461182,  0.216797,  0.927246,  0.394043,   0.685547,  0.576172,
    0.207642,   0.541016,   0.460449,   0.181885,   0.663086,  0.628906,  0.143066,  0.916504,   0.362061,  0.513184,
    0.842285,   0.223389,   0.380127,   0.0245361,  0.193726,  0.235962,  0.322021,  0.479736,   0.620117,  0.694336,
    0.436035,   0.0795898,  0.574707,   0.0999756,  0.424805,  0.953125,  0.666504,  0.868164,   0.526855,  0.628418,
    0.85791,    0.486572,   0.827148,   0.0883789,  0.692871,  0.417969,  0.597168,  0.0760498,  0.141968,  0.490479,
    0.440186,   0.625488,   0.271729,   0.0999146,  0.0983887, 0.924805,  0.241943,  0.907227,   0.752441,  0.775879,
    0.284912,   0.963379,   0.588379,   0.749512,   0.0640259, 0.4375,    0.347656,  0.600586,   0.0349731, 0.0668335,
    0.979004,   0.555664,   0.976074,   0.0257111,  0.537598,  0.24939,   0.110962,  0.859863,   0.78125,   0.695801,
    0.946289,   0.749512,   0.694824,   0.00910187, 0.910645,  0.729004,  0.259277,  0.194824,   0.52832,   0.924316,
    0.0563049,  0.743164,   0.437256,   0.295654,   0.859863,  0.0297089, 0.363037,  0.196777,   0.37793,   0.0940552,
    0.22998,    0.444092,   0.420166,   0.397217,   0.137939,  0.0639038, 0.953613,  0.898438,   0.24646,   0.673828,
    0.800781,   0.0861206,  0.661133,   0.794922,   0.792969,  0.083252,  0.577148,  0.586426,   0.119751,  0.00126743};

static const std::vector<float> expected_float16_idft3d_result = {
    0.859375,  0.00984192, 0.00445557, 0.545898,   0.827148,  0.497803,  0.903809,   0.192749,   0.858887,  0.887695,
    0.723633,  0.0576477,  0.916016,   0.634766,   0.250977,  0.0456543, 0.298096,   0.849121,   0.688477,  0.574219,
    0.009758,  0.682129,   0.761719,   0.376221,   0.518066,  0.0330811, 0.637207,   0.789551,   0.500977,  0.724609,
    0.204224,  0.121155,   0.760254,   0.440186,   0.956543,  0.829102,  0.225464,   0.00920105, 0.0546875, 0.588379,
    0.878418,  0.143433,   0.686035,   0.327393,   0.916992,  0.731934,  0.0462646,  0.0350647,  0.845215,  0.997559,
    0.897949,  0.999023,   0.246338,   0.346924,   0.227295,  0.958008,  0.596191,   0.854004,   0.91748,   0.174438,
    0.866699,  0.15918,    0.664062,   0.522461,   0.515625,  0.0249634, 0.173218,   0.0679932,  0.612793,  0.486572,
    0.287354,  0.359375,   0.644531,   0.279541,   0.600098,  0.631348,  0.507812,   0.234009,   0.774902,  0.446289,
    0.531738,  0.65332,    0.430664,   0.214722,   0.692871,  0.111938,  0.253418,   0.220947,   0.806152,  0.347168,
    0.790039,  0.161011,   0.637695,   0.787598,   0.90918,   0.691406,  0.247437,   0.324463,   0.227295,  0.239746,
    0.251953,  0.28418,    0.324707,   0.519043,   0.839355,  0.632324,  0.285156,   0.888672,   0.902344,  0.160522,
    0.322021,  0.676758,   0.859375,   0.289062,   0.193237,  0.828125,  0.148804,   0.180908,   0.849121,  0.235718,
    0.861816,  0.574219,   0.947266,   0.0922241,  0.345215,  0.471924,  0.600586,   0.226074,   0.306885,  0.495361,
    0.111389,  0.476318,   0.132446,   0.696777,   0.849121,  0.668457,  0.242065,   0.79834,    0.375977,  0.745117,
    0.16748,   0.841309,   0.930664,   0.55957,    0.674805,  0.0770874, 0.69043,    0.069458,   0.680664,  0.797852,
    0.490234,  0.884277,   0.566406,   0.917969,   0.479004,  0.947266,  0.98291,    0.620605,   0.813477,  0.0134964,
    0.783203,  0.994141,   0.992188,   0.666992,   0.388428,  0.383545,  0.120361,   0.547852,   0.133179,  0.946777,
    0.246094,  0.00577164, 0.145752,   0.312256,   0.416992,  0.679688,  0.573242,   0.738281,   0.0133972, 0.346191,
    0.928223,  0.485107,   0.460449,   0.817383,   0.504395,  0.375977,  0.124512,   0.420166,   0.74707,   0.733887,
    0.844727,  0.137451,   0.147827,   0.996582,   0.217896,  0.412354,  0.865723,   0.328613,   0.757324,  0.952148,
    0.580566,  0.959961,   0.718262,   0.0429382,  0.292725,  0.588379,  0.0520935,  0.321533,   0.520508,  0.509766,
    0.0115204, 0.520996,   0.782227,   0.345703,   0.796875,  0.462158,  0.00487137, 0.0290985,  0.767578,  0.571777,
    0.444092,  0.825195,   0.293457,   0.344238,   0.481201,  0.383057,  0.310791,   0.717773,   0.410645,  0.0619812,
    0.874512,  0.602051,   0.448242,   0.161865,   0.861328,  0.797363,  0.421875,   0.956055,   0.553711,  0.334717,
    0.368164,  0.607422,   0.658203,   0.679199,   0.871094,  0.887207,  0.438721,   0.324707,   0.776367,  0.340332,
    0.141479,  0.230225,   0.0718994,  0.069458,   0.373535,  0.912109,  0.65918,    0.777344,   0.45166,   0.45459,
    0.159302,  0.801758,   0.625977,   0.611816,   0.77002,   0.085022,  0.303955,   0.552734,   0.849609,  0.917969,
    0.86084,   0.881836,   0.900879,   0.972168,   0.941406,  0.779297,  0.805664,   0.140381,   0.665039,  0.665039,
    0.0485535, 0.750488,   0.452148,   0.987793,   0.461182,  0.216675,  0.927246,   0.394043,   0.685547,  0.576172,
    0.20752,   0.541016,   0.460449,   0.182129,   0.663086,  0.628906,  0.143311,   0.916992,   0.362061,  0.513184,
    0.842285,  0.223389,   0.380127,   0.024704,   0.193726,  0.235962,  0.322266,   0.479736,   0.620605,  0.694336,
    0.436279,  0.0796509,  0.574707,   0.0999756,  0.424805,  0.953613,  0.666504,   0.868652,   0.526855,  0.628418,
    0.858398,  0.486572,   0.827148,   0.0883179,  0.692871,  0.417969,  0.597168,   0.0759277,  0.141968,  0.490479,
    0.44043,   0.625488,   0.271729,   0.0999146,  0.0983887, 0.924805,  0.242065,   0.907227,   0.752441,  0.775879,
    0.284912,  0.963379,   0.588379,   0.749512,   0.0641479, 0.437744,  0.347412,   0.600586,   0.0350037, 0.0668945,
    0.979004,  0.555664,   0.976074,   0.0257721,  0.537598,  0.249268,  0.111145,   0.859863,   0.78125,   0.695801,
    0.946289,  0.749512,   0.694824,   0.00917816, 0.910645,  0.729004,  0.259521,   0.19458,    0.52832,   0.924316,
    0.0562744, 0.743164,   0.437256,   0.295898,   0.859863,  0.0297546, 0.363037,   0.196899,   0.37793,   0.0941162,
    0.230225,  0.443848,   0.42041,    0.397217,   0.137817,  0.0637817, 0.953613,   0.898438,   0.246338,  0.673828,
    0.800781,  0.0863037,  0.661621,   0.794922,   0.793457,  0.083252,  0.577148,   0.586426,   0.119751,  0.00126934};

static const std::vector<float> data_1 = {
    0.9795938,   0.14046684,  0.9819369,    0.51320475,  0.9390985,  0.06454252,  0.48972926,   0.042538255,
    0.3341647,   0.14752749,  0.44628578,   0.8509109,   0.6611515,  0.5711897,   0.10807402,   0.67733586,
    0.4091941,   0.23590194,  0.4385734,    0.40270114,  0.75568604, 0.9842337,   0.82819414,   0.49742407,
    0.7397849,   0.6104118,   0.019504193,  0.7756829,   0.9271429,  0.6423316,   0.3300541,    0.8688829,
    0.21220064,  0.76539195,  0.8143789,    0.70724595,  0.54020476, 0.29437974,  0.19398275,   0.20308031,
    0.30458412,  0.040420562, 0.36627868,   0.61882246,  0.3416973,  0.5482437,   0.68851316,   0.5670022,
    0.58812225,  0.6487681,   0.88266903,   0.07287276,  0.7716641,  0.12443388,  0.4170407,    0.8380076,
    0.17115247,  0.8118648,   0.7704737,    0.5179812,   0.9407177,  0.7311383,   0.4538601,    0.01992845,
    0.4758718,   0.25867644,  0.55573237,   0.89606065,  0.8505143,  0.47349417,  0.3970769,    0.3293097,
    0.7601557,   0.24247961,  0.8102311,    0.7387785,   0.15742134, 0.8387721,   0.100493915,  0.3733577,
    0.4904671,   0.9106489,   0.0049963384, 0.89285916,  0.24380954, 0.7329451,   0.9373891,    0.52886724,
    0.65965563,  0.7307209,   0.5160155,    0.97944415,  0.43991584, 0.9839402,   0.6350642,    0.16712844,
    0.40538687,  0.80509937,  0.4988989,    0.02185218,  0.74142575, 0.8026278,   0.28912508,   0.50405765,
    0.7768013,   0.9817653,   0.9995751,    0.74799776,  0.8615655,  0.058107413, 0.27611437,   0.76372087,
    0.93234706,  0.7603203,   0.30816016,   0.80595773,  0.8843074,  0.46457228,  0.43644127,   0.6553406,
    0.9050378,   0.5044161,   0.49364874,   0.59174323,  0.2650881,  0.78394204,  0.57706285,   0.33071348,
    0.7140054,   0.5885716,   0.60252094,   0.92644346,  0.91704935, 0.64020824,  0.99652874,   0.8375778,
    0.45622328,  0.3755286,   0.8324417,    0.77270067,  0.50742614, 0.7814994,   0.30720684,   0.36613366,
    0.9426107,   0.12557131,  0.87243265,   0.002567238, 0.8350289,  0.1262151,   0.35253504,   0.07578735,
    0.34082502,  0.9211622,   0.38055828,   0.3247621,   0.5061271,  0.87862396,  0.1869049,    0.7774487,
    0.030804915, 0.25322768,  0.06073754,   0.27092665,  0.9209875,  0.86690956,  0.74456835,   0.42403135,
    0.61839956,  0.9004572,   0.94674456,   0.17315134,  0.74403226, 0.30930993,  0.23992635,   0.9080931,
    0.4886828,   0.9973065,   0.32888287,   0.32976696,  0.09137513, 0.1410893,   0.4248779,    0.019689998,
    0.6828394,   0.47350892,  0.02358055,   0.94660497,  0.9253734,  0.1509718,   0.540138,     0.7050524,
    0.20855357,  0.9753569,   0.0044813985, 0.5063834,   0.6836877,  0.2784342,   0.0139586115, 0.8785785,
    0.4754602,   0.38955635,  0.151705,     0.5694773,   0.14548586, 0.6798502,   0.057361145,  0.031760257,
    0.91168743,  0.5762714,   0.54128575,   0.53421247,  0.5860678,  0.97197753,  0.940639,     0.18688098,
    0.54635745,  0.513619,    0.5645304,    0.91558236,  0.8496063,  0.6258071,   0.31261826,   0.20282389,
    0.2723365,   0.5039135,   0.6405068,    0.65471405,  0.5857442,  0.57205665,  0.23835625,   0.32288164,
    0.068663165, 0.43674967,  0.049117915,  0.78065604,  0.98437595, 0.60793245,  0.38907775,   0.6610265,
    0.5587009,   0.89418066,  0.6170649,    0.1305905,   0.5760506,  0.10885323,  0.5303117,    0.16950679,
    0.9630447,   0.9476875,   0.22327174,   0.87473476,  0.917824,   0.44959846,  0.055421904,  0.22361691,
    0.9334828,   0.16427046,  0.5914317,    0.81789917,  0.48431975, 0.3067152,   0.53250873,   0.19298424,
    0.23529118,  0.4841604,   0.24943262,   0.41821656,  0.59484303, 0.4535004,   0.50373393,   0.6057788,
    0.6799498,   0.21368006,  0.17095569,   0.97966874,  0.3839032,  0.48672524,  0.9375583,    0.84598905,
    0.049092207, 0.47682214,  0.56488436,   0.7817405,   0.8975917,  0.75874424,  0.43242812,   0.8459973,
    0.7138231,   0.9834999,   0.7273379,    0.05828699,  0.6884237,  0.07486352,  0.4326547,    0.78577167,
    0.8844588,   0.9474644,   0.542272,     0.49642876,  0.48886803, 0.11854455,  0.01492267,   0.22648218,
    0.7607531,   0.5930919,   0.9450968,    0.02894685,  0.67735505, 0.46363172,  0.18415985,   0.66824925,
    0.6137258,   0.6086626,   0.6422855,    0.7637218,   0.56419605, 0.74026155,  0.18709394,   0.14683136,
    0.32289994,  0.15482259,  0.11222768,   0.9085655,   0.43263617, 0.32097924,  0.29690787,   0.77809244,
    0.2413839,   0.8267769,   0.82795614,   0.018312717, 0.9958108,  0.769578,    0.13144562,   0.45904484,
    0.38071582,  0.24182741,  0.7200288,    0.20737973,  0.5285696,  0.3680287,   0.46252182,   0.89153767,
    0.13071166,  0.84319293,  0.10841641,   0.40668696,  0.7636801,  0.42153865,  0.65055484,   0.86845386,
    0.6452055,   0.6112245,   0.84526664,   0.15358071,  0.7889171,  0.6356269,   0.2515375,    0.86599886,
    0.20071381,  0.20584217,  0.24220705,   0.049883988, 0.77259976, 0.49566683,  0.8112268,    0.49028614,
    0.2187354,   0.70172536,  0.47309682,   0.12539592,  0.13696012, 0.33588144,  0.98134226,   0.537496,
    0.9999663,   0.13245043,  0.5659106,    0.39207155,  0.48483336, 0.49371332,  0.12930158,   0.103645995};

static const std::vector<float> idft1d_input_data_1 = {
    4.940035,    3.0077164,    -0.43999052,  -0.72356576,  0.35775006,  -1.1781573,    0.35552078,  -0.5878226,
    0.8879826,   -1.1602633,   0.71755445,   0.15355057,   -0.9307331,  0.48268145,    1.9486318,   1.1295953,
    4.4481335,   5.01757,      -0.57099926,  -0.85269475,  -0.7217729,  -0.08008081,   -1.1429803,  -1.1934547,
    1.2154821,   -0.07181215,  0.59362185,   0.44658875,   -0.345927,   -1.480422,     -0.20200539, 0.10152125,
    3.4618404,   3.744587,     0.12548631,   0.8791516,    0.19086862,  -0.33497274,   -0.69986683, 0.6364535,
    -0.6644666,  -0.44771492,  -0.8179812,   0.17377639,   -0.92110324, 0.26135075,    1.0228279,   1.2105042,
    4.9957,      3.764995,     0.17936486,   -0.9405504,   -1.2201893,  -0.17718112,   1.1820351,   0.5077594,
    -0.052387,   0.86741495,   -0.55883414,  0.9524643,    -0.68602496, 1.3873026,     0.8653134,   -1.17206,
    4.107497,    4.150929,     -0.95916677,  -0.56429225,  1.1602635,   -1.679503,     0.5507363,   0.53716975,
    0.38042903,  -0.5240841,   -0.33995685,  -0.78949994,  -0.7040798,  0.05728233,    -0.38874817, 0.8814098,
    3.9273133,   5.9265537,    -0.80074155,  0.20659067,   1.642705,    0.9759259,     0.85149074,  0.44840366,
    -0.25961697, 0.78995633,   -0.039625674, 0.545478,     -0.70991015, -1.1269569,    -0.68787766, -0.48076022,
    4.848893,    4.6852283,    -0.6871975,   -0.041547477, -0.91873163, -0.0071051717, -1.4497755,  0.3778788,
    0.7214658,   0.6099715,    1.4334095,    -0.07150489,  0.07712549,  1.859364,      -0.78209424, -0.97149,
    4.8020935,   4.897006,     0.05723229,   -0.21079391,  1.0996364,   0.22791737,    0.7594234,   1.1837918,
    1.1714673,   0.12949562,   -0.64135337,  -0.5158363,   0.2763425,   -0.19547313,   -0.06606534, 0.56645525,
    5.3334026,   5.288664,     -0.09143779,  -0.7460747,   0.2411859,   -0.5888344,    1.4911498,   0.52246934,
    -0.1439941,  -0.51704764,  0.32441977,   0.35291424,   -0.7496793,  -0.32638037,   -0.6930033,  0.72286314,
    4.4170227,   3.232138,     -0.64390934,  -1.3210952,   -0.58362705, -0.6716566,    0.39952934,  -1.1999383,
    0.83216095,  0.8710072,    0.34266293,   -0.92789006,  0.46818644,  0.7554455,     2.3088598,   0.26656008,
    4.306201,    4.1061068,    -1.286478,    -0.14309806,  -1.9038618,  -0.045521975,  -0.43500268, -0.6120295,
    0.3222475,   0.5537019,    1.2264881,    -1.5052714,   -0.12776875, 0.00045275688, -1.8553859,  -0.32851917,
    3.50575,     3.7639909,    -0.8274512,   1.2718699,    0.7064032,   1.7913067,     -1.4024514,  -0.49303484,
    0.8707912,   -0.23823786,  0.41937304,   1.443722,     -0.396856,   0.56620187,    1.0339032,   -0.12736642,
    1.7406936,   4.309397,     -0.18755847,  -0.46101326,  0.020362198, 0.3217622,     0.7620988,   1.9022003,
    1.2856812,   0.3369981,    -1.149087,    0.5562107,    -0.31068176, 0.4914955,     -0.49307993, 0.34580433,
    5.2527924,   4.527175,     -0.029956281, -0.35984623,  1.0824606,   -0.360453,     0.19873115,  -0.3701315,
    0.53464556,  0.8481753,    1.4529572,    1.012228,     -1.037719,   -0.6553353,    -0.16041204, -0.03164065,
    3.2281785,   4.5399303,    0.3643899,    0.30424678,   -0.7776585,  -0.3015166,    -0.61336106, -0.7931169,
    0.5940609,   -0.29862595,  -0.02879478,  0.6273444,    -1.6805825,  -0.17713517,   1.0924593,   0.1301811,
    4.4416904,   3.7987688,    -1.3668841,   -0.81391287,  0.64007276,  1.0288135,     -0.57070565, -0.52160406,
    1.58955,     1.0018709,    -0.123293996, 1.390446,     -0.5843305,  1.5380195,     0.44350854,  -0.26895642,
    4.125044,    3.443525,     0.7636179,    0.10296479,   0.52696383,  0.08359367,    0.6142223,   -1.2670981,
    0.3708297,   -0.6262324,   0.339195,     -0.5216981,   -0.34774148, -0.30716318,   1.0757314,   0.4062716,
    4.1163635,   5.389367,     -0.1369414,   0.3118773,    -0.48302984, 0.07917905,    1.6785579,   -0.9954241,
    -0.09528947, -1.517424,    0.85461855,   0.18921553,   -0.62187576, -1.1891136,    0.12719542,  -0.558237,
    4.492761,    3.6913419,    -0.29317212,  -1.2950531,   -0.03654802, 0.91552365,    0.123229444, 0.514639,
    1.0583864,   0.5574026,    -0.13546133,  0.9680127,    0.87852824,  2.559589,      -0.3771388,  -0.043456703,
    4.574666,    4.013397,     -0.06427416,  -1.2290373,   0.11051571,  -1.2182673,    0.05659631,  0.77380556,
    0.65739393,  0.7978984,    -0.19493088,  0.9715779,    0.1553396,   1.2139899,     0.79071796,  -0.57862896,
    3.361268,    4.236172,     -0.13507411,  0.6842204,    -1.1744223,  -0.62078804,   2.008315,    -1.2499349,
    0.62419355,  -0.091858864, -0.5990913,   -0.90177983,  -0.55390406, 0.40287262,    -0.94808567, -1.2203228,
    3.745199,    4.248646,     0.63732016,   -0.82302505,  -1.9267471,  0.58008444,    -0.38652933, -0.9787377,
    -0.1378448,  -0.4994706,   -0.24433172,  0.09051508,   0.3651026,   0.010821462,   0.9935576,   -0.69421434,
    4.5576744,   3.50811,      1.745633,     0.16605312,   -1.8684177,  -0.33893645,   -0.17866233, 0.5833766,
    0.2571981,   0.38861072,   -0.5767294,   0.61207676,   0.43722266,  -0.28951776,   0.78772557,  0.26002276,
    3.9901466,   2.82238,      -1.4889656,   -0.1150527,   0.47323376,  0.07621753,    0.16292411,  0.17989358,
    -0.30915606, 0.50516117,   -0.38916004,  1.9493489,    0.72058266,  -0.067055345,  -1.4097221,  0.26290974};

static const std::vector<float> idft2d_input_data_1 = {
    25.880518,   25.61235,    -2.4660468,  -1.9953613,   1.409625,    -2.473969,   1.0969357,    0.34850854,
    1.5074215,   -0.546504,   -0.44522142, 1.482357,     -4.297778,   -0.41876173, 2.5581412,    1.6702101,
    -0.79171646, 0.87513673,  -0.5556948,  -1.4017968,   1.6127079,   3.341402,    -2.2336023,   0.7553977,
    0.8801464,   -1.5552741,  2.8809369,   -0.12944597,  -0.08941656, -2.4948978,  1.1106122,    -0.5771601,
    1.5280423,   -3.6573076,  -1.325342,   -0.75811887,  -4.0773964,  0.41215408,  0.24999249,   0.3498589,
    -0.31276864, -2.3484,     -0.4591713,  -0.04454562,  -0.7590859,  2.5111904,   3.1611128,    -0.09711918,
    -0.8617741,  -3.8058863,  -0.0812951,  1.1779473,    2.0081396,   -3.9112964,  -0.6841551,   0.82309175,
    -0.2995335,  -3.7176208,  -0.43554613, -2.4067037,   -0.81405425, 2.0213914,   2.6072812,    4.772808,
    2.3986423,   -1.6369095,  3.009512,    -2.2388682,   0.08045465,  -2.0042,     3.2657382,    -0.93855727,
    1.3121321,   2.0163581,   1.3805538,   1.8802332,    0.20659024,  3.5175233,   2.7225797,    -1.7004844,
    1.4864945,   0.6589138,   -1.221076,   0.8748032,    1.1129706,   -2.4330344,  0.43821555,   -4.865236,
    2.2404962,   -0.81013864, 1.3837745,   0.13940874,   0.16934663,  -2.240356,   -0.46793693,  2.7093167,
    27.21336,    25.973133,   -3.4792416,  -1.1907393,   -1.358995,   0.70610523,  -0.63712704,  -0.22086221,
    3.7741385,   1.4088898,   3.1050003,   -1.2238663,   -0.45265055, 2.6596098,   -0.053786665, 0.12850314,
    1.7713342,   -0.92604506, -1.5456572,  0.4535787,    -0.4252041,  -0.20687354, -0.26421398,  -1.5723603,
    0.21247786,  -0.19034994, 0.116511285, 3.5963366,    -0.9552689,  1.4078308,   0.17855054,   -1.2697299,
    0.24928832,  -1.3436013,  -1.018871,   -1.1798176,   -2.4574528,  0.14592099,  -0.7871367,   -1.3267987,
    1.6891341,   0.8528522,   -2.194655,   -0.7497572,   0.66770875,  1.4708114,   2.0073843,    0.8376069,
    1.7636304,   2.1868649,   -0.65098536, -0.6707011,   -3.8038197,  -1.9890289,  -0.15012956,  0.7975005,
    -1.9746995,  -0.11563957, 2.8636346,   -1.2238576,   -1.1479954,  0.40726233,  -6.6071806,   -1.2827955,
    0.335096,    -0.8774332,  0.5047921,   -1.7173706,   -0.6906272,  -2.8883119,  -1.7264752,   -0.91851616,
    -0.8023921,  2.1811929,   4.4178715,   -1.0245608,   1.4208769,   3.714005,    2.626697,     -3.0808997,
    -2.2393522,  3.0984519,   2.0667777,   4.0557647,    3.22371,     4.1895566,   -5.1335697,   5.5083103,
    1.4301378,   -0.47711706, 0.29209352,  0.19667566,   0.9300822,   1.4966636,   -2.8442304,   -1.1616251,
    22.90476,    26.008162,   -0.59333247, -0.9156835,   1.009171,    0.85137844,  2.0695426,    -2.0451744,
    4.279478,    -0.2552395,  1.3455946,   3.2537463,    -4.582932,   -0.29923248, 2.0854027,    0.023423433,
    -1.4901955,  1.2697036,   0.12445855,  0.37839913,   -0.90889513, -0.96464497, 3.2230172,    5.11582,
    1.7657483,   -1.2759314,  1.6806445,   -0.48582482,  1.0328861,   -0.21219438, -1.8203479,   0.28618455,
    -3.8749995,  -2.6027172,  -2.7910428,  -1.8929406,   0.43884945,  -0.8854169,  -0.6166424,   3.3119302,
    3.9380612,   1.783706,    -2.8637185,  0.45624626,   1.298605,    2.399745,    -0.42191154,  0.3671223,
    -4.7169294,  -1.4224572,  2.4742305,   0.80807984,   -1.4698355,  -0.64370054, -0.54362357,  1.729145,
    0.2216661,   -0.920482,   -3.022968,   -1.9300321,   -0.09508008, 0.31362647,  1.264819,     1.741091,
    -0.48260987, 0.91905135,  -1.2789521,  -1.0161536,   0.53328425,  4.0857644,   -0.8787215,   2.8750324,
    0.4081546,   2.4881384,   -2.2990177,  2.1299765,    0.59928864,  3.988031,    -1.8122058,   -0.16000175,
    -1.8958641,  1.6846397,   0.9392875,   -0.12778088,  0.51960033,  -0.5128077,  1.3190198,    0.42644808,
    -2.8990207,  0.20179635,  -1.7350545,  -0.08684918,  -0.11685985, -3.241004,   -2.2542362,   -0.18299285,
    24.721714,   22.520046,   0.40146637,  -2.611894,    -4.422385,   -0.6061659,  1.7858734,    -0.17695832,
    2.1501722,   1.6577435,   -2.1397042,  3.6897519,    2.0028722,   3.830699,    -0.16294527,  -2.0136907,
    2.7324684,   -0.48164713, -3.0283842,  -1.1742884,   2.3383465,   -0.04261756, -1.3686588,   0.50161046,
    0.76707345,  0.40514386,  -1.7530769,  2.333971,     2.7187724,   4.413412,    -3.610829,    0.57066756,
    0.3970142,   -0.89236856, -1.0979964,  -4.7337284,   -1.6107149,  3.461636,    0.8141506,    1.3783914,
    0.97534364,  -1.261203,   0.9644269,   -0.4446571,   1.3737998,   1.5714393,   1.5593243,    -3.5085554,
    0.10169166,  0.3512014,   2.2333064,   1.7223357,    -1.7363904,  0.5177647,   2.1198907,    -0.12688133,
    1.7293842,   0.05056551,  -0.4828595,  -2.333132,    -0.4791782,  1.5151871,   -0.91205263,  0.0061766207,
    -0.4048485,  2.1922839,   1.728973,    0.9913887,    0.14321594,  1.6313545,   -3.389923,    -2.5937288,
    -0.36389086, -0.2227447,  0.03589952,  -0.069511056, 0.3542207,   2.3090918,   0.45287704,   3.309232,
    -0.59147376, -1.541465,   -1.9963981,  -1.9641305,   5.0686407,   0.53117156,  0.77804404,   4.1053996,
    1.0922346,   2.7149107,   2.5625482,   2.6316533,    -0.69931746, 1.7177012,   0.4107918,    1.375428787};

static const std::vector<float> idft3d_input_data_1 = {
    100.72035,   100.11369,    -6.1371527,  -6.7136793,   -3.3625832,  -1.5226498,   4.315223,    -2.0944858,
    11.711212,   2.2648964,    1.8656702,   7.201989,     -7.3304863,  5.772318,     4.4268136,   -0.1915536,
    2.2218904,   0.7371478,    -5.005279,   -1.7441093,   2.6169558,   2.1272666,    -0.64345765, 4.800469,
    3.6254454,   -2.6164103,   2.9250154,   5.315038,     2.706975,    3.1141493,    -4.1420155,  -0.99003804,
    -1.7006547,  -8.495995,    -6.2332516,  -8.564608,    -7.7067156,  3.134292,     -0.33963704, 3.7133825,
    6.2897673,   -0.9730439,   -4.5531178,  -0.7827141,   2.581028,    7.953187,     6.305909,    -2.4009464,
    -3.7133813,  -2.690277,    3.9752584,   3.0376637,    -5.0019054,  -6.0262628,   0.7419828,   3.2228546,
    -0.32318294, -4.7031775,   -1.0777391,  -7.8937283,   -2.5363095,  4.257466,     -3.6471338,  5.237282,
    1.8462799,   0.5969913,    3.9643247,   -3.981004,    0.0663265,   0.82460785,   -2.7293837,  -1.5757694,
    0.55400586,  6.462948,     3.5353048,   2.9161394,    2.580977,    13.528652,    3.98995,     -1.632153,
    -3.240194,   3.900541,     -0.21140909, 2.8386583,    9.924921,    1.7748868,    -2.5982907,  5.174925,
    1.8638494,   1.6294506,    2.5033612,   2.8808913,    0.2832532,   -2.2669961,   -5.155612,   2.7401254,
    6.428844,    -2.8874602,   -0.45156026, 2.8010314,    1.7127267,   -6.3887377,   -1.0165125,  4.816684,
    -3.0209088,  -1.9152341,   -6.7044344,  -7.0160933,   -0.8859364,  2.3359919,    2.614932,    1.5376289,
    0.2540813,   0.56656873,   0.947714,    -3.2629232,   2.3573487,   7.069599,     -7.53059,    -5.4648676,
    -1.4810953,  0.27525342,   2.4626575,   -1.5132098,   -4.127886,   1.3913381,    1.090563,    -4.6527243,
    4.9518104,   -0.906865,    5.0196123,   1.055696,     -7.831962,   2.144308,     -1.838556,   -1.3607846,
    -2.1367745,  -4.8458967,   2.0994475,   2.6582882,    -2.158319,   0.8175374,    7.929186,    -0.9123031,
    5.690818,    -4.0453672,   -4.948562,   3.2541595,    0.9711809,   -1.2001665,   0.78384995,  1.3639677,
    -0.6874037,  0.9069457,    3.6966968,   -3.823165,    -1.826899,   2.3765814,    0.0534904,   8.726845,
    -0.18846548, -3.2959056,   1.5797036,   0.0014669895, -4.9724956,  -5.2561207,   5.819672,    -5.477039,
    3.3079143,   -0.033277154, 2.7245224,   -4.631716,    1.0122153,   -1.5371637,   -1.8553452,  -3.7143025,
    8.022276,    0.62215286,   3.8595328,   -3.060592,    4.2517557,   -0.075296044, 0.5221062,   0.6199312,
    1.9474881,   -1.3498385,   0.6838516,   2.4967105,    0.06516862,  -0.6287519,   -0.7507546,  6.147333,
    -3.149796,   3.1273334,    0.018394649, 0.8915896,    8.200176,    -1.7225304,   2.0177326,   -1.2988436,
    -0.13740933, -3.868376,    -0.06492156, 2.2702193,    -10.430931,  -7.2083035,   4.860276,    3.578821,
    -6.7857146,  3.5525331,    4.142806,    -0.3026886,   -1.20933,    2.6262493,    2.6222875,   6.941968,
    1.6663432,   -3.0459986,   6.198147,    -6.5455766,   -0.8200346,  -8.528335,    2.722542,    0.4080863,
    -2.993259,   -4.024056,    -1.999518,   3.2624865,    0.42962015,  -4.08082,     -0.39366418, 3.6101956,
    0.9608154,   -0.15634394,  -2.0926623,  1.6061159,    -1.5019901,  1.8686844,    -0.8275065,  2.9409513,
    -7.4440265,  -7.7664104,   0.8106141,   0.9343933,    6.078513,    -3.0837321,   -3.1975398,  1.8816166,
    0.16744804,  -4.573029,    -5.839288,   -0.7797469,   0.71803975,  0.41256714,   11.391333,   7.790516,
    1.9857845,   -2.0327086,   -0.5032053,  -2.5290394,   1.16115,     3.3385215,    7.5034156,   5.4487205,
    2.886569,    2.5460477,    -5.3722363,  5.1042805,    -0.9692185,  1.4824567,    -2.1692014,  -2.0888186,
    2.4214573,   0.78656745,   -0.3521694,  -1.3446121,   -6.659781,   -7.66657,     6.1127615,   -14.052498,
    -3.1808968,  -2.8461368,   -3.2059226,  -2.7757711,   -0.17827892, -8.695724,    -0.2887354,  2.312519,
    -0.4773283,  2.095835,     -3.293869,   -4.960386,    -0.9118179,  -0.2619573,   -0.92870337, -0.029317379,
    -2.5232022,  1.3327014,    3.1228013,   3.4733155,    1.4562413,   -2.5750513,   -1.6694541,  1.7559463,
    1.142877,    -1.3557005,   -2.30802,    -0.29746848,  2.6858592,   1.5424967,    -3.3826494,  -3.2559767,
    -0.2901088,  -0.83393717,  -0.06207335, 2.225967,     1.8832793,   -5.9567456,   4.7713566,   2.9260354,
    5.854274,    -1.2023156,   -2.0882115,  1.2139479,    -1.2005312,  0.4508332,    3.571826,    -4.5633574,
    -6.3648844,  -3.4183156,   2.7096481,   -3.659875,    -1.957063,   -0.5946456,   -0.76313734, -0.016180754,
    2.0194921,   -0.72149086,  -0.16249031, -2.5144238,   5.9847684,   -5.335026,    -1.0649127,  -3.176074,
    -0.3549943,  -6.501223,    1.4781482,   2.8698225,    0.3889513,   1.0389466,    2.6314335,   -2.6634102,
    5.950971,    -1.8160157,   6.9972243,   -2.4468954,   4.066836,    -6.923808,    2.4692469,   -2.1501422,
    -1.4999585,  -0.91028214,  4.634622,    4.132228,     -1.7976125,  0.59614825,   10.924917,   0.63333595,
    -1.2575581,  -2.6736045,   -8.180259,   5.0657587,    -3.065015,   -3.7651565,   -2.2837136,  -11.203299,
    8.331546,    -0.6740327,   5.5538063,   -2.0441968,   0.5072439,   2.630047,     4.323353,    -0.3627143};

static const std::vector<float> expected_idft2d_signal_size_results_1 = {
    0.35605296,    0.34874853,     -0.05017762,   0.019962106,   0.019251583,   0.13608973,    -0.014922479,
    0.04271815,    0.010398432,    0.025212493,   0.0796304,     0.11609333,    0.038195945,   0.044737395,
    0.03563052,    -0.026620463,   -0.01258824,   -0.12605664,   0.4239486,     0.41011444,    -0.061647452,
    0.00044958064, -0.06847432,    0.047905806,   0.062185094,   0.06219103,    0.058658686,   0.021337016,
    0.052691586,   0.08597042,     0.047829624,   -0.0079172375, 0.08269539,    -0.049927484,  0.004909183,
    -0.13700642,   0.34188202,     0.3957634,     -0.05613245,   0.002547276,   -0.12465724,   0.078391984,
    -0.024422595,  0.034788877,    -0.009830318,  0.008629954,   0.045807023,   0.0015473965,  0.08764248,
    0.026886743,   0.048212234,    -0.10233939,   -0.00421264,   -0.043676328,  0.39110252,    0.3343233,
    0.022979043,   -0.059183322,   -0.009643566,  0.13861291,    -0.0074428986, 0.080656774,   0.03761088,
    0.025686212,   -0.0036547848,  0.07562487,    0.107067585,   0.009425696,   -0.02972116,   -0.06653358,
    -0.014267657,  -0.15654333,    0.37992036,    0.3379705,     -0.05845514,   0.02229193,    -0.0664153,
    0.01909803,    -0.03232333,    0.008668827,   0.050338943,   0.030897861,   0.054762725,   0.03359122,
    0.071066484,   -0.00017446809, 0.02377109,    -0.11749724,   -0.005592843,  -0.077366814,  0.34532386,
    0.39782873,    -0.09539609,    -0.024212658,  -0.08361015,   0.028037548,   0.06844571,    0.08882976,
    -0.024732176,  0.059495773,    0.05300445,    0.030497046,   0.11334491,    -0.010473651,  0.07064287,
    0.03975949,    -0.07145635,    -0.045089867,  -0.07608327,   0.021705426,   0.07362527,    -0.041599143,
    -0.054502584,  -0.02229002,    0.02848531,    -0.065663904,  -0.035630893,  0.024472319,   -0.04410272,
    -0.0011307478, 0.0016347846,   -0.067072116,  0.04176219,    -0.057980362,  -0.015640466,  -0.011828707,
    -0.1550631,    0.10193367,     -0.023138944,  -0.020066634,  -0.014149333,  -0.098568134,  -0.0329638,
    -0.0011906491, -0.013988529,   0.011451623,   0.0027864417,  0.05226567,    0.023412103,   0.00374918,
    -0.056059547,  0.036750186,    -0.0046249777, 0.013725244,   -0.12065297,   0.07273325,    0.0042614806,
    0.046293974,   -0.023597008,   -0.057033,     -0.02165605,   0.051990334,   -0.033027273,  0.022346405,
    0.021624766,   0.017643817,    -0.059393693,  -0.011430228,  0.029003588,   0.05565236,    0.04070292,
    0.014539276,   -0.0682714,     0.06363358,    0.0029064484,  0.02769554,    -0.043081302,  0.024188243,
    -0.006092675,  -0.0041437447,  0.025472214,   0.039816964,   -0.044346157,  -0.060146388,  -0.020552356,
    0.075017385,   0.009120548,    0.0109255845,  0.068151034,   -0.023064155,  -0.121187955,  0.08082092,
    -0.043018054,  -0.02795456,    0.011519067,   -0.01887376,   0.03047559,    -0.06496317,   -0.019677995,
    0.004774744,   -0.021283617,   -0.009544955,  -0.027493106,  0.061126392,   0.0617069,     0.031720735,
    -0.025023961,  -0.075449966,   -0.14747246,   0.08565785,    0.039090306,   0.0038568007,  -0.048530135,
    -0.019006917,  -0.0034149592,  0.014780279,   0.024263024,   -0.00827444,   -0.030654684,  0.06013131,
    0.009361195,   0.04277391,     0.010400491,   0.054690998,   -0.0024832455, 0.008213211,   0.017274184,
    0.15904737,    0.058829676,    0.007716746,   -0.07915838,   0.08207793,    0.015012445,   0.01614127,
    -0.05450456,   -0.037097495,   -0.014707568,  -0.023024952,  0.05446966,    0.03376894,    0.049058244,
    -0.021138508,  0.04486409,     -0.054837674,  0.026894543,   0.13050343,    -0.006895314,  -0.006578704,
    -0.052172694,  0.024099395,    0.00078074407, -0.04804568,   -0.0020129893, 0.028822804,   -0.034898415,
    0.07210272,    -0.0097800465,  -0.051137194,  0.0023688707,  -0.033898205,  0.017479278,   -0.03854542,
    -0.036365107,  0.12087566,     0.024605379,   -0.07259585,   -0.027661474,  0.05020386,    -0.02654937,
    0.002109358,   0.0036475442,   0.026421547,   0.045489445,   0.027975991,   -0.046253216,  0.018246949,
    0.015081541,   -0.0035609512,  0.008914807,   0.022435194,   0.036099557,   0.123795256,   -0.019682797,
    0.0081835855,  0.01935591,     0.052482475,   0.016099557,   0.014442227,   -0.038650487,  0.014238742,
    0.030578831,   0.013300121,    0.009864425,   0.0598111,     0.046959084,   -0.042435378,  0.03186508,
    0.012782564,   0.018486261,    0.13155045,    0.031053057,   -0.02132215,   -0.023451997,  0.02523235,
    -0.02468166,   0.02662655,     -0.024940018,  -0.0108131105, -0.0042828214, -0.028867293,  -0.04255884,
    -0.024206718,  0.11273433,     -0.066240355,  0.0306156,     -0.024481589,  0.08500185,    0.16354619,
    -0.088644646,  -0.024336763,   -0.05789602,   6.998115e-05,  -0.043813612,  0.026284585,   0.0068632537,
    0.0004248606,  -0.05440126,    -0.06935504,   0.030324984,   0.0013237847,  -0.012008667,  0.02037527,
    -0.0009333448, -0.071275614,   0.1135454,     -0.08395175,   0.047969148,   0.055604927,   0.0189245,
    -0.03606843,   0.0006258389,   0.022486897,   0.03366298,    -0.031173265,  0.07229443,    -0.014791926,
    -0.0021743802, -0.012238768,   0.024743382,   0.0043328307,  -0.026065504,  -0.02907504,   0.13310593,
    0.033718396,   -0.0198984,     -0.025202168,  -0.002696252,  -0.030676337,  0.08850493,    0.032282893,
    -0.036327668,  -0.0015227132,  -0.017257914,  0.016897412,   0.046358045,   -0.011470813,  -0.022871837,
    -0.04586317,   -0.041991945,   -0.052629452,  0.08895524,    -0.024078613,  0.022111418,   0.05226525,
    -0.040480733,  0.008305485,    -0.03975329,   0.03907805,    -0.022432908,  -0.020524755,  -0.045322362,
    0.015144711,   0.0486015,      -0.062878706,  -0.018669453,  -0.0444292,    -0.026520465,  0.04212131,
    0.13955119,    0.037846,       -0.069351114,  -0.062554084,  -0.039485786,  0.06294745,    0.014536775,
    0.07071033,    0.0020498342,   -0.003970201,  -0.021590997,  -0.007638035,  -0.002107262,  -0.0009632535,
    -0.099295944,  0.04266729,     -0.010463878,  -0.036356337,  0.12848328,    0.00076921255, 0.029934544,
    0.062316064,   -0.036580022,   0.019605635,   0.019597089,   0.043524865,   0.02382575,    -0.035777867,
    0.07316227,    -0.023159584,   0.027851636,   0.020242432,   0.017187908,   0.038080517,   0.019080264,
    -0.015698584,  0.07482712,     0.077297814,   -0.0320183,    0.016496757,   -0.011164234,  -0.050667573,
    0.07790394,    0.008236007,    0.040544845,   0.021768719,   0.054870743,   -0.027450528,  -0.00052189163,
    0.066775165,   0.08357497,     -0.0057824906, -0.0033803424, 0.0009435627,  0.13810344,    -0.11135888,
    0.044955265,   0.033883553,    -0.017544864,  0.0026577704,  0.01076552,    -0.053345937,  0.0032790357,
    0.0026602354,  0.04282249,     -0.04054511,   0.029172834,   -0.04432436,   0.003725277,   -0.063187025,
    -0.031367887,  0.01675014,     0.06535116,    -0.118762106,  0.017369088,   0.035345916,   -0.029028835,
    -0.02536158,   -0.016617399,   -0.024160875,  -0.008315794,  -0.056838598,  0.017103666,   -0.0198483,
    0.017564027,   -0.05325589,    -0.024262477,  -0.034148794,  -0.027664784,  0.006907692,   0.11082979,
    -0.1492285,    0.056463942,    0.048155528,   0.013185428,   0.045351695,   0.0060724937,  0.0034848584,
    0.011533654,   0.052700933,    0.03514185,    -0.040892825,  -0.05964364,   -0.012769441,  0.0062652444,
    0.01570539,    -0.036600873,   -0.009507085,  0.056595936,   -0.14126535,   0.0154320365,  -0.034443073,
    0.051318757,   0.00806037,     0.0044386787,  -0.0044951793, 0.020322146,   -0.034489576,  0.03268032,
    -0.031927057,  -0.022262735,   0.020510584,   0.012746443,   -0.062045496,  -0.046817604,  0.033580557,
    0.050686605,   -0.089896716,   -0.04283018,   0.061590493,   -0.0042343163, 0.0010036799,  -0.07836513,
    -0.030185085,  0.013559818,    -0.031685624,  0.005115925,   -0.00680789,   -0.056500684,  -0.026287887,
    0.03808572,    0.020346357,    -0.088252544,  0.0040820944,  0.078687795,   -0.06582461,   -0.022862282,
    -0.028989157,  0.006826234,    0.055067796,   -0.013546096,  -0.011523222,  0.051062543,   0.020392258,
    0.029074619,   -0.020586526,   -0.051861648,  -0.07769388,   0.014526297,   0.049357798,   0.023303697,
    0.028278539};

static const std::vector<float> expected_idft2d_signal_size_results_2 = {
    0.5531721,     0.5459923,     0.53077924,    0.45617008,    0.62682444,    0.4433094,     0.5129144,
    0.50332,       0.5066039,     0.56471866,    0.47238868,    0.60442054,    0.6557239,     0.57886696,
    0.33827496,    0.4746061,     0.017551398,   0.09725249,    -0.07586281,   0.052343242,   -0.048901137,
    0.08368729,    0.0050727627,  -0.02984579,   0.07444677,    -0.044774584,  -0.019407284,  0.07174803,
    -0.071770966,  -0.020957068,  -0.016136957,  -0.04693111,   0.07191557,    0.08927296,    0.12441727,
    -0.092424,     -0.01244832,   -0.005490934,  -0.060569115,  -0.0150122,    -0.01933651,   0.13378121,
    0.021377748,   0.0051260735,  0.009875215,   -0.070481166,  -0.058303483,  -0.019897204,  -0.060325067,
    -0.028948829,  0.0055089346,  0.08009351,    0.087592974,   0.00092710555, -0.10650317,   0.0104219075,
    -0.06024557,   -0.06649129,   0.06629865,    -0.084809154,  -0.05111746,   -0.05951735,   -0.0359336,
    0.0362291,     -0.027899336,  -0.03341895,   0.007909834,   -0.17460391,   0.10611749,    -0.19431858,
    -0.078860685,  0.034915417,   0.022309989,   0.050277736,   -0.040946655,  -0.010924172,  -0.004921656,
    -0.019811844,  -0.054569744,  -0.006115485,  0.022424622,   0.055955783,   -0.03958866,   -0.0466471,
    0.0039734766,  -0.022997241,  0.034312718,   0.0489372,     0.0939427,     -0.02093338,   0.035248347,
    0.080642335,   0.0014786869,  -0.05117652,   -0.059213143,  -0.013066596,  -0.027215272,  0.004328589,
    0.009883429,   -0.053747736,  -0.0017252862, 0.04527414,    0.08669756,    0.019889932,   0.0016308328,
    0.0018331807,  -0.012666961,  0.037029736,   -0.035201322,  0.019991955,   -0.041291635,  0.01272665,
    0.020465642,   -0.06784999,   -0.029143205,  -0.037824143,  -0.025811557,  -0.035826147,  -0.04326462,
    -0.13283135,   0.026578283,   0.021154713,   -0.0041265786, -0.02021577,   0.12614238,    0.012778981,
    -0.123238616,  0.1492135,     0.16534506,    -0.04627829,   0.15229645,    -0.010591989,  -0.005314323,
    0.009578362,   -0.005720541,  -0.04372243,   -0.09534259,   -0.04803379,   0.0036502792,  0.062245272,
    0.028041288,   0.027935838,   0.0050014853,  -0.026800316,  0.056877077,   -0.08855667,   0.07485318,
    0.02165105,    -0.056792222,  -0.016549474,  -0.0066841,    -0.0007561048, 0.026854377,   -0.031430766,
    0.016276611,   0.09684428,    0.0077378256,  -0.0139363315, -0.034977213,  0.0026719074,  -0.0012400001,
    -0.06140579,   0.023358472,   0.04847327,    0.0026738036,  -0.034708705,  0.10151848,    -0.02879948,
    -0.036970478,  0.011776146,   0.09303403,    -0.017509384,  0.024898935,   -0.011924837,  0.03665485,
    0.044002272,   0.023127712,   -0.026069423,  0.022224244,   -0.023364877,  -0.0311188,    -0.007990571,
    0.056824386,   -0.058673047,  0.033350483,   -0.042721074,  -0.085414425,  0.026290055,   -0.0075933933,
    0.031163601,   0.03621967,    0.04487782,    0.0068412423,  -0.022031706,  -0.010444781,  0.13619763,
    -0.010095239,  0.0065898895,  0.025147736,   -0.08364924,   -0.06432001,   -0.07871698,   -0.07807772,
    0.13074839,    -0.0009094676, 0.07872045,    0.000616769,   -0.037552796,  -0.027768552,  -0.17855327,
    0.065752104,   0.084788814,   -0.008025914,  0.025628176,   0.10260405,    0.021223223,   -0.05137205,
    -0.01334117,   -0.018114587,  -0.1144638,    0.071344875,   0.12335855,    -0.03352583,   0.08413311,
    0.023557603,   0.058695108,   -0.03169672,   -0.015069341,  0.028463965,   -0.08081926,   0.023019042,
    0.07448329,    0.029195376,   -0.0031292937, 0.002979553,   -0.01043473,   0.020290421,   0.035948016,
    -0.013068214,  -0.14437002,   0.013984859,   -0.026999082,  0.072234645,   0.010123427,   -0.023727309,
    -0.119222626,  -0.120623514,  -0.101863846,  -0.018985119,  -0.12936139,   -0.11510106,   0.012450467,
    -0.12286782,   0.01851142,    0.0049175275,  0.012761181,   -0.043844607,  0.026183426,   0.0005732278,
    -0.004507631,  0.0037628661,  -0.02520454,   -0.058891308,  -0.031899434,  0.011502794,   -0.092772305,
    0.020212494,   -0.0156601,    -0.013763661,  0.0047018034,  -0.044270985,  -0.028510189,  0.044463784,
    -0.025488352,  -0.15637311,   0.023329355,   0.057855785,   0.05939971,    -0.025778957,  0.029527873,
    -0.11892979,   -0.030715317,  -0.04786273,   0.04135018,    -0.090043366,  0.0073567554,  0.053930283,
    0.043262,      -0.00688985,   -0.03498016,   0.08282997,    0.042204063,   0.06849226,    -0.049501877,
    -0.08085754,   -0.03302276,   0.05100158,    -0.05726403,   0.10648756,    -0.025262028,  -0.041605346,
    0.041689955,   0.08840984,    -0.004173999,  0.09688242,    0.024852356,   0.07658803,    0.048349112,
    0.06762494,    0.024731234,   0.068337604,   0.0049590766,  0.05193726,    -0.039766233,  -0.019074153,
    0.015802756,   -0.00874117,   -0.024977459,  0.000706623,   -0.028027793,  0.03868606,    -0.124708645,
    -0.041894287,  -0.0005582869, 0.0131178945,  0.019812506,   -0.015669182,  0.014672056,   -0.047444712,
    0.019266268,   0.0442128,     -0.09576817,   0.060629286,   0.016271904,   -0.059490584,  0.013682982,
    0.019443592,   0.03937396,    0.018371474,   0.01281172,    0.086036205,   -0.033354387,  0.05017164,
    -0.122697085,  0.02376157,    -0.05981153,   0.08532616,    -0.03525166,   -0.025003593,  -0.011591822,
    -0.020989612,  0.041963816,   0.031429403,   -0.007031359,  0.08429086,    -0.03850344,   0.035511702,
    -0.04003218,   0.004656136,   0.02909784,    -0.10617155,   -0.02204068,   -0.02427197,   0.020434562,
    0.09787702,    -0.050744973,  0.08300381,    0.0710342,     -0.031391703,  -0.029119622,  0.015069126,
    -0.0054880977, -0.024989244,  0.011621267,   0.021875506,   -0.06613677,   0.00016721214, 0.12602335,
    -0.05614292,   0.07607715,    0.030650944,   -0.09098706,   0.055347808,   0.0016167786};

static const std::vector<float> expected_idft2d_signal_size_results_3 = {
    0.51965636,   0.46109518,    -0.03224531,   0.122318566,  0.13996802,   0.04999275,    -0.09620099,  0.0069011925,
    0.664382,     0.58964956,    -0.031338193,  -0.14712875,  0.082166344,  0.07589669,    0.03586609,   0.0057471097,
    0.5606513,    0.5847055,     -0.1474433,    0.08164762,   -0.02039446,  -0.023951799,  0.0067006624, -0.023108874,
    0.6620194,    0.27644622,    0.06533013,    0.19844143,   0.050010286,  0.0615575,     -0.08088175,  0.019728234,
    0.54816973,   0.43787545,    -0.1545186,    -0.11976641,  0.06782712,   -0.050659865,  0.018574938,  -0.042057663,
    0.43190402,   0.5899364,     0.011021944,   -0.08617622,  -0.035539385, -0.009537175,  0.1456467,    0.21298885,
    0.20042025,   -0.06154151,   -0.0033541308, 0.027834859,  -0.10164561,  -0.10600591,   0.1779365,    -0.053386994,
    -0.05831848,  -0.05998933,   -0.08526638,   -0.074515656, 0.022936836,  0.07827924,    -0.10342336,  -0.08194167,
    -0.12801322,  0.06166912,    0.025966948,   0.12110167,   0.023929987,  0.034466177,   -0.039979175, -0.016688354,
    0.07164687,   0.12428331,    0.06549715,    0.03601643,   0.020718133,  -0.0071917605, 0.009838374,  0.0040146103,
    -0.013233974, -0.030871946,  0.047197368,   0.0123527115, -0.056864895, -0.046955965,  -0.004868932, -0.17846301,
    0.06855715,   0.027679857,   -0.1500722,    -0.056670498, -0.14638877,  0.023189232,   -0.029592723, 0.052305903,
    0.12751302,   -0.2093656,    -0.07194336,   -0.008120443, 0.07343415,   -0.031670257,  0.046054944,  -0.057585027,
    0.0018484419, 0.00040499022, 0.053662326,   -0.06284372,  -0.13057502,  -0.07417334,   -0.042746447, -0.013483183,
    0.007553713,  -0.15385008,   -0.08656611,   0.0701028,    -0.067524604, 0.026846979,   0.07731889,   0.002451243,
    -0.068792164, 0.020291083,   0.014570983,   0.0130328,    -0.0557093,   -0.08878534,   -0.1661258,   -0.009066362,
    0.03486307,   0.08238177,    -0.12802714,   0.093373135,  0.08243198,   -0.02568413,   0.034321107,  0.12715237,
    -0.08129562,  0.1487139,     0.10971667,    -0.045825522, 0.12990095,   0.041814834,   0.03660839,   0.012229371};

static const std::vector<float> expected_idft2d_signal_size_results_4 = {
    0.50069946,    0.4904276,     0.009259958,   0.01438915,    -0.008930033,  0.1685353,     0.027075492,
    0.05019591,    0.123547375,   0.010753404,   -0.006528951,  0.06896549,    -0.018036481,  0.0016257297,
    -0.05024746,   -0.078786865,  0.5961777,     0.57672334,    0.011319853,   0.0018033399,  -0.029748872,
    -0.03491377,   0.03782171,    0.05982995,    0.11184341,    0.05324241,    -0.0040071756, 0.012312844,
    0.049088746,   -0.044715118,  -0.01899992,   -0.082886636,  0.48077154,    0.5565423,     0.014818694,
    0.026350802,   -0.12203966,   0.0050221197,  -0.035045236,  0.007882977,   0.012806054,   -0.04235148,
    0.06831992,    -0.027629035,  -0.04750082,   -0.057690993,  0.008230133,   0.035048254,   0.5499879,
    0.4701421,     0.14410123,    -0.058395956,  -0.013658326,  0.11536215,    -0.018243652,  0.04704798,
    0.069733724,   0.0700257,     0.019510295,   -0.06851628,   -0.09657783,   0.023751881,   -0.03731589,
    -0.12183073,   0.534263,      0.47527096,    -0.0118961595, 0.038099527,   -0.023198979,  -0.016842058,
    0.020281151,   -0.06888728,   0.04158451,    -0.006500121,  0.017227922,   -0.023705663,  -0.065157875,
    -0.061886493,  0.00823769,    -0.013699137,  0.48561165,    0.55944663,    -0.029265663,  -0.028232932,
    -0.031501852,  -0.056778878,  0.026412688,   0.12899265,    0.006460272,   -0.014392011,  0.04032879,
    -0.026880052,  0.0731035,     0.09133217,    -0.10169062,   0.05235266,    0.06892046,    -0.0295484,
    0.105304524,   0.0118386075,  0.0025054235,  -0.025319275,  0.09081747,    0.036444083,   -0.014070861,
    -0.05731819,   -0.0084318025, -0.12696962,   0.03937552,    -0.074440725,  -0.047060557,  -0.020518059,
    -0.052758366,  0.022431955,   -0.03708367,   -0.02261317,   0.06566422,    -0.022002619,  0.019627463,
    -0.031626925,  0.04216373,    -0.012684815,  -0.054740556,  -0.0037655067, -0.10157557,   0.039671652,
    -0.0487277,    -0.0116044395, -0.02558843,   0.036774725,   -0.06290054,   0.041731425,   0.046522886,
    0.007584708,   -0.063871264,  0.014685718,   -0.026041836,  -0.028664894,  -0.05809094,   0.028512653,
    0.029267922,   0.043192253,   0.037230983,   0.019329412,   0.049078666,   0.019939065,   -0.016842797,
    0.012881201,   -0.026447453,  -0.0014885329, 0.018214535,   0.004656654,   -0.09413797,   0.026110932,
    0.061685666,   0.056731943,   -0.019016284,  0.0042851716,  0.06388497,    -0.043995842,  -0.01923573,
    0.014247827,   -0.027373053,  -0.067749165,  -0.020197151,  -0.0062670596, 0.044943634,   0.04798174,
    -0.0048591197, 0.0052249283,  0.035372503,   0.048372734,   0.010621414,   -0.056204394,  -0.044176046,
    -0.11560526,   -0.035333157,  0.0016496852,  -0.013274903,  0.078784436,   -0.022540363,  -0.032977562,
    -0.012144292,  0.03639986,    0.018095985,   0.10897899,    -0.004816832,  -0.0037983656, 0.012832675,
    0.035309885,   -0.06408509,   0.017382117,   -0.08315392,   -0.033107996,  0.08171202,    0.07782332,
    -0.06865839,   -0.10764925,   -0.054046266,  -0.005835831,  0.012306612,   -0.06220748,   0.07638017,
    0.013183124,   0.041668497,   -0.055150427,  0.011025649,   0.0047506727,  0.010130197,   0.019823164,
    -0.03397094,   0.002564197,   -0.056729004,  -0.09857107,   0.09008947,    0.031346098,   -0.0024604239,
    -0.0047197156, -0.0550084,    -0.110036984,  -0.026545763,  0.017181754,   -0.018559802,  0.007102821,
    -0.062645376,  -0.038759947,  0.11738676,    0.057442024,   -0.0405657,    0.00024135411, -0.017878262,
    0.042187072,   -0.017206412,  -0.004294824,  -0.15039666,   0.017837571,   0.010826452,   0.017910408,
    0.022387126,   0.038914144,   0.039849013,   0.0025931299,  -0.062299855,  -0.031667564,  -0.06573889,
    0.067470476,   -0.024389353,  0.09938391,    0.026338965,   0.046804663,   0.018697795,   0.06765098,
    0.06032055,    0.029475141,   -0.036904067,  0.012176782,   -0.019729197,  -0.0006175935, 0.05483261,
    0.042380556,   -0.04253985,   0.0012245048,  -0.020328768,  -0.013062598,  0.0053691603,  -0.065394655,
    0.05558199,    -0.021914857,  0.1706096,     -0.03785784,   -0.020459495,  -0.01513383,   0.017118126,
    0.14779843,    -0.005776977,  -0.03670439,   -0.051734768,  -0.08797553,   0.024524372,   -0.083074294,
    -0.028641924,  -0.031074718,  0.11779925,    -0.0073087253, -0.0006238073, -0.025388103,  0.043085426,
    -0.019948414,  0.13103837,    -0.05180665,   0.04730244,    0.037148323,   -0.041258633,  0.024768397,
    0.025847614,   -0.06161034,   -0.010785304,  -0.03626065,   -0.016979318,  -0.02865681,   -0.01828878,
    -0.019304248,  0.03128355,    0.0041085277,  0.0024671853,  0.008217737,   0.03448409,    0.030935789,
    -0.022427216,  -0.0295653,    -0.073335916,  -0.0037255269, 0.00038855523, -0.0448144,    -0.029116416,
    -0.047692195,  -0.011189029,  -0.022148391,  0.01491251,    -0.01919859,   0.0401923,     -0.08648369,
    0.058548566,   0.02578877,    0.00094457157, 0.019820662,   0.0145471115,  -0.04303372,   -0.052616134,
    0.019346833,   0.052684337,   0.060835492,   0.031265024,   -0.045283258,  -0.052162454,  0.016602138,
    -0.014453076,  -0.022049919,  0.043205604,   -0.069325164,  0.020091549,   -0.007931273,  -0.04543579,
    -0.03203051,   -0.008483101,  -0.034514427,  0.04786063,    0.007603269,   -0.097250104,  -0.079659835,
    0.032755584,   0.036081,      0.018139064,   0.029964916,   -0.06415692,   0.0974453,     -0.002073994,
    0.029044904,   -0.08739063,   -0.06471936,   0.0054590777,  0.0011593401,  -0.039340377,  0.064394005,
    0.028959818,   -0.053989142,  -0.06349799,   0.07390169,    0.02351752,    0.03192448,    -0.037667163,
    -0.07394214,   0.017038213,   0.036862355,   -0.043745972,  -0.014133457,  -0.028366454,  0.03523228,
    -0.04687486,   0.0940376,     0.120025754,   0.020736795,   0.022597576,   -0.023962533};

static const std::vector<float> expected_idft2d_signal_size_results_5 = {
    0.43990576,    0.37177825,    0.018297018,   0.13355713,    0.09609352,    0.04071089,   -0.092824854,
    0.03483837,    0.52699745,    0.4414857,     0.017332105,   -0.043757387,  0.06474239,   0.06351439,
    -0.0062755626, -0.028125575,  0.39462435,    0.4589463,     -0.100476444,  0.031446725,  0.0050833533,
    -0.07491388,   0.0050572394,  -0.012939147,  0.5018034,     0.2513064,     0.06601329,   0.12563014,
    0.01634291,    0.012481438,   -0.09012963,   -0.0073484736, 0.45544818,    0.3760468,    -0.06427583,
    -0.043393508,  0.057562195,   -0.019032322,  -0.031661525,  -0.056141183,  0.34964913,   0.43898678,
    0.03130693,    -0.058825802,  -0.076260805,  0.013013445,   0.070871785,   0.17149778,   -0.06417048,
    -0.06922141,   -0.040472697,  -0.062998466,  -0.058540687,  0.031437594,   0.08273152,   -0.12060501,
    -0.2061106,    0.118685186,   -0.0009581372, -0.08492219,   -0.0041653924, 0.0067129806, -0.06255558,
    0.05957415,    -0.118542455,  0.14462362,    -0.009236974,  0.022580916,   -0.03849992,  0.03493386,
    0.0035450687,  0.010597763,   -0.0013405303, 0.11584599,    -0.03715597,   0.010403095,  -0.020171534,
    0.021085449,   -0.018025603,  0.006588485,   -0.1273406,    0.058172766,   0.0065336376, -0.09943622,
    -0.0022615304, 0.00024407124, -0.030914608,  0.022675768,   -0.15166818,   0.07935919,   -0.040845793,
    0.016231818,   0.039729055,   0.07636168,    0.0033444613,  0.07087032,    0.117220305,  0.15729488,
    -0.048147924,  0.094027564,   -0.023140147,  -0.07949994,   0.045205563,   -0.00916888,  0.0492486,
    0.115183815,   -0.08267819,   -0.029814515,  -0.014258375,  0.08933942,    -0.010548074, -0.09738556,
    -0.09092414,   0.11030196,    0.0021819957,  0.08301844,    0.0689266,     0.030335644,  -0.019274916,
    -0.0315443,    0.050106198,   0.19658162,    0.0503511,     0.07378791,    0.027816363,  -0.0067463894,
    0.0042154863,  -0.0070224535, 0.030608267,   0.1334948,     -0.0040946133, 0.070922405,  -0.02647118,
    -0.040317588,  0.072931446,   -0.15662149,   0.086943075,   0.18035986,    -0.064264745, -0.06842194,
    -0.11555961,   -0.08764592,   -0.04262618,   0.022765271,   0.1726031,     -0.11350446,  0.01339009,
    0.00404248,    0.07330416,    -0.04052531,   0.024228431,   0.025112806,   0.15811935,   0.0038504712,
    0.08746172,    -0.09458299,   -0.072628036,  -0.0015422329, -0.04602814,   0.007808782,  0.095344745,
    -0.025132503,  -0.089144886,  0.061655734,   -0.087049335,  -0.0064667296, 0.047338445,  -0.025052967,
    0.10809028,    0.038190633,   0.010420183,   0.014206182,   -0.075355306,  -0.014443267, -0.12931237,
    0.06473562,    0.21191315,    0.031243857,   -0.01827092,   0.043444555,   0.071482405,  -0.053060792,
    0.03741809,    0.088275075,   0.095881015,   0.13641034,    0.06575319,    -0.10507756,  0.07346845,
    0.022813302,   0.049534187,   0.0534714,     0.1820309,     -0.15615918,   -0.05060928,  -0.026595712,
    0.024039716,   -0.039806653,  0.06844979,    -0.034248125,  0.07965717,    -0.14914,     -0.08409975,
    -0.031411037,  0.00083757995, -0.078021966,  0.01510364,    -0.031549543,  0.15968925,   -0.19621487,
    -0.011366145,  0.07415028,    -0.012449766,  0.053472463,   0.007374543,   0.02159267,   0.0062146843,
    -0.18090405,   0.05576966,    0.02346333,    0.066386685,   -0.04679683,   -0.003917059, -0.042276695,
    -0.0008301586, -0.10957299,   -0.15524064,   0.014422208,   -0.0069176825, -0.011133326, 0.00025371276,
    0.008443515,   0.038360506,   -0.06878601,   -0.021283172,  0.027421257,   0.02659573,   0.030924385,
    0.07153812,    -0.04108064};

static const std::vector<float> expected_idft3d_signal_size_results = {
    0.47360885,    0.43109864,    -0.015375684,   0.037675954,   0.032973632,   -0.02922261,   0.011034667,
    0.010564784,   -0.014908149,  0.017389147,    -0.09910002,   0.06296675,    -0.03663863,   -0.018516757,
    -0.0011850556, 0.008715436,   -0.016253775,   0.020636003,   0.047444995,   -0.0024833402, -0.054806184,
    0.063673705,   0.08815336,    -0.019982928,   0.023201711,   -0.0112207085, -0.015865667,  0.032094162,
    -0.04799268,   -0.0009248348, -0.0005214396,  0.11000099,    -0.009374426,  0.07988078,    -0.03512358,
    0.027058495,   0.01562349,    0.010030186,    -0.0323284,    -0.015812825,  0.06827773,    0.017847355,
    -0.0024901072, -0.024341352,  -0.0009585398,  -0.009607499,  0.015930153,   0.011760445,   0.034274444,
    0.027811723,   0.12735432,    -0.09866718,    -0.004353502,  0.05861713,    -0.0083505465, -0.056518648,
    -0.026644187,  0.019356728,   -0.012300177,   -0.039047725,  0.006673645,   -0.11678744,   0.020548198,
    -0.016014362,  -0.0039747558, 0.010135546,    0.020794585,   0.04208699,    -0.016170124,  5.5018212e-05,
    0.007777319,   0.008232947,   -0.01095827,    0.018842613,   -0.010135147,  0.0097200265,  -0.01967326,
    -0.026630066,  0.0063641453,  -0.0345983,     0.010849577,   -0.04473317,   -0.011817071,  0.02476529,
    0.0155312605,  -0.009187386,  -0.036735293,   -0.01913485,   0.0043119635,  0.018222943,   0.03163258,
    -0.022928724,  0.01679045,    0.00042682493,  0.02686337,    0.026272336,   0.05542246,    -0.070824265,
    0.049857102,   -0.057350874,  0.004540915,    0.024931736,   -0.0066266567, -0.0060906713, -0.018460626,
    0.030870803,   0.02064466,    -0.027539674,   -0.0001978664, -0.009443704,  0.032834202,   -0.018512815,
    0.0006455081,  0.05246358,    -0.00452247,    -0.002136842,  0.006376774,   -0.04188078,   0.024743602,
    0.027263625,   0.009806405,   -0.022745509,   -0.0042487574, 0.015461739,   -0.020853115,  0.012601041,
    0.01992702,    0.008631575,   0.028762273,    -0.014992784,  0.009239552,   0.008044719,   -0.005379937,
    -0.0020708065, 0.018219728,   -0.012548579,   0.03744698,    -0.03307585,   -0.0156224305, -0.003425684,
    0.009117395,   -0.035289288,  -0.02348312,    -0.0032391632, -0.03008389,   -0.0019097558, 0.018912613,
    0.01636466,    0.0156074455,  0.0038367372,   0.05714075,    -0.04073001,   -0.008098723,  -0.017560972,
    0.005987327,   -0.019382497,  0.029533077,    -0.039648466,  0.025838673,   -0.03902175,   0.009098698,
    -0.0016117467, 0.0051164175,  0.0037725042,   -0.014878556,  -0.028334552,  -0.005686572,  0.013636959,
    -0.03393201,   0.012674245,   -0.020131484,   -0.06613781,   0.031012673,   0.03862036,    0.009845047,
    -0.007944993,  0.008455606,   0.017170515,    0.063504405,   -0.023375345,  0.046734862,   -0.009648957,
    -0.037776463,  -0.015671387,  0.026074437,    -0.021695163,  -0.0055521606, 0.0117665045,  -0.0033777568,
    0.032647174,   0.031938843,   -0.03217729,    -0.027973937,  0.023887817,   -0.0062627294, -0.014567699,
    0.023382662,   -0.013312814,  -0.00063125056, -0.030556878,  -0.01716187,   -0.035170853,  0.016255094,
    -0.0032832257, 0.028565489,   0.01738093,     -0.046995874,  -0.00492523,   -0.0050363657, 0.006696142};

static const std::vector<float> expected_idft1d_signal_size_results = {
    0.45077854,   0.42283636,    0.5254696,     0.49784923,    0.5858539,     0.38200417,    0.46620184,
    0.34846103,   0.42421392,    0.36906072,    0.41839844,    0.64023143,    0.49543527,    0.61651653,
    0.33086562,   0.38194793,    -0.013190938,  -0.11509278,   5.4674492e-05, 0.1155831,     -0.059459202,
    0.1080654,    -0.038735352,  -0.13293009,   -0.044217605,  -0.08810778,   -0.1887316,    -0.04305593,
    0.0064542103, 0.02646665,    -0.19942014,   0.16640756,    0.13297448,    -0.060627896,  -0.0105416095,
    -0.08473419,  -0.04953826,   -0.0053528296, -0.062126894,  0.15922469,    -0.0811423,    0.1871564,
    -0.06467852,  0.019565826,   -0.06630114,   0.05153398,    -0.2517371,    0.012869691,   0.030862298,
    0.18221883,   0.18004268,    0.115521945,   -0.006216203,  -0.04221012,   -0.099812254,  0.10728653,
    -0.18543659,  -0.018842628,  0.03197636,    0.13652334,    -0.063658535,  0.2630217,     0.113256425,
    0.10151879,   0.20121424,    0.038065623,   0.04212692,    0.15088516,    0.11653197,    -0.047717404,
    0.17632447,   -0.16700539,   0.20303649,    -0.025800077,  0.14718783,    0.111731626,   0.083313815,
    -0.09204115,  0.03319706,    -0.025887946,  0.12955333,    -0.1840305,    0.30852506,    -0.033531368,
    0.3569616,    -0.19672023,   0.003184589,   -0.15916768,   0.047273237,   -0.1945002,    0.24541551,
    -0.013310499, 0.19067605,    -0.043925222,  0.019669108,   0.006493839,   0.047401913,   -0.1429027,
    -0.06374044,  -0.24836907,   -0.0050352793, -0.13352643,   0.04469287,    -0.11333085,   -0.029562438,
    -0.08143895,  -0.1432822,    -0.0007748668, 0.015231887,   -0.2503828,    0.062242992,   0.033986002,
    0.50197685,   0.5042996,     0.38166192,    0.33678776,    0.6271677,     0.4345175,     0.4634395,
    0.35949785,   0.5400181,     0.5938341,     0.5252213,     0.50813717,    0.5442304,     0.42320782,
    0.30390763,   0.5501661,     -0.04264457,   0.27560052,    -0.20748335,   0.12261487,    -0.14888233,
    0.2118158,    -0.27173674,   0.14285998,    -0.0120631205, -0.023952818,  -0.06255174,   0.17351858,
    -0.27042615,  -0.14776227,   0.027333874,   -0.08371357,   -0.18530807,   -0.06548692,   -0.08052732,
    -0.22452545,  -0.012073594,  0.1693748,     -0.13727319,   0.054494813,   -0.052982204,  0.34828788,
    0.058887307,  -0.0070416844, 0.035075426,   0.058569312,   -0.13160224,   0.2115795,     -0.060909573,
    0.30598193,   0.13896666,    0.104170956,   0.15620923,    0.04350682,    0.17609014,    -0.043268614,
    0.020355552,  0.13579452,    0.036094327,   0.029404454,   0.119380735,   -0.017330358,  -0.013081285,
    0.16276608,   0.08908949,    -0.013249721,  0.030841062,   -0.09900363,   -0.104013704,  0.14226541,
    -0.022373218, 0.13952398,    0.110540666,   0.050296895,   0.19052894,    0.30660385,    0.23429887,
    -0.07728504,  0.06723292,    0.053966362,   0.10566526,    -0.09358089,   0.12995106,    -0.09512488,
    0.22091591,   0.006204946,   0.021672454,   -0.08378312,   0.09504456,    0.0060912203,  0.21140935,
    -0.170751,    0.048105046,   -0.12022613,   0.031716354,   0.010118494,   -0.0024824888, -0.10846503,
    0.105488904,  -0.12306744,   0.0021025285,  -0.2050574,    0.059306182,   -0.06526719,   0.0758878,
    -0.12858643,  0.039985638,   -0.09187355,   0.15090118,    -0.06106589,   -0.009392858,  -0.14116207,
    0.50924444,   0.47538188,    0.36653236,    0.5176384,     0.4713962,     0.38925174,    0.4561904,
    0.37097445,   0.33398703,    0.46408504,    0.25756317,    0.62005824,    0.6271038,     0.51077616,
    0.2500916,    0.36728626,    -0.26136193,   0.053446535,   -0.10429463,   0.07333,       -0.125614,
    0.13846059,   -0.034961026,  0.06588599,    -0.12191007,   -0.0030780777, -0.1388123,    -0.063579924,
    -0.20671263,  0.12017858,    -0.035247978,  -0.13096417,   0.030405179,   0.31703302,    -0.18467914,
    -0.027100164, -0.10349936,   -0.03151396,   -0.14566809,   0.1760046,     0.119887486,   0.042870738,
    -0.10979464,  0.07983213,    -0.21066193,   0.058638398,   -0.14007244,   -0.025705764,  -0.15512581,
    0.14527217,   -0.008849559,  0.103681795,   0.05173718,    0.022327188,   -0.11486209,   0.33013803,
    -0.13087325,  0.15843569,    -0.13177976,   0.15052482,    -0.024118312,  0.19754727,    0.043348696,
    0.048077006,  0.05479733,    0.045596164,   -0.061486844,  0.13432872,    0.09805413,    -0.03079012,
    0.018617248,  0.057953924,   0.07086571,    -0.107639074,  0.10912783,    -0.027383184,  0.086062506,
    0.021643614,  0.0033594542,  0.1245168,     -0.008100906,  -0.071167745,  0.15794107,    -0.11394106,
    0.20086603,   -0.14925118,   -0.023519892,  -0.17317979,   0.16438125,    0.055674464,   0.043398805,
    -0.14720002,  -0.016775567,  -0.09511416,   0.024722071,   -0.21830398,   0.038695235,   0.009794927,
    -0.16068184,  -0.18155417,   0.09074758,    -0.060050044,  -0.14183791,   0.050801322,   0.039122116,
    -0.22079241,  0.122001916,   -0.04277471,   -0.10941203,   -0.13381958,   -0.08884025,   -0.1331459,
    0.43459043,   0.46945596,    0.5461507,     0.21173653,    0.46469456,    0.31414452,    0.372732,
    0.64673513,   0.43870863,    0.50919855,    0.4184354,     0.3038723,     0.5814269,     0.43418613,
    0.27493507,   0.32782063,    0.05329496,    0.13938421,    -0.14763448,   -0.020887244,  -0.06212297,
    0.0026264533, -0.16405852,   -0.15793757,   -0.10529227,   -0.017391797,  -0.010122827,  0.16707306,
    -0.24121732,  0.058287155,   -0.2669763,    -0.13610138,   -0.058303684,  0.21642019,    0.18846658,
    0.074460745,  0.025465574,   0.105897866,   -0.120273784,  0.12273249,    0.0060386043,  0.22629398,
    -0.18080746,  0.12264217,    -0.09611444,   -0.039596815,  0.13664094,    0.12015629,    -0.02670856,
    0.11202149,   0.062823564,   0.16666472,    0.02688467,    -0.0496769,    0.1111543,     0.20479687,
    0.09844535,   0.22211954,    -0.030972388,  -0.013431635,  0.075517,      -0.01832214,   -0.067268424,
    0.009361826,  0.066086665,   0.114655524,   -0.093547,     0.0037781172,  0.013148758,   0.0049218494,
    0.14736433,   -0.029691944,  0.19633494,    -0.0039153695, 0.24658194,    0.0027496407,  0.085013606,
    0.02836323,   -0.023116423,  0.031402636,   0.25721642,    -0.041657127,  0.08742507,    -0.26150823,
    0.21623416,   -0.10428249,   0.030336773,   0.033717245,   -0.052863047,  -0.05433597,   -0.0014412233,
    -0.05366828,  0.09244677,    -0.13540235,   0.049008,      0.032415926,   -0.012353104,  -0.026780326,
    0.08365354,   -0.11595764,   0.0041189813,  -0.19876777,   0.05539961,    -0.03458054,   0.3030866,
    0.06549558,   0.100598566,   -0.0328084,    -0.008204469,  -0.20897065,   -0.08830016,   -0.15857375};

template <class T>
static std::vector<T> convert(const std::vector<float>& v) {
    if (v.empty()) {
        return std::vector<T>();
    }

    size_t num_of_elems = v.size();
    std::vector<T> converted(num_of_elems);
    for (size_t i = 0; i < num_of_elems; ++i) {
        converted[i] = static_cast<T>(v[i]);
    }
    return converted;
}

template <class T>
static std::vector<T> convert(const std::vector<float16>& v) {
    if (v.empty()) {
        return std::vector<T>();
    }

    size_t num_of_elems = v.size();
    std::vector<T> converted(num_of_elems);
    for (size_t i = 0; i < num_of_elems; ++i) {
        converted[i] = static_cast<T>(v[i]);
    }
    return converted;
}

template <class T>
static std::vector<T> convert(const std::vector<bfloat16>& v) {
    if (v.empty()) {
        return std::vector<T>();
    }

    size_t num_of_elems = v.size();
    std::vector<T> converted(num_of_elems);
    for (size_t i = 0; i < num_of_elems; ++i) {
        converted[i] = static_cast<T>(v[i]);
    }
    return converted;
}

template <element::Type_t ET>
std::vector<IDFTParams> generateParamsForIDFT() {
    std::vector<IDFTParams> params{
        // idft1d_eval_1
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                  ET,
                  ET,
                  idft1d_input_data_1,
                  data_1,
                  op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{1}, {2}),
                  NULL),
        // idft1d_eval
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   idft1d_input_data,
                   expected_result,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{1}, {2}),
                   NULL),
        // idft1d_eval_i32
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   idft1d_input_data,
                   expected_result,
                   op::v0::Constant::create<int64_t>(element::Type_t::i32, Shape{1}, {2}),
                   NULL),
        // idft2d_eval_1
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   idft2d_input_data_1,
                   data_1,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {1, 2}),
                   NULL),
        // idft2d_eval
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   idft2d_input_data,
                   expected_result,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {1, 2}),
                   NULL),
        // idft2d_eval_i32
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   idft2d_input_data,
                   expected_result,
                   op::v0::Constant::create<int64_t>(element::Type_t::i32, Shape{2}, {1, 2}),
                   NULL),
        // idft3d_eval_1
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   idft3d_input_data_1,
                   data_1,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{3}, {0, 1, 2}),
                   NULL),
        // idft3d_eval
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   idft3d_input_data,
                   expected_result,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{3}, {0, 1, 2}),
                   NULL),
        // idft3d_eval_i32
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   idft3d_input_data,
                   expected_result,
                   op::v0::Constant::create<int64_t>(element::Type_t::i32, Shape{3}, {0, 1, 2}),
                   NULL),
        // idft2d_signal_size_eval_1
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft2d_signal_size_results_1,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {0, 2}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {5, 9})),
        // idft2d_signal_size_eval_2
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft2d_signal_size_results_2,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {0, 1}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {4, 6})),
        // idft2d_signal_size_eval_3
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft2d_signal_size_results_3,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {0, 2}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {3, 4})),
        // idft2d_signal_size_eval_4
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft2d_signal_size_results_4,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {0, 2}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {4, 8})),
        // idft2d_signal_size_eval_5
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft2d_signal_size_results_5,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {0, 2}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {5, 4})),
        // idft3d_signal_size_eval
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft3d_signal_size_results,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{3}, {0, 1, 2}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{3}, {3, 7, 5})),
        // idft1d_signal_size_eval
        IDFTParams(Shape{4, 6, 8, 2},
                   Shape{4, 6, 8, 2},
                   ET,
                   ET,
                   data_1,
                   expected_idft1d_signal_size_results,
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{1}, {-2}),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{1}, {7})),
    };

    return params;
}

template <element::Type_t ET>
std::vector<IDFTParams> generateParamsForIDFT_float16() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<IDFTParams> params{
        // idft1d_eval_float16
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   convert<T>(idft1d_input_data),
                   convert<T>(expected_float16_idft1d_result),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{1}, {2}),
                   NULL),
        // idft2d_eval_float16
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   convert<T>(idft2d_input_data),
                   convert<T>(expected_float16_idft2d_result),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {1, 2}),
                   NULL),
        // idft3d_eval_float16
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   convert<T>(idft3d_input_data),
                   convert<T>(expected_float16_idft3d_result),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{3}, {0, 1, 2}),
                   NULL),
    };

    return params;
}

template <element::Type_t ET>
std::vector<IDFTParams> generateParamsForIDFT_bfloat16() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<IDFTParams> params{
        // dft1d_eval_bfloat16
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   convert<T>(idft1d_input_data),
                   convert<T>(expected_bfloat_idft1d_result),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{1}, {2}),
                   NULL),
        // idft2d_eval_bfloat16
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   convert<T>(idft2d_input_data),
                   convert<T>(expected_bfloat_idft2d_result),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{2}, {1, 2}),
                   NULL),
        // idft3d_eval_bfloat16
        IDFTParams(Shape{2, 10, 10, 2},
                   Shape{2, 10, 10, 2},
                   ET,
                   ET,
                   convert<T>(idft3d_input_data),
                   convert<T>(expected_bfloat_idft3d_result),
                   op::v0::Constant::create<int64_t>(element::Type_t::i64, Shape{3}, {0, 1, 2}),
                   NULL),
    };

    return params;
}

std::vector<IDFTParams> generateCombinedParamsForIDFT() {
    const std::vector<std::vector<IDFTParams>> allTypeParams{
        generateParamsForIDFT<element::Type_t::f32>(),
        generateParamsForIDFT_float16<element::Type_t::f16>(),
        generateParamsForIDFT_bfloat16<element::Type_t::bf16>(),
    };

    std::vector<IDFTParams> combinedParams;

    for (const auto& params : allTypeParams) {
        combinedParams.insert(combinedParams.end(), params.begin(), params.end());
    }

    return combinedParams;
}

INSTANTIATE_TEST_SUITE_P(
    smoke_IDFT_With_Hardcoded_Refs,
    ReferenceIDFTLayerTest,
    ::testing::ValuesIn(generateCombinedParamsForIDFT()),
    ReferenceIDFTLayerTest::getTestCaseName);

}  // namespace
