/*******************************************************************************
* Copyright 2016-2021 Intel Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#include <assert.h>

#include <initializer_list>

#include "c_types_map.hpp"
#include "memory_desc_wrapper.hpp"
#include "type_helpers.hpp"
#include "utils.hpp"

namespace dnnl {
namespace impl {

template<typename T>
static status_t fill_blocked_impl(memory_desc_t &md, T&& perm, T&& inner_blks, T&& inner_idxs) {
    const bool ok = true && perm.size() == (size_t)md.ndims
                    && inner_blks.size() == inner_idxs.size();
    if (!ok) return status::invalid_arguments;

    md.offset0 = 0;

    blocking_desc_t &blk = md.format_desc.blocking;

    dim_t block_size = 1;
    dims_t blocks = {0};
    utils::array_set(blocks, 1, md.ndims);

    blk.inner_nblks = (int)inner_blks.size();

    int iblk = 0;
    for (const auto &b : inner_idxs)
        blk.inner_idxs[iblk++] = b;

    iblk = 0;
    for (const auto &b : inner_blks) {
        int dim = blk.inner_idxs[iblk];
        block_size *= b;
        blocks[dim] *= b;
        blk.inner_blks[iblk++] = b;
    }

    utils::array_set(md.padded_offsets, 0, md.ndims);
    for (int d = 0; d < md.ndims; ++d)
        md.padded_dims[d] = md.dims[d] == DNNL_RUNTIME_DIM_VAL
                ? DNNL_RUNTIME_DIM_VAL
                : utils::rnd_up(md.dims[d], blocks[d]);

    // setting the strides
    {
        dim_t stride = block_size;
        auto iter_d = perm.end(); // reverse iterator over perm
        do {
            const int d = *(--iter_d);
            blk.strides[d] = stride;

            const dim_t pdim = md.padded_dims[d];
            if (utils::one_of(DNNL_RUNTIME_DIM_VAL, stride, pdim))
                stride = DNNL_RUNTIME_DIM_VAL;
            else if (pdim != 0)
                stride *= pdim / blocks[d];

        } while (iter_d != perm.begin());
    }

    return status::success;
}

status_t fill_blocked(memory_desc_t &md, std::initializer_list<int> perm,
        std::initializer_list<int> inner_blks,
        std::initializer_list<int> inner_idxs) {
    return fill_blocked_impl(md, perm, inner_blks, inner_idxs);
}

status_t fill_blocked(memory_desc_t &md, std::vector<int>& perm,
                      std::vector<int>& inner_blks,
                      std::vector<int>& inner_idxs) {
    return fill_blocked_impl(md, perm, inner_blks, inner_idxs);
}

void memory_desc_wrapper::compute_strides_compat(dims_t *strides_compat) const {

    if (ndims() == 0) return;

    const blocking_desc_t &blk = md_->format_desc.blocking;

    dims_t blocks;
    compute_blocks(blocks);

    int perm_idx = 0;
    int inner_perm[DNNL_MAX_NDIMS] = {};
    bool seen[DNNL_MAX_NDIMS] = {};

    for (int i = 0; i < blk.inner_nblks; i++) {
        int blk_idx = blk.inner_idxs[i];
        if (seen[blk_idx]) continue;

        seen[blk_idx] = true;
        inner_perm[perm_idx] = blk_idx;
        perm_idx++;
    }

    for (int i = 0; i < ndims(); i++)
        if (!seen[i]) {
            inner_perm[perm_idx] = i;
            perm_idx++;
        }

    dims_t inner_strides;
    inner_strides[inner_perm[ndims() - 1]] = 1;
    for (int d = 1; d < ndims(); ++d) {
        const int prev_idx = inner_perm[ndims() - d];
        const int curr_idx = inner_perm[ndims() - 1 - d];

        inner_strides[curr_idx] = blocks[curr_idx] == 0
                ? 1
                : inner_strides[prev_idx]
                        * nstl::max((dim_t)1, blocks[prev_idx]);
    }

    utils::array_copy(strides_compat[0], blk.strides, ndims());
    utils::array_copy(strides_compat[1], inner_strides, ndims());
}

template<typename F, typename... Args>
status_t process_tag(F f, format_tag_t tag, Args&&... args) {
    using namespace format_tag;

#define C(tag, ... /* perm, inner_blks, inner_idxs */) \
    case tag: return f(std::forward<Args>(args)..., __VA_ARGS__)

    switch (tag) {
        C(a, {0}, {}, {});
        C(ab, {0, 1}, {}, {});
        C(abc, {0, 1, 2}, {}, {});
        C(abcd, {0, 1, 2, 3}, {}, {});
        C(acbd, {0, 2, 1, 3}, {}, {});
        C(abcde, {0, 1, 2, 3, 4}, {}, {});
        C(abcdef, {0, 1, 2, 3, 4, 5}, {}, {});
        C(abcdefg, {0, 1, 2, 3, 4, 5, 6}, {}, {});
        C(abcdefgh, {0, 1, 2, 3, 4, 5, 6, 7}, {}, {});
        C(abcdefghi, {0, 1, 2, 3, 4, 5, 6, 7, 8}, {}, {});
        C(abcdefghij, {0, 1, 2, 3, 4, 5, 6, 7, 8, 9}, {}, {});
        C(abcdefghijk, {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10}, {}, {});
        C(abcdefghijkl, {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11}, {}, {});
        C(abcdefghijlk, {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 11, 10}, {}, {});
        C(abcdefghikj, {0, 1, 2, 3, 4, 5, 6, 7, 8, 10, 9}, {}, {});
        C(abcdefghji, {0, 1, 2, 3, 4, 5, 6, 7, 9, 8}, {}, {});
        C(abcdefgih, {0, 1, 2, 3, 4, 5, 6, 8, 7}, {}, {});
        C(abcdefhg, {0, 1, 2, 3, 4, 5, 7, 6}, {}, {});
        C(abcdegf, {0, 1, 2, 3, 4, 6, 5}, {}, {});
        C(abcdfe, {0, 1, 2, 3, 5, 4}, {}, {});
        C(abced, {0, 1, 2, 4, 3}, {}, {});
        C(abdc, {0, 1, 3, 2}, {}, {});
        C(abdec, {0, 1, 3, 4, 2}, {}, {});
        C(abdfce, {0, 1, 3, 5, 2, 4}, {}, {});
        C(acb, {0, 2, 1}, {}, {});
        C(acbde, {0, 2, 1, 3, 4}, {}, {});
        C(acbdef, {0, 2, 1, 3, 4, 5}, {}, {});
        C(abdefc, {0, 1, 3, 4, 5, 2}, {}, {});
        C(acdb, {0, 2, 3, 1}, {}, {});
        C(acdeb, {0, 2, 3, 4, 1}, {}, {});
        C(adbc, {0, 3, 1, 2}, {}, {});
        C(ba, {1, 0}, {}, {});
        C(bac, {1, 0, 2}, {}, {});
        C(bacd, {1, 0, 2, 3}, {}, {});
        C(bacde, {1, 0, 2, 3, 4}, {}, {});
        C(bca, {1, 2, 0}, {}, {});
        C(bcda, {1, 2, 3, 0}, {}, {});
        C(bcdea, {1, 2, 3, 4, 0}, {}, {});
        C(cba, {2, 1, 0}, {}, {});
        C(cdba, {2, 3, 1, 0}, {}, {});
        C(dcab, {3, 2, 0, 1}, {}, {});
        C(cdeba, {2, 3, 4, 1, 0}, {}, {});
        C(decab, {3, 4, 2, 0, 1}, {}, {});
        C(defcab, {3, 4, 5, 2, 0, 1}, {}, {});

        C(Abc4a, {0, 1, 2}, {4}, {0});
        C(aBc4b, {0, 1, 2}, {4}, {1});

        C(BA16a16b, {1, 0}, {16, 16}, {0, 1});
        C(BA16a32b, {1, 0}, {16, 32}, {0, 1});
        C(BA16a48b, {1, 0}, {16, 48}, {0, 1});
        C(BA16a64b, {1, 0}, {16, 64}, {0, 1});
        C(BA16a16b2a, {1, 0}, {16, 16, 2}, {0, 1, 0});
        C(BA16a32b2a, {1, 0}, {16, 32, 2}, {0, 1, 0});
        C(BA16a48b2a, {1, 0}, {16, 48, 2}, {0, 1, 0});
        C(BA16a64b2a, {1, 0}, {16, 64, 2}, {0, 1, 0});
        C(BA16a16b4a, {1, 0}, {16, 16, 4}, {0, 1, 0});
        C(BA16a32b4a, {1, 0}, {16, 32, 4}, {0, 1, 0});
        C(BA16a48b4a, {1, 0}, {16, 48, 4}, {0, 1, 0});
        C(BA16a64b4a, {1, 0}, {16, 64, 4}, {0, 1, 0});

        C(ABc4b16a4b, {0, 1, 2}, {4, 16, 4}, {1, 0, 1});
        C(ABc4b32a4b, {0, 1, 2}, {4, 32, 4}, {1, 0, 1});
        C(ABc4b64a4b, {0, 1, 2}, {4, 64, 4}, {1, 0, 1});
        C(ABc2b8a4b, {0, 1, 2}, {2, 8, 4}, {1, 0, 1});
        C(ABc16b16a4b, {0, 1, 2}, {16, 16, 4}, {1, 0, 1});
        C(ABc16b32a4b, {0, 1, 2}, {16, 32, 4}, {1, 0, 1});
        C(ABc16b48a4b, {0, 1, 2}, {16, 48, 4}, {1, 0, 1});
        C(ABc16b64a4b, {0, 1, 2}, {16, 64, 4}, {1, 0, 1});
        C(ABc16b16a2b, {0, 1, 2}, {16, 16, 2}, {1, 0, 1});
        C(ABc16b32a2b, {0, 1, 2}, {16, 32, 2}, {1, 0, 1});
        C(ABc16b48a2b, {0, 1, 2}, {16, 48, 2}, {1, 0, 1});
        C(ABc16b64a2b, {0, 1, 2}, {16, 64, 2}, {1, 0, 1});
        C(ABc16a16b2a, {0, 1, 2}, {16, 16, 2}, {0, 1, 0});
        C(ABc2b8a16b4a, {0, 1, 2}, {2, 8, 16, 4}, {1, 0, 1, 0});
        C(ABcd2b8a16b4a, {0, 1, 2, 3}, {2, 8, 16, 4}, {1, 0, 1, 0});
        C(ABcde2b8a16b4a, {0, 1, 2, 3, 4}, {2, 8, 16, 4}, {1, 0, 1, 0});
        C(ABc2a8b16a4b, {0, 1, 2}, {2, 8, 16, 4}, {0, 1, 0, 1});
        C(ABcd2a8b16a4b, {0, 1, 2, 3}, {2, 8, 16, 4}, {0, 1, 0, 1});
        C(ABcde2a8b16a4b, {0, 1, 2, 3, 4}, {2, 8, 16, 4}, {0, 1, 0, 1});
        C(ABc2a8b16a2b, {0, 1, 2}, {2, 8, 16, 2}, {0, 1, 0, 1});
        C(ABcd2a8b16a2b, {0, 1, 2, 3}, {2, 8, 16, 2}, {0, 1, 0, 1});
        C(ABcde2a8b16a2b, {0, 1, 2, 3, 4}, {2, 8, 16, 2}, {0, 1, 0, 1});
        C(aCBd2c8b16c2b, {0, 2, 1, 3}, {2, 8, 16, 2}, {2, 1, 2, 1});
        C(aCBde2c8b16c2b, {0, 2, 1, 3, 4}, {2, 8, 16, 2}, {2, 1, 2, 1});
        C(aCBdef2c8b16c2b, {0, 2, 1, 3, 4, 5}, {2, 8, 16, 2}, {2, 1, 2, 1});
        C(aBCd2b8c16b2c, {0, 1, 2, 3}, {2, 8, 16, 2}, {1, 2, 1, 2});
        C(aBCde2b8c16b2c, {0, 1, 2, 3, 4}, {2, 8, 16, 2}, {1, 2, 1, 2});
        C(aBCdef2b8c16b2c, {0, 1, 2, 3, 4, 5}, {2, 8, 16, 2}, {1, 2, 1, 2});
        C(aBCd2b8c16b4c, {0, 1, 2, 3}, {2, 8, 16, 4}, {1, 2, 1, 2});
        C(aBCd2c8b16c2b, {0, 1, 2, 3}, {2, 8, 16, 2}, {2, 1, 2, 1});
        C(aBCde2b8c16b4c, {0, 1, 2, 3, 4}, {2, 8, 16, 4}, {1, 2, 1, 2});
        C(ABc4b4a, {0, 1, 2}, {4, 4}, {1, 0});
        C(Abcd4a, {0, 1, 2, 3}, {4}, {0});
        C(aBcd4b, {0, 1, 2, 3}, {4}, {1});
        C(ABcd4b4a, {0, 1, 2, 3}, {4, 4}, {1, 0});
        C(ABcd4a2b, {0, 1, 2, 3}, {4, 2}, {0, 1});
        C(ABcd4a4b, {0, 1, 2, 3}, {4, 4}, {0, 1});
        C(aBCd4c16b4c, {0, 1, 2, 3}, {4, 16, 4}, {2, 1, 2});
        C(aBCd2c8b4c, {0, 1, 2, 3}, {2, 8, 4}, {2, 1, 2});
        C(aBCd16c16b4c, {0, 1, 2, 3}, {16, 16, 4}, {2, 1, 2});
        C(aBCd16c16b2c, {0, 1, 2, 3}, {16, 16, 2}, {2, 1, 2});
        C(aBCd16b16c2b, {0, 1, 2, 3}, {16, 16, 2}, {1, 2, 1});
        C(aBCdef16c16b4c, {0, 1, 2, 3, 4, 5}, {16, 16, 4}, {2, 1, 2});
        C(aBCd4c4b, {0, 1, 2, 3}, {4, 4}, {2, 1});
        C(aBCd4b4c, {0, 1, 2, 3}, {4, 4}, {1, 2});
        C(Abcde4a, {0, 1, 2, 3, 4}, {4}, {0});
        C(aBcde4b, {0, 1, 2, 3, 4}, {4}, {1});
        C(ABcde4b4a, {0, 1, 2, 3, 4}, {4, 4}, {1, 0});
        C(ABcde4a2b, {0, 1, 2, 3, 4}, {4, 2}, {0, 1});
        C(ABcde4a4b, {0, 1, 2, 3, 4}, {4, 4}, {0, 1});
        C(aBCde4c4b, {0, 1, 2, 3, 4}, {4, 4}, {2, 1});
        C(aBcdef4b, {0, 1, 2, 3, 4, 5}, {4}, {1});
        C(aBCdef4c4b, {0, 1, 2, 3, 4, 5}, {4, 4}, {2, 1});
        C(aBCdef4b4c, {0, 1, 2, 3, 4, 5}, {4, 4}, {1, 2});
        C(aBdc4b, {0, 1, 3, 2}, {4}, {1});
        C(aBdec4b, {0, 1, 3, 4, 2}, {4}, {1});
        C(aBdefc4b, {0, 1, 3, 4, 5, 2}, {4}, {1});
        C(Acb4a, {0, 2, 1}, {4}, {0});
        C(Acdb4a, {0, 2, 3, 1}, {4}, {0});
        C(Acdeb4a, {0, 2, 3, 4, 1}, {4}, {0});

        C(AB16b16a, {0, 1}, {16, 16}, {1, 0});
        C(AB16b32a, {0, 1}, {16, 32}, {1, 0});
        C(AB16b64a, {0, 1}, {16, 64}, {1, 0});
        C(AB8b16a2b, {0, 1}, {8, 16, 2}, {1, 0, 1});
        C(AB8b32a2b, {0, 1}, {8, 32, 2}, {1, 0, 1});
        C(AB8b64a2b, {0, 1}, {8, 64, 2}, {1, 0, 1});
        C(AB4b16a4b, {0, 1}, {4, 16, 4}, {1, 0, 1});
        C(AB16b16a4b, {0, 1}, {16, 16, 4}, {1, 0, 1});
        C(AB16b32a4b, {0, 1}, {16, 32, 4}, {1, 0, 1});
        C(AB16b48a4b, {0, 1}, {16, 48, 4}, {1, 0, 1});
        C(AB16b64a4b, {0, 1}, {16, 64, 4}, {1, 0, 1});
        C(AB16b16a2b, {0, 1}, {16, 16, 2}, {1, 0, 1});
        C(AB16b32a2b, {0, 1}, {16, 32, 2}, {1, 0, 1});
        C(AB16b48a2b, {0, 1}, {16, 48, 2}, {1, 0, 1});
        C(AB16b64a2b, {0, 1}, {16, 64, 2}, {1, 0, 1});
        C(AB4b32a4b, {0, 1}, {4, 32, 4}, {1, 0, 1});
        C(AB4b64a4b, {0, 1}, {4, 64, 4}, {1, 0, 1});
        C(AB32a16b, {0, 1}, {32, 16}, {0, 1});
        C(AB32a32b, {0, 1}, {32, 32}, {0, 1});
        C(AB48a16b, {0, 1}, {48, 16}, {0, 1});
        C(AB48a32b, {0, 1}, {48, 32}, {0, 1});
        C(BA4b8a8b2a, {1, 0}, {4, 8, 8, 2}, {1, 0, 1, 0});
        C(BA4b8a8b4a, {1, 0}, {4, 8, 8, 4}, {1, 0, 1, 0});
        C(BA4b8a16b2a, {1, 0}, {4, 8, 16, 2}, {1, 0, 1, 0});
        C(BA4b8a16b4a, {1, 0}, {4, 8, 16, 4}, {1, 0, 1, 0});
        C(aBC32b16c, {0, 1, 2}, {32, 16}, {1, 2});
        C(aBC32b32c, {0, 1, 2}, {32, 32}, {1, 2});
        C(aBC48b16c, {0, 1, 2}, {48, 16}, {1, 2});
        C(aBC48b32c, {0, 1, 2}, {48, 32}, {1, 2});
        C(aCB4c8b8c2b, {0, 2, 1}, {4, 8, 8, 2}, {2, 1, 2, 1});
        C(aCB4c8b8c4b, {0, 2, 1}, {4, 8, 8, 4}, {2, 1, 2, 1});
        C(aCB4c8b16c2b, {0, 2, 1}, {4, 8, 16, 2}, {2, 1, 2, 1});
        C(aCB4c8b16c4b, {0, 2, 1}, {4, 8, 16, 4}, {2, 1, 2, 1});
        C(Abc16a, {0, 1, 2}, {16}, {0});
        C(ABc16a16b, {0, 1, 2}, {16, 16}, {0, 1});
        C(ABc4a2b, {0, 1, 2}, {4, 2}, {0, 1});
        C(ABc4a4b, {0, 1, 2}, {4, 4}, {0, 1});
        C(aBc16b, {0, 1, 2}, {16}, {1});
        C(aBc32b, {0, 1, 2}, {32}, {1});
        C(ABc16b16a, {0, 1, 2}, {16, 16}, {1, 0});
        C(ABc16b32a, {0, 1, 2}, {16, 32}, {1, 0});
        C(ABc16b64a, {0, 1, 2}, {16, 64}, {1, 0});
        C(ABc8a16b2a, {0, 1, 2}, {8, 16, 2}, {0, 1, 0});
        C(BAc8a16b2a, {1, 0, 2}, {8, 16, 2}, {0, 1, 0});
        C(ABc8a8b, {0, 1, 2}, {8, 8}, {0, 1});
        C(ABc8a2b, {0, 1, 2}, {8, 2}, {0, 1});
        C(ABc8a4b, {0, 1, 2}, {8, 4}, {0, 1});
        C(aBc8b, {0, 1, 2}, {8}, {1});
        C(ABc8b16a2b, {0, 1, 2}, {8, 16, 2}, {1, 0, 1});
        C(ABc8b32a2b, {0, 1, 2}, {8, 32, 2}, {1, 0, 1});
        C(ABc8b64a2b, {0, 1, 2}, {8, 64, 2}, {1, 0, 1});
        C(ABc8b8a, {0, 1, 2}, {8, 8}, {1, 0});
        C(Abcd16a, {0, 1, 2, 3}, {16}, {0});
        C(Abcd8a, {0, 1, 2, 3}, {8}, {0});
        C(Abcd32a, {0, 1, 2, 3}, {32}, {0});
        C(ABcd16a16b, {0, 1, 2, 3}, {16, 16}, {0, 1});
        C(aBcd16b, {0, 1, 2, 3}, {16}, {1});
        C(aBcd32b, {0, 1, 2, 3}, {32}, {1});
        C(ABcd16b16a, {0, 1, 2, 3}, {16, 16}, {1, 0});
        C(ABcd16b32a, {0, 1, 2, 3}, {16, 32}, {1, 0});
        C(ABcd16b64a, {0, 1, 2, 3}, {16, 64}, {1, 0});
        C(aBCd16b16c, {0, 1, 2, 3}, {16, 16}, {1, 2});
        C(aBCd16c16b, {0, 1, 2, 3}, {16, 16}, {2, 1});
        C(ABcd4b16a4b, {0, 1, 2, 3}, {4, 16, 4}, {1, 0, 1});
        C(ABcd4b32a4b, {0, 1, 2, 3}, {4, 32, 4}, {1, 0, 1});
        C(ABcd4b64a4b, {0, 1, 2, 3}, {4, 64, 4}, {1, 0, 1});
        C(ABcd2b8a4b, {0, 1, 2, 3}, {2, 8, 4}, {1, 0, 1});
        C(ABcd16b16a4b, {0, 1, 2, 3}, {16, 16, 4}, {1, 0, 1});
        C(ABcd16b32a4b, {0, 1, 2, 3}, {16, 32, 4}, {1, 0, 1});
        C(ABcd16b48a4b, {0, 1, 2, 3}, {16, 48, 4}, {1, 0, 1});
        C(ABcd16b64a4b, {0, 1, 2, 3}, {16, 64, 4}, {1, 0, 1});
        C(ABcd16b16a2b, {0, 1, 2, 3}, {16, 16, 2}, {1, 0, 1});
        C(ABcd16b32a2b, {0, 1, 2, 3}, {16, 32, 2}, {1, 0, 1});
        C(ABcd16b48a2b, {0, 1, 2, 3}, {16, 48, 2}, {1, 0, 1});
        C(ABcd16b64a2b, {0, 1, 2, 3}, {16, 64, 2}, {1, 0, 1});
        C(ABcd16a16b2a, {0, 1, 2, 3}, {16, 16, 2}, {0, 1, 0});
        C(ABcde16b16a2b, {0, 1, 2, 3, 4}, {16, 16, 2}, {1, 0, 1});
        C(ABcde16b32a2b, {0, 1, 2, 3, 4}, {16, 32, 2}, {1, 0, 1});
        C(ABcde16b48a2b, {0, 1, 2, 3, 4}, {16, 48, 2}, {1, 0, 1});
        C(ABcde16b64a2b, {0, 1, 2, 3, 4}, {16, 64, 2}, {1, 0, 1});
        C(ABcd8a16b2a, {0, 1, 2, 3}, {8, 16, 2}, {0, 1, 0});
        C(BAcd8a16b2a, {1, 0, 2, 3}, {8, 16, 2}, {0, 1, 0});
        C(ABcd8a8b, {0, 1, 2, 3}, {8, 8}, {0, 1});
        C(ABcd8a32b, {0, 1, 2, 3}, {8, 32}, {0, 1});
        C(ABcd16a32b, {0, 1, 2, 3}, {16, 32}, {0, 1});
        C(ABcd8a4b, {0, 1, 2, 3}, {8, 4}, {0, 1});
        C(ABcd8a2b, {0, 1, 2, 3}, {8, 2}, {0, 1});
        C(aBcd8b, {0, 1, 2, 3}, {8}, {1});
        C(ABcd8b16a2b, {0, 1, 2, 3}, {8, 16, 2}, {1, 0, 1});
        C(ABcd8b32a2b, {0, 1, 2, 3}, {8, 32, 2}, {1, 0, 1});
        C(ABcd8b64a2b, {0, 1, 2, 3}, {8, 64, 2}, {1, 0, 1});
        C(aBCd8b16c2b, {0, 1, 2, 3}, {8, 16, 2}, {1, 2, 1});
        C(aCBd8b16c2b, {0, 2, 1, 3}, {8, 16, 2}, {1, 2, 1});
        C(ABcd8b8a, {0, 1, 2, 3}, {8, 8}, {1, 0});
        C(aBCd8b8c, {0, 1, 2, 3}, {8, 8}, {1, 2});
        C(aBCd8b2c, {0, 1, 2, 3}, {8, 2}, {1, 2});
        C(aBCd8b4c, {0, 1, 2, 3}, {8, 4}, {1, 2});
        C(aBCd8c16b2c, {0, 1, 2, 3}, {8, 16, 2}, {2, 1, 2});
        C(aBCd8c8b, {0, 1, 2, 3}, {8, 8}, {2, 1});
        C(Abcde16a, {0, 1, 2, 3, 4}, {16}, {0});
        C(Abcde32a, {0, 1, 2, 3, 4}, {32}, {0});
        C(ABcde16a16b, {0, 1, 2, 3, 4}, {16, 16}, {0, 1});
        C(ABcde16a16b2a, {0, 1, 2, 3, 4}, {16, 16, 2}, {0, 1, 0});
        C(aBcde16b, {0, 1, 2, 3, 4}, {16}, {1});
        C(aBcde32b, {0, 1, 2, 3, 4}, {32}, {1});
        C(ABcde16b16a, {0, 1, 2, 3, 4}, {16, 16}, {1, 0});
        C(ABcde16b32a, {0, 1, 2, 3, 4}, {16, 32}, {1, 0});
        C(ABcde16b64a, {0, 1, 2, 3, 4}, {16, 64}, {1, 0});
        C(aBCde16b16c, {0, 1, 2, 3, 4}, {16, 16}, {1, 2});
        C(aBCde16c16b, {0, 1, 2, 3, 4}, {16, 16}, {2, 1});
        C(aBCde2c8b4c, {0, 1, 2, 3, 4}, {2, 8, 4}, {2, 1, 2});
        C(aBCde4b4c, {0, 1, 2, 3, 4}, {4, 4}, {1, 2});
        C(aBCde4c16b4c, {0, 1, 2, 3, 4}, {4, 16, 4}, {2, 1, 2});
        C(aBCde16c16b4c, {0, 1, 2, 3, 4}, {16, 16, 4}, {2, 1, 2});
        C(aBCde16c16b2c, {0, 1, 2, 3, 4}, {16, 16, 2}, {2, 1, 2});
        C(aBCde16b16c2b, {0, 1, 2, 3, 4}, {16, 16, 2}, {1, 2, 1});
        C(aBCdef16c16b2c, {0, 1, 2, 3, 4, 5}, {16, 16, 2}, {2, 1, 2});
        C(Abcde8a, {0, 1, 2, 3, 4}, {8}, {0});
        C(ABcde8a8b, {0, 1, 2, 3, 4}, {8, 8}, {0, 1});
        C(ABcde8a2b, {0, 1, 2, 3, 4}, {8, 2}, {0, 1});
        C(ABcde8a4b, {0, 1, 2, 3, 4}, {8, 4}, {0, 1});
        C(ABcde32a32b, {0, 1, 2, 3, 4}, {32, 32}, {0, 1});
        C(ABcde40a32b, {0, 1, 2, 3, 4}, {40, 32}, {0, 1});
        C(aBcde8b, {0, 1, 2, 3, 4}, {8}, {1});
        C(ABcde8a16b2a, {0, 1, 2, 3, 4}, {8, 16, 2}, {0, 1, 0});
        C(ABcde8b16a2b, {0, 1, 2, 3, 4}, {8, 16, 2}, {1, 0, 1});
        C(ABcde8b32a2b, {0, 1, 2, 3, 4}, {8, 32, 2}, {1, 0, 1});
        C(ABcde8b64a2b, {0, 1, 2, 3, 4}, {8, 64, 2}, {1, 0, 1});
        C(BAcde8a16b2a, {1, 0, 2, 3, 4}, {8, 16, 2}, {0, 1, 0});
        C(ABcde4b16a4b, {0, 1, 2, 3, 4}, {4, 16, 4}, {1, 0, 1});
        C(ABcde4b32a4b, {0, 1, 2, 3, 4}, {4, 32, 4}, {1, 0, 1});
        C(ABcde4b64a4b, {0, 1, 2, 3, 4}, {4, 64, 4}, {1, 0, 1});
        C(ABcde16b16a4b, {0, 1, 2, 3, 4}, {16, 16, 4}, {1, 0, 1});
        C(ABcde16b32a4b, {0, 1, 2, 3, 4}, {16, 32, 4}, {1, 0, 1});
        C(ABcde16b48a4b, {0, 1, 2, 3, 4}, {16, 48, 4}, {1, 0, 1});
        C(ABcde16b64a4b, {0, 1, 2, 3, 4}, {16, 64, 4}, {1, 0, 1});
        C(ABcde2b8a4b, {0, 1, 2, 3, 4}, {2, 8, 4}, {1, 0, 1});
        C(aBCde8b16c2b, {0, 1, 2, 3, 4}, {8, 16, 2}, {1, 2, 1});
        C(aCBde8b16c2b, {0, 2, 1, 3, 4}, {8, 16, 2}, {1, 2, 1});
        C(ABcde8b8a, {0, 1, 2, 3, 4}, {8, 8}, {1, 0});
        C(aBCde8b8c, {0, 1, 2, 3, 4}, {8, 8}, {1, 2});
        C(aBCde8b2c, {0, 1, 2, 3, 4}, {8, 2}, {1, 2});
        C(aBCde8b4c, {0, 1, 2, 3, 4}, {8, 4}, {1, 2});
        C(aBCde8c16b2c, {0, 1, 2, 3, 4}, {8, 16, 2}, {2, 1, 2});
        C(aBCde8c8b, {0, 1, 2, 3, 4}, {8, 8}, {2, 1});
        C(aBcdef16b, {0, 1, 2, 3, 4, 5}, {16}, {1});
        C(aBCdef16b16c, {0, 1, 2, 3, 4, 5}, {16, 16}, {1, 2});
        C(aBCdef16b16c2b, {0, 1, 2, 3, 4, 5}, {16, 16, 2}, {1, 2, 1});
        C(aBCdef16c16b, {0, 1, 2, 3, 4, 5}, {16, 16}, {2, 1});
        C(aBCdef4c16b4c, {0, 1, 2, 3, 4, 5}, {4, 16, 4}, {2, 1, 2});
        C(aBCdef2c8b4c, {0, 1, 2, 3, 4, 5}, {2, 8, 4}, {2, 1, 2});
        C(aBCdef8b8c, {0, 1, 2, 3, 4, 5}, {8, 8}, {1, 2});
        C(aBCdef8b2c, {0, 1, 2, 3, 4, 5}, {8, 2}, {1, 2});
        C(aBCdef8b4c, {0, 1, 2, 3, 4, 5}, {8, 4}, {1, 2});
        C(aBCdef8b16c2b, {0, 1, 2, 3, 4, 5}, {8, 16, 2}, {1, 2, 1});
        C(aBCdef8c16b2c, {0, 1, 2, 3, 4, 5}, {8, 16, 2}, {2, 1, 2});
        C(aCBdef8b16c2b, {0, 2, 1, 3, 4, 5}, {8, 16, 2}, {1, 2, 1});
        C(aBCdef8c8b, {0, 1, 2, 3, 4, 5}, {8, 8}, {2, 1});
        C(aBdc16b, {0, 1, 3, 2}, {16}, {1});
        C(aBdC16b2c, {0, 1, 3, 2}, {16, 2}, {1, 2});
        C(aBdC16b4c, {0, 1, 3, 2}, {16, 4}, {1, 2});
        C(aBdc8b, {0, 1, 3, 2}, {8}, {1});
        C(aBdec16b, {0, 1, 3, 4, 2}, {16}, {1});
        C(aBdeC16b2c, {0, 1, 3, 4, 2}, {16, 2}, {1, 2});
        C(aBdeC16b4c, {0, 1, 3, 4, 2}, {16, 4}, {1, 2});
        C(aBdec8b, {0, 1, 3, 4, 2}, {8}, {1});
        C(aBdefc16b, {0, 1, 3, 4, 5, 2}, {16}, {1});
        C(aBdefC16b2c, {0, 1, 3, 4, 5, 2}, {16, 2}, {1, 2});
        C(aBdefC16b4c, {0, 1, 3, 4, 5, 2}, {16, 4}, {1, 2});
        C(aBdefc8b, {0, 1, 3, 4, 5, 2}, {8}, {1});
        C(aBedc16b, {0, 1, 4, 3, 2}, {16}, {1});
        C(Acb16a, {0, 2, 1}, {16}, {0});
        C(AcB16a2b, {0, 2, 1}, {16, 2}, {0, 1});
        C(AcB16a4b, {0, 2, 1}, {16, 4}, {0, 1});
        C(Acb8a, {0, 2, 1}, {8}, {0});
        C(aCBd16b16c, {0, 2, 1, 3}, {16, 16}, {1, 2});
        C(aCBde16b16c, {0, 2, 1, 3, 4}, {16, 16}, {1, 2});
        C(Acdb16a, {0, 2, 3, 1}, {16}, {0});
        C(AcdB16a2b, {0, 2, 3, 1}, {16, 2}, {0, 1});
        C(AcdB16a4b, {0, 2, 3, 1}, {16, 4}, {0, 1});
        C(Acdb8a, {0, 2, 3, 1}, {8}, {0});
        C(Acdeb16a, {0, 2, 3, 4, 1}, {16}, {0});
        C(AcdeB16a2b, {0, 2, 3, 4, 1}, {16, 2}, {0, 1});
        C(AcdeB16a4b, {0, 2, 3, 4, 1}, {16, 4}, {0, 1});
        C(Acdeb8a, {0, 2, 3, 4, 1}, {8}, {0});
        C(Adcb16a, {0, 3, 2, 1}, {16}, {0});
        C(BAc16a16b, {1, 0, 2}, {16, 16}, {0, 1});
        C(BAcd16a16b, {1, 0, 2, 3}, {16, 16}, {0, 1});
        C(ABc32a16b, {0, 1, 2}, {32, 16}, {0, 1});
        C(ABcd32a16b, {0, 1, 2, 3}, {32, 16}, {0, 1});
        C(ABcde32a16b, {0, 1, 2, 3, 4}, {32, 16}, {0, 1});
        C(ABc40a16b, {0, 1, 2}, {40, 16}, {0, 1});
        C(ABcd40a16b, {0, 1, 2, 3}, {40, 16}, {0, 1});
        C(ABcde40a16b, {0, 1, 2, 3, 4}, {40, 16}, {0, 1});
        C(ABc2b32a8b, {0, 1, 2}, {2, 32, 8}, {1, 0, 1});
        C(ABcd2b32a8b, {0, 1, 2, 3}, {2, 32, 8}, {1, 0, 1});
        C(ABcde2b32a8b, {0, 1, 2, 3, 4}, {2, 32, 8}, {1, 0, 1});
        C(aBC2b8c16b2c, {0, 1, 2}, {2, 8, 16, 2}, {1, 2, 1, 2});
        C(ABc40a32b, {0, 1, 2}, {40, 32}, {0, 1});
        C(ABcd40a32b, {0, 1, 2, 3}, {40, 32}, {0, 1});
        C(ABc32a32b, {0, 1, 2}, {32, 32}, {0, 1});
        C(ABcd32a32b, {0, 1, 2, 3}, {32, 32}, {0, 1});
        C(BAcde16b16a, {1, 0, 2, 3, 4}, {16, 16}, {1, 0});
        C(ABc4a8b8a4b, {0, 1, 2}, {4, 8, 8, 4}, {0, 1, 0, 1});
        C(ABcd4a8b8a4b, {0, 1, 2, 3}, {4, 8, 8, 4}, {0, 1, 0, 1});
        C(ABcde4a8b8a4b, {0, 1, 2, 3, 4}, {4, 8, 8, 4}, {0, 1, 0, 1});
        C(BAc4b8a8b4a, {1, 0, 2}, {4, 8, 8, 4}, {1, 0, 1, 0});
        C(BAcd4b8a8b4a, {1, 0, 2, 3}, {4, 8, 8, 4}, {1, 0, 1, 0});
        C(BAcde4b8a8b4a, {1, 0, 2, 3, 4}, {4, 8, 8, 4}, {1, 0, 1, 0});
        C(BAc2b8a16b4a, {1, 0, 2}, {2, 8, 16, 4}, {1, 0, 1, 0});
        C(BAcd2b8a16b4a, {1, 0, 2, 3}, {2, 8, 16, 4}, {1, 0, 1, 0});
        C(BAcde2b8a16b4a, {1, 0, 2, 3, 4}, {2, 8, 16, 4}, {1, 0, 1, 0});
        C(BAc2b8a16b2a, {1, 0, 2}, {2, 8, 16, 2}, {1, 0, 1, 0});
        C(aBCde2c8b16c2b, {0, 1, 2, 3, 4}, {2, 8, 16, 2}, {2, 1, 2, 1});
        C(aBCdef2c8b16c2b, {0, 1, 2, 3, 4, 5}, {2, 8, 16, 2}, {2, 1, 2, 1});
        C(BAcd2b8a16b2a, {1, 0, 2, 3}, {2, 8, 16, 2}, {1, 0, 1, 0});
        C(BAcde2b8a16b2a, {1, 0, 2, 3, 4}, {2, 8, 16, 2}, {1, 0, 1, 0});
        C(aCBd4c8b8c4b, {0, 2, 1, 3}, {4, 8, 8, 4}, {2, 1, 2, 1});
        C(aCBde4c8b8c4b, {0, 2, 1, 3, 4}, {4, 8, 8, 4}, {2, 1, 2, 1});
        C(aCBdef4c8b8c4b, {0, 2, 1, 3, 4, 5}, {4, 8, 8, 4}, {2, 1, 2, 1});
        C(aBCdef4b8c8b4c, {0, 1, 2, 3, 4, 5}, {4, 8, 8, 4}, {1, 2, 1, 2});
        C(ABcd2a8b8a2b, {0, 1, 2, 3}, {2, 8, 8, 2}, {0, 1, 0, 1});
        C(ABcde4a8b8a2b, {0, 1, 2, 3, 4}, {4, 8, 8, 2}, {0, 1, 0, 1});
        C(ABcd4a8b8a2b, {0, 1, 2, 3}, {4, 8, 8, 2}, {0, 1, 0, 1});
        C(ABc4a8b8a2b, {0, 1, 2}, {4, 8, 8, 2}, {0, 1, 0, 1});
        C(aBCdef4b8c8b2c, {0, 1, 2, 3, 4, 5}, {4, 8, 8, 2}, {1, 2, 1, 2});
        C(aBCde4b8c8b2c, {0, 1, 2, 3, 4}, {4, 8, 8, 2}, {1, 2, 1, 2});
        C(aBCd4b8c8b2c, {0, 1, 2, 3}, {4, 8, 8, 2}, {1, 2, 1, 2});
        C(BAcde4b8a8b2a, {1, 0, 2, 3, 4}, {4, 8, 8, 2}, {1, 0, 1, 0});
        C(BAcd4b8a8b2a, {1, 0, 2, 3}, {4, 8, 8, 2}, {1, 0, 1, 0});
        C(BAc4b8a8b2a, {1, 0, 2}, {4, 8, 8, 2}, {1, 0, 1, 0});
        C(aCBdef4c8b8c2b, {0, 2, 1, 3, 4, 5}, {4, 8, 8, 2}, {2, 1, 2, 1});
        C(aCBde4c8b8c2b, {0, 2, 1, 3, 4}, {4, 8, 8, 2}, {2, 1, 2, 1});
        C(aCBd4c8b8c2b, {0, 2, 1, 3}, {4, 8, 8, 2}, {2, 1, 2, 1});
        C(aBCd4b8c8b4c, {0, 1, 2, 3}, {4, 8, 8, 4}, {1, 2, 1, 2});
        C(aBCde4b8c8b4c, {0, 1, 2, 3, 4}, {4, 8, 8, 4}, {1, 2, 1, 2});
        C(aBCde2b8c8b2c, {0, 1, 2, 3, 4}, {2, 8, 8, 2}, {1, 2, 1, 2});
        C(aBdec32b, {0, 1, 3, 4, 2}, {32}, {1});
        C(aCBdef16c16b, {0, 2, 1, 3, 4, 5}, {16, 16}, {2, 1});
        C(aCBdef16b16c, {0, 2, 1, 3, 4, 5}, {16, 16}, {1, 2});
        C(Abcdef4a, {0, 1, 2, 3, 4, 5}, {4}, {0});
        C(Abcdef8a, {0, 1, 2, 3, 4, 5}, {8}, {0});
        C(Abcdef16a, {0, 1, 2, 3, 4, 5}, {16}, {0});
        C(Abcdef32a, {0, 1, 2, 3, 4, 5}, {32}, {0});
        C(aCBd16c16b, {0, 2, 1, 3}, {16, 16}, {2, 1});
        C(aCBde16c16b, {0, 2, 1, 3, 4}, {16, 16}, {2, 1});
        C(Acdb32a, {0, 2, 3, 1}, {32}, {0});
        C(BAcd16b16a, {1, 0, 2, 3}, {16, 16}, {1, 0});
        C(BAcde16a16b, {1, 0, 2, 3, 4}, {16, 16}, {0, 1});
        C(BAc16b16a, {1, 0, 2}, {16, 16}, {1, 0});
        C(aBCd2b4c2b, {0, 1, 2, 3}, {2, 4, 2}, {1, 2, 1});
        C(aBCde2b4c2b, {0, 1, 2, 3, 4}, {2, 4, 2}, {1, 2, 1});
        C(aBCdef2b4c2b, {0, 1, 2, 3, 4, 5}, {2, 4, 2}, {1, 2, 1});
        C(aBCd2c4b2c, {0, 1, 2, 3}, {2, 4, 2}, {2, 1, 2});
        C(aBCde2c4b2c, {0, 1, 2, 3, 4}, {2, 4, 2}, {2, 1, 2});
        C(aBCdef2c4b2c, {0, 1, 2, 3, 4, 5}, {2, 4, 2}, {2, 1, 2});
        C(aBCd4b8c2b, {0, 1, 2, 3}, {4, 8, 2}, {1, 2, 1});
        C(aBCde4b8c2b, {0, 1, 2, 3, 4}, {4, 8, 2}, {1, 2, 1});
        C(aBCdef4b8c2b, {0, 1, 2, 3, 4, 5}, {4, 8, 2}, {1, 2, 1});
        C(aBCd4c8b2c, {0, 1, 2, 3}, {4, 8, 2}, {2, 1, 2});
        C(aBCde4c8b2c, {0, 1, 2, 3, 4}, {4, 8, 2}, {2, 1, 2});
        C(aBCdef4c8b2c, {0, 1, 2, 3, 4, 5}, {4, 8, 2}, {2, 1, 2});
        C(AB32a32b8a4b, {0, 1}, {32, 32, 8, 4}, {0, 1, 0, 1});
        C(AB8a4b, {0, 1}, {8, 4}, {0, 1});
        C(AB32a32b8a2b, {0, 1}, {32, 32, 8, 2}, {0, 1, 0, 1});
        C(AB8a2b, {0, 1}, {8, 2}, {0, 1});
        C(abDc32d, {0, 1, 3, 2}, {32}, {3});
        C(abDC32d4c, {0, 1, 3, 2}, {32, 4}, {3, 2});
        C(abCd32c, {0, 1, 2, 3}, {32}, {2});
        C(abdEc32e, {0, 1, 3, 4, 2}, {32}, {4});
        C(abdEC32e2c, {0, 1, 3, 4, 2}, {32, 2}, {4, 2});
        C(abdEC32e4c, {0, 1, 3, 4, 2}, {32, 4}, {4, 2});
        C(abdCe32c, {0, 1, 3, 2, 4}, {32}, {2});
        C(abdCE32c2e, {0, 1, 3, 2, 4}, {32, 2}, {2, 4});

        C(cBa2b, {2, 1, 0}, {2}, {1});
        C(cBa4b, {2, 1, 0}, {4}, {1});
        C(adcb, {0, 3, 2, 1}, {}, {});
        C(adCb2c, {0, 3, 2, 1}, {2}, {2});
        C(adCb4c, {0, 3, 2, 1}, {4}, {2});
        C(cdBa2b, {2, 3, 1, 0}, {2}, {1});
        C(cdBa4b, {2, 3, 1, 0}, {4}, {1});
        C(adecb, {0, 3, 4, 2, 1}, {}, {});
        C(adeCb2c, {0, 3, 4, 2, 1}, {2}, {2});
        C(adeCb4c, {0, 3, 4, 2, 1}, {4}, {2});
        C(cdeBa2b, {2, 3, 4, 1, 0}, {2}, {1});
        C(cdeBa4b, {2, 3, 4, 1, 0}, {4}, {1});
        C(adefcb, {0, 3, 4, 5, 2, 1}, {}, {});
        C(adefCb2c, {0, 3, 4, 5, 2, 1}, {2}, {2});
        C(adefCb4c, {0, 3, 4, 5, 2, 1}, {4}, {2});
        C(aBdc32b, {0, 1, 3, 2}, {32}, {1});
        C(aBdC32b2c, {0, 1, 3, 2}, {32, 2}, {1, 2});
        C(aBdC32b4c, {0, 1, 3, 2}, {32, 4}, {1, 2});
        C(aBdc48b, {0, 1, 3, 2}, {48}, {1});
        C(aBdC48b2c, {0, 1, 3, 2}, {48, 2}, {1, 2});
        C(aBdC48b4c, {0, 1, 3, 2}, {48, 4}, {1, 2});
        C(aBdc64b, {0, 1, 3, 2}, {64}, {1});
        C(aBdC64b2c, {0, 1, 3, 2}, {64, 2}, {1, 2});
        C(aBdC64b4c, {0, 1, 3, 2}, {64, 4}, {1, 2});
        C(aBdeC32b2c, {0, 1, 3, 4, 2}, {32, 2}, {1, 2});
        C(aBdeC32b4c, {0, 1, 3, 4, 2}, {32, 4}, {1, 2});
        C(aBdec48b, {0, 1, 3, 4, 2}, {48}, {1});
        C(aBdeC48b2c, {0, 1, 3, 4, 2}, {48, 2}, {1, 2});
        C(aBdeC48b4c, {0, 1, 3, 4, 2}, {48, 4}, {1, 2});
        C(aBdec64b, {0, 1, 3, 4, 2}, {64}, {1});
        C(aBdeC64b2c, {0, 1, 3, 4, 2}, {64, 2}, {1, 2});
        C(aBdeC64b4c, {0, 1, 3, 4, 2}, {64, 4}, {1, 2});
        C(aBdefc32b, {0, 1, 3, 4, 5, 2}, {32}, {1});
        C(aBdefC32b2c, {0, 1, 3, 4, 5, 2}, {32, 2}, {1, 2});
        C(aBdefC32b4c, {0, 1, 3, 4, 5, 2}, {32, 4}, {1, 2});
        C(aBdefc48b, {0, 1, 3, 4, 5, 2}, {48}, {1});
        C(aBdefC48b2c, {0, 1, 3, 4, 5, 2}, {48, 2}, {1, 2});
        C(aBdefC48b4c, {0, 1, 3, 4, 5, 2}, {48, 4}, {1, 2});
        C(aBdefc64b, {0, 1, 3, 4, 5, 2}, {64}, {1});
        C(aBdefC64b2c, {0, 1, 3, 4, 5, 2}, {64, 2}, {1, 2});
        C(aBdefC64b4c, {0, 1, 3, 4, 5, 2}, {64, 4}, {1, 2});
        C(Acb32a, {0, 2, 1}, {32}, {0});
        C(AcB32a2b, {0, 2, 1}, {32, 2}, {0, 1});
        C(AcB32a4b, {0, 2, 1}, {32, 4}, {0, 1});
        C(Acb48a, {0, 2, 1}, {48}, {0});
        C(AcB48a2b, {0, 2, 1}, {48, 2}, {0, 1});
        C(AcB48a4b, {0, 2, 1}, {48, 4}, {0, 1});
        C(Acb64a, {0, 2, 1}, {64}, {0});
        C(AcB64a2b, {0, 2, 1}, {64, 2}, {0, 1});
        C(AcB64a4b, {0, 2, 1}, {64, 4}, {0, 1});
        C(AcdB32a2b, {0, 2, 3, 1}, {32, 2}, {0, 1});
        C(AcdB32a4b, {0, 2, 3, 1}, {32, 4}, {0, 1});
        C(Acdb48a, {0, 2, 3, 1}, {48}, {0});
        C(AcdB48a2b, {0, 2, 3, 1}, {48, 2}, {0, 1});
        C(AcdB48a4b, {0, 2, 3, 1}, {48, 4}, {0, 1});
        C(Acdb64a, {0, 2, 3, 1}, {64}, {0});
        C(AcdB64a2b, {0, 2, 3, 1}, {64, 2}, {0, 1});
        C(AcdB64a4b, {0, 2, 3, 1}, {64, 4}, {0, 1});
        C(Acdeb32a, {0, 2, 3, 4, 1}, {32}, {0});
        C(AcdeB32a2b, {0, 2, 3, 4, 1}, {32, 2}, {0, 1});
        C(AcdeB32a4b, {0, 2, 3, 4, 1}, {32, 4}, {0, 1});
        C(Acdeb48a, {0, 2, 3, 4, 1}, {48}, {0});
        C(AcdeB48a2b, {0, 2, 3, 4, 1}, {48, 2}, {0, 1});
        C(AcdeB48a4b, {0, 2, 3, 4, 1}, {48, 4}, {0, 1});
        C(Acdeb64a, {0, 2, 3, 4, 1}, {64}, {0});
        C(AcdeB64a2b, {0, 2, 3, 4, 1}, {64, 2}, {0, 1});
        C(AcdeB64a4b, {0, 2, 3, 4, 1}, {64, 4}, {0, 1});
        C(aBdeC16c16b2c, {0, 1, 3, 4, 2}, {16, 16, 2}, {2, 1, 2});
        C(aBdeC16c16b4c, {0, 1, 3, 4, 2}, {16, 16, 4}, {2, 1, 2});
        C(aBdefC16c16b2c, {0, 1, 3, 4, 5, 2}, {16, 16, 2}, {2, 1, 2});
        C(aBdefC16c16b4c, {0, 1, 3, 4, 5, 2}, {16, 16, 4}, {2, 1, 2});
        C(AcB16b16a2b, {0, 2, 1}, {16, 16, 2}, {1, 0, 1});
        C(AcB16b16a4b, {0, 2, 1}, {16, 16, 4}, {1, 0, 1});
        C(AcdB16b16a2b, {0, 2, 3, 1}, {16, 16, 2}, {1, 0, 1});
        C(AcdB16b16a4b, {0, 2, 3, 1}, {16, 16, 4}, {1, 0, 1});
        C(AcdeB16b16a2b, {0, 2, 3, 4, 1}, {16, 16, 2}, {1, 0, 1});
        C(AcdeB16b16a4b, {0, 2, 3, 4, 1}, {16, 16, 4}, {1, 0, 1});
        C(aBdC16c16b2c, {0, 1, 3, 2}, {16, 16, 2}, {2, 1, 2});
        C(aBdC16c16b4c, {0, 1, 3, 2}, {16, 16, 4}, {2, 1, 2});
        C(aBdC16c32b2c, {0, 1, 3, 2}, {16, 32, 2}, {2, 1, 2});
        C(aBdC16c32b4c, {0, 1, 3, 2}, {16, 32, 4}, {2, 1, 2});
        C(aBdC16c48b2c, {0, 1, 3, 2}, {16, 48, 2}, {2, 1, 2});
        C(aBdC16c48b4c, {0, 1, 3, 2}, {16, 48, 4}, {2, 1, 2});
        C(aBdC16c64b2c, {0, 1, 3, 2}, {16, 64, 2}, {2, 1, 2});
        C(aBdC16c64b4c, {0, 1, 3, 2}, {16, 64, 4}, {2, 1, 2});
        C(aBdeC16c32b2c, {0, 1, 3, 4, 2}, {16, 32, 2}, {2, 1, 2});
        C(aBdeC16c32b4c, {0, 1, 3, 4, 2}, {16, 32, 4}, {2, 1, 2});
        C(aBdeC16c48b2c, {0, 1, 3, 4, 2}, {16, 48, 2}, {2, 1, 2});
        C(aBdeC16c48b4c, {0, 1, 3, 4, 2}, {16, 48, 4}, {2, 1, 2});
        C(aBdeC16c64b2c, {0, 1, 3, 4, 2}, {16, 64, 2}, {2, 1, 2});
        C(aBdeC16c64b4c, {0, 1, 3, 4, 2}, {16, 64, 4}, {2, 1, 2});
        C(aBdefC16c32b2c, {0, 1, 3, 4, 5, 2}, {16, 32, 2}, {2, 1, 2});
        C(aBdefC16c32b4c, {0, 1, 3, 4, 5, 2}, {16, 32, 4}, {2, 1, 2});
        C(aBdefC16c48b2c, {0, 1, 3, 4, 5, 2}, {16, 48, 2}, {2, 1, 2});
        C(aBdefC16c48b4c, {0, 1, 3, 4, 5, 2}, {16, 48, 4}, {2, 1, 2});
        C(aBdefC16c64b2c, {0, 1, 3, 4, 5, 2}, {16, 64, 2}, {2, 1, 2});
        C(aBdefC16c64b4c, {0, 1, 3, 4, 5, 2}, {16, 64, 4}, {2, 1, 2});
        C(AcB16b32a2b, {0, 2, 1}, {16, 32, 2}, {1, 0, 1});
        C(AcB16b32a4b, {0, 2, 1}, {16, 32, 4}, {1, 0, 1});
        C(AcB16b48a2b, {0, 2, 1}, {16, 48, 2}, {1, 0, 1});
        C(AcB16b48a4b, {0, 2, 1}, {16, 48, 4}, {1, 0, 1});
        C(AcB16b64a2b, {0, 2, 1}, {16, 64, 2}, {1, 0, 1});
        C(AcB16b64a4b, {0, 2, 1}, {16, 64, 4}, {1, 0, 1});
        C(AcdB16b32a2b, {0, 2, 3, 1}, {16, 32, 2}, {1, 0, 1});
        C(AcdB16b32a4b, {0, 2, 3, 1}, {16, 32, 4}, {1, 0, 1});
        C(AcdB16b48a2b, {0, 2, 3, 1}, {16, 48, 2}, {1, 0, 1});
        C(AcdB16b48a4b, {0, 2, 3, 1}, {16, 48, 4}, {1, 0, 1});
        C(AcdB16b64a2b, {0, 2, 3, 1}, {16, 64, 2}, {1, 0, 1});
        C(AcdB16b64a4b, {0, 2, 3, 1}, {16, 64, 4}, {1, 0, 1});
        C(AcdeB16b32a2b, {0, 2, 3, 4, 1}, {16, 32, 2}, {1, 0, 1});
        C(AcdeB16b32a4b, {0, 2, 3, 4, 1}, {16, 32, 4}, {1, 0, 1});
        C(AcdeB16b48a2b, {0, 2, 3, 4, 1}, {16, 48, 2}, {1, 0, 1});
        C(AcdeB16b48a4b, {0, 2, 3, 4, 1}, {16, 48, 4}, {1, 0, 1});
        C(AcdeB16b64a2b, {0, 2, 3, 4, 1}, {16, 64, 2}, {1, 0, 1});
        C(AcdeB16b64a4b, {0, 2, 3, 4, 1}, {16, 64, 4}, {1, 0, 1});
        C(decbA16a, {3, 4, 2, 1, 0}, {16}, {0});
        C(aCB16c2b, {0, 2, 1}, {16, 2}, {2, 1});
        C(aCB16c4b, {0, 2, 1}, {16, 4}, {2, 1});
        C(BA16b2a, {1, 0}, {16, 2}, {1, 0});
        C(BA16b4a, {1, 0}, {16, 4}, {1, 0});
        C(aBC16b16c, {0, 1, 2}, {16, 16}, {1, 2});
        C(aBC16b32c, {0, 1, 2}, {16, 32}, {1, 2});
        C(AB16a16b, {0, 1}, {16, 16}, {0, 1});
        C(AB16a32b, {0, 1}, {16, 32}, {0, 1});

        default: break;
    }

#undef C

    return status::invalid_arguments;
}

status_t memory_desc_wrapper::compute_blocking(memory_desc_t &memory_desc, format_tag_t tag) {
    using fill_blocked_t = status_t(memory_desc_t&, std::initializer_list<int>, std::initializer_list<int>, std::initializer_list<int>);
    if (memory_desc.ndims == 0) return status::invalid_arguments;
    return process_tag<fill_blocked_t>(fill_blocked, tag, memory_desc);
}

status_t memory_desc_wrapper::compute_blocking(format_tag_t tag,
                          std::vector<size_t> &perm,
                          std::vector<size_t> &inner_blks,
                          std::vector<size_t> &inner_idxs) {

    auto extract_data = [&](std::initializer_list<int> _perm,
                            std::initializer_list<int> _inner_blks,
                            std::initializer_list<int> _inner_idxs) -> status_t {
        perm = {_perm.begin(), _perm.end()};
        inner_blks = {_inner_blks.begin(), _inner_blks.end()};
        inner_idxs = {_inner_idxs.begin(), _inner_idxs.end()};
        return status::success;
    };
    return process_tag(extract_data, tag);
}

} // namespace impl
} // namespace dnnl

// vim: et ts=4 sw=4 cindent cino+=l0,\:4,N-s
