// Copyright (C) 2018-2022 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma once
#include "primitive.hpp"

namespace cldnn {
/// @addtogroup cpp_api C++ API
/// @{
/// @addtogroup cpp_topology Network Topology
/// @{
/// @addtogroup cpp_primitives Primitives
/// @{

/// @brief Marker type indicating that instead of reference input size left, top,
///        right and bottom borders (to cut out) should be specified.
///
/// @details Used to differentiate constructors.
struct crop_borders_t {};

/// @brief Marker indicating that instead of reference input size left, top,
///        right and bottom borders (to cut out) should be specified.
constexpr auto crop_borders = crop_borders_t{};

/// @brief Performs crop operation on input.
/// @details Crops the input to the shape of reference_input across all dimensions taking into account specified input offsets.
/// @n       Borders variant calculated output shape from input shape minus the specified borders.
/// @n\b Requirements (reference size variant)
/// @n - Input size cannot be greater than reference size in any dimension
/// @n - All sizes have to have positive numbers
/// @n - Reference size plus offset cannot exceed input size
/// @n
/// @n\b Requirements (borders variant)
/// @n - Borders support batch, feature and spatial dimensions (rest of dimensions ignored).
/// @n - Input size cannot be greater than reference size in any dimension
/// @n - All sizes specified in borders have to have non-negative values (positive or @c 0).
/// @n - Sum of sizes of opposite borders must be lower than input size (on all non-ignored dimensions).
/// @n
/// @n Breaking any of this conditions will cause exception throw.
struct crop : public primitive_base<crop> {
    CLDNN_DECLARE_PRIMITIVE(crop)

    /// @brief Constructs crop primitive.
    /// @param id This primitive id.
    /// @param input Input primitive id.
    /// @param reference_input Reference input tensor with the required dimensions.
    /// @param offsets Input offsets.
    crop(const primitive_id& id,
         const primitive_id& input,
         const tensor& reference_input,
         const tensor& offsets,
         const primitive_id& ext_prim_id = "",
         const padding& output_padding = padding())
        : primitive_base(id, {input}, ext_prim_id, output_padding), reference_input(reference_input), offsets(offsets) {}

    /// @brief Constructs crop primitive (borders variant).
    ///
    /// @details Allows to specify borders from each side that should be cut out
    ///          by the primitive.
    /// @n       NOTE: Borders variant supports only up to four dimensions.
    ///
    /// @param id         Identifier of newly created primitive.
    /// @param input      Identifier of input primitive which dimensions will be cropped.
    /// @param lt_borders Border sizes (spatial dimensions define left (X) and top (Y)
    ///                   borders, non-spatial dimensions - lower borders)
    /// @param rb_borders Border sizes (spatial dimensions define right (X) and bottom (Y)
    ///                   borders, non-spatial dimensions - upper borders)
    crop(const primitive_id& id,
         const primitive_id& input,
         const tensor& lt_borders,
         const tensor& rb_borders,
         const crop_borders_t,
         const primitive_id& ext_prim_id = "",
         const padding& output_padding = padding())
        : primitive_base(id, {input}, ext_prim_id, output_padding), reference_input(rb_borders.negate()), offsets(lt_borders) {}

    /// @brief Constructs crop primitive (symmetric borders variant).
    ///
    /// @details Allows to specify borders from each side that should be cut out
    ///          by the primitive.
    /// @n       NOTE: Borders variant supports only up to four dimensions.
    ///
    /// @param id         Identifier of newly created primitive.
    /// @param input      Identifier of input primitive which dimensions will be cropped.
    /// @param xy_borders Border sizes (symmetric; spatial dimensions define left/right (X)
    ///                   and top/bottom (Y) borders, non-spatial dimensions - lower/upper borders).
    crop(const primitive_id& id,
         const primitive_id& input,
         const tensor& xy_borders,
         const crop_borders_t,
         const primitive_id& ext_prim_id = "",
         const padding& output_padding = padding())
        : primitive_base(id, {input}, ext_prim_id, output_padding), reference_input(xy_borders.negate()), offsets(xy_borders) {}

    /// @brief Reference input tensor with the required dimensions.
    tensor reference_input;
    /// @brief Input offsets.
    tensor offsets;
};
/// @}
/// @}
/// @}
}  // namespace cldnn
