// Copyright (C) 2022 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

#include <test_utils/test_utils.h>

#include <intel_gpu/primitives/dft.hpp>
#include <vector>

using namespace cldnn;
using namespace tests;

namespace {

template <typename vecElementType>
std::string vec2str(const std::vector<vecElementType>& vec) {
    if (!vec.empty()) {
        std::ostringstream result;
        result << "(";
        std::copy(vec.begin(), vec.end() - 1, std::ostream_iterator<vecElementType>(result, "."));
        result << vec.back() << ")";
        return result.str();
    }
    return "()";
}

template <class T>
std::vector<T> convert(const std::vector<float>& v) {
    std::vector<T> result(v.begin(), v.end());
    return result;
}

struct dft_test_params {
    std::vector<int32_t> input_shape;
    std::vector<size_t> output_shape;
    std::vector<int64_t> axes;
    std::vector<float> input_values;
    std::vector<float> expected_values;
    std::string test_name;
};

struct idft_test_params {
    std::vector<int32_t> input_shape;
    std::vector<size_t> output_shape;
    std::vector<int64_t> axes;
    std::vector<float> expected_values;
    std::vector<float> input_values;
    std::string test_name;
};

template <class P>
dft_kind getKind();

template <>
dft_kind getKind<dft_test_params>() {
    return dft_kind::forward;
}

template <>
dft_kind getKind<idft_test_params>() {
    return dft_kind::inverse;
}

template <class P, class T>
struct dft_gpu_test : public testing::TestWithParam<P> {
public:
    void test() {
        auto p = testing::TestWithParam<P>::GetParam();
        auto& engine = get_test_engine();

        const auto input_format = format::get_default_format(p.input_shape.size());
        const layout data_layout(type_to_data_type<T>::value, input_format, tensor(input_format, p.input_shape));
        auto input = engine.allocate_memory(data_layout);
        set_values(input, convert<T>(p.input_values));

        topology topology;
        topology.add(input_layout("input", input->get_layout()));
        topology.add(dft("dft", "input", std::move(p.axes), p.output_shape, getKind<P>()));

        network network(engine, topology);
        network.set_input_data("input", input);
        const auto outputs = network.execute();

        EXPECT_EQ(outputs.size(), size_t(1));
        EXPECT_EQ(outputs.begin()->first, "dft");

        auto output = outputs.at("dft").get_memory();
        cldnn::mem_lock<T> output_ptr(output, get_test_stream());

        const auto expected_values = convert<T>(p.expected_values);
        ASSERT_EQ(output_ptr.size(), expected_values.size());
        for (size_t i = 0; i < output_ptr.size(); ++i) {
            EXPECT_NEAR(expected_values[i], output_ptr[i], getThreshold());
        }
    }

    static std::string PrintToStringParamName(const testing::TestParamInfo<P>& info) {
        auto& p = info.param;
        std::ostringstream result;
        result << "InputShape=" << vec2str(p.input_shape) << "_";
        result << "OutputShape=" << vec2str(p.output_shape) << "_";
        result << "Precision=" << data_type_traits::name(type_to_data_type<T>::value) << "_";
        result << "Axes=" << vec2str(p.axes) << "_";
        result << "Inverse=" << (getKind<P>() == dft_kind::inverse);
        if (!p.test_name.empty()) {
            result << "_TestName=" << p.test_name;
        }
        return result.str();
    }

    float getThreshold() const;
};

template <>
float dft_gpu_test<dft_test_params, float>::getThreshold() const {
    return 1e-4f;
}

template <>
float dft_gpu_test<dft_test_params, half_t>::getThreshold() const {
    return 4e-2f;
}

template <>
float dft_gpu_test<idft_test_params, float>::getThreshold() const {
    return 4e-6f;
}

template <>
float dft_gpu_test<idft_test_params, half_t>::getThreshold() const {
    return 5e-4f;
}

using dft_gpu_test_f32 = dft_gpu_test<dft_test_params, float>;
using dft_gpu_test_f16 = dft_gpu_test<dft_test_params, half_t>;
using idft_gpu_test_f32 = dft_gpu_test<idft_test_params, float>;
using idft_gpu_test_f16 = dft_gpu_test<idft_test_params, half_t>;

TEST_P(dft_gpu_test_f32, dft_f32) {
    test();
}

TEST_P(dft_gpu_test_f16, dft_f16) {
    test();
}

TEST_P(idft_gpu_test_f32, idft_f32) {
    test();
}

TEST_P(idft_gpu_test_f16, idft_f16) {
    test();
}

const std::vector<float> input_data = {
    0.85943836,   0.009941814, 0.004292889, 0.54598427,   0.8270831,   0.49770153,  0.9035636,   0.19274887,
    0.8589833,    0.88759327,  0.72343576,  0.057539318,  0.915801,    0.63455844,  0.25069925,  0.045601673,
    0.29793364,   0.8492151,   0.6885839,   0.57419384,   0.009737609, 0.68192583,  0.7614807,   0.37603703,
    0.51804876,   0.033039097, 0.63702065,  0.78960556,   0.5007368,   0.7248742,   0.2040932,   0.1211606,
    0.76035476,   0.44004318,  0.95635134,  0.82913375,   0.225465,    0.009166263, 0.05445403,  0.5885675,
    0.87822133,   0.14324947,  0.68606305,  0.3274419,    0.9169595,   0.732179,    0.04614906,  0.03505424,
    0.84526163,   0.9972937,   0.89781004,  0.9987864,    0.24641308,  0.34678686,  0.22731997,  0.95805293,
    0.595993,     0.8537836,   0.9174756,   0.17441267,   0.86681056,  0.15913424,  0.6638066,   0.522398,
    0.51548326,   0.024979044, 0.1731268,   0.068090245,  0.6125645,   0.4865482,   0.2873719,   0.35936728,
    0.64452374,   0.27963468,  0.59981745,  0.6309508,    0.507604,    0.23389837,  0.77500635,  0.4462004,
    0.53165394,   0.6535075,   0.4306448,   0.21468966,   0.6925882,   0.11183031,  0.25347117,  0.2209481,
    0.8060583,    0.34712377,  0.78980505,  0.16110454,   0.6376819,   0.78736854,  0.909368,    0.6915289,
    0.24747796,   0.32442623,  0.22714981,  0.23976989,   0.25199527,  0.28412706,  0.32461873,  0.51917267,
    0.8394496,    0.6324911,   0.28498915,  0.8887276,    0.90213394,  0.16050571,  0.32190812,  0.67677563,
    0.8594967,    0.28917953,  0.1931407,   0.8282108,    0.14881423,  0.18073067,  0.8490643,   0.2356146,
    0.86200285,   0.57409924,  0.94718546,  0.092213534,  0.34502912,  0.4719212,   0.60031396,  0.22602181,
    0.3067876,    0.49529344,  0.11133887,  0.47633907,   0.13236542,  0.69677263,  0.8490109,   0.6685073,
    0.24199674,   0.7983137,   0.37593383,  0.74520975,   0.16743147,  0.84144354,  0.93073046,  0.55940866,
    0.67484015,   0.077098235, 0.69045097,  0.06949082,   0.6804774,   0.79804176,  0.49027568,  0.8843709,
    0.5665486,    0.91798306,  0.47884017,  0.94707423,   0.98279756,  0.62054926,  0.8134105,   0.01336217,
    0.78324115,   0.9938295,   0.99227554,  0.66681916,   0.38842493,  0.3835454,   0.120395586, 0.5478275,
    0.13309076,   0.9468553,   0.24595714,  0.0057277656, 0.14570542,  0.31220108,  0.41687667,  0.679465,
    0.5731583,    0.7383743,   0.013198466, 0.34619793,   0.9278514,   0.48510832,  0.46039802,  0.8171329,
    0.5041023,    0.37600085,  0.124404594, 0.4201713,    0.7470036,   0.7340853,   0.8449047,   0.137517,
    0.14771219,   0.99655616,  0.2178388,   0.4121613,    0.8655656,   0.32849622,  0.7574791,   0.95230037,
    0.5806251,    0.9598742,   0.7183528,   0.042957753,  0.2926446,   0.5882527,   0.05208914,  0.3216481,
    0.5205192,    0.5095992,   0.011508227, 0.5209922,    0.78207654,  0.34570032,  0.7968098,   0.4619513,
    0.0047925604, 0.029039407, 0.7673424,   0.571703,     0.44400942,  0.82529145,  0.29335254,  0.34418115,
    0.48119327,   0.38321403,  0.31083322,  0.7179562,    0.41055596,  0.06207573,  0.8747831,   0.6018095,
    0.4483476,    0.16189687,  0.8611539,   0.79723805,   0.42178747,  0.95597315,  0.5534858,   0.33466807,
    0.36827618,   0.60728735,  0.6582703,   0.6790265,    0.870856,    0.8868432,   0.43860948,  0.32468447,
    0.77624434,   0.3403061,   0.14144918,  0.23022941,   0.07176102,  0.06941459,  0.37346482,  0.9120822,
    0.65890974,   0.77746564,  0.4515671,   0.45455948,   0.15909587,  0.8017096,   0.6259673,   0.6117355,
    0.77020043,   0.08495594,  0.30376136,  0.55266386,   0.8497134,   0.91790336,  0.86088765,  0.88179666,
    0.9009849,    0.97200614,  0.94119,     0.77911216,   0.8057816,   0.14040896,  0.66522235,  0.6649202,
    0.048396785,  0.75035393,  0.4520953,   0.9877601,    0.46115568,  0.2167145,   0.9271302,   0.39395386,
    0.68578094,   0.576275,    0.20754486,  0.5408786,    0.46040633,  0.18199016,  0.66303253,  0.6288556,
    0.14313427,   0.91675115,  0.36198065,  0.51337945,   0.84241706,  0.22333568,  0.38011634,  0.024615016,
    0.19370414,   0.23593484,  0.32207185,  0.47971123,   0.6202779,   0.6944977,   0.43612957,  0.07961436,
    0.57468814,   0.100025274, 0.42476946,  0.95338464,   0.666547,    0.8683905,   0.52689695,  0.6284723,
    0.85813546,   0.4865953,   0.8269112,   0.08833949,   0.69269264,  0.41784903,  0.5969149,   0.07599888,
    0.14184453,   0.49042618,  0.44027725,  0.6256328,    0.2716237,   0.0999099,   0.09831784,  0.92469853,
    0.24196884,   0.9073526,   0.7523511,   0.7761173,    0.28489882,  0.96349007,  0.5884645,   0.74933976,
    0.06400105,   0.4376275,   0.34752035,  0.6006149,    0.034923803, 0.066874385, 0.9790322,   0.5558188,
    0.97579825,   0.025802653, 0.537738,    0.24921915,   0.111012295, 0.85987717,  0.781183,    0.69588315,
    0.94621634,   0.74946797,  0.6949375,   0.009165181,  0.91075164,  0.72913235,  0.25934777,  0.19463088,
    0.5283304,    0.9241759,   0.0563183,   0.74323857,   0.43722472,  0.2958358,   0.85980684,  0.029655656,
    0.362904,     0.19682994,  0.37778872,  0.09406928,   0.23010127,  0.44393733,  0.420214,    0.39723217,
    0.13777487,   0.06385251,  0.9535715,   0.89861375,   0.2463547,   0.673834,    0.8008994,   0.0861585,
    0.6613363,    0.79498637,  0.79322547,  0.083214305,  0.577025,    0.58655965,  0.119723536, 0.0012204717};

const std::vector<float> expected_dft1d_results = {
    6.329814,     4.2950764,   -0.8105316,   -0.7187835,   -0.059136264, 0.2709784,     0.82793635,  0.57937646,
    0.5997731,    -1.3291739,  1.188664,     1.462941,     -0.01811248,  -1.8314927,    0.16004556,  -2.219835,
    1.0620322,    -1.0679832,  -0.68610185,  0.658314,     4.627743,     4.5935497,     -0.78950775, -0.32600924,
    -1.4706655,   -1.1615934,  0.708719,     1.4568751,    -1.0970218,   -0.39268675,   -0.5990571,  -0.81545514,
    -0.39174145,  -0.420258,   0.55279106,   2.339339,     -0.59915966,  1.3964193,     -0.8447231,  0.14907542,
    6.2576666,    5.5670385,   0.25636938,   -1.7026355,   1.161571,     0.12042561,    0.19768336,  -1.3421875,
    -0.90698814,  1.4111948,   0.70803046,   0.5795436,    1.2021728,    -0.5199567,    -2.558736,   -0.80762154,
    1.1657354,    -0.8685272,  1.2987087,    -1.0047817,   5.6461143,    3.2111988,     0.2361581,   0.3099669,
    0.6179653,    0.099535145, 1.0438079,    -0.016701937, -0.88529384,  -0.12907594,   0.64785606,  -0.8428119,
    -0.058392793, -1.0488291,  -0.4019828,   0.20333555,   0.45051938,   0.45967662,    1.3713523,   -0.6549525,
    5.5258985,    3.7522945,   -1.8860855,   -0.2230255,   0.8160669,    -0.46607828,   0.123957604, 0.61024696,
    0.26978388,   0.9723815,   0.3050212,    0.69621503,   0.27244493,   -1.0805726,    0.20593566,  1.5653824,
    -0.27690098,  0.8950307,   -0.039584313, -0.18680441,  4.975611,     4.6955333,     0.19031112,  -0.8860659,
    0.91665065,   -0.5264673,  -0.4547393,   1.1623507,    -1.4774656,   1.671129,      1.028168,    -1.6014669,
    -1.2178835,   -0.13447604, -0.14712845,  -0.6739672,   -0.3273949,   -0.9012072,    -0.9661755,  0.03590688,
    4.771964,     5.244689,    -0.03415192,  -0.37281254,  -0.49070793,  -0.65789306,   0.8143984,   -0.8913989,
    -0.19890547,  0.17876014,  -0.9956009,   0.82810897,   0.55270624,   -0.023768127,  1.5358362,   0.6981953,
    0.23165298,   0.51040155,  2.4328363,    0.2267083,    6.4758024,    5.72882,       -0.8707881,  -1.110683,
    0.12478554,   1.3484334,   0.3689712,    0.29180524,   -0.8149491,   -0.0922713,    -0.33161288, 0.78140867,
    -0.9623072,   0.8999919,   -2.1120539,   0.84492886,   -1.5347936,   0.7440938,     1.3312622,   -1.0220959,
    3.8123238,    5.62084,     1.3551373,    0.6460793,    -0.21639234,  -1.2077228,    1.1639122,   -0.05263084,
    0.48105645,   -0.5892652,  0.2349168,    1.128768,     0.42568994,   0.36398163,    -1.2250046,  2.3513904,
    0.64331245,   0.8099514,   1.1574583,    0.8668997,    5.59726,      5.659527,      0.48095328,  0.59446967,
    1.1849049,    1.4709316,   -1.2589264,   -0.11577609,  0.6299068,    -1.4621243,    0.7872094,   0.18096408,
    0.5553762,    -2.0060503,  -0.4373122,   0.9938256,    0.89633095,   -0.5491595,    0.8428093,   0.084472984,
    4.52676,      4.351716,    0.73079205,   0.8098516,    0.27877963,   -0.0073297992, 0.36545974,  0.6745955,
    -2.3818088,   1.5816333,   -0.16544427,  0.51321346,   -0.23699868,  -0.13254744,   1.551896,    0.62098134,
    0.7739359,    1.6108581,   0.36288044,   -0.42423314,  5.0995026,    5.1843014,     -1.1968713,  1.1790991,
    -0.018864498, -0.7500831,  0.0879575,    0.22010106,   1.1136081,    2.2893274,     -0.6877146,  -0.40740123,
    0.046427906,  0.8681825,   -0.50678635,  0.23051873,   0.35328788,   -0.45622703,   0.1495475,   -0.104907334,
    4.8094087,    5.2818966,   0.49697292,   0.29568392,   -0.4144543,   -0.64546454,   0.31737912,  -0.8962374,
    -1.0404948,   0.91764164,  0.6826862,    0.08073502,   0.33942595,   0.053232975,   -1.1867946,  0.51120156,
    -1.1452568,   -1.4197243,  0.82389224,   1.8939058,    6.882805,     6.4072084,     -1.3024135,  -0.22483894,
    -0.22082287,  1.0370905,   -0.7639439,   0.6950346,    -0.731326,    0.16821115,    0.0887468,   -0.5732441,
    -0.40715322,  -0.96244293, -0.89126545,  1.3140129,    -0.42358512,  1.7674587,     -0.6400819,  -1.6113993,
    4.4106574,    5.706909,    -1.1110737,   0.10560027,   -1.1108764,   0.34190884,    2.1167603,   -0.067495525,
    -0.16237324,  0.2604496,   -0.8129095,   -0.42274237,  -1.1412699,   -0.0011268258, -0.63462454, -0.15172139,
    -0.7164279,   0.14801888,  -0.3538928,   1.583736,     4.9876184,    4.2879796,     -0.8491325,  0.5345522,
    -0.60507995,  -0.9020085,  1.0447598,    0.21135187,   -0.4787205,   -0.3230412,    0.8076494,   -0.04361339,
    0.62797767,   0.15487206,  -0.23772183,  0.69546384,   1.8609382,    -1.7030516,    1.2658813,   -0.6791475,
    4.921037,     4.8929176,   -0.0124401,   -0.6873918,   -0.21879943,  -0.48610657,   0.36776963,  0.12423802,
    -0.7854952,   0.48838156,  -0.5085067,   -0.08865434,  1.1653454,    0.81965554,    -0.6399579,  -1.0967884,
    1.4099771,    -0.15370974, 2.8824244,    1.0534087,    4.7045717,    5.2045445,     -0.6350576,  2.5321684,
    0.6987691,    -0.53839976, -0.09889791,  0.5662097,    0.4088725,    0.635128,      -1.763303,   -0.49720347,
    -1.0772469,   1.2422445,   -0.3619956,   -1.311133,    1.5846866,    1.0530244,     -0.61141044, 0.74831486,
    5.433625,     3.9661994,   2.006918,     -0.8703619,   -0.7658511,   0.0811044,     0.83877516,  -0.63553256,
    -0.67563355,  1.7368636,   0.9372277,    1.8246815,    0.8615329,    -0.18161502,   0.62479717,  0.2028623,
    0.159001,     1.860977,    0.04177074,   -0.49050322,  4.9402246,    4.0296063,     -0.74729615, -0.27802998,
    -0.8077982,   -0.5414143,  0.467114,     0.9016136,    2.1971147,    -1.466963,     -1.2350414,  1.0967304,
    -0.95607626,  0.51462483,  0.28838068,   1.0117096,    -0.21846394,  0.114624545,   -1.627146,   -0.9431294};

const std::vector<float> expected_dft2d_results = {
    54.020195,    48.368538,   -1.8721353,   -3.7894967,   2.5850394,   -0.7094516,   3.5357249,    1.6819549,
    -3.4001002,   0.23887074,  2.9735894,    2.3982158,    0.3599546,   -5.801426,    -4.427606,    5.2949734,
    1.7113355,    1.428697,    5.8978443,    -0.8472582,   -3.288164,   -0.099487126, -0.33851182,  2.614974,
    -2.766882,    0.18681616,  0.34976268,   -0.2601711,   4.998401,    -2.9831958,   -1.6652081,   0.53361464,
    -0.9001389,   -3.6454318,  -3.7148805,   -0.68562484,  2.0633714,   -2.2154818,   -3.3697965,   3.5273929,
    1.5474558,    -1.6305131,  -5.3327236,   0.54002213,   -1.6671672,  2.4493377,    -2.2604918,   1.4117424,
    2.1797671,    2.5013056,   0.8525213,    1.6570821,    1.717532,    -2.101283,    4.6570606,    -3.6786642,
    0.8912736,    -0.4010569,  -5.9480867,   1.441097,     2.1150498,   -1.4524796,   -3.5035098,   3.0815587,
    -3.3185432,   4.7882123,   5.64722,      -1.1192517,   1.8302126,   -2.5760055,   -0.41363025,  3.2350469,
    1.4296081,    0.8722873,   6.1752787,    -1.7328868,   2.312786,    4.4069357,    1.7721124,    3.3802934,
    -0.53283703,  3.7646027,   4.440572,     -4.353462,    -2.7639425,  3.6855025,    1.8912748,    -2.5849285,
    -2.9895856,   1.1341677,   1.4818796,    0.7482485,    -1.3077981,  1.0669674,    -0.76039124,  -10.8791685,
    2.998129,     -4.2489543,  0.41752052,   -0.45298803,  -0.62486386, 0.5913104,    -0.36638862,  -0.9528576,
    -0.16223967,  -3.171127,   2.7200532,    -3.8751457,   3.8895426,   1.0489256,    -0.091531515, 6.992935,
    4.5098467,    -0.38218838, 0.6637606,    -2.1199496,   3.9403267,   -0.870952,    2.4287906,    1.9679271,
    3.652341,     -4.4909067,  -1.4710087,   0.5256169,    5.4580984,   -2.6554706,   -0.058261395, 3.6613276,
    0.5612789,    1.0594783,   4.5429516,    -1.447232,    -2.388829,   0.52541757,   -6.1111097,   -2.3621864,
    -1.4885365,   -2.6265867,  -4.4030347,   0.27728367,   3.9584684,   -3.7618577,   -3.128574,    -2.8671994,
    1.4171265,    0.02298975,  -2.0790722,   1.6526843,    0.59488124,  -3.2548752,   -0.82249254,  1.3645289,
    -2.9066925,   -3.4377484,  -2.501403,    -2.821631,    -4.427053,   -2.3529994,   0.6670886,    -4.7455816,
    -2.160026,    -1.0587022,  1.1341916,    -0.9469211,   0.67554307,  -4.0473633,   -1.2422556,   4.538533,
    -0.739814,    -3.22405,    1.2332113,    -4.0489397,   -4.560828,   -3.5195189,   6.7066355,    -2.8439593,
    -0.43901098,  -3.9980454,  -4.2256207,   3.0529652,    4.6105156,   2.720234,     2.3327744,    -1.0400636,
    -0.048398018, 2.1603358,   -0.22459112,  0.6870126,    -0.926849,   -7.2363615,   3.7953386,    3.195907,
    3.8662248,    -1.8919971,  0.91311014,   -0.36923724,  3.0576966,   0.19861764,   -0.09782998,  -1.0179963,
    50.71621,     49.313248,   -2.6195984,   3.396334,     -3.1849973,  -2.4107025,   4.7431326,    1.7938776,
    -2.5362587,   6.287631,    -2.656609,    1.4825039,    -0.77803206, 2.3750808,    -1.9940716,   2.0271082,
    3.6380908,    2.822246,    2.2938647,    1.0260472,    3.248794,    -3.05949,     2.0979533,    3.565119,
    1.9497933,    0.2390036,   -2.255065,    0.7849397,    1.9622431,   4.2382064,    -3.2529292,   0.78572094,
    -2.9386084,   0.66875017,  5.743927,     4.850876,     -4.8014383,  6.371132,     -2.6618924,   -1.8847032,
    -1.7702236,   -1.1031301,  1.4129921,    -0.080709964, -2.7634878,  -3.6456683,   1.4174454,    3.4085226,
    3.10102,      0.114031196, -2.4092412,   0.27725983,   2.8974152,   -1.866328,    -0.68216217,  2.249536,
    -0.42875588,  -5.8182187,  5.347006,     -6.2936745,   0.8000201,   3.651592,     1.3155181,    2.3413098,
    2.1600244,    1.8733575,   -2.4694557,   0.39358342,   2.020084,    -0.062472403, -4.131041,    -1.5137839,
    -2.0354557,   1.1957052,   -0.6644075,   -2.0442688,   2.0753646,   4.874056,     -0.090800405, 1.3911223,
    0.68129027,   -4.0028048,  -0.8021738,   0.43866205,   2.7812133,   0.4525791,    -0.87565154,  1.2364697,
    -2.725146,    2.7965212,   4.148448,     -1.9204504,   -0.61004305, -4.790703,    3.1498234,    0.79403657,
    5.305445,     0.2803253,   -3.67164,     -4.3974924,   -2.5132315,  -0.9139994,   6.841936,     -4.089568,
    -1.2774054,   0.9789283,   3.269153,     -3.3947415,   -7.5553513,  3.682307,     2.9227152,    2.3319635,
    2.754105,     -1.2598821,  1.4247041,    -1.8540356,   -2.675635,   1.2705915,    5.2202816,    6.206577,
    0.4957786,    2.1150033,   5.8791704,    2.8043785,    -0.37886655, 0.011162788,  -1.0408137,   -1.5385519,
    -8.079001,    -0.68451786, 2.3513699,    3.0877895,    2.6497078,   1.3670976,    0.77233493,   2.2921152,
    -1.2679763,   2.113087,    4.990262,     -0.046566606, 0.015865922, 1.1569002,    -4.8347507,   1.9560149,
    1.979923,     2.34512,     -0.9634773,   4.3939066,    -6.2031984,  0.8311275,    -2.7740612,   -2.9296994,
    -3.4624243,   -1.4588313,  2.4724,       -0.79477566,  -0.4295609,  5.8110385,    -2.6649034,   -2.270977,
    -2.5511568,   -3.1238616,  -4.46209,     0.16335368,   1.9146351,   1.0459399,    2.8069792,    -0.4705832,
    -4.0632596,   -2.220704,   1.7770543,    -0.5791014,   -2.2041528,  3.026476,     5.324942,     -0.7805673,
    5.9275556,    0.14159381,  -0.81569004,  4.1947803,    -3.8557377,  -0.5163199,   2.478963,     -2.396379,
    -0.3930376,   -0.96302,    -0.9776549,   0.13852966,   0.26078847,  0.8342015,    2.3698487,    4.109933,
    1.3575013,    -0.5828376,  -0.028537825, -0.53020877,  0.39626116,  -1.7572733,   -4.31769,     -2.1674476};

const std::vector<float> expected_dft3d_results = {
    104.7364,    97.68179,     -4.491728,    -0.39316452, -0.59995466, -3.1201572,  8.278858,    3.4758341,
    -5.9363585,  6.5265055,    0.3169801,    3.8807175,   -0.418082,   -3.4263492,  -6.4216776,  7.3220854,
    5.3494234,   4.2509427,    8.191702,     0.17879319,  -0.03937006, -3.1589758,  1.7594413,   6.180092,
    -0.8170867,  0.42582142,   -1.9053001,   0.52476853,  6.9606423,   1.255014,    -4.9181366,  1.319335,
    -3.838747,   -2.9766817,   2.0290484,    4.16525,     -2.7380676,  4.155652,    -6.0316873,  1.6426877,
    -0.2227689,  -2.7336447,   -3.919732,    0.45931256,  -4.4306555,  -1.1963288,  -0.8430467,  4.8202653,
    5.280785,    2.6153364,    -1.556721,    1.9343407,   4.614946,    -3.96761,    3.9748988,   -1.4291265,
    0.46251905,  -6.2192726,   -0.60107887,  -4.852579,   2.9150705,   2.1991146,   -2.1879911,  5.4228687,
    -1.158518,   6.661569,     3.1777658,    -0.7256692,  3.8502965,   -2.6384768,  -4.544671,   1.721262,
    -0.6058461,  2.067991,     5.5108714,    -3.7771575,  4.388153,    9.280992,    1.681312,    4.7714148,
    0.14845347,  -0.23820269,  3.6383984,    -3.9147997,  0.017270446, 4.138083,    1.0156215,   -1.3484575,
    -5.7147317,  3.9306912,    5.630328,     -1.1722009,  -1.9178381,  -3.7237349,  2.3894331,   -10.085134,
    8.303572,    -3.9686286,   -3.2541199,   -4.850478,   -3.1380959,  -0.32268947, 6.475547,    -5.0424256,
    -1.4396465,  -2.1921992,   5.9892044,    -7.269888,   -3.665809,   4.7312326,   2.8311844,   9.324896,
    7.2639513,   -1.6420703,   2.0884657,    -3.9739842,  1.2646922,   0.39964193,  7.649071,    8.174507,
    4.148118,    -2.3759027,   4.4081597,    3.3299959,   5.0792284,   -2.6443086,  -1.0990746,  2.1227744,
    -7.517721,   0.3749615,    6.894322,     1.6405574,   0.26087707,  1.8925169,   -5.3387756,  -0.07007182,
    -2.7565134,  -0.51350284,  0.5872268,    0.23071745,  3.9743357,   -2.6049578,  -7.963324,   -0.9111862,
    3.3970497,   2.368112,     -3.0425484,   6.0465913,   -5.608317,   -2.4237492,  -3.5965526,  -1.5651696,
    -6.369116,   -4.896579,    -0.029001951, -3.616405,   -4.8566127,  3.4580388,   -1.9978137,  -7.016559,
    -4.71118,    -4.1825647,   -3.3278992,   -0.7835678,  2.5901778,   -3.0014238,  1.5647203,   4.06795,
    -4.803074,   -5.444754,    3.0102665,    -4.6280394,  -6.764982,   -0.49304247, 12.031577,   -3.6245267,
    5.488541,    -3.8564541,   -5.04131,     7.2477474,   0.7547778,   2.2039144,   4.8117356,   -3.4364424,
    -0.44143593, 1.1973162,    -1.2022457,   0.8255428,   -0.66605973, -6.4021583,  6.1651874,   7.3058405,
    5.2237253,   -2.4748354,   0.88457155,   -0.89944726, 3.453958,    -1.558656,   -4.4155188,  -3.1854444,
    3.303988,    -0.9447114,   0.7474582,    -7.185831,   5.770039,    1.7012511,   -1.2074116,  -0.11192033,
    -0.86384296, -6.048759,    5.6302013,    0.9157127,   1.1379871,   -8.176507,   -2.433535,   3.2678652,
    -1.9267552,  -1.393548,    3.6039736,    -1.873306,   -6.536957,   2.9600024,   -2.4364662,  -0.95014465,
    -4.716674,   -0.052186966, 2.6048284,    -1.0451086,  3.036159,    -7.221403,   1.5877211,   -0.25210607,
    2.0384693,   -4.3141813,   -9.458808,    -5.5365014,  6.8648105,   -8.586614,   -0.7079052,  5.412094,
    3.3176801,   -0.5273831,   -6.745717,    0.62073076,  1.0963198,   6.0950055,   -3.677938,   -1.9967818,
    -0.921252,   2.387275,     3.261763,     1.3798212,   -1.1798835,  -0.23495495, 5.339221,    -5.928199,
    1.3200281,   5.417163,     -11.295093,   7.7347717,   1.3150296,   -5.1040716,  -4.8190293,  0.74024755,
    -5.4785676,  2.914854,     8.116676,     -1.5128357,  -0.1898706,  -2.5135324,  3.7174103,   4.7488313,
    3.4650638,   -0.32341766,  6.8396864,    0.31138325,  0.2374219,   -0.46712062, 1.8629129,   1.9891711,
    -1.2141278,  7.7674093,    5.2427464,    -4.792124,   -5.5451555,  3.2329237,   2.766926,    -3.8213987,
    -0.26443875, -1.6623533,   -2.6665692,   2.6686997,   -0.6977545,  5.85767,     -3.9102163,  -11.673204,
    -2.3073153,  -4.529278,    4.0891604,    3.9445055,   1.8883687,   1.50531,     -7.2083244,  3.1367111,
    1.1151649,   -4.1500554,   -0.54910004,  -0.48040384, 11.444895,   -2.6333811,  -3.0142484,  4.6609726,
    1.755743,    0.87769306,   -0.7609439,   -0.26591438, 6.615961,    -2.141545,   -2.7914915,  -4.2386503,
    3.1565619,   -6.6059103,   -7.35018,     -2.2787585,  5.836963,    -2.6666338,  0.98255026,  5.199881,
    8.640279,    1.7439961,    2.191582,     -4.535021,   -5.038538,   -0.841679,   -6.8834453,  -4.654301,
    -0.220559,   -4.7396717,   -9.393296,    0.32385087,  3.9426038,   -4.9187584,  1.7061774,   -4.8232145,
    -0.5627973,  -2.3221302,   -1.1155958,   -2.7412212,  6.798079,    -4.0860014,  1.9515686,   4.2942266,
    0.5557329,   -1.9789174,   -4.973804,    -2.0268555,  -3.9974911,  -8.164038,   3.3319929,   -2.474605,
    0.39113098,  2.0651584,    5.5962815,    -1.1102749,  -1.2390921,  -5.0933027,  -4.0492353,  5.009116,
    3.323446,    -1.0033474,   -0.54384375,  -3.4698372,  -2.3566747,  -6.545992,   1.3816929,   -2.0633929,
    -6.3665648,  -4.13964,     -3.4099324,   -1.1418146,  8.466255,    3.2365537,   -0.14618888, 1.3563147,
    0.3446387,   3.1233552,    0.7530624,    0.548483,    -1.1876376,  -8.070564,   1.4254899,   -0.9140264,
    2.5087235,   -1.3091599,   0.9416502,    0.16097029,  2.6614356,   1.9558911,   4.219861,    1.1494511};

const std::vector<float> expected_dft1d_signal_size_results = {
    6.138384,      4.8263664,    6.2014966,    4.641298,     6.2220087,    3.340786,    3.8338857,   3.458686,
    6.393098,      6.578215,     4.9169006,    3.8786886,    5.0566025,    5.701084,    5.099263,    6.690686,
    4.686806,      4.9369535,    5.471756,     4.315829,     3.6622288,    -4.547995,   2.3657713,   -4.4210963,
    3.3341353,     -3.560755,    3.0613456,    -2.0019536,   4.9098253,    -3.27648,    3.6913419,   -2.365142,
    5.2827687,     -3.2966752,   5.633893,     -2.990755,    2.4099903,    -2.5792742,  3.009334,    -3.318112,
    -0.8632047,    -0.302661,    -0.9658433,   1.3884914,    -0.12056512,  -0.5264965,  0.053616166, 0.5239285,
    -0.37204745,   0.6682581,    0.88452375,   -1.4486976,   -0.9331777,   -0.6865864,  -0.32639223, -1.2646291,
    -0.187691,     1.0645473,    -0.45738214,  0.48435384,   1.2216191,    -0.61395854, 2.4932637,   -1.6152657,
    0.99030006,    -0.45764852,  2.4245698,    0.31936115,   2.9254415,    0.4994774,   0.2869299,   -0.82977176,
    1.759331,      0.66088116,   2.0010936,    -0.18261093,  1.5729225,    -0.6416664,  1.2219726,   -0.4185537,
    -0.33628678,   0.21890742,   -2.2292616,   -0.9053817,   0.53581333,   0.36471185,  0.90989465,  -0.067255855,
    0.9978778,     -0.6023144,   1.2700583,    -0.055348396, 0.7769584,    0.20883593,  0.68046755,  -1.3861976,
    -0.7743764,    -0.17685926,  -0.28369236,  0.7703819,    0.88469267,   0.7845876,   0.4245007,   1.0558772,
    1.5855613,     -0.88230014,  2.5918908,    0.5176118,    0.9809585,    -0.16784734, 0.44176394,  -1.8440124,
    1.8485708,     0.13407728,   0.99209386,   -0.49863797,  -0.05547434,  0.51047516,  0.95244277,  0.16795856,
    1.4046597,     1.2883723,    -0.4217211,   -0.30820233,  -0.94360316,  -1.0276735,  1.8597782,   -1.7973311,
    0.17352016,    0.14103556,   -0.53083634,  -0.08058083,  0.58797073,   -0.1623843,  1.0300912,   -1.594127,
    -0.37183756,   0.6519355,    -0.67296886,  1.4364773,    2.9115105,    -0.62121296, 0.10118961,  0.4055282,
    -0.765203,     1.1095873,    0.25468233,   -0.8044969,   0.37809694,   0.47051764,  -0.5070367,  -0.69294405,
    1.678687,      -0.05850029,  -0.15289319,  -2.1158576,   -0.28707075,  0.64672077,  2.1430318,   1.8936268,
    0.287481,      -1.212002,    -0.8066146,   -0.024840236, 0.4002909,    1.5536453,   0.90662,     -0.1200099,
    0.2907222,     1.3641009,    -1.2066911,   2.2924597,    -0.10727149,  -0.90693283, -1.7303126,  -0.9309965,
    -0.39670166,   1.4576144,    1.8296418,    0.29156286,   0.914652,     0.48379737,  0.35427743,  1.0552206,
    1.0729686,     0.66110367,   1.1590368,    -0.883232,    1.5702324,    0.37410414,  2.7553983,   1.3418052,
    0.4280968,     0.43797877,   -0.42501903,  0.6896758,    0.17888534,   0.7881189,   1.906157,    -0.893877,
    1.6907314,     -0.07711154,  -0.08057277,  -0.94700074,  0.118160814,  1.0535691,   0.013901293, -1.0134851,
    -0.49273467,   0.77010435,   0.61979324,   -0.4796943,   -0.9006692,   -0.14570916, 0.20728627,  -0.6043751,
    -0.77368677,   2.1912723,    -1.0270727,   -0.15626097,  1.6779256,    -1.3633705,  -1.419299,   0.4458414,
    1.8119955,     1.3894738,    -0.0533064,   -0.2651497,   2.156881,     1.774823,    1.6229321,   0.83354133,
    0.6217755,     2.7520647,    -0.8899409,   -0.5549828,   2.2334838,    1.866106,    2.2245884,   1.6349602,
    -0.17061183,   -0.75332606,  -0.7192313,   1.011065,     1.1424315,    -0.14698744, -0.5063292,  0.047560766,
    0.8158023,     -0.99613833,  1.3294827,    -0.884288,    1.9334476,    -0.82400334, -1.0622213,  0.45689362,
    0.3765804,     -0.2486183,   0.5129931,    -2.1530728,   1.6927868,    -0.4909035,  0.07076289,  1.1461573,
    1.2108444,     0.5570269,    0.57290983,   1.0781552,    0.2522127,    0.9315722,   0.82063264,  -0.27504963,
    -1.298832,     0.5996642,    0.035723105,  0.9061513,    0.7085681,    1.3870897,   -0.33059365, 1.4296496,
    -0.9227723,    -2.0201705,   -0.25814092,  -0.044265598, 0.74225616,   -0.7740435,  0.56227565,  -0.7865786,
    0.16598742,    -0.13509351,  0.65011877,   -0.5367288,   0.7644322,    1.754046,    0.14904708,  0.0060333014,
    0.81808805,    -0.023402452, 1.2871823,    -1.2016544,   -0.016474128, 1.0952724,   -0.83657134, 0.959798,
    -0.29334623,   0.46025404,   -1.329956,    0.88328505,   0.311208,     1.5458176,   1.058334,    -0.65749556,
    0.7922486,     1.2470598,    0.009132266,  0.07870856,   0.6166347,    0.009361565, -1.6813973,  0.3131196,
    -0.3617983,    -1.6096013,   -0.80183095,  0.60364366,   0.032118678,  0.53880775,  0.79869264,  2.0884013,
    0.30808622,    -1.1033678,   -1.0830308,   -1.5599371,   1.2167512,    0.439706,    -0.76799685, -0.46132383,
    -1.6585693,    -0.8193617,   0.15754253,   0.82434106,   -1.4365332,   2.5602462,   -0.59798455, 2.2706695,
    0.094361365,   1.5161843,    1.576273,     0.8282173,    -2.615784,    2.0659475,   -0.70808023, 1.8205551,
    -0.23570198,   1.0002637,    -0.84214133,  1.1558707,    -0.8486479,   3.3955946,   -0.9163475,  1.2400286,
    1.7278013,     -0.2593556,   0.12464893,   0.045035288,  0.14191893,   0.60069644,  0.6033013,   -0.40642756,
    0.30952126,    2.1911335,    0.38403878,   -0.5504798,   0.7629653,    0.96752846,  -0.77223957, -0.45594752,
    1.2607243,     -0.5419304,   0.06783953,   1.1299804,    -2.9180245,   2.812955,    -2.912982,   4.157113,
    -0.7707863,    4.184089,     -1.2218096,   2.2556906,    -2.2792397,   4.6580005,   -2.2278588,  3.2439072,
    -1.7189349,    2.8687704,    -3.8549495,   3.9684548,    -3.5499556,   3.1096249,   -1.6433489,  3.6931372,
    4.762172,      6.8113427,    5.6586823,    3.9343526,    4.874974,     4.044377,    4.5118494,   4.560476,
    4.814545,      5.255967,     4.8088293,    4.8661695,    5.5842476,    3.047568,    6.3495092,   5.8194113,
    3.9938629,     6.2386484,    5.357541,     4.734993,     4.009847,     -1.85078,    3.257053,    -2.863433,
    3.2807407,     -2.4543116,   2.3266344,    -2.7742774,   5.0006027,    -3.1107163,  3.1461582,   -2.4130437,
    1.9839633,     -3.2893786,   4.9680586,    -1.5064957,   4.93627,      -2.2536325,  4.4328127,   -2.371289,
    -0.09072271,   1.6559569,    0.9273602,    -0.16627279,  -0.15694867,  -0.16094846, -0.30682713, 0.62848985,
    -0.16314042,   -1.793005,    -0.025120497, 0.035565466,  0.4509227,    1.029233,    2.6076002,   -1.3557681,
    -0.042177677,  -1.8681216,   0.047852248,  -1.0646176,   3.5719476,    0.61097944,  1.9404714,   -1.8996478,
    1.4736449,     -0.3556636,   0.7955406,    -0.70645106,  0.106241465,  -1.4871876,  1.4906516,   -0.5542434,
    1.8523693,     -1.8795702,   0.20563364,   -1.7517232,   -0.2156514,   -0.42685848, 1.2532125,   0.29982674,
    0.6122022,     -1.2758396,   -0.7942943,   -1.2208992,   -0.28703833,  -0.6799724,  -0.22529805, 0.88721895,
    -1.6784416,    0.6357958,    -0.40500844,  -1.1880269,   -1.3736604,   0.27873987,  0.9415196,   1.5691454,
    0.637203,      0.6283375,    0.8025869,    -0.73763883,  0.07619148,   0.29328048,  -0.21484284, 0.39326593,
    0.2614212,     0.25093663,   1.1460452,    -0.42564535,  1.2621714,    0.7867665,   -0.9763881,  0.67735475,
    1.3954227,     0.8466128,    2.6533723,    -1.3165393,   1.0205896,    -1.2907634,  -0.09324902, 0.19477385,
    -0.10201472,   1.2100208,    0.8927874,    1.1437554,    -0.27520463,  -0.18907726, -0.1451918,  0.3773734,
    -1.0439434,    -0.35780138,  1.1060231,    1.0964282,    0.2501399,    0.31307727,  -0.13760762, -0.86377877,
    -0.49448854,   0.09268577,   0.74952096,   0.82891256,   1.9546115,    1.2895672,   2.1169534,   -1.0286292,
    0.0854094,     0.63047266,   1.0325564,    -1.0493125,   0.31182784,   2.3592472,   0.69874203,  -0.92576516,
    1.5970948,     0.7720525,    0.9282105,    -0.13239098,  1.5795128,    -0.7387508,  0.9950645,   0.11649346,
    0.7381579,     -0.9112861,   -1.0479062,   -0.9858517,   -0.31902313,  -0.43754727, -1.9271741,  0.41418013,
    1.5709126,     0.12488553,   0.34839654,   -0.14153089,  1.2988961,    -0.17553245, 0.36363417,  -0.4992725,
    -0.87406987,   -1.5621511,   0.52947193,   0.17129752,   -0.19691896,  0.88363117,  0.5983897,   1.4228462,
    -1.309372,     1.6419725,    2.096242,     1.3451272,    0.21495643,   0.16032922,  -0.21870668, -2.3123596,
    1.511457,      -1.2067473,   0.30244982,   -0.5896039,   -0.20020528,  -0.17678946, 0.646347,    0.12540624,
    0.8411275,     0.29581466,   1.0424525,    -0.3198546,   1.5812268,    1.633207,    0.036333233, -1.9386438,
    0.4908937,     0.4255972,    -3.0946343,   0.4557737,    -1.538063,    -1.0618666,  -0.766645,   0.09507492,
    -1.1704439,    -0.58377063,  0.06451824,   0.084664315,  -0.33639127,  0.43388176,  0.7445558,   0.56547284,
    0.20360313,    -0.52746487,  -0.22392502,  0.10510802,   0.2932141,    0.13039428,  0.2339833,   1.1078603,
    0.07111454,    1.674398,     0.24977088,   0.7773813,    0.10618341,   1.3232847,   0.07770634,  0.8410483,
    0.6371973,     1.1520991,    1.6076822,    -0.553284,    0.0399023,    1.6575105,   -1.002435,   -1.153806,
    -0.338341,     0.75674164,   -1.9532704,   -0.16773497,  -0.09083623,  -0.09499304, -0.15297657, 0.6092089,
    1.1756519,     -0.8699633,   0.57320195,   0.77921844,   0.38325477,   -0.4647501,  0.16934802,  -0.9541189,
    1.8387299,     -0.2722485,   -0.9011179,   1.2189366,    1.0526755,    1.2198145,   -0.66335034, 2.4358046,
    -0.0044496655, 2.4705029,    0.7398137,    1.1581391,    -0.08892931,  -1.3800118,  0.39516693,  0.7783501,
    -1.6864198,    0.90398335,   -0.09139767,  0.18852606,   -0.7292757,   -0.7595531,  -0.30982962, -0.37637365,
    0.27046034,    -0.2601264,   0.06654024,   0.83308995,   2.1443768,    0.7846114,   0.72724646,  0.43702295,
    -1.3782393,    -1.555314,    1.0024056,    0.96103704,   0.62146187,   2.4383464,   0.97525114,  0.1517681,
    -0.05941461,   0.20787807,   -0.7399595,   1.4447442,    0.370912,     1.5718691,   0.36367816,  1.2211394,
    1.2772232,     0.46179056,   1.0423609,    -0.1160976,   -1.8006848,   0.2063675,   0.699636,    -0.2978242,
    0.36548108,    0.13973325,   0.06818205,   -0.8364538,   -1.8770711,   -0.46342957, 0.5138623,   -0.7012725,
    1.3353106,     -0.7529058,   -0.5607584,   -0.3658438,   1.3651763,    -0.8271546,  -1.3937892,  -0.4218138,
    -1.5759501,    0.052277893,  -0.79160595,  1.0530065,    -0.25257057,  1.7259041,   -0.09510864, 0.31564656,
    -1.4286227,    2.806394,     -2.0088015,   0.6337404,    -1.4553217,   0.3904129,   -0.8321003,  2.0365574,
    -0.47588396,   0.03407097,   0.08727789,   2.440409,     -1.3018095,   1.9136591,   1.5979958,   1.496789,
    -0.2709299,    -0.38308293,  -1.0800201,   -0.7544405,   0.074904405,  1.2379612,   -0.62439823, 0.5188385,
    -0.05306366,   1.060843,     -0.17591527,  -0.21396813,  -0.27043432,  0.16332026,  -0.57039225, -0.76971775,
    -0.22342275,   -0.28223512,  -0.66207,     -1.0938429,   -4.0251827,   4.238682,    -2.3085427,  4.3264065,
    -1.419694,     3.9545622,    -3.0023227,   3.424511,     -1.9520879,   3.0750623,   -3.127586,   3.9366179,
    -1.3875456,    3.5732715,    -3.2088501,   5.656434,     -3.9873497,   3.1138892,   -2.331269,   4.533456};

const std::vector<float> expected_dft1d_bfloat16_results = {
    6.3125,    4.28125,   -0.804688, -0.722656,  -0.0551758, 0.271484,    0.832031,  0.578125,    0.601563,
    -1.32031,  1.1875,    1.45313,   -0.0197754, -1.82813,   0.15332,     -2.21875,  1.05469,     -1.0625,
    -0.6875,   0.660156,  4.625,     4.5625,     -0.785156,  -0.332031,   -1.46875,  -1.15625,    0.710938,
    1.45313,   -1.09375,  -0.394531, -0.59375,   -0.8125,    -0.388672,   -0.419922, 0.546875,    2.32813,
    -0.59375,  1.39063,   -0.84375,  0.143555,   6.25,       5.5625,      0.251953,  -1.70313,    1.16406,
    0.120117,  0.195313,  -1.34375,  -0.90625,   1.40625,    0.699219,    0.574219,  1.19531,     -0.515625,
    -2.5625,   -0.804688, 1.15625,   -0.859375,  1.28906,    -1,          5.625,     3.21875,     0.240234,
    0.308594,  0.617188,  0.0947266, 1.04688,    -0.0205078, -0.875,      -0.126953, 0.640625,    -0.84375,
    -0.050293, -1.04688,  -0.40625,  0.207031,   0.443359,   0.458984,    1.375,     -0.65625,    5.5,
    3.75,      -1.88281,  -0.226563, 0.816406,   -0.464844,  0.121582,    0.609375,  0.269531,    0.960938,
    0.304688,  0.695313,  0.273438,  -1.07813,   0.207031,   1.5625,      -0.277344, 0.890625,    -0.0373535,
    -0.185547, 4.9375,    4.6875,    0.191406,   -0.882813,  0.914063,    -0.53125,  -0.455078,   1.16406,
    -1.46875,  1.66406,   1.01563,   -1.59375,   -1.21875,   -0.126953,   -0.137695, -0.671875,   -0.324219,
    -0.902344, -0.960938, 0.0281982, 4.75,       5.25,       -0.034668,   -0.378906, -0.492188,   -0.65625,
    0.816406,  -0.890625, -0.201172, 0.173828,   -0.996094,  0.828125,    0.554688,  -0.020752,   1.53125,
    0.691406,  0.227539,  0.503906,  2.42188,    0.220703,   6.4375,      5.6875,    -0.867188,   -1.10156,
    0.128906,  1.34375,   0.363281,  0.289063,   -0.8125,    -0.0976563,  -0.328125, 0.78125,     -0.960938,
    0.898438,  -2.09375,  0.847656,  -1.53125,   0.742188,   1.32813,     -1.03125,  3.79688,     5.625,
    1.34375,   0.640625,  -0.213867, -1.20313,   1.15625,    -0.0522461,  0.476563,  -0.585938,   0.228516,
    1.125,     0.421875,  0.363281,  -1.21875,   2.34375,    0.644531,    0.804688,  1.15625,     0.863281,
    5.59375,   5.65625,   0.484375,  0.59375,    1.17969,    1.46875,     -1.25781,  -0.115723,   0.628906,
    -1.46875,  0.789063,  0.179688,  0.554688,   -2,         -0.435547,   0.992188,  0.898438,    -0.546875,
    0.847656,  0.0820313, 4.5,       4.3125,     0.726563,   0.8125,      0.273438,  -0.00793457, 0.365234,
    0.671875,  -2.375,    1.57813,   -0.167969,  0.511719,   -0.239258,   -0.128906, 1.54688,     0.625,
    0.769531,  1.60938,   0.363281,  -0.417969,  5.09375,    5.1875,      -1.1875,   1.17188,     -0.0154419,
    -0.746094, 0.0834961, 0.225586,  1.10938,    2.28125,    -0.6875,     -0.410156, 0.0449219,   0.867188,
    -0.507813, 0.229492,  0.353516,  -0.457031,  0.145508,   -0.108887,   4.78125,   5.25,        0.498047,
    0.296875,  -0.410156, -0.644531, 0.320313,   -0.898438,  -1.03125,    0.914063,  0.675781,    0.0810547,
    0.335938,  0.0527344, -1.1875,   0.503906,   -1.14063,   -1.42188,    0.820313,  1.89063,     6.875,
    6.375,     -1.29688,  -0.229492, -0.220703,  1.04688,    -0.765625,   0.6875,    -0.734375,   0.173828,
    0.0917969, -0.574219, -0.408203, -0.953125,  -0.890625,  1.3125,      -0.421875, 1.75781,     -0.640625,
    -1.59375,  4.40625,   5.6875,    -1.10938,   0.103516,   -1.10938,    0.34375,   2.10938,     -0.0664063,
    -0.164063, 0.261719,  -0.808594, -0.414063,  -1.14063,   -0.00567627, -0.625,    -0.146484,   -0.710938,
    0.149414,  -0.363281, 1.57813,   4.96875,    4.28125,    -0.84375,    0.53125,   -0.601563,   -0.90625,
    1.04688,   0.213867,  -0.472656, -0.320313,  0.808594,   -0.0415039,  0.632813,  0.15625,     -0.238281,
    0.695313,  1.85938,   -1.69531,  1.25781,    -0.679688,  4.90625,     4.875,     -0.00488281, -0.6875,
    -0.213867, -0.488281, 0.367188,  0.118164,   -0.78125,   0.488281,    -0.5,      -0.0839844,  1.15625,
    0.820313,  -0.640625, -1.09375,  1.40625,    -0.148438,  2.875,       1.04688,   4.6875,      5.1875,
    -0.632813, 2.53125,   0.695313,  -0.539063,  -0.09375,   0.566406,    0.410156,  0.632813,    -1.75781,
    -0.5,      -1.07813,  1.23438,   -0.355469,  -1.3125,    1.57813,     1.04688,   -0.613281,   0.742188,
    5.4375,    3.95313,   2,         -0.863281,  -0.765625,  0.0791016,   0.835938,  -0.632813,   -0.671875,
    1.73438,   0.9375,    1.82031,   0.855469,   -0.178711,  0.621094,    0.206055,  0.15918,     1.85938,
    0.0454102, -0.488281, 4.9375,    4,          -0.746094,  -0.277344,   -0.804688, -0.539063,   0.460938,
    0.898438,  2.1875,    -1.46875,  -1.23438,   1.09375,    -0.953125,   0.515625,  0.291016,    1.01563,
    -0.22168,  0.113281,  -1.625,    -0.945313};

const std::vector<float> expected_dft2d_bfloat16_results = {
    54,         48.25,      -1.85938,   -3.8125,    2.59375,    -0.714844, 3.53125,    1.67188,    -3.39062,
    0.214844,   2.95312,    2.39062,    0.369141,   -5.78125,   -4.4375,   5.3125,     1.70312,    1.41406,
    5.875,      -0.875,     -3.25,      -0.0917969, -0.34375,   2.59375,   -2.75,      0.199219,   0.355469,
    -0.271484,  5,          -2.96875,   -1.65625,   0.539062,   -0.90625,  -3.65625,   -3.71875,   -0.671875,
    2.0625,     -2.1875,    -3.34375,   3.53125,    1.53125,    -1.60938,  -5.3125,    0.53125,    -1.66406,
    2.4375,     -2.25,      1.42188,    2.17188,    2.5,        0.867188,  1.65625,    1.71875,    -2.09375,
    4.625,      -3.67188,   0.890625,   -0.412109,  -5.9375,    1.46875,   2.125,      -1.4375,    -3.48438,
    3.09375,    -3.29688,   4.78125,    5.65625,    -1.11719,   1.82812,   -2.5625,    -0.386719,  3.21875,
    1.42969,    0.859375,   6.125,      -1.73438,   2.28125,    4.375,     1.76562,    3.375,      -0.535156,
    3.75,       4.4375,     -4.3125,    -2.76562,   3.67188,    1.89062,   -2.59375,   -2.96875,   1.14062,
    1.46875,    0.75,       -1.3125,    1.0625,     -0.765625,  -10.875,   2.96875,    -4.21875,   0.417969,
    -0.457031,  -0.625,     0.585938,   -0.388672,  -0.980469,  -0.147461, -3.15625,   2.71875,    -3.875,
    3.875,      1.04688,    -0.0986328, 7,          4.5,        -0.378906, 0.648438,   -2.125,     3.9375,
    -0.859375,  2.40625,    1.98438,    3.65625,    -4.5,       -1.45312,  0.53125,    5.4375,     -2.67188,
    -0.0605469, 3.67188,    0.546875,   1.07812,    4.5,        -1.46094,  -2.39062,   0.539062,   -6.0625,
    -2.34375,   -1.46875,   -2.60938,   -4.375,     0.283203,   3.96875,   -3.78125,   -3.10938,   -2.85938,
    1.40625,    0.0375977,  -2.07812,   1.64062,    0.601562,   -3.25,     -0.820312,  1.35938,    -2.89062,
    -3.4375,    -2.51562,   -2.8125,    -4.4375,    -2.34375,   0.664062,  -4.75,      -2.125,     -1.07812,
    1.15625,    -0.953125,  0.65625,    -4.03125,   -1.21875,   4.5625,    -0.734375,  -3.21875,   1.25,
    -4.03125,   -4.5625,    -3.51562,   6.6875,     -2.84375,   -0.429688, -4,         -4.1875,    3.01562,
    4.59375,    2.6875,     2.34375,    -1.03906,   -0.0419922, 2.17188,   -0.214844,  0.695312,   -0.921875,
    -7.1875,    3.79688,    3.1875,     3.84375,    -1.89062,   0.898438,  -0.371094,  3.04688,    0.197266,
    -0.102539,  -1,         50.5,       49,         -2.59375,   3.39062,   -3.17188,   -2.40625,   4.75,
    1.78906,    -2.51562,   6.28125,    -2.64062,   1.48438,    -0.789062, 2.375,      -1.98438,   2.03125,
    3.625,      2.8125,     2.28125,    1.01562,    3.25,       -3.03125,  2.0625,     3.5625,     1.96094,
    0.248047,   -2.26562,   0.792969,   1.96094,    4.25,       -3.25,     0.78125,    -2.9375,    0.667969,
    5.71875,    4.84375,    -4.8125,    6.34375,    -2.64062,   -1.85938,  -1.75781,   -1.09375,   1.42188,
    -0.0986328, -2.76562,   -3.65625,   1.42188,    3.40625,    3.09375,   0.113281,   -2.40625,   0.291016,
    2.90625,    -1.85938,   -0.695312,  2.26562,    -0.425781,  -5.8125,   5.3125,     -6.28125,   0.8125,
    3.625,      1.3125,     2.34375,    2.14062,    1.89062,    -2.4375,   0.382812,   2,          -0.0454102,
    -4.125,     -1.51562,   -2.04688,   1.19531,    -0.65625,   -2.03125,  2.0625,     4.875,      -0.0996094,
    1.42188,    0.648438,   -4,         -0.8125,    0.445312,   2.78125,   0.4375,     -0.867188,  1.25,
    -2.70312,   2.8125,     4.125,      -1.9375,    -0.585938,  -4.75,     3.14062,    0.796875,   5.3125,
    0.277344,   -3.64062,   -4.375,     -2.51562,   -0.925781,  6.8125,    -4.0625,    -1.28125,   0.972656,
    3.26562,    -3.40625,   -7.5625,    3.6875,     2.90625,    2.34375,   2.73438,    -1.26562,   1.41406,
    -1.83594,   -2.65625,   1.29688,    5.1875,     6.1875,     0.484375,  2.10938,    5.875,      2.79688,
    -0.386719,  0.00540161, -1.01562,   -1.54688,   -8.0625,    -0.679688, 2.34375,    3.07812,    2.625,
    1.375,      0.75,       2.26562,    -1.28125,   2.125,      4.96875,   -0.0222168, 0.0286865,  1.15625,
    -4.8125,    1.95312,    1.96094,    2.34375,    -0.984375,  4.375,     -6.1875,    0.828125,   -2.75,
    -2.92188,   -3.45312,   -1.45312,   2.46875,    -0.789062,  -0.433594, 5.8125,     -2.65625,   -2.26562,
    -2.54688,   -3.125,     -4.4375,    0.167969,   1.92188,    1.04688,   2.79688,    -0.453125,  -4.0625,
    -2.21875,   1.78125,    -0.570312,  -2.1875,    3.01562,    5.3125,    -0.765625,  5.9375,     0.157227,
    -0.8125,    4.1875,     -3.84375,   -0.523438,  2.46875,    -2.375,    -0.408203,  -0.953125,  -0.984375,
    0.144531,   0.253906,   0.816406,   2.34375,    4.0625,     1.34375,   -0.574219,  -0.0258789, -0.53125,
    0.390625,   -1.75,      -4.3125,    -2.15625};

const std::vector<float> expected_dft3d_bfloat16_results = {
    104.5,      97.5,      -4.4375,   -0.414063, -0.582031,  -3.125,     8.25,      3.46875,   -5.90625,  6.5,
    0.3125,     3.875,     -0.419922, -3.40625,  -6.40625,   7.3125,     5.34375,   4.25,      8.125,     0.138672,
    -0.0197754, -3.125,    1.71875,   6.1875,    -0.796875,  0.445313,   -1.90625,  0.523438,  6.9375,    1.25,
    -4.9375,    1.32031,   -3.84375,  -2.98438,  2,          4.15625,    -2.75,     4.125,     -6,        1.66406,
    -0.223633,  -2.70313,  -3.90625,  0.435547,  -4.4375,    -1.21875,   -0.832031, 4.8125,    5.25,      2.625,
    -1.54688,   1.95313,   4.625,     -3.95313,  3.9375,     -1.40625,   0.464844,  -6.1875,   -0.632813, -4.8125,
    2.9375,     2.1875,    -2.17188,  5.4375,    -1.15625,   6.6875,     3.20313,   -0.734375, 3.84375,   -2.60938,
    -4.5,       1.70313,   -0.617188, 2.0625,    5.5,        -3.78125,   4.34375,   9.25,      1.67188,   4.8125,
    0.115234,   -0.242188, 3.60938,   -3.90625,  0.0158691,  4.125,      1.01563,   -1.35156,  -5.6875,   3.95313,
    5.625,      -1.1875,   -1.89063,  -3.71875,  2.375,      -10,        8.25,      -3.9375,   -3.21875,  -4.84375,
    -3.14063,   -0.341797, 6.4375,    -5.0625,   -1.42969,   -2.1875,    6,         -7.25,     -3.67188,  4.71875,
    2.8125,     9.3125,    7.21875,   -1.64063,  2.0625,     -3.96875,   1.26563,   0.4375,    7.625,     8.1875,
    4.125,      -2.375,    4.40625,   3.32813,   5.0625,     -2.65625,   -1.07813,  2.125,     -7.5,      0.394531,
    6.875,      1.625,     0.240234,  1.90625,   -5.3125,    -0.0849609, -2.75,     -0.492188, 0.574219,  0.261719,
    4,          -2.625,    -7.9375,   -0.90625,  3.375,      2.375,      -3.0625,   6,         -5.59375,  -2.40625,
    -3.57813,   -1.5625,   -6.34375,  -4.90625,  -0.0458984, -3.59375,   -4.875,    3.45313,   -1.98438,  -7,
    -4.6875,    -4.1875,   -3.29688,  -0.78125,  2.57813,    -3,         1.57031,   4.09375,   -4.78125,  -5.4375,
    3.03125,    -4.59375,  -6.75,     -0.492188, 12,         -3.625,     5.5,       -3.84375,  -5,        7.1875,
    0.761719,   2.17188,   4.8125,    -3.42188,  -0.449219,  1.21875,    -1.20313,  0.835938,  -0.664063, -6.375,
    6.125,      7.25,      5.1875,    -2.46875,  0.871094,   -0.902344,  3.4375,    -1.54688,  -4.40625,  -3.17188,
    3.28125,    -0.925781, 0.742188,  -7.1875,   5.75,       1.70313,    -1.20313,  -0.112305, -0.867188, -6.0625,
    5.59375,    0.90625,   1.15625,   -8.125,    -2.4375,    3.26563,    -1.92969,  -1.40625,  3.60938,   -1.89063,
    -6.5,       2.9375,    -2.40625,  -0.96875,  -4.71875,   -0.0493164, 2.625,     -1.0625,   3.03125,   -7.21875,
    1.60156,    -0.238281, 2.03125,   -4.3125,   -9.4375,    -5.5,       6.875,     -8.5,      -0.710938, 5.375,
    3.28125,    -0.523438, -6.75,     0.632813,  1.10156,    6.125,      -3.65625,  -1.98438,  -0.914063, 2.39063,
    3.28125,    1.375,     -1.17969,  -0.226563, 5.34375,    -5.9375,    1.3125,    5.375,     -11.25,    7.75,
    1.3125,     -5.0625,   -4.8125,   0.746094,  -5.4375,    2.89063,    8.125,     -1.5,      -0.171875, -2.51563,
    3.73438,    4.75,      3.48438,   -0.335938, 6.8125,     0.308594,   0.214844,  -0.46875,  1.86719,   1.96094,
    -1.1875,    7.75,      5.25,      -4.78125,  -5.5625,    3.23438,    2.75,      -3.84375,  -0.257813, -1.65625,
    -2.65625,   2.6875,    -0.726563, 5.84375,   -3.90625,   -11.625,    -2.3125,   -4.5,      4.0625,    3.9375,
    1.89063,    1.50781,   -7.21875,  3.09375,   1.13281,    -4.125,     -0.550781, -0.472656, 11.375,    -2.625,
    -3,         4.625,     1.74219,   0.882813,  -0.761719,  -0.291016,  6.59375,   -2.15625,  -2.8125,   -4.1875,
    3.15625,    -6.625,    -7.3125,   -2.26563,  5.84375,    -2.6875,    0.953125,  5.21875,   8.625,     1.75781,
    2.17188,    -4.53125,  -5,        -0.832031, -6.84375,   -4.625,     -0.195313, -4.75,     -9.375,    0.304688,
    3.9375,     -4.9375,   1.72656,   -4.8125,   -0.550781,  -2.3125,    -1.09375,  -2.75,     6.8125,    -4.0625,
    1.94531,    4.28125,   0.5625,    -1.99219,  -5,         -2.01563,   -4,        -8.125,    3.3125,    -2.46875,
    0.414063,   2.04688,   5.59375,   -1.11719,  -1.26563,   -5.0625,    -4,        5,         3.3125,    -1,
    -0.539063,  -3.46875,  -2.375,    -6.53125,  1.39063,    -2.09375,   -6.34375,  -4.15625,  -3.40625,  -1.15625,
    8.375,      3.21875,   -0.126953, 1.34375,   0.367188,   3.125,      0.773438,  0.546875,  -1.17188,  -8,
    1.45313,    -0.902344, 2.51563,   -1.3125,   0.921875,   0.157227,   2.65625,   1.94531,   4.1875,    1.17188};

const std::vector<float> expected_dft1d_float16_results = {
    6.32812,    4.29688,   -0.811035, -0.71875,   -0.0587158, 0.270508,     0.827148,  0.57959,     0.599609,
    -1.3291,    1.18848,   1.46289,   -0.0178833, -1.83203,   0.160034,     -2.2207,   1.0625,      -1.06738,
    -0.686523,  0.657715,  4.62891,   4.59375,    -0.789062,  -0.326416,    -1.4707,   -1.16113,    0.709473,
    1.45703,    -1.09766,  -0.392334, -0.599609,  -0.814941,  -0.391846,    -0.420166, 0.552734,    2.33984,
    -0.599121,  1.39648,   -0.845215, 0.149414,   6.25781,    5.56641,      0.256348,  -1.70215,    1.16211,
    0.120178,   0.197266,  -1.34277,  -0.906738,  1.41113,    0.708496,     0.57959,   1.20215,     -0.519043,
    -2.55859,   -0.808594, 1.16602,   -0.868652,  1.29883,    -1.00488,     5.64453,   3.21094,     0.235962,
    0.310059,   0.617676,  0.100159,  1.04297,    -0.0167999, -0.885254,    -0.12915,  0.648926,    -0.842773,
    -0.0584106, -1.04883,  -0.402344, 0.203247,   0.450439,   0.459229,     1.37109,   -0.654785,   5.52734,
    3.75195,    -1.88672,  -0.223389, 0.816895,   -0.46582,   0.123657,     0.609863,  0.27002,     0.97168,
    0.304932,   0.696289,  0.272217,  -1.08105,   0.205933,   1.56543,      -0.276367, 0.89502,     -0.0397949,
    -0.187134,  4.97656,   4.69531,   0.190063,   -0.88623,   0.916504,     -0.525879, -0.454834,   1.16211,
    -1.47754,   1.67188,   1.02832,   -1.60156,   -1.21777,   -0.134521,    -0.147461, -0.67334,    -0.32666,
    -0.901367,  -0.966309, 0.0360413, 4.77344,    5.24609,    -0.0340881,   -0.372559, -0.490723,   -0.657715,
    0.814453,   -0.891602, -0.199219, 0.179199,   -0.996094,  0.828613,     0.552246,  -0.0240173,  1.53516,
    0.69873,    0.231689,  0.510742,  2.43164,    0.226562,   6.47656,      5.73047,   -0.871094,   -1.11035,
    0.125244,   1.34863,   0.369141,  0.291504,   -0.814941,  -0.0922852,   -0.331299, 0.780762,    -0.962402,
    0.899902,   -2.11133,  0.845215,  -1.53516,   0.743164,   1.33203,      -1.02148,  3.8125,      5.62109,
    1.35449,    0.645996,  -0.216187, -1.20801,   1.16406,    -0.0525208,   0.480957,  -0.589355,   0.234863,
    1.12793,    0.425781,  0.36377,   -1.22461,   2.35156,    0.642578,     0.80957,   1.15723,     0.866699,
    5.59766,    5.66016,   0.480469,  0.594727,   1.18457,    1.4707,       -1.25879,  -0.116394,   0.629883,
    -1.46191,   0.787109,  0.181274,  0.554688,   -2.00586,   -0.437012,    0.994141,  0.895996,    -0.549316,
    0.842773,   0.0840454, 4.52734,   4.35156,    0.730957,   0.810059,     0.278564,  -0.00706482, 0.365479,
    0.674805,   -2.38281,  1.58203,   -0.165527,  0.513672,   -0.236938,    -0.132935, 1.55176,     0.620605,
    0.773926,   1.61133,   0.362793,  -0.424316,  5.10156,    5.18359,      -1.19727,  1.17871,     -0.0184937,
    -0.75,      0.0879517, 0.219238,  1.11328,    2.28906,    -0.688477,    -0.407715, 0.0468445,   0.868164,
    -0.507324,  0.230225,  0.353271,  -0.456299,  0.149902,   -0.105286,    4.80859,   5.28125,     0.496826,
    0.29541,    -0.414551, -0.64502,  0.317627,   -0.895508,  -1.04102,     0.917969,  0.682617,    0.0803833,
    0.339111,   0.0525818, -1.18652,  0.51123,    -1.14551,   -1.41895,     0.82373,   1.89453,     6.88281,
    6.40625,    -1.30273,  -0.224731, -0.220825,  1.03711,    -0.763672,    0.694824,  -0.731445,   0.168091,
    0.0882568,  -0.573242, -0.406982, -0.962891,  -0.890625,  1.31445,      -0.423828, 1.76758,     -0.640137,
    -1.61133,   4.41016,   5.70703,   -1.11133,   0.105713,   -1.11133,     0.341553,  2.11719,     -0.0671997,
    -0.161743,  0.260742,  -0.813477, -0.422607,  -1.1416,    -0.000315189, -0.634766, -0.1521,     -0.716797,
    0.148071,   -0.353516, 1.58398,   4.98828,    4.28906,    -0.849121,    0.534668,  -0.60498,    -0.902344,
    1.04492,    0.211914,  -0.479004, -0.322754,  0.807617,   -0.0440674,   0.628418,  0.155029,    -0.237671,
    0.695801,   1.86035,   -1.70312,  1.26562,    -0.679199,  4.92188,      4.89453,   -0.012001,   -0.6875,
    -0.218506,  -0.486816, 0.367432,  0.124451,   -0.786621,  0.48877,      -0.508301, -0.0883179,  1.16504,
    0.819336,   -0.640625, -1.09668,  1.40918,    -0.153564,  2.88281,      1.05371,   4.70312,     5.20312,
    -0.634766,  2.53125,   0.699219,  -0.538574,  -0.098938,  0.565918,     0.408936,  0.634766,    -1.76367,
    -0.49707,   -1.07715,  1.24219,   -0.362549,  -1.31152,   1.58398,      1.05273,   -0.611328,   0.748535,
    5.43359,    3.9668,    2.00586,   -0.870117,  -0.765625,  0.0811768,    0.839355,  -0.635254,   -0.675293,
    1.73633,    0.9375,    1.8252,    0.861816,   -0.182007,  0.625,        0.203125,  0.159302,    1.86133,
    0.041687,   -0.490723, 4.94141,   4.02734,    -0.74707,   -0.277832,    -0.808105, -0.541016,   0.467285,
    0.901367,   2.19727,   -1.4668,   -1.23535,   1.09668,    -0.955566,    0.51416,   0.288574,    1.01172,
    -0.219116,  0.114624,  -1.62695,  -0.943359};

const std::vector<float> expected_dft2d_float16_results = {
    54.0312,    48.375,    -1.87305,  -3.78906,   2.58594,    -0.708984, 3.53516,    1.67969,   -3.40039,   0.239502,
    2.97461,    2.39844,   0.358887,  -5.80078,   -4.42969,   5.29688,   1.71191,    1.42676,   5.89844,    -0.848145,
    -3.28711,   -0.10022,  -0.339111, 2.61523,    -2.76562,   0.185791,  0.349365,   -0.259277, 5,          -2.98438,
    -1.66406,   0.533203,  -0.898438, -3.64453,   -3.71484,   -0.685547, 2.0625,     -2.2168,   -3.36914,   3.52734,
    1.54785,    -1.63281,  -5.33203,  0.538086,   -1.66699,   2.44922,   -2.26172,   1.41113,   2.17969,    2.50195,
    0.851562,   1.66016,   1.7168,    -2.10156,   4.65625,    -3.67578,  0.893555,   -0.398926, -5.94922,   1.44043,
    2.11523,    -1.45312,  -3.50391,  3.08203,    -3.31836,   4.78516,   5.64844,    -1.11914,  1.8291,     -2.57617,
    -0.412842,  3.23633,   1.42773,   0.870117,   6.17578,    -1.73242,  2.3125,     4.40625,   1.77148,    3.38086,
    -0.532715,  3.76562,   4.44141,   -4.35156,   -2.76367,   3.68555,   1.8916,     -2.58398,  -2.99023,   1.13477,
    1.4834,     0.74707,   -1.30762,  1.06641,    -0.759766,  -10.8828,  3,          -4.25,     0.417969,   -0.451416,
    -0.625,     0.592773,  -0.366211, -0.952637,  -0.15979,   -3.17383,  2.71875,    -3.875,    3.89062,    1.04785,
    -0.0922241, 6.99219,   4.51172,   -0.383301,  0.664062,   -2.12305,  3.94141,    -0.869629, 2.42773,    1.96582,
    3.65234,    -4.49219,  -1.47168,  0.526367,   5.45703,    -2.65625,  -0.0605774, 3.66211,   0.561523,   1.05859,
    4.54297,    -1.44824,  -2.38672,  0.524902,   -6.11328,   -2.36328,  -1.48926,   -2.62695,  -4.40234,   0.276123,
    3.95703,    -3.75977,  -3.12695,  -2.86719,   1.41699,    0.0242462, -2.08008,   1.6543,    0.595703,   -3.25586,
    -0.820801,  1.36328,   -2.90625,  -3.43945,   -2.50195,   -2.82227,  -4.42578,   -2.35352,  0.666016,   -4.74609,
    -2.16016,   -1.05859,  1.13379,   -0.946777,  0.675781,   -4.04688,  -1.24219,   4.53906,   -0.742188,  -3.22461,
    1.23145,    -4.04688,  -4.5625,   -3.51953,   6.70703,    -2.84375,  -0.4375,    -3.99609,  -4.22656,   3.05273,
    4.60938,    2.7207,    2.33398,   -1.04004,   -0.0499268, 2.1582,    -0.223877,  0.686523,  -0.927246,  -7.23438,
    3.79492,    3.19727,   3.86719,   -1.89062,   0.915039,   -0.369629, 3.05664,    0.201294,  -0.0986938, -1.01953,
    50.7188,    49.3125,   -2.61914,  3.39648,    -3.18359,   -2.41016,  4.74219,    1.79492,   -2.53711,   6.28906,
    -2.6582,    1.48242,   -0.777344, 2.37305,    -1.99512,   2.02734,   3.63477,    2.82227,   2.29492,    1.02539,
    3.25195,    -3.06055,  2.09766,   3.56641,    1.94922,    0.241577,  -2.25391,   0.783203,  1.96289,    4.23828,
    -3.25391,   0.787109,  -2.9375,   0.66748,    5.74219,    4.84766,   -4.80078,   6.37109,   -2.66211,   -1.88477,
    -1.76953,   -1.10449,  1.41309,   -0.0806274, -2.76367,   -3.64648,  1.41602,    3.41016,   3.09961,    0.116394,
    -2.41016,   0.277344,  2.89648,   -1.86426,   -0.683105,  2.25,      -0.430176,  -5.81641,  5.34766,    -6.29297,
    0.800781,   3.65234,   1.31641,   2.3418,     2.16016,    1.87207,   -2.4707,    0.392822,  2.01953,    -0.0645142,
    -4.13281,   -1.5127,   -2.03516,  1.19531,    -0.666016,  -2.04492,  2.07422,    4.87109,   -0.0910034, 1.38965,
    0.680664,   -4,        -0.802734, 0.439697,   2.78125,    0.452393,  -0.876465,  1.23828,   -2.72656,   2.79688,
    4.14844,    -1.9209,   -0.609863, -4.79297,   3.15234,    0.793457,  5.30859,    0.279785,  -3.67383,   -4.39844,
    -2.51172,   -0.914062, 6.83984,   -4.08984,   -1.27832,   0.978516,  3.26953,    -3.39258,  -7.55469,   3.68359,
    2.92383,    2.33398,   2.75195,   -1.25977,   1.42383,    -1.85449,  -2.67383,   1.27148,   5.21875,    6.20703,
    0.493652,   2.11523,   5.87891,   2.80469,    -0.37793,   0.0119629, -1.04004,   -1.53711,  -8.07812,   -0.686035,
    2.35156,    3.08789,   2.65039,   1.36719,    0.771973,   2.29297,   -1.26855,   2.11523,   4.99219,    -0.0458984,
    0.0148392,  1.1582,    -4.83594,  1.95605,    1.97949,    2.3457,    -0.963379,  4.39453,   -6.20312,   0.832031,
    -2.77344,   -2.92773,  -3.46484,  -1.45898,   2.47266,    -0.795898, -0.428467,  5.8125,    -2.66406,   -2.27148,
    -2.54883,   -3.12305,  -4.46094,  0.164429,   1.91504,    1.04688,   2.80664,    -0.47168,  -4.06641,   -2.22266,
    1.77734,    -0.578613, -2.20312,  3.02734,    5.32422,    -0.783203, 5.92578,    0.141968,  -0.815918,  4.19531,
    -3.85547,   -0.515137, 2.47852,   -2.39648,   -0.394043,  -0.960938, -0.977051,  0.137573,  0.259766,   0.834961,
    2.37109,    4.10938,   1.3584,    -0.584473,  -0.0278931, -0.530762, 0.396484,   -1.75781,  -4.31641,   -2.16797};

const std::vector<float> expected_dft3d_float16_results = {
    104.75,     97.6875,   -4.49219,  -0.392578, -0.598633,  -3.11914,   8.28125,   3.47461,   -5.9375,   6.52734,
    0.316162,   3.88086,   -0.418213, -3.42773,  -6.42578,   7.32422,    5.34766,   4.25,      8.1875,    0.17688,
    -0.0348816, -3.16016,  1.75781,   6.17969,   -0.817383,  0.427246,   -1.90527,  0.523926,  6.96094,   1.25391,
    -4.91797,   1.32031,   -3.83594,  -2.97852,  2.0293,     4.16406,    -2.73633,  4.15625,   -6.03125,  1.64258,
    -0.221558,  -2.73633,  -3.91992,  0.45752,   -4.42969,   -1.19727,   -0.844238, 4.82031,   5.28125,   2.61914,
    -1.55762,   1.9375,    4.61328,   -3.9668,   3.97461,    -1.42676,   0.463135,  -6.21875,  -0.599609, -4.85547,
    2.91406,    2.19922,   -2.1875,   5.42188,   -1.15723,   6.66016,    3.17578,   -0.726562, 3.84961,   -2.64062,
    -4.54297,   1.72363,   -0.606445, 2.06641,   5.50781,    -3.77734,   4.38672,   9.28125,   1.68066,   4.76953,
    0.148071,   -0.237183, 3.63672,   -3.91406,  0.0173492,  4.14062,    1.01465,   -1.34668,  -5.71484,  3.93164,
    5.63281,    -1.17285,  -1.91797,  -3.72656,  2.39258,    -10.0859,   8.30469,   -3.96875,  -3.25586,  -4.84766,
    -3.13672,   -0.321045, 6.47656,   -5.04297,  -1.4375,    -2.19531,   5.98828,   -7.26562,  -3.66602,  4.73047,
    2.83203,    9.32812,   7.26172,   -1.64258,  2.08789,    -3.97656,   1.26562,   0.402588,  7.64844,   8.17188,
    4.14453,    -2.375,    4.40625,   3.33008,   5.07812,    -2.64453,   -1.10059,  2.125,     -7.51953,  0.372559,
    6.89453,    1.63867,   0.263672,  1.8916,    -5.33984,   -0.071228,  -2.75781,  -0.512695, 0.587402,  0.230225,
    3.97266,    -2.60156,  -7.96094,  -0.911133, 3.39648,    2.36914,    -3.04297,  6.04688,   -5.60547,  -2.42383,
    -3.5957,    -1.56543,  -6.37109,  -4.89844,  -0.0288696, -3.61719,   -4.85547,  3.45898,   -1.99805,  -7.01562,
    -4.71094,   -4.17969,  -3.32812,  -0.782227, 2.58984,    -3.00195,   1.56348,   4.06641,   -4.80859,  -5.44531,
    3.00781,    -4.625,    -6.76562,  -0.492432, 12.0312,    -3.62695,   5.48828,   -3.85352,  -5.04297,  7.24609,
    0.754883,   2.20703,   4.8125,    -3.4375,   -0.443848,  1.19727,    -1.20117,  0.824219,  -0.66748,  -6.40234,
    6.16406,    7.30469,   5.22266,   -2.47461,  0.887207,   -0.900391,  3.45312,   -1.55566,  -4.41797,  -3.1875,
    3.30469,    -0.945801, 0.745605,  -7.1875,   5.76953,    1.70215,    -1.20898,  -0.115417, -0.862305, -6.05078,
    5.63281,    0.916504,  1.13574,   -8.17188,  -2.43359,   3.26953,    -1.92383,  -1.39551,  3.60156,   -1.87305,
    -6.53906,   2.96094,   -2.43555,  -0.952148, -4.71484,   -0.0558167, 2.60352,   -1.04297,  3.03516,   -7.22266,
    1.58984,    -0.253906, 2.03906,   -4.3125,   -9.46094,   -5.53516,   6.86328,   -8.58594,  -0.70752,  5.41406,
    3.31641,    -0.527832, -6.74609,  0.618652,  1.09668,    6.09766,    -3.67773,  -1.99902,  -0.919434, 2.38672,
    3.26172,    1.38281,   -1.17969,  -0.237305, 5.33984,    -5.92578,   1.32324,   5.41797,   -11.2969,  7.73438,
    1.31348,    -5.10547,  -4.82031,  0.739258,  -5.47656,   2.91406,    8.11719,   -1.51172,  -0.191284, -2.51172,
    3.71875,    4.75,      3.46289,   -0.325439, 6.83984,    0.311768,   0.237549,  -0.465088, 1.8623,    1.99023,
    -1.21289,   7.76562,   5.24219,   -4.79297,  -5.54297,   3.23438,    2.76758,   -3.82227,  -0.263428, -1.66113,
    -2.66602,   2.66797,   -0.698242, 5.85938,   -3.91211,   -11.6719,   -2.30664,  -4.52734,  4.09375,   3.94531,
    1.88672,    1.50684,   -7.20703,  3.13672,   1.11816,    -4.15234,   -0.55127,  -0.482422, 11.4453,   -2.63477,
    -3.01562,   4.66016,   1.75879,   0.876465,  -0.759766,  -0.267334,  6.61719,   -2.14062,  -2.79102,  -4.24219,
    3.1582,     -6.60547,  -7.35156,  -2.27734,  5.83594,    -2.66992,   0.979492,  5.19922,   8.64062,   1.74414,
    2.19141,    -4.53516,  -5.03516,  -0.841797, -6.88672,   -4.65625,   -0.220825, -4.74219,  -9.39062,  0.322021,
    3.94336,    -4.91797,  1.70703,   -4.82422,  -0.562012,  -2.32031,   -1.11719,  -2.73828,  6.80078,   -4.08594,
    1.95312,    4.29297,   0.558105,  -1.97949,  -4.97656,   -2.02539,   -3.99805,  -8.16406,  3.33008,   -2.47461,
    0.388672,   2.06445,   5.59375,   -1.11133,  -1.23828,   -5.09375,   -4.04688,  5.01172,   3.32227,   -1.00293,
    -0.545898,  -3.46875,  -2.35938,  -6.54688,  1.38086,    -2.06055,   -6.36328,  -4.13672,  -3.41211,  -1.14062,
    8.46875,    3.23633,   -0.145142, 1.35742,   0.343994,   3.11914,    0.753418,  0.548828,  -1.18652,  -8.07031,
    1.42383,    -0.911621, 2.50781,   -1.30566,  0.942871,   0.161255,   2.66016,   1.95898,   4.21875,   1.14844};

const std::vector<float> input_data_1 = {
    0.9795938,   0.14046684,  0.9819369,    0.51320475,  0.9390985,  0.06454252,  0.48972926,   0.042538255,
    0.3341647,   0.14752749,  0.44628578,   0.8509109,   0.6611515,  0.5711897,   0.10807402,   0.67733586,
    0.4091941,   0.23590194,  0.4385734,    0.40270114,  0.75568604, 0.9842337,   0.82819414,   0.49742407,
    0.7397849,   0.6104118,   0.019504193,  0.7756829,   0.9271429,  0.6423316,   0.3300541,    0.8688829,
    0.21220064,  0.76539195,  0.8143789,    0.70724595,  0.54020476, 0.29437974,  0.19398275,   0.20308031,
    0.30458412,  0.040420562, 0.36627868,   0.61882246,  0.3416973,  0.5482437,   0.68851316,   0.5670022,
    0.58812225,  0.6487681,   0.88266903,   0.07287276,  0.7716641,  0.12443388,  0.4170407,    0.8380076,
    0.17115247,  0.8118648,   0.7704737,    0.5179812,   0.9407177,  0.7311383,   0.4538601,    0.01992845,
    0.4758718,   0.25867644,  0.55573237,   0.89606065,  0.8505143,  0.47349417,  0.3970769,    0.3293097,
    0.7601557,   0.24247961,  0.8102311,    0.7387785,   0.15742134, 0.8387721,   0.100493915,  0.3733577,
    0.4904671,   0.9106489,   0.0049963384, 0.89285916,  0.24380954, 0.7329451,   0.9373891,    0.52886724,
    0.65965563,  0.7307209,   0.5160155,    0.97944415,  0.43991584, 0.9839402,   0.6350642,    0.16712844,
    0.40538687,  0.80509937,  0.4988989,    0.02185218,  0.74142575, 0.8026278,   0.28912508,   0.50405765,
    0.7768013,   0.9817653,   0.9995751,    0.74799776,  0.8615655,  0.058107413, 0.27611437,   0.76372087,
    0.93234706,  0.7603203,   0.30816016,   0.80595773,  0.8843074,  0.46457228,  0.43644127,   0.6553406,
    0.9050378,   0.5044161,   0.49364874,   0.59174323,  0.2650881,  0.78394204,  0.57706285,   0.33071348,
    0.7140054,   0.5885716,   0.60252094,   0.92644346,  0.91704935, 0.64020824,  0.99652874,   0.8375778,
    0.45622328,  0.3755286,   0.8324417,    0.77270067,  0.50742614, 0.7814994,   0.30720684,   0.36613366,
    0.9426107,   0.12557131,  0.87243265,   0.002567238, 0.8350289,  0.1262151,   0.35253504,   0.07578735,
    0.34082502,  0.9211622,   0.38055828,   0.3247621,   0.5061271,  0.87862396,  0.1869049,    0.7774487,
    0.030804915, 0.25322768,  0.06073754,   0.27092665,  0.9209875,  0.86690956,  0.74456835,   0.42403135,
    0.61839956,  0.9004572,   0.94674456,   0.17315134,  0.74403226, 0.30930993,  0.23992635,   0.9080931,
    0.4886828,   0.9973065,   0.32888287,   0.32976696,  0.09137513, 0.1410893,   0.4248779,    0.019689998,
    0.6828394,   0.47350892,  0.02358055,   0.94660497,  0.9253734,  0.1509718,   0.540138,     0.7050524,
    0.20855357,  0.9753569,   0.0044813985, 0.5063834,   0.6836877,  0.2784342,   0.0139586115, 0.8785785,
    0.4754602,   0.38955635,  0.151705,     0.5694773,   0.14548586, 0.6798502,   0.057361145,  0.031760257,
    0.91168743,  0.5762714,   0.54128575,   0.53421247,  0.5860678,  0.97197753,  0.940639,     0.18688098,
    0.54635745,  0.513619,    0.5645304,    0.91558236,  0.8496063,  0.6258071,   0.31261826,   0.20282389,
    0.2723365,   0.5039135,   0.6405068,    0.65471405,  0.5857442,  0.57205665,  0.23835625,   0.32288164,
    0.068663165, 0.43674967,  0.049117915,  0.78065604,  0.98437595, 0.60793245,  0.38907775,   0.6610265,
    0.5587009,   0.89418066,  0.6170649,    0.1305905,   0.5760506,  0.10885323,  0.5303117,    0.16950679,
    0.9630447,   0.9476875,   0.22327174,   0.87473476,  0.917824,   0.44959846,  0.055421904,  0.22361691,
    0.9334828,   0.16427046,  0.5914317,    0.81789917,  0.48431975, 0.3067152,   0.53250873,   0.19298424,
    0.23529118,  0.4841604,   0.24943262,   0.41821656,  0.59484303, 0.4535004,   0.50373393,   0.6057788,
    0.6799498,   0.21368006,  0.17095569,   0.97966874,  0.3839032,  0.48672524,  0.9375583,    0.84598905,
    0.049092207, 0.47682214,  0.56488436,   0.7817405,   0.8975917,  0.75874424,  0.43242812,   0.8459973,
    0.7138231,   0.9834999,   0.7273379,    0.05828699,  0.6884237,  0.07486352,  0.4326547,    0.78577167,
    0.8844588,   0.9474644,   0.542272,     0.49642876,  0.48886803, 0.11854455,  0.01492267,   0.22648218,
    0.7607531,   0.5930919,   0.9450968,    0.02894685,  0.67735505, 0.46363172,  0.18415985,   0.66824925,
    0.6137258,   0.6086626,   0.6422855,    0.7637218,   0.56419605, 0.74026155,  0.18709394,   0.14683136,
    0.32289994,  0.15482259,  0.11222768,   0.9085655,   0.43263617, 0.32097924,  0.29690787,   0.77809244,
    0.2413839,   0.8267769,   0.82795614,   0.018312717, 0.9958108,  0.769578,    0.13144562,   0.45904484,
    0.38071582,  0.24182741,  0.7200288,    0.20737973,  0.5285696,  0.3680287,   0.46252182,   0.89153767,
    0.13071166,  0.84319293,  0.10841641,   0.40668696,  0.7636801,  0.42153865,  0.65055484,   0.86845386,
    0.6452055,   0.6112245,   0.84526664,   0.15358071,  0.7889171,  0.6356269,   0.2515375,    0.86599886,
    0.20071381,  0.20584217,  0.24220705,   0.049883988, 0.77259976, 0.49566683,  0.8112268,    0.49028614,
    0.2187354,   0.70172536,  0.47309682,   0.12539592,  0.13696012, 0.33588144,  0.98134226,   0.537496,
    0.9999663,   0.13245043,  0.5659106,    0.39207155,  0.48483336, 0.49371332,  0.12930158,   0.103645995};

const std::vector<float> expected_dft1d_results_1 = {
    4.940035,    3.0077164,    -0.43999052,  -0.72356576,  0.35775006,  -1.1781573,    0.35552078,  -0.5878226,
    0.8879826,   -1.1602633,   0.71755445,   0.15355057,   -0.9307331,  0.48268145,    1.9486318,   1.1295953,
    4.4481335,   5.01757,      -0.57099926,  -0.85269475,  -0.7217729,  -0.08008081,   -1.1429803,  -1.1934547,
    1.2154821,   -0.07181215,  0.59362185,   0.44658875,   -0.345927,   -1.480422,     -0.20200539, 0.10152125,
    3.4618404,   3.744587,     0.12548631,   0.8791516,    0.19086862,  -0.33497274,   -0.69986683, 0.6364535,
    -0.6644666,  -0.44771492,  -0.8179812,   0.17377639,   -0.92110324, 0.26135075,    1.0228279,   1.2105042,
    4.9957,      3.764995,     0.17936486,   -0.9405504,   -1.2201893,  -0.17718112,   1.1820351,   0.5077594,
    -0.052387,   0.86741495,   -0.55883414,  0.9524643,    -0.68602496, 1.3873026,     0.8653134,   -1.17206,
    4.107497,    4.150929,     -0.95916677,  -0.56429225,  1.1602635,   -1.679503,     0.5507363,   0.53716975,
    0.38042903,  -0.5240841,   -0.33995685,  -0.78949994,  -0.7040798,  0.05728233,    -0.38874817, 0.8814098,
    3.9273133,   5.9265537,    -0.80074155,  0.20659067,   1.642705,    0.9759259,     0.85149074,  0.44840366,
    -0.25961697, 0.78995633,   -0.039625674, 0.545478,     -0.70991015, -1.1269569,    -0.68787766, -0.48076022,
    4.848893,    4.6852283,    -0.6871975,   -0.041547477, -0.91873163, -0.0071051717, -1.4497755,  0.3778788,
    0.7214658,   0.6099715,    1.4334095,    -0.07150489,  0.07712549,  1.859364,      -0.78209424, -0.97149,
    4.8020935,   4.897006,     0.05723229,   -0.21079391,  1.0996364,   0.22791737,    0.7594234,   1.1837918,
    1.1714673,   0.12949562,   -0.64135337,  -0.5158363,   0.2763425,   -0.19547313,   -0.06606534, 0.56645525,
    5.3334026,   5.288664,     -0.09143779,  -0.7460747,   0.2411859,   -0.5888344,    1.4911498,   0.52246934,
    -0.1439941,  -0.51704764,  0.32441977,   0.35291424,   -0.7496793,  -0.32638037,   -0.6930033,  0.72286314,
    4.4170227,   3.232138,     -0.64390934,  -1.3210952,   -0.58362705, -0.6716566,    0.39952934,  -1.1999383,
    0.83216095,  0.8710072,    0.34266293,   -0.92789006,  0.46818644,  0.7554455,     2.3088598,   0.26656008,
    4.306201,    4.1061068,    -1.286478,    -0.14309806,  -1.9038618,  -0.045521975,  -0.43500268, -0.6120295,
    0.3222475,   0.5537019,    1.2264881,    -1.5052714,   -0.12776875, 0.00045275688, -1.8553859,  -0.32851917,
    3.50575,     3.7639909,    -0.8274512,   1.2718699,    0.7064032,   1.7913067,     -1.4024514,  -0.49303484,
    0.8707912,   -0.23823786,  0.41937304,   1.443722,     -0.396856,   0.56620187,    1.0339032,   -0.12736642,
    1.7406936,   4.309397,     -0.18755847,  -0.46101326,  0.020362198, 0.3217622,     0.7620988,   1.9022003,
    1.2856812,   0.3369981,    -1.149087,    0.5562107,    -0.31068176, 0.4914955,     -0.49307993, 0.34580433,
    5.2527924,   4.527175,     -0.029956281, -0.35984623,  1.0824606,   -0.360453,     0.19873115,  -0.3701315,
    0.53464556,  0.8481753,    1.4529572,    1.012228,     -1.037719,   -0.6553353,    -0.16041204, -0.03164065,
    3.2281785,   4.5399303,    0.3643899,    0.30424678,   -0.7776585,  -0.3015166,    -0.61336106, -0.7931169,
    0.5940609,   -0.29862595,  -0.02879478,  0.6273444,    -1.6805825,  -0.17713517,   1.0924593,   0.1301811,
    4.4416904,   3.7987688,    -1.3668841,   -0.81391287,  0.64007276,  1.0288135,     -0.57070565, -0.52160406,
    1.58955,     1.0018709,    -0.123293996, 1.390446,     -0.5843305,  1.5380195,     0.44350854,  -0.26895642,
    4.125044,    3.443525,     0.7636179,    0.10296479,   0.52696383,  0.08359367,    0.6142223,   -1.2670981,
    0.3708297,   -0.6262324,   0.339195,     -0.5216981,   -0.34774148, -0.30716318,   1.0757314,   0.4062716,
    4.1163635,   5.389367,     -0.1369414,   0.3118773,    -0.48302984, 0.07917905,    1.6785579,   -0.9954241,
    -0.09528947, -1.517424,    0.85461855,   0.18921553,   -0.62187576, -1.1891136,    0.12719542,  -0.558237,
    4.492761,    3.6913419,    -0.29317212,  -1.2950531,   -0.03654802, 0.91552365,    0.123229444, 0.514639,
    1.0583864,   0.5574026,    -0.13546133,  0.9680127,    0.87852824,  2.559589,      -0.3771388,  -0.043456703,
    4.574666,    4.013397,     -0.06427416,  -1.2290373,   0.11051571,  -1.2182673,    0.05659631,  0.77380556,
    0.65739393,  0.7978984,    -0.19493088,  0.9715779,    0.1553396,   1.2139899,     0.79071796,  -0.57862896,
    3.361268,    4.236172,     -0.13507411,  0.6842204,    -1.1744223,  -0.62078804,   2.008315,    -1.2499349,
    0.62419355,  -0.091858864, -0.5990913,   -0.90177983,  -0.55390406, 0.40287262,    -0.94808567, -1.2203228,
    3.745199,    4.248646,     0.63732016,   -0.82302505,  -1.9267471,  0.58008444,    -0.38652933, -0.9787377,
    -0.1378448,  -0.4994706,   -0.24433172,  0.09051508,   0.3651026,   0.010821462,   0.9935576,   -0.69421434,
    4.5576744,   3.50811,      1.745633,     0.16605312,   -1.8684177,  -0.33893645,   -0.17866233, 0.5833766,
    0.2571981,   0.38861072,   -0.5767294,   0.61207676,   0.43722266,  -0.28951776,   0.78772557,  0.26002276,
    3.9901466,   2.82238,      -1.4889656,   -0.1150527,   0.47323376,  0.07621753,    0.16292411,  0.17989358,
    -0.30915606, 0.50516117,   -0.38916004,  1.9493489,    0.72058266,  -0.067055345,  -1.4097221,  0.26290974};

const std::vector<float> expected_dft2d_results_1 = {
    25.880518,   25.61235,    -2.4660468,  -1.9953613,   1.409625,    -2.473969,   1.0969357,    0.34850854,
    1.5074215,   -0.546504,   -0.44522142, 1.482357,     -4.297778,   -0.41876173, 2.5581412,    1.6702101,
    -0.79171646, 0.87513673,  -0.5556948,  -1.4017968,   1.6127079,   3.341402,    -2.2336023,   0.7553977,
    0.8801464,   -1.5552741,  2.8809369,   -0.12944597,  -0.08941656, -2.4948978,  1.1106122,    -0.5771601,
    1.5280423,   -3.6573076,  -1.325342,   -0.75811887,  -4.0773964,  0.41215408,  0.24999249,   0.3498589,
    -0.31276864, -2.3484,     -0.4591713,  -0.04454562,  -0.7590859,  2.5111904,   3.1611128,    -0.09711918,
    -0.8617741,  -3.8058863,  -0.0812951,  1.1779473,    2.0081396,   -3.9112964,  -0.6841551,   0.82309175,
    -0.2995335,  -3.7176208,  -0.43554613, -2.4067037,   -0.81405425, 2.0213914,   2.6072812,    4.772808,
    2.3986423,   -1.6369095,  3.009512,    -2.2388682,   0.08045465,  -2.0042,     3.2657382,    -0.93855727,
    1.3121321,   2.0163581,   1.3805538,   1.8802332,    0.20659024,  3.5175233,   2.7225797,    -1.7004844,
    1.4864945,   0.6589138,   -1.221076,   0.8748032,    1.1129706,   -2.4330344,  0.43821555,   -4.865236,
    2.2404962,   -0.81013864, 1.3837745,   0.13940874,   0.16934663,  -2.240356,   -0.46793693,  2.7093167,
    27.21336,    25.973133,   -3.4792416,  -1.1907393,   -1.358995,   0.70610523,  -0.63712704,  -0.22086221,
    3.7741385,   1.4088898,   3.1050003,   -1.2238663,   -0.45265055, 2.6596098,   -0.053786665, 0.12850314,
    1.7713342,   -0.92604506, -1.5456572,  0.4535787,    -0.4252041,  -0.20687354, -0.26421398,  -1.5723603,
    0.21247786,  -0.19034994, 0.116511285, 3.5963366,    -0.9552689,  1.4078308,   0.17855054,   -1.2697299,
    0.24928832,  -1.3436013,  -1.018871,   -1.1798176,   -2.4574528,  0.14592099,  -0.7871367,   -1.3267987,
    1.6891341,   0.8528522,   -2.194655,   -0.7497572,   0.66770875,  1.4708114,   2.0073843,    0.8376069,
    1.7636304,   2.1868649,   -0.65098536, -0.6707011,   -3.8038197,  -1.9890289,  -0.15012956,  0.7975005,
    -1.9746995,  -0.11563957, 2.8636346,   -1.2238576,   -1.1479954,  0.40726233,  -6.6071806,   -1.2827955,
    0.335096,    -0.8774332,  0.5047921,   -1.7173706,   -0.6906272,  -2.8883119,  -1.7264752,   -0.91851616,
    -0.8023921,  2.1811929,   4.4178715,   -1.0245608,   1.4208769,   3.714005,    2.626697,     -3.0808997,
    -2.2393522,  3.0984519,   2.0667777,   4.0557647,    3.22371,     4.1895566,   -5.1335697,   5.5083103,
    1.4301378,   -0.47711706, 0.29209352,  0.19667566,   0.9300822,   1.4966636,   -2.8442304,   -1.1616251,
    22.90476,    26.008162,   -0.59333247, -0.9156835,   1.009171,    0.85137844,  2.0695426,    -2.0451744,
    4.279478,    -0.2552395,  1.3455946,   3.2537463,    -4.582932,   -0.29923248, 2.0854027,    0.023423433,
    -1.4901955,  1.2697036,   0.12445855,  0.37839913,   -0.90889513, -0.96464497, 3.2230172,    5.11582,
    1.7657483,   -1.2759314,  1.6806445,   -0.48582482,  1.0328861,   -0.21219438, -1.8203479,   0.28618455,
    -3.8749995,  -2.6027172,  -2.7910428,  -1.8929406,   0.43884945,  -0.8854169,  -0.6166424,   3.3119302,
    3.9380612,   1.783706,    -2.8637185,  0.45624626,   1.298605,    2.399745,    -0.42191154,  0.3671223,
    -4.7169294,  -1.4224572,  2.4742305,   0.80807984,   -1.4698355,  -0.64370054, -0.54362357,  1.729145,
    0.2216661,   -0.920482,   -3.022968,   -1.9300321,   -0.09508008, 0.31362647,  1.264819,     1.741091,
    -0.48260987, 0.91905135,  -1.2789521,  -1.0161536,   0.53328425,  4.0857644,   -0.8787215,   2.8750324,
    0.4081546,   2.4881384,   -2.2990177,  2.1299765,    0.59928864,  3.988031,    -1.8122058,   -0.16000175,
    -1.8958641,  1.6846397,   0.9392875,   -0.12778088,  0.51960033,  -0.5128077,  1.3190198,    0.42644808,
    -2.8990207,  0.20179635,  -1.7350545,  -0.08684918,  -0.11685985, -3.241004,   -2.2542362,   -0.18299285,
    24.721714,   22.520046,   0.40146637,  -2.611894,    -4.422385,   -0.6061659,  1.7858734,    -0.17695832,
    2.1501722,   1.6577435,   -2.1397042,  3.6897519,    2.0028722,   3.830699,    -0.16294527,  -2.0136907,
    2.7324684,   -0.48164713, -3.0283842,  -1.1742884,   2.3383465,   -0.04261756, -1.3686588,   0.50161046,
    0.76707345,  0.40514386,  -1.7530769,  2.333971,     2.7187724,   4.413412,    -3.610829,    0.57066756,
    0.3970142,   -0.89236856, -1.0979964,  -4.7337284,   -1.6107149,  3.461636,    0.8141506,    1.3783914,
    0.97534364,  -1.261203,   0.9644269,   -0.4446571,   1.3737998,   1.5714393,   1.5593243,    -3.5085554,
    0.10169166,  0.3512014,   2.2333064,   1.7223357,    -1.7363904,  0.5177647,   2.1198907,    -0.12688133,
    1.7293842,   0.05056551,  -0.4828595,  -2.333132,    -0.4791782,  1.5151871,   -0.91205263,  0.0061766207,
    -0.4048485,  2.1922839,   1.728973,    0.9913887,    0.14321594,  1.6313545,   -3.389923,    -2.5937288,
    -0.36389086, -0.2227447,  0.03589952,  -0.069511056, 0.3542207,   2.3090918,   0.45287704,   3.309232,
    -0.59147376, -1.541465,   -1.9963981,  -1.9641305,   5.0686407,   0.53117156,  0.77804404,   4.1053996,
    1.0922346,   2.7149107,   2.5625482,   2.6316533,    -0.69931746, 1.7177012,   0.4107918,    1.375428787};

const std::vector<float> expected_dft3d_results_1 = {
    100.72035,   100.11369,    -6.1371527,  -6.7136793,   -3.3625832,  -1.5226498,   4.315223,    -2.0944858,
    11.711212,   2.2648964,    1.8656702,   7.201989,     -7.3304863,  5.772318,     4.4268136,   -0.1915536,
    2.2218904,   0.7371478,    -5.005279,   -1.7441093,   2.6169558,   2.1272666,    -0.64345765, 4.800469,
    3.6254454,   -2.6164103,   2.9250154,   5.315038,     2.706975,    3.1141493,    -4.1420155,  -0.99003804,
    -1.7006547,  -8.495995,    -6.2332516,  -8.564608,    -7.7067156,  3.134292,     -0.33963704, 3.7133825,
    6.2897673,   -0.9730439,   -4.5531178,  -0.7827141,   2.581028,    7.953187,     6.305909,    -2.4009464,
    -3.7133813,  -2.690277,    3.9752584,   3.0376637,    -5.0019054,  -6.0262628,   0.7419828,   3.2228546,
    -0.32318294, -4.7031775,   -1.0777391,  -7.8937283,   -2.5363095,  4.257466,     -3.6471338,  5.237282,
    1.8462799,   0.5969913,    3.9643247,   -3.981004,    0.0663265,   0.82460785,   -2.7293837,  -1.5757694,
    0.55400586,  6.462948,     3.5353048,   2.9161394,    2.580977,    13.528652,    3.98995,     -1.632153,
    -3.240194,   3.900541,     -0.21140909, 2.8386583,    9.924921,    1.7748868,    -2.5982907,  5.174925,
    1.8638494,   1.6294506,    2.5033612,   2.8808913,    0.2832532,   -2.2669961,   -5.155612,   2.7401254,
    6.428844,    -2.8874602,   -0.45156026, 2.8010314,    1.7127267,   -6.3887377,   -1.0165125,  4.816684,
    -3.0209088,  -1.9152341,   -6.7044344,  -7.0160933,   -0.8859364,  2.3359919,    2.614932,    1.5376289,
    0.2540813,   0.56656873,   0.947714,    -3.2629232,   2.3573487,   7.069599,     -7.53059,    -5.4648676,
    -1.4810953,  0.27525342,   2.4626575,   -1.5132098,   -4.127886,   1.3913381,    1.090563,    -4.6527243,
    4.9518104,   -0.906865,    5.0196123,   1.055696,     -7.831962,   2.144308,     -1.838556,   -1.3607846,
    -2.1367745,  -4.8458967,   2.0994475,   2.6582882,    -2.158319,   0.8175374,    7.929186,    -0.9123031,
    5.690818,    -4.0453672,   -4.948562,   3.2541595,    0.9711809,   -1.2001665,   0.78384995,  1.3639677,
    -0.6874037,  0.9069457,    3.6966968,   -3.823165,    -1.826899,   2.3765814,    0.0534904,   8.726845,
    -0.18846548, -3.2959056,   1.5797036,   0.0014669895, -4.9724956,  -5.2561207,   5.819672,    -5.477039,
    3.3079143,   -0.033277154, 2.7245224,   -4.631716,    1.0122153,   -1.5371637,   -1.8553452,  -3.7143025,
    8.022276,    0.62215286,   3.8595328,   -3.060592,    4.2517557,   -0.075296044, 0.5221062,   0.6199312,
    1.9474881,   -1.3498385,   0.6838516,   2.4967105,    0.06516862,  -0.6287519,   -0.7507546,  6.147333,
    -3.149796,   3.1273334,    0.018394649, 0.8915896,    8.200176,    -1.7225304,   2.0177326,   -1.2988436,
    -0.13740933, -3.868376,    -0.06492156, 2.2702193,    -10.430931,  -7.2083035,   4.860276,    3.578821,
    -6.7857146,  3.5525331,    4.142806,    -0.3026886,   -1.20933,    2.6262493,    2.6222875,   6.941968,
    1.6663432,   -3.0459986,   6.198147,    -6.5455766,   -0.8200346,  -8.528335,    2.722542,    0.4080863,
    -2.993259,   -4.024056,    -1.999518,   3.2624865,    0.42962015,  -4.08082,     -0.39366418, 3.6101956,
    0.9608154,   -0.15634394,  -2.0926623,  1.6061159,    -1.5019901,  1.8686844,    -0.8275065,  2.9409513,
    -7.4440265,  -7.7664104,   0.8106141,   0.9343933,    6.078513,    -3.0837321,   -3.1975398,  1.8816166,
    0.16744804,  -4.573029,    -5.839288,   -0.7797469,   0.71803975,  0.41256714,   11.391333,   7.790516,
    1.9857845,   -2.0327086,   -0.5032053,  -2.5290394,   1.16115,     3.3385215,    7.5034156,   5.4487205,
    2.886569,    2.5460477,    -5.3722363,  5.1042805,    -0.9692185,  1.4824567,    -2.1692014,  -2.0888186,
    2.4214573,   0.78656745,   -0.3521694,  -1.3446121,   -6.659781,   -7.66657,     6.1127615,   -14.052498,
    -3.1808968,  -2.8461368,   -3.2059226,  -2.7757711,   -0.17827892, -8.695724,    -0.2887354,  2.312519,
    -0.4773283,  2.095835,     -3.293869,   -4.960386,    -0.9118179,  -0.2619573,   -0.92870337, -0.029317379,
    -2.5232022,  1.3327014,    3.1228013,   3.4733155,    1.4562413,   -2.5750513,   -1.6694541,  1.7559463,
    1.142877,    -1.3557005,   -2.30802,    -0.29746848,  2.6858592,   1.5424967,    -3.3826494,  -3.2559767,
    -0.2901088,  -0.83393717,  -0.06207335, 2.225967,     1.8832793,   -5.9567456,   4.7713566,   2.9260354,
    5.854274,    -1.2023156,   -2.0882115,  1.2139479,    -1.2005312,  0.4508332,    3.571826,    -4.5633574,
    -6.3648844,  -3.4183156,   2.7096481,   -3.659875,    -1.957063,   -0.5946456,   -0.76313734, -0.016180754,
    2.0194921,   -0.72149086,  -0.16249031, -2.5144238,   5.9847684,   -5.335026,    -1.0649127,  -3.176074,
    -0.3549943,  -6.501223,    1.4781482,   2.8698225,    0.3889513,   1.0389466,    2.6314335,   -2.6634102,
    5.950971,    -1.8160157,   6.9972243,   -2.4468954,   4.066836,    -6.923808,    2.4692469,   -2.1501422,
    -1.4999585,  -0.91028214,  4.634622,    4.132228,     -1.7976125,  0.59614825,   10.924917,   0.63333595,
    -1.2575581,  -2.6736045,   -8.180259,   5.0657587,    -3.065015,   -3.7651565,   -2.2837136,  -11.203299,
    8.331546,    -0.6740327,   5.5538063,   -2.0441968,   0.5072439,   2.630047,     4.323353,    -0.3627143};

const std::vector<float> expected_dft2d_signal_size_results_1 = {
    16.022383,    15.693684,   -0.5664709,  -5.672549,   1.6033735,   -1.197921,    1.7188175,    2.0131826,
    3.5833678,    5.2242,      0.46792942,  1.134562,    -0.67151153, 1.9223167,    0.8663217,    6.124037,
    -2.2579927,   0.89829487,  19.077686,   18.455149,   0.22091317,  -6.1652884,   3.7212925,    -2.246737,
    2.152333,     -0.35627568, 2.3711212,   3.8686695,   2.6396408,   0.9601658,    2.798329,     2.7985961,
    -3.0813444,   2.1557612,   -2.7741354,  0.020231158, 15.38469,    17.809353,    -0.18956879,  -1.9654348,
    2.1695504,    -4.6052723,  3.9439118,   1.2099034,   2.0613155,   0.06963302,   -0.44236425,  0.388348,
    -1.0990168,   1.5654994,   -5.6095753,  3.5276392,   -2.5259602,  0.11462754,   17.599613,    15.044548,
    -0.6420444,   -7.04445,    -1.3374522,  -2.994011,   4.8180413,   0.4241563,    -0.16446547,  3.4031193,
    1.6924896,    1.1558795,   -0.33493042, 3.6295547,   -0.43396032, 6.2375803,    1.0340569,    -2.6632495,
    17.096416,    15.208672,   -0.25167805, -3.4815063,  1.0696993,   -5.2873764,   3.1979918,    -0.007851064,
    2.4643226,    1.5116049,   2.2652526,   1.3904041,   -1.4545498,  0.3900972,    -2.9886885,   0.8594112,
    -2.6304812,   1.0031368,   15.539574,   17.902292,   -3.2155356,  -2.029044,    3.1789296,    1.7891771,
    5.1005206,    -0.47131428, 2.3852003,   1.3723673,   -1.1129482,  2.6773098,    3.0800571,    3.9973392,
    -3.7624567,   1.2616894,   -4.292824,   -1.0895696,  6.214655,    -5.0111494,   -1.411555,    0.7537558,
    0.16763747,   -2.843416,   1.3127775,   -1.9945961,  1.9270117,   -1.8245299,   0.14755654,   0.119710535,
    0.4844484,    -2.400567,   -0.7895191,  0.11959961,  2.022987,    1.5247601,    2.940802,     -5.3442945,
    -1.244915,    0.3108465,   -1.0918118,  -1.5366958,  0.7903812,   -2.396515,    0.7696649,    -0.8931735,
    -0.37421072,  -2.557737,   -0.7477829,  -1.0872394,  -1.3062975,  -1.1412712,   0.78160894,   1.5905662,
    4.9873405,    -6.715282,   -1.6470392,  -0.42781863, 0.28193593,  0.7067425,    -2.6839638,   -0.57462484,
    1.5813833,    -1.840177,   0.5190145,   2.371542,    0.2732622,   0.15681863,   0.59334445,   2.0408263,
    2.5408773,    2.1669984,   2.546817,    -6.3569403,  -2.1067922,  1.5111246,    0.5735901,    -2.7920475,
    -1.0018231,   0.92297626,  1.4706146,   -1.4367175,  0.91449654,  -1.5520309,   0.19974053,   -0.20228307,
    2.309344,     0.36271697,  0.69444144,  -1.5499384,  2.2808971,   -4.045352,    -3.9713645,   0.18369448,
    1.7138574,    0.91558576,  -2.5425308,  -1.1829549,  0.2302165,   -0.30635485,  0.61019206,   -1.425853,
    -3.5264308,   -1.3583288,  -0.19054441, 0.04516536,  -1.9273582,  2.771572,     3.5409505,    -2.9621074,
    1.0486665,    1.2725343,   0.65368336,  2.2211008,   -2.333774,   -3.4962246,   1.3083582,    -0.9263934,
    2.2978144,    0.9176514,   -0.6095743,  -0.518545,   0.30718052,  2.4780507,    -1.0288026,   -1.3045118,
    5.109543,     -3.7778285,  -1.1729475,  -1.3083767,  1.1134523,   0.1949772,    -0.097847104, -0.55074453,
    3.2532492,    -0.6656364,  1.5148337,   -1.4027967,  0.028162748, 1.0119103,    0.8516027,    -1.6230793,
    2.1586115,    2.5022216,   5.989767,    1.5173278,   -1.8896376,  -2.3683255,   -1.0292325,   -2.0638425,
    2.086112,     -0.5161866,  -0.77660584, 0.7603841,   -1.6347449,  -0.068522215, 3.9827218,    1.4527302,
    -0.121331334, -1.3804352,  -0.89542794, -1.1340978,  4.0029855,   -1.0835376,   -1.1934209,   1.8954589,
    -0.8401254,   -1.9993141,  2.1870675,   -2.8295417,  -2.0395064,  0.68151194,   -1.0094807,   -0.923614,
    -1.7888981,   1.7585123,   -1.8216331,  0.37374687,  0.9950139,   2.351936,     6.2798033,    1.7030699,
    -0.47087464,  -1.6360345,  -4.468318,   1.9200281,   -0.09482679, -0.043346405, -0.9715949,   -0.34371126,
    0.09224248,   -0.1786593,  0.65415484,  3.1819649,   -1.7768605,  2.832635,     -3.1208003,   -2.8149338,
    5.781748,     0.034614563, 0.8586121,   -0.7064364,  0.77345586,  1.713623,     1.2533236,    0.9109094,
    3.2923024,    -1.0421813,  1.0721587,   -1.610004,   0.881869,    1.9586189,    -1.6461011,   0.88225365,
    1.3470546,    2.804223,    3.3672202,   3.4784017,   -0.15211546, 0.04246044,   3.7608738,    -0.26021224,
    -0.023485124, 3.0048823,   2.4691834,   -1.2352742,  1.824518,    0.97959256,   3.5056772,    0.37062028,
    -0.5023904,   -2.280041,   -1.4408232,  0.74235415,  0.77733827,  7.157131,     2.0188837,    -2.4676952,
    2.2076206,    -0.9512329,  2.4511347,   1.5196023,   -0.66184056, -1.0361224,   -2.452705,    -1.6693871,
    0.67556,      0.7263571,   -3.562127,   3.6935072,   2.6473353,   0.34725356,   1.2102544,    5.872654,
    0.78656757,   -1.734544,   0.10659918,  -1.5254192,  -0.4401021,  -2.3011737,   -1.5704286,   3.2446222,
    -0.09058446,  1.2970262,   0.03513348,  -2.1620555,  -2.3477712,  1.0844729,    -0.31028935,  -0.2960415,
    -1.6364298,   5.4394045,   0.4011662,   1.0095837,   0.6786694,   -0.16024262,  -2.0813947,   0.82111263,
    2.0470252,    1.2589197,   0.16413942,  1.1889694,   -1.1947216,  0.09492111,   -1.2447665,   2.2591739,
    1.107242,     -3.266813,   1.62448,     5.5707865,   1.4339288,   0.5752156,    2.1131587,    -1.9095923,
    0.44389912,   2.6914995,   1.3760473,   0.59850556,  -1.7392722,  0.6407434,    0.7244801,    0.6499002,
    0.8710161,    2.361711,    -0.88572574, 0.36826122,  0.83188176,  5.91977,      1.3777022,    -1.1016716,
    5.073045,     -2.9808154,  -1.9151478,  -1.0893023,  -0.19272709, -1.2990279,   -1.1223007,   -0.48658985,
    -1.1106746,   1.1981947,   -1.0553399,  1.1354556,   1.3973879,   -0.95949686,  3.825083,     7.3595786,
    -0.042000458, -3.2074027,  -0.54039013, 0.91688734,  1.3646243,   0.059570312,  -2.4480567,   -3.120977,
    0.3088463,    0.019118637, -1.9716125,  1.1828063,   -2.605321,   0.0031490326, -3.989009,    -1.0951542,
    -3.423747,    0.9767442,   -0.703821,   -0.53229225, 1.8792986,   -2.6091163,   0.073565304,  -3.0182452,
    -1.9846222,   -0.05088371, -1.6033902,  1.1012542,   1.2818389,   -2.9548755,   -2.4526162,   -1.0030507,
    3.313137,     -1.8719615,  -6.977839,   4.587015,    -0.20812428, 0.6176362,    -2.5226798,   1.6537584,
    1.0535446,    0.1687131,   0.12539,     2.351955,    -0.62948394, 0.5153229,    -1.483371,    -0.05357921,
    -0.6367198,   -4.435566,   -1.0412523,  -0.90299845, -5.4293838,  3.2729964,    1.8316314,    0.65426755,
    1.3051615,    2.5043561,   -2.6727161,  -0.51436025, 0.97311413,  0.7939715,    -1.4862274,   1.005588,
    -0.9745222,   2.339565,    -1.0618652,  -2.5664845,  0.19176674,  2.0832286,    -3.072213,    2.863511,
    3.0667965,    -1.0378871,  0.41042465,  0.49165142,  -0.92485595, 3.3757823,    -1.9955772,   -2.7065873,
    1.1462497,    1.7917634,   -0.27417037, -0.1864685,  -1.9386586,  1.0884712,    0.13079014,   1.2462993,
    -5.453458,    3.6369412,   -1.1260786,  -3.395249,   2.7768104,   1.427433,     -1.2371898,   2.7506876,
    -0.9577626,   -0.4295229,  -0.8855097,  0.2148636,   1.3714015,   -2.9233427,   0.51835805,   -0.8493191,
    -1.9358122,   -1.2579556,  -6.63626,    3.854603,    -0.1117461,  0.36959448,   0.46802208,   2.4610949,
    0.4212538,    1.9248259,   -1.3794608,  2.7059088,   1.091836,    -0.37234998,  -0.15367316,  0.66511256,
    -2.183856,    -0.85531116, 1.759064,    0.17355615};

const std::vector<float> expected_dft2d_signal_size_results_2 = {
    13.276131,    13.103815,    12.738701,    10.948082,    15.043786,   10.639425,   12.309946,   12.0796795,
    12.158493,    13.553247,    11.337328,    14.506093,    15.737373,   13.892806,   8.118599,    11.390546,
    0.5381909,    1.3429388,    -0.95012784,  -1.1195304,   0.09536344,  -0.55193377, 0.8235052,   1.1744928,
    2.2546248,    -0.5024011,   0.8459603,    1.935416,     0.035488486, -1.2282364,  -1.4211154,  -0.3135983,
    -0.66958404,  -0.80205476,  0.18983603,   -4.1904936,   2.5468197,   -4.6636457,  -1.8926563,  0.83797,
    0.5354397,    1.2066656,    -0.9827197,   -0.26218012,  -0.11811975, -0.47548425, -1.3096738,  -0.14677164,
    -1.4478016,   -0.6947719,   0.13221443,   1.9222442,    2.1022313,   0.022250533, -2.556076,   0.25012577,
    -1.4458936,   -1.5957909,   1.5911676,    -2.0354197,   -1.226819,   -1.4284164,  -0.8624064,  0.8694984,
    1.7259736,    2.142551,     2.9860144,    -2.218176,    -0.29875967, -0.13178241, -1.4536587,  -0.3602928,
    -0.46407622,  3.210749,     0.51306593,   0.12302576,   0.23700514,  -1.691548,   -1.3992835,  -0.47753286,
    0.42123353,   2.3340597,    -1.8207074,   1.2562377,    -1.1736273,  2.0084949,   0.1217463,   -0.71629894,
    1.7867224,    -1.07459,     -0.46577477,  1.7219527,    -1.7225032,  -0.5029696,  -0.38728696, -1.1263466,
    -0.1653564,   -0.8395238,   1.9879192,    1.0128975,    1.6438143,   -1.188045,   -1.9405808,  -0.7925463,
    1.2240379,    -1.3743367,   2.5557013,    -0.6062887,   -0.99852824, 1.0005589,   2.1218362,   -0.10017598,
    -0.75340086,  -0.6988709,   0.36165902,   -0.13171434,  -0.5997418,  0.2789104,   0.52501214,  -1.5872824,
    0.0040130913, 3.0245605,    -1.34743,     1.8258516,    0.73562264,  -2.1836894,  1.3283473,   0.038802683,
    0.7543056,    -0.16875261,  2.0229807,    -0.9240825,   0.85228086,  -0.9607723,  0.111747265, 0.69834816,
    -2.5481172,   -0.5289763,   -0.5825273,   0.49042946,   2.3490484,   -1.2178793,  1.9920914,   1.7048208,
    0.4666462,    0.944975,     0.44091535,   0.3074813,    2.064869,    -0.8005053,  1.2041193,   -2.94473,
    0.5702777,    -1.4354767,   2.0478277,    -0.8460398,   -0.6000862,  -0.27820373, -0.5037507,  1.0071316,
    0.92846537,   -2.9930074,   -1.0054629,   -0.013398886, 0.31482947,  0.4755001,   -0.37606037, 0.35212934,
    -1.1386731,   0.46239042,   1.0611072,    -2.298436,    1.4551028,   0.3905257,   -1.427774,   0.32839155,
    2.325178,     0.5964565,    1.8381126,    1.1603787,    1.6229985,   0.5935496,   1.6401024,   0.11901784,
    1.2464942,    -0.9543896,   -0.45777965,  0.37926614,   -0.20978808, -0.599459,   0.016958952, -0.672667,
    0.16418982,   -0.5287609,   -0.25067472,  3.268743,     -0.24228573, 0.15815735,  0.60354567,  -2.0075817,
    -1.5436802,   -1.8892074,   -1.8738651,   3.1379614,    -0.02182722, 1.8892908,   0.014802456, -0.90126705,
    1.0671308,    -0.61172044,  -3.7529547,   0.5599045,    1.3885388,   1.425593,    -0.61869496, 0.70866895,
    -2.8543148,   -0.7371676,   -1.1487055,   0.9924042,    -2.1610408,  0.17656213,  1.2943268,   1.038288,
    -1.0522705,   0.62840223,   0.013757467,  -0.108183146, 0.090308785, -0.60490894, -1.4133914,  -0.7655864,
    0.27606705,   -2.2265353,   0.48509985,   -0.37584236,  -0.33032784, 0.112843275, -1.0625036,  -0.6842445,
    0.33563662,   -0.64797795,  1.7336315,    0.24296224,   -0.5694554,  -2.861343,   -2.8949642,  -2.4447322,
    -0.45564282,  -3.1046734,   -2.7624254,   0.2988112,    -2.9488277,  0.44427407,  0.11802065,  0.30626833,
    0.5653825,    1.4086826,    -0.7607212,   -0.36166418,  0.68313515,  -1.9396621,  0.552457,    1.787599,
    0.700689,     -0.075103045, 0.07150927,   -0.2504335,   0.4869701,   0.8627523,   -0.31363714, -3.4648805,
    -0.66644526,  -4.2852783,   1.5780505,    2.0349314,    -0.19262195, 0.6150762,   2.462497,    0.50935733,
    -1.2329292,   -0.32018808,  -0.43475008,  -2.747131,    1.7122769,   2.9606051,   -0.80461985, 2.0191946,
    -0.65316653,  0.10388613,   0.23720229,   -1.2899456,   -0.04140687, 1.0865793,   2.0807414,   0.47735834,
    0.039139986,  0.043996334,  -0.30400705,  0.8887136,    -0.8448317,  0.4798069,   -0.9909992,  0.3054396,
    0.5550651,    -0.62566614,  0.5333818,    -0.56075704,  -0.7468512,  -0.19177368, 1.3637853,   -1.408153,
    0.8004116,    -1.0253057,   -2.049946,    0.6309613,    -0.18224144, 0.7479264,   0.8692721,   1.0770676,
    -0.029760003, -1.4737389,   0.5606033,    1.1633584,    0.064171284, -0.8330089,  2.4364436,   -0.6911875,
    -0.88729143,  0.2826275,    2.2328167,    -0.4202252,   0.5975744,   -0.28619608, 0.87971634,  1.0560545,
    1.3650498,    -2.12536,     1.7964764,    0.5196252,    -1.3630133,  -0.39718735, -0.16041839, -0.018146515,
    0.644505,     -0.7543384,   0.39063865,   2.3242626,    0.18570781,  -0.33447194, -0.8394531,  0.06412578,
    3.9682813,    -1.1106789,   3.655115,     -0.25420773,  -0.12754375, 0.22988068,  -0.13729298, -1.0493382,
    -2.288222,    -1.1528109,   0.0876067,    1.4938865,    0.6729909,   0.6704601,   0.12003565,  -0.64320755,
    0.4911754,    -1.6283998,   -0.6994369,   -0.9077794,   -0.61947733, -0.8598275,  -1.0383508,  -3.1879523,
    0.6378788,    0.5077131,    -0.099037886, -0.48517847,  3.027417,    0.30669552,  -2.9577267,  3.5811238};

const std::vector<float> expected_dft2d_signal_size_results_3 = {
    6.235876,    5.533142,     -1.1544118,   0.082814306, 1.6796162,   0.599913,     -0.38694367,  1.4678228,
    7.972584,    7.075794,     0.43039304,   0.068965316, 0.9859961,   0.9107602,    -0.3760583,   -1.7655449,
    6.727815,    7.016465,     0.08040795,   -0.27730647, -0.24473351, -0.28742158,  -1.7693194,   0.9797714,
    7.944232,    3.3173547,    -0.970581,    0.2367388,   0.6001234,   0.73869,      0.78396153,   2.381297,
    6.578037,    5.254505,     0.22289926,   -0.50469196, 0.81392545,  -0.6079184,   -1.8542231,   -1.4371969,
    5.182848,    7.0792365,    1.7477604,    2.5558662,   -0.4264726,  -0.1144461,   0.13226332,   -1.0341146,
    1.5301563,   -2.5123873,   0.55265933,   -0.6910203,  0.8812098,   -0.38004306,  -0.86332023,  -0.09744531,
    0.022181302, 0.0048598824, -0.51295733,  -0.1617982,  -1.5669001,  -0.89008003,  0.6439479,    -0.7541246,
    0.09064455,  -1.8462008,   0.9278267,    0.029414916, -0.8102952,  0.32216373,   -1.0387933,   0.8412336,
    -0.8255059,  0.24349298,   -1.9935097,   -0.10879634, -0.66851157, -1.0654241,   0.17485179,   0.15639359,
    0.4183568,   0.9885812,    0.41185328,   1.5258284,   0.9891837,   -0.30820954,  -1.5363256,   1.1204776,
    -0.9755474,  1.7845668,    0.43930066,   0.14675245,  1.5588113,   0.501778,     1.3166,       -0.54990625,
    2.405043,    -0.7384981,   2.135238,     -0.6406439,  -1.2197473,  -1.2720709,   -0.040249567, 0.3340183,
    -0.69982177, -0.71987194,  -1.2410803,   -0.98330003, 0.27524203,  0.9393509,    -1.0231966,   -0.89418787,
    -1.5361586,  0.7400294,    -0.4797501,   -0.20026025, 0.28715983,  0.4135941,    0.31160337,   1.45322,
    0.8597624,   1.4913996,    0.118060485,  0.04817532,  0.24861759,  -0.086301126, 0.7859658,    0.43219715,
    -0.15880768, -0.37046334,  -0.058427185, -2.141556,   -0.6823787,  -0.56347156,  0.5663684,    0.14823253,
    0.8226858,   0.33215827,   -0.35511267,  0.6276708,   -1.7566651,  0.27827078,   -1.8008664,   -0.68004596};

const std::vector<float> expected_dft2d_signal_size_results_4 = {
    16.022383,   15.693684,    -1.6079187,  -2.5211797,  -0.5771674,   0.05202335,   -0.20892644, 2.2068956,
    3.953516,    0.34410894,   0.86641574,  1.6062691,   -0.28576106,  5.39313,      0.29631865,  0.4604528,
    19.077686,   18.455147,    -0.6079974,  -2.6523724,  1.5708399,    -1.4308838,   -0.12822962, 0.39401102,
    3.578989,    1.7037572,    1.2102947,   1.9145584,   -0.9519639,   -1.1172407,   0.3622353,   0.057706878,
    15.384689,   17.809353,    0.26336426,  1.1215441,   -1.5200262,   -1.8461118,   2.1862373,   -0.8841291,
    0.40979373,  -1.3552474,   -1.1214476,  0.25225526,  -3.9052691,   0.16070783,   0.47419822,  0.84322566,
    17.599613,   15.044547,    -1.1941085,  -3.8985834,  -3.0904906,   0.7600602,    0.62432945,  -2.1925209,
    2.2314792,   2.2408223,    -0.58379686, 1.5055354,   -0.43706644,  3.6915889,    4.6112394,   -1.8686706,
    17.096416,   15.208671,    0.26360607,  -0.43837237, -2.085052,    -1.9803678,   0.5512935,   -0.7585812,
    1.3307043,   -0.20800388,  0.64899683,  -2.204393,   -0.7423673,   -0.53894585,  -0.3806771,  1.2191849,
    15.539573,   17.902292,    -3.2540998,  1.6752851,   2.339312,     2.9226294,    1.2905213,   -0.86016166,
    0.2067287,   -0.46054435,  0.845206,    4.1277647,   -1.0080593,   -1.8169241,   -0.9365012,  -0.9034538,
    4.193228,    -1.6578128,   1.0010736,   0.13147289,  -0.58524096,  -0.6177359,   -0.5433382,  -0.91701794,
    -0.34512973, -1.1603408,   0.82712364,  -1.9715309,  -1.3202763,   0.7925887,    1.5136781,   1.1887463,
    0.07894993,  0.2629676,    0.47720033,  -0.6143549,  -0.35804892,  -0.7087485,   -0.9317253,  -1.5261502,
    0.012433767, -1.4340608,   -2.3467493,  -0.11921686, -0.7176709,   -0.9460896,   1.1034908,   0.98994523,
    1.2861536,   -2.767478,    -1.6691985,  0.5312684,   1.0004808,    -1.4490643,   1.6858988,   1.9467357,
    -1.6837163,  0.61909866,   0.46550757,  -1.377079,   0.03022629,   0.6342612,    1.8735541,   0.8252406,
    -0.46249843, -0.7055974,   1.0481787,   1.154592,    -3.1120033,   -2.5491147,   1.5315402,   0.24330461,
    -0.27145922, -1.1044617,   -1.4539453,  -1.0249764,  0.64292955,   -0.25380075,  1.3825793,   -2.2184052,
    0.58045006,  0.9588773,    -2.0319357,  2.364854,    0.9267142,    -1.7276525,   -1.258892,   2.0606081,
    0.17469049,  0.037098885,  -2.7965002,  -2.0710194,  -0.066367805, 0.9294369,    -2.0530214,  3.1182497,
    0.7525606,   1.0215833,    0.7231224,   -0.76680106, 3.8408241,    0.66357744,   -1.4999955,  3.0092032,
    -0.9077265,  1.127433,     -1.3998711,  -0.45227063, 0.5452228,    1.1795954,    -1.2053492,  -2.3661485,
    -2.6609254,  -1.0594559,   0.35282075,  0.15202153,  1.3333919,    -1.7648137,   2.4441655,   0.42185998,
    0.39381158,  -1.9906393,   -1.7294805,  -0.1867466,  -2.1970685,   -3.444776,    2.6147847,   2.4903462,
    0.3241663,   0.63434124,   -0.5939137,  0.22729027,  -0.8494644,   0.54981613,   -1.7602688,  -3.5211835,
    -0.07873356, -0.1510309,   2.882863,    1.0030751,   -1.8153281,   -3.1542742,   -1.0870701,  0.0820543,
    -2.004652,   -1.2403183,   0.71638805,  1.2452526,   0.34644645,   0.57313305,   -4.812693,   0.5708023,
    -0.5506052,  -0.13743436,  -0.5721044,  1.3499863,   -1.2981024,   0.0077233315, 3.7563763,   1.8381448,
    1.2751684,   0.082980156,  -1.1809301,  0.38965702,  1.9302576,    0.9432045,    0.5983294,   2.1648314,
    0.8428469,   1.4977492,    -0.7804593,  3.1802852,   -2.1036444,   2.1590552,    -1.9935954,  -1.013362,
    -0.6313343,  -0.019762993, -0.65470386, -0.48428255, 5.459507,     -1.2114509,   1.7786237,   -0.7012754,
    0.17181313,  -2.092629,    -0.65052056, -0.41800314, -1.3612752,   0.039184153,  1.7546436,   1.3561778,
    0.54778004,  4.72955,      1.3787336,   -0.63834924, -0.019961834, -0.8124193,   3.769576,    -0.23387921,
    -0.9165416,  -0.99439096,  0.7847799,   -2.6583774,  -1.6555126,   -2.815217,    -0.18486327, -1.1745405,
    2.2054548,   -0.9455488,   -1.5059378,  -0.6565779,  1.2600167,    -2.3821032,   -0.26981768, -4.063028,
    -0.45026755, -1.834182,    2.906159,    1.1662107,   0.08017355,   -0.8102168,   3.3697448,   0.37883544,
    -1.6882677,  0.71782255,   -1.5592864,  -0.37134206, -3.2504182,   1.2694929,    -1.7516978,  -0.12049621,
    1.3492393,   -0.40591407,  0.6280788,   -1.0120616,  2.101255,     -0.7040838,   -1.1866775,  -0.7236214,
    -0.8188298,  1.1767912,    1.1913915,   0.6185412,   0.9365735,    1.3821521,    -1.8589101,  0.9124049,
    -0.83333874, -0.9172766,   -2.0438805,  0.469943,    1.4887323,    0.24271065,   -2.0128174,  1.3354056,
    1.5705173,   0.6380501,    2.0443192,   -1.407867,   -0.6085211,   0.13712549,   1.9739413,   1.8154222,
    -3.012415,   0.83554983,   0.5828651,   0.14901292,  -0.8463185,   -0.047633052, -0.5389695,  0.41219842,
    -0.61554337, 0.45593047,   -1.4136335,  -3.6993682,  0.33988523,   -1.7985406,   1.1319201,   1.5479275,
    -0.15549183, 0.1671977,    1.4381963,   1.5354156,   -0.64630884,  -0.2005459,   -0.8759377,  -2.1679733,
    -1.130661,   0.052789927,  -2.0507228,  0.55622774,  0.4106456,    1.1299163,    -0.15413862, -0.1215477,
    0.5790715,   3.4873276,    -0.38861734, 1.1647955,   -0.7212916,   -1.055282,    -0.4247969,  2.521102};

const std::vector<float> expected_dft2d_signal_size_results_5 = {
    8.798115,    7.435564,   -1.856497,    0.6967675,   1.9218704,    0.8142178,    0.36594042,   2.6711423,
    10.539949,   8.829714,   -0.12551129,  -0.56251144, 1.2948477,    1.2702878,    0.34664208,   -0.8751477,
    7.8924866,   9.178925,   0.10114479,   -0.25878292, 0.10166702,   -1.4982777,   -2.0095286,   0.6289345,
    10.036068,   5.0261283,  -1.8025928,   -0.1469695,  0.32685816,   0.24962878,   1.3202658,    2.5126028,
    9.108963,    7.5209365,  -0.6332305,   -1.1228236,  1.1512439,    -0.38064647,  -1.2855166,   -0.8678701,
    6.9929824,   8.779736,   1.4174356,    3.4299555,   -1.5252162,   0.26026887,   0.6261386,    -1.176516,
    3.6406178,   -3.1231828, 1.3689959,    -0.68496245, 0.48079437,   -0.796133,    -1.0121856,   -0.5319141,
    1.5931433,   -2.9828,    0.30207276,   -0.6309908,  0.016751591,  -1.5604393,   -1.6819947,   -0.6282208,
    3.1937852,   -3.9242973, 0.14749089,   0.43185335,  -0.2489954,   1.0694493,    -0.22732289,  1.4830055,
    0.12429348,  -3.6180806, -0.078341216, -0.84553397, 1.3277338,    -0.93593657,  1.1153932,    0.46926653,
    -0.01660335, -2.19146,   0.0050742924, 0.16887031,  -0.13835368,  -0.22266653,  -3.1048126,   0.28844416,
    0.7672101,   -1.3757203, 1.4307625,    -0.8216129,  0.5319145,    0.6184877,    -0.42566353,  0.5484251,
    3.4520624,   -2.2700894, 0.48456866,   0.50225604,  1.4660833,    -0.8105061,   0.26780176,   0.08084947,
    3.162387,    0.07700956, -0.92056286,  0.15617561,  -1.4525607,   -0.030844629, 1.7492342,    -1.8916597,
    1.9068949,   -0.5026501, 0.9467689,    -0.5010593,  -1.7409866,   -0.12933461,  -1.7828977,   1.2331147,
    2.161806,    0.7638128,  -2.586247,    1.2947125,   -1.5071061,   -0.28886533,  0.20840368,   0.28412366,
    4.238263,    0.624877,   0.7483618,    1.7655015,   1.4296482,    -1.0612158,   -0.36541831,  0.8688911,
    1.9176203,   2.7282064,  0.99068373,   1.069428,    1.4693688,    0.456266,     1.3150638,    -2.101551,
    2.3444064,   3.1458972,  0.9041112,    -0.18337753, -0.4628029,   -1.5899987,   -0.96295846,  1.8805511,
    0.9849721,   2.3036761,  -0.21096146,  -1.9477112,  -0.28516757,  1.7867885,    -1.6535637,   -0.59629035,
    -1.8184824,  2.2060392,  -0.3854983,   -0.6308861,  1.3785319,    0.6067129,    0.0436399,    1.6603687,
    1.0021243,   3.9316323,  0.0843097,    -0.1404491,  0.55632716,   -0.1349278,   1.0070219,    1.4757582,
    0.6121656,   2.6698961,  1.4586289,    -3.1324296,  -0.5294236,   -0.8063517,   -0.08189219,  1.4184481,
    1.7388614,   3.607197,   -0.8525236,   0.45530546,  -2.311192,    -1.7529185,   -1.2852949,   -1.3684388,
    -1.2834095,  -1.3844275, 1.6546304,    -2.4121,     -1.1708138,   0.62875193,   -0.80945396,  -1.2599692,
    -4.1222115,  2.373704,   -1.2511115,   1.191483,    -0.0833078,   0.1342597,    -0.019162834, -1.6984439,
    -2.3708487,  2.8924727,  0.07090135,   0.21195525,  -0.7699983,   0.69867724,   -0.18473946,  0.4516183,
    -0.02681084, 2.3169198,  -0.36051208,  0.13176969,  -0.40343064,  0.42170894,   -0.7431194,   0.20806183,
    -2.546812,   1.1634552,  -0.6182922,   0.45351535,  -0.045230594, 0.0048814267, 0.13067257,   -1.9887246,
    -3.0333633,  1.5871837,  0.06688923,   1.4174063,   0.79458106,   1.5272335,    -0.8169159,   0.32463634};

const std::vector<float> expected_dft3d_signal_size_results = {
    49.728928,    45.265354,   -1.5653555,   1.8258603,  1.1586399,    1.1093023,   3.4622312,   -3.068374,
    -1.6144468,   3.955975,    0.7007327,    -12.262681, -1.6978629,   0.005776912, 2.1834314,   4.4191337,
    -0.41734934,  1.0642323,   2.1575608,    -1.6815078, 13.372203,    -10.360054,  -1.2915184,  -4.100011,
    -2.7976396,   2.0324564,   -0.87680733,  -5.934458,  -0.45711768,  6.1547985,   7.1691613,   1.8739722,
    3.5988164,    2.9202309,   1.6726661,    1.2348467,  -0.100646675, -1.0087874,  -0.26146126, -2.555842,
    -0.054751158, 11.550103,   -3.3944821,   -1.6603465, 1.6404665,    1.0531695,   -3.6879756,  2.841142,
    -0.9843147,   8.387482,    -5.754649,    6.6857386,  -5.0392313,   -0.09710765, -1.665895,   3.3698869,
    2.4361796,    -1.1781744,  9.256103,     -2.0982075, -10.405502,   6.6115084,   4.9817243,   -0.2607507,
    -1.7066463,   2.1667802,   -0.124430835, 0.9151207,  -3.847056,    -1.9442594,  0.9573264,   -3.7053752,
    1.6387817,    0.4028574,   1.9858243,    1.7182893,  -3.1588082,   -0.20052435, -2.4657276,  -0.34011212,
    -1.8019962,   -3.6929393,  -0.52881837,  0.70309484, -4.9345665,   -0.51714915, 2.9993763,   1.8249977,
    1.7067848,    -0.3447387,  3.3535786,    -3.3786156, -0.066281304, -3.208472,   2.4551795,   -1.3978454,
    -0.6575866,   -1.5296084,  -2.9372633,   2.5082207,  4.9071603,    -1.0131404,  -0.35466444, 3.4279532,
    -0.5829768,   1.2354829,   2.7378159,    -2.277992,  -3.9665284,   -1.6454957,  -2.1138058,  -6.9444704,
    6.667962,     -2.4544113,  0.8878386,    1.802904,   1.0337299,    -0.83422416, 3.2563305,   4.0551376,
    0.9553633,    -0.1692334,  -3.562861,    1.3307956,  -0.59709,     1.4318806,   -1.5622483,  -2.975128,
    0.5372238,    0.39611292,  5.9997783,    -4.276651,  2.7130606,    -4.0972834,  3.100973,    -4.163089,
    0.6286693,    -2.0351622,  -0.85036594,  -1.843902,  0.81661844,   0.86445946,  0.66823524,  -3.632821,
    -2.0656922,   -2.796157,   -1.0641904,   1.0206027,  -1.1506183,   1.9784743,   0.97015285,  0.84469545,
    -1.6403551,   -0.3596968,  3.9319327,    -3.472964,  1.9130713,    -1.3176007,  -0.56489336, -0.21743466,
    1.0296725,    -2.3882782,  3.0200386,    -1.5742422, 2.092337,     0.9063153,   -2.189577,   1.3231093,
    -0.4461195,   1.6234826,   3.447591,     -1.9438455, 2.598078,     2.8626804,   0.66956127,  -4.3974814,
    -0.47485933,  -0.22436841, 0.06777835,   5.5086756,  0.47679606,   2.6178322,   -0.02077597, -0.9915889,
    2.167689,     -2.8916657,  -1.9383656,   3.241434,   -0.69579893,  -0.63952047, 3.3214207,   -2.407516,
    5.2349954,    -6.0218415,  5.819358,     -7.4365478, 2.8206537,    2.7585952,   1.7629972,   0.044816617,
    1.1392056,    -4.696983,   0.45275614,   1.9134089,  -3.8572056,   -2.009159,   1.6307822,   -0.9646755,
    -1.2407924,   2.6003554};

const std::vector<float> expected_idft2d_signal_size_results_1 = {
    0.35605296,    0.34874853,     -0.05017762,   0.019962106,   0.019251583,   0.13608973,    -0.014922479,
    0.04271815,    0.010398432,    0.025212493,   0.0796304,     0.11609333,    0.038195945,   0.044737395,
    0.03563052,    -0.026620463,   -0.01258824,   -0.12605664,   0.4239486,     0.41011444,    -0.061647452,
    0.00044958064, -0.06847432,    0.047905806,   0.062185094,   0.06219103,    0.058658686,   0.021337016,
    0.052691586,   0.08597042,     0.047829624,   -0.0079172375, 0.08269539,    -0.049927484,  0.004909183,
    -0.13700642,   0.34188202,     0.3957634,     -0.05613245,   0.002547276,   -0.12465724,   0.078391984,
    -0.024422595,  0.034788877,    -0.009830318,  0.008629954,   0.045807023,   0.0015473965,  0.08764248,
    0.026886743,   0.048212234,    -0.10233939,   -0.00421264,   -0.043676328,  0.39110252,    0.3343233,
    0.022979043,   -0.059183322,   -0.009643566,  0.13861291,    -0.0074428986, 0.080656774,   0.03761088,
    0.025686212,   -0.0036547848,  0.07562487,    0.107067585,   0.009425696,   -0.02972116,   -0.06653358,
    -0.014267657,  -0.15654333,    0.37992036,    0.3379705,     -0.05845514,   0.02229193,    -0.0664153,
    0.01909803,    -0.03232333,    0.008668827,   0.050338943,   0.030897861,   0.054762725,   0.03359122,
    0.071066484,   -0.00017446809, 0.02377109,    -0.11749724,   -0.005592843,  -0.077366814,  0.34532386,
    0.39782873,    -0.09539609,    -0.024212658,  -0.08361015,   0.028037548,   0.06844571,    0.08882976,
    -0.024732176,  0.059495773,    0.05300445,    0.030497046,   0.11334491,    -0.010473651,  0.07064287,
    0.03975949,    -0.07145635,    -0.045089867,  -0.07608327,   0.021705426,   0.07362527,    -0.041599143,
    -0.054502584,  -0.02229002,    0.02848531,    -0.065663904,  -0.035630893,  0.024472319,   -0.04410272,
    -0.0011307478, 0.0016347846,   -0.067072116,  0.04176219,    -0.057980362,  -0.015640466,  -0.011828707,
    -0.1550631,    0.10193367,     -0.023138944,  -0.020066634,  -0.014149333,  -0.098568134,  -0.0329638,
    -0.0011906491, -0.013988529,   0.011451623,   0.0027864417,  0.05226567,    0.023412103,   0.00374918,
    -0.056059547,  0.036750186,    -0.0046249777, 0.013725244,   -0.12065297,   0.07273325,    0.0042614806,
    0.046293974,   -0.023597008,   -0.057033,     -0.02165605,   0.051990334,   -0.033027273,  0.022346405,
    0.021624766,   0.017643817,    -0.059393693,  -0.011430228,  0.029003588,   0.05565236,    0.04070292,
    0.014539276,   -0.0682714,     0.06363358,    0.0029064484,  0.02769554,    -0.043081302,  0.024188243,
    -0.006092675,  -0.0041437447,  0.025472214,   0.039816964,   -0.044346157,  -0.060146388,  -0.020552356,
    0.075017385,   0.009120548,    0.0109255845,  0.068151034,   -0.023064155,  -0.121187955,  0.08082092,
    -0.043018054,  -0.02795456,    0.011519067,   -0.01887376,   0.03047559,    -0.06496317,   -0.019677995,
    0.004774744,   -0.021283617,   -0.009544955,  -0.027493106,  0.061126392,   0.0617069,     0.031720735,
    -0.025023961,  -0.075449966,   -0.14747246,   0.08565785,    0.039090306,   0.0038568007,  -0.048530135,
    -0.019006917,  -0.0034149592,  0.014780279,   0.024263024,   -0.00827444,   -0.030654684,  0.06013131,
    0.009361195,   0.04277391,     0.010400491,   0.054690998,   -0.0024832455, 0.008213211,   0.017274184,
    0.15904737,    0.058829676,    0.007716746,   -0.07915838,   0.08207793,    0.015012445,   0.01614127,
    -0.05450456,   -0.037097495,   -0.014707568,  -0.023024952,  0.05446966,    0.03376894,    0.049058244,
    -0.021138508,  0.04486409,     -0.054837674,  0.026894543,   0.13050343,    -0.006895314,  -0.006578704,
    -0.052172694,  0.024099395,    0.00078074407, -0.04804568,   -0.0020129893, 0.028822804,   -0.034898415,
    0.07210272,    -0.0097800465,  -0.051137194,  0.0023688707,  -0.033898205,  0.017479278,   -0.03854542,
    -0.036365107,  0.12087566,     0.024605379,   -0.07259585,   -0.027661474,  0.05020386,    -0.02654937,
    0.002109358,   0.0036475442,   0.026421547,   0.045489445,   0.027975991,   -0.046253216,  0.018246949,
    0.015081541,   -0.0035609512,  0.008914807,   0.022435194,   0.036099557,   0.123795256,   -0.019682797,
    0.0081835855,  0.01935591,     0.052482475,   0.016099557,   0.014442227,   -0.038650487,  0.014238742,
    0.030578831,   0.013300121,    0.009864425,   0.0598111,     0.046959084,   -0.042435378,  0.03186508,
    0.012782564,   0.018486261,    0.13155045,    0.031053057,   -0.02132215,   -0.023451997,  0.02523235,
    -0.02468166,   0.02662655,     -0.024940018,  -0.0108131105, -0.0042828214, -0.028867293,  -0.04255884,
    -0.024206718,  0.11273433,     -0.066240355,  0.0306156,     -0.024481589,  0.08500185,    0.16354619,
    -0.088644646,  -0.024336763,   -0.05789602,   6.998115e-05,  -0.043813612,  0.026284585,   0.0068632537,
    0.0004248606,  -0.05440126,    -0.06935504,   0.030324984,   0.0013237847,  -0.012008667,  0.02037527,
    -0.0009333448, -0.071275614,   0.1135454,     -0.08395175,   0.047969148,   0.055604927,   0.0189245,
    -0.03606843,   0.0006258389,   0.022486897,   0.03366298,    -0.031173265,  0.07229443,    -0.014791926,
    -0.0021743802, -0.012238768,   0.024743382,   0.0043328307,  -0.026065504,  -0.02907504,   0.13310593,
    0.033718396,   -0.0198984,     -0.025202168,  -0.002696252,  -0.030676337,  0.08850493,    0.032282893,
    -0.036327668,  -0.0015227132,  -0.017257914,  0.016897412,   0.046358045,   -0.011470813,  -0.022871837,
    -0.04586317,   -0.041991945,   -0.052629452,  0.08895524,    -0.024078613,  0.022111418,   0.05226525,
    -0.040480733,  0.008305485,    -0.03975329,   0.03907805,    -0.022432908,  -0.020524755,  -0.045322362,
    0.015144711,   0.0486015,      -0.062878706,  -0.018669453,  -0.0444292,    -0.026520465,  0.04212131,
    0.13955119,    0.037846,       -0.069351114,  -0.062554084,  -0.039485786,  0.06294745,    0.014536775,
    0.07071033,    0.0020498342,   -0.003970201,  -0.021590997,  -0.007638035,  -0.002107262,  -0.0009632535,
    -0.099295944,  0.04266729,     -0.010463878,  -0.036356337,  0.12848328,    0.00076921255, 0.029934544,
    0.062316064,   -0.036580022,   0.019605635,   0.019597089,   0.043524865,   0.02382575,    -0.035777867,
    0.07316227,    -0.023159584,   0.027851636,   0.020242432,   0.017187908,   0.038080517,   0.019080264,
    -0.015698584,  0.07482712,     0.077297814,   -0.0320183,    0.016496757,   -0.011164234,  -0.050667573,
    0.07790394,    0.008236007,    0.040544845,   0.021768719,   0.054870743,   -0.027450528,  -0.00052189163,
    0.066775165,   0.08357497,     -0.0057824906, -0.0033803424, 0.0009435627,  0.13810344,    -0.11135888,
    0.044955265,   0.033883553,    -0.017544864,  0.0026577704,  0.01076552,    -0.053345937,  0.0032790357,
    0.0026602354,  0.04282249,     -0.04054511,   0.029172834,   -0.04432436,   0.003725277,   -0.063187025,
    -0.031367887,  0.01675014,     0.06535116,    -0.118762106,  0.017369088,   0.035345916,   -0.029028835,
    -0.02536158,   -0.016617399,   -0.024160875,  -0.008315794,  -0.056838598,  0.017103666,   -0.0198483,
    0.017564027,   -0.05325589,    -0.024262477,  -0.034148794,  -0.027664784,  0.006907692,   0.11082979,
    -0.1492285,    0.056463942,    0.048155528,   0.013185428,   0.045351695,   0.0060724937,  0.0034848584,
    0.011533654,   0.052700933,    0.03514185,    -0.040892825,  -0.05964364,   -0.012769441,  0.0062652444,
    0.01570539,    -0.036600873,   -0.009507085,  0.056595936,   -0.14126535,   0.0154320365,  -0.034443073,
    0.051318757,   0.00806037,     0.0044386787,  -0.0044951793, 0.020322146,   -0.034489576,  0.03268032,
    -0.031927057,  -0.022262735,   0.020510584,   0.012746443,   -0.062045496,  -0.046817604,  0.033580557,
    0.050686605,   -0.089896716,   -0.04283018,   0.061590493,   -0.0042343163, 0.0010036799,  -0.07836513,
    -0.030185085,  0.013559818,    -0.031685624,  0.005115925,   -0.00680789,   -0.056500684,  -0.026287887,
    0.03808572,    0.020346357,    -0.088252544,  0.0040820944,  0.078687795,   -0.06582461,   -0.022862282,
    -0.028989157,  0.006826234,    0.055067796,   -0.013546096,  -0.011523222,  0.051062543,   0.020392258,
    0.029074619,   -0.020586526,   -0.051861648,  -0.07769388,   0.014526297,   0.049357798,   0.023303697,
    0.028278539};

const std::vector<float> expected_idft2d_signal_size_results_2 = {
    0.5531721,     0.5459923,     0.53077924,    0.45617008,    0.62682444,    0.4433094,     0.5129144,
    0.50332,       0.5066039,     0.56471866,    0.47238868,    0.60442054,    0.6557239,     0.57886696,
    0.33827496,    0.4746061,     0.017551398,   0.09725249,    -0.07586281,   0.052343242,   -0.048901137,
    0.08368729,    0.0050727627,  -0.02984579,   0.07444677,    -0.044774584,  -0.019407284,  0.07174803,
    -0.071770966,  -0.020957068,  -0.016136957,  -0.04693111,   0.07191557,    0.08927296,    0.12441727,
    -0.092424,     -0.01244832,   -0.005490934,  -0.060569115,  -0.0150122,    -0.01933651,   0.13378121,
    0.021377748,   0.0051260735,  0.009875215,   -0.070481166,  -0.058303483,  -0.019897204,  -0.060325067,
    -0.028948829,  0.0055089346,  0.08009351,    0.087592974,   0.00092710555, -0.10650317,   0.0104219075,
    -0.06024557,   -0.06649129,   0.06629865,    -0.084809154,  -0.05111746,   -0.05951735,   -0.0359336,
    0.0362291,     -0.027899336,  -0.03341895,   0.007909834,   -0.17460391,   0.10611749,    -0.19431858,
    -0.078860685,  0.034915417,   0.022309989,   0.050277736,   -0.040946655,  -0.010924172,  -0.004921656,
    -0.019811844,  -0.054569744,  -0.006115485,  0.022424622,   0.055955783,   -0.03958866,   -0.0466471,
    0.0039734766,  -0.022997241,  0.034312718,   0.0489372,     0.0939427,     -0.02093338,   0.035248347,
    0.080642335,   0.0014786869,  -0.05117652,   -0.059213143,  -0.013066596,  -0.027215272,  0.004328589,
    0.009883429,   -0.053747736,  -0.0017252862, 0.04527414,    0.08669756,    0.019889932,   0.0016308328,
    0.0018331807,  -0.012666961,  0.037029736,   -0.035201322,  0.019991955,   -0.041291635,  0.01272665,
    0.020465642,   -0.06784999,   -0.029143205,  -0.037824143,  -0.025811557,  -0.035826147,  -0.04326462,
    -0.13283135,   0.026578283,   0.021154713,   -0.0041265786, -0.02021577,   0.12614238,    0.012778981,
    -0.123238616,  0.1492135,     0.16534506,    -0.04627829,   0.15229645,    -0.010591989,  -0.005314323,
    0.009578362,   -0.005720541,  -0.04372243,   -0.09534259,   -0.04803379,   0.0036502792,  0.062245272,
    0.028041288,   0.027935838,   0.0050014853,  -0.026800316,  0.056877077,   -0.08855667,   0.07485318,
    0.02165105,    -0.056792222,  -0.016549474,  -0.0066841,    -0.0007561048, 0.026854377,   -0.031430766,
    0.016276611,   0.09684428,    0.0077378256,  -0.0139363315, -0.034977213,  0.0026719074,  -0.0012400001,
    -0.06140579,   0.023358472,   0.04847327,    0.0026738036,  -0.034708705,  0.10151848,    -0.02879948,
    -0.036970478,  0.011776146,   0.09303403,    -0.017509384,  0.024898935,   -0.011924837,  0.03665485,
    0.044002272,   0.023127712,   -0.026069423,  0.022224244,   -0.023364877,  -0.0311188,    -0.007990571,
    0.056824386,   -0.058673047,  0.033350483,   -0.042721074,  -0.085414425,  0.026290055,   -0.0075933933,
    0.031163601,   0.03621967,    0.04487782,    0.0068412423,  -0.022031706,  -0.010444781,  0.13619763,
    -0.010095239,  0.0065898895,  0.025147736,   -0.08364924,   -0.06432001,   -0.07871698,   -0.07807772,
    0.13074839,    -0.0009094676, 0.07872045,    0.000616769,   -0.037552796,  -0.027768552,  -0.17855327,
    0.065752104,   0.084788814,   -0.008025914,  0.025628176,   0.10260405,    0.021223223,   -0.05137205,
    -0.01334117,   -0.018114587,  -0.1144638,    0.071344875,   0.12335855,    -0.03352583,   0.08413311,
    0.023557603,   0.058695108,   -0.03169672,   -0.015069341,  0.028463965,   -0.08081926,   0.023019042,
    0.07448329,    0.029195376,   -0.0031292937, 0.002979553,   -0.01043473,   0.020290421,   0.035948016,
    -0.013068214,  -0.14437002,   0.013984859,   -0.026999082,  0.072234645,   0.010123427,   -0.023727309,
    -0.119222626,  -0.120623514,  -0.101863846,  -0.018985119,  -0.12936139,   -0.11510106,   0.012450467,
    -0.12286782,   0.01851142,    0.0049175275,  0.012761181,   -0.043844607,  0.026183426,   0.0005732278,
    -0.004507631,  0.0037628661,  -0.02520454,   -0.058891308,  -0.031899434,  0.011502794,   -0.092772305,
    0.020212494,   -0.0156601,    -0.013763661,  0.0047018034,  -0.044270985,  -0.028510189,  0.044463784,
    -0.025488352,  -0.15637311,   0.023329355,   0.057855785,   0.05939971,    -0.025778957,  0.029527873,
    -0.11892979,   -0.030715317,  -0.04786273,   0.04135018,    -0.090043366,  0.0073567554,  0.053930283,
    0.043262,      -0.00688985,   -0.03498016,   0.08282997,    0.042204063,   0.06849226,    -0.049501877,
    -0.08085754,   -0.03302276,   0.05100158,    -0.05726403,   0.10648756,    -0.025262028,  -0.041605346,
    0.041689955,   0.08840984,    -0.004173999,  0.09688242,    0.024852356,   0.07658803,    0.048349112,
    0.06762494,    0.024731234,   0.068337604,   0.0049590766,  0.05193726,    -0.039766233,  -0.019074153,
    0.015802756,   -0.00874117,   -0.024977459,  0.000706623,   -0.028027793,  0.03868606,    -0.124708645,
    -0.041894287,  -0.0005582869, 0.0131178945,  0.019812506,   -0.015669182,  0.014672056,   -0.047444712,
    0.019266268,   0.0442128,     -0.09576817,   0.060629286,   0.016271904,   -0.059490584,  0.013682982,
    0.019443592,   0.03937396,    0.018371474,   0.01281172,    0.086036205,   -0.033354387,  0.05017164,
    -0.122697085,  0.02376157,    -0.05981153,   0.08532616,    -0.03525166,   -0.025003593,  -0.011591822,
    -0.020989612,  0.041963816,   0.031429403,   -0.007031359,  0.08429086,    -0.03850344,   0.035511702,
    -0.04003218,   0.004656136,   0.02909784,    -0.10617155,   -0.02204068,   -0.02427197,   0.020434562,
    0.09787702,    -0.050744973,  0.08300381,    0.0710342,     -0.031391703,  -0.029119622,  0.015069126,
    -0.0054880977, -0.024989244,  0.011621267,   0.021875506,   -0.06613677,   0.00016721214, 0.12602335,
    -0.05614292,   0.07607715,    0.030650944,   -0.09098706,   0.055347808,   0.0016167786};

const std::vector<float> expected_idft2d_signal_size_results_3 = {
    0.51965636,   0.46109518,    -0.03224531,   0.122318566,  0.13996802,   0.04999275,    -0.09620099,  0.0069011925,
    0.664382,     0.58964956,    -0.031338193,  -0.14712875,  0.082166344,  0.07589669,    0.03586609,   0.0057471097,
    0.5606513,    0.5847055,     -0.1474433,    0.08164762,   -0.02039446,  -0.023951799,  0.0067006624, -0.023108874,
    0.6620194,    0.27644622,    0.06533013,    0.19844143,   0.050010286,  0.0615575,     -0.08088175,  0.019728234,
    0.54816973,   0.43787545,    -0.1545186,    -0.11976641,  0.06782712,   -0.050659865,  0.018574938,  -0.042057663,
    0.43190402,   0.5899364,     0.011021944,   -0.08617622,  -0.035539385, -0.009537175,  0.1456467,    0.21298885,
    0.20042025,   -0.06154151,   -0.0033541308, 0.027834859,  -0.10164561,  -0.10600591,   0.1779365,    -0.053386994,
    -0.05831848,  -0.05998933,   -0.08526638,   -0.074515656, 0.022936836,  0.07827924,    -0.10342336,  -0.08194167,
    -0.12801322,  0.06166912,    0.025966948,   0.12110167,   0.023929987,  0.034466177,   -0.039979175, -0.016688354,
    0.07164687,   0.12428331,    0.06549715,    0.03601643,   0.020718133,  -0.0071917605, 0.009838374,  0.0040146103,
    -0.013233974, -0.030871946,  0.047197368,   0.0123527115, -0.056864895, -0.046955965,  -0.004868932, -0.17846301,
    0.06855715,   0.027679857,   -0.1500722,    -0.056670498, -0.14638877,  0.023189232,   -0.029592723, 0.052305903,
    0.12751302,   -0.2093656,    -0.07194336,   -0.008120443, 0.07343415,   -0.031670257,  0.046054944,  -0.057585027,
    0.0018484419, 0.00040499022, 0.053662326,   -0.06284372,  -0.13057502,  -0.07417334,   -0.042746447, -0.013483183,
    0.007553713,  -0.15385008,   -0.08656611,   0.0701028,    -0.067524604, 0.026846979,   0.07731889,   0.002451243,
    -0.068792164, 0.020291083,   0.014570983,   0.0130328,    -0.0557093,   -0.08878534,   -0.1661258,   -0.009066362,
    0.03486307,   0.08238177,    -0.12802714,   0.093373135,  0.08243198,   -0.02568413,   0.034321107,  0.12715237,
    -0.08129562,  0.1487139,     0.10971667,    -0.045825522, 0.12990095,   0.041814834,   0.03660839,   0.012229371};

const std::vector<float> expected_idft2d_signal_size_results_4 = {
    0.50069946,    0.4904276,     0.009259958,   0.01438915,    -0.008930033,  0.1685353,     0.027075492,
    0.05019591,    0.123547375,   0.010753404,   -0.006528951,  0.06896549,    -0.018036481,  0.0016257297,
    -0.05024746,   -0.078786865,  0.5961777,     0.57672334,    0.011319853,   0.0018033399,  -0.029748872,
    -0.03491377,   0.03782171,    0.05982995,    0.11184341,    0.05324241,    -0.0040071756, 0.012312844,
    0.049088746,   -0.044715118,  -0.01899992,   -0.082886636,  0.48077154,    0.5565423,     0.014818694,
    0.026350802,   -0.12203966,   0.0050221197,  -0.035045236,  0.007882977,   0.012806054,   -0.04235148,
    0.06831992,    -0.027629035,  -0.04750082,   -0.057690993,  0.008230133,   0.035048254,   0.5499879,
    0.4701421,     0.14410123,    -0.058395956,  -0.013658326,  0.11536215,    -0.018243652,  0.04704798,
    0.069733724,   0.0700257,     0.019510295,   -0.06851628,   -0.09657783,   0.023751881,   -0.03731589,
    -0.12183073,   0.534263,      0.47527096,    -0.0118961595, 0.038099527,   -0.023198979,  -0.016842058,
    0.020281151,   -0.06888728,   0.04158451,    -0.006500121,  0.017227922,   -0.023705663,  -0.065157875,
    -0.061886493,  0.00823769,    -0.013699137,  0.48561165,    0.55944663,    -0.029265663,  -0.028232932,
    -0.031501852,  -0.056778878,  0.026412688,   0.12899265,    0.006460272,   -0.014392011,  0.04032879,
    -0.026880052,  0.0731035,     0.09133217,    -0.10169062,   0.05235266,    0.06892046,    -0.0295484,
    0.105304524,   0.0118386075,  0.0025054235,  -0.025319275,  0.09081747,    0.036444083,   -0.014070861,
    -0.05731819,   -0.0084318025, -0.12696962,   0.03937552,    -0.074440725,  -0.047060557,  -0.020518059,
    -0.052758366,  0.022431955,   -0.03708367,   -0.02261317,   0.06566422,    -0.022002619,  0.019627463,
    -0.031626925,  0.04216373,    -0.012684815,  -0.054740556,  -0.0037655067, -0.10157557,   0.039671652,
    -0.0487277,    -0.0116044395, -0.02558843,   0.036774725,   -0.06290054,   0.041731425,   0.046522886,
    0.007584708,   -0.063871264,  0.014685718,   -0.026041836,  -0.028664894,  -0.05809094,   0.028512653,
    0.029267922,   0.043192253,   0.037230983,   0.019329412,   0.049078666,   0.019939065,   -0.016842797,
    0.012881201,   -0.026447453,  -0.0014885329, 0.018214535,   0.004656654,   -0.09413797,   0.026110932,
    0.061685666,   0.056731943,   -0.019016284,  0.0042851716,  0.06388497,    -0.043995842,  -0.01923573,
    0.014247827,   -0.027373053,  -0.067749165,  -0.020197151,  -0.0062670596, 0.044943634,   0.04798174,
    -0.0048591197, 0.0052249283,  0.035372503,   0.048372734,   0.010621414,   -0.056204394,  -0.044176046,
    -0.11560526,   -0.035333157,  0.0016496852,  -0.013274903,  0.078784436,   -0.022540363,  -0.032977562,
    -0.012144292,  0.03639986,    0.018095985,   0.10897899,    -0.004816832,  -0.0037983656, 0.012832675,
    0.035309885,   -0.06408509,   0.017382117,   -0.08315392,   -0.033107996,  0.08171202,    0.07782332,
    -0.06865839,   -0.10764925,   -0.054046266,  -0.005835831,  0.012306612,   -0.06220748,   0.07638017,
    0.013183124,   0.041668497,   -0.055150427,  0.011025649,   0.0047506727,  0.010130197,   0.019823164,
    -0.03397094,   0.002564197,   -0.056729004,  -0.09857107,   0.09008947,    0.031346098,   -0.0024604239,
    -0.0047197156, -0.0550084,    -0.110036984,  -0.026545763,  0.017181754,   -0.018559802,  0.007102821,
    -0.062645376,  -0.038759947,  0.11738676,    0.057442024,   -0.0405657,    0.00024135411, -0.017878262,
    0.042187072,   -0.017206412,  -0.004294824,  -0.15039666,   0.017837571,   0.010826452,   0.017910408,
    0.022387126,   0.038914144,   0.039849013,   0.0025931299,  -0.062299855,  -0.031667564,  -0.06573889,
    0.067470476,   -0.024389353,  0.09938391,    0.026338965,   0.046804663,   0.018697795,   0.06765098,
    0.06032055,    0.029475141,   -0.036904067,  0.012176782,   -0.019729197,  -0.0006175935, 0.05483261,
    0.042380556,   -0.04253985,   0.0012245048,  -0.020328768,  -0.013062598,  0.0053691603,  -0.065394655,
    0.05558199,    -0.021914857,  0.1706096,     -0.03785784,   -0.020459495,  -0.01513383,   0.017118126,
    0.14779843,    -0.005776977,  -0.03670439,   -0.051734768,  -0.08797553,   0.024524372,   -0.083074294,
    -0.028641924,  -0.031074718,  0.11779925,    -0.0073087253, -0.0006238073, -0.025388103,  0.043085426,
    -0.019948414,  0.13103837,    -0.05180665,   0.04730244,    0.037148323,   -0.041258633,  0.024768397,
    0.025847614,   -0.06161034,   -0.010785304,  -0.03626065,   -0.016979318,  -0.02865681,   -0.01828878,
    -0.019304248,  0.03128355,    0.0041085277,  0.0024671853,  0.008217737,   0.03448409,    0.030935789,
    -0.022427216,  -0.0295653,    -0.073335916,  -0.0037255269, 0.00038855523, -0.0448144,    -0.029116416,
    -0.047692195,  -0.011189029,  -0.022148391,  0.01491251,    -0.01919859,   0.0401923,     -0.08648369,
    0.058548566,   0.02578877,    0.00094457157, 0.019820662,   0.0145471115,  -0.04303372,   -0.052616134,
    0.019346833,   0.052684337,   0.060835492,   0.031265024,   -0.045283258,  -0.052162454,  0.016602138,
    -0.014453076,  -0.022049919,  0.043205604,   -0.069325164,  0.020091549,   -0.007931273,  -0.04543579,
    -0.03203051,   -0.008483101,  -0.034514427,  0.04786063,    0.007603269,   -0.097250104,  -0.079659835,
    0.032755584,   0.036081,      0.018139064,   0.029964916,   -0.06415692,   0.0974453,     -0.002073994,
    0.029044904,   -0.08739063,   -0.06471936,   0.0054590777,  0.0011593401,  -0.039340377,  0.064394005,
    0.028959818,   -0.053989142,  -0.06349799,   0.07390169,    0.02351752,    0.03192448,    -0.037667163,
    -0.07394214,   0.017038213,   0.036862355,   -0.043745972,  -0.014133457,  -0.028366454,  0.03523228,
    -0.04687486,   0.0940376,     0.120025754,   0.020736795,   0.022597576,   -0.023962533};

const std::vector<float> expected_idft2d_signal_size_results_5 = {
    0.43990576,    0.37177825,    0.018297018,   0.13355713,    0.09609352,    0.04071089,   -0.092824854,
    0.03483837,    0.52699745,    0.4414857,     0.017332105,   -0.043757387,  0.06474239,   0.06351439,
    -0.0062755626, -0.028125575,  0.39462435,    0.4589463,     -0.100476444,  0.031446725,  0.0050833533,
    -0.07491388,   0.0050572394,  -0.012939147,  0.5018034,     0.2513064,     0.06601329,   0.12563014,
    0.01634291,    0.012481438,   -0.09012963,   -0.0073484736, 0.45544818,    0.3760468,    -0.06427583,
    -0.043393508,  0.057562195,   -0.019032322,  -0.031661525,  -0.056141183,  0.34964913,   0.43898678,
    0.03130693,    -0.058825802,  -0.076260805,  0.013013445,   0.070871785,   0.17149778,   -0.06417048,
    -0.06922141,   -0.040472697,  -0.062998466,  -0.058540687,  0.031437594,   0.08273152,   -0.12060501,
    -0.2061106,    0.118685186,   -0.0009581372, -0.08492219,   -0.0041653924, 0.0067129806, -0.06255558,
    0.05957415,    -0.118542455,  0.14462362,    -0.009236974,  0.022580916,   -0.03849992,  0.03493386,
    0.0035450687,  0.010597763,   -0.0013405303, 0.11584599,    -0.03715597,   0.010403095,  -0.020171534,
    0.021085449,   -0.018025603,  0.006588485,   -0.1273406,    0.058172766,   0.0065336376, -0.09943622,
    -0.0022615304, 0.00024407124, -0.030914608,  0.022675768,   -0.15166818,   0.07935919,   -0.040845793,
    0.016231818,   0.039729055,   0.07636168,    0.0033444613,  0.07087032,    0.117220305,  0.15729488,
    -0.048147924,  0.094027564,   -0.023140147,  -0.07949994,   0.045205563,   -0.00916888,  0.0492486,
    0.115183815,   -0.08267819,   -0.029814515,  -0.014258375,  0.08933942,    -0.010548074, -0.09738556,
    -0.09092414,   0.11030196,    0.0021819957,  0.08301844,    0.0689266,     0.030335644,  -0.019274916,
    -0.0315443,    0.050106198,   0.19658162,    0.0503511,     0.07378791,    0.027816363,  -0.0067463894,
    0.0042154863,  -0.0070224535, 0.030608267,   0.1334948,     -0.0040946133, 0.070922405,  -0.02647118,
    -0.040317588,  0.072931446,   -0.15662149,   0.086943075,   0.18035986,    -0.064264745, -0.06842194,
    -0.11555961,   -0.08764592,   -0.04262618,   0.022765271,   0.1726031,     -0.11350446,  0.01339009,
    0.00404248,    0.07330416,    -0.04052531,   0.024228431,   0.025112806,   0.15811935,   0.0038504712,
    0.08746172,    -0.09458299,   -0.072628036,  -0.0015422329, -0.04602814,   0.007808782,  0.095344745,
    -0.025132503,  -0.089144886,  0.061655734,   -0.087049335,  -0.0064667296, 0.047338445,  -0.025052967,
    0.10809028,    0.038190633,   0.010420183,   0.014206182,   -0.075355306,  -0.014443267, -0.12931237,
    0.06473562,    0.21191315,    0.031243857,   -0.01827092,   0.043444555,   0.071482405,  -0.053060792,
    0.03741809,    0.088275075,   0.095881015,   0.13641034,    0.06575319,    -0.10507756,  0.07346845,
    0.022813302,   0.049534187,   0.0534714,     0.1820309,     -0.15615918,   -0.05060928,  -0.026595712,
    0.024039716,   -0.039806653,  0.06844979,    -0.034248125,  0.07965717,    -0.14914,     -0.08409975,
    -0.031411037,  0.00083757995, -0.078021966,  0.01510364,    -0.031549543,  0.15968925,   -0.19621487,
    -0.011366145,  0.07415028,    -0.012449766,  0.053472463,   0.007374543,   0.02159267,   0.0062146843,
    -0.18090405,   0.05576966,    0.02346333,    0.066386685,   -0.04679683,   -0.003917059, -0.042276695,
    -0.0008301586, -0.10957299,   -0.15524064,   0.014422208,   -0.0069176825, -0.011133326, 0.00025371276,
    0.008443515,   0.038360506,   -0.06878601,   -0.021283172,  0.027421257,   0.02659573,   0.030924385,
    0.07153812,    -0.04108064};

const std::vector<float> expected_idft3d_signal_size_results = {
    0.47360885,    0.43109864,    -0.015375684,   0.037675954,   0.032973632,   -0.02922261,   0.011034667,
    0.010564784,   -0.014908149,  0.017389147,    -0.09910002,   0.06296675,    -0.03663863,   -0.018516757,
    -0.0011850556, 0.008715436,   -0.016253775,   0.020636003,   0.047444995,   -0.0024833402, -0.054806184,
    0.063673705,   0.08815336,    -0.019982928,   0.023201711,   -0.0112207085, -0.015865667,  0.032094162,
    -0.04799268,   -0.0009248348, -0.0005214396,  0.11000099,    -0.009374426,  0.07988078,    -0.03512358,
    0.027058495,   0.01562349,    0.010030186,    -0.0323284,    -0.015812825,  0.06827773,    0.017847355,
    -0.0024901072, -0.024341352,  -0.0009585398,  -0.009607499,  0.015930153,   0.011760445,   0.034274444,
    0.027811723,   0.12735432,    -0.09866718,    -0.004353502,  0.05861713,    -0.0083505465, -0.056518648,
    -0.026644187,  0.019356728,   -0.012300177,   -0.039047725,  0.006673645,   -0.11678744,   0.020548198,
    -0.016014362,  -0.0039747558, 0.010135546,    0.020794585,   0.04208699,    -0.016170124,  5.5018212e-05,
    0.007777319,   0.008232947,   -0.01095827,    0.018842613,   -0.010135147,  0.0097200265,  -0.01967326,
    -0.026630066,  0.0063641453,  -0.0345983,     0.010849577,   -0.04473317,   -0.011817071,  0.02476529,
    0.0155312605,  -0.009187386,  -0.036735293,   -0.01913485,   0.0043119635,  0.018222943,   0.03163258,
    -0.022928724,  0.01679045,    0.00042682493,  0.02686337,    0.026272336,   0.05542246,    -0.070824265,
    0.049857102,   -0.057350874,  0.004540915,    0.024931736,   -0.0066266567, -0.0060906713, -0.018460626,
    0.030870803,   0.02064466,    -0.027539674,   -0.0001978664, -0.009443704,  0.032834202,   -0.018512815,
    0.0006455081,  0.05246358,    -0.00452247,    -0.002136842,  0.006376774,   -0.04188078,   0.024743602,
    0.027263625,   0.009806405,   -0.022745509,   -0.0042487574, 0.015461739,   -0.020853115,  0.012601041,
    0.01992702,    0.008631575,   0.028762273,    -0.014992784,  0.009239552,   0.008044719,   -0.005379937,
    -0.0020708065, 0.018219728,   -0.012548579,   0.03744698,    -0.03307585,   -0.0156224305, -0.003425684,
    0.009117395,   -0.035289288,  -0.02348312,    -0.0032391632, -0.03008389,   -0.0019097558, 0.018912613,
    0.01636466,    0.0156074455,  0.0038367372,   0.05714075,    -0.04073001,   -0.008098723,  -0.017560972,
    0.005987327,   -0.019382497,  0.029533077,    -0.039648466,  0.025838673,   -0.03902175,   0.009098698,
    -0.0016117467, 0.0051164175,  0.0037725042,   -0.014878556,  -0.028334552,  -0.005686572,  0.013636959,
    -0.03393201,   0.012674245,   -0.020131484,   -0.06613781,   0.031012673,   0.03862036,    0.009845047,
    -0.007944993,  0.008455606,   0.017170515,    0.063504405,   -0.023375345,  0.046734862,   -0.009648957,
    -0.037776463,  -0.015671387,  0.026074437,    -0.021695163,  -0.0055521606, 0.0117665045,  -0.0033777568,
    0.032647174,   0.031938843,   -0.03217729,    -0.027973937,  0.023887817,   -0.0062627294, -0.014567699,
    0.023382662,   -0.013312814,  -0.00063125056, -0.030556878,  -0.01716187,   -0.035170853,  0.016255094,
    -0.0032832257, 0.028565489,   0.01738093,     -0.046995874,  -0.00492523,   -0.0050363657, 0.006696142};

const std::vector<float> expected_idft1d_signal_size_results = {
    0.45077854,   0.42283636,    0.5254696,     0.49784923,    0.5858539,     0.38200417,    0.46620184,
    0.34846103,   0.42421392,    0.36906072,    0.41839844,    0.64023143,    0.49543527,    0.61651653,
    0.33086562,   0.38194793,    -0.013190938,  -0.11509278,   5.4674492e-05, 0.1155831,     -0.059459202,
    0.1080654,    -0.038735352,  -0.13293009,   -0.044217605,  -0.08810778,   -0.1887316,    -0.04305593,
    0.0064542103, 0.02646665,    -0.19942014,   0.16640756,    0.13297448,    -0.060627896,  -0.0105416095,
    -0.08473419,  -0.04953826,   -0.0053528296, -0.062126894,  0.15922469,    -0.0811423,    0.1871564,
    -0.06467852,  0.019565826,   -0.06630114,   0.05153398,    -0.2517371,    0.012869691,   0.030862298,
    0.18221883,   0.18004268,    0.115521945,   -0.006216203,  -0.04221012,   -0.099812254,  0.10728653,
    -0.18543659,  -0.018842628,  0.03197636,    0.13652334,    -0.063658535,  0.2630217,     0.113256425,
    0.10151879,   0.20121424,    0.038065623,   0.04212692,    0.15088516,    0.11653197,    -0.047717404,
    0.17632447,   -0.16700539,   0.20303649,    -0.025800077,  0.14718783,    0.111731626,   0.083313815,
    -0.09204115,  0.03319706,    -0.025887946,  0.12955333,    -0.1840305,    0.30852506,    -0.033531368,
    0.3569616,    -0.19672023,   0.003184589,   -0.15916768,   0.047273237,   -0.1945002,    0.24541551,
    -0.013310499, 0.19067605,    -0.043925222,  0.019669108,   0.006493839,   0.047401913,   -0.1429027,
    -0.06374044,  -0.24836907,   -0.0050352793, -0.13352643,   0.04469287,    -0.11333085,   -0.029562438,
    -0.08143895,  -0.1432822,    -0.0007748668, 0.015231887,   -0.2503828,    0.062242992,   0.033986002,
    0.50197685,   0.5042996,     0.38166192,    0.33678776,    0.6271677,     0.4345175,     0.4634395,
    0.35949785,   0.5400181,     0.5938341,     0.5252213,     0.50813717,    0.5442304,     0.42320782,
    0.30390763,   0.5501661,     -0.04264457,   0.27560052,    -0.20748335,   0.12261487,    -0.14888233,
    0.2118158,    -0.27173674,   0.14285998,    -0.0120631205, -0.023952818,  -0.06255174,   0.17351858,
    -0.27042615,  -0.14776227,   0.027333874,   -0.08371357,   -0.18530807,   -0.06548692,   -0.08052732,
    -0.22452545,  -0.012073594,  0.1693748,     -0.13727319,   0.054494813,   -0.052982204,  0.34828788,
    0.058887307,  -0.0070416844, 0.035075426,   0.058569312,   -0.13160224,   0.2115795,     -0.060909573,
    0.30598193,   0.13896666,    0.104170956,   0.15620923,    0.04350682,    0.17609014,    -0.043268614,
    0.020355552,  0.13579452,    0.036094327,   0.029404454,   0.119380735,   -0.017330358,  -0.013081285,
    0.16276608,   0.08908949,    -0.013249721,  0.030841062,   -0.09900363,   -0.104013704,  0.14226541,
    -0.022373218, 0.13952398,    0.110540666,   0.050296895,   0.19052894,    0.30660385,    0.23429887,
    -0.07728504,  0.06723292,    0.053966362,   0.10566526,    -0.09358089,   0.12995106,    -0.09512488,
    0.22091591,   0.006204946,   0.021672454,   -0.08378312,   0.09504456,    0.0060912203,  0.21140935,
    -0.170751,    0.048105046,   -0.12022613,   0.031716354,   0.010118494,   -0.0024824888, -0.10846503,
    0.105488904,  -0.12306744,   0.0021025285,  -0.2050574,    0.059306182,   -0.06526719,   0.0758878,
    -0.12858643,  0.039985638,   -0.09187355,   0.15090118,    -0.06106589,   -0.009392858,  -0.14116207,
    0.50924444,   0.47538188,    0.36653236,    0.5176384,     0.4713962,     0.38925174,    0.4561904,
    0.37097445,   0.33398703,    0.46408504,    0.25756317,    0.62005824,    0.6271038,     0.51077616,
    0.2500916,    0.36728626,    -0.26136193,   0.053446535,   -0.10429463,   0.07333,       -0.125614,
    0.13846059,   -0.034961026,  0.06588599,    -0.12191007,   -0.0030780777, -0.1388123,    -0.063579924,
    -0.20671263,  0.12017858,    -0.035247978,  -0.13096417,   0.030405179,   0.31703302,    -0.18467914,
    -0.027100164, -0.10349936,   -0.03151396,   -0.14566809,   0.1760046,     0.119887486,   0.042870738,
    -0.10979464,  0.07983213,    -0.21066193,   0.058638398,   -0.14007244,   -0.025705764,  -0.15512581,
    0.14527217,   -0.008849559,  0.103681795,   0.05173718,    0.022327188,   -0.11486209,   0.33013803,
    -0.13087325,  0.15843569,    -0.13177976,   0.15052482,    -0.024118312,  0.19754727,    0.043348696,
    0.048077006,  0.05479733,    0.045596164,   -0.061486844,  0.13432872,    0.09805413,    -0.03079012,
    0.018617248,  0.057953924,   0.07086571,    -0.107639074,  0.10912783,    -0.027383184,  0.086062506,
    0.021643614,  0.0033594542,  0.1245168,     -0.008100906,  -0.071167745,  0.15794107,    -0.11394106,
    0.20086603,   -0.14925118,   -0.023519892,  -0.17317979,   0.16438125,    0.055674464,   0.043398805,
    -0.14720002,  -0.016775567,  -0.09511416,   0.024722071,   -0.21830398,   0.038695235,   0.009794927,
    -0.16068184,  -0.18155417,   0.09074758,    -0.060050044,  -0.14183791,   0.050801322,   0.039122116,
    -0.22079241,  0.122001916,   -0.04277471,   -0.10941203,   -0.13381958,   -0.08884025,   -0.1331459,
    0.43459043,   0.46945596,    0.5461507,     0.21173653,    0.46469456,    0.31414452,    0.372732,
    0.64673513,   0.43870863,    0.50919855,    0.4184354,     0.3038723,     0.5814269,     0.43418613,
    0.27493507,   0.32782063,    0.05329496,    0.13938421,    -0.14763448,   -0.020887244,  -0.06212297,
    0.0026264533, -0.16405852,   -0.15793757,   -0.10529227,   -0.017391797,  -0.010122827,  0.16707306,
    -0.24121732,  0.058287155,   -0.2669763,    -0.13610138,   -0.058303684,  0.21642019,    0.18846658,
    0.074460745,  0.025465574,   0.105897866,   -0.120273784,  0.12273249,    0.0060386043,  0.22629398,
    -0.18080746,  0.12264217,    -0.09611444,   -0.039596815,  0.13664094,    0.12015629,    -0.02670856,
    0.11202149,   0.062823564,   0.16666472,    0.02688467,    -0.0496769,    0.1111543,     0.20479687,
    0.09844535,   0.22211954,    -0.030972388,  -0.013431635,  0.075517,      -0.01832214,   -0.067268424,
    0.009361826,  0.066086665,   0.114655524,   -0.093547,     0.0037781172,  0.013148758,   0.0049218494,
    0.14736433,   -0.029691944,  0.19633494,    -0.0039153695, 0.24658194,    0.0027496407,  0.085013606,
    0.02836323,   -0.023116423,  0.031402636,   0.25721642,    -0.041657127,  0.08742507,    -0.26150823,
    0.21623416,   -0.10428249,   0.030336773,   0.033717245,   -0.052863047,  -0.05433597,   -0.0014412233,
    -0.05366828,  0.09244677,    -0.13540235,   0.049008,      0.032415926,   -0.012353104,  -0.026780326,
    0.08365354,   -0.11595764,   0.0041189813,  -0.19876777,   0.05539961,    -0.03458054,   0.3030866,
    0.06549558,   0.100598566,   -0.0328084,    -0.008204469,  -0.20897065,   -0.08830016,   -0.15857375};

const std::vector<dft_test_params> dft_params = {
    // With original shape
    {{2, 10, 10, 2}, {2, 10, 10, 2}, {2}, input_data, expected_dft1d_results},
    {{2, 10, 10, 2}, {2, 10, 10, 2}, {1, 2}, input_data, expected_dft2d_results},
    {{2, 10, 10, 2}, {2, 10, 10, 2}, {0, 1, 2}, input_data, expected_dft3d_results},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {2}, input_data_1, expected_dft1d_results_1},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {1, 2}, input_data_1, expected_dft2d_results_1},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 1}, input_data_1, expected_dft2d_signal_size_results_2},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 2}, input_data_1, expected_dft2d_signal_size_results_4},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 1, 2}, input_data_1, expected_dft3d_results_1},
    // With changed shape
    {{2, 10, 10, 2}, {2, 20, 10, 2}, {1}, input_data, expected_dft1d_signal_size_results},
    {{4, 6, 8, 2}, {5, 6, 9, 2}, {0, 2}, input_data_1, expected_dft2d_signal_size_results_1},
    {{4, 6, 8, 2}, {3, 6, 4, 2}, {0, 2}, input_data_1, expected_dft2d_signal_size_results_3},
    {{4, 6, 8, 2}, {5, 6, 4, 2}, {0, 2}, input_data_1, expected_dft2d_signal_size_results_5},
    {{4, 6, 8, 2}, {3, 7, 5, 2}, {0, 1, 2}, input_data_1, expected_dft3d_signal_size_results},
};

const std::vector<idft_test_params> idft_params = {
    // Inversed to DFT with original shape cases
    {{2, 10, 10, 2}, {2, 10, 10, 2}, {2}, input_data, expected_dft1d_results},
    {{2, 10, 10, 2}, {2, 10, 10, 2}, {1, 2}, input_data, expected_dft2d_results},
    {{2, 10, 10, 2}, {2, 10, 10, 2}, {0, 1, 2}, input_data, expected_dft3d_results},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {2}, input_data_1, expected_dft1d_results_1},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {1, 2}, input_data_1, expected_dft2d_results_1},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 1}, input_data_1, expected_dft2d_signal_size_results_2},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 2}, input_data_1, expected_dft2d_signal_size_results_4},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 1, 2}, input_data_1, expected_dft3d_results_1},
    // Other cases with original shape
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 1}, expected_idft2d_signal_size_results_2, input_data_1, "v2"},
    {{4, 6, 8, 2}, {4, 6, 8, 2}, {0, 2}, expected_idft2d_signal_size_results_4, input_data_1, "v2"},
    // With changed shape
    {{4, 6, 8, 2}, {4, 7, 8, 2}, {1}, expected_idft1d_signal_size_results, input_data_1},
    {{4, 6, 8, 2}, {5, 6, 9, 2}, {0, 2}, expected_idft2d_signal_size_results_1, input_data_1},
    {{4, 6, 8, 2}, {3, 6, 4, 2}, {0, 2}, expected_idft2d_signal_size_results_3, input_data_1},
    {{4, 6, 8, 2}, {5, 6, 4, 2}, {0, 2}, expected_idft2d_signal_size_results_5, input_data_1},
    {{4, 6, 8, 2}, {3, 7, 5, 2}, {0, 1, 2}, expected_idft3d_signal_size_results, input_data_1},
};

INSTANTIATE_TEST_SUITE_P(smoke_dft_float32,
                         dft_gpu_test_f32,
                         testing::ValuesIn(dft_params),
                         dft_gpu_test_f32::PrintToStringParamName);

INSTANTIATE_TEST_SUITE_P(smoke_dft_float16,
                         dft_gpu_test_f16,
                         testing::ValuesIn(dft_params),
                         dft_gpu_test_f16::PrintToStringParamName);

INSTANTIATE_TEST_SUITE_P(smoke_idft_float32,
                         idft_gpu_test_f32,
                         testing::ValuesIn(idft_params),
                         idft_gpu_test_f32::PrintToStringParamName);

INSTANTIATE_TEST_SUITE_P(smoke_idft_float16,
                         idft_gpu_test_f16,
                         testing::ValuesIn(idft_params),
                         idft_gpu_test_f16::PrintToStringParamName);

}  // namespace
